#include <alloc.h>
#include <time.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "sun.h"

/*-------------------------------------------------------------------------
  Algotithm is source of Almanac for Computers, 1990
  published by Nautical Almanac Office
  United States Naval Observatory
  credit : http://www.best.com/~williams/sunrise_sunset_algorithm.htm
  -------------------------------------------------------------------------*/

#define ZENITH	90.83333
#define RAD(x)	(x * 3.1415926535 / 180)
#define SIN(x)	sin(x * M_PI / 180.0)
#define COS(x)	cos(x * M_PI / 180.0)
#define TAN(x)  tan(x * M_PI / 180.0)
#define ACOS(x) acos(x) * 180.0 / M_PI
#define ATAN(x) atan(x) * 180.0 / M_PI

/* Calculate sunrise and sunset */

/* -----------------------------------------------------------------------
   Calcule le jour de l'annee
   -----------------------------------------------------------------------*/
static int SUN_dayOfYear(void)
{
	struct tm * tblock;
	time_t t;

	t = time(NULL);
	tblock = localtime(&t);
	return tblock->tm_yday + 1;
}

static double SUN_compute(double latitude, double longitude, int day, int nRiseSet)
{
	/* Convert the longitude to hour value and calculate an approximate
	   time */
	double lngHour = longitude / 15.0;

	double t;
	if( nRiseSet == SUN_RISE )
		t = day + ((6 - lngHour) / 24);
	else
		t = day + ((18- lngHour) / 24);

	/* Calculate the sun mean anomaly */
	double M = (0.9856 * t) - 3.289;

	/* Calculate the sun's true longitude */
	double L = M + (1.916 * SIN(M)) + (0.020 * SIN(2 * M)) + 282.634;

	while( L >= 360.0 )
		L -= 360.0;

	/* Calculate the right ascension */
	double RA = ATAN(0.91764 * TAN(L));

	/* RA value needs to be in the same quadrant as L */
	double Lquadrant  = (floor( L/90.0)) * 90.0;
	double RAquadrant = (floor(RA/90.0)) * 90.0;

	RA = RA + (Lquadrant - RAquadrant);

	/* RA value needs to be converted into hours */
	RA /= 15.0;

	/* Calculate the sun declinaison */
	double sinDec = 0.39782 * SIN(L);
	double cosDec = cos(asin(sinDec));

	/* Calculate the sun's local hour angle */
	double cosH = (COS(ZENITH) - (sinDec * SIN(latitude))) / (cosDec * COS(latitude));

	double H;
	if( nRiseSet == SUN_RISE )
	{
		if( cosH > 1.0 )
			return -1.0;	/* The sun never rises today */
		H = 360.0 - ACOS(cosH);
	}
	else
	{
		if( cosH < -1.0 )
			return -1.0;	/* The sun never sets today */
		H = ACOS(cosH);
	}


	H /= 15.0;

	/* Calculate local mean time of rising/setting */
	double T = H + RA - (0.06571 * t) - 6.622;

	/* Ajust back UTC */
	double UT = T - lngHour;

	while( UT < 0.0 )
		UT += 24.0;
	while( UT >= 24.0 )
		UT -= 24.0;

	return UT;
}

static char* SUN_time2str(double dTime, char* pszString)
{
	if( dTime < 0.0 )
	{
		strcpy(pszString, "--:--");
		return pszString;
	}

	int hour = (int) dTime;
	dTime *= 100.0;
	int min  = ((int) dTime % 100) * 60 / 100;
	sprintf(pszString, "%02d:%02d", hour, min);
	return pszString;
}

char * SUN_calculate(double dLatitude, double dLongitude, int nRiseSet, char * pszBuffer)
{

	/* Quel jour depuis le debut de l'annee ? */
	int day = SUN_dayOfYear();

	SUN_time2str(SUN_compute(dLatitude, dLongitude, day, nRiseSet),
		pszBuffer);

	return pszBuffer;
}
