#include <alloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "protocol.h"
#include "tools.h"
#include "hops.h"
#include "memory.h"
#include "dx.h"
#include "announce.h"
#include "node.h"
#include "buffers.h"
#include "mail.h"
#include "clulink.h"
#include "users.h"
#include "database.h"
#include "message.h"
#include "pavillon.h"
#include "log.h"

/*------ TALK -------------------------------------------------------------
  PC10^FromUser^ToUser1^text^BellFlag^ToUser2^FromPC^~
*/
int PAVILLON_pc10(int StreamNum, int nParams, char **pArg)
{
	char* pszFromUser = pArg[0];
	char* pszToUser1  = pArg[1];
	char* pszText     = pArg[2];
	char* pszBellFlag = pArg[3];
	char* pszToUser2  = pArg[4];
	char* pszFromPC   = pArg[5];
	char  cFlag;

	/* Verifier la validite du protocole */
	if( nParams != 6 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromUser) || ! TOOLS_isCall(pszToUser1) ||
	    ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC10, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traitement du BellFlag */
	if( *pszBellFlag == '*' )
		cFlag = PROTOCOL_TALK_BELL;
	else
		cFlag = 0;

	/* Link external ? */
	if( *pszToUser2 != ' ' )
	{
		/* OUI */
		PROTOCOL_sendTalk(StreamNum, pszFromPC, pszFromUser,
				  pszToUser1, pszToUser2, pszText,
				  cFlag, TOOLS_whatDateTime(),
				  HOPS_probable(pszFromPC));
	}
	else
	{
		/* NON */
		PROTOCOL_sendTalk(StreamNum, pszFromPC, pszFromUser, "",
				  pszToUser1, pszText, cFlag,
				  TOOLS_whatDateTime(),
				  HOPS_probable(pszFromPC));
	}
	return PROTOCOL_OK;
}

/*------ DX info -----------------------------------------------------------
  PC11^DXFreq^DXCall^Date^Time^DXInfoText^Logger^FromPc^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc11(int StreamNum, int nParams, char **pArg)
{
	char* pszDxFrequency	= pArg[0];
	char* pszDxCall		= pArg[1];
	char* pszDate		= pArg[2];
	char* pszTime		= pArg[3];
	char* pszDxInfoText	= pArg[4];
	char* pszLogger		= pArg[5];
	char* pszFromPC		= pArg[6];
	char* pszHops		= pArg[7];

	int   nHops;
	ulong lFrequency;
	char  szTime[10];

	/* Verifier la validite du protocole */
	if( nParams != 8 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszLogger) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, pszFromPC, StreamNum, PC11, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lire le hop count */
	nHops = HOPS_str2int(StreamNum, pszFromPC, pszHops, PC11);

	/* Decoder la frequence */
	lFrequency = (unsigned long) (atof(pszDxFrequency) * 10.0 + 0.000001);
	if( ! DX_checkFrequency(lFrequency) )
		return PROTOCOL_BADFREQUENCY;	/* Frequence incorrecte */

	/* Enlever le Z a la fin de l'heure */
	strncpy(szTime, pszTime, 4);
	szTime[4] = SNULL;

	PROTOCOL_sendDx(StreamNum, pszFromPC, pszLogger, lFrequency,
			pszDate, szTime, pszDxCall, pszDxInfoText, nHops);

	return PROTOCOL_OK;
}

/*------ Announcement ------------------------------------------------------
  PC12^FromUser^ToPC^text^SysopFlag^FromPC^WxFlag^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc12(int StreamNum, int nParams, char **pArg)
{
	char* pszFromUser	= pArg[0];
	char* pszToPC		= pArg[1];
	char* pszText		= pArg[2];
	char* pszSysopFlag	= pArg[3];
	char* pszFromPC		= pArg[4];
	char* pszWxFlag		= pArg[5];
	char* pszHops		= pArg[6];

	int	nHops;
	char	cFlag;

	/* Verifier la validite du protocole */
	if( nParams != 7 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromUser) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, pszFromPC, StreamNum, PC12, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, pszFromPC, pszHops, PC12);

	/* Lecture des flags */
	if( *pszSysopFlag != ' ' )
		cFlag = ANNOUNCE_TOSYSOP;	/* flag SysOp */
	else if( *pszToPC == '*' )
		cFlag = ANNOUNCE_ALL;		/* flag ClusterWide */
	else
	{
		/* Est-ce pour un cluster ou une liste de distribution ? */
		/* ICI */
		cFlag = ANNOUNCE_TOCLUSTER;
	}

	if( *pszWxFlag == '1' )
		cFlag = ANNOUNCE_WX;		/* flag Wx */

	/* Faire suivre le protocole */
	PROTOCOL_sendAnnounce(StreamNum, pszFromPC, pszFromUser, pszToPC,
			      TOOLS_whatDateTime(), cFlag, pszText, nHops);

	return PROTOCOL_OK;
}

/*------ cluster-wide user login -----------------------------------------
  PC13^FromUser^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc13(int StreamNum, int nParams, char **pArg)
{
	char* pszFromUser	= pArg[0];
	char* pszHops		= pArg[1];

	int	nHops;

	/* Verifier la validite du protocole */
	if( nParams != 2 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromUser) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC13, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC13);

	/* Faire suivre le protocole */
	PROTOCOL_sendClusterConfLogin(StreamNum, "", pszFromUser, nHops);

	return PROTOCOL_OK;
}

/*------ cluster-wide user logout ----------------------------------------
  PC14^FromUser^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc14(int StreamNum, int nParams, char **pArg)
{
	char* pszFromUser	= pArg[0];
	char* pszHops		= pArg[1];

	int	nHops;

	/* Verifier la validite du protocole */
	if( nParams != 2 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromUser) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC14, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC14);

	/* Faire suivre le protocole */
	PROTOCOL_sendClusterConfLogout(StreamNum, "", pszFromUser, nHops);

	return PROTOCOL_OK;
}

/*------ cluster-wide conference message ---------------------------------
  PC15^FromUser^Message^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc15(int StreamNum, int nParams, char **pArg)
{
	char* pszFromUser	= pArg[0];
	char* pszMessage	= pArg[1];
	char* pszHops		= pArg[2];

	int	nHops;

	/* Verifier la validite du protocole */
	if( nParams != 3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromUser) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC15, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC15);

	/* Faire suivre le protocole */
	PROTOCOL_sendClusterConferenceMessage(StreamNum, "", pszFromUser,
		pszMessage, nHops);

	return PROTOCOL_OK;
}


/*------ Add User ----------------------------------------------------------
  PC16^FromPC^User ConfMode Here^User ConfMode Here^...^...^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc16(int StreamNum, int nParams, char **pArg)
{

	char  * pszFromPC = pArg[0];
	char	szPavillon[1024];
	char    szCluLink[1024];
	int	nCluLinkPos;
	int	index;
	int	nHops;
	extern	char	PROTOCOL_initDone[65];

	/* Verifier la validite du protocole */
	if( nParams < 3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC16, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Vrifier que FromPC est bien dans la table */
	if( ! NODE_isNodeConnected(pszFromPC) )
	{
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "*** Protocol error : Can't add users on an unknown node (%s).\n", pszFromPC);
		return PROTOCOL_NODEUNKNOWN;
	}

	/* Prparer les protocoles qu'il faudra envoyer aux adjacents */
	sprintf(szPavillon, "PC16^%1s^", pszFromPC);
	nCluLinkPos = PROTOCOL_setCluLink(szCluLink, CL(3));
	nCluLinkPos = PROTOCOL_addPascalString(szCluLink, pszFromPC, nCluLinkPos);

	/* Lire maintenant la liste des utilisateurs */
	for(index = 1; index < PPARAM-1; index++)
	{
		char	szBuffer[128];
		char	szUserCall[16];
		char*	pPtr1;
		char*	pPtr2;
		char	cFlag;

		/* Tester si la fin de la liste  t atteinte */
		if( pArg[index + 1] == NULL || *pArg[index + 1] == SNULL )
			break;

		/* Lire l'indicatif */
		pPtr1 = pArg[index];
		pPtr2 = szUserCall;
		while( * pPtr1 != ' ' && * pPtr1 != SNULL )
			*pPtr2++ = *pPtr1++;

		if( * pPtr1 == SNULL )	/* Manque les user flags */
			return PROTOCOL_BADARGNUMBER;

		*pPtr2 = SNULL;	/* NULL de fin de la chaine */

		/* Lire la config utilisateur (ConfMode) */
		if( * ++pPtr1 == SNULL )	/* Passer l'espace */
			return PROTOCOL_BADARGNUMBER;
		cFlag = (*pPtr1++ == '*' ? PROTOCOL_USER_CLUSTERCONF : 0);

		/* Lire la config utilisateur (Here) */
		if( * ++pPtr1 == SNULL )	/* Passer l'espace */		if( * ++pPtr1 == SNULL )	/* Passer l'espace */
			return PROTOCOL_BADARGNUMBER;
		cFlag |= (*pPtr1 == '1' ? PROTOCOL_USER_HERE : 0);

		/* Ajouter cet utilisateur dans la table (si l'indicatif
		   est valide) */
		if( TOOLS_isCall(szUserCall) )
		{
			NODE_newUser(StreamNum, 0, pszFromPC, szUserCall, cFlag);

			/* Forward pour cet utilisateur ? */
			MAIL_fwd2User(StreamNum, szUserCall,
				      PROTOCOL_initDone[StreamNum]);

			/* Afficher le message sur la console */
			BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node %s> Adding %s\n", pszFromPC, szUserCall);

			/* Ajouter au protocol Pavillon */
			sprintf(szBuffer, "%s %c %c^", szUserCall,
				(cFlag & PROTOCOL_USER_CONF ? '*' : '-'),
				(cFlag & PROTOCOL_USER_HERE ? '1' : '0'));
			strcat(szPavillon, szBuffer);

			/* Ajouter au protocol CluLink */
			nCluLinkPos = PROTOCOL_addPascalString(szCluLink,
				      szUserCall, nCluLinkPos);
			szCluLink[nCluLinkPos++] = cFlag;
		}/*End IF*/
	}/*End FOR*/

	/* Lecture du hop count */
	nHops = HOPS_str2int(StreamNum, NULL, pArg[index], PC16);

	/* Updater le fichier des hops count (sauf si le link est en cours
	   d'initialisation) */
	if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DONE )
		HOPS_updateHops(pszFromPC, nHops, PC16);

	/* Faire suivre le protocole */
	PROTOCOL_sendProt(StreamNum, NULL, PC16, szPavillon, szCluLink, nHops,
			  nCluLinkPos, PROTOCOL_ALL, -1);

	return PROTOCOL_OK;
}

/*------ Delete User -----------------------------------------------------
  PC17^User^FromPC^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc17(int StreamNum, int nParams, char **pArg)
{
	char* pszUser	= pArg[0];
	char* pszFromPC = pArg[1];
	char* pszHops	= pArg[2];

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszUser) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC17, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Enlever l'utilisateur de la table */
	if( ! NODE_deleteUser(StreamNum, pszFromPC, pszUser) )
	{
		/* Lecture du hops */
		int nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC17);

		/* Updater le fichier des hops count */
		HOPS_updateHops(pszFromPC, nHops, PC17);

		/* A pu tre ffac, faire suivre aux adjacents */
		PROTOCOL_deleteUser(StreamNum, pszFromPC, pszUser, nHops);

		/* Message sur la console */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"Node %s> Removing %s\n", pszFromPC, pszUser);
		return PROTOCOL_OK;
	}
	else
	{
		/* Erreur, cet utilisateur n'tait pas dans la table */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"*** Protocol error : Can't remove %s from node %s.\n",
			pszUser, pszFromPC);
		return PROTOCOL_USERUNKNOWN;
	}
}

/*------ Request init  -----------------------------------------------------
  PC18^Cluster information / clulink id^PCVersion^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc18(int StreamNum, int nParams, char **pArg)
{
	char* pszClusterInfo	= pArg[0];
	char* pszPCVersion	= pArg[1];

	char*	pPtr;
	extern	int 	PROTOCOL_adjacentVersion[65];
	extern	char	PROTOCOL_linkType[65];
	extern	char	STREAMS_callsign[MAX_STREAMS][10];
	extern  char	PROTOCOL_initDone[65];

	/* Verifier la validite du protocole */
	if( nParams != 2 )
		return PROTOCOL_BADARGNUMBER;

      /* Verifier s'il s'agit d'un init suite a la l'etablissement d'un
	 connexion ou si cela a ete demand par le sysop d l'adjacent */
      if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DONE )
      {
		/* Demande par le sysop, informer les adjacents */
		char szReason[256];
		char szAdjacentCallSign[16];

		NODE_getNodeCall(StreamNum, 0, szAdjacentCallSign);
		sprintf(szReason, "Remote init request by %s", szAdjacentCallSign);

		PROTOCOL_clusterStreamDisconnected(StreamNum, szReason, -1);
      }


	/* L'adjacent supporte t-il CluLink ? */
	pPtr = strstr(pszClusterInfo, "<CLU-");

	if( pPtr && HOPS_fix(StreamNum, NULL, StreamNum, PCCLULINK, HOPS_OUT, 100) )
	{

		/* CluLink valid : Lire le numro de version */
		PROTOCOL_adjacentVersion[StreamNum] = 1000 * atoi(pPtr+5) + atoi(pPtr+7);

		/* Accorder les versions du protocol CluLink */
		if( PROTOCOL_adjacentVersion[StreamNum] > VERSION_CLU_INT )
			PROTOCOL_adjacentVersion[StreamNum] = VERSION_CLU_INT;

		PROTOCOL_linkType[StreamNum]  = CLU_LINK;
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node %s> CluLink v%d.%d implemented.\n", STREAMS_callsign[StreamNum], PROTOCOL_adjacentVersion[StreamNum]/1000, PROTOCOL_adjacentVersion[StreamNum]%1000);

		/* Valider le protocol auprs de l'adjacent */
		BUFFERS_printBuff(StreamNum, OUT, "<CLU-%d.%d>\r", PROTOCOL_adjacentVersion[StreamNum]/1000, PROTOCOL_adjacentVersion[StreamNum]%1000);
		BUFFERS_addBuff(BUFFER_SCREEN1, "Transmitting local configuration...\n", OUT);

		/* Transmettre la configuration locale */
		PROTOCOL_sendInit(StreamNum);
	}
	else
	{
		BUFFERS_addBuff(BUFFER_SCREEN1, "Transmitting local configuration...\n", OUT);
		/* Transmettre la liste des clusters connectes */
		PROTOCOL_sendConnectedNodeList(StreamNum);
		/* Transmettre l'init */
		PROTOCOL_adjacentVersion[StreamNum] = atoi(pszPCVersion);	/* Numro de version */
		PROTOCOL_sendInit(StreamNum);
		BUFFERS_addBuff(StreamNum, "PC20^\n", OUT);
	}

	return PROTOCOL_OK;
}

/*------ Add Node ----------------------------------------------------------
  PC19^Here^PcCall^ConfMode^PCVersion^...more...^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc19(int StreamNum, int nParams, char **pArg)
{
	char	szPavillon[1024];
	char	szCluLink[1024];
	int	nCluLinkPos;
	int	index;
	int	nHops;

	extern	char	PROTOCOL_initDone[65];
	extern	char	SCRIPT_fileName[65][9];

	/* Verifier la validite du protocole */
	if( nParams < 5 )
		return PROTOCOL_BADARGNUMBER;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC19, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Pour ne plus afficher "can't connect twice" */
	SCRIPT_fileName[StreamNum][0] = SNULL;

	/*Prparer les protocoles qu'il faudra forwarder aux autres adjacents */
	strcpy(szPavillon, "PC19^");
	nCluLinkPos = PROTOCOL_setCluLink(szCluLink, CL(1));

	/* Balayer la liste de call dans le PC19 */
	for(index = 0; /*index < nParams ,*/ index < PPARAM-4; index += 4)
	{
		char* 	pszHere      = pArg[index];
		char* 	pszPcCall    = pArg[index + 1];
		char* 	pszConfMode  = pArg[index + 2];
		char* 	pszPCVersion = pArg[index + 3];
		char	szBuffer[128];
		unsigned short	shVersion;
		char	cFlag;
		
			
		/* Tester si la fin de la liste est atteinte */
		if( pArg[index][0] == 'H' )
			break;

		/* Dtecter une boucle */
		if( NODE_isNodeConnected(pszPcCall) )
			BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "*** Loop detected : %s\n", pszPcCall);
		else if( TOOLS_isCall(pszPcCall) )	/* Verifier la validite de l'indicatif */
		{
			/* Verification des arguments */
			if( pszHere == NULL || strlen(pszHere) > 1 || 
				(*pszHere != '0' && *pszHere != '1') )
		   		return PROTOCOL_BADARGNUMBER;
				
			if( pszConfMode == NULL || strlen(pszConfMode) > 1 ||
				(*pszConfMode != '0' && *pszConfMode != '1') )
				return PROTOCOL_BADARGNUMBER;
				
			if( pszPCVersion == NULL || atoi(pszPCVersion) < 100 )
				return PROTOCOL_UNKNOWNERROR;

			/* Le dernier octet de NodeCall contient les params Here & ConfMode */
			cFlag = 0;
			cFlag |= *pszHere     == '0' ? 0 : BIT6;
			cFlag |= *pszConfMode == '0' ? 0 : BIT7;

			/* Ajouter le cluster dans la table */
			NODE_newNode(StreamNum, pszPcCall, atoi(pszPCVersion), cFlag);

			/* Forwarder un mail perso pour ce cluster ? */
			MAIL_fwd2User(StreamNum, pszPcCall,
				      PROTOCOL_initDone[StreamNum]);

			/* Forward pour ce cluster ? */
			if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DONE )
				MAIL_fwd(pszPcCall, 0L, BUFFER_SCREEN1);

			/* Afficher le message sur la console */
			BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node logged on : %s\n", pszPcCall);

			/* Ajouter au protocole pavillon */
			sprintf(szBuffer, "%s^%s^%s^%s^", pszHere, pszPcCall,
				pszConfMode, pszPCVersion);
			strcat(szPavillon, szBuffer);

			/* Ajouter au protocol CluLink */
			nCluLinkPos = PROTOCOL_addPascalString(szCluLink,
				pszPcCall, nCluLinkPos);
			szCluLink[nCluLinkPos++] = cFlag + FLAG_SOFTWAREUNKNOWN;
			shVersion = (unsigned short) atoi(pszPCVersion);
			nCluLinkPos = CLULINK_addShort(szCluLink, &shVersion, nCluLinkPos);
		}
		else
		{
			return PROTOCOL_BADCALL;
		}/*End IF*/
	}/*End FOR*/

	/* Lecture du Hops */
	nHops = HOPS_str2int(StreamNum, NULL, pArg[index], PC19);

	/* Updater le fichier des hops (sauf si le link est en
	   cours d'initialisation)
	   Il faut cependant updater le hops de l'adjacent tout de suite,
	   car on ne recevra plus de PC19 plus tard. */
	if( PROTOCOL_initDone[StreamNum] != PROTOCOL_INIT_DO )
	{
		if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_NOTSTART )
			PROTOCOL_initDone[StreamNum] = PROTOCOL_INIT_DO;
		HOPS_updateHops(pArg[1], nHops, PC19);
	}

	/* Transmettre ce protocole aux autres clusters */
	if( strlen( szPavillon ) > 5 )
		PROTOCOL_sendProt(StreamNum, NULL, PC19, szPavillon, szCluLink, nHops, nCluLinkPos, PROTOCOL_ALL, -1);

	/* Lancer les fwd lorsqu'une nouvelle station se connecte */
	if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DONE )
		MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ Init Done ---------------------------------------------------------
  PC20^
  ------------------------------------------------------------------------*/
#pragma argsused
int PAVILLON_pc20(int StreamNum, int nParams, char **pArg)
{
	char	szPcCall[16];
	int	nMRdx, nMRwwv;
	extern	char	PROTOCOL_initDone[65];

	/* Quel est le call du cluster ? */
	NODE_getNodeCall(StreamNum, 0, szPcCall);

	/* Affichage */
	BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node %s> Remote initialization completed.\n",   szPcCall);
	BUFFERS_addBuff(BUFFER_SCREEN1, "Transmitting local configuration...\n", OUT);
	BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node %s> Initialization sequence completed.\n", szPcCall);

	/* Envoyer l'init */
	PROTOCOL_sendInit(StreamNum);
	BUFFERS_addBuff(StreamNum, "PC22^\n", OUT);
	PROTOCOL_initDone[StreamNum] = PROTOCOL_INIT_DONE;

	/* Envoyer le DX/WWV merge request */
	nMRwwv = HOPS_getMRwwvCount(StreamNum);
	nMRdx  = HOPS_getMRdxCount (StreamNum);
	PROTOCOL_WWVDX_mergeRequest(0, NULL, szPcCall, nMRdx, nMRwwv);

	/* Lancer les forward */
	MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ Delete node -------------------------------------------------------
  PC21^PcCall^Reason^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc21(int StreamNum, int nParams, char **pArg)
{
	char*	pszPcCall	= pArg[0];
	char*	pszReason	= pArg[1];
	char*	pszHops		= pArg[2];
	int	nHops;

	/* Verifier la validite du protocole */
	if( nParams != 3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszPcCall) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC21, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Enlever le cluster de la table */
	if( NODE_deleteNode(StreamNum, pszPcCall) )
	{
		nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC21);

		/* Updater le fichier des hops count */
		HOPS_updateHops(pszPcCall, nHops, PC21);

		/* Faire suivre le protocole aux adjacents */
		PROTOCOL_nodeDisconnect(StreamNum, pszPcCall, pszReason,
			255, nHops);

		/* Message sur la console */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node logged out : %s (%s)\n", pszPcCall, pszReason);

		return PROTOCOL_OK;
      }
      else
      {
	BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "*** Protocol error : unable to delete %s (not connected).\n", pszPcCall);
	return PROTOCOL_NODEUNKNOWN;
      }/*End IF*/
}

/*------ PC Done ---------------------------------------------------------
  PC22^
  ------------------------------------------------------------------------*/
#pragma argsused
int PAVILLON_pc22(int StreamNum, int nParams, char **pArg)
{
	char	szPcCall[16];
	int	nMRdx, nMRwwv;
	extern	char	PROTOCOL_initDone[65];

	/* Quel est le call du cluster ? */
	NODE_getNodeCall(StreamNum, 0, szPcCall);

	BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "Node %s> Initialization sequence completed.\n", szPcCall);
	PROTOCOL_initDone[StreamNum] = PROTOCOL_INIT_DONE;

	/* Envoyer le DX/WWV merge request */
	nMRwwv = HOPS_getMRwwvCount(StreamNum);
	nMRdx  = HOPS_getMRdxCount (StreamNum);
	PROTOCOL_WWVDX_mergeRequest(0, NULL, szPcCall, nMRdx, nMRwwv);

	/* Lancer les fwd */
	MAIL_fwd();
	return PROTOCOL_OK;
}

/*------ WWV info ----------------------------------------------------------
  PC23^Date^Hour^SFI^A^K^Forecast^Logger^FromPC^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc23(int StreamNum, int nParams, char **pArg)
{
	char* pszDate	  = pArg[0];
	char* pszHour     = pArg[1];
	char* pszSFI	  = pArg[2];
	char* pszA	  = pArg[3];
	char* pszK	  = pArg[4];
	char* pszForecast = pArg[5];
	char* pszLogger	  = pArg[6];
	char* pszFromPC	  = pArg[7];
	char* pszHops	  = pArg[8];

	int	nHops;
	int	nSFI, nK, nA;
	int	nHour;
	char	szTime[16];

	/* Verifier la validite du protocole */
	if( nParams !=  9 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszLogger) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	if( pszForecast == NULL )
		return PROTOCOL_UNKNOWNERROR;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, pszFromPC, StreamNum, PC23, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, pszFromPC, pszHops, PC23);

	/* Lecture de SFI, A et K */
	nSFI = atoi (pszSFI);
	nA   = atoi (pszA);
	nK   = atoi (pszK);

	/* Lecture de l'heure (ajouter 00) */
	if( strlen(pszHour) > 2 )
		return PROTOCOL_UNKNOWNERROR;
	if( pszHour[0] == ' ' )
		nHour = atoi(pszHour+1);
	else
		nHour = atoi(pszHour);
	sprintf(szTime, "%02d00", nHour);

	/* Faire suivre le protocole */
	PROTOCOL_sendWwv(StreamNum, pszFromPC, pszLogger, nSFI, nA, nK,
			 pszDate, szTime, pszForecast, nHops);

	return PROTOCOL_OK;
}

/*------ Here status -------------------------------------------------------
  PC24^User^Here^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc24(int StreamNum, int nParams, char **pArg)
{
	char* pszUser 	= pArg[0];
	char* pszHere   = pArg[1];
	char* pszHops   = pArg[2];

	int	nHops;

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszUser) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC24, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC24);

	/* Faire suivre le protocole */
	PROTOCOL_sendUserFlag(StreamNum, "", pszUser, PROTOCOL_USER_HERE,
		(*pszHere == '1'), nHops);

	return PROTOCOL_OK;
}

/*------ DX merge info ---------------------------------------------------
  PC25^ToPC^FromPC^DXcount^WWVcount^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc25(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC	  = pArg[0];
	char* pszFromPC	  = pArg[1];
	char* pszDXcount  = pArg[2];
	char* pszWWVcount = pArg[3];

	/* Verifier la validite du protocole */
	if( nParams != 4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC25, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traiter ... */
	if( PROTOCOL_WWVDX_mergeRequest(StreamNum, pszFromPC, pszToPC,
					atoi(pszDXcount),
					atoi(pszWWVcount)) == FALSE )
		return PROTOCOL_NODEUNKNOWN;	/* Routage impossible */

	return PROTOCOL_OK;
}

/*------ DX merge info ---------------------------------------------------
  PC26^DXFreq^DXCall^Date^Time^DXInfoText^Logger^ToPC^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc26(int StreamNum, int nParams, char **pArg)
{
	char* pszDxFrequency	= pArg[0];
	char* pszDxCall		= pArg[1];
	char* pszDate		= pArg[2];
	char* pszTime		= pArg[3];
	char* pszDxInfoText	= pArg[4];
	char* pszLogger		= pArg[5];
	char* pszToPC		= pArg[6];

	ulong lFrequency;

	/* Verifier la validite du protocole */
	if( nParams != 8 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszLogger) || ! TOOLS_isCall(pszToPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC26, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Decoder la frequence */
	lFrequency = (unsigned long) (atof(pszDxFrequency) * 10.0 + 0.000001);
	if( ! DX_checkFrequency(lFrequency) )
		return PROTOCOL_BADFREQUENCY;	/* Frequence incorrecte */

	/* Enlever le Z a la fin de l'heure */
	TOOLS_maxLength(pszTime, 4);

	/* Traiter ... */
	if( PROTOCOL_DXMergeInfo(StreamNum, "?", pszToPC, pszLogger,
				 lFrequency, pszDate,  pszTime, pszDxCall,
				 pszDxInfoText) == FALSE )
		return PROTOCOL_NODEUNKNOWN;	/* Routage impossible */

	return PROTOCOL_OK;
}

/*------ WWV merge info ---------------------------------------------------
  PC27^Date^Hour^SFI^A^K^Forecast^Logger^ToPC^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc27(int StreamNum, int nParams, char **pArg)
{
	char* pszDate	  = pArg[0];
	char* pszHour     = pArg[1];
	char* pszSFI	  = pArg[2];
	char* pszA	  = pArg[3];
	char* pszK	  = pArg[4];
	char* pszForecast = pArg[5];
	char* pszLogger	  = pArg[6];
	char* pszToPC	  = pArg[7];

	int	nSFI, nK, nA;
	int	nHour;
	char	szTime[8];

	/* Verifier la validite du protocole */
	if( nParams !=  9 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszLogger) || ! TOOLS_isCall(pszToPC) )
		return PROTOCOL_BADCALL;

	if( pszForecast == NULL )
		return PROTOCOL_UNKNOWNERROR;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC27, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture de SFI, A et K */
	nSFI = atoi (pszSFI);
	nA   = atoi (pszA);
	nK   = atoi (pszK);

	/* Lecture de l'heure (ajouter 00) */
	if( strlen(pszHour) > 2 )
		return PROTOCOL_UNKNOWNERROR;
	if( pszHour[0] == ' ' )
		nHour = atoi(pszHour+1);
	else
		nHour = atoi(pszHour);
	sprintf(szTime, "%02d00", nHour);

	/* Traiter ... */
	if( PROTOCOL_WWVMergeInfo(StreamNum, "?", pszToPC, pszLogger,
				  nSFI, nA, nK, pszDate, szTime,
				  pszForecast) == FALSE )
		return PROTOCOL_NODEUNKNOWN;	/* Routage impossible */

	return PROTOCOL_OK;
}

/*------ PC mail - Send Subject --------------------------------------------
  PC28^ToPC^ViaPC^ToStn^FromStn^Date^Time^PrivateFlag^Subject^ ^LineAck^ReceiptFlag^ ^OriginatedFromPC^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc28(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC 		= pArg[0];
	char* pszViaPC		= pArg[1];
	char* pszToStn  	= pArg[2];
	char* pszFromStn 	= pArg[3];
	char* pszDate 		= pArg[4];
	char* pszTime		= pArg[5];
	char* pszPrivateFlag	= pArg[6];
	char* pszSubject	= pArg[7];
	char* pszLineAck	= pArg[9];
	char* pszReceiptFlag	= pArg[10];
	char* pszOriginatedPC	= pArg[12];

	char  szType[10];
	char  szBuffer[256];
	int   nDupe = FALSE;

	extern	char	PROTOCOL_mailEveryNLines[65];
	extern	char	PROTOCOL_mailLinesCounter[65];

	/* Liste des champs
		pArg[0]	PC destinataire
		pArg[1]	PC expediteur
		pArg[2]	Recipient
		pArg[3]	Expediteur (utilisateur)
		pArg[4]	Date
		pArg[5]	Heure
		pArg[6]	Private-flag (1->prive, 0->bull)
		pArg[7]	sujet
		pArg[8]	?  A priori un espace
		pArg[9]	Nombre de ligne avant un ack
		pArg[10]	1 si Return Receipt demande, 0 sinon
		pArg[11]	?  A priori un espace
		pArg[12]	PC qui est a l'origine du message (StartNode)
	*/

	/* Verifier la validite du protocole */
	if( nParams !=  13 )
		return PROTOCOL_BADARGNUMBER;

	/* Packet cluster envoie les mails perso sans preciser
	   pszOriginatedPC ... Ce protocol est dfinitivement tres
	   difficile a hacker */

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszViaPC) ||
	    ! TOOLS_isCall(pszFromStn) ||
	   (! TOOLS_isCall(pszOriginatedPC) && strcmp(pszOriginatedPC, " ")) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC28, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Recherche de doublons */
	if( MAIL_isDupe(pszToStn, pszFromStn, pszSubject, pszDate, pszTime) )
		nDupe = TRUE;

	/* Champ vide ? Alors vider pour de bon */
	if( ! strcmp(pszOriginatedPC, " ") )
		*pszOriginatedPC = SNULL;

	/* Prive ou public ? */
	if( *pszPrivateFlag == '1' )
		strcpy(szType, "sp");
	else
		strcpy(szType, "sb");

	/* Destinataire & expediteur */
	sprintf(szBuffer, "%s < %s", pszToStn, pszFromStn);

	/* Enregistrer le message */
	if( ! MAIL_rcvRecip(StreamNum, szType, szBuffer, atoi(pszReceiptFlag), nDupe) )
		return PROTOCOL_OK;	/* Le message n'a pas pu tre ouvert */

	/* Ajouter le sujet */
	if( * pszOriginatedPC )
	{
		if( ! MAIL_addSubject2Tmp(StreamNum, pszSubject, pszDate,
		      pszTime, pszOriginatedPC, pszViaPC) )
			return PROTOCOL_OK;	/* Probleme ... */
	}
	else
	{
		if( ! MAIL_addSubject2Tmp(StreamNum, pszSubject, pszDate,
		      pszTime, "?", pszViaPC) )
		      return PROTOCOL_OK;	/* Probleme ... */
	}/*End IF*/

	/* Accuser reception du PC28 */
	PROTOCOL_sendAckSubject(StreamNum, pszViaPC, pszToPC, (unsigned long) StreamNum);

	/* Nombre de lignes par blocs */
	PROTOCOL_mailEveryNLines[StreamNum]  = (char) atoi (pszLineAck);
	PROTOCOL_mailLinesCounter[StreamNum] = PROTOCOL_mailEveryNLines[StreamNum];

	return PROTOCOL_OK;
}

/*------ PC mail - Send Text -----------------------------------------------
  PC29^ToPC^FromPC^MsgNum^Text^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc29(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszMsgNum = pArg[2];
	char* pszText	= pArg[3];

	extern	char	PROTOCOL_mailEveryNLines[65];
	extern	char	PROTOCOL_mailLinesCounter[65];

	/* Verifier la validite du protocole */
	if( nParams !=  4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC29, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Verifier la validite du numero de message */
	if( StreamNum != (int) atoi (pszMsgNum) )
		return PROTOCOL_BADMSGNUMBER;

	/* Ajouter la ligne de texte ds le fichier temp */
	MAIL_addText2Tmp(StreamNum, pszText);

	/* Gestion du compteur de lignes - faut-il envoyer un ACK ? */
	if( --PROTOCOL_mailLinesCounter[StreamNum] == 0 )
	{
		PROTOCOL_sendAckText(StreamNum, pszFromPC, pszToPC, (ulong) StreamNum);
		PROTOCOL_mailLinesCounter[StreamNum] = PROTOCOL_mailEveryNLines[StreamNum];
	}

	return PROTOCOL_OK;
}

/*------ PC mail - Ack Subject ---------------------------------------------
  PC30^ToPC^FromPC^MsgNum^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc30(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszMsgNum = pArg[2];

	extern	unsigned long PROTOCOL_lFwdMesNum[MAX_STREAMS];

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC30, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Ligne suivante */
	PROTOCOL_lFwdMesNum[StreamNum] = atol (pszMsgNum);
	MAIL_sendNextBloc(StreamNum, pszToPC, pszFromPC, PROTOCOL_lFwdMesNum[StreamNum]);

	return PROTOCOL_OK;
}

/*------ PC mail - Ack Text ------------------------------------------------
  PC31^ToPC^FromPC^MsgNum^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc31(int StreamNum, int nParams, char **pArg)
{
	/* Traitement identique a PC30 */
	return PAVILLON_pc30(StreamNum, nParams, pArg);
}

/*------ PC mail - CompleteText --------------------------------------------
  PC32^ToPC^FromPC^MsgNum^
  ------------------------------------------------------------------------*/
int PAVILLON_pc32(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszMsgNum = pArg[2];

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC32, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Verifier la validite du numero de message */
	if( StreamNum != (int) atoi (pszMsgNum) )
		return PROTOCOL_BADMSGNUMBER;

	/* Valider le message */
	if( MAIL_tmp2Msg((ulong) StreamNum) )
		PROTOCOL_sendAckCompleteText(StreamNum, pszFromPC, pszToPC,
					     (ulong) StreamNum);
	return PROTOCOL_OK;
}

/*------ PC mail - Ack CompleteText ----------------------------------------
  PC33^ToPC^FromPC^MsgNum^
  ------------------------------------------------------------------------*/
int PAVILLON_pc33(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
      /*char* pszMsgNum = pArg[2];*/

	extern  unsigned long	MAIL_lFwdMsgNumber[MAX_STREAMS];


	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC33, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	MAIL_changeStatut(MAIL_lFwdMsgNumber[StreamNum], STATUT_F);
	MAIL_deQueue(pszFromPC, MAIL_lFwdMsgNumber[StreamNum]);
	MAIL_lFwdMsgNumber[StreamNum] = 0L;
	MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ Remote commands - commands ----------------------------------------
  PC34^ToPC^FromPC^cmd^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc34(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszCmd    = pArg[2];
	char  szMyNodeCall[16];

	/* Indicatif du serveur */
	NODE_getNodeCall(0, 0, szMyNodeCall);

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	if( ! strcmp(pszToPC, pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Verifier qu'on ne recoit pas un rcmd de soit de l'exterieur ! */
	if( ! strcmp(szMyNodeCall, pszFromPC) )
		return PROTOCOL_LOOP;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC34, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traiter ce protocole */
	PROTOCOL_rCmd(StreamNum, pszFromPC, pszToPC, NULL, pszCmd);

	return PROTOCOL_OK;
}

/*------ Remote commands - reponse -----------------------------------------
  PC35^ToPC^FromPC^cmd^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc35(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszCmd    = pArg[2];
	char  szMyNodeCall[16];

	/* Indicatif du serveur */
	NODE_getNodeCall(0, 0, szMyNodeCall);

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	if( ! strcmp(pszToPC, pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Verifier qu'on ne recoit pas un rcmd de soit de l'exterieur ! */
	if( ! strcmp(szMyNodeCall, pszFromPC) )
		return PROTOCOL_LOOP;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC35, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traiter ce protocole */
	PROTOCOL_rCmdResponse(StreamNum, pszFromPC, pszToPC, NULL, pszCmd);

	return PROTOCOL_OK;
}

/*------ Delete node requested by op -------------------------------------
  PC39^FromPC^reason^
  ------------------------------------------------------------------------*/
int PAVILLON_pc39(int StreamNum, int nParams, char **pArg)
{
	char* pszFromPC	= pArg[0];
	char* pszReason = pArg[1];
	char* pszHops   = pArg[2];
	int   nHops;

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC39, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC24);

	PROTOCOL_rcvExplicitDisconnect(StreamNum, pszFromPC, pszReason, nHops);

	return PROTOCOL_OK;
}


/*------ Addit. user info --------------------------------------------------
  PC41^User^InfoType^Info^Hops^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc41(int StreamNum, int nParams, char **pArg)
{
	char* pszUser  		= pArg[0];
	char* pszInfoType	= pArg[1];
	char* pszInfo		= pArg[2];
	char* pszHops		= pArg[3];

	int	nHops;
	int	nInfoType;

	/* Verifier la validite du protocole */
	if( nParams !=  4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszUser) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC41, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, NULL, pszHops, PC24);

	/* Type d'info ? */
	nInfoType = atoi(pszInfoType);

	/* Updater la config de cet utilisateur */
	USERS_changeConfig(pszUser, pszInfo, atoi(pszInfoType));

	/* Faire suivre le protocole */
	PROTOCOL_sendUserConfig(StreamNum, "", pszUser, pszInfo, nInfoType,
				nHops);

	return PROTOCOL_OK;
}

/*------ Forwarding abort --------------------------------------------------
  PC42^ToPC^FromPC^MsgNum^
  ------------------------------------------------------------------------*/
int PAVILLON_pc42(int StreamNum, int nParams, char **pArg)
{
	extern  unsigned long	MAIL_lFwdMsgNumber[MAX_STREAMS];

	char* pszToPC   = pArg[0];
	char* pszFromPC = pArg[1];
	/* char* pszMsgNum = pArg[2]; */

	/* Verifier la validite du protocole */
	if( nParams != 3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC44, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Enlever le message de la queue et continuer le forward */
	MAIL_deQueue(pszFromPC, MAIL_lFwdMsgNumber[StreamNum]);
	MAIL_lFwdMsgNumber[StreamNum] = 0L;
	MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ Remote DB request -------------------------------------------------
  PC44^ToPC^FromPC^Stream^Qualifier^Key^FromStn^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc44(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  		= pArg[0];
	char* pszFromPC		= pArg[1];
	char* pszStream 	= pArg[2];
	char* pszQualifier	= pArg[3];
	char* pszKey		= pArg[4];
	char* pszFromStn	= pArg[5];

	char	szNodeCall[16];
	int	nStream;

	/* Verifier la validite du protocole */
	if( nParams !=  6 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) ||
	    ! TOOLS_isCall(pszFromStn) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC44, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Numero de stream */
	nStream = atoi(pszStream);

	/*Repondre ou transmettre a un adjacent*/
	NODE_getNodeCall(0, 0, szNodeCall);
	if( ! strcmp(szNodeCall, pszToPC) )
	{
		/* Interroger la base de donnees */
		if( ! DATABASE_remote(nStream, pszQualifier, pszKey, pszFromPC) )
		{
			/* Erreur acces base de donnees */
			PROTOCOL_dbResponse(0, pszToPC, pszFromPC, nStream,
			      "*** Error accessing the database : database unknown");
		}

		PROTOCOL_dbComplete(0, pszToPC, pszFromPC, nStream);
	}
	else
	{
		/* Faire suivre le protocole  l'adjacent concern */
		PROTOCOL_dbRequest(StreamNum, pszToPC, pszFromPC, nStream,
			   pszQualifier, pszKey, pszFromStn);
	}

	return PROTOCOL_OK;
}

/*------ Remote DB response ------------------------------------------------
  PC45^ToPC^FromPC^Stream^Info^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc45(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  		= pArg[0];
	char* pszFromPC		= pArg[1];
	char* pszStream 	= pArg[2];
	char* pszInfo		= pArg[3];

	char	szNodeCall[16];
	int	nStream;

	/* Verifier la validite du protocole */
	if( nParams !=  4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC45, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Numero de stream */
	nStream = atoi(pszStream);

	NODE_getNodeCall(0, 0, szNodeCall);
	if( ! strcmp(szNodeCall, pszToPC) )
	{
		/* Envoyer a l'utilisateur */
		BUFFERS_printBuff(nStream, OUT, "%s\n", pszInfo);
	}
	else
	{
		/* Faire suivre a l'adjacent concerne */
		PROTOCOL_dbResponse(StreamNum, pszFromPC, pszToPC, nStream,
				    pszInfo);
	}

	return PROTOCOL_OK;
}

/*------ Remote DB complete ------------------------------------------------
  PC46^ToPC^FromPC^Stream^
  ------------------------------------------------------------------------*/
int PAVILLON_pc46(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  		= pArg[0];
	char* pszFromPC		= pArg[1];
	char* pszStream 	= pArg[2];

	char	szNodeCall[16];
	int	nStream;

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC46, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Numero de stream */
	nStream = atoi(pszStream);

	/*Recuperer ou transmettre a un adjacent*/
	NODE_getNodeCall(0, 0, szNodeCall);
	if( ! strcmp(szNodeCall, pszToPC) )
	{
		/* Envoyer a l'utilisateur */
		MSG_send(nStream, MSG62);
	}
	else
	{
		/* Faire suivre a l'adjacent concerne */
		PROTOCOL_dbComplete(StreamNum, pszFromPC, pszToPC, nStream);
	}

	return PROTOCOL_OK;
}

/*------ Update user count -------------------------------------------------
  PC50^FromPC^UserCount^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc50(int StreamNum, int nParams, char **pArg)
{
	char* pszFromPC		= pArg[0];
	char* pszUserCount	= pArg[1];
	char* pszHops		= pArg[2];

	int	nHops;
        int	nUserCount;

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, pszFromPC, StreamNum, PC50, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture de nombre d'utilisateurs et du hops */
	nUserCount = atoi(pszUserCount);
	nHops = HOPS_str2int(StreamNum, pszFromPC, pszHops, PC50);

	/* UserCount correct ? */
	if( nUserCount < 0 )
		return PROTOCOL_BADUSERCOUNT;

	PROTOCOL_userCount(StreamNum, pszFromPC, TRUE, (unsigned short) nUserCount, nHops);

	return PROTOCOL_OK;
}

/*------ Ping --------------------------------------------------------------
  PC51^ToPC^FromPC^PingFlag^
  ------------------------------------------------------------------------*/
int PAVILLON_pc51(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC		= pArg[0];
	char* pszFromPC		= pArg[1];
	char* pszPingFlag	= pArg[2];

	/* Verifier la validite du protocole */
	if( nParams !=  3 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC51, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	PROTOCOL_ping(StreamNum, pszFromPC, pszToPC,
		      (*pszPingFlag == '1' ? PROTOCOL_PING_ASK : PROTOCOL_PING_ANSWER),
		      HOPS_probable(pszFromPC) );

	return PROTOCOL_OK;
}

/*------ WCY info ----------------------------------------------------------
  PC73^Date^Hour^SFI^A^K^Exp.K^R^SA^GMF^Aurora^Logger^FromPC^Hops^
  ------------------------------------------------------------------------*/
int PAVILLON_pc73(int StreamNum, int nParams, char **pArg)
{
	char* pszDate	  = pArg[0];
	char* pszHour     = pArg[1];
	char* pszSFI	  = pArg[2];
	char* pszA	  = pArg[3];
	char* pszK	  = pArg[4];
	char* pszExpK	  = pArg[5];
	char* pszR	  = pArg[6];
	char* pszSA	  = pArg[7];
	char* pszGMF	  = pArg[8];
	char* pszAurora   = pArg[9];
	char* pszLogger	  = pArg[10];
	char* pszFromPC   = pArg[11];
	char* pszHops	  = pArg[12];

	int	nHops;
	int	nSFI, nK, nA, nR, nExpK;
	int	nHour;
	char	szTime[16];

	/* Verifier la validite du protocole */
	if( nParams !=  13 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszLogger) || ! TOOLS_isCall(pszFromPC) )
		return PROTOCOL_BADCALL;

	if( pszSA == NULL || pszGMF == NULL || pszAurora == NULL )
		return PROTOCOL_UNKNOWNERROR;

	if( strlen(pszSA) > 3 || strlen(pszGMF) > 3 || strlen(pszAurora) > 6 )
		return PROTOCOL_UNKNOWNERROR;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, pszFromPC, StreamNum, PC73, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = HOPS_str2int(StreamNum, pszFromPC, pszHops, PC73);

	/* Lecture de SFI, A et K */
	while( *pszSFI  == ' ' && *pszSFI  != SNULL ) pszSFI++;
	while( *pszA    == ' ' && *pszA    != SNULL ) pszA++;
	while( *pszK    == ' ' && *pszK    != SNULL ) pszK++;
	while( *pszR    == ' ' && *pszR    != SNULL ) pszR++;
	while( *pszExpK == ' ' && *pszExpK != SNULL ) pszExpK++;
	while( *pszHour == ' ' && *pszHour != SNULL ) pszHour++;
	nSFI  = atoi(pszSFI);
	nA    = atoi(pszA);
	nK    = atoi(pszK);
	nR    = atoi(pszR);
	nExpK = atoi(pszExpK);
	nHour = atoi(pszHour);

	sprintf(szTime, "%02d00", nHour);

	/* Faire suivre le protocole */
	PROTOCOL_sendWcy(StreamNum, pszDate, szTime, nSFI, nA, nK, nExpK,
		nR, pszSA, pszGMF, pszAurora, pszLogger, pszFromPC, nHops);

	return PROTOCOL_OK;
}

/*------ Remote commands - CLX way - commands ------------------------------
  PC84^ToPC^FromPC^User^cmd^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc84(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszUser   = pArg[2];
	char* pszCmd    = pArg[3];
	char  szMyNodeCall[16];

	/* Indicatif du serveur */
	NODE_getNodeCall(0, 0, szMyNodeCall);

	/* Verifier la validite du protocole */
	if( nParams !=  4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) ||
	    ! TOOLS_isCall(pszUser) )
		return PROTOCOL_BADCALL;

	if( ! strcmp(pszToPC, pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Verifier qu'on ne recoit pas un rcmd de soit de l'exterieur ! */
	if( ! strcmp(szMyNodeCall, pszFromPC) )
		return PROTOCOL_LOOP;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC84, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traiter ce protocole */
	PROTOCOL_rCmd(StreamNum, pszFromPC, pszToPC, pszUser, pszCmd);

	return PROTOCOL_OK;
}

/*------ Remote commands - CLX way - reponse -------------------------------
  PC85^ToPC^FromPC^User^cmd^~
  ------------------------------------------------------------------------*/
int PAVILLON_pc85(int StreamNum, int nParams, char **pArg)
{
	char* pszToPC  	= pArg[0];
	char* pszFromPC	= pArg[1];
	char* pszUser   = pArg[2];
	char* pszCmd    = pArg[3];
	char  szMyNodeCall[16];

	/* Indicatif du serveur */
	NODE_getNodeCall(0, 0, szMyNodeCall);

	/* Verifier la validite du protocole */
	if( nParams !=  4 )
		return PROTOCOL_BADARGNUMBER;

	if( ! TOOLS_isCall(pszToPC) || ! TOOLS_isCall(pszFromPC) ||
	    ! TOOLS_isCall(pszUser) )
		return PROTOCOL_BADCALL;

	if( ! strcmp(pszToPC, pszFromPC) )
		return PROTOCOL_BADCALL;

	/* Verifier qu'on ne recoit pas un rcmd de soit de l'exterieur ! */
	if( ! strcmp(szMyNodeCall, pszFromPC) )
		return PROTOCOL_LOOP;

	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC35, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Traiter ce protocole */
	PROTOCOL_rCmdResponse(StreamNum, pszFromPC, pszToPC, pszUser, pszCmd);

	return PROTOCOL_OK;
}

