#include <alloc.h>
#include <string.h>
#include <stdio.h>
#include "define.h"
#include "tools.h"
#include "filter.h"
#include "dxcc.h"
#include "users.h"

/* FILTER_cWAZ est utilise pour gerrer le filtrage des protocoles
   FILTER_cWAZ[STREAM][WAZ_NUMBER]

   FILTER_cWAZ[STREAM][0] indique si la fonction filtre est mise en place
   (un 0 indique que le filtre n'est pas en place)

   . Pour un cluster     -> bit 0 : IN  bit 1 : OUT
   . Pour un utilisateur -> seul le bit 0 est significatif

   Si FILTER_cWAZ[STREAM][0] est null alors aucun filtre n'est
   mit en place
*/

unsigned char FILTER_cWAZ[66][FILTER_MAXWAZ];
int           FILTER_nMyWAZ = 0;
int	      FILTER_nIsEnabled = FALSE;

/*--------------------------------------------------------------------------
  ------ Initialiser -------------------------------------------------------
  --------------------------------------------------------------------------
  Cette fonction est appelee en premier - si un filtre doit etre mis en
  place, il faudra appeler ensuite FILTER_set                              */
void FILTER_init(int StreamNum)
{
	/* Pas de filtre */
	FILTER_cWAZ[StreamNum][0] = 0;
}

/*--------------------------------------------------------------------------
  ------ Definir les filtres a appliquer sur le stream ---------------------
  --------------------------------------------------------------------------*/
/*
   nStreamNum : numero du stream
   nDirection : pour un utilisateur : FILTER_USER
		pour un cluster     : FILTER_IN & FILTER_OUT
   szFilter   : chaine de caracteres contenant les zones WAZ ou les
		continents
		ATTENTION : le contenu de szFilter sera modifie par cette
		fonction
   Retour : TRUE si OK et FALSE si erreur */
int FILTER_set(int StreamNum, int nDirection, char * pszFilter)
{
	char * token;
	int    index;

	/* Valider le filtrage */
	FILTER_setWAZ2ON(StreamNum, nDirection, 0);
	for(index = 1; index < FILTER_MAXWAZ; index++)
		FILTER_setWAZ2OFF(StreamNum, nDirection, index);

	/* tout passer en minuscule */
	strlwr(pszFilter);

	/* Parcourir la chaine a la recherche des elements du filtre */
	token = strtok(pszFilter, " ");
	while( token )
	{
		int nWAZ = atoi(token);

		if( nWAZ >= 1 && nWAZ <= FILTER_MAXWAZ )
			FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		else if( ! strcmp(token, "na") )
		{
			/* toutes les zones NA */
			for(nWAZ = 1; nWAZ <= 8; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( ! strcmp(token, "sa") )
		{
			/* toutes les zones SA */
			for(nWAZ = 9; nWAZ <= 13; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( ! strcmp(token, "eu") )
		{
			/* toutes les zones EU */
			for(nWAZ = 14; nWAZ <= 20; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( ! strcmp(token, "as") )
		{
			/* toutes les zones AS */
			for(nWAZ = 21; nWAZ <= 26; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( ! strcmp(token, "oc") )
		{
			/* toutes les zones OC */
			for(nWAZ = 27; nWAZ <= 32; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( ! strcmp(token, "af") )
		{
			/* toutes les zones AF */
			for(nWAZ = 33; nWAZ <= 39; nWAZ++)
				FILTER_setWAZ2ON(StreamNum, nDirection, nWAZ);
		}
		else if( nWAZ != -1 )
			return FALSE;	/* ERREUR ! */

		token = strtok(NULL, " ");
	}

	return TRUE;
}

/*--------------------------------------------------------------------------
  ------ Definir les filtres a appliquer sur le stream pour un utilisateur -
  --------------------------------------------------------------------------*/
void FILTER_setUser(int StreamNum, unsigned int nFlags)
{
	char szBuffer[64];

	/* Creer la chaine de definition du filtrage */
	FILTER_flags2str(nFlags, szBuffer);

	if( szBuffer[0] == SNULL )
		FILTER_init(StreamNum);	/* Pas de filtre a appliquer */
	else
		FILTER_set(StreamNum, FILTER_USER, szBuffer);
}

/*--------------------------------------------------------------------------
  ------ Valider une zone WAZ ----------------------------------------------
  --------------------------------------------------------------------------*/
void FILTER_setWAZ2ON(int StreamNum, int nDirection, int nWAZ)
{
	switch(nDirection)
	{
	case FILTER_USER :
		FILTER_cWAZ[StreamNum][nWAZ] = 255;
		break;

	case FILTER_IN :
		FILTER_cWAZ[StreamNum][nWAZ] |= 1;	/* bit 0 */
		break;

	case FILTER_OUT :
		FILTER_cWAZ[StreamNum][nWAZ] |= 2;	/* bit 1 */
		break;

	default:
		break;
	}
}

/*--------------------------------------------------------------------------
  ------ Devalider une zone WAZ --------------------------------------------
  --------------------------------------------------------------------------*/
void FILTER_setWAZ2OFF(int StreamNum, int nDirection, int nWAZ)
{
	switch(nDirection)
	{
	case FILTER_USER :
		FILTER_cWAZ[StreamNum][nWAZ] = 0;
		break;

	case FILTER_IN :
		FILTER_cWAZ[StreamNum][nWAZ] &= 0xFE;	/* bit 0 */
		break;

	case FILTER_OUT :
		FILTER_cWAZ[StreamNum][nWAZ] &= 0XFD;	/* bit 1 */
		break;

	default:
		break;
	}
}

/*--------------------------------------------------------------------------
  ------ Rechercher la zone WAZ pour un indicatif --------------------------
  --------------------------------------------------------------------------
  Retour : numero de la zone CQ                                            */
int FILTER_searchWAZ(char * pszCallSign)
{
	char szCtyLine[256];
	char szBuffer[256];

	/* Filtrage active ? Pas d'indicatif a rechercher ? */
	if( FILTER_nIsEnabled == FALSE || pszCallSign == NULL )
		return -1;

	if( DXCC_lookup(pszCallSign, szCtyLine) == -1 )
		return -1;	/* prefixe inconnu */

	DXCC_getField(szCtyLine, szBuffer, DXCC_CQZONE_P, DXCC_CQZONE_L);

	return atoi(szBuffer);
}

/*-------------------------------------------------------------------------
  ------ Definir la zone WAZ de mon cluster -------------------------------
  -------------------------------------------------------------------------*/
void FILTER_setMyWAZ(char * pszCallSign)
{
	int nIsEnabled_SAVE;

	nIsEnabled_SAVE = FILTER_nIsEnabled;
	FILTER_nIsEnabled = TRUE; /* Au cas ou ca ne serait pas deja active */

	/* Rechercher ma propre zone WAZ */
	FILTER_nMyWAZ = FILTER_searchWAZ(pszCallSign);

	FILTER_nIsEnabled = nIsEnabled_SAVE;	/* Restorer etat initial */
}

/*-------------------------------------------------------------------------
  ------ Doit etre filtre ? -----------------------------------------------
  -------------------------------------------------------------------------
  Retour : TRUE si OK (pas filtre)  et  FALSE si filtre                    */
int FILTER_isOK(int StreamNum, int nDirection, int nWAZ)
{
	if( StreamNum == 0 )
		return TRUE;

	if( FILTER_nIsEnabled == FALSE || nWAZ == -1 || FILTER_cWAZ[StreamNum][0] == 0 )
		return TRUE;

	switch( nDirection )
	{
	case FILTER_USER :
		return FILTER_cWAZ[StreamNum][nWAZ];

	case FILTER_IN :
		return FILTER_cWAZ[StreamNum][nWAZ] & 0x01;	/* bit 0 */

	case FILTER_OUT :
		return FILTER_cWAZ[StreamNum][nWAZ] & 0x02;	/* bit 1 */

	default :
		break;
	}

	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Les filtrage est il active ? ------------------------------------
  ------------------------------------------------------------------------*/
int FILTER_isEnabled(void)
{
	return FILTER_nIsEnabled;
}

/*------------------------------------------------------------------------
  ------ Convertir les flags USERS en une chaine -------------------------
  ------------------------------------------------------------------------*/
char * FILTER_flags2str(unsigned int nFlags, char * pszBuffer)
{
	/* Creer la chaine */
	strcpy(pszBuffer, "");
	if( nFlags & USERS_FILTER_NA )
		strcat(pszBuffer, "NA ");
	if( nFlags & USERS_FILTER_SA )
		strcat(pszBuffer, "SA ");
	if( nFlags & USERS_FILTER_EU )
		strcat(pszBuffer, "EU ");
	if( nFlags & USERS_FILTER_AS )
		strcat(pszBuffer, "AS ");
	if( nFlags & USERS_FILTER_OC )
		strcat(pszBuffer, "OC ");
	if (nFlags & USERS_FILTER_AF )
		strcat(pszBuffer, "AF ");

	return pszBuffer;
}
