#include <alloc.h>
#include <string.h>
#include "clulink.h"
#include "node.h"
#include "buffers.h"
#include "protocol.h"
#include "mail.h"
#include "hops.h"
#include "memory.h"
#include "tools.h"
#include "users.h"
#include "database.h"
#include "message.h"
#include "dx.h"

//------------------------------------------------------------------------
//------ Lire la longueur d'un protocole CluLink -------------------------
//------------------------------------------------------------------------
unsigned int CLULINK_getSize(char * Prot)
{
  unsigned short	  shLen;
  char  *pLen = (char *) &shLen;

  pLen[1] = Prot[0];
  pLen[0] = Prot[1];

  return (unsigned int) shLen & 0x3FFF;	//Masquer CL_COMP et CL_BROADCAST
}

//------------------------------------------------------------------------
//------ Mettre la longueur d'un protocole CluLink au debut d'une trame --
//------------------------------------------------------------------------
void CLULINK_putSize(char * Prot, unsigned int Len)
{
  unsigned short shLen = (unsigned short) Len;
  char *pLen = (char *) &shLen;

  Prot[0] = pLen[1];
  Prot[1] = pLen[0];

  //Calcul du CRC
  Prot[2] = CLULINK_fixCRC(Prot + 3, Len);
}

//------------------------------------------------------------------------
//------ Calculer le CRC -------------------------------------------------
//------------------------------------------------------------------------
char CLULINK_fixCRC(char * Prot, unsigned int Size)
{
  unsigned char crc = 0;
  unsigned int	index;

  for(index = 1; index < Size; index++)
    crc += *Prot++;

  return crc;
}

//--------------------------------------------------------------------------
//------ Ajouter un short  une chaine de caractres -----------------------
//--------------------------------------------------------------------------
int CLULINK_addShort(char * Dest, unsigned short * Val, unsigned int Pos)
{
  char *pShort = (char *) Val;
  char *pPtr = Dest + Pos;

  pPtr[1] = *pShort++;
  pPtr[0] = *pShort++;

  return Pos + 2;

}

//--------------------------------------------------------------------------
//------ Lire un short dans une chaine de caracteres -----------------------
//--------------------------------------------------------------------------
//La fonction retourne un pointeur sur l'emplacement suivant le short
char * CLULINK_getShort(char * Ptr, unsigned short * Val)
{
  unsigned char *pShort = (unsigned char *) Val;

  pShort[1] = *Ptr++;
  pShort[0] = *Ptr++;

  return Ptr;
}

//--------------------------------------------------------------------------
//------ Ajouter un short  une chaine de caractres -----------------------
//--------------------------------------------------------------------------
int CLULINK_addLong(char * Dest, unsigned long * Val, unsigned int Pos)
{
  char *pLong = (char *) Val;
  char *pPtr  = Dest + Pos;

  pPtr[3] = *pLong++;
  pPtr[2] = *pLong++;
  pPtr[1] = *pLong++;
  pPtr[0] = *pLong++;

  return Pos + 4;
}

//--------------------------------------------------------------------------
//------ Lire un short dans une chaine de caracteres -----------------------
//--------------------------------------------------------------------------
//La fonction retourne un pointeur sur l'emplacement suivant le short
char * CLULINK_getLong(char * Ptr, unsigned long * Val)
{
  char *pLong = (char *) Val;

  pLong[3] = *Ptr++;
  pLong[2] = *Ptr++;
  pLong[1] = *Ptr++;
  pLong[0] = *Ptr++;

  return Ptr;
}

/*------ Remote Initialization complete ------------------------------------
*/
#pragma argsused
int CLULINK_cl0(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
	extern	char	PROTOCOL_initDone[65];
	int	nMRwwv, nMRdx;

	BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			  "Node %s> Remote initialization completed.\n",
			  NODE_getNodeCall(StreamNum, 0, szNodeCall));

	/* Flag init */
	PROTOCOL_initDone[StreamNum] = PROTOCOL_INIT_DONE;

	/* Envoyer le DX/WWV merge request */
	nMRwwv = HOPS_getMRwwvCount(StreamNum);
	nMRdx  = HOPS_getMRdxCount (StreamNum);
	PROTOCOL_WWVDX_mergeRequest(0, NULL, szNodeCall, nMRdx, nMRwwv);

	/* Lancer les forwards */
	MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ Add nodes ---------------------------------------------------------
	Hop count:      byte
	NodeCall:       pascal-type string
	Software ID &
	Node status:	byte
				bit 0 to 5 :
					1 - unknown
					2 - PacketCluster - type
					3 - Clusse
					4 - DxNet
				bit 6 - nohere ON
				bit 7 - conference mode ON
	Version:	unsigned int
				version number is multiplied by 1000

	If there are more than one node to add, they'll be added following.
  ------------------------------------------------------------------------*/
int CLULINK_cl1(int StreamNum, char* pszCL, int nSize)
{
	char	szPavillon[1024];
	char    szCluLink[1024];
	char*   pszEndProt = pszCL + nSize;	/* Fin du protocol */
	int	nCluLinkPos;
	int	nHops;
	extern	char	SCRIPT_fileName[65][9];


	/* Ce protocol est-il autoris ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC19, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Ne plus afficher "can't connect twice" */
	SCRIPT_fileName[StreamNum][0] = SNULL;

	/* Preparer les protocoles qu'il faudra forwarder aux autres
	   adjacents */
	strcpy(szPavillon, "PC19^");
	nCluLinkPos = PROTOCOL_setCluLink(szCluLink, CL(1));

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC19);

	/* Lire la liste de nodes a ajouter */
	while( pszCL < pszEndProt )
	{
		char	szBuffer[256];
		char	szNodeCall[16];
		char	szVersion[8];
		char	cFlag;
		unsigned short	shVersion;
		extern	char	PROTOCOL_initDone[65];

		/* Lecture de l'indicatif */
		pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

		if( ! TOOLS_isCall(szNodeCall) )
			return PROTOCOL_BADCALL;

		/* Lecture du node flag */
		cFlag = *pszCL++;

		/* Lecture du numro de version */
		pszCL = CLULINK_getShort(pszCL, &shVersion);
		sprintf(szVersion, "%04d", shVersion);


		/* Dtecter une boucle */
		if( NODE_isNodeConnected( szNodeCall ) )
		{
			/* boucle detectee - l'afficher et ignorer cet
			   indicatif */
			BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
					"*** Loop detected : %s\n",
					szNodeCall);
		}
		else
		{
			/* Ajouter ce cluster dans la table */
			NODE_newNode(StreamNum, szNodeCall, (int) shVersion,
				     cFlag);

			/* Updater le fichier des hops count (sauf si le
			   link est en cours d'init)
			   Il faut cependant updater le hops de l'adjacent
			   tout de suite, car on ne recevra plus de PC19
			   plus tard. */
			if( PROTOCOL_initDone[StreamNum] != PROTOCOL_INIT_DO )
			{
				if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_NOTSTART )
					PROTOCOL_initDone[StreamNum] = PROTOCOL_INIT_DO;

				HOPS_updateHops(szNodeCall, nHops, PC19);
			}

			/* Forwarder un mail perso pour ce cluster ? */
			MAIL_fwd2User(StreamNum, szNodeCall,
				      PROTOCOL_initDone[StreamNum]);

			/* Forward pour ce cluster ? */
			if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DO )
				MAIL_fwd(szNodeCall, 0L, BUFFER_SCREEN1);

			/* Afficher le message sur la console */
			BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
					"Node logged on : %s\n",
					szNodeCall);

			/* Ajouter au protocole pavillon */
			sprintf(szBuffer, "%c^%s^%c^%s^",
				(cFlag & BIT6 ? '1' : '0'),
				szNodeCall,
				(cFlag & BIT7 ? '*' : '-'),
				szVersion);
			strcat(szPavillon, szBuffer);

			/* Ajouter au protocole CluLink */
			nCluLinkPos = PROTOCOL_addPascalString(szCluLink,
							szNodeCall,
							nCluLinkPos);

			/* Ajouter Node Flag */
			szCluLink[nCluLinkPos++] = cFlag;


			/* Ajouter le numero de Version */
			nCluLinkPos = CLULINK_addShort(szCluLink,
						       &shVersion,
						       nCluLinkPos);
		}/*End IF*/
	}/*End WHILE*/

	/* Marquer la fin de chaine */
	szCluLink[nCluLinkPos] = SNULL;

	/* Transmettre ce protocole aux autres autres adjacents clusters,
	   seulement si au moins un des nouveaux clusters n'a pas ete
	   detecte en tant que boucle et que la longueur du protocol
	   CluLink est correcte. */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	if( strlen(szPavillon) > 5 )
		PROTOCOL_sendProt(StreamNum, NULL, PC19, szPavillon, szCluLink,
				  nHops, nCluLinkPos, PROTOCOL_ALL, -1);

	return PROTOCOL_OK;
}

/*------ Delete Node -------------------------------------------------------
	Hop count:	byte
	NodeCall:	pascal-type string
	Reason:		byte
				0   - Shutdown
				1   - Disconnected by operateur
				2   - Hard disconnect
				255 - Special reason, the reason string follows ...
	Reason String:	pascal-type string (maxlength 40)
				only send if Reason is 255.
  ------------------------------------------------------------------------*/
int CLULINK_cl2(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
	char	szReason[41];
	int	nHops;
	BYTE	byReason;
	char*   pszEndProt = pszCL + nSize;	/* Longueur trame clulink */

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC21, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC21);

	/* Lecture de l'indicatif */
	pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

	if( ! TOOLS_isCall(szNodeCall) )
		return PROTOCOL_BADCALL;

	/* Updater le fichier des hops count */
	HOPS_updateHops(szNodeCall, nHops, PC21);

	/* Lecture de la raison */
	byReason = (BYTE) *pszCL++;

	switch( byReason )
	{
		case 0 :
			strcpy(szReason, PROTOCOL_REASON_STR0);
			break;

		case 1 :
			strcpy(szReason, PROTOCOL_REASON_STR1);
			 break;

		case 2 :
			strcpy(szReason, PROTOCOL_REASON_STR2);
			break;

		case 255 : /* Raison particuliere, a lire */
			pszCL = PROTOCOL_readPascalString(szReason, pszCL, 40);
			if( szReason == NULL )
				return PROTOCOL_STROVERFLOW;
			break;

		default : /* Ne sait pas laquelle */
			byReason = 255;
			strcpy(szReason, "Unknown");
			break;
	}/*End SWITCH*/

	/* Verifier que la longueur du protocol recue est conforme */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Enlever le cluster de la table */
	if( NODE_deleteNode(StreamNum, szNodeCall) )
	{
		/* Faire suivre le protocol aux adjacents */
		PROTOCOL_nodeDisconnect(StreamNum, szNodeCall, szReason,
			byReason, nHops);

		/* Message sur la console */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"Node logged out : %s (%s)\n", szNodeCall,
			szReason);
	}
	else
	{
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"*** Protocol error : delete node for %s can't be performed.\n",
			szNodeCall);
		return PROTOCOL_NODEUNKNOWN;
	}/*End IF*/

	return PROTOCOL_OK;
}

/*------ Add users ----------------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	UserCall:	pascal-type string
	UserFlag:	byte
				Bit	Meaning		1	0
				bit 0 - Here status	here	not here
				bit 1 - Conf status	in	not in
				others are reserved

	If there are more than one user to add, they'll be added following.
  -------------------------------------------------------------------------*/
int CLULINK_cl3(int StreamNum, char* pszCL, int nSize)
{
	char	szPavillon[1024];
	char	szCluLink[1024];
	char	szNodeCall[16];
	int	nHops;
	int	nCluLinkPos;
	char*	pEndProt = pszCL + nSize;
	extern	char	PROTOCOL_initDone[65];

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC16, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC16);

	/* Lecture de l'indicatif du home node */
	pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

	if( ! TOOLS_isCall(szNodeCall) )
		return PROTOCOL_BADCALL;

	/* Updater le fichier des hop count (sauf si le link est en cours
	   d'init) */
	if( PROTOCOL_initDone[StreamNum] == PROTOCOL_INIT_DONE )
		HOPS_updateHops(szNodeCall, nHops, PC16);

	/* Vrifier que ce cluster est connu */
	if( ! NODE_isNodeConnected(szNodeCall) )
	{
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"*** Protocol error : Can't add users on an unknown node (%s).\n",
			szNodeCall);
		return PROTOCOL_NODEUNKNOWN;
	}

	/* Prparer les protocoles qu'il faudra envoyer aux adjacents */
	sprintf(szPavillon, "PC16^%s^", szNodeCall);
	nCluLinkPos = PROTOCOL_setCluLink(szCluLink, CL(3));
	nCluLinkPos = PROTOCOL_addPascalString(szCluLink, szNodeCall,
					       nCluLinkPos);

	/* Lire la liste des utilisateurs a ajouter */
	while( pszCL < pEndProt )
	{
		char	szUserCall[16];
                char	cFlag;
		char	szBuffer[256];

		/* Lecture de l'indicatif */
			pszCL = PROTOCOL_readPascalString(szUserCall, pszCL, 9);
			if( ! TOOLS_isCall(szUserCall) )
				return PROTOCOL_BADCALL;
		/* Lecture du user flag */
		cFlag = *pszCL++;

		/* Ajouter cet utilisateur dans la table */
		NODE_newUser(StreamNum, 0, szNodeCall, szUserCall, cFlag);

		/* Forward pour cet utilisateur ? */
		MAIL_fwd2User(StreamNum, szUserCall,
			      PROTOCOL_initDone[StreamNum]);

		/* Afficher le message sur la console */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"Node %s> Adding %s\n", szNodeCall, szUserCall);

		/* Ajouter au protocol Pavillon */
		sprintf(szBuffer, "%s %c %c^", szUserCall,
			(cFlag & PROTOCOL_USER_CLUSTERCONF ? '*' : '-'),
			(cFlag & PROTOCOL_USER_HERE ? '1' : '0'));
		strcat(szPavillon, szBuffer);

		/*  Ajouter au protocol CluLink */
		nCluLinkPos = PROTOCOL_addPascalString(szCluLink, szUserCall,
			      nCluLinkPos);
		szCluLink[nCluLinkPos++] = cFlag;
	} /* End WHILE */

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pEndProt )
		return PROTOCOL_BADSIZE;

	/* Transmettre ce protocole aux adjacents */
	szCluLink[nCluLinkPos] = SNULL;
	PROTOCOL_sendProt(StreamNum, NULL, PC16, szPavillon, szCluLink, nHops,
			nCluLinkPos, PROTOCOL_ALL, -1);

	return PROTOCOL_OK;
}

/*------ Delete user --------------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	UserCall:	pascal-type string
  -------------------------------------------------------------------------*/
int CLULINK_cl4(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
	char	szUserCall[16];
	int 	nHops;
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC17, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC17);

	/* Lecture de l'indicatif du node */
	pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

	if( ! TOOLS_isCall(szNodeCall) )
		return PROTOCOL_BADCALL;

	/* Updater le fichier des hop count */
	HOPS_updateHops(szNodeCall, nHops, PC17);

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szUserCall, pszCL, 9);

	if( ! TOOLS_isCall(szUserCall) )
        	return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Effacer l'indicatif de l'utilisateur de la table */
	if( ! NODE_deleteUser(StreamNum, szNodeCall, szUserCall) )
	{
		/* A pu etre efface, faire suivre aux adjacents */
		PROTOCOL_deleteUser(StreamNum, szNodeCall, szUserCall, nHops);

		/* Message sur la console */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"Node %s> Removing %s\n", szNodeCall, szUserCall);
	}
	else
	{
		/* Erreur, cet utilisateur n'tait pas dans la table */
		BUFFERS_printBuff(BUFFER_SCREEN1, OUT,
			"*** Protocol error : Can't remove %s from node %s.\n",
			szUserCall, szNodeCall);

		return PROTOCOL_USERUNKNOWN;
      } /* End IF */

      return PROTOCOL_OK;
}

/*------ Delete node requested by operator --------------------------------
	FromNode:	pascal-type string
	ToNode:		pascal-type string
  -------------------------------------------------------------------------*/
int CLULINK_cl5(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char*	pszEndProt = pszCL + nSize;
	int	nHops;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC39, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC39);

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Interpreter le protocole */
	PROTOCOL_rcvExplicitDisconnect(StreamNum, szFromNode, "Disconnected by operator", nHops);

	return PROTOCOL_OK;
}



/*------ Change User Flag ---------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	UserCall:	pascal-type string
	UserFlag:	byte (see message #3)
  -------------------------------------------------------------------------*/
int CLULINK_cl10(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
	char	szUserCall[16];
	int     nHops;
	int	nType;
	char	cFlag;
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC24, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC24);

	/* Lecture de l'indicatif du node */
	pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

	if( ! TOOLS_isCall(szNodeCall) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szUserCall, pszCL, 9);

	/* Updater de UserFlag et regarder quelle info a changee */
	cFlag = NODE_getUserFlag(szNodeCall, szUserCall);
	nType = 0;

	if( (cFlag & PROTOCOL_USER_HERE) != (*pszCL & PROTOCOL_USER_HERE) )
		nType = PROTOCOL_USER_HERE;
	else if( (cFlag & PROTOCOL_USER_CONF) != (*pszCL & PROTOCOL_USER_CONF) )
		nType = PROTOCOL_USER_CONF;

	cFlag = *pszCL++;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Transmettre l'information aux clusters adjacents */
	PROTOCOL_sendUserFlag(StreamNum, szNodeCall, szUserCall, nType,
		cFlag & nType, nHops);

	return PROTOCOL_OK;
}

/*------ Change User data ---------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	UserCall:	pascal-type string
	DataType:	byte
				1 - Name (max length 22)
				2 - Qth  (max length 80)
				3 - Coordinates (not yet implemented in DxNet)
				4 - Home node (max length 9)
				5 - Locator (max length 6)
	Info:		pascal-type string

	Remark : if FromNode is unknown (this protocol has been send with
		 a PC-type protocol), CluLink sends a '?' in place of.
  -------------------------------------------------------------------------*/
int CLULINK_cl11(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
	char	szUserCall[16];
	char	szBuffer[128];
	int	nType;
	int	nHops;
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC41, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC41);

	/* Lecture de l'indicatif du node */
	pszCL = PROTOCOL_readPascalString(szNodeCall, pszCL, 9);

	if( szNodeCall == NULL ) /* Pour proteger le strcmp du test suivant */
		return PROTOCOL_BADCALL;

	/* Remarque : lorsque FromNode n'a pas pu etre determiner par
	   CluLink, FromNode contient '?' */
	if( strcmp(szNodeCall, "?") && ! TOOLS_isCall(szNodeCall) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szUserCall, pszCL, 9);

	if( ! TOOLS_isCall(szUserCall) )
		return PROTOCOL_BADCALL;

	/* Lecture du type d'information (data type) */
	nType = (unsigned int) *pszCL++;

	/* Lecture de la chaine d'information */
	pszCL = PROTOCOL_readPascalString(szBuffer, pszCL, 80);

	if( pszCL == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Sauvegarder l'information */
	USERS_changeConfig(szUserCall, szBuffer, nType);

	/* Transmettre l'information aux clusters adjacents */
	PROTOCOL_sendUserConfig(StreamNum, szNodeCall, szUserCall, szBuffer,
				nType, nHops);

	return PROTOCOL_OK;
}

/*------ Ping message ------------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	ToNode:		pascal-type string
	pingFlag:	byte
				0 - Answer for a ping
				1 - Ping
  -------------------------------------------------------------------------*/
int CLULINK_cl12(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	int	nHops;
	char* 	pszEndProt = pszCL + nSize;
	char	cPingFlag;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC51, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC51);

	/* Lecture de l'indicatif de FromNode */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de ToNode */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture du ping flag */
	cPingFlag = *pszCL++;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole ... ou repondre si besoin. */
	PROTOCOL_ping(StreamNum, szFromNode, szToNode,
		      (cPingFlag == 1 ? PROTOCOL_PING_ASK : PROTOCOL_PING_ANSWER),
		      nHops);

	return PROTOCOL_OK;
}

/*------ Local user count message -------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	User count:     unsigned short
  -------------------------------------------------------------------------*/
int CLULINK_cl13(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	int	nHops;
	char*	pszEndProt = pszCL + nSize;
	unsigned short	shUserCount;

	/* Lecture du hop count */
	nHops = HOPS_char2int(StreamNum, NULL, *pszCL++, PC50);

	/* Lecture de l'indicatif de l'expediteur */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, szFromNode, StreamNum, PC50, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du user count */
	pszCL = CLULINK_getShort(pszCL, &shUserCount);

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole ... */
	PROTOCOL_userCount(StreamNum, szFromNode, TRUE, shUserCount, nHops);

	return PROTOCOL_OK;
}

/*------ Remote Initialization complete ------------------------------------
*/
#pragma argsused
int CLULINK_cl60(int StreamNum, char* pszCL, int nSize)
{
	char	szNodeCall[16];
//	extern	char	PROTOCOL_initDone[65];
	char szReason[256];

	/* Informer les adjacents */
	NODE_getNodeCall(StreamNum, 0, szNodeCall);
	sprintf(szReason, "Remote init request by %s", szNodeCall);
	PROTOCOL_clusterStreamDisconnected(StreamNum, szReason, -1);

	BUFFERS_printBuff(BUFFER_SCREEN1, OUT, "%s : Transmitting local configuration...\n",
			  szNodeCall);

	PROTOCOL_sendInit(StreamNum);

	return PROTOCOL_OK;
}


/*------ DX info -----------------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 9 bytes)
	DxFreq:		unsigned long (4 bytes)
				Frequency of the DX station in KHz, multiplied by 10
				An integer is used to prevent rouding errors.
	DxCall:		pascal-type string (max 14 bytes)
	DateTime:	unsigned long
	Comments:	pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl64(int StreamNum, char* pszCL, int nSize)
{
	char    szFromNode[16];
	char	szFromUser[16];
	char	szDxCall[16];
	char	szComments[128];
	char	szDate[16];
	char	szTime[8];
	ulong	lFrequency;
	ulong	lDateTime;
	int 	nHops;
	char*	pszEndProt = pszCL + nSize;
	char    cHops;

	/* Lecture du hop count */
	cHops = *pszCL++;

	/* Lecture de l'indicatif de FromNode */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture de la frquence */
	pszCL = CLULINK_getLong(pszCL, &lFrequency);
	if( DX_checkFrequency(lFrequency) == 0 )
		return PROTOCOL_BADFREQUENCY;

	/* Lecture de l'indicatif du DX */
	pszCL = PROTOCOL_readPascalString(szDxCall, pszCL, 14);

	if( szDxCall == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture du commentaire */
	pszCL = PROTOCOL_readPascalString(szComments, pszCL, 127);

	if( szComments == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole */
	nHops = HOPS_char2int(StreamNum, szFromNode, cHops, PC11);
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);
	PROTOCOL_sendDx(StreamNum, szFromNode, szFromUser, lFrequency, szDate,
			szTime, szDxCall, szComments, nHops);

	return PROTOCOL_OK;
}

/*------ WWV info -----------------------------------------------------------
	Hop count:	byte
	FromNode:	poscal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 6 bytes, without SSID)
	SFI, A, K:	unsigned long
				Bit	Meaning
				0-8	SFI
				9-17	A
				18-21	K

			Some info about SFI, A and K range :

				SFI: 	normally from 64 to 300 but could
					be greater
				A:	from 0 to 400
				K	from 0 to 9
	DateTime:	unsigned long
	Forcast:	pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl65(int StreamNum, char* pszCL, int nSize)
{
	char		szFromNode[16];
	char		szFromUser[16];
	char		szForecast[128];
	char		szDate[16];
	char		szTime[8];
	char*		pszEndProt = pszCL + nSize;
	ulong		lDateTime;
	int		nHops;
	ulong		lDatas;
	unsigned int	iSfi,
			iA,
			iK;
	char		cHops;

	/* Lecture du hop count */
	cHops = *pszCL++;

	/* Lecture de l'indicatif de FromNode */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, szFromNode, StreamNum, PC23, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture de SFI, A et K */
	pszCL = CLULINK_getLong(pszCL, &lDatas);
	iSfi  = lDatas         & 511;      /* 9 premiers bits */
	iA    = (lDatas >> 9)  & 511; 	  /* 9 bits suivants */
	iK    = (lDatas >> 18) & 15;	  /* 4 derniers bits */

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture de la prevision */
	pszCL = PROTOCOL_readPascalString(szForecast, pszCL, 80);

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole */
	nHops = HOPS_char2int(StreamNum, szFromNode, cHops, PC23);
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);
	PROTOCOL_sendWwv(StreamNum, szFromNode, szFromUser, iSfi, iA, iK,
			 szDate, szTime, szForecast, nHops);

	return PROTOCOL_OK;
}

/*------ Annoucement --------------------------------------------------------
	Hop count:	byte
	FromNode:	pascal-type string
	FromUser:	pascal-type string
	ToNode or
	Distribution:	pascal-type string (max 20 bytes)
	DateTime:	unsigned long
	Message flags:	byte
				Bit	Meaning
				bit 0 - Cluster-wide announcement
				bit 1 - To users connected to the specified cluster only
				bit 2 - To a distribution list of users
	Message:	pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl66(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char	szDistrib[32];
	char	szMessage[128];
	char* pszEndProt = pszCL + nSize;
	ulong	lDateTime;
	char	cMessageFlags;
	int	nHops;
	char	cHops;

	/* Lecture du hop count */
	cHops = *pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, szFromNode, StreamNum, PC12, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture du cluster destinataire ou de la distribution */
	pszCL = PROTOCOL_readPascalString(szDistrib, pszCL, 20);

	if( szDistrib == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture du Message Flags */
	cMessageFlags = *pszCL++;

	/* Lecture du message */
	pszCL = PROTOCOL_readPascalString(szMessage, pszCL, 255);

	if( szMessage == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole */
	nHops = HOPS_char2int(StreamNum, szFromNode, cHops, PC12);
	PROTOCOL_sendAnnounce(StreamNum, szFromNode, szFromUser, szDistrib,
			      lDateTime, cMessageFlags, szMessage, nHops);

	return PROTOCOL_OK;
}

/*------ Talk --------------------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
	ToUser:		pascal-type string (max 9 bytes)
	DateTime:	unsigned long
	Message flags:	byte
				Bit	Meaning		ON	OFF
				bit 0 - Bell flag	1	0
	Message:	pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl67(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char	szToNode[16];
	char	szToUser[16];
	ulong	lDateTime;
	char	cMessageFlags;
	char	szMessage[256];
	char*	pszEndProt = pszCL + nSize;
	int	nHops;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC10, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hop count */
	nHops = (unsigned int) *pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'expediteur */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du destinataire */
	pszCL = PROTOCOL_readPascalString(szToUser, pszCL, 9);

	if( ! TOOLS_isCall(szToUser) )
		return PROTOCOL_BADCALL;

	/* Lecture de la date */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture du flag */
	cMessageFlags = *pszCL++;

	/* Lecture du message */
	pszCL = PROTOCOL_readPascalString(szMessage, pszCL, 255);	/*Pour etre large*/

	if( szMessage == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole */
	PROTOCOL_sendTalk(StreamNum, szFromNode, szFromUser, szToNode,
			  szToUser, szMessage, cMessageFlags, lDateTime,
			  nHops);

	return PROTOCOL_OK;
}

/*------ 68 - DX/WWV merge request ----------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
	DXcount:	byte
	WWVCount:	byte
  -------------------------------------------------------------------------*/
int CLULINK_cl68(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	BYTE    byDxCount;
	BYTE	byWwvCount;
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC25, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hops (inutilise) */
	pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);
	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);
	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture des compteurs */
	byDxCount  = (BYTE) *pszCL++;
	byWwvCount = (BYTE) *pszCL++;

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Traiter le protocole */
	if( PROTOCOL_WWVDX_mergeRequest(StreamNum, szFromNode, szToNode,
				    byDxCount, byWwvCount) == FALSE )
		return PROTOCOL_NODEUNKNOWN; /* Routage impossible */

	return PROTOCOL_OK;
}

/*------ 69 - DX merge info ------------------------------------------------

	FromNode:	pascal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	DxFreq:		unsigned long (4 bytes)
				Frequency of the DX station in KHz, multiplied by 10
				An integer is used to prevent rouding errors.
	DxCall:		pascal-type string (max 14 bytes)
	DateTime:	unsigned long
	Comments:	pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl69(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szLogger[16];
	char	szToNode[16];
	ulong 	lFrequency;
	char	szDxCall[16];
	ulong   lDateTime;
	char	szComments[128];
	char*	pszEndProt = pszCL + nSize;

	char	szDate[16];
	char	szTime[8];

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC26, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hops (inutilise) */
	pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);
	if( ! TOOLS_isCall(szFromNode) && strcmp(szFromNode, "?") )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szLogger, pszCL, 9);
	if( ! TOOLS_isCall(szLogger) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);
	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de la frequence */
	pszCL = CLULINK_getLong(pszCL, &lFrequency);
	if( DX_checkFrequency(lFrequency) == 0 )
		return PROTOCOL_BADFREQUENCY;

	/* Lecture de l'indicatif du dx */
	pszCL = PROTOCOL_readPascalString(szDxCall, pszCL, 14);

	/* Lecture de la date/heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture du commentaire */
	pszCL = PROTOCOL_readPascalString(szComments, pszCL, 80);

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Traiter le protocole */
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);

	if( PROTOCOL_DXMergeInfo(StreamNum, szFromNode, szToNode, szLogger,
				 lFrequency, szDate,  szTime, szDxCall,
				 szComments) == FALSE )
		return PROTOCOL_NODEUNKNOWN;	/* Routage impossible */

	return PROTOCOL_OK;
}

/*------ 70 - WWV merge info -----------------------------------------------
	FromNode:	poscal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	SFI, A, K:	unsigned long
				Bit	Meaning
				0-8	SFI
				9-17	A
				18-21	K
	DateTime:	unsigned long
	Forecast:	pascal-type string (max 80 bytes)

  -------------------------------------------------------------------------*/
int CLULINK_cl70(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szLogger[16];
	char	szToNode[16];
	int	nSFI, nA, nK;
	ulong 	lDateTime;
	char    szForecast[128];
	char*	pszEndProt = pszCL + nSize;

	ulong   lDatas;
	char	szDate[16];
	char	szTime[8];

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC27, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hops (inutilise) */
	pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);
	if( ! TOOLS_isCall(szFromNode) && strcmp(szFromNode, "?") )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szLogger, pszCL, 9);
	if( ! TOOLS_isCall(szLogger) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);
	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de SFI, A et K */
	pszCL = CLULINK_getLong(pszCL, &lDatas);
	nSFI  = lDatas         & 511;      /* 9 premiers bits */
	nA    = (lDatas >> 9)  & 511; 	   /* 9 bits suivants */
	nK    = (lDatas >> 18) & 15;	   /* 4 derniers bits */

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture de la prevision */
	pszCL = PROTOCOL_readPascalString(szForecast, pszCL, 80);

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;


	/* Traiter le protocole */
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);

	if( PROTOCOL_WWVMergeInfo(StreamNum, szFromNode, szToNode, szLogger,
				  nSFI, nA, nK, szDate, szTime,
				  szForecast) == FALSE )

		return PROTOCOL_NODEUNKNOWN;	/* Routage impossible */

	return PROTOCOL_OK;
}

/*------ WCY info -----------------------------------------------------------
	Hop count:	byte
	FromNode:	poscal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 6 bytes, without SSID)
	DateTime:	unsigned long
	SFI, A, K:	unsigned long
				Bit	Meaning
				0-8	SFI
				9-17	A
				18-21	K

			Some info about SFI, A and K range :

				SFI: 	normally from 64 to 300 but could
					be greater
				A:	from 0 to 400
				K	from 0 to 9
	R, ExpK:	unsigned short
				Bit	Meaning
				0-8	R
				9-12	ExpK
				13-16	Reserved

			Some info about R and ExpK
				R:	from 0 to 300
				ExpK	from 0 to 9

	SA:             pascal-type string (max 3 bytes)
	GMF:		pascal-type string (max 3 bytes)
	Aurora:		pascal-type string (max 6 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl71(int StreamNum, char* pszCL, int nSize)
{
	char		szFromPC[16];
	char		szLogger[16];
	char		szDate[16];
	char		szTime[8];
	char*		pszEndProt = pszCL + nSize;
	ulong		lDateTime;
	int		nHops;
	ulong		lData;
	unsigned short	shData;
	unsigned int	nSFI,
			nA,
			nK,
			nR,
			nExpK;
	char		szSA[8],
			szGMF[8],
			szAurora[8];

	char		cHops;

	/* Lecture du hop count */
	cHops = *pszCL++;

	/* Lecture de l'indicatif de FromNode */
	pszCL = PROTOCOL_readPascalString(szFromPC, pszCL, 9);
	if( ! TOOLS_isCall(szFromPC) )
		return PROTOCOL_BADCALL;

	/* Ce protocol est-il autorise ? */
	if( HOPS_fix(StreamNum, szFromPC, StreamNum, PC23, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture de l'indicatif du logger */
	pszCL = PROTOCOL_readPascalString(szLogger, pszCL, 9);
	if( ! TOOLS_isCall(szLogger) )
		return PROTOCOL_BADCALL;

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture de SFI, A et K */
	pszCL = CLULINK_getLong(pszCL, &lData);
	nSFI  =  lData        & 511;      /* 9 premiers bits */
	nA    = (lData >> 9)  & 511; 	  /* 9 bits suivants */
	nK    = (lData >> 18) & 15;	  /* 4 derniers bits */

	/* Lecture de R et de ExpK */
	pszCL = CLULINK_getShort(pszCL, &shData);
	nR    =  shData       & 511;
	nExpK = (shData >> 9) & 15;

	/* Lecture des chaines SA, GMF et Aurora */
	pszCL = PROTOCOL_readPascalString(szSA,     pszCL, 3);
	pszCL = PROTOCOL_readPascalString(szGMF,    pszCL, 3);
	pszCL = PROTOCOL_readPascalString(szAurora, pszCL, 6);

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Faire suivre le protocole */
	nHops = HOPS_char2int(StreamNum, szFromPC, cHops, PC73);
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);

	/* Faire suivre le protocole */
	PROTOCOL_sendWcy(StreamNum, szDate, szTime, nSFI, nA, nK, nExpK,
		nR, szSA, szGMF, szAurora, szLogger, szFromPC, nHops);

	return PROTOCOL_OK;
}



/*------ SendSubject --------------------------------------------------------
	FromNode: 	pascal-type string (max 9 bytes)
	FromUser:	pascal-type string (without ssid - max 6 bytes)
	ToNode:		pascal-type string (max 9 bytes)
	Recipient:	pascal-type string (without ssid - max 8 bytes)
	StartNode:	pascal-type string (max 9 bytes)
	Title:		pascal-type string (max 60 bytes)
	DateTime:       unsigned long
	Message flags:	byte
				Bit	Meaning
				0-4     Number of lines before sending ack
				6	Return Receipt
				7	PrivateFlag (1=Priv 0=Bull)
  -------------------------------------------------------------------------*/
int CLULINK_cl95(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char	szToNode[16];
	char	szRecipient[16];
	char	szStartNode[16];
	char	szTitle[128];
	char	szType[8];
	char	szDate[16];
	char	szTime[8];
	char	cNumLines;
	ulong	lDateTime;
	char	cMessageFlags;
	BYTE	byRetReceipt;
      /*int	nHops;*/
	char*	pszEndProt = pszCL + nSize;
	char    szBuffer[256];
	int     nDupe = FALSE;

	extern	char	PROTOCOL_mailEveryNLines[65];
	extern	char	PROTOCOL_mailLinesCounter[65];

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC28, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hops count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'expediteur du msg */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du recipient */
	pszCL = PROTOCOL_readPascalString(szRecipient, pszCL, 8);

	if( szRecipient == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Lecture du StartNode du msg */
	pszCL = PROTOCOL_readPascalString(szStartNode, pszCL, 9);

	if( szStartNode == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Lecture du titre du msg */
	pszCL = PROTOCOL_readPascalString(szTitle, pszCL, 60);

	if( szTitle == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Lecture de la date et de l'heure */
	pszCL = CLULINK_getLong(pszCL, &lDateTime);

	/* Lecture du MessageFlags */
	cMessageFlags = *pszCL++;

	cNumLines = cMessageFlags & 31;/* Bits 0-4 */

	if( cMessageFlags & 64 )       /* Tester le bit 6 (return receipt) */
		byRetReceipt = TRUE;
	else
		byRetReceipt = FALSE;

	if( cMessageFlags & 128 )      /* Tester le bit 7 (perso) */
		strcpy(szType, "sp");
	else
		strcpy(szType, "sb");

	/* Verifier que la longueur du protocol est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Convertir lDateTime en chaines */
	TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);

	/* Recherche de doublons */
	if( MAIL_isDupe(szRecipient, szFromUser, szTitle, szDate, szTime) )
		nDupe = TRUE;

	/* Creer le fichier temporaire */
	/* Destinataire & expediteur */
	sprintf(szBuffer, "%s < %s", szRecipient, szFromUser);

	if( ! MAIL_rcvRecip(StreamNum, szType, szBuffer, byRetReceipt, nDupe) )
		return PROTOCOL_OK;	/* Probleme ... */

	/* Ajouter les datas concernant le mail ds le fichier temp */
	if( ! MAIL_addSubject2Tmp(StreamNum, szTitle, szDate, szTime,
				  szStartNode, szFromNode) )
		return PROTOCOL_OK;	/* Probleme ... */

	/* Repondre au protocole */
	if( nDupe == FALSE )
	{
		PROTOCOL_sendAckSubject(StreamNum, szFromNode, szToNode,
			(unsigned long) StreamNum);
	}
	else
	{
		PROTOCOL_sendForwardingAbort(StreamNum, szFromNode,
			szToNode, 0L);
	}

	/* Nombre de lignes par blocs */
	PROTOCOL_mailEveryNLines[StreamNum]  = cNumLines;
	PROTOCOL_mailLinesCounter[StreamNum] = PROTOCOL_mailEveryNLines[StreamNum];

	return PROTOCOL_OK;
}

/*------ AckSubject --------------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl96(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char*	pszEndProt = pszCL + nSize;

	extern	ulong  	PROTOCOL_lFwdMesNum[MAX_STREAMS];

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC30, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passage du hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'expediteur du msg */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	PROTOCOL_lFwdMesNum[StreamNum] = (unsigned long) StreamNum;
	MAIL_sendNextBloc(StreamNum, szToNode, szFromNode,
			  PROTOCOL_lFwdMesNum[StreamNum]);

	return PROTOCOL_OK;
}

/*------ SendText -----------------------------------------------------------
	Fromnode:	pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	Text:		pascal-type string (max 80 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl97(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szText[256];
	char*	pszEndProt = pszCL + nSize;

	extern	char	PROTOCOL_mailEveryNLines[65];
	extern	char	PROTOCOL_mailLinesCounter[65];

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC30, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif du cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Lecture du texte */
	pszCL = PROTOCOL_readPascalString(szText, pszCL, 255); 	/* 255 pour etre large */

	if( szText == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Ajouter la ligne de texte ds le fichier temp */
	MAIL_addText2Tmp(StreamNum, szText);

	/* Gestion du conteur de ligne - faut-il envoyer un ACK ? */
	if( --PROTOCOL_mailLinesCounter[StreamNum] == 0 )
	{
		PROTOCOL_sendAckText(StreamNum, szFromNode, szToNode, (unsigned long) StreamNum);
		PROTOCOL_mailLinesCounter[StreamNum] = PROTOCOL_mailEveryNLines[StreamNum];
	}

	return PROTOCOL_OK;
}

/*------ AckText ------------------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl98(int StreamNum, char* pszCL, int nSize)
{
	/* Traitement identique a CL96 */
	return  CLULINK_cl96(StreamNum, pszCL, nSize);
}

/*------ CompleteText -------------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl99(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC32, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Valider le message */
	if( MAIL_tmp2Msg(StreamNum) )
		PROTOCOL_sendAckCompleteText(StreamNum, szFromNode, szToNode,
					     (unsigned long) StreamNum);

	return PROTOCOL_OK;
}

/*------ AckCompleteText ----------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl100(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char*	pszEndProt = pszCL + nSize;

	extern	unsigned long 	MAIL_lFwdMsgNumber[MAX_STREAMS];

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC33, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'expditeur du msg */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	MAIL_changeStatut(MAIL_lFwdMsgNumber[StreamNum], STATUT_F);
	MAIL_deQueue(szFromNode, MAIL_lFwdMsgNumber[StreamNum]);
	MAIL_lFwdMsgNumber[StreamNum] = 0L;
	MAIL_fwd();

	return PROTOCOL_OK;
}

/*------ ForwardingAbort ----------------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
  -------------------------------------------------------------------------*/
int CLULINK_cl101(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char*	pszEndProt = pszCL + nSize;

	extern	unsigned long 	MAIL_lFwdMsgNumber[MAX_STREAMS];

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC33, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'expditeur du msg */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	MAIL_deQueue(szFromNode, MAIL_lFwdMsgNumber[StreamNum]);
	MAIL_lFwdMsgNumber[StreamNum] = 0L;
	MAIL_fwd();

	return PROTOCOL_OK;
}


/*------ conference : Add user -------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	FromUser:       pascal-type string (max 9 bytes)

	Remark : FromNode is set to "" if it's unknown (ie if the
		 protocol has been received via a PC13)
  ------------------------------------------------------------------------*/
int CLULINK_cl140(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char*	pszEndProt = pszCL + nSize;
	int	nHops;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC13, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = (unsigned int) pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! szFromNode[0] && ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Traiter le protocole */
	PROTOCOL_sendClusterConfLogin(StreamNum, szFromNode, szFromUser,
		nHops);

	return PROTOCOL_OK;
}


/*------ conference : remove user ----------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	FromUser:       pascal-type string (max 9 bytes)

	Remark : FromNode is set to "" if it's unknown (ie if the
		 protocol has been received via a PC13)
  ------------------------------------------------------------------------*/
int CLULINK_cl141(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char*	pszEndProt = pszCL + nSize;
	int	nHops;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC14, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = (unsigned int) pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! szFromNode[0] && ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Traiter le protocole */
	PROTOCOL_sendClusterConfLogout(StreamNum, szFromNode, szFromUser,
		nHops);

	return PROTOCOL_OK;
}

/*------ conference : user message ---------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	FromUser:       pascal-type string (max 9 bytes)
	Message:        pascal-type string (max 128 bytes)

	Remark : FromNode is set to "" if it's unknown (ie if the
		 protocol has been received via a PC13)
  ------------------------------------------------------------------------*/
int CLULINK_cl142(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szFromUser[16];
	char 	szMessage[256];
	char*	pszEndProt = pszCL + nSize;
	int	nHops;

	/* Ce protocole est-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC15, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Lecture du hops */
	nHops = (unsigned int) pszCL++;

	/* Lecture de l'indicatif du cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! szFromNode[0] && ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Lecture de l'indicatif de l'utilisateur */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Lecture du message */
	pszCL = PROTOCOL_readPascalString(szMessage, pszCL, 128);

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Traiter le protocole */
	PROTOCOL_sendClusterConferenceMessage(StreamNum, szFromNode,
		szFromUser, szMessage, nHops);

	return PROTOCOL_OK;
}



/*------ Remote Database Request --------------------------------------------
	FromNode:	pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
	FromUser:	pascal-type string (max 9 bytes)
	Qualifier:	pascal-type string (max 20 bytes)
	Key:		pascal-type string (max 20 bytes)
	BYTE:		user stream number
  -------------------------------------------------------------------------*/
int CLULINK_cl160(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szFromUser[16];
	char	szServer[16];
	char	szQualifier[32];
	char	szKey[32];
	int	nUserStream;
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC44, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Utilisateur */
	pszCL = PROTOCOL_readPascalString(szFromUser, pszCL, 9);

	if( ! TOOLS_isCall(szFromUser) )
		return PROTOCOL_BADCALL;

	/* Base de donnees - "Qualifier" */
	pszCL = PROTOCOL_readPascalString(szQualifier, pszCL, 20);

	if( szQualifier == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Clef - "Key" */
	pszCL = PROTOCOL_readPascalString(szKey, pszCL, 20);

	if( szKey == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Numero du stream de l'utilisateur */
	nUserStream = (unsigned int) *pszCL++;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Repondre ou transmettre a un adjacent ? */
	NODE_getNodeCall(0, 0, szServer);
	if( ! strcmp(szToNode, szServer) )
	{
		/* Interroger la base de donnees */
		if( ! DATABASE_remote(nUserStream, szQualifier, szKey,
				      szFromNode) )
		{
			/* Erreur acces database */
			PROTOCOL_dbResponse(0, szToNode, szFromNode,
				nUserStream,
				"*** Error accessing the database : database unknown");
		}
		PROTOCOL_dbComplete(StreamNum, szToNode, szFromNode,
			nUserStream);
	}
	else
	{
		/* Faire suivre ... */
		PROTOCOL_dbRequest(StreamNum, szFromNode, szToNode,
			nUserStream, szQualifier, szKey, szFromUser);
	}

	return PROTOCOL_OK;
}

/*------ Remote Database Response -------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	String:         pascal-type string (max 128 bytes)
	UserStream:	BYTE
  -------------------------------------------------------------------------*/
int CLULINK_cl161(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szString[256];
	char    szServer[16];
	char*	pszEndProt = pszCL + nSize;
	int	nUserStream;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC45, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Chaine contenant la reponse */
	pszCL = PROTOCOL_readPascalString(szString, pszCL, 128);

	if( szString == NULL )
		return PROTOCOL_STROVERFLOW;

	/* Numero du stream de l'utilisateur */
	nUserStream = (unsigned int) *pszCL++;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Repondre ou transmettre a un adjacent ? */
	NODE_getNodeCall(0, 0, szServer);
	if( ! strcmp(szToNode, szServer) )
	{
		/* Transmettre a l'utilisateur */
		BUFFERS_printBuff(nUserStream, OUT, "%s\n", szString);
	}
	else
	{
		/* Faire suivre ... */
		PROTOCOL_dbResponse(StreamNum, szFromNode, szToNode,
			nUserStream, szString);
	}

	return PROTOCOL_OK;
}

/*------ Remote Database Complete ------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	UserStream:	BYTE
  -------------------------------------------------------------------------*/
int CLULINK_cl162(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char    szServer[16];
	char*	pszEndProt = pszCL + nSize;
	int	nUserStream;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC45, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/*nHops = (unsigned int)*/ /* * */ pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Numero du stream de l'utilisateur */
	nUserStream = (unsigned int) *pszCL++;

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Repondre ou transmettre a un adjacent ? */
	NODE_getNodeCall(0, 0, szServer);
	if( ! strcmp(szToNode, szServer) )
	{
		/* Transmettre a l'utilisateur */
		MSG_send(nUserStream, MSG62);
	}
	else
	{
		/* Faire suivre ... */
		PROTOCOL_dbComplete(StreamNum, szFromNode, szToNode,
			nUserStream);
	}

	return PROTOCOL_OK;
}

/*------ Remote command request -------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	Command:        pascal-type string (max 200 bytes)

  -------------------------------------------------------------------------*/
int CLULINK_cl200(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szCmd[256];
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC34, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/* * */ pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Commande */
	pszCL = PROTOCOL_readPascalString(szCmd, pszCL, 200);

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Interpreter le protocole */
	PROTOCOL_rCmd(StreamNum, szFromNode, szToNode, NULL, szCmd);

	return PROTOCOL_OK;
}

/*------ Remote command response ------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	Command:        pascal-type string (max 200 bytes)

  -------------------------------------------------------------------------*/
int CLULINK_cl201(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szCmd[256];
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC35, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	/* * */ pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Commande */
	pszCL = PROTOCOL_readPascalString(szCmd, pszCL, 200);

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Interpreter le protocole */
	PROTOCOL_rCmdResponse(StreamNum, szFromNode, szToNode, NULL, szCmd);

	return PROTOCOL_OK;
}

/*------ Remote command request (2nd version) -----------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:		pascal-type string (max 9 bytes)
	FmUser:         pascal-type string (max 9 bytes)
	Command:        pascal-type string (max 200 bytes)

  -------------------------------------------------------------------------*/
int CLULINK_cl202(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szFmUser[16];
	char	szCmd[256];
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC34, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Utilisateur */
	pszCL = PROTOCOL_readPascalString(szFmUser, pszCL, 9);

	if( ! TOOLS_isCall(szFmUser) )
		return PROTOCOL_BADCALL;

	/* Commande */
	pszCL = PROTOCOL_readPascalString(szCmd, pszCL, 200);

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Interpreter le protocole */
	PROTOCOL_rCmd(StreamNum, szFromNode, szToNode, szFmUser, szCmd);

	return PROTOCOL_OK;
}

/*------ Remote command response ------------------------------------------
	FromNode:       pascal-type string (max 9 bytes)
	ToNode:         pascal-type string (max 9 bytes)
	ToUser:         pascal-type string (max 9 bytes)
	Command:        pascal-type string (max 200 bytes)

  -------------------------------------------------------------------------*/
int CLULINK_cl203(int StreamNum, char* pszCL, int nSize)
{
	char	szFromNode[16];
	char	szToNode[16];
	char	szToUser[16];
	char	szCmd[256];
	char*	pszEndProt = pszCL + nSize;

	/* Ce protocole est t-il autorise ? */
	if( HOPS_fix(StreamNum, NULL, StreamNum, PC35, HOPS_IN, 100) == 0 )
		return PROTOCOL_OK;

	/* Passer le hop count (inutilise) */
	pszCL++;

	/* Cluster d'origine */
	pszCL = PROTOCOL_readPascalString(szFromNode, pszCL, 9);

	if( ! TOOLS_isCall(szFromNode) )
		return PROTOCOL_BADCALL;

	/* Cluster destinataire */
	pszCL = PROTOCOL_readPascalString(szToNode, pszCL, 9);

	if( ! TOOLS_isCall(szToNode) )
		return PROTOCOL_BADCALL;

	/* Utilisateur */
	pszCL = PROTOCOL_readPascalString(szToUser, pszCL, 9);

	if( ! TOOLS_isCall(szToUser) )
		return PROTOCOL_BADCALL;

	/* Commande */
	pszCL = PROTOCOL_readPascalString(szCmd, pszCL, 200);

	/* Verifier que la longueur du protocole est OK */
	if( pszCL != pszEndProt )
		return PROTOCOL_BADSIZE;

	/* Interpreter le protocole */
	PROTOCOL_rCmdResponse(StreamNum, szFromNode, szToNode, szToUser, szCmd);

	return PROTOCOL_OK;
}

