#include <alloc.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <dos.h>
#include "buffers.h"
#include "define.h"
#include "protocol.h"
#include "clulink.h"
#include "users.h"
#include "set.h"
#include "log.h"
#include "switch.h"
#include "message.h"
#include "commands.h"
#include "fbbsrv.h"

/*

- BUFFERS 1  64 : les changes avec le SWITCH
- BUFFER  0      : fentre F1 uniquement
- BUFFER  65     : fentre F2 uniquement
*/

//--------------------------------------------------------------------------
//------ Variables globales ------------------------------------------------
//--------------------------------------------------------------------------
CHANNEL Stream[MAX_BUFFER+1]; /* Reserver 65 Structures Channel */

extern	char	STREAMS_level[MAX_STREAMS];
extern  char	PROTOCOL_linkType[65];

extern  tUserCfg * pUserCfg;

/*-------------------------------------------------------------------------
  ------ Initialiser les buffers ------------------------------------------
  -------------------------------------------------------------------------*/
void BUFFERS_init(void)
{
	memset(&Stream, 0, sizeof(Stream));
}

/*--------------------------------------------------------------------------
  ------ openbuff - ouvre un nouveau buffer pour le stream N "streamnum" --
  ------ et le charge avec une nouvelle chaine de caracteres ---------------
  --------------------------------------------------------------------------
  Reserve de la mmoire tampon
  Retourne DONE si OK
  Retourne ERR_NOT_ENOUGH_MEMORY si la memoire disponible est insuffisante  */
int BUFFERS_openBuff(int StreamNum, char * String, int Direction, int StrSize)
{

	Stream[StreamNum].pBuffer[Direction] = (char far*) farmalloc( (unsigned long) StrSize + 1L );

	/* Le buffer a t-il pu etre ouvert ? */
	if( ! Stream[StreamNum].pBuffer[Direction] )
		return ERR_NOT_ENOUGH_MEMORY;

	memcpy(Stream[StreamNum].pBuffer[Direction], String, StrSize);
	Stream[StreamNum].pBuffer[Direction][StrSize] = SNULL;
	Stream[StreamNum].nSize[Direction] = StrSize;

	return DONE;
}

/*--------------------------------------------------------------------------
  ------ addbuff - ajoute des donnees de longueur StrSize dans un buffer ---
  --------------------------------------------------------------------------
  Agrandit la mmoire Tampon
  Empile le nouveau string
  Retourne DONE si OK
  Retourne ERR_NOT_ENOUGH_MEMORY si la mmoire disponible est insuffisante  */
int BUFFERS_addBuff_sized(int StreamNum, char * String, int Direction, int StrSize)
{
	char far * pPtr;
	int	   nLastSize;

	if( StreamNum >= RCMD_VIRTSTREAM )
	{
		/* C'est pour un remote cmd - ecrire le contenu du buffer dans
		   le fichier tmp */
		FILE *	fPtr;

		fPtr = fopen(TEMP_RCMD_FILE, "a+t");
		if( ! fPtr )
		{
			/* Erreur d'ouverture du fichier */
			perror("fopen in BUFFERS_addBuff_sized (RCMD)");
			return FALSE;
		}

		fwrite(String, StrSize, 1, fPtr);
		fclose(fPtr);
		return TRUE;
	}

	/* Eviter les erreurs de bords */
	if( StreamNum < 0 || StreamNum > 65 )
		return FALSE;

#ifndef DOS
	/* Monitor sous linux */
	if( StreamNum >= 1 && StreamNum <= 64 )
		FBBSRV_monitor(StreamNum, String, StrSize, Direction);
#endif

	if( ! Stream[StreamNum].pBuffer[Direction] )
	{
		/* Buffer pas encore ouvert */
		if( BUFFERS_openBuff(StreamNum, String, Direction, StrSize) == DONE )
			return DONE;
		else
			return ERR_NOT_ENOUGH_MEMORY;
	}

	/* Calcule la nouvelle longueur des chaines */
	nLastSize = Stream[StreamNum].nSize[Direction];
	Stream[StreamNum].nSize[Direction] += StrSize;

#ifdef DOS
	/* Ne pas laisser farrealloc utiliser toute la memoire */
	if( coreleft() < 7500L )
		return ERR_NOT_ENOUGH_MEMORY;
#endif

	/* Reallouer un bloc */
	pPtr = (char far *) farrealloc( Stream[StreamNum].pBuffer[Direction], (unsigned long) Stream[StreamNum].nSize[Direction] + 1L);

	/* Si la reallocation a echouee, addbuff retourne la valeur NULL
	   et l'information a bufferiser est perdue */
	if( ! pPtr )
		return ERR_NOT_ENOUGH_MEMORY;

	Stream[StreamNum].pBuffer[Direction] = pPtr;

	/* Copie du bloc au nouvel emplacement */
	memcpy(Stream[StreamNum].pBuffer[Direction]+nLastSize, String, StrSize);
	Stream[StreamNum].pBuffer[Direction][nLastSize + StrSize] = SNULL;

	return DONE;
}

/*-------------------------------------------------------------------------
  ------ getRow - lit une ligne COMPLETE dans le buffer -------------------
  -------------------------------------------------------------------------
  Retourne 0 si le buffer est vide
  Retourne la longueur de la chaine sinon.
  La ligne est retournee sans le caractre de RC                           */
int BUFFERS_getRow(int StreamNum, char * String, int Direction)
{
	char far * 	pStrOrg;
	char far * 	pStrPos;
	unsigned int  	index;
	unsigned int  	nLen;
	int 		bCluLinkFrame = FALSE;
	int 		nStrSize = 0; /* Longueur de la chaine retournee */

	/* Initialisation de String */
	String[0] = SNULL;

	/* Est-ce une trame CluLink ? */
	if( (STREAMS_level[StreamNum] & LEVEL_cluster) && (PROTOCOL_linkType[StreamNum] == CLU_LINK) )
		bCluLinkFrame = TRUE;

	/* Le buffer est-il ouvert ? */
	if( ! Stream[StreamNum].pBuffer[Direction] )
	{
		/* Non */
		String[0] = SNULL;
		return 0;
	}

	if( bCluLinkFrame )
	{
		/* Le premier caractere indique la longueur du protocole
		   Ne pas lire si l'integralit de ce protocole n'a pas
		   ete reue */
		if( Stream[StreamNum].nSize[Direction] < CLULINK_getSize(Stream[StreamNum].pBuffer[Direction]) )
			return FALSE;

		/* Definir le nombre de caracteres a copier */
		nLen = CLULINK_getSize(Stream[StreamNum].pBuffer[Direction]) + 2;
	}
	else
	{
		/* Ne pas lire le buffer tant que la fin d'une ligne n'a pas
		   ete reue */
		if( ! strchr( Stream[StreamNum].pBuffer[Direction], '\r') )
			return FALSE;

		/* Definir le nombre de caracteres a copier (au maximum) */
		if( STREAMS_level[StreamNum] & LEVEL_cluster )
			nLen = 1023;
		else
			nLen = 255;
	}

	/*  Copier la ligne dans String */
	pStrOrg = Stream[StreamNum].pBuffer[Direction];
	pStrPos = String;

	for(index = 0; index < nLen && Stream[StreamNum].nSize[Direction] != 0; index++)
	{
		*pStrPos = *pStrOrg;
		Stream[StreamNum].nSize[Direction]--;
		nStrSize++;

		if( *pStrPos || bCluLinkFrame )
			pStrOrg++;

		if( (*pStrPos == SNULL || *pStrPos == '\r') && ! bCluLinkFrame )
		{
			*pStrPos = SNULL; /* Ne pas retourner le caractere \r */
			break;
		}
		else
			pStrPos++;
	}

	/* Ajouter un caractere SNULL  la fin de String, si ncessaire */
	if( *pStrPos != SNULL )
		*++pStrPos = SNULL;

	/* La fin du buffer est-elle atteinte ? */
	if( Stream[StreamNum].nSize[Direction] == 0 )
	{
		farfree( Stream[StreamNum].pBuffer[Direction] );
		Stream[StreamNum].pBuffer[Direction] = NULL;
		return nStrSize;
	}

	/* Sinon, reallouer le bloc avec la chaine restante dans le buffer */
	memmove(Stream[StreamNum].pBuffer[Direction], pStrOrg, Stream[StreamNum].nSize[Direction]);
	Stream[StreamNum].pBuffer[Direction][Stream[StreamNum].nSize[Direction]] = SNULL;
	Stream[StreamNum].pBuffer[Direction] = (char far *) farrealloc(Stream[StreamNum].pBuffer[Direction], (unsigned long) Stream[StreamNum].nSize[Direction] + 1L);

	return nStrSize;
}

/*-------------------------------------------------------------------------
  ------ getBloc - lit un bloc dans le buffer -----------------------------
  -------------------------------------------------------------------------
  Retourne 0 si le buffer est vide;
  Retourne la longueur de la chaine sinon.                                 */
int BUFFERS_getBloc(int StreamNum, char * String, int Direction)
{
	char far *	pStrOrg;
	char far *	pStrPos;
	unsigned int   	index;
	int		bCluLinkFrame = FALSE;
	int		nStrSize = 0;	//Longueur de la chaine retourne
	unsigned int	nPaclen = 255;
	int		nPage;

	/* Pagination */
	if( Direction == OUT && STREAMS_level[StreamNum] & LEVEL_user )
	{
		extern tUser*   pUser;
		USERS_getRecord(StreamNum);
		nPage = pUser->byPage;

		if( nPage != 0 && Stream[StreamNum].byPageCount >= nPage )
		{
			extern unsigned long COMMANDS_statut[MAX_STREAMS];

			if( COMMANDS_statut[StreamNum] & COMMANDS_statutPageWaitCR )
				return 0;

			COMMANDS_statut[StreamNum] |= COMMANDS_statutPageWaitCR;
			MSG_get(StreamNum, MSG80, String);
			if( String[0] ) /* Si buffer non vide */
				String[strlen(String) - 1] = SNULL;
			return strlen(String);
		}

	}
	else
		nPage = 0;


	/* Initialisation de String */
	String[0] = SNULL;

	/* Est-ce une trame CluLink ? */
	if( STREAMS_level[StreamNum] & LEVEL_cluster && PROTOCOL_linkType[StreamNum] == CLU_LINK )
		bCluLinkFrame = TRUE;

	/* Le buffer est-il ouvert ? */
	if( ! Stream[StreamNum].pBuffer[Direction] )
	{
		String[0] = SNULL;
		return 0; /* Retourne 0 (buffer vide) */
	}

#ifdef LINUX
	/* Gestion du paclen - pour linux uniquement */
	if( Direction == OUT && StreamNum >= 1 && StreamNum <= 64 )
		nPaclen = SWITCH_getPaclen(StreamNum);
#endif

	/* Copier la ligne dans String SANS jamais depasser 255 caractres */
	pStrOrg = Stream[StreamNum].pBuffer[Direction];
	pStrPos = String;

	for(index = 0; index < nPaclen && Stream[StreamNum].nSize[Direction] != 0; index++)
	{
		*pStrPos = *pStrOrg;
		Stream[StreamNum].nSize[Direction]--;
		nStrSize++;

		if( *pStrPos || bCluLinkFrame )
			pStrOrg++;

		/* Pagination */
		if( nPage != 0 && StreamNum > BUFFER_SCREEN1 && StreamNum <= BUFFER_SCREEN2 && StreamNum && *pStrPos == '\n' )
		{
			if( Stream[StreamNum].byPageCount++ >= nPage )
			{
				pStrPos++;
				break;
			}
		}

		if( ((*pStrPos == SNULL || *pStrPos == '\r') && ! bCluLinkFrame) || Stream[StreamNum].nSize[Direction] == 0 )
			break;
		else
			pStrPos++;
	}

	/* Ajouter un caractere SNULL  la fin de String, si necessaire */
	String[nStrSize] = SNULL;
	if( *pStrPos != SNULL )
		*++pStrPos = SNULL;

	/* La fin du buffer est-elle atteinte ? */
	if( Stream[StreamNum].nSize[Direction] == 0 )
	{
		Stream[StreamNum].byPageCount = 0;
		farfree(Stream[StreamNum].pBuffer[Direction] );
		Stream[StreamNum].pBuffer[Direction] = NULL;
		return nStrSize;
	}

	/* Sinon, reallouer le bloc avec la chaine restante dans le buffer */
	memmove(Stream[StreamNum].pBuffer[Direction], pStrOrg, Stream[StreamNum].nSize[Direction]);
	Stream[StreamNum].pBuffer[Direction] = (char far *) farrealloc(Stream[StreamNum].pBuffer[Direction], (unsigned long) Stream[StreamNum].nSize[Direction] + 1L);

	return nStrSize;
}
/*-------------------------------------------------------------------------
  ------ printfbuff - ajoute des donnees dans un buffer -------------------
  -------------------------------------------------------------------------*/
int BUFFERS_printBuff(int StreamNum, int Direction, char * String, ...)
{
	va_list     args;
	static char buf[256];
	int nReturnValue;

	va_start(args, String);
	vsprintf(buf, String, args);
	nReturnValue = BUFFERS_addBuff(StreamNum, buf, Direction);
	va_end(args);

	return nReturnValue;
}

/*-------------------------------------------------------------------------
  ------ addbuff - ajoute des donnees dans un buffer ----------------------
  -------------------------------------------------------------------------*/
int BUFFERS_addBuff(int StreamNum, char * String, int Direction)
{
	return BUFFERS_addBuff_sized(StreamNum, String, Direction, strlen(String));
}

/*-------------------------------------------------------------------------
  ------ Fermer un buffer -------------------------------------------------
  -------------------------------------------------------------------------*/
void BUFFERS_close(int StreamNum)
{
	int Direction;

	for(Direction = 0; Direction < 2; Direction++)
	{
		if( Stream[StreamNum].pBuffer[Direction] )
		{
			farfree( Stream[StreamNum].pBuffer[Direction] );
			Stream[StreamNum].pBuffer[Direction] = NULL;
			Stream[StreamNum].nSize[Direction]   = 0;
		}
	}

	Stream[StreamNum].byPageCount = 0;
}

/*************************************************************************
 * Les fonctions qui suivent sont utilisees pour gerrer des buffers      *
 * temporaires (pour les commandes UPDATE des databases par exemples)    *
 *************************************************************************/
/*-------------------------------------------------------------------------
  ------ Ajouter une chaine dans un buffer temporaire ---------------------
  -------------------------------------------------------------------------*/
void BUFFERS_addTemp(int StreamNum, char * String, int Size)
{
	int nOldBloc;
	int nNewBloc;

	/* Acceder a la config du stream */
	USERS_getRecord(StreamNum);

	/* Au premier appel, creer un nouveau buffer ... */
	if( pUserCfg->iBufferLength == 0 )
	{
		/* Un buffer etait-il deja alloue ? Si oui, le virer ... */
		if( pUserCfg->pBuffer )
			free( pUserCfg->pBuffer );

		pUserCfg->pBuffer = (char *) malloc(1024);
	}
	else
	{
		/* Faut-il augmenter la taille du buffer ? */
		/* Calcul du nombre de blocs necessaires pour l'ancien
		   buffer et le nouveau */
		nOldBloc = pUserCfg->iBufferLength / 1024;
		nNewBloc = (pUserCfg->iBufferLength + Size) / 1024;

		/* Faut-il augmenter la taille du buffer ? */
		if( nOldBloc != nNewBloc )
			pUserCfg->pBuffer = (char *) realloc(pUserCfg->pBuffer, (nNewBloc + 1) * 1024);
	}

	/* Copier le nouveau bloc */
	memmove(pUserCfg->pBuffer + pUserCfg->iBufferLength, String, Size);

	/* Mettre a jour et sauvegarder la config utilisateur */
	pUserCfg->iBufferLength += Size;
	USERS_updateLocalConfig(StreamNum);
}

/*-------------------------------------------------------------------------
  ------ Ajouter une chaine dans un buffer temporaire ---------------------
  -------------------------------------------------------------------------
  La fonction retourne une ligne lue dans le buffer et retourne la longueur
  de la chaine. Si le buffer est vide, la fonction retourne 0.
  Si le buffer n'est pas vide, un caractere terminal NULL est ajoute a
  la fin de la chaine retournee.                                           */
int BUFFERS_getTemp(int StreamNum, char * String, int MaxSize)
{
	int   nSize = 0;
	char* pScr;
	char* pDest;

	/* Acceder a la config du stream */
	USERS_getRecord(StreamNum);

	if( pUserCfg->iBufferLength == 0 )
		return 0; /* Buffer vide */

	/* Pointer sur le debut du buffer */
	pScr  = pUserCfg->pBuffer;
	pDest = String;

	/* La fonction ne retourne jamais de chaine suprieure a 255 octets */
	for(nSize = 0; nSize < MaxSize; nSize++)
	{
		*pDest = *pScr;

		pUserCfg->iBufferLength--;

		/* Fin de la ligne atteinte ? */
		if( *pScr == '\r' || pUserCfg->iBufferLength == 0 )
			break;

		pScr++;
		pDest++;

	}

	/* Mettre a jour et sauvegarder la config utilisateur */
	USERS_updateLocalConfig(StreamNum);

	/* Deplacer le reste du buffer au debut */
	memmove(pUserCfg->pBuffer, ++pScr, pUserCfg->iBufferLength);

	/* Ajouter un NULL a la chaine lue et retourner sa longueur */
	*++pDest = SNULL;
	return nSize;
}

/*-------------------------------------------------------------------------
  ------ Fermer (killer) un buffer temporaire -----------------------------
  -------------------------------------------------------------------------*/
void BUFFERS_killTemp(int StreamNum)
{
	/* Acceder a la config du stream */
	USERS_getRecord(StreamNum);

	if( pUserCfg->pBuffer )
		free( pUserCfg->pBuffer );

	pUserCfg->pBuffer = NULL;
	pUserCfg->iBufferLength = 0;

	/* Mettre a jour et sauvegarder la config utilisateur */
	USERS_updateLocalConfig(StreamNum);
}

/*-------------------------------------------------------------------------
  ------ Copier un buffer dans un fichier et fermer le buffer -------------
  -------------------------------------------------------------------------
  La fonction retourne TRUE si OK et FALSE sinon                           */
int BUFFERS_saveTemp(int StreamNum)
{
	FILE* fPtr;
	int   nLength;
	char  szBuffer[256];

	/* Acceder a la config du stream */
	USERS_getRecord(StreamNum);

	fPtr = fopen(pUserCfg->sFileName, "wt");
	if( ! fPtr )
	{
		perror("fopen in BUFFERS_saveTemp");
		return FALSE;
	}

	/* Copier dans le fichier */
	for(;;)
	{
		nLength = BUFFERS_getTemp(StreamNum, szBuffer, 255);

		if( nLength == 0 )
			break; /* Fin du buffer atteinte */

		/* Enregistrer la ligne dans le fichier */
		fwrite(szBuffer, nLength, 1, fPtr);
	}

	fclose (fPtr);

	/* Killer le buffer */
	BUFFERS_killTemp(StreamNum);

	return TRUE; /* OK */
}

/*-------------------------------------------------------------------------
  ------ Vider un buffer --------------------------------------------------
  -------------------------------------------------------------------------*/
void BUFFERS_empty(int StreamNum, int nDirection)
{
	if( Stream[StreamNum].pBuffer[nDirection] == NULL )
		return;

	farfree(Stream[StreamNum].pBuffer[nDirection]);
	Stream[StreamNum].pBuffer[nDirection] = NULL;
	Stream[StreamNum].nSize[nDirection]   = 0;
}
