(* ---------------------------------------------------------------
Title         Q&D Life
Author        PhG
Overview      Yet Another Version Of Conway's Life
Usage         see help
Notes         COMPACT model or more to avoid 64 Kb overflow
              assume normal color palette
              minimal check for QDLIFE.DAT
Bugs          M2 bug when cellsize=3 for box and fillmasktype is useless
Wish List     detect static condition (hash code of successive states of array ?)

--------------------------------------------------------------- *)

MODULE QDLife;

IMPORT Lib;
IMPORT MsMouse;
IMPORT Str;
IMPORT Graph;
IMPORT IO;
IMPORT SYSTEM;
IMPORT FIO;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetString, CharCount, same,
aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles,LtrimBlanks,RtrimBlanks,cleantabs;

FROM QD_rand IMPORT InitRnd, GetRnd, GetRndCardRange;

FROM IO IMPORT WrStr,WrLn;

(* ------------------------------------------------------------ *)

CONST
    gmode    = Graph._MRES256COLOR;
    colorEmpty= 0;
    minproba = 1;
    maxproba = 10;
    minsize  = 0;
    maxsize  = 32;
    mindelay = 0;
    maxdelay = 5000; (* was 64000 ! tss... *)
    minspeed    = 0;
    maxspeed    = 10;

    nopattern = 0;

    firstColor = 20H; (* with standard palette, only this range is useful *)
    lastColor  = 67H;
    pausevalue = 1000; (* in milliseconds, before restart *)
    maxWait    = 10; (* seconds *)
    maxWaitMS  = 1000*maxWait;
CONST
    cBLACK	       =0;
    cBLUE 	       =1;
    cGREEN         =2;
    cCYAN          =3;
    cRED           =4;
    cMAGENTA       =5;
    cBROWN         =6;
    cWHITE         =7;
    cGRAY          =8;
    cLIGHTBLUE     =9;
    cLIGHTGREEN    =10;
    cLIGHTCYAN     =11;
    cLIGHTRED      =12;
    cLIGHTMAGENTA  =13;
    cLIGHTYELLOW   =14;
    cBRIGHTWHITE   =15;
    minink         = cBLUE;
    maxink         = cBRIGHTWHITE;
CONST
    empty=000H;
    alive=080H;
CONST
    screenX = 320;
    screenY = 200;
    xmin = 0;
    xmax = screenX-1;
    ymin = 0;
    ymax = screenY-1;
CONST
    firstX = 0;
    firstY = 0;
(*
    maxLastX = (screenX DIV (minsize+1))-1;
    maxLastY = (screenY DIV (minsize+1))-1;
*)
    maxLastX = xmax;
    maxLastY = ymax;
VAR
    cell         : ARRAY [firstX..maxLastX],[firstY..maxLastY] OF SHORTCARD; (* BYTE *)
    survivalrule : str128;
    birthrule    : str128;
    rndbirthrule : str128;
    cellsize     : CARDINAL;
    box          : BOOLEAN;
    datfile      : str128;
VAR
    generation   : LONGCARD;
    lastX        : CARDINAL;
    lastY        : CARDINAL;
    currentCount : LONGCARD;
    currentColor : CARDINAL;
CONST
    space = " ";
    tab   = CHR(9);
    dash  = "-";
    coma  = ",";
    slash = "/";
    colon = ":";
    escape= CHR(27);
    special=CHR(0);
    star   = "*";
    extEXE = ".EXE"; (* upper case *)
    extDAT = ".DAT"; (* idem *)
CONST
    ProgEXEname   = "QDLIFE";
    ProgTitle     = "Q&D Life 320x200x256";
    ProgVersion   = "v1.1e";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errParameter    = 3;
    errConversion   = 4;
    errRange        = 5;
    errRule         = 6;
    errVGAneeded    = 7;
    errMouseNeeded  = 8;
    errCannotSet    = 9;
    errCannotReset  = 10;
    errNotFound     = 11;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    cr = CHR(13);
    lf = CHR(10);
    nl = cr+lf;
(*
00000000011111111112222222222333333333344444444445555555555666666666677777777778
1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msghelp = Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [option]..."+nl+
nl+
"  m:# mouse (0=no, 1=yes)"+nl+
"  w:# wrap (0=no, 1=yes)"+nl+
"  z:# speed ([0..10])"+nl+
"  p:# birth probability ([1..10])"+nl+
"  u:# rule random birth probability ([1..10])"+nl+
"  c:# cycle colors (0=yes, else [1..15])"+nl+
"  s:# cell size (0 or [1..16])"+nl+
"  b:# box (0=no, 1=yes)"+nl+
"  t:# initial delay in milliseconds ([0..5000])"+nl+
"  d:# refresh delay in milliseconds ([0..5000])"+nl+
"  g:# last generation before restart (0=unlimited)"+nl+
"  i:# initial pattern found in "+ProgEXEname+extDAT+" (0=random)"+nl+
"  r:S rule (Survival/Birth[/RandomBirth] : Conway's is 23/3)"+nl+
"      Try these : /2, 01245678/34, 245/368, 1358/357, 125/36,"+nl+
"      23/36, 238/357, 235678/378, 125678/367 and 45678/3"+nl+
"  i   show current parameters and wait until keypress or 10 seconds"+nl+
"  l   show list of available patterns"+nl+
nl+
'When running : " " = single step, "-" = slower, "+" = faster, "*" = init.'+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msghelp);
    | errOption :
        Str.Concat(S,"Unknown ",einfo);
        Str.Append(S," option");
    | errParameter :
        Str.Concat(S,"Unknown ",einfo);
        Str.Append(S," parameter");
    | errConversion :
        Str.Concat(S,"StrToCard would not convert ",einfo);
    | errRange :
        Str.Concat(S,"Illegal range for ",einfo);
    | errRule :
        Str.Concat(S,"Illegal rule in ",einfo);
    | errVGAneeded :
        S := "VGA card required";
    | errMouseNeeded :
        S := "Mouse hardware and software required";
    | errCannotSet :
        S := "Cannot set 320x200x256 video mode";
    | errCannotReset :
        S := "Cannot reset original video mode";
    | errNotFound :
        Str.Concat(S,"File ",einfo);
        Str.Append(S," not found");
    ELSE
        S := "How Can Such A Thing B(i)e(rce) ???";
    END;
    CASE e OF
    | errNone, errHelp :
        ;
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrStr(" !"); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBufferType  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn : ioBufferType;

(* ------------------------------------------------------------ *)

PROCEDURE WaitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END WaitVGAretrace;


PROCEDURE Set(x,y,color : CARDINAL);
VAR
    xpos,ypos,r:CARDINAL;
BEGIN
    IF cellsize=0 THEN Graph.Plot(x,y,color); RETURN; END;

    IF cellsize=1 THEN (* we have to because of crash if we let graphics library handle this size ! *)
        Graph.Plot(x<<1,y<<1,color); (* supposedly faster than *2 *)
        RETURN;
    END;

    CASE box OF
    | TRUE :
        xpos := x * (cellsize +1);
        ypos := y * (cellsize +1);
        Graph.Rectangle(xpos,ypos,xpos+cellsize-1,ypos+cellsize-1,color,TRUE);
        IF cellsize=3 THEN (* fix TopSpeed bug for cellsize=3 !!! *)
            Graph.Plot(xpos+1,ypos+1,color);
        END;
    | FALSE :
        r := (cellsize-1) DIV 2;
        xpos := x * (cellsize +1) + r;
        ypos := y * (cellsize +1) + r;
        Graph.Disc(xpos,ypos,r,color);
        IF r=1 THEN (* another TopSpeed bug for cellsize=3 or 4 !!! *)
            Graph.Plot(xpos,ypos,color);
        END;
    END;
END Set;

(* ------------------------------------------------------------ *)

PROCEDURE MakeAlive (x,y:CARDINAL);
BEGIN
    cell[x,y]:=alive;
END MakeAlive;

PROCEDURE MakeEmpty (x,y:CARDINAL);
BEGIN
    cell[x,y]:=empty;
END MakeEmpty;

PROCEDURE isAlive (x,y:CARDINAL) : BOOLEAN;
BEGIN
    IF cell[x,y] < alive THEN
        RETURN FALSE;
    ELSE
        RETURN TRUE;
    END;
END isAlive;

PROCEDURE Add (x,y:CARDINAL);
BEGIN
    INC (cell[x,y]);
END Add;

PROCEDURE wasEmpty (x,y:CARDINAL) : BOOLEAN;
BEGIN
    IF cell[x,y] < alive THEN
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
END wasEmpty;

PROCEDURE GetNeighbours (x,y:CARDINAL  ) : SHORTCARD;
VAR
    v : SHORTCARD;
BEGIN
    v := cell[x,y];
    IF cell[x,y] < alive THEN
        RETURN v;
    ELSE
        RETURN (v-alive);
    END;
END GetNeighbours;

(* ------------------------------------------------------------ *)

PROCEDURE initrandom (proba:CARDINAL);
VAR
    x,y:CARDINAL;
BEGIN
    FOR x := firstX TO lastX DO
        FOR y := firstY TO lastY DO
            IF GetRndCardRange(minproba,maxproba) <= proba THEN
                MakeAlive(x,y);
                INC(currentCount);
            END;
        END;
    END;
END initrandom;

CONST
    rem1 = "#";
    rem2 = ";";
    dot  = ".";

PROCEDURE initfile (initialpattern:CARDINAL);
VAR
    x,y:CARDINAL;
    hnd : FIO.File;
    S   : str128; (* should be enough *)
    i,len,rows,columns:CARDINAL;
    fpos:LONGCARD;
    state : (waiting,getting);
    c:CHAR;
BEGIN
    FIO.EOF:=FALSE;
    hnd:=FIO.OpenRead(datfile);
    FIO.AssignBuffer(hnd,bufferIn);
    i := 0;
    state := waiting;
    LOOP
        IF FIO.EOF=TRUE THEN EXIT; END;
        FIO.RdStr(hnd,S);
        Ltrim(S,space);
        Rtrim(S,space);
        Ltrim(S,tab);
        Rtrim(S,tab);
        CASE state OF
        | waiting :
            CASE S[0] OF
            | rem1,rem2,CHR(0) :
            ELSE
                state   :=getting;
                rows    :=0;
                columns :=0;
                fpos    :=FIO.GetPos(hnd);
(* WrStr("Name : ");WrStr(S);WrLn; *)
            END;
        | getting :
            CASE S[0] OF
            | CHR(0) :
                INC(i);
                IF i = initialpattern THEN EXIT; END;
                state := waiting;
            ELSE
                len:= Str.Length(S);
                IF len > columns THEN columns:=len; END;
                INC(rows);
            END;
        END;
    END;
    (* should check here for last line without CR but who cares ? *)
    IF state=waiting THEN
        currentCount:=0;
    ELSE
        rows:=rows DIV 2;
        columns:=columns DIV 2;
        x:=lastX DIV 2;
        y:=lastY DIV 2;
        IF rows > y THEN
            y:=firstY;
        ELSE
            y:=y - rows;
        END;
        IF columns > x THEN
            x:=firstX;
        ELSE
            x:=x - columns;
        END;
        FIO.Seek(hnd,fpos);
        LOOP
            IF FIO.EOF=TRUE THEN EXIT; END;
            FIO.RdStr(hnd,S);
            Ltrim(S,space);
            Rtrim(S,space);
            Ltrim(S,tab);
            Rtrim(S,tab);
(* WrStr("data : ");WrStr(S);WrLn; *)
            len := Str.Length(S);
            IF len = 0 THEN EXIT; END;
            FOR i := 0 TO len-1 DO
                 c:=S[i];
                 IF (c # space) AND (c # dot) THEN
                     IF (y<= lastY) AND ( (x+i) <= lastX) THEN
                         MakeAlive(x+i,y);
                         INC(currentCount);
                     END;
                 END;
            END;
            INC(y);
        END;
    END;
    FIO.Close(hnd);
END initfile;

PROCEDURE InitWorld (proba:CARDINAL; initialpattern:CARDINAL);
VAR
    x,y : CARDINAL;
BEGIN
    InitRnd();

    lastX := (screenX DIV (cellsize+1))-1;
    lastY := (screenY DIV (cellsize+1))-1;
    generation := 0;

    FOR x := firstX TO lastX DO
        FOR y := firstY TO lastY DO
           MakeEmpty(x,y);
        END;
    END;

    currentCount := 0;

    CASE initialpattern OF
    | 0 :
        initrandom(proba);
    ELSE
        initfile(initialpattern);
        IF currentCount = 0 THEN initrandom(proba); END;
    END;

END InitWorld;

(* ------------------------------------------------------------ *)

PROCEDURE ShowWorld ();
VAR
    i,x,y:CARDINAL;
BEGIN
    FOR x := firstX TO lastX DO
        FOR y := firstY TO lastY DO
            CASE isAlive(x,y) OF
            | FALSE :
                Set(x,y,colorEmpty);
            | TRUE :
                Set(x,y,currentColor);
            END;
        END;
    END;
END ShowWorld;

(* ------------------------------------------------------------ *)

PROCEDURE InitColor (cycle:CARDINAL);
(*
CONST
    blockfill=Graph.FillMaskType(0FFH,0FFH,0FFH,0FFH,0FFH,0FFH,0FFH,0FFH);
*)
BEGIN
    IF cycle = 0 THEN
        currentColor := firstColor;
    ELSE
        currentColor := cycle;
    END;
(*
    Graph.SetFillMask(blockfill);
*)
END InitColor;

PROCEDURE CycleColor (cycle:CARDINAL);
BEGIN
    IF cycle = 0 THEN
        INC(currentColor);
        IF currentColor = lastColor THEN currentColor:=firstColor; END;
    END;
END CycleColor;

(* ------------------------------------------------------------ *)

PROCEDURE NextGeneration (wrap:BOOLEAN;proba,speed:CARDINAL ) : LONGCARD;
CONST
    basechar="0";
VAR
    i,x,y:CARDINAL;
    content : SHORTCARD;
    changes : LONGCARD;
    c:CHAR;
BEGIN

    (* pass 1 : create # of neighbours for this cell *)

    FOR x := firstX+1 TO lastX-1 DO
        FOR y := firstY+1 TO lastY-1 DO
            IF isAlive(x,y) THEN
               Add (x-1,y-1);
               Add (x  ,y-1);
               Add (x+1,y-1);
               Add (x-1,y  );
               Add (x+1,y  );
               Add (x-1,y+1);
               Add (x  ,y+1);
               Add (x+1,y+1);
            END;
        END;
    END;

    (* north *)

    y := firstY;
    FOR x := firstX+1 TO lastX-1 DO
        IF isAlive(x,y) THEN
            IF wrap=TRUE THEN
                Add (x-1,lastY);
                Add (x  ,lastY);
                Add (x+1,lastY);
            END;
            Add (x-1,y  );
            Add (x+1,y  );
            Add (x-1,y+1);
            Add (x  ,y+1);
            Add (x+1,y+1);
        END;
    END;

    (* south *)

    y := lastY;
    FOR x := firstX+1 TO lastX-1 DO
        IF isAlive(x,y) THEN
            Add (x-1,y-1);
            Add (x  ,y-1);
            Add (x+1,y-1);
            Add (x-1,y  );
            Add (x+1,y  );
            IF wrap=TRUE THEN
                Add (x-1,firstY);
                Add (x  ,firstY);
                Add (x+1,firstY);
            END;
        END;
    END;

    (* west *)

    x := firstX;
    FOR y := firstY+1 TO lastY-1 DO
        IF isAlive(x,y) THEN
            IF wrap=TRUE THEN
                Add (lastX,y-1);
                Add (lastX,y  );
                Add (lastX,y+1);
            END;
            Add (x  ,y-1);
            Add (x  ,y+1);
            Add (x+1,y-1);
            Add (x+1,y  );
            Add (x+1,y+1);
        END;
    END;

    (* east *)

    x := lastX;
    FOR y := firstY+1 TO lastY-1 DO
        IF isAlive(x,y) THEN
            IF wrap=TRUE THEN
                Add (firstX,y-1);
                Add (firstX,y  );
                Add (firstX,y+1);
            END;
            Add (x  ,y-1);
            Add (x  ,y+1);
            Add (x-1,y-1);
            Add (x-1,y  );
            Add (x-1,y+1);
        END;
    END;

    (* nw *)

    x := firstX;
    y := firstY;
    IF isAlive(x,y) THEN
        IF wrap = TRUE THEN
            Add(lastX ,lastY );
            Add(x     ,lastY );
            Add(x+1   ,lastY );
            Add(lastX ,y     );
            Add(lastX ,y+1   );
        END;
        Add (x+1,y  );
        Add (x  ,y+1);
        Add (x+1,y+1);
    END;

    (* ne *)

    x := lastX;
    y := firstY;
    IF isAlive(x,y) THEN
        IF wrap = TRUE THEN
            Add(x-1   ,lastY );
            Add(x     ,lastY );
            Add(firstX,lastY );
            Add(firstX,y     );
            Add(firstX,y+1   );
        END;
        Add (x-1,y  );
        Add (x-1,y+1);
        Add (x  ,y+1);
    END;

    (* sw *)

    x := firstX;
    y := lastY;
    IF isAlive(x,y) THEN
        IF wrap = TRUE THEN
            Add(lastX ,y-1    );
            Add(lastX ,y      );
            Add(lastX ,firstY );
            Add(x     ,firstY );
            Add(x+1   ,firstY );
        END;
        Add (x  ,y-1);
        Add (x+1,y-1);
        Add (x+1,y  );
    END;

    (* se *)

    x := lastX;
    y := lastY;
    IF isAlive(x,y) THEN
        IF wrap = TRUE THEN
            Add(firstX,y-1   );
            Add(firstX,y     );
            Add(x-1   ,firstY);
            Add(x     ,firstY);
            Add(firstX,firstY);
        END;
        Add (x-1,y-1);
        Add (x  ,y-1);
        Add (x-1,y  );
    END;

    (* pass 2 : update display and reset array *)

    FOR i:=minspeed+1 TO speed DO WaitVGAretrace; END;

    changes := 0;
    FOR x := firstX TO lastX DO
        FOR y := firstY TO lastY DO
            CASE wasEmpty(x,y) OF
            | TRUE :
                c:=basechar+CHR(GetNeighbours(x,y));
                IF Str.CharPos(birthrule,c) # MAX(CARDINAL) THEN
                    MakeAlive(x,y);
                    Set(x,y,currentColor);
                    INC(currentCount);
                    INC(changes);
                ELSE

                    (* // v1.1a ... *)
                    IF Str.CharPos(rndbirthrule,c) # MAX(CARDINAL) THEN
                        IF GetRndCardRange(minproba,maxproba) <= proba THEN
                            MakeAlive(x,y);
                            Set(x,y,currentColor);
                            INC(currentCount);
                            INC(changes);
                        ELSE
                            MakeEmpty(x,y); (* reset neighbour counter *)
                        END;
                    ELSE
                    (* ... v1.1a // *)

                        MakeEmpty(x,y); (* reset neighbour counter *)
                    END;
                END;
            | FALSE :
                c:=basechar+CHR(GetNeighbours(x,y));
                IF Str.CharPos(survivalrule,c) # MAX(CARDINAL) THEN
                    MakeAlive(x,y); (* reset neighbour counter *)
                ELSE
                    MakeEmpty(x,y);
                    Set(x,y,colorEmpty);
                    DEC(currentCount);
                    INC(changes);
                END;
            END;
        END;
    END;

    INC(generation);

    RETURN changes;
END NextGeneration;

(* ------------------------------------------------------------ *)

PROCEDURE MouseClick (  ) : BOOLEAN;
VAR
    info : MsMouse.MsData;
BEGIN
    MsMouse.GetStatus(info);
    IF (info.left_pressed=TRUE) OR (info.right_pressed=TRUE) THEN
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
END MouseClick;

PROCEDURE isKeyWaiting (mouse:BOOLEAN) : BOOLEAN;
BEGIN
    IF mouse=TRUE THEN
        IF MouseClick()=TRUE THEN RETURN TRUE; END;
    END;
    RETURN IO.KeyPressed();
END isKeyWaiting;

PROCEDURE ReadNormalKey (mouse:BOOLEAN) : CHAR;
VAR
    ch : CHAR;
    c  : CHAR;
BEGIN
    IF mouse=TRUE THEN
        IF MouseClick()=TRUE THEN RETURN escape; END;
    END;
    ch := IO.RdKey();
    IF ch = special THEN c := IO.RdKey(); END;
    RETURN ch;
END ReadNormalKey;

(* ------------------------------------------------------------ *)

PROCEDURE HiresOn (  ) : BOOLEAN;
BEGIN
    RETURN Graph.SetVideoMode(gmode);
END HiresOn;

PROCEDURE TextOn (  ) : BOOLEAN;
BEGIN
    RETURN Graph.SetVideoMode(Graph._DEFAULTMODE);
END TextOn;

(* ------------------------------------------------------------ *)

PROCEDURE chkRule (VAR s:ARRAY OF CHAR) : BOOLEAN;
CONST
    legalrange="012345678";
VAR
    len,i:CARDINAL;
    p:CARDINAL;
BEGIN
    RtrimBlanks(s);
    LtrimBlanks(s);
    len:=Str.Length(s);
    IF len=0 THEN RETURN TRUE; END; (* allow empty ryle *)
    FOR i := 0 TO len-1 DO
        IF Str.CharPos(legalrange,s[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    RETURN TRUE;
END chkRule;

PROCEDURE ParseRule (rule : ARRAY OF CHAR) : BOOLEAN;
CONST
    separators = dash+coma+slash+colon;
BEGIN
    IF Str.Length(rule)=0 THEN RETURN FALSE; END;
    (* trick *)
    Str.Prepend(rule,space);
    Str.Append(rule,space);
    Str.ItemS(survivalrule,rule,separators,0);
    Str.ItemS(birthrule,rule,separators,1);
    Str.ItemS(rndbirthrule,rule,separators,2);
    IF chkRule(survivalrule)=FALSE THEN RETURN FALSE; END;
    IF chkRule(birthrule)=FALSE THEN RETURN FALSE; END;
    IF chkRule(rndbirthrule)=FALSE THEN RETURN FALSE; END;
    RETURN TRUE;
END ParseRule;

(* ------------------------------------------------------------ *)

PROCEDURE showlong (v:LONGCARD);
VAR
    S  : str128;
    ok : BOOLEAN;
BEGIN
    Str.CardToStr(v,S,10,ok);
    Ltrim(S,space);
    WrStr(S);
END showlong;

PROCEDURE showcard (v:CARDINAL);
VAR
    S  : str128;
    ok : BOOLEAN;
BEGIN
    Str.CardToStr(LONGCARD(v),S,10,ok);
    Ltrim(S,space);
    WrStr(S);
END showcard;

PROCEDURE showbool (flag:BOOLEAN);
VAR
    S : str128;
BEGIN
    IF flag=TRUE THEN
        S:="Enabled";
    ELSE
        S:="Disabled";
    END;
    WrStr(S);
END showbool;

PROCEDURE displayparms (mouse,wrap:BOOLEAN;
                        pBirth,cycle,cellsize:CARDINAL;
                        box:BOOLEAN;
                        rule:ARRAY OF CHAR;
                        initialdelay,refreshdelay:CARDINAL;
                        lastgeneration:LONGCARD;
                        initialpattern:CARDINAL;
                        rndPbirth:CARDINAL);
VAR
    S : str128;
BEGIN
    WrStr(Banner); WrLn;
    WrLn;
    WrStr("Size of cells in pixels : "); showcard(cellsize); WrLn;
    WrStr("Survival/Birth[/Random] : "); WrStr(rule); WrLn;
    WrStr("Initial pause           : ");
    IF initialdelay=0 THEN
        WrStr("None");
    ELSE
        showcard(initialdelay);
    END;
    WrLn;
    WrStr("Refresh pause           : ");
    IF refreshdelay=0 THEN
        WrStr("None");
    ELSE
        showcard(refreshdelay);
    END;
    WrLn;
    WrStr("Last generation         : ");
    IF lastgeneration = 0 THEN
        WrStr("Unlimited");
    ELSE
        showlong(lastgeneration);
    END;
    WrLn;

    WrStr("Initial pattern         : ");
    IF initialpattern = 0 THEN
        WrStr("Random");
    ELSE
        showcard(initialpattern);
    END;
    WrLn;
    WrStr("Probability of birth    : "); showcard(pBirth); WrLn;
    WrStr("Probability of RNDbirth : "); showcard(rndPbirth); WrLn;

    WrStr   ("Shapes                  : ");
    IF box=TRUE THEN
        S:="Boxes";
    ELSE
        S:="Disks";
    END;
    WrStr(S);
    WrLn;
    WrStr   ("Color cycling           : ");
    IF cycle=0 THEN
        WrStr("Enabled");
    ELSE
        WrStr("Disabled, color is "); showcard(cycle);
    END;
    WrLn;
    WrStr("Abort on mouse click    : "); showbool(mouse); WrLn;
    WrStr("Wrap coordinates        : "); showbool(wrap); WrLn;
END displayparms;

PROCEDURE displaylist ();
CONST
    wi  = 5;
    sep = " : ";
VAR
    hnd : FIO.File;
    S   : str128; (* should be enough *)
    state : (waiting,skipping);
    currpattern:CARDINAL;
BEGIN
    WrStr("index"+sep+"pattern");WrLn;
    WrLn;

    currpattern:=0;

    FIO.EOF:=FALSE;
    hnd:=FIO.OpenRead(datfile);
    FIO.AssignBuffer(hnd,bufferIn);
    state := waiting;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hnd,S);
        Ltrim(S,space);Rtrim(S,space);
        Ltrim(S,tab);  Rtrim(S,tab);
        CASE state OF
        | waiting :
            CASE S[0] OF
            | rem1,rem2,CHR(0) :
                ;
            ELSE
                INC(currpattern);
                IO.WrCard(currpattern,wi);WrStr(sep);WrStr(S);WrLn;
                state   :=skipping;
            END;
        | skipping :
            IF S[0] = CHR(0) THEN state := waiting; END;
        END;
    END;
    FIO.Close(hnd);
END displaylist;

PROCEDURE newspeed ( VAR speed:CARDINAL; minspeed,maxspeed:CARDINAL; motion:INTEGER );
BEGIN
    IF motion < 0 THEN
        IF speed > minspeed THEN DEC(speed);END;
    ELSE
        IF (speed+1) <= maxspeed THEN INC(speed);END;
    END;
END newspeed;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    mouse  : BOOLEAN;
    wrap   : BOOLEAN;
    pBirth : CARDINAL;
    rndPbirth : CARDINAL;
    cycle  : CARDINAL;
    (* cellsize:CARDINAL; *)
    (* box    : BOOLEAN; *)
    initialdelay : CARDINAL;
    refreshdelay : CARDINAL;
    speed        : CARDINAL;
    lastgeneration : LONGCARD;
    initialpattern : CARDINAL;
    rule   : str128;
    cmd    : (show,list);
    (* datfile: str128; *)
VAR
    singlestep : BOOLEAN;
    c          : CHAR;
    showparms  : BOOLEAN;
    anykey     : str2;
VAR
    parmcount : CARDINAL;
    i         : CARDINAL;
    opt       : CARDINAL;
    S         : str128;
    R         : str128;
    n         : LONGCARD;
    infovideo : Graph.VideoConfig;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn; (* must be here *)

    mouse          := FALSE;
    wrap           := TRUE;
    pBirth         := 2;
    rndPbirth      := 7;
    cycle          := 0;
    cellsize       := 2;
    box            := TRUE;
    initialdelay   := mindelay;
    refreshdelay   := mindelay;
    lastgeneration := 0;
    initialpattern := nopattern; (* random *)
    rule           := "23/3";
    speed          := minspeed;
    cmd            := show;

    singlestep := FALSE;
    showparms := FALSE;

    IF ParseRule(rule)=FALSE THEN abort(errRule,S); END; (* cannot happen here *)

    parmcount := Lib.ParamCount();
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S); cleantabs(R);
        UpperCase(R);
        IF isOption(R)=FALSE THEN Str.Prepend(R,dash); END; (* trick ! *)

            opt := GetOptIndex(R,"?"+delim+"H"+delim+"HELP"+delim+
                                 "M:"+delim+"MOUSE:"+delim+
                                 "W:"+delim+"WRAP:"+delim+
                                 "P:"+delim+"PROBA:"+delim+
                                 "C:"+delim+"CYCLE:"+delim+
                                 "S:"+delim+"SIZE:"+delim+
                                 "B:"+delim+"BOX:"+delim+
                                 "R:"+delim+"RULE:"+delim+
                                 "I"+delim+"INFO"+delim+
                                 "D:"+delim+"DELAY:"+delim+
                                 "G:"+delim+"GENERATION:"+delim+
                                 "I:"+delim+"INITIAL:"+delim+
                                 "U:"+delim+"RNDPROBA:"+delim+
                                 "L"+delim+"LIST"+delim+
                                 "T:"+delim+"TEMPO:"+delim+
                                 "Z:"+delim+"SPEED:"
                              );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5 :   IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | 0 : mouse := FALSE;
                      | 1 : mouse := TRUE;
                      ELSE
                         abort(errRange,S);
                      END;
            | 6,7 :
                      IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | 0 : wrap := FALSE;
                      | 1 : wrap := TRUE;
                      ELSE
                         abort(errRange,S);
                      END;
            | 8,9 :   IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | minproba..maxproba :  pBirth:= CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 10,11 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | 0 : cycle := 0;
                      | minink..maxink : cycle := CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 12,13 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | minsize..maxsize : cellsize := CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 14,15 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | 0 : box := FALSE;
                      | 1 : box := TRUE;
                      ELSE
                          abort(errRange,S);
                      END;
            | 16,17 : GetString(S,rule);
                      IF ParseRule(rule)=FALSE THEN abort(errRule,S); END;
            | 18,19 : showparms := TRUE;
            | 20,21 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | mindelay..maxdelay : refreshdelay := CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 22,23 : IF GetLongCard(S,lastgeneration)=FALSE THEN abort(errConversion,S);END;
            | 24,25 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      initialpattern := CARDINAL(n);
            | 26,27:  IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | minproba..maxproba : rndPbirth:= CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 28,29:  cmd:=list;
            | 30,31 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | mindelay..maxdelay : initialdelay := CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            | 32,33 : IF GetLongCard(S,n)=FALSE THEN abort(errConversion,S);END;
                      CASE CARDINAL(n) OF
                      | minspeed..maxspeed : speed := CARDINAL(n);
                      ELSE
                          abort(errRange,S);
                      END;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
    END;

CASE cmd OF
| list:
    Lib.ParamStr(datfile,0);
    Str.Caps(datfile); (* useless ! *)
    Str.Subst(datfile,extEXE,extDAT);
    IF FIO.Exists(datfile)=FALSE THEN abort (errNotFound,datfile); END;
    displaylist();
    abort(errNone,"");
ELSE
    IF initialpattern # 0 THEN
        Lib.ParamStr(datfile,0);
        Str.Caps(datfile); (* useless ! *)
        Str.Subst(datfile,extEXE,extDAT);
        IF FIO.Exists(datfile)=FALSE THEN abort (errNotFound,datfile); END;
    END;

    Graph.GetVideoConfig(infovideo);
    IF infovideo.adapter # Graph._VGA THEN abort(errVGAneeded,""); END;

    IF mouse=TRUE THEN
        IF MsMouse.Reset()=MAX(INTEGER) THEN abort(errMouseNeeded,""); END;
    END;

    IF showparms=TRUE THEN
        displayparms (mouse,wrap,pBirth,cycle,cellsize,box,rule,
                     initialdelay,refreshdelay,lastgeneration,
                     initialpattern,rndPbirth);
        WrLn;
        WrStr("Please hit ESCAPE to abort, or almost any key to continue... ");
        (* anykey:=Waitkey(); (* do not use WaitkeyDelay here after all... NOT ! ;-) *) *)
        anykey := WaitkeyDelay("",maxWaitMS);
        WrLn;
    END;
    IF same(anykey,escape)=TRUE THEN abort(errNone,""); END;

    IF HiresOn()=FALSE THEN abort(errCannotSet,""); END;

    InitWorld(pBirth,initialpattern);
    InitColor(cycle);
    ShowWorld();

    IF initialdelay # 0 THEN Lib.Delay(initialdelay); END; (* useless : 0 means no delay anyway *)

    LOOP

        CycleColor(cycle);

        CASE singlestep OF
        | TRUE :
            WHILE isKeyWaiting(mouse)=FALSE DO
            END;
            c := ReadNormalKey(mouse);
            CASE c OF
            | escape :
                EXIT;
            | space :
                ;
            | star :
                Graph.ClearScreen(Graph._GCLEARSCREEN);
                InitWorld(pBirth,initialpattern);
                InitColor(cycle);
                ShowWorld();
                singlestep := FALSE;
            | "-" :
                newspeed(speed,minspeed,maxspeed,1);  singlestep:=FALSE;
            | "+" :
                newspeed(speed,minspeed,maxspeed,-1); singlestep:=FALSE;
            ELSE
                singlestep := FALSE;
            END;
        | FALSE :
            IF isKeyWaiting(mouse) THEN
                c := ReadNormalKey(mouse);
                CASE c OF
                | space :
                    singlestep := TRUE;
                | escape :
                    EXIT;
                | star :
                    Graph.ClearScreen(Graph._GCLEARSCREEN);
                    InitWorld(pBirth,initialpattern);
                    InitColor(cycle);
                    ShowWorld();
                | "-" : newspeed(speed,minspeed,maxspeed,1);
                | "+" : newspeed(speed,minspeed,maxspeed,-1);
                ELSE
                    EXIT;
                END;
            END;
        END;

        (* WaitVGAretrace(); *)
        n:=NextGeneration(wrap,rndPbirth,speed);

        IF (currentCount = 0) OR (n=0) THEN
            (* c := ReadNormalKey(mouse);
            EXIT; *)
            (* restart after 2 seconds ! *)
            Lib.Delay(pausevalue);
            Graph.ClearScreen(Graph._GCLEARSCREEN);
            InitWorld(pBirth,initialpattern);
            InitColor(cycle);
            ShowWorld();
        END;

        IF refreshdelay # 0 THEN Lib.Delay(refreshdelay); END; (* useless : 0 means no delay anyway *)
        (* WaitVGAretrace(); *)

        IF (lastgeneration #0) AND (lastgeneration < generation) THEN
            (* restart after 2 seconds ! *)
            Lib.Delay(pausevalue);
            Graph.ClearScreen(Graph._GCLEARSCREEN);
            InitWorld(pBirth,initialpattern);
            InitColor(cycle);
            ShowWorld();
        END;

    END;

    IF TextOn()=FALSE THEN abort(errCannotReset,""); END;
(*
    IF c = escape THEN
        WrStr("The Wrath of GOD has striken !");
        WrLn;
    END;
*)
    abort(errNone,"");
END;

END QDLife.


(*
                    IF (c # space) AND (c # dot) THEN
                        IF y <= lastY THEN
                            IF x+i <= lastX THEN
                               MakeAlive(x+i,y);
                               INC(currentCount);
                            ELSE
                               IF wrap=TRUE THEN
                                   MakeAlive(x+i-lastX+firstX-1 ,y); (* $$$ *)
                                   INC(currentCount);
                               END;
                            END;
                        END;
                    END;
                END;
                INC(y);
                IF wrap = TRUE THEN
                    IF y > lastY THEN y := firstY; END;
                END;
            END;
        ELSE
            INC(y); (* empty line, so advance *)
            IF wrap = TRUE THEN
                IF y > lastY THEN y := firstY; END;
            END;
        END;
    END;
    FIO.Close(hnd);
END InitWordFile;


*)

