
(* ---------------------------------------------------------------
Title         Q&D FON <-> TXT Font Tool
Overview      self-explanatory !
Notes         
Bugs
Wish List

--------------------------------------------------------------- *)

MODULE FontTool;

IMPORT Lib;
IMPORT Str;
IMPORT IO;
IMPORT FIO;
IMPORT SYSTEM;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim,
completedInit, completedShow, completedSHOW, completedEnd, completed;

FROM IO IMPORT WrStr, WrLn;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "FONTTOOL";
    ProgTitle     = "Q&D FON <-> TXT Font Tool";
    ProgVersion   = "v1.0c";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
    cr            = CHR(13);
    lf            = CHR(10);
    nl            = cr+lf;
    dot           = ".";
    blank         = " "; (* space *)
    doublequote   = '"';
    singlequote   = "'";
    semicolon     = ";";
CONST
    extFON        = ".FON";
    extDAT        = ".DAT";

(* ------------------------------------------------------------ *)

CONST
    errNone               = 0;
    errHelp               = 1;
    errTooManyParms       = 2;
    errOption             = 3;
    errNonsense           = 4;
    errSyntax             = 5;
    errNotFound           = 6;
    errTooMany            = 7;
    errBadFont            = 8;
    errExists             = 9;
    errBadExt             = 10;
    errEither             = 11;
    errNotSpecific        = 12;
    errBuildOnly          = 13;
    errIrrelevant         = 14;
    errJokers             = 15;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" <-build> <binary_font["+extFON+"]> [-p<2|5>] [-a] [-o] [-v]"+nl+
"Syntax 2 : "+ProgEXEname+" <-rebuild> <text_definition["+extDAT+"]> [-p] [-o] [-v]"+nl+
"Syntax 3 : "+ProgEXEname+" <-squash> <binary_font["+extFON+"]> <target> [-m#]"+nl+
nl+
"    -p  allow partial set of characters (-p2=25 lines, -p5=50 lines)"+nl+
"    -a  automatically adjust font height if different from 8 or 16"+nl+
"    -m# squash method [1..6]"+nl+
"    -o  overwrite"+nl+
"    -v  verbose"+nl;
VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errTooManyParms :
        Str.Concat(S,"Uneeded ",einfo);Str.Append(S," parameter !");
    | errOption :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," option !");
    | errNonsense:
        Str.Concat(S,einfo," are mutually exclusive !");
    | errSyntax:
        S := "Syntax error in command line !";
    | errNotFound :
        Str.Concat(S,"No file matches ",einfo);Str.Append(S," specification !");
    | errTooMany:
        Str.Concat(S,"Too many files match ",einfo);Str.Append(S," specification !");
    | errBadFont :
        Str.Concat(S,einfo," font file should be 4096 or 2048 bytes long !");
    | errExists :
        Str.Concat(S,einfo," target already exists !");
    | errBadExt:
        Str.Concat(S,"Reserved target extension in ",einfo);
        Str.Append(S," source specification !");
    | errEither:
        S := "-p2 and -p5 options are mutually exclusive !";
    | errNotSpecific:
        S := "-p option is not specific enough here !";
    | errBuildOnly:
        S := "-p2 and -p5 options apply to -build command only !";
    | errIrrelevant:
        S := "-p option is irrevelant here !";
    | errJokers:
        Str.Concat(S,'Illegal joker(s) in "',einfo);Str.Append(S,'" !');
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp : ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
VAR
    ioBuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)

CONST
    charsize50       = 8;
    charsize25       = 16;
    fontsize50       = 2048; (* 8 *)
    fontsize25       = 4096; (* 16 *)
    fontsizemax      = fontsize25;
    firstdata        = 0;
    lastdata         = fontsize25-1; (* biggest *)
VAR
    fontdata : ARRAY [firstdata..lastdata] OF BYTE;

PROCEDURE initFontBuffer (  );
VAR
    i:CARDINAL;
BEGIN
    FOR i:=firstdata TO lastdata DO
        fontdata[i]:=BYTE(00H);
    END;
END initFontBuffer;

PROCEDURE chkFont (relaxed,autoheight:BOOLEAN; S:ARRAY OF CHAR):BOOLEAN;
VAR
    fsize:LONGCARD;
BEGIN
    fsize:=getFileSize(S);
    IF (relaxed OR autoheight) THEN RETURN (fsize <= fontsizemax); END;
    IF fsize=fontsize25 THEN RETURN TRUE; END;
    IF fsize=fontsize50 THEN RETURN TRUE; END;
    RETURN FALSE;
END chkFont;

(* ------------------------------------------------------------ *)

TYPE
    f8e3 = ARRAY [0..8+1+3-1] OF CHAR;
CONST
    firstfile = 1;
    maxfile   = 1000;
VAR
    fileArray : ARRAY [firstfile..maxfile] OF f8e3;

PROCEDURE buildFileList (S:ARRAY OF CHAR):CARDINAL;
VAR
    found     : BOOLEAN;
    direntry  : FIO.DirEntry;
    countFile : CARDINAL;
BEGIN
    countFile := 0;
    found := FIO.ReadFirstEntry(S,allfiles,direntry);
    WHILE found DO
        IF countFile = (maxfile-firstfile+1) THEN RETURN MAX(CARDINAL); END;
        fileArray[firstfile+countFile]:=f8e3(direntry.Name);
        Str.Caps(fileArray[firstfile+countFile]); (* useless ! *)
        INC (countFile);
        found :=FIO.ReadNextEntry(direntry);
    END;
    RETURN countFile;
END buildFileList;

PROCEDURE makebase (spec:ARRAY OF CHAR;VAR basepath:ARRAY OF CHAR );
CONST
    defaultdrive = SHORTCARD(0);
VAR
    u,d,n,e:str128;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Str.Concat(basepath,u,d);
    IF same(basepath,"") THEN
        Str.Concat(basepath, CHAR( ORD("A")-1+ FIO.GetDrive()), ":");
        FIO.GetDir(defaultdrive,d);
        Str.Append(basepath,d);
        fixDirectory(basepath); (* in case *)
    END;
END makebase;

PROCEDURE defaultExtension (ext:ARRAY OF CHAR;VAR R:ARRAY OF CHAR);
BEGIN
    IF Str.CharPos(R,dot)=MAX(CARDINAL) THEN
        Str.Append(R,ext);
    END;
END defaultExtension;

PROCEDURE makeTarget (S,ext:ARRAY OF CHAR;VAR R:ARRAY OF CHAR);
VAR
    u,d,n,e:str128;
BEGIN
    Lib.SplitAllPath(S,u,d,n,e);
    Lib.MakeAllPath(R,u,d,n,ext);
END makeTarget;

PROCEDURE padval (v : CARDINAL; digits,base : CARDINAL;padchar:CHAR) : str16;
VAR
    S,R,padStr : str16;
    ok : BOOLEAN;
    len,delta,i : CARDINAL;
BEGIN
    FOR i:=0 TO 15 DO
        padStr[i]:=padchar;
    END;
    digits := digits MOD 16; (* better safe than sorry! *)
    Str.CardToStr (LONGCARD(v),S,base,ok);
    len := Str.Length(S);
    IF len > digits THEN digits := len; END;
    delta := digits - len;
    Str.Slice (R,padStr,0,delta);
    Str.Append (R,S);
    RETURN R;
END padval;

PROCEDURE Filter (c : CHAR) : str16;
VAR
    R : str16;
BEGIN
    CASE ORD(c) OF
    | 0..ORD(blank)-1  : R:="'?'";
    | ORD(doublequote) : R:=singlequote+doublequote+singlequote;
    | 255              : R:="'?'";
    ELSE
                         Str.Concat(R,doublequote,c);Str.Append(R,doublequote);
    END;
    RETURN R;
END Filter;

PROCEDURE binstrToVal (S:ARRAY OF CHAR):BYTE;
VAR
    i:CARDINAL;
    ok:BOOLEAN;
BEGIN
    FOR i:=1 TO Str.Length(S) DO
        CASE S[i-1] OF
        | blank :
            S[i-1]:="0";
        ELSE
            S[i-1]:="1";
        END;
    END;
    RETURN BYTE( Str.StrToCard(S,2,ok) );
END binstrToVal;

(* ------------------------------------------------------------ *)

PROCEDURE doBuild (force25,force50,verbose,autoheight:BOOLEAN;
                   infile:ARRAY OF CHAR;
                   VAR outfile:ARRAY OF CHAR):BOOLEAN;
VAR
    S:str128;
    hnd:FIO.File;
    fsize,got,i,j,p,charsize,v,targetcharsize:CARDINAL;
BEGIN
    IF verbose THEN Work(cmdInit);END;

    fsize := CARDINAL( getFileSize(infile) ); (* buffer/legal size already checked *)
    hnd:=FIO.OpenRead(infile);
    FIO.AssignBuffer(hnd,ioBuffer);
    got:= FIO.RdBin(hnd,fontdata,fsize);
    FIO.Close(hnd);

    IF force25 THEN
        fsize := fontsize25;
    ELSIF force50 THEN
        fsize := fontsize50;
    END;
    charsize := fsize DIV 256;

    IF autoheight THEN
        IF charsize > charsize50 THEN
            targetcharsize := charsize25;
        ELSE
            targetcharsize := charsize50;
        END;
    END;

    makeTarget(infile,extDAT,outfile);
    hnd:=FIO.Create(outfile);
    FIO.AssignBuffer(hnd,ioBuffer);

    Str.Concat(S,"; Font file : ",infile);
    FIO.WrStr(hnd,S);FIO.WrLn(hnd);
    FOR i:=0 TO 255 DO
        IF verbose THEN Work(cmdSHOW);END;
        FIO.WrLn(hnd);
        FIO.WrStr(hnd,";");FIO.WrLn(hnd);

        Str.Concat (S,"; ::: ",padval(i,3,10," "));
        Str.Append (S,"  $");Str.Append(S,padval(i,2,16,"0"));
        Str.Append (S,"  ");Str.Append(S,Filter(CHR(i)));
        FIO.WrStr(hnd,S);FIO.WrLn(hnd);

        FIO.WrStr(hnd,";");FIO.WrLn(hnd);
        FIO.WrLn(hnd);

        p := i * charsize;
        FOR j:=1 TO charsize DO
            v := CARDINAL( fontdata[p+j-1] );
            Str.Concat(S,doublequote,padval(v,8,2,"0"));Str.Append(S,doublequote);
            ReplaceChar (S,"0"," ");
            ReplaceChar (S,"1","*");
            FIO.WrStr(hnd,S);FIO.WrLn(hnd);
        END;
        (* ugly, ugly : call the Code Police ! *)
        FOR j:=(charsize+1) TO targetcharsize DO
            v := 0;
            Str.Concat(S,doublequote,padval(v,8,2,"0"));Str.Append(S,doublequote);
            ReplaceChar (S,"0"," ");
            ReplaceChar (S,"1","*");
            FIO.WrStr(hnd,S);FIO.WrLn(hnd);
        END;
    END;
    FIO.Flush(hnd);
    FIO.Close(hnd);
    IF verbose THEN Work(cmdStop);END;
    RETURN TRUE;
END doBuild;

(* ------------------------------------------------------------ *)

PROCEDURE doRebuild (relaxed,verbose:BOOLEAN; infile:ARRAY OF CHAR;
                     VAR outfile:ARRAY OF CHAR):BOOLEAN;
VAR
    S:str128;
    p,fsize,currline,len:CARDINAL;
    hnd:FIO.File;
    ok:BOOLEAN;
BEGIN
    IF verbose THEN Work(cmdInit);END;

    hnd:=FIO.OpenRead(infile);
    FIO.AssignBuffer(hnd,ioBuffer);
    currline:=0;
    p:=0;
    LOOP
        IF verbose THEN Work(cmdSHOW);END;
        FIO.RdStr(hnd,S);
        INC(currline);
        LtrimBlanks(S);
        RtrimBlanks(S);
        ok:=FALSE;
        CASE S[0] OF
        | CHR(0)      : ok:=TRUE;   (* empty string *)
        | semicolon   : ok:=TRUE;   (* comment *)
        | doublequote :
            IF CharCount(S,doublequote) = 2 THEN
                len:=Str.Length(S);
                IF S[len-1]=S[0] THEN
                    S[len-1]:=0C;
                    Str.Delete(S,0,1);
                    IF Str.Length(S) = 8 THEN
                        fontdata[p]:=binstrToVal(S);
                        ok:=TRUE;
                        INC(p);
                        IF p > lastdata THEN EXIT; END; (* force here *)
                    END;
                END;
            END;
        END;
        IF FIO.EOF THEN ok:=TRUE; EXIT; END; (* legitimate exit ! *)
        IF ok=FALSE THEN EXIT; END;
    END;
    FIO.Close(hnd);
    IF verbose THEN Work(cmdStop);END;

    IF ok THEN
        CASE p OF
        | fontsize25,fontsize50:
            fsize:=p;
        ELSE
            IF relaxed THEN
                fsize:=p;
                ok:=TRUE;
            ELSE
                ok:=FALSE;
            END;
        END;
    END;
    IF ok=FALSE THEN
        Str.Copy(outfile,padval(currline,4,10,"0"));
        RETURN FALSE;
    END;

    makeTarget(infile,extFON,outfile);
    hnd:=FIO.Create(outfile);
    FIO.AssignBuffer(hnd,ioBuffer);
    FIO.WrBin(hnd,fontdata,fsize);
    FIO.Flush(hnd);
    FIO.Close(hnd);
    RETURN TRUE;
END doRebuild;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    method1 = 1;
    method2 = 2;
    method3 = 3;
    method4 = 4;
    method5 = 5;
    method6 = 6;
TYPE
    FontDef      = ARRAY [firstdata..lastdata] OF BYTE;
VAR
    gFont25 : FontDef;

(* see 25to50.bas and no, won't bother to write a cleaner bit-manipulating version ! *)

PROCEDURE squash (squashmethod:CARDINAL; VAR fontbuff:FontDef);
TYPE
    ktype     = ARRAY [1..3] OF CARDINAL;
    matrixtype= ARRAY [1..4] OF ktype;
CONST
    transfo1 = matrixtype ( ktype(0,1,0),
                            ktype(1,2,1),
                            ktype(1,2,1),
                            ktype(0,1,0));
    divisor1 = 2+2+1+1+1+1+1+1-1;
    transfo2 = matrixtype ( ktype(0,0,0),
                            ktype(0,2,0),
                            ktype(0,2,0),
                            ktype(0,0,0));
    divisor2 = 2+2-1;
    transfo3 = matrixtype ( ktype(0,0,0),
                            ktype(0,2,0),
                            ktype(0,2,0),
                            ktype(0,0,0));
    divisor3 = 2+2-1-1;
    transfo4 = matrixtype ( ktype(1,0,1),
                            ktype(0,2,0),
                            ktype(0,2,0),
                            ktype(1,0,1));
    divisor4 = 1+1+2+2+1+1-1;
TYPE
    str8 = ARRAY [1..8] OF CHAR;
CONST
    blanc = "00000000"; (* 8 blanks *)
    combi1= "0"+delim+"1";
    combi2= "00"+delim+"10"+delim+"01"+delim+"11";
VAR
    old : ARRAY [1..16] OF str8;
    now : ARRAY [1..8] OF str8;
    R,a,b : str8;
    ca,cb,c:CHAR;
    sa,sb,sc:str2;
    psrc,pto,n,i,w,j,y,x : CARDINAL;
    v:BYTE;
    lv:LONGCARD;
    ok:BOOLEAN;
    ii,jj,sigma,k,bval,divisor:CARDINAL;
    transfo:matrixtype;
BEGIN
    psrc := firstdata;
    pto  := firstdata;
    FOR n := 0 TO 255 DO
        FOR i:=1 TO 16 DO
            v:=fontbuff[psrc];
            Str.CardToStr( LONGCARD(v),R,2,ok);
            FOR w:=Str.Length(R)+1 TO 8 DO Str.Prepend(R,"0");END;
            old[i]:=R;
            INC(psrc);
        END;
        FOR i:=1 TO 8 DO
            now[i]:=blanc;
        END;

        CASE squashmethod OF
        | method1: (* raw squashing considering 2 dots (x,y x,y+1) *)
            j:=1;
            FOR y:=1 TO 16 BY 2 DO
		        a:=old[y];
		        b:=old[y+1];
                R:="";
		        FOR x:=1 TO 8 DO
        	        ca:=a[x];
                    cb:=b[x];
                    CASE getStrIndex(delim,ca,  combi1) OF
                    | 1:
                        CASE getStrIndex(delim,cb, combi1) OF
                        | 1: c:="0";
                        | 2: c:="1";
                        END;
                    | 2:
                         CASE getStrIndex(delim,cb, combi1) OF
                        | 1: c:="1";
                        | 2: c:="1";
                        END;
                    END;
                    Str.Append(R,c);
                END;
                now[j]:=R;
                INC(j);
            END;
        | method2: (* raw squashing considering 4 dots (x,y x,y+1 x+1,y x+1,y+1) *)
            j:=1;
            FOR y:=1 TO 16 BY 2 DO
		        a:=old[y];
                b:=old[y+1];
		        R:="";
		        FOR x:=1 TO 8 BY 2 DO
        	        Str.Slice (sa, a, x,2);
                    Str.Slice (sb, b, x,2);
                    CASE getStrIndex(delim,sa,  combi2) OF
                    | 1 :
                        CASE getStrIndex(delim,sb,  combi2) OF
                        | 1 : sc:="00";
                        | 2 : sc:="10";sc:="00";
                        | 3 : sc:="01";sc:="00";
                        | 4 : sc:="11";
                        END;
                    | 2 :
                        CASE getStrIndex(delim,sb,  combi2) OF
                        | 1 : sc:="10";sc:="00";
                        | 2 : sc:="10";
                        | 3 : sc:="01";
                        | 4 : sc:="11";
                        END;
                    | 3 :
                        CASE getStrIndex(delim,sb,  combi2) OF
                        | 1 : sc:="01"; sc:="00";
                        | 2 : sc:="10";
                        | 3 : sc:="01";
                        | 4 : sc:="11";
                        END;
                    | 4 :
                        CASE getStrIndex(delim,sb,  combi2) OF
                        | 1 : sc:="11";
                        | 2 : sc:="11";
                        | 3 : sc:="11";
                        | 4 : sc:="11";
                        END;
                    END;
                    Str.Append(R,sc);
                END;
                now[j]:=R;
                INC(j);
            END;
        | method3,method4,method5,method6:
            CASE squashmethod OF
            | method3 : transfo:=transfo1; divisor:=divisor1;
            | method4 : transfo:=transfo2; divisor:=divisor2;
            | method5 : transfo:=transfo3; divisor:=divisor3;
            | method6 : transfo:=transfo4; divisor:=divisor4;
            END;
            j:=1;
            FOR y:=1+1 TO 16-1 BY 2 DO
                a:=old[y];
                R:="";
		        FOR x:=1 TO 8 DO
		            CASE a[x] OF
		            | "0": bval:=0;
		            | "1": bval:=1;
		            END;
		            sigma:=0;
		            FOR jj:=1 TO 4 DO
		                CASE (y+jj-1-1) OF
		                | 1..16:
		                    FOR ii:=1 TO 3 DO
		                        CASE (x+ii-1-1) OF
		                        | 1..8:
		                            k:=transfo[jj][ii];
		                            INC(sigma,bval*k);
		                        END;
		                    END;
		                END;
		            END;
		            sigma:=sigma DIV divisor;
		            IF sigma=0 THEN
		                Str.Append(R,"0");
		            ELSE
		                Str.Append(R,"1");
		            END;
		        END;
		        now[j]:=R;
		        INC(j);
            END;
        END;

        (* update *)
        FOR i:=1 TO 8 DO
            R:=now[i];
            lv:=Str.StrToCard(R,2,ok);
            fontbuff[pto]:=BYTE(lv);
            INC(pto);
        END;
    END;
END squash;

PROCEDURE doSquash (method:CARDINAL;infile:ARRAY OF CHAR;
                   VAR R:ARRAY OF CHAR):BOOLEAN;
VAR
    hnd:FIO.File;
    got:CARDINAL;
BEGIN
    Lib.Fill(ADR(gFont25),fontsize25, 0);
    hnd:=FIO.OpenRead(infile);
    FIO.AssignBuffer(hnd,ioBuffer);
    got:=FIO.RdBin(hnd,gFont25,SIZE(gFont25));
    FIO.Close(hnd);

    squash (method, gFont25);

    hnd:=FIO.Create(R);
    FIO.AssignBuffer(hnd,ioBuffer);
    FIO.WrBin(hnd,gFont25,fontsize50);
    FIO.Flush(hnd);
    FIO.Close(hnd);

    RETURN TRUE;
END doSquash;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    prefixok      = "+++ ";
    prefixerr     = "--- ";
    msgProcessing = "Processing ";
    msgCreated    = " created !";
    msgProblemAt  = "Problem at line ";
    msgInFile     = " while processing ";
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : str128;
    infile,outfile,base : str128;
    state           : (waiting,gotparm1,gotparm2);
    cmd             : (nocmd,build,rebuild,squashfont);
    overwrite,verbose,relaxed,relaxed25,relaxed50,autoheight : BOOLEAN;
    method          : CARDINAL;
    countFile       : CARDINAL;
    ok              : BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    cmd       := nocmd;
    overwrite := FALSE;
    verbose   := FALSE;
    relaxed   := FALSE;
    relaxed25 := FALSE;
    relaxed50 := FALSE;
    autoheight:= FALSE;
    method    := method1;
    state     := waiting;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    FOR i := 1 TO parmcount DO (* for future extension ! *)
        Lib.ParamStr(S,i);
        RtrimBlanks(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "B"+delim+"BUILD"+delim+
                                  "R"+delim+"REBUILD"+delim+
                                  "O"+delim+"OVERWRITE"+delim+
                                  "V"+delim+"VERBOSE"+delim+
                                  "P"+delim+"PARTIAL"+delim+
                                  "P2"+delim+"25"+delim+
                                  "P5"+delim+"50"+delim+
                                  "A"+delim+"AUTO"+delim+
                                  "S"+delim+"SQUASH"+delim+
                                  "M1"+delim+
                                  "M2"+delim+
                                  "M3"+delim+
                                  "M4"+delim+
                                  "M5"+delim+
                                  "M6"
                              );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5   : CASE cmd OF
                      | nocmd,build:cmd:=build;
                      ELSE abort(errNonsense,"-b, -r and -s");END;
            | 6,7   : CASE cmd OF
                      | nocmd,rebuild:cmd:=rebuild;
                      ELSE abort(errNonsense,"-b, -r and -s");END;
            | 8,9   : overwrite := TRUE;
            | 10,11 : verbose := TRUE;
            | 12,13 : relaxed:=TRUE;
            | 14,15 : relaxed25:=TRUE;
            | 16,17 : relaxed50:=TRUE;
            | 18,19 : autoheight:=TRUE;
            | 20,21 : CASE cmd OF
                      | nocmd,squashfont:cmd:=squashfont;
                      ELSE abort(errNonsense,"-b, -r and -s");END;
            | 22:     method    :=method1;
            | 23:     method    :=method2;
            | 24:     method    :=method3;
            | 25:     method    :=method4;
            | 26:     method    :=method5;
            | 27:     method    :=method6;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting:  Str.Copy(infile,R);
            | gotparm1: Str.Copy(outfile,R);
            | gotparm2: abort(errTooManyParms,S); (* errSyntax would do *)
            END;
            INC(state);
        END;
    END;
    CASE state OF
    | waiting: abort(errSyntax,"");
    | gotparm1:
        CASE cmd OF
        | nocmd:
            abort(errSyntax,"");
        | build:
            defaultExtension(extFON,infile);
            IF Str.Pos(infile,extDAT) # MAX(CARDINAL) THEN
                abort(errBadExt,infile);
            END;
            IF (relaxed25 AND relaxed50) THEN abort(errEither,"");END;
            IF relaxed THEN abort(errNotSpecific,"");END;
            IF (relaxed25 OR relaxed50) THEN relaxed:=TRUE;END; (* trick, eh eh *)
        | rebuild:
            defaultExtension(extDAT,infile);
            IF Str.Pos(infile,extFON) # MAX(CARDINAL) THEN
                abort(errBadExt,infile);
            END;
            IF (relaxed25 OR relaxed50) THEN abort(errBuildOnly,"");END;
        | squashfont:
            abort(errSyntax,"");
        END;
    | gotparm2:
        CASE cmd OF
        | nocmd,build,rebuild :
            abort(errSyntax,"");
        | squashfont :
            defaultExtension(extFON,infile);
            IF Str.Pos(infile,extDAT) # MAX(CARDINAL) THEN
                abort(errBadExt,infile);
            END;
            IF chkJoker(infile) THEN abort(errJokers,infile);END;
            IF chkJoker(outfile) THEN abort(errJokers,outfile);END;
            IF relaxed THEN abort(errIrrelevant,"");END;
            IF (relaxed25 OR relaxed50) THEN abort(errBuildOnly,"");END;
        END;
    END;

    makebase(infile,base);
    countFile:=buildFileList(infile);
    CASE countFile OF
    | 0 : abort(errNotFound,infile);
    | MAX(CARDINAL):abort(errTooMany,infile);
    END;

    (* first, we check for possible problems *)

    FOR i := 1 TO countFile DO
        Str.Copy(S,fileArray[firstfile+i-1]);
        Str.Concat(infile,base,S);
        CASE cmd OF
        | build:
            IF chkFont(relaxed,autoheight,infile)=FALSE THEN abort(errBadFont,infile);END;
            IF NOT(overwrite) THEN
                makeTarget(infile,extDAT,S);
                IF FIO.Exists(S) THEN abort(errExists,S);END;
            END;
        | rebuild:
            IF NOT(overwrite) THEN
                makeTarget(infile,extFON,S);
                IF FIO.Exists(S) THEN abort(errExists,S);END;
            END;
        | squashfont:
            (* infile does not contain any joker *)
            IF NOT(overwrite) THEN
                IF FIO.Exists(outfile) THEN abort(errExists,outfile);END;
            END;
        END;
    END;

    (* now we can proceed *)

    WrStr(Banner);WrLn;
    WrLn;

    FOR i := 1 TO countFile DO
        Str.Copy(S,fileArray[firstfile+i-1]);
        Str.Concat(infile,base,S);
        Str.Concat(S,msgProcessing,infile);
        video(S,TRUE);
        initFontBuffer; (* in case we are in relaxed mode *)
        CASE cmd OF
        | build:
            ok:=doBuild(relaxed25,relaxed50,verbose,autoheight,infile,R);
        | rebuild:
            ok:=doRebuild(relaxed,verbose,infile,R);
        | squashfont:
            Str.Copy(R,outfile);
            ok:=doSquash(method,infile,R);
        END;
        video(S,FALSE);
        IF ok THEN
            Str.Concat(S,prefixok,R);Str.Append(S,msgCreated);
        ELSE
            Str.Concat(S,prefixerr,msgProblemAt);Str.Append(S,R);
            Str.Append(S,msgInFile);Str.Append(S,infile);
        END;
        WrStr(S);WrLn;
    END;

    abort(errNone,"");
END FontTool.
