(* ---------------------------------------------------------------
Title         Q&D VESA library
Author        PhG
Overview
Notes         we assume 8 or 32 bits modes
Bugs
Wish List

--------------------------------------------------------------- *)

IMPLEMENTATION MODULE QD_VESA;

IMPORT SYSTEM;
IMPORT Lib;
IMPORT IO;

(* ------------------------------------------------------------ *)

CONST
    CO80 = BYTE(3); (* color 80x25 *)
    MONO = BYTE(7); (* monochrome *)
CONST
    iv = 010H; (* video interrupt is $10 *)
    sb = 040H; (* segBiosData *)
VAR
    biosCurrentVideoMode  [sb:049H] : BYTE;
    biosColumnsOnScreen   [sb:04AH] : CARDINAL; (* a WORD *)
    biosRowsOnScreen      [sb:084H] : SHORTCARD; (* a BYTE, rows-1 in fact *)

PROCEDURE isMonoMode ():BOOLEAN;
BEGIN
    RETURN (biosCurrentVideoMode = MONO);
END isMonoMode;

PROCEDURE setVideoMode (mode:SHORTCARD); (* private *)
VAR
    R : SYSTEM.Registers;
BEGIN
    (* set video mode *)
    R.AH := 00H;
    R.AL := mode;
    Lib.Intr(R,iv);
END setVideoMode;

PROCEDURE waitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END waitVGAretrace;

PROCEDURE setDAC(index:CARDINAL;red,green,blue:SHORTCARD);
CONST
    DACWriteIndex  = 03C8H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACWriteIndex,SHORTCARD(index));
    SYSTEM.Out (DACDataRegister, red);
    SYSTEM.Out (DACDataRegister, green);
    SYSTEM.Out (DACDataRegister, blue);
END setDAC;

PROCEDURE resetDAC (index,r,g,b:CARDINAL);
BEGIN
    setDAC(index, SHORTCARD(r), SHORTCARD(g), SHORTCARD(b) );
END resetDAC;

PROCEDURE getDAC (index:CARDINAL;VAR r,g,b:SHORTCARD);
CONST
    DACReadIndex   = 03C7H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACReadIndex,SHORTCARD(index));
    r:=SYSTEM.In (DACDataRegister);
    g:=SYSTEM.In (DACDataRegister);
    b:=SYSTEM.In (DACDataRegister);
END getDAC;

VAR
    syspal : ARRAY [minpalndx..maxpalndx] OF paltriplettype;

PROCEDURE savesystempal (  );
VAR
    i:CARDINAL;
BEGIN
    FOR i := minpalndx TO maxpalndx DO
        getDAC(i,syspal[i].r,syspal[i].g,syspal[i].b);
    END;
END savesystempal;

PROCEDURE restoresystempal (  ); (* assume it was saved ! *)
VAR
    i : CARDINAL;
BEGIN
    FOR i := minpalndx TO maxpalndx DO
        setDAC(i,syspal[i].r,syspal[i].g,syspal[i].b);
    END;
END restoresystempal;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(*# save *)
(*# data(near_ptr => off)  *)
TYPE
    vpointer = POINTER TO WORD; (* well, glad M2 is case-sensitive ! *)
(*# restore *)

VAR
    vesalist:vpointer;
    modelist:vpointer; (* duplicate for getModeFromList *)
TYPE
    vesamainbuffertype = RECORD
        id             : ARRAY [0..3] OF CHAR;
        majmin         : WORD;
        pOEMname       : LONGWORD;  (* asciiz *)
        capabilities   : LONGWORD;
        pList          : LONGWORD;
        VRAM64KBblocks :WORD;
        OEMmajmin      : WORD;
        pOEM           : LONGWORD;
        pProduct       : LONGWORD;
        pRevision      : LONGWORD;
        VBEAFversion   : WORD;
        pAcceleratedModes:LONGWORD;
        dummy          : ARRAY [1..216] OF BYTE;
        OEMdummy       : ARRAY [1..256] OF BYTE;
    END;

VAR
    vesaBuffer : vesamainbuffertype;

PROCEDURE CheckVesaHere (forceVBE2:BOOLEAN;VAR listPtr : vpointer) : BOOLEAN; (* private *)
VAR
    R : SYSTEM.Registers;
BEGIN
    IF forceVBE2 THEN
        vesaBuffer.id[0]:="V";
        vesaBuffer.id[1]:="B";
        vesaBuffer.id[2]:="E";
        vesaBuffer.id[3]:="2";
    END;
    R.AX := 4F00H;
    R.ES := Seg(vesaBuffer); (* buffer segment *)
    R.DI := Ofs(vesaBuffer); (* buffer offset *)
    Lib.Intr(R,iv);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    IF vesaBuffer.id[0] # "V" THEN RETURN FALSE; END;
    IF vesaBuffer.id[1] # "E" THEN RETURN FALSE; END;
    IF vesaBuffer.id[2] # "S" THEN RETURN FALSE; END;
    IF vesaBuffer.id[3] # "A" THEN RETURN FALSE; END;
    Lib.Move (ADR(vesaBuffer.pList),ADR(listPtr),SIZE(vesaBuffer.pList));
    RETURN TRUE;
END CheckVesaHere;

(* warning ! a mode may be supposed "available" while not being implemented ! matrox idiosyncrasy ! *)

PROCEDURE ModeAvailable (listPtr : vpointer; mode : CARDINAL) : BOOLEAN; (* private *)
BEGIN
    LOOP
        IF listPtr^ = WORD(0FFFFH) THEN EXIT; END;
        IF listPtr^ = WORD(mode) THEN RETURN TRUE; END;
        Lib.IncFarAddr(listPtr,SIZE(WORD));
    END;
    RETURN FALSE;
END ModeAvailable;

PROCEDURE SetVesaMode (mode : CARDINAL) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AX := 4F02H;
    R.BX := mode;
    Lib.Intr(R,iv);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    RETURN TRUE;
END SetVesaMode;

(* ------------------------------------------------------------ *)

(* vesamodeinfobuffertype MUST be 256 bytes *)

TYPE
    vesamodeinfobuffertype = RECORD
        modeattributes      : WORD; (* 79 *)
        wA                  : BYTE; (* 81 *)
        wB                  : BYTE; (* id *)
        wgranularityKB      : WORD;
        wsizeKB             : WORD;
        segmentwA           : WORD; (* 0 if unsupported *)
        segmentwB           : WORD; (* id *)
        wpositioningfunction: LONGWORD;
        bytesperscanline    : WORD;
    (* ---remainder is optional for VESA modes in v1.0/1.1, needed for OEM modes--- *)
        width               : WORD; (* pixels or chars *)
        height              : WORD; (* id *)
        wcell               : BYTE;
        hcell               : BYTE;
        memplanes           : BYTE;
        bpp                 : BYTE;
        banks               : BYTE;
        memorymodeltype     : BYTE; (* 82 *)
        banksizeKB          : BYTE;
        imagepages          : BYTE; (* -1 *)
        reservedflag        : BYTE; (* 1 for VBE 3.0, ELSE 0 *)
    (* ---VBE v1.2+ --- *)
        rmasksize           : BYTE;
        rfield              : BYTE;
        gmasksize           : BYTE;
        gfield              : BYTE;
        bmasksize           : BYTE;
        bfield              : BYTE;
        reservedmasksize    : BYTE;
        reservedmaskposition: BYTE;
        directcolormode     : BYTE; (* bits 0 and 1 *)
    (* ---VBE v2.0+ --- *)
        linearvideobufferaddr: LONGWORD;
        offscreenmemoryaddr : LONGWORD;
        offscreenmemoryKB   : WORD;
    (* ---VBE v3.0 --- *)
        bytesperscanlinelinear:WORD;
        imagesforbankedmodes: BYTE; (* -1 *)
        imagesforlinearmodes: BYTE; (* -1 *)
        rmasklinear         : BYTE;
        rmaskLSBlinear      : BYTE;
        gmasklinear         : BYTE;
        gmaskLSBlinear      : BYTE;
        bmasklinear         : BYTE;
        bmaskLSBlinear      : BYTE;
        reservedmasklinear  : BYTE;
        reservedmaskLSBlinear:BYTE;
        maxclockHz          : LONGWORD;
        unused              : ARRAY [1..190] OF BYTE;
    END;

VAR
    vesaInfos   : vesamodeinfobuffertype;
    currbank    : CARDINAL; (* a word in fact *)
    segwin      : CARDINAL;
    granularity : LONGCARD; (* should be 65536 *)
    granshift   : LONGCARD; (* should be 16 *)
CONST
    vesaminy = 0;
    vesamaxy = 1024-1; (* 1280x1024 at most *)
VAR
    vesabaseaddr : ARRAY [vesaminy..vesamaxy] OF LONGCARD;

(* ------------------------------------------------------------ *)

PROCEDURE initVESAlookup (  );
VAR
    y,pixelsize,i : CARDINAL;
    addr,xcount : LONGCARD;
BEGIN
    xcount:=LONGCARD(vesaInfos.width);
    pixelsize := CARDINAL(vesaInfos.bpp) >> 3; (* DIV 8 *)
    addr:=0;
    FOR y:=vesaminy TO vesamaxy DO
        vesabaseaddr[y]  := addr;
        FOR i:=1 TO pixelsize DO
            INC (addr, xcount);   (* addr  := x + y*xcount*pixelsize *)
        END;
    END;
END initVESAlookup;

PROCEDURE getVESAmodeInfos (mode:CARDINAL):BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    Lib.Fill(ADR(vesaInfos),SIZE(vesaInfos),00H);

    R.AX := 4F01H;      (* GET SuperVGA MODE INFORMATION *)
    R.CX := mode;
    R.ES := Seg(vesaInfos); (* buffer segment *)
    R.DI := Ofs(vesaInfos); (* buffer offset *)
    Lib.Intr(R,iv);

    IF R.AL # 4FH THEN RETURN FALSE; END; (* unsupported function *)
    IF R.AH = 00H THEN initVESAlookup; END; (* ok, init if needed *)
    RETURN (R.AH=00H); (* FALSE if function failure *)
END getVESAmodeInfos;

PROCEDURE setVESAbank (bank:CARDINAL):BOOLEAN;
VAR
    R:SYSTEM.Registers;
BEGIN
    currbank := bank;
    R.AX := 04F05H;
    R.BH := 00H;     (* select video memory window -- dx = window address in granularity units *)
    R.BL := 00H;     (* window A should always work *)
    R.DX := currbank;
    Lib.Intr(R,iv);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    RETURN (R.AH=00H);
END setVESAbank;

(* ------------------------------------------------------------ *)

PROCEDURE VESAgetpixel8 (x,y:CARDINAL; VAR ink:SHORTCARD);
VAR
    addr:LONGCARD;
    bank,offset:CARDINAL;
    rc:BOOLEAN;
BEGIN
    addr  := LONGCARD(x)+vesabaseaddr[y];
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarMove([segwin:offset],FarADR(ink),1);
END VESAgetpixel8;

PROCEDURE VESAgetpixel32 (x,y:CARDINAL; VAR r,g,b:SHORTCARD);
VAR
    addr:LONGCARD;
    bank,offset,bpp,pixelsize:CARDINAL;
    rc:BOOLEAN;
BEGIN
    bpp       := CARDINAL(vesaInfos.bpp);
    pixelsize := bpp >> 3; (* DIV 8 *)
    x:=x * pixelsize;
    addr  := LONGCARD(x) +vesabaseaddr[y];
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarMove([segwin:offset],FarADR(b),1);

    INC(addr);
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarMove([segwin:offset],FarADR(g),1);

    INC(addr);
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarMove([segwin:offset],FarADR(r),1);
END VESAgetpixel32;

(* ------------------------------------------------------------ *)

PROCEDURE VESAputpixel8 (x,y:CARDINAL;ink:SHORTCARD);
VAR
    addr:LONGCARD;
    bank,offset:CARDINAL;
    rc:BOOLEAN;
    p:FarADDRESS;
BEGIN
    addr  := LONGCARD(x)+vesabaseaddr[y];
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarFill([segwin:offset],1,ink);
END VESAputpixel8;

PROCEDURE VESAputpixel32 (x,y:CARDINAL; r,g,b:SHORTCARD);
TYPE
    rgbtype = RECORD
        r,g,b:SHORTCARD;
    END;
VAR
    addr:LONGCARD;
    bank,offset,bpp,pixelsize:CARDINAL;
    rc:BOOLEAN;
    p:FarADDRESS;
    pixel:rgbtype;
BEGIN
    bpp       := CARDINAL(vesaInfos.bpp);
    pixelsize := bpp >> 3; (* DIV 8 *)
    x:=x * pixelsize;
    addr  := LONGCARD(x) +vesabaseaddr[y];
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarFill([segwin:offset],1,r);

    INC(addr);
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarFill([segwin:offset],1,g);

    INC(addr);
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);
    IF currbank # bank THEN rc:=setVESAbank(bank);END;
    (* ugly hack *)
    Lib.FarFill([segwin:offset],1,b);
END VESAputpixel32;

(* ------------------------------------------------------------ *)

CONST
    vesaminx = 0;
    vesamaxx = 1600-1; (* 1600x1200 at most ! *)
    quadsize = 4;      (* RGB? *)
VAR
    VESAhline : ARRAY [vesaminx..vesamaxx*quadsize] OF BYTE;

PROCEDURE initVESAhline (width:CARDINAL;v:SHORTCARD);
VAR
    bpp,pixelsize,count:CARDINAL;
BEGIN
    bpp       := CARDINAL(vesaInfos.bpp);
    pixelsize := bpp >> 3; (* DIV 8 *)
    count     := width * pixelsize;
    Lib.FarFill(FarADR(VESAhline),count,v);
END initVESAhline;

PROCEDURE updateVESAhline (width,y:CARDINAL);
VAR
    addr,addrUP : LONGCARD;
    bank,bankUP,offset:CARDINAL;
    rc:BOOLEAN;
    bpp,pixelsize,count:CARDINAL;
    x,xx:CARDINAL;
    r,g,b:SHORTCARD;
BEGIN
    bpp       := CARDINAL(vesaInfos.bpp);
    pixelsize := bpp >> 3; (* DIV 8 *)
    count     := width * pixelsize;
    addr  := vesabaseaddr[y];
    bank  := CARDINAL (addr >> granshift);
    offset:= CARDINAL (addr MOD granularity);

    addrUP:= vesabaseaddr[y]+LONGCARD(count);
    bankUP:= CARDINAL (addrUP >> granshift);

    IF bank=bankUP THEN
        IF currbank # bank THEN rc:=setVESAbank(bank);END;
        IF ODD(count) THEN
            Lib.FarMove(FarADR(VESAhline),[segwin:offset],count);
        ELSE
            Lib.FarWordMove(FarADR(VESAhline),[segwin:offset],(count>>1) );
        END;
    ELSE
        xx:=0;
        FOR x:=0 TO (width-1) DO
            CASE bpp OF
            | 8:
                VESAputpixel8 (x,y, VESAhline[x] );
            | 32: (* pay attention to order ! *)
                r:=VESAhline[xx];
                g:=VESAhline[xx+1];
                b:=VESAhline[xx+2];
                VESAputpixel32(x,y, r,g,b);
                INC(xx,pixelsize);
            END;
        END;
    END;
END updateVESAhline;

PROCEDURE VESAputpixel8hline (x,y:CARDINAL;ink:SHORTCARD);
VAR
    addr:LONGCARD;
    bank,offset:CARDINAL;
    rc:BOOLEAN;
    p:FarADDRESS;
BEGIN
    p:=FarADR(VESAhline);
    Lib.IncFarAddr(p,x);
    Lib.FarFill(p,SIZE(ink),ink);
END VESAputpixel8hline;

PROCEDURE VESAputpixel32hline (x,y:CARDINAL; r,g,b:SHORTCARD);
TYPE
    rgbquadtype = RECORD
        r,g,b,foobar:SHORTCARD;
    END;
VAR
    addr:LONGCARD;
    bank,offset,bpp,pixelsize:CARDINAL;
    rc:BOOLEAN;
    p:FarADDRESS;
    pixel:rgbquadtype;
BEGIN
    pixel.r:=r;
    pixel.g:=g;
    pixel.b:=b;
    pixel.foobar:=0;
    bpp       := CARDINAL(vesaInfos.bpp);
    pixelsize := bpp >> 3; (* DIV 8 *)
    x:=x * pixelsize;
    p:=FarADR(VESAhline);
    Lib.IncFarAddr(p,x);
    Lib.FarMove(FarADR(pixel), p, SIZE(pixel) );
END VESAputpixel32hline;

(* ------------------------------------------------------------ *)

PROCEDURE initVESAmode (m:CARDINAL ):BOOLEAN;
VAR
    n : CARDINAL;
    lc : LONGCARD;
BEGIN
    CASE m OF
    | 1 :
        RETURN setVESAbank(0);
    | 2 :
        segwin        := vesaInfos.segmentwA;
        n             := vesaInfos.wgranularityKB;
        granularity   := LONGCARD(n) * 1024;

        lc:=granularity;
        granshift:=0;
        LOOP
            IF lc = 0 THEN EXIT; END;
            lc := lc >> 1;
            INC(granshift);
        END;
        DEC(granshift);

        n             := vesaInfos.wgranularityKB;
        lc            := 1 << granshift;
        RETURN ( lc=granularity );
    END;
END initVESAmode;

PROCEDURE getVESAgeometry (VAR xmin,ymin,xmax,ymax,bpp,bitsum:CARDINAL);
BEGIN
    xmin          := 0;
    ymin          := 0;
    xmax          := vesaInfos.width;  DEC(xmax);
    ymax          := vesaInfos.height; DEC(ymax);
    bpp           := CARDINAL(vesaInfos.bpp); (* 0=text, 8=256 colors, *)
    CASE bpp OF
    | 8 :
        bitsum:=8;
    | 16,32:
        bitsum:=CARDINAL(vesaInfos.rmasksize+vesaInfos.gmasksize+vesaInfos.bmasksize);
        (* 15,16, 24=16M colors, 32=4G colors *)
    ELSE
        bitsum:=0;
    END;
END getVESAgeometry;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE chkVESAhere () : BOOLEAN;
BEGIN
    RETURN CheckVesaHere(TRUE, vesalist); (* we may need VBE2 1600x1200 mode *)
END chkVESAhere;

PROCEDURE isModeHere(mode:CARDINAL):BOOLEAN;
BEGIN
    IF vesalist=vpointer(NIL) THEN RETURN FALSE; END; (* if we forgot to check for VESA first *)
    RETURN ModeAvailable (vesalist,mode);
END isModeHere;

PROCEDURE set80x25 ();
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 02H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN; END;
    setVideoMode(CO80);
END set80x25;

PROCEDURE set80x50 ();
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 02H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN; END;
    setVideoMode(CO80);
    (* load rom 8x8 dbl-dot patterns *)
    R.AH := 11H;
    R.AL := 12H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,iv);
END set80x50;

PROCEDURE getTextmodeDimensions(VAR hcount,vcount:CARDINAL);
BEGIN
    hcount := biosColumnsOnScreen;
    vcount := CARDINAL(biosRowsOnScreen) +1;
END getTextmodeDimensions;

PROCEDURE textclearscreen (attr:SHORTCARD);
VAR
    R : SYSTEM.Registers;
    savBP,savDS : CARDINAL;
    hcount,vcount:CARDINAL;
    activepage:SHORTCARD;
BEGIN
    getTextmodeDimensions(hcount,vcount);

    (* INT 10 - VIDEO - SCROLL DOWN WINDOW *)
    savBP := R.BP;
    savDS := R.DS;
    R.AH  := 07H;
    R.AL  := 00H; (* entire window *)
    R.BH  := attr; (* attribute used to write blank lines at top of window *)
    (* CH,CL = row,column of window's upper left corner 0-based *)
    R.CH  := 0;
    R.CL  := 0;
    (* DH,DL = row,column of window's lower right corner 0-based *)
    R.DH  := SHORTCARD(vcount-1);
    R.DL  := SHORTCARD(hcount-1);
    Lib.Intr(R,iv);
    (*
    some implementations (including the original IBM PC)
    have a bug which destroys BP
    the Trident TVGA8900CL (BIOS dated 1992/9/8) clears DS to 0000h
    when scrolling in an SVGA mode (800x600 or higher)
    *)
    IF R.BP # savBP THEN R.BP := savBP; END;
    IF R.DS # savDS THEN R.DS := savDS; END;

    (* INT 10 - VIDEO - GET CURRENT VIDEO MODE *)
    R.AH := 0FH;
    Lib.Intr(R,iv);
    activepage := R.BH;

    (* INT 10 - VIDEO - SET CURSOR POSITION *)
    R.AH := 02H;
    R.BH := activepage;
    R.DH := 0; (* row (00h is top) *)
    R.DL := 0; (* column (00h is left) *)
    Lib.Intr(R,iv);
END textclearscreen;

PROCEDURE getModeFromList (firstcall:BOOLEAN;VAR vesamode:CARDINAL):BOOLEAN;
BEGIN
    IF vesalist=vpointer(NIL) THEN RETURN FALSE; END; (* if we forgot to check for VESA first *)
    IF firstcall THEN modelist:=vesalist; END;
    IF modelist=vpointer(NIL) THEN RETURN FALSE; END; (* no firstcall ! *)
    IF modelist^ = WORD(0FFFFH) THEN RETURN FALSE; END;
    vesamode:=modelist^;
    Lib.IncFarAddr(modelist,SIZE(WORD));
    RETURN TRUE;
END getModeFromList;

(* ------------------------------------------------------------ *)

BEGIN
    vesalist:=vpointer(NIL);
    modelist:=vpointer(NIL);
END QD_VESA.





