/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2012, 2013
 */

#include <stdio.h>
#include <stdlib.h>    /* system() */
#include <string.h>    /* strcpy() */
#include <unistd.h>    /* read() */
#include <sys/types.h> /* struct utimbuf */
#include "crc32.h"     /* all CRC32 related stuff */
#include "http.h"
#include "helpers.h"   /* slash2backslash(), strtolower() */
#include "fileexst.h"
#include "kprintf.h"
#include "pkginst.h"
#include "libunzip.h"  /* zip_listfiles()... */


/* checks that pkgname is NOT installed. return 0 on success, non-zero otherwise. */
int validate_package_not_installed(char *pkgname, char *dosdir) {
  char fname[512];
  sprintf(fname, "%s\\packages\\%s.lst", dosdir, pkgname);
  if (fileexists(fname) != 0) { /* file exists -> package already installed */
    kitten_printf(3, 18, "Package %s is already installed! You might want to use the 'update' action.", pkgname);
    puts("");
    return(-1);
  }
  return(0);
}



/* prepare a package for installation. this is mandatory before actually installing it!
 * returns a pointer to the zip file's index on success, NULL on failure. the *zipfile pointer is updated with a file descriptor to the open zip file to install. */
struct ziplist *pkginstall_preparepackage(struct pkgdb *pkgdb, char *pkgname, char *tempdir, char *localfile, int nosourceflag, char **repolist, FILE **zipfd, char *proxy, int proxyport, char *downloadingstring) {
  char fname[512];
  char zipfile[512];
  char appinfofile[64];
  int appinfopresence;
  char *instrepo;
  struct pkgdb *pkgnode, *lastnode;
  struct pkgrepo *pkgrepo;
  int repoid;
  unsigned long zipfilecrc;
  unsigned char *buff;
  int buffreadres;
  struct ziplist *ziplinkedlist, *curzipnode, *prevzipnode;

  strtolower(pkgname); /* convert pkgname to lower case, because the http repo is probably case sensitive */
  sprintf(appinfofile, "appinfo\\%s.lsm", pkgname); /* Prepare the appinfo/xxxx.lsm filename string for later use */

  if (localfile != NULL) {  /* if it's a local file, then we will have to skip all the network stuff */
      strcpy(zipfile, localfile);
    } else {
      zipfile[0] = 0;
  }

  if (zipfile[0] == 0) { /* need to download the package from a repository */
    /* look into the db to find the package */
    pkgnode = findpkg(pkgdb, pkgname, &lastnode);
    if (pkgnode == NULL) { /* no such package found in repositories */
      kitten_printf(3, 1, "No package '%s' found in online repositories.", pkgname);
      puts("");
      return(NULL);
    }

    /* if found - check the list of repositories */
    if (pkgnode->repolist == NULL) {
      kitten_printf(3, 2, "Package '%s' is not available in repositories.", pkgname);
      puts("");
      return(NULL);
    }

    if (pkgnode->repolist->nextrepo != NULL) { /* available from more than 1 repo.. */
        char userchoicestr[8];
        int userchoice, latestver_repoid = 1;
        struct pkgrepo *xrepo, *latestver = pkgnode->repolist;
        /* check out if we are able to find out the newest version */
        repoid = 2; /* setting to 2, because we start iterating at the second repo entry */
        for (xrepo = pkgnode->repolist->nextrepo; xrepo != NULL; xrepo = xrepo->nextrepo) {
          int versionnewerres = isversionnewer(latestver->version, xrepo->version);
          if (versionnewerres > 0) {
              latestver = xrepo;
              latestver_repoid = repoid;
            } else if (versionnewerres < 0) {  /* unable to tell which version is newer */
              latestver_repoid = -1;
              break;
          }
          repoid += 1;
        }
        if (latestver_repoid > 0) {
            repoid = latestver_repoid;
          } else { /* newest version could not be figured out, so let's ask the user to choose */
            puts("");
            kitten_printf(3, 3, "%s is available from several repositories. Choose which one to use:", pkgname);
            puts("");
            repoid = 1;
            for (xrepo = pkgnode->repolist; xrepo != NULL; xrepo = xrepo->nextrepo) {
              printf(" %d) %s %s (%s)\n", repoid, pkgnode->name, xrepo->version, repolist[xrepo->repo]);
              repoid += 1;
            }
            for (;;) {
              kitten_printf(3, 4, "Your choice:");
              printf(" ");
              fgets(userchoicestr, 6, stdin);
              userchoice = atoi(userchoicestr);
              if ((userchoice < 1) || (userchoice >= repoid)) {
                  kitten_puts(3, 5, "Invalid choice!");
                } else {
                  break;
              }
            }
            repoid = userchoice;
        }
      } else { /* available only from one repository - get it there */
        repoid = 1;
    }
    pkgrepo = pkgnode->repolist;
    for (; repoid > 1; repoid--) pkgrepo = pkgrepo->nextrepo;
    instrepo = repolist[pkgrepo->repo];

    /* if it's a network repo, download the package from repoid into the temp directory */
    if (detect_localpath(instrepo) == 0) {
        sprintf(fname, "%s%s.zip", instrepo, pkgname);
        sprintf(zipfile, "%s\\fdnpkg.tmp", tempdir);
        kitten_printf(3, 6, "Downloading package %s...", fname);
        puts("");
        if (http_get(fname, zipfile, proxy, proxyport, downloadingstring) <= 0) {
          kitten_puts(3, 7, "Error downloading package. Aborted.");
          return(NULL);
        }
      } else { /* else it's an on-disk repo, so we can use the package right from there */
        sprintf(zipfile, "%s%s.zip", instrepo, pkgname);
    }
    /* check the CRC of the downloaded file */
    buff = malloc(4096);  /* use a 4K buffer to compute file's CRC */
    if (buff == NULL) {
      kitten_puts(3, 15, "Error: Out of memory while computing the CRC of the package!");
      return(NULL);
    }
    *zipfd = fopen(zipfile, "rb");
    if (*zipfd == NULL) {
      kitten_puts(3, 14, "Error: Failed to open the downloaded package. Installation aborted.");
      free(buff);
      return(NULL);
    }
    zipfilecrc = crc32_init();
    while ((buffreadres = read(fileno(*zipfd), buff, 4096)) > 0) {
      crc32_feed(&zipfilecrc, buff, buffreadres);
    }
    crc32_finish(&zipfilecrc);
    fclose(*zipfd);
    free(buff);
    if (zipfilecrc != pkgrepo->crc32) {
      kitten_puts(3, 13, "Error: Downloaded package had wrong CRC. Installation aborted.");
      return(NULL);
    }
  } /* if (zipfile[0] == 0) */

  /* Now let's check the content of the zip file */

  *zipfd = fopen(zipfile, "rb");
  if (*zipfd == NULL) {
    kitten_puts(3, 8, "Error: Invalid zip archive! Package not installed.");
    return(NULL);
  }
  ziplinkedlist = zip_listfiles(*zipfd);
  if (ziplinkedlist == NULL) {
    kitten_puts(3, 8, "Error: Invalid zip archive! Package not installed.");
    fclose(*zipfd);
    return(NULL);
  }
  /* Verify that there's no collision with existing local files, look for the appinfo presence and get rid of sources if required */
  appinfopresence = 0;
  prevzipnode = NULL;
  for (curzipnode = ziplinkedlist; curzipnode != NULL;) {
    slash2backslash(curzipnode->filename);  /* change all slashes to backslashes */
    strtolower(curzipnode->filename);  /* put everything into lowercase */
    if (nosourceflag != 0) { /* if --nosource specified, skip sources */
      if (fdnpkg_strcasestr(curzipnode->filename, "source\\") == curzipnode->filename) { /* drop this file */
        curzipnode->filename[0] = 0; /* in fact, we just mark the file as 'empty' on the filename.. see below */
      }
    }
    if (curzipnode->filename[0] == 0) { /* ignore empty filenames (maybe it was empty originally, or has been emptied because it's a dropped source) */
      if (prevzipnode == NULL) {  /* take the item out of the list */
          ziplinkedlist = curzipnode->nextfile;
          free(curzipnode); /* free the item */
          curzipnode = ziplinkedlist;
        } else {
          prevzipnode->nextfile = curzipnode->nextfile;
          free(curzipnode); /* free the item */
          curzipnode = prevzipnode->nextfile;
      }
      continue; /* go to the next item */
    }
    if ((curzipnode->flags & ZIP_FLAG_ENCRYPTED) != 0) {
      kitten_printf(3, 20, "Error: Package contains an encrypted file:");
      puts("");
      printf(" %s\n", curzipnode->filename);
      zip_freelist(&ziplinkedlist);
      fclose(*zipfd);
      return(NULL);
    }
    if ((curzipnode->compmethod != 0/*store*/) && (curzipnode->compmethod != 8/*deflate*/) && (curzipnode->compmethod != 14/*lzma*/)) { /* unsupported compression method */
      kitten_printf(8, 2, "Error: Package contains a file compressed with an unsupported method (%d):", curzipnode->compmethod);
      puts("");
      printf(" %s\n", curzipnode->filename);
      zip_freelist(&ziplinkedlist);
      fclose(*zipfd);
      return(NULL);
    }
    if (strcmp(curzipnode->filename, appinfofile) == 0) appinfopresence = 1;
    if ((curzipnode->flags & ZIP_FLAG_ISADIR) != 0) { /* it it's a directory, make sure it will appear first in the list */
      if (prevzipnode != NULL) { /* do it only if it's not already at the top of the list */
        prevzipnode->nextfile = curzipnode->nextfile;
        curzipnode->nextfile = ziplinkedlist;
        ziplinkedlist = curzipnode;
        curzipnode = prevzipnode->nextfile;
        continue;
      }
    }
    prevzipnode = curzipnode;
    curzipnode = curzipnode->nextfile;
  }
  if (appinfopresence != 1) { /* if appinfo file not found, this is not a real FreeDOS package */
    kitten_printf(3, 12, "Error: Package do not contain the %s file! Not a valid FreeDOS package.", appinfofile);
    zip_freelist(&ziplinkedlist);
    fclose(*zipfd);
    return(NULL);
  }

  return(ziplinkedlist);
}


/* install a package that has been prepared already */
void pkginstall_installpackage(char *pkgname, char *dosdir, struct customdirs *dirlist, struct ziplist *ziplinkedlist, FILE *zipfd) {
  char fname[512];
  char shellcmd[512];
  char packageslst[64];
  char *shortfile;
  long filesextractedsucees = 0, filesextractedfailure = 0;
  struct ziplist *curzipnode;
  FILE *lstfd;

  sprintf(packageslst, "packages\\%s.lst", pkgname); /* Prepare the packages/xxxx.lst filename string for later use */

  /* check if not already installed, if already here, print a message "you might want to use update instead" */
  if (validate_package_not_installed(pkgname, dosdir) != 0) {
    zip_freelist(&ziplinkedlist);
    fclose(zipfd);
    return;
  }

  /* look out for collisions with already existing files */
  for (curzipnode = ziplinkedlist; curzipnode != NULL; curzipnode = curzipnode->nextfile) {
    shortfile = computelocalpath(curzipnode->filename, shellcmd, dosdir, dirlist);
    strcat(shellcmd, shortfile);
    if (fileexists(shellcmd)) {
      kitten_puts(3, 9, "Error: Package contains a file that already exists locally:");
      printf(" %s\n", shellcmd);
      zip_freelist(&ziplinkedlist);
      fclose(zipfd);
      return;
    }
  }

  /* create the %DOSDIR%/packages directory, just in case it doesn't exist yet */
  sprintf(fname, "%s\\packages\\", dosdir);
  mkpath(fname);

  /* open the lst file */
  sprintf(shellcmd, "%s\\%s", dosdir, packageslst);
  lstfd = fopen(shellcmd, "wb"); /* opening it in binary mode, because I like to have control over line terminators (CR/LF) */
  if (lstfd == NULL) {
    kitten_printf(3, 10, "Error: Could not create %s!", shellcmd);
    puts("");
    zip_freelist(&ziplinkedlist);
    fclose(zipfd);
    return;
  }
  /* write list of files in zip into the lst, and create the directories structure */
  for (curzipnode = ziplinkedlist; curzipnode != NULL; curzipnode = curzipnode->nextfile) {
    if (strcasecmp(curzipnode->filename, packageslst) == 0) continue; /* skip silently the package.lst file, if found */
    shortfile = computelocalpath(curzipnode->filename, fname, dosdir, dirlist); /* substitute paths to custom dirs */
    fprintf(lstfd, "%s%s\r\n", fname, shortfile); /* log the filename to packages\pkg.lst */
    if ((curzipnode->flags & ZIP_FLAG_ISADIR) != 0) { /* if a dir, create it */
        /* printf(" mkdir %s\n", fname); */
        mkpath(fname);
      } else { /* if a file, extract it */
        char fulldestfilename[1024];
        int unzip_result;
        sprintf(fulldestfilename, "%s%s", fname, shortfile);
        /* Now unzip the file */
        unzip_result = zip_unzip(zipfd, curzipnode, fulldestfilename);
        if (unzip_result != 0) {
            kitten_printf(8, 3, "Error while extracting '%s' to '%s'!", curzipnode->filename, fulldestfilename);
            printf(" [%d]", unzip_result);
            puts("");
            filesextractedfailure += 1;
          } else {
            printf(" %s -> %s\n", curzipnode->filename, fname);
            filesextractedsucees += 1;
        }
    }
  }
  fclose(zipfd);
  fclose(lstfd);
  /* free the ziplist and close file descriptor */
  zip_freelist(&ziplinkedlist);
  kitten_printf(3, 19, "Package %s installed: %d files extracted, %d errors.", pkgname, filesextractedsucees, filesextractedfailure);
  puts("");
}
