/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/

package org.jpc.emulator.memory.codeblock.basic;

import org.jpc.emulator.memory.codeblock.*;

public final class RealModeDecoder implements FirstStageCodeSet, Decoder, InstructionSource
{
    private static final boolean[] modrmArray = new boolean[] { // true for opcodes that require a modrm byte
	true, true, true, true, false, false, false, false, true, true, true, true, false, false, false, false,
	true, true, true, true, false, false, false, false, true, true, true, true, false, false, false, false,
	true, true, true, true, false, false, false, false, true, true, true, true, false, false, false, false,
	true, true, true, true, false, false, false, false, true, true, true, true, false, false, false, false,
	
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, true, true, false, false, false, false, false, true, false, true, false, false, false, false,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	
	true, true, false, false, true, true, true, true, false, false, false, false, false, false, false, false,
	true, true, true, true, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
	false, false, false, false, false, false, true, true, false, false, false, false, false, false, true, true
    };
    
    private static final boolean[] sibArray = new boolean[] { // true for modrm values that require a sib byte (32 bit addressing only)
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false
    };
    
    private static final boolean[] twoByte_0f_modrmArray = new boolean[] { // true for opcodes that require a modrm byte
	true, true, true, true, false, false, false, false, false, false, false, true,  false, false, false, false,
	true, true, true, true, true, true, true, true, true, false, false, false, false, false, false, false,
	true, true, true, true, true, false, true, false, true, true, true, true, true, true, true, true,
	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,

	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	true, true, true, true, true, true, true, false, false, false, false, false, true, true, true, true, 

	false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, 
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	false, false, false, true, true, true, false, false, false, false, false, true, true, true, true, true,
	true, true, true, true, true, true, true, true, false, true, true, true, true, true, true, true, 

	true, true, true, true, true, true, true, true, false, false, false, false, false, false, false, false, 
 	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, 
	true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true
    };

    private static final boolean[] twoByte_0f_sibArray = new boolean[] { // true for modrm values that require a sib byte (32 bit addressing only)
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false,
	false, false, false, false, true, false, false, false, false, false, false, false, true, false, false, false
    };

    private static final int PREFICES_SG = 0x7;
    private static final int PREFICES_ES = 0x1;
    private static final int PREFICES_CS = 0x2;
    private static final int PREFICES_SS = 0x3;
    private static final int PREFICES_DS = 0x4;
    private static final int PREFICES_FS = 0x5;
    private static final int PREFICES_GS = 0x6;

    private static final int PREFICES_OPERAND = 0x8;
    private static final int PREFICES_ADDRESS = 0x10;

    private static final int PREFICES_REPNE = 0x20;
    private static final int PREFICES_REPE = 0x40;

    private static final int PREFICES_REP = PREFICES_REPNE | PREFICES_REPE;

    private static final int PREFICES_LOCK = 0x80;

    private ByteSource source;
    private Operation current;
    private Operation waiting;
    private Operation working;

    private boolean blockComplete;

    public RealModeDecoder()
    {
	this.current = new Operation();
	this.waiting = new Operation();
	this.working = new Operation();
    }

    public InstructionSource decodeReal(ByteSource source)
    {
	reset();
	this.source = source;
	return this;
    }

    public InstructionSource decodeProtected(ByteSource source, boolean operandSize)
    {
	return null;
    }

    private void blockFinished()
    {
	blockComplete = true;
    }

    private void rotate()
    {
	Operation temp = current;
	current = waiting;
	waiting = working;
	working = temp;
    }

    public boolean getNext()
    {
	decode(); //will put new block in working
	rotate(); //moves buffer around
	if (current.decoded()) {
	    return true;
	} else if (current.terminal()) {
	    reset();
	    return false;
	} else {
            return getNext();
        }
    }

    private void reset()
    {
	working.reset();
	waiting.reset();
	current.reset();
	blockComplete = false;
    }

    public int getMicrocode()
    {
	return current.getMicrocode();
    }
 
    public int getLength()
    {
	return current.getLength();
    }
 
    public int getX86Length()
    {
	return current.getX86Length();
    }
           
    private void decodeComplete(int position)
    {
	working.finish(position);
    }
       
    private void decode()
    {
	working.reset();

	if (blockComplete) {
	    working.makeTerminal();
	    return;
	}

	int length = 0;
        try {
	    length = decodeOpcode();
        } catch (ArrayIndexOutOfBoundsException e) {
	    if (!waiting.decoded())
		throw e;

	    waiting.write(EIP_UPDATE);
	    working.makeTerminal();
	    blockFinished();
	    return;
	}

	if (length < 0) {
	    decodeComplete(-length);
	    blockFinished();
	} else {
	    decodeComplete(length);                    
	}
    }
    
    private int decodeOpcode()
    {
	int opcode = 0;
	int opcodePrefix = 0;
	int prefices = 0;
	int bytesRead = 0;
	int modrm = -1;
	int sib = -1;

	while (true) {
	    bytesRead += 1;
	    switch (opcode = 0xff & source.getByte()) {
	    case 0x0f:
	    case 0xd8:
	    case 0xd9:
	    case 0xda:
	    case 0xdb:
	    case 0xdc:
	    case 0xdd:
	    case 0xde:
	    case 0xdf:
		opcodePrefix = (opcodePrefix << 8) | opcode;
		opcode = 0xff & source.getByte();
		bytesRead += 1;
		modrm = opcode;
		break;
	    default:
		break;
		//Prefices
	    case 0x2e:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_CS;
		continue;
	    case 0x3e:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_DS;
		continue;
	    case 0x26:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_ES;
		continue;
	    case 0x36:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_SS;
		continue;
	    case 0x64:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_FS;
		continue;
	    case 0x65:
		prefices &= ~PREFICES_SG;
		prefices |= PREFICES_GS;
		continue;
	    case 0x66:
		prefices = prefices ^ PREFICES_OPERAND;
		continue;
	    case 0x67:
		prefices = prefices ^ PREFICES_ADDRESS;
		continue;
	    case 0xf2:
		prefices |= PREFICES_REPNE;
		continue;
	    case 0xf3:
		prefices |= PREFICES_REPE;
		continue;
	    case 0xf0:
		prefices |= PREFICES_LOCK;
		continue;
	    }
	    break;
	}

	opcode = (opcodePrefix << 8) | opcode;

	switch (opcodePrefix) {
	case 0x00:
	    if (modrmArray[opcode]) {
		modrm = 0xff & source.getByte();
		bytesRead += 1;
	    } else {
		modrm = -1;
	    }
	    if ((modrm == -1) || ((prefices & PREFICES_ADDRESS) == 0)) {
		sib = -1;
	    } else {
		if (sibArray[modrm]) {
		    sib = 0xff & source.getByte();
		    bytesRead += 1;
		} else {
		    sib = -1;
		}
	    }
	    break;
	case 0x0f:
	    if (twoByte_0f_modrmArray[0xff & opcode]) {
		modrm = 0xff & source.getByte();
		bytesRead += 1;
	    } else {
		modrm = -1;
	    }
	    if ((modrm == -1) || ((prefices & PREFICES_ADDRESS) == 0)) {
		sib = -1;
	    } else {
		if (twoByte_0f_sibArray[modrm]) {
		    sib = 0xff & source.getByte();
		    bytesRead += 1;
		} else {
		    sib = -1;
		}
	    }
	    break;
	case 0xd8:
	case 0xd9:
	case 0xda:
	case 0xdb:
	case 0xdc:
	case 0xdd:
	case 0xde:
	case 0xdf:
	    if (sibArray[modrm]) {
		sib = 0xff & source.getByte();
		bytesRead += 1;
	    } else {
		sib = -1;
	    }
	    break;
	default:
	    modrm = -1;
	    sib = -1;
	    break;
	}

	int operation = decodeOperation(prefices, opcode, modrm);

	if (isJump(operation))
	    working.write(EIP_UPDATE);
	
	working.write(operation);

	int displacement = 0;
	int immediate = 0;
	int moreImmediate = 0;

	int numberOfOperands = operandCount(operation);

	for (int i = 0; i < numberOfOperands; i++) {
	    switch (operandHasDisplacement(decodeOperand(i, prefices, opcode, modrm, sib))) {
	    case 0:
		break;
	    case 1:
		displacement = source.getByte();
		bytesRead += 1;
		i = numberOfOperands;
		break;
	    case 2:
		displacement = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00);
		bytesRead += 2;
		i = numberOfOperands;
		break;
	    case 4:
		displacement = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00) | ((source.getByte() << 16) & 0xff0000) | ((source.getByte() << 24) & 0xff000000);
		bytesRead += 4;
		i = numberOfOperands;
		break;
	    default:
		System.err.println("Displacement Longer Than 4-bytes");
		break;
	    }
	}

	for (int i = 0; i < numberOfOperands; i++) {
	    switch (operandHasImmediate(decodeOperand(i, prefices, opcode, modrm, sib))) {
	    case 0:
		break;
	    case 1:
		immediate = source.getByte();
		bytesRead += 1;
		i = numberOfOperands;
		break;
	    case 2:
		immediate = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00);
		bytesRead += 2;
		i = numberOfOperands;
		break;
	    case 3:
		immediate = ((source.getByte() << 16) & 0xff0000) | ((source.getByte() << 24) & 0xff000000) | (source.getByte() & 0xff);
		bytesRead += 3;
		i = numberOfOperands;
		break;
	    case 4:
		immediate = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00) | ((source.getByte() << 16) & 0xff0000) | ((source.getByte() << 24) & 0xff000000);
		bytesRead += 4;
		i = numberOfOperands;
		break;
	    case 6:
		immediate = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00) | ((source.getByte() << 16) & 0xff0000) | ((source.getByte() << 24) & 0xff000000);
		bytesRead += 4;
		moreImmediate = (source.getByte() & 0xff) | ((source.getByte() << 8) & 0xff00);
		bytesRead += 2;
		i = numberOfOperands;
		break;
	    default:
		System.err.println("Immediate Longer Than 6-bytes");
		break;
	    }
	}

	for (int i = 0; i < numberOfOperands; i++) {
	    int operand = decodeOperand(i, prefices, opcode, modrm, sib);
	    working.write(operand);

	    if (operandHasDisplacement(operand) > 0)
		working.write(displacement);

	    if (operandHasImmediate(operand) > 0)
		working.write(immediate);

	    if (operandHasImmediate(operand) > 4)
		working.write(moreImmediate);
	}
	    
	if (isJump(operation))
	    return -bytesRead;
	else {
	    return bytesRead;
	}
    }

    private static int decodeOperation(int prefices, int opcode, int modrm)
    {
	if ((opcode & 0xff00) == 0x0f00)
	    return decodeTwoByteOperation(prefices, opcode, modrm);
	
	if ((opcode & 0xf800) == 0xd800)
	    return decodeFPUOperation(prefices, opcode, modrm);

	//Normal One Byte Operation
	switch (opcode) {
	case 0x00:
	case 0x02:
	case 0x04:
	    return ADD_O8;
	case 0x01:
	case 0x03:
	case 0x05:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return ADD_O32;
	    else
		return ADD_O16;

	case 0x08:
	case 0x0a:
	case 0x0c:
	    return OR_O8;
	case 0x09:
	case 0x0b:
	case 0x0d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return OR_O32;
	    else
		return OR_O16;

	case 0x10:
	case 0x12:
	case 0x14:
	    return ADC_O8;
	case 0x11:
	case 0x13:
	case 0x15:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return ADC_O32;
	    else
		return ADC_O16;

	case 0x18:
	case 0x1a:
	case 0x1c:
	    return SBB_O8;
	case 0x19:
	case 0x1b:
	case 0x1d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return SBB_O32;
	    else
		return SBB_O16;

	case 0x20:
	case 0x22:
	case 0x24:
	    return AND_O8;
	case 0x21:
	case 0x23:
	case 0x25:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return AND_O32;
	    else
		return AND_O16;
	case 0x27:
	    return DAA;
	case 0x28:
	case 0x2a:
	case 0x2c:
	    return SUB_O8;
	case 0x29:
	case 0x2b:
	case 0x2d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return SUB_O32;
	    else
		return SUB_O16;
	case 0x2f:
	    return DAS;

	case 0x30:
	case 0x32:
	case 0x34:
	    return XOR_O8;
	case 0x31:
	case 0x33:
	case 0x35:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return XOR_O32;
	    else
		return XOR_O16;
	case 0x37:
	    return AAA;
	case 0x38:
	case 0x3a:
	case 0x3c:
	    return CMP_O8;
	case 0x39:
	case 0x3b:
	case 0x3d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMP_O32;
	    else
		return CMP_O16;

	case 0x3f:
	    return AAS;

	case 0x40:
	case 0x41:
	case 0x42:
	case 0x43:
	case 0x44:
	case 0x45:
	case 0x46:
	case 0x47:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return INC_O32;
	    else
		return INC_O16;
	case 0x48:
	case 0x49:
	case 0x4a:
	case 0x4b:
	case 0x4c:
	case 0x4d:
	case 0x4e:
	case 0x4f:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return DEC_O32;
	    else
		return DEC_O16;

	case 0x50:
	case 0x51:
	case 0x52:
	case 0x53:
	case 0x54:
	case 0x55:
	case 0x56:
	case 0x57:
	case 0x68:
	case 0x6a:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return PUSH_O16_A16;
	    case PREFICES_OPERAND:
		return PUSH_O32_A16;
	    case PREFICES_ADDRESS:
		return PUSH_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return PUSH_O32_A32;
	    }
	    break;
	case 0x58:
	case 0x59:
	case 0x5a:
	case 0x5b:
	case 0x5c:
	case 0x5d:
	case 0x5e:
	case 0x5f:
	case 0x8f:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return POP_O16_A16;
	    case PREFICES_OPERAND:
		return POP_O32_A16;
	    case PREFICES_ADDRESS:
		return POP_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return POP_O32_A32;
	    }
	    break;

	case 0x60:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return PUSHA_O16_A16;
	    case PREFICES_OPERAND:
		return PUSHA_O32_A16;
	    case PREFICES_ADDRESS:
		return PUSHA_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return PUSHA_O32_A32;
	    }
	    break;
	case 0x61:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return POPA_O16_A16;
	    case PREFICES_OPERAND:
		return POPA_O32_A16;
	    case PREFICES_ADDRESS:
		return POPA_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return POPA_O32_A32;
	    }
	    break;
	case 0x62:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return BOUND_O16_A16;
	    case PREFICES_OPERAND:
		return BOUND_O32_A16;
	    case PREFICES_ADDRESS:
		return BOUND_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return BOUND_O32_A32;
	    }
	    break;
	case 0x63:
	    break; // ARPL (not valid in real-mode)
	case 0x69:
	case 0x6b:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return IMUL_O32;
	    else
		return IMUL_O16;

	case 0x6c:
	    if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_INS_O8_A32;
		else
		    return REP_INS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return INS_O8_A32;
		else
		    return INS_O8_A16;
	    }

	case 0x6d:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_INS_O32_A32;
		    else
			return REP_INS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return INS_O32_A32;
		    else
			return INS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_INS_O16_A32;
		    else
			return REP_INS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return INS_O16_A32;
		    else
			return INS_O16_A16;
		}
	    }

	case 0x6e:
	    if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_OUTS_O8_A32;
		else
		    return REP_OUTS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return OUTS_O8_A32;
		else
		    return OUTS_O8_A16;
	    }

	case 0x6f:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_OUTS_O32_A32;
		    else
			return REP_OUTS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return OUTS_O32_A32;
		    else
			return OUTS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_OUTS_O16_A32;
		    else
			return REP_OUTS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return OUTS_O16_A32;
		    else
			return OUTS_O16_A16;
		}
	    }

	case 0x70:
	    return JO_O8;
	case 0x71:
	    return JNO_O8;
	case 0x72:
	    return JB_O8;
	case 0x73:
	    return JNB_O8;
	case 0x74:
	    return JZ_O8;
	case 0x75:
	    return JNZ_O8;
	case 0x76:
	    return JBE_O8;
	case 0x77:
	    return JNBE_O8;
	case 0x78:
	    return JS_O8;
	case 0x79:
	    return JNS_O8;
	case 0x7a:
	    return JP_O8;
	case 0x7b:
	    return JNP_O8;
	case 0x7c:
	    return JL_O8;
	case 0x7d:
	    return JNL_O8;
	case 0x7e:
	    return JLE_O8;
	case 0x7f:
	    return JNLE_O8;

	case 0x84:
	    return TEST_O8;
	case 0x85:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return TEST_O32;
	    else
		return TEST_O16;
	case 0x86:
	    return XCHG_O8;
	case 0x88:
	case 0x8a:
	case 0xa0:
	case 0xa2:
	case 0xb0:
	case 0xb1:
	case 0xb2:
	case 0xb3:
	case 0xb4:
	case 0xb5:
	case 0xb6:
	case 0xb7:
	case 0xc6:
	    return MOV_O8;
	case 0x89:
	case 0x8b:
	case 0xa1:
	case 0xa3:
	case 0xb8:
	case 0xb9:
	case 0xba:
	case 0xbb:
	case 0xbc:
	case 0xbd:
	case 0xbe:
	case 0xbf:
	case 0xc7:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return MOV_O32;
	    else
		return MOV_O16;
	case 0x8c:
	case 0x8e:
	    return MOV_O16;
	case 0x8d:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LEA_O16_A16;
	    case PREFICES_OPERAND:
		return LEA_O32_A16;
	    case PREFICES_ADDRESS:
		return LEA_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LEA_O32_A32;
	    }
	    break;
	case 0x90:
	    return NOP;
	case 0x87:
	case 0x91:
	case 0x92:
	case 0x93:
	case 0x94:
	case 0x95:
	case 0x96:
	case 0x97:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return XCHG_O32;
	    else
		return XCHG_O16;
	case 0x98:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CWDE;
	    else
		return CBW;
	case 0x99:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CDQ;
	    else
		return CWD;
	case 0x9a:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return CALLF_O16_A16;
	    case PREFICES_OPERAND:
		return CALLF_O32_A16;
	    case PREFICES_ADDRESS:
		return CALLF_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return CALLF_O32_A32;
	    }
	    break;
	case 0x9b:
	    return WAIT;
	case 0x9c:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return PUSHF_O16_A16;
	    case PREFICES_OPERAND:
		return PUSHF_O32_A16;
	    case PREFICES_ADDRESS:
		return PUSHF_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return PUSHF_O32_A32;
	    }
	    break;
	case 0x9d:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return POPF_O16_A16;
	    case PREFICES_OPERAND:
		return POPF_O32_A16;
	    case PREFICES_ADDRESS:
		return POPF_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return POPF_O32_A32;
	    }
	    break;
	case 0x9e:
	    return SAHF;
	case 0x9f:
	    return LAHF;

	case 0xa4:
	    if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_MOVS_O8_A32;
		else
		    return REP_MOVS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return MOVS_O8_A32;
		else
		    return MOVS_O8_A16;
	    }

	case 0xa5:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_MOVS_O32_A32;
		    else
			return REP_MOVS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return MOVS_O32_A32;
		    else
			return MOVS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_MOVS_O16_A32;
		    else
			return REP_MOVS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return MOVS_O16_A32;
		    else
			return MOVS_O16_A16;
		}
	    }

	case 0xa6:
	    if ((prefices & PREFICES_REPE) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REPE_CMPS_O8_A32;
		else
		    return REPE_CMPS_O8_A16;
	    } else if ((prefices & PREFICES_REPNE) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REPNE_CMPS_O8_A32;
		else
		    return REPNE_CMPS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return CMPS_O8_A32;
		else
		    return CMPS_O8_A16;
	    }

	case 0xa7:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REPE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPE_CMPS_O32_A32;
		    else
			return REPE_CMPS_O32_A16;
		} else if ((prefices & PREFICES_REPNE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPNE_CMPS_O32_A32;
		    else
			return REPNE_CMPS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CMPS_O32_A32;
		    else
			return CMPS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REPE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPE_CMPS_O16_A32;
		    else
			return REPE_CMPS_O16_A16;
		} else if ((prefices & PREFICES_REPNE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPNE_CMPS_O16_A32;
		    else
			return REPNE_CMPS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CMPS_O16_A32;
		    else
			return CMPS_O16_A16;
		}
	    }
	    
	case 0xa8:
	    return TEST_O8;
	case 0xa9:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return TEST_O32;
	    else
		return TEST_O16;

	case 0xaa:
	    if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_STOS_O8_A32;
		else
		    return REP_STOS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return STOS_O8_A32;
		else
		    return STOS_O8_A16;
	    }

	case 0xab:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_STOS_O32_A32;
		    else
			return REP_STOS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return STOS_O32_A32;
		    else
			return STOS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_STOS_O16_A32;
		    else
			return REP_STOS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return STOS_O16_A32;
		    else
			return STOS_O16_A16;
		}
	    }

	case 0xac:
	    if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_LODS_O8_A32;
		else
		    return REP_LODS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return LODS_O8_A32;
		else
		    return LODS_O8_A16;
	    }

	case 0xad:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REP) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REP_LODS_O32_A32;
		    else
			return REP_LODS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return LODS_O32_A32;
		    else
			return LODS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REP) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REP_LODS_O16_A32;
		else
		    return REP_LODS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return LODS_O16_A32;
		    else
			return LODS_O16_A16;
		}
	    }

	case 0xae:
	    if ((prefices & PREFICES_REPE) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REPE_SCAS_O8_A32;
		else
		    return REPE_SCAS_O8_A16;
	    } else if ((prefices & PREFICES_REPNE) != 0) {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return REPNE_SCAS_O8_A32;
		else
		    return REPNE_SCAS_O8_A16;
	    } else {
		if ((prefices & PREFICES_ADDRESS) != 0)
		    return SCAS_O8_A32;
		else
		    return SCAS_O8_A16;
	    }

	case 0xaf:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		if ((prefices & PREFICES_REPE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPE_SCAS_O32_A32;
		    else
			return REPE_SCAS_O32_A16;
		} else if ((prefices & PREFICES_REPNE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPNE_SCAS_O32_A32;
		    else
			return REPNE_SCAS_O32_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return SCAS_O32_A32;
		    else
			return SCAS_O32_A16;
		}
	    } else {
		if ((prefices & PREFICES_REPE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPE_SCAS_O16_A32;
		    else
			return REPE_SCAS_O16_A16;
		} else if ((prefices & PREFICES_REPNE) != 0) {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return REPNE_SCAS_O16_A32;
		    else
			return REPNE_SCAS_O16_A16;
		} else {
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return SCAS_O16_A32;
		    else
			return SCAS_O16_A16;
		}
	    }

	case 0xc0:
	case 0xd0:
	case 0xd2:
	    switch (modrm & 0x38) {
	    case 0x00:
		return ROL_O8;
	    case 0x08:
		return ROR_O8;
	    case 0x10:
		return RCL_O8;
	    case 0x18:
		return RCR_O8;
	    case 0x20:
		return SHL_O8;
	    case 0x28:
		return SHR_O8;
	    case 0x30:
		return SHL_O8;
	    case 0x38:
		return SAR_O8;
	    }
	    break;

	case 0xc1:
	case 0xd1:
	case 0xd3:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00:
		    return ROL_O32;
		case 0x08:
		    return ROR_O32;
		case 0x10:
		    return RCL_O32;
		case 0x18:
		    return RCR_O32;
		case 0x20:
		    return SHL_O32;
		case 0x28:
		    return SHR_O32;
		case 0x30:
		    return SHL_O32;
		case 0x38:
		    return SAR_O32;
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		    return ROL_O16;
		case 0x08:
		    return ROR_O16;
		case 0x10:
		    return RCL_O16;
		case 0x18:
		    return RCR_O16;
		case 0x20:
		    return SHL_O16;
		case 0x28:
		    return SHR_O16;
		case 0x30:
		    return SHL_O16;
		case 0x38:
		    return SAR_O16;
		}
	    }
	    break;

	case 0xc2:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return RET_IW_O16_A16;
	    case PREFICES_OPERAND:
		return RET_IW_O32_A16;
	    case PREFICES_ADDRESS:
		return RET_IW_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return RET_IW_O32_A32;
	    }
	    break;
	case 0xc3:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return RET_O16_A16;
	    case PREFICES_OPERAND:
		return RET_O32_A16;
	    case PREFICES_ADDRESS:
		return RET_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return RET_O32_A32;
	    }
	    break;
	case 0xc8:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return ENTER_O16_A16;
	    case PREFICES_OPERAND:
		return ENTER_O32_A16;
	    case PREFICES_ADDRESS:
		return ENTER_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return ENTER_O32_A32;
	    }
	    break;
	case 0xc9:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LEAVE_O16_A16;
	    case PREFICES_OPERAND:
		return LEAVE_O32_A16;
	    case PREFICES_ADDRESS:
		return LEAVE_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LEAVE_O32_A32;
	    }
	    break;
	case 0xca:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return RETF_IW_O16_A16;
	    case PREFICES_OPERAND:
		return RETF_IW_O32_A16;
	    case PREFICES_ADDRESS:
		return RETF_IW_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return RETF_IW_O32_A32;
	    }
	    break;
	case 0xcb:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return RETF_O16_A16;
	    case PREFICES_OPERAND:
		return RETF_O32_A16;
	    case PREFICES_ADDRESS:
		return RETF_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return RETF_O32_A32;
	    }
	    break;

	case 0xcc: //INT3 (distinct from INT 3 when not in real mode)
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return INT3_O16_A16;
	    case PREFICES_OPERAND:
		return INT3_O32_A16;
	    case PREFICES_ADDRESS:
		return INT3_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return INT3_O32_A32;
	    }
	    break;

	case 0xcd:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return INT_O16_A16;
	    case PREFICES_OPERAND:
		return INT_O32_A16;
	    case PREFICES_ADDRESS:
		return INT_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return INT_O32_A32;
	    }
	    break;
	case 0xce:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return INTO_O16_A16;
	    case PREFICES_OPERAND:
		return INTO_O32_A16;
	    case PREFICES_ADDRESS:
		return INTO_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return INTO_O32_A32;
	    }
	    break;
	case 0xcf:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return IRET_O16_A16;
	    case PREFICES_OPERAND:
		return IRET_O32_A16;
	    case PREFICES_ADDRESS:
		return IRET_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return IRET_O32_A32;
	    }
	    break;

	case 0xd4:
	    return AAM;
	case 0xd5:
	    return AAD;
	case 0xd6:
	    return SALC;
	case 0xe0:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LOOPNZ_O16_A16;
	    case PREFICES_OPERAND:
		return LOOPNZ_O32_A16;
	    case PREFICES_ADDRESS:
		return LOOPNZ_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LOOPNZ_O32_A32;
	    }
	    break;
	case 0xe1:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LOOPZ_O16_A16;
	    case PREFICES_OPERAND:
		return LOOPZ_O32_A16;
	    case PREFICES_ADDRESS:
		return LOOPZ_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LOOPZ_O32_A32;
	    }
	    break;
	case 0xe2:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LOOP_O16_A16;
	    case PREFICES_OPERAND:
		return LOOP_O32_A16;
	    case PREFICES_ADDRESS:
		return LOOP_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LOOP_O32_A32;
	    }
	    break;
	case 0xe3:
	    if ((prefices & PREFICES_ADDRESS) != 0)
		return JCXZ_A32;
	    else
		return JCXZ_A16;
	case 0xe4:
	    return IN_O8_O8;
	case 0xec:
	    return IN_O8_O16;
	case 0xe5:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return IN_O32_O8;
	    else
		return IN_O16_O8;
	case 0xed:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return IN_O32_O16;
	    else
		return IN_O16_O16;
	case 0xe6:
	    return OUT_O8_O8;
	case 0xee:
	    return OUT_O16_O8;
	case 0xe7:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return OUT_O8_O32;
	    else
		return OUT_O8_O16;
	case 0xef:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return OUT_O16_O32;
	    else
		return OUT_O16_O16;
	case 0xe8:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return CALL_O16_A16;
	    case PREFICES_OPERAND:
		return CALL_O32_A16;
	    case PREFICES_ADDRESS:
		return CALL_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return CALL_O32_A32;
	    }
	    break;
	case 0xe9:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JMP_O32_NEAR_RELATIVE;
	    else
		return JMP_O16_NEAR_RELATIVE;
	case 0xea:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JMP_O32_FAR;
	    else
		return JMP_O16_FAR;
	case 0xeb:
	    return JMP_O8_SHORT;

	case 0xf1: // INT1 (undocumented must check behaviour)
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return INT1_O16_A16;
	    case PREFICES_OPERAND:
		return INT1_O32_A16;
	    case PREFICES_ADDRESS:
		return INT1_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return INT1_O32_A32;
	    }
	    break;
	case 0xf4:
	    return HLT;
	case 0xf5:
	    return CMC;
	case 0xf6:
	    switch (modrm & 0x38) {
	    case 0x00:
		return TEST_O8;
	    case 0x10:
		return NOT_O8;
	    case 0x18:
		return NEG_O8;
	    case 0x20:
		return MUL_O8;
	    case 0x28:
		return IMUL_REGA_O8;
	    case 0x30:
		return DIV_O8;
	    case 0x38:
		return IDIV_O8;
	    }
	    return -1;
	case 0xf7:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00:
		    return TEST_O32;
		case 0x10:
		    return NOT_O32;
		case 0x18:
		    return NEG_O32;
		case 0x20:
		    return MUL_O32;
		case 0x28:
		    return IMUL_REGA_O32;
		case 0x30:
		    return DIV_O32;
		case 0x38:
		    return IDIV_O32;
		}
		return -1;
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		    return TEST_O16;
		case 0x10:
		    return NOT_O16;
		case 0x18:
		    return NEG_O16;
		case 0x20:
		    return MUL_O16;
		case 0x28:
		    return IMUL_REGA_O16;
		case 0x30:
		    return DIV_O16;
		case 0x38:
		    return IDIV_O16;
		}
		return -1;
	    }

	case 0xf8:
	    return CLC;
	case 0xf9:
	    return STC;
	case 0xfa:
	    return CLI;
	case 0xfb:
	    return STI;
	case 0xfc:
	    return CLD;
	case 0xfd:
	    return STD;

	case 0xfe:
	    switch (modrm & 0x38) {
	    case 0x00:
		return INC_O8;
	    case 0x08:
		return DEC_O8;
	    }
	    return -1;

	case 0xd7:
	    return XLAT;

	case 0x06:
	case 0x16:
	case 0x0e:
	case 0x1e:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return PUSH_O16_A16;
	    case PREFICES_OPERAND:
		return PUSH_O32_A16;
	    case PREFICES_ADDRESS:
		return PUSH_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return PUSH_O32_A32;
	    }
	    break;

	case 0x07:
	case 0x17:
	case 0x1f:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return POP_O16_A16;
	    case PREFICES_OPERAND:
		return POP_O32_A16;
	    case PREFICES_ADDRESS:
		return POP_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return POP_O32_A32;
	    }
	    break;

	case 0x80:
	case 0x82:
	    switch (modrm & 0x38) {
	    case 0x00:
		return ADD_O8;
	    case 0x08:
		return OR_O8;
	    case 0x10:
		return ADC_O8;
	    case 0x18:
		return SBB_O8;
	    case 0x20:
		return AND_O8;
	    case 0x28:
		return SUB_O8;
	    case 0x30:
		return XOR_O8;
	    case 0x38:
		return CMP_O8;
	    }
	    break;

	case 0x81:
	case 0x83:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00:
		    return ADD_O32;
		case 0x08:
		    return OR_O32;
		case 0x10:
		    return ADC_O32;
		case 0x18:
		    return SBB_O32;
		case 0x20:
		    return AND_O32;
		case 0x28:
		    return SUB_O32;
		case 0x30:
		    return XOR_O32;
		case 0x38:
		    return CMP_O32;
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		    return ADD_O16;
		case 0x08:
		    return OR_O16;
		case 0x10:
		    return ADC_O16;
		case 0x18:
		    return SBB_O16;
		case 0x20:
		    return AND_O16;
		case 0x28:
		    return SUB_O16;
		case 0x30:
		    return XOR_O16;
		case 0x38:
		    return CMP_O16;
		}
	    }
	    break;

	case 0xff:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00:
		    return INC_O32;
		case 0x08:
		    return DEC_O32;
		case 0x10:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CALLN_O32_A32;
		    else
			return CALLN_O32_A16;
		case 0x18:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CALLF_O32_A32;
		    else
			return CALLF_O32_A16;
		case 0x20:
		    return JMP_O32_NEAR_ABSOLUTE;
		case 0x28:
		    return JMP_O32_FAR;
		case 0x30:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return PUSH_O32_A32;
		    else
			return PUSH_O32_A16;
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		    return INC_O16;
		case 0x08:
		    return DEC_O16;
		case 0x10:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CALLN_O16_A32;
		    else
			return CALLN_O16_A16;
		case 0x18:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return CALLF_O16_A32;
		    else
			return CALLF_O16_A16;
		case 0x20:
		    return JMP_O16_NEAR_ABSOLUTE;
		case 0x28:
		    return JMP_O16_FAR;
		case 0x30:
		    if ((prefices & PREFICES_ADDRESS) != 0)
			return PUSH_O16_A32;
		    else
			return PUSH_O16_A16;
		}
	    }
	    break;

	case 0xc4:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LES_O16_A16;
	    case PREFICES_OPERAND:
		return LES_O32_A16;
	    case PREFICES_ADDRESS:
		return LES_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LES_O32_A32;
	    }
	    break;

	case 0xc5:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LDS_O16_A16;
	    case PREFICES_OPERAND:
		return LDS_O32_A16;
	    case PREFICES_ADDRESS:
		return LDS_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LDS_O32_A32;
	    }
	    break;

	default:
	    return -1;
	}
	return -1;
    }

    private static int decodeTwoByteOperation(int prefices, int opcode, int modrm)
    {
	switch (opcode & 0xff) {
	case 0x01:
	    switch (modrm & 0x38) {
	    case 0x00:
		if ((prefices & PREFICES_OPERAND) != 0)
		    return SGDT_O32;
		else
		    return SGDT_O16;
	    case 0x08:
		if ((prefices & PREFICES_OPERAND) != 0)
		    return SIDT_O32;
		else
		    return SIDT_O16;
	    case 0x10:
		if ((prefices & PREFICES_OPERAND) != 0)
		    return LGDT_O32;
		else
		    return LGDT_O16;
	    case 0x18:
		if ((prefices & PREFICES_OPERAND) != 0)
		    return LIDT_O32;
		else
		    return LIDT_O16;
	    case 0x20:
		return SMSW;
	    case 0x30:
		return LMSW;
	    case 0x38:
		return INVLPG;
	    default:
		return -1;
	    }
	case 0x06:
	    return CLTS;

	case 0x20:
	    return MOV_O32;
	case 0x22:
	    return MOV_TO_CR_O32;

	case 0x40:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVO_O32;
	    else
		return CMOVO_O16;
	case 0x41:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNO_O32;
	    else
		return CMOVNO_O16;
	case 0x42:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVB_O32;
	    else
		return CMOVB_O16;
	case 0x43:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNB_O32;
	    else
		return CMOVNB_O16;
	case 0x44:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVZ_O32;
	    else
		return CMOVZ_O16;
	case 0x45:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNZ_O32;
	    else
		return CMOVNZ_O16;
	case 0x46:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVBE_O32;
	    else
		return CMOVBE_O16;
	case 0x47:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNBE_O32;
	    else
		return CMOVNBE_O16;
	case 0x48:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVS_O32;
	    else
		return CMOVS_O16;
	case 0x49:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNS_O32;
	    else
		return CMOVNS_O16;
	case 0x4a:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVP_O32;
	    else
		return CMOVP_O16;
	case 0x4b:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNP_O32;
	    else
		return CMOVNP_O16;
	case 0x4c:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVL_O32;
	    else
		return CMOVL_O16;
	case 0x4d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNL_O32;
	    else
		return CMOVNL_O16;
	case 0x4e:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVLE_O32;
	    else
		return CMOVLE_O16;
	case 0x4f:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMOVNLE_O32;
	    else
		return CMOVNLE_O16;

	case 0x80:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JO_O32;
	    else
		return JO_O16;
	case 0x81:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNO_O32;
	    else
		return JNO_O16;
	case 0x82:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JB_O32;
	    else
		return JB_O16;
	case 0x83:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNB_O32;
	    else
		return JNB_O16;
	case 0x84:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JZ_O32;
	    else
		return JZ_O16;
	case 0x85:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNZ_O32;
	    else
		return JNZ_O16;
	case 0x86:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JBE_O32;
	    else
		return JBE_O16;
	case 0x87:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNBE_O32;
	    else
		return JNBE_O16;
	case 0x88:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JS_O32;
	    else
		return JS_O16;
	case 0x89:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNS_O32;
	    else
		return JNS_O16;
	case 0x8a:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JP_O32;
	    else
		return JP_O16;
	case 0x8b:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNP_O32;
	    else
		return JNP_O16;
	case 0x8c:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JL_O32;
	    else
		return JL_O16;
	case 0x8d:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNL_O32;
	    else
		return JNL_O16;
	case 0x8e:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JLE_O32;
	    else
		return JLE_O16;
	case 0x8f:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return JNLE_O32;
	    else
		return JNLE_O16;

	case 0x90:
	    return SETO;
	case 0x91:
	    return SETNO;
	case 0x92:
	    return SETB;
	case 0x93:
	    return SETNB;
	case 0x94:
	    return SETZ;
	case 0x95:
	    return SETNZ;
	case 0x96:
	    return SETBE;
	case 0x97:
	    return SETNBE;
	case 0x98:
	    return SETS;
	case 0x99:
	    return SETNS;
	case 0x9a:
	    return SETP;
	case 0x9b:
	    return SETNP;
	case 0x9c:
	    return SETL;
	case 0x9d:
	    return SETNL;
	case 0x9e:
	    return SETLE;
	case 0x9f:
	    return SETNLE;

	case 0xa0:
	case 0xa8:
	    if ((prefices & PREFICES_ADDRESS) != 0)
		return PUSH_O16_A32;
	    else
		return PUSH_O16_A16;
	case 0xa1:
	case 0xa9:
	    if ((prefices & PREFICES_ADDRESS) != 0)
		return POP_O16_A32;
	    else
		return POP_O16_A16;
	case 0xa2:
	    return CPUID;
	case 0xa3:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BT_O32;
	    else
		return BT_O16;
	case 0xa4:
	case 0xa5:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return SHLD_O32;
	    else
		return SHLD_O16;
	case 0xab:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BTS_O32;
	    else
		return BTS_O16;
	case 0xac:
	case 0xad:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return SHRD_O32;
	    else
		return SHRD_O16;
	case 0xaf:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return IMUL_O32;
	    else
		return IMUL_O16;
	case 0xb0:
	    return CMPXCHG_O8;
	case 0xb1:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return CMPXCHG_O32;
	    else
		return CMPXCHG_O16;
	case 0xb2:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LSS_O16_A16;
	    case PREFICES_OPERAND:
		return LSS_O32_A16;
	    case PREFICES_ADDRESS:
		return LSS_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LSS_O32_A32;
	    }
	    break;

	case 0xb3:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BTR_O32;
	    else
		return BTR_O16;
	case 0xb4:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LFS_O16_A16;
	    case PREFICES_OPERAND:
		return LFS_O32_A16;
	    case PREFICES_ADDRESS:
		return LFS_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LFS_O32_A32;
	    }
	    break;

	case 0xb5:
	    switch (prefices & (PREFICES_OPERAND | PREFICES_ADDRESS)) {
	    case 0:
		return LGS_O16_A16;
	    case PREFICES_OPERAND:
		return LGS_O32_A16;
	    case PREFICES_ADDRESS:
		return LGS_O16_A32;
	    case PREFICES_ADDRESS | PREFICES_OPERAND:
		return LGS_O32_A32;
	    }
	    break;

	case 0xb6:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return MOVZX_O32_O8;
	    else
		return MOVZX_O16_O8;
	case 0xb7:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return MOVZX_O32_O16;
	    else
		return MOVZX_O16_O16;
	case 0xba: //Group 8, BitTest
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00:
		case 0x08:
		case 0x10:
		case 0x18:
		default:
		    return -1;
		case 0x20:
		    return BT_O32_O8;
		case 0x28:
		    return BTS_O32_O8;
		case 0x30:
		    return BTR_O32_O8;
		case 0x38:
		    return BTC_O32_O8;
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		case 0x08:
		case 0x10:
		case 0x18:
		default:
		    return -1;
		case 0x20:
		    return BT_O16_O8;
		case 0x28:
		    return BTS_O16_O8;
		case 0x30:
		    return BTR_O16_O8;
		case 0x38:
		    return BTC_O16_O8;
		}
	    }
	case 0xbb:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BTC_O32;
	    else
		return BTC_O16;
	case 0xbc:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BSF_O32;
	    else
		return BSF_O16;
	case 0xbd:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return BSR_O32;
	    else
		return BSR_O16;
	case 0xbe:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return MOVSX_O32_O8;
	    else
		return MOVSX_O16_O8;
	case 0xbf:
	    if ((prefices & PREFICES_OPERAND) != 0)
		return MOVSX_O32_O16;
	    else
		return MOVSX_O16_O16;
	}
	return -1;
    }

    private static int decodeFPUOperation(int prefices, int opcode, int modrm)
    {
	if ((modrm & 0xc0) != 0xc0) {
	    switch (opcode & 0xff38) { //operations using the reg portion of modrm for opcode encoding
	    case 0xd800: return FADD_SR;
	    case 0xd808: return FMUL_SR;
	    case 0xd810: return FCOM_SR;
	    case 0xd818: return FCOMP_SR;
	    case 0xd820: return FSUB_SR;
	    case 0xd828: return FSUBR_SR;
	    case 0xd830: return FDIV_SR;
	    case 0xd838: return FDIVR_SR;

	    case 0xd900: return FLD_SR;
	    case 0xd910: return FST_SR;
	    case 0xd918: return FSTP_SR;
	    case 0xd920:
		if ((prefices & PREFICES_OPERAND) == 0)
		    return FNLDENV_14;
		else
		    return FNLDENV_28;
	    case 0xd928: return FLDCW;
	    case 0xd930:
		if ((prefices & PREFICES_OPERAND) == 0)
		    return FNSTENV_14;
		else
		    return FNSTENV_28;
	    case 0xd938: return FSTCW;

	    case 0xda00: return FIADD_DI;
	    case 0xda08: return FIMUL_DI;
	    case 0xda10: return FICOM_DI;
	    case 0xda18: return FICOMP_DI;
	    case 0xda20: return FISUB_DI;
	    case 0xda28: return FISUBR_DI;
	    case 0xda30: return FIDIV_DI;
	    case 0xda38: return FIDIVR_DI;

	    case 0xdb00: return FILD_DI;
	    case 0xdb08: return FISTTP_DI;
	    case 0xdb10: return FIST_DI;
	    case 0xdb18: return FISTP_DI;
	    case 0xdb28: return FLD_XR;
	    case 0xdb38: return FSTP_XR;

	    case 0xdc00: return FADD_DR;
	    case 0xdc08: return FMUL_DR;
	    case 0xdc10: return FCOM_DR;
	    case 0xdc18: return FCOMP_DR;
	    case 0xdc20: return FSUB_DR;
	    case 0xdc28: return FSUBR_DR;
	    case 0xdc30: return FDIV_DR;
	    case 0xdc38: return FDIVR_DR;

	    case 0xdd00: return FLD_DR;
	    case 0xdd08: return FISTTP_QI;
	    case 0xdd10: return FST_DR;
	    case 0xdd18: return FSTP_DR;
	    case 0xdd20: return FRSTOR_98108;
	    case 0xdd30: return FSAVE_98108;
	    case 0xdd38: return FSTSW;

	    case 0xde00: return FIADD_WI;
	    case 0xde08: return FIMUL_WI;
	    case 0xde10: return FICOM_WI;
	    case 0xde18: return FICOMP_WI;
	    case 0xde20: return FISUB_WI;
	    case 0xde28: return FISUBR_WI;
	    case 0xde30: return FIDIV_WI;
	    case 0xde38: return FIDIVR_WI;

	    case 0xdf00: return FILD_WI;
	    case 0xdf08: return FISTTP_WI;
	    case 0xdf10: return FIST_WI;
	    case 0xdf18: return FISTP_WI;
	    case 0xdf20: return FBLD;
	    case 0xdf28: return FILD_QI;
	    case 0xdf30: return FBSTP;
	    case 0xdf38: return FISTP_QI;
	    }
	} else {
	    switch (opcode & 0xfff8) { //operations that act on two fpu stack operands
	    case 0xd8c0: return FADD;
	    case 0xd8c8: return FMUL;
	    case 0xd8d0: return FCOM;
	    case 0xd8d8: return FCOMP;
	    case 0xd8e0: return FSUB;
	    case 0xd8e8: return FSUBR;
	    case 0xd8f0: return FDIV;
	    case 0xd8f8: return FDIVR;

	    case 0xd9c0: return FLD;
	    case 0xd9c8: return FXCH;

	    case 0xdac0: return FCMOVB;
	    case 0xdac8: return FCMOVE;
	    case 0xdad0: return FCMOVBE;
	    case 0xdad8: return FCMOVU;

	    case 0xdbc0: return FCMOVNB;
	    case 0xdbc8: return FCMOVNE;
	    case 0xdbd0: return FCMOVNBE;
	    case 0xdbd8: return FCMOVNU;
	    case 0xdbe8: return FUCOMI;
	    case 0xdbf0: return FCOMI;

	    case 0xdcc0: return FADD;
	    case 0xdcc8: return FMUL;
	    case 0xdce0: return FSUBR;
	    case 0xdce8: return FSUB;
	    case 0xdcf0: return FDIVR;
	    case 0xdcf8: return FDIV;

	    case 0xddc0: return FFREE;
	    case 0xddd0: return FST;
	    case 0xddd8: return FSTP;
	    case 0xdde0: return FUCOM;
	    case 0xdde8: return FUCOMP;

	    case 0xdec0: return FADDP;
	    case 0xdec8: return FMULP;
	    case 0xdee0: return FSUBRP;
	    case 0xdee8: return FSUBP;
	    case 0xdef0: return FDIVRP;
	    case 0xdef8: return FDIVP;

	    case 0xdfe8: return FUCOMIP;
	    case 0xdff0: return FCOMIP;
	    }

	    switch (opcode) { //operations that use the whole opcode + modrm combination to encode themselves
	    case 0xd9d0: return FNOP;

	    case 0xd9e0: return FCHS;
	    case 0xd9e1: return FABS;
	    case 0xd9e4: return FTST;
	    case 0xd9e5: return FXAM;
	    case 0xd9e8: return FLD1;
	    case 0xd9e9: return FLDL2T;
	    case 0xd9ea: return FLDL2E;
	    case 0xd9eb: return FLDPI;
	    case 0xd9ec: return FLDLG2;
	    case 0xd9ed: return FLDLN2;
	    case 0xd9ee: return FLDZ;

	    case 0xd9f0: return F2XM1;
	    case 0xd9f1: return FYL2X;
	    case 0xd9f2: return FPTAN;
	    case 0xd9f3: return FPATAN;
	    case 0xd9f4: return FXTRACT;
	    case 0xd9f5: return FPREM1;
	    case 0xd9f6: return FDECSTP;
	    case 0xd9f7: return FINCSTP;
	    case 0xd9f8: return FPREM;
	    case 0xd9f9: return FYL2XP1;
	    case 0xd9fa: return FSQRT;
	    case 0xd9fb: return FSINCOS;
	    case 0xd9fc: return FRNDINT;
	    case 0xd9fd: return FSCALE;
	    case 0xd9fe: return FSIN;
	    case 0xd9ff: return FCOS;

	    case 0xdae9: return FUCOMPP;

	    case 0xdbe2: return FCLEX;
	    case 0xdbe3: return FNINIT;
	    case 0xdbe4: return NOP; //kernel comment says: fsetpm for 287, ignored by 387

	    case 0xded9: return FCOMPP;

	    case 0xdfe0: return FSTSW;
	    }

	}
	System.err.println("Unknown FPU Operation 0x" + Integer.toHexString(0xffff & opcode) + " [modrm:0x" + Integer.toHexString(modrm) + "]");
	return -1;
    }

    private static int decodeOperand(int index, int prefices, int opcode, int modrm, int sib)
    {
	if ((opcode & 0xff00) == 0x0f00)
	    return decodeTwoByteOperand(index, prefices, opcode, modrm, sib);
	
	if ((opcode & 0xf800) == 0xd800)
	    return decodeFPUOperand(index, prefices, opcode, modrm, sib);

	//Normal One Byte Operation
	switch (opcode) {
	case 0x00: //ADD Eb, Gb
	case 0x08: //OR  Eb, Gb
	case 0x10: //ADC Eb, Gb
	case 0x18: //SBB Eb, Gb
	case 0x20: //AND Eb, Gb
	case 0x28: //SUB Eb, Gb
	case 0x30: //XOR Eb, Gb

	case 0x38: //CMP  Eb, Gb
	case 0x84: //TEST Eb, Gb
	case 0x88: //MOV  Eb, Gb
	    switch (index) {
	    case 0:
		return decodeGb(modrm);
	    case 1:
		return decodeEb(prefices, modrm, sib);
	    case 2:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0x86: //XCHG Eb, Gb
	    switch (index) {
	    case 0:
		return decodeGb(modrm);
	    case 1:
		return decodeEb(prefices, modrm, sib);
	    case 2:
		return decodeGb(modrm);
	    case 3:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }



	case 0x02: //ADD Gb, Eb
	case 0x0a: //OR  Gb, Eb
	case 0x12: //ADC Gb, Eb
	case 0x1a: //SBB Gb, Eb
	case 0x22: //AND Gb, Eb
	case 0x2a: //SUB Gb, Eb
	case 0x32: //XOR Gb, Eb
	case 0x3a: //CMP Gb, Eb
	case 0x8a: //MOV Gb, Eb
	    switch (index) {
	    case 0:
		return decodeEb(prefices, modrm, sib);
	    case 1:
		return decodeGb(modrm);
	    case 2:
		return decodeGb(modrm);
	    default:
		return -1;
	    }

	case 0x01: //ADD Ev, Gv
	case 0x09: //OR  Ev, Gv
	case 0x11: //ADC Ev, Gv
	case 0x19: //SBB Ev, Gv
	case 0x21: //AND Ev, Gv
	case 0x29: //SUB Ev, Gv
	case 0x31: //XOR Ev, Gv
	case 0x39: //CMP  Ev, Gv
	case 0x85: //TEST Ev, Gv
	case 0x89: //MOV  Ev, Gv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0x87: //XCHG Ev, Gv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeGd(modrm);
		case 3:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeGw(modrm);
		case 3:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0x03: //ADD Gv, Ev
	case 0x0b: //OR  Gv, Ev
	case 0x13: //ADC Gv, Ev
	case 0x1b: //SBB Gv, Ev
	case 0x23: //AND Gv, Ev
	case 0x2b: //SUB Gv, Ev
	case 0x33: //XOR Gv, Ev
	case 0x3b: //CMP Gv, Ev
	case 0x8b: //MOV Gv, Ev
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		case 2:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		case 2:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x8d: //LEA Gv, M
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x80: //IMM G1 Eb, Ib
	case 0x82: //IMM G1 Eb, Ib
	case 0xc0: //SFT G2 Eb, Ib
	case 0xc6: //MOV G11 Eb, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return decodeEb(prefices, modrm, sib);
	    case 2:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0x81: //IMM G1 Ev, Iv
	case 0xc7: //MOV G11 Ev, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0x83: //IMM G1 Ev, Ib
	case 0xc1: //SFT G2 Ev, Ib
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0x8f: //POP Ev
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return decodeEd(prefices, modrm, sib);
	    } else {
		return decodeEw(prefices, modrm, sib);
	    }
	    
	case 0xc2: //RET Iw
	case 0xca: //RETF Iw
	    return IW;

	case 0x68: //PUSH Iv
	case 0xe8: //CALL Jv
	case 0xe9: //JMP  Jv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return ID;
	    } else {
		return IW;
	    }

	case 0x9a: //CALLF Ap
	case 0xea: //JMPF Ap
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return IA48;
	    } else {
		return IA32;
	    }

	case 0xec: //IN AL, DX
	    switch (index) {
	    case 0:
		return DX;
	    case 1:
		return AL;
	    default:
		return -1;
	    }

	case 0xee: //OUT DX, AL
	    switch (index) {
	    case 0:
		return AL;
	    case 1:
		return DX;
	    default:
		return -1;
	    }

	case 0xed: //IN eAX, DX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0xef: //OUT DX, eAX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EAX;
		case 1:
		    return DX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return DX;
		default:
		    return -1;
		}
	    }

	case 0x04: //ADD AL, Ib
	case 0x0c: //OR  AL, Ib
	case 0x14: //ADC AL, Ib
	case 0x1c: //SBB AL, Ib
	case 0x24: //AND AL, Ib
	case 0x2c: //SUB AL, Ib
	case 0x34: //XOR AL, Ib
	case 0x3c: //CMP AL, Ib
	case 0xe4: //IN  AL, Ib
	case 0xa8: //TEST AL, Ib
	case 0xb0: //MOV AL, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return AL;
	    case 2:
		return AL;
	    default:
		return -1;
	    }

	case 0xb1: //MOV CL, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return CL;
	    default:
		return -1;
	    }

	case 0xb2: //MOV DL, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return DL;
	    default:
		return -1;
	    }

	case 0xb3: //MOV BL, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return BL;
	    default:
		return -1;
	    }

	case 0xb4: //MOV AH, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return AH;
	    default:
		return -1;
	    }

	case 0xb5: //MOV CH, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return CH;
	    default:
		return -1;
	    }

	case 0xb6: //MOV DH, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return DH;
	    default:
		return -1;
	    }

	case 0xb7: //MOV BH, Ib
	    switch (index) {
	    case 0:
		return IB;
	    case 1:
		return BH;
	    default:
		return -1;
	    }

	case 0xc8: //ENTER Iw, Ib
	    return IWIB;

	case 0x69: //IMUL Gv, Ev, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return ID;
		case 2:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return IW;
		case 2:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x6b: //IMUL Gv, Ev, Ib
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return IB;
		case 2:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return IB;
		case 2:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x6a: //PUSH Ib
	case 0x70: //Jcc Jb
	case 0x71:
	case 0x72:
	case 0x73:
	case 0x74:
	case 0x75:
	case 0x76:
	case 0x77:
	case 0x78:
	case 0x79:
	case 0x7a:
	case 0x7b:
	case 0x7c:
	case 0x7d:
	case 0x7e:
	case 0x7f:
	case 0xcd: //INT Ib
	case 0xd4: //AAM Ib
	case 0xd5: //AAD Ib
	case 0xe0: //LOOPNZ Jb
	case 0xe1: //LOOPZ Jb
	case 0xe2: //LOOP Jb
	case 0xe3: //JCXZ Jb
	case 0xeb: //JMP Jb
	    return IB;

	case 0xe6: //OUT Ib, AL
	    switch (index) {
	    case 0:
		return AL;
	    case 1:
		return IB;
	    default:
		return -1;
	    }

	case 0x05: //ADD eAX, Iv
	case 0x0d: //OR  eAX, Iv
	case 0x15: //ADC eAX, Iv
	case 0x1d: //SBB eAX, Iv
	case 0x25: //AND eAX, Iv
	case 0x2d: //SUB eAX, Iv
	case 0x35: //XOR eAX, Iv
	case 0x3d: //CMP eAX, Iv
	case 0xa9: //TEST eAX, Iv
	case 0xb8: //MOV eAX, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return EAX;
		case 2:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return AX;
		case 2:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0xb9: //MOV eCX, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return ECX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return CX;
		default:
		    return -1;
		}
	    }

	case 0xba: //MOV eDX, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return EDX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return DX;
		default:
		    return -1;
		}
	    }

	case 0xbb: //MOV eBX, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return EBX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return BX;
		default:
		    return -1;
		}
	    }

	case 0xbc: //MOV eSP, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return ESP;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return SP;
		default:
		    return -1;
		}
	    }

	case 0xbd: //MOV eBP, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return EBP;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return BP;
		default:
		    return -1;
		}
	    }

	case 0xbe: //MOV eSI, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return ESI;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return SI;
		default:
		    return -1;
		}
	    }

	case 0xbf: //MOV eDI, Iv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ID;
		case 1:
		    return EDI;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IW;
		case 1:
		    return DI;
		default:
		    return -1;
		}
	    }

	case 0xe5: //IN eAX, Ib
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return AX;
		default:
		    return -1;
		}
	    }
	case 0xe7: //OUT Ib, eAX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EAX;
		case 1:
		    return IB;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return IB;
		default:
		    return -1;
		}
	    }

	case 0x40: //INC eAX
	case 0x48: //DEC eAX
	case 0x50: //PUSH eAX
	case 0x58: //POP eAX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return EAX;
	    } else {
		return AX;
	    }

	case 0x41: //INC eCX	
	case 0x49: //DEC eCX
	case 0x51: //PUSH eCX
	case 0x59: //POP eCX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return ECX;
	    } else {
		return CX;
	    }

	case 0x42: //INC eDX	
	case 0x4a: //DEC eDX
	case 0x52: //PUSH eDX
	case 0x5a: //POP eDX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return EDX;
	    } else {
		return DX;
	    }

	case 0x43: //INC eBX
	case 0x4b: //DEC eBX
	case 0x53: //PUSH eBX
	case 0x5b: //POP eBX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return EBX;
	    } else {
		return BX;
	    }

	case 0x44: //INC eSP	
	case 0x4c: //DEC eSP
	case 0x54: //PUSH eSP
	case 0x5c: //POP eSP
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return ESP;
	    } else {
		return SP;
	    }

	case 0x45: //INC eBP		
	case 0x4d: //DEC eBP
	case 0x55: //PUSH eBP
	case 0x5d: //POP eBP
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return EBP;
	    } else {
		return BP;
	    }

	case 0x46: //INC eSI	
	case 0x4e: //DEC eSI
	case 0x56: //PUSH eSI
	case 0x5e: //POP eSI
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return ESI;
	    } else {
		return SI;
	    }

	case 0x47: //INC eDI		
	case 0x4f: //DEC eDI
	case 0x57: //PUSH eDI
	case 0x5f: //POP eDI
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return EDI;
	    } else {
		return DI;
	    }

	case 0x91: //XCHG eAX, eCX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ECX;
		case 1:
		    return EAX;
		case 2:
		    return ECX;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return CX;
		case 1:
		    return AX;
		case 2:
		    return CX;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0x92: //XCHG eAX, eDX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EDX;
		case 1:
		    return EAX;
		case 2:
		    return EDX;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return DX;
		case 2:
		    return AX;
		case 3:
		    return DX;
		default:
		    return -1;
		}
	    }

	case 0x93: //XCHG eAX, eBX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EBX;
		case 1:
		    return EAX;
		case 2:
		    return EBX;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return BX;
		case 1:
		    return AX;
		case 2:
		    return BX;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0x94: //XCHG eAX, eSP
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ESP;
		case 1:
		    return EAX;
		case 2:
		    return ESP;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return SP;
		case 1:
		    return AX;
		case 2:
		    return SP;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0x95: //XCHG eAX, eBP
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EBP;
		case 1:
		    return EAX;
		case 2:
		    return EBP;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return BP;
		case 1:
		    return AX;
		case 2:
		    return BP;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0x96: //XCHG eAX, eSI
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return ESI;
		case 1:
		    return EAX;
		case 2:
		    return ESI;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return SI;
		case 1:
		    return AX;
		case 2:
		    return SI;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0x97: //XCHG eAX, eDI
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EDI;
		case 1:
		    return EAX;
		case 2:
		    return EDI;
		case 3:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return DI;
		case 1:
		    return AX;
		case 2:
		    return DI;
		case 3:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0xd0: //SFT G2 Eb, 1
	    switch (index) {
	    case 0:
		return CONSTANT_1;
	    case 1:
		return decodeEb(prefices, modrm, sib);
	    case 2:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0xd2: //SFT G2 Eb, CL
	    switch (index) {
	    case 0:
		return CL;
	    case 1:
		return decodeEb(prefices, modrm, sib);
	    case 2:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0xd1: //SFT G2 Ev, 1
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return CONSTANT_1;
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return CONSTANT_1;
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0xd3: //SFT G2 Ev, CL
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return CL;
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return CL;
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0xf6: //UNA G3 Eb, ?
	    switch (modrm & 0x38) {
	    case 0x00: //TEST Eb, Ib
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return decodeEb(prefices, modrm, sib);
		default:
		    return -1;
		}
	    default:
		return decodeEb(prefices, modrm, sib);
	    }

	case 0xf7: //UNA G3 Ev, ?
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00: //TEST Ed, Id
		    switch (index) {
		    case 0:
			return ID;
		    case 1:
			return decodeEd(prefices, modrm, sib);
		    default:
			return -1;
		    }
		default:
		    return decodeEd(prefices, modrm, sib);
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00: //TEST Ew, Iw
		    switch (index) {
		    case 0:
			return IW;
		    case 1:
			return decodeEw(prefices, modrm, sib);
		    default:
			return -1;
		    }
		default:
		    return decodeEw(prefices, modrm, sib);
		}
	    }

	case 0xfe: //INC/DEC G4 Eb
	    return decodeEb(prefices, modrm, sib);

	case 0x06: //PUSH ES
	case 0x07: //POP ES
	    return ES;

	case 0x0e: //PUSH CS
	    return CS;

	case 0x16: //PUSH SS
	case 0x17: //POP SS
	    return SS;

	case 0x1e: //PUSH DS
	case 0x1f: //POP DS
	    return DS;

	case 0x62: //BOUND Gv, Ma
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x8c: //MOV Ew, Sw
		switch (index) {
		case 0:
		    return decodeSw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	case 0x8e: //MOV Sw, Ew
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeSw(modrm);
		default:
		    return -1;
		}

	case 0xa0: //MOV AL, Ob
	    switch (index) {
	    case 0:
		return decodeOb(prefices);
	    case 1:
		return AL;
	    default:
		return -1;
	    }
	    
	case 0xa2: //MOV Ob, AL
	    switch (index) {
	    case 0:
		return AL;
	    case 1:
		return decodeOb(prefices);
	    default:
		return -1;
	    }

	case 0xa1: //MOV eAX, Ov
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeOd(prefices);
		case 1:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeOw(prefices);
		case 1:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0xa3: //MOV Ov, eAX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EAX;
		case 1:
		    return decodeOd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return decodeOw(prefices);
		default:
		    return -1;
		}
	    }

	case 0x6c: //INS Yb, DX (prefices do not override segment)
	    switch (index) {
	    case 0:
		return DX;
	    case 1:
		return decodeYb(prefices);
	    default:
		return -1;
	    }

	case 0x6d: //INS Yv, DX (prefices do not override segment)
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return decodeYd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return decodeYw(prefices);
		default:
		    return -1;
		}
	    }

	case 0x6e: //OUTS DX, Xb
	    switch (index) {
	    case 0:
		return DX;
	    case 1:
		return decodeXb(prefices);
	    default:
		return -1;
	    }

	case 0x6f: //OUTS DX, Xv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return decodeXd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return DX;
		case 1:
		    return decodeXw(prefices);
		default:
		    return -1;
		}
	    }

	case 0xa4: //MOVS Yb, Xb
	case 0xa6: //CMPS Yb, Xb
	    switch (index) {
	    case 0:
		return decodeXb(prefices);
	    case 1:
		return decodeYb(prefices);
	    default:
		return -1;
	    }

	case 0xa5: //MOVS Yv, Xv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeXd(prefices);
		case 1:
		    return decodeYd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeXw(prefices);
		case 1:
		    return decodeYw(prefices);
		default:
		    return -1;
		}
	    }

	case 0xa7: //CMPS Xv, Yv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeYd(prefices);
		case 1:
		    return decodeXd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeYw(prefices);
		case 1:
		    return decodeXw(prefices);
		default:
		    return -1;
		}
	    }

	case 0xaa: //STOS Yb, AL (prefices do not override segment)
	    switch (index) {
	    case 0:
		return AL;
	    case 1:
		return decodeYb(prefices);
	    default:
		return -1;
	    }

	case 0xab: //STOS Yv, eAX
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EAX;
		case 1:
		    return decodeYd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return decodeYw(prefices);
		default:
		    return -1;
		}
	    }

	case 0xac: //LODS AL, Xb
	    switch (index) {
	    case 0:
		return decodeXb(prefices);
	    case 1:
		return AL;
	    default:
		return -1;
	    }

	case 0xad: //LODS eAX, Xv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeXd(prefices);
		case 1:
		    return EAX;
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeXw(prefices);
		case 1:
		    return AX;
		default:
		    return -1;
		}
	    }

	case 0xae: //SCAS AL, Yb (prefices do not override segment)
	    switch (index) {
	    case 0:
		return AL;
	    case 1:
		return decodeYb(prefices);
	    default:
		return -1;
	    }

	case 0xaf: //SCAS eAX, Yv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return EAX;
		case 1:
		    return decodeYd(prefices);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return AX;
		case 1:
		    return decodeYw(prefices);
		default:
		    return -1;
		}
	    }

	case 0xff: //INC/DEC G5
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (modrm & 0x38) {
		case 0x00: //INC Ed
		case 0x08: //DEC Ed
		case 0x10: //CALLN Ed
		case 0x20: //JMPN Ed
		case 0x30: //PUSH Ed
		    return decodeEd(prefices, modrm, sib);
		case 0x18: //CALLF Ep
		case 0x28: //JMPF Ep
		    return decodeM(prefices, modrm, sib);
		}
	    } else {
		switch (modrm & 0x38) {
		case 0x00:
		case 0x08:
		case 0x10:
		case 0x20:
		case 0x30:
		    return decodeEw(prefices, modrm, sib);
		case 0x18:
		case 0x28:
		    return decodeM(prefices, modrm, sib);
		}
	    }
	    break;

	case 0xc4: //LES Gv, Mp
	case 0xc5: //LDS Gv, Mp
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0xd7: // XLAT
	    switch (index) {
	    case 0:
		if ((prefices & PREFICES_ADDRESS) != 0) {
		    switch (prefices & PREFICES_SG) {
		    case PREFICES_ES:
			return M_ES_EBX_uAL;
		    case PREFICES_CS:
			return M_CS_EBX_uAL;
		    case PREFICES_SS:
			return M_SS_EBX_uAL;
		    default:
		    case PREFICES_DS:
			return M_DS_EBX_uAL;
		    case PREFICES_FS:
			return M_FS_EBX_uAL;
		    case PREFICES_GS:
			return M_GS_EBX_uAL;
		    }
		} else {
		    switch (prefices & PREFICES_SG) {
		    case PREFICES_ES:
			return M_ES_BX_uAL;
		    case PREFICES_CS:
			return M_CS_BX_uAL;
		    case PREFICES_SS:
			return M_SS_BX_uAL;
		    default:
		    case PREFICES_DS:
			return M_DS_BX_uAL;
		    case PREFICES_FS:
			return M_FS_BX_uAL;
		    case PREFICES_GS:
			return M_GS_BX_uAL;
		    }
		}
	    case 1:
		return AL;
	    default:
		return -1;
	    }

	default:
	    return -1;
	}
	return -1;
    }

    private static int decodeTwoByteOperand(int index, int prefices, int opcode, int modrm, int sib)
    {
	switch (opcode & 0xff) {
	case 0x01:
	    switch (modrm & 0x38) {
	    case 0x00:
	    case 0x08:
	    case 0x10:
	    case 0x18:
		return decodeM(prefices, modrm, sib);
	    case 0x20:
	    case 0x30:
		return decodeEw(prefices, modrm, sib);
	    case 0x38:
		return decodeEb(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0x20: //MOV Rd, Cd
	    switch (index) {
	    case 0:
		return decodeCd(modrm);
	    case 1:
		return decodeEd(prefices, modrm, sib);
	    default:
		return -1;
	    }
	case 0x22: //MOV Cd, Rd
	    switch (index) {
	    case 0:
		return decodeEd(prefices, modrm, sib);
	    case 1:
		return decodeCd(modrm);
	    default:
		return -1;
	    }

	case 0x40:
	case 0x41:
	case 0x42:
	case 0x43:
	case 0x44:
	case 0x45:
	case 0x46:
	case 0x47:
	case 0x48:
	case 0x49:
	case 0x4a:
	case 0x4b:
	case 0x4c:
	case 0x4d:
	case 0x4e:
	case 0x4f:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		case 2:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		case 2:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0x80: //Jcc Jv
	case 0x81:
	case 0x82:
	case 0x83:
	case 0x84:
	case 0x85:
	case 0x86:
	case 0x87:
	case 0x88:
	case 0x89:
	case 0x8a:
	case 0x8b:
	case 0x8c:
	case 0x8d:
	case 0x8e:
	case 0x8f:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		return ID;
	    } else {
		return IW;
	    }

	case 0x90:
	case 0x91:
	case 0x92:
	case 0x93:
	case 0x94:
	case 0x95:
	case 0x96:
	case 0x97:
	case 0x98:
	case 0x99:
	case 0x9a:
	case 0x9b:
	case 0x9c:
	case 0x9d:
	case 0x9e:
	case 0x9f:
	    return decodeEb(prefices, modrm, sib);

	case 0xa0: //PUSH FS
	case 0xa1: //POP FS
	    return FS;

	case 0xa3:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0xa4:
	    	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return IB;
		case 2:
		    return decodeEd(prefices, modrm, sib);
		case 3:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return IB;
		case 2:
		    return decodeEw(prefices, modrm, sib);
		case 3:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xa5:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return CL;
		case 2:
		    return decodeEd(prefices, modrm, sib);
		case 3:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return CL;
		case 2:
		    return decodeEw(prefices, modrm, sib);
		case 3:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0xa8: //PUSH GS
	case 0xa9: //POP GS
	    return GS;
	case 0xab:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xac:
	    	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return IB;
		case 2:
		    return decodeEd(prefices, modrm, sib);
		case 3:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return IB;
		case 2:
		    return decodeEw(prefices, modrm, sib);
		case 3:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xad:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return CL;
		case 2:
		    return decodeEd(prefices, modrm, sib);
		case 3:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return CL;
		case 2:
		    return decodeEw(prefices, modrm, sib);
		case 3:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }

	case 0xaf:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		case 2:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		case 2:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }
	    	case 0xb0: //CMPXCHG_O8 Ev, Gv
	    switch (index) {
	    case 0:
		return decodeEd(prefices, modrm, sib);
	    case 1:
		return decodeGd(modrm);
	    case 2:
		return decodeEd(prefices, modrm, sib);
	    default:
		return -1;
	    }

	case 0xb1: //CMPXCHG_O16/O32 Ev, Gv
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	    
	case 0xb2: //LSS Gv, Mp
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }
	 
	case 0xb3:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xb4:
	case 0xb5:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeM(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }
	case 0xb6: //MOVZX Gv, Eb
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEb(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEb(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0xb7: //MOVZX Gv, Ew
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }
	case 0xba: //BitTest Ev, Ib
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return IB;
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xbb:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeGd(modrm);
		case 1:
		    return decodeEd(prefices, modrm, sib);
		case 2:
		    return decodeEd(prefices, modrm, sib);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeGw(modrm);
		case 1:
		    return decodeEw(prefices, modrm, sib);
		case 2:
		    return decodeEw(prefices, modrm, sib);
		default:
		    return -1;
		}
	    }
	case 0xbc:
	case 0xbd:
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEd(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }
	case 0xbe: //MOVSX Gv, Eb
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEb(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEb(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }

	case 0xbf: //MOVSX Gv, Ew
	    if ((prefices & PREFICES_OPERAND) != 0) {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGd(modrm);
		default:
		    return -1;
		}
	    } else {
		switch (index) {
		case 0:
		    return decodeEw(prefices, modrm, sib);
		case 1:
		    return decodeGw(modrm);
		default:
		    return -1;
		}
	    }   
	}
	return -1;
    }

    private static int decodeFPUOperand(int index, int prefices, int opcode, int modrm, int sib)
    {
	if ((modrm & 0xc0) != 0xc0) {
	    switch (opcode & 0xff38) { //operations using the reg portion of modrm for opcode encoding
	    case 0xd810: //FCOM_SR   ST0, m32fp
	    case 0xd818: //FCOMP_SR  ST0, m32fp
	    case 0xd900: //FLD_SR    m32fp
	    case 0xd910: //FST_SR    m32fp
	    case 0xd918: //FSTP_SR   m32fp
	    case 0xd928: //FLDCW     m2byte
	    case 0xd938: //FSTCW     m2byte
	    case 0xda10: //FICOM_DI  m32int
	    case 0xda18: //FICOMP_DI m32int
	    case 0xdb00: //FILD_DI   m32int
	    case 0xdb08: //FISTTP_DI m32int
	    case 0xdb10: //FIST_DI   m32int
	    case 0xdb18: //FISTP_DI  m32int
	    case 0xdb28: //FLD_XR    m80fp
	    case 0xdb38: //FSTP_XR   m80fp
	    case 0xdc10: //FCOM_DR   ST0, m64fp
	    case 0xdc18: //FCOMP_DR  ST0, m64fp
	    case 0xdd00: //FLD_DR    m64fp
	    case 0xdd08: //FISTTP_QI m64int
	    case 0xdd10: //FST_DR    m64fp
	    case 0xdd18: //FSTP_DR   m64fp
	    case 0xdd38: //FSTSW     m2byte
	    case 0xde10: //FICOM_WI  m16int
	    case 0xde18: //FICOMP_WI m16int
	    case 0xdf00: //FILD_WI   m16int
	    case 0xdf08: //FISTTP_WI m16int
	    case 0xdf10: //FIST_WI   m16int
	    case 0xdf18: //FISTP_WI  m16int
	    case 0xdf20: //FBLD      m80bcd
	    case 0xdf28: //FILD_QI   m64int
	    case 0xdf30: //FBSTP     m80bcd
	    case 0xdf38: //FISTP_QI  m64int

	    case 0xd800: //FADD_SR  m32fp
	    case 0xdc00: //FADD_DR  m64fp
	    case 0xd830: //FDIV_SR  m32fp
	    case 0xdc30: //FDIV_DR  m64fp
	    case 0xd838: //FDIVR_SR  m32fp
	    case 0xdc38: //FDIVR_DR  m64fp
	    case 0xd808: //FMUL_SR  m32fp
	    case 0xdc08: //FMUL_DR  m64fp
	    case 0xd820: //FSUB_SR  m32fp
	    case 0xdc20: //FSUB_DR  m64fp
	    case 0xd828: //FSUBR_SR m32fp
	    case 0xd920: //FNLDENV_14/28 m(14/28b)
	    case 0xd930: //FNSTENV_14/28 m(14/28b)
	    case 0xdc28: //FSUBR_DR m64fp
	    case 0xde00: //FIADD_WI m16int
	    case 0xda00: //FIADD_DI m32int
	    case 0xde30: //FIDIV_WI  m16int
	    case 0xda30: //FIDIV_DI  m32int
	    case 0xde38: //FIDIVR_WI m16int
	    case 0xda38: //FIDIVR_DI m32int
	    case 0xda08: //FIMUL_DI m32int
	    case 0xde08: //FIMUL_WI m16int
	    case 0xde20: //FISUB_WI m16int
	    case 0xda20: //FISUB_DI m32int
	    case 0xde28: //FISUBR_WI m16int
	    case 0xda28: //FISUBR_DI m32int
		return decodeM(prefices, modrm, sib);


	    case 0xdd20: return -1; //FRSTOR_98108;
	    case 0xdd30: return -1; //FSAVE_98108;
	    }
	} else {
	    switch (opcode & 0xfff8) { //operations that act on two fpu stack operands
	    case 0xd8c0: //FADD ST0, STi
	    case 0xd8c8: //FMUL ST0, STi
	    case 0xd8e0: //FSUB ST0, STi
	    case 0xd8e8: //FSUBR ST0, STi
	    case 0xd8f0: //FDIV ST0, STi
	    case 0xd8f8: //FDIVR ST0, STi
 		switch (index) {
		case 0:
		    return decodeST(modrm);
		case 1:
		    return FPU_ST0;
		case 2:
		    return FPU_ST0;
		default:
		    return -1;
		}

	    case 0xdcc0: //FADD  STi, ST0
	    case 0xdcc8: //FMUL  STi, ST0
	    case 0xdce0: //FSUBR STi, ST0
	    case 0xdce8: //FSUB  STi, ST0
	    case 0xdcf0: //FDIVR STi, ST0
	    case 0xdcf8: //FDIV  STi, ST0
	    case 0xdec0: //FADDP STi, ST0
	    case 0xdec8: //FMULP STi, ST0
	    case 0xdee0: //FSUBRP STi, ST0
	    case 0xdee8: //FSUBP STi, ST0
	    case 0xdef0: //FDIVRP STi, ST0
	    case 0xdef8: //FDIVP STi, ST0
 		switch (index) {
		case 0:
		    return FPU_ST0;
		case 1:
		    return decodeST(modrm);
		case 2:
		    return decodeST(modrm);
		default:
		    return -1;
		}

	    case 0xd8d0: //FCOM     ST0, STi
	    case 0xd8d8: //FCOMP    ST0, STi
	    case 0xd9c0: //FLD      STi
	    case 0xd9c8: //FXCH     ST0, STi
	    case 0xdac0: //FCMOVB   ST0, STi  
	    case 0xdac8: //FCMOVE   ST0, STi
	    case 0xdad0: //FCMOVBE  ST0, STi
	    case 0xdad8: //FCMOVU   ST0, STi
	    case 0xdbc0: //FCMOVNB  ST0, STi
	    case 0xdbc8: //FCMOVNE  ST0, STi
	    case 0xdbd0: //FCMOVNBE ST0, STi
	    case 0xdbd8: //FCMOVNU  ST0, STi
	    case 0xdbe8: //FUCOMI   ST0, STi
	    case 0xdbf0: //FCOMI    ST0, STi
	    case 0xddc0: //FFREE    STi
	    case 0xddd0: //FST      STi
	    case 0xddd8: //FSTP     STi
	    case 0xdde0: //FUCOM    ST0, STi
	    case 0xdde8: //FUCOMP   ST0, STi
	    case 0xdfe8: //FUCOMIP  ST0, STi
	    case 0xdff0: //FCOMIP   ST0, STi
		return decodeST(modrm); 
	    }

	    switch (opcode) { //operations that use the whole opcode + modrm combination to encode themselves
	    case 0xdae9: //FUCOMPP ST0, ST1;
	    case 0xded9: //FCOMPP  ST0, ST1;
		return FPU_ST1;
	    case 0xdfe0: return AX; //FSTSW AX
	    }

	}
	return -1;
    }

    public static int operandCount(int operation)
    {
	switch (operation) {
	default:
	    return -1;

	case EIP_UPDATE:
	case DAA:
	case DAS:
	case AAA:
	case AAS:
	case PUSHA_O16_A16:
	case PUSHA_O32_A16:
	case PUSHA_O16_A32:
	case PUSHA_O32_A32:
	case POPA_O16_A16:
	case POPA_O32_A16:
	case POPA_O16_A32:
	case POPA_O32_A32:
	case NOP:
	case CWDE:
	case CBW:
	case CDQ:
	case CWD:
	case WAIT:
	case PUSHF_O16_A16:
	case PUSHF_O32_A16:
	case PUSHF_O16_A32:
	case PUSHF_O32_A32:
	case POPF_O16_A16:
	case POPF_O32_A16:
	case POPF_O16_A32:
	case POPF_O32_A32:
	case SAHF:
	case LAHF:
	case RET_O16_A16:
	case RET_O32_A16:
	case RET_O16_A32:
	case RET_O32_A32:
	case LEAVE_O16_A16:
	case LEAVE_O32_A16:
	case LEAVE_O16_A32:
	case LEAVE_O32_A32:
	case RETF_O16_A16:
	case RETF_O32_A16:
	case RETF_O16_A32:
	case RETF_O32_A32:
	case INT3_O16_A16:
	case INT3_O32_A16:
	case INT3_O16_A32:
	case INT3_O32_A32:
	case INTO_O16_A16:
	case INTO_O32_A16:
	case INTO_O16_A32:
	case INTO_O32_A32:
	case SALC:
	case INT1_O16_A16:
	case INT1_O32_A16:
	case INT1_O16_A32:
	case INT1_O32_A32:
	case HLT:
	case CMC:
	case CLC:
	case STC:
	case CLI:
	case STI:
	case CLD:
	case STD:
	case IRET_O16_A16:
	case IRET_O32_A16:
	case IRET_O16_A32:
	case IRET_O32_A32:
	case CPUID:
	case FNOP:
	case FCHS:
	case FABS:
	case FTST:
	case FXAM:
	case FLD1:
	case FLDL2T:
	case FLDL2E:
	case FLDPI:
	case FLDLG2:
	case FLDLN2:
	case FLDZ:
	case F2XM1:
	case FYL2X:
	case FPTAN:
	case FPATAN:
	case FXTRACT:
	case FDECSTP:
	case FINCSTP:
	case FPREM1:
	case FPREM:
	case FYL2XP1:
	case FSQRT:
	case FRNDINT:
	case FSCALE:
	case FSINCOS:
	case FSIN:
	case FCOS:
	case FCLEX:
	case FNINIT:
	    return 0;

	case PUSH_O16_A16:
	case PUSH_O32_A16:
	case PUSH_O16_A32:
	case PUSH_O32_A32:
	case POP_O16_A16:
	case POP_O32_A16:
	case POP_O16_A32:
	case POP_O32_A32:
	case JO_O8:
	case JNO_O8:
	case JB_O8:
	case JNB_O8:
	case JZ_O8:
	case JNZ_O8:
	case JBE_O8:
	case JNBE_O8:
	case JS_O8:
	case JNS_O8:
	case JP_O8:
	case JNP_O8:
	case JL_O8:
	case JNL_O8:
	case JLE_O8:
	case JNLE_O8:
	case JO_O16:
	case JNO_O16:
	case JB_O16:
	case JNB_O16:
	case JZ_O16:
	case JNZ_O16:
	case JBE_O16:
	case JNBE_O16:
	case JS_O16:
	case JNS_O16:
	case JP_O16:
	case JNP_O16:
	case JL_O16:
	case JNL_O16:
	case JLE_O16:
	case JNLE_O16:
	case JO_O32:
	case JNO_O32:
	case JB_O32:
	case JNB_O32:
	case JZ_O32:
	case JNZ_O32:
	case JBE_O32:
	case JNBE_O32:
	case JS_O32:
	case JNS_O32:
	case JP_O32:
	case JNP_O32:
	case JL_O32:
	case JNL_O32:
	case JLE_O32:
	case JNLE_O32:
	case CALLF_O16_A16:
	case CALLF_O32_A16:
	case CALLF_O16_A32:
	case CALLF_O32_A32:
	case RET_IW_O16_A16:
	case RET_IW_O32_A16:
	case RET_IW_O16_A32:
	case RET_IW_O32_A32:
	case RETF_IW_O16_A16:
	case RETF_IW_O32_A16:
	case RETF_IW_O16_A32:
	case RETF_IW_O32_A32:
	case INT_O16_A16:
	case INT_O32_A16:
	case INT_O16_A32:
	case INT_O32_A32:
	case LOOPNZ_O16_A16:
	case LOOPNZ_O32_A16:
	case LOOPNZ_O16_A32:
	case LOOPNZ_O32_A32:
	case LOOPZ_O16_A16:
	case LOOPZ_O32_A16:
	case LOOPZ_O16_A32:
	case LOOPZ_O32_A32:
	case LOOP_O16_A16:
	case LOOP_O32_A16:
	case LOOP_O16_A32:
	case LOOP_O32_A32:
	case JCXZ_A16:
	case JCXZ_A32:
	case JMP_O8_SHORT:
	case CALL_O16_A16:
	case CALL_O32_A16:
	case CALL_O16_A32:
	case CALL_O32_A32:
	case CALLN_O16_A16:
	case CALLN_O32_A16:
	case CALLN_O16_A32:
	case CALLN_O32_A32:
	case JMP_O32_NEAR_RELATIVE:
	case JMP_O16_NEAR_RELATIVE:
	case JMP_O32_NEAR_ABSOLUTE:
	case JMP_O16_NEAR_ABSOLUTE:
	case JMP_O32_FAR:
	case JMP_O16_FAR:
	case IMUL_REGA_O8:
	case IMUL_REGA_O16:
	case IMUL_REGA_O32:
	case DIV_O8:
	case DIV_O16:
	case DIV_O32:
	case MUL_O8:
	case MUL_O16:
	case MUL_O32:
	case IDIV_O8:
	case IDIV_O16:
	case IDIV_O32:
	case LGDT_O16:
	case LGDT_O32:
	case SGDT_O16:
	case SGDT_O32:
	case LIDT_O16:
	case LIDT_O32:
	case SIDT_O16:
	case SIDT_O32:
	case SMSW:
	case LMSW:
	case INVLPG:
	case SETO:
	case SETNO:
	case SETB:
	case SETNB:
	case SETZ:
	case SETNZ:
	case SETBE:
	case SETNBE:
	case SETS:
	case SETNS:
	case SETP:
	case SETNP:
	case SETL:
	case SETNL:
	case SETLE:
	case SETNLE:
	case ENTER_O16_A16:
	case ENTER_O32_A16:
	case ENTER_O16_A32:
	case ENTER_O32_A32:
	case AAM:
	case AAD:
	case FUCOMI:
	case FCOMI:
	case FUCOMIP:
	case FCOMIP:
	case FUCOM:
	case FUCOMP:
	case FUCOMPP:
	case FCOM_SR:
	case FCOMP_SR:
	case FCOM_DR:
	case FCOMP_DR:
	case FCOM:
	case FCOMP:
	case FCOMPP:
	case FSTSW:
	case FLDCW:
	case FRSTOR_98108:
	case FSAVE_98108:
	case FSTCW:
	case FIST_DI:
	case FISTP_DI:
	case FIST_WI:
	case FISTP_WI:
	case FISTP_QI:
	case FILD_WI:
	case FILD_DI:
	case FILD_QI:
	case FICOM_WI:
	case FICOMP_WI:
	case FICOM_DI:
	case FICOMP_DI:
	case FFREE:
	case FCMOVB:
	case FCMOVE:
	case FCMOVBE:
	case FCMOVU:
	case FCMOVNB:
	case FCMOVNE:
	case FCMOVNBE:
	case FCMOVNU:
	case FBSTP:
	case FBLD:
	case FST:
	case FSTP:
	case FST_SR:
	case FSTP_SR:
	case FST_DR:
	case FSTP_DR:
	case FSTP_XR:
	case FLD:
	case FLD_SR:
	case FLD_DR:
	case FLD_XR:
	case FISTTP_WI:
	case FISTTP_DI:
	case FISTTP_QI:
	case FADD_SR:
	case FADD_DR:
	case FMUL_SR:
	case FMUL_DR:
	case FDIV_SR:
	case FDIV_DR:
	case FDIVR_SR:
	case FDIVR_DR:
	case FSUB_SR:
	case FSUB_DR:
	case FSUBR_SR:
	case FSUBR_DR:
	case FIADD_DI:
	case FIADD_WI:
	case FIDIV_DI:
	case FIDIV_WI:
	case FIDIVR_DI:    
	case FIDIVR_WI:
	case FISUB_DI:
	case FISUBR_DI:
	case FISUB_WI:
	case FISUBR_WI:
	case FIMUL_WI:
	case FIMUL_DI:
	    return 1;

	case INC_O32:
	case INC_O16:
	case DEC_O32:
	case DEC_O16:
	case INC_O8:
	case DEC_O8:
	case NEG_O16:
	case NOT_O16:
	case NEG_O32:
	case NOT_O32:
	case NEG_O8:
	case NOT_O8:
	case CMP_O8:
	case CMP_O32:
	case CMP_O16:
	case BOUND_O16_A16:
	case BOUND_O32_A16:
	case BOUND_O16_A32:
	case BOUND_O32_A32:
	case TEST_O8:
	case TEST_O32:
	case TEST_O16:
	case MOV_O8:
	case MOV_O32:
	case MOV_O16:
	case MOVZX_O16_O8:
	case MOVZX_O32_O8:
	case MOVZX_O16_O16:
	case MOVZX_O32_O16:
	case MOV_TO_SEG:
	case CMOVO_O16:
	case CMOVNO_O16:
	case CMOVB_O16:
	case CMOVNB_O16:
	case CMOVZ_O16:
	case CMOVNZ_O16:
	case CMOVBE_O16:
	case CMOVNBE_O16:
	case CMOVS_O16:
	case CMOVNS_O16:
	case CMOVP_O16:
	case CMOVNP_O16:
	case CMOVL_O16:
	case CMOVNL_O16:
	case CMOVLE_O16:
	case CMOVNLE_O16:
	case CMOVO_O32:
	case CMOVNO_O32:
	case CMOVB_O32:
	case CMOVNB_O32:
	case CMOVZ_O32:
	case CMOVNZ_O32:
	case CMOVBE_O32:
	case CMOVNBE_O32:
	case CMOVS_O32:
	case CMOVNS_O32:
	case CMOVP_O32:
	case CMOVNP_O32:
	case CMOVL_O32:
	case CMOVNL_O32:
	case CMOVLE_O32:
	case CMOVNLE_O32:
	case LEA_O16_A16:
	case LEA_O32_A16:
	case LEA_O16_A32:
	case LEA_O32_A32:
	case IN_O8_O8:
	case IN_O32_O8:
	case IN_O16_O8:
	case IN_O8_O16:
	case IN_O32_O16:
	case IN_O16_O16:
	case OUT_O8_O8:
	case OUT_O8_O32:
	case OUT_O8_O16:
	case OUT_O16_O8:
	case OUT_O16_O32:
	case OUT_O16_O16:
	case INS_O8_A16:
	case INS_O16_A16:
	case INS_O32_A16:
	case INS_O8_A32:
	case INS_O16_A32:
	case INS_O32_A32:
	case REP_INS_O8_A16:
	case REP_INS_O8_A32:
	case REP_INS_O16_A16:
	case REP_INS_O16_A32:
	case REP_INS_O32_A16:
	case REP_INS_O32_A32:
	case OUTS_O8_A16:
	case OUTS_O16_A16:
	case OUTS_O32_A16:
	case OUTS_O8_A32:
	case OUTS_O16_A32:
	case OUTS_O32_A32:
	case REP_OUTS_O8_A16:
	case REP_OUTS_O8_A32:
	case REP_OUTS_O16_A16:
	case REP_OUTS_O16_A32:
	case REP_OUTS_O32_A16:
	case REP_OUTS_O32_A32:
	case MOVS_O8_A16:
	case MOVS_O16_A16:
	case MOVS_O32_A16:
	case MOVS_O8_A32:
	case MOVS_O16_A32:
	case MOVS_O32_A32:
	case REP_MOVS_O8_A16:
	case REP_MOVS_O8_A32:
	case REP_MOVS_O16_A16:
	case REP_MOVS_O16_A32:
	case REP_MOVS_O32_A16:
	case REP_MOVS_O32_A32:
	case CMPS_O8_A16:
	case CMPS_O16_A16:
	case CMPS_O32_A16:
	case CMPS_O8_A32:
	case CMPS_O16_A32:
	case CMPS_O32_A32:
	case REPE_CMPS_O8_A16:
	case REPE_CMPS_O16_A16:
	case REPE_CMPS_O32_A16:
	case REPNE_CMPS_O8_A16:
	case REPNE_CMPS_O16_A16:
	case REPNE_CMPS_O32_A16:
	case REPE_CMPS_O8_A32:
	case REPE_CMPS_O16_A32:
	case REPE_CMPS_O32_A32:
	case REPNE_CMPS_O8_A32:
	case REPNE_CMPS_O16_A32:
	case REPNE_CMPS_O32_A32:
	case STOS_O8_A16:
	case STOS_O16_A16:
	case STOS_O32_A16:
	case STOS_O8_A32:
	case STOS_O16_A32:
	case STOS_O32_A32:
	case REP_STOS_O8_A16:
	case REP_STOS_O16_A16:
	case REP_STOS_O32_A16:
	case REP_STOS_O8_A32:
	case REP_STOS_O16_A32:
	case REP_STOS_O32_A32:
	case LODS_O8_A16:
	case LODS_O16_A16:
	case LODS_O32_A16:
	case LODS_O8_A32:
	case LODS_O16_A32:
	case LODS_O32_A32:
	case REP_LODS_O8_A16:
	case REP_LODS_O16_A16:
	case REP_LODS_O32_A16:
	case REP_LODS_O8_A32:
	case REP_LODS_O16_A32:
	case REP_LODS_O32_A32:
	case SCAS_O8_A16:
	case SCAS_O16_A16:
	case SCAS_O32_A16:
	case SCAS_O8_A32:
	case SCAS_O16_A32:
	case SCAS_O32_A32:
	case REPE_SCAS_O8_A16:
	case REPE_SCAS_O16_A16:
	case REPE_SCAS_O32_A16:
	case REPNE_SCAS_O8_A16:
	case REPNE_SCAS_O16_A16:
	case REPNE_SCAS_O32_A16:
	case REPE_SCAS_O8_A32:
	case REPE_SCAS_O16_A32:
	case REPE_SCAS_O32_A32:
	case REPNE_SCAS_O8_A32:
	case REPNE_SCAS_O16_A32:
	case REPNE_SCAS_O32_A32:
	case LES_O16_A16:
	case LES_O32_A16:
	case LES_O16_A32:
	case LES_O32_A32:
	case LFS_O16_A16:
	case LFS_O32_A16:
	case LFS_O16_A32:
	case LFS_O32_A32:
	case LGS_O16_A16:
	case LGS_O32_A16:
	case LGS_O16_A32:
	case LGS_O32_A32:
	case LDS_O16_A16:
	case LDS_O32_A16:
	case LDS_O16_A32:
	case LDS_O32_A32:
	case LSS_O16_A16:
	case LSS_O32_A16:
	case LSS_O16_A32:
	case LSS_O32_A32:
	case XLAT:
	case MOV_TO_CR_O32:
	case MOVSX_O16_O8:
	case MOVSX_O32_O8:
	case MOVSX_O16_O16:
	case MOVSX_O32_O16:
	case BSF_O16:
	case BSF_O32:
	case FXCH:
	case FNSTENV_14:
	case FNLDENV_14:
	case FNSTENV_28:
	case FNLDENV_28:
	    return 2;


	case ADD_O8:
	case ADD_O32:
	case ADD_O16:
	case OR_O8:
	case OR_O32:
	case OR_O16:
	case ADC_O8:
	case ADC_O32:
	case ADC_O16:
	case SBB_O8:
	case SBB_O32:
	case SBB_O16:
	case AND_O8:
	case AND_O32:
	case AND_O16:
	case SUB_O8:
	case SUB_O32:
	case SUB_O16:
	case XOR_O8:
	case XOR_O32:
	case XOR_O16:
	case ROL_O8:
	case ROR_O8:
	case RCL_O8:
	case RCR_O8:
	case SHL_O8:
	case SHR_O8:
	case SAR_O8:
	case ROL_O16:
	case ROR_O16:
	case RCL_O16:
	case RCR_O16:
	case SHL_O16:
	case SHR_O16:
	case SAR_O16:
	case ROL_O32:
	case ROR_O32:
	case RCL_O32:
	case RCR_O32:
	case SHL_O32:
	case SHR_O32:
	case SAR_O32:
	case IMUL_O32:
	case IMUL_O16:
	case BTR_O16:
	case BTR_O32:
	case BTS_O16:
	case BTS_O32:
	case BTC_O16:
	case BTC_O32:
	case BT_O16:
	case BT_O32:
	case BT_O16_O8:
	case BT_O32_O8:
	case BTS_O16_O8:
	case BTS_O32_O8:
	case BTR_O16_O8:
	case BTR_O32_O8:
	case BTC_O16_O8:
	case BTC_O32_O8:
	case FDIV:
	case FDIVP:
	case FDIVR:
	case FDIVRP:
	case FADD:
	case FADDP:
	case FSUBP:
	case FSUB:
	case FSUBR:
	case FSUBRP:
	case FMUL:
	case FMULP:
	case CMPXCHG_O8:
	case CMPXCHG_O16:
	case CMPXCHG_O32:
	    return 3;

	case XCHG_O8:
	case XCHG_O32:
	case XCHG_O16:
	case SHRD_O16:
	case SHRD_O32:
	case SHLD_O16:
	case SHLD_O32:
	    return 4;
	}
    }

    public static int immediateCount(int operand)
    {
	switch (operandHasImmediate(operand)) {
	default:
	case 0:
	    break;
	case 1:
	case 2:
	case 3:
	case 4:
	    return 1;
	case 6:
	    return 2;
	}

	switch (operandHasDisplacement(operand)) {
	default:
	case 0:
	    return 0;
	case 1:
	case 2:
	case 4:
	    return 1;
	case 6:
	    return 2;
	}
    }

    private static int operandHasImmediate(int operand)
    {
	switch (operand) {
	case IB:
	    return 1;
	case IW:
	    return 2;
	case IWIB:
	    return 3;
	case ID:
	case IA32:
	    return 4;
	case IA48:
	    return 6;
	}
	return 0;
    }

    private static int operandHasDisplacement(int operand)
    {
	switch (operand) {
	case M_ES_EAX_IB:
	case M_ES_ECX_IB:
	case M_ES_EDX_IB:
	case M_ES_EBX_IB:
	case M_ES_EBP_IB:
	case M_ES_ESI_IB:
	case M_ES_EDI_IB:
	case M_CS_EAX_IB:
	case M_CS_ECX_IB:
	case M_CS_EDX_IB:
	case M_CS_EBX_IB:
	case M_CS_EBP_IB:
	case M_CS_ESI_IB:
	case M_CS_EDI_IB:
	case M_SS_EAX_IB:
	case M_SS_ECX_IB:
	case M_SS_EDX_IB:
	case M_SS_EBX_IB:
	case M_SS_EBP_IB:
	case M_SS_ESI_IB:
	case M_SS_EDI_IB:
	case M_DS_EAX_IB:
	case M_DS_ECX_IB:
	case M_DS_EDX_IB:
	case M_DS_EBX_IB:
	case M_DS_EBP_IB:
	case M_DS_ESI_IB:
	case M_DS_EDI_IB:
	case M_FS_EAX_IB:
	case M_FS_ECX_IB:
	case M_FS_EDX_IB:
	case M_FS_EBX_IB:
	case M_FS_EBP_IB:
	case M_FS_ESI_IB:
	case M_FS_EDI_IB:
	case M_GS_EAX_IB:
	case M_GS_ECX_IB:
	case M_GS_EDX_IB:
	case M_GS_EBX_IB:
	case M_GS_EBP_IB:
	case M_GS_ESI_IB:
	case M_GS_EDI_IB:
	case M_ES_BX_SI_IB:
	case M_ES_BX_DI_IB:
	case M_ES_BP_SI_IB:
	case M_ES_BP_DI_IB:
	case M_ES_SI_IB:
	case M_ES_DI_IB:
	case M_ES_BP_IB:
	case M_ES_BX_IB:
	case M_CS_BX_SI_IB:
	case M_CS_BX_DI_IB:
	case M_CS_BP_SI_IB:
	case M_CS_BP_DI_IB:
	case M_CS_SI_IB:
	case M_CS_DI_IB:
	case M_CS_BP_IB:
	case M_CS_BX_IB:
	case M_SS_BX_SI_IB:
	case M_SS_BX_DI_IB:
	case M_SS_BP_SI_IB:
	case M_SS_BP_DI_IB:
	case M_SS_SI_IB:
	case M_SS_DI_IB:
	case M_SS_BP_IB:
	case M_SS_BX_IB:
	case M_DS_BX_SI_IB:
	case M_DS_BX_DI_IB:
	case M_DS_BP_SI_IB:
	case M_DS_BP_DI_IB:
	case M_DS_SI_IB:
	case M_DS_DI_IB:
	case M_DS_BP_IB:
	case M_DS_BX_IB:
	case M_FS_BX_SI_IB:
	case M_FS_BX_DI_IB:
	case M_FS_BP_SI_IB:
	case M_FS_BP_DI_IB:
	case M_FS_SI_IB:
	case M_FS_DI_IB:
	case M_FS_BP_IB:
	case M_FS_BX_IB:
	case M_GS_BX_SI_IB:
	case M_GS_BX_DI_IB:
	case M_GS_BP_SI_IB:
	case M_GS_BP_DI_IB:
	case M_GS_SI_IB:
	case M_GS_DI_IB:
	case M_GS_BP_IB:
	case M_GS_BX_IB:
	case M_ES_2EAX_IB:
	case M_ES_EAX_ECX_IB:
	case M_ES_EAX_EDX_IB:
	case M_ES_EAX_EBX_IB:
	case M_ES_EAX_ESP_IB:
	case M_ES_EAX_EBP_IB:
	case M_ES_EAX_ESI_IB:
	case M_ES_EAX_EDI_IB:
	case M_ES_2ECX_IB:
	case M_ES_ECX_EDX_IB:
	case M_ES_ECX_EBX_IB:
	case M_ES_ECX_ESP_IB:
	case M_ES_ECX_EBP_IB:
	case M_ES_ECX_ESI_IB:
	case M_ES_ECX_EDI_IB:
	case M_ES_2EDX_IB:
	case M_ES_EDX_EBX_IB:
	case M_ES_EDX_ESP_IB:
	case M_ES_EDX_EBP_IB:
	case M_ES_EDX_ESI_IB:
	case M_ES_EDX_EDI_IB:
	case M_ES_2EBX_IB:
	case M_ES_EBX_ESP_IB:
	case M_ES_EBX_EBP_IB:
	case M_ES_EBX_ESI_IB:
	case M_ES_EBX_EDI_IB:
	case M_ES_ESP_IB:
	case M_ES_EBP_ESP_IB:
	case M_ES_2EBP_IB:
	case M_ES_EBP_ESI_IB:
	case M_ES_EBP_EDI_IB:
	case M_ES_ESI_ESP_IB:
	case M_ES_2ESI_IB:
	case M_ES_ESI_EDI_IB:
	case M_ES_EDI_ESP_IB:
	case M_ES_2EDI_IB:
	case M_ES_3EAX_IB:
	case M_ES_2EAX_ECX_IB:
	case M_ES_2EAX_EDX_IB:
	case M_ES_2EAX_EBX_IB:
	case M_ES_2EAX_ESP_IB:
	case M_ES_2EAX_EBP_IB:
	case M_ES_2EAX_ESI_IB:
	case M_ES_2EAX_EDI_IB:
	case M_ES_2ECX_EAX_IB:
	case M_ES_3ECX_IB:
	case M_ES_2ECX_EDX_IB:
	case M_ES_2ECX_EBX_IB:
	case M_ES_2ECX_ESP_IB:
	case M_ES_2ECX_EBP_IB:
	case M_ES_2ECX_ESI_IB:
	case M_ES_2ECX_EDI_IB:
	case M_ES_2EDX_EAX_IB:
	case M_ES_2EDX_ECX_IB:
	case M_ES_3EDX_IB:
	case M_ES_2EDX_EBX_IB:
	case M_ES_2EDX_ESP_IB:
	case M_ES_2EDX_EBP_IB:
	case M_ES_2EDX_ESI_IB:
	case M_ES_2EDX_EDI_IB:
	case M_ES_2EBX_EAX_IB:
	case M_ES_2EBX_ECX_IB:
	case M_ES_2EBX_EDX_IB:
	case M_ES_3EBX_IB:
	case M_ES_2EBX_ESP_IB:
	case M_ES_2EBX_EBP_IB:
	case M_ES_2EBX_ESI_IB:
	case M_ES_2EBX_EDI_IB:
	case M_ES_2EBP_EAX_IB:
	case M_ES_2EBP_ECX_IB:
	case M_ES_2EBP_EDX_IB:
	case M_ES_2EBP_EBX_IB:
	case M_ES_2EBP_ESP_IB:
	case M_ES_3EBP_IB:
	case M_ES_2EBP_ESI_IB:
	case M_ES_2EBP_EDI_IB:
	case M_ES_2ESI_EAX_IB:
	case M_ES_2ESI_ECX_IB:
	case M_ES_2ESI_EDX_IB:
	case M_ES_2ESI_EBX_IB:
	case M_ES_2ESI_ESP_IB:
	case M_ES_2ESI_EBP_IB:
	case M_ES_3ESI_IB:
	case M_ES_2ESI_EDI_IB:
	case M_ES_2EDI_EAX_IB:
	case M_ES_2EDI_ECX_IB:
	case M_ES_2EDI_EDX_IB:
	case M_ES_2EDI_EBX_IB:
	case M_ES_2EDI_ESP_IB:
	case M_ES_2EDI_EBP_IB:
	case M_ES_2EDI_ESI_IB:
	case M_ES_3EDI_IB:
	case M_ES_5EAX_IB:
	case M_ES_4EAX_ECX_IB:
	case M_ES_4EAX_EDX_IB:
	case M_ES_4EAX_EBX_IB:
	case M_ES_4EAX_ESP_IB:
	case M_ES_4EAX_EBP_IB:
	case M_ES_4EAX_ESI_IB:
	case M_ES_4EAX_EDI_IB:
	case M_ES_4ECX_EAX_IB:
	case M_ES_5ECX_IB:
	case M_ES_4ECX_EDX_IB:
	case M_ES_4ECX_EBX_IB:
	case M_ES_4ECX_ESP_IB:
	case M_ES_4ECX_EBP_IB:
	case M_ES_4ECX_ESI_IB:
	case M_ES_4ECX_EDI_IB:
	case M_ES_4EDX_EAX_IB:
	case M_ES_4EDX_ECX_IB:
	case M_ES_5EDX_IB:
	case M_ES_4EDX_EBX_IB:
	case M_ES_4EDX_ESP_IB:
	case M_ES_4EDX_EBP_IB:
	case M_ES_4EDX_ESI_IB:
	case M_ES_4EDX_EDI_IB:
	case M_ES_4EBX_EAX_IB:
	case M_ES_4EBX_ECX_IB:
	case M_ES_4EBX_EDX_IB:
	case M_ES_5EBX_IB:
	case M_ES_4EBX_ESP_IB:
	case M_ES_4EBX_EBP_IB:
	case M_ES_4EBX_ESI_IB:
	case M_ES_4EBX_EDI_IB:
	case M_ES_4EBP_EAX_IB:
	case M_ES_4EBP_ECX_IB:
	case M_ES_4EBP_EDX_IB:
	case M_ES_4EBP_EBX_IB:
	case M_ES_4EBP_ESP_IB:
	case M_ES_5EBP_IB:
	case M_ES_4EBP_ESI_IB:
	case M_ES_4EBP_EDI_IB:
	case M_ES_4ESI_EAX_IB:
	case M_ES_4ESI_ECX_IB:
	case M_ES_4ESI_EDX_IB:
	case M_ES_4ESI_EBX_IB:
	case M_ES_4ESI_ESP_IB:
	case M_ES_4ESI_EBP_IB:
	case M_ES_5ESI_IB:
	case M_ES_4ESI_EDI_IB:
	case M_ES_4EDI_EAX_IB:
	case M_ES_4EDI_ECX_IB:
	case M_ES_4EDI_EDX_IB:
	case M_ES_4EDI_EBX_IB:
	case M_ES_4EDI_ESP_IB:
	case M_ES_4EDI_EBP_IB:
	case M_ES_4EDI_ESI_IB:
	case M_ES_5EDI_IB:
	case M_ES_9EAX_IB:
	case M_ES_8EAX_ECX_IB:
	case M_ES_8EAX_EDX_IB:
	case M_ES_8EAX_EBX_IB:
	case M_ES_8EAX_ESP_IB:
	case M_ES_8EAX_EBP_IB:
	case M_ES_8EAX_ESI_IB:
	case M_ES_8EAX_EDI_IB:
	case M_ES_8ECX_EAX_IB:
	case M_ES_9ECX_IB:
	case M_ES_8ECX_EDX_IB:
	case M_ES_8ECX_EBX_IB:
	case M_ES_8ECX_ESP_IB:
	case M_ES_8ECX_EBP_IB:
	case M_ES_8ECX_ESI_IB:
	case M_ES_8ECX_EDI_IB:
	case M_ES_8EDX_EAX_IB:
	case M_ES_8EDX_ECX_IB:
	case M_ES_9EDX_IB:
	case M_ES_8EDX_EBX_IB:
	case M_ES_8EDX_ESP_IB:
	case M_ES_8EDX_EBP_IB:
	case M_ES_8EDX_ESI_IB:
	case M_ES_8EDX_EDI_IB:
	case M_ES_8EBX_EAX_IB:
	case M_ES_8EBX_ECX_IB:
	case M_ES_8EBX_EDX_IB:
	case M_ES_9EBX_IB:
	case M_ES_8EBX_ESP_IB:
	case M_ES_8EBX_EBP_IB:
	case M_ES_8EBX_ESI_IB:
	case M_ES_8EBX_EDI_IB:
	case M_ES_8EBP_EAX_IB:
	case M_ES_8EBP_ECX_IB:
	case M_ES_8EBP_EDX_IB:
	case M_ES_8EBP_EBX_IB:
	case M_ES_8EBP_ESP_IB:
	case M_ES_9EBP_IB:
	case M_ES_8EBP_ESI_IB:
	case M_ES_8EBP_EDI_IB:
	case M_ES_8ESI_EAX_IB:
	case M_ES_8ESI_ECX_IB:
	case M_ES_8ESI_EDX_IB:
	case M_ES_8ESI_EBX_IB:
	case M_ES_8ESI_ESP_IB:
	case M_ES_8ESI_EBP_IB:
	case M_ES_9ESI_IB:
	case M_ES_8ESI_EDI_IB:
	case M_ES_8EDI_EAX_IB:
	case M_ES_8EDI_ECX_IB:
	case M_ES_8EDI_EDX_IB:
	case M_ES_8EDI_EBX_IB:
	case M_ES_8EDI_ESP_IB:
	case M_ES_8EDI_EBP_IB:
	case M_ES_8EDI_ESI_IB:
	case M_ES_9EDI_IB:
	case M_CS_2EAX_IB:
	case M_CS_EAX_ECX_IB:
	case M_CS_EAX_EDX_IB:
	case M_CS_EAX_EBX_IB:
	case M_CS_EAX_ESP_IB:
	case M_CS_EAX_EBP_IB:
	case M_CS_EAX_ESI_IB:
	case M_CS_EAX_EDI_IB:
	case M_CS_2ECX_IB:
	case M_CS_ECX_EDX_IB:
	case M_CS_ECX_EBX_IB:
	case M_CS_ECX_ESP_IB:
	case M_CS_ECX_EBP_IB:
	case M_CS_ECX_ESI_IB:
	case M_CS_ECX_EDI_IB:
	case M_CS_2EDX_IB:
	case M_CS_EDX_EBX_IB:
	case M_CS_EDX_ESP_IB:
	case M_CS_EDX_EBP_IB:
	case M_CS_EDX_ESI_IB:
	case M_CS_EDX_EDI_IB:
	case M_CS_2EBX_IB:
	case M_CS_EBX_ESP_IB:
	case M_CS_EBX_EBP_IB:
	case M_CS_EBX_ESI_IB:
	case M_CS_EBX_EDI_IB:
	case M_CS_ESP_IB:
	case M_CS_EBP_ESP_IB:
	case M_CS_2EBP_IB:
	case M_CS_EBP_ESI_IB:
	case M_CS_EBP_EDI_IB:
	case M_CS_ESI_ESP_IB:
	case M_CS_2ESI_IB:
	case M_CS_ESI_EDI_IB:
	case M_CS_EDI_ESP_IB:
	case M_CS_2EDI_IB:
	case M_CS_3EAX_IB:
	case M_CS_2EAX_ECX_IB:
	case M_CS_2EAX_EDX_IB:
	case M_CS_2EAX_EBX_IB:
	case M_CS_2EAX_ESP_IB:
	case M_CS_2EAX_EBP_IB:
	case M_CS_2EAX_ESI_IB:
	case M_CS_2EAX_EDI_IB:
	case M_CS_2ECX_EAX_IB:
	case M_CS_3ECX_IB:
	case M_CS_2ECX_EDX_IB:
	case M_CS_2ECX_EBX_IB:
	case M_CS_2ECX_ESP_IB:
	case M_CS_2ECX_EBP_IB:
	case M_CS_2ECX_ESI_IB:
	case M_CS_2ECX_EDI_IB:
	case M_CS_2EDX_EAX_IB:
	case M_CS_2EDX_ECX_IB:
	case M_CS_3EDX_IB:
	case M_CS_2EDX_EBX_IB:
	case M_CS_2EDX_ESP_IB:
	case M_CS_2EDX_EBP_IB:
	case M_CS_2EDX_ESI_IB:
	case M_CS_2EDX_EDI_IB:
	case M_CS_2EBX_EAX_IB:
	case M_CS_2EBX_ECX_IB:
	case M_CS_2EBX_EDX_IB:
	case M_CS_3EBX_IB:
	case M_CS_2EBX_ESP_IB:
	case M_CS_2EBX_EBP_IB:
	case M_CS_2EBX_ESI_IB:
	case M_CS_2EBX_EDI_IB:
	case M_CS_2EBP_EAX_IB:
	case M_CS_2EBP_ECX_IB:
	case M_CS_2EBP_EDX_IB:
	case M_CS_2EBP_EBX_IB:
	case M_CS_2EBP_ESP_IB:
	case M_CS_3EBP_IB:
	case M_CS_2EBP_ESI_IB:
	case M_CS_2EBP_EDI_IB:
	case M_CS_2ESI_EAX_IB:
	case M_CS_2ESI_ECX_IB:
	case M_CS_2ESI_EDX_IB:
	case M_CS_2ESI_EBX_IB:
	case M_CS_2ESI_ESP_IB:
	case M_CS_2ESI_EBP_IB:
	case M_CS_3ESI_IB:
	case M_CS_2ESI_EDI_IB:
	case M_CS_2EDI_EAX_IB:
	case M_CS_2EDI_ECX_IB:
	case M_CS_2EDI_EDX_IB:
	case M_CS_2EDI_EBX_IB:
	case M_CS_2EDI_ESP_IB:
	case M_CS_2EDI_EBP_IB:
	case M_CS_2EDI_ESI_IB:
	case M_CS_3EDI_IB:
	case M_CS_5EAX_IB:
	case M_CS_4EAX_ECX_IB:
	case M_CS_4EAX_EDX_IB:
	case M_CS_4EAX_EBX_IB:
	case M_CS_4EAX_ESP_IB:
	case M_CS_4EAX_EBP_IB:
	case M_CS_4EAX_ESI_IB:
	case M_CS_4EAX_EDI_IB:
	case M_CS_4ECX_EAX_IB:
	case M_CS_5ECX_IB:
	case M_CS_4ECX_EDX_IB:
	case M_CS_4ECX_EBX_IB:
	case M_CS_4ECX_ESP_IB:
	case M_CS_4ECX_EBP_IB:
	case M_CS_4ECX_ESI_IB:
	case M_CS_4ECX_EDI_IB:
	case M_CS_4EDX_EAX_IB:
	case M_CS_4EDX_ECX_IB:
	case M_CS_5EDX_IB:
	case M_CS_4EDX_EBX_IB:
	case M_CS_4EDX_ESP_IB:
	case M_CS_4EDX_EBP_IB:
	case M_CS_4EDX_ESI_IB:
	case M_CS_4EDX_EDI_IB:
	case M_CS_4EBX_EAX_IB:
	case M_CS_4EBX_ECX_IB:
	case M_CS_4EBX_EDX_IB:
	case M_CS_5EBX_IB:
	case M_CS_4EBX_ESP_IB:
	case M_CS_4EBX_EBP_IB:
	case M_CS_4EBX_ESI_IB:
	case M_CS_4EBX_EDI_IB:
	case M_CS_4EBP_EAX_IB:
	case M_CS_4EBP_ECX_IB:
	case M_CS_4EBP_EDX_IB:
	case M_CS_4EBP_EBX_IB:
	case M_CS_4EBP_ESP_IB:
	case M_CS_5EBP_IB:
	case M_CS_4EBP_ESI_IB:
	case M_CS_4EBP_EDI_IB:
	case M_CS_4ESI_EAX_IB:
	case M_CS_4ESI_ECX_IB:
	case M_CS_4ESI_EDX_IB:
	case M_CS_4ESI_EBX_IB:
	case M_CS_4ESI_ESP_IB:
	case M_CS_4ESI_EBP_IB:
	case M_CS_5ESI_IB:
	case M_CS_4ESI_EDI_IB:
	case M_CS_4EDI_EAX_IB:
	case M_CS_4EDI_ECX_IB:
	case M_CS_4EDI_EDX_IB:
	case M_CS_4EDI_EBX_IB:
	case M_CS_4EDI_ESP_IB:
	case M_CS_4EDI_EBP_IB:
	case M_CS_4EDI_ESI_IB:
	case M_CS_5EDI_IB:
	case M_CS_9EAX_IB:
	case M_CS_8EAX_ECX_IB:
	case M_CS_8EAX_EDX_IB:
	case M_CS_8EAX_EBX_IB:
	case M_CS_8EAX_ESP_IB:
	case M_CS_8EAX_EBP_IB:
	case M_CS_8EAX_ESI_IB:
	case M_CS_8EAX_EDI_IB:
	case M_CS_8ECX_EAX_IB:
	case M_CS_9ECX_IB:
	case M_CS_8ECX_EDX_IB:
	case M_CS_8ECX_EBX_IB:
	case M_CS_8ECX_ESP_IB:
	case M_CS_8ECX_EBP_IB:
	case M_CS_8ECX_ESI_IB:
	case M_CS_8ECX_EDI_IB:
	case M_CS_8EDX_EAX_IB:
	case M_CS_8EDX_ECX_IB:
	case M_CS_9EDX_IB:
	case M_CS_8EDX_EBX_IB:
	case M_CS_8EDX_ESP_IB:
	case M_CS_8EDX_EBP_IB:
	case M_CS_8EDX_ESI_IB:
	case M_CS_8EDX_EDI_IB:
	case M_CS_8EBX_EAX_IB:
	case M_CS_8EBX_ECX_IB:
	case M_CS_8EBX_EDX_IB:
	case M_CS_9EBX_IB:
	case M_CS_8EBX_ESP_IB:
	case M_CS_8EBX_EBP_IB:
	case M_CS_8EBX_ESI_IB:
	case M_CS_8EBX_EDI_IB:
	case M_CS_8EBP_EAX_IB:
	case M_CS_8EBP_ECX_IB:
	case M_CS_8EBP_EDX_IB:
	case M_CS_8EBP_EBX_IB:
	case M_CS_8EBP_ESP_IB:
	case M_CS_9EBP_IB:
	case M_CS_8EBP_ESI_IB:
	case M_CS_8EBP_EDI_IB:
	case M_CS_8ESI_EAX_IB:
	case M_CS_8ESI_ECX_IB:
	case M_CS_8ESI_EDX_IB:
	case M_CS_8ESI_EBX_IB:
	case M_CS_8ESI_ESP_IB:
	case M_CS_8ESI_EBP_IB:
	case M_CS_9ESI_IB:
	case M_CS_8ESI_EDI_IB:
	case M_CS_8EDI_EAX_IB:
	case M_CS_8EDI_ECX_IB:
	case M_CS_8EDI_EDX_IB:
	case M_CS_8EDI_EBX_IB:
	case M_CS_8EDI_ESP_IB:
	case M_CS_8EDI_EBP_IB:
	case M_CS_8EDI_ESI_IB:
	case M_CS_9EDI_IB:
	case M_SS_2EAX_IB:
	case M_SS_EAX_ECX_IB:
	case M_SS_EAX_EDX_IB:
	case M_SS_EAX_EBX_IB:
	case M_SS_EAX_ESP_IB:
	case M_SS_EAX_EBP_IB:
	case M_SS_EAX_ESI_IB:
	case M_SS_EAX_EDI_IB:
	case M_SS_2ECX_IB:
	case M_SS_ECX_EDX_IB:
	case M_SS_ECX_EBX_IB:
	case M_SS_ECX_ESP_IB:
	case M_SS_ECX_EBP_IB:
	case M_SS_ECX_ESI_IB:
	case M_SS_ECX_EDI_IB:
	case M_SS_2EDX_IB:
	case M_SS_EDX_EBX_IB:
	case M_SS_EDX_ESP_IB:
	case M_SS_EDX_EBP_IB:
	case M_SS_EDX_ESI_IB:
	case M_SS_EDX_EDI_IB:
	case M_SS_2EBX_IB:
	case M_SS_EBX_ESP_IB:
	case M_SS_EBX_EBP_IB:
	case M_SS_EBX_ESI_IB:
	case M_SS_EBX_EDI_IB:
	case M_SS_ESP_IB:
	case M_SS_EBP_ESP_IB:
	case M_SS_2EBP_IB:
	case M_SS_EBP_ESI_IB:
	case M_SS_EBP_EDI_IB:
	case M_SS_ESI_ESP_IB:
	case M_SS_2ESI_IB:
	case M_SS_ESI_EDI_IB:
	case M_SS_EDI_ESP_IB:
	case M_SS_2EDI_IB:
	case M_SS_3EAX_IB:
	case M_SS_2EAX_ECX_IB:
	case M_SS_2EAX_EDX_IB:
	case M_SS_2EAX_EBX_IB:
	case M_SS_2EAX_ESP_IB:
	case M_SS_2EAX_EBP_IB:
	case M_SS_2EAX_ESI_IB:
	case M_SS_2EAX_EDI_IB:
	case M_SS_2ECX_EAX_IB:
	case M_SS_3ECX_IB:
	case M_SS_2ECX_EDX_IB:
	case M_SS_2ECX_EBX_IB:
	case M_SS_2ECX_ESP_IB:
	case M_SS_2ECX_EBP_IB:
	case M_SS_2ECX_ESI_IB:
	case M_SS_2ECX_EDI_IB:
	case M_SS_2EDX_EAX_IB:
	case M_SS_2EDX_ECX_IB:
	case M_SS_3EDX_IB:
	case M_SS_2EDX_EBX_IB:
	case M_SS_2EDX_ESP_IB:
	case M_SS_2EDX_EBP_IB:
	case M_SS_2EDX_ESI_IB:
	case M_SS_2EDX_EDI_IB:
	case M_SS_2EBX_EAX_IB:
	case M_SS_2EBX_ECX_IB:
	case M_SS_2EBX_EDX_IB:
	case M_SS_3EBX_IB:
	case M_SS_2EBX_ESP_IB:
	case M_SS_2EBX_EBP_IB:
	case M_SS_2EBX_ESI_IB:
	case M_SS_2EBX_EDI_IB:
	case M_SS_2EBP_EAX_IB:
	case M_SS_2EBP_ECX_IB:
	case M_SS_2EBP_EDX_IB:
	case M_SS_2EBP_EBX_IB:
	case M_SS_2EBP_ESP_IB:
	case M_SS_3EBP_IB:
	case M_SS_2EBP_ESI_IB:
	case M_SS_2EBP_EDI_IB:
	case M_SS_2ESI_EAX_IB:
	case M_SS_2ESI_ECX_IB:
	case M_SS_2ESI_EDX_IB:
	case M_SS_2ESI_EBX_IB:
	case M_SS_2ESI_ESP_IB:
	case M_SS_2ESI_EBP_IB:
	case M_SS_3ESI_IB:
	case M_SS_2ESI_EDI_IB:
	case M_SS_2EDI_EAX_IB:
	case M_SS_2EDI_ECX_IB:
	case M_SS_2EDI_EDX_IB:
	case M_SS_2EDI_EBX_IB:
	case M_SS_2EDI_ESP_IB:
	case M_SS_2EDI_EBP_IB:
	case M_SS_2EDI_ESI_IB:
	case M_SS_3EDI_IB:
	case M_SS_5EAX_IB:
	case M_SS_4EAX_ECX_IB:
	case M_SS_4EAX_EDX_IB:
	case M_SS_4EAX_EBX_IB:
	case M_SS_4EAX_ESP_IB:
	case M_SS_4EAX_EBP_IB:
	case M_SS_4EAX_ESI_IB:
	case M_SS_4EAX_EDI_IB:
	case M_SS_4ECX_EAX_IB:
	case M_SS_5ECX_IB:
	case M_SS_4ECX_EDX_IB:
	case M_SS_4ECX_EBX_IB:
	case M_SS_4ECX_ESP_IB:
	case M_SS_4ECX_EBP_IB:
	case M_SS_4ECX_ESI_IB:
	case M_SS_4ECX_EDI_IB:
	case M_SS_4EDX_EAX_IB:
	case M_SS_4EDX_ECX_IB:
	case M_SS_5EDX_IB:
	case M_SS_4EDX_EBX_IB:
	case M_SS_4EDX_ESP_IB:
	case M_SS_4EDX_EBP_IB:
	case M_SS_4EDX_ESI_IB:
	case M_SS_4EDX_EDI_IB:
	case M_SS_4EBX_EAX_IB:
	case M_SS_4EBX_ECX_IB:
	case M_SS_4EBX_EDX_IB:
	case M_SS_5EBX_IB:
	case M_SS_4EBX_ESP_IB:
	case M_SS_4EBX_EBP_IB:
	case M_SS_4EBX_ESI_IB:
	case M_SS_4EBX_EDI_IB:
	case M_SS_4EBP_EAX_IB:
	case M_SS_4EBP_ECX_IB:
	case M_SS_4EBP_EDX_IB:
	case M_SS_4EBP_EBX_IB:
	case M_SS_4EBP_ESP_IB:
	case M_SS_5EBP_IB:
	case M_SS_4EBP_ESI_IB:
	case M_SS_4EBP_EDI_IB:
	case M_SS_4ESI_EAX_IB:
	case M_SS_4ESI_ECX_IB:
	case M_SS_4ESI_EDX_IB:
	case M_SS_4ESI_EBX_IB:
	case M_SS_4ESI_ESP_IB:
	case M_SS_4ESI_EBP_IB:
	case M_SS_5ESI_IB:
	case M_SS_4ESI_EDI_IB:
	case M_SS_4EDI_EAX_IB:
	case M_SS_4EDI_ECX_IB:
	case M_SS_4EDI_EDX_IB:
	case M_SS_4EDI_EBX_IB:
	case M_SS_4EDI_ESP_IB:
	case M_SS_4EDI_EBP_IB:
	case M_SS_4EDI_ESI_IB:
	case M_SS_5EDI_IB:
	case M_SS_9EAX_IB:
	case M_SS_8EAX_ECX_IB:
	case M_SS_8EAX_EDX_IB:
	case M_SS_8EAX_EBX_IB:
	case M_SS_8EAX_ESP_IB:
	case M_SS_8EAX_EBP_IB:
	case M_SS_8EAX_ESI_IB:
	case M_SS_8EAX_EDI_IB:
	case M_SS_8ECX_EAX_IB:
	case M_SS_9ECX_IB:
	case M_SS_8ECX_EDX_IB:
	case M_SS_8ECX_EBX_IB:
	case M_SS_8ECX_ESP_IB:
	case M_SS_8ECX_EBP_IB:
	case M_SS_8ECX_ESI_IB:
	case M_SS_8ECX_EDI_IB:
	case M_SS_8EDX_EAX_IB:
	case M_SS_8EDX_ECX_IB:
	case M_SS_9EDX_IB:
	case M_SS_8EDX_EBX_IB:
	case M_SS_8EDX_ESP_IB:
	case M_SS_8EDX_EBP_IB:
	case M_SS_8EDX_ESI_IB:
	case M_SS_8EDX_EDI_IB:
	case M_SS_8EBX_EAX_IB:
	case M_SS_8EBX_ECX_IB:
	case M_SS_8EBX_EDX_IB:
	case M_SS_9EBX_IB:
	case M_SS_8EBX_ESP_IB:
	case M_SS_8EBX_EBP_IB:
	case M_SS_8EBX_ESI_IB:
	case M_SS_8EBX_EDI_IB:
	case M_SS_8EBP_EAX_IB:
	case M_SS_8EBP_ECX_IB:
	case M_SS_8EBP_EDX_IB:
	case M_SS_8EBP_EBX_IB:
	case M_SS_8EBP_ESP_IB:
	case M_SS_9EBP_IB:
	case M_SS_8EBP_ESI_IB:
	case M_SS_8EBP_EDI_IB:
	case M_SS_8ESI_EAX_IB:
	case M_SS_8ESI_ECX_IB:
	case M_SS_8ESI_EDX_IB:
	case M_SS_8ESI_EBX_IB:
	case M_SS_8ESI_ESP_IB:
	case M_SS_8ESI_EBP_IB:
	case M_SS_9ESI_IB:
	case M_SS_8ESI_EDI_IB:
	case M_SS_8EDI_EAX_IB:
	case M_SS_8EDI_ECX_IB:
	case M_SS_8EDI_EDX_IB:
	case M_SS_8EDI_EBX_IB:
	case M_SS_8EDI_ESP_IB:
	case M_SS_8EDI_EBP_IB:
	case M_SS_8EDI_ESI_IB:
	case M_SS_9EDI_IB:
	case M_DS_2EAX_IB:
	case M_DS_EAX_ECX_IB:
	case M_DS_EAX_EDX_IB:
	case M_DS_EAX_EBX_IB:
	case M_DS_EAX_ESP_IB:
	case M_DS_EAX_EBP_IB:
	case M_DS_EAX_ESI_IB:
	case M_DS_EAX_EDI_IB:
	case M_DS_2ECX_IB:
	case M_DS_ECX_EDX_IB:
	case M_DS_ECX_EBX_IB:
	case M_DS_ECX_ESP_IB:
	case M_DS_ECX_EBP_IB:
	case M_DS_ECX_ESI_IB:
	case M_DS_ECX_EDI_IB:
	case M_DS_2EDX_IB:
	case M_DS_EDX_EBX_IB:
	case M_DS_EDX_ESP_IB:
	case M_DS_EDX_EBP_IB:
	case M_DS_EDX_ESI_IB:
	case M_DS_EDX_EDI_IB:
	case M_DS_2EBX_IB:
	case M_DS_EBX_ESP_IB:
	case M_DS_EBX_EBP_IB:
	case M_DS_EBX_ESI_IB:
	case M_DS_EBX_EDI_IB:
	case M_DS_ESP_IB:
	case M_DS_EBP_ESP_IB:
	case M_DS_2EBP_IB:
	case M_DS_EBP_ESI_IB:
	case M_DS_EBP_EDI_IB:
	case M_DS_ESI_ESP_IB:
	case M_DS_2ESI_IB:
	case M_DS_ESI_EDI_IB:
	case M_DS_EDI_ESP_IB:
	case M_DS_2EDI_IB:
	case M_DS_3EAX_IB:
	case M_DS_2EAX_ECX_IB:
	case M_DS_2EAX_EDX_IB:
	case M_DS_2EAX_EBX_IB:
	case M_DS_2EAX_ESP_IB:
	case M_DS_2EAX_EBP_IB:
	case M_DS_2EAX_ESI_IB:
	case M_DS_2EAX_EDI_IB:
	case M_DS_2ECX_EAX_IB:
	case M_DS_3ECX_IB:
	case M_DS_2ECX_EDX_IB:
	case M_DS_2ECX_EBX_IB:
	case M_DS_2ECX_ESP_IB:
	case M_DS_2ECX_EBP_IB:
	case M_DS_2ECX_ESI_IB:
	case M_DS_2ECX_EDI_IB:
	case M_DS_2EDX_EAX_IB:
	case M_DS_2EDX_ECX_IB:
	case M_DS_3EDX_IB:
	case M_DS_2EDX_EBX_IB:
	case M_DS_2EDX_ESP_IB:
	case M_DS_2EDX_EBP_IB:
	case M_DS_2EDX_ESI_IB:
	case M_DS_2EDX_EDI_IB:
	case M_DS_2EBX_EAX_IB:
	case M_DS_2EBX_ECX_IB:
	case M_DS_2EBX_EDX_IB:
	case M_DS_3EBX_IB:
	case M_DS_2EBX_ESP_IB:
	case M_DS_2EBX_EBP_IB:
	case M_DS_2EBX_ESI_IB:
	case M_DS_2EBX_EDI_IB:
	case M_DS_2EBP_EAX_IB:
	case M_DS_2EBP_ECX_IB:
	case M_DS_2EBP_EDX_IB:
	case M_DS_2EBP_EBX_IB:
	case M_DS_2EBP_ESP_IB:
	case M_DS_3EBP_IB:
	case M_DS_2EBP_ESI_IB:
	case M_DS_2EBP_EDI_IB:
	case M_DS_2ESI_EAX_IB:
	case M_DS_2ESI_ECX_IB:
	case M_DS_2ESI_EDX_IB:
	case M_DS_2ESI_EBX_IB:
	case M_DS_2ESI_ESP_IB:
	case M_DS_2ESI_EBP_IB:
	case M_DS_3ESI_IB:
	case M_DS_2ESI_EDI_IB:
	case M_DS_2EDI_EAX_IB:
	case M_DS_2EDI_ECX_IB:
	case M_DS_2EDI_EDX_IB:
	case M_DS_2EDI_EBX_IB:
	case M_DS_2EDI_ESP_IB:
	case M_DS_2EDI_EBP_IB:
	case M_DS_2EDI_ESI_IB:
	case M_DS_3EDI_IB:
	case M_DS_5EAX_IB:
	case M_DS_4EAX_ECX_IB:
	case M_DS_4EAX_EDX_IB:
	case M_DS_4EAX_EBX_IB:
	case M_DS_4EAX_ESP_IB:
	case M_DS_4EAX_EBP_IB:
	case M_DS_4EAX_ESI_IB:
	case M_DS_4EAX_EDI_IB:
	case M_DS_4ECX_EAX_IB:
	case M_DS_5ECX_IB:
	case M_DS_4ECX_EDX_IB:
	case M_DS_4ECX_EBX_IB:
	case M_DS_4ECX_ESP_IB:
	case M_DS_4ECX_EBP_IB:
	case M_DS_4ECX_ESI_IB:
	case M_DS_4ECX_EDI_IB:
	case M_DS_4EDX_EAX_IB:
	case M_DS_4EDX_ECX_IB:
	case M_DS_5EDX_IB:
	case M_DS_4EDX_EBX_IB:
	case M_DS_4EDX_ESP_IB:
	case M_DS_4EDX_EBP_IB:
	case M_DS_4EDX_ESI_IB:
	case M_DS_4EDX_EDI_IB:
	case M_DS_4EBX_EAX_IB:
	case M_DS_4EBX_ECX_IB:
	case M_DS_4EBX_EDX_IB:
	case M_DS_5EBX_IB:
	case M_DS_4EBX_ESP_IB:
	case M_DS_4EBX_EBP_IB:
	case M_DS_4EBX_ESI_IB:
	case M_DS_4EBX_EDI_IB:
	case M_DS_4EBP_EAX_IB:
	case M_DS_4EBP_ECX_IB:
	case M_DS_4EBP_EDX_IB:
	case M_DS_4EBP_EBX_IB:
	case M_DS_4EBP_ESP_IB:
	case M_DS_5EBP_IB:
	case M_DS_4EBP_ESI_IB:
	case M_DS_4EBP_EDI_IB:
	case M_DS_4ESI_EAX_IB:
	case M_DS_4ESI_ECX_IB:
	case M_DS_4ESI_EDX_IB:
	case M_DS_4ESI_EBX_IB:
	case M_DS_4ESI_ESP_IB:
	case M_DS_4ESI_EBP_IB:
	case M_DS_5ESI_IB:
	case M_DS_4ESI_EDI_IB:
	case M_DS_4EDI_EAX_IB:
	case M_DS_4EDI_ECX_IB:
	case M_DS_4EDI_EDX_IB:
	case M_DS_4EDI_EBX_IB:
	case M_DS_4EDI_ESP_IB:
	case M_DS_4EDI_EBP_IB:
	case M_DS_4EDI_ESI_IB:
	case M_DS_5EDI_IB:
	case M_DS_9EAX_IB:
	case M_DS_8EAX_ECX_IB:
	case M_DS_8EAX_EDX_IB:
	case M_DS_8EAX_EBX_IB:
	case M_DS_8EAX_ESP_IB:
	case M_DS_8EAX_EBP_IB:
	case M_DS_8EAX_ESI_IB:
	case M_DS_8EAX_EDI_IB:
	case M_DS_8ECX_EAX_IB:
	case M_DS_9ECX_IB:
	case M_DS_8ECX_EDX_IB:
	case M_DS_8ECX_EBX_IB:
	case M_DS_8ECX_ESP_IB:
	case M_DS_8ECX_EBP_IB:
	case M_DS_8ECX_ESI_IB:
	case M_DS_8ECX_EDI_IB:
	case M_DS_8EDX_EAX_IB:
	case M_DS_8EDX_ECX_IB:
	case M_DS_9EDX_IB:
	case M_DS_8EDX_EBX_IB:
	case M_DS_8EDX_ESP_IB:
	case M_DS_8EDX_EBP_IB:
	case M_DS_8EDX_ESI_IB:
	case M_DS_8EDX_EDI_IB:
	case M_DS_8EBX_EAX_IB:
	case M_DS_8EBX_ECX_IB:
	case M_DS_8EBX_EDX_IB:
	case M_DS_9EBX_IB:
	case M_DS_8EBX_ESP_IB:
	case M_DS_8EBX_EBP_IB:
	case M_DS_8EBX_ESI_IB:
	case M_DS_8EBX_EDI_IB:
	case M_DS_8EBP_EAX_IB:
	case M_DS_8EBP_ECX_IB:
	case M_DS_8EBP_EDX_IB:
	case M_DS_8EBP_EBX_IB:
	case M_DS_8EBP_ESP_IB:
	case M_DS_9EBP_IB:
	case M_DS_8EBP_ESI_IB:
	case M_DS_8EBP_EDI_IB:
	case M_DS_8ESI_EAX_IB:
	case M_DS_8ESI_ECX_IB:
	case M_DS_8ESI_EDX_IB:
	case M_DS_8ESI_EBX_IB:
	case M_DS_8ESI_ESP_IB:
	case M_DS_8ESI_EBP_IB:
	case M_DS_9ESI_IB:
	case M_DS_8ESI_EDI_IB:
	case M_DS_8EDI_EAX_IB:
	case M_DS_8EDI_ECX_IB:
	case M_DS_8EDI_EDX_IB:
	case M_DS_8EDI_EBX_IB:
	case M_DS_8EDI_ESP_IB:
	case M_DS_8EDI_EBP_IB:
	case M_DS_8EDI_ESI_IB:
	case M_DS_9EDI_IB:
	case M_FS_2EAX_IB:
	case M_FS_EAX_ECX_IB:
	case M_FS_EAX_EDX_IB:
	case M_FS_EAX_EBX_IB:
	case M_FS_EAX_ESP_IB:
	case M_FS_EAX_EBP_IB:
	case M_FS_EAX_ESI_IB:
	case M_FS_EAX_EDI_IB:
	case M_FS_2ECX_IB:
	case M_FS_ECX_EDX_IB:
	case M_FS_ECX_EBX_IB:
	case M_FS_ECX_ESP_IB:
	case M_FS_ECX_EBP_IB:
	case M_FS_ECX_ESI_IB:
	case M_FS_ECX_EDI_IB:
	case M_FS_2EDX_IB:
	case M_FS_EDX_EBX_IB:
	case M_FS_EDX_ESP_IB:
	case M_FS_EDX_EBP_IB:
	case M_FS_EDX_ESI_IB:
	case M_FS_EDX_EDI_IB:
	case M_FS_2EBX_IB:
	case M_FS_EBX_ESP_IB:
	case M_FS_EBX_EBP_IB:
	case M_FS_EBX_ESI_IB:
	case M_FS_EBX_EDI_IB:
	case M_FS_ESP_IB:
	case M_FS_EBP_ESP_IB:
	case M_FS_2EBP_IB:
	case M_FS_EBP_ESI_IB:
	case M_FS_EBP_EDI_IB:
	case M_FS_ESI_ESP_IB:
	case M_FS_2ESI_IB:
	case M_FS_ESI_EDI_IB:
	case M_FS_EDI_ESP_IB:
	case M_FS_2EDI_IB:
	case M_FS_3EAX_IB:
	case M_FS_2EAX_ECX_IB:
	case M_FS_2EAX_EDX_IB:
	case M_FS_2EAX_EBX_IB:
	case M_FS_2EAX_ESP_IB:
	case M_FS_2EAX_EBP_IB:
	case M_FS_2EAX_ESI_IB:
	case M_FS_2EAX_EDI_IB:
	case M_FS_2ECX_EAX_IB:
	case M_FS_3ECX_IB:
	case M_FS_2ECX_EDX_IB:
	case M_FS_2ECX_EBX_IB:
	case M_FS_2ECX_ESP_IB:
	case M_FS_2ECX_EBP_IB:
	case M_FS_2ECX_ESI_IB:
	case M_FS_2ECX_EDI_IB:
	case M_FS_2EDX_EAX_IB:
	case M_FS_2EDX_ECX_IB:
	case M_FS_3EDX_IB:
	case M_FS_2EDX_EBX_IB:
	case M_FS_2EDX_ESP_IB:
	case M_FS_2EDX_EBP_IB:
	case M_FS_2EDX_ESI_IB:
	case M_FS_2EDX_EDI_IB:
	case M_FS_2EBX_EAX_IB:
	case M_FS_2EBX_ECX_IB:
	case M_FS_2EBX_EDX_IB:
	case M_FS_3EBX_IB:
	case M_FS_2EBX_ESP_IB:
	case M_FS_2EBX_EBP_IB:
	case M_FS_2EBX_ESI_IB:
	case M_FS_2EBX_EDI_IB:
	case M_FS_2EBP_EAX_IB:
	case M_FS_2EBP_ECX_IB:
	case M_FS_2EBP_EDX_IB:
	case M_FS_2EBP_EBX_IB:
	case M_FS_2EBP_ESP_IB:
	case M_FS_3EBP_IB:
	case M_FS_2EBP_ESI_IB:
	case M_FS_2EBP_EDI_IB:
	case M_FS_2ESI_EAX_IB:
	case M_FS_2ESI_ECX_IB:
	case M_FS_2ESI_EDX_IB:
	case M_FS_2ESI_EBX_IB:
	case M_FS_2ESI_ESP_IB:
	case M_FS_2ESI_EBP_IB:
	case M_FS_3ESI_IB:
	case M_FS_2ESI_EDI_IB:
	case M_FS_2EDI_EAX_IB:
	case M_FS_2EDI_ECX_IB:
	case M_FS_2EDI_EDX_IB:
	case M_FS_2EDI_EBX_IB:
	case M_FS_2EDI_ESP_IB:
	case M_FS_2EDI_EBP_IB:
	case M_FS_2EDI_ESI_IB:
	case M_FS_3EDI_IB:
	case M_FS_5EAX_IB:
	case M_FS_4EAX_ECX_IB:
	case M_FS_4EAX_EDX_IB:
	case M_FS_4EAX_EBX_IB:
	case M_FS_4EAX_ESP_IB:
	case M_FS_4EAX_EBP_IB:
	case M_FS_4EAX_ESI_IB:
	case M_FS_4EAX_EDI_IB:
	case M_FS_4ECX_EAX_IB:
	case M_FS_5ECX_IB:
	case M_FS_4ECX_EDX_IB:
	case M_FS_4ECX_EBX_IB:
	case M_FS_4ECX_ESP_IB:
	case M_FS_4ECX_EBP_IB:
	case M_FS_4ECX_ESI_IB:
	case M_FS_4ECX_EDI_IB:
	case M_FS_4EDX_EAX_IB:
	case M_FS_4EDX_ECX_IB:
	case M_FS_5EDX_IB:
	case M_FS_4EDX_EBX_IB:
	case M_FS_4EDX_ESP_IB:
	case M_FS_4EDX_EBP_IB:
	case M_FS_4EDX_ESI_IB:
	case M_FS_4EDX_EDI_IB:
	case M_FS_4EBX_EAX_IB:
	case M_FS_4EBX_ECX_IB:
	case M_FS_4EBX_EDX_IB:
	case M_FS_5EBX_IB:
	case M_FS_4EBX_ESP_IB:
	case M_FS_4EBX_EBP_IB:
	case M_FS_4EBX_ESI_IB:
	case M_FS_4EBX_EDI_IB:
	case M_FS_4EBP_EAX_IB:
	case M_FS_4EBP_ECX_IB:
	case M_FS_4EBP_EDX_IB:
	case M_FS_4EBP_EBX_IB:
	case M_FS_4EBP_ESP_IB:
	case M_FS_5EBP_IB:
	case M_FS_4EBP_ESI_IB:
	case M_FS_4EBP_EDI_IB:
	case M_FS_4ESI_EAX_IB:
	case M_FS_4ESI_ECX_IB:
	case M_FS_4ESI_EDX_IB:
	case M_FS_4ESI_EBX_IB:
	case M_FS_4ESI_ESP_IB:
	case M_FS_4ESI_EBP_IB:
	case M_FS_5ESI_IB:
	case M_FS_4ESI_EDI_IB:
	case M_FS_4EDI_EAX_IB:
	case M_FS_4EDI_ECX_IB:
	case M_FS_4EDI_EDX_IB:
	case M_FS_4EDI_EBX_IB:
	case M_FS_4EDI_ESP_IB:
	case M_FS_4EDI_EBP_IB:
	case M_FS_4EDI_ESI_IB:
	case M_FS_5EDI_IB:
	case M_FS_9EAX_IB:
	case M_FS_8EAX_ECX_IB:
	case M_FS_8EAX_EDX_IB:
	case M_FS_8EAX_EBX_IB:
	case M_FS_8EAX_ESP_IB:
	case M_FS_8EAX_EBP_IB:
	case M_FS_8EAX_ESI_IB:
	case M_FS_8EAX_EDI_IB:
	case M_FS_8ECX_EAX_IB:
	case M_FS_9ECX_IB:
	case M_FS_8ECX_EDX_IB:
	case M_FS_8ECX_EBX_IB:
	case M_FS_8ECX_ESP_IB:
	case M_FS_8ECX_EBP_IB:
	case M_FS_8ECX_ESI_IB:
	case M_FS_8ECX_EDI_IB:
	case M_FS_8EDX_EAX_IB:
	case M_FS_8EDX_ECX_IB:
	case M_FS_9EDX_IB:
	case M_FS_8EDX_EBX_IB:
	case M_FS_8EDX_ESP_IB:
	case M_FS_8EDX_EBP_IB:
	case M_FS_8EDX_ESI_IB:
	case M_FS_8EDX_EDI_IB:
	case M_FS_8EBX_EAX_IB:
	case M_FS_8EBX_ECX_IB:
	case M_FS_8EBX_EDX_IB:
	case M_FS_9EBX_IB:
	case M_FS_8EBX_ESP_IB:
	case M_FS_8EBX_EBP_IB:
	case M_FS_8EBX_ESI_IB:
	case M_FS_8EBX_EDI_IB:
	case M_FS_8EBP_EAX_IB:
	case M_FS_8EBP_ECX_IB:
	case M_FS_8EBP_EDX_IB:
	case M_FS_8EBP_EBX_IB:
	case M_FS_8EBP_ESP_IB:
	case M_FS_9EBP_IB:
	case M_FS_8EBP_ESI_IB:
	case M_FS_8EBP_EDI_IB:
	case M_FS_8ESI_EAX_IB:
	case M_FS_8ESI_ECX_IB:
	case M_FS_8ESI_EDX_IB:
	case M_FS_8ESI_EBX_IB:
	case M_FS_8ESI_ESP_IB:
	case M_FS_8ESI_EBP_IB:
	case M_FS_9ESI_IB:
	case M_FS_8ESI_EDI_IB:
	case M_FS_8EDI_EAX_IB:
	case M_FS_8EDI_ECX_IB:
	case M_FS_8EDI_EDX_IB:
	case M_FS_8EDI_EBX_IB:
	case M_FS_8EDI_ESP_IB:
	case M_FS_8EDI_EBP_IB:
	case M_FS_8EDI_ESI_IB:
	case M_FS_9EDI_IB:
	case M_GS_2EAX_IB:
	case M_GS_EAX_ECX_IB:
	case M_GS_EAX_EDX_IB:
	case M_GS_EAX_EBX_IB:
	case M_GS_EAX_ESP_IB:
	case M_GS_EAX_EBP_IB:
	case M_GS_EAX_ESI_IB:
	case M_GS_EAX_EDI_IB:
	case M_GS_2ECX_IB:
	case M_GS_ECX_EDX_IB:
	case M_GS_ECX_EBX_IB:
	case M_GS_ECX_ESP_IB:
	case M_GS_ECX_EBP_IB:
	case M_GS_ECX_ESI_IB:
	case M_GS_ECX_EDI_IB:
	case M_GS_2EDX_IB:
	case M_GS_EDX_EBX_IB:
	case M_GS_EDX_ESP_IB:
	case M_GS_EDX_EBP_IB:
	case M_GS_EDX_ESI_IB:
	case M_GS_EDX_EDI_IB:
	case M_GS_2EBX_IB:
	case M_GS_EBX_ESP_IB:
	case M_GS_EBX_EBP_IB:
	case M_GS_EBX_ESI_IB:
	case M_GS_EBX_EDI_IB:
	case M_GS_ESP_IB:
	case M_GS_EBP_ESP_IB:
	case M_GS_2EBP_IB:
	case M_GS_EBP_ESI_IB:
	case M_GS_EBP_EDI_IB:
	case M_GS_ESI_ESP_IB:
	case M_GS_2ESI_IB:
	case M_GS_ESI_EDI_IB:
	case M_GS_EDI_ESP_IB:
	case M_GS_2EDI_IB:
	case M_GS_3EAX_IB:
	case M_GS_2EAX_ECX_IB:
	case M_GS_2EAX_EDX_IB:
	case M_GS_2EAX_EBX_IB:
	case M_GS_2EAX_ESP_IB:
	case M_GS_2EAX_EBP_IB:
	case M_GS_2EAX_ESI_IB:
	case M_GS_2EAX_EDI_IB:
	case M_GS_2ECX_EAX_IB:
	case M_GS_3ECX_IB:
	case M_GS_2ECX_EDX_IB:
	case M_GS_2ECX_EBX_IB:
	case M_GS_2ECX_ESP_IB:
	case M_GS_2ECX_EBP_IB:
	case M_GS_2ECX_ESI_IB:
	case M_GS_2ECX_EDI_IB:
	case M_GS_2EDX_EAX_IB:
	case M_GS_2EDX_ECX_IB:
	case M_GS_3EDX_IB:
	case M_GS_2EDX_EBX_IB:
	case M_GS_2EDX_ESP_IB:
	case M_GS_2EDX_EBP_IB:
	case M_GS_2EDX_ESI_IB:
	case M_GS_2EDX_EDI_IB:
	case M_GS_2EBX_EAX_IB:
	case M_GS_2EBX_ECX_IB:
	case M_GS_2EBX_EDX_IB:
	case M_GS_3EBX_IB:
	case M_GS_2EBX_ESP_IB:
	case M_GS_2EBX_EBP_IB:
	case M_GS_2EBX_ESI_IB:
	case M_GS_2EBX_EDI_IB:
	case M_GS_2EBP_EAX_IB:
	case M_GS_2EBP_ECX_IB:
	case M_GS_2EBP_EDX_IB:
	case M_GS_2EBP_EBX_IB:
	case M_GS_2EBP_ESP_IB:
	case M_GS_3EBP_IB:
	case M_GS_2EBP_ESI_IB:
	case M_GS_2EBP_EDI_IB:
	case M_GS_2ESI_EAX_IB:
	case M_GS_2ESI_ECX_IB:
	case M_GS_2ESI_EDX_IB:
	case M_GS_2ESI_EBX_IB:
	case M_GS_2ESI_ESP_IB:
	case M_GS_2ESI_EBP_IB:
	case M_GS_3ESI_IB:
	case M_GS_2ESI_EDI_IB:
	case M_GS_2EDI_EAX_IB:
	case M_GS_2EDI_ECX_IB:
	case M_GS_2EDI_EDX_IB:
	case M_GS_2EDI_EBX_IB:
	case M_GS_2EDI_ESP_IB:
	case M_GS_2EDI_EBP_IB:
	case M_GS_2EDI_ESI_IB:
	case M_GS_3EDI_IB:
	case M_GS_5EAX_IB:
	case M_GS_4EAX_ECX_IB:
	case M_GS_4EAX_EDX_IB:
	case M_GS_4EAX_EBX_IB:
	case M_GS_4EAX_ESP_IB:
	case M_GS_4EAX_EBP_IB:
	case M_GS_4EAX_ESI_IB:
	case M_GS_4EAX_EDI_IB:
	case M_GS_4ECX_EAX_IB:
	case M_GS_5ECX_IB:
	case M_GS_4ECX_EDX_IB:
	case M_GS_4ECX_EBX_IB:
	case M_GS_4ECX_ESP_IB:
	case M_GS_4ECX_EBP_IB:
	case M_GS_4ECX_ESI_IB:
	case M_GS_4ECX_EDI_IB:
	case M_GS_4EDX_EAX_IB:
	case M_GS_4EDX_ECX_IB:
	case M_GS_5EDX_IB:
	case M_GS_4EDX_EBX_IB:
	case M_GS_4EDX_ESP_IB:
	case M_GS_4EDX_EBP_IB:
	case M_GS_4EDX_ESI_IB:
	case M_GS_4EDX_EDI_IB:
	case M_GS_4EBX_EAX_IB:
	case M_GS_4EBX_ECX_IB:
	case M_GS_4EBX_EDX_IB:
	case M_GS_5EBX_IB:
	case M_GS_4EBX_ESP_IB:
	case M_GS_4EBX_EBP_IB:
	case M_GS_4EBX_ESI_IB:
	case M_GS_4EBX_EDI_IB:
	case M_GS_4EBP_EAX_IB:
	case M_GS_4EBP_ECX_IB:
	case M_GS_4EBP_EDX_IB:
	case M_GS_4EBP_EBX_IB:
	case M_GS_4EBP_ESP_IB:
	case M_GS_5EBP_IB:
	case M_GS_4EBP_ESI_IB:
	case M_GS_4EBP_EDI_IB:
	case M_GS_4ESI_EAX_IB:
	case M_GS_4ESI_ECX_IB:
	case M_GS_4ESI_EDX_IB:
	case M_GS_4ESI_EBX_IB:
	case M_GS_4ESI_ESP_IB:
	case M_GS_4ESI_EBP_IB:
	case M_GS_5ESI_IB:
	case M_GS_4ESI_EDI_IB:
	case M_GS_4EDI_EAX_IB:
	case M_GS_4EDI_ECX_IB:
	case M_GS_4EDI_EDX_IB:
	case M_GS_4EDI_EBX_IB:
	case M_GS_4EDI_ESP_IB:
	case M_GS_4EDI_EBP_IB:
	case M_GS_4EDI_ESI_IB:
	case M_GS_5EDI_IB:
	case M_GS_9EAX_IB:
	case M_GS_8EAX_ECX_IB:
	case M_GS_8EAX_EDX_IB:
	case M_GS_8EAX_EBX_IB:
	case M_GS_8EAX_ESP_IB:
	case M_GS_8EAX_EBP_IB:
	case M_GS_8EAX_ESI_IB:
	case M_GS_8EAX_EDI_IB:
	case M_GS_8ECX_EAX_IB:
	case M_GS_9ECX_IB:
	case M_GS_8ECX_EDX_IB:
	case M_GS_8ECX_EBX_IB:
	case M_GS_8ECX_ESP_IB:
	case M_GS_8ECX_EBP_IB:
	case M_GS_8ECX_ESI_IB:
	case M_GS_8ECX_EDI_IB:
	case M_GS_8EDX_EAX_IB:
	case M_GS_8EDX_ECX_IB:
	case M_GS_9EDX_IB:
	case M_GS_8EDX_EBX_IB:
	case M_GS_8EDX_ESP_IB:
	case M_GS_8EDX_EBP_IB:
	case M_GS_8EDX_ESI_IB:
	case M_GS_8EDX_EDI_IB:
	case M_GS_8EBX_EAX_IB:
	case M_GS_8EBX_ECX_IB:
	case M_GS_8EBX_EDX_IB:
	case M_GS_9EBX_IB:
	case M_GS_8EBX_ESP_IB:
	case M_GS_8EBX_EBP_IB:
	case M_GS_8EBX_ESI_IB:
	case M_GS_8EBX_EDI_IB:
	case M_GS_8EBP_EAX_IB:
	case M_GS_8EBP_ECX_IB:
	case M_GS_8EBP_EDX_IB:
	case M_GS_8EBP_EBX_IB:
	case M_GS_8EBP_ESP_IB:
	case M_GS_9EBP_IB:
	case M_GS_8EBP_ESI_IB:
	case M_GS_8EBP_EDI_IB:
	case M_GS_8ESI_EAX_IB:
	case M_GS_8ESI_ECX_IB:
	case M_GS_8ESI_EDX_IB:
	case M_GS_8ESI_EBX_IB:
	case M_GS_8ESI_ESP_IB:
	case M_GS_8ESI_EBP_IB:
	case M_GS_9ESI_IB:
	case M_GS_8ESI_EDI_IB:
	case M_GS_8EDI_EAX_IB:
	case M_GS_8EDI_ECX_IB:
	case M_GS_8EDI_EDX_IB:
	case M_GS_8EDI_EBX_IB:
	case M_GS_8EDI_ESP_IB:
	case M_GS_8EDI_EBP_IB:
	case M_GS_8EDI_ESI_IB:
	case M_GS_9EDI_IB:
	    return 1;

	case M_ES_IW:
	case M_ES_BX_SI_IW:
	case M_ES_BX_DI_IW:
	case M_ES_BP_SI_IW:
	case M_ES_BP_DI_IW:
	case M_ES_SI_IW:
	case M_ES_DI_IW:
	case M_ES_BP_IW:
	case M_ES_BX_IW:
	case M_CS_IW:
	case M_CS_BX_SI_IW:
	case M_CS_BX_DI_IW:
	case M_CS_BP_SI_IW:
	case M_CS_BP_DI_IW:
	case M_CS_SI_IW:
	case M_CS_DI_IW:
	case M_CS_BP_IW:
	case M_CS_BX_IW:
	case M_SS_IW:
	case M_SS_BX_SI_IW:
	case M_SS_BX_DI_IW:
	case M_SS_BP_SI_IW:
	case M_SS_BP_DI_IW:
	case M_SS_SI_IW:
	case M_SS_DI_IW:
	case M_SS_BP_IW:
	case M_SS_BX_IW:
	case M_DS_IW:
	case M_DS_BX_SI_IW:
	case M_DS_BX_DI_IW:
	case M_DS_BP_SI_IW:
	case M_DS_BP_DI_IW:
	case M_DS_SI_IW:
	case M_DS_DI_IW:
	case M_DS_BP_IW:
	case M_DS_BX_IW:
	case M_FS_IW:
	case M_FS_BX_SI_IW:
	case M_FS_BX_DI_IW:
	case M_FS_BP_SI_IW:
	case M_FS_BP_DI_IW:
	case M_FS_SI_IW:
	case M_FS_DI_IW:
	case M_FS_BP_IW:
	case M_FS_BX_IW:
	case M_GS_IW:
	case M_GS_BX_SI_IW:
	case M_GS_BX_DI_IW:
	case M_GS_BP_SI_IW:
	case M_GS_BP_DI_IW:
	case M_GS_SI_IW:
	case M_GS_DI_IW:
	case M_GS_BP_IW:
	case M_GS_BX_IW:
	    return 2;

	case M_ES_ID:
	case M_ES_EAX_ID:
	case M_ES_ECX_ID:
	case M_ES_EDX_ID:
	case M_ES_EBX_ID:
	case M_ES_EBP_ID:
	case M_ES_ESI_ID:
	case M_ES_EDI_ID:
	case M_CS_ID:
	case M_CS_EAX_ID:
	case M_CS_ECX_ID:
	case M_CS_EDX_ID:
	case M_CS_EBX_ID:
	case M_CS_EBP_ID:
	case M_CS_ESI_ID:
	case M_CS_EDI_ID:
	case M_SS_ID:
	case M_SS_EAX_ID:
	case M_SS_ECX_ID:
	case M_SS_EDX_ID:
	case M_SS_EBX_ID:
	case M_SS_EBP_ID:
	case M_SS_ESI_ID:
	case M_SS_EDI_ID:
	case M_DS_ID:
	case M_DS_EAX_ID:
	case M_DS_ECX_ID:
	case M_DS_EDX_ID:
	case M_DS_EBX_ID:
	case M_DS_EBP_ID:
	case M_DS_ESI_ID:
	case M_DS_EDI_ID:
	case M_FS_ID:
	case M_FS_EAX_ID:
	case M_FS_ECX_ID:
	case M_FS_EDX_ID:
	case M_FS_EBX_ID:
	case M_FS_EBP_ID:
	case M_FS_ESI_ID:
	case M_FS_EDI_ID:
	case M_GS_ID:
	case M_GS_EAX_ID:
	case M_GS_ECX_ID:
	case M_GS_EDX_ID:
	case M_GS_EBX_ID:
	case M_GS_EBP_ID:
	case M_GS_ESI_ID:
	case M_GS_EDI_ID:
	case M_ES_2EAX_ID:
	case M_ES_2ECX_ID:
	case M_ES_2EDX_ID:
	case M_ES_2EBX_ID:
	case M_ES_2EBP_ID:
	case M_ES_2ESI_ID:
	case M_ES_2EDI_ID:
	case M_ES_4EAX_ID:
	case M_ES_4ECX_ID:
	case M_ES_4EDX_ID:
	case M_ES_4EBX_ID:
	case M_ES_4EBP_ID:
	case M_ES_4ESI_ID:
	case M_ES_4EDI_ID:
	case M_ES_8EAX_ID:
	case M_ES_8ECX_ID:
	case M_ES_8EDX_ID:
	case M_ES_8EBX_ID:
	case M_ES_8EBP_ID:
	case M_ES_8ESI_ID:
	case M_ES_8EDI_ID:
	case M_ES_EAX_ECX_ID:
	case M_ES_EAX_EDX_ID:
	case M_ES_EAX_EBX_ID:
	case M_ES_EAX_ESP_ID:
	case M_ES_EAX_EBP_ID:
	case M_ES_EAX_ESI_ID:
	case M_ES_EAX_EDI_ID:
	case M_ES_ECX_EDX_ID:
	case M_ES_ECX_EBX_ID:
	case M_ES_ECX_ESP_ID:
	case M_ES_ECX_EBP_ID:
	case M_ES_ECX_ESI_ID:
	case M_ES_ECX_EDI_ID:
	case M_ES_EDX_EBX_ID:
	case M_ES_EDX_ESP_ID:
	case M_ES_EDX_EBP_ID:
	case M_ES_EDX_ESI_ID:
	case M_ES_EDX_EDI_ID:
	case M_ES_EBX_ESP_ID:
	case M_ES_EBX_EBP_ID:
	case M_ES_EBX_ESI_ID:
	case M_ES_EBX_EDI_ID:
	case M_ES_ESP_ID:
	case M_ES_EBP_ESP_ID:
	case M_ES_EBP_ESI_ID:
	case M_ES_EBP_EDI_ID:
	case M_ES_ESI_ESP_ID:
	case M_ES_ESI_EDI_ID:
	case M_ES_EDI_ESP_ID:
	case M_ES_3EAX_ID:
	case M_ES_2EAX_ECX_ID:
	case M_ES_2EAX_EDX_ID:
	case M_ES_2EAX_EBX_ID:
	case M_ES_2EAX_ESP_ID:
	case M_ES_2EAX_EBP_ID:
	case M_ES_2EAX_ESI_ID:
	case M_ES_2EAX_EDI_ID:
	case M_ES_2ECX_EAX_ID:
	case M_ES_3ECX_ID:
	case M_ES_2ECX_EDX_ID:
	case M_ES_2ECX_EBX_ID:
	case M_ES_2ECX_ESP_ID:
	case M_ES_2ECX_EBP_ID:
	case M_ES_2ECX_ESI_ID:
	case M_ES_2ECX_EDI_ID:
	case M_ES_2EDX_EAX_ID:
	case M_ES_2EDX_ECX_ID:
	case M_ES_3EDX_ID:
	case M_ES_2EDX_EBX_ID:
	case M_ES_2EDX_ESP_ID:
	case M_ES_2EDX_EBP_ID:
	case M_ES_2EDX_ESI_ID:
	case M_ES_2EDX_EDI_ID:
	case M_ES_2EBX_EAX_ID:
	case M_ES_2EBX_ECX_ID:
	case M_ES_2EBX_EDX_ID:
	case M_ES_3EBX_ID:
	case M_ES_2EBX_ESP_ID:
	case M_ES_2EBX_EBP_ID:
	case M_ES_2EBX_ESI_ID:
	case M_ES_2EBX_EDI_ID:
	case M_ES_2EBP_EAX_ID:
	case M_ES_2EBP_ECX_ID:
	case M_ES_2EBP_EDX_ID:
	case M_ES_2EBP_EBX_ID:
	case M_ES_2EBP_ESP_ID:
	case M_ES_3EBP_ID:
	case M_ES_2EBP_ESI_ID:
	case M_ES_2EBP_EDI_ID:
	case M_ES_2ESI_EAX_ID:
	case M_ES_2ESI_ECX_ID:
	case M_ES_2ESI_EDX_ID:
	case M_ES_2ESI_EBX_ID:
	case M_ES_2ESI_ESP_ID:
	case M_ES_2ESI_EBP_ID:
	case M_ES_3ESI_ID:
	case M_ES_2ESI_EDI_ID:
	case M_ES_2EDI_EAX_ID:
	case M_ES_2EDI_ECX_ID:
	case M_ES_2EDI_EDX_ID:
	case M_ES_2EDI_EBX_ID:
	case M_ES_2EDI_ESP_ID:
	case M_ES_2EDI_EBP_ID:
	case M_ES_2EDI_ESI_ID:
	case M_ES_3EDI_ID:
	case M_ES_5EAX_ID:
	case M_ES_4EAX_ECX_ID:
	case M_ES_4EAX_EDX_ID:
	case M_ES_4EAX_EBX_ID:
	case M_ES_4EAX_ESP_ID:
	case M_ES_4EAX_EBP_ID:
	case M_ES_4EAX_ESI_ID:
	case M_ES_4EAX_EDI_ID:
	case M_ES_4ECX_EAX_ID:
	case M_ES_5ECX_ID:
	case M_ES_4ECX_EDX_ID:
	case M_ES_4ECX_EBX_ID:
	case M_ES_4ECX_ESP_ID:
	case M_ES_4ECX_EBP_ID:
	case M_ES_4ECX_ESI_ID:
	case M_ES_4ECX_EDI_ID:
	case M_ES_4EDX_EAX_ID:
	case M_ES_4EDX_ECX_ID:
	case M_ES_5EDX_ID:
	case M_ES_4EDX_EBX_ID:
	case M_ES_4EDX_ESP_ID:
	case M_ES_4EDX_EBP_ID:
	case M_ES_4EDX_ESI_ID:
	case M_ES_4EDX_EDI_ID:
	case M_ES_4EBX_EAX_ID:
	case M_ES_4EBX_ECX_ID:
	case M_ES_4EBX_EDX_ID:
	case M_ES_5EBX_ID:
	case M_ES_4EBX_ESP_ID:
	case M_ES_4EBX_EBP_ID:
	case M_ES_4EBX_ESI_ID:
	case M_ES_4EBX_EDI_ID:
	case M_ES_4EBP_EAX_ID:
	case M_ES_4EBP_ECX_ID:
	case M_ES_4EBP_EDX_ID:
	case M_ES_4EBP_EBX_ID:
	case M_ES_4EBP_ESP_ID:
	case M_ES_5EBP_ID:
	case M_ES_4EBP_ESI_ID:
	case M_ES_4EBP_EDI_ID:
	case M_ES_4ESI_EAX_ID:
	case M_ES_4ESI_ECX_ID:
	case M_ES_4ESI_EDX_ID:
	case M_ES_4ESI_EBX_ID:
	case M_ES_4ESI_ESP_ID:
	case M_ES_4ESI_EBP_ID:
	case M_ES_5ESI_ID:
	case M_ES_4ESI_EDI_ID:
	case M_ES_4EDI_EAX_ID:
	case M_ES_4EDI_ECX_ID:
	case M_ES_4EDI_EDX_ID:
	case M_ES_4EDI_EBX_ID:
	case M_ES_4EDI_ESP_ID:
	case M_ES_4EDI_EBP_ID:
	case M_ES_4EDI_ESI_ID:
	case M_ES_5EDI_ID:
	case M_ES_9EAX_ID:
	case M_ES_8EAX_ECX_ID:
	case M_ES_8EAX_EDX_ID:
	case M_ES_8EAX_EBX_ID:
	case M_ES_8EAX_ESP_ID:
	case M_ES_8EAX_EBP_ID:
	case M_ES_8EAX_ESI_ID:
	case M_ES_8EAX_EDI_ID:
	case M_ES_8ECX_EAX_ID:
	case M_ES_9ECX_ID:
	case M_ES_8ECX_EDX_ID:
	case M_ES_8ECX_EBX_ID:
	case M_ES_8ECX_ESP_ID:
	case M_ES_8ECX_EBP_ID:
	case M_ES_8ECX_ESI_ID:
	case M_ES_8ECX_EDI_ID:
	case M_ES_8EDX_EAX_ID:
	case M_ES_8EDX_ECX_ID:
	case M_ES_9EDX_ID:
	case M_ES_8EDX_EBX_ID:
	case M_ES_8EDX_ESP_ID:
	case M_ES_8EDX_EBP_ID:
	case M_ES_8EDX_ESI_ID:
	case M_ES_8EDX_EDI_ID:
	case M_ES_8EBX_EAX_ID:
	case M_ES_8EBX_ECX_ID:
	case M_ES_8EBX_EDX_ID:
	case M_ES_9EBX_ID:
	case M_ES_8EBX_ESP_ID:
	case M_ES_8EBX_EBP_ID:
	case M_ES_8EBX_ESI_ID:
	case M_ES_8EBX_EDI_ID:
	case M_ES_8EBP_EAX_ID:
	case M_ES_8EBP_ECX_ID:
	case M_ES_8EBP_EDX_ID:
	case M_ES_8EBP_EBX_ID:
	case M_ES_8EBP_ESP_ID:
	case M_ES_9EBP_ID:
	case M_ES_8EBP_ESI_ID:
	case M_ES_8EBP_EDI_ID:
	case M_ES_8ESI_EAX_ID:
	case M_ES_8ESI_ECX_ID:
	case M_ES_8ESI_EDX_ID:
	case M_ES_8ESI_EBX_ID:
	case M_ES_8ESI_ESP_ID:
	case M_ES_8ESI_EBP_ID:
	case M_ES_9ESI_ID:
	case M_ES_8ESI_EDI_ID:
	case M_ES_8EDI_EAX_ID:
	case M_ES_8EDI_ECX_ID:
	case M_ES_8EDI_EDX_ID:
	case M_ES_8EDI_EBX_ID:
	case M_ES_8EDI_ESP_ID:
	case M_ES_8EDI_EBP_ID:
	case M_ES_8EDI_ESI_ID:
	case M_ES_9EDI_ID:
	case M_CS_2EAX_ID:
	case M_CS_2ECX_ID:
	case M_CS_2EDX_ID:
	case M_CS_2EBX_ID:
	case M_CS_2EBP_ID:
	case M_CS_2ESI_ID:
	case M_CS_2EDI_ID:
	case M_CS_4EAX_ID:
	case M_CS_4ECX_ID:
	case M_CS_4EDX_ID:
	case M_CS_4EBX_ID:
	case M_CS_4EBP_ID:
	case M_CS_4ESI_ID:
	case M_CS_4EDI_ID:
	case M_CS_8EAX_ID:
	case M_CS_8ECX_ID:
	case M_CS_8EDX_ID:
	case M_CS_8EBX_ID:
	case M_CS_8EBP_ID:
	case M_CS_8ESI_ID:
	case M_CS_8EDI_ID:
	case M_CS_EAX_ECX_ID:
	case M_CS_EAX_EDX_ID:
	case M_CS_EAX_EBX_ID:
	case M_CS_EAX_ESP_ID:
	case M_CS_EAX_EBP_ID:
	case M_CS_EAX_ESI_ID:
	case M_CS_EAX_EDI_ID:
	case M_CS_ECX_EDX_ID:
	case M_CS_ECX_EBX_ID:
	case M_CS_ECX_ESP_ID:
	case M_CS_ECX_EBP_ID:
	case M_CS_ECX_ESI_ID:
	case M_CS_ECX_EDI_ID:
	case M_CS_EDX_EBX_ID:
	case M_CS_EDX_ESP_ID:
	case M_CS_EDX_EBP_ID:
	case M_CS_EDX_ESI_ID:
	case M_CS_EDX_EDI_ID:
	case M_CS_EBX_ESP_ID:
	case M_CS_EBX_EBP_ID:
	case M_CS_EBX_ESI_ID:
	case M_CS_EBX_EDI_ID:
	case M_CS_ESP_ID:
	case M_CS_EBP_ESP_ID:
	case M_CS_EBP_ESI_ID:
	case M_CS_EBP_EDI_ID:
	case M_CS_ESI_ESP_ID:
	case M_CS_ESI_EDI_ID:
	case M_CS_EDI_ESP_ID:
	case M_CS_3EAX_ID:
	case M_CS_2EAX_ECX_ID:
	case M_CS_2EAX_EDX_ID:
	case M_CS_2EAX_EBX_ID:
	case M_CS_2EAX_ESP_ID:
	case M_CS_2EAX_EBP_ID:
	case M_CS_2EAX_ESI_ID:
	case M_CS_2EAX_EDI_ID:
	case M_CS_2ECX_EAX_ID:
	case M_CS_3ECX_ID:
	case M_CS_2ECX_EDX_ID:
	case M_CS_2ECX_EBX_ID:
	case M_CS_2ECX_ESP_ID:
	case M_CS_2ECX_EBP_ID:
	case M_CS_2ECX_ESI_ID:
	case M_CS_2ECX_EDI_ID:
	case M_CS_2EDX_EAX_ID:
	case M_CS_2EDX_ECX_ID:
	case M_CS_3EDX_ID:
	case M_CS_2EDX_EBX_ID:
	case M_CS_2EDX_ESP_ID:
	case M_CS_2EDX_EBP_ID:
	case M_CS_2EDX_ESI_ID:
	case M_CS_2EDX_EDI_ID:
	case M_CS_2EBX_EAX_ID:
	case M_CS_2EBX_ECX_ID:
	case M_CS_2EBX_EDX_ID:
	case M_CS_3EBX_ID:
	case M_CS_2EBX_ESP_ID:
	case M_CS_2EBX_EBP_ID:
	case M_CS_2EBX_ESI_ID:
	case M_CS_2EBX_EDI_ID:
	case M_CS_2EBP_EAX_ID:
	case M_CS_2EBP_ECX_ID:
	case M_CS_2EBP_EDX_ID:
	case M_CS_2EBP_EBX_ID:
	case M_CS_2EBP_ESP_ID:
	case M_CS_3EBP_ID:
	case M_CS_2EBP_ESI_ID:
	case M_CS_2EBP_EDI_ID:
	case M_CS_2ESI_EAX_ID:
	case M_CS_2ESI_ECX_ID:
	case M_CS_2ESI_EDX_ID:
	case M_CS_2ESI_EBX_ID:
	case M_CS_2ESI_ESP_ID:
	case M_CS_2ESI_EBP_ID:
	case M_CS_3ESI_ID:
	case M_CS_2ESI_EDI_ID:
	case M_CS_2EDI_EAX_ID:
	case M_CS_2EDI_ECX_ID:
	case M_CS_2EDI_EDX_ID:
	case M_CS_2EDI_EBX_ID:
	case M_CS_2EDI_ESP_ID:
	case M_CS_2EDI_EBP_ID:
	case M_CS_2EDI_ESI_ID:
	case M_CS_3EDI_ID:
	case M_CS_5EAX_ID:
	case M_CS_4EAX_ECX_ID:
	case M_CS_4EAX_EDX_ID:
	case M_CS_4EAX_EBX_ID:
	case M_CS_4EAX_ESP_ID:
	case M_CS_4EAX_EBP_ID:
	case M_CS_4EAX_ESI_ID:
	case M_CS_4EAX_EDI_ID:
	case M_CS_4ECX_EAX_ID:
	case M_CS_5ECX_ID:
	case M_CS_4ECX_EDX_ID:
	case M_CS_4ECX_EBX_ID:
	case M_CS_4ECX_ESP_ID:
	case M_CS_4ECX_EBP_ID:
	case M_CS_4ECX_ESI_ID:
	case M_CS_4ECX_EDI_ID:
	case M_CS_4EDX_EAX_ID:
	case M_CS_4EDX_ECX_ID:
	case M_CS_5EDX_ID:
	case M_CS_4EDX_EBX_ID:
	case M_CS_4EDX_ESP_ID:
	case M_CS_4EDX_EBP_ID:
	case M_CS_4EDX_ESI_ID:
	case M_CS_4EDX_EDI_ID:
	case M_CS_4EBX_EAX_ID:
	case M_CS_4EBX_ECX_ID:
	case M_CS_4EBX_EDX_ID:
	case M_CS_5EBX_ID:
	case M_CS_4EBX_ESP_ID:
	case M_CS_4EBX_EBP_ID:
	case M_CS_4EBX_ESI_ID:
	case M_CS_4EBX_EDI_ID:
	case M_CS_4EBP_EAX_ID:
	case M_CS_4EBP_ECX_ID:
	case M_CS_4EBP_EDX_ID:
	case M_CS_4EBP_EBX_ID:
	case M_CS_4EBP_ESP_ID:
	case M_CS_5EBP_ID:
	case M_CS_4EBP_ESI_ID:
	case M_CS_4EBP_EDI_ID:
	case M_CS_4ESI_EAX_ID:
	case M_CS_4ESI_ECX_ID:
	case M_CS_4ESI_EDX_ID:
	case M_CS_4ESI_EBX_ID:
	case M_CS_4ESI_ESP_ID:
	case M_CS_4ESI_EBP_ID:
	case M_CS_5ESI_ID:
	case M_CS_4ESI_EDI_ID:
	case M_CS_4EDI_EAX_ID:
	case M_CS_4EDI_ECX_ID:
	case M_CS_4EDI_EDX_ID:
	case M_CS_4EDI_EBX_ID:
	case M_CS_4EDI_ESP_ID:
	case M_CS_4EDI_EBP_ID:
	case M_CS_4EDI_ESI_ID:
	case M_CS_5EDI_ID:
	case M_CS_9EAX_ID:
	case M_CS_8EAX_ECX_ID:
	case M_CS_8EAX_EDX_ID:
	case M_CS_8EAX_EBX_ID:
	case M_CS_8EAX_ESP_ID:
	case M_CS_8EAX_EBP_ID:
	case M_CS_8EAX_ESI_ID:
	case M_CS_8EAX_EDI_ID:
	case M_CS_8ECX_EAX_ID:
	case M_CS_9ECX_ID:
	case M_CS_8ECX_EDX_ID:
	case M_CS_8ECX_EBX_ID:
	case M_CS_8ECX_ESP_ID:
	case M_CS_8ECX_EBP_ID:
	case M_CS_8ECX_ESI_ID:
	case M_CS_8ECX_EDI_ID:
	case M_CS_8EDX_EAX_ID:
	case M_CS_8EDX_ECX_ID:
	case M_CS_9EDX_ID:
	case M_CS_8EDX_EBX_ID:
	case M_CS_8EDX_ESP_ID:
	case M_CS_8EDX_EBP_ID:
	case M_CS_8EDX_ESI_ID:
	case M_CS_8EDX_EDI_ID:
	case M_CS_8EBX_EAX_ID:
	case M_CS_8EBX_ECX_ID:
	case M_CS_8EBX_EDX_ID:
	case M_CS_9EBX_ID:
	case M_CS_8EBX_ESP_ID:
	case M_CS_8EBX_EBP_ID:
	case M_CS_8EBX_ESI_ID:
	case M_CS_8EBX_EDI_ID:
	case M_CS_8EBP_EAX_ID:
	case M_CS_8EBP_ECX_ID:
	case M_CS_8EBP_EDX_ID:
	case M_CS_8EBP_EBX_ID:
	case M_CS_8EBP_ESP_ID:
	case M_CS_9EBP_ID:
	case M_CS_8EBP_ESI_ID:
	case M_CS_8EBP_EDI_ID:
	case M_CS_8ESI_EAX_ID:
	case M_CS_8ESI_ECX_ID:
	case M_CS_8ESI_EDX_ID:
	case M_CS_8ESI_EBX_ID:
	case M_CS_8ESI_ESP_ID:
	case M_CS_8ESI_EBP_ID:
	case M_CS_9ESI_ID:
	case M_CS_8ESI_EDI_ID:
	case M_CS_8EDI_EAX_ID:
	case M_CS_8EDI_ECX_ID:
	case M_CS_8EDI_EDX_ID:
	case M_CS_8EDI_EBX_ID:
	case M_CS_8EDI_ESP_ID:
	case M_CS_8EDI_EBP_ID:
	case M_CS_8EDI_ESI_ID:
	case M_CS_9EDI_ID:
	case M_SS_2EAX_ID:
	case M_SS_2ECX_ID:
	case M_SS_2EDX_ID:
	case M_SS_2EBX_ID:
	case M_SS_2EBP_ID:
	case M_SS_2ESI_ID:
	case M_SS_2EDI_ID:
	case M_SS_4EAX_ID:
	case M_SS_4ECX_ID:
	case M_SS_4EDX_ID:
	case M_SS_4EBX_ID:
	case M_SS_4EBP_ID:
	case M_SS_4ESI_ID:
	case M_SS_4EDI_ID:
	case M_SS_8EAX_ID:
	case M_SS_8ECX_ID:
	case M_SS_8EDX_ID:
	case M_SS_8EBX_ID:
	case M_SS_8EBP_ID:
	case M_SS_8ESI_ID:
	case M_SS_8EDI_ID:
	case M_SS_EAX_ECX_ID:
	case M_SS_EAX_EDX_ID:
	case M_SS_EAX_EBX_ID:
	case M_SS_EAX_ESP_ID:
	case M_SS_EAX_EBP_ID:
	case M_SS_EAX_ESI_ID:
	case M_SS_EAX_EDI_ID:
	case M_SS_ECX_EDX_ID:
	case M_SS_ECX_EBX_ID:
	case M_SS_ECX_ESP_ID:
	case M_SS_ECX_EBP_ID:
	case M_SS_ECX_ESI_ID:
	case M_SS_ECX_EDI_ID:
	case M_SS_EDX_EBX_ID:
	case M_SS_EDX_ESP_ID:
	case M_SS_EDX_EBP_ID:
	case M_SS_EDX_ESI_ID:
	case M_SS_EDX_EDI_ID:
	case M_SS_EBX_ESP_ID:
	case M_SS_EBX_EBP_ID:
	case M_SS_EBX_ESI_ID:
	case M_SS_EBX_EDI_ID:
	case M_SS_ESP_ID:
	case M_SS_EBP_ESP_ID:
	case M_SS_EBP_ESI_ID:
	case M_SS_EBP_EDI_ID:
	case M_SS_ESI_ESP_ID:
	case M_SS_ESI_EDI_ID:
	case M_SS_EDI_ESP_ID:
	case M_SS_3EAX_ID:
	case M_SS_2EAX_ECX_ID:
	case M_SS_2EAX_EDX_ID:
	case M_SS_2EAX_EBX_ID:
	case M_SS_2EAX_ESP_ID:
	case M_SS_2EAX_EBP_ID:
	case M_SS_2EAX_ESI_ID:
	case M_SS_2EAX_EDI_ID:
	case M_SS_2ECX_EAX_ID:
	case M_SS_3ECX_ID:
	case M_SS_2ECX_EDX_ID:
	case M_SS_2ECX_EBX_ID:
	case M_SS_2ECX_ESP_ID:
	case M_SS_2ECX_EBP_ID:
	case M_SS_2ECX_ESI_ID:
	case M_SS_2ECX_EDI_ID:
	case M_SS_2EDX_EAX_ID:
	case M_SS_2EDX_ECX_ID:
	case M_SS_3EDX_ID:
	case M_SS_2EDX_EBX_ID:
	case M_SS_2EDX_ESP_ID:
	case M_SS_2EDX_EBP_ID:
	case M_SS_2EDX_ESI_ID:
	case M_SS_2EDX_EDI_ID:
	case M_SS_2EBX_EAX_ID:
	case M_SS_2EBX_ECX_ID:
	case M_SS_2EBX_EDX_ID:
	case M_SS_3EBX_ID:
	case M_SS_2EBX_ESP_ID:
	case M_SS_2EBX_EBP_ID:
	case M_SS_2EBX_ESI_ID:
	case M_SS_2EBX_EDI_ID:
	case M_SS_2EBP_EAX_ID:
	case M_SS_2EBP_ECX_ID:
	case M_SS_2EBP_EDX_ID:
	case M_SS_2EBP_EBX_ID:
	case M_SS_2EBP_ESP_ID:
	case M_SS_3EBP_ID:
	case M_SS_2EBP_ESI_ID:
	case M_SS_2EBP_EDI_ID:
	case M_SS_2ESI_EAX_ID:
	case M_SS_2ESI_ECX_ID:
	case M_SS_2ESI_EDX_ID:
	case M_SS_2ESI_EBX_ID:
	case M_SS_2ESI_ESP_ID:
	case M_SS_2ESI_EBP_ID:
	case M_SS_3ESI_ID:
	case M_SS_2ESI_EDI_ID:
	case M_SS_2EDI_EAX_ID:
	case M_SS_2EDI_ECX_ID:
	case M_SS_2EDI_EDX_ID:
	case M_SS_2EDI_EBX_ID:
	case M_SS_2EDI_ESP_ID:
	case M_SS_2EDI_EBP_ID:
	case M_SS_2EDI_ESI_ID:
	case M_SS_3EDI_ID:
	case M_SS_5EAX_ID:
	case M_SS_4EAX_ECX_ID:
	case M_SS_4EAX_EDX_ID:
	case M_SS_4EAX_EBX_ID:
	case M_SS_4EAX_ESP_ID:
	case M_SS_4EAX_EBP_ID:
	case M_SS_4EAX_ESI_ID:
	case M_SS_4EAX_EDI_ID:
	case M_SS_4ECX_EAX_ID:
	case M_SS_5ECX_ID:
	case M_SS_4ECX_EDX_ID:
	case M_SS_4ECX_EBX_ID:
	case M_SS_4ECX_ESP_ID:
	case M_SS_4ECX_EBP_ID:
	case M_SS_4ECX_ESI_ID:
	case M_SS_4ECX_EDI_ID:
	case M_SS_4EDX_EAX_ID:
	case M_SS_4EDX_ECX_ID:
	case M_SS_5EDX_ID:
	case M_SS_4EDX_EBX_ID:
	case M_SS_4EDX_ESP_ID:
	case M_SS_4EDX_EBP_ID:
	case M_SS_4EDX_ESI_ID:
	case M_SS_4EDX_EDI_ID:
	case M_SS_4EBX_EAX_ID:
	case M_SS_4EBX_ECX_ID:
	case M_SS_4EBX_EDX_ID:
	case M_SS_5EBX_ID:
	case M_SS_4EBX_ESP_ID:
	case M_SS_4EBX_EBP_ID:
	case M_SS_4EBX_ESI_ID:
	case M_SS_4EBX_EDI_ID:
	case M_SS_4EBP_EAX_ID:
	case M_SS_4EBP_ECX_ID:
	case M_SS_4EBP_EDX_ID:
	case M_SS_4EBP_EBX_ID:
	case M_SS_4EBP_ESP_ID:
	case M_SS_5EBP_ID:
	case M_SS_4EBP_ESI_ID:
	case M_SS_4EBP_EDI_ID:
	case M_SS_4ESI_EAX_ID:
	case M_SS_4ESI_ECX_ID:
	case M_SS_4ESI_EDX_ID:
	case M_SS_4ESI_EBX_ID:
	case M_SS_4ESI_ESP_ID:
	case M_SS_4ESI_EBP_ID:
	case M_SS_5ESI_ID:
	case M_SS_4ESI_EDI_ID:
	case M_SS_4EDI_EAX_ID:
	case M_SS_4EDI_ECX_ID:
	case M_SS_4EDI_EDX_ID:
	case M_SS_4EDI_EBX_ID:
	case M_SS_4EDI_ESP_ID:
	case M_SS_4EDI_EBP_ID:
	case M_SS_4EDI_ESI_ID:
	case M_SS_5EDI_ID:
	case M_SS_9EAX_ID:
	case M_SS_8EAX_ECX_ID:
	case M_SS_8EAX_EDX_ID:
	case M_SS_8EAX_EBX_ID:
	case M_SS_8EAX_ESP_ID:
	case M_SS_8EAX_EBP_ID:
	case M_SS_8EAX_ESI_ID:
	case M_SS_8EAX_EDI_ID:
	case M_SS_8ECX_EAX_ID:
	case M_SS_9ECX_ID:
	case M_SS_8ECX_EDX_ID:
	case M_SS_8ECX_EBX_ID:
	case M_SS_8ECX_ESP_ID:
	case M_SS_8ECX_EBP_ID:
	case M_SS_8ECX_ESI_ID:
	case M_SS_8ECX_EDI_ID:
	case M_SS_8EDX_EAX_ID:
	case M_SS_8EDX_ECX_ID:
	case M_SS_9EDX_ID:
	case M_SS_8EDX_EBX_ID:
	case M_SS_8EDX_ESP_ID:
	case M_SS_8EDX_EBP_ID:
	case M_SS_8EDX_ESI_ID:
	case M_SS_8EDX_EDI_ID:
	case M_SS_8EBX_EAX_ID:
	case M_SS_8EBX_ECX_ID:
	case M_SS_8EBX_EDX_ID:
	case M_SS_9EBX_ID:
	case M_SS_8EBX_ESP_ID:
	case M_SS_8EBX_EBP_ID:
	case M_SS_8EBX_ESI_ID:
	case M_SS_8EBX_EDI_ID:
	case M_SS_8EBP_EAX_ID:
	case M_SS_8EBP_ECX_ID:
	case M_SS_8EBP_EDX_ID:
	case M_SS_8EBP_EBX_ID:
	case M_SS_8EBP_ESP_ID:
	case M_SS_9EBP_ID:
	case M_SS_8EBP_ESI_ID:
	case M_SS_8EBP_EDI_ID:
	case M_SS_8ESI_EAX_ID:
	case M_SS_8ESI_ECX_ID:
	case M_SS_8ESI_EDX_ID:
	case M_SS_8ESI_EBX_ID:
	case M_SS_8ESI_ESP_ID:
	case M_SS_8ESI_EBP_ID:
	case M_SS_9ESI_ID:
	case M_SS_8ESI_EDI_ID:
	case M_SS_8EDI_EAX_ID:
	case M_SS_8EDI_ECX_ID:
	case M_SS_8EDI_EDX_ID:
	case M_SS_8EDI_EBX_ID:
	case M_SS_8EDI_ESP_ID:
	case M_SS_8EDI_EBP_ID:
	case M_SS_8EDI_ESI_ID:
	case M_SS_9EDI_ID:
	case M_DS_2EAX_ID:
	case M_DS_2ECX_ID:
	case M_DS_2EDX_ID:
	case M_DS_2EBX_ID:
	case M_DS_2EBP_ID:
	case M_DS_2ESI_ID:
	case M_DS_2EDI_ID:
	case M_DS_4EAX_ID:
	case M_DS_4ECX_ID:
	case M_DS_4EDX_ID:
	case M_DS_4EBX_ID:
	case M_DS_4EBP_ID:
	case M_DS_4ESI_ID:
	case M_DS_4EDI_ID:
	case M_DS_8EAX_ID:
	case M_DS_8ECX_ID:
	case M_DS_8EDX_ID:
	case M_DS_8EBX_ID:
	case M_DS_8EBP_ID:
	case M_DS_8ESI_ID:
	case M_DS_8EDI_ID:
	case M_DS_EAX_ECX_ID:
	case M_DS_EAX_EDX_ID:
	case M_DS_EAX_EBX_ID:
	case M_DS_EAX_ESP_ID:
	case M_DS_EAX_EBP_ID:
	case M_DS_EAX_ESI_ID:
	case M_DS_EAX_EDI_ID:
	case M_DS_ECX_EDX_ID:
	case M_DS_ECX_EBX_ID:
	case M_DS_ECX_ESP_ID:
	case M_DS_ECX_EBP_ID:
	case M_DS_ECX_ESI_ID:
	case M_DS_ECX_EDI_ID:
	case M_DS_EDX_EBX_ID:
	case M_DS_EDX_ESP_ID:
	case M_DS_EDX_EBP_ID:
	case M_DS_EDX_ESI_ID:
	case M_DS_EDX_EDI_ID:
	case M_DS_EBX_ESP_ID:
	case M_DS_EBX_EBP_ID:
	case M_DS_EBX_ESI_ID:
	case M_DS_EBX_EDI_ID:
	case M_DS_ESP_ID:
	case M_DS_EBP_ESP_ID:
	case M_DS_EBP_ESI_ID:
	case M_DS_EBP_EDI_ID:
	case M_DS_ESI_ESP_ID:
	case M_DS_ESI_EDI_ID:
	case M_DS_EDI_ESP_ID:
	case M_DS_3EAX_ID:
	case M_DS_2EAX_ECX_ID:
	case M_DS_2EAX_EDX_ID:
	case M_DS_2EAX_EBX_ID:
	case M_DS_2EAX_ESP_ID:
	case M_DS_2EAX_EBP_ID:
	case M_DS_2EAX_ESI_ID:
	case M_DS_2EAX_EDI_ID:
	case M_DS_2ECX_EAX_ID:
	case M_DS_3ECX_ID:
	case M_DS_2ECX_EDX_ID:
	case M_DS_2ECX_EBX_ID:
	case M_DS_2ECX_ESP_ID:
	case M_DS_2ECX_EBP_ID:
	case M_DS_2ECX_ESI_ID:
	case M_DS_2ECX_EDI_ID:
	case M_DS_2EDX_EAX_ID:
	case M_DS_2EDX_ECX_ID:
	case M_DS_3EDX_ID:
	case M_DS_2EDX_EBX_ID:
	case M_DS_2EDX_ESP_ID:
	case M_DS_2EDX_EBP_ID:
	case M_DS_2EDX_ESI_ID:
	case M_DS_2EDX_EDI_ID:
	case M_DS_2EBX_EAX_ID:
	case M_DS_2EBX_ECX_ID:
	case M_DS_2EBX_EDX_ID:
	case M_DS_3EBX_ID:
	case M_DS_2EBX_ESP_ID:
	case M_DS_2EBX_EBP_ID:
	case M_DS_2EBX_ESI_ID:
	case M_DS_2EBX_EDI_ID:
	case M_DS_2EBP_EAX_ID:
	case M_DS_2EBP_ECX_ID:
	case M_DS_2EBP_EDX_ID:
	case M_DS_2EBP_EBX_ID:
	case M_DS_2EBP_ESP_ID:
	case M_DS_3EBP_ID:
	case M_DS_2EBP_ESI_ID:
	case M_DS_2EBP_EDI_ID:
	case M_DS_2ESI_EAX_ID:
	case M_DS_2ESI_ECX_ID:
	case M_DS_2ESI_EDX_ID:
	case M_DS_2ESI_EBX_ID:
	case M_DS_2ESI_ESP_ID:
	case M_DS_2ESI_EBP_ID:
	case M_DS_3ESI_ID:
	case M_DS_2ESI_EDI_ID:
	case M_DS_2EDI_EAX_ID:
	case M_DS_2EDI_ECX_ID:
	case M_DS_2EDI_EDX_ID:
	case M_DS_2EDI_EBX_ID:
	case M_DS_2EDI_ESP_ID:
	case M_DS_2EDI_EBP_ID:
	case M_DS_2EDI_ESI_ID:
	case M_DS_3EDI_ID:
	case M_DS_5EAX_ID:
	case M_DS_4EAX_ECX_ID:
	case M_DS_4EAX_EDX_ID:
	case M_DS_4EAX_EBX_ID:
	case M_DS_4EAX_ESP_ID:
	case M_DS_4EAX_EBP_ID:
	case M_DS_4EAX_ESI_ID:
	case M_DS_4EAX_EDI_ID:
	case M_DS_4ECX_EAX_ID:
	case M_DS_5ECX_ID:
	case M_DS_4ECX_EDX_ID:
	case M_DS_4ECX_EBX_ID:
	case M_DS_4ECX_ESP_ID:
	case M_DS_4ECX_EBP_ID:
	case M_DS_4ECX_ESI_ID:
	case M_DS_4ECX_EDI_ID:
	case M_DS_4EDX_EAX_ID:
	case M_DS_4EDX_ECX_ID:
	case M_DS_5EDX_ID:
	case M_DS_4EDX_EBX_ID:
	case M_DS_4EDX_ESP_ID:
	case M_DS_4EDX_EBP_ID:
	case M_DS_4EDX_ESI_ID:
	case M_DS_4EDX_EDI_ID:
	case M_DS_4EBX_EAX_ID:
	case M_DS_4EBX_ECX_ID:
	case M_DS_4EBX_EDX_ID:
	case M_DS_5EBX_ID:
	case M_DS_4EBX_ESP_ID:
	case M_DS_4EBX_EBP_ID:
	case M_DS_4EBX_ESI_ID:
	case M_DS_4EBX_EDI_ID:
	case M_DS_4EBP_EAX_ID:
	case M_DS_4EBP_ECX_ID:
	case M_DS_4EBP_EDX_ID:
	case M_DS_4EBP_EBX_ID:
	case M_DS_4EBP_ESP_ID:
	case M_DS_5EBP_ID:
	case M_DS_4EBP_ESI_ID:
	case M_DS_4EBP_EDI_ID:
	case M_DS_4ESI_EAX_ID:
	case M_DS_4ESI_ECX_ID:
	case M_DS_4ESI_EDX_ID:
	case M_DS_4ESI_EBX_ID:
	case M_DS_4ESI_ESP_ID:
	case M_DS_4ESI_EBP_ID:
	case M_DS_5ESI_ID:
	case M_DS_4ESI_EDI_ID:
	case M_DS_4EDI_EAX_ID:
	case M_DS_4EDI_ECX_ID:
	case M_DS_4EDI_EDX_ID:
	case M_DS_4EDI_EBX_ID:
	case M_DS_4EDI_ESP_ID:
	case M_DS_4EDI_EBP_ID:
	case M_DS_4EDI_ESI_ID:
	case M_DS_5EDI_ID:
	case M_DS_9EAX_ID:
	case M_DS_8EAX_ECX_ID:
	case M_DS_8EAX_EDX_ID:
	case M_DS_8EAX_EBX_ID:
	case M_DS_8EAX_ESP_ID:
	case M_DS_8EAX_EBP_ID:
	case M_DS_8EAX_ESI_ID:
	case M_DS_8EAX_EDI_ID:
	case M_DS_8ECX_EAX_ID:
	case M_DS_9ECX_ID:
	case M_DS_8ECX_EDX_ID:
	case M_DS_8ECX_EBX_ID:
	case M_DS_8ECX_ESP_ID:
	case M_DS_8ECX_EBP_ID:
	case M_DS_8ECX_ESI_ID:
	case M_DS_8ECX_EDI_ID:
	case M_DS_8EDX_EAX_ID:
	case M_DS_8EDX_ECX_ID:
	case M_DS_9EDX_ID:
	case M_DS_8EDX_EBX_ID:
	case M_DS_8EDX_ESP_ID:
	case M_DS_8EDX_EBP_ID:
	case M_DS_8EDX_ESI_ID:
	case M_DS_8EDX_EDI_ID:
	case M_DS_8EBX_EAX_ID:
	case M_DS_8EBX_ECX_ID:
	case M_DS_8EBX_EDX_ID:
	case M_DS_9EBX_ID:
	case M_DS_8EBX_ESP_ID:
	case M_DS_8EBX_EBP_ID:
	case M_DS_8EBX_ESI_ID:
	case M_DS_8EBX_EDI_ID:
	case M_DS_8EBP_EAX_ID:
	case M_DS_8EBP_ECX_ID:
	case M_DS_8EBP_EDX_ID:
	case M_DS_8EBP_EBX_ID:
	case M_DS_8EBP_ESP_ID:
	case M_DS_9EBP_ID:
	case M_DS_8EBP_ESI_ID:
	case M_DS_8EBP_EDI_ID:
	case M_DS_8ESI_EAX_ID:
	case M_DS_8ESI_ECX_ID:
	case M_DS_8ESI_EDX_ID:
	case M_DS_8ESI_EBX_ID:
	case M_DS_8ESI_ESP_ID:
	case M_DS_8ESI_EBP_ID:
	case M_DS_9ESI_ID:
	case M_DS_8ESI_EDI_ID:
	case M_DS_8EDI_EAX_ID:
	case M_DS_8EDI_ECX_ID:
	case M_DS_8EDI_EDX_ID:
	case M_DS_8EDI_EBX_ID:
	case M_DS_8EDI_ESP_ID:
	case M_DS_8EDI_EBP_ID:
	case M_DS_8EDI_ESI_ID:
	case M_DS_9EDI_ID:
	case M_FS_2EAX_ID:
	case M_FS_2ECX_ID:
	case M_FS_2EDX_ID:
	case M_FS_2EBX_ID:
	case M_FS_2EBP_ID:
	case M_FS_2ESI_ID:
	case M_FS_2EDI_ID:
	case M_FS_4EAX_ID:
	case M_FS_4ECX_ID:
	case M_FS_4EDX_ID:
	case M_FS_4EBX_ID:
	case M_FS_4EBP_ID:
	case M_FS_4ESI_ID:
	case M_FS_4EDI_ID:
	case M_FS_8EAX_ID:
	case M_FS_8ECX_ID:
	case M_FS_8EDX_ID:
	case M_FS_8EBX_ID:
	case M_FS_8EBP_ID:
	case M_FS_8ESI_ID:
	case M_FS_8EDI_ID:
	case M_FS_EAX_ECX_ID:
	case M_FS_EAX_EDX_ID:
	case M_FS_EAX_EBX_ID:
	case M_FS_EAX_ESP_ID:
	case M_FS_EAX_EBP_ID:
	case M_FS_EAX_ESI_ID:
	case M_FS_EAX_EDI_ID:
	case M_FS_ECX_EDX_ID:
	case M_FS_ECX_EBX_ID:
	case M_FS_ECX_ESP_ID:
	case M_FS_ECX_EBP_ID:
	case M_FS_ECX_ESI_ID:
	case M_FS_ECX_EDI_ID:
	case M_FS_EDX_EBX_ID:
	case M_FS_EDX_ESP_ID:
	case M_FS_EDX_EBP_ID:
	case M_FS_EDX_ESI_ID:
	case M_FS_EDX_EDI_ID:
	case M_FS_EBX_ESP_ID:
	case M_FS_EBX_EBP_ID:
	case M_FS_EBX_ESI_ID:
	case M_FS_EBX_EDI_ID:
	case M_FS_ESP_ID:
	case M_FS_EBP_ESP_ID:
	case M_FS_EBP_ESI_ID:
	case M_FS_EBP_EDI_ID:
	case M_FS_ESI_ESP_ID:
	case M_FS_ESI_EDI_ID:
	case M_FS_EDI_ESP_ID:
	case M_FS_3EAX_ID:
	case M_FS_2EAX_ECX_ID:
	case M_FS_2EAX_EDX_ID:
	case M_FS_2EAX_EBX_ID:
	case M_FS_2EAX_ESP_ID:
	case M_FS_2EAX_EBP_ID:
	case M_FS_2EAX_ESI_ID:
	case M_FS_2EAX_EDI_ID:
	case M_FS_2ECX_EAX_ID:
	case M_FS_3ECX_ID:
	case M_FS_2ECX_EDX_ID:
	case M_FS_2ECX_EBX_ID:
	case M_FS_2ECX_ESP_ID:
	case M_FS_2ECX_EBP_ID:
	case M_FS_2ECX_ESI_ID:
	case M_FS_2ECX_EDI_ID:
	case M_FS_2EDX_EAX_ID:
	case M_FS_2EDX_ECX_ID:
	case M_FS_3EDX_ID:
	case M_FS_2EDX_EBX_ID:
	case M_FS_2EDX_ESP_ID:
	case M_FS_2EDX_EBP_ID:
	case M_FS_2EDX_ESI_ID:
	case M_FS_2EDX_EDI_ID:
	case M_FS_2EBX_EAX_ID:
	case M_FS_2EBX_ECX_ID:
	case M_FS_2EBX_EDX_ID:
	case M_FS_3EBX_ID:
	case M_FS_2EBX_ESP_ID:
	case M_FS_2EBX_EBP_ID:
	case M_FS_2EBX_ESI_ID:
	case M_FS_2EBX_EDI_ID:
	case M_FS_2EBP_EAX_ID:
	case M_FS_2EBP_ECX_ID:
	case M_FS_2EBP_EDX_ID:
	case M_FS_2EBP_EBX_ID:
	case M_FS_2EBP_ESP_ID:
	case M_FS_3EBP_ID:
	case M_FS_2EBP_ESI_ID:
	case M_FS_2EBP_EDI_ID:
	case M_FS_2ESI_EAX_ID:
	case M_FS_2ESI_ECX_ID:
	case M_FS_2ESI_EDX_ID:
	case M_FS_2ESI_EBX_ID:
	case M_FS_2ESI_ESP_ID:
	case M_FS_2ESI_EBP_ID:
	case M_FS_3ESI_ID:
	case M_FS_2ESI_EDI_ID:
	case M_FS_2EDI_EAX_ID:
	case M_FS_2EDI_ECX_ID:
	case M_FS_2EDI_EDX_ID:
	case M_FS_2EDI_EBX_ID:
	case M_FS_2EDI_ESP_ID:
	case M_FS_2EDI_EBP_ID:
	case M_FS_2EDI_ESI_ID:
	case M_FS_3EDI_ID:
	case M_FS_5EAX_ID:
	case M_FS_4EAX_ECX_ID:
	case M_FS_4EAX_EDX_ID:
	case M_FS_4EAX_EBX_ID:
	case M_FS_4EAX_ESP_ID:
	case M_FS_4EAX_EBP_ID:
	case M_FS_4EAX_ESI_ID:
	case M_FS_4EAX_EDI_ID:
	case M_FS_4ECX_EAX_ID:
	case M_FS_5ECX_ID:
	case M_FS_4ECX_EDX_ID:
	case M_FS_4ECX_EBX_ID:
	case M_FS_4ECX_ESP_ID:
	case M_FS_4ECX_EBP_ID:
	case M_FS_4ECX_ESI_ID:
	case M_FS_4ECX_EDI_ID:
	case M_FS_4EDX_EAX_ID:
	case M_FS_4EDX_ECX_ID:
	case M_FS_5EDX_ID:
	case M_FS_4EDX_EBX_ID:
	case M_FS_4EDX_ESP_ID:
	case M_FS_4EDX_EBP_ID:
	case M_FS_4EDX_ESI_ID:
	case M_FS_4EDX_EDI_ID:
	case M_FS_4EBX_EAX_ID:
	case M_FS_4EBX_ECX_ID:
	case M_FS_4EBX_EDX_ID:
	case M_FS_5EBX_ID:
	case M_FS_4EBX_ESP_ID:
	case M_FS_4EBX_EBP_ID:
	case M_FS_4EBX_ESI_ID:
	case M_FS_4EBX_EDI_ID:
	case M_FS_4EBP_EAX_ID:
	case M_FS_4EBP_ECX_ID:
	case M_FS_4EBP_EDX_ID:
	case M_FS_4EBP_EBX_ID:
	case M_FS_4EBP_ESP_ID:
	case M_FS_5EBP_ID:
	case M_FS_4EBP_ESI_ID:
	case M_FS_4EBP_EDI_ID:
	case M_FS_4ESI_EAX_ID:
	case M_FS_4ESI_ECX_ID:
	case M_FS_4ESI_EDX_ID:
	case M_FS_4ESI_EBX_ID:
	case M_FS_4ESI_ESP_ID:
	case M_FS_4ESI_EBP_ID:
	case M_FS_5ESI_ID:
	case M_FS_4ESI_EDI_ID:
	case M_FS_4EDI_EAX_ID:
	case M_FS_4EDI_ECX_ID:
	case M_FS_4EDI_EDX_ID:
	case M_FS_4EDI_EBX_ID:
	case M_FS_4EDI_ESP_ID:
	case M_FS_4EDI_EBP_ID:
	case M_FS_4EDI_ESI_ID:
	case M_FS_5EDI_ID:
	case M_FS_9EAX_ID:
	case M_FS_8EAX_ECX_ID:
	case M_FS_8EAX_EDX_ID:
	case M_FS_8EAX_EBX_ID:
	case M_FS_8EAX_ESP_ID:
	case M_FS_8EAX_EBP_ID:
	case M_FS_8EAX_ESI_ID:
	case M_FS_8EAX_EDI_ID:
	case M_FS_8ECX_EAX_ID:
	case M_FS_9ECX_ID:
	case M_FS_8ECX_EDX_ID:
	case M_FS_8ECX_EBX_ID:
	case M_FS_8ECX_ESP_ID:
	case M_FS_8ECX_EBP_ID:
	case M_FS_8ECX_ESI_ID:
	case M_FS_8ECX_EDI_ID:
	case M_FS_8EDX_EAX_ID:
	case M_FS_8EDX_ECX_ID:
	case M_FS_9EDX_ID:
	case M_FS_8EDX_EBX_ID:
	case M_FS_8EDX_ESP_ID:
	case M_FS_8EDX_EBP_ID:
	case M_FS_8EDX_ESI_ID:
	case M_FS_8EDX_EDI_ID:
	case M_FS_8EBX_EAX_ID:
	case M_FS_8EBX_ECX_ID:
	case M_FS_8EBX_EDX_ID:
	case M_FS_9EBX_ID:
	case M_FS_8EBX_ESP_ID:
	case M_FS_8EBX_EBP_ID:
	case M_FS_8EBX_ESI_ID:
	case M_FS_8EBX_EDI_ID:
	case M_FS_8EBP_EAX_ID:
	case M_FS_8EBP_ECX_ID:
	case M_FS_8EBP_EDX_ID:
	case M_FS_8EBP_EBX_ID:
	case M_FS_8EBP_ESP_ID:
	case M_FS_9EBP_ID:
	case M_FS_8EBP_ESI_ID:
	case M_FS_8EBP_EDI_ID:
	case M_FS_8ESI_EAX_ID:
	case M_FS_8ESI_ECX_ID:
	case M_FS_8ESI_EDX_ID:
	case M_FS_8ESI_EBX_ID:
	case M_FS_8ESI_ESP_ID:
	case M_FS_8ESI_EBP_ID:
	case M_FS_9ESI_ID:
	case M_FS_8ESI_EDI_ID:
	case M_FS_8EDI_EAX_ID:
	case M_FS_8EDI_ECX_ID:
	case M_FS_8EDI_EDX_ID:
	case M_FS_8EDI_EBX_ID:
	case M_FS_8EDI_ESP_ID:
	case M_FS_8EDI_EBP_ID:
	case M_FS_8EDI_ESI_ID:
	case M_FS_9EDI_ID:
	case M_GS_2EAX_ID:
	case M_GS_2ECX_ID:
	case M_GS_2EDX_ID:
	case M_GS_2EBX_ID:
	case M_GS_2EBP_ID:
	case M_GS_2ESI_ID:
	case M_GS_2EDI_ID:
	case M_GS_4EAX_ID:
	case M_GS_4ECX_ID:
	case M_GS_4EDX_ID:
	case M_GS_4EBX_ID:
	case M_GS_4EBP_ID:
	case M_GS_4ESI_ID:
	case M_GS_4EDI_ID:
	case M_GS_8EAX_ID:
	case M_GS_8ECX_ID:
	case M_GS_8EDX_ID:
	case M_GS_8EBX_ID:
	case M_GS_8EBP_ID:
	case M_GS_8ESI_ID:
	case M_GS_8EDI_ID:
	case M_GS_EAX_ECX_ID:
	case M_GS_EAX_EDX_ID:
	case M_GS_EAX_EBX_ID:
	case M_GS_EAX_ESP_ID:
	case M_GS_EAX_EBP_ID:
	case M_GS_EAX_ESI_ID:
	case M_GS_EAX_EDI_ID:
	case M_GS_ECX_EDX_ID:
	case M_GS_ECX_EBX_ID:
	case M_GS_ECX_ESP_ID:
	case M_GS_ECX_EBP_ID:
	case M_GS_ECX_ESI_ID:
	case M_GS_ECX_EDI_ID:
	case M_GS_EDX_EBX_ID:
	case M_GS_EDX_ESP_ID:
	case M_GS_EDX_EBP_ID:
	case M_GS_EDX_ESI_ID:
	case M_GS_EDX_EDI_ID:
	case M_GS_EBX_ESP_ID:
	case M_GS_EBX_EBP_ID:
	case M_GS_EBX_ESI_ID:
	case M_GS_EBX_EDI_ID:
	case M_GS_ESP_ID:
	case M_GS_EBP_ESP_ID:
	case M_GS_EBP_ESI_ID:
	case M_GS_EBP_EDI_ID:
	case M_GS_ESI_ESP_ID:
	case M_GS_ESI_EDI_ID:
	case M_GS_EDI_ESP_ID:
	case M_GS_3EAX_ID:
	case M_GS_2EAX_ECX_ID:
	case M_GS_2EAX_EDX_ID:
	case M_GS_2EAX_EBX_ID:
	case M_GS_2EAX_ESP_ID:
	case M_GS_2EAX_EBP_ID:
	case M_GS_2EAX_ESI_ID:
	case M_GS_2EAX_EDI_ID:
	case M_GS_2ECX_EAX_ID:
	case M_GS_3ECX_ID:
	case M_GS_2ECX_EDX_ID:
	case M_GS_2ECX_EBX_ID:
	case M_GS_2ECX_ESP_ID:
	case M_GS_2ECX_EBP_ID:
	case M_GS_2ECX_ESI_ID:
	case M_GS_2ECX_EDI_ID:
	case M_GS_2EDX_EAX_ID:
	case M_GS_2EDX_ECX_ID:
	case M_GS_3EDX_ID:
	case M_GS_2EDX_EBX_ID:
	case M_GS_2EDX_ESP_ID:
	case M_GS_2EDX_EBP_ID:
	case M_GS_2EDX_ESI_ID:
	case M_GS_2EDX_EDI_ID:
	case M_GS_2EBX_EAX_ID:
	case M_GS_2EBX_ECX_ID:
	case M_GS_2EBX_EDX_ID:
	case M_GS_3EBX_ID:
	case M_GS_2EBX_ESP_ID:
	case M_GS_2EBX_EBP_ID:
	case M_GS_2EBX_ESI_ID:
	case M_GS_2EBX_EDI_ID:
	case M_GS_2EBP_EAX_ID:
	case M_GS_2EBP_ECX_ID:
	case M_GS_2EBP_EDX_ID:
	case M_GS_2EBP_EBX_ID:
	case M_GS_2EBP_ESP_ID:
	case M_GS_3EBP_ID:
	case M_GS_2EBP_ESI_ID:
	case M_GS_2EBP_EDI_ID:
	case M_GS_2ESI_EAX_ID:
	case M_GS_2ESI_ECX_ID:
	case M_GS_2ESI_EDX_ID:
	case M_GS_2ESI_EBX_ID:
	case M_GS_2ESI_ESP_ID:
	case M_GS_2ESI_EBP_ID:
	case M_GS_3ESI_ID:
	case M_GS_2ESI_EDI_ID:
	case M_GS_2EDI_EAX_ID:
	case M_GS_2EDI_ECX_ID:
	case M_GS_2EDI_EDX_ID:
	case M_GS_2EDI_EBX_ID:
	case M_GS_2EDI_ESP_ID:
	case M_GS_2EDI_EBP_ID:
	case M_GS_2EDI_ESI_ID:
	case M_GS_3EDI_ID:
	case M_GS_5EAX_ID:
	case M_GS_4EAX_ECX_ID:
	case M_GS_4EAX_EDX_ID:
	case M_GS_4EAX_EBX_ID:
	case M_GS_4EAX_ESP_ID:
	case M_GS_4EAX_EBP_ID:
	case M_GS_4EAX_ESI_ID:
	case M_GS_4EAX_EDI_ID:
	case M_GS_4ECX_EAX_ID:
	case M_GS_5ECX_ID:
	case M_GS_4ECX_EDX_ID:
	case M_GS_4ECX_EBX_ID:
	case M_GS_4ECX_ESP_ID:
	case M_GS_4ECX_EBP_ID:
	case M_GS_4ECX_ESI_ID:
	case M_GS_4ECX_EDI_ID:
	case M_GS_4EDX_EAX_ID:
	case M_GS_4EDX_ECX_ID:
	case M_GS_5EDX_ID:
	case M_GS_4EDX_EBX_ID:
	case M_GS_4EDX_ESP_ID:
	case M_GS_4EDX_EBP_ID:
	case M_GS_4EDX_ESI_ID:
	case M_GS_4EDX_EDI_ID:
	case M_GS_4EBX_EAX_ID:
	case M_GS_4EBX_ECX_ID:
	case M_GS_4EBX_EDX_ID:
	case M_GS_5EBX_ID:
	case M_GS_4EBX_ESP_ID:
	case M_GS_4EBX_EBP_ID:
	case M_GS_4EBX_ESI_ID:
	case M_GS_4EBX_EDI_ID:
	case M_GS_4EBP_EAX_ID:
	case M_GS_4EBP_ECX_ID:
	case M_GS_4EBP_EDX_ID:
	case M_GS_4EBP_EBX_ID:
	case M_GS_4EBP_ESP_ID:
	case M_GS_5EBP_ID:
	case M_GS_4EBP_ESI_ID:
	case M_GS_4EBP_EDI_ID:
	case M_GS_4ESI_EAX_ID:
	case M_GS_4ESI_ECX_ID:
	case M_GS_4ESI_EDX_ID:
	case M_GS_4ESI_EBX_ID:
	case M_GS_4ESI_ESP_ID:
	case M_GS_4ESI_EBP_ID:
	case M_GS_5ESI_ID:
	case M_GS_4ESI_EDI_ID:
	case M_GS_4EDI_EAX_ID:
	case M_GS_4EDI_ECX_ID:
	case M_GS_4EDI_EDX_ID:
	case M_GS_4EDI_EBX_ID:
	case M_GS_4EDI_ESP_ID:
	case M_GS_4EDI_EBP_ID:
	case M_GS_4EDI_ESI_ID:
	case M_GS_5EDI_ID:
	case M_GS_9EAX_ID:
	case M_GS_8EAX_ECX_ID:
	case M_GS_8EAX_EDX_ID:
	case M_GS_8EAX_EBX_ID:
	case M_GS_8EAX_ESP_ID:
	case M_GS_8EAX_EBP_ID:
	case M_GS_8EAX_ESI_ID:
	case M_GS_8EAX_EDI_ID:
	case M_GS_8ECX_EAX_ID:
	case M_GS_9ECX_ID:
	case M_GS_8ECX_EDX_ID:
	case M_GS_8ECX_EBX_ID:
	case M_GS_8ECX_ESP_ID:
	case M_GS_8ECX_EBP_ID:
	case M_GS_8ECX_ESI_ID:
	case M_GS_8ECX_EDI_ID:
	case M_GS_8EDX_EAX_ID:
	case M_GS_8EDX_ECX_ID:
	case M_GS_9EDX_ID:
	case M_GS_8EDX_EBX_ID:
	case M_GS_8EDX_ESP_ID:
	case M_GS_8EDX_EBP_ID:
	case M_GS_8EDX_ESI_ID:
	case M_GS_8EDX_EDI_ID:
	case M_GS_8EBX_EAX_ID:
	case M_GS_8EBX_ECX_ID:
	case M_GS_8EBX_EDX_ID:
	case M_GS_9EBX_ID:
	case M_GS_8EBX_ESP_ID:
	case M_GS_8EBX_EBP_ID:
	case M_GS_8EBX_ESI_ID:
	case M_GS_8EBX_EDI_ID:
	case M_GS_8EBP_EAX_ID:
	case M_GS_8EBP_ECX_ID:
	case M_GS_8EBP_EDX_ID:
	case M_GS_8EBP_EBX_ID:
	case M_GS_8EBP_ESP_ID:
	case M_GS_9EBP_ID:
	case M_GS_8EBP_ESI_ID:
	case M_GS_8EBP_EDI_ID:
	case M_GS_8ESI_EAX_ID:
	case M_GS_8ESI_ECX_ID:
	case M_GS_8ESI_EDX_ID:
	case M_GS_8ESI_EBX_ID:
	case M_GS_8ESI_ESP_ID:
	case M_GS_8ESI_EBP_ID:
	case M_GS_9ESI_ID:
	case M_GS_8ESI_EDI_ID:
	case M_GS_8EDI_EAX_ID:
	case M_GS_8EDI_ECX_ID:
	case M_GS_8EDI_EDX_ID:
	case M_GS_8EDI_EBX_ID:
	case M_GS_8EDI_ESP_ID:
	case M_GS_8EDI_EBP_ID:
	case M_GS_8EDI_ESI_ID:
	case M_GS_9EDI_ID:
	    return 4;
	}
	return 0;
    }

    public static boolean isFarJump(int operation)
    {
        switch (operation) 
        {
	case JMP_O32_FAR:
	case JMP_O16_FAR:
	case CALLF_O16_A16:
	case CALLF_O32_A16:
	case CALLF_O16_A32:
	case CALLF_O32_A32:
	case RETF_IW_O16_A16:
	case RETF_IW_O32_A16:
	case RETF_IW_O16_A32:
	case RETF_IW_O32_A32:
	case RETF_O16_A16:
	case RETF_O32_A16:
	case RETF_O16_A32:
	case RETF_O32_A32:	
	case INT3_O16_A16:
	case INT3_O32_A16:
	case INT3_O16_A32:
	case INT3_O32_A32:
	case INT_O16_A16:
	case INT_O32_A16:
	case INT_O16_A32:
	case INT_O32_A32:
	case INTO_O16_A16:
	case INTO_O32_A16:
	case INTO_O16_A32:
	case INTO_O32_A32:
	case INT1_O16_A16:
	case INT1_O32_A16:
	case INT1_O16_A32:
	case INT1_O32_A32:	    
	case IRET_O16_A16:
	case IRET_O32_A16:
	case IRET_O16_A32:
	case IRET_O32_A32:
	    return true;
	default:
	    return false;
	}
    }

    public static boolean isNearJump(int operation)
    {
        switch (operation) 
        {
	case JMP_O32_NEAR_RELATIVE:
	case JMP_O16_NEAR_RELATIVE:	    
	case JMP_O32_NEAR_ABSOLUTE:
	case JMP_O16_NEAR_ABSOLUTE:	    
	case JMP_O8_SHORT:
	case JCXZ_A16:
	case JCXZ_A32:
	case JO_O8:
	case JNO_O8:
	case JB_O8:
	case JNB_O8:
	case JZ_O8:
	case JNZ_O8:
	case JBE_O8:
	case JNBE_O8:
	case JS_O8:
	case JNS_O8:
	case JP_O8:
	case JNP_O8:
	case JL_O8:
	case JNL_O8:
	case JLE_O8:
	case JNLE_O8:
	case JO_O16:
	case JNO_O16:
	case JB_O16:
	case JNB_O16:
	case JZ_O16:
	case JNZ_O16:
	case JBE_O16:
	case JNBE_O16:
	case JS_O16:
	case JNS_O16:
	case JP_O16:
	case JNP_O16:
	case JL_O16:
	case JNL_O16:
	case JLE_O16:
	case JNLE_O16:
	case JO_O32:
	case JNO_O32:
	case JB_O32:
	case JNB_O32:
	case JZ_O32:
	case JNZ_O32:
	case JBE_O32:
	case JNBE_O32:
	case JS_O32:
	case JNS_O32:
	case JP_O32:
	case JNP_O32:
	case JL_O32:
	case JNL_O32:
	case JLE_O32:
	case JNLE_O32:
	case CALL_O16_A16:
	case CALL_O32_A16:
	case CALL_O16_A32:
	case CALL_O32_A32:
	case CALLN_O16_A16:
	case CALLN_O32_A16:
	case CALLN_O16_A32:
	case CALLN_O32_A32:   
	case RET_IW_O16_A16:
	case RET_IW_O32_A16:
	case RET_IW_O16_A32:
	case RET_IW_O32_A32:
	case RET_O16_A16:
	case RET_O32_A16:
	case RET_O16_A32:
	case RET_O32_A32:
	case LOOPNZ_O16_A16:
	case LOOPNZ_O32_A16:
	case LOOPNZ_O16_A32:
	case LOOPNZ_O32_A32:
	case LOOPZ_O16_A16:
	case LOOPZ_O32_A16:
	case LOOPZ_O16_A32:
	case LOOPZ_O32_A32:
	case LOOP_O16_A16:
	case LOOP_O32_A16:
	case LOOP_O16_A32:
	case LOOP_O32_A32:
	    return true;
	default:
	    return false;
	}
    }

    public static boolean isModeSwitch(int operation)
    {
	switch (operation) {
	case MOV_TO_CR_O32:
	case LMSW:
	    return true;
	default:
	    return false;
	}
    }

    public static boolean isBlockTerminating(int operation)
    {
	switch (operation) {
	case HLT:
	    return true;
	default:
	    return false;
	}
    }

    public static boolean isJump(int operation)
    {
        return isNearJump(operation) || isFarJump(operation) || isModeSwitch(operation) || isBlockTerminating(operation);
    }

    private static int decodeEb(int prefices, int modrm, int sib)
    {
	switch(modrm & 0xc7) {
	case 0xc0: return AL;
	case 0xc1: return CL;
	case 0xc2: return DL;
	case 0xc3: return BL;
	case 0xc4: return AH;
	case 0xc5: return CH;
	case 0xc6: return DH;
	case 0xc7: return BH;
	}

	return decodeM(prefices, modrm, sib);
    }

    private static int decodeEw(int prefices, int modrm, int sib)
    {
	switch (modrm & 0xc7) {
	case 0xc0: return AX;
	case 0xc1: return CX;
	case 0xc2: return DX;
	case 0xc3: return BX;
	case 0xc4: return SP;
	case 0xc5: return BP;
	case 0xc6: return SI;
	case 0xc7: return DI;
	}

	return decodeM(prefices, modrm, sib);
    }

    private static int decodeEd(int prefices, int modrm, int sib)
    {
	switch (modrm & 0xc7) {
	case 0xc0: return EAX;
	case 0xc1: return ECX;
	case 0xc2: return EDX;
	case 0xc3: return EBX;
	case 0xc4: return ESP;
	case 0xc5: return EBP;
	case 0xc6: return ESI;
	case 0xc7: return EDI;
	}

	return decodeM(prefices, modrm, sib);
    }

    private static int decodeGb(int modrm)
    {
	switch(modrm & 0x38) {
	case 0x00: return AL;
	case 0x08: return CL;
	case 0x10: return DL;
	case 0x18: return BL;
	case 0x20: return AH;
	case 0x28: return CH;
	case 0x30: return DH;
	case 0x38: return BH;
	default: return -1;
	}
    }
    private static int decodeGw(int modrm)
    {
	switch(modrm & 0x38) {
	case 0x00: return AX;
	case 0x08: return CX;
	case 0x10: return DX;
	case 0x18: return BX;
	case 0x20: return SP;
	case 0x28: return BP;
	case 0x30: return SI;
	case 0x38: return DI;
	default: return -1;
	}
    }
    private static int decodeGd(int modrm)
    {
	switch(modrm & 0x38) {
	case 0x00: return EAX;
	case 0x08: return ECX;
	case 0x10: return EDX;
	case 0x18: return EBX;
	case 0x20: return ESP;
	case 0x28: return EBP;
	case 0x30: return ESI;
	case 0x38: return EDI;
	default: return -1;
	}
    }
    private static int decodeSw(int modrm)
    {
	switch(modrm & 0x38) {
	case 0x00: return ES;
	case 0x08: return CS;
	case 0x10: return SS;
	case 0x18: return DS;
	case 0x20: return FS;
	case 0x28: return GS;
	default: return -1;
	}
    }

    private static int decodeOb(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ID;
	    } else {
		return M_ES_IW;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ID;
	    } else {
		return M_CS_IW;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ID;
	    } else {
		return M_SS_IW;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ID;
	    } else {
		return M_FS_IW;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ID;
	    } else {
		return M_GS_IW;
	    }
	}
    }

    private static int decodeOw(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ID;
	    } else {
		return M_ES_IW;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ID;
	    } else {
		return M_CS_IW;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ID;
	    } else {
		return M_SS_IW;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ID;
	    } else {
		return M_FS_IW;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ID;
	    } else {
		return M_GS_IW;
	    }
	}
    }

    private static int decodeOd(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ID;
	    } else {
		return M_ES_IW;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ID;
	    } else {
		return M_CS_IW;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ID;
	    } else {
		return M_SS_IW;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ID;
	    } else {
		return M_DS_IW;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ID;
	    } else {
		return M_FS_IW;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ID;
	    } else {
		return M_GS_IW;
	    }
	}
    }

    private static int decodeYb(int prefices)
    {
	if ((prefices & PREFICES_ADDRESS) != 0) {
	    return M_ES_EDI;
	} else {
	    return M_ES_DI;
	}
    }

    private static int decodeYw(int prefices)
    {
	if ((prefices & PREFICES_ADDRESS) != 0) {
	    return M_ES_EDI;
	} else {
	    return M_ES_DI;
	}
    }

    private static int decodeYd(int prefices)
    {
	if ((prefices & PREFICES_ADDRESS) != 0) {
	    return M_ES_EDI;
	} else {
	    return M_ES_DI;
	}
    }

    private static int decodeXb(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ESI;
	    } else {
		return M_ES_SI;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ESI;
	    } else {
		return M_CS_SI;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ESI;
	    } else {
		return M_SS_SI;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ESI;
	    } else {
		return M_FS_SI;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ESI;
	    } else {
		return M_GS_SI;
	    }
	}
    }

    private static int decodeXw(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ESI;
	    } else {
		return M_ES_SI;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ESI;
	    } else {
		return M_CS_SI;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ESI;
	    } else {
		return M_SS_SI;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ESI;
	    } else {
		return M_FS_SI;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ESI;
	    } else {
		return M_GS_SI;
	    }
	}
    }

    private static int decodeXd(int prefices)
    {
	switch (prefices & PREFICES_SG) {
	default:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_ES:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_ES_ESI;
	    } else {
		return M_ES_SI;
	    }
	case PREFICES_CS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_CS_ESI;
	    } else {
		return M_CS_SI;
	    }
	case PREFICES_SS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_SS_ESI;
	    } else {
		return M_SS_SI;
	    }
	case PREFICES_DS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_DS_ESI;
	    } else {
		return M_DS_SI;
	    }
	case PREFICES_FS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_FS_ESI;
	    } else {
		return M_FS_SI;
	    }
	case PREFICES_GS:
	    if ((prefices & PREFICES_ADDRESS) != 0) {
		return M_GS_ESI;
	    } else {
		return M_GS_SI;
	    }
	}
    }

    private static int decodeM(int prefices, int modrm, int sib)
    {
	if ((prefices & PREFICES_ADDRESS) != 0) {
	    //32 bit address size
	    switch (prefices & PREFICES_SG) {
	    default:
		switch(modrm & 0xc7) {
		case 0x00: return M_DS_EAX;
		case 0x01: return M_DS_ECX;
		case 0x02: return M_DS_EDX;
		case 0x03: return M_DS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_DS_ID;
		case 0x06: return M_DS_ESI;
		case 0x07: return M_DS_EDI;
		case 0x40: return M_DS_EAX_IB;
		case 0x41: return M_DS_ECX_IB;
		case 0x42: return M_DS_EDX_IB;
		case 0x43: return M_DS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_SS_EBP_IB; //SS?
		case 0x46: return M_DS_ESI_IB;
		case 0x47: return M_DS_EDI_IB;
		case 0x80: return M_DS_EAX_ID;
		case 0x81: return M_DS_ECX_ID;
		case 0x82: return M_DS_EDX_ID;
		case 0x83: return M_DS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_SS_EBP_ID; //SS?
		case 0x86: return M_DS_ESI_ID;
		case 0x87: return M_DS_EDI_ID;
		}
	    case PREFICES_ES:
		switch(modrm & 0xc7) {
		case 0x00: return M_ES_EAX;
		case 0x01: return M_ES_ECX;
		case 0x02: return M_ES_EDX;
		case 0x03: return M_ES_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_ES_ID;
		case 0x06: return M_ES_ESI;
		case 0x07: return M_ES_EDI;
		case 0x40: return M_ES_EAX_IB;
		case 0x41: return M_ES_ECX_IB;
		case 0x42: return M_ES_EDX_IB;
		case 0x43: return M_ES_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_ES_EBP_IB;
		case 0x46: return M_ES_ESI_IB;
		case 0x47: return M_ES_EDI_IB;
		case 0x80: return M_ES_EAX_ID;
		case 0x81: return M_ES_ECX_ID;
		case 0x82: return M_ES_EDX_ID;
		case 0x83: return M_ES_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_ES_EBP_ID;
		case 0x86: return M_ES_ESI_ID;
		case 0x87: return M_ES_EDI_ID;
		}
	    case PREFICES_CS:
		switch(modrm & 0xc7) {
		case 0x00: return M_CS_EAX;
		case 0x01: return M_CS_ECX;
		case 0x02: return M_CS_EDX;
		case 0x03: return M_CS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_CS_ID;
		case 0x06: return M_CS_ESI;
		case 0x07: return M_CS_EDI;
		case 0x40: return M_CS_EAX_IB;
		case 0x41: return M_CS_ECX_IB;
		case 0x42: return M_CS_EDX_IB;
		case 0x43: return M_CS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_CS_EBP_IB;
		case 0x46: return M_CS_ESI_IB;
		case 0x47: return M_CS_EDI_IB;
		case 0x80: return M_CS_EAX_ID;
		case 0x81: return M_CS_ECX_ID;
		case 0x82: return M_CS_EDX_ID;
		case 0x83: return M_CS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_CS_EBP_ID;
		case 0x86: return M_CS_ESI_ID;
		case 0x87: return M_CS_EDI_ID;
		}
	    case PREFICES_SS:
		switch(modrm & 0xc7) {
		case 0x00: return M_SS_EAX;
		case 0x01: return M_SS_ECX;
		case 0x02: return M_SS_EDX;
		case 0x03: return M_SS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_SS_ID;
		case 0x06: return M_SS_ESI;
		case 0x07: return M_SS_EDI;
		case 0x40: return M_SS_EAX_IB;
		case 0x41: return M_SS_ECX_IB;
		case 0x42: return M_SS_EDX_IB;
		case 0x43: return M_SS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_SS_EBP_IB;
		case 0x46: return M_SS_ESI_IB;
		case 0x47: return M_SS_EDI_IB;
		case 0x80: return M_SS_EAX_ID;
		case 0x81: return M_SS_ECX_ID;
		case 0x82: return M_SS_EDX_ID;
		case 0x83: return M_SS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_SS_EBP_ID;
		case 0x86: return M_SS_ESI_ID;
		case 0x87: return M_SS_EDI_ID;
		}
	    case PREFICES_DS:
		switch(modrm & 0xc7) {
		case 0x00: return M_DS_EAX;
		case 0x01: return M_DS_ECX;
		case 0x02: return M_DS_EDX;
		case 0x03: return M_DS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_DS_ID;
		case 0x06: return M_DS_ESI;
		case 0x07: return M_DS_EDI;
		case 0x40: return M_DS_EAX_IB;
		case 0x41: return M_DS_ECX_IB;
		case 0x42: return M_DS_EDX_IB;
		case 0x43: return M_DS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_DS_EBP_IB;
		case 0x46: return M_DS_ESI_IB;
		case 0x47: return M_DS_EDI_IB;
		case 0x80: return M_DS_EAX_ID;
		case 0x81: return M_DS_ECX_ID;
		case 0x82: return M_DS_EDX_ID;
		case 0x83: return M_DS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_DS_EBP_ID;
		case 0x86: return M_DS_ESI_ID;
		case 0x87: return M_DS_EDI_ID;
		}
	    case PREFICES_FS:
		switch(modrm & 0xc7) {
		case 0x00: return M_FS_EAX;
		case 0x01: return M_FS_ECX;
		case 0x02: return M_FS_EDX;
		case 0x03: return M_FS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_FS_ID;
		case 0x06: return M_FS_ESI;
		case 0x07: return M_FS_EDI;
		case 0x40: return M_FS_EAX_IB;
		case 0x41: return M_FS_ECX_IB;
		case 0x42: return M_FS_EDX_IB;
		case 0x43: return M_FS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_FS_EBP_IB;
		case 0x46: return M_FS_ESI_IB;
		case 0x47: return M_FS_EDI_IB;
		case 0x80: return M_FS_EAX_ID;
		case 0x81: return M_FS_ECX_ID;
		case 0x82: return M_FS_EDX_ID;
		case 0x83: return M_FS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_FS_EBP_ID;
		case 0x86: return M_FS_ESI_ID;
		case 0x87: return M_FS_EDI_ID;
		}
	    case PREFICES_GS:
		switch(modrm & 0xc7) {
		case 0x00: return M_GS_EAX;
		case 0x01: return M_GS_ECX;
		case 0x02: return M_GS_EDX;
		case 0x03: return M_GS_EBX;
		    case 0x04: return decodeSIB(prefices, modrm, sib);
		case 0x05: return M_GS_ID;
		case 0x06: return M_GS_ESI;
		case 0x07: return M_GS_EDI;
		case 0x40: return M_GS_EAX_IB;
		case 0x41: return M_GS_ECX_IB;
		case 0x42: return M_GS_EDX_IB;
		case 0x43: return M_GS_EBX_IB;
		    case 0x44: return decodeSIB(prefices, modrm, sib);
		case 0x45: return M_GS_EBP_IB;
		case 0x46: return M_GS_ESI_IB;
		case 0x47: return M_GS_EDI_IB;
		case 0x80: return M_GS_EAX_ID;
		case 0x81: return M_GS_ECX_ID;
		case 0x82: return M_GS_EDX_ID;
		case 0x83: return M_GS_EBX_ID;
		    case 0x84: return decodeSIB(prefices, modrm, sib);
		case 0x85: return M_GS_EBP_ID;
		case 0x86: return M_GS_ESI_ID;
		case 0x87: return M_GS_EDI_ID;
		}
	    }
	} else {
	    //16 bit address size
	    switch (prefices & PREFICES_SG) {
	    default:
		switch(modrm & 0xc7) {
		case 0x00: return M_DS_BX_SI;
		case 0x01: return M_DS_BX_DI;
		case 0x02: return M_SS_BP_SI;
		case 0x03: return M_SS_BP_DI;
		case 0x04: return M_DS_SI;
		case 0x05: return M_DS_DI;
		case 0x06: return M_DS_IW;
		case 0x07: return M_DS_BX;
		case 0x40: return M_DS_BX_SI_IB;
		case 0x41: return M_DS_BX_DI_IB;
		case 0x42: return M_SS_BP_SI_IB;
		case 0x43: return M_SS_BP_DI_IB;
		case 0x44: return M_DS_SI_IB;
		case 0x45: return M_DS_DI_IB;
		case 0x46: return M_SS_BP_IB;
		case 0x47: return M_DS_BX_IB;
		case 0x80: return M_DS_BX_SI_IW;
		case 0x81: return M_DS_BX_DI_IW;
		case 0x82: return M_SS_BP_SI_IW;
		case 0x83: return M_SS_BP_DI_IW;
		case 0x84: return M_DS_SI_IW;
		case 0x85: return M_DS_DI_IW;
		case 0x86: return M_SS_BP_IW;
		case 0x87: return M_DS_BX_IW;
		}
	    case PREFICES_ES:
		switch(modrm & 0xc7) {
		case 0x00: return M_ES_BX_SI;
		case 0x01: return M_ES_BX_DI;
		case 0x02: return M_ES_BP_SI;
		case 0x03: return M_ES_BP_DI;
		case 0x04: return M_ES_SI;
		case 0x05: return M_ES_DI;
		case 0x06: return M_ES_IW;
		case 0x07: return M_ES_BX;
		case 0x40: return M_ES_BX_SI_IB;
		case 0x41: return M_ES_BX_DI_IB;
		case 0x42: return M_ES_BP_SI_IB;
		case 0x43: return M_ES_BP_DI_IB;
		case 0x44: return M_ES_SI_IB;
		case 0x45: return M_ES_DI_IB;
		case 0x46: return M_ES_BP_IB;
		case 0x47: return M_ES_BX_IB;
		case 0x80: return M_ES_BX_SI_IW;
		case 0x81: return M_ES_BX_DI_IW;
		case 0x82: return M_ES_BP_SI_IW;
		case 0x83: return M_ES_BP_DI_IW;
		case 0x84: return M_ES_SI_IW;
		case 0x85: return M_ES_DI_IW;
		case 0x86: return M_ES_BP_IW;
		case 0x87: return M_ES_BX_IW;
		}
	    case PREFICES_CS:
		switch(modrm & 0xc7) {
		case 0x00: return M_CS_BX_SI;
		case 0x01: return M_CS_BX_DI;
		case 0x02: return M_CS_BP_SI;
		case 0x03: return M_CS_BP_DI;
		case 0x04: return M_CS_SI;
		case 0x05: return M_CS_DI;
		case 0x06: return M_CS_IW;
		case 0x07: return M_CS_BX;
		case 0x40: return M_CS_BX_SI_IB;
		case 0x41: return M_CS_BX_DI_IB;
		case 0x42: return M_CS_BP_SI_IB;
		case 0x43: return M_CS_BP_DI_IB;
		case 0x44: return M_CS_SI_IB;
		case 0x45: return M_CS_DI_IB;
		case 0x46: return M_CS_BP_IB;
		case 0x47: return M_CS_BX_IB;
		case 0x80: return M_CS_BX_SI_IW;
		case 0x81: return M_CS_BX_DI_IW;
		case 0x82: return M_CS_BP_SI_IW;
		case 0x83: return M_CS_BP_DI_IW;
		case 0x84: return M_CS_SI_IW;
		case 0x85: return M_CS_DI_IW;
		case 0x86: return M_CS_BP_IW;
		case 0x87: return M_CS_BX_IW;
		}
	    case PREFICES_SS:
		switch(modrm & 0xc7) {
		case 0x00: return M_SS_BX_SI;
		case 0x01: return M_SS_BX_DI;
		case 0x02: return M_SS_BP_SI;
		case 0x03: return M_SS_BP_DI;
		case 0x04: return M_SS_SI;
		case 0x05: return M_SS_DI;
		case 0x06: return M_SS_IW;
		case 0x07: return M_SS_BX;
		case 0x40: return M_SS_BX_SI_IB;
		case 0x41: return M_SS_BX_DI_IB;
		case 0x42: return M_SS_BP_SI_IB;
		case 0x43: return M_SS_BP_DI_IB;
		case 0x44: return M_SS_SI_IB;
		case 0x45: return M_SS_DI_IB;
		case 0x46: return M_SS_BP_IB;
		case 0x47: return M_SS_BX_IB;
		case 0x80: return M_SS_BX_SI_IW;
		case 0x81: return M_SS_BX_DI_IW;
		case 0x82: return M_SS_BP_SI_IW;
		case 0x83: return M_SS_BP_DI_IW;
		case 0x84: return M_SS_SI_IW;
		case 0x85: return M_SS_DI_IW;
		case 0x86: return M_SS_BP_IW;
		case 0x87: return M_SS_BX_IW;
		}
	    case PREFICES_DS:
		switch(modrm & 0xc7) {
		case 0x00: return M_DS_BX_SI;
		case 0x01: return M_DS_BX_DI;
		case 0x02: return M_DS_BP_SI;
		case 0x03: return M_DS_BP_DI;
		case 0x04: return M_DS_SI;
		case 0x05: return M_DS_DI;
		case 0x06: return M_DS_IW;
		case 0x07: return M_DS_BX;
		case 0x40: return M_DS_BX_SI_IB;
		case 0x41: return M_DS_BX_DI_IB;
		case 0x42: return M_DS_BP_SI_IB;
		case 0x43: return M_DS_BP_DI_IB;
		case 0x44: return M_DS_SI_IB;
		case 0x45: return M_DS_DI_IB;
		case 0x46: return M_DS_BP_IB;
		case 0x47: return M_DS_BX_IB;
		case 0x80: return M_DS_BX_SI_IW;
		case 0x81: return M_DS_BX_DI_IW;
		case 0x82: return M_DS_BP_SI_IW;
		case 0x83: return M_DS_BP_DI_IW;
		case 0x84: return M_DS_SI_IW;
		case 0x85: return M_DS_DI_IW;
		case 0x86: return M_DS_BP_IW;
		case 0x87: return M_DS_BX_IW;
		}
	    case PREFICES_FS:
		switch(modrm & 0xc7) {
		case 0x00: return M_FS_BX_SI;
		case 0x01: return M_FS_BX_DI;
		case 0x02: return M_FS_BP_SI;
		case 0x03: return M_FS_BP_DI;
		case 0x04: return M_FS_SI;
		case 0x05: return M_FS_DI;
		case 0x06: return M_FS_IW;
		case 0x07: return M_FS_BX;
		case 0x40: return M_FS_BX_SI_IB;
		case 0x41: return M_FS_BX_DI_IB;
		case 0x42: return M_FS_BP_SI_IB;
		case 0x43: return M_FS_BP_DI_IB;
		case 0x44: return M_FS_SI_IB;
		case 0x45: return M_FS_DI_IB;
		case 0x46: return M_FS_BP_IB;
		case 0x47: return M_FS_BX_IB;
		case 0x80: return M_FS_BX_SI_IW;
		case 0x81: return M_FS_BX_DI_IW;
		case 0x82: return M_FS_BP_SI_IW;
		case 0x83: return M_FS_BP_DI_IW;
		case 0x84: return M_FS_SI_IW;
		case 0x85: return M_FS_DI_IW;
		case 0x86: return M_FS_BP_IW;
		case 0x87: return M_FS_BX_IW;
		}
	    case PREFICES_GS:
		switch(modrm & 0xc7) {
		case 0x00: return M_GS_BX_SI;
		case 0x01: return M_GS_BX_DI;
		case 0x02: return M_GS_BP_SI;
		case 0x03: return M_GS_BP_DI;
		case 0x04: return M_GS_SI;
		case 0x05: return M_GS_DI;
		case 0x06: return M_GS_IW;
		case 0x07: return M_GS_BX;
		case 0x40: return M_GS_BX_SI_IB;
		case 0x41: return M_GS_BX_DI_IB;
		case 0x42: return M_GS_BP_SI_IB;
		case 0x43: return M_GS_BP_DI_IB;
		case 0x44: return M_GS_SI_IB;
		case 0x45: return M_GS_DI_IB;
		case 0x46: return M_GS_BP_IB;
		case 0x47: return M_GS_BX_IB;
		case 0x80: return M_GS_BX_SI_IW;
		case 0x81: return M_GS_BX_DI_IW;
		case 0x82: return M_GS_BP_SI_IW;
		case 0x83: return M_GS_BP_DI_IW;
		case 0x84: return M_GS_SI_IW;
		case 0x85: return M_GS_DI_IW;
		case 0x86: return M_GS_BP_IW;
		case 0x87: return M_GS_BX_IW;
		}
	    }
	}
	return -1;
    }

    private static int decodeCd(int modrm)
    {
	switch(modrm & 0x38) {
	case 0x00: return CR0;
	case 0x10: return CR2;
	case 0x18: return CR3;
	case 0x20: return CR4;
	default: return -1;
	}
    }

    private static int decodeSIB(int prefices, int modrm, int sib)
    {
	switch (prefices & PREFICES_SG) {
	case 0: //no segment prefix
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX;
		case 0x01: return M_DS_EAX_ECX;
		case 0x02: return M_DS_EAX_EDX;
		case 0x03: return M_DS_EAX_EBX;
		case 0x04: return M_SS_EAX_ESP;
		case 0x05: return M_DS_EAX_ID;
		case 0x06: return M_DS_EAX_ESI;
		case 0x07: return M_DS_EAX_EDI;

		case 0x08: return M_DS_EAX_ECX;
		case 0x09: return M_DS_2ECX;
		case 0x0a: return M_DS_ECX_EDX;
		case 0x0b: return M_DS_ECX_EBX;
		case 0x0c: return M_SS_ECX_ESP;
		case 0x0d: return M_DS_ECX_ID;
		case 0x0e: return M_DS_ECX_ESI;
		case 0x0f: return M_DS_ECX_EDI;

		case 0x10: return M_DS_EAX_EDX;
		case 0x11: return M_DS_ECX_EDX;
		case 0x12: return M_DS_2EDX;
		case 0x13: return M_DS_EDX_EBX;
		case 0x14: return M_SS_EDX_ESP;
		case 0x15: return M_DS_EDX_ID;
		case 0x16: return M_DS_EDX_ESI;
		case 0x17: return M_DS_EDX_EDI;

		case 0x18: return M_DS_EAX_EBX;
		case 0x19: return M_DS_ECX_EBX;
		case 0x1a: return M_DS_EDX_EBX;
		case 0x1b: return M_DS_2EBX;
		case 0x1c: return M_SS_EBX_ESP;
		case 0x1d: return M_DS_EBX_ID;
		case 0x1e: return M_DS_EBX_ESI;
		case 0x1f: return M_DS_EBX_EDI;

		case 0x20: return M_DS_EAX;
		case 0x21: return M_DS_ECX;
		case 0x22: return M_DS_EDX;
		case 0x23: return M_DS_EBX;
		case 0x24: return M_SS_ESP;
		case 0x25: return M_DS_ID;
		case 0x26: return M_DS_ESI;
		case 0x27: return M_DS_EDI;

		case 0x28: return M_DS_EAX_EBP;
		case 0x29: return M_DS_ECX_EBP;
		case 0x2a: return M_DS_EDX_EBP;
		case 0x2b: return M_DS_EBX_EBP;
		case 0x2c: return M_SS_EBP_ESP;
		case 0x2d: return M_DS_EBP_ID;
		case 0x2e: return M_DS_EBP_ESI;
		case 0x2f: return M_DS_EBP_EDI;

		case 0x30: return M_DS_EAX_ESI;
		case 0x31: return M_DS_ECX_ESI;
		case 0x32: return M_DS_EDX_ESI;
		case 0x33: return M_DS_EBX_ESI;
		case 0x34: return M_SS_ESI_ESP;
		case 0x35: return M_DS_ESI_ID;
		case 0x36: return M_DS_2ESI;
		case 0x37: return M_DS_ESI_EDI;

		case 0x38: return M_DS_EAX_EDI;
		case 0x39: return M_DS_ECX_EDI;
		case 0x3a: return M_DS_EDX_EDI;
		case 0x3b: return M_DS_EBX_EDI;
		case 0x3c: return M_SS_EDI_ESP;
		case 0x3d: return M_DS_EDI_ID;
		case 0x3e: return M_DS_ESI_EDI;
		case 0x3f: return M_DS_2EDI;

		case 0x40: return M_DS_3EAX;
		case 0x41: return M_DS_2EAX_ECX;
		case 0x42: return M_DS_2EAX_EDX;
		case 0x43: return M_DS_2EAX_EBX;
		case 0x44: return M_SS_2EAX_ESP;
		case 0x45: return M_DS_2EAX_ID;
		case 0x46: return M_DS_2EAX_ESI;
		case 0x47: return M_DS_2EAX_EDI;

		case 0x48: return M_DS_2ECX_EAX;
		case 0x49: return M_DS_3ECX;
		case 0x4a: return M_DS_2ECX_EDX;
		case 0x4b: return M_DS_2ECX_EBX;
		case 0x4c: return M_SS_2ECX_ESP;
		case 0x4d: return M_DS_2ECX_ID;
		case 0x4e: return M_DS_2ECX_ESI;
		case 0x4f: return M_DS_2ECX_EDI;

		case 0x50: return M_DS_2EDX_EAX;
		case 0x51: return M_DS_2EDX_ECX;
		case 0x52: return M_DS_3EDX;
		case 0x53: return M_DS_2EDX_EBX;
		case 0x54: return M_SS_2EDX_ESP;
		case 0x55: return M_DS_2EDX_ID;
		case 0x56: return M_DS_2EDX_ESI;
		case 0x57: return M_DS_2EDX_EDI;

		case 0x58: return M_DS_2EBX_EAX;
		case 0x59: return M_DS_2EBX_ECX;
		case 0x5a: return M_DS_2EBX_EDX;
		case 0x5b: return M_DS_3EBX;
		case 0x5c: return M_SS_2EBX_ESP;
		case 0x5d: return M_DS_2EBX_ID;
		case 0x5e: return M_DS_2EBX_ESI;
		case 0x5f: return M_DS_2EBX_EDI;

		case 0x60: return M_DS_EAX;
		case 0x61: return M_DS_ECX;
		case 0x62: return M_DS_EDX;
		case 0x63: return M_DS_EBX;
		case 0x64: return M_SS_ESP;
		case 0x65: return M_DS_ID;
		case 0x66: return M_DS_ESI;
		case 0x67: return M_DS_EDI;

		case 0x68: return M_DS_2EBP_EAX;
		case 0x69: return M_DS_2EBP_ECX;
		case 0x6a: return M_DS_2EBP_EDX;
		case 0x6b: return M_DS_2EBP_EBX;
		case 0x6c: return M_SS_2EBP_ESP;
		case 0x6d: return M_DS_2EBP_ID;
		case 0x6e: return M_DS_2EBP_ESI;
		case 0x6f: return M_DS_2EBP_EDI;

		case 0x70: return M_DS_2ESI_EAX;
		case 0x71: return M_DS_2ESI_ECX;
		case 0x72: return M_DS_2ESI_EDX;
		case 0x73: return M_DS_2ESI_EBX;
		case 0x74: return M_SS_2ESI_ESP;
		case 0x75: return M_DS_2ESI_ID;
		case 0x76: return M_DS_3ESI;
		case 0x77: return M_DS_2ESI_EDI;

		case 0x78: return M_DS_2EDI_EAX;
		case 0x79: return M_DS_2EDI_ECX;
		case 0x7a: return M_DS_2EDI_EDX;
		case 0x7b: return M_DS_2EDI_EBX;
		case 0x7c: return M_SS_2EDI_ESP;
		case 0x7d: return M_DS_2EDI_ID;
		case 0x7e: return M_DS_2EDI_ESI;
		case 0x7f: return M_DS_3EDI;

		case 0x80: return M_DS_5EAX;
		case 0x81: return M_DS_4EAX_ECX;
		case 0x82: return M_DS_4EAX_EDX;
		case 0x83: return M_DS_4EAX_EBX;
		case 0x84: return M_SS_4EAX_ESP;
		case 0x85: return M_DS_4EAX_ID;
		case 0x86: return M_DS_4EAX_ESI;
		case 0x87: return M_DS_4EAX_EDI;

		case 0x88: return M_DS_4ECX_EAX;
		case 0x89: return M_DS_5ECX;
		case 0x8a: return M_DS_4ECX_EDX;
		case 0x8b: return M_DS_4ECX_EBX;
		case 0x8c: return M_SS_4ECX_ESP;
		case 0x8d: return M_DS_4ECX_ID;
		case 0x8e: return M_DS_4ECX_ESI;
		case 0x8f: return M_DS_4ECX_EDI;

		case 0x90: return M_DS_4EDX_EAX;
		case 0x91: return M_DS_4EDX_ECX;
		case 0x92: return M_DS_5EDX;
		case 0x93: return M_DS_4EDX_EBX;
		case 0x94: return M_SS_4EDX_ESP;
		case 0x95: return M_DS_4EDX_ID;
		case 0x96: return M_DS_4EDX_ESI;
		case 0x97: return M_DS_4EDX_EDI;

		case 0x98: return M_DS_4EBX_EAX;
		case 0x99: return M_DS_4EBX_ECX;
		case 0x9a: return M_DS_4EBX_EDX;
		case 0x9b: return M_DS_5EBX;
		case 0x9c: return M_SS_4EBX_ESP;
		case 0x9d: return M_DS_4EBX_ID;
		case 0x9e: return M_DS_4EBX_ESI;
		case 0x9f: return M_DS_4EBX_EDI;

		case 0xa0: return M_DS_EAX;
		case 0xa1: return M_DS_ECX;
		case 0xa2: return M_DS_EDX;
		case 0xa3: return M_DS_EBX;
		case 0xa4: return M_SS_ESP;
		case 0xa5: return M_DS_ID;
		case 0xa6: return M_DS_ESI;
		case 0xa7: return M_DS_EDI;

		case 0xa8: return M_DS_4EBP_EAX;
		case 0xa9: return M_DS_4EBP_ECX;
		case 0xaa: return M_DS_4EBP_EDX;
		case 0xab: return M_DS_4EBP_EBX;
		case 0xac: return M_SS_4EBP_ESP;
		case 0xad: return M_DS_4EBP_ID;
		case 0xae: return M_DS_4EBP_ESI;
		case 0xaf: return M_DS_4EBP_EDI;

		case 0xb0: return M_DS_4ESI_EAX;
		case 0xb1: return M_DS_4ESI_ECX;
		case 0xb2: return M_DS_4ESI_EDX;
		case 0xb3: return M_DS_4ESI_EBX;
		case 0xb4: return M_SS_4ESI_ESP;
		case 0xb5: return M_DS_4ESI_ID;
		case 0xb6: return M_DS_5ESI;
		case 0xb7: return M_DS_4ESI_EDI;

		case 0xb8: return M_DS_4EDI_EAX;
		case 0xb9: return M_DS_4EDI_ECX;
		case 0xba: return M_DS_4EDI_EDX;
		case 0xbb: return M_DS_4EDI_EBX;
		case 0xbc: return M_SS_4EDI_ESP;
		case 0xbd: return M_DS_4EDI_ID;
		case 0xbe: return M_DS_4EDI_ESI;
		case 0xbf: return M_DS_5EDI;

		case 0xc0: return M_DS_9EAX;
		case 0xc1: return M_DS_8EAX_ECX;
		case 0xc2: return M_DS_8EAX_EDX;
		case 0xc3: return M_DS_8EAX_EBX;
		case 0xc4: return M_SS_8EAX_ESP;
		case 0xc5: return M_DS_8EAX_ID;
		case 0xc6: return M_DS_8EAX_ESI;
		case 0xc7: return M_DS_8EAX_EDI;

		case 0xc8: return M_DS_8ECX_EAX;
		case 0xc9: return M_DS_9ECX;
		case 0xca: return M_DS_8ECX_EDX;
		case 0xcb: return M_DS_8ECX_EBX;
		case 0xcc: return M_SS_8ECX_ESP;
		case 0xcd: return M_DS_8ECX_ID;
		case 0xce: return M_DS_8ECX_ESI;
		case 0xcf: return M_DS_8ECX_EDI;

		case 0xd0: return M_DS_8EDX_EAX;
		case 0xd1: return M_DS_8EDX_ECX;
		case 0xd2: return M_DS_9EDX;
		case 0xd3: return M_DS_8EDX_EBX;
		case 0xd4: return M_SS_8EDX_ESP;
		case 0xd5: return M_DS_8EDX_ID;
		case 0xd6: return M_DS_8EDX_ESI;
		case 0xd7: return M_DS_8EDX_EDI;

		case 0xd8: return M_DS_8EBX_EAX;
		case 0xd9: return M_DS_8EBX_ECX;
		case 0xda: return M_DS_8EBX_EDX;
		case 0xdb: return M_DS_9EBX;
		case 0xdc: return M_SS_8EBX_ESP;
		case 0xdd: return M_DS_8EBX_ID;
		case 0xde: return M_DS_8EBX_ESI;
		case 0xdf: return M_DS_8EBX_EDI;

		case 0xe0: return M_DS_EAX;
		case 0xe1: return M_DS_ECX;
		case 0xe2: return M_DS_EDX;
		case 0xe3: return M_DS_EBX;
		case 0xe4: return M_SS_ESP;
		case 0xe5: return M_DS_ID;
		case 0xe6: return M_DS_ESI;
		case 0xe7: return M_DS_EDI;

		case 0xe8: return M_DS_8EBP_EAX;
		case 0xe9: return M_DS_8EBP_ECX;
		case 0xea: return M_DS_8EBP_EDX;
		case 0xeb: return M_DS_8EBP_EBX;
		case 0xec: return M_SS_8EBP_ESP;
		case 0xed: return M_DS_8EBP_ID;
		case 0xee: return M_DS_8EBP_ESI;
		case 0xef: return M_DS_8EBP_EDI;

		case 0xf0: return M_DS_8ESI_EAX;
		case 0xf1: return M_DS_8ESI_ECX;
		case 0xf2: return M_DS_8ESI_EDX;
		case 0xf3: return M_DS_8ESI_EBX;
		case 0xf4: return M_SS_8ESI_ESP;
		case 0xf5: return M_DS_8ESI_ID;
		case 0xf6: return M_DS_9ESI;
		case 0xf7: return M_DS_8ESI_EDI;

		case 0xf8: return M_DS_8EDI_EAX;
		case 0xf9: return M_DS_8EDI_ECX;
		case 0xfa: return M_DS_8EDI_EDX;
		case 0xfb: return M_DS_8EDI_EBX;
		case 0xfc: return M_SS_8EDI_ESP;
		case 0xfd: return M_DS_8EDI_ID;
		case 0xfe: return M_DS_8EDI_ESI;
		case 0xff: return M_DS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX_IB;
		case 0x01: return M_DS_EAX_ECX_IB;
		case 0x02: return M_DS_EAX_EDX_IB;
		case 0x03: return M_DS_EAX_EBX_IB;
		case 0x04: return M_SS_EAX_ESP_IB;
		case 0x05: return M_SS_EAX_EBP_IB;
		case 0x06: return M_DS_EAX_ESI_IB;
		case 0x07: return M_DS_EAX_EDI_IB;

		case 0x08: return M_DS_EAX_ECX_IB;
		case 0x09: return M_DS_2ECX_IB;
		case 0x0a: return M_DS_ECX_EDX_IB;
		case 0x0b: return M_DS_ECX_EBX_IB;
		case 0x0c: return M_SS_ECX_ESP_IB;
		case 0x0d: return M_SS_ECX_EBP_IB;
		case 0x0e: return M_DS_ECX_ESI_IB;
		case 0x0f: return M_DS_ECX_EDI_IB;

		case 0x10: return M_DS_EAX_EDX_IB;
		case 0x11: return M_DS_ECX_EDX_IB;
		case 0x12: return M_DS_2EDX_IB;
		case 0x13: return M_DS_EDX_EBX_IB;
		case 0x14: return M_SS_EDX_ESP_IB;
		case 0x15: return M_SS_EDX_EBP_IB;
		case 0x16: return M_DS_EDX_ESI_IB;
		case 0x17: return M_DS_EDX_EDI_IB;

		case 0x18: return M_DS_EAX_EBX_IB;
		case 0x19: return M_DS_ECX_EBX_IB;
		case 0x1a: return M_DS_EDX_EBX_IB;
		case 0x1b: return M_DS_2EBX_IB;
		case 0x1c: return M_SS_EBX_ESP_IB;
		case 0x1d: return M_SS_EBX_EBP_IB;
		case 0x1e: return M_DS_EBX_ESI_IB;
		case 0x1f: return M_DS_EBX_EDI_IB;

		case 0x20: return M_DS_EAX_IB;
		case 0x21: return M_DS_ECX_IB;
		case 0x22: return M_DS_EDX_IB;
		case 0x23: return M_DS_EBX_IB;
		case 0x24: return M_SS_ESP_IB;
		case 0x25: return M_SS_EBP_IB;
		case 0x26: return M_DS_ESI_IB;
		case 0x27: return M_DS_EDI_IB;

		case 0x28: return M_DS_EAX_EBP_IB;
		case 0x29: return M_DS_ECX_EBP_IB;
		case 0x2a: return M_DS_EDX_EBP_IB;
		case 0x2b: return M_DS_EBX_EBP_IB;
		case 0x2c: return M_SS_EBP_ESP_IB;
		case 0x2d: return M_SS_2EBP_IB;
		case 0x2e: return M_DS_EBP_ESI_IB;
		case 0x2f: return M_DS_EBP_EDI_IB;

		case 0x30: return M_DS_EAX_ESI_IB;
		case 0x31: return M_DS_ECX_ESI_IB;
		case 0x32: return M_DS_EDX_ESI_IB;
		case 0x33: return M_DS_EBX_ESI_IB;
		case 0x34: return M_SS_ESI_ESP_IB;
		case 0x35: return M_SS_EBP_ESI_IB;
		case 0x36: return M_DS_2ESI_IB;
		case 0x37: return M_DS_ESI_EDI_IB;

		case 0x38: return M_DS_EAX_EDI_IB;
		case 0x39: return M_DS_ECX_EDI_IB;
		case 0x3a: return M_DS_EDX_EDI_IB;
		case 0x3b: return M_DS_EBX_EDI_IB;
		case 0x3c: return M_SS_EDI_ESP_IB;
		case 0x3d: return M_SS_EBP_EDI_IB;
		case 0x3e: return M_DS_ESI_EDI_IB;
		case 0x3f: return M_DS_2EDI_IB;

		case 0x40: return M_DS_3EAX_IB;
		case 0x41: return M_DS_2EAX_ECX_IB;
		case 0x42: return M_DS_2EAX_EDX_IB;
		case 0x43: return M_DS_2EAX_EBX_IB;
		case 0x44: return M_SS_2EAX_ESP_IB;
		case 0x45: return M_SS_2EAX_EBP_IB;
		case 0x46: return M_DS_2EAX_ESI_IB;
		case 0x47: return M_DS_2EAX_EDI_IB;

		case 0x48: return M_DS_2ECX_EAX_IB;
		case 0x49: return M_DS_3ECX_IB;
		case 0x4a: return M_DS_2ECX_EDX_IB;
		case 0x4b: return M_DS_2ECX_EBX_IB;
		case 0x4c: return M_SS_2ECX_ESP_IB;
		case 0x4d: return M_SS_2ECX_EBP_IB;
		case 0x4e: return M_DS_2ECX_ESI_IB;
		case 0x4f: return M_DS_2ECX_EDI_IB;

		case 0x50: return M_DS_2EDX_EAX_IB;
		case 0x51: return M_DS_2EDX_ECX_IB;
		case 0x52: return M_DS_3EDX_IB;
		case 0x53: return M_DS_2EDX_EBX_IB;
		case 0x54: return M_SS_2EDX_ESP_IB;
		case 0x55: return M_SS_2EDX_EBP_IB;
		case 0x56: return M_DS_2EDX_ESI_IB;
		case 0x57: return M_DS_2EDX_EDI_IB;

		case 0x58: return M_DS_2EBX_EAX_IB;
		case 0x59: return M_DS_2EBX_ECX_IB;
		case 0x5a: return M_DS_2EBX_EDX_IB;
		case 0x5b: return M_DS_3EBX_IB;
		case 0x5c: return M_SS_2EBX_ESP_IB;
		case 0x5d: return M_SS_2EBX_EBP_IB;
		case 0x5e: return M_DS_2EBX_ESI_IB;
		case 0x5f: return M_DS_2EBX_EDI_IB;

		case 0x60: return M_DS_EAX_IB;
		case 0x61: return M_DS_ECX_IB;
		case 0x62: return M_DS_EDX_IB;
		case 0x63: return M_DS_EBX_IB;
		case 0x64: return M_SS_ESP_IB;
		case 0x65: return M_SS_EBP_IB;
		case 0x66: return M_DS_ESI_IB;
		case 0x67: return M_DS_EDI_IB;

		case 0x68: return M_DS_2EBP_EAX_IB;
		case 0x69: return M_DS_2EBP_ECX_IB;
		case 0x6a: return M_DS_2EBP_EDX_IB;
		case 0x6b: return M_DS_2EBP_EBX_IB;
		case 0x6c: return M_SS_2EBP_ESP_IB;
		case 0x6d: return M_SS_3EBP_IB;
		case 0x6e: return M_DS_2EBP_ESI_IB;
		case 0x6f: return M_DS_2EBP_EDI_IB;

		case 0x70: return M_DS_2ESI_EAX_IB;
		case 0x71: return M_DS_2ESI_ECX_IB;
		case 0x72: return M_DS_2ESI_EDX_IB;
		case 0x73: return M_DS_2ESI_EBX_IB;
		case 0x74: return M_SS_2ESI_ESP_IB;
		case 0x75: return M_SS_2ESI_EBP_IB;
		case 0x76: return M_DS_3ESI_IB;
		case 0x77: return M_DS_2ESI_EDI_IB;

		case 0x78: return M_DS_2EDI_EAX_IB;
		case 0x79: return M_DS_2EDI_ECX_IB;
		case 0x7a: return M_DS_2EDI_EDX_IB;
		case 0x7b: return M_DS_2EDI_EBX_IB;
		case 0x7c: return M_SS_2EDI_ESP_IB;
		case 0x7d: return M_SS_2EDI_EBP_IB;
		case 0x7e: return M_DS_2EDI_ESI_IB;
		case 0x7f: return M_DS_3EDI_IB;

		case 0x80: return M_DS_5EAX_IB;
		case 0x81: return M_DS_4EAX_ECX_IB;
		case 0x82: return M_DS_4EAX_EDX_IB;
		case 0x83: return M_DS_4EAX_EBX_IB;
		case 0x84: return M_SS_4EAX_ESP_IB;
		case 0x85: return M_SS_4EAX_EBP_IB;
		case 0x86: return M_DS_4EAX_ESI_IB;
		case 0x87: return M_DS_4EAX_EDI_IB;

		case 0x88: return M_DS_4ECX_EAX_IB;
		case 0x89: return M_DS_5ECX_IB;
		case 0x8a: return M_DS_4ECX_EDX_IB;
		case 0x8b: return M_DS_4ECX_EBX_IB;
		case 0x8c: return M_SS_4ECX_ESP_IB;
		case 0x8d: return M_SS_4ECX_EBP_IB;
		case 0x8e: return M_DS_4ECX_ESI_IB;
		case 0x8f: return M_DS_4ECX_EDI_IB;

		case 0x90: return M_DS_4EDX_EAX_IB;
		case 0x91: return M_DS_4EDX_ECX_IB;
		case 0x92: return M_DS_5EDX_IB;
		case 0x93: return M_DS_4EDX_EBX_IB;
		case 0x94: return M_SS_4EDX_ESP_IB;
		case 0x95: return M_SS_4EDX_EBP_IB;
		case 0x96: return M_DS_4EDX_ESI_IB;
		case 0x97: return M_DS_4EDX_EDI_IB;

		case 0x98: return M_DS_4EBX_EAX_IB;
		case 0x99: return M_DS_4EBX_ECX_IB;
		case 0x9a: return M_DS_4EBX_EDX_IB;
		case 0x9b: return M_DS_5EBX_IB;
		case 0x9c: return M_SS_4EBX_ESP_IB;
		case 0x9d: return M_SS_4EBX_EBP_IB;
		case 0x9e: return M_DS_4EBX_ESI_IB;
		case 0x9f: return M_DS_4EBX_EDI_IB;

		case 0xa0: return M_DS_EAX_IB;
		case 0xa1: return M_DS_ECX_IB;
		case 0xa2: return M_DS_EDX_IB;
		case 0xa3: return M_DS_EBX_IB;
		case 0xa4: return M_SS_ESP_IB;
		case 0xa5: return M_SS_EBP_IB;
		case 0xa6: return M_DS_ESI_IB;
		case 0xa7: return M_DS_EDI_IB;

		case 0xa8: return M_DS_4EBP_EAX_IB;
		case 0xa9: return M_DS_4EBP_ECX_IB;
		case 0xaa: return M_DS_4EBP_EDX_IB;
		case 0xab: return M_DS_4EBP_EBX_IB;
		case 0xac: return M_SS_4EBP_ESP_IB;
		case 0xad: return M_SS_5EBP_IB;
		case 0xae: return M_DS_4EBP_ESI_IB;
		case 0xaf: return M_DS_4EBP_EDI_IB;

		case 0xb0: return M_DS_4ESI_EAX_IB;
		case 0xb1: return M_DS_4ESI_ECX_IB;
		case 0xb2: return M_DS_4ESI_EDX_IB;
		case 0xb3: return M_DS_4ESI_EBX_IB;
		case 0xb4: return M_SS_4ESI_ESP_IB;
		case 0xb5: return M_SS_4ESI_EBP_IB;
		case 0xb6: return M_DS_5ESI_IB;
		case 0xb7: return M_DS_4ESI_EDI_IB;

		case 0xb8: return M_DS_4EDI_EAX_IB;
		case 0xb9: return M_DS_4EDI_ECX_IB;
		case 0xba: return M_DS_4EDI_EDX_IB;
		case 0xbb: return M_DS_4EDI_EBX_IB;
		case 0xbc: return M_SS_4EDI_ESP_IB;
		case 0xbd: return M_SS_4EDI_EBP_IB;
		case 0xbe: return M_DS_4EDI_ESI_IB;
		case 0xbf: return M_DS_5EDI_IB;

		case 0xc0: return M_DS_9EAX_IB;
		case 0xc1: return M_DS_8EAX_ECX_IB;
		case 0xc2: return M_DS_8EAX_EDX_IB;
		case 0xc3: return M_DS_8EAX_EBX_IB;
		case 0xc4: return M_SS_8EAX_ESP_IB;
		case 0xc5: return M_SS_8EAX_EBP_IB;
		case 0xc6: return M_DS_8EAX_ESI_IB;
		case 0xc7: return M_DS_8EAX_EDI_IB;

		case 0xc8: return M_DS_8ECX_EAX_IB;
		case 0xc9: return M_DS_9ECX_IB;
		case 0xca: return M_DS_8ECX_EDX_IB;
		case 0xcb: return M_DS_8ECX_EBX_IB;
		case 0xcc: return M_SS_8ECX_ESP_IB;
		case 0xcd: return M_SS_8ECX_EBP_IB;
		case 0xce: return M_DS_8ECX_ESI_IB;
		case 0xcf: return M_DS_8ECX_EDI_IB;

		case 0xd0: return M_DS_8EDX_EAX_IB;
		case 0xd1: return M_DS_8EDX_ECX_IB;
		case 0xd2: return M_DS_9EDX_IB;
		case 0xd3: return M_DS_8EDX_EBX_IB;
		case 0xd4: return M_SS_8EDX_ESP_IB;
		case 0xd5: return M_SS_8EDX_EBP_IB;
		case 0xd6: return M_DS_8EDX_ESI_IB;
		case 0xd7: return M_DS_8EDX_EDI_IB;

		case 0xd8: return M_DS_8EBX_EAX_IB;
		case 0xd9: return M_DS_8EBX_ECX_IB;
		case 0xda: return M_DS_8EBX_EDX_IB;
		case 0xdb: return M_DS_9EBX_IB;
		case 0xdc: return M_SS_8EBX_ESP_IB;
		case 0xdd: return M_SS_8EBX_EBP_IB;
		case 0xde: return M_DS_8EBX_ESI_IB;
		case 0xdf: return M_DS_8EBX_EDI_IB;

		case 0xe0: return M_DS_EAX_IB;
		case 0xe1: return M_DS_ECX_IB;
		case 0xe2: return M_DS_EDX_IB;
		case 0xe3: return M_DS_EBX_IB;
		case 0xe4: return M_SS_ESP_IB;
		case 0xe5: return M_SS_EBP_IB;
		case 0xe6: return M_DS_ESI_IB;
		case 0xe7: return M_DS_EDI_IB;

		case 0xe8: return M_DS_8EBP_EAX_IB;
		case 0xe9: return M_DS_8EBP_ECX_IB;
		case 0xea: return M_DS_8EBP_EDX_IB;
		case 0xeb: return M_DS_8EBP_EBX_IB;
		case 0xec: return M_SS_8EBP_ESP_IB;
		case 0xed: return M_SS_9EBP_IB;
		case 0xee: return M_DS_8EBP_ESI_IB;
		case 0xef: return M_DS_8EBP_EDI_IB;

		case 0xf0: return M_DS_8ESI_EAX_IB;
		case 0xf1: return M_DS_8ESI_ECX_IB;
		case 0xf2: return M_DS_8ESI_EDX_IB;
		case 0xf3: return M_DS_8ESI_EBX_IB;
		case 0xf4: return M_SS_8ESI_ESP_IB;
		case 0xf5: return M_SS_8ESI_EBP_IB;
		case 0xf6: return M_DS_9ESI_IB;
		case 0xf7: return M_DS_8ESI_EDI_IB;

		case 0xf8: return M_DS_8EDI_EAX_IB;
		case 0xf9: return M_DS_8EDI_ECX_IB;
		case 0xfa: return M_DS_8EDI_EDX_IB;
		case 0xfb: return M_DS_8EDI_EBX_IB;
		case 0xfc: return M_SS_8EDI_ESP_IB;
		case 0xfd: return M_SS_8EDI_EBP_IB;
		case 0xfe: return M_DS_8EDI_ESI_IB;
		case 0xff: return M_DS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX_ID;
		case 0x01: return M_DS_EAX_ECX_ID;
		case 0x02: return M_DS_EAX_EDX_ID;
		case 0x03: return M_DS_EAX_EBX_ID;
		case 0x04: return M_SS_EAX_ESP_ID;
		case 0x05: return M_SS_EAX_EBP_ID;
		case 0x06: return M_DS_EAX_ESI_ID;
		case 0x07: return M_DS_EAX_EDI_ID;

		case 0x08: return M_DS_EAX_ECX_ID;
		case 0x09: return M_DS_2ECX_ID;
		case 0x0a: return M_DS_ECX_EDX_ID;
		case 0x0b: return M_DS_ECX_EBX_ID;
		case 0x0c: return M_SS_ECX_ESP_ID;
		case 0x0d: return M_SS_ECX_EBP_ID;
		case 0x0e: return M_DS_ECX_ESI_ID;
		case 0x0f: return M_DS_ECX_EDI_ID;

		case 0x10: return M_DS_EAX_EDX_ID;
		case 0x11: return M_DS_ECX_EDX_ID;
		case 0x12: return M_DS_2EDX_ID;
		case 0x13: return M_DS_EDX_EBX_ID;
		case 0x14: return M_SS_EDX_ESP_ID;
		case 0x15: return M_SS_EDX_EBP_ID;
		case 0x16: return M_DS_EDX_ESI_ID;
		case 0x17: return M_DS_EDX_EDI_ID;

		case 0x18: return M_DS_EAX_EBX_ID;
		case 0x19: return M_DS_ECX_EBX_ID;
		case 0x1a: return M_DS_EDX_EBX_ID;
		case 0x1b: return M_DS_2EBX_ID;
		case 0x1c: return M_SS_EBX_ESP_ID;
		case 0x1d: return M_SS_EBX_EBP_ID;
		case 0x1e: return M_DS_EBX_ESI_ID;
		case 0x1f: return M_DS_EBX_EDI_ID;

		case 0x20: return M_DS_EAX_ID;
		case 0x21: return M_DS_ECX_ID;
		case 0x22: return M_DS_EDX_ID;
		case 0x23: return M_DS_EBX_ID;
		case 0x24: return M_SS_ESP_ID;
		case 0x25: return M_SS_EBP_ID;
		case 0x26: return M_DS_ESI_ID;
		case 0x27: return M_DS_EDI_ID;

		case 0x28: return M_DS_EAX_EBP_ID;
		case 0x29: return M_DS_ECX_EBP_ID;
		case 0x2a: return M_DS_EDX_EBP_ID;
		case 0x2b: return M_DS_EBX_EBP_ID;
		case 0x2c: return M_SS_EBP_ESP_ID;
		case 0x2d: return M_SS_2EBP_ID;
		case 0x2e: return M_DS_EBP_ESI_ID;
		case 0x2f: return M_DS_EBP_EDI_ID;

		case 0x30: return M_DS_EAX_ESI_ID;
		case 0x31: return M_DS_ECX_ESI_ID;
		case 0x32: return M_DS_EDX_ESI_ID;
		case 0x33: return M_DS_EBX_ESI_ID;
		case 0x34: return M_SS_ESI_ESP_ID;
		case 0x35: return M_SS_EBP_ESI_ID;
		case 0x36: return M_DS_2ESI_ID;
		case 0x37: return M_DS_ESI_EDI_ID;

		case 0x38: return M_DS_EAX_EDI_ID;
		case 0x39: return M_DS_ECX_EDI_ID;
		case 0x3a: return M_DS_EDX_EDI_ID;
		case 0x3b: return M_DS_EBX_EDI_ID;
		case 0x3c: return M_SS_EDI_ESP_ID;
		case 0x3d: return M_SS_EBP_EDI_ID;
		case 0x3e: return M_DS_ESI_EDI_ID;
		case 0x3f: return M_DS_2EDI_ID;

		case 0x40: return M_DS_3EAX_ID;
		case 0x41: return M_DS_2EAX_ECX_ID;
		case 0x42: return M_DS_2EAX_EDX_ID;
		case 0x43: return M_DS_2EAX_EBX_ID;
		case 0x44: return M_SS_2EAX_ESP_ID;
		case 0x45: return M_SS_2EAX_EBP_ID;
		case 0x46: return M_DS_2EAX_ESI_ID;
		case 0x47: return M_DS_2EAX_EDI_ID;

		case 0x48: return M_DS_2ECX_EAX_ID;
		case 0x49: return M_DS_3ECX_ID;
		case 0x4a: return M_DS_2ECX_EDX_ID;
		case 0x4b: return M_DS_2ECX_EBX_ID;
		case 0x4c: return M_SS_2ECX_ESP_ID;
		case 0x4d: return M_SS_2ECX_EBP_ID;
		case 0x4e: return M_DS_2ECX_ESI_ID;
		case 0x4f: return M_DS_2ECX_EDI_ID;

		case 0x50: return M_DS_2EDX_EAX_ID;
		case 0x51: return M_DS_2EDX_ECX_ID;
		case 0x52: return M_DS_3EDX_ID;
		case 0x53: return M_DS_2EDX_EBX_ID;
		case 0x54: return M_SS_2EDX_ESP_ID;
		case 0x55: return M_SS_2EDX_EBP_ID;
		case 0x56: return M_DS_2EDX_ESI_ID;
		case 0x57: return M_DS_2EDX_EDI_ID;

		case 0x58: return M_DS_2EBX_EAX_ID;
		case 0x59: return M_DS_2EBX_ECX_ID;
		case 0x5a: return M_DS_2EBX_EDX_ID;
		case 0x5b: return M_DS_3EBX_ID;
		case 0x5c: return M_SS_2EBX_ESP_ID;
		case 0x5d: return M_SS_2EBX_EBP_ID;
		case 0x5e: return M_DS_2EBX_ESI_ID;
		case 0x5f: return M_DS_2EBX_EDI_ID;

		case 0x60: return M_DS_EAX_ID;
		case 0x61: return M_DS_ECX_ID;
		case 0x62: return M_DS_EDX_ID;
		case 0x63: return M_DS_EBX_ID;
		case 0x64: return M_SS_ESP_ID;
		case 0x65: return M_SS_EBP_ID;
		case 0x66: return M_DS_ESI_ID;
		case 0x67: return M_DS_EDI_ID;

		case 0x68: return M_DS_2EBP_EAX_ID;
		case 0x69: return M_DS_2EBP_ECX_ID;
		case 0x6a: return M_DS_2EBP_EDX_ID;
		case 0x6b: return M_DS_2EBP_EBX_ID;
		case 0x6c: return M_SS_2EBP_ESP_ID;
		case 0x6d: return M_SS_3EBP_ID;
		case 0x6e: return M_DS_2EBP_ESI_ID;
		case 0x6f: return M_DS_2EBP_EDI_ID;

		case 0x70: return M_DS_2ESI_EAX_ID;
		case 0x71: return M_DS_2ESI_ECX_ID;
		case 0x72: return M_DS_2ESI_EDX_ID;
		case 0x73: return M_DS_2ESI_EBX_ID;
		case 0x74: return M_SS_2ESI_ESP_ID;
		case 0x75: return M_SS_2ESI_EBP_ID;
		case 0x76: return M_DS_3ESI_ID;
		case 0x77: return M_DS_2ESI_EDI_ID;

		case 0x78: return M_DS_2EDI_EAX_ID;
		case 0x79: return M_DS_2EDI_ECX_ID;
		case 0x7a: return M_DS_2EDI_EDX_ID;
		case 0x7b: return M_DS_2EDI_EBX_ID;
		case 0x7c: return M_SS_2EDI_ESP_ID;
		case 0x7d: return M_SS_2EDI_EBP_ID;
		case 0x7e: return M_DS_2EDI_ESI_ID;
		case 0x7f: return M_DS_3EDI_ID;

		case 0x80: return M_DS_5EAX_ID;
		case 0x81: return M_DS_4EAX_ECX_ID;
		case 0x82: return M_DS_4EAX_EDX_ID;
		case 0x83: return M_DS_4EAX_EBX_ID;
		case 0x84: return M_SS_4EAX_ESP_ID;
		case 0x85: return M_SS_4EAX_EBP_ID;
		case 0x86: return M_DS_4EAX_ESI_ID;
		case 0x87: return M_DS_4EAX_EDI_ID;

		case 0x88: return M_DS_4ECX_EAX_ID;
		case 0x89: return M_DS_5ECX_ID;
		case 0x8a: return M_DS_4ECX_EDX_ID;
		case 0x8b: return M_DS_4ECX_EBX_ID;
		case 0x8c: return M_SS_4ECX_ESP_ID;
		case 0x8d: return M_SS_4ECX_EBP_ID;
		case 0x8e: return M_DS_4ECX_ESI_ID;
		case 0x8f: return M_DS_4ECX_EDI_ID;

		case 0x90: return M_DS_4EDX_EAX_ID;
		case 0x91: return M_DS_4EDX_ECX_ID;
		case 0x92: return M_DS_5EDX_ID;
		case 0x93: return M_DS_4EDX_EBX_ID;
		case 0x94: return M_SS_4EDX_ESP_ID;
		case 0x95: return M_SS_4EDX_EBP_ID;
		case 0x96: return M_DS_4EDX_ESI_ID;
		case 0x97: return M_DS_4EDX_EDI_ID;

		case 0x98: return M_DS_4EBX_EAX_ID;
		case 0x99: return M_DS_4EBX_ECX_ID;
		case 0x9a: return M_DS_4EBX_EDX_ID;
		case 0x9b: return M_DS_5EBX_ID;
		case 0x9c: return M_SS_4EBX_ESP_ID;
		case 0x9d: return M_SS_4EBX_EBP_ID;
		case 0x9e: return M_DS_4EBX_ESI_ID;
		case 0x9f: return M_DS_4EBX_EDI_ID;

		case 0xa0: return M_DS_EAX_ID;
		case 0xa1: return M_DS_ECX_ID;
		case 0xa2: return M_DS_EDX_ID;
		case 0xa3: return M_DS_EBX_ID;
		case 0xa4: return M_SS_ESP_ID;
		case 0xa5: return M_SS_EBP_ID;
		case 0xa6: return M_DS_ESI_ID;
		case 0xa7: return M_DS_EDI_ID;

		case 0xa8: return M_DS_4EBP_EAX_ID;
		case 0xa9: return M_DS_4EBP_ECX_ID;
		case 0xaa: return M_DS_4EBP_EDX_ID;
		case 0xab: return M_DS_4EBP_EBX_ID;
		case 0xac: return M_SS_4EBP_ESP_ID;
		case 0xad: return M_SS_5EBP_ID;
		case 0xae: return M_DS_4EBP_ESI_ID;
		case 0xaf: return M_DS_4EBP_EDI_ID;

		case 0xb0: return M_DS_4ESI_EAX_ID;
		case 0xb1: return M_DS_4ESI_ECX_ID;
		case 0xb2: return M_DS_4ESI_EDX_ID;
		case 0xb3: return M_DS_4ESI_EBX_ID;
		case 0xb4: return M_SS_4ESI_ESP_ID;
		case 0xb5: return M_SS_4ESI_EBP_ID;
		case 0xb6: return M_DS_5ESI_ID;
		case 0xb7: return M_DS_4ESI_EDI_ID;

		case 0xb8: return M_DS_4EDI_EAX_ID;
		case 0xb9: return M_DS_4EDI_ECX_ID;
		case 0xba: return M_DS_4EDI_EDX_ID;
		case 0xbb: return M_DS_4EDI_EBX_ID;
		case 0xbc: return M_SS_4EDI_ESP_ID;
		case 0xbd: return M_SS_4EDI_EBP_ID;
		case 0xbe: return M_DS_4EDI_ESI_ID;
		case 0xbf: return M_DS_5EDI_ID;

		case 0xc0: return M_DS_9EAX_ID;
		case 0xc1: return M_DS_8EAX_ECX_ID;
		case 0xc2: return M_DS_8EAX_EDX_ID;
		case 0xc3: return M_DS_8EAX_EBX_ID;
		case 0xc4: return M_SS_8EAX_ESP_ID;
		case 0xc5: return M_SS_8EAX_EBP_ID;
		case 0xc6: return M_DS_8EAX_ESI_ID;
		case 0xc7: return M_DS_8EAX_EDI_ID;

		case 0xc8: return M_DS_8ECX_EAX_ID;
		case 0xc9: return M_DS_9ECX_ID;
		case 0xca: return M_DS_8ECX_EDX_ID;
		case 0xcb: return M_DS_8ECX_EBX_ID;
		case 0xcc: return M_SS_8ECX_ESP_ID;
		case 0xcd: return M_SS_8ECX_EBP_ID;
		case 0xce: return M_DS_8ECX_ESI_ID;
		case 0xcf: return M_DS_8ECX_EDI_ID;

		case 0xd0: return M_DS_8EDX_EAX_ID;
		case 0xd1: return M_DS_8EDX_ECX_ID;
		case 0xd2: return M_DS_9EDX_ID;
		case 0xd3: return M_DS_8EDX_EBX_ID;
		case 0xd4: return M_SS_8EDX_ESP_ID;
		case 0xd5: return M_SS_8EDX_EBP_ID;
		case 0xd6: return M_DS_8EDX_ESI_ID;
		case 0xd7: return M_DS_8EDX_EDI_ID;

		case 0xd8: return M_DS_8EBX_EAX_ID;
		case 0xd9: return M_DS_8EBX_ECX_ID;
		case 0xda: return M_DS_8EBX_EDX_ID;
		case 0xdb: return M_DS_9EBX_ID;
		case 0xdc: return M_SS_8EBX_ESP_ID;
		case 0xdd: return M_SS_8EBX_EBP_ID;
		case 0xde: return M_DS_8EBX_ESI_ID;
		case 0xdf: return M_DS_8EBX_EDI_ID;

		case 0xe0: return M_DS_EAX_ID;
		case 0xe1: return M_DS_ECX_ID;
		case 0xe2: return M_DS_EDX_ID;
		case 0xe3: return M_DS_EBX_ID;
		case 0xe4: return M_SS_ESP_ID;
		case 0xe5: return M_SS_EBP_ID;
		case 0xe6: return M_DS_ESI_ID;
		case 0xe7: return M_DS_EDI_ID;

		case 0xe8: return M_DS_8EBP_EAX_ID;
		case 0xe9: return M_DS_8EBP_ECX_ID;
		case 0xea: return M_DS_8EBP_EDX_ID;
		case 0xeb: return M_DS_8EBP_EBX_ID;
		case 0xec: return M_SS_8EBP_ESP_ID;
		case 0xed: return M_SS_9EBP_ID;
		case 0xee: return M_DS_8EBP_ESI_ID;
		case 0xef: return M_DS_8EBP_EDI_ID;

		case 0xf0: return M_DS_8ESI_EAX_ID;
		case 0xf1: return M_DS_8ESI_ECX_ID;
		case 0xf2: return M_DS_8ESI_EDX_ID;
		case 0xf3: return M_DS_8ESI_EBX_ID;
		case 0xf4: return M_SS_8ESI_ESP_ID;
		case 0xf5: return M_SS_8ESI_EBP_ID;
		case 0xf6: return M_DS_9ESI_ID;
		case 0xf7: return M_DS_8ESI_EDI_ID;

		case 0xf8: return M_DS_8EDI_EAX_ID;
		case 0xf9: return M_DS_8EDI_ECX_ID;
		case 0xfa: return M_DS_8EDI_EDX_ID;
		case 0xfb: return M_DS_8EDI_EBX_ID;
		case 0xfc: return M_SS_8EDI_ESP_ID;
		case 0xfd: return M_SS_8EDI_EBP_ID;
		case 0xfe: return M_DS_8EDI_ESI_ID;
		case 0xff: return M_DS_9EDI_ID;
		}
	    }
	case PREFICES_DS:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX;
		case 0x01: return M_DS_EAX_ECX;
		case 0x02: return M_DS_EAX_EDX;
		case 0x03: return M_DS_EAX_EBX;
		case 0x04: return M_DS_EAX_ESP;
		case 0x05: return M_DS_EAX_ID;
		case 0x06: return M_DS_EAX_ESI;
		case 0x07: return M_DS_EAX_EDI;

		case 0x08: return M_DS_EAX_ECX;
		case 0x09: return M_DS_2ECX;
		case 0x0a: return M_DS_ECX_EDX;
		case 0x0b: return M_DS_ECX_EBX;
		case 0x0c: return M_DS_ECX_ESP;
		case 0x0d: return M_DS_ECX_ID;
		case 0x0e: return M_DS_ECX_ESI;
		case 0x0f: return M_DS_ECX_EDI;

		case 0x10: return M_DS_EAX_EDX;
		case 0x11: return M_DS_ECX_EDX;
		case 0x12: return M_DS_2EDX;
		case 0x13: return M_DS_EDX_EBX;
		case 0x14: return M_DS_EDX_ESP;
		case 0x15: return M_DS_EDX_ID;
		case 0x16: return M_DS_EDX_ESI;
		case 0x17: return M_DS_EDX_EDI;

		case 0x18: return M_DS_EAX_EBX;
		case 0x19: return M_DS_ECX_EBX;
		case 0x1a: return M_DS_EDX_EBX;
		case 0x1b: return M_DS_2EBX;
		case 0x1c: return M_DS_EBX_ESP;
		case 0x1d: return M_DS_EBX_ID;
		case 0x1e: return M_DS_EBX_ESI;
		case 0x1f: return M_DS_EBX_EDI;

		case 0x20: return M_DS_EAX;
		case 0x21: return M_DS_ECX;
		case 0x22: return M_DS_EDX;
		case 0x23: return M_DS_EBX;
		case 0x24: return M_DS_ESP;
		case 0x25: return M_DS_ID;
		case 0x26: return M_DS_ESI;
		case 0x27: return M_DS_EDI;

		case 0x28: return M_DS_EAX_EBP;
		case 0x29: return M_DS_ECX_EBP;
		case 0x2a: return M_DS_EDX_EBP;
		case 0x2b: return M_DS_EBX_EBP;
		case 0x2c: return M_DS_EBP_ESP;
		case 0x2d: return M_DS_EBP_ID;
		case 0x2e: return M_DS_EBP_ESI;
		case 0x2f: return M_DS_EBP_EDI;

		case 0x30: return M_DS_EAX_ESI;
		case 0x31: return M_DS_ECX_ESI;
		case 0x32: return M_DS_EDX_ESI;
		case 0x33: return M_DS_EBX_ESI;
		case 0x34: return M_DS_ESI_ESP;
		case 0x35: return M_DS_ESI_ID;
		case 0x36: return M_DS_2ESI;
		case 0x37: return M_DS_ESI_EDI;

		case 0x38: return M_DS_EAX_EDI;
		case 0x39: return M_DS_ECX_EDI;
		case 0x3a: return M_DS_EDX_EDI;
		case 0x3b: return M_DS_EBX_EDI;
		case 0x3c: return M_DS_EDI_ESP;
		case 0x3d: return M_DS_EDI_ID;
		case 0x3e: return M_DS_ESI_EDI;
		case 0x3f: return M_DS_2EDI;

		case 0x40: return M_DS_3EAX;
		case 0x41: return M_DS_2EAX_ECX;
		case 0x42: return M_DS_2EAX_EDX;
		case 0x43: return M_DS_2EAX_EBX;
		case 0x44: return M_DS_2EAX_ESP;
		case 0x45: return M_DS_2EAX_ID;
		case 0x46: return M_DS_2EAX_ESI;
		case 0x47: return M_DS_2EAX_EDI;

		case 0x48: return M_DS_2ECX_EAX;
		case 0x49: return M_DS_3ECX;
		case 0x4a: return M_DS_2ECX_EDX;
		case 0x4b: return M_DS_2ECX_EBX;
		case 0x4c: return M_DS_2ECX_ESP;
		case 0x4d: return M_DS_2ECX_ID;
		case 0x4e: return M_DS_2ECX_ESI;
		case 0x4f: return M_DS_2ECX_EDI;

		case 0x50: return M_DS_2EDX_EAX;
		case 0x51: return M_DS_2EDX_ECX;
		case 0x52: return M_DS_3EDX;
		case 0x53: return M_DS_2EDX_EBX;
		case 0x54: return M_DS_2EDX_ESP;
		case 0x55: return M_DS_2EDX_ID;
		case 0x56: return M_DS_2EDX_ESI;
		case 0x57: return M_DS_2EDX_EDI;

		case 0x58: return M_DS_2EBX_EAX;
		case 0x59: return M_DS_2EBX_ECX;
		case 0x5a: return M_DS_2EBX_EDX;
		case 0x5b: return M_DS_3EBX;
		case 0x5c: return M_DS_2EBX_ESP;
		case 0x5d: return M_DS_2EBX_ID;
		case 0x5e: return M_DS_2EBX_ESI;
		case 0x5f: return M_DS_2EBX_EDI;

		case 0x60: return M_DS_EAX;
		case 0x61: return M_DS_ECX;
		case 0x62: return M_DS_EDX;
		case 0x63: return M_DS_EBX;
		case 0x64: return M_DS_ESP;
		case 0x65: return M_DS_ID;
		case 0x66: return M_DS_ESI;
		case 0x67: return M_DS_EDI;

		case 0x68: return M_DS_2EBP_EAX;
		case 0x69: return M_DS_2EBP_ECX;
		case 0x6a: return M_DS_2EBP_EDX;
		case 0x6b: return M_DS_2EBP_EBX;
		case 0x6c: return M_DS_2EBP_ESP;
		case 0x6d: return M_DS_2EBP_ID;
		case 0x6e: return M_DS_2EBP_ESI;
		case 0x6f: return M_DS_2EBP_EDI;

		case 0x70: return M_DS_2ESI_EAX;
		case 0x71: return M_DS_2ESI_ECX;
		case 0x72: return M_DS_2ESI_EDX;
		case 0x73: return M_DS_2ESI_EBX;
		case 0x74: return M_DS_2ESI_ESP;
		case 0x75: return M_DS_2ESI_ID;
		case 0x76: return M_DS_3ESI;
		case 0x77: return M_DS_2ESI_EDI;

		case 0x78: return M_DS_2EDI_EAX;
		case 0x79: return M_DS_2EDI_ECX;
		case 0x7a: return M_DS_2EDI_EDX;
		case 0x7b: return M_DS_2EDI_EBX;
		case 0x7c: return M_DS_2EDI_ESP;
		case 0x7d: return M_DS_2EDI_ID;
		case 0x7e: return M_DS_2EDI_ESI;
		case 0x7f: return M_DS_3EDI;

		case 0x80: return M_DS_5EAX;
		case 0x81: return M_DS_4EAX_ECX;
		case 0x82: return M_DS_4EAX_EDX;
		case 0x83: return M_DS_4EAX_EBX;
		case 0x84: return M_DS_4EAX_ESP;
		case 0x85: return M_DS_4EAX_ID;
		case 0x86: return M_DS_4EAX_ESI;
		case 0x87: return M_DS_4EAX_EDI;

		case 0x88: return M_DS_4ECX_EAX;
		case 0x89: return M_DS_5ECX;
		case 0x8a: return M_DS_4ECX_EDX;
		case 0x8b: return M_DS_4ECX_EBX;
		case 0x8c: return M_DS_4ECX_ESP;
		case 0x8d: return M_DS_4ECX_ID;
		case 0x8e: return M_DS_4ECX_ESI;
		case 0x8f: return M_DS_4ECX_EDI;

		case 0x90: return M_DS_4EDX_EAX;
		case 0x91: return M_DS_4EDX_ECX;
		case 0x92: return M_DS_5EDX;
		case 0x93: return M_DS_4EDX_EBX;
		case 0x94: return M_DS_4EDX_ESP;
		case 0x95: return M_DS_4EDX_ID;
		case 0x96: return M_DS_4EDX_ESI;
		case 0x97: return M_DS_4EDX_EDI;

		case 0x98: return M_DS_4EBX_EAX;
		case 0x99: return M_DS_4EBX_ECX;
		case 0x9a: return M_DS_4EBX_EDX;
		case 0x9b: return M_DS_5EBX;
		case 0x9c: return M_DS_4EBX_ESP;
		case 0x9d: return M_DS_4EBX_ID;
		case 0x9e: return M_DS_4EBX_ESI;
		case 0x9f: return M_DS_4EBX_EDI;

		case 0xa0: return M_DS_EAX;
		case 0xa1: return M_DS_ECX;
		case 0xa2: return M_DS_EDX;
		case 0xa3: return M_DS_EBX;
		case 0xa4: return M_DS_ESP;
		case 0xa5: return M_DS_ID;
		case 0xa6: return M_DS_ESI;
		case 0xa7: return M_DS_EDI;

		case 0xa8: return M_DS_4EBP_EAX;
		case 0xa9: return M_DS_4EBP_ECX;
		case 0xaa: return M_DS_4EBP_EDX;
		case 0xab: return M_DS_4EBP_EBX;
		case 0xac: return M_DS_4EBP_ESP;
		case 0xad: return M_DS_4EBP_ID;
		case 0xae: return M_DS_4EBP_ESI;
		case 0xaf: return M_DS_4EBP_EDI;

		case 0xb0: return M_DS_4ESI_EAX;
		case 0xb1: return M_DS_4ESI_ECX;
		case 0xb2: return M_DS_4ESI_EDX;
		case 0xb3: return M_DS_4ESI_EBX;
		case 0xb4: return M_DS_4ESI_ESP;
		case 0xb5: return M_DS_4ESI_ID;
		case 0xb6: return M_DS_5ESI;
		case 0xb7: return M_DS_4ESI_EDI;

		case 0xb8: return M_DS_4EDI_EAX;
		case 0xb9: return M_DS_4EDI_ECX;
		case 0xba: return M_DS_4EDI_EDX;
		case 0xbb: return M_DS_4EDI_EBX;
		case 0xbc: return M_DS_4EDI_ESP;
		case 0xbd: return M_DS_4EDI_ID;
		case 0xbe: return M_DS_4EDI_ESI;
		case 0xbf: return M_DS_5EDI;

		case 0xc0: return M_DS_9EAX;
		case 0xc1: return M_DS_8EAX_ECX;
		case 0xc2: return M_DS_8EAX_EDX;
		case 0xc3: return M_DS_8EAX_EBX;
		case 0xc4: return M_DS_8EAX_ESP;
		case 0xc5: return M_DS_8EAX_ID;
		case 0xc6: return M_DS_8EAX_ESI;
		case 0xc7: return M_DS_8EAX_EDI;

		case 0xc8: return M_DS_8ECX_EAX;
		case 0xc9: return M_DS_9ECX;
		case 0xca: return M_DS_8ECX_EDX;
		case 0xcb: return M_DS_8ECX_EBX;
		case 0xcc: return M_DS_8ECX_ESP;
		case 0xcd: return M_DS_8ECX_ID;
		case 0xce: return M_DS_8ECX_ESI;
		case 0xcf: return M_DS_8ECX_EDI;

		case 0xd0: return M_DS_8EDX_EAX;
		case 0xd1: return M_DS_8EDX_ECX;
		case 0xd2: return M_DS_9EDX;
		case 0xd3: return M_DS_8EDX_EBX;
		case 0xd4: return M_DS_8EDX_ESP;
		case 0xd5: return M_DS_8EDX_ID;
		case 0xd6: return M_DS_8EDX_ESI;
		case 0xd7: return M_DS_8EDX_EDI;

		case 0xd8: return M_DS_8EBX_EAX;
		case 0xd9: return M_DS_8EBX_ECX;
		case 0xda: return M_DS_8EBX_EDX;
		case 0xdb: return M_DS_9EBX;
		case 0xdc: return M_DS_8EBX_ESP;
		case 0xdd: return M_DS_8EBX_ID;
		case 0xde: return M_DS_8EBX_ESI;
		case 0xdf: return M_DS_8EBX_EDI;

		case 0xe0: return M_DS_EAX;
		case 0xe1: return M_DS_ECX;
		case 0xe2: return M_DS_EDX;
		case 0xe3: return M_DS_EBX;
		case 0xe4: return M_DS_ESP;
		case 0xe5: return M_DS_ID;
		case 0xe6: return M_DS_ESI;
		case 0xe7: return M_DS_EDI;

		case 0xe8: return M_DS_8EBP_EAX;
		case 0xe9: return M_DS_8EBP_ECX;
		case 0xea: return M_DS_8EBP_EDX;
		case 0xeb: return M_DS_8EBP_EBX;
		case 0xec: return M_DS_8EBP_ESP;
		case 0xed: return M_DS_8EBP_ID;
		case 0xee: return M_DS_8EBP_ESI;
		case 0xef: return M_DS_8EBP_EDI;

		case 0xf0: return M_DS_8ESI_EAX;
		case 0xf1: return M_DS_8ESI_ECX;
		case 0xf2: return M_DS_8ESI_EDX;
		case 0xf3: return M_DS_8ESI_EBX;
		case 0xf4: return M_DS_8ESI_ESP;
		case 0xf5: return M_DS_8ESI_ID;
		case 0xf6: return M_DS_9ESI;
		case 0xf7: return M_DS_8ESI_EDI;

		case 0xf8: return M_DS_8EDI_EAX;
		case 0xf9: return M_DS_8EDI_ECX;
		case 0xfa: return M_DS_8EDI_EDX;
		case 0xfb: return M_DS_8EDI_EBX;
		case 0xfc: return M_DS_8EDI_ESP;
		case 0xfd: return M_DS_8EDI_ID;
		case 0xfe: return M_DS_8EDI_ESI;
		case 0xff: return M_DS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX_IB;
		case 0x01: return M_DS_EAX_ECX_IB;
		case 0x02: return M_DS_EAX_EDX_IB;
		case 0x03: return M_DS_EAX_EBX_IB;
		case 0x04: return M_DS_EAX_ESP_IB;
		case 0x05: return M_DS_EAX_EBP_IB;
		case 0x06: return M_DS_EAX_ESI_IB;
		case 0x07: return M_DS_EAX_EDI_IB;

		case 0x08: return M_DS_EAX_ECX_IB;
		case 0x09: return M_DS_2ECX_IB;
		case 0x0a: return M_DS_ECX_EDX_IB;
		case 0x0b: return M_DS_ECX_EBX_IB;
		case 0x0c: return M_DS_ECX_ESP_IB;
		case 0x0d: return M_DS_ECX_EBP_IB;
		case 0x0e: return M_DS_ECX_ESI_IB;
		case 0x0f: return M_DS_ECX_EDI_IB;

		case 0x10: return M_DS_EAX_EDX_IB;
		case 0x11: return M_DS_ECX_EDX_IB;
		case 0x12: return M_DS_2EDX_IB;
		case 0x13: return M_DS_EDX_EBX_IB;
		case 0x14: return M_DS_EDX_ESP_IB;
		case 0x15: return M_DS_EDX_EBP_IB;
		case 0x16: return M_DS_EDX_ESI_IB;
		case 0x17: return M_DS_EDX_EDI_IB;

		case 0x18: return M_DS_EAX_EBX_IB;
		case 0x19: return M_DS_ECX_EBX_IB;
		case 0x1a: return M_DS_EDX_EBX_IB;
		case 0x1b: return M_DS_2EBX_IB;
		case 0x1c: return M_DS_EBX_ESP_IB;
		case 0x1d: return M_DS_EBX_EBP_IB;
		case 0x1e: return M_DS_EBX_ESI_IB;
		case 0x1f: return M_DS_EBX_EDI_IB;

		case 0x20: return M_DS_EAX_IB;
		case 0x21: return M_DS_ECX_IB;
		case 0x22: return M_DS_EDX_IB;
		case 0x23: return M_DS_EBX_IB;
		case 0x24: return M_DS_ESP_IB;
		case 0x25: return M_DS_EBP_IB;
		case 0x26: return M_DS_ESI_IB;
		case 0x27: return M_DS_EDI_IB;

		case 0x28: return M_DS_EAX_EBP_IB;
		case 0x29: return M_DS_ECX_EBP_IB;
		case 0x2a: return M_DS_EDX_EBP_IB;
		case 0x2b: return M_DS_EBX_EBP_IB;
		case 0x2c: return M_DS_EBP_ESP_IB;
		case 0x2d: return M_DS_2EBP_IB;
		case 0x2e: return M_DS_EBP_ESI_IB;
		case 0x2f: return M_DS_EBP_EDI_IB;

		case 0x30: return M_DS_EAX_ESI_IB;
		case 0x31: return M_DS_ECX_ESI_IB;
		case 0x32: return M_DS_EDX_ESI_IB;
		case 0x33: return M_DS_EBX_ESI_IB;
		case 0x34: return M_DS_ESI_ESP_IB;
		case 0x35: return M_DS_EBP_ESI_IB;
		case 0x36: return M_DS_2ESI_IB;
		case 0x37: return M_DS_ESI_EDI_IB;

		case 0x38: return M_DS_EAX_EDI_IB;
		case 0x39: return M_DS_ECX_EDI_IB;
		case 0x3a: return M_DS_EDX_EDI_IB;
		case 0x3b: return M_DS_EBX_EDI_IB;
		case 0x3c: return M_DS_EDI_ESP_IB;
		case 0x3d: return M_DS_EBP_EDI_IB;
		case 0x3e: return M_DS_ESI_EDI_IB;
		case 0x3f: return M_DS_2EDI_IB;

		case 0x40: return M_DS_3EAX_IB;
		case 0x41: return M_DS_2EAX_ECX_IB;
		case 0x42: return M_DS_2EAX_EDX_IB;
		case 0x43: return M_DS_2EAX_EBX_IB;
		case 0x44: return M_DS_2EAX_ESP_IB;
		case 0x45: return M_DS_2EAX_EBP_IB;
		case 0x46: return M_DS_2EAX_ESI_IB;
		case 0x47: return M_DS_2EAX_EDI_IB;

		case 0x48: return M_DS_2ECX_EAX_IB;
		case 0x49: return M_DS_3ECX_IB;
		case 0x4a: return M_DS_2ECX_EDX_IB;
		case 0x4b: return M_DS_2ECX_EBX_IB;
		case 0x4c: return M_DS_2ECX_ESP_IB;
		case 0x4d: return M_DS_2ECX_EBP_IB;
		case 0x4e: return M_DS_2ECX_ESI_IB;
		case 0x4f: return M_DS_2ECX_EDI_IB;

		case 0x50: return M_DS_2EDX_EAX_IB;
		case 0x51: return M_DS_2EDX_ECX_IB;
		case 0x52: return M_DS_3EDX_IB;
		case 0x53: return M_DS_2EDX_EBX_IB;
		case 0x54: return M_DS_2EDX_ESP_IB;
		case 0x55: return M_DS_2EDX_EBP_IB;
		case 0x56: return M_DS_2EDX_ESI_IB;
		case 0x57: return M_DS_2EDX_EDI_IB;

		case 0x58: return M_DS_2EBX_EAX_IB;
		case 0x59: return M_DS_2EBX_ECX_IB;
		case 0x5a: return M_DS_2EBX_EDX_IB;
		case 0x5b: return M_DS_3EBX_IB;
		case 0x5c: return M_DS_2EBX_ESP_IB;
		case 0x5d: return M_DS_2EBX_EBP_IB;
		case 0x5e: return M_DS_2EBX_ESI_IB;
		case 0x5f: return M_DS_2EBX_EDI_IB;

		case 0x60: return M_DS_EAX_IB;
		case 0x61: return M_DS_ECX_IB;
		case 0x62: return M_DS_EDX_IB;
		case 0x63: return M_DS_EBX_IB;
		case 0x64: return M_DS_ESP_IB;
		case 0x65: return M_DS_EBP_IB;
		case 0x66: return M_DS_ESI_IB;
		case 0x67: return M_DS_EDI_IB;

		case 0x68: return M_DS_2EBP_EAX_IB;
		case 0x69: return M_DS_2EBP_ECX_IB;
		case 0x6a: return M_DS_2EBP_EDX_IB;
		case 0x6b: return M_DS_2EBP_EBX_IB;
		case 0x6c: return M_DS_2EBP_ESP_IB;
		case 0x6d: return M_DS_3EBP_IB;
		case 0x6e: return M_DS_2EBP_ESI_IB;
		case 0x6f: return M_DS_2EBP_EDI_IB;

		case 0x70: return M_DS_2ESI_EAX_IB;
		case 0x71: return M_DS_2ESI_ECX_IB;
		case 0x72: return M_DS_2ESI_EDX_IB;
		case 0x73: return M_DS_2ESI_EBX_IB;
		case 0x74: return M_DS_2ESI_ESP_IB;
		case 0x75: return M_DS_2ESI_EBP_IB;
		case 0x76: return M_DS_3ESI_IB;
		case 0x77: return M_DS_2ESI_EDI_IB;

		case 0x78: return M_DS_2EDI_EAX_IB;
		case 0x79: return M_DS_2EDI_ECX_IB;
		case 0x7a: return M_DS_2EDI_EDX_IB;
		case 0x7b: return M_DS_2EDI_EBX_IB;
		case 0x7c: return M_DS_2EDI_ESP_IB;
		case 0x7d: return M_DS_2EDI_EBP_IB;
		case 0x7e: return M_DS_2EDI_ESI_IB;
		case 0x7f: return M_DS_3EDI_IB;

		case 0x80: return M_DS_5EAX_IB;
		case 0x81: return M_DS_4EAX_ECX_IB;
		case 0x82: return M_DS_4EAX_EDX_IB;
		case 0x83: return M_DS_4EAX_EBX_IB;
		case 0x84: return M_DS_4EAX_ESP_IB;
		case 0x85: return M_DS_4EAX_EBP_IB;
		case 0x86: return M_DS_4EAX_ESI_IB;
		case 0x87: return M_DS_4EAX_EDI_IB;

		case 0x88: return M_DS_4ECX_EAX_IB;
		case 0x89: return M_DS_5ECX_IB;
		case 0x8a: return M_DS_4ECX_EDX_IB;
		case 0x8b: return M_DS_4ECX_EBX_IB;
		case 0x8c: return M_DS_4ECX_ESP_IB;
		case 0x8d: return M_DS_4ECX_EBP_IB;
		case 0x8e: return M_DS_4ECX_ESI_IB;
		case 0x8f: return M_DS_4ECX_EDI_IB;

		case 0x90: return M_DS_4EDX_EAX_IB;
		case 0x91: return M_DS_4EDX_ECX_IB;
		case 0x92: return M_DS_5EDX_IB;
		case 0x93: return M_DS_4EDX_EBX_IB;
		case 0x94: return M_DS_4EDX_ESP_IB;
		case 0x95: return M_DS_4EDX_EBP_IB;
		case 0x96: return M_DS_4EDX_ESI_IB;
		case 0x97: return M_DS_4EDX_EDI_IB;

		case 0x98: return M_DS_4EBX_EAX_IB;
		case 0x99: return M_DS_4EBX_ECX_IB;
		case 0x9a: return M_DS_4EBX_EDX_IB;
		case 0x9b: return M_DS_5EBX_IB;
		case 0x9c: return M_DS_4EBX_ESP_IB;
		case 0x9d: return M_DS_4EBX_EBP_IB;
		case 0x9e: return M_DS_4EBX_ESI_IB;
		case 0x9f: return M_DS_4EBX_EDI_IB;

		case 0xa0: return M_DS_EAX_IB;
		case 0xa1: return M_DS_ECX_IB;
		case 0xa2: return M_DS_EDX_IB;
		case 0xa3: return M_DS_EBX_IB;
		case 0xa4: return M_DS_ESP_IB;
		case 0xa5: return M_DS_EBP_IB;
		case 0xa6: return M_DS_ESI_IB;
		case 0xa7: return M_DS_EDI_IB;

		case 0xa8: return M_DS_4EBP_EAX_IB;
		case 0xa9: return M_DS_4EBP_ECX_IB;
		case 0xaa: return M_DS_4EBP_EDX_IB;
		case 0xab: return M_DS_4EBP_EBX_IB;
		case 0xac: return M_DS_4EBP_ESP_IB;
		case 0xad: return M_DS_5EBP_IB;
		case 0xae: return M_DS_4EBP_ESI_IB;
		case 0xaf: return M_DS_4EBP_EDI_IB;

		case 0xb0: return M_DS_4ESI_EAX_IB;
		case 0xb1: return M_DS_4ESI_ECX_IB;
		case 0xb2: return M_DS_4ESI_EDX_IB;
		case 0xb3: return M_DS_4ESI_EBX_IB;
		case 0xb4: return M_DS_4ESI_ESP_IB;
		case 0xb5: return M_DS_4ESI_EBP_IB;
		case 0xb6: return M_DS_5ESI_IB;
		case 0xb7: return M_DS_4ESI_EDI_IB;

		case 0xb8: return M_DS_4EDI_EAX_IB;
		case 0xb9: return M_DS_4EDI_ECX_IB;
		case 0xba: return M_DS_4EDI_EDX_IB;
		case 0xbb: return M_DS_4EDI_EBX_IB;
		case 0xbc: return M_DS_4EDI_ESP_IB;
		case 0xbd: return M_DS_4EDI_EBP_IB;
		case 0xbe: return M_DS_4EDI_ESI_IB;
		case 0xbf: return M_DS_5EDI_IB;

		case 0xc0: return M_DS_9EAX_IB;
		case 0xc1: return M_DS_8EAX_ECX_IB;
		case 0xc2: return M_DS_8EAX_EDX_IB;
		case 0xc3: return M_DS_8EAX_EBX_IB;
		case 0xc4: return M_DS_8EAX_ESP_IB;
		case 0xc5: return M_DS_8EAX_EBP_IB;
		case 0xc6: return M_DS_8EAX_ESI_IB;
		case 0xc7: return M_DS_8EAX_EDI_IB;

		case 0xc8: return M_DS_8ECX_EAX_IB;
		case 0xc9: return M_DS_9ECX_IB;
		case 0xca: return M_DS_8ECX_EDX_IB;
		case 0xcb: return M_DS_8ECX_EBX_IB;
		case 0xcc: return M_DS_8ECX_ESP_IB;
		case 0xcd: return M_DS_8ECX_EBP_IB;
		case 0xce: return M_DS_8ECX_ESI_IB;
		case 0xcf: return M_DS_8ECX_EDI_IB;

		case 0xd0: return M_DS_8EDX_EAX_IB;
		case 0xd1: return M_DS_8EDX_ECX_IB;
		case 0xd2: return M_DS_9EDX_IB;
		case 0xd3: return M_DS_8EDX_EBX_IB;
		case 0xd4: return M_DS_8EDX_ESP_IB;
		case 0xd5: return M_DS_8EDX_EBP_IB;
		case 0xd6: return M_DS_8EDX_ESI_IB;
		case 0xd7: return M_DS_8EDX_EDI_IB;

		case 0xd8: return M_DS_8EBX_EAX_IB;
		case 0xd9: return M_DS_8EBX_ECX_IB;
		case 0xda: return M_DS_8EBX_EDX_IB;
		case 0xdb: return M_DS_9EBX_IB;
		case 0xdc: return M_DS_8EBX_ESP_IB;
		case 0xdd: return M_DS_8EBX_EBP_IB;
		case 0xde: return M_DS_8EBX_ESI_IB;
		case 0xdf: return M_DS_8EBX_EDI_IB;

		case 0xe0: return M_DS_EAX_IB;
		case 0xe1: return M_DS_ECX_IB;
		case 0xe2: return M_DS_EDX_IB;
		case 0xe3: return M_DS_EBX_IB;
		case 0xe4: return M_DS_ESP_IB;
		case 0xe5: return M_DS_EBP_IB;
		case 0xe6: return M_DS_ESI_IB;
		case 0xe7: return M_DS_EDI_IB;

		case 0xe8: return M_DS_8EBP_EAX_IB;
		case 0xe9: return M_DS_8EBP_ECX_IB;
		case 0xea: return M_DS_8EBP_EDX_IB;
		case 0xeb: return M_DS_8EBP_EBX_IB;
		case 0xec: return M_DS_8EBP_ESP_IB;
		case 0xed: return M_DS_9EBP_IB;
		case 0xee: return M_DS_8EBP_ESI_IB;
		case 0xef: return M_DS_8EBP_EDI_IB;

		case 0xf0: return M_DS_8ESI_EAX_IB;
		case 0xf1: return M_DS_8ESI_ECX_IB;
		case 0xf2: return M_DS_8ESI_EDX_IB;
		case 0xf3: return M_DS_8ESI_EBX_IB;
		case 0xf4: return M_DS_8ESI_ESP_IB;
		case 0xf5: return M_DS_8ESI_EBP_IB;
		case 0xf6: return M_DS_9ESI_IB;
		case 0xf7: return M_DS_8ESI_EDI_IB;

		case 0xf8: return M_DS_8EDI_EAX_IB;
		case 0xf9: return M_DS_8EDI_ECX_IB;
		case 0xfa: return M_DS_8EDI_EDX_IB;
		case 0xfb: return M_DS_8EDI_EBX_IB;
		case 0xfc: return M_DS_8EDI_ESP_IB;
		case 0xfd: return M_DS_8EDI_EBP_IB;
		case 0xfe: return M_DS_8EDI_ESI_IB;
		case 0xff: return M_DS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_DS_2EAX_ID;
		case 0x01: return M_DS_EAX_ECX_ID;
		case 0x02: return M_DS_EAX_EDX_ID;
		case 0x03: return M_DS_EAX_EBX_ID;
		case 0x04: return M_DS_EAX_ESP_ID;
		case 0x05: return M_DS_EAX_EBP_ID;
		case 0x06: return M_DS_EAX_ESI_ID;
		case 0x07: return M_DS_EAX_EDI_ID;

		case 0x08: return M_DS_EAX_ECX_ID;
		case 0x09: return M_DS_2ECX_ID;
		case 0x0a: return M_DS_ECX_EDX_ID;
		case 0x0b: return M_DS_ECX_EBX_ID;
		case 0x0c: return M_DS_ECX_ESP_ID;
		case 0x0d: return M_DS_ECX_EBP_ID;
		case 0x0e: return M_DS_ECX_ESI_ID;
		case 0x0f: return M_DS_ECX_EDI_ID;

		case 0x10: return M_DS_EAX_EDX_ID;
		case 0x11: return M_DS_ECX_EDX_ID;
		case 0x12: return M_DS_2EDX_ID;
		case 0x13: return M_DS_EDX_EBX_ID;
		case 0x14: return M_DS_EDX_ESP_ID;
		case 0x15: return M_DS_EDX_EBP_ID;
		case 0x16: return M_DS_EDX_ESI_ID;
		case 0x17: return M_DS_EDX_EDI_ID;

		case 0x18: return M_DS_EAX_EBX_ID;
		case 0x19: return M_DS_ECX_EBX_ID;
		case 0x1a: return M_DS_EDX_EBX_ID;
		case 0x1b: return M_DS_2EBX_ID;
		case 0x1c: return M_DS_EBX_ESP_ID;
		case 0x1d: return M_DS_EBX_EBP_ID;
		case 0x1e: return M_DS_EBX_ESI_ID;
		case 0x1f: return M_DS_EBX_EDI_ID;

		case 0x20: return M_DS_EAX_ID;
		case 0x21: return M_DS_ECX_ID;
		case 0x22: return M_DS_EDX_ID;
		case 0x23: return M_DS_EBX_ID;
		case 0x24: return M_DS_ESP_ID;
		case 0x25: return M_DS_EBP_ID;
		case 0x26: return M_DS_ESI_ID;
		case 0x27: return M_DS_EDI_ID;

		case 0x28: return M_DS_EAX_EBP_ID;
		case 0x29: return M_DS_ECX_EBP_ID;
		case 0x2a: return M_DS_EDX_EBP_ID;
		case 0x2b: return M_DS_EBX_EBP_ID;
		case 0x2c: return M_DS_EBP_ESP_ID;
		case 0x2d: return M_DS_2EBP_ID;
		case 0x2e: return M_DS_EBP_ESI_ID;
		case 0x2f: return M_DS_EBP_EDI_ID;

		case 0x30: return M_DS_EAX_ESI_ID;
		case 0x31: return M_DS_ECX_ESI_ID;
		case 0x32: return M_DS_EDX_ESI_ID;
		case 0x33: return M_DS_EBX_ESI_ID;
		case 0x34: return M_DS_ESI_ESP_ID;
		case 0x35: return M_DS_EBP_ESI_ID;
		case 0x36: return M_DS_2ESI_ID;
		case 0x37: return M_DS_ESI_EDI_ID;

		case 0x38: return M_DS_EAX_EDI_ID;
		case 0x39: return M_DS_ECX_EDI_ID;
		case 0x3a: return M_DS_EDX_EDI_ID;
		case 0x3b: return M_DS_EBX_EDI_ID;
		case 0x3c: return M_DS_EDI_ESP_ID;
		case 0x3d: return M_DS_EBP_EDI_ID;
		case 0x3e: return M_DS_ESI_EDI_ID;
		case 0x3f: return M_DS_2EDI_ID;

		case 0x40: return M_DS_3EAX_ID;
		case 0x41: return M_DS_2EAX_ECX_ID;
		case 0x42: return M_DS_2EAX_EDX_ID;
		case 0x43: return M_DS_2EAX_EBX_ID;
		case 0x44: return M_DS_2EAX_ESP_ID;
		case 0x45: return M_DS_2EAX_EBP_ID;
		case 0x46: return M_DS_2EAX_ESI_ID;
		case 0x47: return M_DS_2EAX_EDI_ID;

		case 0x48: return M_DS_2ECX_EAX_ID;
		case 0x49: return M_DS_3ECX_ID;
		case 0x4a: return M_DS_2ECX_EDX_ID;
		case 0x4b: return M_DS_2ECX_EBX_ID;
		case 0x4c: return M_DS_2ECX_ESP_ID;
		case 0x4d: return M_DS_2ECX_EBP_ID;
		case 0x4e: return M_DS_2ECX_ESI_ID;
		case 0x4f: return M_DS_2ECX_EDI_ID;

		case 0x50: return M_DS_2EDX_EAX_ID;
		case 0x51: return M_DS_2EDX_ECX_ID;
		case 0x52: return M_DS_3EDX_ID;
		case 0x53: return M_DS_2EDX_EBX_ID;
		case 0x54: return M_DS_2EDX_ESP_ID;
		case 0x55: return M_DS_2EDX_EBP_ID;
		case 0x56: return M_DS_2EDX_ESI_ID;
		case 0x57: return M_DS_2EDX_EDI_ID;

		case 0x58: return M_DS_2EBX_EAX_ID;
		case 0x59: return M_DS_2EBX_ECX_ID;
		case 0x5a: return M_DS_2EBX_EDX_ID;
		case 0x5b: return M_DS_3EBX_ID;
		case 0x5c: return M_DS_2EBX_ESP_ID;
		case 0x5d: return M_DS_2EBX_EBP_ID;
		case 0x5e: return M_DS_2EBX_ESI_ID;
		case 0x5f: return M_DS_2EBX_EDI_ID;

		case 0x60: return M_DS_EAX_ID;
		case 0x61: return M_DS_ECX_ID;
		case 0x62: return M_DS_EDX_ID;
		case 0x63: return M_DS_EBX_ID;
		case 0x64: return M_DS_ESP_ID;
		case 0x65: return M_DS_EBP_ID;
		case 0x66: return M_DS_ESI_ID;
		case 0x67: return M_DS_EDI_ID;

		case 0x68: return M_DS_2EBP_EAX_ID;
		case 0x69: return M_DS_2EBP_ECX_ID;
		case 0x6a: return M_DS_2EBP_EDX_ID;
		case 0x6b: return M_DS_2EBP_EBX_ID;
		case 0x6c: return M_DS_2EBP_ESP_ID;
		case 0x6d: return M_DS_3EBP_ID;
		case 0x6e: return M_DS_2EBP_ESI_ID;
		case 0x6f: return M_DS_2EBP_EDI_ID;

		case 0x70: return M_DS_2ESI_EAX_ID;
		case 0x71: return M_DS_2ESI_ECX_ID;
		case 0x72: return M_DS_2ESI_EDX_ID;
		case 0x73: return M_DS_2ESI_EBX_ID;
		case 0x74: return M_DS_2ESI_ESP_ID;
		case 0x75: return M_DS_2ESI_EBP_ID;
		case 0x76: return M_DS_3ESI_ID;
		case 0x77: return M_DS_2ESI_EDI_ID;

		case 0x78: return M_DS_2EDI_EAX_ID;
		case 0x79: return M_DS_2EDI_ECX_ID;
		case 0x7a: return M_DS_2EDI_EDX_ID;
		case 0x7b: return M_DS_2EDI_EBX_ID;
		case 0x7c: return M_DS_2EDI_ESP_ID;
		case 0x7d: return M_DS_2EDI_EBP_ID;
		case 0x7e: return M_DS_2EDI_ESI_ID;
		case 0x7f: return M_DS_3EDI_ID;

		case 0x80: return M_DS_5EAX_ID;
		case 0x81: return M_DS_4EAX_ECX_ID;
		case 0x82: return M_DS_4EAX_EDX_ID;
		case 0x83: return M_DS_4EAX_EBX_ID;
		case 0x84: return M_DS_4EAX_ESP_ID;
		case 0x85: return M_DS_4EAX_EBP_ID;
		case 0x86: return M_DS_4EAX_ESI_ID;
		case 0x87: return M_DS_4EAX_EDI_ID;

		case 0x88: return M_DS_4ECX_EAX_ID;
		case 0x89: return M_DS_5ECX_ID;
		case 0x8a: return M_DS_4ECX_EDX_ID;
		case 0x8b: return M_DS_4ECX_EBX_ID;
		case 0x8c: return M_DS_4ECX_ESP_ID;
		case 0x8d: return M_DS_4ECX_EBP_ID;
		case 0x8e: return M_DS_4ECX_ESI_ID;
		case 0x8f: return M_DS_4ECX_EDI_ID;

		case 0x90: return M_DS_4EDX_EAX_ID;
		case 0x91: return M_DS_4EDX_ECX_ID;
		case 0x92: return M_DS_5EDX_ID;
		case 0x93: return M_DS_4EDX_EBX_ID;
		case 0x94: return M_DS_4EDX_ESP_ID;
		case 0x95: return M_DS_4EDX_EBP_ID;
		case 0x96: return M_DS_4EDX_ESI_ID;
		case 0x97: return M_DS_4EDX_EDI_ID;

		case 0x98: return M_DS_4EBX_EAX_ID;
		case 0x99: return M_DS_4EBX_ECX_ID;
		case 0x9a: return M_DS_4EBX_EDX_ID;
		case 0x9b: return M_DS_5EBX_ID;
		case 0x9c: return M_DS_4EBX_ESP_ID;
		case 0x9d: return M_DS_4EBX_EBP_ID;
		case 0x9e: return M_DS_4EBX_ESI_ID;
		case 0x9f: return M_DS_4EBX_EDI_ID;

		case 0xa0: return M_DS_EAX_ID;
		case 0xa1: return M_DS_ECX_ID;
		case 0xa2: return M_DS_EDX_ID;
		case 0xa3: return M_DS_EBX_ID;
		case 0xa4: return M_DS_ESP_ID;
		case 0xa5: return M_DS_EBP_ID;
		case 0xa6: return M_DS_ESI_ID;
		case 0xa7: return M_DS_EDI_ID;

		case 0xa8: return M_DS_4EBP_EAX_ID;
		case 0xa9: return M_DS_4EBP_ECX_ID;
		case 0xaa: return M_DS_4EBP_EDX_ID;
		case 0xab: return M_DS_4EBP_EBX_ID;
		case 0xac: return M_DS_4EBP_ESP_ID;
		case 0xad: return M_DS_5EBP_ID;
		case 0xae: return M_DS_4EBP_ESI_ID;
		case 0xaf: return M_DS_4EBP_EDI_ID;

		case 0xb0: return M_DS_4ESI_EAX_ID;
		case 0xb1: return M_DS_4ESI_ECX_ID;
		case 0xb2: return M_DS_4ESI_EDX_ID;
		case 0xb3: return M_DS_4ESI_EBX_ID;
		case 0xb4: return M_DS_4ESI_ESP_ID;
		case 0xb5: return M_DS_4ESI_EBP_ID;
		case 0xb6: return M_DS_5ESI_ID;
		case 0xb7: return M_DS_4ESI_EDI_ID;

		case 0xb8: return M_DS_4EDI_EAX_ID;
		case 0xb9: return M_DS_4EDI_ECX_ID;
		case 0xba: return M_DS_4EDI_EDX_ID;
		case 0xbb: return M_DS_4EDI_EBX_ID;
		case 0xbc: return M_DS_4EDI_ESP_ID;
		case 0xbd: return M_DS_4EDI_EBP_ID;
		case 0xbe: return M_DS_4EDI_ESI_ID;
		case 0xbf: return M_DS_5EDI_ID;

		case 0xc0: return M_DS_9EAX_ID;
		case 0xc1: return M_DS_8EAX_ECX_ID;
		case 0xc2: return M_DS_8EAX_EDX_ID;
		case 0xc3: return M_DS_8EAX_EBX_ID;
		case 0xc4: return M_DS_8EAX_ESP_ID;
		case 0xc5: return M_DS_8EAX_EBP_ID;
		case 0xc6: return M_DS_8EAX_ESI_ID;
		case 0xc7: return M_DS_8EAX_EDI_ID;

		case 0xc8: return M_DS_8ECX_EAX_ID;
		case 0xc9: return M_DS_9ECX_ID;
		case 0xca: return M_DS_8ECX_EDX_ID;
		case 0xcb: return M_DS_8ECX_EBX_ID;
		case 0xcc: return M_DS_8ECX_ESP_ID;
		case 0xcd: return M_DS_8ECX_EBP_ID;
		case 0xce: return M_DS_8ECX_ESI_ID;
		case 0xcf: return M_DS_8ECX_EDI_ID;

		case 0xd0: return M_DS_8EDX_EAX_ID;
		case 0xd1: return M_DS_8EDX_ECX_ID;
		case 0xd2: return M_DS_9EDX_ID;
		case 0xd3: return M_DS_8EDX_EBX_ID;
		case 0xd4: return M_DS_8EDX_ESP_ID;
		case 0xd5: return M_DS_8EDX_EBP_ID;
		case 0xd6: return M_DS_8EDX_ESI_ID;
		case 0xd7: return M_DS_8EDX_EDI_ID;

		case 0xd8: return M_DS_8EBX_EAX_ID;
		case 0xd9: return M_DS_8EBX_ECX_ID;
		case 0xda: return M_DS_8EBX_EDX_ID;
		case 0xdb: return M_DS_9EBX_ID;
		case 0xdc: return M_DS_8EBX_ESP_ID;
		case 0xdd: return M_DS_8EBX_EBP_ID;
		case 0xde: return M_DS_8EBX_ESI_ID;
		case 0xdf: return M_DS_8EBX_EDI_ID;

		case 0xe0: return M_DS_EAX_ID;
		case 0xe1: return M_DS_ECX_ID;
		case 0xe2: return M_DS_EDX_ID;
		case 0xe3: return M_DS_EBX_ID;
		case 0xe4: return M_DS_ESP_ID;
		case 0xe5: return M_DS_EBP_ID;
		case 0xe6: return M_DS_ESI_ID;
		case 0xe7: return M_DS_EDI_ID;

		case 0xe8: return M_DS_8EBP_EAX_ID;
		case 0xe9: return M_DS_8EBP_ECX_ID;
		case 0xea: return M_DS_8EBP_EDX_ID;
		case 0xeb: return M_DS_8EBP_EBX_ID;
		case 0xec: return M_DS_8EBP_ESP_ID;
		case 0xed: return M_DS_9EBP_ID;
		case 0xee: return M_DS_8EBP_ESI_ID;
		case 0xef: return M_DS_8EBP_EDI_ID;

		case 0xf0: return M_DS_8ESI_EAX_ID;
		case 0xf1: return M_DS_8ESI_ECX_ID;
		case 0xf2: return M_DS_8ESI_EDX_ID;
		case 0xf3: return M_DS_8ESI_EBX_ID;
		case 0xf4: return M_DS_8ESI_ESP_ID;
		case 0xf5: return M_DS_8ESI_EBP_ID;
		case 0xf6: return M_DS_9ESI_ID;
		case 0xf7: return M_DS_8ESI_EDI_ID;

		case 0xf8: return M_DS_8EDI_EAX_ID;
		case 0xf9: return M_DS_8EDI_ECX_ID;
		case 0xfa: return M_DS_8EDI_EDX_ID;
		case 0xfb: return M_DS_8EDI_EBX_ID;
		case 0xfc: return M_DS_8EDI_ESP_ID;
		case 0xfd: return M_DS_8EDI_EBP_ID;
		case 0xfe: return M_DS_8EDI_ESI_ID;
		case 0xff: return M_DS_9EDI_ID;
		}
	    }
	case PREFICES_ES:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_ES_2EAX;
		case 0x01: return M_ES_EAX_ECX;
		case 0x02: return M_ES_EAX_EDX;
		case 0x03: return M_ES_EAX_EBX;
		case 0x04: return M_ES_EAX_ESP;
		case 0x05: return M_ES_EAX_ID;
		case 0x06: return M_ES_EAX_ESI;
		case 0x07: return M_ES_EAX_EDI;

		case 0x08: return M_ES_EAX_ECX;
		case 0x09: return M_ES_2ECX;
		case 0x0a: return M_ES_ECX_EDX;
		case 0x0b: return M_ES_ECX_EBX;
		case 0x0c: return M_ES_ECX_ESP;
		case 0x0d: return M_ES_ECX_ID;
		case 0x0e: return M_ES_ECX_ESI;
		case 0x0f: return M_ES_ECX_EDI;

		case 0x10: return M_ES_EAX_EDX;
		case 0x11: return M_ES_ECX_EDX;
		case 0x12: return M_ES_2EDX;
		case 0x13: return M_ES_EDX_EBX;
		case 0x14: return M_ES_EDX_ESP;
		case 0x15: return M_ES_EDX_ID;
		case 0x16: return M_ES_EDX_ESI;
		case 0x17: return M_ES_EDX_EDI;

		case 0x18: return M_ES_EAX_EBX;
		case 0x19: return M_ES_ECX_EBX;
		case 0x1a: return M_ES_EDX_EBX;
		case 0x1b: return M_ES_2EBX;
		case 0x1c: return M_ES_EBX_ESP;
		case 0x1d: return M_ES_EBX_ID;
		case 0x1e: return M_ES_EBX_ESI;
		case 0x1f: return M_ES_EBX_EDI;

		case 0x20: return M_ES_EAX;
		case 0x21: return M_ES_ECX;
		case 0x22: return M_ES_EDX;
		case 0x23: return M_ES_EBX;
		case 0x24: return M_ES_ESP;
		case 0x25: return M_ES_ID;
		case 0x26: return M_ES_ESI;
		case 0x27: return M_ES_EDI;

		case 0x28: return M_ES_EAX_EBP;
		case 0x29: return M_ES_ECX_EBP;
		case 0x2a: return M_ES_EDX_EBP;
		case 0x2b: return M_ES_EBX_EBP;
		case 0x2c: return M_ES_EBP_ESP;
		case 0x2d: return M_ES_EBP_ID;
		case 0x2e: return M_ES_EBP_ESI;
		case 0x2f: return M_ES_EBP_EDI;

		case 0x30: return M_ES_EAX_ESI;
		case 0x31: return M_ES_ECX_ESI;
		case 0x32: return M_ES_EDX_ESI;
		case 0x33: return M_ES_EBX_ESI;
		case 0x34: return M_ES_ESI_ESP;
		case 0x35: return M_ES_ESI_ID;
		case 0x36: return M_ES_2ESI;
		case 0x37: return M_ES_ESI_EDI;

		case 0x38: return M_ES_EAX_EDI;
		case 0x39: return M_ES_ECX_EDI;
		case 0x3a: return M_ES_EDX_EDI;
		case 0x3b: return M_ES_EBX_EDI;
		case 0x3c: return M_ES_EDI_ESP;
		case 0x3d: return M_ES_EDI_ID;
		case 0x3e: return M_ES_ESI_EDI;
		case 0x3f: return M_ES_2EDI;

		case 0x40: return M_ES_3EAX;
		case 0x41: return M_ES_2EAX_ECX;
		case 0x42: return M_ES_2EAX_EDX;
		case 0x43: return M_ES_2EAX_EBX;
		case 0x44: return M_ES_2EAX_ESP;
		case 0x45: return M_ES_2EAX_ID;
		case 0x46: return M_ES_2EAX_ESI;
		case 0x47: return M_ES_2EAX_EDI;

		case 0x48: return M_ES_2ECX_EAX;
		case 0x49: return M_ES_3ECX;
		case 0x4a: return M_ES_2ECX_EDX;
		case 0x4b: return M_ES_2ECX_EBX;
		case 0x4c: return M_ES_2ECX_ESP;
		case 0x4d: return M_ES_2ECX_ID;
		case 0x4e: return M_ES_2ECX_ESI;
		case 0x4f: return M_ES_2ECX_EDI;

		case 0x50: return M_ES_2EDX_EAX;
		case 0x51: return M_ES_2EDX_ECX;
		case 0x52: return M_ES_3EDX;
		case 0x53: return M_ES_2EDX_EBX;
		case 0x54: return M_ES_2EDX_ESP;
		case 0x55: return M_ES_2EDX_ID;
		case 0x56: return M_ES_2EDX_ESI;
		case 0x57: return M_ES_2EDX_EDI;

		case 0x58: return M_ES_2EBX_EAX;
		case 0x59: return M_ES_2EBX_ECX;
		case 0x5a: return M_ES_2EBX_EDX;
		case 0x5b: return M_ES_3EBX;
		case 0x5c: return M_ES_2EBX_ESP;
		case 0x5d: return M_ES_2EBX_ID;
		case 0x5e: return M_ES_2EBX_ESI;
		case 0x5f: return M_ES_2EBX_EDI;

		case 0x60: return M_ES_EAX;
		case 0x61: return M_ES_ECX;
		case 0x62: return M_ES_EDX;
		case 0x63: return M_ES_EBX;
		case 0x64: return M_ES_ESP;
		case 0x65: return M_ES_ID;
		case 0x66: return M_ES_ESI;
		case 0x67: return M_ES_EDI;

		case 0x68: return M_ES_2EBP_EAX;
		case 0x69: return M_ES_2EBP_ECX;
		case 0x6a: return M_ES_2EBP_EDX;
		case 0x6b: return M_ES_2EBP_EBX;
		case 0x6c: return M_ES_2EBP_ESP;
		case 0x6d: return M_ES_2EBP_ID;
		case 0x6e: return M_ES_2EBP_ESI;
		case 0x6f: return M_ES_2EBP_EDI;

		case 0x70: return M_ES_2ESI_EAX;
		case 0x71: return M_ES_2ESI_ECX;
		case 0x72: return M_ES_2ESI_EDX;
		case 0x73: return M_ES_2ESI_EBX;
		case 0x74: return M_ES_2ESI_ESP;
		case 0x75: return M_ES_2ESI_ID;
		case 0x76: return M_ES_3ESI;
		case 0x77: return M_ES_2ESI_EDI;

		case 0x78: return M_ES_2EDI_EAX;
		case 0x79: return M_ES_2EDI_ECX;
		case 0x7a: return M_ES_2EDI_EDX;
		case 0x7b: return M_ES_2EDI_EBX;
		case 0x7c: return M_ES_2EDI_ESP;
		case 0x7d: return M_ES_2EDI_ID;
		case 0x7e: return M_ES_2EDI_ESI;
		case 0x7f: return M_ES_3EDI;

		case 0x80: return M_ES_5EAX;
		case 0x81: return M_ES_4EAX_ECX;
		case 0x82: return M_ES_4EAX_EDX;
		case 0x83: return M_ES_4EAX_EBX;
		case 0x84: return M_ES_4EAX_ESP;
		case 0x85: return M_ES_4EAX_ID;
		case 0x86: return M_ES_4EAX_ESI;
		case 0x87: return M_ES_4EAX_EDI;

		case 0x88: return M_ES_4ECX_EAX;
		case 0x89: return M_ES_5ECX;
		case 0x8a: return M_ES_4ECX_EDX;
		case 0x8b: return M_ES_4ECX_EBX;
		case 0x8c: return M_ES_4ECX_ESP;
		case 0x8d: return M_ES_4ECX_ID;
		case 0x8e: return M_ES_4ECX_ESI;
		case 0x8f: return M_ES_4ECX_EDI;

		case 0x90: return M_ES_4EDX_EAX;
		case 0x91: return M_ES_4EDX_ECX;
		case 0x92: return M_ES_5EDX;
		case 0x93: return M_ES_4EDX_EBX;
		case 0x94: return M_ES_4EDX_ESP;
		case 0x95: return M_ES_4EDX_ID;
		case 0x96: return M_ES_4EDX_ESI;
		case 0x97: return M_ES_4EDX_EDI;

		case 0x98: return M_ES_4EBX_EAX;
		case 0x99: return M_ES_4EBX_ECX;
		case 0x9a: return M_ES_4EBX_EDX;
		case 0x9b: return M_ES_5EBX;
		case 0x9c: return M_ES_4EBX_ESP;
		case 0x9d: return M_ES_4EBX_ID;
		case 0x9e: return M_ES_4EBX_ESI;
		case 0x9f: return M_ES_4EBX_EDI;

		case 0xa0: return M_ES_EAX;
		case 0xa1: return M_ES_ECX;
		case 0xa2: return M_ES_EDX;
		case 0xa3: return M_ES_EBX;
		case 0xa4: return M_ES_ESP;
		case 0xa5: return M_ES_ID;
		case 0xa6: return M_ES_ESI;
		case 0xa7: return M_ES_EDI;

		case 0xa8: return M_ES_4EBP_EAX;
		case 0xa9: return M_ES_4EBP_ECX;
		case 0xaa: return M_ES_4EBP_EDX;
		case 0xab: return M_ES_4EBP_EBX;
		case 0xac: return M_ES_4EBP_ESP;
		case 0xad: return M_ES_4EBP_ID;
		case 0xae: return M_ES_4EBP_ESI;
		case 0xaf: return M_ES_4EBP_EDI;

		case 0xb0: return M_ES_4ESI_EAX;
		case 0xb1: return M_ES_4ESI_ECX;
		case 0xb2: return M_ES_4ESI_EDX;
		case 0xb3: return M_ES_4ESI_EBX;
		case 0xb4: return M_ES_4ESI_ESP;
		case 0xb5: return M_ES_4ESI_ID;
		case 0xb6: return M_ES_5ESI;
		case 0xb7: return M_ES_4ESI_EDI;

		case 0xb8: return M_ES_4EDI_EAX;
		case 0xb9: return M_ES_4EDI_ECX;
		case 0xba: return M_ES_4EDI_EDX;
		case 0xbb: return M_ES_4EDI_EBX;
		case 0xbc: return M_ES_4EDI_ESP;
		case 0xbd: return M_ES_4EDI_ID;
		case 0xbe: return M_ES_4EDI_ESI;
		case 0xbf: return M_ES_5EDI;

		case 0xc0: return M_ES_9EAX;
		case 0xc1: return M_ES_8EAX_ECX;
		case 0xc2: return M_ES_8EAX_EDX;
		case 0xc3: return M_ES_8EAX_EBX;
		case 0xc4: return M_ES_8EAX_ESP;
		case 0xc5: return M_ES_8EAX_ID;
		case 0xc6: return M_ES_8EAX_ESI;
		case 0xc7: return M_ES_8EAX_EDI;

		case 0xc8: return M_ES_8ECX_EAX;
		case 0xc9: return M_ES_9ECX;
		case 0xca: return M_ES_8ECX_EDX;
		case 0xcb: return M_ES_8ECX_EBX;
		case 0xcc: return M_ES_8ECX_ESP;
		case 0xcd: return M_ES_8ECX_ID;
		case 0xce: return M_ES_8ECX_ESI;
		case 0xcf: return M_ES_8ECX_EDI;

		case 0xd0: return M_ES_8EDX_EAX;
		case 0xd1: return M_ES_8EDX_ECX;
		case 0xd2: return M_ES_9EDX;
		case 0xd3: return M_ES_8EDX_EBX;
		case 0xd4: return M_ES_8EDX_ESP;
		case 0xd5: return M_ES_8EDX_ID;
		case 0xd6: return M_ES_8EDX_ESI;
		case 0xd7: return M_ES_8EDX_EDI;

		case 0xd8: return M_ES_8EBX_EAX;
		case 0xd9: return M_ES_8EBX_ECX;
		case 0xda: return M_ES_8EBX_EDX;
		case 0xdb: return M_ES_9EBX;
		case 0xdc: return M_ES_8EBX_ESP;
		case 0xdd: return M_ES_8EBX_ID;
		case 0xde: return M_ES_8EBX_ESI;
		case 0xdf: return M_ES_8EBX_EDI;

		case 0xe0: return M_ES_EAX;
		case 0xe1: return M_ES_ECX;
		case 0xe2: return M_ES_EDX;
		case 0xe3: return M_ES_EBX;
		case 0xe4: return M_ES_ESP;
		case 0xe5: return M_ES_ID;
		case 0xe6: return M_ES_ESI;
		case 0xe7: return M_ES_EDI;

		case 0xe8: return M_ES_8EBP_EAX;
		case 0xe9: return M_ES_8EBP_ECX;
		case 0xea: return M_ES_8EBP_EDX;
		case 0xeb: return M_ES_8EBP_EBX;
		case 0xec: return M_ES_8EBP_ESP;
		case 0xed: return M_ES_8EBP_ID;
		case 0xee: return M_ES_8EBP_ESI;
		case 0xef: return M_ES_8EBP_EDI;

		case 0xf0: return M_ES_8ESI_EAX;
		case 0xf1: return M_ES_8ESI_ECX;
		case 0xf2: return M_ES_8ESI_EDX;
		case 0xf3: return M_ES_8ESI_EBX;
		case 0xf4: return M_ES_8ESI_ESP;
		case 0xf5: return M_ES_8ESI_ID;
		case 0xf6: return M_ES_9ESI;
		case 0xf7: return M_ES_8ESI_EDI;

		case 0xf8: return M_ES_8EDI_EAX;
		case 0xf9: return M_ES_8EDI_ECX;
		case 0xfa: return M_ES_8EDI_EDX;
		case 0xfb: return M_ES_8EDI_EBX;
		case 0xfc: return M_ES_8EDI_ESP;
		case 0xfd: return M_ES_8EDI_ID;
		case 0xfe: return M_ES_8EDI_ESI;
		case 0xff: return M_ES_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_ES_2EAX_IB;
		case 0x01: return M_ES_EAX_ECX_IB;
		case 0x02: return M_ES_EAX_EDX_IB;
		case 0x03: return M_ES_EAX_EBX_IB;
		case 0x04: return M_ES_EAX_ESP_IB;
		case 0x05: return M_ES_EAX_EBP_IB;
		case 0x06: return M_ES_EAX_ESI_IB;
		case 0x07: return M_ES_EAX_EDI_IB;

		case 0x08: return M_ES_EAX_ECX_IB;
		case 0x09: return M_ES_2ECX_IB;
		case 0x0a: return M_ES_ECX_EDX_IB;
		case 0x0b: return M_ES_ECX_EBX_IB;
		case 0x0c: return M_ES_ECX_ESP_IB;
		case 0x0d: return M_ES_ECX_EBP_IB;
		case 0x0e: return M_ES_ECX_ESI_IB;
		case 0x0f: return M_ES_ECX_EDI_IB;

		case 0x10: return M_ES_EAX_EDX_IB;
		case 0x11: return M_ES_ECX_EDX_IB;
		case 0x12: return M_ES_2EDX_IB;
		case 0x13: return M_ES_EDX_EBX_IB;
		case 0x14: return M_ES_EDX_ESP_IB;
		case 0x15: return M_ES_EDX_EBP_IB;
		case 0x16: return M_ES_EDX_ESI_IB;
		case 0x17: return M_ES_EDX_EDI_IB;

		case 0x18: return M_ES_EAX_EBX_IB;
		case 0x19: return M_ES_ECX_EBX_IB;
		case 0x1a: return M_ES_EDX_EBX_IB;
		case 0x1b: return M_ES_2EBX_IB;
		case 0x1c: return M_ES_EBX_ESP_IB;
		case 0x1d: return M_ES_EBX_EBP_IB;
		case 0x1e: return M_ES_EBX_ESI_IB;
		case 0x1f: return M_ES_EBX_EDI_IB;

		case 0x20: return M_ES_EAX_IB;
		case 0x21: return M_ES_ECX_IB;
		case 0x22: return M_ES_EDX_IB;
		case 0x23: return M_ES_EBX_IB;
		case 0x24: return M_ES_ESP_IB;
		case 0x25: return M_ES_EBP_IB;
		case 0x26: return M_ES_ESI_IB;
		case 0x27: return M_ES_EDI_IB;

		case 0x28: return M_ES_EAX_EBP_IB;
		case 0x29: return M_ES_ECX_EBP_IB;
		case 0x2a: return M_ES_EDX_EBP_IB;
		case 0x2b: return M_ES_EBX_EBP_IB;
		case 0x2c: return M_ES_EBP_ESP_IB;
		case 0x2d: return M_ES_2EBP_IB;
		case 0x2e: return M_ES_EBP_ESI_IB;
		case 0x2f: return M_ES_EBP_EDI_IB;

		case 0x30: return M_ES_EAX_ESI_IB;
		case 0x31: return M_ES_ECX_ESI_IB;
		case 0x32: return M_ES_EDX_ESI_IB;
		case 0x33: return M_ES_EBX_ESI_IB;
		case 0x34: return M_ES_ESI_ESP_IB;
		case 0x35: return M_ES_EBP_ESI_IB;
		case 0x36: return M_ES_2ESI_IB;
		case 0x37: return M_ES_ESI_EDI_IB;

		case 0x38: return M_ES_EAX_EDI_IB;
		case 0x39: return M_ES_ECX_EDI_IB;
		case 0x3a: return M_ES_EDX_EDI_IB;
		case 0x3b: return M_ES_EBX_EDI_IB;
		case 0x3c: return M_ES_EDI_ESP_IB;
		case 0x3d: return M_ES_EBP_EDI_IB;
		case 0x3e: return M_ES_ESI_EDI_IB;
		case 0x3f: return M_ES_2EDI_IB;

		case 0x40: return M_ES_3EAX_IB;
		case 0x41: return M_ES_2EAX_ECX_IB;
		case 0x42: return M_ES_2EAX_EDX_IB;
		case 0x43: return M_ES_2EAX_EBX_IB;
		case 0x44: return M_ES_2EAX_ESP_IB;
		case 0x45: return M_ES_2EAX_EBP_IB;
		case 0x46: return M_ES_2EAX_ESI_IB;
		case 0x47: return M_ES_2EAX_EDI_IB;

		case 0x48: return M_ES_2ECX_EAX_IB;
		case 0x49: return M_ES_3ECX_IB;
		case 0x4a: return M_ES_2ECX_EDX_IB;
		case 0x4b: return M_ES_2ECX_EBX_IB;
		case 0x4c: return M_ES_2ECX_ESP_IB;
		case 0x4d: return M_ES_2ECX_EBP_IB;
		case 0x4e: return M_ES_2ECX_ESI_IB;
		case 0x4f: return M_ES_2ECX_EDI_IB;

		case 0x50: return M_ES_2EDX_EAX_IB;
		case 0x51: return M_ES_2EDX_ECX_IB;
		case 0x52: return M_ES_3EDX_IB;
		case 0x53: return M_ES_2EDX_EBX_IB;
		case 0x54: return M_ES_2EDX_ESP_IB;
		case 0x55: return M_ES_2EDX_EBP_IB;
		case 0x56: return M_ES_2EDX_ESI_IB;
		case 0x57: return M_ES_2EDX_EDI_IB;

		case 0x58: return M_ES_2EBX_EAX_IB;
		case 0x59: return M_ES_2EBX_ECX_IB;
		case 0x5a: return M_ES_2EBX_EDX_IB;
		case 0x5b: return M_ES_3EBX_IB;
		case 0x5c: return M_ES_2EBX_ESP_IB;
		case 0x5d: return M_ES_2EBX_EBP_IB;
		case 0x5e: return M_ES_2EBX_ESI_IB;
		case 0x5f: return M_ES_2EBX_EDI_IB;

		case 0x60: return M_ES_EAX_IB;
		case 0x61: return M_ES_ECX_IB;
		case 0x62: return M_ES_EDX_IB;
		case 0x63: return M_ES_EBX_IB;
		case 0x64: return M_ES_ESP_IB;
		case 0x65: return M_ES_EBP_IB;
		case 0x66: return M_ES_ESI_IB;
		case 0x67: return M_ES_EDI_IB;

		case 0x68: return M_ES_2EBP_EAX_IB;
		case 0x69: return M_ES_2EBP_ECX_IB;
		case 0x6a: return M_ES_2EBP_EDX_IB;
		case 0x6b: return M_ES_2EBP_EBX_IB;
		case 0x6c: return M_ES_2EBP_ESP_IB;
		case 0x6d: return M_ES_3EBP_IB;
		case 0x6e: return M_ES_2EBP_ESI_IB;
		case 0x6f: return M_ES_2EBP_EDI_IB;

		case 0x70: return M_ES_2ESI_EAX_IB;
		case 0x71: return M_ES_2ESI_ECX_IB;
		case 0x72: return M_ES_2ESI_EDX_IB;
		case 0x73: return M_ES_2ESI_EBX_IB;
		case 0x74: return M_ES_2ESI_ESP_IB;
		case 0x75: return M_ES_2ESI_EBP_IB;
		case 0x76: return M_ES_3ESI_IB;
		case 0x77: return M_ES_2ESI_EDI_IB;

		case 0x78: return M_ES_2EDI_EAX_IB;
		case 0x79: return M_ES_2EDI_ECX_IB;
		case 0x7a: return M_ES_2EDI_EDX_IB;
		case 0x7b: return M_ES_2EDI_EBX_IB;
		case 0x7c: return M_ES_2EDI_ESP_IB;
		case 0x7d: return M_ES_2EDI_EBP_IB;
		case 0x7e: return M_ES_2EDI_ESI_IB;
		case 0x7f: return M_ES_3EDI_IB;

		case 0x80: return M_ES_5EAX_IB;
		case 0x81: return M_ES_4EAX_ECX_IB;
		case 0x82: return M_ES_4EAX_EDX_IB;
		case 0x83: return M_ES_4EAX_EBX_IB;
		case 0x84: return M_ES_4EAX_ESP_IB;
		case 0x85: return M_ES_4EAX_EBP_IB;
		case 0x86: return M_ES_4EAX_ESI_IB;
		case 0x87: return M_ES_4EAX_EDI_IB;

		case 0x88: return M_ES_4ECX_EAX_IB;
		case 0x89: return M_ES_5ECX_IB;
		case 0x8a: return M_ES_4ECX_EDX_IB;
		case 0x8b: return M_ES_4ECX_EBX_IB;
		case 0x8c: return M_ES_4ECX_ESP_IB;
		case 0x8d: return M_ES_4ECX_EBP_IB;
		case 0x8e: return M_ES_4ECX_ESI_IB;
		case 0x8f: return M_ES_4ECX_EDI_IB;

		case 0x90: return M_ES_4EDX_EAX_IB;
		case 0x91: return M_ES_4EDX_ECX_IB;
		case 0x92: return M_ES_5EDX_IB;
		case 0x93: return M_ES_4EDX_EBX_IB;
		case 0x94: return M_ES_4EDX_ESP_IB;
		case 0x95: return M_ES_4EDX_EBP_IB;
		case 0x96: return M_ES_4EDX_ESI_IB;
		case 0x97: return M_ES_4EDX_EDI_IB;

		case 0x98: return M_ES_4EBX_EAX_IB;
		case 0x99: return M_ES_4EBX_ECX_IB;
		case 0x9a: return M_ES_4EBX_EDX_IB;
		case 0x9b: return M_ES_5EBX_IB;
		case 0x9c: return M_ES_4EBX_ESP_IB;
		case 0x9d: return M_ES_4EBX_EBP_IB;
		case 0x9e: return M_ES_4EBX_ESI_IB;
		case 0x9f: return M_ES_4EBX_EDI_IB;

		case 0xa0: return M_ES_EAX_IB;
		case 0xa1: return M_ES_ECX_IB;
		case 0xa2: return M_ES_EDX_IB;
		case 0xa3: return M_ES_EBX_IB;
		case 0xa4: return M_ES_ESP_IB;
		case 0xa5: return M_ES_EBP_IB;
		case 0xa6: return M_ES_ESI_IB;
		case 0xa7: return M_ES_EDI_IB;

		case 0xa8: return M_ES_4EBP_EAX_IB;
		case 0xa9: return M_ES_4EBP_ECX_IB;
		case 0xaa: return M_ES_4EBP_EDX_IB;
		case 0xab: return M_ES_4EBP_EBX_IB;
		case 0xac: return M_ES_4EBP_ESP_IB;
		case 0xad: return M_ES_5EBP_IB;
		case 0xae: return M_ES_4EBP_ESI_IB;
		case 0xaf: return M_ES_4EBP_EDI_IB;

		case 0xb0: return M_ES_4ESI_EAX_IB;
		case 0xb1: return M_ES_4ESI_ECX_IB;
		case 0xb2: return M_ES_4ESI_EDX_IB;
		case 0xb3: return M_ES_4ESI_EBX_IB;
		case 0xb4: return M_ES_4ESI_ESP_IB;
		case 0xb5: return M_ES_4ESI_EBP_IB;
		case 0xb6: return M_ES_5ESI_IB;
		case 0xb7: return M_ES_4ESI_EDI_IB;

		case 0xb8: return M_ES_4EDI_EAX_IB;
		case 0xb9: return M_ES_4EDI_ECX_IB;
		case 0xba: return M_ES_4EDI_EDX_IB;
		case 0xbb: return M_ES_4EDI_EBX_IB;
		case 0xbc: return M_ES_4EDI_ESP_IB;
		case 0xbd: return M_ES_4EDI_EBP_IB;
		case 0xbe: return M_ES_4EDI_ESI_IB;
		case 0xbf: return M_ES_5EDI_IB;

		case 0xc0: return M_ES_9EAX_IB;
		case 0xc1: return M_ES_8EAX_ECX_IB;
		case 0xc2: return M_ES_8EAX_EDX_IB;
		case 0xc3: return M_ES_8EAX_EBX_IB;
		case 0xc4: return M_ES_8EAX_ESP_IB;
		case 0xc5: return M_ES_8EAX_EBP_IB;
		case 0xc6: return M_ES_8EAX_ESI_IB;
		case 0xc7: return M_ES_8EAX_EDI_IB;

		case 0xc8: return M_ES_8ECX_EAX_IB;
		case 0xc9: return M_ES_9ECX_IB;
		case 0xca: return M_ES_8ECX_EDX_IB;
		case 0xcb: return M_ES_8ECX_EBX_IB;
		case 0xcc: return M_ES_8ECX_ESP_IB;
		case 0xcd: return M_ES_8ECX_EBP_IB;
		case 0xce: return M_ES_8ECX_ESI_IB;
		case 0xcf: return M_ES_8ECX_EDI_IB;

		case 0xd0: return M_ES_8EDX_EAX_IB;
		case 0xd1: return M_ES_8EDX_ECX_IB;
		case 0xd2: return M_ES_9EDX_IB;
		case 0xd3: return M_ES_8EDX_EBX_IB;
		case 0xd4: return M_ES_8EDX_ESP_IB;
		case 0xd5: return M_ES_8EDX_EBP_IB;
		case 0xd6: return M_ES_8EDX_ESI_IB;
		case 0xd7: return M_ES_8EDX_EDI_IB;

		case 0xd8: return M_ES_8EBX_EAX_IB;
		case 0xd9: return M_ES_8EBX_ECX_IB;
		case 0xda: return M_ES_8EBX_EDX_IB;
		case 0xdb: return M_ES_9EBX_IB;
		case 0xdc: return M_ES_8EBX_ESP_IB;
		case 0xdd: return M_ES_8EBX_EBP_IB;
		case 0xde: return M_ES_8EBX_ESI_IB;
		case 0xdf: return M_ES_8EBX_EDI_IB;

		case 0xe0: return M_ES_EAX_IB;
		case 0xe1: return M_ES_ECX_IB;
		case 0xe2: return M_ES_EDX_IB;
		case 0xe3: return M_ES_EBX_IB;
		case 0xe4: return M_ES_ESP_IB;
		case 0xe5: return M_ES_EBP_IB;
		case 0xe6: return M_ES_ESI_IB;
		case 0xe7: return M_ES_EDI_IB;

		case 0xe8: return M_ES_8EBP_EAX_IB;
		case 0xe9: return M_ES_8EBP_ECX_IB;
		case 0xea: return M_ES_8EBP_EDX_IB;
		case 0xeb: return M_ES_8EBP_EBX_IB;
		case 0xec: return M_ES_8EBP_ESP_IB;
		case 0xed: return M_ES_9EBP_IB;
		case 0xee: return M_ES_8EBP_ESI_IB;
		case 0xef: return M_ES_8EBP_EDI_IB;

		case 0xf0: return M_ES_8ESI_EAX_IB;
		case 0xf1: return M_ES_8ESI_ECX_IB;
		case 0xf2: return M_ES_8ESI_EDX_IB;
		case 0xf3: return M_ES_8ESI_EBX_IB;
		case 0xf4: return M_ES_8ESI_ESP_IB;
		case 0xf5: return M_ES_8ESI_EBP_IB;
		case 0xf6: return M_ES_9ESI_IB;
		case 0xf7: return M_ES_8ESI_EDI_IB;

		case 0xf8: return M_ES_8EDI_EAX_IB;
		case 0xf9: return M_ES_8EDI_ECX_IB;
		case 0xfa: return M_ES_8EDI_EDX_IB;
		case 0xfb: return M_ES_8EDI_EBX_IB;
		case 0xfc: return M_ES_8EDI_ESP_IB;
		case 0xfd: return M_ES_8EDI_EBP_IB;
		case 0xfe: return M_ES_8EDI_ESI_IB;
		case 0xff: return M_ES_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_ES_2EAX_ID;
		case 0x01: return M_ES_EAX_ECX_ID;
		case 0x02: return M_ES_EAX_EDX_ID;
		case 0x03: return M_ES_EAX_EBX_ID;
		case 0x04: return M_ES_EAX_ESP_ID;
		case 0x05: return M_ES_EAX_EBP_ID;
		case 0x06: return M_ES_EAX_ESI_ID;
		case 0x07: return M_ES_EAX_EDI_ID;

		case 0x08: return M_ES_EAX_ECX_ID;
		case 0x09: return M_ES_2ECX_ID;
		case 0x0a: return M_ES_ECX_EDX_ID;
		case 0x0b: return M_ES_ECX_EBX_ID;
		case 0x0c: return M_ES_ECX_ESP_ID;
		case 0x0d: return M_ES_ECX_EBP_ID;
		case 0x0e: return M_ES_ECX_ESI_ID;
		case 0x0f: return M_ES_ECX_EDI_ID;

		case 0x10: return M_ES_EAX_EDX_ID;
		case 0x11: return M_ES_ECX_EDX_ID;
		case 0x12: return M_ES_2EDX_ID;
		case 0x13: return M_ES_EDX_EBX_ID;
		case 0x14: return M_ES_EDX_ESP_ID;
		case 0x15: return M_ES_EDX_EBP_ID;
		case 0x16: return M_ES_EDX_ESI_ID;
		case 0x17: return M_ES_EDX_EDI_ID;

		case 0x18: return M_ES_EAX_EBX_ID;
		case 0x19: return M_ES_ECX_EBX_ID;
		case 0x1a: return M_ES_EDX_EBX_ID;
		case 0x1b: return M_ES_2EBX_ID;
		case 0x1c: return M_ES_EBX_ESP_ID;
		case 0x1d: return M_ES_EBX_EBP_ID;
		case 0x1e: return M_ES_EBX_ESI_ID;
		case 0x1f: return M_ES_EBX_EDI_ID;

		case 0x20: return M_ES_EAX_ID;
		case 0x21: return M_ES_ECX_ID;
		case 0x22: return M_ES_EDX_ID;
		case 0x23: return M_ES_EBX_ID;
		case 0x24: return M_ES_ESP_ID;
		case 0x25: return M_ES_EBP_ID;
		case 0x26: return M_ES_ESI_ID;
		case 0x27: return M_ES_EDI_ID;

		case 0x28: return M_ES_EAX_EBP_ID;
		case 0x29: return M_ES_ECX_EBP_ID;
		case 0x2a: return M_ES_EDX_EBP_ID;
		case 0x2b: return M_ES_EBX_EBP_ID;
		case 0x2c: return M_ES_EBP_ESP_ID;
		case 0x2d: return M_ES_2EBP_ID;
		case 0x2e: return M_ES_EBP_ESI_ID;
		case 0x2f: return M_ES_EBP_EDI_ID;

		case 0x30: return M_ES_EAX_ESI_ID;
		case 0x31: return M_ES_ECX_ESI_ID;
		case 0x32: return M_ES_EDX_ESI_ID;
		case 0x33: return M_ES_EBX_ESI_ID;
		case 0x34: return M_ES_ESI_ESP_ID;
		case 0x35: return M_ES_EBP_ESI_ID;
		case 0x36: return M_ES_2ESI_ID;
		case 0x37: return M_ES_ESI_EDI_ID;

		case 0x38: return M_ES_EAX_EDI_ID;
		case 0x39: return M_ES_ECX_EDI_ID;
		case 0x3a: return M_ES_EDX_EDI_ID;
		case 0x3b: return M_ES_EBX_EDI_ID;
		case 0x3c: return M_ES_EDI_ESP_ID;
		case 0x3d: return M_ES_EBP_EDI_ID;
		case 0x3e: return M_ES_ESI_EDI_ID;
		case 0x3f: return M_ES_2EDI_ID;

		case 0x40: return M_ES_3EAX_ID;
		case 0x41: return M_ES_2EAX_ECX_ID;
		case 0x42: return M_ES_2EAX_EDX_ID;
		case 0x43: return M_ES_2EAX_EBX_ID;
		case 0x44: return M_ES_2EAX_ESP_ID;
		case 0x45: return M_ES_2EAX_EBP_ID;
		case 0x46: return M_ES_2EAX_ESI_ID;
		case 0x47: return M_ES_2EAX_EDI_ID;

		case 0x48: return M_ES_2ECX_EAX_ID;
		case 0x49: return M_ES_3ECX_ID;
		case 0x4a: return M_ES_2ECX_EDX_ID;
		case 0x4b: return M_ES_2ECX_EBX_ID;
		case 0x4c: return M_ES_2ECX_ESP_ID;
		case 0x4d: return M_ES_2ECX_EBP_ID;
		case 0x4e: return M_ES_2ECX_ESI_ID;
		case 0x4f: return M_ES_2ECX_EDI_ID;

		case 0x50: return M_ES_2EDX_EAX_ID;
		case 0x51: return M_ES_2EDX_ECX_ID;
		case 0x52: return M_ES_3EDX_ID;
		case 0x53: return M_ES_2EDX_EBX_ID;
		case 0x54: return M_ES_2EDX_ESP_ID;
		case 0x55: return M_ES_2EDX_EBP_ID;
		case 0x56: return M_ES_2EDX_ESI_ID;
		case 0x57: return M_ES_2EDX_EDI_ID;

		case 0x58: return M_ES_2EBX_EAX_ID;
		case 0x59: return M_ES_2EBX_ECX_ID;
		case 0x5a: return M_ES_2EBX_EDX_ID;
		case 0x5b: return M_ES_3EBX_ID;
		case 0x5c: return M_ES_2EBX_ESP_ID;
		case 0x5d: return M_ES_2EBX_EBP_ID;
		case 0x5e: return M_ES_2EBX_ESI_ID;
		case 0x5f: return M_ES_2EBX_EDI_ID;

		case 0x60: return M_ES_EAX_ID;
		case 0x61: return M_ES_ECX_ID;
		case 0x62: return M_ES_EDX_ID;
		case 0x63: return M_ES_EBX_ID;
		case 0x64: return M_ES_ESP_ID;
		case 0x65: return M_ES_EBP_ID;
		case 0x66: return M_ES_ESI_ID;
		case 0x67: return M_ES_EDI_ID;

		case 0x68: return M_ES_2EBP_EAX_ID;
		case 0x69: return M_ES_2EBP_ECX_ID;
		case 0x6a: return M_ES_2EBP_EDX_ID;
		case 0x6b: return M_ES_2EBP_EBX_ID;
		case 0x6c: return M_ES_2EBP_ESP_ID;
		case 0x6d: return M_ES_3EBP_ID;
		case 0x6e: return M_ES_2EBP_ESI_ID;
		case 0x6f: return M_ES_2EBP_EDI_ID;

		case 0x70: return M_ES_2ESI_EAX_ID;
		case 0x71: return M_ES_2ESI_ECX_ID;
		case 0x72: return M_ES_2ESI_EDX_ID;
		case 0x73: return M_ES_2ESI_EBX_ID;
		case 0x74: return M_ES_2ESI_ESP_ID;
		case 0x75: return M_ES_2ESI_EBP_ID;
		case 0x76: return M_ES_3ESI_ID;
		case 0x77: return M_ES_2ESI_EDI_ID;

		case 0x78: return M_ES_2EDI_EAX_ID;
		case 0x79: return M_ES_2EDI_ECX_ID;
		case 0x7a: return M_ES_2EDI_EDX_ID;
		case 0x7b: return M_ES_2EDI_EBX_ID;
		case 0x7c: return M_ES_2EDI_ESP_ID;
		case 0x7d: return M_ES_2EDI_EBP_ID;
		case 0x7e: return M_ES_2EDI_ESI_ID;
		case 0x7f: return M_ES_3EDI_ID;

		case 0x80: return M_ES_5EAX_ID;
		case 0x81: return M_ES_4EAX_ECX_ID;
		case 0x82: return M_ES_4EAX_EDX_ID;
		case 0x83: return M_ES_4EAX_EBX_ID;
		case 0x84: return M_ES_4EAX_ESP_ID;
		case 0x85: return M_ES_4EAX_EBP_ID;
		case 0x86: return M_ES_4EAX_ESI_ID;
		case 0x87: return M_ES_4EAX_EDI_ID;

		case 0x88: return M_ES_4ECX_EAX_ID;
		case 0x89: return M_ES_5ECX_ID;
		case 0x8a: return M_ES_4ECX_EDX_ID;
		case 0x8b: return M_ES_4ECX_EBX_ID;
		case 0x8c: return M_ES_4ECX_ESP_ID;
		case 0x8d: return M_ES_4ECX_EBP_ID;
		case 0x8e: return M_ES_4ECX_ESI_ID;
		case 0x8f: return M_ES_4ECX_EDI_ID;

		case 0x90: return M_ES_4EDX_EAX_ID;
		case 0x91: return M_ES_4EDX_ECX_ID;
		case 0x92: return M_ES_5EDX_ID;
		case 0x93: return M_ES_4EDX_EBX_ID;
		case 0x94: return M_ES_4EDX_ESP_ID;
		case 0x95: return M_ES_4EDX_EBP_ID;
		case 0x96: return M_ES_4EDX_ESI_ID;
		case 0x97: return M_ES_4EDX_EDI_ID;

		case 0x98: return M_ES_4EBX_EAX_ID;
		case 0x99: return M_ES_4EBX_ECX_ID;
		case 0x9a: return M_ES_4EBX_EDX_ID;
		case 0x9b: return M_ES_5EBX_ID;
		case 0x9c: return M_ES_4EBX_ESP_ID;
		case 0x9d: return M_ES_4EBX_EBP_ID;
		case 0x9e: return M_ES_4EBX_ESI_ID;
		case 0x9f: return M_ES_4EBX_EDI_ID;

		case 0xa0: return M_ES_EAX_ID;
		case 0xa1: return M_ES_ECX_ID;
		case 0xa2: return M_ES_EDX_ID;
		case 0xa3: return M_ES_EBX_ID;
		case 0xa4: return M_ES_ESP_ID;
		case 0xa5: return M_ES_EBP_ID;
		case 0xa6: return M_ES_ESI_ID;
		case 0xa7: return M_ES_EDI_ID;

		case 0xa8: return M_ES_4EBP_EAX_ID;
		case 0xa9: return M_ES_4EBP_ECX_ID;
		case 0xaa: return M_ES_4EBP_EDX_ID;
		case 0xab: return M_ES_4EBP_EBX_ID;
		case 0xac: return M_ES_4EBP_ESP_ID;
		case 0xad: return M_ES_5EBP_ID;
		case 0xae: return M_ES_4EBP_ESI_ID;
		case 0xaf: return M_ES_4EBP_EDI_ID;

		case 0xb0: return M_ES_4ESI_EAX_ID;
		case 0xb1: return M_ES_4ESI_ECX_ID;
		case 0xb2: return M_ES_4ESI_EDX_ID;
		case 0xb3: return M_ES_4ESI_EBX_ID;
		case 0xb4: return M_ES_4ESI_ESP_ID;
		case 0xb5: return M_ES_4ESI_EBP_ID;
		case 0xb6: return M_ES_5ESI_ID;
		case 0xb7: return M_ES_4ESI_EDI_ID;

		case 0xb8: return M_ES_4EDI_EAX_ID;
		case 0xb9: return M_ES_4EDI_ECX_ID;
		case 0xba: return M_ES_4EDI_EDX_ID;
		case 0xbb: return M_ES_4EDI_EBX_ID;
		case 0xbc: return M_ES_4EDI_ESP_ID;
		case 0xbd: return M_ES_4EDI_EBP_ID;
		case 0xbe: return M_ES_4EDI_ESI_ID;
		case 0xbf: return M_ES_5EDI_ID;

		case 0xc0: return M_ES_9EAX_ID;
		case 0xc1: return M_ES_8EAX_ECX_ID;
		case 0xc2: return M_ES_8EAX_EDX_ID;
		case 0xc3: return M_ES_8EAX_EBX_ID;
		case 0xc4: return M_ES_8EAX_ESP_ID;
		case 0xc5: return M_ES_8EAX_EBP_ID;
		case 0xc6: return M_ES_8EAX_ESI_ID;
		case 0xc7: return M_ES_8EAX_EDI_ID;

		case 0xc8: return M_ES_8ECX_EAX_ID;
		case 0xc9: return M_ES_9ECX_ID;
		case 0xca: return M_ES_8ECX_EDX_ID;
		case 0xcb: return M_ES_8ECX_EBX_ID;
		case 0xcc: return M_ES_8ECX_ESP_ID;
		case 0xcd: return M_ES_8ECX_EBP_ID;
		case 0xce: return M_ES_8ECX_ESI_ID;
		case 0xcf: return M_ES_8ECX_EDI_ID;

		case 0xd0: return M_ES_8EDX_EAX_ID;
		case 0xd1: return M_ES_8EDX_ECX_ID;
		case 0xd2: return M_ES_9EDX_ID;
		case 0xd3: return M_ES_8EDX_EBX_ID;
		case 0xd4: return M_ES_8EDX_ESP_ID;
		case 0xd5: return M_ES_8EDX_EBP_ID;
		case 0xd6: return M_ES_8EDX_ESI_ID;
		case 0xd7: return M_ES_8EDX_EDI_ID;

		case 0xd8: return M_ES_8EBX_EAX_ID;
		case 0xd9: return M_ES_8EBX_ECX_ID;
		case 0xda: return M_ES_8EBX_EDX_ID;
		case 0xdb: return M_ES_9EBX_ID;
		case 0xdc: return M_ES_8EBX_ESP_ID;
		case 0xdd: return M_ES_8EBX_EBP_ID;
		case 0xde: return M_ES_8EBX_ESI_ID;
		case 0xdf: return M_ES_8EBX_EDI_ID;

		case 0xe0: return M_ES_EAX_ID;
		case 0xe1: return M_ES_ECX_ID;
		case 0xe2: return M_ES_EDX_ID;
		case 0xe3: return M_ES_EBX_ID;
		case 0xe4: return M_ES_ESP_ID;
		case 0xe5: return M_ES_EBP_ID;
		case 0xe6: return M_ES_ESI_ID;
		case 0xe7: return M_ES_EDI_ID;

		case 0xe8: return M_ES_8EBP_EAX_ID;
		case 0xe9: return M_ES_8EBP_ECX_ID;
		case 0xea: return M_ES_8EBP_EDX_ID;
		case 0xeb: return M_ES_8EBP_EBX_ID;
		case 0xec: return M_ES_8EBP_ESP_ID;
		case 0xed: return M_ES_9EBP_ID;
		case 0xee: return M_ES_8EBP_ESI_ID;
		case 0xef: return M_ES_8EBP_EDI_ID;

		case 0xf0: return M_ES_8ESI_EAX_ID;
		case 0xf1: return M_ES_8ESI_ECX_ID;
		case 0xf2: return M_ES_8ESI_EDX_ID;
		case 0xf3: return M_ES_8ESI_EBX_ID;
		case 0xf4: return M_ES_8ESI_ESP_ID;
		case 0xf5: return M_ES_8ESI_EBP_ID;
		case 0xf6: return M_ES_9ESI_ID;
		case 0xf7: return M_ES_8ESI_EDI_ID;

		case 0xf8: return M_ES_8EDI_EAX_ID;
		case 0xf9: return M_ES_8EDI_ECX_ID;
		case 0xfa: return M_ES_8EDI_EDX_ID;
		case 0xfb: return M_ES_8EDI_EBX_ID;
		case 0xfc: return M_ES_8EDI_ESP_ID;
		case 0xfd: return M_ES_8EDI_EBP_ID;
		case 0xfe: return M_ES_8EDI_ESI_ID;
		case 0xff: return M_ES_9EDI_ID;
		}
	    }
	case PREFICES_CS:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_CS_2EAX;
		case 0x01: return M_CS_EAX_ECX;
		case 0x02: return M_CS_EAX_EDX;
		case 0x03: return M_CS_EAX_EBX;
		case 0x04: return M_CS_EAX_ESP;
		case 0x05: return M_CS_EAX_ID;
		case 0x06: return M_CS_EAX_ESI;
		case 0x07: return M_CS_EAX_EDI;

		case 0x08: return M_CS_EAX_ECX;
		case 0x09: return M_CS_2ECX;
		case 0x0a: return M_CS_ECX_EDX;
		case 0x0b: return M_CS_ECX_EBX;
		case 0x0c: return M_CS_ECX_ESP;
		case 0x0d: return M_CS_ECX_ID;
		case 0x0e: return M_CS_ECX_ESI;
		case 0x0f: return M_CS_ECX_EDI;

		case 0x10: return M_CS_EAX_EDX;
		case 0x11: return M_CS_ECX_EDX;
		case 0x12: return M_CS_2EDX;
		case 0x13: return M_CS_EDX_EBX;
		case 0x14: return M_CS_EDX_ESP;
		case 0x15: return M_CS_EDX_ID;
		case 0x16: return M_CS_EDX_ESI;
		case 0x17: return M_CS_EDX_EDI;

		case 0x18: return M_CS_EAX_EBX;
		case 0x19: return M_CS_ECX_EBX;
		case 0x1a: return M_CS_EDX_EBX;
		case 0x1b: return M_CS_2EBX;
		case 0x1c: return M_CS_EBX_ESP;
		case 0x1d: return M_CS_EBX_ID;
		case 0x1e: return M_CS_EBX_ESI;
		case 0x1f: return M_CS_EBX_EDI;

		case 0x20: return M_CS_EAX;
		case 0x21: return M_CS_ECX;
		case 0x22: return M_CS_EDX;
		case 0x23: return M_CS_EBX;
		case 0x24: return M_CS_ESP;
		case 0x25: return M_CS_ID;
		case 0x26: return M_CS_ESI;
		case 0x27: return M_CS_EDI;

		case 0x28: return M_CS_EAX_EBP;
		case 0x29: return M_CS_ECX_EBP;
		case 0x2a: return M_CS_EDX_EBP;
		case 0x2b: return M_CS_EBX_EBP;
		case 0x2c: return M_CS_EBP_ESP;
		case 0x2d: return M_CS_EBP_ID;
		case 0x2e: return M_CS_EBP_ESI;
		case 0x2f: return M_CS_EBP_EDI;

		case 0x30: return M_CS_EAX_ESI;
		case 0x31: return M_CS_ECX_ESI;
		case 0x32: return M_CS_EDX_ESI;
		case 0x33: return M_CS_EBX_ESI;
		case 0x34: return M_CS_ESI_ESP;
		case 0x35: return M_CS_ESI_ID;
		case 0x36: return M_CS_2ESI;
		case 0x37: return M_CS_ESI_EDI;

		case 0x38: return M_CS_EAX_EDI;
		case 0x39: return M_CS_ECX_EDI;
		case 0x3a: return M_CS_EDX_EDI;
		case 0x3b: return M_CS_EBX_EDI;
		case 0x3c: return M_CS_EDI_ESP;
		case 0x3d: return M_CS_EDI_ID;
		case 0x3e: return M_CS_ESI_EDI;
		case 0x3f: return M_CS_2EDI;

		case 0x40: return M_CS_3EAX;
		case 0x41: return M_CS_2EAX_ECX;
		case 0x42: return M_CS_2EAX_EDX;
		case 0x43: return M_CS_2EAX_EBX;
		case 0x44: return M_CS_2EAX_ESP;
		case 0x45: return M_CS_2EAX_ID;
		case 0x46: return M_CS_2EAX_ESI;
		case 0x47: return M_CS_2EAX_EDI;

		case 0x48: return M_CS_2ECX_EAX;
		case 0x49: return M_CS_3ECX;
		case 0x4a: return M_CS_2ECX_EDX;
		case 0x4b: return M_CS_2ECX_EBX;
		case 0x4c: return M_CS_2ECX_ESP;
		case 0x4d: return M_CS_2ECX_ID;
		case 0x4e: return M_CS_2ECX_ESI;
		case 0x4f: return M_CS_2ECX_EDI;

		case 0x50: return M_CS_2EDX_EAX;
		case 0x51: return M_CS_2EDX_ECX;
		case 0x52: return M_CS_3EDX;
		case 0x53: return M_CS_2EDX_EBX;
		case 0x54: return M_CS_2EDX_ESP;
		case 0x55: return M_CS_2EDX_ID;
		case 0x56: return M_CS_2EDX_ESI;
		case 0x57: return M_CS_2EDX_EDI;

		case 0x58: return M_CS_2EBX_EAX;
		case 0x59: return M_CS_2EBX_ECX;
		case 0x5a: return M_CS_2EBX_EDX;
		case 0x5b: return M_CS_3EBX;
		case 0x5c: return M_CS_2EBX_ESP;
		case 0x5d: return M_CS_2EBX_ID;
		case 0x5e: return M_CS_2EBX_ESI;
		case 0x5f: return M_CS_2EBX_EDI;

		case 0x60: return M_CS_EAX;
		case 0x61: return M_CS_ECX;
		case 0x62: return M_CS_EDX;
		case 0x63: return M_CS_EBX;
		case 0x64: return M_CS_ESP;
		case 0x65: return M_CS_ID;
		case 0x66: return M_CS_ESI;
		case 0x67: return M_CS_EDI;

		case 0x68: return M_CS_2EBP_EAX;
		case 0x69: return M_CS_2EBP_ECX;
		case 0x6a: return M_CS_2EBP_EDX;
		case 0x6b: return M_CS_2EBP_EBX;
		case 0x6c: return M_CS_2EBP_ESP;
		case 0x6d: return M_CS_2EBP_ID;
		case 0x6e: return M_CS_2EBP_ESI;
		case 0x6f: return M_CS_2EBP_EDI;

		case 0x70: return M_CS_2ESI_EAX;
		case 0x71: return M_CS_2ESI_ECX;
		case 0x72: return M_CS_2ESI_EDX;
		case 0x73: return M_CS_2ESI_EBX;
		case 0x74: return M_CS_2ESI_ESP;
		case 0x75: return M_CS_2ESI_ID;
		case 0x76: return M_CS_3ESI;
		case 0x77: return M_CS_2ESI_EDI;

		case 0x78: return M_CS_2EDI_EAX;
		case 0x79: return M_CS_2EDI_ECX;
		case 0x7a: return M_CS_2EDI_EDX;
		case 0x7b: return M_CS_2EDI_EBX;
		case 0x7c: return M_CS_2EDI_ESP;
		case 0x7d: return M_CS_2EDI_ID;
		case 0x7e: return M_CS_2EDI_ESI;
		case 0x7f: return M_CS_3EDI;

		case 0x80: return M_CS_5EAX;
		case 0x81: return M_CS_4EAX_ECX;
		case 0x82: return M_CS_4EAX_EDX;
		case 0x83: return M_CS_4EAX_EBX;
		case 0x84: return M_CS_4EAX_ESP;
		case 0x85: return M_CS_4EAX_ID;
		case 0x86: return M_CS_4EAX_ESI;
		case 0x87: return M_CS_4EAX_EDI;

		case 0x88: return M_CS_4ECX_EAX;
		case 0x89: return M_CS_5ECX;
		case 0x8a: return M_CS_4ECX_EDX;
		case 0x8b: return M_CS_4ECX_EBX;
		case 0x8c: return M_CS_4ECX_ESP;
		case 0x8d: return M_CS_4ECX_ID;
		case 0x8e: return M_CS_4ECX_ESI;
		case 0x8f: return M_CS_4ECX_EDI;

		case 0x90: return M_CS_4EDX_EAX;
		case 0x91: return M_CS_4EDX_ECX;
		case 0x92: return M_CS_5EDX;
		case 0x93: return M_CS_4EDX_EBX;
		case 0x94: return M_CS_4EDX_ESP;
		case 0x95: return M_CS_4EDX_ID;
		case 0x96: return M_CS_4EDX_ESI;
		case 0x97: return M_CS_4EDX_EDI;

		case 0x98: return M_CS_4EBX_EAX;
		case 0x99: return M_CS_4EBX_ECX;
		case 0x9a: return M_CS_4EBX_EDX;
		case 0x9b: return M_CS_5EBX;
		case 0x9c: return M_CS_4EBX_ESP;
		case 0x9d: return M_CS_4EBX_ID;
		case 0x9e: return M_CS_4EBX_ESI;
		case 0x9f: return M_CS_4EBX_EDI;

		case 0xa0: return M_CS_EAX;
		case 0xa1: return M_CS_ECX;
		case 0xa2: return M_CS_EDX;
		case 0xa3: return M_CS_EBX;
		case 0xa4: return M_CS_ESP;
		case 0xa5: return M_CS_ID;
		case 0xa6: return M_CS_ESI;
		case 0xa7: return M_CS_EDI;

		case 0xa8: return M_CS_4EBP_EAX;
		case 0xa9: return M_CS_4EBP_ECX;
		case 0xaa: return M_CS_4EBP_EDX;
		case 0xab: return M_CS_4EBP_EBX;
		case 0xac: return M_CS_4EBP_ESP;
		case 0xad: return M_CS_4EBP_ID;
		case 0xae: return M_CS_4EBP_ESI;
		case 0xaf: return M_CS_4EBP_EDI;

		case 0xb0: return M_CS_4ESI_EAX;
		case 0xb1: return M_CS_4ESI_ECX;
		case 0xb2: return M_CS_4ESI_EDX;
		case 0xb3: return M_CS_4ESI_EBX;
		case 0xb4: return M_CS_4ESI_ESP;
		case 0xb5: return M_CS_4ESI_ID;
		case 0xb6: return M_CS_5ESI;
		case 0xb7: return M_CS_4ESI_EDI;

		case 0xb8: return M_CS_4EDI_EAX;
		case 0xb9: return M_CS_4EDI_ECX;
		case 0xba: return M_CS_4EDI_EDX;
		case 0xbb: return M_CS_4EDI_EBX;
		case 0xbc: return M_CS_4EDI_ESP;
		case 0xbd: return M_CS_4EDI_ID;
		case 0xbe: return M_CS_4EDI_ESI;
		case 0xbf: return M_CS_5EDI;

		case 0xc0: return M_CS_9EAX;
		case 0xc1: return M_CS_8EAX_ECX;
		case 0xc2: return M_CS_8EAX_EDX;
		case 0xc3: return M_CS_8EAX_EBX;
		case 0xc4: return M_CS_8EAX_ESP;
		case 0xc5: return M_CS_8EAX_ID;
		case 0xc6: return M_CS_8EAX_ESI;
		case 0xc7: return M_CS_8EAX_EDI;

		case 0xc8: return M_CS_8ECX_EAX;
		case 0xc9: return M_CS_9ECX;
		case 0xca: return M_CS_8ECX_EDX;
		case 0xcb: return M_CS_8ECX_EBX;
		case 0xcc: return M_CS_8ECX_ESP;
		case 0xcd: return M_CS_8ECX_ID;
		case 0xce: return M_CS_8ECX_ESI;
		case 0xcf: return M_CS_8ECX_EDI;

		case 0xd0: return M_CS_8EDX_EAX;
		case 0xd1: return M_CS_8EDX_ECX;
		case 0xd2: return M_CS_9EDX;
		case 0xd3: return M_CS_8EDX_EBX;
		case 0xd4: return M_CS_8EDX_ESP;
		case 0xd5: return M_CS_8EDX_ID;
		case 0xd6: return M_CS_8EDX_ESI;
		case 0xd7: return M_CS_8EDX_EDI;

		case 0xd8: return M_CS_8EBX_EAX;
		case 0xd9: return M_CS_8EBX_ECX;
		case 0xda: return M_CS_8EBX_EDX;
		case 0xdb: return M_CS_9EBX;
		case 0xdc: return M_CS_8EBX_ESP;
		case 0xdd: return M_CS_8EBX_ID;
		case 0xde: return M_CS_8EBX_ESI;
		case 0xdf: return M_CS_8EBX_EDI;

		case 0xe0: return M_CS_EAX;
		case 0xe1: return M_CS_ECX;
		case 0xe2: return M_CS_EDX;
		case 0xe3: return M_CS_EBX;
		case 0xe4: return M_CS_ESP;
		case 0xe5: return M_CS_ID;
		case 0xe6: return M_CS_ESI;
		case 0xe7: return M_CS_EDI;

		case 0xe8: return M_CS_8EBP_EAX;
		case 0xe9: return M_CS_8EBP_ECX;
		case 0xea: return M_CS_8EBP_EDX;
		case 0xeb: return M_CS_8EBP_EBX;
		case 0xec: return M_CS_8EBP_ESP;
		case 0xed: return M_CS_8EBP_ID;
		case 0xee: return M_CS_8EBP_ESI;
		case 0xef: return M_CS_8EBP_EDI;

		case 0xf0: return M_CS_8ESI_EAX;
		case 0xf1: return M_CS_8ESI_ECX;
		case 0xf2: return M_CS_8ESI_EDX;
		case 0xf3: return M_CS_8ESI_EBX;
		case 0xf4: return M_CS_8ESI_ESP;
		case 0xf5: return M_CS_8ESI_ID;
		case 0xf6: return M_CS_9ESI;
		case 0xf7: return M_CS_8ESI_EDI;

		case 0xf8: return M_CS_8EDI_EAX;
		case 0xf9: return M_CS_8EDI_ECX;
		case 0xfa: return M_CS_8EDI_EDX;
		case 0xfb: return M_CS_8EDI_EBX;
		case 0xfc: return M_CS_8EDI_ESP;
		case 0xfd: return M_CS_8EDI_ID;
		case 0xfe: return M_CS_8EDI_ESI;
		case 0xff: return M_CS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_CS_2EAX_IB;
		case 0x01: return M_CS_EAX_ECX_IB;
		case 0x02: return M_CS_EAX_EDX_IB;
		case 0x03: return M_CS_EAX_EBX_IB;
		case 0x04: return M_CS_EAX_ESP_IB;
		case 0x05: return M_CS_EAX_EBP_IB;
		case 0x06: return M_CS_EAX_ESI_IB;
		case 0x07: return M_CS_EAX_EDI_IB;

		case 0x08: return M_CS_EAX_ECX_IB;
		case 0x09: return M_CS_2ECX_IB;
		case 0x0a: return M_CS_ECX_EDX_IB;
		case 0x0b: return M_CS_ECX_EBX_IB;
		case 0x0c: return M_CS_ECX_ESP_IB;
		case 0x0d: return M_CS_ECX_EBP_IB;
		case 0x0e: return M_CS_ECX_ESI_IB;
		case 0x0f: return M_CS_ECX_EDI_IB;

		case 0x10: return M_CS_EAX_EDX_IB;
		case 0x11: return M_CS_ECX_EDX_IB;
		case 0x12: return M_CS_2EDX_IB;
		case 0x13: return M_CS_EDX_EBX_IB;
		case 0x14: return M_CS_EDX_ESP_IB;
		case 0x15: return M_CS_EDX_EBP_IB;
		case 0x16: return M_CS_EDX_ESI_IB;
		case 0x17: return M_CS_EDX_EDI_IB;

		case 0x18: return M_CS_EAX_EBX_IB;
		case 0x19: return M_CS_ECX_EBX_IB;
		case 0x1a: return M_CS_EDX_EBX_IB;
		case 0x1b: return M_CS_2EBX_IB;
		case 0x1c: return M_CS_EBX_ESP_IB;
		case 0x1d: return M_CS_EBX_EBP_IB;
		case 0x1e: return M_CS_EBX_ESI_IB;
		case 0x1f: return M_CS_EBX_EDI_IB;

		case 0x20: return M_CS_EAX_IB;
		case 0x21: return M_CS_ECX_IB;
		case 0x22: return M_CS_EDX_IB;
		case 0x23: return M_CS_EBX_IB;
		case 0x24: return M_CS_ESP_IB;
		case 0x25: return M_CS_EBP_IB;
		case 0x26: return M_CS_ESI_IB;
		case 0x27: return M_CS_EDI_IB;

		case 0x28: return M_CS_EAX_EBP_IB;
		case 0x29: return M_CS_ECX_EBP_IB;
		case 0x2a: return M_CS_EDX_EBP_IB;
		case 0x2b: return M_CS_EBX_EBP_IB;
		case 0x2c: return M_CS_EBP_ESP_IB;
		case 0x2d: return M_CS_2EBP_IB;
		case 0x2e: return M_CS_EBP_ESI_IB;
		case 0x2f: return M_CS_EBP_EDI_IB;

		case 0x30: return M_CS_EAX_ESI_IB;
		case 0x31: return M_CS_ECX_ESI_IB;
		case 0x32: return M_CS_EDX_ESI_IB;
		case 0x33: return M_CS_EBX_ESI_IB;
		case 0x34: return M_CS_ESI_ESP_IB;
		case 0x35: return M_CS_EBP_ESI_IB;
		case 0x36: return M_CS_2ESI_IB;
		case 0x37: return M_CS_ESI_EDI_IB;

		case 0x38: return M_CS_EAX_EDI_IB;
		case 0x39: return M_CS_ECX_EDI_IB;
		case 0x3a: return M_CS_EDX_EDI_IB;
		case 0x3b: return M_CS_EBX_EDI_IB;
		case 0x3c: return M_CS_EDI_ESP_IB;
		case 0x3d: return M_CS_EBP_EDI_IB;
		case 0x3e: return M_CS_ESI_EDI_IB;
		case 0x3f: return M_CS_2EDI_IB;

		case 0x40: return M_CS_3EAX_IB;
		case 0x41: return M_CS_2EAX_ECX_IB;
		case 0x42: return M_CS_2EAX_EDX_IB;
		case 0x43: return M_CS_2EAX_EBX_IB;
		case 0x44: return M_CS_2EAX_ESP_IB;
		case 0x45: return M_CS_2EAX_EBP_IB;
		case 0x46: return M_CS_2EAX_ESI_IB;
		case 0x47: return M_CS_2EAX_EDI_IB;

		case 0x48: return M_CS_2ECX_EAX_IB;
		case 0x49: return M_CS_3ECX_IB;
		case 0x4a: return M_CS_2ECX_EDX_IB;
		case 0x4b: return M_CS_2ECX_EBX_IB;
		case 0x4c: return M_CS_2ECX_ESP_IB;
		case 0x4d: return M_CS_2ECX_EBP_IB;
		case 0x4e: return M_CS_2ECX_ESI_IB;
		case 0x4f: return M_CS_2ECX_EDI_IB;

		case 0x50: return M_CS_2EDX_EAX_IB;
		case 0x51: return M_CS_2EDX_ECX_IB;
		case 0x52: return M_CS_3EDX_IB;
		case 0x53: return M_CS_2EDX_EBX_IB;
		case 0x54: return M_CS_2EDX_ESP_IB;
		case 0x55: return M_CS_2EDX_EBP_IB;
		case 0x56: return M_CS_2EDX_ESI_IB;
		case 0x57: return M_CS_2EDX_EDI_IB;

		case 0x58: return M_CS_2EBX_EAX_IB;
		case 0x59: return M_CS_2EBX_ECX_IB;
		case 0x5a: return M_CS_2EBX_EDX_IB;
		case 0x5b: return M_CS_3EBX_IB;
		case 0x5c: return M_CS_2EBX_ESP_IB;
		case 0x5d: return M_CS_2EBX_EBP_IB;
		case 0x5e: return M_CS_2EBX_ESI_IB;
		case 0x5f: return M_CS_2EBX_EDI_IB;

		case 0x60: return M_CS_EAX_IB;
		case 0x61: return M_CS_ECX_IB;
		case 0x62: return M_CS_EDX_IB;
		case 0x63: return M_CS_EBX_IB;
		case 0x64: return M_CS_ESP_IB;
		case 0x65: return M_CS_EBP_IB;
		case 0x66: return M_CS_ESI_IB;
		case 0x67: return M_CS_EDI_IB;

		case 0x68: return M_CS_2EBP_EAX_IB;
		case 0x69: return M_CS_2EBP_ECX_IB;
		case 0x6a: return M_CS_2EBP_EDX_IB;
		case 0x6b: return M_CS_2EBP_EBX_IB;
		case 0x6c: return M_CS_2EBP_ESP_IB;
		case 0x6d: return M_CS_3EBP_IB;
		case 0x6e: return M_CS_2EBP_ESI_IB;
		case 0x6f: return M_CS_2EBP_EDI_IB;

		case 0x70: return M_CS_2ESI_EAX_IB;
		case 0x71: return M_CS_2ESI_ECX_IB;
		case 0x72: return M_CS_2ESI_EDX_IB;
		case 0x73: return M_CS_2ESI_EBX_IB;
		case 0x74: return M_CS_2ESI_ESP_IB;
		case 0x75: return M_CS_2ESI_EBP_IB;
		case 0x76: return M_CS_3ESI_IB;
		case 0x77: return M_CS_2ESI_EDI_IB;

		case 0x78: return M_CS_2EDI_EAX_IB;
		case 0x79: return M_CS_2EDI_ECX_IB;
		case 0x7a: return M_CS_2EDI_EDX_IB;
		case 0x7b: return M_CS_2EDI_EBX_IB;
		case 0x7c: return M_CS_2EDI_ESP_IB;
		case 0x7d: return M_CS_2EDI_EBP_IB;
		case 0x7e: return M_CS_2EDI_ESI_IB;
		case 0x7f: return M_CS_3EDI_IB;

		case 0x80: return M_CS_5EAX_IB;
		case 0x81: return M_CS_4EAX_ECX_IB;
		case 0x82: return M_CS_4EAX_EDX_IB;
		case 0x83: return M_CS_4EAX_EBX_IB;
		case 0x84: return M_CS_4EAX_ESP_IB;
		case 0x85: return M_CS_4EAX_EBP_IB;
		case 0x86: return M_CS_4EAX_ESI_IB;
		case 0x87: return M_CS_4EAX_EDI_IB;

		case 0x88: return M_CS_4ECX_EAX_IB;
		case 0x89: return M_CS_5ECX_IB;
		case 0x8a: return M_CS_4ECX_EDX_IB;
		case 0x8b: return M_CS_4ECX_EBX_IB;
		case 0x8c: return M_CS_4ECX_ESP_IB;
		case 0x8d: return M_CS_4ECX_EBP_IB;
		case 0x8e: return M_CS_4ECX_ESI_IB;
		case 0x8f: return M_CS_4ECX_EDI_IB;

		case 0x90: return M_CS_4EDX_EAX_IB;
		case 0x91: return M_CS_4EDX_ECX_IB;
		case 0x92: return M_CS_5EDX_IB;
		case 0x93: return M_CS_4EDX_EBX_IB;
		case 0x94: return M_CS_4EDX_ESP_IB;
		case 0x95: return M_CS_4EDX_EBP_IB;
		case 0x96: return M_CS_4EDX_ESI_IB;
		case 0x97: return M_CS_4EDX_EDI_IB;

		case 0x98: return M_CS_4EBX_EAX_IB;
		case 0x99: return M_CS_4EBX_ECX_IB;
		case 0x9a: return M_CS_4EBX_EDX_IB;
		case 0x9b: return M_CS_5EBX_IB;
		case 0x9c: return M_CS_4EBX_ESP_IB;
		case 0x9d: return M_CS_4EBX_EBP_IB;
		case 0x9e: return M_CS_4EBX_ESI_IB;
		case 0x9f: return M_CS_4EBX_EDI_IB;

		case 0xa0: return M_CS_EAX_IB;
		case 0xa1: return M_CS_ECX_IB;
		case 0xa2: return M_CS_EDX_IB;
		case 0xa3: return M_CS_EBX_IB;
		case 0xa4: return M_CS_ESP_IB;
		case 0xa5: return M_CS_EBP_IB;
		case 0xa6: return M_CS_ESI_IB;
		case 0xa7: return M_CS_EDI_IB;

		case 0xa8: return M_CS_4EBP_EAX_IB;
		case 0xa9: return M_CS_4EBP_ECX_IB;
		case 0xaa: return M_CS_4EBP_EDX_IB;
		case 0xab: return M_CS_4EBP_EBX_IB;
		case 0xac: return M_CS_4EBP_ESP_IB;
		case 0xad: return M_CS_5EBP_IB;
		case 0xae: return M_CS_4EBP_ESI_IB;
		case 0xaf: return M_CS_4EBP_EDI_IB;

		case 0xb0: return M_CS_4ESI_EAX_IB;
		case 0xb1: return M_CS_4ESI_ECX_IB;
		case 0xb2: return M_CS_4ESI_EDX_IB;
		case 0xb3: return M_CS_4ESI_EBX_IB;
		case 0xb4: return M_CS_4ESI_ESP_IB;
		case 0xb5: return M_CS_4ESI_EBP_IB;
		case 0xb6: return M_CS_5ESI_IB;
		case 0xb7: return M_CS_4ESI_EDI_IB;

		case 0xb8: return M_CS_4EDI_EAX_IB;
		case 0xb9: return M_CS_4EDI_ECX_IB;
		case 0xba: return M_CS_4EDI_EDX_IB;
		case 0xbb: return M_CS_4EDI_EBX_IB;
		case 0xbc: return M_CS_4EDI_ESP_IB;
		case 0xbd: return M_CS_4EDI_EBP_IB;
		case 0xbe: return M_CS_4EDI_ESI_IB;
		case 0xbf: return M_CS_5EDI_IB;

		case 0xc0: return M_CS_9EAX_IB;
		case 0xc1: return M_CS_8EAX_ECX_IB;
		case 0xc2: return M_CS_8EAX_EDX_IB;
		case 0xc3: return M_CS_8EAX_EBX_IB;
		case 0xc4: return M_CS_8EAX_ESP_IB;
		case 0xc5: return M_CS_8EAX_EBP_IB;
		case 0xc6: return M_CS_8EAX_ESI_IB;
		case 0xc7: return M_CS_8EAX_EDI_IB;

		case 0xc8: return M_CS_8ECX_EAX_IB;
		case 0xc9: return M_CS_9ECX_IB;
		case 0xca: return M_CS_8ECX_EDX_IB;
		case 0xcb: return M_CS_8ECX_EBX_IB;
		case 0xcc: return M_CS_8ECX_ESP_IB;
		case 0xcd: return M_CS_8ECX_EBP_IB;
		case 0xce: return M_CS_8ECX_ESI_IB;
		case 0xcf: return M_CS_8ECX_EDI_IB;

		case 0xd0: return M_CS_8EDX_EAX_IB;
		case 0xd1: return M_CS_8EDX_ECX_IB;
		case 0xd2: return M_CS_9EDX_IB;
		case 0xd3: return M_CS_8EDX_EBX_IB;
		case 0xd4: return M_CS_8EDX_ESP_IB;
		case 0xd5: return M_CS_8EDX_EBP_IB;
		case 0xd6: return M_CS_8EDX_ESI_IB;
		case 0xd7: return M_CS_8EDX_EDI_IB;

		case 0xd8: return M_CS_8EBX_EAX_IB;
		case 0xd9: return M_CS_8EBX_ECX_IB;
		case 0xda: return M_CS_8EBX_EDX_IB;
		case 0xdb: return M_CS_9EBX_IB;
		case 0xdc: return M_CS_8EBX_ESP_IB;
		case 0xdd: return M_CS_8EBX_EBP_IB;
		case 0xde: return M_CS_8EBX_ESI_IB;
		case 0xdf: return M_CS_8EBX_EDI_IB;

		case 0xe0: return M_CS_EAX_IB;
		case 0xe1: return M_CS_ECX_IB;
		case 0xe2: return M_CS_EDX_IB;
		case 0xe3: return M_CS_EBX_IB;
		case 0xe4: return M_CS_ESP_IB;
		case 0xe5: return M_CS_EBP_IB;
		case 0xe6: return M_CS_ESI_IB;
		case 0xe7: return M_CS_EDI_IB;

		case 0xe8: return M_CS_8EBP_EAX_IB;
		case 0xe9: return M_CS_8EBP_ECX_IB;
		case 0xea: return M_CS_8EBP_EDX_IB;
		case 0xeb: return M_CS_8EBP_EBX_IB;
		case 0xec: return M_CS_8EBP_ESP_IB;
		case 0xed: return M_CS_9EBP_IB;
		case 0xee: return M_CS_8EBP_ESI_IB;
		case 0xef: return M_CS_8EBP_EDI_IB;

		case 0xf0: return M_CS_8ESI_EAX_IB;
		case 0xf1: return M_CS_8ESI_ECX_IB;
		case 0xf2: return M_CS_8ESI_EDX_IB;
		case 0xf3: return M_CS_8ESI_EBX_IB;
		case 0xf4: return M_CS_8ESI_ESP_IB;
		case 0xf5: return M_CS_8ESI_EBP_IB;
		case 0xf6: return M_CS_9ESI_IB;
		case 0xf7: return M_CS_8ESI_EDI_IB;

		case 0xf8: return M_CS_8EDI_EAX_IB;
		case 0xf9: return M_CS_8EDI_ECX_IB;
		case 0xfa: return M_CS_8EDI_EDX_IB;
		case 0xfb: return M_CS_8EDI_EBX_IB;
		case 0xfc: return M_CS_8EDI_ESP_IB;
		case 0xfd: return M_CS_8EDI_EBP_IB;
		case 0xfe: return M_CS_8EDI_ESI_IB;
		case 0xff: return M_CS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_CS_2EAX_ID;
		case 0x01: return M_CS_EAX_ECX_ID;
		case 0x02: return M_CS_EAX_EDX_ID;
		case 0x03: return M_CS_EAX_EBX_ID;
		case 0x04: return M_CS_EAX_ESP_ID;
		case 0x05: return M_CS_EAX_EBP_ID;
		case 0x06: return M_CS_EAX_ESI_ID;
		case 0x07: return M_CS_EAX_EDI_ID;

		case 0x08: return M_CS_EAX_ECX_ID;
		case 0x09: return M_CS_2ECX_ID;
		case 0x0a: return M_CS_ECX_EDX_ID;
		case 0x0b: return M_CS_ECX_EBX_ID;
		case 0x0c: return M_CS_ECX_ESP_ID;
		case 0x0d: return M_CS_ECX_EBP_ID;
		case 0x0e: return M_CS_ECX_ESI_ID;
		case 0x0f: return M_CS_ECX_EDI_ID;

		case 0x10: return M_CS_EAX_EDX_ID;
		case 0x11: return M_CS_ECX_EDX_ID;
		case 0x12: return M_CS_2EDX_ID;
		case 0x13: return M_CS_EDX_EBX_ID;
		case 0x14: return M_CS_EDX_ESP_ID;
		case 0x15: return M_CS_EDX_EBP_ID;
		case 0x16: return M_CS_EDX_ESI_ID;
		case 0x17: return M_CS_EDX_EDI_ID;

		case 0x18: return M_CS_EAX_EBX_ID;
		case 0x19: return M_CS_ECX_EBX_ID;
		case 0x1a: return M_CS_EDX_EBX_ID;
		case 0x1b: return M_CS_2EBX_ID;
		case 0x1c: return M_CS_EBX_ESP_ID;
		case 0x1d: return M_CS_EBX_EBP_ID;
		case 0x1e: return M_CS_EBX_ESI_ID;
		case 0x1f: return M_CS_EBX_EDI_ID;

		case 0x20: return M_CS_EAX_ID;
		case 0x21: return M_CS_ECX_ID;
		case 0x22: return M_CS_EDX_ID;
		case 0x23: return M_CS_EBX_ID;
		case 0x24: return M_CS_ESP_ID;
		case 0x25: return M_CS_EBP_ID;
		case 0x26: return M_CS_ESI_ID;
		case 0x27: return M_CS_EDI_ID;

		case 0x28: return M_CS_EAX_EBP_ID;
		case 0x29: return M_CS_ECX_EBP_ID;
		case 0x2a: return M_CS_EDX_EBP_ID;
		case 0x2b: return M_CS_EBX_EBP_ID;
		case 0x2c: return M_CS_EBP_ESP_ID;
		case 0x2d: return M_CS_2EBP_ID;
		case 0x2e: return M_CS_EBP_ESI_ID;
		case 0x2f: return M_CS_EBP_EDI_ID;

		case 0x30: return M_CS_EAX_ESI_ID;
		case 0x31: return M_CS_ECX_ESI_ID;
		case 0x32: return M_CS_EDX_ESI_ID;
		case 0x33: return M_CS_EBX_ESI_ID;
		case 0x34: return M_CS_ESI_ESP_ID;
		case 0x35: return M_CS_EBP_ESI_ID;
		case 0x36: return M_CS_2ESI_ID;
		case 0x37: return M_CS_ESI_EDI_ID;

		case 0x38: return M_CS_EAX_EDI_ID;
		case 0x39: return M_CS_ECX_EDI_ID;
		case 0x3a: return M_CS_EDX_EDI_ID;
		case 0x3b: return M_CS_EBX_EDI_ID;
		case 0x3c: return M_CS_EDI_ESP_ID;
		case 0x3d: return M_CS_EBP_EDI_ID;
		case 0x3e: return M_CS_ESI_EDI_ID;
		case 0x3f: return M_CS_2EDI_ID;

		case 0x40: return M_CS_3EAX_ID;
		case 0x41: return M_CS_2EAX_ECX_ID;
		case 0x42: return M_CS_2EAX_EDX_ID;
		case 0x43: return M_CS_2EAX_EBX_ID;
		case 0x44: return M_CS_2EAX_ESP_ID;
		case 0x45: return M_CS_2EAX_EBP_ID;
		case 0x46: return M_CS_2EAX_ESI_ID;
		case 0x47: return M_CS_2EAX_EDI_ID;

		case 0x48: return M_CS_2ECX_EAX_ID;
		case 0x49: return M_CS_3ECX_ID;
		case 0x4a: return M_CS_2ECX_EDX_ID;
		case 0x4b: return M_CS_2ECX_EBX_ID;
		case 0x4c: return M_CS_2ECX_ESP_ID;
		case 0x4d: return M_CS_2ECX_EBP_ID;
		case 0x4e: return M_CS_2ECX_ESI_ID;
		case 0x4f: return M_CS_2ECX_EDI_ID;

		case 0x50: return M_CS_2EDX_EAX_ID;
		case 0x51: return M_CS_2EDX_ECX_ID;
		case 0x52: return M_CS_3EDX_ID;
		case 0x53: return M_CS_2EDX_EBX_ID;
		case 0x54: return M_CS_2EDX_ESP_ID;
		case 0x55: return M_CS_2EDX_EBP_ID;
		case 0x56: return M_CS_2EDX_ESI_ID;
		case 0x57: return M_CS_2EDX_EDI_ID;

		case 0x58: return M_CS_2EBX_EAX_ID;
		case 0x59: return M_CS_2EBX_ECX_ID;
		case 0x5a: return M_CS_2EBX_EDX_ID;
		case 0x5b: return M_CS_3EBX_ID;
		case 0x5c: return M_CS_2EBX_ESP_ID;
		case 0x5d: return M_CS_2EBX_EBP_ID;
		case 0x5e: return M_CS_2EBX_ESI_ID;
		case 0x5f: return M_CS_2EBX_EDI_ID;

		case 0x60: return M_CS_EAX_ID;
		case 0x61: return M_CS_ECX_ID;
		case 0x62: return M_CS_EDX_ID;
		case 0x63: return M_CS_EBX_ID;
		case 0x64: return M_CS_ESP_ID;
		case 0x65: return M_CS_EBP_ID;
		case 0x66: return M_CS_ESI_ID;
		case 0x67: return M_CS_EDI_ID;

		case 0x68: return M_CS_2EBP_EAX_ID;
		case 0x69: return M_CS_2EBP_ECX_ID;
		case 0x6a: return M_CS_2EBP_EDX_ID;
		case 0x6b: return M_CS_2EBP_EBX_ID;
		case 0x6c: return M_CS_2EBP_ESP_ID;
		case 0x6d: return M_CS_3EBP_ID;
		case 0x6e: return M_CS_2EBP_ESI_ID;
		case 0x6f: return M_CS_2EBP_EDI_ID;

		case 0x70: return M_CS_2ESI_EAX_ID;
		case 0x71: return M_CS_2ESI_ECX_ID;
		case 0x72: return M_CS_2ESI_EDX_ID;
		case 0x73: return M_CS_2ESI_EBX_ID;
		case 0x74: return M_CS_2ESI_ESP_ID;
		case 0x75: return M_CS_2ESI_EBP_ID;
		case 0x76: return M_CS_3ESI_ID;
		case 0x77: return M_CS_2ESI_EDI_ID;

		case 0x78: return M_CS_2EDI_EAX_ID;
		case 0x79: return M_CS_2EDI_ECX_ID;
		case 0x7a: return M_CS_2EDI_EDX_ID;
		case 0x7b: return M_CS_2EDI_EBX_ID;
		case 0x7c: return M_CS_2EDI_ESP_ID;
		case 0x7d: return M_CS_2EDI_EBP_ID;
		case 0x7e: return M_CS_2EDI_ESI_ID;
		case 0x7f: return M_CS_3EDI_ID;

		case 0x80: return M_CS_5EAX_ID;
		case 0x81: return M_CS_4EAX_ECX_ID;
		case 0x82: return M_CS_4EAX_EDX_ID;
		case 0x83: return M_CS_4EAX_EBX_ID;
		case 0x84: return M_CS_4EAX_ESP_ID;
		case 0x85: return M_CS_4EAX_EBP_ID;
		case 0x86: return M_CS_4EAX_ESI_ID;
		case 0x87: return M_CS_4EAX_EDI_ID;

		case 0x88: return M_CS_4ECX_EAX_ID;
		case 0x89: return M_CS_5ECX_ID;
		case 0x8a: return M_CS_4ECX_EDX_ID;
		case 0x8b: return M_CS_4ECX_EBX_ID;
		case 0x8c: return M_CS_4ECX_ESP_ID;
		case 0x8d: return M_CS_4ECX_EBP_ID;
		case 0x8e: return M_CS_4ECX_ESI_ID;
		case 0x8f: return M_CS_4ECX_EDI_ID;

		case 0x90: return M_CS_4EDX_EAX_ID;
		case 0x91: return M_CS_4EDX_ECX_ID;
		case 0x92: return M_CS_5EDX_ID;
		case 0x93: return M_CS_4EDX_EBX_ID;
		case 0x94: return M_CS_4EDX_ESP_ID;
		case 0x95: return M_CS_4EDX_EBP_ID;
		case 0x96: return M_CS_4EDX_ESI_ID;
		case 0x97: return M_CS_4EDX_EDI_ID;

		case 0x98: return M_CS_4EBX_EAX_ID;
		case 0x99: return M_CS_4EBX_ECX_ID;
		case 0x9a: return M_CS_4EBX_EDX_ID;
		case 0x9b: return M_CS_5EBX_ID;
		case 0x9c: return M_CS_4EBX_ESP_ID;
		case 0x9d: return M_CS_4EBX_EBP_ID;
		case 0x9e: return M_CS_4EBX_ESI_ID;
		case 0x9f: return M_CS_4EBX_EDI_ID;

		case 0xa0: return M_CS_EAX_ID;
		case 0xa1: return M_CS_ECX_ID;
		case 0xa2: return M_CS_EDX_ID;
		case 0xa3: return M_CS_EBX_ID;
		case 0xa4: return M_CS_ESP_ID;
		case 0xa5: return M_CS_EBP_ID;
		case 0xa6: return M_CS_ESI_ID;
		case 0xa7: return M_CS_EDI_ID;

		case 0xa8: return M_CS_4EBP_EAX_ID;
		case 0xa9: return M_CS_4EBP_ECX_ID;
		case 0xaa: return M_CS_4EBP_EDX_ID;
		case 0xab: return M_CS_4EBP_EBX_ID;
		case 0xac: return M_CS_4EBP_ESP_ID;
		case 0xad: return M_CS_5EBP_ID;
		case 0xae: return M_CS_4EBP_ESI_ID;
		case 0xaf: return M_CS_4EBP_EDI_ID;

		case 0xb0: return M_CS_4ESI_EAX_ID;
		case 0xb1: return M_CS_4ESI_ECX_ID;
		case 0xb2: return M_CS_4ESI_EDX_ID;
		case 0xb3: return M_CS_4ESI_EBX_ID;
		case 0xb4: return M_CS_4ESI_ESP_ID;
		case 0xb5: return M_CS_4ESI_EBP_ID;
		case 0xb6: return M_CS_5ESI_ID;
		case 0xb7: return M_CS_4ESI_EDI_ID;

		case 0xb8: return M_CS_4EDI_EAX_ID;
		case 0xb9: return M_CS_4EDI_ECX_ID;
		case 0xba: return M_CS_4EDI_EDX_ID;
		case 0xbb: return M_CS_4EDI_EBX_ID;
		case 0xbc: return M_CS_4EDI_ESP_ID;
		case 0xbd: return M_CS_4EDI_EBP_ID;
		case 0xbe: return M_CS_4EDI_ESI_ID;
		case 0xbf: return M_CS_5EDI_ID;

		case 0xc0: return M_CS_9EAX_ID;
		case 0xc1: return M_CS_8EAX_ECX_ID;
		case 0xc2: return M_CS_8EAX_EDX_ID;
		case 0xc3: return M_CS_8EAX_EBX_ID;
		case 0xc4: return M_CS_8EAX_ESP_ID;
		case 0xc5: return M_CS_8EAX_EBP_ID;
		case 0xc6: return M_CS_8EAX_ESI_ID;
		case 0xc7: return M_CS_8EAX_EDI_ID;

		case 0xc8: return M_CS_8ECX_EAX_ID;
		case 0xc9: return M_CS_9ECX_ID;
		case 0xca: return M_CS_8ECX_EDX_ID;
		case 0xcb: return M_CS_8ECX_EBX_ID;
		case 0xcc: return M_CS_8ECX_ESP_ID;
		case 0xcd: return M_CS_8ECX_EBP_ID;
		case 0xce: return M_CS_8ECX_ESI_ID;
		case 0xcf: return M_CS_8ECX_EDI_ID;

		case 0xd0: return M_CS_8EDX_EAX_ID;
		case 0xd1: return M_CS_8EDX_ECX_ID;
		case 0xd2: return M_CS_9EDX_ID;
		case 0xd3: return M_CS_8EDX_EBX_ID;
		case 0xd4: return M_CS_8EDX_ESP_ID;
		case 0xd5: return M_CS_8EDX_EBP_ID;
		case 0xd6: return M_CS_8EDX_ESI_ID;
		case 0xd7: return M_CS_8EDX_EDI_ID;

		case 0xd8: return M_CS_8EBX_EAX_ID;
		case 0xd9: return M_CS_8EBX_ECX_ID;
		case 0xda: return M_CS_8EBX_EDX_ID;
		case 0xdb: return M_CS_9EBX_ID;
		case 0xdc: return M_CS_8EBX_ESP_ID;
		case 0xdd: return M_CS_8EBX_EBP_ID;
		case 0xde: return M_CS_8EBX_ESI_ID;
		case 0xdf: return M_CS_8EBX_EDI_ID;

		case 0xe0: return M_CS_EAX_ID;
		case 0xe1: return M_CS_ECX_ID;
		case 0xe2: return M_CS_EDX_ID;
		case 0xe3: return M_CS_EBX_ID;
		case 0xe4: return M_CS_ESP_ID;
		case 0xe5: return M_CS_EBP_ID;
		case 0xe6: return M_CS_ESI_ID;
		case 0xe7: return M_CS_EDI_ID;

		case 0xe8: return M_CS_8EBP_EAX_ID;
		case 0xe9: return M_CS_8EBP_ECX_ID;
		case 0xea: return M_CS_8EBP_EDX_ID;
		case 0xeb: return M_CS_8EBP_EBX_ID;
		case 0xec: return M_CS_8EBP_ESP_ID;
		case 0xed: return M_CS_9EBP_ID;
		case 0xee: return M_CS_8EBP_ESI_ID;
		case 0xef: return M_CS_8EBP_EDI_ID;

		case 0xf0: return M_CS_8ESI_EAX_ID;
		case 0xf1: return M_CS_8ESI_ECX_ID;
		case 0xf2: return M_CS_8ESI_EDX_ID;
		case 0xf3: return M_CS_8ESI_EBX_ID;
		case 0xf4: return M_CS_8ESI_ESP_ID;
		case 0xf5: return M_CS_8ESI_EBP_ID;
		case 0xf6: return M_CS_9ESI_ID;
		case 0xf7: return M_CS_8ESI_EDI_ID;

		case 0xf8: return M_CS_8EDI_EAX_ID;
		case 0xf9: return M_CS_8EDI_ECX_ID;
		case 0xfa: return M_CS_8EDI_EDX_ID;
		case 0xfb: return M_CS_8EDI_EBX_ID;
		case 0xfc: return M_CS_8EDI_ESP_ID;
		case 0xfd: return M_CS_8EDI_EBP_ID;
		case 0xfe: return M_CS_8EDI_ESI_ID;
		case 0xff: return M_CS_9EDI_ID;
		}
	    }
	case PREFICES_SS:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_SS_2EAX;
		case 0x01: return M_SS_EAX_ECX;
		case 0x02: return M_SS_EAX_EDX;
		case 0x03: return M_SS_EAX_EBX;
		case 0x04: return M_SS_EAX_ESP;
		case 0x05: return M_SS_EAX_ID;
		case 0x06: return M_SS_EAX_ESI;
		case 0x07: return M_SS_EAX_EDI;

		case 0x08: return M_SS_EAX_ECX;
		case 0x09: return M_SS_2ECX;
		case 0x0a: return M_SS_ECX_EDX;
		case 0x0b: return M_SS_ECX_EBX;
		case 0x0c: return M_SS_ECX_ESP;
		case 0x0d: return M_SS_ECX_ID;
		case 0x0e: return M_SS_ECX_ESI;
		case 0x0f: return M_SS_ECX_EDI;

		case 0x10: return M_SS_EAX_EDX;
		case 0x11: return M_SS_ECX_EDX;
		case 0x12: return M_SS_2EDX;
		case 0x13: return M_SS_EDX_EBX;
		case 0x14: return M_SS_EDX_ESP;
		case 0x15: return M_SS_EDX_ID;
		case 0x16: return M_SS_EDX_ESI;
		case 0x17: return M_SS_EDX_EDI;

		case 0x18: return M_SS_EAX_EBX;
		case 0x19: return M_SS_ECX_EBX;
		case 0x1a: return M_SS_EDX_EBX;
		case 0x1b: return M_SS_2EBX;
		case 0x1c: return M_SS_EBX_ESP;
		case 0x1d: return M_SS_EBX_ID;
		case 0x1e: return M_SS_EBX_ESI;
		case 0x1f: return M_SS_EBX_EDI;

		case 0x20: return M_SS_EAX;
		case 0x21: return M_SS_ECX;
		case 0x22: return M_SS_EDX;
		case 0x23: return M_SS_EBX;
		case 0x24: return M_SS_ESP;
		case 0x25: return M_SS_ID;
		case 0x26: return M_SS_ESI;
		case 0x27: return M_SS_EDI;

		case 0x28: return M_SS_EAX_EBP;
		case 0x29: return M_SS_ECX_EBP;
		case 0x2a: return M_SS_EDX_EBP;
		case 0x2b: return M_SS_EBX_EBP;
		case 0x2c: return M_SS_EBP_ESP;
		case 0x2d: return M_SS_EBP_ID;
		case 0x2e: return M_SS_EBP_ESI;
		case 0x2f: return M_SS_EBP_EDI;

		case 0x30: return M_SS_EAX_ESI;
		case 0x31: return M_SS_ECX_ESI;
		case 0x32: return M_SS_EDX_ESI;
		case 0x33: return M_SS_EBX_ESI;
		case 0x34: return M_SS_ESI_ESP;
		case 0x35: return M_SS_ESI_ID;
		case 0x36: return M_SS_2ESI;
		case 0x37: return M_SS_ESI_EDI;

		case 0x38: return M_SS_EAX_EDI;
		case 0x39: return M_SS_ECX_EDI;
		case 0x3a: return M_SS_EDX_EDI;
		case 0x3b: return M_SS_EBX_EDI;
		case 0x3c: return M_SS_EDI_ESP;
		case 0x3d: return M_SS_EDI_ID;
		case 0x3e: return M_SS_ESI_EDI;
		case 0x3f: return M_SS_2EDI;

		case 0x40: return M_SS_3EAX;
		case 0x41: return M_SS_2EAX_ECX;
		case 0x42: return M_SS_2EAX_EDX;
		case 0x43: return M_SS_2EAX_EBX;
		case 0x44: return M_SS_2EAX_ESP;
		case 0x45: return M_SS_2EAX_ID;
		case 0x46: return M_SS_2EAX_ESI;
		case 0x47: return M_SS_2EAX_EDI;

		case 0x48: return M_SS_2ECX_EAX;
		case 0x49: return M_SS_3ECX;
		case 0x4a: return M_SS_2ECX_EDX;
		case 0x4b: return M_SS_2ECX_EBX;
		case 0x4c: return M_SS_2ECX_ESP;
		case 0x4d: return M_SS_2ECX_ID;
		case 0x4e: return M_SS_2ECX_ESI;
		case 0x4f: return M_SS_2ECX_EDI;

		case 0x50: return M_SS_2EDX_EAX;
		case 0x51: return M_SS_2EDX_ECX;
		case 0x52: return M_SS_3EDX;
		case 0x53: return M_SS_2EDX_EBX;
		case 0x54: return M_SS_2EDX_ESP;
		case 0x55: return M_SS_2EDX_ID;
		case 0x56: return M_SS_2EDX_ESI;
		case 0x57: return M_SS_2EDX_EDI;

		case 0x58: return M_SS_2EBX_EAX;
		case 0x59: return M_SS_2EBX_ECX;
		case 0x5a: return M_SS_2EBX_EDX;
		case 0x5b: return M_SS_3EBX;
		case 0x5c: return M_SS_2EBX_ESP;
		case 0x5d: return M_SS_2EBX_ID;
		case 0x5e: return M_SS_2EBX_ESI;
		case 0x5f: return M_SS_2EBX_EDI;

		case 0x60: return M_SS_EAX;
		case 0x61: return M_SS_ECX;
		case 0x62: return M_SS_EDX;
		case 0x63: return M_SS_EBX;
		case 0x64: return M_SS_ESP;
		case 0x65: return M_SS_ID;
		case 0x66: return M_SS_ESI;
		case 0x67: return M_SS_EDI;

		case 0x68: return M_SS_2EBP_EAX;
		case 0x69: return M_SS_2EBP_ECX;
		case 0x6a: return M_SS_2EBP_EDX;
		case 0x6b: return M_SS_2EBP_EBX;
		case 0x6c: return M_SS_2EBP_ESP;
		case 0x6d: return M_SS_2EBP_ID;
		case 0x6e: return M_SS_2EBP_ESI;
		case 0x6f: return M_SS_2EBP_EDI;

		case 0x70: return M_SS_2ESI_EAX;
		case 0x71: return M_SS_2ESI_ECX;
		case 0x72: return M_SS_2ESI_EDX;
		case 0x73: return M_SS_2ESI_EBX;
		case 0x74: return M_SS_2ESI_ESP;
		case 0x75: return M_SS_2ESI_ID;
		case 0x76: return M_SS_3ESI;
		case 0x77: return M_SS_2ESI_EDI;

		case 0x78: return M_SS_2EDI_EAX;
		case 0x79: return M_SS_2EDI_ECX;
		case 0x7a: return M_SS_2EDI_EDX;
		case 0x7b: return M_SS_2EDI_EBX;
		case 0x7c: return M_SS_2EDI_ESP;
		case 0x7d: return M_SS_2EDI_ID;
		case 0x7e: return M_SS_2EDI_ESI;
		case 0x7f: return M_SS_3EDI;

		case 0x80: return M_SS_5EAX;
		case 0x81: return M_SS_4EAX_ECX;
		case 0x82: return M_SS_4EAX_EDX;
		case 0x83: return M_SS_4EAX_EBX;
		case 0x84: return M_SS_4EAX_ESP;
		case 0x85: return M_SS_4EAX_ID;
		case 0x86: return M_SS_4EAX_ESI;
		case 0x87: return M_SS_4EAX_EDI;

		case 0x88: return M_SS_4ECX_EAX;
		case 0x89: return M_SS_5ECX;
		case 0x8a: return M_SS_4ECX_EDX;
		case 0x8b: return M_SS_4ECX_EBX;
		case 0x8c: return M_SS_4ECX_ESP;
		case 0x8d: return M_SS_4ECX_ID;
		case 0x8e: return M_SS_4ECX_ESI;
		case 0x8f: return M_SS_4ECX_EDI;

		case 0x90: return M_SS_4EDX_EAX;
		case 0x91: return M_SS_4EDX_ECX;
		case 0x92: return M_SS_5EDX;
		case 0x93: return M_SS_4EDX_EBX;
		case 0x94: return M_SS_4EDX_ESP;
		case 0x95: return M_SS_4EDX_ID;
		case 0x96: return M_SS_4EDX_ESI;
		case 0x97: return M_SS_4EDX_EDI;

		case 0x98: return M_SS_4EBX_EAX;
		case 0x99: return M_SS_4EBX_ECX;
		case 0x9a: return M_SS_4EBX_EDX;
		case 0x9b: return M_SS_5EBX;
		case 0x9c: return M_SS_4EBX_ESP;
		case 0x9d: return M_SS_4EBX_ID;
		case 0x9e: return M_SS_4EBX_ESI;
		case 0x9f: return M_SS_4EBX_EDI;

		case 0xa0: return M_SS_EAX;
		case 0xa1: return M_SS_ECX;
		case 0xa2: return M_SS_EDX;
		case 0xa3: return M_SS_EBX;
		case 0xa4: return M_SS_ESP;
		case 0xa5: return M_SS_ID;
		case 0xa6: return M_SS_ESI;
		case 0xa7: return M_SS_EDI;

		case 0xa8: return M_SS_4EBP_EAX;
		case 0xa9: return M_SS_4EBP_ECX;
		case 0xaa: return M_SS_4EBP_EDX;
		case 0xab: return M_SS_4EBP_EBX;
		case 0xac: return M_SS_4EBP_ESP;
		case 0xad: return M_SS_4EBP_ID;
		case 0xae: return M_SS_4EBP_ESI;
		case 0xaf: return M_SS_4EBP_EDI;

		case 0xb0: return M_SS_4ESI_EAX;
		case 0xb1: return M_SS_4ESI_ECX;
		case 0xb2: return M_SS_4ESI_EDX;
		case 0xb3: return M_SS_4ESI_EBX;
		case 0xb4: return M_SS_4ESI_ESP;
		case 0xb5: return M_SS_4ESI_ID;
		case 0xb6: return M_SS_5ESI;
		case 0xb7: return M_SS_4ESI_EDI;

		case 0xb8: return M_SS_4EDI_EAX;
		case 0xb9: return M_SS_4EDI_ECX;
		case 0xba: return M_SS_4EDI_EDX;
		case 0xbb: return M_SS_4EDI_EBX;
		case 0xbc: return M_SS_4EDI_ESP;
		case 0xbd: return M_SS_4EDI_ID;
		case 0xbe: return M_SS_4EDI_ESI;
		case 0xbf: return M_SS_5EDI;

		case 0xc0: return M_SS_9EAX;
		case 0xc1: return M_SS_8EAX_ECX;
		case 0xc2: return M_SS_8EAX_EDX;
		case 0xc3: return M_SS_8EAX_EBX;
		case 0xc4: return M_SS_8EAX_ESP;
		case 0xc5: return M_SS_8EAX_ID;
		case 0xc6: return M_SS_8EAX_ESI;
		case 0xc7: return M_SS_8EAX_EDI;

		case 0xc8: return M_SS_8ECX_EAX;
		case 0xc9: return M_SS_9ECX;
		case 0xca: return M_SS_8ECX_EDX;
		case 0xcb: return M_SS_8ECX_EBX;
		case 0xcc: return M_SS_8ECX_ESP;
		case 0xcd: return M_SS_8ECX_ID;
		case 0xce: return M_SS_8ECX_ESI;
		case 0xcf: return M_SS_8ECX_EDI;

		case 0xd0: return M_SS_8EDX_EAX;
		case 0xd1: return M_SS_8EDX_ECX;
		case 0xd2: return M_SS_9EDX;
		case 0xd3: return M_SS_8EDX_EBX;
		case 0xd4: return M_SS_8EDX_ESP;
		case 0xd5: return M_SS_8EDX_ID;
		case 0xd6: return M_SS_8EDX_ESI;
		case 0xd7: return M_SS_8EDX_EDI;

		case 0xd8: return M_SS_8EBX_EAX;
		case 0xd9: return M_SS_8EBX_ECX;
		case 0xda: return M_SS_8EBX_EDX;
		case 0xdb: return M_SS_9EBX;
		case 0xdc: return M_SS_8EBX_ESP;
		case 0xdd: return M_SS_8EBX_ID;
		case 0xde: return M_SS_8EBX_ESI;
		case 0xdf: return M_SS_8EBX_EDI;

		case 0xe0: return M_SS_EAX;
		case 0xe1: return M_SS_ECX;
		case 0xe2: return M_SS_EDX;
		case 0xe3: return M_SS_EBX;
		case 0xe4: return M_SS_ESP;
		case 0xe5: return M_SS_ID;
		case 0xe6: return M_SS_ESI;
		case 0xe7: return M_SS_EDI;

		case 0xe8: return M_SS_8EBP_EAX;
		case 0xe9: return M_SS_8EBP_ECX;
		case 0xea: return M_SS_8EBP_EDX;
		case 0xeb: return M_SS_8EBP_EBX;
		case 0xec: return M_SS_8EBP_ESP;
		case 0xed: return M_SS_8EBP_ID;
		case 0xee: return M_SS_8EBP_ESI;
		case 0xef: return M_SS_8EBP_EDI;

		case 0xf0: return M_SS_8ESI_EAX;
		case 0xf1: return M_SS_8ESI_ECX;
		case 0xf2: return M_SS_8ESI_EDX;
		case 0xf3: return M_SS_8ESI_EBX;
		case 0xf4: return M_SS_8ESI_ESP;
		case 0xf5: return M_SS_8ESI_ID;
		case 0xf6: return M_SS_9ESI;
		case 0xf7: return M_SS_8ESI_EDI;

		case 0xf8: return M_SS_8EDI_EAX;
		case 0xf9: return M_SS_8EDI_ECX;
		case 0xfa: return M_SS_8EDI_EDX;
		case 0xfb: return M_SS_8EDI_EBX;
		case 0xfc: return M_SS_8EDI_ESP;
		case 0xfd: return M_SS_8EDI_ID;
		case 0xfe: return M_SS_8EDI_ESI;
		case 0xff: return M_SS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_SS_2EAX_IB;
		case 0x01: return M_SS_EAX_ECX_IB;
		case 0x02: return M_SS_EAX_EDX_IB;
		case 0x03: return M_SS_EAX_EBX_IB;
		case 0x04: return M_SS_EAX_ESP_IB;
		case 0x05: return M_SS_EAX_EBP_IB;
		case 0x06: return M_SS_EAX_ESI_IB;
		case 0x07: return M_SS_EAX_EDI_IB;

		case 0x08: return M_SS_EAX_ECX_IB;
		case 0x09: return M_SS_2ECX_IB;
		case 0x0a: return M_SS_ECX_EDX_IB;
		case 0x0b: return M_SS_ECX_EBX_IB;
		case 0x0c: return M_SS_ECX_ESP_IB;
		case 0x0d: return M_SS_ECX_EBP_IB;
		case 0x0e: return M_SS_ECX_ESI_IB;
		case 0x0f: return M_SS_ECX_EDI_IB;

		case 0x10: return M_SS_EAX_EDX_IB;
		case 0x11: return M_SS_ECX_EDX_IB;
		case 0x12: return M_SS_2EDX_IB;
		case 0x13: return M_SS_EDX_EBX_IB;
		case 0x14: return M_SS_EDX_ESP_IB;
		case 0x15: return M_SS_EDX_EBP_IB;
		case 0x16: return M_SS_EDX_ESI_IB;
		case 0x17: return M_SS_EDX_EDI_IB;

		case 0x18: return M_SS_EAX_EBX_IB;
		case 0x19: return M_SS_ECX_EBX_IB;
		case 0x1a: return M_SS_EDX_EBX_IB;
		case 0x1b: return M_SS_2EBX_IB;
		case 0x1c: return M_SS_EBX_ESP_IB;
		case 0x1d: return M_SS_EBX_EBP_IB;
		case 0x1e: return M_SS_EBX_ESI_IB;
		case 0x1f: return M_SS_EBX_EDI_IB;

		case 0x20: return M_SS_EAX_IB;
		case 0x21: return M_SS_ECX_IB;
		case 0x22: return M_SS_EDX_IB;
		case 0x23: return M_SS_EBX_IB;
		case 0x24: return M_SS_ESP_IB;
		case 0x25: return M_SS_EBP_IB;
		case 0x26: return M_SS_ESI_IB;
		case 0x27: return M_SS_EDI_IB;

		case 0x28: return M_SS_EAX_EBP_IB;
		case 0x29: return M_SS_ECX_EBP_IB;
		case 0x2a: return M_SS_EDX_EBP_IB;
		case 0x2b: return M_SS_EBX_EBP_IB;
		case 0x2c: return M_SS_EBP_ESP_IB;
		case 0x2d: return M_SS_2EBP_IB;
		case 0x2e: return M_SS_EBP_ESI_IB;
		case 0x2f: return M_SS_EBP_EDI_IB;

		case 0x30: return M_SS_EAX_ESI_IB;
		case 0x31: return M_SS_ECX_ESI_IB;
		case 0x32: return M_SS_EDX_ESI_IB;
		case 0x33: return M_SS_EBX_ESI_IB;
		case 0x34: return M_SS_ESI_ESP_IB;
		case 0x35: return M_SS_EBP_ESI_IB;
		case 0x36: return M_SS_2ESI_IB;
		case 0x37: return M_SS_ESI_EDI_IB;

		case 0x38: return M_SS_EAX_EDI_IB;
		case 0x39: return M_SS_ECX_EDI_IB;
		case 0x3a: return M_SS_EDX_EDI_IB;
		case 0x3b: return M_SS_EBX_EDI_IB;
		case 0x3c: return M_SS_EDI_ESP_IB;
		case 0x3d: return M_SS_EBP_EDI_IB;
		case 0x3e: return M_SS_ESI_EDI_IB;
		case 0x3f: return M_SS_2EDI_IB;

		case 0x40: return M_SS_3EAX_IB;
		case 0x41: return M_SS_2EAX_ECX_IB;
		case 0x42: return M_SS_2EAX_EDX_IB;
		case 0x43: return M_SS_2EAX_EBX_IB;
		case 0x44: return M_SS_2EAX_ESP_IB;
		case 0x45: return M_SS_2EAX_EBP_IB;
		case 0x46: return M_SS_2EAX_ESI_IB;
		case 0x47: return M_SS_2EAX_EDI_IB;

		case 0x48: return M_SS_2ECX_EAX_IB;
		case 0x49: return M_SS_3ECX_IB;
		case 0x4a: return M_SS_2ECX_EDX_IB;
		case 0x4b: return M_SS_2ECX_EBX_IB;
		case 0x4c: return M_SS_2ECX_ESP_IB;
		case 0x4d: return M_SS_2ECX_EBP_IB;
		case 0x4e: return M_SS_2ECX_ESI_IB;
		case 0x4f: return M_SS_2ECX_EDI_IB;

		case 0x50: return M_SS_2EDX_EAX_IB;
		case 0x51: return M_SS_2EDX_ECX_IB;
		case 0x52: return M_SS_3EDX_IB;
		case 0x53: return M_SS_2EDX_EBX_IB;
		case 0x54: return M_SS_2EDX_ESP_IB;
		case 0x55: return M_SS_2EDX_EBP_IB;
		case 0x56: return M_SS_2EDX_ESI_IB;
		case 0x57: return M_SS_2EDX_EDI_IB;

		case 0x58: return M_SS_2EBX_EAX_IB;
		case 0x59: return M_SS_2EBX_ECX_IB;
		case 0x5a: return M_SS_2EBX_EDX_IB;
		case 0x5b: return M_SS_3EBX_IB;
		case 0x5c: return M_SS_2EBX_ESP_IB;
		case 0x5d: return M_SS_2EBX_EBP_IB;
		case 0x5e: return M_SS_2EBX_ESI_IB;
		case 0x5f: return M_SS_2EBX_EDI_IB;

		case 0x60: return M_SS_EAX_IB;
		case 0x61: return M_SS_ECX_IB;
		case 0x62: return M_SS_EDX_IB;
		case 0x63: return M_SS_EBX_IB;
		case 0x64: return M_SS_ESP_IB;
		case 0x65: return M_SS_EBP_IB;
		case 0x66: return M_SS_ESI_IB;
		case 0x67: return M_SS_EDI_IB;

		case 0x68: return M_SS_2EBP_EAX_IB;
		case 0x69: return M_SS_2EBP_ECX_IB;
		case 0x6a: return M_SS_2EBP_EDX_IB;
		case 0x6b: return M_SS_2EBP_EBX_IB;
		case 0x6c: return M_SS_2EBP_ESP_IB;
		case 0x6d: return M_SS_3EBP_IB;
		case 0x6e: return M_SS_2EBP_ESI_IB;
		case 0x6f: return M_SS_2EBP_EDI_IB;

		case 0x70: return M_SS_2ESI_EAX_IB;
		case 0x71: return M_SS_2ESI_ECX_IB;
		case 0x72: return M_SS_2ESI_EDX_IB;
		case 0x73: return M_SS_2ESI_EBX_IB;
		case 0x74: return M_SS_2ESI_ESP_IB;
		case 0x75: return M_SS_2ESI_EBP_IB;
		case 0x76: return M_SS_3ESI_IB;
		case 0x77: return M_SS_2ESI_EDI_IB;

		case 0x78: return M_SS_2EDI_EAX_IB;
		case 0x79: return M_SS_2EDI_ECX_IB;
		case 0x7a: return M_SS_2EDI_EDX_IB;
		case 0x7b: return M_SS_2EDI_EBX_IB;
		case 0x7c: return M_SS_2EDI_ESP_IB;
		case 0x7d: return M_SS_2EDI_EBP_IB;
		case 0x7e: return M_SS_2EDI_ESI_IB;
		case 0x7f: return M_SS_3EDI_IB;

		case 0x80: return M_SS_5EAX_IB;
		case 0x81: return M_SS_4EAX_ECX_IB;
		case 0x82: return M_SS_4EAX_EDX_IB;
		case 0x83: return M_SS_4EAX_EBX_IB;
		case 0x84: return M_SS_4EAX_ESP_IB;
		case 0x85: return M_SS_4EAX_EBP_IB;
		case 0x86: return M_SS_4EAX_ESI_IB;
		case 0x87: return M_SS_4EAX_EDI_IB;

		case 0x88: return M_SS_4ECX_EAX_IB;
		case 0x89: return M_SS_5ECX_IB;
		case 0x8a: return M_SS_4ECX_EDX_IB;
		case 0x8b: return M_SS_4ECX_EBX_IB;
		case 0x8c: return M_SS_4ECX_ESP_IB;
		case 0x8d: return M_SS_4ECX_EBP_IB;
		case 0x8e: return M_SS_4ECX_ESI_IB;
		case 0x8f: return M_SS_4ECX_EDI_IB;

		case 0x90: return M_SS_4EDX_EAX_IB;
		case 0x91: return M_SS_4EDX_ECX_IB;
		case 0x92: return M_SS_5EDX_IB;
		case 0x93: return M_SS_4EDX_EBX_IB;
		case 0x94: return M_SS_4EDX_ESP_IB;
		case 0x95: return M_SS_4EDX_EBP_IB;
		case 0x96: return M_SS_4EDX_ESI_IB;
		case 0x97: return M_SS_4EDX_EDI_IB;

		case 0x98: return M_SS_4EBX_EAX_IB;
		case 0x99: return M_SS_4EBX_ECX_IB;
		case 0x9a: return M_SS_4EBX_EDX_IB;
		case 0x9b: return M_SS_5EBX_IB;
		case 0x9c: return M_SS_4EBX_ESP_IB;
		case 0x9d: return M_SS_4EBX_EBP_IB;
		case 0x9e: return M_SS_4EBX_ESI_IB;
		case 0x9f: return M_SS_4EBX_EDI_IB;

		case 0xa0: return M_SS_EAX_IB;
		case 0xa1: return M_SS_ECX_IB;
		case 0xa2: return M_SS_EDX_IB;
		case 0xa3: return M_SS_EBX_IB;
		case 0xa4: return M_SS_ESP_IB;
		case 0xa5: return M_SS_EBP_IB;
		case 0xa6: return M_SS_ESI_IB;
		case 0xa7: return M_SS_EDI_IB;

		case 0xa8: return M_SS_4EBP_EAX_IB;
		case 0xa9: return M_SS_4EBP_ECX_IB;
		case 0xaa: return M_SS_4EBP_EDX_IB;
		case 0xab: return M_SS_4EBP_EBX_IB;
		case 0xac: return M_SS_4EBP_ESP_IB;
		case 0xad: return M_SS_5EBP_IB;
		case 0xae: return M_SS_4EBP_ESI_IB;
		case 0xaf: return M_SS_4EBP_EDI_IB;

		case 0xb0: return M_SS_4ESI_EAX_IB;
		case 0xb1: return M_SS_4ESI_ECX_IB;
		case 0xb2: return M_SS_4ESI_EDX_IB;
		case 0xb3: return M_SS_4ESI_EBX_IB;
		case 0xb4: return M_SS_4ESI_ESP_IB;
		case 0xb5: return M_SS_4ESI_EBP_IB;
		case 0xb6: return M_SS_5ESI_IB;
		case 0xb7: return M_SS_4ESI_EDI_IB;

		case 0xb8: return M_SS_4EDI_EAX_IB;
		case 0xb9: return M_SS_4EDI_ECX_IB;
		case 0xba: return M_SS_4EDI_EDX_IB;
		case 0xbb: return M_SS_4EDI_EBX_IB;
		case 0xbc: return M_SS_4EDI_ESP_IB;
		case 0xbd: return M_SS_4EDI_EBP_IB;
		case 0xbe: return M_SS_4EDI_ESI_IB;
		case 0xbf: return M_SS_5EDI_IB;

		case 0xc0: return M_SS_9EAX_IB;
		case 0xc1: return M_SS_8EAX_ECX_IB;
		case 0xc2: return M_SS_8EAX_EDX_IB;
		case 0xc3: return M_SS_8EAX_EBX_IB;
		case 0xc4: return M_SS_8EAX_ESP_IB;
		case 0xc5: return M_SS_8EAX_EBP_IB;
		case 0xc6: return M_SS_8EAX_ESI_IB;
		case 0xc7: return M_SS_8EAX_EDI_IB;

		case 0xc8: return M_SS_8ECX_EAX_IB;
		case 0xc9: return M_SS_9ECX_IB;
		case 0xca: return M_SS_8ECX_EDX_IB;
		case 0xcb: return M_SS_8ECX_EBX_IB;
		case 0xcc: return M_SS_8ECX_ESP_IB;
		case 0xcd: return M_SS_8ECX_EBP_IB;
		case 0xce: return M_SS_8ECX_ESI_IB;
		case 0xcf: return M_SS_8ECX_EDI_IB;

		case 0xd0: return M_SS_8EDX_EAX_IB;
		case 0xd1: return M_SS_8EDX_ECX_IB;
		case 0xd2: return M_SS_9EDX_IB;
		case 0xd3: return M_SS_8EDX_EBX_IB;
		case 0xd4: return M_SS_8EDX_ESP_IB;
		case 0xd5: return M_SS_8EDX_EBP_IB;
		case 0xd6: return M_SS_8EDX_ESI_IB;
		case 0xd7: return M_SS_8EDX_EDI_IB;

		case 0xd8: return M_SS_8EBX_EAX_IB;
		case 0xd9: return M_SS_8EBX_ECX_IB;
		case 0xda: return M_SS_8EBX_EDX_IB;
		case 0xdb: return M_SS_9EBX_IB;
		case 0xdc: return M_SS_8EBX_ESP_IB;
		case 0xdd: return M_SS_8EBX_EBP_IB;
		case 0xde: return M_SS_8EBX_ESI_IB;
		case 0xdf: return M_SS_8EBX_EDI_IB;

		case 0xe0: return M_SS_EAX_IB;
		case 0xe1: return M_SS_ECX_IB;
		case 0xe2: return M_SS_EDX_IB;
		case 0xe3: return M_SS_EBX_IB;
		case 0xe4: return M_SS_ESP_IB;
		case 0xe5: return M_SS_EBP_IB;
		case 0xe6: return M_SS_ESI_IB;
		case 0xe7: return M_SS_EDI_IB;

		case 0xe8: return M_SS_8EBP_EAX_IB;
		case 0xe9: return M_SS_8EBP_ECX_IB;
		case 0xea: return M_SS_8EBP_EDX_IB;
		case 0xeb: return M_SS_8EBP_EBX_IB;
		case 0xec: return M_SS_8EBP_ESP_IB;
		case 0xed: return M_SS_9EBP_IB;
		case 0xee: return M_SS_8EBP_ESI_IB;
		case 0xef: return M_SS_8EBP_EDI_IB;

		case 0xf0: return M_SS_8ESI_EAX_IB;
		case 0xf1: return M_SS_8ESI_ECX_IB;
		case 0xf2: return M_SS_8ESI_EDX_IB;
		case 0xf3: return M_SS_8ESI_EBX_IB;
		case 0xf4: return M_SS_8ESI_ESP_IB;
		case 0xf5: return M_SS_8ESI_EBP_IB;
		case 0xf6: return M_SS_9ESI_IB;
		case 0xf7: return M_SS_8ESI_EDI_IB;

		case 0xf8: return M_SS_8EDI_EAX_IB;
		case 0xf9: return M_SS_8EDI_ECX_IB;
		case 0xfa: return M_SS_8EDI_EDX_IB;
		case 0xfb: return M_SS_8EDI_EBX_IB;
		case 0xfc: return M_SS_8EDI_ESP_IB;
		case 0xfd: return M_SS_8EDI_EBP_IB;
		case 0xfe: return M_SS_8EDI_ESI_IB;
		case 0xff: return M_SS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_SS_2EAX_ID;
		case 0x01: return M_SS_EAX_ECX_ID;
		case 0x02: return M_SS_EAX_EDX_ID;
		case 0x03: return M_SS_EAX_EBX_ID;
		case 0x04: return M_SS_EAX_ESP_ID;
		case 0x05: return M_SS_EAX_EBP_ID;
		case 0x06: return M_SS_EAX_ESI_ID;
		case 0x07: return M_SS_EAX_EDI_ID;

		case 0x08: return M_SS_EAX_ECX_ID;
		case 0x09: return M_SS_2ECX_ID;
		case 0x0a: return M_SS_ECX_EDX_ID;
		case 0x0b: return M_SS_ECX_EBX_ID;
		case 0x0c: return M_SS_ECX_ESP_ID;
		case 0x0d: return M_SS_ECX_EBP_ID;
		case 0x0e: return M_SS_ECX_ESI_ID;
		case 0x0f: return M_SS_ECX_EDI_ID;

		case 0x10: return M_SS_EAX_EDX_ID;
		case 0x11: return M_SS_ECX_EDX_ID;
		case 0x12: return M_SS_2EDX_ID;
		case 0x13: return M_SS_EDX_EBX_ID;
		case 0x14: return M_SS_EDX_ESP_ID;
		case 0x15: return M_SS_EDX_EBP_ID;
		case 0x16: return M_SS_EDX_ESI_ID;
		case 0x17: return M_SS_EDX_EDI_ID;

		case 0x18: return M_SS_EAX_EBX_ID;
		case 0x19: return M_SS_ECX_EBX_ID;
		case 0x1a: return M_SS_EDX_EBX_ID;
		case 0x1b: return M_SS_2EBX_ID;
		case 0x1c: return M_SS_EBX_ESP_ID;
		case 0x1d: return M_SS_EBX_EBP_ID;
		case 0x1e: return M_SS_EBX_ESI_ID;
		case 0x1f: return M_SS_EBX_EDI_ID;

		case 0x20: return M_SS_EAX_ID;
		case 0x21: return M_SS_ECX_ID;
		case 0x22: return M_SS_EDX_ID;
		case 0x23: return M_SS_EBX_ID;
		case 0x24: return M_SS_ESP_ID;
		case 0x25: return M_SS_EBP_ID;
		case 0x26: return M_SS_ESI_ID;
		case 0x27: return M_SS_EDI_ID;

		case 0x28: return M_SS_EAX_EBP_ID;
		case 0x29: return M_SS_ECX_EBP_ID;
		case 0x2a: return M_SS_EDX_EBP_ID;
		case 0x2b: return M_SS_EBX_EBP_ID;
		case 0x2c: return M_SS_EBP_ESP_ID;
		case 0x2d: return M_SS_2EBP_ID;
		case 0x2e: return M_SS_EBP_ESI_ID;
		case 0x2f: return M_SS_EBP_EDI_ID;

		case 0x30: return M_SS_EAX_ESI_ID;
		case 0x31: return M_SS_ECX_ESI_ID;
		case 0x32: return M_SS_EDX_ESI_ID;
		case 0x33: return M_SS_EBX_ESI_ID;
		case 0x34: return M_SS_ESI_ESP_ID;
		case 0x35: return M_SS_EBP_ESI_ID;
		case 0x36: return M_SS_2ESI_ID;
		case 0x37: return M_SS_ESI_EDI_ID;

		case 0x38: return M_SS_EAX_EDI_ID;
		case 0x39: return M_SS_ECX_EDI_ID;
		case 0x3a: return M_SS_EDX_EDI_ID;
		case 0x3b: return M_SS_EBX_EDI_ID;
		case 0x3c: return M_SS_EDI_ESP_ID;
		case 0x3d: return M_SS_EBP_EDI_ID;
		case 0x3e: return M_SS_ESI_EDI_ID;
		case 0x3f: return M_SS_2EDI_ID;

		case 0x40: return M_SS_3EAX_ID;
		case 0x41: return M_SS_2EAX_ECX_ID;
		case 0x42: return M_SS_2EAX_EDX_ID;
		case 0x43: return M_SS_2EAX_EBX_ID;
		case 0x44: return M_SS_2EAX_ESP_ID;
		case 0x45: return M_SS_2EAX_EBP_ID;
		case 0x46: return M_SS_2EAX_ESI_ID;
		case 0x47: return M_SS_2EAX_EDI_ID;

		case 0x48: return M_SS_2ECX_EAX_ID;
		case 0x49: return M_SS_3ECX_ID;
		case 0x4a: return M_SS_2ECX_EDX_ID;
		case 0x4b: return M_SS_2ECX_EBX_ID;
		case 0x4c: return M_SS_2ECX_ESP_ID;
		case 0x4d: return M_SS_2ECX_EBP_ID;
		case 0x4e: return M_SS_2ECX_ESI_ID;
		case 0x4f: return M_SS_2ECX_EDI_ID;

		case 0x50: return M_SS_2EDX_EAX_ID;
		case 0x51: return M_SS_2EDX_ECX_ID;
		case 0x52: return M_SS_3EDX_ID;
		case 0x53: return M_SS_2EDX_EBX_ID;
		case 0x54: return M_SS_2EDX_ESP_ID;
		case 0x55: return M_SS_2EDX_EBP_ID;
		case 0x56: return M_SS_2EDX_ESI_ID;
		case 0x57: return M_SS_2EDX_EDI_ID;

		case 0x58: return M_SS_2EBX_EAX_ID;
		case 0x59: return M_SS_2EBX_ECX_ID;
		case 0x5a: return M_SS_2EBX_EDX_ID;
		case 0x5b: return M_SS_3EBX_ID;
		case 0x5c: return M_SS_2EBX_ESP_ID;
		case 0x5d: return M_SS_2EBX_EBP_ID;
		case 0x5e: return M_SS_2EBX_ESI_ID;
		case 0x5f: return M_SS_2EBX_EDI_ID;

		case 0x60: return M_SS_EAX_ID;
		case 0x61: return M_SS_ECX_ID;
		case 0x62: return M_SS_EDX_ID;
		case 0x63: return M_SS_EBX_ID;
		case 0x64: return M_SS_ESP_ID;
		case 0x65: return M_SS_EBP_ID;
		case 0x66: return M_SS_ESI_ID;
		case 0x67: return M_SS_EDI_ID;

		case 0x68: return M_SS_2EBP_EAX_ID;
		case 0x69: return M_SS_2EBP_ECX_ID;
		case 0x6a: return M_SS_2EBP_EDX_ID;
		case 0x6b: return M_SS_2EBP_EBX_ID;
		case 0x6c: return M_SS_2EBP_ESP_ID;
		case 0x6d: return M_SS_3EBP_ID;
		case 0x6e: return M_SS_2EBP_ESI_ID;
		case 0x6f: return M_SS_2EBP_EDI_ID;

		case 0x70: return M_SS_2ESI_EAX_ID;
		case 0x71: return M_SS_2ESI_ECX_ID;
		case 0x72: return M_SS_2ESI_EDX_ID;
		case 0x73: return M_SS_2ESI_EBX_ID;
		case 0x74: return M_SS_2ESI_ESP_ID;
		case 0x75: return M_SS_2ESI_EBP_ID;
		case 0x76: return M_SS_3ESI_ID;
		case 0x77: return M_SS_2ESI_EDI_ID;

		case 0x78: return M_SS_2EDI_EAX_ID;
		case 0x79: return M_SS_2EDI_ECX_ID;
		case 0x7a: return M_SS_2EDI_EDX_ID;
		case 0x7b: return M_SS_2EDI_EBX_ID;
		case 0x7c: return M_SS_2EDI_ESP_ID;
		case 0x7d: return M_SS_2EDI_EBP_ID;
		case 0x7e: return M_SS_2EDI_ESI_ID;
		case 0x7f: return M_SS_3EDI_ID;

		case 0x80: return M_SS_5EAX_ID;
		case 0x81: return M_SS_4EAX_ECX_ID;
		case 0x82: return M_SS_4EAX_EDX_ID;
		case 0x83: return M_SS_4EAX_EBX_ID;
		case 0x84: return M_SS_4EAX_ESP_ID;
		case 0x85: return M_SS_4EAX_EBP_ID;
		case 0x86: return M_SS_4EAX_ESI_ID;
		case 0x87: return M_SS_4EAX_EDI_ID;

		case 0x88: return M_SS_4ECX_EAX_ID;
		case 0x89: return M_SS_5ECX_ID;
		case 0x8a: return M_SS_4ECX_EDX_ID;
		case 0x8b: return M_SS_4ECX_EBX_ID;
		case 0x8c: return M_SS_4ECX_ESP_ID;
		case 0x8d: return M_SS_4ECX_EBP_ID;
		case 0x8e: return M_SS_4ECX_ESI_ID;
		case 0x8f: return M_SS_4ECX_EDI_ID;

		case 0x90: return M_SS_4EDX_EAX_ID;
		case 0x91: return M_SS_4EDX_ECX_ID;
		case 0x92: return M_SS_5EDX_ID;
		case 0x93: return M_SS_4EDX_EBX_ID;
		case 0x94: return M_SS_4EDX_ESP_ID;
		case 0x95: return M_SS_4EDX_EBP_ID;
		case 0x96: return M_SS_4EDX_ESI_ID;
		case 0x97: return M_SS_4EDX_EDI_ID;

		case 0x98: return M_SS_4EBX_EAX_ID;
		case 0x99: return M_SS_4EBX_ECX_ID;
		case 0x9a: return M_SS_4EBX_EDX_ID;
		case 0x9b: return M_SS_5EBX_ID;
		case 0x9c: return M_SS_4EBX_ESP_ID;
		case 0x9d: return M_SS_4EBX_EBP_ID;
		case 0x9e: return M_SS_4EBX_ESI_ID;
		case 0x9f: return M_SS_4EBX_EDI_ID;

		case 0xa0: return M_SS_EAX_ID;
		case 0xa1: return M_SS_ECX_ID;
		case 0xa2: return M_SS_EDX_ID;
		case 0xa3: return M_SS_EBX_ID;
		case 0xa4: return M_SS_ESP_ID;
		case 0xa5: return M_SS_EBP_ID;
		case 0xa6: return M_SS_ESI_ID;
		case 0xa7: return M_SS_EDI_ID;

		case 0xa8: return M_SS_4EBP_EAX_ID;
		case 0xa9: return M_SS_4EBP_ECX_ID;
		case 0xaa: return M_SS_4EBP_EDX_ID;
		case 0xab: return M_SS_4EBP_EBX_ID;
		case 0xac: return M_SS_4EBP_ESP_ID;
		case 0xad: return M_SS_5EBP_ID;
		case 0xae: return M_SS_4EBP_ESI_ID;
		case 0xaf: return M_SS_4EBP_EDI_ID;

		case 0xb0: return M_SS_4ESI_EAX_ID;
		case 0xb1: return M_SS_4ESI_ECX_ID;
		case 0xb2: return M_SS_4ESI_EDX_ID;
		case 0xb3: return M_SS_4ESI_EBX_ID;
		case 0xb4: return M_SS_4ESI_ESP_ID;
		case 0xb5: return M_SS_4ESI_EBP_ID;
		case 0xb6: return M_SS_5ESI_ID;
		case 0xb7: return M_SS_4ESI_EDI_ID;

		case 0xb8: return M_SS_4EDI_EAX_ID;
		case 0xb9: return M_SS_4EDI_ECX_ID;
		case 0xba: return M_SS_4EDI_EDX_ID;
		case 0xbb: return M_SS_4EDI_EBX_ID;
		case 0xbc: return M_SS_4EDI_ESP_ID;
		case 0xbd: return M_SS_4EDI_EBP_ID;
		case 0xbe: return M_SS_4EDI_ESI_ID;
		case 0xbf: return M_SS_5EDI_ID;

		case 0xc0: return M_SS_9EAX_ID;
		case 0xc1: return M_SS_8EAX_ECX_ID;
		case 0xc2: return M_SS_8EAX_EDX_ID;
		case 0xc3: return M_SS_8EAX_EBX_ID;
		case 0xc4: return M_SS_8EAX_ESP_ID;
		case 0xc5: return M_SS_8EAX_EBP_ID;
		case 0xc6: return M_SS_8EAX_ESI_ID;
		case 0xc7: return M_SS_8EAX_EDI_ID;

		case 0xc8: return M_SS_8ECX_EAX_ID;
		case 0xc9: return M_SS_9ECX_ID;
		case 0xca: return M_SS_8ECX_EDX_ID;
		case 0xcb: return M_SS_8ECX_EBX_ID;
		case 0xcc: return M_SS_8ECX_ESP_ID;
		case 0xcd: return M_SS_8ECX_EBP_ID;
		case 0xce: return M_SS_8ECX_ESI_ID;
		case 0xcf: return M_SS_8ECX_EDI_ID;

		case 0xd0: return M_SS_8EDX_EAX_ID;
		case 0xd1: return M_SS_8EDX_ECX_ID;
		case 0xd2: return M_SS_9EDX_ID;
		case 0xd3: return M_SS_8EDX_EBX_ID;
		case 0xd4: return M_SS_8EDX_ESP_ID;
		case 0xd5: return M_SS_8EDX_EBP_ID;
		case 0xd6: return M_SS_8EDX_ESI_ID;
		case 0xd7: return M_SS_8EDX_EDI_ID;

		case 0xd8: return M_SS_8EBX_EAX_ID;
		case 0xd9: return M_SS_8EBX_ECX_ID;
		case 0xda: return M_SS_8EBX_EDX_ID;
		case 0xdb: return M_SS_9EBX_ID;
		case 0xdc: return M_SS_8EBX_ESP_ID;
		case 0xdd: return M_SS_8EBX_EBP_ID;
		case 0xde: return M_SS_8EBX_ESI_ID;
		case 0xdf: return M_SS_8EBX_EDI_ID;

		case 0xe0: return M_SS_EAX_ID;
		case 0xe1: return M_SS_ECX_ID;
		case 0xe2: return M_SS_EDX_ID;
		case 0xe3: return M_SS_EBX_ID;
		case 0xe4: return M_SS_ESP_ID;
		case 0xe5: return M_SS_EBP_ID;
		case 0xe6: return M_SS_ESI_ID;
		case 0xe7: return M_SS_EDI_ID;

		case 0xe8: return M_SS_8EBP_EAX_ID;
		case 0xe9: return M_SS_8EBP_ECX_ID;
		case 0xea: return M_SS_8EBP_EDX_ID;
		case 0xeb: return M_SS_8EBP_EBX_ID;
		case 0xec: return M_SS_8EBP_ESP_ID;
		case 0xed: return M_SS_9EBP_ID;
		case 0xee: return M_SS_8EBP_ESI_ID;
		case 0xef: return M_SS_8EBP_EDI_ID;

		case 0xf0: return M_SS_8ESI_EAX_ID;
		case 0xf1: return M_SS_8ESI_ECX_ID;
		case 0xf2: return M_SS_8ESI_EDX_ID;
		case 0xf3: return M_SS_8ESI_EBX_ID;
		case 0xf4: return M_SS_8ESI_ESP_ID;
		case 0xf5: return M_SS_8ESI_EBP_ID;
		case 0xf6: return M_SS_9ESI_ID;
		case 0xf7: return M_SS_8ESI_EDI_ID;

		case 0xf8: return M_SS_8EDI_EAX_ID;
		case 0xf9: return M_SS_8EDI_ECX_ID;
		case 0xfa: return M_SS_8EDI_EDX_ID;
		case 0xfb: return M_SS_8EDI_EBX_ID;
		case 0xfc: return M_SS_8EDI_ESP_ID;
		case 0xfd: return M_SS_8EDI_EBP_ID;
		case 0xfe: return M_SS_8EDI_ESI_ID;
		case 0xff: return M_SS_9EDI_ID;
		}
	    }
	default: return decodeSIBUnusualSegments(prefices, modrm, sib);
	}
    }

    //This helps pull the bytecode length of the method below 32K so that MIDP devices will load it.
    private static int decodeSIBUnusualSegments(int prefices, int modrm, int sib)
    {
	switch (prefices & PREFICES_SG) {
	case PREFICES_FS:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_FS_2EAX;
		case 0x01: return M_FS_EAX_ECX;
		case 0x02: return M_FS_EAX_EDX;
		case 0x03: return M_FS_EAX_EBX;
		case 0x04: return M_FS_EAX_ESP;
		case 0x05: return M_FS_EAX_ID;
		case 0x06: return M_FS_EAX_ESI;
		case 0x07: return M_FS_EAX_EDI;

		case 0x08: return M_FS_EAX_ECX;
		case 0x09: return M_FS_2ECX;
		case 0x0a: return M_FS_ECX_EDX;
		case 0x0b: return M_FS_ECX_EBX;
		case 0x0c: return M_FS_ECX_ESP;
		case 0x0d: return M_FS_ECX_ID;
		case 0x0e: return M_FS_ECX_ESI;
		case 0x0f: return M_FS_ECX_EDI;

		case 0x10: return M_FS_EAX_EDX;
		case 0x11: return M_FS_ECX_EDX;
		case 0x12: return M_FS_2EDX;
		case 0x13: return M_FS_EDX_EBX;
		case 0x14: return M_FS_EDX_ESP;
		case 0x15: return M_FS_EDX_ID;
		case 0x16: return M_FS_EDX_ESI;
		case 0x17: return M_FS_EDX_EDI;

		case 0x18: return M_FS_EAX_EBX;
		case 0x19: return M_FS_ECX_EBX;
		case 0x1a: return M_FS_EDX_EBX;
		case 0x1b: return M_FS_2EBX;
		case 0x1c: return M_FS_EBX_ESP;
		case 0x1d: return M_FS_EBX_ID;
		case 0x1e: return M_FS_EBX_ESI;
		case 0x1f: return M_FS_EBX_EDI;

		case 0x20: return M_FS_EAX;
		case 0x21: return M_FS_ECX;
		case 0x22: return M_FS_EDX;
		case 0x23: return M_FS_EBX;
		case 0x24: return M_FS_ESP;
		case 0x25: return M_FS_ID;
		case 0x26: return M_FS_ESI;
		case 0x27: return M_FS_EDI;

		case 0x28: return M_FS_EAX_EBP;
		case 0x29: return M_FS_ECX_EBP;
		case 0x2a: return M_FS_EDX_EBP;
		case 0x2b: return M_FS_EBX_EBP;
		case 0x2c: return M_FS_EBP_ESP;
		case 0x2d: return M_FS_EBP_ID;
		case 0x2e: return M_FS_EBP_ESI;
		case 0x2f: return M_FS_EBP_EDI;

		case 0x30: return M_FS_EAX_ESI;
		case 0x31: return M_FS_ECX_ESI;
		case 0x32: return M_FS_EDX_ESI;
		case 0x33: return M_FS_EBX_ESI;
		case 0x34: return M_FS_ESI_ESP;
		case 0x35: return M_FS_ESI_ID;
		case 0x36: return M_FS_2ESI;
		case 0x37: return M_FS_ESI_EDI;

		case 0x38: return M_FS_EAX_EDI;
		case 0x39: return M_FS_ECX_EDI;
		case 0x3a: return M_FS_EDX_EDI;
		case 0x3b: return M_FS_EBX_EDI;
		case 0x3c: return M_FS_EDI_ESP;
		case 0x3d: return M_FS_EDI_ID;
		case 0x3e: return M_FS_ESI_EDI;
		case 0x3f: return M_FS_2EDI;

		case 0x40: return M_FS_3EAX;
		case 0x41: return M_FS_2EAX_ECX;
		case 0x42: return M_FS_2EAX_EDX;
		case 0x43: return M_FS_2EAX_EBX;
		case 0x44: return M_FS_2EAX_ESP;
		case 0x45: return M_FS_2EAX_ID;
		case 0x46: return M_FS_2EAX_ESI;
		case 0x47: return M_FS_2EAX_EDI;

		case 0x48: return M_FS_2ECX_EAX;
		case 0x49: return M_FS_3ECX;
		case 0x4a: return M_FS_2ECX_EDX;
		case 0x4b: return M_FS_2ECX_EBX;
		case 0x4c: return M_FS_2ECX_ESP;
		case 0x4d: return M_FS_2ECX_ID;
		case 0x4e: return M_FS_2ECX_ESI;
		case 0x4f: return M_FS_2ECX_EDI;

		case 0x50: return M_FS_2EDX_EAX;
		case 0x51: return M_FS_2EDX_ECX;
		case 0x52: return M_FS_3EDX;
		case 0x53: return M_FS_2EDX_EBX;
		case 0x54: return M_FS_2EDX_ESP;
		case 0x55: return M_FS_2EDX_ID;
		case 0x56: return M_FS_2EDX_ESI;
		case 0x57: return M_FS_2EDX_EDI;

		case 0x58: return M_FS_2EBX_EAX;
		case 0x59: return M_FS_2EBX_ECX;
		case 0x5a: return M_FS_2EBX_EDX;
		case 0x5b: return M_FS_3EBX;
		case 0x5c: return M_FS_2EBX_ESP;
		case 0x5d: return M_FS_2EBX_ID;
		case 0x5e: return M_FS_2EBX_ESI;
		case 0x5f: return M_FS_2EBX_EDI;

		case 0x60: return M_FS_EAX;
		case 0x61: return M_FS_ECX;
		case 0x62: return M_FS_EDX;
		case 0x63: return M_FS_EBX;
		case 0x64: return M_FS_ESP;
		case 0x65: return M_FS_ID;
		case 0x66: return M_FS_ESI;
		case 0x67: return M_FS_EDI;

		case 0x68: return M_FS_2EBP_EAX;
		case 0x69: return M_FS_2EBP_ECX;
		case 0x6a: return M_FS_2EBP_EDX;
		case 0x6b: return M_FS_2EBP_EBX;
		case 0x6c: return M_FS_2EBP_ESP;
		case 0x6d: return M_FS_2EBP_ID;
		case 0x6e: return M_FS_2EBP_ESI;
		case 0x6f: return M_FS_2EBP_EDI;

		case 0x70: return M_FS_2ESI_EAX;
		case 0x71: return M_FS_2ESI_ECX;
		case 0x72: return M_FS_2ESI_EDX;
		case 0x73: return M_FS_2ESI_EBX;
		case 0x74: return M_FS_2ESI_ESP;
		case 0x75: return M_FS_2ESI_ID;
		case 0x76: return M_FS_3ESI;
		case 0x77: return M_FS_2ESI_EDI;

		case 0x78: return M_FS_2EDI_EAX;
		case 0x79: return M_FS_2EDI_ECX;
		case 0x7a: return M_FS_2EDI_EDX;
		case 0x7b: return M_FS_2EDI_EBX;
		case 0x7c: return M_FS_2EDI_ESP;
		case 0x7d: return M_FS_2EDI_ID;
		case 0x7e: return M_FS_2EDI_ESI;
		case 0x7f: return M_FS_3EDI;

		case 0x80: return M_FS_5EAX;
		case 0x81: return M_FS_4EAX_ECX;
		case 0x82: return M_FS_4EAX_EDX;
		case 0x83: return M_FS_4EAX_EBX;
		case 0x84: return M_FS_4EAX_ESP;
		case 0x85: return M_FS_4EAX_ID;
		case 0x86: return M_FS_4EAX_ESI;
		case 0x87: return M_FS_4EAX_EDI;

		case 0x88: return M_FS_4ECX_EAX;
		case 0x89: return M_FS_5ECX;
		case 0x8a: return M_FS_4ECX_EDX;
		case 0x8b: return M_FS_4ECX_EBX;
		case 0x8c: return M_FS_4ECX_ESP;
		case 0x8d: return M_FS_4ECX_ID;
		case 0x8e: return M_FS_4ECX_ESI;
		case 0x8f: return M_FS_4ECX_EDI;

		case 0x90: return M_FS_4EDX_EAX;
		case 0x91: return M_FS_4EDX_ECX;
		case 0x92: return M_FS_5EDX;
		case 0x93: return M_FS_4EDX_EBX;
		case 0x94: return M_FS_4EDX_ESP;
		case 0x95: return M_FS_4EDX_ID;
		case 0x96: return M_FS_4EDX_ESI;
		case 0x97: return M_FS_4EDX_EDI;

		case 0x98: return M_FS_4EBX_EAX;
		case 0x99: return M_FS_4EBX_ECX;
		case 0x9a: return M_FS_4EBX_EDX;
		case 0x9b: return M_FS_5EBX;
		case 0x9c: return M_FS_4EBX_ESP;
		case 0x9d: return M_FS_4EBX_ID;
		case 0x9e: return M_FS_4EBX_ESI;
		case 0x9f: return M_FS_4EBX_EDI;

		case 0xa0: return M_FS_EAX;
		case 0xa1: return M_FS_ECX;
		case 0xa2: return M_FS_EDX;
		case 0xa3: return M_FS_EBX;
		case 0xa4: return M_FS_ESP;
		case 0xa5: return M_FS_ID;
		case 0xa6: return M_FS_ESI;
		case 0xa7: return M_FS_EDI;

		case 0xa8: return M_FS_4EBP_EAX;
		case 0xa9: return M_FS_4EBP_ECX;
		case 0xaa: return M_FS_4EBP_EDX;
		case 0xab: return M_FS_4EBP_EBX;
		case 0xac: return M_FS_4EBP_ESP;
		case 0xad: return M_FS_4EBP_ID;
		case 0xae: return M_FS_4EBP_ESI;
		case 0xaf: return M_FS_4EBP_EDI;

		case 0xb0: return M_FS_4ESI_EAX;
		case 0xb1: return M_FS_4ESI_ECX;
		case 0xb2: return M_FS_4ESI_EDX;
		case 0xb3: return M_FS_4ESI_EBX;
		case 0xb4: return M_FS_4ESI_ESP;
		case 0xb5: return M_FS_4ESI_ID;
		case 0xb6: return M_FS_5ESI;
		case 0xb7: return M_FS_4ESI_EDI;

		case 0xb8: return M_FS_4EDI_EAX;
		case 0xb9: return M_FS_4EDI_ECX;
		case 0xba: return M_FS_4EDI_EDX;
		case 0xbb: return M_FS_4EDI_EBX;
		case 0xbc: return M_FS_4EDI_ESP;
		case 0xbd: return M_FS_4EDI_ID;
		case 0xbe: return M_FS_4EDI_ESI;
		case 0xbf: return M_FS_5EDI;

		case 0xc0: return M_FS_9EAX;
		case 0xc1: return M_FS_8EAX_ECX;
		case 0xc2: return M_FS_8EAX_EDX;
		case 0xc3: return M_FS_8EAX_EBX;
		case 0xc4: return M_FS_8EAX_ESP;
		case 0xc5: return M_FS_8EAX_ID;
		case 0xc6: return M_FS_8EAX_ESI;
		case 0xc7: return M_FS_8EAX_EDI;

		case 0xc8: return M_FS_8ECX_EAX;
		case 0xc9: return M_FS_9ECX;
		case 0xca: return M_FS_8ECX_EDX;
		case 0xcb: return M_FS_8ECX_EBX;
		case 0xcc: return M_FS_8ECX_ESP;
		case 0xcd: return M_FS_8ECX_ID;
		case 0xce: return M_FS_8ECX_ESI;
		case 0xcf: return M_FS_8ECX_EDI;

		case 0xd0: return M_FS_8EDX_EAX;
		case 0xd1: return M_FS_8EDX_ECX;
		case 0xd2: return M_FS_9EDX;
		case 0xd3: return M_FS_8EDX_EBX;
		case 0xd4: return M_FS_8EDX_ESP;
		case 0xd5: return M_FS_8EDX_ID;
		case 0xd6: return M_FS_8EDX_ESI;
		case 0xd7: return M_FS_8EDX_EDI;

		case 0xd8: return M_FS_8EBX_EAX;
		case 0xd9: return M_FS_8EBX_ECX;
		case 0xda: return M_FS_8EBX_EDX;
		case 0xdb: return M_FS_9EBX;
		case 0xdc: return M_FS_8EBX_ESP;
		case 0xdd: return M_FS_8EBX_ID;
		case 0xde: return M_FS_8EBX_ESI;
		case 0xdf: return M_FS_8EBX_EDI;

		case 0xe0: return M_FS_EAX;
		case 0xe1: return M_FS_ECX;
		case 0xe2: return M_FS_EDX;
		case 0xe3: return M_FS_EBX;
		case 0xe4: return M_FS_ESP;
		case 0xe5: return M_FS_ID;
		case 0xe6: return M_FS_ESI;
		case 0xe7: return M_FS_EDI;

		case 0xe8: return M_FS_8EBP_EAX;
		case 0xe9: return M_FS_8EBP_ECX;
		case 0xea: return M_FS_8EBP_EDX;
		case 0xeb: return M_FS_8EBP_EBX;
		case 0xec: return M_FS_8EBP_ESP;
		case 0xed: return M_FS_8EBP_ID;
		case 0xee: return M_FS_8EBP_ESI;
		case 0xef: return M_FS_8EBP_EDI;

		case 0xf0: return M_FS_8ESI_EAX;
		case 0xf1: return M_FS_8ESI_ECX;
		case 0xf2: return M_FS_8ESI_EDX;
		case 0xf3: return M_FS_8ESI_EBX;
		case 0xf4: return M_FS_8ESI_ESP;
		case 0xf5: return M_FS_8ESI_ID;
		case 0xf6: return M_FS_9ESI;
		case 0xf7: return M_FS_8ESI_EDI;

		case 0xf8: return M_FS_8EDI_EAX;
		case 0xf9: return M_FS_8EDI_ECX;
		case 0xfa: return M_FS_8EDI_EDX;
		case 0xfb: return M_FS_8EDI_EBX;
		case 0xfc: return M_FS_8EDI_ESP;
		case 0xfd: return M_FS_8EDI_ID;
		case 0xfe: return M_FS_8EDI_ESI;
		case 0xff: return M_FS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_FS_2EAX_IB;
		case 0x01: return M_FS_EAX_ECX_IB;
		case 0x02: return M_FS_EAX_EDX_IB;
		case 0x03: return M_FS_EAX_EBX_IB;
		case 0x04: return M_FS_EAX_ESP_IB;
		case 0x05: return M_FS_EAX_EBP_IB;
		case 0x06: return M_FS_EAX_ESI_IB;
		case 0x07: return M_FS_EAX_EDI_IB;

		case 0x08: return M_FS_EAX_ECX_IB;
		case 0x09: return M_FS_2ECX_IB;
		case 0x0a: return M_FS_ECX_EDX_IB;
		case 0x0b: return M_FS_ECX_EBX_IB;
		case 0x0c: return M_FS_ECX_ESP_IB;
		case 0x0d: return M_FS_ECX_EBP_IB;
		case 0x0e: return M_FS_ECX_ESI_IB;
		case 0x0f: return M_FS_ECX_EDI_IB;

		case 0x10: return M_FS_EAX_EDX_IB;
		case 0x11: return M_FS_ECX_EDX_IB;
		case 0x12: return M_FS_2EDX_IB;
		case 0x13: return M_FS_EDX_EBX_IB;
		case 0x14: return M_FS_EDX_ESP_IB;
		case 0x15: return M_FS_EDX_EBP_IB;
		case 0x16: return M_FS_EDX_ESI_IB;
		case 0x17: return M_FS_EDX_EDI_IB;

		case 0x18: return M_FS_EAX_EBX_IB;
		case 0x19: return M_FS_ECX_EBX_IB;
		case 0x1a: return M_FS_EDX_EBX_IB;
		case 0x1b: return M_FS_2EBX_IB;
		case 0x1c: return M_FS_EBX_ESP_IB;
		case 0x1d: return M_FS_EBX_EBP_IB;
		case 0x1e: return M_FS_EBX_ESI_IB;
		case 0x1f: return M_FS_EBX_EDI_IB;

		case 0x20: return M_FS_EAX_IB;
		case 0x21: return M_FS_ECX_IB;
		case 0x22: return M_FS_EDX_IB;
		case 0x23: return M_FS_EBX_IB;
		case 0x24: return M_FS_ESP_IB;
		case 0x25: return M_FS_EBP_IB;
		case 0x26: return M_FS_ESI_IB;
		case 0x27: return M_FS_EDI_IB;

		case 0x28: return M_FS_EAX_EBP_IB;
		case 0x29: return M_FS_ECX_EBP_IB;
		case 0x2a: return M_FS_EDX_EBP_IB;
		case 0x2b: return M_FS_EBX_EBP_IB;
		case 0x2c: return M_FS_EBP_ESP_IB;
		case 0x2d: return M_FS_2EBP_IB;
		case 0x2e: return M_FS_EBP_ESI_IB;
		case 0x2f: return M_FS_EBP_EDI_IB;

		case 0x30: return M_FS_EAX_ESI_IB;
		case 0x31: return M_FS_ECX_ESI_IB;
		case 0x32: return M_FS_EDX_ESI_IB;
		case 0x33: return M_FS_EBX_ESI_IB;
		case 0x34: return M_FS_ESI_ESP_IB;
		case 0x35: return M_FS_EBP_ESI_IB;
		case 0x36: return M_FS_2ESI_IB;
		case 0x37: return M_FS_ESI_EDI_IB;

		case 0x38: return M_FS_EAX_EDI_IB;
		case 0x39: return M_FS_ECX_EDI_IB;
		case 0x3a: return M_FS_EDX_EDI_IB;
		case 0x3b: return M_FS_EBX_EDI_IB;
		case 0x3c: return M_FS_EDI_ESP_IB;
		case 0x3d: return M_FS_EBP_EDI_IB;
		case 0x3e: return M_FS_ESI_EDI_IB;
		case 0x3f: return M_FS_2EDI_IB;

		case 0x40: return M_FS_3EAX_IB;
		case 0x41: return M_FS_2EAX_ECX_IB;
		case 0x42: return M_FS_2EAX_EDX_IB;
		case 0x43: return M_FS_2EAX_EBX_IB;
		case 0x44: return M_FS_2EAX_ESP_IB;
		case 0x45: return M_FS_2EAX_EBP_IB;
		case 0x46: return M_FS_2EAX_ESI_IB;
		case 0x47: return M_FS_2EAX_EDI_IB;

		case 0x48: return M_FS_2ECX_EAX_IB;
		case 0x49: return M_FS_3ECX_IB;
		case 0x4a: return M_FS_2ECX_EDX_IB;
		case 0x4b: return M_FS_2ECX_EBX_IB;
		case 0x4c: return M_FS_2ECX_ESP_IB;
		case 0x4d: return M_FS_2ECX_EBP_IB;
		case 0x4e: return M_FS_2ECX_ESI_IB;
		case 0x4f: return M_FS_2ECX_EDI_IB;

		case 0x50: return M_FS_2EDX_EAX_IB;
		case 0x51: return M_FS_2EDX_ECX_IB;
		case 0x52: return M_FS_3EDX_IB;
		case 0x53: return M_FS_2EDX_EBX_IB;
		case 0x54: return M_FS_2EDX_ESP_IB;
		case 0x55: return M_FS_2EDX_EBP_IB;
		case 0x56: return M_FS_2EDX_ESI_IB;
		case 0x57: return M_FS_2EDX_EDI_IB;

		case 0x58: return M_FS_2EBX_EAX_IB;
		case 0x59: return M_FS_2EBX_ECX_IB;
		case 0x5a: return M_FS_2EBX_EDX_IB;
		case 0x5b: return M_FS_3EBX_IB;
		case 0x5c: return M_FS_2EBX_ESP_IB;
		case 0x5d: return M_FS_2EBX_EBP_IB;
		case 0x5e: return M_FS_2EBX_ESI_IB;
		case 0x5f: return M_FS_2EBX_EDI_IB;

		case 0x60: return M_FS_EAX_IB;
		case 0x61: return M_FS_ECX_IB;
		case 0x62: return M_FS_EDX_IB;
		case 0x63: return M_FS_EBX_IB;
		case 0x64: return M_FS_ESP_IB;
		case 0x65: return M_FS_EBP_IB;
		case 0x66: return M_FS_ESI_IB;
		case 0x67: return M_FS_EDI_IB;

		case 0x68: return M_FS_2EBP_EAX_IB;
		case 0x69: return M_FS_2EBP_ECX_IB;
		case 0x6a: return M_FS_2EBP_EDX_IB;
		case 0x6b: return M_FS_2EBP_EBX_IB;
		case 0x6c: return M_FS_2EBP_ESP_IB;
		case 0x6d: return M_FS_3EBP_IB;
		case 0x6e: return M_FS_2EBP_ESI_IB;
		case 0x6f: return M_FS_2EBP_EDI_IB;

		case 0x70: return M_FS_2ESI_EAX_IB;
		case 0x71: return M_FS_2ESI_ECX_IB;
		case 0x72: return M_FS_2ESI_EDX_IB;
		case 0x73: return M_FS_2ESI_EBX_IB;
		case 0x74: return M_FS_2ESI_ESP_IB;
		case 0x75: return M_FS_2ESI_EBP_IB;
		case 0x76: return M_FS_3ESI_IB;
		case 0x77: return M_FS_2ESI_EDI_IB;

		case 0x78: return M_FS_2EDI_EAX_IB;
		case 0x79: return M_FS_2EDI_ECX_IB;
		case 0x7a: return M_FS_2EDI_EDX_IB;
		case 0x7b: return M_FS_2EDI_EBX_IB;
		case 0x7c: return M_FS_2EDI_ESP_IB;
		case 0x7d: return M_FS_2EDI_EBP_IB;
		case 0x7e: return M_FS_2EDI_ESI_IB;
		case 0x7f: return M_FS_3EDI_IB;

		case 0x80: return M_FS_5EAX_IB;
		case 0x81: return M_FS_4EAX_ECX_IB;
		case 0x82: return M_FS_4EAX_EDX_IB;
		case 0x83: return M_FS_4EAX_EBX_IB;
		case 0x84: return M_FS_4EAX_ESP_IB;
		case 0x85: return M_FS_4EAX_EBP_IB;
		case 0x86: return M_FS_4EAX_ESI_IB;
		case 0x87: return M_FS_4EAX_EDI_IB;

		case 0x88: return M_FS_4ECX_EAX_IB;
		case 0x89: return M_FS_5ECX_IB;
		case 0x8a: return M_FS_4ECX_EDX_IB;
		case 0x8b: return M_FS_4ECX_EBX_IB;
		case 0x8c: return M_FS_4ECX_ESP_IB;
		case 0x8d: return M_FS_4ECX_EBP_IB;
		case 0x8e: return M_FS_4ECX_ESI_IB;
		case 0x8f: return M_FS_4ECX_EDI_IB;

		case 0x90: return M_FS_4EDX_EAX_IB;
		case 0x91: return M_FS_4EDX_ECX_IB;
		case 0x92: return M_FS_5EDX_IB;
		case 0x93: return M_FS_4EDX_EBX_IB;
		case 0x94: return M_FS_4EDX_ESP_IB;
		case 0x95: return M_FS_4EDX_EBP_IB;
		case 0x96: return M_FS_4EDX_ESI_IB;
		case 0x97: return M_FS_4EDX_EDI_IB;

		case 0x98: return M_FS_4EBX_EAX_IB;
		case 0x99: return M_FS_4EBX_ECX_IB;
		case 0x9a: return M_FS_4EBX_EDX_IB;
		case 0x9b: return M_FS_5EBX_IB;
		case 0x9c: return M_FS_4EBX_ESP_IB;
		case 0x9d: return M_FS_4EBX_EBP_IB;
		case 0x9e: return M_FS_4EBX_ESI_IB;
		case 0x9f: return M_FS_4EBX_EDI_IB;

		case 0xa0: return M_FS_EAX_IB;
		case 0xa1: return M_FS_ECX_IB;
		case 0xa2: return M_FS_EDX_IB;
		case 0xa3: return M_FS_EBX_IB;
		case 0xa4: return M_FS_ESP_IB;
		case 0xa5: return M_FS_EBP_IB;
		case 0xa6: return M_FS_ESI_IB;
		case 0xa7: return M_FS_EDI_IB;

		case 0xa8: return M_FS_4EBP_EAX_IB;
		case 0xa9: return M_FS_4EBP_ECX_IB;
		case 0xaa: return M_FS_4EBP_EDX_IB;
		case 0xab: return M_FS_4EBP_EBX_IB;
		case 0xac: return M_FS_4EBP_ESP_IB;
		case 0xad: return M_FS_5EBP_IB;
		case 0xae: return M_FS_4EBP_ESI_IB;
		case 0xaf: return M_FS_4EBP_EDI_IB;

		case 0xb0: return M_FS_4ESI_EAX_IB;
		case 0xb1: return M_FS_4ESI_ECX_IB;
		case 0xb2: return M_FS_4ESI_EDX_IB;
		case 0xb3: return M_FS_4ESI_EBX_IB;
		case 0xb4: return M_FS_4ESI_ESP_IB;
		case 0xb5: return M_FS_4ESI_EBP_IB;
		case 0xb6: return M_FS_5ESI_IB;
		case 0xb7: return M_FS_4ESI_EDI_IB;

		case 0xb8: return M_FS_4EDI_EAX_IB;
		case 0xb9: return M_FS_4EDI_ECX_IB;
		case 0xba: return M_FS_4EDI_EDX_IB;
		case 0xbb: return M_FS_4EDI_EBX_IB;
		case 0xbc: return M_FS_4EDI_ESP_IB;
		case 0xbd: return M_FS_4EDI_EBP_IB;
		case 0xbe: return M_FS_4EDI_ESI_IB;
		case 0xbf: return M_FS_5EDI_IB;

		case 0xc0: return M_FS_9EAX_IB;
		case 0xc1: return M_FS_8EAX_ECX_IB;
		case 0xc2: return M_FS_8EAX_EDX_IB;
		case 0xc3: return M_FS_8EAX_EBX_IB;
		case 0xc4: return M_FS_8EAX_ESP_IB;
		case 0xc5: return M_FS_8EAX_EBP_IB;
		case 0xc6: return M_FS_8EAX_ESI_IB;
		case 0xc7: return M_FS_8EAX_EDI_IB;

		case 0xc8: return M_FS_8ECX_EAX_IB;
		case 0xc9: return M_FS_9ECX_IB;
		case 0xca: return M_FS_8ECX_EDX_IB;
		case 0xcb: return M_FS_8ECX_EBX_IB;
		case 0xcc: return M_FS_8ECX_ESP_IB;
		case 0xcd: return M_FS_8ECX_EBP_IB;
		case 0xce: return M_FS_8ECX_ESI_IB;
		case 0xcf: return M_FS_8ECX_EDI_IB;

		case 0xd0: return M_FS_8EDX_EAX_IB;
		case 0xd1: return M_FS_8EDX_ECX_IB;
		case 0xd2: return M_FS_9EDX_IB;
		case 0xd3: return M_FS_8EDX_EBX_IB;
		case 0xd4: return M_FS_8EDX_ESP_IB;
		case 0xd5: return M_FS_8EDX_EBP_IB;
		case 0xd6: return M_FS_8EDX_ESI_IB;
		case 0xd7: return M_FS_8EDX_EDI_IB;

		case 0xd8: return M_FS_8EBX_EAX_IB;
		case 0xd9: return M_FS_8EBX_ECX_IB;
		case 0xda: return M_FS_8EBX_EDX_IB;
		case 0xdb: return M_FS_9EBX_IB;
		case 0xdc: return M_FS_8EBX_ESP_IB;
		case 0xdd: return M_FS_8EBX_EBP_IB;
		case 0xde: return M_FS_8EBX_ESI_IB;
		case 0xdf: return M_FS_8EBX_EDI_IB;

		case 0xe0: return M_FS_EAX_IB;
		case 0xe1: return M_FS_ECX_IB;
		case 0xe2: return M_FS_EDX_IB;
		case 0xe3: return M_FS_EBX_IB;
		case 0xe4: return M_FS_ESP_IB;
		case 0xe5: return M_FS_EBP_IB;
		case 0xe6: return M_FS_ESI_IB;
		case 0xe7: return M_FS_EDI_IB;

		case 0xe8: return M_FS_8EBP_EAX_IB;
		case 0xe9: return M_FS_8EBP_ECX_IB;
		case 0xea: return M_FS_8EBP_EDX_IB;
		case 0xeb: return M_FS_8EBP_EBX_IB;
		case 0xec: return M_FS_8EBP_ESP_IB;
		case 0xed: return M_FS_9EBP_IB;
		case 0xee: return M_FS_8EBP_ESI_IB;
		case 0xef: return M_FS_8EBP_EDI_IB;

		case 0xf0: return M_FS_8ESI_EAX_IB;
		case 0xf1: return M_FS_8ESI_ECX_IB;
		case 0xf2: return M_FS_8ESI_EDX_IB;
		case 0xf3: return M_FS_8ESI_EBX_IB;
		case 0xf4: return M_FS_8ESI_ESP_IB;
		case 0xf5: return M_FS_8ESI_EBP_IB;
		case 0xf6: return M_FS_9ESI_IB;
		case 0xf7: return M_FS_8ESI_EDI_IB;

		case 0xf8: return M_FS_8EDI_EAX_IB;
		case 0xf9: return M_FS_8EDI_ECX_IB;
		case 0xfa: return M_FS_8EDI_EDX_IB;
		case 0xfb: return M_FS_8EDI_EBX_IB;
		case 0xfc: return M_FS_8EDI_ESP_IB;
		case 0xfd: return M_FS_8EDI_EBP_IB;
		case 0xfe: return M_FS_8EDI_ESI_IB;
		case 0xff: return M_FS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_FS_2EAX_ID;
		case 0x01: return M_FS_EAX_ECX_ID;
		case 0x02: return M_FS_EAX_EDX_ID;
		case 0x03: return M_FS_EAX_EBX_ID;
		case 0x04: return M_FS_EAX_ESP_ID;
		case 0x05: return M_FS_EAX_EBP_ID;
		case 0x06: return M_FS_EAX_ESI_ID;
		case 0x07: return M_FS_EAX_EDI_ID;

		case 0x08: return M_FS_EAX_ECX_ID;
		case 0x09: return M_FS_2ECX_ID;
		case 0x0a: return M_FS_ECX_EDX_ID;
		case 0x0b: return M_FS_ECX_EBX_ID;
		case 0x0c: return M_FS_ECX_ESP_ID;
		case 0x0d: return M_FS_ECX_EBP_ID;
		case 0x0e: return M_FS_ECX_ESI_ID;
		case 0x0f: return M_FS_ECX_EDI_ID;

		case 0x10: return M_FS_EAX_EDX_ID;
		case 0x11: return M_FS_ECX_EDX_ID;
		case 0x12: return M_FS_2EDX_ID;
		case 0x13: return M_FS_EDX_EBX_ID;
		case 0x14: return M_FS_EDX_ESP_ID;
		case 0x15: return M_FS_EDX_EBP_ID;
		case 0x16: return M_FS_EDX_ESI_ID;
		case 0x17: return M_FS_EDX_EDI_ID;

		case 0x18: return M_FS_EAX_EBX_ID;
		case 0x19: return M_FS_ECX_EBX_ID;
		case 0x1a: return M_FS_EDX_EBX_ID;
		case 0x1b: return M_FS_2EBX_ID;
		case 0x1c: return M_FS_EBX_ESP_ID;
		case 0x1d: return M_FS_EBX_EBP_ID;
		case 0x1e: return M_FS_EBX_ESI_ID;
		case 0x1f: return M_FS_EBX_EDI_ID;

		case 0x20: return M_FS_EAX_ID;
		case 0x21: return M_FS_ECX_ID;
		case 0x22: return M_FS_EDX_ID;
		case 0x23: return M_FS_EBX_ID;
		case 0x24: return M_FS_ESP_ID;
		case 0x25: return M_FS_EBP_ID;
		case 0x26: return M_FS_ESI_ID;
		case 0x27: return M_FS_EDI_ID;

		case 0x28: return M_FS_EAX_EBP_ID;
		case 0x29: return M_FS_ECX_EBP_ID;
		case 0x2a: return M_FS_EDX_EBP_ID;
		case 0x2b: return M_FS_EBX_EBP_ID;
		case 0x2c: return M_FS_EBP_ESP_ID;
		case 0x2d: return M_FS_2EBP_ID;
		case 0x2e: return M_FS_EBP_ESI_ID;
		case 0x2f: return M_FS_EBP_EDI_ID;

		case 0x30: return M_FS_EAX_ESI_ID;
		case 0x31: return M_FS_ECX_ESI_ID;
		case 0x32: return M_FS_EDX_ESI_ID;
		case 0x33: return M_FS_EBX_ESI_ID;
		case 0x34: return M_FS_ESI_ESP_ID;
		case 0x35: return M_FS_EBP_ESI_ID;
		case 0x36: return M_FS_2ESI_ID;
		case 0x37: return M_FS_ESI_EDI_ID;

		case 0x38: return M_FS_EAX_EDI_ID;
		case 0x39: return M_FS_ECX_EDI_ID;
		case 0x3a: return M_FS_EDX_EDI_ID;
		case 0x3b: return M_FS_EBX_EDI_ID;
		case 0x3c: return M_FS_EDI_ESP_ID;
		case 0x3d: return M_FS_EBP_EDI_ID;
		case 0x3e: return M_FS_ESI_EDI_ID;
		case 0x3f: return M_FS_2EDI_ID;

		case 0x40: return M_FS_3EAX_ID;
		case 0x41: return M_FS_2EAX_ECX_ID;
		case 0x42: return M_FS_2EAX_EDX_ID;
		case 0x43: return M_FS_2EAX_EBX_ID;
		case 0x44: return M_FS_2EAX_ESP_ID;
		case 0x45: return M_FS_2EAX_EBP_ID;
		case 0x46: return M_FS_2EAX_ESI_ID;
		case 0x47: return M_FS_2EAX_EDI_ID;

		case 0x48: return M_FS_2ECX_EAX_ID;
		case 0x49: return M_FS_3ECX_ID;
		case 0x4a: return M_FS_2ECX_EDX_ID;
		case 0x4b: return M_FS_2ECX_EBX_ID;
		case 0x4c: return M_FS_2ECX_ESP_ID;
		case 0x4d: return M_FS_2ECX_EBP_ID;
		case 0x4e: return M_FS_2ECX_ESI_ID;
		case 0x4f: return M_FS_2ECX_EDI_ID;

		case 0x50: return M_FS_2EDX_EAX_ID;
		case 0x51: return M_FS_2EDX_ECX_ID;
		case 0x52: return M_FS_3EDX_ID;
		case 0x53: return M_FS_2EDX_EBX_ID;
		case 0x54: return M_FS_2EDX_ESP_ID;
		case 0x55: return M_FS_2EDX_EBP_ID;
		case 0x56: return M_FS_2EDX_ESI_ID;
		case 0x57: return M_FS_2EDX_EDI_ID;

		case 0x58: return M_FS_2EBX_EAX_ID;
		case 0x59: return M_FS_2EBX_ECX_ID;
		case 0x5a: return M_FS_2EBX_EDX_ID;
		case 0x5b: return M_FS_3EBX_ID;
		case 0x5c: return M_FS_2EBX_ESP_ID;
		case 0x5d: return M_FS_2EBX_EBP_ID;
		case 0x5e: return M_FS_2EBX_ESI_ID;
		case 0x5f: return M_FS_2EBX_EDI_ID;

		case 0x60: return M_FS_EAX_ID;
		case 0x61: return M_FS_ECX_ID;
		case 0x62: return M_FS_EDX_ID;
		case 0x63: return M_FS_EBX_ID;
		case 0x64: return M_FS_ESP_ID;
		case 0x65: return M_FS_EBP_ID;
		case 0x66: return M_FS_ESI_ID;
		case 0x67: return M_FS_EDI_ID;

		case 0x68: return M_FS_2EBP_EAX_ID;
		case 0x69: return M_FS_2EBP_ECX_ID;
		case 0x6a: return M_FS_2EBP_EDX_ID;
		case 0x6b: return M_FS_2EBP_EBX_ID;
		case 0x6c: return M_FS_2EBP_ESP_ID;
		case 0x6d: return M_FS_3EBP_ID;
		case 0x6e: return M_FS_2EBP_ESI_ID;
		case 0x6f: return M_FS_2EBP_EDI_ID;

		case 0x70: return M_FS_2ESI_EAX_ID;
		case 0x71: return M_FS_2ESI_ECX_ID;
		case 0x72: return M_FS_2ESI_EDX_ID;
		case 0x73: return M_FS_2ESI_EBX_ID;
		case 0x74: return M_FS_2ESI_ESP_ID;
		case 0x75: return M_FS_2ESI_EBP_ID;
		case 0x76: return M_FS_3ESI_ID;
		case 0x77: return M_FS_2ESI_EDI_ID;

		case 0x78: return M_FS_2EDI_EAX_ID;
		case 0x79: return M_FS_2EDI_ECX_ID;
		case 0x7a: return M_FS_2EDI_EDX_ID;
		case 0x7b: return M_FS_2EDI_EBX_ID;
		case 0x7c: return M_FS_2EDI_ESP_ID;
		case 0x7d: return M_FS_2EDI_EBP_ID;
		case 0x7e: return M_FS_2EDI_ESI_ID;
		case 0x7f: return M_FS_3EDI_ID;

		case 0x80: return M_FS_5EAX_ID;
		case 0x81: return M_FS_4EAX_ECX_ID;
		case 0x82: return M_FS_4EAX_EDX_ID;
		case 0x83: return M_FS_4EAX_EBX_ID;
		case 0x84: return M_FS_4EAX_ESP_ID;
		case 0x85: return M_FS_4EAX_EBP_ID;
		case 0x86: return M_FS_4EAX_ESI_ID;
		case 0x87: return M_FS_4EAX_EDI_ID;

		case 0x88: return M_FS_4ECX_EAX_ID;
		case 0x89: return M_FS_5ECX_ID;
		case 0x8a: return M_FS_4ECX_EDX_ID;
		case 0x8b: return M_FS_4ECX_EBX_ID;
		case 0x8c: return M_FS_4ECX_ESP_ID;
		case 0x8d: return M_FS_4ECX_EBP_ID;
		case 0x8e: return M_FS_4ECX_ESI_ID;
		case 0x8f: return M_FS_4ECX_EDI_ID;

		case 0x90: return M_FS_4EDX_EAX_ID;
		case 0x91: return M_FS_4EDX_ECX_ID;
		case 0x92: return M_FS_5EDX_ID;
		case 0x93: return M_FS_4EDX_EBX_ID;
		case 0x94: return M_FS_4EDX_ESP_ID;
		case 0x95: return M_FS_4EDX_EBP_ID;
		case 0x96: return M_FS_4EDX_ESI_ID;
		case 0x97: return M_FS_4EDX_EDI_ID;

		case 0x98: return M_FS_4EBX_EAX_ID;
		case 0x99: return M_FS_4EBX_ECX_ID;
		case 0x9a: return M_FS_4EBX_EDX_ID;
		case 0x9b: return M_FS_5EBX_ID;
		case 0x9c: return M_FS_4EBX_ESP_ID;
		case 0x9d: return M_FS_4EBX_EBP_ID;
		case 0x9e: return M_FS_4EBX_ESI_ID;
		case 0x9f: return M_FS_4EBX_EDI_ID;

		case 0xa0: return M_FS_EAX_ID;
		case 0xa1: return M_FS_ECX_ID;
		case 0xa2: return M_FS_EDX_ID;
		case 0xa3: return M_FS_EBX_ID;
		case 0xa4: return M_FS_ESP_ID;
		case 0xa5: return M_FS_EBP_ID;
		case 0xa6: return M_FS_ESI_ID;
		case 0xa7: return M_FS_EDI_ID;

		case 0xa8: return M_FS_4EBP_EAX_ID;
		case 0xa9: return M_FS_4EBP_ECX_ID;
		case 0xaa: return M_FS_4EBP_EDX_ID;
		case 0xab: return M_FS_4EBP_EBX_ID;
		case 0xac: return M_FS_4EBP_ESP_ID;
		case 0xad: return M_FS_5EBP_ID;
		case 0xae: return M_FS_4EBP_ESI_ID;
		case 0xaf: return M_FS_4EBP_EDI_ID;

		case 0xb0: return M_FS_4ESI_EAX_ID;
		case 0xb1: return M_FS_4ESI_ECX_ID;
		case 0xb2: return M_FS_4ESI_EDX_ID;
		case 0xb3: return M_FS_4ESI_EBX_ID;
		case 0xb4: return M_FS_4ESI_ESP_ID;
		case 0xb5: return M_FS_4ESI_EBP_ID;
		case 0xb6: return M_FS_5ESI_ID;
		case 0xb7: return M_FS_4ESI_EDI_ID;

		case 0xb8: return M_FS_4EDI_EAX_ID;
		case 0xb9: return M_FS_4EDI_ECX_ID;
		case 0xba: return M_FS_4EDI_EDX_ID;
		case 0xbb: return M_FS_4EDI_EBX_ID;
		case 0xbc: return M_FS_4EDI_ESP_ID;
		case 0xbd: return M_FS_4EDI_EBP_ID;
		case 0xbe: return M_FS_4EDI_ESI_ID;
		case 0xbf: return M_FS_5EDI_ID;

		case 0xc0: return M_FS_9EAX_ID;
		case 0xc1: return M_FS_8EAX_ECX_ID;
		case 0xc2: return M_FS_8EAX_EDX_ID;
		case 0xc3: return M_FS_8EAX_EBX_ID;
		case 0xc4: return M_FS_8EAX_ESP_ID;
		case 0xc5: return M_FS_8EAX_EBP_ID;
		case 0xc6: return M_FS_8EAX_ESI_ID;
		case 0xc7: return M_FS_8EAX_EDI_ID;

		case 0xc8: return M_FS_8ECX_EAX_ID;
		case 0xc9: return M_FS_9ECX_ID;
		case 0xca: return M_FS_8ECX_EDX_ID;
		case 0xcb: return M_FS_8ECX_EBX_ID;
		case 0xcc: return M_FS_8ECX_ESP_ID;
		case 0xcd: return M_FS_8ECX_EBP_ID;
		case 0xce: return M_FS_8ECX_ESI_ID;
		case 0xcf: return M_FS_8ECX_EDI_ID;

		case 0xd0: return M_FS_8EDX_EAX_ID;
		case 0xd1: return M_FS_8EDX_ECX_ID;
		case 0xd2: return M_FS_9EDX_ID;
		case 0xd3: return M_FS_8EDX_EBX_ID;
		case 0xd4: return M_FS_8EDX_ESP_ID;
		case 0xd5: return M_FS_8EDX_EBP_ID;
		case 0xd6: return M_FS_8EDX_ESI_ID;
		case 0xd7: return M_FS_8EDX_EDI_ID;

		case 0xd8: return M_FS_8EBX_EAX_ID;
		case 0xd9: return M_FS_8EBX_ECX_ID;
		case 0xda: return M_FS_8EBX_EDX_ID;
		case 0xdb: return M_FS_9EBX_ID;
		case 0xdc: return M_FS_8EBX_ESP_ID;
		case 0xdd: return M_FS_8EBX_EBP_ID;
		case 0xde: return M_FS_8EBX_ESI_ID;
		case 0xdf: return M_FS_8EBX_EDI_ID;

		case 0xe0: return M_FS_EAX_ID;
		case 0xe1: return M_FS_ECX_ID;
		case 0xe2: return M_FS_EDX_ID;
		case 0xe3: return M_FS_EBX_ID;
		case 0xe4: return M_FS_ESP_ID;
		case 0xe5: return M_FS_EBP_ID;
		case 0xe6: return M_FS_ESI_ID;
		case 0xe7: return M_FS_EDI_ID;

		case 0xe8: return M_FS_8EBP_EAX_ID;
		case 0xe9: return M_FS_8EBP_ECX_ID;
		case 0xea: return M_FS_8EBP_EDX_ID;
		case 0xeb: return M_FS_8EBP_EBX_ID;
		case 0xec: return M_FS_8EBP_ESP_ID;
		case 0xed: return M_FS_9EBP_ID;
		case 0xee: return M_FS_8EBP_ESI_ID;
		case 0xef: return M_FS_8EBP_EDI_ID;

		case 0xf0: return M_FS_8ESI_EAX_ID;
		case 0xf1: return M_FS_8ESI_ECX_ID;
		case 0xf2: return M_FS_8ESI_EDX_ID;
		case 0xf3: return M_FS_8ESI_EBX_ID;
		case 0xf4: return M_FS_8ESI_ESP_ID;
		case 0xf5: return M_FS_8ESI_EBP_ID;
		case 0xf6: return M_FS_9ESI_ID;
		case 0xf7: return M_FS_8ESI_EDI_ID;

		case 0xf8: return M_FS_8EDI_EAX_ID;
		case 0xf9: return M_FS_8EDI_ECX_ID;
		case 0xfa: return M_FS_8EDI_EDX_ID;
		case 0xfb: return M_FS_8EDI_EBX_ID;
		case 0xfc: return M_FS_8EDI_ESP_ID;
		case 0xfd: return M_FS_8EDI_EBP_ID;
		case 0xfe: return M_FS_8EDI_ESI_ID;
		case 0xff: return M_FS_9EDI_ID;
		}
	    }
	case PREFICES_GS:
	    switch (modrm & 0xc0) {
	    default:
		return -1;
	    case 0x00:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_GS_2EAX;
		case 0x01: return M_GS_EAX_ECX;
		case 0x02: return M_GS_EAX_EDX;
		case 0x03: return M_GS_EAX_EBX;
		case 0x04: return M_GS_EAX_ESP;
		case 0x05: return M_GS_EAX_ID;
		case 0x06: return M_GS_EAX_ESI;
		case 0x07: return M_GS_EAX_EDI;

		case 0x08: return M_GS_EAX_ECX;
		case 0x09: return M_GS_2ECX;
		case 0x0a: return M_GS_ECX_EDX;
		case 0x0b: return M_GS_ECX_EBX;
		case 0x0c: return M_GS_ECX_ESP;
		case 0x0d: return M_GS_ECX_ID;
		case 0x0e: return M_GS_ECX_ESI;
		case 0x0f: return M_GS_ECX_EDI;

		case 0x10: return M_GS_EAX_EDX;
		case 0x11: return M_GS_ECX_EDX;
		case 0x12: return M_GS_2EDX;
		case 0x13: return M_GS_EDX_EBX;
		case 0x14: return M_GS_EDX_ESP;
		case 0x15: return M_GS_EDX_ID;
		case 0x16: return M_GS_EDX_ESI;
		case 0x17: return M_GS_EDX_EDI;

		case 0x18: return M_GS_EAX_EBX;
		case 0x19: return M_GS_ECX_EBX;
		case 0x1a: return M_GS_EDX_EBX;
		case 0x1b: return M_GS_2EBX;
		case 0x1c: return M_GS_EBX_ESP;
		case 0x1d: return M_GS_EBX_ID;
		case 0x1e: return M_GS_EBX_ESI;
		case 0x1f: return M_GS_EBX_EDI;

		case 0x20: return M_GS_EAX;
		case 0x21: return M_GS_ECX;
		case 0x22: return M_GS_EDX;
		case 0x23: return M_GS_EBX;
		case 0x24: return M_GS_ESP;
		case 0x25: return M_GS_ID;
		case 0x26: return M_GS_ESI;
		case 0x27: return M_GS_EDI;

		case 0x28: return M_GS_EAX_EBP;
		case 0x29: return M_GS_ECX_EBP;
		case 0x2a: return M_GS_EDX_EBP;
		case 0x2b: return M_GS_EBX_EBP;
		case 0x2c: return M_GS_EBP_ESP;
		case 0x2d: return M_GS_EBP_ID;
		case 0x2e: return M_GS_EBP_ESI;
		case 0x2f: return M_GS_EBP_EDI;

		case 0x30: return M_GS_EAX_ESI;
		case 0x31: return M_GS_ECX_ESI;
		case 0x32: return M_GS_EDX_ESI;
		case 0x33: return M_GS_EBX_ESI;
		case 0x34: return M_GS_ESI_ESP;
		case 0x35: return M_GS_ESI_ID;
		case 0x36: return M_GS_2ESI;
		case 0x37: return M_GS_ESI_EDI;

		case 0x38: return M_GS_EAX_EDI;
		case 0x39: return M_GS_ECX_EDI;
		case 0x3a: return M_GS_EDX_EDI;
		case 0x3b: return M_GS_EBX_EDI;
		case 0x3c: return M_GS_EDI_ESP;
		case 0x3d: return M_GS_EDI_ID;
		case 0x3e: return M_GS_ESI_EDI;
		case 0x3f: return M_GS_2EDI;

		case 0x40: return M_GS_3EAX;
		case 0x41: return M_GS_2EAX_ECX;
		case 0x42: return M_GS_2EAX_EDX;
		case 0x43: return M_GS_2EAX_EBX;
		case 0x44: return M_GS_2EAX_ESP;
		case 0x45: return M_GS_2EAX_ID;
		case 0x46: return M_GS_2EAX_ESI;
		case 0x47: return M_GS_2EAX_EDI;

		case 0x48: return M_GS_2ECX_EAX;
		case 0x49: return M_GS_3ECX;
		case 0x4a: return M_GS_2ECX_EDX;
		case 0x4b: return M_GS_2ECX_EBX;
		case 0x4c: return M_GS_2ECX_ESP;
		case 0x4d: return M_GS_2ECX_ID;
		case 0x4e: return M_GS_2ECX_ESI;
		case 0x4f: return M_GS_2ECX_EDI;

		case 0x50: return M_GS_2EDX_EAX;
		case 0x51: return M_GS_2EDX_ECX;
		case 0x52: return M_GS_3EDX;
		case 0x53: return M_GS_2EDX_EBX;
		case 0x54: return M_GS_2EDX_ESP;
		case 0x55: return M_GS_2EDX_ID;
		case 0x56: return M_GS_2EDX_ESI;
		case 0x57: return M_GS_2EDX_EDI;

		case 0x58: return M_GS_2EBX_EAX;
		case 0x59: return M_GS_2EBX_ECX;
		case 0x5a: return M_GS_2EBX_EDX;
		case 0x5b: return M_GS_3EBX;
		case 0x5c: return M_GS_2EBX_ESP;
		case 0x5d: return M_GS_2EBX_ID;
		case 0x5e: return M_GS_2EBX_ESI;
		case 0x5f: return M_GS_2EBX_EDI;

		case 0x60: return M_GS_EAX;
		case 0x61: return M_GS_ECX;
		case 0x62: return M_GS_EDX;
		case 0x63: return M_GS_EBX;
		case 0x64: return M_GS_ESP;
		case 0x65: return M_GS_ID;
		case 0x66: return M_GS_ESI;
		case 0x67: return M_GS_EDI;

		case 0x68: return M_GS_2EBP_EAX;
		case 0x69: return M_GS_2EBP_ECX;
		case 0x6a: return M_GS_2EBP_EDX;
		case 0x6b: return M_GS_2EBP_EBX;
		case 0x6c: return M_GS_2EBP_ESP;
		case 0x6d: return M_GS_2EBP_ID;
		case 0x6e: return M_GS_2EBP_ESI;
		case 0x6f: return M_GS_2EBP_EDI;

		case 0x70: return M_GS_2ESI_EAX;
		case 0x71: return M_GS_2ESI_ECX;
		case 0x72: return M_GS_2ESI_EDX;
		case 0x73: return M_GS_2ESI_EBX;
		case 0x74: return M_GS_2ESI_ESP;
		case 0x75: return M_GS_2ESI_ID;
		case 0x76: return M_GS_3ESI;
		case 0x77: return M_GS_2ESI_EDI;

		case 0x78: return M_GS_2EDI_EAX;
		case 0x79: return M_GS_2EDI_ECX;
		case 0x7a: return M_GS_2EDI_EDX;
		case 0x7b: return M_GS_2EDI_EBX;
		case 0x7c: return M_GS_2EDI_ESP;
		case 0x7d: return M_GS_2EDI_ID;
		case 0x7e: return M_GS_2EDI_ESI;
		case 0x7f: return M_GS_3EDI;

		case 0x80: return M_GS_5EAX;
		case 0x81: return M_GS_4EAX_ECX;
		case 0x82: return M_GS_4EAX_EDX;
		case 0x83: return M_GS_4EAX_EBX;
		case 0x84: return M_GS_4EAX_ESP;
		case 0x85: return M_GS_4EAX_ID;
		case 0x86: return M_GS_4EAX_ESI;
		case 0x87: return M_GS_4EAX_EDI;

		case 0x88: return M_GS_4ECX_EAX;
		case 0x89: return M_GS_5ECX;
		case 0x8a: return M_GS_4ECX_EDX;
		case 0x8b: return M_GS_4ECX_EBX;
		case 0x8c: return M_GS_4ECX_ESP;
		case 0x8d: return M_GS_4ECX_ID;
		case 0x8e: return M_GS_4ECX_ESI;
		case 0x8f: return M_GS_4ECX_EDI;

		case 0x90: return M_GS_4EDX_EAX;
		case 0x91: return M_GS_4EDX_ECX;
		case 0x92: return M_GS_5EDX;
		case 0x93: return M_GS_4EDX_EBX;
		case 0x94: return M_GS_4EDX_ESP;
		case 0x95: return M_GS_4EDX_ID;
		case 0x96: return M_GS_4EDX_ESI;
		case 0x97: return M_GS_4EDX_EDI;

		case 0x98: return M_GS_4EBX_EAX;
		case 0x99: return M_GS_4EBX_ECX;
		case 0x9a: return M_GS_4EBX_EDX;
		case 0x9b: return M_GS_5EBX;
		case 0x9c: return M_GS_4EBX_ESP;
		case 0x9d: return M_GS_4EBX_ID;
		case 0x9e: return M_GS_4EBX_ESI;
		case 0x9f: return M_GS_4EBX_EDI;

		case 0xa0: return M_GS_EAX;
		case 0xa1: return M_GS_ECX;
		case 0xa2: return M_GS_EDX;
		case 0xa3: return M_GS_EBX;
		case 0xa4: return M_GS_ESP;
		case 0xa5: return M_GS_ID;
		case 0xa6: return M_GS_ESI;
		case 0xa7: return M_GS_EDI;

		case 0xa8: return M_GS_4EBP_EAX;
		case 0xa9: return M_GS_4EBP_ECX;
		case 0xaa: return M_GS_4EBP_EDX;
		case 0xab: return M_GS_4EBP_EBX;
		case 0xac: return M_GS_4EBP_ESP;
		case 0xad: return M_GS_4EBP_ID;
		case 0xae: return M_GS_4EBP_ESI;
		case 0xaf: return M_GS_4EBP_EDI;

		case 0xb0: return M_GS_4ESI_EAX;
		case 0xb1: return M_GS_4ESI_ECX;
		case 0xb2: return M_GS_4ESI_EDX;
		case 0xb3: return M_GS_4ESI_EBX;
		case 0xb4: return M_GS_4ESI_ESP;
		case 0xb5: return M_GS_4ESI_ID;
		case 0xb6: return M_GS_5ESI;
		case 0xb7: return M_GS_4ESI_EDI;

		case 0xb8: return M_GS_4EDI_EAX;
		case 0xb9: return M_GS_4EDI_ECX;
		case 0xba: return M_GS_4EDI_EDX;
		case 0xbb: return M_GS_4EDI_EBX;
		case 0xbc: return M_GS_4EDI_ESP;
		case 0xbd: return M_GS_4EDI_ID;
		case 0xbe: return M_GS_4EDI_ESI;
		case 0xbf: return M_GS_5EDI;

		case 0xc0: return M_GS_9EAX;
		case 0xc1: return M_GS_8EAX_ECX;
		case 0xc2: return M_GS_8EAX_EDX;
		case 0xc3: return M_GS_8EAX_EBX;
		case 0xc4: return M_GS_8EAX_ESP;
		case 0xc5: return M_GS_8EAX_ID;
		case 0xc6: return M_GS_8EAX_ESI;
		case 0xc7: return M_GS_8EAX_EDI;

		case 0xc8: return M_GS_8ECX_EAX;
		case 0xc9: return M_GS_9ECX;
		case 0xca: return M_GS_8ECX_EDX;
		case 0xcb: return M_GS_8ECX_EBX;
		case 0xcc: return M_GS_8ECX_ESP;
		case 0xcd: return M_GS_8ECX_ID;
		case 0xce: return M_GS_8ECX_ESI;
		case 0xcf: return M_GS_8ECX_EDI;

		case 0xd0: return M_GS_8EDX_EAX;
		case 0xd1: return M_GS_8EDX_ECX;
		case 0xd2: return M_GS_9EDX;
		case 0xd3: return M_GS_8EDX_EBX;
		case 0xd4: return M_GS_8EDX_ESP;
		case 0xd5: return M_GS_8EDX_ID;
		case 0xd6: return M_GS_8EDX_ESI;
		case 0xd7: return M_GS_8EDX_EDI;

		case 0xd8: return M_GS_8EBX_EAX;
		case 0xd9: return M_GS_8EBX_ECX;
		case 0xda: return M_GS_8EBX_EDX;
		case 0xdb: return M_GS_9EBX;
		case 0xdc: return M_GS_8EBX_ESP;
		case 0xdd: return M_GS_8EBX_ID;
		case 0xde: return M_GS_8EBX_ESI;
		case 0xdf: return M_GS_8EBX_EDI;

		case 0xe0: return M_GS_EAX;
		case 0xe1: return M_GS_ECX;
		case 0xe2: return M_GS_EDX;
		case 0xe3: return M_GS_EBX;
		case 0xe4: return M_GS_ESP;
		case 0xe5: return M_GS_ID;
		case 0xe6: return M_GS_ESI;
		case 0xe7: return M_GS_EDI;

		case 0xe8: return M_GS_8EBP_EAX;
		case 0xe9: return M_GS_8EBP_ECX;
		case 0xea: return M_GS_8EBP_EDX;
		case 0xeb: return M_GS_8EBP_EBX;
		case 0xec: return M_GS_8EBP_ESP;
		case 0xed: return M_GS_8EBP_ID;
		case 0xee: return M_GS_8EBP_ESI;
		case 0xef: return M_GS_8EBP_EDI;

		case 0xf0: return M_GS_8ESI_EAX;
		case 0xf1: return M_GS_8ESI_ECX;
		case 0xf2: return M_GS_8ESI_EDX;
		case 0xf3: return M_GS_8ESI_EBX;
		case 0xf4: return M_GS_8ESI_ESP;
		case 0xf5: return M_GS_8ESI_ID;
		case 0xf6: return M_GS_9ESI;
		case 0xf7: return M_GS_8ESI_EDI;

		case 0xf8: return M_GS_8EDI_EAX;
		case 0xf9: return M_GS_8EDI_ECX;
		case 0xfa: return M_GS_8EDI_EDX;
		case 0xfb: return M_GS_8EDI_EBX;
		case 0xfc: return M_GS_8EDI_ESP;
		case 0xfd: return M_GS_8EDI_ID;
		case 0xfe: return M_GS_8EDI_ESI;
		case 0xff: return M_GS_9EDI;
		}

	    case 0x40:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_GS_2EAX_IB;
		case 0x01: return M_GS_EAX_ECX_IB;
		case 0x02: return M_GS_EAX_EDX_IB;
		case 0x03: return M_GS_EAX_EBX_IB;
		case 0x04: return M_GS_EAX_ESP_IB;
		case 0x05: return M_GS_EAX_EBP_IB;
		case 0x06: return M_GS_EAX_ESI_IB;
		case 0x07: return M_GS_EAX_EDI_IB;

		case 0x08: return M_GS_EAX_ECX_IB;
		case 0x09: return M_GS_2ECX_IB;
		case 0x0a: return M_GS_ECX_EDX_IB;
		case 0x0b: return M_GS_ECX_EBX_IB;
		case 0x0c: return M_GS_ECX_ESP_IB;
		case 0x0d: return M_GS_ECX_EBP_IB;
		case 0x0e: return M_GS_ECX_ESI_IB;
		case 0x0f: return M_GS_ECX_EDI_IB;

		case 0x10: return M_GS_EAX_EDX_IB;
		case 0x11: return M_GS_ECX_EDX_IB;
		case 0x12: return M_GS_2EDX_IB;
		case 0x13: return M_GS_EDX_EBX_IB;
		case 0x14: return M_GS_EDX_ESP_IB;
		case 0x15: return M_GS_EDX_EBP_IB;
		case 0x16: return M_GS_EDX_ESI_IB;
		case 0x17: return M_GS_EDX_EDI_IB;

		case 0x18: return M_GS_EAX_EBX_IB;
		case 0x19: return M_GS_ECX_EBX_IB;
		case 0x1a: return M_GS_EDX_EBX_IB;
		case 0x1b: return M_GS_2EBX_IB;
		case 0x1c: return M_GS_EBX_ESP_IB;
		case 0x1d: return M_GS_EBX_EBP_IB;
		case 0x1e: return M_GS_EBX_ESI_IB;
		case 0x1f: return M_GS_EBX_EDI_IB;

		case 0x20: return M_GS_EAX_IB;
		case 0x21: return M_GS_ECX_IB;
		case 0x22: return M_GS_EDX_IB;
		case 0x23: return M_GS_EBX_IB;
		case 0x24: return M_GS_ESP_IB;
		case 0x25: return M_GS_EBP_IB;
		case 0x26: return M_GS_ESI_IB;
		case 0x27: return M_GS_EDI_IB;

		case 0x28: return M_GS_EAX_EBP_IB;
		case 0x29: return M_GS_ECX_EBP_IB;
		case 0x2a: return M_GS_EDX_EBP_IB;
		case 0x2b: return M_GS_EBX_EBP_IB;
		case 0x2c: return M_GS_EBP_ESP_IB;
		case 0x2d: return M_GS_2EBP_IB;
		case 0x2e: return M_GS_EBP_ESI_IB;
		case 0x2f: return M_GS_EBP_EDI_IB;

		case 0x30: return M_GS_EAX_ESI_IB;
		case 0x31: return M_GS_ECX_ESI_IB;
		case 0x32: return M_GS_EDX_ESI_IB;
		case 0x33: return M_GS_EBX_ESI_IB;
		case 0x34: return M_GS_ESI_ESP_IB;
		case 0x35: return M_GS_EBP_ESI_IB;
		case 0x36: return M_GS_2ESI_IB;
		case 0x37: return M_GS_ESI_EDI_IB;

		case 0x38: return M_GS_EAX_EDI_IB;
		case 0x39: return M_GS_ECX_EDI_IB;
		case 0x3a: return M_GS_EDX_EDI_IB;
		case 0x3b: return M_GS_EBX_EDI_IB;
		case 0x3c: return M_GS_EDI_ESP_IB;
		case 0x3d: return M_GS_EBP_EDI_IB;
		case 0x3e: return M_GS_ESI_EDI_IB;
		case 0x3f: return M_GS_2EDI_IB;

		case 0x40: return M_GS_3EAX_IB;
		case 0x41: return M_GS_2EAX_ECX_IB;
		case 0x42: return M_GS_2EAX_EDX_IB;
		case 0x43: return M_GS_2EAX_EBX_IB;
		case 0x44: return M_GS_2EAX_ESP_IB;
		case 0x45: return M_GS_2EAX_EBP_IB;
		case 0x46: return M_GS_2EAX_ESI_IB;
		case 0x47: return M_GS_2EAX_EDI_IB;

		case 0x48: return M_GS_2ECX_EAX_IB;
		case 0x49: return M_GS_3ECX_IB;
		case 0x4a: return M_GS_2ECX_EDX_IB;
		case 0x4b: return M_GS_2ECX_EBX_IB;
		case 0x4c: return M_GS_2ECX_ESP_IB;
		case 0x4d: return M_GS_2ECX_EBP_IB;
		case 0x4e: return M_GS_2ECX_ESI_IB;
		case 0x4f: return M_GS_2ECX_EDI_IB;

		case 0x50: return M_GS_2EDX_EAX_IB;
		case 0x51: return M_GS_2EDX_ECX_IB;
		case 0x52: return M_GS_3EDX_IB;
		case 0x53: return M_GS_2EDX_EBX_IB;
		case 0x54: return M_GS_2EDX_ESP_IB;
		case 0x55: return M_GS_2EDX_EBP_IB;
		case 0x56: return M_GS_2EDX_ESI_IB;
		case 0x57: return M_GS_2EDX_EDI_IB;

		case 0x58: return M_GS_2EBX_EAX_IB;
		case 0x59: return M_GS_2EBX_ECX_IB;
		case 0x5a: return M_GS_2EBX_EDX_IB;
		case 0x5b: return M_GS_3EBX_IB;
		case 0x5c: return M_GS_2EBX_ESP_IB;
		case 0x5d: return M_GS_2EBX_EBP_IB;
		case 0x5e: return M_GS_2EBX_ESI_IB;
		case 0x5f: return M_GS_2EBX_EDI_IB;

		case 0x60: return M_GS_EAX_IB;
		case 0x61: return M_GS_ECX_IB;
		case 0x62: return M_GS_EDX_IB;
		case 0x63: return M_GS_EBX_IB;
		case 0x64: return M_GS_ESP_IB;
		case 0x65: return M_GS_EBP_IB;
		case 0x66: return M_GS_ESI_IB;
		case 0x67: return M_GS_EDI_IB;

		case 0x68: return M_GS_2EBP_EAX_IB;
		case 0x69: return M_GS_2EBP_ECX_IB;
		case 0x6a: return M_GS_2EBP_EDX_IB;
		case 0x6b: return M_GS_2EBP_EBX_IB;
		case 0x6c: return M_GS_2EBP_ESP_IB;
		case 0x6d: return M_GS_3EBP_IB;
		case 0x6e: return M_GS_2EBP_ESI_IB;
		case 0x6f: return M_GS_2EBP_EDI_IB;

		case 0x70: return M_GS_2ESI_EAX_IB;
		case 0x71: return M_GS_2ESI_ECX_IB;
		case 0x72: return M_GS_2ESI_EDX_IB;
		case 0x73: return M_GS_2ESI_EBX_IB;
		case 0x74: return M_GS_2ESI_ESP_IB;
		case 0x75: return M_GS_2ESI_EBP_IB;
		case 0x76: return M_GS_3ESI_IB;
		case 0x77: return M_GS_2ESI_EDI_IB;

		case 0x78: return M_GS_2EDI_EAX_IB;
		case 0x79: return M_GS_2EDI_ECX_IB;
		case 0x7a: return M_GS_2EDI_EDX_IB;
		case 0x7b: return M_GS_2EDI_EBX_IB;
		case 0x7c: return M_GS_2EDI_ESP_IB;
		case 0x7d: return M_GS_2EDI_EBP_IB;
		case 0x7e: return M_GS_2EDI_ESI_IB;
		case 0x7f: return M_GS_3EDI_IB;

		case 0x80: return M_GS_5EAX_IB;
		case 0x81: return M_GS_4EAX_ECX_IB;
		case 0x82: return M_GS_4EAX_EDX_IB;
		case 0x83: return M_GS_4EAX_EBX_IB;
		case 0x84: return M_GS_4EAX_ESP_IB;
		case 0x85: return M_GS_4EAX_EBP_IB;
		case 0x86: return M_GS_4EAX_ESI_IB;
		case 0x87: return M_GS_4EAX_EDI_IB;

		case 0x88: return M_GS_4ECX_EAX_IB;
		case 0x89: return M_GS_5ECX_IB;
		case 0x8a: return M_GS_4ECX_EDX_IB;
		case 0x8b: return M_GS_4ECX_EBX_IB;
		case 0x8c: return M_GS_4ECX_ESP_IB;
		case 0x8d: return M_GS_4ECX_EBP_IB;
		case 0x8e: return M_GS_4ECX_ESI_IB;
		case 0x8f: return M_GS_4ECX_EDI_IB;

		case 0x90: return M_GS_4EDX_EAX_IB;
		case 0x91: return M_GS_4EDX_ECX_IB;
		case 0x92: return M_GS_5EDX_IB;
		case 0x93: return M_GS_4EDX_EBX_IB;
		case 0x94: return M_GS_4EDX_ESP_IB;
		case 0x95: return M_GS_4EDX_EBP_IB;
		case 0x96: return M_GS_4EDX_ESI_IB;
		case 0x97: return M_GS_4EDX_EDI_IB;

		case 0x98: return M_GS_4EBX_EAX_IB;
		case 0x99: return M_GS_4EBX_ECX_IB;
		case 0x9a: return M_GS_4EBX_EDX_IB;
		case 0x9b: return M_GS_5EBX_IB;
		case 0x9c: return M_GS_4EBX_ESP_IB;
		case 0x9d: return M_GS_4EBX_EBP_IB;
		case 0x9e: return M_GS_4EBX_ESI_IB;
		case 0x9f: return M_GS_4EBX_EDI_IB;

		case 0xa0: return M_GS_EAX_IB;
		case 0xa1: return M_GS_ECX_IB;
		case 0xa2: return M_GS_EDX_IB;
		case 0xa3: return M_GS_EBX_IB;
		case 0xa4: return M_GS_ESP_IB;
		case 0xa5: return M_GS_EBP_IB;
		case 0xa6: return M_GS_ESI_IB;
		case 0xa7: return M_GS_EDI_IB;

		case 0xa8: return M_GS_4EBP_EAX_IB;
		case 0xa9: return M_GS_4EBP_ECX_IB;
		case 0xaa: return M_GS_4EBP_EDX_IB;
		case 0xab: return M_GS_4EBP_EBX_IB;
		case 0xac: return M_GS_4EBP_ESP_IB;
		case 0xad: return M_GS_5EBP_IB;
		case 0xae: return M_GS_4EBP_ESI_IB;
		case 0xaf: return M_GS_4EBP_EDI_IB;

		case 0xb0: return M_GS_4ESI_EAX_IB;
		case 0xb1: return M_GS_4ESI_ECX_IB;
		case 0xb2: return M_GS_4ESI_EDX_IB;
		case 0xb3: return M_GS_4ESI_EBX_IB;
		case 0xb4: return M_GS_4ESI_ESP_IB;
		case 0xb5: return M_GS_4ESI_EBP_IB;
		case 0xb6: return M_GS_5ESI_IB;
		case 0xb7: return M_GS_4ESI_EDI_IB;

		case 0xb8: return M_GS_4EDI_EAX_IB;
		case 0xb9: return M_GS_4EDI_ECX_IB;
		case 0xba: return M_GS_4EDI_EDX_IB;
		case 0xbb: return M_GS_4EDI_EBX_IB;
		case 0xbc: return M_GS_4EDI_ESP_IB;
		case 0xbd: return M_GS_4EDI_EBP_IB;
		case 0xbe: return M_GS_4EDI_ESI_IB;
		case 0xbf: return M_GS_5EDI_IB;

		case 0xc0: return M_GS_9EAX_IB;
		case 0xc1: return M_GS_8EAX_ECX_IB;
		case 0xc2: return M_GS_8EAX_EDX_IB;
		case 0xc3: return M_GS_8EAX_EBX_IB;
		case 0xc4: return M_GS_8EAX_ESP_IB;
		case 0xc5: return M_GS_8EAX_EBP_IB;
		case 0xc6: return M_GS_8EAX_ESI_IB;
		case 0xc7: return M_GS_8EAX_EDI_IB;

		case 0xc8: return M_GS_8ECX_EAX_IB;
		case 0xc9: return M_GS_9ECX_IB;
		case 0xca: return M_GS_8ECX_EDX_IB;
		case 0xcb: return M_GS_8ECX_EBX_IB;
		case 0xcc: return M_GS_8ECX_ESP_IB;
		case 0xcd: return M_GS_8ECX_EBP_IB;
		case 0xce: return M_GS_8ECX_ESI_IB;
		case 0xcf: return M_GS_8ECX_EDI_IB;

		case 0xd0: return M_GS_8EDX_EAX_IB;
		case 0xd1: return M_GS_8EDX_ECX_IB;
		case 0xd2: return M_GS_9EDX_IB;
		case 0xd3: return M_GS_8EDX_EBX_IB;
		case 0xd4: return M_GS_8EDX_ESP_IB;
		case 0xd5: return M_GS_8EDX_EBP_IB;
		case 0xd6: return M_GS_8EDX_ESI_IB;
		case 0xd7: return M_GS_8EDX_EDI_IB;

		case 0xd8: return M_GS_8EBX_EAX_IB;
		case 0xd9: return M_GS_8EBX_ECX_IB;
		case 0xda: return M_GS_8EBX_EDX_IB;
		case 0xdb: return M_GS_9EBX_IB;
		case 0xdc: return M_GS_8EBX_ESP_IB;
		case 0xdd: return M_GS_8EBX_EBP_IB;
		case 0xde: return M_GS_8EBX_ESI_IB;
		case 0xdf: return M_GS_8EBX_EDI_IB;

		case 0xe0: return M_GS_EAX_IB;
		case 0xe1: return M_GS_ECX_IB;
		case 0xe2: return M_GS_EDX_IB;
		case 0xe3: return M_GS_EBX_IB;
		case 0xe4: return M_GS_ESP_IB;
		case 0xe5: return M_GS_EBP_IB;
		case 0xe6: return M_GS_ESI_IB;
		case 0xe7: return M_GS_EDI_IB;

		case 0xe8: return M_GS_8EBP_EAX_IB;
		case 0xe9: return M_GS_8EBP_ECX_IB;
		case 0xea: return M_GS_8EBP_EDX_IB;
		case 0xeb: return M_GS_8EBP_EBX_IB;
		case 0xec: return M_GS_8EBP_ESP_IB;
		case 0xed: return M_GS_9EBP_IB;
		case 0xee: return M_GS_8EBP_ESI_IB;
		case 0xef: return M_GS_8EBP_EDI_IB;

		case 0xf0: return M_GS_8ESI_EAX_IB;
		case 0xf1: return M_GS_8ESI_ECX_IB;
		case 0xf2: return M_GS_8ESI_EDX_IB;
		case 0xf3: return M_GS_8ESI_EBX_IB;
		case 0xf4: return M_GS_8ESI_ESP_IB;
		case 0xf5: return M_GS_8ESI_EBP_IB;
		case 0xf6: return M_GS_9ESI_IB;
		case 0xf7: return M_GS_8ESI_EDI_IB;

		case 0xf8: return M_GS_8EDI_EAX_IB;
		case 0xf9: return M_GS_8EDI_ECX_IB;
		case 0xfa: return M_GS_8EDI_EDX_IB;
		case 0xfb: return M_GS_8EDI_EBX_IB;
		case 0xfc: return M_GS_8EDI_ESP_IB;
		case 0xfd: return M_GS_8EDI_EBP_IB;
		case 0xfe: return M_GS_8EDI_ESI_IB;
		case 0xff: return M_GS_9EDI_IB;
		}
	    case 0x80:
		switch (sib) {
		default:
		    return -1;
		case 0x00: return M_GS_2EAX_ID;
		case 0x01: return M_GS_EAX_ECX_ID;
		case 0x02: return M_GS_EAX_EDX_ID;
		case 0x03: return M_GS_EAX_EBX_ID;
		case 0x04: return M_GS_EAX_ESP_ID;
		case 0x05: return M_GS_EAX_EBP_ID;
		case 0x06: return M_GS_EAX_ESI_ID;
		case 0x07: return M_GS_EAX_EDI_ID;

		case 0x08: return M_GS_EAX_ECX_ID;
		case 0x09: return M_GS_2ECX_ID;
		case 0x0a: return M_GS_ECX_EDX_ID;
		case 0x0b: return M_GS_ECX_EBX_ID;
		case 0x0c: return M_GS_ECX_ESP_ID;
		case 0x0d: return M_GS_ECX_EBP_ID;
		case 0x0e: return M_GS_ECX_ESI_ID;
		case 0x0f: return M_GS_ECX_EDI_ID;

		case 0x10: return M_GS_EAX_EDX_ID;
		case 0x11: return M_GS_ECX_EDX_ID;
		case 0x12: return M_GS_2EDX_ID;
		case 0x13: return M_GS_EDX_EBX_ID;
		case 0x14: return M_GS_EDX_ESP_ID;
		case 0x15: return M_GS_EDX_EBP_ID;
		case 0x16: return M_GS_EDX_ESI_ID;
		case 0x17: return M_GS_EDX_EDI_ID;

		case 0x18: return M_GS_EAX_EBX_ID;
		case 0x19: return M_GS_ECX_EBX_ID;
		case 0x1a: return M_GS_EDX_EBX_ID;
		case 0x1b: return M_GS_2EBX_ID;
		case 0x1c: return M_GS_EBX_ESP_ID;
		case 0x1d: return M_GS_EBX_EBP_ID;
		case 0x1e: return M_GS_EBX_ESI_ID;
		case 0x1f: return M_GS_EBX_EDI_ID;

		case 0x20: return M_GS_EAX_ID;
		case 0x21: return M_GS_ECX_ID;
		case 0x22: return M_GS_EDX_ID;
		case 0x23: return M_GS_EBX_ID;
		case 0x24: return M_GS_ESP_ID;
		case 0x25: return M_GS_EBP_ID;
		case 0x26: return M_GS_ESI_ID;
		case 0x27: return M_GS_EDI_ID;

		case 0x28: return M_GS_EAX_EBP_ID;
		case 0x29: return M_GS_ECX_EBP_ID;
		case 0x2a: return M_GS_EDX_EBP_ID;
		case 0x2b: return M_GS_EBX_EBP_ID;
		case 0x2c: return M_GS_EBP_ESP_ID;
		case 0x2d: return M_GS_2EBP_ID;
		case 0x2e: return M_GS_EBP_ESI_ID;
		case 0x2f: return M_GS_EBP_EDI_ID;

		case 0x30: return M_GS_EAX_ESI_ID;
		case 0x31: return M_GS_ECX_ESI_ID;
		case 0x32: return M_GS_EDX_ESI_ID;
		case 0x33: return M_GS_EBX_ESI_ID;
		case 0x34: return M_GS_ESI_ESP_ID;
		case 0x35: return M_GS_EBP_ESI_ID;
		case 0x36: return M_GS_2ESI_ID;
		case 0x37: return M_GS_ESI_EDI_ID;

		case 0x38: return M_GS_EAX_EDI_ID;
		case 0x39: return M_GS_ECX_EDI_ID;
		case 0x3a: return M_GS_EDX_EDI_ID;
		case 0x3b: return M_GS_EBX_EDI_ID;
		case 0x3c: return M_GS_EDI_ESP_ID;
		case 0x3d: return M_GS_EBP_EDI_ID;
		case 0x3e: return M_GS_ESI_EDI_ID;
		case 0x3f: return M_GS_2EDI_ID;

		case 0x40: return M_GS_3EAX_ID;
		case 0x41: return M_GS_2EAX_ECX_ID;
		case 0x42: return M_GS_2EAX_EDX_ID;
		case 0x43: return M_GS_2EAX_EBX_ID;
		case 0x44: return M_GS_2EAX_ESP_ID;
		case 0x45: return M_GS_2EAX_EBP_ID;
		case 0x46: return M_GS_2EAX_ESI_ID;
		case 0x47: return M_GS_2EAX_EDI_ID;

		case 0x48: return M_GS_2ECX_EAX_ID;
		case 0x49: return M_GS_3ECX_ID;
		case 0x4a: return M_GS_2ECX_EDX_ID;
		case 0x4b: return M_GS_2ECX_EBX_ID;
		case 0x4c: return M_GS_2ECX_ESP_ID;
		case 0x4d: return M_GS_2ECX_EBP_ID;
		case 0x4e: return M_GS_2ECX_ESI_ID;
		case 0x4f: return M_GS_2ECX_EDI_ID;

		case 0x50: return M_GS_2EDX_EAX_ID;
		case 0x51: return M_GS_2EDX_ECX_ID;
		case 0x52: return M_GS_3EDX_ID;
		case 0x53: return M_GS_2EDX_EBX_ID;
		case 0x54: return M_GS_2EDX_ESP_ID;
		case 0x55: return M_GS_2EDX_EBP_ID;
		case 0x56: return M_GS_2EDX_ESI_ID;
		case 0x57: return M_GS_2EDX_EDI_ID;

		case 0x58: return M_GS_2EBX_EAX_ID;
		case 0x59: return M_GS_2EBX_ECX_ID;
		case 0x5a: return M_GS_2EBX_EDX_ID;
		case 0x5b: return M_GS_3EBX_ID;
		case 0x5c: return M_GS_2EBX_ESP_ID;
		case 0x5d: return M_GS_2EBX_EBP_ID;
		case 0x5e: return M_GS_2EBX_ESI_ID;
		case 0x5f: return M_GS_2EBX_EDI_ID;

		case 0x60: return M_GS_EAX_ID;
		case 0x61: return M_GS_ECX_ID;
		case 0x62: return M_GS_EDX_ID;
		case 0x63: return M_GS_EBX_ID;
		case 0x64: return M_GS_ESP_ID;
		case 0x65: return M_GS_EBP_ID;
		case 0x66: return M_GS_ESI_ID;
		case 0x67: return M_GS_EDI_ID;

		case 0x68: return M_GS_2EBP_EAX_ID;
		case 0x69: return M_GS_2EBP_ECX_ID;
		case 0x6a: return M_GS_2EBP_EDX_ID;
		case 0x6b: return M_GS_2EBP_EBX_ID;
		case 0x6c: return M_GS_2EBP_ESP_ID;
		case 0x6d: return M_GS_3EBP_ID;
		case 0x6e: return M_GS_2EBP_ESI_ID;
		case 0x6f: return M_GS_2EBP_EDI_ID;

		case 0x70: return M_GS_2ESI_EAX_ID;
		case 0x71: return M_GS_2ESI_ECX_ID;
		case 0x72: return M_GS_2ESI_EDX_ID;
		case 0x73: return M_GS_2ESI_EBX_ID;
		case 0x74: return M_GS_2ESI_ESP_ID;
		case 0x75: return M_GS_2ESI_EBP_ID;
		case 0x76: return M_GS_3ESI_ID;
		case 0x77: return M_GS_2ESI_EDI_ID;

		case 0x78: return M_GS_2EDI_EAX_ID;
		case 0x79: return M_GS_2EDI_ECX_ID;
		case 0x7a: return M_GS_2EDI_EDX_ID;
		case 0x7b: return M_GS_2EDI_EBX_ID;
		case 0x7c: return M_GS_2EDI_ESP_ID;
		case 0x7d: return M_GS_2EDI_EBP_ID;
		case 0x7e: return M_GS_2EDI_ESI_ID;
		case 0x7f: return M_GS_3EDI_ID;

		case 0x80: return M_GS_5EAX_ID;
		case 0x81: return M_GS_4EAX_ECX_ID;
		case 0x82: return M_GS_4EAX_EDX_ID;
		case 0x83: return M_GS_4EAX_EBX_ID;
		case 0x84: return M_GS_4EAX_ESP_ID;
		case 0x85: return M_GS_4EAX_EBP_ID;
		case 0x86: return M_GS_4EAX_ESI_ID;
		case 0x87: return M_GS_4EAX_EDI_ID;

		case 0x88: return M_GS_4ECX_EAX_ID;
		case 0x89: return M_GS_5ECX_ID;
		case 0x8a: return M_GS_4ECX_EDX_ID;
		case 0x8b: return M_GS_4ECX_EBX_ID;
		case 0x8c: return M_GS_4ECX_ESP_ID;
		case 0x8d: return M_GS_4ECX_EBP_ID;
		case 0x8e: return M_GS_4ECX_ESI_ID;
		case 0x8f: return M_GS_4ECX_EDI_ID;

		case 0x90: return M_GS_4EDX_EAX_ID;
		case 0x91: return M_GS_4EDX_ECX_ID;
		case 0x92: return M_GS_5EDX_ID;
		case 0x93: return M_GS_4EDX_EBX_ID;
		case 0x94: return M_GS_4EDX_ESP_ID;
		case 0x95: return M_GS_4EDX_EBP_ID;
		case 0x96: return M_GS_4EDX_ESI_ID;
		case 0x97: return M_GS_4EDX_EDI_ID;

		case 0x98: return M_GS_4EBX_EAX_ID;
		case 0x99: return M_GS_4EBX_ECX_ID;
		case 0x9a: return M_GS_4EBX_EDX_ID;
		case 0x9b: return M_GS_5EBX_ID;
		case 0x9c: return M_GS_4EBX_ESP_ID;
		case 0x9d: return M_GS_4EBX_EBP_ID;
		case 0x9e: return M_GS_4EBX_ESI_ID;
		case 0x9f: return M_GS_4EBX_EDI_ID;

		case 0xa0: return M_GS_EAX_ID;
		case 0xa1: return M_GS_ECX_ID;
		case 0xa2: return M_GS_EDX_ID;
		case 0xa3: return M_GS_EBX_ID;
		case 0xa4: return M_GS_ESP_ID;
		case 0xa5: return M_GS_EBP_ID;
		case 0xa6: return M_GS_ESI_ID;
		case 0xa7: return M_GS_EDI_ID;

		case 0xa8: return M_GS_4EBP_EAX_ID;
		case 0xa9: return M_GS_4EBP_ECX_ID;
		case 0xaa: return M_GS_4EBP_EDX_ID;
		case 0xab: return M_GS_4EBP_EBX_ID;
		case 0xac: return M_GS_4EBP_ESP_ID;
		case 0xad: return M_GS_5EBP_ID;
		case 0xae: return M_GS_4EBP_ESI_ID;
		case 0xaf: return M_GS_4EBP_EDI_ID;

		case 0xb0: return M_GS_4ESI_EAX_ID;
		case 0xb1: return M_GS_4ESI_ECX_ID;
		case 0xb2: return M_GS_4ESI_EDX_ID;
		case 0xb3: return M_GS_4ESI_EBX_ID;
		case 0xb4: return M_GS_4ESI_ESP_ID;
		case 0xb5: return M_GS_4ESI_EBP_ID;
		case 0xb6: return M_GS_5ESI_ID;
		case 0xb7: return M_GS_4ESI_EDI_ID;

		case 0xb8: return M_GS_4EDI_EAX_ID;
		case 0xb9: return M_GS_4EDI_ECX_ID;
		case 0xba: return M_GS_4EDI_EDX_ID;
		case 0xbb: return M_GS_4EDI_EBX_ID;
		case 0xbc: return M_GS_4EDI_ESP_ID;
		case 0xbd: return M_GS_4EDI_EBP_ID;
		case 0xbe: return M_GS_4EDI_ESI_ID;
		case 0xbf: return M_GS_5EDI_ID;

		case 0xc0: return M_GS_9EAX_ID;
		case 0xc1: return M_GS_8EAX_ECX_ID;
		case 0xc2: return M_GS_8EAX_EDX_ID;
		case 0xc3: return M_GS_8EAX_EBX_ID;
		case 0xc4: return M_GS_8EAX_ESP_ID;
		case 0xc5: return M_GS_8EAX_EBP_ID;
		case 0xc6: return M_GS_8EAX_ESI_ID;
		case 0xc7: return M_GS_8EAX_EDI_ID;

		case 0xc8: return M_GS_8ECX_EAX_ID;
		case 0xc9: return M_GS_9ECX_ID;
		case 0xca: return M_GS_8ECX_EDX_ID;
		case 0xcb: return M_GS_8ECX_EBX_ID;
		case 0xcc: return M_GS_8ECX_ESP_ID;
		case 0xcd: return M_GS_8ECX_EBP_ID;
		case 0xce: return M_GS_8ECX_ESI_ID;
		case 0xcf: return M_GS_8ECX_EDI_ID;

		case 0xd0: return M_GS_8EDX_EAX_ID;
		case 0xd1: return M_GS_8EDX_ECX_ID;
		case 0xd2: return M_GS_9EDX_ID;
		case 0xd3: return M_GS_8EDX_EBX_ID;
		case 0xd4: return M_GS_8EDX_ESP_ID;
		case 0xd5: return M_GS_8EDX_EBP_ID;
		case 0xd6: return M_GS_8EDX_ESI_ID;
		case 0xd7: return M_GS_8EDX_EDI_ID;

		case 0xd8: return M_GS_8EBX_EAX_ID;
		case 0xd9: return M_GS_8EBX_ECX_ID;
		case 0xda: return M_GS_8EBX_EDX_ID;
		case 0xdb: return M_GS_9EBX_ID;
		case 0xdc: return M_GS_8EBX_ESP_ID;
		case 0xdd: return M_GS_8EBX_EBP_ID;
		case 0xde: return M_GS_8EBX_ESI_ID;
		case 0xdf: return M_GS_8EBX_EDI_ID;

		case 0xe0: return M_GS_EAX_ID;
		case 0xe1: return M_GS_ECX_ID;
		case 0xe2: return M_GS_EDX_ID;
		case 0xe3: return M_GS_EBX_ID;
		case 0xe4: return M_GS_ESP_ID;
		case 0xe5: return M_GS_EBP_ID;
		case 0xe6: return M_GS_ESI_ID;
		case 0xe7: return M_GS_EDI_ID;

		case 0xe8: return M_GS_8EBP_EAX_ID;
		case 0xe9: return M_GS_8EBP_ECX_ID;
		case 0xea: return M_GS_8EBP_EDX_ID;
		case 0xeb: return M_GS_8EBP_EBX_ID;
		case 0xec: return M_GS_8EBP_ESP_ID;
		case 0xed: return M_GS_9EBP_ID;
		case 0xee: return M_GS_8EBP_ESI_ID;
		case 0xef: return M_GS_8EBP_EDI_ID;

		case 0xf0: return M_GS_8ESI_EAX_ID;
		case 0xf1: return M_GS_8ESI_ECX_ID;
		case 0xf2: return M_GS_8ESI_EDX_ID;
		case 0xf3: return M_GS_8ESI_EBX_ID;
		case 0xf4: return M_GS_8ESI_ESP_ID;
		case 0xf5: return M_GS_8ESI_EBP_ID;
		case 0xf6: return M_GS_9ESI_ID;
		case 0xf7: return M_GS_8ESI_EDI_ID;

		case 0xf8: return M_GS_8EDI_EAX_ID;
		case 0xf9: return M_GS_8EDI_ECX_ID;
		case 0xfa: return M_GS_8EDI_EDX_ID;
		case 0xfb: return M_GS_8EDI_EBX_ID;
		case 0xfc: return M_GS_8EDI_ESP_ID;
		case 0xfd: return M_GS_8EDI_EBP_ID;
		case 0xfe: return M_GS_8EDI_ESI_ID;
		case 0xff: return M_GS_9EDI_ID;
		}
	    }
	default: throw new IllegalStateException("Bug In Decoder: Invalid segment prefix value passed");
	}
    }

    private static int decodeST(int modrm)
    {
	switch(modrm & 0x7) {
	case 0: return FPU_ST0;
	case 1: return FPU_ST1;
	case 2: return FPU_ST2;
	case 3: return FPU_ST3;
	case 4: return FPU_ST4;
	case 5: return FPU_ST5;
	case 6: return FPU_ST6;
	case 7: return FPU_ST7;
	default: return -1;
	}
    }

    private class Operation
    {
	private int[] microcodes;
	private int microcodesLength;
	private int x86Length;
	private int readOffset;
	private boolean decoded;
	private boolean terminal;

	Operation()
	{
	    microcodes = new int[10];
	}

	void write(int microcode)
	{
	    try {
		microcodes[microcodesLength++] = microcode;
	    } catch (ArrayIndexOutOfBoundsException e) {
		int[] temp = new int[2*microcodes.length];
		System.arraycopy(microcodes, 0, temp, 0, microcodes.length);
		microcodes = temp;
		microcodes[microcodesLength++] = microcode;
	    }
	}

	void replace(int offset, int microcode)
	{
	    microcodes[offset] = microcode;
	}

	void finish(int x86Length)
	{
	    this.x86Length = x86Length;
	    decoded = true;
	}

	void makeTerminal()
	{
	    reset();
	    terminal = true;
	}

	boolean terminal()
	{
	    return terminal;
	}

	boolean decoded()
	{
	    return decoded;
	}

	void reset()
	{
	    microcodesLength = 0;
	    x86Length = 0;
	    readOffset = 0;
	    decoded = false;
	    terminal = false;
	}

	int getMicrocodeAt(int offset)
	{
	    return microcodes[offset];
	}
	
	int getMicrocode()
	{
	    if (readOffset < microcodesLength)
		return microcodes[readOffset++];
	    else
		throw new IllegalStateException();
	}

	int getLength()
	{
	    return microcodesLength;
	}

	int getX86Length()
	{
	    return x86Length;
	}
    }
}
