/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/

package org.jpc.emulator.memory;

import org.jpc.emulator.memory.codeblock.*;

public class LazyMemory extends AbstractMemory 
{
    private int size;
    private byte[] buffer;
    private CodeBlockTrigger trigger;

    public LazyMemory(int size)
    {
        this.size = size;
        buffer = null;
        trigger = null;
    }

    public LazyMemory(byte[] data)
    {
	this.size = data.length;
	buffer = data;
    }

    public boolean isCacheable()
    {
        return true;
    }

    private final void allocateBuffer()
    {
        if (buffer == null)
            buffer = new byte[size];
    }

    public void copyContentsInto(int address, byte[] buf, int off, int len)
    {
        try
        {
            System.arraycopy(buffer, address, buf, off, len);
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            System.arraycopy(buffer, address, buf, off, len);
        }
    }

    public void copyContentsFrom(int address, byte[] buf, int off, int len)
    {
        try
        {
            System.arraycopy(buf, off, buffer, address, len);
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            System.arraycopy(buf, off, buffer, address, len);
        }
    }

    public long getSize()
    {
        return size;
    }

    public byte getByte(int offset)
    {
        try
        {
            return buffer[offset];
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            return buffer[offset];
        }
    }

    public void setByte(int offset, byte data)
    {
        try
        {
            buffer[offset] = data;
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            buffer[offset] = data;
        }
    } 

    public short getWord(int offset)
    {
        try
        {
            int result = 0xFF & buffer[offset];
	    offset++;
            result |= buffer[offset] << 8;
            return (short) result;
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            int result = 0xFF & buffer[offset];
	    offset++;
            result |= buffer[offset] << 8;
            return (short) result;
        }
    }

    public int getDoubleWord(int offset)
    {
        try
        {
            int result=  0xFF & buffer[offset];
	    offset++;
            result |= (0xFF & buffer[offset]) << 8;
	    offset++;
            result |= (0xFF & buffer[offset]) << 16;
	    offset++;
            result |= (buffer[offset]) << 24;
            return result;
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            int result=  0xFF & buffer[offset];
	    offset++;
            result |= (0xFF & buffer[offset]) << 8;
	    offset++;
            result |= (0xFF & buffer[offset]) << 16;
	    offset++;
            result |= (buffer[offset]) << 24;
            return result;
        }
    }

    public void setWord(int offset, short data)
    {
        try
        {
            buffer[offset] = (byte) data;
	    offset++;
            buffer[offset] = (byte) (data >> 8);
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            buffer[offset] = (byte) data;
	    offset++;
            buffer[offset] = (byte) (data >> 8);
        }
    }

    public void setDoubleWord(int offset, int data)
    {
        try
        {
            buffer[offset] = (byte) data;
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
        }
        catch (NullPointerException e)
        {
            allocateBuffer();
            buffer[offset] = (byte) data;
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
	    offset++;
            data >>= 8;
            buffer[offset] = (byte) (data);
        }
    }

    public void clear()
    {
	buffer = null;
    }

    public CodeBlock queryCodeBlockAt(int offset)
    {
	return null;
    }

    public RealModeCodeBlock getRealModeCodeBlockAt(int offset)
    {
        if (trigger == null)
            trigger = new CodeBlockTrigger(MemoryManager.getInstance().convertMemory(this));
        return trigger;
    }

    public ProtectedModeCodeBlock getProtectedModeCodeBlockAt(int offset)
    {
        if (trigger == null)
            trigger = new CodeBlockTrigger(MemoryManager.getInstance().convertMemory(this));
        return trigger;
    }
    
    public String toString()
    {
        return "LazyMemory["+getSize()+"] {Allocated="+(buffer != null)+"}";
    }
}
