/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/

package org.jpc.emulator.memory;

import org.jpc.emulator.memory.codeblock.*;

public class LazyCodeBlockMemory extends LazyMemory implements CodeBlockMemory
{
    private static final CodeBlock PLACEHOLDER = new BlankCodeBlock(0, 0);

    protected CodeBlock[] codeBuffer;

    public LazyCodeBlockMemory(Memory src)
    {
        super((int) src.getSize());
        if (src.getSize() > 1024*1024*32)
            throw new IllegalStateException("Cannot create code block of size "+src.getSize());

	constructCodeBlocksArray();

        byte[] temp = new byte[(int) src.getSize()];
        src.copyContentsInto(0, temp, 0, temp.length);
        copyContentsFrom(0, temp, 0, temp.length);
    }

    public LazyCodeBlockMemory(byte[] buf)
    {
        super(buf);
	constructCodeBlocksArray();
    }

    public LazyCodeBlockMemory(int size)
    {
        super(size);
	constructCodeBlocksArray();
    }

    protected void constructCodeBlocksArray()
    {
	codeBuffer = MemoryManager.getInstance().getCodeBlockCacheArray(this);
    }

    public void relinquishCache()
    {
    }

    public CodeBlock queryCodeBlockAt(int offset)
    {
	return codeBuffer[offset];
    }

    public RealModeCodeBlock getRealModeCodeBlockAt(int offset)
    {
	try 
        {
	    return (RealModeCodeBlock) codeBuffer[offset];
	} 
        catch (ClassCastException e) 
        {
            return MemoryManager.getInstance().getRealModeCodeBlockAt(this, offset);
	}
    }

    public ProtectedModeCodeBlock getProtectedModeCodeBlockAt(int offset)
    {
	try 
        {
	    return (ProtectedModeCodeBlock) codeBuffer[offset];
	} 
        catch (ClassCastException e) 
        {
            return MemoryManager.getInstance().getProtectedModeCodeBlockAt(this, offset);
	} 
    }

    private void removeCodeBlockAt(int offset)
    {
        CodeBlock b = codeBuffer[offset];
        if ((b == null) || (b == PLACEHOLDER))
            return;

        codeBuffer[offset] = null;
        int len = b.getX86Length();
        for (int i=offset+1; (i<offset + len) && (i < codeBuffer.length); i++)
        {
            if (codeBuffer[i] == PLACEHOLDER)
                codeBuffer[i] = null;
        }
   
        for (int i=Math.min(offset + len, codeBuffer.length)-1; i>=0; i--)
        {
            if (codeBuffer[i] == null)
            {
                if (i < offset)
                    break;
                else
                    continue;
            }
            if (codeBuffer[i] == PLACEHOLDER)
                continue;

            CodeBlock bb = codeBuffer[i];
            len = bb.getX86Length();

            for (int j=i+1; (j<i+len) && (j<codeBuffer.length); j++)
            {
                if (codeBuffer[j] == null)
                    codeBuffer[j] = PLACEHOLDER;
            }
        }
    }

    public void setCodeBlockAt(int offset, CodeBlock block)
    {
        removeCodeBlockAt(offset);
        if (block == null)
            return;

        codeBuffer[offset] = block;
        int len = block.getX86Length();
        for (int i=offset+1; (i<offset+len) && (i<codeBuffer.length); i++)
        {
            if (codeBuffer[i] == null)
                codeBuffer[i] = PLACEHOLDER;
        }
    }

    protected void regionAltered(int start, int end)
    {
        for (int i=end; i>=0; i--)
        {
            CodeBlock b = codeBuffer[i];
            if (b == null) 
            {
                if (i < start)
                    break;
                else
                    continue;
            }

            if (b == PLACEHOLDER)
                continue;
            
            if (!b.handleMemoryRegionChange(start, end))
                removeCodeBlockAt(i);
        } 
    }

    public void copyContentsFrom(int address, byte[] buf, int off, int len)
    {
        super.copyContentsFrom(address, buf, off, len);
        regionAltered(address, address + len - 1);
    }

    public void setByte(int offset, byte data)
    {
        if (super.getByte(offset) == data)
            return;
        super.setByte(offset, data);
        regionAltered(offset, offset);
    }

    public void setWord(int offset, short data)
    {
        if (super.getWord(offset) == data)
            return;
        super.setWord(offset, data);
        regionAltered(offset, offset + 1);
    }

    public void setDoubleWord(int offset, int data)
    {
        if (super.getDoubleWord(offset) == data)
            return;
        super.setDoubleWord(offset, data);
        regionAltered(offset, offset + 3);
    }

    public void clear()
    {
	constructCodeBlocksArray();
	super.clear();
    }

    public String toString()
    {
        return "LazyCodeBlockMemory["+getSize()+"]";
    }
}
