; ****************************************************************************
; `bootload' - Boot manager's Master Bootstrap Loader
;   Written by Michal H. Tyc
;
; This file is part of `BOOTMGR - multi-BOOT ManaGeR'.
;
; Copyright (c) 1997-2006 BTTR Software.  All rights reserved.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the `MODIFIED' BSD LICENSE.  Please see `legal.txt'
; for details.
; ****************************************************************************
; This file is normally %include'd into the main file (bootmgr.asm), but it
; can be assembled separately as well, to create an MBR image file with
; default BOOTMGR settings (no bootable partitions defined).
; ****************************************************************************


; Equates and macros.

%ifndef VERSIONDATE    ; already defined if this file is %include'd
  %include 'defs.inc'  ; no need to read common definitions again
%endif


; This code will be placed at the beginning of master boot record.
; It is loaded by ROM BIOS Initial Program Loader at linear address 7c00h
; and then run.
; Note: this address is not officially guaranteed to be expressed as
; 0:7c00h (in other words, cs on entry must be treated as unknown).

; Boot sectors of bootable partitions are also loaded at this location,
; so this code must be relocated first to another address
; (traditionally it is 0:600h or 60h:0, we choose the former).

mbr:
  xor cx, cx
  cli              ; disable interrupts
  mov ss, cx       ;   while switching stack (ss = 0)
  mov sp, LOADADR  ; stack just below the code
  sti
  mov ds, cx
  mov es, cx       ; ds = es = 0
  mov si, sp
  mov di, RELOC
  push si          ; push return address for runboot
  mov ch, 1        ; 256 words
  cld              ; string direction UP from here
  rep movsw        ; relocate MBR code

; Newer BIOSes report boot drive number in dl.
; We check and save it to allow the loader to work also on HD2..HD9, etc.

  mov al, [475h]  ; number of hard disks in system
  or al, 80h
  xchg ax, dx
  cmp al, dl      ; check boot drive: allowed 80h to (7fh + [475h])
  jl .hdok        ; drive number ok (signed: 80h < 81h < ... < 0 < ... < 7fh)
  mov al, 80h     ; force HD1 if BIOS reports garbage
.hdok:
  mov [RELOC + rewrite.drv + 1 - mbr], al  ; patch drive number
  sub al, (80h - '1')                      ; make ASCII digit
  mov [RELOC + titletext + 3 - mbr], al    ; update title bar
  add al, (81h - '1')                      ; other (next) drive number
  cmp al, dl
  jl .nowrap
  mov al, 80h                              ; wrap to HD1
.nowrap:
  mov [RELOC + execboot.hdno + 1 - mbr], al  ; patch other drive number

IDLEN Equ .nowrap + 1 - mbr ; the block to be compared by installer

; Configure the display.
; It's unclear whether the int 10h, ah = 2 method of hiding cursor really
; works on all machines. However, a little loss of aesthetics on some exotic
; hardware isn't worth the extra bytes of code...

  mov al, [449h]  ; current video mode
  mov ch, 0b8h    ; cx = screen buffer segment (for color modes)
  test al, 4      ; mode 3 = color, 7 = monochrome
  je .color
  mov ch, 0b0h    ; monochrome: segment = 0b000h
.color:
  mov es, cx        ; es = screen buffer segment
  mov ch, 20h       ; cursor start (invisible), cl = 0
  mov ah, 1         ; al = video mode (workaround for BIOS bugs)
  int BIOSVIDEOINT  ; set cursor shape
  xor di, di        ; screen page 0 at offset 0
  mov ax, 07b1h     ; medium shade = background
  mov cx, 80 * 25   ; screen size in words
  rep stosw         ; fill

; Go to relocated start.

.item:
  mov bp, DEFIT                ; *** menu item to start with
  jmp 0:(RELOC + start - mbr)

sdefit Equ .item + 1 - mbr  ; relative offset for setup


; Print the title bar and all menu lines with shadows and blanks on sides.
; On entry:
;   bp = selected item
;   es = screen segment
; On exit:
;   cx = 0
;   al = ' '
;   dx = number of items (dh = 0)
;   cf = nc
;   si, di, ah = modified

printmenu:
  mov di, SCROFS(35, 10)           ; title position
  mov si, RELOC + titletext - mbr  ;   and text,
  mov dx, -1                       ; initialize line counter
.line:
  mov ah, 7    ; white on black
  cmp dx, bp
  je .bar      ; if selection bar,
  mov ah, 70h  ;   else black on white
.bar:
  mov cx, 1 << 9  ; a smart character counter
.chr:
  shr cx, 1    ; modify the counter
  mov al, ' '  ; one of blank edges
  jpe .spc     ;   if cl = 0,
  lodsb        ;   else get next character
.spc:
  stosw                                  ; store in screen buffer
  mov byte [es:di + SCROFS(1, 1)], 0b0h  ; `light shade' as shadow
  jne .chr                               ; all 10 characters done?
  inc dx
  add di, SCROFS(-10, 1)                 ; one row down, ten columns back
.items
  cmp dl, CITEMS        ; all menu items printed?
  jne .line
; ret

sitems Equ .items + 2 - mbr  ; relative offset for setup


; This ret is used also by sectorrw (to save space)

done:
  ret


; Start the boot manager.
; On entry:
;   bp = selected menu item

start:
  call printmenu  ; show menu
  mov bx, 46ch    ; system clock low word
.delay:
  JMPS readkbd.key  ; *** infinite timeout code variant
; mov dx, DEFTIM    ; dh = 0 after printmenu
; mov dl, DEFTIM    ; *** wait that long
.sec:
  mov di, SCROFS(43, 10)
  mov al, dl
  aam 10                 ; ah:al = (al / 10):(al % 10)
  or ax, '00'            ; convert to ASCII
  mov [es:di - 2], ah    ; print high digit
  stosb                  ; print low digit
  dec dx
  js execboot            ; no more time left
; mov cx, 18             ; ch = 0 after printmenu
  mov cl, 18             ; 18 ticks is approx. 1 sec
.tick:
  mov di, [bx]  ; can't simply wait for (t + 18), it won't work at midnight
.wait:
  mov ah, 1
  int BIOSKEYBINT  ; non-destructive read
  jne readkbd.upd  ; a key was pressed
  cmp di, [bx]
  je .wait         ; no tick so far
  loop .tick       ; wait 18 ticks
  JMPS .sec        ; and re-display the counter digits

sdelay Equ .delay + 1 - mbr  ; relative offset for setup

; Write new settings to Master Boot Record.
; On entry:
;   ah = active/hidden partitions flag:
;     bits: 7 6 5 4 3 2 1 0
;           | | | | | | | |__ #1 is active
;           | | | | | | |____ #2    "
;           | | | | | |______ #3    "
;           | | | | |________ #4    "
;           | | | |__________ #1 has ID with bit 4 set ("hidden" if FAT FS)
;           | | |____________ #2    "
;           | |______________ #3    "
;           |________________ #4    "
;   ch = 0
;   es = 0
; On exit:
;   bx = 600h
;   cx = 1
;   dh = 0
;   dl = boot drive
;   (see below)

rewrite:
.drv:
  mov dx, 80h               ; dl = boot drive (to be patched), dh = head = 0
  mov cl, 4                 ; number of entries in partition table
  mov di, RELOC + PARTTABL  ; address of the first entry
.nxt:
  and byte [di + 4], 0efh  ; clear ID bit 4
  mov al, ah
  and al, 10h              ; extract "hidden" bit
  or byte [di + 4], al     ; put it into ID
  shr ax, 1                ; shift ah and move "active" bit to b7 of al
  and al, 80h              ; extract "active" bit
  jns .na                  ; inactive?
  mov si, di               ; remember active partition data offset
.na:
  stosb              ; set active flag in partition table
  lea di, [di + 15]  ; next entry
  loop .nxt
  mov bx, RELOC      ; buffer offset = 600h
  inc cx             ; cylinder 0, sector 1 = MBR
  mov ah, 3          ; write


; Read or write a sector (CHS mode).
; On entry:
;   ah = command: 2 (read), 3 (write)
;   dl = drive: 0 (floppy), 7fh + n (HDn)
;   dh = head
;   cx = cylinder/sector in BIOS format
;   es:bx = data buffer
; On exit:
;   al = 1
;   di = modified (0 < di <= 3)
;   if an error occurs, Boot Manager is restarted

sectorrw:
  mov di, 3  ; retry count (give some time to diskette motor to spin up?)
  mov al, 1  ; 1 sector
.retry:
  push ax          ; save function number
  int BIOSDISKINT  ; read/write
  pop ax
  jnb done         ; cf = nc means OK
  push ax          ; save function number again
  cbw              ; ah = 0
  int BIOSDISKINT  ; reset drive
  pop ax
  dec di
  jne .retry       ; try once more?

; Signal an error and re-start.

error:
  mov sp, LOADADR - 4  ; restore stack pointer
  pop es               ; get screen buffer segment
  mov ax, 0e07h        ; teletype write ^G = ASCII <BELL>
  xor bx, bx           ; page 0
  push bp              ; old BIOSes may corrupt bp
  int BIOSVIDEOINT
  pop bp


; Process keyboard input.
; On entry:
;   bp = selected menu item
; On exit:
;   bp = new selected menu item
;   dx, si modified by printmenu only

readkbd:
.upd:
  call printmenu   ; update menu
.key:
; xor ah, ah
  cbw              ; al was 20h
  int BIOSKEYBINT  ; read key
  mov al, ah       ; use shorter opcode `cmp al, #' later
  cmp al, 48h      ; [Up-Arrow] key scan code
  je .decr
  cmp al, 50h      ; [Down-Arrow]
  je .incr
  cmp al, 1ch      ; [Enter]
  jne .upd         ; ignore other keys
; mov ch, 0        ; ch = 0 after printmenu
  JMPS execboot    ; boot now
.decr:
  dec bp    ; go to previous menu item
  jns .upd  ; not at the top
.incr:
  inc bp      ; go to next menu item
  cmp bp, dx  ; already at the bottom?
  je .decr    ; go back if so
  JMPS .upd   ; again

sdjump Equ .key - start.sec  ; relative jump distance for infinite timeout


; Boot from selected HD partition or other device.
; On entry:
;   ch = 0
;   bp = selected menu item
; Never return, on error restart boot manager

execboot:
  push es                       ; save screen segment
  push cs
  pop es                        ; es = 0, the buffer segment
  mov ah, [bp + RELOC + sparf]  ; partition hidden/active flags
  call rewrite                  ; update partition table in MBR, cx = al = 1
  mov bh, LOADADR >> 8          ; CHS buffer at 0:7c00h (bl was 0)
  push si                       ; save PT pointer
  xor al, [bp + RELOC + sdevf]  ; boot device flag
  js .hd                        ; other device?
; mov dh, 0                     ; dh = head = 0: not needed after rewrite
; mov cx, 1                     ; cylinder 0, sector 1 = MBR
.hdno:
  mov dl, 80h  ; other hard disk -- to be patched
  dec ax       ; now al = 0 for floppy and 0ffh for other HD
  and dl, al   ; dl unchanged if HD, zeroed if floppy
  JMPS .chsrd
.hd:
  lodsw
  mov dh, ah          ; partition start head
  push si             ; si needed temporarily...
  push es             ; LBA address packet on the stack: 0,
  push es             ;   0,
  mov di, [si + 8]
  push di             ;   LBA sector hi,
  push word [si + 6]  ;   LBA sector lo: qword sector
  push es
  push bx             ; buffer segment:offset
; mov cx, 1           ; not needed after rewrite
  push cx             ; number of sectors
  mov cl, 16
  push cx             ; packet size
  mov si, sp          ; ds:si = LBA address packet (ss = ds)
  mov ah, 42h
  int BIOSDISKINT     ; extended disk read
  lea sp, [si + 16]   ; clean the stack
  pop si              ; restore
  jnb .boot           ; success
  xchg ax, di
  test ah, ah
  jne error           ; sector number > 0ffffffh, cannot use CHS
  mov cx, [si]        ; partition start cylinder/sector
.chsrd:
  mov ah, 2      ; read
  call sectorrw  ; read partition boot sector
.boot:
  cmp word [LOADADR + PARTSIGN], ISVALIDPART  ; check if a valid
  jne error                                   ;   bootable partition


; Run the partition boot loader from its first sector.
; The loader is called with: dl = drive number,
; ds:si still points to partition data at 7beh, 7ceh, 7deh or 7eeh,
; as in standard DOS loaders (for maximum compatibility), es:bx = 0:7c00h.

runboot:
  mov al, [449h]    ; current video mode (should be 3 or 7)
  cbw               ; ah = 0, function `set mode and clear screen'
  push bx
  int BIOSVIDEOINT  ; old BIOSes may corrupt si, bx, but probably not dx
  pop bx
  pop si            ; ds:si = partition info
  pop ax            ; throw away screen buffer segment
  ret               ; run boot code (0:7c00h)

; Filler to 512 bytes (if any).

  Times (PARTTABL - 6 - CFGLEN - 8 - ($ - mbr)) nop

; THE CODE BELOW THIS LINE CANNOT BE MOVED!

; Title bar (8 characters).

titletext:
  Db '[HD1]   '

; Space for menu items (NITEMS * 8 characters).

smenu Equ $ - mbr  ; relative offset for setup

  Times (NITEMS * 8) Db ' '  ; ***

; Device type and hidden/active flags (see rewrite) for the menu items.

sparf Equ $ - mbr  ; relative offset for setup

  Times NITEMS Db 0  ; ***

sdevf Equ $ - mbr  ; relative offset for setup

  Times NITEMS Db OTHERID  ; ***

; end of BOOTMGR area (offset 1b8h)

%ifndef VERSIONDATE  ; only in MBR image file

  Db 0, 0, 0, 0        ; reserved for NT Volume Bytes
  Dw 0                 ; reserved
  Times (4 * 16) Db 0  ; empty partition table
  Dw ISVALIDPART       ; valid partition table signature

%endif

; (end of bootload.inc)
