/****************************************************************************

code.c - Source code for the code command

AUTHOR: Gregory Pietsch
CREATED Sat Aug 01 14:58:14 1998

DESCRIPTION:

code performs the same functions as uuencode, uudecode, xxencode, xxdecode --
it turns a binary file into a printable one suitable for transmission via
e-mail, etc. or back again.

COPYRIGHT NOTICE AND DISCLAIMER:

Copyright (C) 1998 Gregory Pietsch

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write:

The Free Software Foundation, Inc.
59 Temple Place, Suite 330
Cambridge, Massachusetts  02139  USA

Here's my e-mail address, should one need to contact me:

Gregory Pietsch
gkp1@flash.net

****************************************************************************/

/* include files */
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#ifndef GETOPT_H
#include "getopt.h"
#endif

/* macros */
#define LINE_LENGTH     256

#define IN_LINE_LENGTH  45
#define OUT_LINE_LENGTH 61

/* types */
typedef enum DECODE_STATE_T {
    NONE,
    OUTSIDE_FILE,
    INSIDE_FILE,
    SAW_EOF
} DECODE_STATE_T;

/* statically-defined variables */

static int flag_xx = 0;
static int flag_encode = 0;
static int flag_quiet = 0;
static int show_help = 0;
static int show_version = 0;
static char *shortopts = "eqx";
static GETOPT_LONG_OPTION_T longopts[] =
{
    {"encode", NO_ARG, NULL, 'e'},
    {"quiet", NO_ARG, NULL, 'q'},
    {"silent", NO_ARG, NULL, 'q'},
    {"xx", NO_ARG, NULL, 'x'},

    {"help", NO_ARG, &show_help, 1},
    {"version", NO_ARG, &show_version, 1},
    {NULL, 0, 0, 0}
};
static char *program_name;
static char *uucode =
"`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_";
static char *xxcode =
"+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
static int decode[UCHAR_MAX + 1];
static char *encode;

/* functions */

/* allocate memory, die on error */
void *xmalloc(size_t n)
{
    void *p = malloc(n);

    if (p == NULL) {
	fprintf(stderr, "%s: out of memory\n", program_name);
	exit(EXIT_FAILURE);
    }
    return p;
}

/* reallocate memory, die on error */
void *xrealloc(void *p, size_t n)
{
    void *s;

    if (n == 0) {
	if (p != NULL)
	    free(p);
	return NULL;
    }
    if (p == NULL)
	return xmalloc(n);
    s = realloc(p, n);
    if (s == NULL) {
	fprintf(stderr, "%s: out of memory\n", program_name);
	exit(EXIT_FAILURE);
    }
    return s;
}

/* encode a string */
void encode_string(char *dest, char *source, int length)
{
    while ((length % 3) != 0)
	source[length++] = '\0';
    while (length > 0) {
	*dest++ = encode[(source[0] >> 2) & 0x3F];
	*dest++ = encode[((source[0] << 4) & 0x30) | ((source[1] >> 4) & 0xF)];
	*dest++ = encode[((source[1] << 2) & 0x3C) | ((source[2] >> 6) & 0x3)];
	*dest++ = encode[source[2] & 0x3F];
	source += 3;
	length -= 3;
    }
    *dest = '\0';
}

/* decode a string */
void decode_string(char *dest, char *source, int length)
{
    int digit;
    int count = 0;
    unsigned long num;
    int i;
    char *d = dest;

    while (count < length) {
	i = 0;
	digit = 0;
	num = 0;
	while (i < 4) {
	    digit = decode[*source++];
	    if (digit < 0)
		digit = 0;
	    num = num * 64 + digit;
	    i++;
	}
	for (i = 0; i < 3; i++)
	    *dest++ = (num >> (16 - i * 8)) & 0xFF;
	count += 3;
    }
    d[length] = '\0';
}

/* encode a file */
void encode_file(char *filename)
{
    FILE *f = fopen(filename, "rb");
    char src[IN_LINE_LENGTH + 3];
    char dest[OUT_LINE_LENGTH + 1];
    size_t num_read;

    if (f == NULL) {
	fprintf(stderr, "%s: Can't open %s\n", program_name, filename);
	abort();
    }
    printf("begin 644 %s\n", filename);
    while ((num_read = fread(src, 1, IN_LINE_LENGTH, f)) != 0) {
	putc(encode[num_read], stdout);
	memset(src + num_read, '\0', 3);
	encode_string(dest, src, num_read);
	puts(dest);
    }
    printf("%c\nend\n", *encode);
    fclose(f);
}

/* read a line out of a file dynamically */
char *readline(FILE * f)
{
    char *s = xmalloc(LINE_LENGTH);
    size_t slen = LINE_LENGTH;
    size_t i = 0;
    int c;

    while ((c = getc(f)) != '\n' && c != EOF) {
	if (i == slen - 1) {
	    slen += LINE_LENGTH;
	    s = xrealloc(s, slen);
	}
	s[i++] = c;
    }
    if (c == EOF) {
	free(s);
	return NULL;
    }
    else {
	s[i] = '\0';
	s = xrealloc(s, i + 1);
	return s;
    }
}

/* decode a file */
void decode_file(char *infilename)
{
    int i;
    char *name;
    FILE *inf = fopen(infilename, "r");
    FILE *outf = NULL;
    char *s;
    char *t;
    DECODE_STATE_T decode_state = OUTSIDE_FILE;
    int length;

    if (inf == NULL) {
	fprintf(stderr, "%s: Can't open %s\n", program_name, infilename);
	abort();
    }
    /* first, create translation table */
    for (i = 0; i < UCHAR_MAX + 1; i++)
	decode[i] = -1;
    for (i = 0; i < 64; i++)
	decode[encode[i]] = decode[encode[i] | 0x80] = i;
    while (1) {
	s = readline(inf);
	if (s == NULL)
	    break;
	switch (decode_state) {
	case OUTSIDE_FILE:
	    t = strtok(s, " \t\v\f\n");
	    if (strcmp(t, "begin") != 0) {
		free(s);
		break;
	    }
	    t = strtok(NULL, " \t\v\f\n");
	    if (!isdigit(*t) || !isdigit(t[1]) || !isdigit(t[2])) {
		free(s);
		break;
	    }
	    t = strtok(NULL, " \t\v\f\n");	/* t should be a filename */
	    outf = fopen(t, "wb");
	    if (outf == NULL) {
		fprintf(stderr, "%s: Can't open %s\n",
			program_name, t);
		abort();
	    }
	    if (!flag_quiet)
		puts(t);
	    decode_state = INSIDE_FILE;
	    break;
	case INSIDE_FILE:
	    length = decode[*s];
	    if (length == 0) {
		decode_state = SAW_EOF;
		break;
	    }
	    t = xmalloc(length + 1);
	    decode_string(t, s + 1, length);
	    fwrite(t, 1, (size_t) length, outf);
	    free(t);
	    break;
	case SAW_EOF:
	    fclose(outf);
	    outf = NULL;
	    decode_state = OUTSIDE_FILE;
	    break;
	}
	free(s);
    }
}

void parse_args(int argc, char **argv)
{
    int opt;

    do {
	switch ((opt = getopt_long(argc, argv, shortopts, longopts, NULL))) {
	case 'e':		/* encode */
	    flag_encode = 1;
	    break;
        case 'q':               /* quiet */
            flag_quiet = 1;
	    break;
	case 'x':		/* xxencode/xxdecode */
	    flag_xx = 1;
	    break;
        case '?':               /* invalid option */
            fprintf(stderr, "For help, type:\n\t%s --help\n", program_name);
            exit(EXIT_FAILURE);
	case 1:
	case 0:
	    if (show_help || show_version) {
		if (show_help)
		    help();
		if (show_version)
		    version();
		exit(EXIT_SUCCESS);
	    }
	    break;
	default:
	    break;
	}
    } while (opt != EOF);
    encode = flag_xx ? xxcode : uucode;
}

int main(int argc, char **argv)
{
    int i;

    program_name = argv[0];
    parse_args(argc, argv);
    for (i = optind; i < argc; i++)
	flag_encode ? encode_file(argv[i]) : decode_file(argv[i]);
    return EXIT_SUCCESS;
}

/* END OF FILE code.c */
