/* Integer trig functions.
 * In our representation, an angle is specified as in the X protocol,
 * the angle in degrees * 64.  The value of sin and cos are returned
 * as integers between -65536 and 65536.
 */
#define FULLCIRCLE (360 * 64)
#define HALFCIRCLE (360 * 32)
#define QUARTERCIRCLE (360 * 16)

typedef unsigned short u_short;

# define mod(a,b)	((a) >= 0 ? (a) % (b) : (b) - (-a) % (b))

u_short sin_table[] = {
	/* 0 degree */
	0, 36, 71, 107, 143, 179, 214, 250,
	286, 322, 357, 393, 429, 465, 500, 536,
	572, 608, 643, 679, 715, 751, 786, 822,
	858, 894, 929, 965, 1001, 1037, 1072, 1108,

	/* 1 degree */
	1144, 1179, 1215, 1251, 1287, 1322, 1358, 1394,
	1430, 1465, 1501, 1537, 1573, 1608, 1644, 1680,
	1716, 1751, 1787, 1823, 1858, 1894, 1930, 1966,
	2001, 2037, 2073, 2109, 2144, 2180, 2216, 2251,

	/* 2 degrees */
	2287, 2323, 2359, 2394, 2430, 2466, 2501, 2537,
	2573, 2609, 2644, 2680, 2716, 2752, 2787, 2823,
	2859, 2894, 2930, 2966, 3001, 3037, 3073, 3109,
	3144, 3180, 3216, 3251, 3287, 3323, 3358, 3394,

	/* 3 degrees */
	3430, 3466, 3501, 3537, 3573, 3608, 3644, 3680,
	3715, 3751, 3787, 3822, 3858, 3894, 3930, 3965,
	4001, 4037, 4072, 4108, 4144, 4179, 4215, 4251,
	4286, 4322, 4358, 4393, 4429, 4465, 4500, 4536,

	4572, 4607, 4643, 4679, 4714, 4750, 4785, 4821,
	4857, 4892, 4928, 4964, 4999, 5035, 5071, 5106,
	5142, 5178, 5213, 5249, 5284, 5320, 5356, 5391,
	5427, 5463, 5498, 5534, 5569, 5605, 5641, 5676,

	5712, 5747, 5783, 5819, 5854, 5890, 5925, 5961,
	5997, 6032, 6068, 6103, 6139, 6175, 6210, 6246,
	6281, 6317, 6353, 6388, 6424, 6459, 6495, 6530,
	6566, 6601, 6637, 6673, 6708, 6744, 6779, 6815,

	6850, 6886, 6921, 6957, 6993, 7028, 7064, 7099,
	7135, 7170, 7206, 7241, 7277, 7312, 7348, 7383,
	7419, 7454, 7490, 7525, 7561, 7596, 7632, 7667,
	7703, 7738, 7774, 7809, 7845, 7880, 7916, 7951,

	7987, 8022, 8058, 8093, 8129, 8164, 8200, 8235,
	8271, 8306, 8341, 8377, 8412, 8448, 8483, 8519,
	8554, 8590, 8625, 8660, 8696, 8731, 8767, 8802,
	8838, 8873, 8908, 8944, 8979, 9015, 9050, 9085,

	9121, 9156, 9192, 9227, 9262, 9298, 9333, 9369,
	9404, 9439, 9475, 9510, 9545, 9581, 9616, 9651,
	9687, 9722, 9758, 9793, 9828, 9864, 9899, 9934,
	9970, 10005, 10040, 10076, 10111, 10146, 10181, 10217,

	10252, 10287, 10323, 10358, 10393, 10429, 10464, 10499,
	10534, 10570, 10605, 10640, 10676, 10711, 10746, 10781,
	10817, 10852, 10887, 10922, 10958, 10993, 11028, 11063,
	11098, 11134, 11169, 11204, 11239, 11275, 11310, 11345,

	11380, 11415, 11451, 11486, 11521, 11556, 11591, 11627,
	11662, 11697, 11732, 11767, 11802, 11838, 11873, 11908,
	11943, 11978, 12013, 12048, 12084, 12119, 12154, 12189,
	12224, 12259, 12294, 12329, 12364, 12400, 12435, 12470,

	12505, 12540, 12575, 12610, 12645, 12680, 12715, 12750,
	12785, 12820, 12856, 12891, 12926, 12961, 12996, 13031,
	13066, 13101, 13136, 13171, 13206, 13241, 13276, 13311,
	13346, 13381, 13416, 13451, 13486, 13521, 13556, 13591,

	13626, 13661, 13696, 13731, 13766, 13800, 13835, 13870,
	13905, 13940, 13975, 14010, 14045, 14080, 14115, 14150,
	14185, 14219, 14254, 14289, 14324, 14359, 14394, 14429,
	14464, 14498, 14533, 14568, 14603, 14638, 14673, 14708,

	14742, 14777, 14812, 14847, 14882, 14916, 14951, 14986,
	15021, 15056, 15090, 15125, 15160, 15195, 15230, 15264,
	15299, 15334, 15369, 15403, 15438, 15473, 15508, 15542,
	15577, 15612, 15646, 15681, 15716, 15751, 15785, 15820,

	15855, 15889, 15924, 15959, 15993, 16028, 16063, 16097,
	16132, 16167, 16201, 16236, 16270, 16305, 16340, 16374,
	16409, 16444, 16478, 16513, 16547, 16582, 16616, 16651,
	16686, 16720, 16755, 16789, 16824, 16858, 16893, 16927,

	16962, 16996, 17031, 17066, 17100, 17135, 17169, 17204,
	17238, 17272, 17307, 17341, 17376, 17410, 17445, 17479,
	17514, 17548, 17583, 17617, 17651, 17686, 17720, 17755,
	17789, 17824, 17858, 17892, 17927, 17961, 17995, 18030,

	18064, 18099, 18133, 18167, 18202, 18236, 18270, 18305,
	18339, 18373, 18407, 18442, 18476, 18510, 18545, 18579,
	18613, 18647, 18682, 18716, 18750, 18785, 18819, 18853,
	18887, 18921, 18956, 18990, 19024, 19058, 19092, 19127,

	19161, 19195, 19229, 19263, 19298, 19332, 19366, 19400,
	19434, 19468, 19502, 19537, 19571, 19605, 19639, 19673,
	19707, 19741, 19775, 19809, 19843, 19877, 19911, 19946,
	19980, 20014, 20048, 20082, 20116, 20150, 20184, 20218,

	20252, 20286, 20320, 20354, 20388, 20422, 20456, 20490,
	20524, 20557, 20591, 20625, 20659, 20693, 20727, 20761,
	20795, 20829, 20863, 20897, 20930, 20964, 20998, 21032,
	21066, 21100, 21134, 21167, 21201, 21235, 21269, 21303,

	21336, 21370, 21404, 21438, 21472, 21505, 21539, 21573,
	21607, 21640, 21674, 21708, 21742, 21775, 21809, 21843,
	21876, 21910, 21944, 21977, 22011, 22045, 22078, 22112,
	22146, 22179, 22213, 22247, 22280, 22314, 22347, 22381,

	22415, 22448, 22482, 22515, 22549, 22582, 22616, 22650,
	22683, 22717, 22750, 22784, 22817, 22851, 22884, 22918,
	22951, 22985, 23018, 23052, 23085, 23119, 23152, 23185,
	23219, 23252, 23286, 23319, 23352, 23386, 23419, 23453,

	23486, 23519, 23553, 23586, 23619, 23653, 23686, 23719,
	23753, 23786, 23819, 23853, 23886, 23919, 23952, 23986,
	24019, 24052, 24086, 24119, 24152, 24185, 24218, 24252,
	24285, 24318, 24351, 24384, 24418, 24451, 24484, 24517,

	24550, 24583, 24616, 24650, 24683, 24716, 24749, 24782,
	24815, 24848, 24881, 24914, 24947, 24980, 25013, 25047,
	25080, 25113, 25146, 25179, 25212, 25245, 25278, 25311,
	25343, 25376, 25409, 25442, 25475, 25508, 25541, 25574,

	25607, 25640, 25673, 25706, 25739, 25771, 25804, 25837,
	25870, 25903, 25936, 25968, 26001, 26034, 26067, 26100,
	26132, 26165, 26198, 26231, 26263, 26296, 26329, 26362,
	26394, 26427, 26460, 26493, 26525, 26558, 26591, 26623,

	26656, 26689, 26721, 26754, 26786, 26819, 26852, 26884,
	26917, 26949, 26982, 27015, 27047, 27080, 27112, 27145,
	27177, 27210, 27242, 27275, 27307, 27340, 27372, 27405,
	27437, 27470, 27502, 27535, 27567, 27599, 27632, 27664,

	27697, 27729, 27761, 27794, 27826, 27859, 27891, 27923,
	27956, 27988, 28020, 28053, 28085, 28117, 28149, 28182,
	28214, 28246, 28278, 28311, 28343, 28375, 28407, 28440,
	28472, 28504, 28536, 28568, 28601, 28633, 28665, 28697,

	28729, 28761, 28793, 28825, 28858, 28890, 28922, 28954,
	28986, 29018, 29050, 29082, 29114, 29146, 29178, 29210,
	29242, 29274, 29306, 29338, 29370, 29402, 29434, 29466,
	29498, 29530, 29561, 29593, 29625, 29657, 29689, 29721,

	29753, 29785, 29816, 29848, 29880, 29912, 29944, 29975,
	30007, 30039, 30071, 30103, 30134, 30166, 30198, 30229,
	30261, 30293, 30325, 30356, 30388, 30420, 30451, 30483,
	30515, 30546, 30578, 30609, 30641, 30673, 30704, 30736,

	30767, 30799, 30830, 30862, 30893, 30925, 30956, 30988,
	31019, 31051, 31082, 31114, 31145, 31177, 31208, 31240,
	31271, 31302, 31334, 31365, 31397, 31428, 31459, 31491,
	31522, 31553, 31585, 31616, 31647, 31679, 31710, 31741,

	31772, 31804, 31835, 31866, 31897, 31929, 31960, 31991,
	32022, 32053, 32085, 32116, 32147, 32178, 32209, 32240,
	32271, 32303, 32334, 32365, 32396, 32427, 32458, 32489,
	32520, 32551, 32582, 32613, 32644, 32675, 32706, 32737,

	32768, 32799, 32830, 32861, 32892, 32923, 32954, 32984,
	33015, 33046, 33077, 33108, 33139, 33170, 33200, 33231,
	33262, 33293, 33324, 33354, 33385, 33416, 33447, 33477,
	33508, 33539, 33570, 33600, 33631, 33662, 33692, 33723,

	33754, 33784, 33815, 33845, 33876, 33907, 33937, 33968,
	33998, 34029, 34059, 34090, 34120, 34151, 34181, 34212,
	34242, 34273, 34303, 34334, 34364, 34395, 34425, 34456,
	34486, 34516, 34547, 34577, 34607, 34638, 34668, 34698,

	34729, 34759, 34789, 34820, 34850, 34880, 34910, 34941,
	34971, 35001, 35031, 35062, 35092, 35122, 35152, 35182,
	35212, 35243, 35273, 35303, 35333, 35363, 35393, 35423,
	35453, 35483, 35513, 35543, 35573, 35603, 35633, 35663,

	35693, 35723, 35753, 35783, 35813, 35843, 35873, 35903,
	35933, 35963, 35993, 36023, 36052, 36082, 36112, 36142,
	36172, 36202, 36231, 36261, 36291, 36321, 36350, 36380,
	36410, 36440, 36469, 36499, 36529, 36558, 36588, 36618,

	36647, 36677, 36707, 36736, 36766, 36795, 36825, 36854,
	36884, 36914, 36943, 36973, 37002, 37032, 37061, 37091,
	37120, 37149, 37179, 37208, 37238, 37267, 37297, 37326,
	37355, 37385, 37414, 37443, 37473, 37502, 37531, 37561,

	37590, 37619, 37648, 37678, 37707, 37736, 37765, 37795,
	37824, 37853, 37882, 37911, 37940, 37970, 37999, 38028,
	38057, 38086, 38115, 38144, 38173, 38202, 38231, 38260,
	38289, 38318, 38347, 38376, 38405, 38434, 38463, 38492,

	38521, 38550, 38579, 38608, 38637, 38666, 38694, 38723,
	38752, 38781, 38810, 38838, 38867, 38896, 38925, 38954,
	38982, 39011, 39040, 39068, 39097, 39126, 39154, 39183,
	39212, 39240, 39269, 39298, 39326, 39355, 39383, 39412,

	39441, 39469, 39498, 39526, 39555, 39583, 39612, 39640,
	39669, 39697, 39725, 39754, 39782, 39811, 39839, 39867,
	39896, 39924, 39952, 39981, 40009, 40037, 40066, 40094,
	40122, 40151, 40179, 40207, 40235, 40263, 40292, 40320,

	40348, 40376, 40404, 40432, 40461, 40489, 40517, 40545,
	40573, 40601, 40629, 40657, 40685, 40713, 40741, 40769,
	40797, 40825, 40853, 40881, 40909, 40937, 40965, 40993,
	41021, 41048, 41076, 41104, 41132, 41160, 41188, 41215,

	41243, 41271, 41299, 41326, 41354, 41382, 41410, 41437,
	41465, 41493, 41520, 41548, 41576, 41603, 41631, 41658,
	41686, 41714, 41741, 41769, 41796, 41824, 41851, 41879,
	41906, 41934, 41961, 41989, 42016, 42044, 42071, 42098,

	42126, 42153, 42180, 42208, 42235, 42262, 42290, 42317,
	42344, 42372, 42399, 42426, 42453, 42481, 42508, 42535,
	42562, 42589, 42617, 42644, 42671, 42698, 42725, 42752,
	42779, 42806, 42833, 42860, 42887, 42914, 42942, 42969,

	42995, 43022, 43049, 43076, 43103, 43130, 43157, 43184,
	43211, 43238, 43265, 43291, 43318, 43345, 43372, 43399,
	43425, 43452, 43479, 43506, 43532, 43559, 43586, 43613,
	43639, 43666, 43693, 43719, 43746, 43772, 43799, 43826,

	43852, 43879, 43905, 43932, 43958, 43985, 44011, 44038,
	44064, 44091, 44117, 44144, 44170, 44196, 44223, 44249,
	44275, 44302, 44328, 44354, 44381, 44407, 44433, 44460,
	44486, 44512, 44538, 44565, 44591, 44617, 44643, 44669,

	44695, 44722, 44748, 44774, 44800, 44826, 44852, 44878,
	44904, 44930, 44956, 44982, 45008, 45034, 45060, 45086,
	45112, 45138, 45164, 45190, 45216, 45241, 45267, 45293,
	45319, 45345, 45371, 45396, 45422, 45448, 45474, 45499,

	45525, 45551, 45577, 45602, 45628, 45654, 45679, 45705,
	45730, 45756, 45782, 45807, 45833, 45858, 45884, 45909,
	45935, 45960, 45986, 46011, 46037, 46062, 46088, 46113,
	46138, 46164, 46189, 46214, 46240, 46265, 46290, 46316,

	46341, 46366, 46391, 46417, 46442, 46467, 46492, 46518,
	46543, 46568, 46593, 46618, 46643, 46668, 46693, 46719,
	46744, 46769, 46794, 46819, 46844, 46869, 46894, 46919,
	46944, 46968, 46993, 47018, 47043, 47068, 47093, 47118,

	47143, 47167, 47192, 47217, 47242, 47267, 47291, 47316,
	47341, 47366, 47390, 47415, 47440, 47464, 47489, 47514,
	47538, 47563, 47587, 47612, 47636, 47661, 47686, 47710,
	47735, 47759, 47783, 47808, 47832, 47857, 47881, 47906,

	47930, 47954, 47979, 48003, 48027, 48052, 48076, 48100,
	48125, 48149, 48173, 48197, 48221, 48246, 48270, 48294,
	48318, 48342, 48366, 48391, 48415, 48439, 48463, 48487,
	48511, 48535, 48559, 48583, 48607, 48631, 48655, 48679,

	48703, 48727, 48751, 48774, 48798, 48822, 48846, 48870,
	48894, 48917, 48941, 48965, 48989, 49012, 49036, 49060,
	49084, 49107, 49131, 49155, 49178, 49202, 49225, 49249,
	49273, 49296, 49320, 49343, 49367, 49390, 49414, 49437,

	49461, 49484, 49508, 49531, 49554, 49578, 49601, 49624,
	49648, 49671, 49694, 49718, 49741, 49764, 49788, 49811,
	49834, 49857, 49880, 49904, 49927, 49950, 49973, 49996,
	50019, 50042, 50065, 50088, 50111, 50134, 50158, 50181,

	50203, 50226, 50249, 50272, 50295, 50318, 50341, 50364,
	50387, 50410, 50433, 50455, 50478, 50501, 50524, 50546,
	50569, 50592, 50615, 50637, 50660, 50683, 50705, 50728,
	50751, 50773, 50796, 50818, 50841, 50863, 50886, 50909,

	50931, 50954, 50976, 50998, 51021, 51043, 51066, 51088,
	51111, 51133, 51155, 51178, 51200, 51222, 51244, 51267,
	51289, 51311, 51333, 51356, 51378, 51400, 51422, 51444,
	51467, 51489, 51511, 51533, 51555, 51577, 51599, 51621,

	51643, 51665, 51687, 51709, 51731, 51753, 51775, 51797,
	51819, 51841, 51862, 51884, 51906, 51928, 51950, 51971,
	51993, 52015, 52037, 52058, 52080, 52102, 52123, 52145,
	52167, 52188, 52210, 52232, 52253, 52275, 52296, 52318,

	52339, 52361, 52382, 52404, 52425, 52447, 52468, 52490,
	52511, 52532, 52554, 52575, 52596, 52618, 52639, 52660,
	52682, 52703, 52724, 52745, 52766, 52788, 52809, 52830,
	52851, 52872, 52893, 52914, 52936, 52957, 52978, 52999,

	53020, 53041, 53062, 53083, 53104, 53125, 53146, 53166,
	53187, 53208, 53229, 53250, 53271, 53292, 53312, 53333,
	53354, 53375, 53395, 53416, 53437, 53457, 53478, 53499,
	53519, 53540, 53561, 53581, 53602, 53622, 53643, 53663,

	53684, 53704, 53725, 53745, 53766, 53786, 53807, 53827,
	53847, 53868, 53888, 53909, 53929, 53949, 53969, 53990,
	54010, 54030, 54050, 54071, 54091, 54111, 54131, 54151,
	54171, 54191, 54212, 54232, 54252, 54272, 54292, 54312,

	54332, 54352, 54372, 54392, 54412, 54432, 54451, 54471,
	54491, 54511, 54531, 54551, 54570, 54590, 54610, 54630,
	54650, 54669, 54689, 54709, 54728, 54748, 54768, 54787,
	54807, 54826, 54846, 54866, 54885, 54905, 54924, 54944,

	54963, 54983, 55002, 55021, 55041, 55060, 55080, 55099,
	55118, 55138, 55157, 55176, 55196, 55215, 55234, 55253,
	55273, 55292, 55311, 55330, 55349, 55368, 55387, 55407,
	55426, 55445, 55464, 55483, 55502, 55521, 55540, 55559,

	55578, 55597, 55616, 55634, 55653, 55672, 55691, 55710,
	55729, 55747, 55766, 55785, 55804, 55823, 55841, 55860,
	55879, 55897, 55916, 55935, 55953, 55972, 55990, 56009,
	56028, 56046, 56065, 56083, 56102, 56120, 56138, 56157,

	56175, 56194, 56212, 56230, 56249, 56267, 56285, 56304,
	56322, 56340, 56359, 56377, 56395, 56413, 56431, 56450,
	56468, 56486, 56504, 56522, 56540, 56558, 56576, 56594,
	56612, 56630, 56648, 56666, 56684, 56702, 56720, 56738,

	56756, 56774, 56792, 56809, 56827, 56845, 56863, 56881,
	56898, 56916, 56934, 56951, 56969, 56987, 57004, 57022,
	57040, 57057, 57075, 57092, 57110, 57127, 57145, 57162,
	57180, 57197, 57215, 57232, 57250, 57267, 57284, 57302,

	57319, 57336, 57354, 57371, 57388, 57406, 57423, 57440,
	57457, 57474, 57492, 57509, 57526, 57543, 57560, 57577,
	57594, 57611, 57628, 57645, 57662, 57679, 57696, 57713,
	57730, 57747, 57764, 57781, 57798, 57814, 57831, 57848,

	57865, 57882, 57898, 57915, 57932, 57949, 57965, 57982,
	57999, 58015, 58032, 58048, 58065, 58082, 58098, 58115,
	58131, 58148, 58164, 58181, 58197, 58213, 58230, 58246,
	58263, 58279, 58295, 58312, 58328, 58344, 58361, 58377,

	58393, 58409, 58425, 58442, 58458, 58474, 58490, 58506,
	58522, 58538, 58554, 58570, 58586, 58602, 58618, 58634,
	58650, 58666, 58682, 58698, 58714, 58730, 58746, 58762,
	58777, 58793, 58809, 58825, 58841, 58856, 58872, 58888,

	58903, 58919, 58935, 58950, 58966, 58981, 58997, 59013,
	59028, 59044, 59059, 59075, 59090, 59106, 59121, 59136,
	59152, 59167, 59183, 59198, 59213, 59229, 59244, 59259,
	59274, 59290, 59305, 59320, 59335, 59350, 59366, 59381,

	59396, 59411, 59426, 59441, 59456, 59471, 59486, 59501,
	59516, 59531, 59546, 59561, 59576, 59591, 59606, 59620,
	59635, 59650, 59665, 59680, 59694, 59709, 59724, 59739,
	59753, 59768, 59783, 59797, 59812, 59826, 59841, 59856,

	59870, 59885, 59899, 59914, 59928, 59943, 59957, 59971,
	59986, 60000, 60015, 60029, 60043, 60058, 60072, 60086,
	60100, 60115, 60129, 60143, 60157, 60171, 60186, 60200,
	60214, 60228, 60242, 60256, 60270, 60284, 60298, 60312,

	60326, 60340, 60354, 60368, 60382, 60396, 60410, 60424,
	60437, 60451, 60465, 60479, 60493, 60506, 60520, 60534,
	60547, 60561, 60575, 60588, 60602, 60616, 60629, 60643,
	60656, 60670, 60683, 60697, 60710, 60724, 60737, 60751,

	60764, 60777, 60791, 60804, 60817, 60831, 60844, 60857,
	60870, 60884, 60897, 60910, 60923, 60936, 60950, 60963,
	60976, 60989, 61002, 61015, 61028, 61041, 61054, 61067,
	61080, 61093, 61106, 61119, 61132, 61145, 61157, 61170,

	61183, 61196, 61209, 61221, 61234, 61247, 61260, 61272,
	61285, 61298, 61310, 61323, 61336, 61348, 61361, 61373,
	61386, 61398, 61411, 61423, 61436, 61448, 61461, 61473,
	61485, 61498, 61510, 61522, 61535, 61547, 61559, 61571,

	61584, 61596, 61608, 61620, 61632, 61645, 61657, 61669,
	61681, 61693, 61705, 61717, 61729, 61741, 61753, 61765,
	61777, 61789, 61801, 61813, 61825, 61836, 61848, 61860,
	61872, 61884, 61895, 61907, 61919, 61931, 61942, 61954,

	61966, 61977, 61989, 62000, 62012, 62023, 62035, 62047,
	62058, 62069, 62081, 62092, 62104, 62115, 62127, 62138,
	62149, 62161, 62172, 62183, 62195, 62206, 62217, 62228,
	62239, 62251, 62262, 62273, 62284, 62295, 62306, 62317,

	62328, 62339, 62350, 62361, 62372, 62383, 62394, 62405,
	62416, 62427, 62438, 62449, 62460, 62470, 62481, 62492,
	62503, 62514, 62524, 62535, 62546, 62556, 62567, 62578,
	62588, 62599, 62609, 62620, 62630, 62641, 62651, 62662,

	62672, 62683, 62693, 62704, 62714, 62724, 62735, 62745,
	62755, 62766, 62776, 62786, 62796, 62807, 62817, 62827,
	62837, 62847, 62857, 62868, 62878, 62888, 62898, 62908,
	62918, 62928, 62938, 62948, 62958, 62968, 62978, 62987,

	62997, 63007, 63017, 63027, 63037, 63046, 63056, 63066,
	63075, 63085, 63095, 63104, 63114, 63124, 63133, 63143,
	63152, 63162, 63172, 63181, 63191, 63200, 63209, 63219,
	63228, 63238, 63247, 63256, 63266, 63275, 63284, 63294,

	63303, 63312, 63321, 63331, 63340, 63349, 63358, 63367,
	63376, 63385, 63394, 63404, 63413, 63422, 63431, 63440,
	63449, 63457, 63466, 63475, 63484, 63493, 63502, 63511,
	63520, 63528, 63537, 63546, 63555, 63563, 63572, 63581,

	63589, 63598, 63607, 63615, 63624, 63632, 63641, 63649,
	63658, 63666, 63675, 63683, 63692, 63700, 63709, 63717,
	63725, 63734, 63742, 63750, 63758, 63767, 63775, 63783,
	63791, 63800, 63808, 63816, 63824, 63832, 63840, 63848,

	63856, 63864, 63872, 63880, 63888, 63896, 63904, 63912,
	63920, 63928, 63936, 63944, 63951, 63959, 63967, 63975,
	63983, 63990, 63998, 64006, 64013, 64021, 64029, 64036,
	64044, 64051, 64059, 64066, 64074, 64082, 64089, 64096,

	64104, 64111, 64119, 64126, 64133, 64141, 64148, 64155,
	64163, 64170, 64177, 64184, 64192, 64199, 64206, 64213,
	64220, 64227, 64235, 64242, 64249, 64256, 64263, 64270,
	64277, 64284, 64291, 64298, 64304, 64311, 64318, 64325,

	64332, 64339, 64346, 64352, 64359, 64366, 64372, 64379,
	64386, 64393, 64399, 64406, 64412, 64419, 64426, 64432,
	64439, 64445, 64452, 64458, 64464, 64471, 64477, 64484,
	64490, 64496, 64503, 64509, 64515, 64522, 64528, 64534,

	64540, 64547, 64553, 64559, 64565, 64571, 64577, 64583,
	64589, 64595, 64601, 64607, 64613, 64619, 64625, 64631,
	64637, 64643, 64649, 64655, 64661, 64666, 64672, 64678,
	64684, 64690, 64695, 64701, 64707, 64712, 64718, 64724,

	64729, 64735, 64740, 64746, 64751, 64757, 64762, 64768,
	64773, 64779, 64784, 64789, 64795, 64800, 64806, 64811,
	64816, 64821, 64827, 64832, 64837, 64842, 64847, 64853,
	64858, 64863, 64868, 64873, 64878, 64883, 64888, 64893,

	64898, 64903, 64908, 64913, 64918, 64923, 64928, 64933,
	64937, 64942, 64947, 64952, 64957, 64961, 64966, 64971,
	64975, 64980, 64985, 64989, 64994, 64998, 65003, 65008,
	65012, 65017, 65021, 65025, 65030, 65034, 65039, 65043,

	65048, 65052, 65056, 65060, 65065, 65069, 65073, 65078,
	65082, 65086, 65090, 65094, 65098, 65102, 65107, 65111,
	65115, 65119, 65123, 65127, 65131, 65135, 65139, 65143,
	65146, 65150, 65154, 65158, 65162, 65166, 65169, 65173,

	65177, 65181, 65184, 65188, 65192, 65195, 65199, 65203,
	65206, 65210, 65213, 65217, 65220, 65224, 65227, 65231,
	65234, 65238, 65241, 65244, 65248, 65251, 65254, 65258,
	65261, 65264, 65268, 65271, 65274, 65277, 65280, 65283,

	65287, 65290, 65293, 65296, 65299, 65302, 65305, 65308,
	65311, 65314, 65317, 65320, 65323, 65325, 65328, 65331,
	65334, 65337, 65340, 65342, 65345, 65348, 65350, 65353,
	65356, 65358, 65361, 65364, 65366, 65369, 65371, 65374,

	65376, 65379, 65381, 65384, 65386, 65389, 65391, 65393,
	65396, 65398, 65400, 65403, 65405, 65407, 65409, 65412,
	65414, 65416, 65418, 65420, 65422, 65424, 65427, 65429,
	65431, 65433, 65435, 65437, 65439, 65440, 65442, 65444,

	65446, 65448, 65450, 65452, 65454, 65455, 65457, 65459,
	65461, 65462, 65464, 65466, 65467, 65469, 65470, 65472,
	65474, 65475, 65477, 65478, 65480, 65481, 65483, 65484,
	65485, 65487, 65488, 65490, 65491, 65492, 65494, 65495,

	/* 88 degrees */
	65496, 65497, 65499, 65500, 65501, 65502, 65503, 65504,
	65505, 65507, 65508, 65509, 65510, 65511, 65512, 65513,
	65514, 65514, 65515, 65516, 65517, 65518, 65519, 65520,
	65520, 65521, 65522, 65523, 65523, 65524, 65525, 65525,

	/* 89 degrees */
	65526, 65527, 65527, 65528, 65528, 65529, 65529, 65530,
	65530, 65531, 65531, 65532, 65532, 65532, 65533, 65533,
	65534, 65534, 65534, 65534, 65535, 65535, 65535, 65535,
	65535, 65535, 65535, 65535, 65535, 65535, 65535, 65535
};

int
Isin(a)
int a;
{
    int neg;
    int result;

    /* step 1: normalize angle */
    a = mod(a, FULLCIRCLE);

    /* step 2: determine sign of result */
    if (a > HALFCIRCLE) {
	neg = 1;
	a = FULLCIRCLE - a;
    } else
	neg = 0;
    if (a > QUARTERCIRCLE)
	a = HALFCIRCLE - a;
    if (a == 0)
	return 0;
    if (a >= QUARTERCIRCLE-14)
	return neg ? -65536 : 65536;
    result = sin_table[a / 2];
    if (a & 1)
	result += (sin_table[a / 2 + 1] - result) / 2;
    return neg ? -result : result;
}

int
Icos(a)
int a;
{
    /* In this code I set neg to be TRUE if the result is
     * negative or 0 if result is positive. */
    int neg;	/* 0 or 1 */
    int result;

    /* step 1: normalize angle */
    a = mod(a, FULLCIRCLE);

    /* step 2: normalize some more */
    if (a > HALFCIRCLE)
	a = FULLCIRCLE - a;

    /* step 3: determine sign */
    if (a > QUARTERCIRCLE) {
	a = HALFCIRCLE - a;
	neg = 1;
    } else
	neg = 0;
    if (a <= 14)
	return neg ? -65536 : 65536;
    if (a == QUARTERCIRCLE)
	return 0;
    /* Invert angle so we can use the sin table. */
    a = QUARTERCIRCLE - a;
    result = sin_table[a / 2];
    if (a & 1)
	result += (sin_table[a / 2 + 1] - result) / 2;
    return neg ? -result : result;
}

/* arc sin: do a binary search through the table. */
int
Iasin(x)
int x;
{
    u_short *sp, *ep, *mp;
    int ave;
    int neg;

    if (x < 0) {
	x = -x;
	neg = 1;
    } else
	neg = 0;
    if ((u_short)(unsigned)x == 0) {
	if (x == 0)
	    return 0;
	return neg ? -QUARTERCIRCLE : QUARTERCIRCLE;
    }
    sp = sin_table;
    ep = &sin_table[sizeof sin_table / sizeof sin_table[0]];

    mp = (u_short *)(((int)sp / 2 + (int)ep / 2) & ~1);
    while (sp < mp) {
	if (x < *mp)
	    ep = mp;
	else
	    sp = mp;
	mp = (u_short *)(((unsigned)sp / 2 + (unsigned)ep / 2) & ~1);
    }
    /* average the result if necessary */
    ave = (mp[0] + mp[1]) / 2;
    if (x > ave)
	return neg ? -((unsigned)mp - (unsigned)sin_table + 1) :
		    (unsigned)mp - (unsigned)sin_table + 1;
    else
	return neg ? -((unsigned)mp - (unsigned)sin_table) :
		    (unsigned)mp - (unsigned)sin_table;
}

/* arc cos: do a binary search through the table. */
int
Iacos(x)
int x;
{
    u_short *sp, *ep, *mp;
    int neg;
    int ave;

    if (x < 0) {
	neg = 1;
	x = -x;
    } else
	neg = 0;
    if ((u_short)(unsigned)x == 0) {
	if (x == 0)
	    return QUARTERCIRCLE;
	return neg ? HALFCIRCLE : 0;
    }
    sp = sin_table;
    ep = &sin_table[sizeof sin_table / sizeof sin_table[0]];

    mp = (u_short *)(((int)sp / 2 + (int)ep / 2) & ~1);
    while (sp < mp) {
	if (x < *mp)
	    ep = mp;
	else
	    sp = mp;
	mp = (u_short *)(((unsigned)sp / 2 + (unsigned)ep / 2) & ~1);
    }
    /* average the result if necessary */
    ave = (mp[0] + mp[1]) / 2;
    if (x > ave)
	return neg ? HALFCIRCLE -
	    QUARTERCIRCLE + ((unsigned)mp - (unsigned)sin_table + 1) :
	    QUARTERCIRCLE - ((unsigned)mp - (unsigned)sin_table + 1);
    else
	return neg ? HALFCIRCLE -
	    QUARTERCIRCLE + ((unsigned)mp - (unsigned)sin_table) :
	    QUARTERCIRCLE - ((unsigned)mp - (unsigned)sin_table);
}

/* arc tan: we have to evaluate tan as we search through the table.
 * Fortunately, we only do this a maximum of 12 times.
 */
int
Iatan2(y, x)
int y, x;
{
#define table_size (sizeof sin_table / sizeof sin_table[0])
    int lower, upper, current;
    int xneg, yneg;

    if (x == 0)
	return y < 0 ? -QUARTERCIRCLE : QUARTERCIRCLE;
    if (y == 0)
	return x < 0 ? HALFCIRCLE : 0;
    if (x < 0) {
	x = -x;
	xneg = 1;
    } else
	xneg = 0;
    if (y < 0) {
	y = -y;
	yneg = 1;
    } else
	yneg = 0;

    if (x == y) {
	if (yneg)
	    return xneg ? HALFCIRCLE + QUARTERCIRCLE / 2 :
		HALFCIRCLE + QUARTERCIRCLE + QUARTERCIRCLE / 2;
	else
	    return xneg ? QUARTERCIRCLE + QUARTERCIRCLE / 2 : QUARTERCIRCLE / 2;
    }

    lower = 0;
    upper = sizeof sin_table / sizeof sin_table[0];
    current = (upper + lower) / 2;

    while (lower < current) {
	if (y * sin_table[table_size - current] < x * sin_table[current])
	    upper = current;
	else
	    lower = current;
	current = (upper + lower) / 2;
    }
    if (y *
    (sin_table[table_size - current] + sin_table[table_size - 1 - current])/2 <
    x * (sin_table[current] + sin_table[current + 1]) / 2)
	current *= 2;
    else
	current = current * 2 + 1;
    if (xneg)
	if (yneg)
	    return HALFCIRCLE + current;
	else
	    return HALFCIRCLE - current;
    else
	if (yneg)
	    return FULLCIRCLE - current;
	else
	    return current;
}
