/*
//  Program:  Free Unformat
//  Version:  0.5
//  Written By:  Brian E. Reifsnyder
//  Copyright 1998 under the terms of the GNU GPL by Brian E. Reifsnyder
//
//  Note:  This program is free and is without a warranty of any kind.
//
*/


/*
/////////////////////////////////////////////////////////////////////////////
// INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/*
/////////////////////////////////////////////////////////////////////////////
// DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define FOUND 1
#define TRUE  1
#define FALSE 0
#define LIST  1

/*
/////////////////////////////////////////////////////////////////////////////
// GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

int list_file_and_directory_names = 0;
int print_output = 0;
int test = 0;
int unformat_without_mirror_file = 0;

unsigned char control_buffer[512];
unsigned char mirror_map[5120];
unsigned char sector_buffer[512];

long floppy_total_sectors;
long sectors_per_fat;
long total_sectors;


/*
/////////////////////////////////////////////////////////////////////////////
// PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/
int Compare_Sector_Buffers();

long Decimal_Number(int hex1, int hex2, int hex3);

void Copy_Sector_Into_Control_Buffer();
void Display_Help_Screen();
void Read_Sector(int drive, long s_sector);
void Read_Boot_Sector(int drive);
void Read_Mirror_Map(int drive);
void Restore_Partition_Tables(int list);
void Unformat_Drive(int drive);
void Verify_Drive_Mirror(int drive);
void Write_Sector(int drive, long s_sector);

/*
/////////////////////////////////////////////////////////////////////////////
// FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/


/* Compare Sector Buffers ///////////// */
int Compare_Sector_Buffers()
{
  int answer=0;
  int compare=0;
  int loop = 0;

  do
    {
    compare=control_buffer[loop]-sector_buffer[loop];
    if(compare!=0) answer = 1;
    loop++;
    } while(loop<=512);

  return(answer);
}

/* Copy sector into control buffer //// */
void Copy_Sector_Into_Control_Buffer()
{
  int loop=0;

  do
    {
    control_buffer[loop]=sector_buffer[loop];
    loop++;
    } while(loop<=512);
}

/* Convert Hexidecimal Number to a Decimal Number */
long Decimal_Number(int hex1, int hex2, int hex3)
{
  return((hex1) + (hex2*256) + (hex3*65536));
}

/* Help Routine  ////////////////////// */
void Display_Help_Screen()
{
  printf("\nUnformat Version 0.5               Restores a disk that has been formatted.\n\n");
  printf("Syntax:\n\n");
  printf("UNFORMAT drive: [/J]\n");
  printf("UNFORMAT drive: [/U] [/L] [/TEST] [/P]\n");
  printf("UNFORMAT /PARTN [/L]\n");
  printf("UNFORMAT /?\n\n");
  printf("  drive:    Drive letter to unformat.\n");
  printf("  /J        Verifies that the mirror files are synchronized with the disk.\n");
  printf("  /U      * Unformats without using MIRROR files.\n");
  printf("  /L      * Lists all files and directories found, or, when used with the\n");
  printf("              /PART switch, displays current partition tables.\n");
  printf("  /TEST   * Displays information but does not write changes to disk.\n");
  printf("  /P      * Sends output messages to LPT1.\n");
  printf("  /PARTN  * Restores disk partition tables.\n\n");
  printf("  /?        Displays this help screen.\n");
  printf("This program is copyright 1998 by Brian E. Reifsnyder under the terms of\n");
  printf("GNU General Public License and is without warranty of any kind.\n");
  printf("\n* Indicates functions not yet available.\n");
}

/* Read Sector From Disk ////////////// */
void Read_Sector(int drive, long s_sector)
{
  int err;
  err = absread(drive, 1, s_sector, &sector_buffer);
  if (err!=0)
    {
    printf("\nRead Error...Operation Terminated\n");
    exit(1);
    }
}

/* Read Boot Sector /////////////////// */
void Read_Boot_Sector(int drive)
{
  Read_Sector(drive,0);

  sectors_per_fat = sector_buffer[22];
  total_sectors=Decimal_Number(sector_buffer[32],sector_buffer[33],sector_buffer[34]);
  floppy_total_sectors=Decimal_Number(sector_buffer[19],sector_buffer[20],0);
}


/* Read Mirror Map Into Buffer //////// */
void Read_Mirror_Map(int drive)
{
  int flag = 0;
  int loop;

  long mirror_map_pointer;
  long index;

  long current_sector;
  long mirror_map_start_sector;

  long source_sector;
  long destination_sector;

  Read_Boot_Sector(drive);

  /*Find "pointer sector" at end of drive */

  current_sector=total_sectors-32;
  loop=30;

  /*If drive is a floppy drive, change the following: */
  if((0==drive) || (1==drive))
    {
    current_sector=floppy_total_sectors-10;
    loop=10;
    }


  do
    {
    Read_Sector(drive,current_sector);
    if((65==sector_buffer[4])&&(77==sector_buffer[5])&&(83==sector_buffer[6])&&(69==sector_buffer[7]))
      {
      mirror_map_start_sector=Decimal_Number(sector_buffer[0],sector_buffer[1],0);
      flag=FOUND;
      loop=0;
      }
    current_sector++;
    loop--;
    } while(loop>=1);


  if(FOUND==flag)
    {
    /*Read Mirror Map into buffer */
    int err;
    err = absread(drive, 10, mirror_map_start_sector, &mirror_map);
    if (err!=0)
      {
      printf("\nRead Error...Operation Terminated\n");
      exit(0);
      }
    }

  else
    {
    printf("\nThe sector that points to the drive mirror has not been found.\n");
    printf("Operation Terminated.\n");
    exit(1);
    }


}

/* Restore Partition Tables /////////// */
void Restore_Partition_Tables(int list)
{
  printf("Restore Partition Tables Function will be added at a later date.\n");
  exit(0);
}

/* Unformat Drive ///////////////////// */
void Unformat_Drive(int drive)
{
  int loop;

  long mirror_map_pointer;
  long index;

  long current_sector;
  long mirror_map_start_sector;

  long source_sector;
  long destination_sector;


  Read_Mirror_Map(drive);

  /*Re-create the boot sector */
  destination_sector=Decimal_Number(mirror_map[84],mirror_map[85],mirror_map[86]);
  source_sector=Decimal_Number(mirror_map[88],mirror_map[89],mirror_map[90]);

  Read_Sector(drive,source_sector);
  Write_Sector(drive,destination_sector);

  /*Re-create the FAT tables */
  mirror_map_pointer = 92;
  index=1;

  do
    {
    destination_sector=Decimal_Number(mirror_map[mirror_map_pointer],mirror_map[(mirror_map_pointer+1)],mirror_map[(mirror_map_pointer+2)]);
    source_sector=Decimal_Number(mirror_map[(mirror_map_pointer+4)],mirror_map[(mirror_map_pointer+5)],mirror_map[(mirror_map_pointer+6)]);

    Read_Sector(drive,source_sector);
    Write_Sector(drive,destination_sector);
    Write_Sector(drive,(destination_sector+sectors_per_fat));

    mirror_map_pointer=mirror_map_pointer + 8;
    index++;
    } while(index<=sectors_per_fat);

  /*Re-create the Root Directory */

  loop=1;

  /*If drive is a floppy drive, change the loop for a smaller root dir*/
  if( (0==drive) || (1==drive) ) loop=14;


  do
    {
    destination_sector=Decimal_Number(mirror_map[mirror_map_pointer],mirror_map[(mirror_map_pointer+1)],mirror_map[(mirror_map_pointer+2)]);
    source_sector=Decimal_Number(mirror_map[(mirror_map_pointer+4)],mirror_map[(mirror_map_pointer+5)],mirror_map[(mirror_map_pointer+6)]);


    if( (source_sector>floppy_total_sectors) && ((0==drive) || (1==drive)) )
      {
      printf("\nUnformat operation complete!\n");
      exit(0);
      }

    Read_Sector(drive,source_sector);
    Write_Sector(drive,destination_sector);

    mirror_map_pointer=mirror_map_pointer + 8;

    loop++;
    } while(loop<=31);
  printf("\nUnformat operation complete!\n");
}


/* Verify Drive Mirror //////////////// */
void Verify_Drive_Mirror(int drive)
{
  long mirrored_sector;
  long system_sector;

  int loop;

  long mirror_map_pointer;
  long index;

  long current_sector;
  long mirror_map_start_sector;

  Read_Mirror_Map(drive);

  /*Compare the boot sector */
  system_sector=Decimal_Number(mirror_map[84],mirror_map[85],mirror_map[86]);
  mirrored_sector=Decimal_Number(mirror_map[88],mirror_map[89],mirror_map[90]);

  Read_Sector(drive,system_sector);

  Copy_Sector_Into_Control_Buffer();

  Read_Sector(drive,mirrored_sector);

  if(Compare_Sector_Buffers()!=0)
    {
    printf("\nVerification failed at boot sector...operation terminated.\n");
    exit(1);
    }

  /*Compare the FAT tables */
  mirror_map_pointer = 92;
  index=1;

  do
    {
    system_sector=Decimal_Number(mirror_map[mirror_map_pointer],mirror_map[(mirror_map_pointer+1)],mirror_map[(mirror_map_pointer+2)]);
    mirrored_sector=Decimal_Number(mirror_map[(mirror_map_pointer+4)],mirror_map[(mirror_map_pointer+5)],mirror_map[(mirror_map_pointer+6)]);

    Read_Sector(drive,system_sector);

    Copy_Sector_Into_Control_Buffer();

    Read_Sector(drive,mirrored_sector);

    if(Compare_Sector_Buffers()!=0)
      {
      printf("\nVerification failed on FAT tables...operation terminated.\n");
      exit(1);
      }

    Read_Sector(drive,(system_sector+sectors_per_fat));

    if(Compare_Sector_Buffers()!=0)
      {
      printf("\nVerification failed on FAT tables...operation terminated.\n");
      exit(1);
      }

    mirror_map_pointer=mirror_map_pointer + 8;
    index++;
    } while(index<=sectors_per_fat);

  /*Compare the Root Directory */

  loop=1;

  /*If drive is a floppy drive, change the loop for a smaller root dir*/
  if( (0==drive) || (1==drive) ) loop=14;


  do
    {
    system_sector=Decimal_Number(mirror_map[mirror_map_pointer],mirror_map[(mirror_map_pointer+1)],mirror_map[(mirror_map_pointer+2)]);
    mirrored_sector=Decimal_Number(mirror_map[(mirror_map_pointer+4)],mirror_map[(mirror_map_pointer+5)],mirror_map[(mirror_map_pointer+6)]);

    if( (system_sector>floppy_total_sectors) && ((0==drive) || (1==drive)) )
      {
      printf("\nMirror image has been verified.\n");
      exit(0);
      }

    Read_Sector(drive,system_sector);

    Copy_Sector_Into_Control_Buffer();

    Read_Sector(drive,mirrored_sector);

    if(Compare_Sector_Buffers()!=0)
      {
      printf("\nVerification failed at root directory...operation terminated.\n");
      exit(1);
      }

    mirror_map_pointer=mirror_map_pointer + 8;

    loop++;
    } while(loop<=31);

  printf("\nMirror image has been verified.\n");
  exit(0);
}

/* Write Sector To Disk /////////////// */
void Write_Sector(int drive, long s_sector)
{
  int err;
  err = abswrite(drive, 1, s_sector, &sector_buffer);
  if (err!=0)
    {
    printf("\nWrite Error...Operation Terminated\n");
    exit(1);
    }
}

/*
/////////////////////////////////////////////////////////////////////////////
// MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/

void main(int argc, char *argv[])
{
  int true=1;
  int true1=1;
  int true2=1;


  /* if UNFORMAT is typed without any options */
  if(argc==1)
    {
    Display_Help_Screen();
    exit(1);
    }

  /* if UNFORMAT is typed with more than one option and
  // the first option is "/?" or "/PARTN" or "/partn" */
  if(argc>=2)
    {

    /* if "UNFORMAT /?" is entered  */
    true=strcmp("/?",argv[1]);
    if(0==true)
      {
      Display_Help_Screen();
      exit(0);
      }


    /* if "UNFORMAT /PARTN" or "UNFORMAT /partn" is entered */
    true=1;
    true=strcmp("/PARTN",argv[1]);

    int mask=1;
    mask=strcmp("/partn",argv[1]);

    true=true*mask;

    if(0==true)
      {
      if(argc==3)
	{
	int true=1;

	/*if "UNFORMAT /PARTN /L" is entered */
	true=strcmp("/L",argv[2]);
	if(0==true)
	  {
	  Restore_Partition_Tables(LIST);
	  exit(0);
	  }
	true=1;
	true=strcmp("/l",argv[2]);
	if(0==true)
	  {
	  Restore_Partition_Tables(LIST);
	  exit(0);
	  }
	printf("\nSyntax error, type \"UNFORMAT /?\" for help.\n");
	exit(1);
	}
      else
	{
	Restore_Partition_Tables(FALSE);
	exit(0);
	}
      }
    }

  /*if UNFORMAT is typed with a drive letter */
  if(argc>=2)
    {
    int drive;

    char drive_letter[1];
    char *input;

    drive=argv[1] [0];

    if(drive>=97) drive = drive - 97;
    if(drive>=65) drive = drive - 65;

    if( (drive<0) || (drive> 25) )
      {
      printf("\nSyntax error, type \"UNFORMAT /?\" for help.\n");
      exit(1);
      }

    drive_letter[0] = drive+65;
    drive_letter[1] = 58;


    /*place valid drive checking code here*/

    true1=1;
    true2=1;
    true1=strcmp("/j",argv[2]);
    true2=strcmp("/J",argv[2]);
    true=true1*true2;
    if(0==true) Verify_Drive_Mirror(drive);

    printf("\nUnformatting drive %c%c\n",drive_letter [0],drive_letter [1]);

    true=0;
    if(0==true)
      {
      if(argc>=3)
	{
	int switch_to_test = (argc-1);

	do
	  {
	  true1=1;
	  true2=1;

	  true1=strcmp("/u",argv[switch_to_test]);
	  true2=strcmp("/U",argv[switch_to_test]);
	  true1=true1*true2;

	  if(0==true1) unformat_without_mirror_file = 1;


	  true1=1;
	  true2=2;

	  true1=strcmp("/l",argv[switch_to_test]);
	  true2=strcmp("/L",argv[switch_to_test]);
	  true1=true1*true2;

	  if(0==true1) list_file_and_directory_names = 1;

	  true1=1;
	  true2=2;

	  true1=strcmp("/test",argv[switch_to_test]);
	  true2=strcmp("/TEST",argv[switch_to_test]);
	  true1=true1*true2;

	  if(0==true1) test = 1;

	  true1=1;
	  true2=2;

	  true1=strcmp("/p",argv[switch_to_test]);
	  true2=strcmp("/P",argv[switch_to_test]);
	  true1=true1*true2;

	  if(0==true1) print_output = 1;

	  switch_to_test--;
	  } while (switch_to_test>=2);

	}

      Unformat_Drive(drive);
      }
    }
}

