;
; "DVRCORE.ASM" -- Common UltraDMA Driver "Core" Logic.
; General Program Equations.
;
RDYTO	equ	008h		;384-msec minimum I-O timeout.
RBYTES	equ	(2*1024*65536*12/14318180+1)*512 ;Read-test byte count.
	;( = microseconds per tick, but adjusted for binary megabytes).
	;Number of reads this length per tick = transfer rate in MB/s.
RS_SC	equ	RBYTES/512	;"Read speed" input sector count.
RC_SC	equ	58		;"Read compare" total sector count.
RC_CYL	equ	3		;"Read compare" starting disk address.
RC_HD	equ	15
RC_SEC	equ	5
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Return Codes.
;
DMATIMO equ	0E8h		;DMA timeout code, 008h at exit.
DMAERR	equ	0EFh		;DMA error   code, 00Fh at exit.
CTLRERR equ	000h		;Ctlr. busy  code, 020h/021h at exit.
DRQTIMO equ	080h		;DRQ timeout code.
DISKERR equ	08Ah		;Disk-busy   code, 0AAh/0ABh at exit.
WFLTERR equ	0ACh		;Write-fault code, 0CCh/0CDh at exit.
HARDERR equ	0BFh		;Hard-error  code, 0E0H at exit.
XMSERR	equ	0FFh		;XMS memory-error code.
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Disk-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
SETM	equ	003h		;Set Mode subcommand.
SETF	equ	0EFh		;Set Features command.
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; LBA "Device Address Packet" Layout.
;
struc	DAP
DapPL	resb	1		;Packet length.
	resb	1		;(Reserved).
DapSC	resb	1		;I-O sector count.
	resb	1		;(Reserved).
DapBuf	resd	1		;I-O buffer address (segment:offset).
DapLBA	resw	1		;48-bit logical block address (LBA).
DapLBA1	resw	1
DapLBA2	resw	1
endstruc
;
; DOS "Request Packet" Layout.
;
struc	RP
	resb	2		;(Unused by us).
RPOp	resb	1		;Opcode.
RPStat	resw	1		;Status word.
	resb	9		;(Unused by us).
RPSize	resd	1		;Resident driver size.
RPCL	resd	1		;Command-line data pointer.
endstruc
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	Strat		;"Strategy" routine offset.
Timer	equ	$-2		;(400-msec timeout counter, after Init).
IDEAd	equ	$-1		;(Lower IDE status address, after Init).
	dw	DevInt		;"Device-Interrupt" routine offset.
DMAAd	equ	$-2		;(DMA command-reg. address, after Init).
	db	DVRNAME		;Driver name.
;
; Resident Driver "Data Page" Variables.   The 0BAh/0B0h bytes in the
;   init class-code "interface byte" table (SecCt2) are for ALi M5529
;   chips.   MANY THANKS to David Muller for this VALUABLE upgrade!
;
VLF	db	0		;VDS lock flag (1 = user buffer locked).
Flags	db	0		;Driver control flags (usage varies).
Units	dd	0000000FFh	;IDE "active units" table, set by Init:
	dd	0000000FFh	;  Byte 0:    BIOS unit (0FFh inactive).
	dd	0000000FFh	;  Byte 1:    CHS sectors per head.
	dd	0000000FFh	;  Byte 2-3:  CHS sectors per cylinder.
PRDAd	dd	IOAdr		;PRD 32-bit command addr. (Init set).
SecCt2	db	0FAh		;IDE "upper" sector count.
LBA2	db	0F0h,08Ah,080h	;IDE "upper" LBA bits 24-47.
SecCt	db	0BAh		;IDE "lower" sector count.
LBA	db	0B0h,0,0	;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
XMSInp	dw	XMINPUT		;"XMS input" routine pointer (set to
				;  "VDExit" during any "read tests").
XMHdl	dw	0		;XMS buffer handle number (Init set).
XMOfs	dw	0		;XMS 16-bit buffer offset (Init set).
VDSLn	dd	VLENGTH		;VDS and XMS buffer length.
VDSOf	dd	0		;VDS 32-bit offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
XMSSH	equ	VDSOf		;(XMS parameters share the VDS block).
XMSSA	equ	VDSOf+2
XMSDH	equ	VDSSg+2
;
; Driver Entry Routine.  For CHS requests, the registers contain:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   ES:BX   I-O buffer address.
;
; For LBA requests, the registers contain:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   DS:SI   Pointer to Device Address Packet ("DAP"), described above.
;
Entry	pusha			;Entry -- save all CPU registers.
	mov	bp,16		;Reset active-units table index.
NextU	sub	bp,byte 4	;Any more active units to check?
	js	QuickX		;No, request NOT for us -- exit quick!
	cmp	dl,[cs:bp+Units-@] ;Does request unit match our table?
	jne	NextU		;No, see if more table entries remain.
@LS1	sti			;Ensure CPU interrupts are enabled.
	push	ds		;Save CPU segment registers.
	push	es
TestRW	mov	dl,0BEh		;Mask out LBA and write request bits.
	and	dl,ah
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	jne	Pass		;No, let BIOS handle this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns	ValSC		;No, go validate CHS sector count.
	mov	al,[si+DapSC]	;Get "DAP" I-O sector count.
	cmp	dword [si+DapBuf],byte -1 ;64-bit "DAP" buffer address?
	jne	GetDAP		;No, go get remaining "DAP" parameters.
Pass	pop	es		;"Pass" request -- reload segment regs.
	pop	ds
	nop			;(Unused alignment "filler").
QuickX	mov	bp,sp		;Reload CPU flags saved by Int 13h.
	push	word [bp+20]
	popf
	popa			;Reload all CPU registers.
	db	0EAh		;Go to next routine in Int 13h chain.
@PrvI13	dd	0FFFFFFFFh	;(Previous Int 13h vector, Init set).
GetDAP	les	cx,[si+DapBuf]	;Get "DAP" I-O buffer address.
	mov	di,[si+DapLBA2]	;Get "DAP" logical block address.
	mov	dx,[si+DapLBA1]
	mov	si,[si+DapLBA]
ValSC	dec	al		;Is sector count zero or over 128?
	js	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js	SetDS		;Yes, go set our DS-register.
	xchg	ax,cx		;CHS -- save request code and sectors.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		     ;Swap cylinder and head values.
	mov	al,[cs:bp+Units+1-@] ;Convert head to sectors.
	mul	ah
	add	si,ax		;Add to starting sector.
	xchg	ax,dx		;Convert cylinder to sectors.
	mul	word [cs:bp+Units+2-@]
	add	si,ax		;Get final 24-bit CHS disk address.
	adc	dx,di
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
SetDS	push	cs		;Set our DS-register.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative commands.
	mov	[bx+LBA-@],si	;Save 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	shr	dx,12		;Shift out LBA bits 16-27.
	or	di,dx		;Anything in LBA bits 28-47?
	mov	dl,(LBABITS/32)	;(Initialize LBA command bits).
	jnz	LBA48		;Yes, use LBA48 read/write command.
	xchg	dh,[bx+LBA2-@]	;LBA28 -- reload & reset bits 24-27.
	or	ah,(DRCMD+1)	;Get LBA28 read/write command + 5.
	jmp	short GetAdr	;Go get IDE and LBA address bytes.
LBA48	shl	ah,3		;LBA48 -- get command as 020h/030h.
GetAdr	mov	di,CDSEL-00100h	;Get primary device-address bytes.
@DMALo1	equ	$-1		;(Lower DMA command address, Init set).
	shr	bp,3		;Is this a primary-channel request?
	jz	DevAdr		;Yes, set IDE & PCI address bytes.
	mov	di,CDSEL+00680h	;Get secondary device-address bytes.
@DMALo2	equ	$-1		;(Lower DMA command address, Init set).
DevAdr	mov	[bx+IDEAd-@],di ;Set current IDE & PCI address bytes.
	rcl	dl,5		;Shift disk-select into LBA commands.
	or	dl,dh		;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		;Get final IDE command byte.
	xor	dh,ah		;(LBA28 = C8h/CAh, LBA48 = 25h/35h).
	mov	[bx+DSCmd-@],dx	;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	;Set I-O sector count.
	mov	ah,0		;Set VDS/XMS and DMA buffer lengths.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	mov	[bx+IOLen+1-@],ax
	mov	[bx+VDSOf-@],cx	;Set VDS offset and segment.
	mov	[bx+VDSOf+2-@],bx
	mov	[bx+VDSSg-@],es
	or	dword [bx+IOAdr-@],byte -1 ;Invalidate VDS address.
	mov	ax,08103h	           ;VDS "lock" user buffer.
	mov	dx,0000Ch
	call	VDLock
	jc	GoBuf		           ;Error -- do buffered I-O.
	cmp	dword [bx+IOAdr-@],byte -1 ;Got a valid VDS address?
	je	NoVDS			   ;No, set 20-bit address.
	mov	byte [bx+VLF-@],001h	   ;Set VDS "lock" flag.
	mov	ax,[bx+IOAdr-@]	;Get low-order VDS buffer address.
ChkOdd	test	al,003h		;Is user I-O buffer 32-bit aligned?
	jnz	NoLock		;No, use buffered I-O routines below.
	mov	cx,[bx+IOLen-@]	;Get lower ending DMA address.
	dec	cx		;(IOLen - 1 + IOAdr).
	add	ax,cx		;Would this I-O cross a 64K boundary?
	jc	NoLock		;Yes, use buffered I-O routines below.
	cmp	word [bx+IOAdr+2-@],byte -1 ;DMA I-O beyond our limit?
@DMALmt	equ	$-1		;(If 640K limit, set to 009h by Init).
	ja	NoLock		;Yes, use buffered I-O routines below.
	call	DoDMA		;Do direct DMA I-O with user's buffer.
Done	mov	bp,sp		;Done -- point to saved registers.
	mov	[bp+19],al	;Set error code in exiting AH-reg.
@LS2	rcr	byte [bp+24],1	;Set error flag in exiting carry
GetOut	rol	byte [bp+24],1	;  bit, in flags saved by Int 13h.
	call	VDUnlk		;If needed, "unlock" user I-O buffer.
	pop	es		;Reload all CPU registers and exit.
	pop	ds
	popa
	iret
NoVDS	mov	ax,16		;No VDS -- get 20-bit buffer segment.
	mul	word [bx+VDSSg-@]
	add	ax,[bx+VDSOf-@]	;Add in buffer offset value.
	adc	dx,bx
	mov	[bx+IOAdr-@],ax	;Set 20-bit user buffer address.
	mov	[bx+IOAdr+2-@],dx
	jmp	short ChkOdd	;Go check for "odd" buffer alignment.
NoLock	call	VDUnlk		;Buffered I-O -- "unlock" user buffer.
GoBuf	jmp	BufIO		;Go to buffered I-O routines below.
	db	0		;(Unused alignment "filler").
;
; Subroutine to execute read and write commands.
;
BufDMA	mov	dword [bx+IOAdr-@],0  ;Buffered -- set XMS buffer addr.
@XBufAd	equ	$-4		;(XMS 32-bit buffer address, Init set).
DoDMA	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	mov	dl,[bx+IDEAd-@]
	mov	dh,001h
	out	dx,al
	mov	di,dx		;Save IDE drive-select address.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,(RDYTO*256)+FLT   ;Get timeout count & "fault" mask.
	add	ch,[es:si]	     ;Set CH-reg. with timeout limit.
	call	ChkRdy		     ;Await controller- and disk-ready.
	jc	DoDMAX		     ;If error, exit immediately!
	test	byte [bx+IOCmd-@],012h  ;Is this a write request?
	jnz	SetDMA		;Yes, reset DMA command register.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA	mov	dx,[bx+DMAAd-@]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	push	si		;Save BIOS timer pointer.
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	mov	si,PRDAd
	outsd
	mov	bx,001F7h	;Set IDE parameter-output flags.
NxtPar	lea	dx,[di+CSECCT-CDSEL-1] ;Point to IDE sector count -1.
IDEPar	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	bx,1		;More parameters to go in this group?
	jc	IDEPar		;Yes, loop back and output next one.
	jnz	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ	cmp	ch,[es:si]	;Too long without 1st data-request?
	je	ErrDRQ		;Yes?  Return carry and DRQ error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz	ChkDRQ		;No, loop back and check again.
	mov	dx,[bx+DMAAd-@]	;Set DMA Start/Stop bit (starts DMA).
	in	al,dx
	inc	ax
	out	dx,al
ChkDMA	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,[es:si]	;Has our DMA transfer timed out?
	jne	ChkDMA		;No, loop back and check again.
HltDMA	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne	ErrDMA		;No?  Go see what went wrong.
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy	lea	dx,[di+CSTAT-CDSEL] ;Read IDE primary status.
	in	al,dx
	test	al,BSY+RDY	;Controller or disk still busy?
	jg	ChkErr		;No, check for "fault" or hard-error.
	cmp	ch,[es:si]	;Too long without becoming ready?
	jne	ChkRdy		;No, loop back and check again.
	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp	short WhichE	;Go see which error code to return.
ErrDRQ	mov	al,DRQTIMO	;BAAAD News!  Get DRQ error code.
	stc			;Set carry flag (error) and exit!
DoDMAX	ret
ChkErr	and	al,cl		;Disk "fault" or hard-error?
	jz	ChkDMAX		;No, all is well -- go exit below.
	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*WFLTERR)+HARDERR ;(Get hardware error codes).
WhichE	jz	EndErr		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
EndErr	add	al,cl		;Add 1 if error occurred at I-O end.
	stc			;Set carry flag to denote "error"!
ChkDMAX	ret			;Exit.
ErrDMA	test	al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp	short WhichE	;Go see which error code to return.
;
; Subroutine to do a VDS "lock" or "unlock".
;
VDUnlk	shr	byte [bx+VLF-@],1 ;Was input buffer "locked" by VDS?
	jnc	VDExit		;No, just exit below.
VDUnlX	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
VDLock	mov	di,VDSLn	;Point to VDS parameter block.
VDInit	push	cs
	pop	es
	push	bx		;Save BX-reg. and execute VDS request.
	int	04Bh
VDDone	pop	bx		;Reload our BX-register.
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
VDExit	ret			;Exit.
NonXEnd	equ	$		;End of resident non-XMS driver.
;
; Buffered I-O routines.   This logic and the "XMSMov" subroutine are
;   "dismissed" by driver init if the non-XMS driver is used AND if a
;   local-stack was NOT requested.
;
BufIO	shl	dword [bx+VDSOf-@],16  ;Convert to XMS handle & offset.
	test	byte [bx+IOCmd-@],012h ;Is this a write request?
	jnz	BufOut		;Yes, use output routine below.
	call	BufDMA		;Input all data to driver XMS buffer.
	jc	DoneJ		;If error, post return code & exit!
	call	[XMSInp]	;Move XMS data to user input buffer.
	jmp	short DoneJ	;Done -- post any return code & exit.
BufOut	call	XMSMov		;Output -- move data to XMS buffer.
	jc	DoneJ		;If error, post return code & exit!
	call	BufDMA		;Output all data from XMS buffer.
DoneJ	jmp	Done		;Done -- post any return code & exit.
;
; Subroutine to move data to or from the driver's XMS buffer.
;
XMSMov	push	cs		;Point ES-reg. to our data.
	pop	es
	mov	di,XMSDH	;Point to XMS destination field.
	jnz	XMSet		;If output, just set XMS destination!
	mov	si,XMSSH	;Point to XMS user-buffer address.
	push	si		;Save pointer for below.
	movsw			;Move user-buffer address from
	movsw			;  XMS source to XMS destination.
	movsw
	pop	di		;Point to XMS source field.
XMSet	mov	si,XMHdl	;Set XMS handle and buffer offset as
	movsw			;  input source or output destination.
	movsw
	mov	[di],bx		;(Hi-order always 0 due to alignment).
XMGo	mov	ah,00Bh		;Get XMS "move data" request code.
XMCall	push	bx		;Save BX-reg. and execute XMS request.
	db	09Ah		;(SI-reg. points to IOLen after move).
@XEntry	dd	0		;(XMS "entry" address, set by init).
	dec	ax		;Return 0 if success, -1 if error.
	sar	ax,1		;If error, set carry flag on.
	jmp	short VDDone	;Go reload BX-reg. and exit above.
ResEnd	equ	$		;End of resident driver.
