
/************************************************************************/
/*                                                                      */
/* FILE: command.c                                                      */
/* command.com Top level driver                                         */
/*                                                                      */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* This file is part of G-COM.                                          */
/* G-COM is derived from DOS-C source (GPL).                            */
/*                                                                      */
/* (C) Copyright 1999-2000  Roberto Gordo Saez   (GCOM)                 */
/* (C) Copyright 1995-1998  Pasquale J. Villani  (DOSC)                 */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option)  */
/* any later version.                                                   */
/*                                                                      */
/* This program is distributed in the hope that it will be useful, but  */
/* WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU     */
/* General Public License for more details.                             */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.                  */
/*                                                                      */
/************************************************************************/


#include "globals.h"


/* -- Function prototypes --------------------------------------------- */

static BOOL CmdLineProc (void);
BOOL do_command (char*);
BOOL TmpFileName (char*);
BOOL Redirect (char*, char*, char*, BOOL, DWORD*, DWORD*);
BOOL RedirectInput (char*, DWORD*);
BOOL RedirectOutput (char*, BOOL, DWORD*);
void RestoreInput (DWORD*);
void RestoreOutput (DWORD*);
struct table *LookupCmd (char*);
BOOL cmd_exit (char*, int, char*);


/* -- Global variables ------------------------------------------------ */

int default_drive;
char FAR *lpEnviron;
char *error_msg_str;
int rtn_errlvl;
char switchchar;
char *newline;
char prompt_string[MAX_CMDLINE];
char cmd_line[MAX_CMDLINE];
char cmd_buffer[MAX_CMDBUFFER];
char *cmd_tail;
BOOL batch_flag;
BOOL echo_flag;
BOOL break_flag;

static BOOL pflag = FALSE, cflag = FALSE;
static BOOL pipe_flag = FALSE;
psp FAR *p_psp;


/* -------------------------------------------------------------------- */

struct table
{
    char    *entry;
    BOOL    (*func)(char*, int, char*);
};


static struct table commands[] =
{
    {"BEEP",    cmd_beep},
    {"BREAK",   cmd_break},
    {"CALL",    cmd_call_bat},
    {"CD",      cmd_cd},
    {"CHDIR",   cmd_cd},
    {"ECHO",    cmd_echo},
    {"EXIT",    cmd_exit},
    {"GOTO",    cmd_goto_bat},
    {"PATH",    cmd_path},
    {"PAUSE",   cmd_pause},
    {"PROMPT",  cmd_prompt},
    {"REM",     cmd_rem_bat},
    {"SET",     cmd_set},
    {"SHIFT",   cmd_shift_bat},
    {"UNSET",   cmd_unset},
    {"VER",     cmd_ver},
    {"VERIFY",  cmd_verify},
    {"",        ExecCmd}
};


/* -------------------------------------------------------------------- */
/* main - command.com main function                                     */
/* -------------------------------------------------------------------- */

void main(void)
{
    char *pointer, FAR *fpointer;
    DWORD nread;

    switchchar = '/';
    newline = "\r\n";

    InitHandler();
    p_psp = DosGetPSP();
    batch_flag = FALSE;
    break_flag = FALSE;
    lpEnviron = (char FAR *)MK_FP(p_psp->ps_environ, 0);
    cmd_tail = "";

    if (!CmdLineProc())
        DosExit(1);

    /* Check what PROMPT is set in environment to override default */
    fpointer = EnvLookup("PROMPT");
    if(*fpointer != '\0')
        fstrncpy((char FAR *)prompt_string, fpointer, MAX_CMDLINE);
    else
        strncpy(prompt_string, default_prompt, MAX_CMDLINE);

    if(!cflag)
    {
        if(pflag)
        {
            /* A permanent shell terminates onto itself */
            p_psp->ps_parent = FP_SEG(p_psp);

            /* Try to exec AUTOEXEC.BAT */
            if(batch("AUTOEXEC.BAT",0,"") != 1) {}
        }

        /* Announce G-COM */
        DosWrite(STDOUT, (char FAR *)newline, 2);
        DosWrite(STDOUT, (char FAR *)str_line, 68);
        printf(announce, (unsigned int)MAJOR_RELEASE, (unsigned int)MINOR_RELEASE);
        DosWrite(STDOUT, (char FAR *)str_line, 68);
        DosWrite(STDOUT, (char FAR *)copyright, strlen(copyright));
        DosWrite(STDOUT, (char FAR *)newline, 2);
        DosWrite(STDOUT, (char FAR *)str_line, 68);
        DosWrite(STDOUT, (char FAR *)"\r\n\r\n", 4);

        FOREVER
        {
            default_drive = DosGetDrive();
            put_prompt(prompt_string);
            if((nread = DosRead(STDIN,(char FAR*)cmd_line,MAX_CMDLINE)) < 2)
                continue;
            cmd_line[(UWORD)(nread-1)] = '\0';
            if(issep(cmd_line[(UWORD)(nread-2)]))
                cmd_line[(UWORD)(nread-2)] = '\0';
            if(!do_command(cmd_line))
                DosWrite(STDOUT, (char FAR *)newline, 2);
        }
    }
    else
    {
        default_drive = DosGetDrive();
        fpointer = p_psp->ps_cmd;
        while(*fpointer != '\r' && *fpointer != '\n' && *fpointer != '\0')
        {
            if(*fpointer == switchchar && (*(fpointer+1) == 'c' || *(fpointer+1) == 'C'))
                break;
            fpointer++;
        }
        fpointer += 2;
        pointer = cmd_buffer;
        nread = 1;
        while(*fpointer != '\r' && *fpointer != '\n' && *fpointer != '\0' && nread < MAX_CMDBUFFER)
        {
            nread++;
            *pointer++ = *fpointer++;
        }
        if(nread == MAX_CMDBUFFER)
            error_message(CMDLINE_LONG);
        else
        {
            cmd_line[(UWORD)(--nread)] = '\0';
            if(nread > 0)
                do_command(cmd_buffer);
        }
    }
}


/* -------------------------------------------------------------------- */
/* CmdLineProc - Process command line options for command.com           */
/* -------------------------------------------------------------------- */

static BOOL CmdLineProc(void)
{
    long size;
    char env_size[8];

    *env_size = '\0';
    parsing("[E:7PC.]+", p_psp->ps_cmd, env_size, &pflag, &cflag);
    if(*env_size == ' ')
    {
        error_message(PARAM_FORMAT);
        return FALSE;
    }

    /* Get the passed-in environment size and make certain we allocate */
    /* enough space */
    if(*env_size != '\0')
    {
        size = atoi(env_size);
        if(strlen(env_size) > 5 || size < 64 || size > 32000)
        {
            error_message(PARAM_RANGE_ERR);
            return FALSE;
        }
    }
    else
    {
        size = EnvGetSize();
        if(size < ENV_DEFAULT)
            size = ENV_DEFAULT;
    }
    EnvAlloc((int)size);
    return TRUE;
}


/* -------------------------------------------------------------------- */
/* do_command                                                           */
/* -------------------------------------------------------------------- */

BOOL do_command(char *command_line)
{
    REG char *p_cmd;
    BOOL AppendMode = FALSE;
    char *p_Input = NULL, *p_Output = NULL;
    BOOL IORedirected = FALSE;
    char *pointer;
    struct table *p_table;
    DWORD OldStdin = -1, OldStdout = -1;
    BOOL status = TRUE;
    BOOL invalid_pipe = TRUE;
    int c;
    static char cmd_name[FNAME_MAX+1];

    p_cmd = skipwh(command_line);
    if(*p_cmd == '\0')
        return TRUE;

    /* Pre-scan the command line and look for any re-directs */
    while(*p_cmd != '\0')
    {
        switch(*p_cmd++)
        {
        case '<':
            p_cmd = skipwh(p_cmd);
            if(p_Input != NULL || *p_cmd == '>' || *p_cmd == '<' || *p_cmd == '|' || *p_cmd == '\0')
            {
                error_message(SYNTAX_ERR);
                return FALSE;
            }
            p_Input = p_cmd;
            while(!issep(*p_cmd))
                ++p_cmd;
            break;

        case '>':
            if(*p_cmd == '>')
            {
                ++p_cmd;
                AppendMode = TRUE;
            }
            p_cmd = skipwh(p_cmd);
            if(p_Output != NULL || *p_cmd == '>' || *p_cmd == '<' || *p_cmd == '|' || *p_cmd == '\0')
            {
                error_message(SYNTAX_ERR);
                return FALSE;
            }
            p_Output = p_cmd;
            while(!issep(*p_cmd))
                ++p_cmd;
            break;

        case '|':
            if(invalid_pipe)
            {
                error_message(SYNTAX_ERR);
                return FALSE;
            }
            if(!IORedirected)
            {
                if(*EnvLookup("TEMP") == '\0')
                {
                    error_message(TEMP_VAR);
                    return FALSE;
                }
                IORedirected = TRUE;
            }
            invalid_pipe = TRUE;
            break;

        case ' ':
        case '\t':
            break;

        default:
            invalid_pipe = FALSE;
            break;
        }
    }
    if(IORedirected && invalid_pipe)
    {
        error_message(SYNTAX_ERR);
        return FALSE;
    }

    IORedirected |= (p_Input != NULL || p_Output != NULL);
    if(IORedirected)
    {
        if(!Redirect(command_line, p_Input, p_Output, AppendMode, &OldStdin, &OldStdout))
        {
            if(IORedirected)
            {
                RestoreInput(&OldStdin);
                RestoreOutput(&OldStdout);
            }
            return FALSE;
        }
        if(*command_line == '\0')
        {
            if(IORedirected)
            {
                RestoreInput(&OldStdin);
                RestoreOutput(&OldStdout);
            }
            return TRUE;
        }
    }

    p_cmd = skipwh(command_line);

    if(batch_flag)
    {
        if(*p_cmd == '@')
            p_cmd = skipwh(++p_cmd);
    }

    cmd_tail = p_cmd;
    while(!issep(*cmd_tail))
        cmd_tail++;

    /* Look for just a drive change command, and execute it if found */
    if(isalpha(*p_cmd) && *(p_cmd+1) == ':' && issep(*(p_cmd+2)))
        DosSetDrive(toupper(*p_cmd) - 'A');
    else if(strnicmp(p_cmd, "ECHO.", 5))
        cmd_echo_dot("ECHO.", 1, p_cmd+5);
    else if((p_table = LookupCmd(p_cmd)) == NULL)
    {
        error_message(SYNTAX_ERR);
        status = FALSE;
    }
    else
    {
        pointer = skipwh(cmd_tail);

        /* It may be a help command request */
        if(*(p_table -> entry) != '\0' && *pointer == switchchar && *(pointer+1) == '?' && issep(*pointer+2))
        {
            *(cmd_tail+1) = '\0';
            do
            {
                --cmd_tail;
                *(cmd_tail+1) = *cmd_tail;
            }
            while(cmd_tail > p_cmd);
            *cmd_tail = ' ';

            ExecCmd("HELP.EXE", 1, skipwh(cmd_tail));
        }
        /* Do a command execution */
        else
        {
            break_flag = FALSE;

            if(checkfile(pointer = scanfile(p_cmd)))
            {
                if(pointer != p_cmd)
                {
                    c = num_args(skipwh(cmd_tail));
                    if (!Execute(p_cmd, c, skipwh(cmd_tail)))
                        status = FALSE;
                }
                else
                {
                    for(c = 0; !issep(*pointer); c++, pointer++)
                        cmd_name[c] = toupper(*pointer);
                    cmd_name[c] = '\0';

                    c = num_args(skipwh(cmd_tail));
                    if (!(*(p_table -> func))(cmd_name, c, skipwh(cmd_tail)))
                        status = FALSE;
                }
            }
            else
            {
                error_message(FILE_NOT_FOUND);
                status = FALSE;
            }
        }
    }

    if(p_Output == NULL && !batch_flag && !pipe_flag && status)
        DosWrite(STDOUT, (char FAR *)newline, 2);
    if(IORedirected)
    {
        RestoreInput(&OldStdin);
        RestoreOutput(&OldStdout);
    }

    return status;
}


/* -------------------------------------------------------------------- */
/* TmpFileName                                                          */
/* -------------------------------------------------------------------- */

BOOL TmpFileName(char *string)
{
    char FAR *TmpDir;
    char *pointer, *p_name;
    UBYTE n1, n2, n3, n4;
    DWORD ret;
    unsigned int c = 0, len;
    UDWORD n;
    static UDWORD num = 0;

    if(num == 0)
    {
        DosGetTime(&n1, &n2, &n3, &n4);
        num = (((UDWORD)n1 + (UDWORD)n4*(UDWORD)n2) << 16) | ((UDWORD)n4 + (UDWORD)n1*(UDWORD)n3);
    }
    n = num;

    TmpDir = EnvLookup("TEMP");

    if(*TmpDir != '\0')
    {
        if(!fstrncpy((char FAR *)string, TmpDir, MAX_PATH))
            return FALSE;
    }

    pointer = string + strlen(string);
    if(*string != '\0')
    {
        if(*(pointer-1) != '\\')
            *pointer++ = '\\';
    }
    p_name = pointer;

    do
    {
        len = 0;
        while(n > 0 && len < 8)
        {
            *pointer++ = 'A' + (char)(n % 26);
            n /= 26;
            len++;
        }
        n = num;
        while(len < 8)
        {
            *pointer++ = '0';
            len++;
        }
        strcpy(pointer, ".TMP");
        num++;
        c++;
        pointer = p_name;

        if((ret = DosOpen(string, O_RDONLY)) > 0)
            DosClose((UWORD)ret);
    }
    while(ret != DE_FILENOTFND && c < 100);

    if(c == 100)
        return FALSE;

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* Input-Output redirection                                             */
/* -------------------------------------------------------------------- */

BOOL Redirect(char *command_line, char *p_Input, char *p_Output, BOOL AppendMode, DWORD *OldStdin, DWORD *OldStdout)
{
    REG char *p_cmd;
    BOOL status = TRUE;
    char *pointer;
    char buffer[MAX_CMDLINE];
    char TmpFile1[MAX_PATH+FNAME_MAX+1];
    char TmpFile2[MAX_PATH+FNAME_MAX+1];

    *TmpFile1 = '\0';
    *TmpFile2 = '\0';

    if(p_Input != NULL)
    {
        p_cmd = skipwh(p_Input);
        pointer = buffer;
        while(!issep(*p_cmd))
            *pointer++ = *p_cmd++;
        *pointer = '\0';
        if(!RedirectInput(buffer,OldStdin))
            return FALSE;
    }

    if(p_Output != NULL)
    {
        p_cmd = skipwh(p_Output);
        pointer = buffer;
        while(!issep(*p_cmd))
            *pointer++ = *p_cmd++;
        *pointer = '\0';
    }

    pointer = command_line;
    p_cmd = skipwh(pointer);

    while(*p_cmd != '\0')
    {
        switch(*p_cmd)
        {
        case '>':
            if(*(p_cmd+1) == '>')
                ++p_cmd;
        case '<':
            p_cmd = skipwh(++p_cmd);
            while(!issep(*p_cmd))
                ++p_cmd;
            break;

        case '|':
            pipe_flag = TRUE;
            *pointer++ = '|';
            ++p_cmd;
            break;

        default:
            *pointer++ = *p_cmd++;
        }
    }
    *pointer = '\0';

    p_cmd = pointer = command_line;
    while(*p_cmd != '\0')
    {
        if(*p_cmd == '|')
        {
            *p_cmd++ = '\0';
            if(*TmpFile1 != '\0')
                strcpy(TmpFile2, TmpFile1);
            if(!TmpFileName(TmpFile1))
            {
                error_message(FILE_PIPE_ERR);
                status = FALSE;
                break;
            }
            if(!RedirectOutput(TmpFile1, FALSE, OldStdout))
            {
                status = FALSE;
                break;
            }
            if(!do_command(pointer))
            {
                status = FALSE;
                break;
            }
            RestoreInput(OldStdin);
            RestoreOutput(OldStdout);
            if(*TmpFile2 != '\0')
                DosDelete((char FAR *)TmpFile2);
            if(!RedirectInput(TmpFile1,OldStdin))
            {
                status = FALSE;
                break;
            }
            pointer = p_cmd;
        }
        else
            p_cmd++;
    }

    if(p_Output != NULL && status)
    {
        if(!RedirectOutput(buffer, AppendMode, OldStdout))
            status = FALSE;
    }

    if(*TmpFile1 != '\0')
    {
        pipe_flag = FALSE;
        if(status)
        {
            if(!do_command(pointer))
                return FALSE;
            *command_line = '\0';
        }
        RestoreInput(OldStdin);
        RestoreOutput(OldStdout);
        DosDelete((char FAR *)TmpFile1);
    }

    return status;
}


BOOL RedirectInput(char *FileName, DWORD *OldStdin)
{
    DWORD Handle;

    if(DosDupHandle(STDIN, (DWORD FAR *)OldStdin) != DE_SUCCESS)
    {
        RestoreInput(OldStdin);
        error_message(INTERNAL_ERR);
        return FALSE;
    }

    if((Handle = DosOpen((char FAR *)FileName, O_RDONLY)) < 0)
    {
        RestoreInput(OldStdin);
        switch(Handle)
        {
        case DE_FILENOTFND:
        case DE_PATHNOTFND:
            error_message(FILE_NOT_FOUND);
            break;

        case DE_TOOMANY:
            error_message(TOO_FILES_OPEN);
            break;

        case DE_ACCESS:
            error_message(ACCESS_DENIED);
            break;

        default:
            error_message(INTERNAL_ERR);
        }
        return FALSE;
    }

    if(DosForceDupHandle(Handle, STDIN) != DE_SUCCESS)
    {
        RestoreInput(OldStdin);
        error_message(INTERNAL_ERR);
        return FALSE;
    }

    DosClose((UWORD)Handle);
    return TRUE;
}


BOOL RedirectOutput(char *FileName, BOOL AppendMode, DWORD *OldStdout)
{
    DWORD Handle;

    if(DosDupHandle(STDOUT, (DWORD FAR *)OldStdout) != DE_SUCCESS)
    {
        RestoreOutput(OldStdout);
        error_message(INTERNAL_ERR);
        return FALSE;
    }

    if(AppendMode)
    {
        if((Handle = DosOpen((char FAR *)FileName, O_RDWR)) >= 0)
            DosSeek((UWORD)Handle, S_END, 0L);
    }
    else
        Handle = DosCreate((char FAR *)FileName, F_NORMAL);

    if(Handle < 0)
    {
        RestoreOutput(OldStdout);
        switch(Handle)
        {
        case DE_FILENOTFND:
        case DE_PATHNOTFND:
            error_message(FILE_NOT_FOUND);
            break;

        case DE_TOOMANY:
            error_message(TOO_FILES_OPEN);
            break;

        case DE_ACCESS:
            error_message(ACCESS_DENIED);
            break;

        default:
            error_message(INTERNAL_ERR);
        }
        return FALSE;
    }

    if(DosForceDupHandle(Handle, STDOUT) != DE_SUCCESS)
    {
        RestoreOutput(OldStdout);
        error_message(INTERNAL_ERR);
        return FALSE;
    }

    DosClose((UWORD)Handle);
    return TRUE;
}


void RestoreInput(DWORD *DupStdin)
{
    if(*DupStdin >= 0)
    {
        if(DosForceDupHandle(*DupStdin, STDIN) != DE_SUCCESS)
            error_message(INTERNAL_ERR);
        DosClose((UWORD)*DupStdin);
        *DupStdin = -1;
    }
}


void RestoreOutput(DWORD *DupStdout)
{
    if(*DupStdout >= 0)
    {
        if(DosForceDupHandle(*DupStdout, STDOUT) != DE_SUCCESS)
            error_message(INTERNAL_ERR);
        DosClose((UWORD)*DupStdout);
        *DupStdout = -1;
    }
}


/* -------------------------------------------------------------------- */
/* LookupCmd - Search for command.com internal commands                 */
/* -------------------------------------------------------------------- */

struct table *LookupCmd(char *token)
{
    struct table *pointer = (struct table *)commands;
    char *p_token = token;
    char cmd_name[FNAME_MAX+1];
    unsigned int c;

    c = 0;
    while(!issep(*p_token) && c < FNAME_MAX)
    {
        if(*p_token == ':' || *p_token == '\\')
            c = 0;
        else
            cmd_name[c++] = toupper(*p_token);
        p_token++;
    }
    cmd_name[c] = '\0';

    if(c == FNAME_MAX)
        return NULL;

    while(*(pointer -> entry) != '\0')
    {
        if(strcmp(pointer -> entry, cmd_name))
            break;
        else ++pointer;
    }

    return pointer;
}


/* -------------------------------------------------------------------- */
/* cmd_exit - EXIT command                                              */
/* -------------------------------------------------------------------- */

BOOL cmd_exit(char *cmd_name, int argc, char *arguments)
{
    int ret_val;
    char *p;

    /* Don't exit from a permanent shell */
    if(pflag) return TRUE;

    /* If no values passed, return zero ... */
    if(argc == 0)
        DosExit(0);

    /* ... otherwise return what the user asked for */
    else
    {
        for(p = arguments, ret_val = 0; isdigit(*p); p++)
        {
            ret_val = ret_val * 10 + tonum(*p);
            if(ret_val > 0xFF)
            {
                ret_val = 0;
                break;
            }
        }
        DosExit(ret_val);
    }
    return TRUE;
}
