;This file provides macros for FASM that disallow the inadvertent
;use of non-existent instructions for older CPU architectures.
;
;Both invalidation and revalidation are implemented
;
;The intended use is something like this:
;
;	include 'compatibility.inc'
;  .386					;<-- stop us from accidentally using SSE instructions
;	format etc.
;	;put code to test for "cpuid" existence here
;	jxx	sse_not_allowed
;  .486					;<-- if cpuid exists then we have at least a 486
;	mov	eax,1
;	cpuid
;	bt	edx,25
;	jnc	sse_not_allowed
;	;put code to check OS support for SSE here
;	jxx	sse_not_allowed
;     sse_allowed:
;  .pentium3				;<-- SSE code okay in here, but SSE2 not allowed
;	;SSE code goes here
;  .386					;<-- switch back to non-SSE code
;	jmp	done
;     sse_not_allowed:
;	;non-SSE code goes here
;     done:
;
;You can also use like this:
;
;  .086					;<-- Now only 8086 instructions are valid
;	xor	ax,ax
;	;shl	bx,2			;<-- Error, only immediate shifts by one allowed
;	;mov	ecx,edx			;<-- Error, 32 bit not allowed
;  .286f				;<-- Now 80286 with FPU
;	shl	bx,2			;<-- Okay with 80286
;	;fsin				;<-- Error, 80287 doesn't have fsin
;  .486f				;<-- Now 80486dx or 80486sx with 80487sx
;	bswap	edx
;	fsin				;<-- Okay with 80387+
;  .EM64T				;<-- Now all instructions available
;	addsubpd xmm0,xmm1
;	use64
;	mov	r14,rsi
;  .286					;<-- 80286 without FPU (implicit use16 generated)
;	jz	somelabel		;<-- generates 2 or 5 byte jumps for 16 bit CPU
;	;use32				;<-- Error, 32 bit not allowed
;  .386
;	jz	somelabel		;<-- generates 2 or 4 byte jumps for 32 bit CPU
;	use32
;	jz	somelabel		;<-- generates 2 or 6 byte jumps for 32 bit CPU
;
;
;All instruction macros assume lower case names only, this means you can get access to
;the original instruction at any time by using one or more upper case letters.
;eg. Use32  moV [r],4  puSh EAX  etc. will bypass these macros.
;
;
;There are probably lots of ways to upset this code, it might not be bullet-proof,
;so be careful and aware. I tested it as much as I could but you never know
;about those sneaky little bugs and oversights that seem to pop up at the most
;inconvenient times. As usual, it's your responsibility, use at your own risk.
;
;
;If you find a problem or can improve on this code contact "revolution" at:
;http://board.flatassembler.net


;masm/tasm style macros for convenience

compatibility@undo equ

macro .086 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_186
	invalidate_fpu
	compatibility@undo equ revalidate_186,revalidate_fpu
}
macro .086f {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_186
	invalidate_fpu_287
	compatibility@undo equ revalidate_186,revalidate_fpu_287
}
macro .186 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_286
	invalidate_fpu
	compatibility@undo equ revalidate_286,revalidate_fpu
}
macro .186f {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_286
	invalidate_fpu_287
	compatibility@undo equ revalidate_286,revalidate_fpu_287
}
macro .286 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_386
	invalidate_fpu
	compatibility@undo equ revalidate_386,revalidate_fpu
}
macro .286f {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_386
	invalidate_fpu_387
	compatibility@undo equ revalidate_386,revalidate_fpu_387
}
macro .386 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_486
	invalidate_fpu
	compatibility@undo equ revalidate_486,revalidate_fpu
}
macro .386f {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_486
	invalidate_fpu_487
	compatibility@undo equ revalidate_486,revalidate_fpu_487
}
macro .486 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_pentium
	invalidate_fpu
	compatibility@undo equ revalidate_pentium,revalidate_fpu
}
macro .486f {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_pentium
	invalidate_mmx
	compatibility@undo equ revalidate_pentium,revalidate_mmx
}
macro .pentium {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_p6
	invalidate_mmx
	compatibility@undo equ revalidate_p6,revalidate_mmx
}
macro .pentium_mmx {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_p6
	compatibility@undo equ revalidate_p6
}
macro .pentium_pro {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_p6_p2
	invalidate_mmx
	compatibility@undo equ revalidate_p6_p2,revalidate_mmx
}
macro .pentium2 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_sse
	compatibility@undo equ revalidate_sse
}
macro .pentium3 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_sse2
	compatibility@undo equ revalidate_sse2
}
macro .pentium4 {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_sse3
	compatibility@undo equ revalidate_sse3
}
macro .pentium4prescott {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	invalidate_ssse3
	compatibility@undo equ revalidate_64bit
}
macro .EM64T {
	match i,compatibility@undo\{irp j,i\\{\\reverse j\\}\}
	compatibility@undo equ
}


macro invalidate [instruction] {forward macro instruction[x]\{\common rb -1\}}

macro invalidatexmm [instruction] {
    forward
    macro instruction [x] \{
      \common
	if ~ x eq
      \forward
	    if x in <xmm0,xmm1,xmm2,xmm3,xmm4,xmm5,xmm6,xmm7>
		rb -1
	    end if
      \common
	end if
	instruction x
    \}
}

macro invalidatemem32 [instruction] {
    common
    local i,.i
    forward
    macro instruction [x] \{
      \common
	.i=$
	instruction x
	load i from .i
	while (i and 0e7h)=026h | (i and 0fch)=064h		;override or segment prefix
	    if i=066h | i=067h
		rb -1 ;32bit not available
	    end if
	    .i=.i+1
	    load i from .i
	end while
    \}
}

macro invalidate_vme {
    invalidate vmptrld,vmptrst,vmclear,vmread,vmwrite,vmcall,vmlaunch,vmresume,vmxoff,vmxon,\
	       clgi,skinit,stgi,vmload,vmmcall,vmrun,vmsave
}
macro revalidate_vme {
    purge vmptrld,vmptrst,vmclear,vmread,vmwrite,vmcall,vmlaunch,vmresume,vmxoff,vmxon,\
	  clgi,skinit,stgi,vmload,vmmcall,vmrun,vmsave
}

macro invalidate_64bit_safe {
    invalidate use64,swapgs,cmpxchg16b
    macro lahf \{
	virtual at 0
	    xchg eax,eax
	    if $=2
		rb -1 ;LAHF not valid in 64 bit mode
	    end if
	end virtual
	lahf
    \}
    macro sahf \{
	virtual at 0
	    xchg eax,eax
	    if $=2
		rb -1 ;SAHF not valid in 64 bit mode
	    end if
	end virtual
	sahf
    \}
    invalidate_vme
}
macro revalidate_64bit_safe {
    purge use64,swapgs,cmpxchg16b,lahf,sahf
    revalidate_vme
}

macro invalidate_64bit {
    invalidate use64,swapgs
}
macro revalidate_64bit {
    purge use64,swapgs
}

macro invalidate_ssse3 {
    invalidate_64bit_safe
    invalidate psignb,psignw,psignd,pabsb,pabsw,pabsd,palignr,pshufb,pmulhrsw,\
	       pmaddubsw,phsubw,phsubd,phsubsw,phaddw,phaddd,phaddsw
}
macro revalidate_ssse3 {
    revalidate_64bit_safe
    purge psignb,psignw,psignd,pabsb,pabsw,pabsd,palignr,pshufb,pmulhrsw,\
	  pmaddubsw,phsubw,phsubd,phsubsw,phaddw,phaddd,phaddsw
}

macro invalidate_sse3 {
    invalidate_ssse3
    invalidate fisttp,lddqu,movshdup,movsldup,movddup,addsubps,addsubpd,haddps,\
	       hsubps,haddpd,hsubpd,monitor,mwait
}
macro revalidate_sse3 {
    revalidate_ssse3
    purge fisttp,lddqu,movshdup,movsldup,movddup,addsubps,addsubpd,haddps,\
	  hsubps,haddpd,hsubpd,monitor,mwait
}

macro invalidate_sse2 {
    invalidate_sse3
    invalidate movapd,movupd,movhpd,movlpd,movmskpd,addpd,addsd,mulpd,mulsd,\
	       divpd,divsd,sqrtpd,sqrtsd,maxpd,maxsd,minpd,minsd,andpd,andnpd,\
	       orpd,xorpd,cmppd,cmpsd,comisd,shufpd,unpckhpd,unpcklpd,cvtps2pd,\
	       cvtpd2ps,cvtss2sd,cvtsd2ss,cvtpd2pi,cvtpi2pd,cvtpd2dq,cvtdq2pd,\
	       cvtsd2si,cvtsi2sd,cvtps2dq,cvtdq2ps,movdqa,movdqu,paddq,psubq,\
	       pmuludq,pshuflw,pshufhw,pshufd,pslldq,psrldq,punpckhqdq,\
	       punpcklqdq,movq2dq,movdq2q,clflush,movntdq,movntpd,movnti,\
	       maskmovdqu,lfence,mfence,pause
    invalidatexmm movsd,paddb,paddw,paddd,psubb,psubw,psubd,pmull,pmulh,pmadd,\
		  paddsb,paddsw,psubsb,psubsw,paddusb,paddusw,psubusb,psubusw,\
		  pcmpeqb,pcmpeqw,pcmpeqd,pcmpgtpb,pcmpgtpw,pcmpgtpd,packsswb,\
		  packssdw,packuswb,punpckhbw,punpckhwd,punpckhdq,punpcklbw,\
		  punpcklwd,punpckldq,pand,pandn,por,pxor,psllw,pslld,psrlw,\
		  psrld,psraw,psrad,psllq,psrlq,pavgb,pextrw,pinsrw,pmaxub,\
		  pminub,pmaxsw,pminsw,pmovmskb,pmulhuw,psadbw,movd,movq
}
macro revalidate_sse2 {
    revalidate_sse3
    purge movapd,movupd,movhpd,movlpd,movmskpd,addpd,addsd,mulpd,mulsd,\
	  divpd,divsd,sqrtpd,sqrtsd,maxpd,maxsd,minpd,minsd,andpd,andnpd,\
	  orpd,xorpd,cmppd,cmpsd,comisd,shufpd,unpckhpd,unpcklpd,cvtps2pd,\
	  cvtpd2ps,cvtss2sd,cvtsd2ss,cvtpd2pi,cvtpi2pd,cvtpd2dq,cvtdq2pd,\
	  cvtsd2si,cvtsi2sd,cvtps2dq,cvtdq2ps,movdqa,movdqu,paddq,psubq,\
	  pmuludq,pshuflw,pshufhw,pshufd,pslldq,psrldq,punpckhqdq,\
	  punpcklqdq,movq2dq,movdq2q,clflush,movntdq,movntpd,movnti,\
	  maskmovdqu,lfence,mfence,pause
    purge movsd,paddb,paddw,paddd,psubb,psubw,psubd,pmull,pmulh,pmadd,\
	  paddsb,paddsw,psubsb,psubsw,paddusb,paddusw,psubusb,psubusw,\
	  pcmpeqb,pcmpeqw,pcmpeqd,pcmpgtpb,pcmpgtpw,pcmpgtpd,packsswb,\
	  packssdw,packuswb,punpckhbw,punpckhwd,punpckhdq,punpcklbw,\
	  punpcklwd,punpckldq,pand,pandn,por,pxor,psllw,pslld,psrlw,\
	  psrld,psraw,psrad,psllq,psrlq,pavgb,pextrw,pinsrw,pmaxub,\
	  pminub,pmaxsw,pminsw,pmovmskb,pmulhuw,psadbw,movd,movq
}

macro invalidate_sse {
    invalidate_sse2
    rept 8 r:0 \{xmm\#r equ error\}
    invalidate prefetcht0,prefetcht1,prefetcht2,prefetchnta,sfence,fxsave,\
	       fxrstor,ldmxcsr,maskmovq,movntq,stmxcsr,pavgb,pextrw,pinsrw,\
	       pmaxub,pminub,pmaxsw,pminsw,pmovmskb,pmulhuw,psadbw,pshufw
}
macro revalidate_sse {
    revalidate_sse2
    rept 8 r:0 \{restore xmm\#r\}
    purge prefetcht0,prefetcht1,prefetcht2,prefetchnta,sfence,fxsave,\
	  fxrstor,ldmxcsr,maskmovq,movntq,stmxcsr,pavgb,pextrw,pinsrw,\
	  pmaxub,pminub,pmaxsw,pminsw,pmovmskb,pmulhuw,psadbw,pshufw
}

macro invalidate_3dnowext {
    invalidate pf2iw,pfnacc,pfpnacc,pi2fw,pswapd
}
macro revalidate_3dnowext {
    purge pf2iw,pfnacc,pfpnacc,pi2fw,pswapd
}

macro invalidate_3dnow {
    invalidate_3dnowext
    invalidate femms,pavgusb,pf2id,pfacc,pfadd,pfcmpeq,pfcmpge,pfcmpgt,pfmax,pfmin,pfmul,\
	       pfrcp,pfrcpit1,pfrpit2,pfrsqit1,pfrsqrt,pfsub,pfsubr,pi2fd,pmulhrw
}
macro revalidate_3dnow {
    revalidate_3dnowext
    purge femms,pavgusb,pf2id,pfacc,pfadd,pfcmpeq,pfcmpge,pfcmpgt,pfmax,pfmin,pfmul,\
	  pfrcp,pfrcpit1,pfrpit2,pfrsqit1,pfrsqrt,pfsub,pfsubr,pi2fd,pmulhrw
}

invalidate_3dnow	;always default to 3dnow as invalid, can be revalidated later if needed

macro invalidate_mmx {
    invalidate_sse
    rept 8 r:0 \{mm\#r equ error\}
    invalidate emms
}
macro revalidate_mmx {
    revalidate_sse
    rept 8 r:0 \{restore mm\#r\}
    purge emms
}

macro invalidate_fpu_p6 {
    invalidate_sse
    invalidate fcmovb,fcmove,fcmovbe,fcmovu,fcmovnb,fcmovne,fcmovnbe,fcmovnu,\
	       fcomi,fcomip,fucomi,fucomip
}
macro revalidate_fpu_p6 {
    revalidate_sse
    purge fcmovb,fcmove,fcmovbe,fcmovu,fcmovnb,fcmovne,fcmovnbe,fcmovnu,\
	  fcomi,fcomip,fucomi,fucomip
}

macro invalidate_fpu_487 {
    invalidate_fpu_p6
    invalidate_mmx
}
macro revalidate_fpu_487 {
    revalidate_fpu_p6
    revalidate_mmx
}

macro invalidate_fpu_387 {
    invalidate_fpu_487
    invalidate fprem1,fucom,fucomp,fucompp,fcos,fsin,fsincos
}
macro revalidate_fpu_387 {
    revalidate_fpu_487
    purge fprem1,fucom,fucomp,fucompp,fcos,fsin,fsincos
}

macro invalidate_fpu_287 {
    invalidate_fpu_387
    invalidate fsetpm
}
macro revalidate_fpu_287 {
    revalidate_fpu_387
    purge fsetpm
}

macro invalidate_fpu {
    invalidate_fpu_287
    rept 8 r:0 \{st\#r equ error\}
    invalidate fst,fist,fstp,fistp,fld,fild,fldz,fld1,fldpi,fldl2t,fldl2e,fldlg2,\
	       fldln2,f2xm1,fabs,fadd,faddp,fiadd,fbld,fbstp,fchs,fclex,fnclex,fcom,\
	       fcomp,fcompp,fdecstp,fdiv,fdivp,fidiv,fdivr,fdivrp,fidivr,ficom,ficomp,\
	       fincstp,finit,fninit,fldcw,fldenv,fmul,fmulp,fimul,fnop,fpatan,fprem,\
	       fptan,frndint,frstor,fsave,fnsave,fscale,fsqrt,fstcw,fnstcw,fstenv,\
	       fnstenv,fstsw,fnstsw,fsub,fsubp,fisub,fsubr,fsubrp,fisubr,ftst,fxam,\
	       fxch,fxtract,fyl2x,fyl2xp1,feni,fdisi,fwait
}
macro revalidate_fpu {
    revalidate_fpu_287
    rept 8 r:0 \{restore st\#r\}
    purge fst,fist,fstp,fistp,fld,fild,fldz,fld1,fldpi,fldl2t,fldl2e,fldlg2,\
	  fldln2,f2xm1,fabs,fadd,faddp,fiadd,fbld,fbstp,fchs,fclex,fnclex,fcom,\
	  fcomp,fcompp,fdecstp,fdiv,fdivp,fidiv,fdivr,fdivrp,fidivr,ficom,ficomp,\
	  fincstp,finit,fninit,fldcw,fldenv,fmul,fmulp,fimul,fnop,fpatan,fprem,\
	  fptan,frndint,frstor,fsave,fnsave,fscale,fsqrt,fstcw,fnstcw,fstenv,\
	  fnstenv,fstsw,fnstsw,fsub,fsubp,fisub,fsubr,fsubrp,fisubr,ftst,fxam,\
	  fxch,fxtract,fyl2x,fyl2xp1,feni,fdisi,fwait
}

macro invalidate_p6_p2 {
    invalidate sysenter
    invalidate sysexit
}
macro revalidate_p6_p2 {
    purge sysenter
    purge sysexit
}

macro invalidate_p6 {
    invalidate_fpu_p6
    invalidate_p6_p2
    invalidate rdpmc
    irp cc,c,z,e,s,a,b,g,l,ae,be,ge,le,o,p,pe,nc,nz,ne,ns,na,nb,ng,nl,nae,nbe,nge,nle,no,np,po \{
	invalidate cmov\#cc
    \}
}
macro revalidate_p6 {
    revalidate_fpu_p6
    revalidate_p6_p2
    purge rdpmc
    irp cc,c,z,e,s,a,b,g,l,ae,be,ge,le,o,p,pe,nc,nz,ne,ns,na,nb,ng,nl,nae,nbe,nge,nle,no,np,po \{
	purge cmov\#cc
    \}
}

macro invalidate_pentium {
    invalidate_p6
    invalidate cmpxchg8b,rdmsr,rdtsc,rsm
    cr4 equ error
}
macro revalidate_pentium {
    revalidate_p6
    purge cmpxchg8b,rdmsr,rdtsc,rsm
    restore cr4
}

macro invalidate_486 {
    invalidate_pentium
    invalidate_mmx
    invalidate bswap,cmpxchg,cpuid,invd,invlpg,wbinvd,wrmsr,xadd
}
macro revalidate_486 {
    revalidate_pentium
    revalidate_mmx
    purge bswap,cmpxchg,cpuid,invd,invlpg,wbinvd,wrmsr,xadd
}

macro invalidate_386 {
    local i,.i
    invalidate_486
    rept 8 r:0 \{cr\#r equ error\}
    rept 8 r:0 \{dr\#r equ error\}
    rept 8 r:0 \{tr\#r equ error\}
    invalidate use32,bsf,bsr,bt,btc,btr,bts,cdq,cwde,insd,iretd,jecxz,lfs,lgs,\
	       lss,lodsd,movsd,movsx,movzx,outsd,popad,popfd,pushad,pushfd,\
	       scasd,shrd,shld,stosd
    macro imul [args] \{
      \common
	.i=$
	imul args
	load i from .i
	if i=0fh
	    rb -1 ;imul not available
	end if
    \}
    irp cc,c,z,e,s,a,b,g,l,ae,be,ge,le,o,p,pe,nc,nz,ne,ns,na,nb,ng,nl,nae,nbe,nge,nle,no,np,po \{
	invalidate set\#cc
	macro j\#cc dest \\{
	    .i=$
	    if ((dest-$-2) > 127) | (($-dest+2) > 128)
		j\#cc $+5
		load i from $-2
		store (i xor 1) at $-2
		jmp dest
	    else
		j\#cc dest
	    end if
	    load i from .i
	    if i=0fh
		rb -1 ;jump too far
	    else if (i and 0feh)=066h
		rb -1 ;32bit not available
	    end if
	\\}
    \}
    invalidatemem32 adc,add,and,call,cmp,cmps,dec,div,idiv,imul,in,inc,ins,jmp,lea,lods,mov,movs,mul,neg,\
		    not,or,out,outs,pop,push,rcl,rcr,rep,rol,ror,sal,sar,sbb,scas,shl,shr,stos,sub,test,xor
    irp reg,fs,gs,eax,ebx,ecx,edx,esi,edi,ebp,esp \{reg equ error\}
    use16
}
macro revalidate_386 {
    revalidate_486
    rept 8 r:0 \{restore cr\#r\}
    rept 8 r:0 \{restore dr\#r\}
    rept 8 r:0 \{restore tr\#r\}
    purge use32,bsf,bsr,bt,btc,btr,bts,cdq,cwde,insd,iretd,jecxz,lfs,lgs,\
	  lss,lodsd,movsd,movsx,movzx,outsd,popad,popfd,pushad,pushfd,\
	  scasd,shrd,shld,stosd,imul
    irp cc,c,z,e,s,a,b,g,l,ae,be,ge,le,o,p,pe,nc,nz,ne,ns,na,nb,ng,nl,nae,nbe,nge,nle,no,np,po \{
	purge set\#cc
	purge j\#cc
    \}
    purge adc,add,and,call,cmp,cmps,dec,div,idiv,imul,in,inc,ins,jmp,lea,lods,mov,movs,mul,neg,\
	  not,or,out,outs,pop,push,rcl,rcr,rep,rol,ror,sal,sar,sbb,scas,shl,shr,stos,sub,test,xor
    restore fs,gs,eax,ebx,ecx,edx,esi,edi,ebp,esp
}

macro invalidate_286 {
    invalidate_386
    invalidate arpl,bound,clts,lar,lgdt,lidt,lldt,lmsw,lsl,ltr,sgdt,sidt,sldt,smsw,str,verr,verw
}
macro revalidate_286 {
    revalidate_386
    purge arpl,bound,clts,lar,lgdt,lidt,lldt,lmsw,lsl,ltr,sgdt,sidt,sldt,smsw,str,verr,verw
}

macro invalidate_186 {
    local i,.i
    invalidate_286
    invalidate enter,leave,ins,insb,insw,outs,outsb,outsw,popa,popaw,pusha,pushaw,ud2
    macro imul [args] \{
      \common
	.i=$
	imuL	args
	load	i from .i
	if i<>0f6h & i<>0f7h
		rb -1 ;imul not available
	end if
    \}
    macro push [args] \{
      \common
	.i=$
	push args
	while .i<$
	    load i from .i
	    while (i and 0e7h)=026h | (i and 0fch)=064h		;override or segment prefix
		if i=066h | i=067h
		    rb -1 ;push 32bit not available
		end if
		.i=.i+1
		load i from .i
	    end while
	    if i=068h
		.i=.i+2
		rb -1 ;push immediate not available
	    else if i=06ah
		.i=.i+1
		rb -1 ;push immediate not available
	    else if i=0ffh
		.i=.i+1
		load i from .i
		if (i and 0c0h)=040h | (i and 0c0h)=080h
		    .i=.i+(i shr 6)
		else if (i and 0c7h)=06h
		    .i=.i+2
		end if
	    end if
	    .i=.i+1
	end while
    \}
    irp instruction,shl,shr,rol,ror,rcl,rcr,sal,sar \{
	macro instruction reg,count \\{
	    if (count in <cl>) | (count=1)
		instruction reg,count
	    else
		rb -1 ;shift/rotate count must be 1 or CL
	    end if
	\\}
    \}
}
macro revalidate_186 {
    revalidate_286
    purge enter,leave,ins,insb,insw,outs,outsb,outsw,popa,pusha,ud2,imul,push
    purge shl,shr,rol,ror,rcl,rcr,sal,sar
}
