/*=============================================
// sqlite for Lua 5.0
// 2003-Feb-28 doug.currie@alum.mit.edu
// The author disclaims copyright to this source code.  In place of
// a legal notice, here is a blessing:
//
//    May you do good and not evil.
//    May you find forgiveness for yourself and forgive others.
//    May you share freely, never taking more than you give.
//
// Thank-you to D. Richard Hipp.
// ============================================
// this Lua extension was built to be used
// with the 'loadmodule' by Thatcher Ulrich
// http://lua-users.org/wiki/LuaBinaryModules
//=============================================*/

// TO DO
//int sqlite_complete(const char *sql); //?
//user functions
//callback api?

#ifdef __cplusplus
extern "C" {
#endif
    
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "sqlite.h"

#ifdef __cplusplus
}
#endif

#ifdef _WIN32
#define LIBEXPORT __declspec(dllexport)
#else
#define LIBEXPORT
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

#define LUA_SQLITELIBNAME "sqlite"

typedef struct dbud
{
	sqlite 	*db;
	sqlite_vm 	*vm;
	int		cnt;		/* Number of columns in row */
	const char **row; 	/* Column data */
	const char **n_t;	/* Column names and datatypes */
} dbud;

static void ls_close_dbud( dbud *ud )
{
	if( ud != NULL )
	{
		if( ud->vm != NULL )
		{
			sqlite_finalize( ud->vm, NULL ); // does a free(ud->vm)
			ud->vm = NULL;
		}
		if( ud->db != NULL )
		{
			sqlite_close( ud->db ); // does a free(ud->db)
			ud->db = NULL;
		}
		ud->cnt = 0;
	}
}

/* gc tag method for the db user data */
int lsql_udgc (lua_State *L) {
	dbud *ud = lua_touserdata(L, -1);
#ifdef DEBUG
	if( ud != NULL && ud->vm!= NULL )
	{
		printf("\n-- ** gc ud->vm ** --\n");
	}
	if( ud != NULL && ud->db != NULL )
	{
		printf("\n-- ** gc ud->db ** --\n");
	}
#endif
	ls_close_dbud( ud );
	lua_pop (L, 1);
	return 0;
}

int lsql_version(lua_State *L)
{
	lua_pushstring(L, sqlite_version);
	return 1;
}

int lsql_encoding(lua_State *L)
{
	lua_pushstring(L, sqlite_encoding);
	return 1;
}

static void lsql_attachmeta(lua_State *L)
{
	// stack has ud
	lua_pushstring (L, LUA_SQLITELIBNAME);
	lua_gettable (L, LUA_GLOBALSINDEX);
	lua_setmetatable(L, -2);
	// return with stack unchanged (stack has ud)
}

int lsql_open(lua_State *L)
{
	dbud *ud;
	sqlite *db = NULL;
	char *errmsg;
	const char *filename;
	int mode = 0;
	int n = lua_gettop(L);  // number of arguments
	if( n<1 || n>2 )
	{
		errmsg = "Wrong number of arguments to sqlite.open(name[,mode])";
		goto arg_error;
	}
	if( n < 2 )
	{
		// use default mode 0
	}
	else if( !lua_isnumber(L,2) )
	{
		errmsg = "Second argument to sqlite.open(name[,mode]) must be a number";
		goto arg_error;
	}
	else
	{
		mode = lua_tonumber(L,2);
	}
	if( !lua_isstring(L,1) )
	{
		errmsg = "First argument to sqlite.open(name[,mode]) must be a string";
	arg_error:
		lua_pushstring(L, errmsg);
		return lua_error(L);
	}
	filename = lua_tostring(L,1);
	db = sqlite_open(filename, mode, &errmsg);
	if( db == NULL )
	{
		lua_pushstring(L, errmsg);
		if( errmsg != NULL ) sqlite_freemem( errmsg );
		return lua_error(L);
	}
	// results
	ud = (dbud *)lua_newuserdata (L, sizeof(dbud));
	if( ud == NULL )
	{
		lua_pushstring(L, "sqlite.open(): Cannot allocate user data!");
		if( errmsg != NULL ) sqlite_freemem( errmsg );
		return lua_error(L);
	}
	ud->db = db;
	ud->vm = NULL;
	ud->cnt = 0;
	lsql_attachmeta(L); // attach gc to automatically close db
	lua_pushstring(L, errmsg);
	if( errmsg != NULL ) sqlite_freemem( errmsg );
	return 2; // dbud,errmsg
}

static dbud *ls_getbdud(lua_State *L, const char *fname)
{
	dbud * ud;
	if( !lua_isuserdata(L,1) )
	{
		luaL_error(L,"First argument to %s must be a db",fname);
	}
	ud = (dbud *)lua_touserdata(L,1);
	if( ud->db == NULL )
	{
		luaL_error(L,"Db passed to %s was closed",fname);
	}
	return ud;
}

static void ls_checknargs(lua_State *L, int x, const char *fname)
{
	int n = lua_gettop(L);  // number of arguments
	if(  n != x )
	{
		luaL_error(L,"Wrong number of arguments to %s",fname);
	}
}

static dbud *
ls_checknargs_getbdud(lua_State *L, int x, const char *fname)
{
	ls_checknargs(L,x,fname);
	return ls_getbdud(L,fname);
}

static void ls_checkvm(lua_State *L, dbud *ud, const char *fname)
{
	if( ud->vm == NULL )
	{
		luaL_error(L,"Db's vm to %s is uncompiled (or finalized)",fname);
	}
}

static dbud *
ls_checknargs_vm_getbdud(lua_State *L, int x, const char *fname)
{
	dbud *ud = ls_checknargs_getbdud(L,x,fname);
	ls_checkvm(L,ud,fname);
	return ud;
}

int lsql_close(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,1,"sqlite.close(db)");
	ls_close_dbud( ud );
	return 0;
}

int lsql_last_insert_rowid(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,1,"sqlite.last_insert_rowid(db)");
	lua_pushnumber(L, sqlite_last_insert_rowid( ud->db ));
	return 1;
}

int lsql_changes(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,1,"sqlite.changes(db)");
	lua_pushnumber(L, sqlite_changes( ud->db ));
	return 1;
}

int lsql_error_string(lua_State *L)
{
	const char *errmsg;
	ls_checknargs(L,1,"sqlite.error_string(x)");
	if( !lua_isnumber(L,1) )
	{
		errmsg = "Argument to sqlite.error_string(x) must be a number";
		lua_pushstring(L, errmsg);
		return lua_error(L);
	}
	lua_pushstring(L, sqlite_error_string(lua_tonumber(L,1)));
	return 1;
}

int lsql_interrupt(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,1,"sqlite.finalize(db)");
	// do it
	sqlite_interrupt( ud->db );
	return 0;
}

int lsql_busy_timeout(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,2,"sqlite.busy_timeout(db)");
	const char *errmsg;
	int ms;
	if( !lua_isnumber(L,2) )
	{
		errmsg = "Second argument to sqlite.busy_timeout(db,sql) must be a number";
		lua_pushstring(L, errmsg);
		return lua_error(L);
	}
	ms = lua_tonumber(L,2);
	// do it
	sqlite_busy_timeout( ud->db, ms );
	return 0;
}

// compile a statement (return status and length of query string processed)
int lsql_compile(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,2,"sqlite.compile(db)");
	const char *sql, *tail, *errmsg;
	int err;
	if( !lua_isstring(L,2) )
	{
		errmsg = "Second argument to sqlite.compile(db,sql) must be a string";
		lua_pushstring(L, errmsg);
		return lua_error(L);
	}
	sql = lua_tostring(L,2);
	// do it
	err = sqlite_compile(
		  ud->db	/* The open database */
		, sql		/* SQL statement to be compiled */
		, &tail	/* OUT: uncompiled tail of zSql */
		, &ud->vm	/* OUT: the virtual machine to execute zSql */
		, NULL	/* OUT: Error message. */
		);
	lua_pushnumber(L, (lua_Number )err);
	//printf( "\nTail: %x Sql: %x Diff: %x\n", tail, sql, tail - (sql == NULL ? tail : sql));
	lua_pushnumber(L, (lua_Number )(tail - (sql == NULL ? tail : sql)));
	return 2;
}

int lsql_step(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.step(db)");
	int err;
	// do it
	err = sqlite_step(
		   ud->vm		/* The virtual machine to execute */
		, &ud->cnt		/* OUT: Number of columns in result */
		, &ud->row	/* OUT: Column data */
		, &ud->n_t		/* OUT: Column names and datatypes */
		);
	lua_pushnumber(L, (lua_Number )err);
	return 1;
}

int lsql_finalize(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.finalize(db)");
	char *errmsg;
	int err;
	// do it
	err = sqlite_finalize( ud->vm , &errmsg );
	lua_pushnumber(L, (lua_Number )err);
	lua_pushstring(L,errmsg);
	if( errmsg != NULL ) sqlite_freemem( errmsg );
	return 2;
}

/*
do a step, if OK (if ud->cnt > 0)...
  row_idata -- integer indexed table of result row
  row_iname -- integer indexed table of result row's column names
  row_itype -- integer indexed table of result row's column types
  row_data -- column name indexed table of result row
  row_type -- column name indexed table of result row's column types
*/

int lsql_row_idata(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.row_idata(db)");
	int x;
	// do it
	lua_newtable(L);
	if( ud->row != NULL )
		for( x = 0;  x < ud->cnt;  x++ )
		{
			lua_pushstring( L, ud->row[x] );
			lua_rawseti( L, -2, x+1 );
		}
	return 1;
}

int lsql_row_iname(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.row_iname(db)");
	int x;
	// do it
	lua_newtable(L);
	if( ud->n_t != NULL )
		for( x = 0;  x < ud->cnt;  x++ )
		{
			lua_pushstring( L, ud->n_t[x] );
			lua_rawseti( L, -2, x+1 );
		}
	return 1;
}

int lsql_row_itype(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.row_itype(db)");
	int x,  cnt;
	// do it
	lua_newtable(L);
	cnt = ud->cnt;
	if( ud->n_t != NULL )
		for( x = 0;  x < cnt;  x++ )
		{
			lua_pushstring( L, ud->n_t[x+cnt] );
			lua_rawseti( L, -2, x+1 );
		}
	return 1;
}

int lsql_row_data(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.row_data(db)");
	int x,  cnt;
	// do it
	lua_newtable(L);
	cnt = ud->cnt;
	if( ud->row != NULL && ud->n_t != NULL )
		for( x = 0;  x < cnt;  x++ )
		{
			lua_pushstring( L, ud->n_t[x] );
			lua_pushstring( L, ud->row[x] );
			lua_rawset( L, -3 );
		}
	return 1;
}

int lsql_row_type(lua_State *L)
{
	dbud *ud = ls_checknargs_vm_getbdud(L,1,"sqlite.row_type(db)");
	int x,  cnt;
	// do it
	lua_newtable(L);
	cnt = ud->cnt;
	if( ud->row != NULL && ud->n_t != NULL )
		for( x = 0;  x < cnt;  x++ )
		{
			lua_pushstring( L, ud->n_t[x] );
			lua_pushstring( L, ud->n_t[x+cnt] );
			lua_rawset( L, -3 );
		}
	return 1;
}

// exeute a statement with no data results (return status and error string)
int lsql_exec(lua_State *L)
{
	dbud *ud = ls_checknargs_getbdud(L,2,"sqlite.exec(db)");
	const char *sql;
	char *errmsg;
	int err;
	if( !lua_isstring(L,2) )
	{
		errmsg = "Second argument to sqlite.exec(db,sql) must be a string";
		lua_pushstring(L, errmsg);
		return lua_error(L);
	}
	sql = lua_tostring(L,2);
	// do it
	err = sqlite_exec(
		  ud->db	/* The open database */
		, sql		/* SQL statement to be exec'd */
		, NULL	/* no callback */
		, NULL	/* no callback arg */
		, &errmsg	/* OUT: Error message. */
		);
	lua_pushnumber(L, (lua_Number )err);
	lua_pushstring(L,errmsg);
	if( errmsg != NULL ) sqlite_freemem( errmsg );
	return 2;
}


static const luaL_reg sqlitelib[] =
{
	{"version",			lsql_version},
	{"encoding",		lsql_encoding},
	{"open",			lsql_open},
	{"close",			lsql_close},
	{"last_insert_rowid",	lsql_last_insert_rowid},
	{"changes",		lsql_changes},
	{"error_string",		lsql_error_string},
	{"interrupt",		lsql_interrupt},
	{"busy_timeout",		lsql_busy_timeout},
	{"compile",			lsql_compile},
	{"step",			lsql_step},
	{"finalize",			lsql_finalize},
	{"row_idata",		lsql_row_idata},
	{"row_iname",		lsql_row_iname},
	{"row_itype",		lsql_row_itype},
	{"row_data",		lsql_row_data},
	{"row_type",		lsql_row_type},
	{"exec",			lsql_exec},
	// sqlite namespace table will be metatable for db userdata...
	{"__gc",			lsql_udgc},
	{NULL, NULL}
};

static int lsqlite_initialize(lua_State *L)
{
	luaL_openlib(L, LUA_SQLITELIBNAME, sqlitelib, 0);
	lua_pushliteral(L, "OK");
	lua_pushnumber(L, SQLITE_OK);
	lua_settable(L, -3);
	lua_pushliteral(L, "BUSY");
	lua_pushnumber(L, SQLITE_BUSY);
	lua_settable(L, -3);
	lua_pushliteral(L, "DONE");
	lua_pushnumber(L, SQLITE_DONE);
	lua_settable(L, -3);
	lua_pushliteral(L, "ERROR");
	lua_pushnumber(L, SQLITE_ERROR);
	lua_settable(L, -3);
	lua_pushliteral(L, "MISUSE");
	lua_pushnumber(L, SQLITE_MISUSE);
	lua_settable(L, -3);
	lua_pushliteral(L, "ROW");
	lua_pushnumber(L, SQLITE_ROW);
	lua_settable(L, -3);	
	return 0;
}

/*=========================================
// now comes the special 'loadmodule' code
//=========================================*/

LIBEXPORT int  luaLM_import(lua_State* L)
{
	return lsqlite_initialize(L);
}

LIBEXPORT const char*  luaLM_version(void)
{
	return LUA_VERSION;
}

