(* ---------------------------------------------------------------
Title         Q&D Read/Write/Compare Unit/Track/Head/Sector
Author        PhG
Overview      see help
Notes
              according to a few docs, headcount should be 255 or 240 :
              we default to best factor, as DiskEdit does too

              INT 13 - DISK - READ SECTOR(S) INTO MEMORY (1302))

              under Windows95, a volume must be locked (see INT 21/AX=440Dh/CX=084Bh)
              in order to perform direct accesses such as INT 13h reads and writes

              all versions of MS-DOS (including MS-DOS 7 [Windows 95]) have a bug
              which prevents booting on hard disks with 256 heads (FFh), so many
              modern BIOSes provide mappings with at most 255 (FEh) heads

              tick threshold was 200 but 600 seems wiser with p4
              (we'll need to check against another similar drive)

              -chk is almost certainly paranoiac
              (even with page crossing buffers, we've NEVER had any problem !)

              a few BIOSes (even modern ones, Ralf !) can't do multitrack operations :
              using oneSector was a good idea here
              (ok, ok : we're too lazy to rewrite rwCHS so it emulates that feature)

              disk last operation status may be unreliable (Award 6.0)

Bugs          we don't get the same total sector count whether using LBA or default !
              anyway, even norton diskedit (both 8.0 and 2000)
              displays such a quirk (/m maintenance mode adds 1 to cylinders !)

              CCCCC  HHH  SS  Mode    Who
                524  255  63  LBA     Award BIOS setup
              14848    9  63  normal  Award BIOS setup
                523  255  63          DTHS /X
              14818    9  63          DTHS

              11345  224  63          DE 2000 normal mode
              11344  224  63          DE 2000 maintenance mode (gives totalSectors)
               9964  255  63          Partinfo (PQ Magic tool)
              16383   16  63          DTHS v1.1a
              11344  224  63          DHTS v1.2 (total is 160086528 ok)

              support for $82 and $83 was not tested for lack of hardware

Wish List     LOCK/UNLOCK volume under Win9X ?
              a log of operations ?
              should -s be default ? or should we on the contrary have a -test option ?
              should we force an -apply option to really update data ?

              real-time read/write from one unit to another ?
              (would required ugly unit letters hacks anyway)

              log completion date/time ?

              more informative fatal() (block, operation) ?

              bigger buffer (track ?) for rawrite mode ?

              saving original data user-specified unit ?

              LFN ? bah, overkill here...

              rewrite DHTS, VITAL and UCLONE so they share common subs ?
              in another unlikely (un)life !

--------------------------------------------------------------- *)

MODULE DTHS;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM IO IMPORT WrStr, WrLn;

FROM FIO IMPORT FIXEDLIBS;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits;

(* FROM QD_LFN IMPORT w9XsupportLFN; *)

(* ------------------------------------------------------------ *)

CONST
    ENABLEREALUPDATE = TRUE; (* if FALSE, hard disk is protected against writing *)
    RESETHD          = FALSE; (* TRUE to really reset hd in resetUnit() *)
CONST
    (* progTitle     = "Q&D Read/Write/Compare Unit/Track/Head/Sector"; *)
    progTitle     = "Q&D Unit/Track/Head/Sector";
    progVersion   = "v1.3g";
    progCopyright = "by PhG";
    Banner        = progTitle+" "+progVersion+" "+progCopyright;
    progEXEname   = "DTHS";  (* must be [1..4] chars *)
CONST
    cr            = CHR(13);
    lf            = CHR(10);
    nl            = cr+lf;
    coma          = ",";
    dot           = ".";
    star          = "*";
    question      = "?";
    backslash     = "\";
    colon         = ":";
    dquote        = '"';
    blank         = " ";
    dotdot        = dot+dot;
    netslash      = backslash+backslash;
    extBIN        = dot+"BIN";
    extSOS        = dot+"SOS";
    extBAT        = dot+"BAT";
    extDAT        = dot+"DAT";
    extLOG        = dot+"LOG";
    extERR        = dot+"ERR";
    extEXE        = dot+"EXE";
    extCFG        = dot+"CFG";
    (* it's fortunate we use terse program names ! 4 chars is good ! ;-) *)
    BACKUPFILE    = progEXEname+"DATA"+extSOS;
    UNDOBATCH     = progEXEname+"UNDO"+extBAT;
    RESTARTFROM   = progEXEname+extDAT;
    REPORTFILE    = progEXEname+extLOG;
    FATALFILE     = progEXEname+extERR;
    wiblock       = 9+2; (* "###.###.###" is 512 Gb for # = 9 *)
    sepdot        = coma;
CONST
    CHKEVERY      = 256; (* let's call chkEscape every CHKEVERY loop *)
CONST
    ioBufferSize      = (8 * 512) + FIO.BufferOverhead; (* 8 or less is ok *)
    firstBufferByte   = 1;
    lastBufferByte    = ioBufferSize;
VAR
    ioBuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)

PROCEDURE sound (freq,duration,pause:CARDINAL);
BEGIN
    Lib.Sound(freq);
    Lib.Delay(duration);
    Lib.NoSound();
    Lib.Delay(pause);
END sound;

PROCEDURE alarm (  );
BEGIN
    sound(55,55,100);
    sound(55,55,10);
END alarm;

PROCEDURE alert (  );
BEGIN
    sound(550,55,100);
    sound(550,55,10);
END alert;

(* globerks because referenced in abort() *)

VAR
    DYNALLOC : BOOLEAN; (* if FALSE, fixed buffers *)
VAR
    AUDIO : BOOLEAN; (* globerk but we don't care *)
VAR
    FATALINAL : BOOLEAN; (* globerk *)
    sFatalStatusExplanation : str128; (* used by getFatalStatus() *)
    bFatalStatus : CARDINAL;

CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errParameter    = 3;
    errLongCard     = 4;
    errSyntax       = 5;
    errCommand      = 6;
    errUnit         = 7;
    errPhantom      = 8;
    errXBIOSvalues  = 9;
    errFixGeometry  = 10;
    errNonsense     = 11;
    errRange        = 12;
    errFile         = 13;
    errAlready      = 14;
    errAlreadyRO    = 15;
    errRedirected   = 16;
    errNotFound     = 17;
    errSectorSize   = 18;
    errBeyondDisk   = 19;
    errCmd          = 20;
    errCorrupted    = 21;
    errBlockRange   = 22;
    errVerifyFailure= 23;
    errBadBackupname= 24;
    errExtAPI       = 25;
    errFactor       = 26;
    errResumeProfiling=27;
    errThreshold    = 28;
    errAborted      = 29;
    errDynChk       = 30;
    errHelper       = 31;

    errBoundary     = 64;

    errInt13h           = 128;
    errXBIOSint13h      = 129;
    errWrongTHSgeometry = 130;

    errMismatch     = 255;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);

    PROCEDURE msg2 (VAR R : ARRAY OF CHAR;S1,S2:ARRAY OF CHAR);
    BEGIN
        Str.Concat(R,S1,S2);
    END msg2;

    PROCEDURE msg3 (VAR R : ARRAY OF CHAR;S1,S2,S3:ARRAY OF CHAR);
    BEGIN
         msg2(R,S1,S2);Str.Append(R,S3);
    END msg3;

CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msgHelp =
Banner+nl+
nl+
"Syntax 1 : "+progEXEname+" <unit> <R|W|C> <block> <file["+extBIN+"]> [option]..."+nl+
"Syntax 2 : "+progEXEname+" <unit> <R|W|C> <track> <head> <sector> <file["+extBIN+"]> [option]..."+nl+
"Syntax 3 : "+progEXEname+" <unit> [P] [-a|-w] [-x] [-j[j]]"+nl+
"Syntax 4 : "+progEXEname+" <unit> <V[V]> [-m] [-p[:#]] [-x] [-j[j]]"+nl+
nl+
"A) <unit> may be $8<0|1|2|3> or 8<0|1|2|3>H (hard disk),"+nl+
"   $00 or 00H (first floppy disk), $01 or 01H (second floppy disk)."+nl+
"B) For compatibility and compiler reasons, <unit> access requires"+nl+
"   512 bytes <sector>, DWORD <block>, WORD <track>, WORD <head>, WORD <sector>."+nl+
"C) IMPORTANT : should best factorization fail, -j would be silently forced."+nl+
(*%F ENABLEREALUPDATE *)
"D) PLEASE NOTE THAT UPDATING HARD DISK DATA IS NOT ENABLED !"+nl+
(*%E  *)
nl+
"R|S     read|save sector(s) to <file>"+nl+
"W|U     write|update sector(s) with <file>"+nl+
"C       compare sector(s) against <file>"+nl+
"        (return code is 255 if any mismatch was found)"+nl+
"P|I     display <unit> parameters and BIOS interrupt $13 informations"+nl+
"V[V]    verify all <unit> blocks"+nl+
"        any error is reported to "+REPORTFILE+" log ; if ESCape was hit to abort,"+nl+
"        "+RESTARTFROM+" is created to allow resuming from last checked block"+nl+
"        (VV = restart from block 0 ignoring existing "+RESTARTFROM+" if any)"+nl+
nl+
"-m      while verifying all <unit> blocks with V[V] command,"+nl+
"        report blocks containing $aa55 word magic signature to "+REPORTFILE+nl+
"-p[:#]  profile sector reading (V or R command required)"+nl+
"        -p:# sets threshold to report slow sector (default is 600, V only)"+nl+
"-t      test mode (existing data is not modified by <W|U>)"+nl+
"-c:#    count of sectors to read (default is 1)"+nl+
"-o[o]   overwrite any existing <file> (-oo = -o -r)"+nl+
"-r      ignore read-only status of existing <file>"+nl+
"-q      do not display any mismatch found"+nl+
"-a|-w   audio warning"+nl+
"-s      save unmodified data to "+BACKUPFILE+" (default) backup"+nl+
"        ("+UNDOBATCH+", created in current directory, WILL require editing)"+nl+
"-n:$    new backup name (default is "+BACKUPFILE+")"+nl+
"-d      alternate display"+nl+
"-yes    do not ask for confirmation"+nl+
"        BE SURE TO KNOW WHAT YOU ARE DOING ! USE THIS OPTION AT YOUR OWN RISK !"+nl+
"-x      ignore BIOS interrupt $13 extensions even if available"+nl+
"-ths    factorize blocks count ignoring unexpected values for TxHxS"+nl+
"-u      ignore fatal errors (most UNWISE and most UNSAFE !)"+nl+
"-j[j]   if rebuilding TxHxS is needed, instead of finding best factorization,"+nl+
"        force headcount=255 (-j) or headcount=240 (-jj)"+nl+
"-ah     assume int $1301 returns last operation status in AH (default is AL)"+nl+
"-dyn    use a dynamic buffer (default is to use a fixed buffer)"+nl+
"-??|-hh more help"+nl;
    verbosehelpmsg=
nl+
"a) -yes and -x options should NOT be used and are NOT recommended."+nl+
"b) For safety, -s option is HIGHLY recommended."+nl+
"c) -u option reports problems to "+FATALFILE+" log (created if necessary)."+nl+
"d) -p[:#] option is experimental (besides, its results may vary)."+nl+
"   Only tick ranges with non-zero sector count are reported."+nl+
"   If specified, threshold is rounded to nearest lower multiple of 100."+nl+
"   Even when resuming, the list of slow blocks is reinitialized :"+nl+
"   check "+REPORTFILE+" log for previous report."+nl+
"   Tick counter overflow is reported as 65535."+nl+
"e) -ah option is forced if "+progEXEname+extCFG+" exists in executable directory."+nl+
"f) By design, "+progEXEname+" handles only one block at a time :"+nl+
"   it is therefore not suited to fast floppy disk imaging."+nl+
"g) Undocumented -chk option checks fixed buffer page boundaries."+nl+
nl+
(*%F FIXEDLIBS *)
"Unfortunately, thanks to a fatal bug in TopSpeed Modula-2 FIO library,"+nl+
"paths are limited to 65 characters (longer ones will NOT be found). :-("+nl+
nl+
(*%E  *)
"Examples : "+progEXEname+" $80 read 0 0 1 mbr -o"+nl+
"           "+progEXEname+" $80 r 0 -c:63 track0"+nl+
"           "+progEXEname+" $80 write 0 track0 -s"+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp,errHelper :
        WrStr(msgHelp);
        IF e=errHelper THEN
            WrStr(verbosehelpmsg);
            e := errHelp;
        END;
    | errOption :
        msg3(S,'Unknown "',einfo,'" option !');
    | errParameter :
        msg3(S,'Unexpected "',einfo,'" parameter !');
    | errLongCard :
        msg3(S,'Illegal "',einfo,'" number !');
    | errSyntax :
        S := "Wrong number of parameters !";
    | errCommand :
        msg3(S,'Unknown "',einfo,'" <command> !');
    | errUnit:
        msg3(S,'Illegal "',einfo,'" <unit> specification !');
    | errPhantom:
        msg3(S,'"',einfo,'" <unit> does not exist !');
    | errXBIOSvalues:
        msg3(S,'Unexpected values returned by "',einfo,'" procedure !');
    | errFixGeometry:
        S:='Failure rebuilding values in "FixGeometry" procedure !';
    | errNonsense:
        msg2(S,einfo," option is a nonsense with specified command !");
    | errRange:
        msg2(S,einfo," range !");
    | errFile :
        msg3(S,'Illegal "',einfo,'" <file> specification !');
    | errAlready:
        msg3(S,'"',einfo,'" <file> already exists !');
    | errAlreadyRO:
        msg3(S,'"',einfo,'" <file> is read-only !');
    | errRedirected:
        S:='Redirection is not allowed with "Write" command !';
        video(progEXEname+" : ",TRUE);video(S,TRUE); (* send message to screen anyway ! *)
        IF AUDIO THEN alarm();END;
    | errNotFound:
        msg3(S,'"',einfo,'" <file> does not exist !');
    | errSectorSize :
        msg3(S,'Length of "',einfo,'" <file> is not a multiple of 512 bytes !');
    | errBeyondDisk:
        msg3(S,'Length of "',einfo,'" <file> would lead beyond <unit> limits !');
    | errCmd:
        msg3(S,'Illegal syntax for specified "',einfo,'" <command> !');
    | errCorrupted:
        S:=dquote+RESTARTFROM+dquote+" is corrupted !";
    | errBlockRange:
        S:=dquote+RESTARTFROM+dquote+" contains an out of range block number !";
    | errVerifyFailure:
        S:="Verify operation failed, please check "+dquote+REPORTFILE+dquote+" log !";
    | errBadBackupname:
        msg3(S,'Illegal "',einfo,'" new backup name !');
    | errExtAPI:
        S:='Current BIOS API (-x) does not match "'+RESTARTFROM+'" content !';
    | errFactor:
        S:='Current geometry (-j[j]) does not match "'+RESTARTFROM+'" content !';
    | errResumeProfiling:
        S:='Specified -profile option does not match "'+RESTARTFROM+'" content !';
    | errThreshold:
        msg3(S,'"',einfo,'" threshold should belong to [0..65535] range !');
    | errAborted:
        S:="Aborted by user !";
    | errDynChk:
        S:="-chk option is irrelevant when using -dyn option !";

    | errBoundary:
        IF NOT ( DYNALLOC ) THEN
            msg2(S,einfo," fixed");
        ELSE
            msg2(S,einfo," dynamic");
        END;
        Str.Append(S," buffer would cross 64Kb page boundary !");

    | errInt13h :
        msg3(S,'Unexpected BIOS interrupt $13 failure in "',einfo,'" procedure !');
        Str.Append(S,sFatalStatusExplanation);
        IF AUDIO THEN alert();END;
    | errXBIOSint13h :
        msg3(S,'Unexpected BIOS interrupt $13 extensions failure in "',einfo,'" procedure !');
        Str.Append(S,sFatalStatusExplanation);
        IF AUDIO THEN alert();END;
    | errWrongTHSgeometry:
        msg3(S,'Unexpected geometry values returned by "',einfo,'" procedure !');
        IF AUDIO THEN alert();END;

    | errMismatch :
        (* never displayed anyway ! *)
        S := "Data from <source> and from <file> do not match !";
        IF AUDIO THEN alert();END;
    ELSE
        S := "How did you get THERE ???";
    END;
    CASE e OF
    | errNone, errHelp, errMismatch :
        ;
    ELSE
        WrStr(progEXEname+" : ");WrStr(S);WrLn;
    END;

    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

PROCEDURE fatal (XBIOS:BOOLEAN;S:ARRAY OF CHAR);
BEGIN
    IF XBIOS THEN
        abort(errXBIOSint13h,S);
    ELSE
        abort(errInt13h,S);
    END;
END fatal;

(* fills global sFatalStatusExplanation and useless bFatalStatus *)

PROCEDURE initFatalStatus (  );
CONST
    msg=nl+progEXEname+" : this message should never be seen !";
BEGIN
    bFatalStatus := 00H;
    sFatalStatusExplanation := msg;
END initFatalStatus;

PROCEDURE getFatalStatus(unit:BYTE);
CONST
    placeholder = "@";
    msg = nl+progEXEname+" : int $1301 @=$@ i.e. @ !";
    sb = 040H; (* segBiosData *)
VAR
    RC:str16;
    S,Z:str128;
    statuscode:CARDINAL;
    ok,rc:BOOLEAN;
    R:SYSTEM.Registers;
    biosFloppyLastRC [sb:041H] : BYTE;
    biosFixedLastRC  [sb:074H] : BYTE;
BEGIN
    IF unit < BYTE(80H) THEN
        bFatalStatus := CARDINAL(biosFloppyLastRC); (* DISKETTE - LAST OPERATION STATUS *)
    ELSE
        bFatalStatus := CARDINAL(biosFixedLastRC);  (* FIXED DISK LAST OPERATION STATUS (except ESDI drives) *)
    END;

    R.AH := 01H; (* get status of last operation *)
    R.DL := unit; (* bit 7 for hard disk *)
    Lib.Intr (R,13H);
    rc:=( NOT (SYSTEM.CarryFlag IN R.Flags) ); (* CF clear if successful status 00h *)
    IF FATALINAL THEN
        statuscode:= CARDINAL(R.AL);
    ELSE
        statuscode:= CARDINAL(R.AH); (* not standard, Ralf ! *)
    END;
    CASE statuscode OF
    | 000H : S:="successful completion";
    | 001H : S:="invalid function in AH or invalid parameter";
    | 002H : S:="address mark not found";
    | 003H : S:="disk write-protected";
    | 004H : S:="sector not found/read error";
    | 005H : S:="reset failed (hard disk)";
    (* | 005H : S:="data did not verify correctly (TI Professional PC)"; *)
    | 006H : S:="disk changed (floppy)";
    | 007H : S:="drive parameter activity failed (hard disk)";
    | 008H : S:="DMA overrun";
    | 009H : S:="data boundary error (attempted DMA across 64K boundary or >80h sectors)";
    | 00AH : S:="bad sector detected (hard disk)";
    | 00BH : S:="bad track detected (hard disk)";
    | 00CH : S:="unsupported track or invalid media";
    | 00DH : S:="invalid number of sectors on format (PS/2 hard disk)";
    | 00EH : S:="control data address mark detected (hard disk)";
    | 00FH : S:="DMA arbitration level out of range (hard disk)";
    | 010H : S:="uncorrectable CRC or ECC error on read";
    | 011H : S:="data ECC corrected (hard disk)";
    | 020H : S:="controller failure";
	| 030H : S:="drive does not support media sense"; (* added by hand from MEMORY.LST *)
    | 031H : S:="no media in drive (IBM/MS INT 13 extensions)";
    | 032H : S:="incorrect drive type stored in CMOS (Compaq)";
    | 040H : S:="seek failed";
    | 080H : S:="timeout (not ready)";
    | 0AAH : S:="drive not ready (hard disk)";
    | 0B0H : S:="volume not locked in drive (INT 13 extensions)";
    | 0B1H : S:="volume locked in drive (INT 13 extensions)";
    | 0B2H : S:="volume not removable (INT 13 extensions)";
    | 0B3H : S:="volume in use (INT 13 extensions)";
    | 0B4H : S:="lock count exceeded (INT 13 extensions)";
    | 0B5H : S:="valid eject request failed (INT 13 extensions)";
    | 0B6H : S:="volume present but read protected (INT 13 extensions)";
    | 0BBH : S:="undefined error (hard disk)";
    | 0CCH : S:="write fault (hard disk)";
    | 0E0H : S:="status register error (hard disk)";
    | 0FFH : S:="sense operation failed (hard disk)";
    ELSE
             S:="unexpected fatal disk operation status";
    END;

    Str.Copy(Z,msg);

    IF FATALINAL THEN
        RC:="AL";
    ELSE
        RC:="AH";
    END;
    Str.Subst(Z,placeholder,RC);

    Str.CardToStr (LONGCARD(statuscode),RC,16,ok);
    IF statuscode < 10H THEN Str.Prepend(RC,"0");END;
    Str.Lows(RC);
    Str.Subst(Z,placeholder,RC);

    (*
    Str.CardToStr (LONGCARD(bFatalStatus),RC,16,ok);
    IF bFatalStatus < 10H THEN Str.Prepend(RC,"0");END;
    Str.Lows(RC);
    Str.Subst(Z,placeholder,RC);
    *)

    Str.Subst(Z,placeholder,S);
    Str.Copy(sFatalStatusExplanation,Z);
END getFatalStatus;

PROCEDURE fixFatalStatusBios ():BOOLEAN;
VAR
    ini:str128;
BEGIN
    Lib.ParamStr(ini,0); (* retrieve executable location : yes, we assume it ! *)
    Str.Caps(ini); (* safety *)
    Str.Subst(ini,extEXE,extCFG);
    RETURN FIO.Exists(ini);
END fixFatalStatusBios;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE withinlc (v,lower,upper:LONGCARD):BOOLEAN;
BEGIN
    IF v < lower THEN RETURN FALSE; END;
    RETURN (v <= upper);
END withinlc;

PROCEDURE within (v,lower,upper:CARDINAL):BOOLEAN;
BEGIN
    IF v < lower THEN RETURN FALSE; END;
    RETURN (v <= upper);
END within;

PROCEDURE getvallc (S:ARRAY OF CHAR;VAR v : LONGCARD ):BOOLEAN;
VAR
    lc   : LONGCARD;
    base : CARDINAL;
    ok   : BOOLEAN;
BEGIN
    IF S[0]="$" THEN
        Str.Delete(S,0,1);
        base := 16;
    ELSE
        base := 10;
    END;
    lc:=Str.StrToCard(S,base,ok);
    IF NOT(ok) THEN RETURN FALSE; END;
    IF lc > MAX(LONGCARD) THEN RETURN FALSE; END;
    v := lc;
    RETURN TRUE;
END getvallc;

PROCEDURE getval (S:ARRAY OF CHAR;VAR v : CARDINAL ):BOOLEAN;
VAR
    lc:LONGCARD;
BEGIN
    lc := LONGCARD(v);
    IF getvallc(S, lc) THEN
        IF lc > MAX(CARDINAL) THEN
            RETURN FALSE;
        ELSE
            v := CARDINAL(lc);
            RETURN TRUE;
        END;
    ELSE
        RETURN FALSE;
    END;
END getval;

PROCEDURE defaultExtension (VAR S : ARRAY OF CHAR;ext:ARRAY OF CHAR);
BEGIN
    IF Str.RCharPos(S,dot) # MAX(CARDINAL) THEN RETURN; END;
    Str.Append(S,ext);
END defaultExtension;

(* try and avoid biggest problems : [u:][[\]path\]f8[.e3]  *)

PROCEDURE chkFilename (S:ARRAY OF CHAR ):BOOLEAN ;
VAR
    u,d,f8,e3:str128;
    n:CARDINAL;
BEGIN
    n:=0;
    IF chkJoker(S) THEN INC(n); END;
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN INC(n);END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN INC(n);END;

    Lib.SplitAllPath(S,u,d,f8,e3);
    IF within(Str.Length(u),0,2)=FALSE THEN INC(n);END;
    IF within(Str.Length(e3),0,1+3)=FALSE THEN INC(n);END;
    IF within(Str.Length(f8),1,8)=FALSE THEN INC(n);END;

    CASE CharCount(S,colon) OF
    | 0 : ;
    | 1 : IF Str.CharPos(S,colon) # 1 THEN INC(n); END;
    ELSE
        INC(n);
    END;

    RETURN (n=0);
END chkFilename;

(* f8[.e3] *)

PROCEDURE chkBackupname (VAR S:ARRAY OF CHAR; ext:ARRAY OF CHAR):BOOLEAN;
VAR
     path,u,d,f8,e3:str128;
     n:CARDINAL;
BEGIN
    n:=0;
    IF chkJoker(S) THEN INC(n);END;
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN INC(n);END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN INC(n);END;

    IF Str.CharPos(S,dot)=MAX(CARDINAL) THEN Str.Append(S,ext);END;

    Lib.SplitAllPath(S,u,d,f8,e3);
    IF within(Str.Length(u),0,2)=FALSE THEN INC(n);END;
    IF within(Str.Length(e3),0,1+3)=FALSE THEN INC(n);END;
    IF within(Str.Length(f8),1,8)=FALSE THEN INC(n);END;

    CASE CharCount(S,colon) OF
    | 0 : ;
    ELSE
        INC(n);
    END;

    Lib.MakeAllPath(path,"","",f8,e3);
    IF same(path,S)=FALSE THEN INC(n); END;

    RETURN (n=0);
END chkBackupname;

(* ------------------------------------------------------------ *)

CONST
    legalUnits = "$80"+delim+"80H"+delim+"128"+delim+"0X80"+delim+
                 "$81"+delim+"81H"+delim+"129"+delim+"0X81"+delim+
                 "$00"+delim+"00H"+delim+"0"+delim+"0X00"+delim+
                 "A:"+delim+"$0"+delim+"0X0"+delim+
                 "$01"+delim+"01H"+delim+"1"+delim+"0X01"+delim+
                 "B:"+delim+"$1"+delim+"0X1"+delim+
                 "$82"+delim+"82H"+delim+"130"+delim+"0X82"+delim+
                 "$83"+delim+"83H"+delim+"131"+delim+"0X83";

PROCEDURE chkUnit (VAR unit:BYTE;S:ARRAY OF CHAR):BOOLEAN;
VAR
    i:CARDINAL;
BEGIN
    i := getStrIndex(delim,S,legalUnits);
    CASE i OF
    |  1..4  : unit := BYTE(80H);
    |  5..8  : unit := BYTE(81H);
    |  9..15 : unit := BYTE(00H);
    | 16..22 : unit := BYTE(01H);
    | 23..26 : unit := BYTE(82H);
    | 27..30 : unit := BYTE(83H);
    ELSE
       RETURN FALSE; (* 0 *)
    END;
    RETURN TRUE;
END chkUnit;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    sectorSize    = 512;
    mintrack      = 0;
    minhead       = 0;
    minsector     = 1;
    minblock      = 0;
    maxheadcount  = 255; (* yes, 255 and not 256 because we want to avoid MessDOS fatal boot bug *)
    mincountofsectors = 1;
CONST
    firstByteInSector = 0;
    lastByteInSector  = sectorSize-1; (* 511 = $1ff *)
    oneSector         = 1;
TYPE
    SectorType        = ARRAY [firstByteInSector..lastByteInSector] OF BYTE;
    ptrToSectorType   = POINTER TO SectorType;
VAR (* globerks *)
    buffSector        : SectorType;
    buffSectorDisk    : SectorType;
    pbuffSector       : ptrToSectorType;
    pbuffSectorDisk   : ptrToSectorType;

(*
T = block AND $03ff
H = block AND $ff
S = block AND $3f

if trackcount <= 1023, perform as above
if not, do
T = 1023
H = maxhead AND $ff
S = maxsector AND $3f
*)

PROCEDURE blockToCHS (trackcount,headcount,sectorcount:CARDINAL;
                     block:LONGCARD;
                     VAR track,head,sector:CARDINAL);
VAR
    v:LONGCARD;
BEGIN
    v      := block;
    sector := minsector + CARDINAL (v MOD LONGCARD(sectorcount));

    v      := block DIV (LONGCARD(sectorcount));
    head   := minhead   + CARDINAL (v MOD LONGCARD(headcount));

    v      := block DIV (LONGCARD(sectorcount) * LONGCARD(headcount));
    track  := mintrack  + CARDINAL (v MOD LONGCARD(trackcount));
END blockToCHS;

PROCEDURE CHStoBlock (trackcount,headcount,sectorcount:CARDINAL;
                     track,head,sector:CARDINAL;
                     VAR block:LONGCARD);
VAR
    v:LONGCARD;
BEGIN
    v:= ( LONGCARD(track) * LONGCARD(headcount) ) + LONGCARD(head);
    v:= v * LONGCARD(sectorcount) + LONGCARD(sector) - minsector; (* i.e. -1 *)
    block :=v;
END CHStoBlock;

(* assume $00, $01, $80, $81 *)

PROCEDURE isFloppy (unit:BYTE  ):BOOLEAN;
BEGIN
    CASE unit OF
    | 80H,81H,82H,83H:
        RETURN FALSE;
    ELSE
        RETURN TRUE;
    END;
END isFloppy;

(*
    first, we try to find headcount (<=highest) and trackcount
    so that their product is v (totalblocks)
    if we fail finding exact values, we rely to headcount=highest
*)

CONST
    FORCEBEST = 0;
    FORCE255  = 255;
    FORCE240  = 240;

PROCEDURE findBestFactors (BESTFIT:CARDINAL;v:LONGCARD;highest:LONGCARD;
                          VAR headcount,trackcount:CARDINAL);
VAR
    a,b : LONGCARD;
    needhack : BOOLEAN;
BEGIN
    needhack := TRUE;
    a        := highest;      (* maxheadcount is 255 *)
    LOOP
        b:=(v DIV a);
        IF (a*b)=v THEN
            needhack := FALSE;
            EXIT;
        END;
        DEC (a);
        IF a < 2 THEN EXIT; END;
    END;
    IF BESTFIT # FORCEBEST THEN needhack:=TRUE; END;
    IF needhack THEN
        IF BESTFIT = FORCEBEST THEN BESTFIT := FORCE255; END; (* fix 0-divide *)
        a := LONGCARD(BESTFIT); (* 255 or 240 *)
        b := (v DIV a);
        IF b > MAX(CARDINAL) THEN b:=MAX(CARDINAL);END; (* ugly fix just in case because it should never happen *)
    END;
    headcount  := CARDINAL(a);
    trackcount := CARDINAL(b);
END findBestFactors;

(* ------------------------------------------------------------ *)

TYPE
    QWORD = RECORD
        lo:LONGWORD;
        hi:LONGWORD;
    END;
    XBIOSinfotype = RECORD
        bufsize         : WORD;   (* always $1A here because v1.x call *)
        flag            : WORD;
        cylinders       : LONGWORD;  (* physical *)
        heads           : LONGWORD;  (* id. *)
        sectorsPerTrack : LONGWORD;  (* id. *)
        totalSectors    : QWORD;
        bytesPerSector  : WORD;
        EDDdata         : LONGWORD;  (* unused because we only call v1.x function *)
    END;

PROCEDURE encodeCS (cylinder,sector:WORD; VAR bcylinder,bsector:BYTE);
CONST
    maskKeepLo = WORD(000FFH); (* 0000000011111111 *)
    maskKeepHi = WORD(0FF00H); (* 1111111100000000 *)
VAR
    v : WORD;
BEGIN
    v := cylinder AND maskKeepLo;
    bcylinder := BYTE(v);
    v := cylinder AND maskKeepHi;
    v := v >> WORD(2);
    bsector := BYTE(v) OR BYTE(sector);
END encodeCS;

PROCEDURE decodeCS (bcylinder,bsector:BYTE; VAR vcylinder,vsector:WORD);
CONST
    maskKeepSector = BYTE(03FH); (* 00111111 *)
    maskKeepCylinder=BYTE(0C0H); (* 11000000 *)
VAR
    v : BYTE;
BEGIN
    v := bsector AND maskKeepSector;
    vsector := WORD(v);
    v := bsector AND maskKeepCylinder;
    vcylinder := WORD(v) << WORD(2);
    vcylinder := vcylinder OR WORD(bcylinder);
END decodeCS;

PROCEDURE resetUnit (DEBUG,enabledResetHD:BOOLEAN;unit:BYTE):BOOLEAN ;
VAR
    R : SYSTEM.Registers;
    rc:BOOLEAN;
BEGIN
IF DEBUG THEN WrStr("::: resetUnit - ");END;
    IF isFloppy(unit) THEN
IF DEBUG THEN WrStr("Floppy");WrLn;END;
        R.AH := 00H; (* reset disk system *)
        R.DL := unit;
        Lib.Intr (R,13H);
        rc:=( NOT (SYSTEM.CarryFlag IN R.Flags) );
    ELSE
IF DEBUG THEN WrStr("Hard disk");WrLn;END;
        IF enabledResetHD THEN
            R.AH := 0DH; (* reset hard disk *)
            R.DL := unit;
            Lib.Intr (R,13H);
            rc:=NOT (SYSTEM.CarryFlag IN R.Flags);
        ELSE
            rc:=TRUE; (* we won't reset hard disk *)
        END;
    END;
    IF NOT(rc) THEN getFatalStatus(unit);END;
    RETURN rc;
END resetUnit;

PROCEDURE chkExtendedSupport (DEBUG:BOOLEAN;unit:BYTE;
                             VAR EDDmajor,EDDflag:BYTE):BOOLEAN;
VAR
    R : SYSTEM.Registers;
    ok: BOOLEAN;
BEGIN
IF DEBUG THEN WrStr("::: chkExtendedSupport");WrLn;END;
    IF isFloppy(unit) THEN RETURN FALSE; END;
    ok:=FALSE;
    R.AH := 41H; (* installation check *)
    R.BX := 55AAH;
    R.DL := unit;
    Lib.Intr (R,13H);
    IF NOT (SYSTEM.CarryFlag IN R.Flags) THEN
        IF R.BX = 0AA55H THEN
            IF (R.CX AND 0001H)=0001H THEN (* extended disk access functions (AH=42h-44h,47h,48h) supported *)
                EDDmajor:=R.AH;
                EDDflag :=BYTE (R.CX AND 07H); (* we care about bits 0..2 i.e. %111 *)
                ok:=TRUE;
            END;
        END;
    END;
    IF NOT(ok) THEN getFatalStatus(unit);END;
    RETURN ok;
END chkExtendedSupport;

PROCEDURE getXBIOSvalues (DEBUG:BOOLEAN;
                         VAR cyl,head,sec,bps:WORD;VAR totalSectors:LONGWORD;
                         v:XBIOSinfotype):BOOLEAN;
VAR
    n:CARDINAL;
BEGIN
IF DEBUG THEN
    WrStr("::: getXBIOSvalues");WrLn;
    WrStr(":::   bytes per sector ");IO.WrCard(v.bytesPerSector,0);WrLn;
    WrStr(":::   cylinders        ");IO.WrLngCard(v.cylinders,0);WrLn;
    WrStr(":::   heads            ");IO.WrLngCard(v.heads,0);WrLn;
    WrStr(":::   sectors          ");IO.WrLngCard(v.sectorsPerTrack,0);WrLn;
    WrStr(":::   total            ");IO.WrLngCard(v.totalSectors.lo,0);WrLn;
END;
    n:=0;
    IF v.bytesPerSector  # WORD(sectorSize)    THEN INC(n);END;
    IF v.cylinders       > LONGWORD(MAX(WORD)) THEN INC(n);END;
    IF v.heads           > LONGWORD(MAX(WORD)) THEN INC(n);END;
    IF v.sectorsPerTrack > LONGWORD(MAX(WORD)) THEN INC(n);END;
    IF n # 0 THEN RETURN FALSE; END;

    cyl :=WORD( v.cylinders);
    head:=WORD( v.heads);
    sec :=WORD( v.sectorsPerTrack);
    bps :=WORD( v.bytesPerSector);

    totalSectors:=v.totalSectors.lo; (* we won't handle more than LONGCARD here *)

    (* don't forget to normalize counts for 0.. range ! *)

    DEC (cyl);
    DEC (head);
    RETURN TRUE;
END getXBIOSvalues;

(*
    CH = low eight bits of maximum cylinder number
    CL = maximum sector number (bits 5-0)
         high two bits of maximum cylinder number (bits 7-6)

    the maximum cylinder number reported in CX is usually two less than
    the total cylinder count reported in the fixed disk parameter table
    (see INT 41h,INT 46h) because early hard disks used the last cylinder
    for testing purposes; however, on some Zenith machines, the maximum
    cylinder number reportedly is three less than the count in the fixed
    disk parameter table.
    for BIOSes which reserve the last cylinder for testing purposes, the
    cylinder count is automatically decremented

    SeeAlso: INT 41"HARD DISK 0"
*)

CONST
    rcNoProblemo       = 0;
    rcPhantom          = 1;
    rcInt13h           = 2;
    rcXBIOSint13h      = 3;
    rcXBIOSvalues      = 4;
    rcWrongTHSgeometry = 5;

PROCEDURE chkTHSgeometry (maxTrack,maxHead,maxSector:CARDINAL):BOOLEAN;
VAR
    i,pb,v:CARDINAL;
BEGIN
    pb:=0;
    FOR i:=1 TO 3 DO
        CASE i OF
        | 1: v:=maxTrack;
        | 2: v:=maxHead;
        | 3: v:=maxSector;
        END;
        CASE v OF
        | 0 , MAX(CARDINAL): INC(pb);
        END;
    END;
    RETURN (pb = 0);
END chkTHSgeometry;

PROCEDURE buildtotal (maxtrack,maxhead,maxsector:CARDINAL):LONGCARD;
VAR
    trackcount,headcount,sectorcount:CARDINAL;
    blockcount,totalSectors : LONGCARD;
BEGIN
    trackcount := maxtrack -mintrack +1;
    headcount  := maxhead  -minhead  +1;
    sectorcount:= maxsector-minsector+1;
    blockcount := LONGCARD(trackcount)*LONGCARD(headcount)*LONGCARD(sectorcount);
    totalSectors := blockcount;
(*
WrStr("::: maxtrack     = ");IO.WrCard(maxtrack,0);WrLn;
WrStr("::: maxhead      = ");IO.WrCard(maxhead,0);WrLn;
WrStr("::: maxsector    = ");IO.WrCard(maxsector,0);WrLn;
WrStr("::: totalSectors = ");IO.WrLngCard(totalSectors,0);WrLn;
*)
    RETURN totalSectors;
END buildtotal;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE trydiv (VAR q : LONGCARD ;v,d:LONGCARD):BOOLEAN ;
BEGIN
    q := v DIV d;
(*
WrStr("value = ");WrLngCard(v,0);
WrStr("  divisor = ");WrLngCard(d,0);
WrStr("  quotient = ");WrLngCard(q,0);WrLn;
*)
    RETURN ( v = (q * d) );
END trydiv;

CONST
    minfactentry = 0;
    maxfactentry = 1;
    minfactor    = 1;    (* so we can flag using minfactor-1=0 *)
    maxfactor    = 32+1; (* should do for at most 2^32 ! *)
TYPE
    factortype = RECORD
       k    : LONGCARD;
       flag : BOOLEAN;
    END;
VAR
    factor : ARRAY[minfactentry..maxfactentry] , [minfactor..maxfactor] OF factortype;

PROCEDURE factorize (where:CARDINAL; v:LONGCARD):CARDINAL;
VAR
    i,ndx:CARDINAL;
    quotient, divisor,added : LONGCARD;
BEGIN
    ndx:=minfactor-1;
    (* handle trivial cases *)
    IF v = 0 THEN RETURN ndx; END;
    IF v = 1 THEN
        INC(ndx);
        factor[where][ndx].k:=v;
        factor[where][ndx].flag:=TRUE;
        RETURN ndx;
    END;

    FOR i:=1 TO 2 DO
        CASE i OF
        | 1: divisor := 2;
        | 2: divisor := 3;
        END;
        LOOP
            IF trydiv (quotient, v,divisor) THEN
                INC(ndx);
                IF ndx > maxfactor THEN RETURN minfactor-1; END; (* should never happen *)
                factor[where][ndx].k:=divisor;
                factor[where][ndx].flag:=TRUE;
                v:=quotient;
                IF quotient < divisor THEN RETURN ndx; END;
            ELSE
                EXIT;
            END;
        END;
    END;

    (*
    after 2 and 3, gen next prime (5, 7, 11, 13, 17, 19, 23, 25, 29, 31...)
    by adding alternatively 2 then 4 to previous value
    *)

    added := 2;
    INC(divisor,added); (* 3+2=5 *)
    LOOP
        IF trydiv (quotient,v,divisor) THEN
            INC(ndx);
            IF ndx  > maxfactor THEN RETURN minfactor-1; END;
            factor[where][ndx].k   :=divisor;
            factor[where][ndx].flag:=TRUE;
            v:=quotient;
            IF quotient < divisor THEN RETURN ndx; END;
        ELSE
            INC(divisor,added);
            IF added=2 THEN
                added:=4;
            ELSE (* is 4 *)
                added:=2;
            END;
        END;
    END;

    RETURN ndx;
END factorize;

PROCEDURE findinfact (where,ndx:CARDINAL; limit:LONGCARD):LONGCARD;
VAR
    v, k: LONGCARD;
    i : CARDINAL;
BEGIN
    (*
    (* lazy eval : we don't find highest value here *)
    v := 1;
    i := minfactor-1;
    LOOP
        INC(i);
        IF i > ndx THEN EXIT; END;
        IF factor[where][i].flag THEN
            k:=factor[where][i].k;
            IF v * k <= limit THEN
                factor[where][i].flag := FALSE; (* no longer useable *)
                v := v * k;
            ELSE
                EXIT;
            END;
        END;
    END;
    *)
    v := 1;
    i := ndx+1;
    LOOP
        DEC(i);
        IF i < minfactor THEN EXIT; END;
        IF factor[where][i].flag THEN
            k:=factor[where][i].k;
            IF v * k <= limit THEN
                factor[where][i].flag := FALSE; (* no longer useable *)
                v := v * k;
            END;
        END;
    END;
    RETURN v;
END findinfact;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE buildTHSfromTOTAL (VAR maxTrack,maxHead,maxSector:WORD;
                            BESTFIT:CARDINAL;totalSectors:LONGCARD ):BOOLEAN;
CONST
    maxsectorcount    = 63;
    maxsectorcountalt = 255; (* desperate try to keep CARDINALs everywhere *)
    maxheadcount      = 255;
    maxtrackcount     = MAX(LONGCARD);
VAR
    rebuilt,v,quotient:LONGCARD;
    t,h,s,smax:LONGCARD;
    headcount,trackcount,sectorcount:CARDINAL;
    retry,ndx,ndxref:CARDINAL;
    rc:BOOLEAN;
BEGIN
    rc:=TRUE;
    retry := 1;
    LOOP
        v:=totalSectors;
        CASE retry OF
        | 1 :
            s:=maxsectorcount; (* try 63 first *)
            IF trydiv(quotient, v,s) THEN
                v:=quotient;
            ELSE
                s:=0;
            END;
            smax := maxsectorcount;
        | 2 :
            s:=0;
            smax := maxsectorcountalt;
        END;
        (*
        FOR now, don't care about BESTFIT which is a BAD idea here
        CASE BESTFIT OF (* should always be FORCEBEST *)
        | FORCEBEST : h:=16;
        | FORCE255:   h:=255;
        | FORCE240:   h:=240;
        ELSE
                      h:=16; (* safety *)
        END;
        *)
        ndx:=factorize(minfactentry,v);

        IF s = 0 THEN s:=findinfact(minfactentry,ndx,smax);END;
        h:=findinfact(minfactentry,ndx,maxheadcount);
        t:=findinfact(minfactentry,ndx,maxtrackcount);

        trackcount  := CARDINAL(t);
        headcount   := CARDINAL(h);
        sectorcount := CARDINAL(s);

        rebuilt     := LONGCARD(trackcount) * LONGCARD(headcount) * LONGCARD(sectorcount); (* safety *)
        IF rebuilt = totalSectors THEN EXIT; END;
        INC(retry);
        IF retry > 2 THEN rc:=FALSE; EXIT; END;
    END;
    maxTrack    := trackcount + mintrack -1;
    maxHead     := headcount  + minhead  -1;
    maxSector   := sectorcount+ minsector-1;
    RETURN rc;
END buildTHSfromTOTAL;

(* ------------------------------------------------------------ *)

PROCEDURE getGeometry (DEBUG:BOOLEAN;unit:BYTE;XBIOS,FIXBADTHS:BOOLEAN;BESTFIT:CARDINAL;
                      VAR maxTrack,maxHead,maxSector,bytesPerSector:WORD;
                      VAR totalSectors:LONGWORD):CARDINAL;
VAR
    R : SYSTEM.Registers;
    bcylinder,bsector:BYTE;
    XBIOSinfo:XBIOSinfotype;
    rc:CARDINAL;
    good:BOOLEAN;
BEGIN
IF DEBUG THEN WrStr("::: getGeometry - ");END;
    IF isFloppy(unit) THEN XBIOS:=FALSE; END; (* safety, though it is already trapped *)

    IF XBIOS THEN
IF DEBUG THEN WrStr("XBIOS");WrLn;END;
        XBIOSinfo.bufsize := 1AH; (* $1A v1.x or $1E v2.x *)

        R.AH := 48H; (* get drive parameters *)
        R.DL := unit;
        R.DS := Seg(XBIOSinfo); (* was seg(faraddress()) *)
        R.SI := Ofs(XBIOSinfo);
        Lib.Intr (R,13H);
        (* IF BYTE(R.AH) = BYTE(00H) THEN *)
        IF NOT (SYSTEM.CarryFlag IN R.Flags) THEN
            IF getXBIOSvalues (DEBUG,maxTrack,maxHead,maxSector,
                              bytesPerSector,totalSectors,XBIOSinfo) THEN
                rc:=rcNoProblemo;
            ELSE
                rc:=rcXBIOSvalues;
            END;
        ELSE
            rc:=rcXBIOSint13h;
        END;
    ELSE
IF DEBUG THEN WrStr("bios");WrLn;END;
        R.ES := 0;
        R.DI := 0; (* guard against BIOS bugs, according to canonical intlist.zip *)
        R.AH := 08H; (* get drive parameters *)
        R.DL := unit;
        Lib.Intr (R,13H);
        (* IF BYTE(R.AH) = BYTE(00H) THEN *)
        IF NOT (SYSTEM.CarryFlag IN R.Flags) THEN
            (*
            mask bit 7 only ! compare $00,$01,$80,$81 with number of drives
            in fact, useful for floppies only, because phantom hd trapped by error
            *)
            IF ((CARDINAL(unit) AND 7FH)+1) > CARDINAL(R.DL) THEN
                rc:=rcPhantom;
            ELSE
                bcylinder:=R.CH;
                bsector  :=R.CL;
                decodeCS(bcylinder,bsector,maxTrack,maxSector);
                maxHead:=WORD(R.DH);
                totalSectors := buildtotal(maxTrack,maxHead,maxSector);
                rc:=rcNoProblemo;
            END;
        ELSE
            rc:=rcInt13h;
        END;
    END;
IF DEBUG THEN
    WrStr(":::   maxtrack         ");IO.WrCard(maxTrack,0);WrLn;
    WrStr(":::   maxhead          ");IO.WrCard(maxHead,0);WrLn;
    WrStr(":::   maxsector        ");IO.WrCard(maxSector,0);WrLn;
END;
    IF rc = rcNoProblemo THEN
        IF chkTHSgeometry(maxTrack,maxHead,maxSector) = FALSE THEN
            IF FIXBADTHS THEN
                good:=buildTHSfromTOTAL(maxTrack,maxHead,maxSector, BESTFIT,totalSectors);
                IF good THEN
                    rc:=rcNoProblemo;
                ELSE
                    rc:=rcWrongTHSgeometry;
                END;
            ELSE
                rc:=rcWrongTHSgeometry;
            END;
        END;
    ELSE
        getFatalStatus(unit);
    END;
    RETURN rc;
END getGeometry;

PROCEDURE fixGeometry (DEBUG:BOOLEAN;unit:BYTE;XBIOS:BOOLEAN;BESTFIT:CARDINAL;
                      maxtrackorg,maxheadorg,maxsectororg:CARDINAL;
                      totalSectors:LONGCARD;
                      VAR maxtrack,maxhead,maxsector:CARDINAL;
                      VAR maxblock:LONGCARD;
                      VAR trackcount,headcount,sectorcount:CARDINAL;
                      VAR blockcount:LONGCARD;
                      VAR fixTHSneeded,fixedPerfect:BOOLEAN):CARDINAL;
VAR
    v:LONGCARD;
BEGIN
    fixTHSneeded:=FALSE;
IF DEBUG THEN WrStr("::: fixGeometry");WrLn;END;
    IF isFloppy(unit) THEN XBIOS:=FALSE; END; (* safety, though it is already trapped *)

    maxtrack   := maxtrackorg;
    maxhead    := maxheadorg;
    maxsector  := maxsectororg;

    trackcount := maxtrack -mintrack +1;
    headcount  := maxhead  -minhead  +1;
    sectorcount:= maxsector-minsector+1;

    blockcount := LONGCARD(trackcount)*LONGCARD(headcount)*LONGCARD(sectorcount);
    maxblock   := blockcount-1+minblock;

IF DEBUG THEN
    WrStr(":::   maxtrack         ");IO.WrCard(maxtrack,0);WrLn;
    WrStr(":::   maxhead          ");IO.WrCard(maxhead,0);WrLn;
    WrStr(":::   maxsector        ");IO.WrCard(maxsector,0);WrLn;
    WrStr(":::   maxblock         ");IO.WrLngCard(maxblock,0);WrLn;
    WrStr(":::");WrLn;
    WrStr(":::   trackcount       ");IO.WrCard(trackcount,0);WrLn;
    WrStr(":::   headcount        ");IO.WrCard(headcount,0);WrLn;
    WrStr(":::   sectorcount      ");IO.WrCard(sectorcount,0);WrLn;
    WrStr(":::   blockcount       ");IO.WrLngCard(blockcount,0);WrLn;
END;

    IF XBIOS THEN
        fixTHSneeded := (blockcount # totalSectors); (* NEWGEOMETRY *)
        IF fixTHSneeded THEN
IF DEBUG THEN WrLn;WrStr("::: rebuilding TxHxS for XBIOS use is needed now...");WrLn;END;
            (* rebuild THS from totalSectors assuming maxsectororg is correct and trying 256 heads *)
            (* should we assume 255 heads or finds a better matching product ? *)
            blockcount := totalSectors;
            maxblock   := blockcount-1+minblock;

            v:=blockcount; (* T*H*S *)
            v:=v DIV LONGCARD(sectorcount); (* T*H *)

            findBestFactors(BESTFIT,v,LONGCARD(maxheadcount), headcount,trackcount);

            maxhead    := headcount + minhead -1;
            maxtrack   := trackcount + mintrack -1;

IF DEBUG THEN
    WrLn;
    WrStr(":::   maxtrack         ");IO.WrCard(maxtrack,0);WrLn;
    WrStr(":::   maxhead          ");IO.WrCard(maxhead,0);WrLn;
    WrStr(":::   maxsector        ");IO.WrCard(maxsector,0);WrLn;
    WrStr(":::   maxblock         ");IO.WrLngCard(maxblock,0);WrLn;
    WrStr(":::");WrLn;
    WrStr(":::   trackcount       ");IO.WrCard(trackcount,0);WrLn;
    WrStr(":::   headcount        ");IO.WrCard(headcount,0);WrLn;
    WrStr(":::   sectorcount      ");IO.WrCard(sectorcount,0);WrLn;
    WrStr(":::   blockcount       ");IO.WrLngCard(blockcount,0);WrLn;
END;
            (*
            v := LONGCARD(trackcount)*LONGCARD(headcount)*LONGCARD(sectorcount);
            IF v # blockcount THEN RETURN errFixGeometry;END;
            *)
        END;
    END;

    v:=LONGCARD(trackcount) * LONGCARD(headcount) * LONGCARD(sectorcount);
    fixedPerfect := (blockcount = v);

    RETURN errNone;
END fixGeometry;

(* ------------------------------------------------------------ *)

CONST
    opRead  = BYTE(02H);
    opWrite = BYTE(03H);

TYPE
    diskAddressPacketType = RECORD
        packetsize : BYTE;
        reserved   : BYTE; (* must be $00 *)
        count      : WORD; (* number of blocks to transfer: updated after call *)
        (* DWORD : transfer buffer *)
        ofsBuf     : WORD;
        segBuf     : WORD;
        startblock : QWORD; (* starting absolute block number *)
        (*
           for non-LBA devices, compute as
           (Cylinder*NumHeads + SelectedHead) * SectorPerTrack +
           SelectedSector - 1
        *)
    END;

PROCEDURE rwCHS (unit:BYTE;XBIOS:BOOLEAN;command:BYTE;block:LONGCARD;
                track,head,sector,count,bufsegment,bufoffset:CARDINAL):BYTE;
VAR
    R : SYSTEM.Registers;
    bcylinder,bsector,LBAcmd:BYTE;
    dap:diskAddressPacketType;
BEGIN
    CASE command OF
    | opRead : ; (* let it be ! *)
    | opWrite: ; (* let it be ! *)
(*%F ENABLEREALUPDATE *)
        command:=opRead; (* don't take chances with hard disk ! *)
(*%E  *)
    ELSE
        RETURN BYTE(0FFH); (* do nothing and force a problem *)
    END;

    IF XBIOS THEN
        dap.packetsize:=10H;
        dap.reserved  :=00H;
        dap.count     :=count;       (* $7f at most ! *)
        dap.segBuf    :=bufsegment;
        dap.ofsBuf    :=bufoffset;
        dap.startblock.lo:=block; (* assume our qword is a longword *)
        dap.startblock.hi:=0;
        CASE command OF
        | opRead:
            LBAcmd:=42H;
        | opWrite:
            LBAcmd:=43H;
            R.AL  :=00H; (* no verify, and we don't care about v2.1 ! *)
        END;
        R.AH:=LBAcmd;
        R.DL:=unit;
        R.DS:=Seg(FarADDRESS(dap));
        R.SI:=Ofs(FarADDRESS(dap));
        Lib.Intr(R,13H);
        (* IF NOT (SYSTEM.CarryFlag IN R.Flags) THEN *)
            RETURN (R.AH); (* 00H *)
        (* ELSE
            RETURN (R.AH);
        END; *)
    ELSE
        encodeCS(track,sector, bcylinder,bsector);
        R.DL := unit;
        R.AH := command;
        R.AL := BYTE(count);
        R.CH := bcylinder;
        R.DH := BYTE(head);
        R.CL := bsector;   (* contains cyl data too *)
        R.ES := bufsegment;   (* buffer segment *)
        R.BX := bufoffset;    (* buffer offset *)
        Lib.Intr (R,13H);
        RETURN (R.AH);
    END;
END rwCHS;

(*
Notes:  errors on a floppy may be due to the motor failing to spin up quickly
        enough; the read should be retried at least three times, resetting
        the disk with AH=00h between attempts
*)

PROCEDURE procSector (DEBUG:BOOLEAN;unit:BYTE;XBIOS:BOOLEAN;opcode:BYTE;
                     block:LONGCARD;track,head,sector,bufsegment,bufoffset:CARDINAL):BOOLEAN;
CONST
    maxretry = 3;
VAR
    retry : CARDINAL;
    rc    : BOOLEAN;
BEGIN
    rc := FALSE;
    IF isFloppy(unit) THEN
        retry := 0;
        LOOP
            IF rwCHS(unit,XBIOS,opcode,block,track,head,sector,oneSector,
                     bufsegment,bufoffset) = BYTE(00H) THEN rc:=TRUE; EXIT;END;
            INC(retry);
            IF retry = maxretry THEN EXIT;END;
            IF resetUnit(DEBUG,RESETHD,unit)=FALSE THEN EXIT;END;
        END;
    ELSE
(*%F ENABLEREALUPDATE *)
        opcode := opRead; (* don't take chances with hard disk ! *)
(*%E  *)
        IF rwCHS(unit,XBIOS,opcode,block,track,head,sector,oneSector,
                 bufsegment,bufoffset) = BYTE(00H) THEN rc:=TRUE;END;
    END;
    IF NOT(rc) THEN getFatalStatus(unit);END;
    RETURN rc;
END procSector;

(* ------------------------------------------------------------ *)

(* same as rwCHS and procSector but with experimental profiling *)

CONST
    TICKSOVERFLOW    = MAX(CARDINAL);
    (* legal ticks threshold range, however silly to allow these values *)
    MINTICKSTHRESHOLD     = 0;
    MAXTICKSTHRESHOLD     = MAX(CARDINAL);
    DEFAULTTICKSTHRESHOLD = 600; (* was 200 *)

PROCEDURE rwCHSalt (unit:BYTE;XBIOS:BOOLEAN;command:BYTE;block:LONGCARD;
                   track,head,sector,count,bufsegment,bufoffset:CARDINAL;
                   VAR ticks:CARDINAL):BYTE;
TYPE
    counterType = RECORD
        CASE : BOOLEAN OF
        | TRUE:
           counter : CARDINAL;
        | FALSE:
           lsb : SHORTCARD;
           msb : SHORTCARD;
        END;
    END;
CONST
    KBportB    = 61H;
    PITmode    = 43H;
    PITcounter = 42H;
VAR
    v:counterType;
    b:SHORTCARD;
VAR
    R : SYSTEM.Registers;
    bcylinder,bsector,LBAcmd:BYTE;
    dap:diskAddressPacketType;
BEGIN
    CASE command OF
    | opRead : ; (* let it be ! *)
    | opWrite: ; (* let it be ! *)
(*%F ENABLEREALUPDATE *)
        command:=opRead; (* don't take chances with hard disk ! *)
(*%E  *)
    ELSE
        RETURN BYTE(0FFH); (* do nothing and force a problem *)
    END;

    IF XBIOS THEN
        dap.packetsize:=10H;
        dap.reserved  :=00H;
        dap.count     :=count;       (* $7f at most ! *)
        dap.segBuf    :=bufsegment;
        dap.ofsBuf    :=bufoffset;
        dap.startblock.lo:=block; (* assume our qword is a longword *)
        dap.startblock.hi:=0;
        CASE command OF
        | opRead:
            LBAcmd:=42H;
        | opWrite:
            LBAcmd:=43H;
            R.AL  :=00H; (* no verify, and we don't care about v2.1 ! *)
        END;
        R.AH:=LBAcmd;
        R.DL:=unit;
        R.DS:=Seg(FarADDRESS(dap));
        R.SI:=Ofs(FarADDRESS(dap));

(*  *)

        (* disable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0FEH);        (* set gate bit off : %11111110 = $fe *)
        SYSTEM.Out(KBportB,b);  (* disable counter *)

        (* init counter *)

        v.counter := 0FFFFH;
        SYSTEM.Out(PITmode,0B0H);    (* tell we'll send timer 2 lsb then msb : %10110000 = $b0 *)
        SYSTEM.Out(PITcounter,v.lsb);
        SYSTEM.Out(PITcounter,v.msb);

        (* enable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b OR 1);            (* set gate bit on : %00000001 = $01 *)
        SYSTEM.Out(KBportB,b);  (* activate counter *)

        (* perform op *)

        Lib.Intr(R,13H);

        (* disable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0FEH);        (* set gate bit off : %11111110 = $fe *)
        SYSTEM.Out(KBportB,b);  (* disable counter *)

        (* read result *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0DFH); (* test %00100000 = $20 *)
        IF b = 0 THEN
            ticks:=TICKSOVERFLOW; (* was 0 to show overflow *)
        ELSE
            SYSTEM.Out(PITmode, 080H); (* read counter 2 : %10000000 = $80 *)
            v.lsb:=SYSTEM.In(PITcounter);
            v.msb:=SYSTEM.In(PITcounter);
            v.counter:=0FFFFH - v.counter;
            ticks:=v.counter; (* microseconds require *838 *)
        END;

(*  *)

        (* IF NOT (SYSTEM.CarryFlag IN R.Flags) THEN *)
            RETURN (R.AH); (* 00H *)
        (* ELSE
            RETURN (R.AH);
        END; *)
    ELSE
        encodeCS(track,sector, bcylinder,bsector);
        R.DL := unit;
        R.AH := command;
        R.AL := BYTE(count);
        R.CH := bcylinder;
        R.DH := BYTE(head);
        R.CL := bsector;   (* contains cyl data too *)
        R.ES := bufsegment;   (* buffer segment *)
        R.BX := bufoffset;    (* buffer offset *)

(*  *)

        (* disable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0FEH);        (* set gate bit off : %11111110 = $fe *)
        SYSTEM.Out(KBportB,b);  (* disable counter *)

        (* init counter *)

        v.counter := 0FFFFH;
        SYSTEM.Out(PITmode,0B0H);    (* tell we'll send timer 2 lsb then msb : %10110000 = $b0 *)
        SYSTEM.Out(PITcounter,v.lsb);
        SYSTEM.Out(PITcounter,v.msb);

        (* enable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b OR 1);            (* set gate bit on : %00000001 = $01 *)
        SYSTEM.Out(KBportB,b);  (* activate counter *)

        (* perform op *)

        Lib.Intr (R,13H);

        (* disable counter *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0FEH);        (* set gate bit off : %11111110 = $fe *)
        SYSTEM.Out(KBportB,b);  (* disable counter *)

        (* read result *)

        b:=SYSTEM.In(KBportB);
        b:=(b AND 0DFH); (* test %00100000 = $20 *)
        IF b = 0 THEN
            ticks:=TICKSOVERFLOW; (* was 0 to show overflow *)
        ELSE
            SYSTEM.Out(PITmode, 080H); (* read counter 2 : %10000000 = $80 *)
            v.lsb:=SYSTEM.In(PITcounter);
            v.msb:=SYSTEM.In(PITcounter);
            v.counter:=0FFFFH - v.counter;
            ticks:=v.counter; (* microseconds require *838 *)
        END;

(*  *)

        RETURN (R.AH);
    END;
END rwCHSalt;

PROCEDURE procSectorALT (DEBUG:BOOLEAN;unit:BYTE;XBIOS:BOOLEAN;opcode:BYTE;
                        block:LONGCARD;track,head,sector,bufsegment,bufoffset:CARDINAL;
                        VAR ticks:CARDINAL):BOOLEAN;
CONST
    maxretry = 3;
VAR
    retry : CARDINAL;
    rc    : BOOLEAN;
BEGIN
    rc := FALSE;
    IF isFloppy(unit) THEN
        retry := 0;
        LOOP
            IF rwCHS(unit,XBIOS,opcode,block,track,head,sector,oneSector,
                     bufsegment,bufoffset) = BYTE(00H) THEN rc:=TRUE; EXIT;END;
            INC(retry);
            IF retry = maxretry THEN EXIT;END;
            IF resetUnit(DEBUG,RESETHD,unit)=FALSE THEN EXIT;END;
        END;
    ELSE
(*%F ENABLEREALUPDATE *)
        opcode := opRead; (* don't take chances with hard disk ! *)
(*%E  *)
        IF rwCHSalt (unit,XBIOS,opcode,block,track,head,sector,oneSector,
                    bufsegment,bufoffset, ticks) = BYTE(00H) THEN rc:=TRUE;END;
    END;
    IF NOT(rc) THEN getFatalStatus(unit);END;
    RETURN rc;
END procSectorALT;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE beautifiedlc (v : LONGCARD;pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.CardToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END beautifiedlc;

PROCEDURE fmtlc (v:LONGCARD;base:CARDINAL;wi:INTEGER;ch,prefix:CHAR) : str80;
VAR
    R : str80;
    ok: BOOLEAN;
    i : CARDINAL;
BEGIN
    Str.CardToStr(v,R,base,ok);
    FOR i:= Str.Length(R)+1 TO ABS(wi) DO
        IF wi < 0 THEN
            Str.Append(R,ch);
        ELSE
            Str.Prepend(R,ch);
        END;
    END;
    IF base=16 THEN Str.Lows(R);END;
    Str.Prepend(R,prefix);
    RETURN R;
END fmtlc;

PROCEDURE fmt (v:CARDINAL;base:CARDINAL;wi:INTEGER;ch,prefix:CHAR) : str80;
BEGIN
    RETURN fmtlc(LONGCARD(v),base,wi,ch,prefix);
END fmt;

PROCEDURE fmtrangelc (lower,upper:LONGCARD):str80;
VAR
    R:str80;
BEGIN
    Str.Concat(R,"[",fmtlc(lower,10,0,"",""));
    Str.Append(R,"..");
    Str.Append(R,fmtlc(upper,10,0,"",""));
    Str.Append(R,"]");
    RETURN R;
END fmtrangelc;

PROCEDURE fmtrange (lower,upper:CARDINAL):str80;
BEGIN
    RETURN fmtrangelc( LONGCARD(lower),LONGCARD(upper) );
END fmtrange;

PROCEDURE fmtDateTimeUS ():str80;
TYPE
    tDays   = ARRAY [1..7] OF str16;
    tMonths = ARRAY [1..12] OF str16;
CONST
    tJours = tDays("Sunday","Monday","Tuesday","Wednesday",
                   "Thursday","Friday","Saturday"
                  );
    tMois = tMonths("January","February","March","April",
                    "May","June","July","August",
                    "September","October","November","December"
                   );
VAR
    Year,Month,Day : CARDINAL;
    DayOfWeek      : Lib.DayType;
    H,M,S,s        : CARDINAL;
    R              : str80;
BEGIN
    (* yes, we know it's not very american a date/time format... so what ? *)
    Lib.GetDate (Year,Month,Day,DayOfWeek);
    Lib.GetTime (H,M,S,s);

    Str.Copy(R,tJours[ORD(DayOfWeek)+1]);
    Str.Append(R,", ");

    Str.Append(R,fmt(Day,10,0,"",""));  Str.Append(R," ");
    Str.Append(R,tMois[Month]);         Str.Append(R," ");
    Str.Append(R,fmt(Year,10,0,"","")); Str.Append(R," at ");
    Str.Append(R,fmt(H,10,0,"",""));    Str.Append(R,"h ");
    Str.Append(R,fmt(M,10,0,"",""));    Str.Append(R,"mn ");
    Str.Append(R,fmt(S,10,0,"",""));    Str.Append(R,"s");
    RETURN R;
END fmtDateTimeUS;


(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE fmtTHS (ALTFORMAT:BOOLEAN; t,h,s:CARDINAL;mul:CHAR):str80;
VAR
    S:str80;
BEGIN
    IF NOT(ALTFORMAT) THEN
        Str.Concat(S,beautifiedlc( (LONGCARD(t)*LONGCARD(h)*LONGCARD(s))," ",sepdot,wiblock), " = ");
        Str.Append(S,fmt(t,10,0,"",""));Str.Append(S,mul);
        Str.Append(S,fmt(h,10,0,"",""));Str.Append(S,mul);
        Str.Append(S,fmt(s,10,0,"",""));
    ELSE
        Str.Concat(S,fmt(t,10,0,"",""),mul);
        Str.Append(S,fmt(h,10,0,"",""));
        Str.Append(S,mul);
        Str.Append(S,fmt(s,10,0,"",""));
        Str.Append(S," = ");
        Str.Append(S,fmtlc( (LONGCARD(t)*LONGCARD(h)*LONGCARD(s)),10,0,"",""));
    END;
    RETURN S;
END fmtTHS;

TYPE
    fixeddiskparametersType = RECORD
        cylcount:WORD;
        headcount:BYTE;
        pad1:WORD; (* starting reduced write current cylinder (XT only, 0 for others) *)
        pad2:WORD; (* starting write precompensation cylinder number *)
        pad3:BYTE; (* maximum ECC burst length (XT only) *)
        pad4:BYTE; (* control BYTE *)
        pad5:BYTE; (* standard timeout (XT only, 0 for others) *)
        pad6:BYTE; (* formatting timeout (XT and WD1002 only, 0 for others) *)
        pad7:BYTE; (* timeout for checking drive (XT and WD1002 only, 0 for others) *)
        pad8:WORD; (* cylinder number of landing zone (AT and later only) *)
        sectorcount:BYTE;
        rsvd:BYTE;
    END;

(* see INT 41 - SYSTEM DATA - HARD DISK 0 PARAMETER TABLE ADDRESS [NOT A VECTOR!] *)

(*
    BIOSes which support four hard drives may store the parameter tables
        for drives 81h-83h immediately following the parameter table pointed
        at by INT 41, with a separate copy of the drive 81h table for INT 46.
        The check for such an arrangement is to test whether INT 46 points
        somewhere other than exactly 16 bytes past INT 41, and the sixteen
        bytes starting at offset 10h from INT 41 are identical to the sixteen
        bytes pointed at by INT 46
    another arrangement for BIOSes which support four IDE drives is to have
        four tables pointed at by INT 41 in the order primary master,
        primary slave, secondary master, and secondary slave, in which case
        (for example) a system with only primary master and secondary master
        will have valid tables at offsets 00h and 20h, with garbage (but
        sectors-per-track = 00h) at offsets 10h and 30h
*)

PROCEDURE fmtBIOSgeometry (ALTFORMAT:BOOLEAN; unit:BYTE):str80;
CONST
    sBadGeom = "?x?x?";
VAR
    S:str80;
    a : FarADDRESS;
    table : fixeddiskparametersType;
    R:SYSTEM.Registers;
    c,h,s:CARDINAL;
    v:BYTE;
BEGIN
    CASE unit OF
    | 80H : v:=41H;
    | 81H : v:=46H;
    | 82H : v:=41H;
    | 83H : v:=41H;
    ELSE
        RETURN sBadGeom; (* cannot happen : already trapped ! *)
    END;
    R.AH := 35H;
    R.AL := v;
    Lib.Dos(R);
    a := [R.ES:R.BX];
    CASE unit OF        (* assumption ! *)
    | 82H: Lib.IncAddr(a,SIZE(table));   (* 10H *)
    | 83H: Lib.IncAddr(a,SIZE(table)*2); (* 20H *)
    END;
    Lib.FarMove ( a, FarADR(table), SIZE(table));
    CASE unit OF
    | 82H: IF CARDINAL(table.sectorcount) = 00H THEN RETURN sBadGeom;END;
    | 83H: IF CARDINAL(table.sectorcount) = 00H THEN RETURN sBadGeom;END;
    END;
    c := CARDINAL(table.cylcount);
    h := CARDINAL(table.headcount);
    s := CARDINAL(table.sectorcount);
    RETURN fmtTHS(ALTFORMAT,c,h,s,"x");
END fmtBIOSgeometry;

PROCEDURE show (wi:CARDINAL;S1,S2:ARRAY OF CHAR);
VAR
    i:CARDINAL;
BEGIN
    WrStr(S1);
    FOR i:=Str.Length(S1)+1 TO wi DO WrStr(" ");END;
    WrStr(" : ");WrStr(S2);WrLn;
END show;

PROCEDURE doCmdParms (DEBUG:BOOLEAN;unit:BYTE;
                     XBIOS,XBIOSavailable,
                     NEWGEOMETRY,PERFECTGEOMETRY,ALTFORMAT:BOOLEAN;
                     EDDmajor,EDDflag:BYTE; BESTFIT:CARDINAL;
                     maxtrackorg,maxheadorg,maxsectororg:CARDINAL;
                     totalSectors:LONGCARD;
                     maxtrack,maxhead,maxsector:CARDINAL;
                     maxblock:LONGCARD;
                     trackcount,headcount,sectorcount:CARDINAL;
                     blockcount:LONGCARD);
CONST
    wi            = 12;
    strMode       = "Access mode";
    strUnit       = "Unit";
    strTHS        = "TxHxS";
    strTotal      = "Total blocks"; (* 12 *)
    strTrack      = "Track";
    strHead       = "Head";
    strSector     = "Sector";
    strBlock      = "Block";
    strBIOSths    = "BIOS TxHxS";
    strBIOStable  = "BIOS table";

    warnFloppyFactor = nl+
"Note specifying -j[j] option with a floppy unit has no effect."+nl;

    warnUselessFactor= nl+
"Note specifying -j[j] option with this system has no effect."+nl;

    warnXBIOS = nl+
"Note specifying -x option with this system is UNSAFE and is NOT recommended :"+nl+
"you should rely on BIOS interrupt $13 extensions instead."+nl;

    warnFactor= nl+
"Note specifying -j[j] option with this system is NOT recommended :"+nl+
"you should let "+progEXEname+" try and rebuild better Track/Head/Sector values."+nl;

    warnXBIOSuselessFactor = warnXBIOS+warnUselessFactor;

VAR
    S,R : str128;
    total:LONGCARD;
BEGIN
IF DEBUG THEN
    WrStr("::: doCmdParms");WrLn;
    WrStr(":::   XBIOS available  ");IF XBIOSavailable THEN WrStr("Yes") ELSE WrStr("no");END;WrLn;
    WrStr(":::   XBIOS used here  ");IF XBIOS THEN WrStr("Yes") ELSE WrStr("no");END;WrLn;
    WrStr(":::   THS autorebuilt  ");IF NEWGEOMETRY THEN WrStr("Yes") ELSE WrStr("no");END;WrLn;
    WrStr(":::   perfect geometry ");IF PERFECTGEOMETRY THEN WrStr("Yes")ELSE WrStr("no");END;WrLn;
    WrStr(":::   -j[j] specified  ");CASE BESTFIT OF
                                     | FORCEBEST : WrStr("no");
                                     | FORCE255:   WrStr("Yes (-j = 255)");
                                     | FORCE240:   WrStr("Yes (-jj = 240)");
                                     END;
                                     WrLn;
    WrStr(":::   blockcount       ");IO.WrLngCard(blockcount,0);WrLn;
    (* redundant with PERFECTGEOMETRY *)
    total:=LONGCARD(maxtrack-mintrack+1)*LONGCARD(maxhead-minhead+1)*LONGCARD(maxsector-minsector+1);
    WrStr(":::   totalsectors     ");IO.WrLngCard(total,0);WrLn;
    WrStr(":::   mismatch         ");IF blockcount # total THEN WrStr("Yes") ELSE WrStr("no");END;WrLn;
END;

    Str.Copy(S,fmt( CARDINAL(unit),16,2,"0","$"));
    IF isFloppy(unit) THEN
        Str.Append(S," (floppy disk)");
    ELSE
        Str.Append(S," (hard disk)");
    END;
    show(wi,strUnit,S);

    IF XBIOS THEN
        S:="BIOS interrupt $13 extensions";
        CASE CARDINAL(EDDmajor) OF (* well, extensions in fact *)
        | 01H : R:="1.x";
        | 20H : R:="2.0 (EDD 1.0)";
        | 21H : R:="2.1 (EDD 1.1)";
        | 30H : R:="3.0 (EDD 3.0)";
        ELSE
                R:="";
        END;
        IF same(R,"")=FALSE THEN
            Str.Append(S," v");Str.Append(S,R);
            Str.Append(S," -- %");
            Str.Append(S,fmt( CARDINAL(EDDflag),2,4,"0","" ));
            (*
            0 extended disk access functions (AH=42h-44h,47h,48h) supported
            1 removable drive controller functions supported (AH=45h,46h,48h,49h,INT 15/AH=52h)
            2 enhanced disk drive (EDD) functions (AH=48h,AH=4Eh) supported
              extended drive parameter table is valid (see #00273,#00278)
            *)
        END;
    ELSE
        S:="BIOS interrupt $13";
    END;
    show(wi,strMode,S);

    WrLn;
    show(wi,strTrack, fmtrange  (mintrack,  maxtrack));
    show(wi,strHead,  fmtrange  (minhead,   maxhead));
    show(wi,strSector,fmtrange  (minsector, maxsector));
    show(wi,strBlock, fmtrangelc(LONGCARD(minblock),  maxblock));

    WrLn;
    IF NOT(ALTFORMAT) THEN
        show(wi,strTotal,beautifiedlc(blockcount," ",sepdot,wiblock));
    ELSE
        show(wi,strTotal,fmtlc(blockcount,10,0,"",""));
    END;

    show(wi,strTHS,fmtTHS(ALTFORMAT,trackcount,headcount,sectorcount,"x"));

    WrLn;
    show(wi,strBIOSths,fmtTHS (ALTFORMAT,
                              maxtrackorg-mintrack+1,
                              maxheadorg-minhead+1,
                              maxsectororg-minsector+1,"x"));
    IF isFloppy(unit)=FALSE THEN
        show (wi,strBIOStable, fmtBIOSgeometry(ALTFORMAT,unit));
    END;

    IF isFloppy(unit) THEN
        IF BESTFIT # FORCEBEST THEN WrStr(warnFloppyFactor); END;
    ELSE
        IF XBIOS THEN
            IF NEWGEOMETRY THEN
                IF PERFECTGEOMETRY=FALSE THEN WrStr(warnFactor); END;
            ELSE
                IF BESTFIT # FORCEBEST THEN WrStr(warnUselessFactor); END;
            END;
        ELSE
            IF XBIOSavailable THEN
                IF BESTFIT = FORCEBEST THEN
                    WrStr(warnXBIOS);
                ELSE
                    WrStr(warnXBIOSuselessFactor);
                END;
            END;
            (*
            older BIOSes would require more checks but who cares ? ;-)
            *)
        END;
    END;
END doCmdParms;

(*
unit $80                 /f         /x    /x /f

XBIOS used here  Yes     Yes        no    no
THS autorebuilt  no      no         no    no
perfect geometry Yes     Yes        Yes   Yes
                 GOOD    GOOD       BAD   BAD

XBIOS used here  Yes     Yes        no    no
THS autorebuilt  Yes     Yes        no    no
perfect geometry Yes     no         Yes   Yes
                 GOOD    BAD        BAD   BAD
*)

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    sFAKE         = " ;-)"; (* suffix *)
    sOK           = "+++ ";
    sINFO         = "::: ";
    sPROBLEM      = "--- ";
    sWARN         = "*** ";
    sDIFF         = "### ";
    sNADA         = "    ";
    sNOTICE       = "    ";  (* question to user *)
    sCANCELLED    = "--- ";  (* not a problem but show nothing was done *)
CONST
    sSrc     = sINFO+"Source : ";
    sDst     = sINFO+"Target : ";
    sSref    = sINFO+"Sector : ";
    sFref    = sINFO+"File   : ";

    msgWorking = sINFO+"Working, please wait...";

PROCEDURE fmtHere (unit:BYTE;track,head,sector:CARDINAL;block,countofsectors:LONGCARD):str128;
VAR
    R:str128;
BEGIN
    Str.Concat(R,"U=",fmt(CARDINAL(unit),16,2,"0","$"));
    Str.Append(R,", T=");Str.Append(R,fmt(track,10,0,"",""));
    Str.Append(R,", H=");Str.Append(R,fmt(head,10,0,"",""));
    Str.Append(R,", S=");Str.Append(R,fmt(sector,10,0,"",""));
    Str.Append(R,", block=");Str.Append(R,fmtlc(block,10,0,"",""));
    Str.Append(R," (");
    Str.Append(R,fmtlc(countofsectors,10,0,"",""));
    Str.Append(R," sector");
    IF countofsectors > 1 THEN Str.Append(R,"s");END;
    Str.Append(R,")");
    RETURN R;
END fmtHere;

CONST
    sTICKPREFIX = sOK; (* or sINFO, or another one ? *)

PROCEDURE fmtTicks (VAR S:ARRAY OF CHAR;ticks:CARDINAL;block:LONGCARD);
BEGIN
    Str.Concat(S,sTICKPREFIX,fmt(ticks,10,5," ",""));
    Str.Append(S," ticks were needed to read block ");
    Str.Append(S,fmtlc(block,10,0,"",""));
END fmtTicks;

PROCEDURE fmtTally (VAR S:ARRAY OF CHAR;
                   profileinterval,iprof:CARDINAL;tickcount:LONGCARD);
VAR
    begticks,endticks:LONGCARD;
BEGIN
    begticks:=LONGCARD(iprof)*LONGCARD(profileinterval);
    endticks:=begticks+LONGCARD(profileinterval)-1;
    IF endticks > MAX(CARDINAL) THEN endticks:=MAX(CARDINAL);END;
    Str.Concat(S,sTICKPREFIX+"Ticks [",fmtlc(begticks,10,5,"0",""));
    Str.Append(S,"..");
    Str.Append(S,fmtlc(endticks,10,5,"0",""));
    Str.Append(S,"] : ");
    Str.Append(S,fmtlc(tickcount,10,10," ",""));
    Str.Append(S," sector(s)");
END fmtTally;

(* ------------------------------------------------------------ *)

PROCEDURE doCmdRead (DEBUG,IGNOREFATAL,PROFILE:BOOLEAN;
                    unit:BYTE;XBIOS:BOOLEAN;
                    track,head,sector:CARDINAL;block,countofsectors:LONGCARD;
                    maxtrack,maxhead,maxsector:CARDINAL;
                    maxblock:LONGCARD;
                    trackcount,headcount,sectorcount:CARDINAL;
                    blockcount:LONGCARD;
                    F:ARRAY OF CHAR):BOOLEAN;
VAR
    S:str128;
    hnd,herr:FIO.File;
    counter:LONGCARD;
    ok,rc:BOOLEAN;
    ticks:CARDINAL;
    pb:LONGCARD;
BEGIN
    Str.Concat(S,sSrc,fmtHere(unit,track,head,sector,block,countofsectors));
    WrStr(S);WrLn;
    Str.Concat(S,sDst,F);
    WrStr(S);WrLn;

    (* handle log *)
    pb:=0;
    IF IGNOREFATAL THEN
        S:=FATALFILE;
        IF FIO.Exists(S) THEN
            setReadWrite(S);
            herr:=FIO.Append(S);
        ELSE
            herr:=FIO.Create(S);
        END;
        Str.Concat(S,nl+sINFO+"Log started on ",fmtDateTimeUS());
        Str.Append(S,nl);
        FIO.WrStr(herr,S);
        FIO.Close(herr);
    END;

    video(msgWorking,TRUE);

    hnd := FIO.Create(F);
    FIO.AssignBuffer(hnd,ioBuffer);
    FOR counter := 1 TO countofsectors DO
        blockToCHS (trackcount,headcount,sectorcount,
                   block,
                   track,head,sector);
        IF PROFILE THEN
            IF NOT( DYNALLOC ) THEN
            ok:=procSectorALT (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                              Seg(buffSector),Ofs(buffSector), ticks);
            ELSE
            ok:=procSectorALT (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                              Seg(pbuffSector^),Ofs(pbuffSector^), ticks);
            END;
            IF ok THEN
                fmtTicks(S, ticks,block);
                video(msgWorking,FALSE);
                WrStr(S);WrLn;
                video(msgWorking,TRUE);
            END;
        ELSE
            IF NOT ( DYNALLOC ) THEN
                ok:=procSector(DEBUG,unit,XBIOS,opRead,block,track,head,sector,Seg(buffSector),Ofs(buffSector));
            ELSE
                ok:=procSector(DEBUG,unit,XBIOS,opRead,block,track,head,sector,Seg(pbuffSector^),Ofs(pbuffSector^));
            END;
        END;
        IF ok = FALSE THEN
            IF IGNOREFATAL THEN
                INC(pb);
                S:=FATALFILE;
                herr:=FIO.Append(S);
                Str.Concat(S,sPROBLEM+"Error reading block ",fmtlc(block,10,0,"",""));
                Str.Append(S,nl);
                FIO.WrStr(herr,S);
                FIO.Close(herr);
                video(msgWorking,FALSE);
                WrStr(S);
                video(msgWorking,TRUE);
            ELSE
                FIO.Close(hnd);
                video(msgWorking,FALSE);
                RETURN FALSE;
            END;
        END;
        IF NOT ( DYNALLOC ) THEN
            FIO.WrBin (hnd,buffSector,sectorSize);
        ELSE
            FIO.WrBin (hnd,pbuffSector^,sectorSize);
        END;
        INC(block);
    END;
    FIO.Flush(hnd);
    FIO.Close(hnd);

    video(msgWorking,FALSE);

    IF IGNOREFATAL THEN
        rc:=(pb = 0);
    ELSE
        rc:=TRUE;
    END;
    IF rc THEN WrStr(sOK+"Data was successfully written from <source> to <file>.");WrLn; END;
    RETURN rc;
END doCmdRead;

(* ------------------------------------------------------------ *)

PROCEDURE getkbd (VAR c1,c2:CHAR);
BEGIN
    BiosFlushkey(); (* useless against DOSKEY ! *)
    BiosWaitkey(c1,c2);
END getkbd;

(* we can use video here because we know redirection cannot happen here *)

PROCEDURE ShallWeProceed (firsttime:BOOLEAN; src:ARRAY OF CHAR):BOOLEAN;
CONST
    acceptA = "<^F9>"; acceptA1 = CHR(0); acceptA2 = CHR(102);
    acceptB = "<^F1>"; acceptB1 = CHR(0); acceptB2 = CHR(94);
VAR
    S : str128;
    c1,c2 , expected1,expected2:CHAR;
BEGIN
    CASE firsttime OF
    | TRUE:
        IF same(src,"") THEN
            S:=sNOTICE+acceptA+" to perform requested operation : ";
        ELSE
            Str.Concat(S,sNOTICE+acceptA+" to update <target> with ",src);
            Str.Append(S," : " );
        END;
        expected1:=acceptA1;
        expected2:=acceptA2;
    | FALSE:
        IF same(src,"") THEN
            S:=sNOTICE+"Now, "+acceptB+" to _perform_ requested operation : ";
        ELSE
            Str.Concat(S,sNOTICE+"Now, "+acceptB+" to _update_ <target> with ",src);
            Str.Append(S," : " );
        END;
        expected1:=acceptB1;
        expected2:=acceptB2;
    END;
    video(S,TRUE);
    getkbd(c1,c2);
    video(S,FALSE);
    IF ((c1=expected1) AND (c2=expected2) ) THEN
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
END ShallWeProceed;

PROCEDURE queryUser (AUTOCONFIRM:BOOLEAN;msg,src:ARRAY OF CHAR):BOOLEAN;
VAR
    letsdoit:BOOLEAN;
BEGIN
    IF AUTOCONFIRM THEN
        letsdoit := TRUE;
    ELSE
        (* WrStr(msg);WrLn; *)
        letsdoit := FALSE;
        IF ShallWeProceed(TRUE, src) THEN
            IF ShallWeProceed(FALSE, src) THEN
                (* dmp(sINFO+"Update confirmed twice !"); *)
                letsdoit:=TRUE;
            ELSE
                WrStr(sCANCELLED+"Update cancelled"+nl); (* " at second thought" *)
            END;
        ELSE
            WrStr(sCANCELLED+"Update cancelled"+nl);
        END;
    END;
    RETURN letsdoit;
END queryUser;

PROCEDURE doCmdWrite (DEBUG,IGNOREFATAL,REALLY,AUTOCONFIRM:BOOLEAN;
                     unit:BYTE;XBIOS:BOOLEAN;
                     track,head,sector:CARDINAL;block,countofsectors:LONGCARD;
                     maxtrack,maxhead,maxsector:CARDINAL;
                     maxblock:LONGCARD;
                     trackcount,headcount,sectorcount:CARDINAL;
                     blockcount:LONGCARD;
                     F:ARRAY OF CHAR):BOOLEAN;
VAR
    S:str128;
    hnd,herr:FIO.File;
    counter:LONGCARD;
    got:CARDINAL;
    pb:LONGCARD;
    rc,result:BOOLEAN;
BEGIN
    rc:=TRUE;

    Str.Concat(S,sSrc,F);
    WrStr(S);WrLn;
    Str.Concat(S,sDst,fmtHere(unit,track,head,sector,block,countofsectors));
    WrStr(S);WrLn;
    IF queryUser (AUTOCONFIRM,"",F) THEN

        (* handle log *)
        pb:=0;
        IF IGNOREFATAL THEN
            S:=FATALFILE;
            IF FIO.Exists(S) THEN
                setReadWrite(S);
                herr:=FIO.Append(S);
            ELSE
                herr:=FIO.Create(S);
            END;
            Str.Concat(S,nl+sINFO+"Log started on ",fmtDateTimeUS());
            Str.Append(S,nl);
            FIO.WrStr(herr,S);
            FIO.Close(herr);
        END;

        video(msgWorking,TRUE);

        hnd := FIO.OpenRead(F);
        FIO.AssignBuffer(hnd,ioBuffer);
        FOR counter := 1 TO countofsectors DO
            blockToCHS (trackcount,headcount,sectorcount,
                       block,
                       track,head,sector);
            IF NOT ( DYNALLOC ) THEN
                got:=FIO.RdBin (hnd,buffSector,sectorSize);
            ELSE
                got:=FIO.RdBin (hnd,pbuffSector^,sectorSize);
            END;

IF REALLY THEN
            IF NOT ( DYNALLOC ) THEN
                result:=procSector(DEBUG,unit,XBIOS,opWrite,block,track,head,sector,Seg(buffSector),Ofs(buffSector));
            ELSE
                result:=procSector(DEBUG,unit,XBIOS,opWrite,block,track,head,sector,Seg(pbuffSector^),Ofs(pbuffSector^));
            END;
            IF result=FALSE THEN
                IF IGNOREFATAL THEN
                    INC(pb);
                    S:=FATALFILE;
                    herr:=FIO.Append(S);
                    Str.Concat(S,sPROBLEM+"Error writing block ",fmtlc(block,10,0,"",""));
                    Str.Append(S,nl);
                    FIO.WrStr(herr,S);
                    FIO.Close(herr);
                    video(msgWorking,FALSE);
                    WrStr(S);
                    video(msgWorking,TRUE);
                ELSE
                    FIO.Close(hnd);
                    video(msgWorking,FALSE);
                    RETURN FALSE;
                END;
            END;
END;
            INC(block);
        END;
        FIO.Close(hnd);

        video(msgWorking,FALSE);

        IF IGNOREFATAL THEN
            rc:=(pb = 0);
        ELSE
            rc:=TRUE;
        END;
        IF rc THEN
            WrStr(sOK+"Data was successfully written from <source> to <target>.");
            IF REALLY=FALSE THEN WrStr(sFAKE);END;
            WrLn;
        END;
    END;
    RETURN rc;
END doCmdWrite;

(* ------------------------------------------------------------ *)

PROCEDURE createUndo (unit:BYTE;XBIOS:BOOLEAN;BESTFIT:CARDINAL;
                     track,head,sector:CARDINAL;block:LONGCARD;
                     BAKFILE,BATCH : ARRAY OF CHAR);
CONST
    placeholder = "~";
    sUndo = "@ECHO OFF"+nl+
            "ECHO."+nl+
            "ECHO Data saved "+placeholder+nl+
            "ECHO."+nl+
            "ECHO "+progEXEname+" "+placeholder+nl+
            "ECHO "+progEXEname+" "+placeholder+nl+
            "ECHO."+nl;
VAR
    hnd:FIO.File;
    S:str1024; (* oversized *)
    R:str128;
BEGIN
    Str.Copy(S,sUndo);

    (* here and now *)
    Str.Copy(R,fmtDateTimeUS() );
    Str.Subst(S,placeholder,R);

    (* block *)

    Str.Copy(R,fmt( CARDINAL(unit),16,2,"0","$"));
    Str.Append(R," WRITE ");
    Str.Append(R,fmtlc( block,10,0,"","") );  Str.Append(R," ");
    Str.Append(R,BAKFILE);
    IF isFloppy(unit)=FALSE THEN
        IF NOT(XBIOS)   THEN Str.Append(R," -x"); END;
        CASE BESTFIT OF
        | FORCE255 : Str.Append(R," -j");
        | FORCE240 : Str.Append(R," -jj");
        END;
    END;
    Str.Subst(S,placeholder,R);

    (* cylinder head sector *)

    Str.Copy(R,fmt( CARDINAL(unit),16,2,"0","$"));
    Str.Append(R," WRITE ");
    Str.Append(R,fmt( track,10,0,"","") );    Str.Append(R," ");
    Str.Append(R,fmt( head,10,0,"","") );     Str.Append(R," ");
    Str.Append(R,fmt( sector,10,0,"","") );   Str.Append(R," ");
    Str.Append(R,BAKFILE);
    IF isFloppy(unit)=FALSE THEN
        IF NOT(XBIOS)   THEN Str.Append(R," -x"); END;
        CASE BESTFIT OF
        | FORCE255 : Str.Append(R," -j");
        | FORCE240 : Str.Append(R," -jj");
        END;
    END;
    Str.Subst(S,placeholder,R);

    hnd := FIO.Create(BATCH);
    FIO.AssignBuffer(hnd,ioBuffer);
    FIO.WrStr(hnd,S);
    FIO.Flush(hnd);
    FIO.Close(hnd);

    Str.Concat(R,sOK,BATCH);
    Str.Append(R," was successfully created : it WILL require editing.");
    WrStr(R);WrLn;
END createUndo;

(* ------------------------------------------------------------ *)

PROCEDURE fmtchar (c : BYTE) : str16;
VAR
    S : str16;
BEGIN
    CASE ORD(c) OF
    | 0..ORD(blank)-1 : S:='"."';
    | ORD(dquote) :     S:="'"+dquote+"'";
    | 255 :             S:='"."';
    ELSE
                        Str.Concat(S,dquote,CHR(c));Str.Append(S,dquote);
    END;
    RETURN S;
END fmtchar;

(* "######## - $#### : ## --> ##   ??? --> ???" *)
PROCEDURE showDifference (counter:LONGCARD; i : CARDINAL;ref,now : BYTE);
VAR
    S:str128;
BEGIN
    Str.Concat(S,fmtlc(counter,10,8," ","")," - ");
    Str.Append(S,fmt(i,16,4,"0","$"));
    Str.Append(S," : ");
    Str.Append(S,fmt(CARDINAL(ref),16,2,"0",""));
    Str.Append(S," --> ");
    Str.Append(S,fmt(CARDINAL(now),16,2,"0",""));
    Str.Append(S,"   ");
    Str.Append(S,fmtchar(ref));
    Str.Append(S," --> ");
    Str.Append(S,fmtchar(now));
    WrStr(sDIFF);WrStr(S);WrLn;
END showDifference;

PROCEDURE doCmdCompare (VAR mismatches:LONGCARD;
                       DEBUG,IGNOREFATAL,VERBOSE:BOOLEAN;
                       unit:BYTE;XBIOS:BOOLEAN;
                       track,head,sector:CARDINAL;block,countofsectors:LONGCARD;
                       maxtrack,maxhead,maxsector:CARDINAL;
                       maxblock:LONGCARD;
                       trackcount,headcount,sectorcount:CARDINAL;
                       blockcount:LONGCARD;
                       F:ARRAY OF CHAR):CARDINAL;
VAR
    S:str128;
    hnd,herr:FIO.File;
    counter:LONGCARD;
    i,got:CARDINAL;
    ref,now:BYTE;
    alcatraz,result:BOOLEAN;
    chkrounds:CARDINAL;
    pb:LONGCARD;
    rc:CARDINAL;
BEGIN
    Str.Concat(S,sSref,fmtHere(unit,track,head,sector,block,countofsectors));
    WrStr(S);WrLn;
    Str.Concat(S,sFref,F);
    WrStr(S);WrLn;

    (* handle log *)
    pb:=0;
    IF IGNOREFATAL THEN
        S:=FATALFILE;
        IF FIO.Exists(S) THEN
            setReadWrite(S);
            herr:=FIO.Append(S);
        ELSE
            herr:=FIO.Create(S);
        END;
        Str.Concat(S,nl+sINFO+"Log started on ",fmtDateTimeUS());
        Str.Append(S,nl);
        FIO.WrStr(herr,S);
        FIO.Close(herr);
    END;

    video(msgWorking,TRUE);

    alcatraz:=FALSE;
    chkrounds:=0;

    mismatches:=0;
    hnd := FIO.OpenRead(F);
    FIO.AssignBuffer(hnd,ioBuffer);
    FOR counter := 1 TO countofsectors DO

        IF NOT ( DYNALLOC ) THEN
            got:=FIO.RdBin (hnd,buffSectorDisk,sectorSize);
        ELSE
            got:=FIO.RdBin (hnd,pbuffSectorDisk^,sectorSize);
        END;

        blockToCHS (trackcount,headcount,sectorcount,
                   block,
                   track,head,sector);
        IF NOT ( DYNALLOC ) THEN
        result:=procSector (DEBUG,unit,XBIOS,opRead,block,track,head,sector,
                       Seg(buffSector),Ofs(buffSector));
        ELSE
        result:=procSector (DEBUG,unit,XBIOS,opRead,block,track,head,sector,
                       Seg(pbuffSector^),Ofs(pbuffSector^));
        END;
        IF result = FALSE THEN
            IF IGNOREFATAL THEN
                INC(pb);
                S:=FATALFILE;
                herr:=FIO.Append(S);
                Str.Concat(S,sPROBLEM+"Error reading block ",fmtlc(block,10,0,"",""));
                Str.Append(S,nl);
                FIO.WrStr(herr,S);
                FIO.Close(herr);
                video(msgWorking,FALSE);
                WrStr(S);
                video(msgWorking,TRUE);
            ELSE
                FIO.Close(hnd);
                video(msgWorking,FALSE);
                RETURN errInt13h;
            END;
        END;

        FOR i := firstByteInSector TO lastByteInSector DO
            IF NOT ( DYNALLOC ) THEN
                ref:=buffSectorDisk[i];
                now:=buffSector[i];
            ELSE
                ref:=pbuffSectorDisk^[i];
                now:=pbuffSector^[i];
            END;

            IF now # ref THEN
                IF VERBOSE THEN
                    video(msgWorking,FALSE);
                    IF mismatches = 0 THEN
                           (* "######## - $#### : ## --> ##   ??? --> ???" *)
                        WrStr(sDIFF+"  sector  offset disk   file  disk    file");WrLn;
                        WrStr(sDIFF+"==========================================");WrLn;
                    END;
                    showDifference(counter,i,ref,now);

                    video(msgWorking,TRUE);
                END;
                INC(mismatches);
            END;
        END;

        INC(block);

        INC(chkrounds);
        IF (chkrounds MOD CHKEVERY) = 0 THEN alcatraz:=ChkEscape(); END;
        IF alcatraz THEN
            FIO.Close(hnd);
            video(msgWorking,FALSE);
            RETURN errAborted;
        END;
    END;
    FIO.Close(hnd);

    video(msgWorking,FALSE);

    IF IGNOREFATAL THEN
        IF pb=0 THEN
            rc:=errNone;
        ELSE
            rc:=errInt13h;
        END;
    ELSE
        rc:=errNone;
    END;

    (* IF (VERBOSE AND (mismatches # 0) ) THEN WrLn; END; *)
    RETURN rc;
END doCmdCompare;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    pBlock = POINTER TO slowBlockType;
    slowBlockType = RECORD
        next  : pBlock;
        ticks : CARDINAL;
        block : LONGCARD;
    END;

PROCEDURE initList (VAR anchor : pBlock );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pBlock);
CONST
    needed = SIZE(slowBlockType);
VAR
    p      : pBlock;
BEGIN
    WHILE anchor # NIL DO
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pBlock):BOOLEAN;
CONST
    needed = SIZE(slowBlockType);
BEGIN
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

PROCEDURE dmpWantedList (anchor:pBlock;
                        iprofwanted,profileinterval:CARDINAL;hnd:FIO.File);
CONST
    (*    "ticks [" *)
    MSG = "      Reading block % required % tick(s) !"+nl;
VAR
    p : pBlock;
    S : str128;
BEGIN
    p:=anchor;
    WHILE anchor # NIL DO
        IF p^.ticks DIV profileinterval = iprofwanted THEN
            Str.Concat(S,sINFO,MSG);
            Str.Subst(S,"%",fmtlc(p^.block,10,10,"",""));
            Str.Subst(S,"%",fmt(p^.ticks,10,5,"",""));
            FIO.WrStr(hnd,S);
            WrStr(S);
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
END dmpWantedList;

PROCEDURE dmpList (anchor:pBlock; hnd:FIO.File);
CONST
    MSG = "% tick(s) required to read block % (probably weak) !"+nl;
VAR
    p : pBlock;
    S : str128;
BEGIN
    p:=anchor;
    WHILE anchor # NIL DO
        Str.Concat(S,sINFO,MSG);
        Str.Subst(S,"%",fmt(p^.ticks,10,5," ",""));
        Str.Subst(S,"%",fmtlc(p^.block,10,10," ",""));
        FIO.WrStr(hnd,S);
        WrStr(S);
        p:=anchor^.next;
        anchor:=p;
    END;
END dmpList;

(* ------------------------------------------------------------ *)

PROCEDURE doCmdVerify (unit:BYTE; XBIOS, FORCERESTART, SCANMAGIC, PROFILE:BOOLEAN;
                      trackcount,headcount,sectorcount:CARDINAL;
                      blockcount, ticksThreshold:LONGCARD):CARDINAL;
CONST
    posmagic1 = 01FEH;    magic1 = BYTE(055H);
    posmagic2 = 01FFH;    magic2 = BYTE(0AAH);
CONST
    profileinterval = 100; (* seems a good value *)
    firstprofile    = 0;
    maxprofile      = (MAX(CARDINAL) DIV profileinterval);
TYPE
    sampleArrayType = ARRAY[firstprofile..maxprofile] OF LONGCARD;
    resumeType = RECORD;
        nextblock : LONGCARD;
        problems  : LONGCARD;
        xtdBIOS   : BOOLEAN;
        tracks    : CARDINAL;
        heads     : CARDINAL;
        sectors   : CARDINAL;
        blocks    : LONGCARD;
        profiling : BOOLEAN;
        tickcount : sampleArrayType;
    END;
CONST
    HDupdateFreq  = 4096;
    FDupdateFreq  = 16;   (* eh, we're already verifying a slow floppy disk ! *)
    placeholder = "~";
VAR
    block,maxblock,v,fsize,pb,pbmem:LONGCARD;
    updateFreq:LONGCARD;
    got:CARDINAL;
    hnd:FIO.File;
    track,head,sector:CARDINAL;
    orgmsg,msg,N,S:str128;
    ok,alcatraz,complete:BOOLEAN;
    resume : resumeType;
    sysblock:BOOLEAN;
    threshold,iprof,ticks,chkrounds:CARDINAL;
    anchor,p:pBlock;
BEGIN
    IF isFloppy (unit) THEN
        updateFreq := FDupdateFreq;
    ELSE
        updateFreq := HDupdateFreq;
    END;

    initList(anchor);
    threshold:=CARDINAL(ticksThreshold) DIV profileinterval;

    maxblock := blockcount-1+minblock;

(* //TEST  maxblock:=500000-1; *)

    (* check if we're resuming operation *)
    S:=RESTARTFROM;
    IF ( FIO.Exists(S) AND (FORCERESTART=FALSE) ) THEN
        setReadWrite(S);
        hnd:=FIO.OpenRead(S);
        fsize:=FIO.Size(hnd);
        got:=FIO.RdBin(hnd,resume,SIZE(resume));
        FIO.Close(hnd);
        IF fsize # SIZE(resume)         THEN RETURN errCorrupted; END;
        IF got # SIZE(resume)           THEN RETURN errCorrupted; END;

        IF resume.xtdBIOS # XBIOS       THEN RETURN errExtAPI; END;
        IF resume.tracks # trackcount   THEN RETURN errFactor; END;
        IF resume.heads # headcount     THEN RETURN errFactor; END;
        IF resume.sectors # sectorcount THEN RETURN errFactor; END;
        IF resume.blocks # blockcount   THEN RETURN errFactor; END;

        IF resume.profiling THEN
            PROFILE:=resume.profiling;
        ELSE
            IF PROFILE                  THEN RETURN errResumeProfiling; END;
        END;

        block:=resume.nextblock;
        pb   :=resume.problems;
        IF block > maxblock   THEN RETURN errBlockRange; END;
        IF block < minblock   THEN RETURN errBlockRange; END;
    ELSE
        block:=minblock;
        pb:=0;
        FOR iprof:=firstprofile TO maxprofile DO
            resume.tickcount[iprof]:=0;
        END;
    END;

    (* handle log *)
    S:=REPORTFILE;
    IF FIO.Exists(S) THEN
        setReadWrite(S);
        hnd:=FIO.Append(S);
    ELSE
        hnd:=FIO.Create(S);
    END;

    Str.Concat(S,nl+sINFO+"Log started on ",fmtDateTimeUS());
    Str.Append(S,nl);
    FIO.WrStr(hnd,S);
    IF block=0 THEN
        S:=sINFO+"Starting ";
    ELSE
        S:=sINFO+"Resuming ";
    END;
    IF PROFILE THEN
        Str.Append(S,"verify/profile (threshold=");
        Str.Append(S,fmtlc(ticksThreshold,10,10,"",""));
        Str.Append(S,")");
    ELSE
        Str.Append(S,"verify");
    END;
    Str.Append(S," operation from block ");
    Str.Append(S,fmtlc(block,10,0,"",""));
    Str.Append(S,nl);
    FIO.WrStr(hnd,S);
    FIO.Close(hnd);
    WrStr(S);

    pbmem   :=0;

    complete:=FALSE;
    alcatraz:=FALSE;
    chkrounds:=0;

    Str.Copy(orgmsg,"Reading block "+placeholder+" out of ");
    Str.Append(orgmsg,beautifiedlc(maxblock +1 ," ",sepdot,wiblock)); (* +1 because we show totalblocks *)
    Str.Copy(N,beautifiedlc(block," ",sepdot,wiblock));
    Str.Copy(msg,orgmsg);
    Str.Subst(msg,placeholder,N);

    video(msg,TRUE);
    LOOP
        blockToCHS (trackcount,headcount,sectorcount,
                   block,
                   track,head,sector);
        IF PROFILE THEN
            IF NOT ( DYNALLOC ) THEN
            ok:=procSectorALT (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                              Seg(buffSector),Ofs(buffSector), ticks);
            ELSE
            ok:=procSectorALT (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                              Seg(pbuffSector^),Ofs(pbuffSector^), ticks);
            END;
        ELSE
            IF NOT ( DYNALLOC ) THEN
            ok:=procSector (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                           Seg(buffSector),Ofs(buffSector));
            ELSE
            ok:=procSector (FALSE,unit,XBIOS,opRead,block,track,head,sector,
                           Seg(pbuffSector^),Ofs(pbuffSector^));
            END;
        END;

(* //TEST IF Lib.RAND() < 0.001 THEN ok:=FALSE; END; *)

        IF ok THEN
            IF SCANMAGIC THEN
                IF NOT ( DYNALLOC ) THEN
                    sysblock :=              (buffSector[posmagic1]=magic1);
                    sysblock :=(sysblock AND (buffSector[posmagic2]=magic2) );
                ELSE
                    sysblock :=              (pbuffSector^[posmagic1]=magic1);
                    sysblock :=(sysblock AND (pbuffSector^[posmagic2]=magic2) );
                END;
                IF sysblock THEN
                    S:=REPORTFILE;
                    hnd:=FIO.Append(S);
                    Str.Concat(S,sOK+"$aa55 magic found in block ",fmtlc(block,10,0,"",""));
                    Str.Append(S,nl);
                    FIO.WrStr(hnd,S);
                    FIO.Close(hnd);
                    video(msg,FALSE);
                    WrStr(S);
                    video(msg,TRUE);
                END;
            END;

            IF PROFILE THEN
                iprof := ticks DIV profileinterval;
                INC (resume.tickcount[iprof]);
                IF iprof >= threshold THEN
                    IF buildNewPtr(anchor,p) THEN
                        p^.ticks:=ticks;
                        p^.block:=block;
                    ELSE
                        INC(pbmem);
                    END;
                END;
                    (*
                    S:=REPORTFILE;
                    hnd:=FIO.Append(S);
                    fmtTicks(S, ticks,block);
                    Str.Append(S,nl);
                    FIO.WrStr(hnd,S);
                    FIO.Close(hnd);
                    video(msg,FALSE);
                    WrStr(S);
                    video(msg,TRUE);
                    *)
            END;
        ELSE
            INC(pb);
            S:=REPORTFILE;
            hnd:=FIO.Append(S);
            Str.Concat(S,sPROBLEM+"Error reading block ",fmtlc(block,10,0,"",""));
            Str.Append(S,nl);
            FIO.WrStr(hnd,S);
            FIO.Close(hnd);
            video(msg,FALSE);
            WrStr(S);
            video(msg,TRUE);
        END;
        INC(block);
        IF block > maxblock THEN complete:=TRUE; EXIT; END;

        INC(chkrounds);
        IF (chkrounds MOD CHKEVERY) = 0 THEN alcatraz:=ChkEscape(); END;
        IF alcatraz THEN EXIT; END;

        IF (block MOD updateFreq) = 0 THEN
            video(msg,FALSE);
            Str.Copy(N,beautifiedlc(block," ",sepdot,wiblock));
            Str.Copy(msg,orgmsg);
            Str.Subst(msg,placeholder,N);
            video(msg,TRUE);
        END;
    END;
    video(msg,FALSE);

    (* let's duplicate code uselessly according to complete or alcatraz ! *)

    IF complete THEN
        (* no resuming here *)
        S:=RESTARTFROM;
        IF FIO.Exists(S) THEN
            setReadWrite(S);
            FIO.Erase(S);
        END;

        S:=REPORTFILE;
        hnd:=FIO.Append(S);
        IF pbmem # 0 THEN
            Str.Concat(S,sINFO,fmtlc(pbmem,10,10,"",""));
            Str.Append(S," slow sector");
            IF pbmem=1 THEN
                Str.Append(S," was");
            ELSE
                Str.Append(S,"s were");
            END;
            Str.Append(S," not reported (Storage.ALLOCATE() failure)."+nl);
            FIO.WrStr(hnd,S);
            WrStr(S);
        END;
        IF pb = 0 THEN
            Str.Copy(S,sOK+"Verify operation complete, no error found !"+nl);
        ELSE
            Str.Concat(S,sINFO+"Verify operation complete, ",fmtlc(pb,10,0,"","") );
            Str.Append(S," error(s) found !"+nl); (* can't we make it singular/plural, eh ? *)
        END;
        FIO.WrStr(hnd,S);
        FIO.Close(hnd);
        WrStr(S);

        IF PROFILE THEN
            S:=REPORTFILE;
            hnd:=FIO.Append(S);
            FOR iprof := firstprofile TO maxprofile DO
                fmtTally(S, profileinterval,iprof, resume.tickcount[iprof]);
                Str.Append(S,nl);
                IF resume.tickcount[iprof] # 0 THEN (* don't show useless *)
                    FIO.WrStr(hnd,S);
                    WrStr(S);
                    (* dmpWantedList(anchor,iprof,profileinterval,hnd); *)
                END;
            END;
            dmpList(anchor,hnd);
            FIO.Close(hnd);
        END;
        freeList(anchor);

        IF pb = 0 THEN
            RETURN errNone;
        ELSE
            WrLn;
            RETURN errVerifyFailure;
        END;
    END;

    IF alcatraz THEN
        resume.nextblock:=block;
        resume.problems:=pb;
        resume.xtdBIOS:=XBIOS;
        resume.tracks:=trackcount;
        resume.heads:=headcount;
        resume.sectors:=sectorcount;
        resume.blocks:=blockcount;
        resume.profiling:=PROFILE;
        (* in case we resume *)
        S:=RESTARTFROM;
        hnd:=FIO.Create(S);
        FIO.WrBin(hnd,resume,SIZE(resume));
        FIO.Close(hnd);

        S:=REPORTFILE;
        hnd:=FIO.Append(S);
        IF pbmem # 0 THEN
            Str.Concat(S,sINFO,fmtlc(pbmem,10,10,"",""));
            Str.Append(S," slow sector");
            IF pbmem=1 THEN
                Str.Append(S," was");
            ELSE
                Str.Append(S,"s were");
            END;
            Str.Append(S," not reported (Storage.ALLOCATE() failure)."+nl);
            FIO.WrStr(hnd,S);
            WrStr(S);
        END;
        IF pb = 0 THEN
            S:=sINFO+"Verify operation interrupted by user !"+nl;
            FIO.WrStr(hnd,S);
        ELSE
            Str.Concat(S,sINFO,fmtlc(pb,10,0,"","") );
            Str.Append(S," error(s) found until now !"+nl);
            FIO.WrStr(hnd,S);
            WrStr(S);
        END;
        FIO.Close(hnd);

        IF PROFILE THEN
            S:=REPORTFILE;
            hnd:=FIO.Append(S);
            FOR iprof := firstprofile TO maxprofile DO
                fmtTally(S, profileinterval,iprof, resume.tickcount[iprof]);
                Str.Append(S,nl);
                IF resume.tickcount[iprof] # 0 THEN (* don't show useless *)
                    FIO.WrStr(hnd,S);
                    WrStr(S);
                    (* dmpWantedList(anchor,iprof,profileinterval,hnd); *)
                END;
            END;
            dmpList(anchor,hnd);
            FIO.Close(hnd);
        END;
        freeList(anchor);

        IF pb = 0 THEN
            WrLn;
            RETURN errAborted;
        ELSE
            WrLn;
            RETURN errVerifyFailure;
        END;
    END;
    RETURN errNone; (* never used ! *)
END doCmdVerify;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE fmtbignum (base,wi:CARDINAL;v:LONGCARD):str16;
VAR
    i : CARDINAL;
    R:str16;
    ok,lo:BOOLEAN;
    pad,prefix,suffix:CHAR;
BEGIN
    CASE base OF
    | 16: pad:="0";prefix:="$"; suffix:=""; lo:=TRUE;
    ELSE
          pad:=" ";prefix:= ""; suffix:=""; lo:=FALSE;
    END;
    Str.CardToStr ( v, R, base, ok);
    FOR i:=Str.Length(R)+1 TO wi DO Str.Prepend(R,pad);END;
    IF lo THEN Str.Lows(R); END;
    Str.Prepend(R,prefix);Str.Append(R,suffix);
    RETURN R;
END fmtbignum;

PROCEDURE fmtnum (base,wi:CARDINAL;v:CARDINAL):str16;
BEGIN
    RETURN fmtbignum(base,wi,LONGCARD(v));
END fmtnum;

PROCEDURE fmtsegofs (hi,lo:CARDINAL;sepa:CHAR ):str16;
VAR
    R:str16;
BEGIN
    Str.Concat(R, fmtnum(16,4,hi), sepa);
    Str.Append(R, fmtnum(16,4,lo));
    ReplaceChar(R,"$","");
    RETURN R;
END fmtsegofs;

PROCEDURE segofs2addr (segment,offset:CARDINAL):LONGCARD;
BEGIN
    RETURN LONGCARD(segment) << 4 + LONGCARD(offset);
END segofs2addr;

(* ssss:oooo to linear address *)

PROCEDURE so2addr (so:LONGCARD):LONGCARD ;
VAR
    segment,offset:LONGCARD;
BEGIN
    segment:=so >> 16;
    offset :=so AND 0000FFFFH;
    RETURN segment << 4 + offset;
END so2addr;

PROCEDURE splitaddr (a:LONGCARD;VAR hi,lo:CARDINAL );
BEGIN
    hi:= CARDINAL(a >> 16);
    lo:= CARDINAL(a AND 0000FFFFH);
END splitaddr;

(* ------------------------------------------------------------ *)

PROCEDURE crosspages (DEBUG:BOOLEAN;a:FarADDRESS; varsize :CARDINAL):BOOLEAN;
VAR
    a1,a2:FarADDRESS;
    page1,page2,lo1,lo2:CARDINAL;
    seg1,seg2,ofs1,ofs2:CARDINAL;

    so1,so2:LONGCARD;
    addr1,addr2:LONGCARD;
BEGIN
    IF DEBUG THEN WrStr("::: crosspages"+nl);END;

    a1:=a;
    a2:=Lib.AddFarAddr(a1,varsize-1);

    so1:=LONGCARD(a1); (* automagically converted to ssss:oooo *)
    so2:=LONGCARD(a2);

    splitaddr(so1, seg1,ofs1);
    splitaddr(so2, seg2,ofs2);

    addr1:=so2addr(so1); (* ssss:oooo to linear *)
    addr2:=so2addr(so2);
    splitaddr(addr1, page1,lo1);
    splitaddr(addr2, page2,lo2);

    IF DEBUG THEN
        WrStr(":::   ");WrStr(fmtbignum(16,8,addr1));
        WrStr("..");WrStr(fmtbignum(16,8,addr2));WrStr("  ");

        WrStr( fmtsegofs(CARDINAL(page1),CARDINAL(lo1),"-" ));
        WrStr("..");
        WrStr( fmtsegofs(CARDINAL(page2),CARDINAL(lo2),"-" ));WrStr("  ");

        WrStr( fmtsegofs(CARDINAL(seg1),CARDINAL(ofs1),":" ));
        WrStr("..");
        WrStr( fmtsegofs(CARDINAL(seg2),CARDINAL(ofs2),":" ));
        WrLn;
    END;

    RETURN ( page1 # page2 );
END crosspages;

(* ------------------------------------------------------------ *)

PROCEDURE onsamepage (DEBUG:BOOLEAN;anchor:ptrToSectorType ):BOOLEAN;
VAR
    a1,a2:FarADDRESS;
    page1,page2,lo1,lo2:CARDINAL;
    seg1,seg2,ofs1,ofs2:CARDINAL;

    so1,so2:LONGCARD;
    addr1,addr2:LONGCARD;
BEGIN
    a1:=FarADDRESS(anchor^);
    a2:=Lib.AddFarAddr(a1,SIZE(ptrToSectorType^)-1);

    so1:=LONGCARD(a1); (* automagically converted to ssss:oooo *)
    so2:=LONGCARD(a2);

    splitaddr(so1, seg1,ofs1);
    splitaddr(so2, seg2,ofs2);

    addr1:=so2addr(so1); (* ssss:oooo to linear *)
    addr2:=so2addr(so2);
    splitaddr(addr1, page1,lo1);
    splitaddr(addr2, page2,lo2);

    IF DEBUG THEN
        WrStr("[");WrStr(fmtbignum(16,8,addr1));
        WrStr("..");WrStr(fmtbignum(16,8,addr2));WrStr("]"+"  ");

        WrStr( fmtsegofs(CARDINAL(page1),CARDINAL(lo1),"-" ));
        WrStr("..");
        WrStr( fmtsegofs(CARDINAL(page2),CARDINAL(lo2),"-" ));WrStr("  ");

        WrStr( fmtsegofs(CARDINAL(seg1),CARDINAL(ofs1),":" ));
        WrStr("..");
        WrStr( fmtsegofs(CARDINAL(seg2),CARDINAL(ofs2),":" ));
        WrLn;
    END;

    RETURN ( page1=page2 );
END onsamepage;

PROCEDURE releaseMem (anchor:ptrToSectorType);
BEGIN
    IF anchor # NIL THEN DEALLOCATE(anchor,SIZE(anchor^)); END;
END releaseMem;

PROCEDURE grabMem (DEBUG:BOOLEAN):ptrToSectorType;
CONST
    firstbuff = 1;
    lastbuff  = 16;
VAR
    buff : ARRAY [firstbuff..lastbuff] OF ptrToSectorType;
    anchor : ptrToSectorType;
    last,i,wanted:CARDINAL;
BEGIN
    IF DEBUG THEN WrStr("::: grabMem");WrLn;END;
    anchor:=NIL;
    wanted:=SIZE(ptrToSectorType^);
    i:=firstbuff-1;
    LOOP
        INC(i);
        IF i > lastbuff THEN EXIT; END;
        IF Available(wanted)=FALSE THEN
            IF DEBUG THEN WrStr("Storage.Available() returned FALSE !");WrLn;END;
            EXIT;
        END;
        ALLOCATE(buff[i],wanted);
        IF DEBUG THEN
            WrStr("::: ");WrStr( fmtnum(10,2,i) );WrStr(" @ ");
        END;
        IF onsamepage(DEBUG,buff[i]) THEN anchor:=buff[i]; EXIT; END;
    END;
    DEC(i);
    last:=i;
    FOR i:= firstbuff TO last DO
        DEALLOCATE( buff[i] ,wanted);
    END;
    RETURN anchor;
END grabMem;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    sPARMS        = "P";
    legalCommands = "R"+delim+"READ"+delim+"S"+delim+"SAVE"+delim+
                    "W"+delim+"WRITE"+delim+"U"+delim+"UPDATE"+delim+
                    "C"+delim+"COMPARE"+delim+
                    "V"+delim+"VERIFY"+delim+
                    "VV"+delim+
                    sPARMS+delim+"PARMS"+delim+"I"+delim+"INFOS";
CONST
    firstparm = 1;
    maxparm   = 6; (* longest is "u cmd t h s file" *)
VAR
    parmcount,i,opt,lastparm:CARDINAL;
    S,R:str128;
    DEBUG,XBIOSavailable:BOOLEAN;
    OVERWRITE,IGNORERO,AUTOCONFIRM,VERBOSE,XBIOS:BOOLEAN;
    REALLY,BACKUP,ALTFORMAT,FORCERESTART,SCANMAGIC,PROFILE,IGNOREFATAL:BOOLEAN;
    BESTFIT:CARDINAL;
    countofsectors,block:LONGCARD;
    track,head,sector:CARDINAL;
    parm : ARRAY [firstparm..maxparm] OF str128;
    cmd : (cmdParms,cmdRead,cmdWrite,cmdCompare,cmdVerify);
    sCmd,sUnit,sBlock,sTrack,sHead,sSector:str128; (* oversized ! *)
    bakfile,sFile:str128;
    access : (noaccess,byblock, byTHS);
    unit,EDDmajor,EDDflag:BYTE;
    maxtrackorg,maxheadorg,maxsectororg,bytespersector:CARDINAL;
    maxtrack,maxhead,maxsector,trackcount,headcount,sectorcount:CARDINAL;
    totalsectors,maxblock,blockcount,fsize,mismatches:LONGCARD;
    NEWGEOMETRY,PERFECTGEOMETRY:BOOLEAN; (* was fixTHSgeometry really needed ? *)
    CHECKBOUNDARIES:BOOLEAN;
    FIXBADTHS : BOOLEAN;
    rcode:CARDINAL;
    ticksThreshold:LONGCARD; (* a CARDINAL would do *)
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck:=FALSE;
    WrLn;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    initFatalStatus;
    DYNALLOC      := FALSE;
    AUDIO         := FALSE;
    FATALINAL     := TRUE;   (* global, globerk *)

    OVERWRITE     := FALSE;
    IGNORERO      := FALSE;
    AUTOCONFIRM   := FALSE;
    VERBOSE       := TRUE;
    XBIOS         := TRUE;
    BESTFIT       := FORCEBEST;
    BACKUP        := FALSE;
    ALTFORMAT     := FALSE;
    bakfile       := BACKUPFILE;
    countofsectors:= MAX(LONGCARD); (* flag to means it was not specified *)
    FORCERESTART  := FALSE; (* initialization useless here but cleaner *)
    REALLY        := TRUE;
    SCANMAGIC     := FALSE;
    IGNOREFATAL   := FALSE;
    ticksThreshold:= MAX(LONGCARD); (* not default but flag *)
    CHECKBOUNDARIES:=FALSE;
    FIXBADTHS     := FALSE;
    DEBUG         := FALSE;
    lastparm      := firstparm-1;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "O"+delim+"OVERWRITE"+delim+
                                  "YES"+delim+"OUI"+delim+"AUTOCONFIRM"+delim+
                                  "D"+delim+"DISPLAY"+delim+
                                  "Q"+delim+"QUIET"+delim+
                                  "C:"+delim+"COUNT:"+delim+
                                  "X"+delim+"LBA"+delim+
                                  "R"+delim+"READONLY"+delim+
                                  "OO"+delim+"OR"+delim+
                                  "J"+delim+"F255"+delim+"MAXHEAD"+delim+
                                  "S"+delim+"SAVE"+delim+"B"+delim+"BACKUP"+delim+
                                  "N:"+delim+"FILE:"+delim+"NAME:"+delim+
                                  "JJ"+delim+"F240"+delim+
                                  "A"+delim+"AUDIO"+delim+"W"+delim+
                                  "T"+delim+"TEST"+delim+"FAKE"+delim+
                                  "M"+delim+"MAGIC"+delim+"AA55"+delim+
                                  "P"+delim+"PROFILE"+delim+
                                  "P:"+delim+"PROFILE:"+delim+
                                  "U"+delim+"UNSAFE"+delim+"UNWISE"+delim+"IGNOREFATAL"+delim+
                                  "??"+delim+"HH"+delim+
                                  "64KB"+delim+"PAGE"+delim+"CROSS"+delim+"CHK"+delim+"PARANOIA"+delim+
                                  "AH"+delim+
                                  "DYN"+delim+"DY"+delim+
                                  "THS"+delim+"FIXBADTHS"+delim+"BAD"+delim+
                                  "DEBUG"
                              );
            CASE opt OF
            | 1,2,3 :   abort(errHelp,"");
            | 4,5 :     OVERWRITE := TRUE;
            | 6,7,8 :   AUTOCONFIRM:=TRUE;
            | 9,10:     ALTFORMAT := TRUE;
            | 11,12:    VERBOSE   := FALSE;
            | 13,14:    IF GetLongCard(R,countofsectors)=FALSE THEN abort(errLongCard,S); END;
                        IF countofsectors < mincountofsectors THEN abort(errLongCard,S);END;
            | 15,16:    XBIOS     := FALSE;
            | 17,18:    IGNORERO  := TRUE;
            | 19,20:    OVERWRITE := TRUE; IGNORERO := TRUE;
            | 21,22,23: BESTFIT   := FORCE255;
            | 24,25,26,27 : BACKUP:= TRUE;
            | 28,29,30: GetString(R,bakfile);
            | 31,32:    BESTFIT   := FORCE240;
            | 33,34,35: AUDIO     := TRUE;
            | 36,37,38: REALLY    := FALSE;
            | 39,40,41: SCANMAGIC := TRUE;
            | 42,43:    PROFILE   := TRUE;
            | 44,45:    IF GetLongCard(R,ticksThreshold)=FALSE THEN abort(errLongCard,S); END;
                        IF ticksThreshold < MINTICKSTHRESHOLD THEN abort(errThreshold,S);END;
                        IF ticksThreshold > MAXTICKSTHRESHOLD THEN abort(errThreshold,S);END;
                        PROFILE   := TRUE;
            | 46,47,48,49: IGNOREFATAL:=TRUE;
            | 50,51 :   abort(errHelper,"");
            | 52,53,54,55,56: CHECKBOUNDARIES:=TRUE;
            | 57 :      FATALINAL := FALSE;
            | 58,59:    DYNALLOC  := TRUE;
            | 60,61,62: FIXBADTHS := TRUE;
            | 63:       DEBUG     := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxparm THEN abort(errParameter,S);END;
            Str.Copy( parm[lastparm], R);
        END;
    END;

    access:=noaccess;
    CASE lastparm OF
    | 1 : (* unit *)
        Str.Copy(sUnit,  parm[1]);
        Str.Copy(sCmd,   sPARMS);   (* fake it *)
    | 2 : (* unit cmd *)
        Str.Copy(sUnit,  parm[1]);
        Str.Copy(sCmd,   parm[2]);
    | 4 : (* unit cmd block *)
        Str.Copy(sUnit,  parm[1]);
        Str.Copy(sCmd,   parm[2]);
        Str.Copy(sBlock, parm[3]);
        Str.Copy(sFile,  parm[4]);
        IF getvallc(sBlock,block)=FALSE THEN abort(errLongCard,sBlock); END;
        access:=byblock;
    | 6 : (* unit cmd track head sector *)
        Str.Copy(sUnit,  parm[1]);
        Str.Copy(sCmd,   parm[2]);
        Str.Copy(sTrack, parm[3]);
        Str.Copy(sHead,  parm[4]);
        Str.Copy(sSector,parm[5]);
        Str.Copy(sFile,  parm[6]);
        IF getval(sTrack,track)=FALSE THEN abort(errLongCard,sTrack); END;
        IF getval(sHead,head)=FALSE THEN abort(errLongCard,sHead); END;
        IF getval(sSector,sector)=FALSE THEN abort(errLongCard,sSector); END;
        access:=byTHS;
    ELSE
        abort(errSyntax,"");
    END;

    CASE getStrIndex(delim,sCmd,legalCommands) OF
    | 1,2,3,4    : cmd:=cmdRead;
    | 5,6,7,8    : cmd:=cmdWrite;
    | 9,10       : cmd:=cmdCompare;
    | 11,12      : cmd:=cmdVerify;  FORCERESTART:=FALSE;
    | 13         : cmd:=cmdVerify;  FORCERESTART:=TRUE;
    | 14,15,16,17: cmd:=cmdParms;
    ELSE
        abort(errCommand,sCmd);
    END;

    CASE cmd OF
    | cmdParms:
        CASE lastparm OF
        | 1,2:
            ;
        ELSE
            abort(errCmd,sCmd);
        END;
    | cmdVerify:
        IF lastparm # 2 THEN abort(errCmd,sCmd);END;
    | cmdRead,cmdWrite,cmdCompare:
        CASE lastparm OF
        | 4,6:
            ;
        ELSE
            abort(errCmd,sCmd);
        END;
    END;

    IF chkUnit(unit,sUnit)=FALSE THEN abort(errUnit,sUnit); END;

IF NOT ( DYNALLOC ) THEN
    (* now is a good time to check if static buffers don't cross 64Kb boundary *)

    IF CHECKBOUNDARIES THEN
        IF crosspages( DEBUG,FarADR(buffSector),SIZE(buffSector)) THEN
            abort(errBoundary,"buffSector");
        END;
        IF crosspages( DEBUG,FarADR(buffSectorDisk),SIZE(buffSectorDisk)) THEN
            abort(errBoundary,"buffSectorDisk");
        END;
    END;
ELSE
    IF CHECKBOUNDARIES THEN abort(errDynChk,"");END;
    (* now is a good time to create (ptrTo)SectorType buffers *)
    (* we'll let good old DOS handle deallocation alone in most abort()cases *)
    (* maybe we should check only if CHECKBOUNDARIES ? bah... *)

    pbuffSector     := NIL;
    pbuffSectorDisk := NIL;

    pbuffSector := grabMem(DEBUG);
    IF pbuffSector = NIL THEN abort(errBoundary,"pbuffSector");END;
    pbuffSectorDisk:=grabMem(DEBUG);
    IF pbuffSectorDisk = NIL THEN abort(errBoundary,"pbuffSectorDisk");END;
END;

    (* let's start checking for Good Old DOS *)

    (* GOD:=NOT ( w9XsupportLFN() ); *)

    IF fixFatalStatusBios () THEN FATALINAL:=FALSE; END; (* if .CFG exists, force -AH *)

    IF resetUnit(DEBUG,RESETHD,unit)=FALSE THEN abort(errInt13h,"resetUnit");END;
    XBIOSavailable:=chkExtendedSupport(DEBUG,unit,EDDmajor,EDDflag);
    XBIOS := (XBIOS AND XBIOSavailable);
    IF isFloppy(unit) THEN XBIOS:=FALSE; END; (* safety, though it is trapped at chkExtendedSupport() *)

    i:= getGeometry (DEBUG,unit,XBIOS,FIXBADTHS,BESTFIT,
                    maxtrackorg,maxheadorg,maxsectororg,bytespersector,
                    totalsectors);
    CASE i OF
    | rcPhantom :    abort(errPhantom,sUnit); (* will only trap phantom floppy ! *)
    | rcInt13h:      abort(errInt13h,"getDiskGeometry");
    | rcXBIOSint13h: abort(errXBIOSint13h,"getDiskGeometry");
    | rcXBIOSvalues: abort(errXBIOSvalues,"getXBIOSvalues");
    | rcWrongTHSgeometry:abort(errWrongTHSgeometry,"getDiskGeometry");
    END;

    i:=fixGeometry (DEBUG,unit,XBIOS,BESTFIT,
                   maxtrackorg,maxheadorg,maxsectororg,totalsectors,
                   maxtrack,maxhead,maxsector,maxblock,
                   trackcount,headcount,sectorcount,blockcount,
                   NEWGEOMETRY,PERFECTGEOMETRY);
    IF i # errNone THEN abort(errFixGeometry,"");END;

    CASE cmd OF
    | cmdParms:
        (* valid : XBIOS, BESTFIT, AUDIO, ALTFORMAT, DEBUG *)

        IF OVERWRITE      THEN abort(errNonsense,"-o"); END;
        IF IGNORERO       THEN abort(errNonsense,"-r"); END;
        IF AUTOCONFIRM    THEN abort(errNonsense,"-yes"); END;
        IF VERBOSE=FALSE  THEN abort(errNonsense,"-q"); END;
        IF BACKUP         THEN abort(errNonsense,"-s"); END;
        IF REALLY=FALSE   THEN abort(errNonsense,"-t"); END;
        IF SCANMAGIC      THEN abort(errNonsense,"-m"); END;
        IF PROFILE        THEN abort(errNonsense,"-p[:#]"); END;
        IF IGNOREFATAL    THEN abort(errNonsense,"-u");END;
        IF same(bakfile,BACKUPFILE)=FALSE THEN abort(errNonsense,"-n:$"); END;
        IF countofsectors # MAX(LONGCARD) THEN abort(errNonsense,"-c:#"); END;

        doCmdParms (DEBUG,unit,XBIOS,XBIOSavailable,
                   NEWGEOMETRY,PERFECTGEOMETRY,ALTFORMAT,
                   EDDmajor,EDDflag,BESTFIT,
                   maxtrackorg,maxheadorg,maxsectororg,totalsectors,
                   maxtrack,maxhead,maxsector,maxblock,
                   trackcount,headcount,sectorcount,blockcount);
        abort(errNone,"");
    | cmdVerify:
        (* valid : AUDIO, XBIOS, BESTFIT, SCANMAGIC, PROFILE, DEBUG *)
        IF OVERWRITE      THEN abort(errNonsense,"-o"); END;
        IF IGNORERO       THEN abort(errNonsense,"-r"); END;
        IF AUTOCONFIRM    THEN abort(errNonsense,"-yes"); END;
        IF VERBOSE=FALSE  THEN abort(errNonsense,"-q"); END;
        IF ALTFORMAT      THEN abort(errNonsense,"-d"); END;
        IF BACKUP         THEN abort(errNonsense,"-s"); END;
        IF REALLY=FALSE   THEN abort(errNonsense,"-t"); END;
        IF IGNOREFATAL    THEN abort(errNonsense,"-u"); END;
        IF same(bakfile,BACKUPFILE)=FALSE THEN abort(errNonsense,"-n:$"); END;
        IF countofsectors # MAX(LONGCARD) THEN abort(errNonsense,"-c:#"); END;

        IF ticksThreshold = MAX(LONGCARD) THEN ticksThreshold:=DEFAULTTICKSTHRESHOLD;END;

        i:=doCmdVerify (unit,XBIOS,FORCERESTART,SCANMAGIC,PROFILE,
                       trackcount,headcount,sectorcount,blockcount,
                       ticksThreshold);
        abort(i,"");
    END;

    (* R, W, C *)

    CASE access OF
    | byblock:
        IF withinlc(block,minblock,maxblock)=FALSE THEN
            Str.Concat(S,"<block> should be within ",fmtrangelc (minblock,maxblock));
            abort(errRange,S);
        END;
        blockToCHS (trackcount,headcount,sectorcount,
                   block,
                   track,head,sector);
IF DEBUG THEN
    WrStr("::: byblock");WrLn;
    WrStr("block            ");IO.WrLngCard(block,0);WrLn;
    WrLn;
    WrStr("track            ");IO.WrCard(track,0);WrLn;
    WrStr("head             ");IO.WrCard(head,0);WrLn;
    WrStr("sector           ");IO.WrCard(sector,0);WrLn;
END;
    | byTHS:
        IF within(track,mintrack,maxtrack)=FALSE THEN
            Str.Concat(S,"<track> should be within ",fmtrange (mintrack,maxtrack));
            abort(errRange,S);
        END;
        IF within(head,minhead,maxhead)=FALSE THEN
            Str.Concat(S,"<head> should be within ",fmtrange (minhead,maxhead));
            abort(errRange,S);
        END;
        IF within(sector,minsector,maxsector)=FALSE THEN
            Str.Concat(S,"<sector> should be within ",fmtrange (minsector,maxsector));
            abort(errRange,S);
        END;
        CHStoBlock (trackcount,headcount,sectorcount,
                   track,head,sector,
                   block);
IF DEBUG THEN
    WrStr("::: byTHS");WrLn;
    WrStr("track            ");IO.WrCard(track,0);WrLn;
    WrStr("head             ");IO.WrCard(head,0);WrLn;
    WrStr("sector           ");IO.WrCard(sector,0);WrLn;
    WrLn;
    WrStr("block            ");IO.WrLngCard(block,0);WrLn;
END;
    END;

    defaultExtension(sFile,extBIN);
    IF chkFilename(sFile)=FALSE THEN abort(errFile,sFile);END;

    CASE cmd OF
    | cmdRead:
        (* valid : OVERWRITE, IGNORERO, XBIOS, BESTFIT, AUDIO, DEBUG, PROFILE *)

        IF AUTOCONFIRM    THEN abort(errNonsense,"-yes");END;
        IF VERBOSE=FALSE  THEN abort(errNonsense,"-q");END;
        IF ALTFORMAT      THEN abort(errNonsense,"-d"); END;
        IF BACKUP         THEN abort(errNonsense,"-s");END;
        IF REALLY=FALSE   THEN abort(errNonsense,"-t");END;
        IF SCANMAGIC      THEN abort(errNonsense,"-m"); END;
        IF ticksThreshold # MAX(LONGCARD) THEN abort(errNonsense,"-p:#");END;
        IF same(bakfile,BACKUPFILE)=FALSE THEN abort(errNonsense,"-n:$"); END;
        IF countofsectors = MAX(LONGCARD) THEN countofsectors:=mincountofsectors;END;
        IF withinlc(block+countofsectors-1,minblock,maxblock)=FALSE THEN
            Str.Concat(S,"-c:# should be within ",fmtrangelc (mincountofsectors,maxblock-block+1));
            abort(errRange,S);
        END;

        IF FIO.Exists(sFile) THEN
            IF OVERWRITE THEN
                IF isReadOnly(sFile) THEN
                    IF IGNORERO THEN
                        setReadWrite(sFile);
                    ELSE
                        abort(errAlreadyRO,sFile);
                    END;
                END;
            ELSE
                abort(errAlready,sFile);
            END;
        END;

        IF doCmdRead (DEBUG,IGNOREFATAL,PROFILE, unit,XBIOS,
                     track,head,sector,block,countofsectors,
                     maxtrack,maxhead,maxsector,maxblock,
                     trackcount,headcount,sectorcount,blockcount,
                     sFile)=FALSE THEN fatal(XBIOS,"procSector (read)");END;

    | cmdWrite:
        (* valid : REALLY, AUTOCONFIRM, XBIOS, BESTFIT, BACKUP, AUDIO, bakfile, DEBUG *)

        IF IsRedirected() THEN abort(errRedirected,""); END;

        IF OVERWRITE      THEN abort(errNonsense,"-o");END;
        IF IGNORERO       THEN abort(errNonsense,"-r");END;
        IF VERBOSE=FALSE  THEN abort(errNonsense,"-q");END;
        IF ALTFORMAT      THEN abort(errNonsense,"-d"); END;
        IF SCANMAGIC      THEN abort(errNonsense,"-m"); END;
        IF PROFILE        THEN abort(errNonsense,"-p[:#]"); END;
        IF countofsectors # MAX(LONGCARD) THEN abort(errNonsense,"-c:#");END;

        IF FIO.Exists(sFile)=FALSE THEN abort(errNotFound,sFile);END;
        fsize := getFileSize(sFile);
        IF fsize = 0 THEN abort(errSectorSize,sFile);END;
        IF (fsize MOD sectorSize) # 0 THEN abort(errSectorSize,sFile);END;
        countofsectors := (fsize DIV sectorSize );
        IF withinlc(block+countofsectors-1,minblock,maxblock)=FALSE THEN
            abort(errBeyondDisk,sFile);
        END;

        IF chkBackupname(bakfile, extSOS)=FALSE THEN
            abort(errBadBackupname,bakfile);
        END;

        IF BACKUP THEN

            IF FIO.Exists(bakfile) THEN
                IF isReadOnly(bakfile) THEN setReadWrite(bakfile);END;
            END;

            IF doCmdRead (DEBUG,IGNOREFATAL, FALSE,  unit,XBIOS,
                         track,head,sector,block,countofsectors,
                         maxtrack,maxhead,maxsector,maxblock,
                         trackcount,headcount,sectorcount,blockcount,
                         bakfile)=FALSE THEN
               fatal(XBIOS,"procSector (read)");
            END;
            WrLn;
            createUndo (unit,XBIOS,BESTFIT,track,head,sector,block,
                       bakfile,UNDOBATCH);
            WrLn;
        END;

        IF doCmdWrite (DEBUG,IGNOREFATAL,REALLY,AUTOCONFIRM,unit,XBIOS,
                      track,head,sector,block,countofsectors,
                      maxtrack,maxhead,maxsector,maxblock,
                      trackcount,headcount,sectorcount,blockcount,
                      sFile)=FALSE THEN fatal(XBIOS,"procSector (write)");END;

    | cmdCompare:
        (* valid : VERBOSE, AUDIO, XBIOS, BESTFIT, DEBUG *)

        IF AUTOCONFIRM  THEN abort(errNonsense,"-yes");END;
        IF OVERWRITE    THEN abort(errNonsense,"-o");END;
        IF IGNORERO     THEN abort(errNonsense,"-r");END;
        IF ALTFORMAT    THEN abort(errNonsense,"-d"); END;
        IF BACKUP       THEN abort(errNonsense,"-s");END;
        IF REALLY=FALSE THEN abort(errNonsense,"-t");END;
        IF SCANMAGIC    THEN abort(errNonsense,"-m"); END;
        IF PROFILE        THEN abort(errNonsense,"-p[:#]"); END;
        IF same(bakfile,BACKUPFILE)=FALSE THEN abort(errNonsense,"-n:$"); END;
        IF countofsectors # MAX(LONGCARD) THEN abort(errNonsense,"-c:#");END;

        IF FIO.Exists(sFile)=FALSE THEN abort(errNotFound,sFile);END;
        fsize := getFileSize(sFile);
        IF fsize = 0 THEN abort(errSectorSize,sFile);END;
        IF (fsize MOD sectorSize) # 0 THEN abort(errSectorSize,sFile);END;
        countofsectors := (fsize DIV sectorSize );
        IF withinlc(block+countofsectors-1,minblock,maxblock)=FALSE THEN
            abort(errBeyondDisk,sFile);
        END;

        rcode:=doCmdCompare (mismatches,
                            DEBUG,IGNOREFATAL,VERBOSE,unit,XBIOS,
                            track,head,sector,block,countofsectors,
                            maxtrack,maxhead,maxsector,maxblock,
                            trackcount,headcount,sectorcount,blockcount,
                            sFile);
        CASE rcode OF
        | errAborted: abort(errAborted,"");
        | errInt13h : fatal(XBIOS,"procSector (read)");
        END;

        (* CASE don't like longcards *)
        IF mismatches = 0 THEN
            WrStr(sOK+"Data from <source> and from <file> are identical.");
        ELSIF mismatches = 1 THEN
            WrStr(sPROBLEM+"1 mismatch found between <source> and <file>.");
        ELSE
            WrStr(sPROBLEM);WrStr(fmtlc(mismatches,10,0,"",""));
            WrStr(" mismatches found between <source> and <file>.");
        END;
        WrLn;
        IF mismatches # 0 THEN abort(errMismatch,"");END;
    END;

IF DYNALLOC THEN
    (* yes, we know we let good old DOS handle this with earlier abort() calls *)
    releaseMem(pbuffSector);
    releaseMem(pbuffSectorDisk);
END;

    abort(errNone,"");
END DTHS.



(*

unit = $00             /f    /x    /x /f

XBIOS available  no    no    no    no
XBIOS used here  no    no    no    no
THS autorebuilt  no    no    no    no
perfect geometry Yes   Yes   Yes   Yes
-f specified     no    Yes   no    Yes
blockcount       2880  2880  2880  2880
totalsectors     2880  2880  2880  2880
mismatch         no    no    no    no


unit $80                    /f         /x        /x /f

getGeometry      XBIOS      XBIOS      bios      bios
trackcount       14818      14818      523       523
headcount        9          9          255       255
sectorcount      63         63         63        63
XBIOS available  Yes        Yes        Yes       Yes
XBIOS used here  Yes        Yes        no        no
THS autorebuilt  no         no         no        no
perfect geometry Yes        Yes        Yes       Yes
-f specified     no         Yes        no        Yes
blockcount       8401806    8401806    8401995   8401995
totalsectors     8401806    8401806    8401995   8401995
mismatch         no         no         no        no


unit $80                    /f         /x        /x /f

getGeometry      XBIOS      XBIOS      bios      bios
trackcount       11344      9964       1024      1024
headcount        224        255        255       255
sectorcount      63         63         63        63
XBIOS available  Yes        Yes        Yes       Yes
XBIOS used here  Yes        Yes        no        no
THS autorebuilt  Yes        Yes        no        no
perfect geometry Yes        no         Yes       Yes
-f specified     no         Yes        no        Yes
blockcount       160086528  160086528  16450560  16450560
totalsectors     160086528  160071660  16450560  16450560
mismatch         no         Yes        no        no

*)








