
(* ---------------------------------------------------------------
Title         Q&D PlayWAV
Overview      YACUP, YACWOT
Usage
Notes         we only play 8 bits PCM sounds
              see PC-GPE v1.0 text about DMA,
              and Programmer's PC Sourcebook 7.067 & 7.068
Bugs          computer freezes when program is run from Win9X
              whatever the audio chip, but it's fine from Win3X !
Wish List     find better infos about SB programming,
              and check fixme in order to play 22kHz and 16 bits files
              a close look at dma.c could be useful, but what for ?
              let's keep this crap as YAUB
              (yet another unfinished business, eh Lara !)

--------------------------------------------------------------- *)

MODULE playWAV;

IMPORT Str;
IMPORT Lib;
IMPORT FIO;
IMPORT SYSTEM;
IMPORT Storage;

FROM IO IMPORT WrStr, WrLn;
FROM Storage IMPORT Available,ALLOCATE,DEALLOCATE;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits;

FROM QD_LFN IMPORT w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileIsDirectorySpec,
fileClose;

(* ------------------------------------------------------------ *)

CONST
    cr          = CHR(13);
    lf          = CHR(10);
    nl          = cr+lf;
    dot         = ".";
    extWAV      = ".WAV";
    sBLASTER    = "BLASTER";
CONST
    progEXEname   = "PLAYWAV";
    progTitle     = "Q&D PlayWAV";
    progVersion   = "v1.1e";
    progCopyright = "by PhG";
    banner        = progTitle+" "+progVersion+" "+progCopyright;
CONST
    errNone            = 0;
    errHelp            = 1;
    errOption          = 2;
    errTooManyParms    = 3;
    errMissingSpec     = 4;
    errNotFound        = 5;
    errNotFile         = 6;
    errJoker           = 7;
    errUselessSpec     = 8;
    errExclusive       = 9;

    errALLOCATEPAGE    = 50;

    errSBnoBlasterVar  = 100;
    errSBbadID         = 101;
    errSBbadAddr       = 102;
    errSBbadDMA        = 103;
    errSBbadIRQ        = 104;
    errSBbadDMA16      = 105;
    errSBbadMidiPort   = 106;
    errSBunsupported   = 107;

    errSBreset         = 200;
    errWAVheaderSize   = 201;
    errWAVheaderSizeChunk=202;
    errWAVheaderID     = 203;
    errWAVformat       = 204;

    errSafetyFirst     = 255;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
errmsg =
banner+nl+
nl+
"Syntax : "+progEXEname+" <file["+extWAV+"]> [-lfn] [-p] [-t|-i]"+nl+
nl+
"  -lfn  disable LFN support even if available"+nl+
"  -t    do not display infos about played file"+nl+
"  -p    display current SoundBlaster parameters then exit"+nl+
"  -i    display infos about specified file then exit without playing it"+nl+
nl+
"This program requires a SoundBlaster Pro (or better) compatible card"+nl+
"and it assumes "+sBLASTER+" environment variable is correctly set."+nl+
"Only 8-bits PCM data is supported for now... and for ever ! ;-)"+nl+
"In order to prevent computer freeze, no sound will be played from Windows 9X."+nl;

VAR
    S  : str1024; (* oversized but we may get a LFN *)
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," is one parameter too many !");
    | errMissingSpec:
        S := "Missing <file["+extWAV+"]> specification !";
    | errNotFound:
        Str.Concat(S,einfo," does not exist !");
    | errNotFile:
        Str.Concat(S,einfo," looks like a directory !");
    | errJoker:
        S := "Jokers are not allowed in <file["+extWAV+"]> !";
    | errUselessSpec:
        S := "Useless <file["+extWAV+"]> specification !";
    | errExclusive:
        Str.Concat(S,einfo," options are mutually exclusive !");

    | errALLOCATEPAGE:
        Str.Concat(S,einfo," buffer would cross 64Kb page boundary !");

    | errSBnoBlasterVar:
        S := "BLASTER environment variable does not exist !";
    | errSBbadID :
        S := "Unsupported or illegal BLASTER card model !";
    | errSBbadAddr :
        S := "Unsupported, missing or illegal BLASTER I/O address !";
    | errSBbadDMA :
        S := "Unsupported, missing or illegal BLASTER DMA value !";
    | errSBbadIRQ :
        S := "Unsupported, missing or illegal BLASTER IRQ value !";
    | errSBbadDMA16 :
        S := "Unsupported, missing or illegal BLASTER DMA-16 value !";
    | errSBbadMidiPort   :
        S := "Unsupported, missing or illegal BLASTER MIDI port addresse !";
    | errSBunsupported:
        S := "Unsupported BLASTER parameter !";

    | errSBreset:
        S := "Audio card would not reset !";
    | errWAVheaderSize:
        Str.Concat(S,"Bad WAV header size for ",einfo);Str.Append(S," !");
    | errWAVheaderSizeChunk:
        Str.Concat(S,"Bad WAV header size for ",einfo);Str.Append(S," !"); (* same msg *)
    | errWAVheaderID:
        Str.Concat(S,"Bad WAV header ID for ",einfo);Str.Append(S," !");
    | errWAVformat:
        Str.Concat(S,"Unsupported WAV format for ",einfo);Str.Append(S," !");

    | errSafetyFirst:
        e:=errNone;
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ;
    ELSE
        WrStr(progEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize      = (8 * 512) + FIO.BufferOverhead;
    firstioBufferByte = 1;
    lastioBufferByte  = ioBufferSize;
TYPE
    ioBufferType = ARRAY [firstioBufferByte..lastioBufferByte] OF BYTE;
VAR
    ioBufferIn   : ioBufferType; (* globerk *)

PROCEDURE dmp (doit:BOOLEAN;S1,S2:ARRAY OF CHAR);
CONST
    wi = 21;
VAR
    i : CARDINAL;
BEGIN
    IF doit THEN
        WrStr(S1);
        FOR i:=Str.Length(S1)+1 TO wi DO WrStr(" ");END;
        WrStr("= ");
        WrStr(S2); WrLn;
    END;
END dmp;

PROCEDURE fmtbignum (base,wi:CARDINAL;v:LONGCARD):str16;
VAR
    i : CARDINAL;
    R:str16;
    ok,lo:BOOLEAN;
    pad,prefix,suffix:CHAR;
BEGIN
    CASE base OF
    | 16: pad:="0";prefix:="$"; suffix:=""; lo:=TRUE;
    ELSE
          pad:=" ";prefix:= ""; suffix:=""; lo:=FALSE;
    END;
    Str.CardToStr ( v, R, base, ok);
    FOR i:=Str.Length(R)+1 TO wi DO Str.Prepend(R,pad);END;
    IF lo THEN Str.Lows(R); END;
    Str.Prepend(R,prefix);Str.Append(R,suffix);
    RETURN R;
END fmtbignum;

PROCEDURE fmtnum (base,wi:CARDINAL;v:CARDINAL):str16;
BEGIN
    RETURN fmtbignum(base,wi,LONGCARD(v));
END fmtnum;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    firstDataByte = 0;
    lastDataByte  = 32*512-1;
    dataBufferSize= lastDataByte - firstDataByte + 1;
TYPE
    dataBufferType = ARRAY [firstDataByte..lastDataByte] OF BYTE;
    pBuff = POINTER TO dataBufferType;

TYPE
    soundType = RECORD
        datapos  : LONGCARD;
        datasize : LONGCARD;
        channels : CARDINAL;
        frequency: LONGCARD; (* bytespersec *)
        anchor   : pBuff;
        (* modified by caller *)
        remaining: LONGCARD;
        wanted   : CARDINAL; (* what we wanted/got from file *)
    END;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE fmtsegofs (hi,lo:CARDINAL;sepa:CHAR ):str16;
VAR
    R:str16;
BEGIN
    Str.Concat(R, fmtnum(16,4,hi), sepa);
    Str.Append(R, fmtnum(16,4,lo));
    ReplaceChar(R,"$","");
    RETURN R;
END fmtsegofs;

PROCEDURE segofs2addr (segment,offset:CARDINAL):LONGCARD;
BEGIN
    RETURN LONGCARD(segment) << 4 + LONGCARD(offset);
END segofs2addr;

(* ssss:oooo to linear address *)

PROCEDURE so2addr (so:LONGCARD):LONGCARD ;
VAR
    segment,offset:LONGCARD;
BEGIN
    segment:=so >> 16;
    offset :=so AND 0000FFFFH;
    RETURN segment << 4 + offset;
END so2addr;

PROCEDURE splitaddr (a:LONGCARD;VAR hi,lo:CARDINAL );
BEGIN
    hi:= CARDINAL(a >> 16);
    lo:= CARDINAL(a AND 0000FFFFH);
END splitaddr;

PROCEDURE onsamepage (DEBUG:BOOLEAN;anchor:pBuff ):BOOLEAN;
VAR
    a1,a2:FarADDRESS;
    page1,page2,lo1,lo2:CARDINAL;
    seg1,seg2,ofs1,ofs2:CARDINAL;

    so1,so2:LONGCARD;
    addr1,addr2:LONGCARD;
BEGIN
    a1:=FarADDRESS(anchor^);
    a2:=Lib.AddFarAddr(a1,SIZE(pBuff^)-1);

    so1:=LONGCARD(a1); (* automagically converted to ssss:oooo *)
    so2:=LONGCARD(a2);

    splitaddr(so1, seg1,ofs1);
    splitaddr(so2, seg2,ofs2);

    addr1:=so2addr(so1); (* ssss:oooo to linear *)
    addr2:=so2addr(so2);
    splitaddr(addr1, page1,lo1);
    splitaddr(addr2, page2,lo2);

    IF DEBUG THEN
        WrStr("[");WrStr(fmtbignum(16,8,addr1));
        WrStr("..");WrStr(fmtbignum(16,8,addr2));WrStr("]"+"  ");

        WrStr( fmtsegofs(CARDINAL(page1),CARDINAL(lo1),"-" ));
        WrStr(" to ");
        WrStr( fmtsegofs(CARDINAL(page2),CARDINAL(lo2),"-" ));WrStr("  ");

        WrStr( fmtsegofs(CARDINAL(seg1),CARDINAL(ofs1),":" ));
        WrStr(" to ");
        WrStr( fmtsegofs(CARDINAL(seg2),CARDINAL(ofs2),":" ));
        WrLn;
    END;

    RETURN ( page1=page2 );
END onsamepage;

PROCEDURE releaseMem (anchor:pBuff);
BEGIN
    Storage.DEALLOCATE(anchor,SIZE(anchor^));
END releaseMem;

PROCEDURE grabMem (DEBUG:BOOLEAN):pBuff;
CONST
    firstbuff = 1;
    lastbuff  = 16; (* grab up to 16K * 16 = 256K *)
VAR
    buff : ARRAY [firstbuff..lastbuff] OF pBuff;
    anchor : pBuff;
    last,i,wanted:CARDINAL;
BEGIN
    anchor:=NIL;
    wanted:=SIZE(pBuff^);
    i:=firstbuff-1;
    LOOP
        INC(i);
        IF i > lastbuff THEN EXIT; END;
        IF Storage.Available(wanted)=FALSE THEN
            IF DEBUG THEN WrStr("Storage.Available() returned FALSE !");WrLn;END;
            EXIT;
        END;
        Storage.ALLOCATE(buff[i],wanted);
        IF DEBUG THEN
            WrStr("# ");WrStr( fmtnum(10,2,i) );WrStr(" @ ");
        END;
        IF onsamepage(DEBUG,buff[i]) THEN anchor:=buff[i]; EXIT; END;
    END;
    DEC(i);
    last:=i;
    FOR i:= firstbuff TO last DO
        Storage.DEALLOCATE( buff[i] ,wanted);
    END;
    RETURN anchor;
END grabMem;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    WavHeaderType = RECORD
        rID               : ARRAY [0..3] OF CHAR; (* "RIFF" *)
        rLen              : LONGCARD;             (* remaining length after this header *)
        wID               : ARRAY [0..3] OF CHAR; (* "WAVE" *)
        fID               : ARRAY [0..3] OF CHAR; (* "fmt " *)
        fLen              : LONGCARD;             (* remaining chunk length after chunk "fmt "header *)
        wFormatTag        : CARDINAL;             (* 1 = pcm, ... *)
        nChannels         : CARDINAL;             (* 1=mono 2 = stereo *)
        nSamplesPerSec    : LONGCARD;             (* playback frequency *)
        nAvgBytesPerSec   : LONGCARD;             (* average bytes / second data should be sent at *)
                                                  (* nchannels * nSamplesPerSec * (nbitspersample/8) *)
        nBlockAlign       : CARDINAL;             (* nchannels * (nbitspersample/8) *)
        (* other data variable according to wFormatTag *)
    END;
    ChunkHeaderType = RECORD
        dID               : ARRAY [0..3] OF CHAR; (* "data" *)
        dLen              : LONGCARD;             (* remaining chunk length after header *)
    END;

PROCEDURE chkWavHeader (VAR h:soundType;  tf,terse,useLFN:BOOLEAN;S:pathtype):CARDINAL;
CONST
    PCM           = 1; (* we only handle this one *)
    WAVKLUDGE     = 4+4+4+4+4; (* [rID..fLen] *)
VAR
    w : WavHeaderType;
    chunk: ChunkHeaderType;
    hin:FIO.File;
    got:CARDINAL;
    R:str16;
    fsize:LONGCARD;
    rc:CARDINAL;
BEGIN
    hin:=fileOpenRead(useLFN,S);
    got:=FIO.RdBin(hin,w,SIZE(w));
    fsize:=FIO.Size(hin);

    rc:=errWAVheaderSize;
    IF got = SIZE(w) THEN
        rc:=errWAVheaderID;
        Str.Copy(R,w.rID);Str.Caps(R);
        IF same (R,"RIFF") THEN
            Str.Copy(R,w.wID);Str.Caps(R);
            IF same (R,"WAVE") THEN
                Str.Copy(R,w.fID);Str.Caps(R);
                IF same (R,"FMT ") THEN
                    FIO.Seek(hin,WAVKLUDGE+w.fLen); (* skip variable data *)
                    got := FIO.RdBin(hin,chunk,SIZE(chunk));
                    rc:=errWAVheaderSizeChunk;
                    IF got = SIZE(chunk) THEN
                        Str.Copy(R,chunk.dID);Str.Caps(R);
                        IF same (R,"DATA") THEN
                            rc:=errNone;
                        ELSE
                            rc:=errWAVheaderID; (* "FACT", etc. *)
                        END;
                    END;
                END;
            END;
        END;
    END;
    fileClose(useLFN,hin);
    IF rc # errNone THEN RETURN rc; END;

    terse := NOT(terse); (* terse is now showinfos ! we can use OR *)

    dmp(tf OR terse,"File",S);
    dmp(tf OR terse,"File size"         ,fmtbignum(10,9,fsize));

    dmp(tf OR terse,"Format"            ,fmtnum(10,9,w.wFormatTag));      (* fmt *)
    dmp(tf OR terse,"Channels"          ,fmtnum(10,9,w.nChannels));       (* channels *)
    dmp(tf OR terse,"Playback frequency",fmtbignum(10,9,w.nSamplesPerSec));  (* sampling *)
    dmp(tf OR terse,"Bytes per second"  ,fmtbignum(10,9,w.nAvgBytesPerSec)); (* frequency *)

    h.datapos  := WAVKLUDGE+w.fLen+SIZE(chunk); (* kludgy ! *)
    h.datasize := chunk.dLen;

    h.channels :=w.nChannels;
    h.frequency:=w.nAvgBytesPerSec;

    dmp(tf,"datapos"           ,fmtbignum(16,8,h.datapos));
    dmp(tf,"datasize"          ,fmtbignum(10,9,h.datasize));

    IF w.wFormatTag # PCM THEN RETURN errWAVformat; END;

    RETURN errNone;
END chkWavHeader;

(* ------------------------------------------------------------ *)

TYPE
    SBtype = RECORD
        SBaddr        : CARDINAL;
        SBdma         : CARDINAL;
        DSPWrite      : CARDINAL;
        DSPmixerIndex : CARDINAL;
        DSPmixerValue : CARDINAL;
        DSPReset      : CARDINAL;
        DSPRead       : CARDINAL;
        DSPAvailable  : CARDINAL;
        regaddr,regcount,regpage,regmask,regmode,regclear,regstatus:CARDINAL;
        terminal:SHORTCARD;
    END;

PROCEDURE initparmsDSP (VAR v : SBtype; addr : CARDINAL);
CONST
    addrSBmixerindex  = 04H;
    addrSBmixervalue  = 05H;
    addrSBreset       = 06H;
    addrSBread        = 0AH;
    addrSBwrite       = 0CH;
    addrSBavailable   = 0EH;
BEGIN
    v.DSPWrite     := addr + addrSBwrite;
    v.DSPReset     := addr + addrSBreset;
    v.DSPRead      := addr + addrSBread;
    v.DSPAvailable := addr + addrSBavailable;
    v.DSPmixerIndex:= addr + addrSBmixerindex;
    v.DSPmixerValue:= addr + addrSBmixervalue;
END initparmsDSP;

PROCEDURE initparmsDMAregisters(VAR v : SBtype; dma:CARDINAL);
VAR
    regaddr,regcount,regpage,regmask,regmode,regclear,regstatus:CARDINAL;
    terminal:SHORTCARD;
BEGIN
    CASE dma OF
    | 0 : regaddr:= 000H ; regcount:= 001H ; regpage:= 087H;
    | 1 : regaddr:= 002H ; regcount:= 003H ; regpage:= 083H;
    | 2 : regaddr:= 004H ; regcount:= 005H ; regpage:= 081H;
    | 3 : regaddr:= 006H ; regcount:= 007H ; regpage:= 082H;
    | 4 : regaddr:= 0C0H ; regcount:= 0C2H ; regpage:= 08FH;
    | 5 : regaddr:= 0C4H ; regcount:= 0C6H ; regpage:= 08BH;
    | 6 : regaddr:= 0C8H ; regcount:= 0CAH ; regpage:= 089H;
    | 7 : regaddr:= 0CCH ; regcount:= 0CEH ; regpage:= 08AH;
    END;
    CASE dma OF
    | 0,1,2,3 : regmask:= 00AH ; regmode:= 00BH ; regclear:= 00CH ; regstatus:= 008H;
    | 4,5,6,7 : regmask:= 0D4H ; regmode:= 0D6H ; regclear:= 0D8H ; regstatus:= 0D0H;
    END;
    CASE dma OF
    | 0,4 : terminal:= 001H;
    | 1,5 : terminal:= 002H;
    | 2,6 : terminal:= 004H;
    | 3,7 : terminal:= 008H;
    END;
    v.regaddr := regaddr;
    v.regcount:= regcount;
    v.regpage := regpage;
    v.regmask := regmask;
    v.regmode := regmode;
    v.regclear:= regclear;
    v.regstatus:=regstatus;
    v.terminal:= terminal;
END initparmsDMAregisters;

PROCEDURE initSBglobal (VAR v: SBtype; tf:BOOLEAN; addr,dma:CARDINAL);
BEGIN
    v.SBaddr := addr;
    v.SBdma  := dma;

    initparmsDSP(v,addr);
    initparmsDMAregisters(v, dma);

    (*
    dmp(tf,"SoundBlaster address",  fmtnum(16,4,v.SBaddr));
    dmp(tf,"SoundBlaster DMA",      fmtnum(16,4,v.SBdma));
    *)
    dmp(tf,"DSPWrite",              fmtnum(16,4,v.DSPWrite));
    dmp(tf,"DSPReset",              fmtnum(16,4,v.DSPReset));
    dmp(tf,"DSPRead",               fmtnum(16,4,v.DSPRead));
    dmp(tf,"DSPAvailable",          fmtnum(16,4,v.DSPAvailable));
    dmp(tf,"DSPmixerIndex",         fmtnum(16,4,v.DSPmixerIndex));
    dmp(tf,"DSPmixerValue",         fmtnum(16,4,v.DSPmixerValue));

    dmp(tf,"regaddr",               fmtnum(16,4,v.regaddr));
    dmp(tf,"regcount",              fmtnum(16,4,v.regcount));
    dmp(tf,"regpage",               fmtnum(16,4,v.regpage));
    dmp(tf,"regmask",               fmtnum(16,4,v.regmask ));
    dmp(tf,"regmode",               fmtnum(16,4,v.regmode ));
    dmp(tf,"regclear",              fmtnum(16,4,v.regclear));
    dmp(tf,"regstatus",             fmtnum(16,4,v.regstatus   ));
    dmp(tf,"terminal",              fmtnum(16,4,CARDINAL(v.terminal)));
END initSBglobal;

PROCEDURE chkBlaster (VAR SB:SBtype;tf,DEBUG:BOOLEAN):CARDINAL;
CONST
    sSBmodels="SB 1.x,SB Pro,SB 2.0,SB Pro 2.0,SB Pro MCV,SB 16 or AWE32";
    NOTFOUND = MAX(CARDINAL);
VAR
    SS,S,R : str128;
    i,rc,base,v : CARDINAL;
    lc:LONGCARD;
    ok:BOOLEAN;
    addr,dma,id,irq,dma16,midiport:CARDINAL;
BEGIN
    rc:=errSBnoBlasterVar;
    Lib.EnvironmentFind(sBLASTER,SS);
    Str.Caps(SS);
    IF same(SS,"") THEN RETURN rc; END;

    dmp(tf,sBLASTER+" variable",SS);

    (* init the only parms we're interested in with unreasonable values *)
    addr:=NOTFOUND;
    dma :=NOTFOUND;
    id  :=NOTFOUND;

    i:=0;
    LOOP
        isoleItemS(S,SS," ",i);
        IF same(S,"") THEN EXIT;END;
        dmp(DEBUG,sBLASTER+" parameter",S);
        CASE S[0] OF
        | "A","P":
            base:=16;
        ELSE
            base:=10;
        END;
        Str.Copy(R,S);
        Str.Delete(R,0,1);
        lc:=Str.StrToCard(R,base,ok);
        IF ok=FALSE THEN lc:=NOTFOUND;END;
        IF lc > MAX(CARDINAL) THEN lc:=NOTFOUND;END;
        v := CARDINAL(lc);
        CASE S[0] OF
        | "A":
                CASE v OF
                | 220H,240H,260H,280H: addr:=v; (* only valid addresses $02x0 where x=2,4,6,8 *)
                | 210H,230H,250H:      addr:=v; (* supposed to be legal too ! *)
                ELSE
                    RETURN errSBbadAddr;
                END;
        | "D":
                CASE v OF
                | 0,1,3: dma:=v; (* 0 is often reserved for RAM refresh ! *)
                ELSE
                    RETURN errSBbadDMA;
                END;
        | "T":
                CASE v OF
                | 1,2,3,4,5,6: id:=v;
                ELSE
                    RETURN errSBbadID;
                END;
        | "I":
                CASE v OF
                | 2,3,5,7,10:  irq:=v;
                ELSE
                    RETURN errSBbadIRQ;
                END;
        | "H":
                CASE v OF
                | 5,6,7:  dma16:=v;
                ELSE
                    RETURN errSBbadDMA16;
                END;
        | "P":
                CASE v OF
                | 300H,330H: midiport:=v;
                ELSE
                    RETURN errSBbadMidiPort;
                END;
        ELSE
                RETURN errSBunsupported; (* unknown/illegal BLASTER parm *)
        END;
        INC(i);
    END;

    IF addr=NOTFOUND THEN RETURN errSBbadAddr;END;
    IF dma =NOTFOUND THEN RETURN errSBbadDMA; END;
    IF id  =NOTFOUND THEN RETURN errSBbadID;  END;

    Str.Copy(S,sSBmodels);
    isoleItemS(R,S, ",", id-1);
    dmp(tf,"SoundBlaster type",R);

    initSBglobal(SB, tf,addr,dma);

    RETURN errNone;
END chkBlaster;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE ReadFromDSP (SB:SBtype):SHORTCARD;
BEGIN
    WHILE (SYSTEM.In(SB.DSPAvailable) AND SHORTCARD(80H)) = SHORTCARD(00H) DO (* till bit 7 = 1 *)
    END;
    RETURN ( SYSTEM.In(SB.DSPRead) );
END ReadFromDSP;

PROCEDURE WriteToDSP (SB:SBtype;  v : SHORTCARD);
BEGIN
    WHILE (SYSTEM.In(SB.DSPWrite) AND SHORTCARD(80H)) # SHORTCARD(00H) DO (* till bit 7 = 0 *)
    END;
    SYSTEM.Out(SB.DSPWrite,v);
END WriteToDSP;

PROCEDURE WriteDAC (SB:SBtype;  v:SHORTCARD);
BEGIN
    WriteToDSP (SB,10H); (* direct mode to DAC *)
    WriteToDSP (SB,v);   (* 8-bit sample *)
END WriteDAC;

PROCEDURE SBCardSpeakerOn(SB:SBtype);
BEGIN
    WriteToDSP (SB,0D1H);
END SBCardSpeakerOn;

PROCEDURE SBCardSpeakerOff(SB:SBtype);
BEGIN
    WriteToDSP (SB,0D3H);
END SBCardSpeakerOff;

PROCEDURE DMAstop (SB:SBtype );
BEGIN
    WriteToDSP (SB,0D0H);
END DMAstop;

PROCEDURE DMAcontinue (SB:SBtype);
BEGIN
    WriteToDSP (SB,0D4H);
END DMAcontinue;

PROCEDURE myDelay (v:CARDINAL);
BEGIN
    Lib.Delay(v);
END myDelay;

PROCEDURE SBCardReset (SB:SBtype ):BOOLEAN;
CONST
    PAUSE = 150; (* was 150 i.e. 0,15 seconds though 3 microseconds should do *)
    DSPREADY = 0AAH;
BEGIN
    SYSTEM.Out (SB.DSPReset,1);
    myDelay(PAUSE);
    SYSTEM.Out (SB.DSPReset,0);
    myDelay(PAUSE);
    RETURN ( ReadFromDSP(SB) = DSPREADY );
END SBCardReset;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE openSB (SB:SBtype):BOOLEAN;
BEGIN
    IF SBCardReset(SB)=FALSE THEN RETURN FALSE; END;
    SBCardSpeakerOn(SB);
    RETURN TRUE;
END openSB;

PROCEDURE closeSB (SB:SBtype ):BOOLEAN;
BEGIN
    SBCardSpeakerOff(SB);
    RETURN SBCardReset(SB);
END closeSB;

PROCEDURE PlaybackDone(SB:SBtype):BOOLEAN;
VAR
    dummy:SHORTCARD;
BEGIN
    (* loop until terminal count bit set to 1 *)

    IF ( SYSTEM.In(SB.regstatus) AND SB.terminal) = SB.terminal THEN
	    (* acknowledge the DSP interrupt by reading the DATA AVAILABLE port once *)
	    dummy:=SYSTEM.In (SB.DSPAvailable);
        DMAstop(SB);     (* safety *)
	    (* send EOI to the interrupt controller port 20h just in case *)
        (* SYSTEM.Out (020H, 020H); *)
        RETURN TRUE; (* let closeSB handle voice off and reset *)
    ELSE
	    RETURN FALSE;
    END;
END PlaybackDone;

PROCEDURE doPlayback (soundinfo:soundType ; SB:SBtype);
CONST
    THRESHOLD = LONGCARD(23000); (* threshold for fast mode *)
VAR
    dsize:CARDINAL;
    losize,hisize,loaddr,hiaddr,stringpageaddr:SHORTCARD;
    segment,offset,stringaddrword:CARDINAL;
    timeconstant:SHORTCARD;
    dummy:SHORTCARD;
    fastmode:BOOLEAN;
    a : FarADDRESS;
    so,stringaddr : LONGCARD;
BEGIN
    dsize          := soundinfo.wanted; (* what we got from file *)
    IF dsize < 2 THEN dsize:=2; END;
    DEC(dsize);
    losize         := SHORTCARD(dsize MOD 256);
    hisize         := SHORTCARD(dsize DIV 256);

    a              := FarADDRESS(soundinfo.anchor);
    so             := LONGCARD(a);             (* automagically ssss:oooo *)
    splitaddr(so,segment,offset);
    stringaddr:=so2addr(so);                   (* ssss:oooo to linear *)

    stringpageaddr := SHORTCARD(stringaddr >> 16);
    stringaddrword := CARDINAL (stringaddr AND 0000FFFFH);
    loaddr         := SHORTCARD(stringaddrword AND 00FFH);
    hiaddr         := SHORTCARD(stringaddrword >> 8);

    SYSTEM.Out(SB.regmask,(SHORTCARD(04H) + SHORTCARD(SB.SBdma MOD 4))); (* cmd=set channel before programming it *)
    SYSTEM.Out(SB.regclear,00H);                                         (* reset internal pointers *)
    SYSTEM.Out(SB.regmode,(SHORTCARD(48H) + SHORTCARD(SB.SBdma MOD 4))); (* set mode to read & signal *)
    SYSTEM.Out(SB.regaddr, loaddr);         (* bits 0-7 of the 20bit address *)
    SYSTEM.Out(SB.regaddr, hiaddr);         (* bits 8-15 of the 20bit address *)
    SYSTEM.Out(SB.regpage, stringpageaddr); (* bits 16-19 of the 20bit address *)
    SYSTEM.Out(SB.regcount, losize);        (* bits 0-7 of size *)
    SYSTEM.Out(SB.regcount, hisize);        (* bits 8-16 of size *)
    SYSTEM.Out(SB.regmask,(SHORTCARD(00H) + SHORTCARD(SB.SBdma MOD 4))); (* cmd=enable channel *)

    (* init DAC here *)

    (* FROM dma.c
        // The time constant equation is basically this:
        //   TimeConst = 256 - 1000000/(Channels * SampPerSec)
        // but to round to the nearest time constant, we effectively
        // add 0.5 by multiplying by 2, adding 1, then dividing by 2.
        //  Then TimeConst = 256 -   (2000000/ (Channels * SampPerSec)+ 1)   / 2
        TimeConstant = 256 - ((2000000L / (Channels * SampPerSec) + 1) >> 1);
    *)

    (* set playback frequency *)
    (* timeconstant := 256 - SHORTCARD( 1000000 DIV soundinfo.frequency); *)
    timeconstant := 256 - SHORTCARD( (2000000 DIV ( LONGCARD(soundinfo.channels)*soundinfo.frequency)+1)>>1);
    WriteToDSP (SB,40H);
    WriteToDSP (SB,timeconstant);

    SYSTEM.Out(SB.DSPmixerIndex,00H); (* reset mixer chip before changes *)
    SYSTEM.Out(SB.DSPmixerValue,00H);
    CASE soundinfo.channels OF
    | 1 :
        SYSTEM.Out(SB.DSPmixerIndex,0EH);
        SYSTEM.Out(SB.DSPmixerValue,SHORTCARD(00H)); (* bypass O/P filter and mono *)
    | 2 :
        SYSTEM.Out(SB.DSPmixerIndex,0EH);
        SYSTEM.Out(SB.DSPmixerValue,SHORTCARD(02H)); (* stereo -- or 20H ? *)
    END;

    fastmode:=(soundinfo.frequency > THRESHOLD);

    IF fastmode THEN
        WriteToDSP (SB,48H); (* set playback type to high speed SBpro *)
    ELSE
        WriteToDSP (SB,14H); (* set playback type to 8-bit DAC -- normal single cycle *)
    END;

    (* write lsb and msb of datalength-1 *)
    WriteToDSP (SB,losize);
    WriteToDSP (SB,hisize);

    IF fastmode THEN WriteToDSP (SB,91H); END; (* high Speed DMA mode 8-bit *)

    (* make sure DMA is going *)
    dummy:=SYSTEM.In(SB.regstatus);

    LOOP
        IF PlaybackDone(SB) THEN EXIT; END;
    END;
END doPlayback;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    firstparm = 1;
    maxparm   = 1;
VAR
    parmcount,i,opt,lastparm:CARDINAL;
    S,R:str256;
    spec:pathtype;
    useLFN,terse,about,DEBUG:BOOLEAN;
    parm:ARRAY [firstparm..maxparm] OF str256;
    cmd : (play,show);
VAR
    SB:SBtype;
    soundinfo:soundType;
    hin:FIO.File;
    got:CARDINAL;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE; (* don't let topspeed handle problems *)
    WrLn;

    parmcount := Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,"");END;

    lastparm    := firstparm-1;
    useLFN      := TRUE;
    terse       := FALSE;
    about       := FALSE;
    DEBUG       := FALSE;
    cmd         := play;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "L"+delim+"LFN"+delim+
                                  "P"+delim+"SB"+delim+
                                  "T"+delim+"TERSE"+delim+
                                  "I"+delim+"INFOS"+delim+
                                  "DEBUG"
                              );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5:    useLFN    := FALSE;
            | 6,7:    cmd       := show;
            | 8,9:    terse     := TRUE;
            |10,11:   about     := TRUE;
            |12:      DEBUG     := TRUE;
            ELSE
                abort(errOption,S);
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxparm THEN abort(errTooManyParms,S);END;
            Str.Copy(parm[lastparm],S); (* keep case *)
        END;
    END;
    CASE cmd OF
    | show:
        i:=chkBlaster(SB,TRUE,DEBUG);
        abort(i,"");
    END;

    (* cmd is play *)

    IF w9XsupportLFN() THEN (* FIXME *)
        WrStr("::: This program will not run from Windows 9X ! :-("+nl);
        abort(errSafetyFirst,""); (* better safe than sorry *)
    END;

    IF lastparm < firstparm THEN abort(errMissingSpec,""); END;

    IF (about AND terse) THEN abort(errExclusive,"-i and -t");END;

    useLFN:=( useLFN AND w9XsupportLFN() );
    IF useLFN=FALSE THEN
        FOR i:=firstparm TO lastparm DO
            UpperCase( parm[i] );
        END;
    END;
    Str.Copy(spec,parm[firstparm]);

    IF Str.RCharPos(spec,dot)=MAX(CARDINAL) THEN Str.Append(spec,extWAV);END;
    IF chkJoker(spec) THEN abort(errJoker,spec);END;
    IF fileIsDirectorySpec(useLFN,spec) THEN abort(errNotFile,spec);END;
    IF fileExists(useLFN,spec)=FALSE THEN abort(errNotFound,spec);END;

    i:=chkBlaster(SB,DEBUG,DEBUG);
    IF i # errNone THEN abort(i,"");END;

    i:=chkWavHeader(soundinfo,DEBUG,terse,useLFN,spec);
    IF i #errNone THEN abort(i,spec);END;

    IF about THEN abort(errNone,"");END;

    (* let's play ! *)

    soundinfo.anchor:=grabMem(DEBUG);
    IF soundinfo.anchor = NIL THEN abort(errALLOCATEPAGE,"soundinfo.anchor");END;

    IF openSB(SB)=FALSE THEN abort(errSBreset,"");END;

    FIO.EOF:=FALSE;
    hin:=fileOpenRead(useLFN,spec);
    FIO.AssignBuffer(hin,ioBufferIn); (* useless for small files but... *)
    FIO.Seek(hin,soundinfo.datapos);

    soundinfo.remaining := soundinfo.datasize;
    LOOP
        IF soundinfo.remaining = 0 THEN EXIT; END;
        IF soundinfo.remaining > dataBufferSize THEN
            soundinfo.wanted := dataBufferSize;
            DEC(soundinfo.remaining,dataBufferSize);
        ELSE
            soundinfo.wanted := CARDINAL(soundinfo.remaining);
            soundinfo.remaining:=0;
        END;
        got:=FIO.RdBin(hin,soundinfo.anchor^,soundinfo.wanted);
        IF got = 0 THEN EXIT; END;
        doPlayback(soundinfo,SB);
        IF got # soundinfo.wanted THEN EXIT; END;
        IF ChkEscape() THEN EXIT; END;
    END;
    fileClose(useLFN,hin);

    releaseMem(soundinfo.anchor);

    IF closeSB(SB)=FALSE THEN abort(errSBreset,"");END;

    abort(errNone,"");
END playWAV.
