(* ---------------------------------------------------------------
Title         Q&D Date/Time/Attributes Synchronizer
Author        PhG
Overview      an ersatz for TOUCH & ATTRIB minus recursion plus synchro
Usage         see help
Notes
Bugs
Wish List     recursion ? could be dangerous

--------------------------------------------------------------- *)

MODULE dtSync;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT IO;
IMPORT SYSTEM;

FROM IO IMPORT WrStr,WrLn;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileGetFileStamp,
fileIsDirectorySpec, fileClose;

(* ------------------------------------------------------------ *)

CONST
    cr                = CHR(13);
    lf                = CHR(10);
    nl                = cr+lf;
    dash              = "-";
    colon             = ":";
    dot               = ".";
    star              = "*";
    question          = "?";
    antislash         = "\";
    dotdot            = dot+dot;
    netslash          = antislash+antislash;
    strdigits         = "0123456789";
    stralphabet       = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    slash             = "/";
    semicolon         = ";";
    pound             = "#";
    arobas            = "@";
    dquote            = '"';
    equal             = "=";
    nullchar          = 0C;
    stardotstar       = star+dot+star;
    nowplaceholder    = star;
CONST
    extLST            = ".LST";
    sUNDOBATCH        = "ROLLBACK.BAT";
    sDMYrestampDefault= "1-Feb-2063";
    sHMSrestampDefault= "00:00:00";
CONST
    mintimeinterval   = LONGCARD(2);
    maxtimeinterval   = LONGCARD(10*60);
    defaulttimeinterval= mintimeinterval;
    sIntervalRange    = "[2..600]"; (* 2 seconds to 10 minutes *)
CONST
    specSWP           = "*.SWP";
    specPAR           = "*.PAR";
    specPAGEFILE      = "PAGEFILE.SYS";
    specROLLBACK      = sUNDOBATCH;
    specPAGEFILEpat   = "*\"+specPAGEFILE;
    specROLLBACKpat   = "*\"+specROLLBACK;
    firstpat          = 1;
    lastpat           = 6;
TYPE
    attrstatetype = (dontcare,required,unwanted);
    masktype = RECORD (* D R H S A *)
        stateD,stateR,stateH,stateS,stateA : attrstatetype;
    END;
CONST
    firstattr      = 1;
    lastattr       = 5;
    allowedAttr    = "DRHSA"; (* 1..5 *)

CONST
    ProgEXEname   = "DTSYNC";
    ProgTitle     = "Q&D Date/Time/Attributes Synchronizer";
    ProgVersion   = "v1.1c";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;

CONST
    errNone             = 0;
    errHelp             = 1;
    errUnknownOption    = 2;
    errParmOverflow     = 3;
    errDosVersion       = 4;
    errBadDate          = 5;
    errBadTime          = 6;
    errSyntaxOption     = 7;
    errCmdConflict      = 8;
    errSourceJokers     = 9;
    errSourceBadName    = 10;
    errSourceNoMatch    = 11;
    errJokerList        = 12;
    errNotFile          = 13;
    errNotFound         = 14;
    errJokerPath        = 15;
    errNoMatch          = 16;
    errTooMany          = 17;
    errTargetBadName    = 18;
    errNoEntry          = 19;
    errTooManyEntries   = 20;
    errListBadName      = 21;
    errJokerInEntry     = 22;
    errMask             = 23;
    errSyntax           = 24;
    errOpening          = 25;
    errEntryIsDir       = 26;
    errBadVal           = 27;
    errInterval         = 28;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    spec="<target(s)|@filelist["+extLST+"]>";
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" <source> "+spec+" [option]..."+nl+
"Syntax 2 : "+ProgEXEname+" <"+nowplaceholder+"|date> <"+nowplaceholder+"|time> "+spec+" [option]..."+nl+
"Syntax 3 : "+ProgEXEname+" [-d:$] "+spec+" [option]..."+nl+
"Syntax 4 : "+ProgEXEname+" <-m:$> "+spec+" [option]..."+nl+
nl+
"This program stamps file(s) with <source> date and time (syntax 1)"+nl+
"or with specified date and time (syntax 2) ; synchronizes file(s) to same date"+nl+
"while incrementing file(s) time from 00:00:00 to 23:59:58"+nl+
"in directory or filelist order (syntax 3) ; changes attributes (syntax 4)."+nl+
nl+
"    -d:$  base date (dd-mm-yyyy format, default is "+sDMYrestampDefault+")"+nl+
"    -m:$  set attributes (<"+allowedAttr+">[-|+|?|!|x]...)"+nl+
"    -q    terse display"+nl+
"    -v    verbose display"+nl+
"    -r    process read-only files instead of skipping them (forced with -m:$)"+nl+
"    -c    add 2000 to year [0..79] and add 1900 to year [80..99]"+nl+
"          (default is to add 1900 to year [0..99])"+nl+
"    -i:#  increment ("+sIntervalRange+" seconds, default is 2)"+nl+
"    -b|-u create undo "+sUNDOBATCH+" batch"+nl+
"    -y[y] actually update data (-yy = -y -b)"+nl+
"    -x    disable LFN support even if available"+nl+
nl+
"a) Unless -y[y] was specified, program will NOT actually update data."+nl+
'b) Date format is dd-mm-yy[yy], time format is hh:mm[:ss], "'+nowplaceholder+'" means now.'+nl+
"c) A filelist should contain either canonical pathnames,"+nl+
"   or mere filenames to be searched for in current directory."+nl+
"   Jokers are not allowed in filelist entries."+nl+
'd) "'+specSWP+'", "'+specPAR+'", "'+specPAGEFILEpat+'", "'+specROLLBACKpat+'" will not be processed.'+nl;

VAR
    S  : str1024; (* we may get a LFN *)
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errUnknownOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errParmOverflow:
        Str.Concat(S,einfo," is one parameter too many !");
    | errDosVersion:
        Str.Concat(S,"DOS ",einfo);Str.Append(S," or later is required !");
    | errBadDate:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," date !");
    | errBadTime:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," time !");
    | errSyntaxOption:
        Str.Concat(S,einfo," option is a nonsense with specified syntax !");
    | errCmdConflict:
        Str.Concat(S,"Syntax conflict involving ",einfo);Str.Append(S," option !");
    | errSourceJokers:
        Str.Concat(S,einfo," <source> cannot contain any joker !");
    | errSourceBadName :
        Str.Concat(S,einfo," <source> is not legal !");
    | errSourceNoMatch :
        Str.Concat(S,einfo," <source> does not exist !");
    | errJokerList:
        S := "Jokers are not allowed in <filelist["+extLST+"]> !";
    | errNotFile:
        Str.Concat(S,einfo," looks like a directory !");
    | errNotFound:
        Str.Concat(S,einfo," does not exist !");
    | errJokerPath:
        S := "Jokers are not allowed in <filelist["+extLST+"]> path !";
    | errNoMatch:
        Str.Concat(S,"No match for ",einfo); Str.Append(S," <target> !");
    | errTooMany:
        Str.Concat(S,"Too many files match ",einfo); Str.Append(S," <target> !");
    | errTargetBadName :
        Str.Concat(S,einfo," <target> is not legal !");
    | errNoEntry:
        Str.Concat(S,"No entry in ",einfo); Str.Append(S," <filelist> !");
    | errTooManyEntries:
        Str.Concat(S,"Too many entries in ",einfo); Str.Append(S," <filelist> !");
    | errListBadName :
        Str.Concat(S,einfo," <filelist> is not legal !");
    | errJokerInEntry:
        Str.Concat(S,'Illegal joker(s) in "',einfo);Str.Append(S,'" <filelist> entry !');
    | errMask  :
        Str.Concat(S,"Illegal or illogical ",einfo);Str.Append(S," attributes !");
    | errSyntax:
        S:="Syntax error !"; (* now, call me laconic : "this is spartan !" *)
    | errOpening :
        Str.Concat(S,"Error while opening ",einfo); Str.Append(S," <source> !");
    | errEntryIsDir:
        Str.Concat(S,'"',einfo);Str.Append(S,'" <filelist> entry looks like a directory !');
    | errBadVal:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," value !");
    | errInterval:
        Str.Concat(S,einfo," interval is out of "+sIntervalRange+" valid range !");
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ; (* nada *)
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

CONST
    ioBufferSize      = (8 * 512) + FIO.BufferOverhead;
    firstioBufferByte = 1;
    lastioBufferByte  = ioBufferSize;
TYPE
    ioBufferType  = ARRAY [firstioBufferByte..lastioBufferByte] OF BYTE;
VAR
    ioBuffer : ioBufferType;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE AtLeastDosVersion (minmajor,minminor:CARDINAL) : BOOLEAN;
VAR
    R             : SYSTEM.Registers;
    minDosVersion : CARDINAL;
    major         : CARDINAL;
    minor         : CARDINAL;
    thisDosVersion: CARDINAL;
BEGIN
    minDosVersion := (minmajor << 8) + minminor;
    R.AX := 3000H;
    Lib.Dos(R);
    major := CARDINAL(R.AL);
    minor := CARDINAL(R.AH);
    thisDosVersion := (major << 8) + minor;
    IF thisDosVersion < minDosVersion THEN RETURN FALSE; END;
    RETURN TRUE;
END AtLeastDosVersion;

PROCEDURE chkAnyJoker (justinpath:BOOLEAN;spec:pathtype  ):BOOLEAN;
VAR
    S,u,d,n,e:pathtype;
BEGIN
    IF justinpath THEN
        Lib.SplitAllPath(spec,u,d,n,e);
        Lib.MakeAllPath(S,u,d,"","");
    ELSE
        Str.Copy(S,spec);
    END;
    RETURN chkJoker(S);
END chkAnyJoker;

PROCEDURE chkValidName (S:ARRAY OF CHAR ) : BOOLEAN;
VAR
    i : CARDINAL;
BEGIN
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN RETURN FALSE; END;
    (* yes, there are lost of other things to check... but who cares ? *)
    RETURN TRUE;
END chkValidName;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* some code adapted from older DTCHK *)

PROCEDURE using (n : CARDINAL; digits : CARDINAL; pad : CHAR) : str80;
VAR
    ok   : BOOLEAN;
    v    : LONGCARD;
    len  : CARDINAL;
    S    : str80;
BEGIN
    v := LONGCARD(n);
    Str.CardToStr(v,S,10,ok);
    len := Str.Length(S);
    LOOP
        IF Str.Length(S) >= digits THEN EXIT; END;
        Str.Prepend(S,pad);
    END;
    RETURN S;
END using;

TYPE
    dttype = RECORD
        CASE : BOOLEAN OF
        | TRUE  :
            t : CARDINAL; (* hms is low  *)
            d : CARDINAL; (* ymd is high *)
        | FALSE :
            dt : LONGCARD;
        END;
    END;

TYPE
    datetype = RECORD
        day   : CARDINAL;
        month : CARDINAL;
        year  : CARDINAL; (* 1980 is already added *)
        dayOfWeek : Lib.DayType;
    END;
    timetype = RECORD
        hours   : CARDINAL;
        minutes : CARDINAL;
        seconds : CARDINAL;
    END;

(*
Year stored relative to 1980 (ex. 1988 stores as 8)
    year      month    day   

 F E D C B A 9 8 7 6 5 4 3 2 1 0   <-- Bit Number

Seconds are 0 to 29 -- DOS stores nearest even / 2
  hours    minutes   seconds 

 F E D C B A 9 8 7 6 5 4 3 2 1 0   <-- Bit Number
*)

CONST
    DOSbaseyear  = 1980;
CONST
    yyMask=BITSET{9..15};    yyShft=9;
    moMask=BITSET{5..8};     moShft=5;
    ddMask=BITSET{0..4};     ddShft=0;
CONST
    hhMask=BITSET{11..15};   hhShft=11;
    miMask=BITSET{5..10};    miShft=5;
    ssMask=BITSET{0..4};     ssShft=0;

PROCEDURE packDate (d,m,y : CARDINAL  ) : CARDINAL;
BEGIN
    y := (y - DOSbaseyear) << yyShft; (* 1980 *)
    m :=                 m << moShft;
    RETURN (y + m + d);
END packDate;

PROCEDURE packTime (h,m,s:CARDINAL  ) : CARDINAL;
BEGIN
    h := h << hhShft;
    m := m << miShft;
    s := s >> 1;
    RETURN (h + m + s);
END packTime;

PROCEDURE unpackDate (ymd:CARDINAL;VAR y,m,d:CARDINAL);
BEGIN
    y :=  CARDINAL(BITSET(ymd) * yyMask) >> yyShft ;
    m :=  CARDINAL(BITSET(ymd) * moMask) >> moShft ;
    d :=  CARDINAL(BITSET(ymd) * ddMask) >> ddShft ;
    INC(y,DOSbaseyear);
END unpackDate;

PROCEDURE unpackTime (hms:CARDINAL;VAR h,m,s:CARDINAL);
BEGIN
    h :=  CARDINAL(BITSET(hms) * hhMask) >> hhShft ;
    m :=  CARDINAL(BITSET(hms) * miMask) >> miShft ;
    s :=  CARDINAL(BITSET(hms) * ssMask) >> ssShft ;
    s := s << 1; (* yes, yes, "* 2" works too... *)
END unpackTime;

PROCEDURE PackDateTime (d:datetype;t:timetype) : dttype;
VAR
    dt : dttype;
BEGIN
    dt.d := packDate(d.day,d.month,d.year);
    dt.t := packTime(t.hours,t.minutes,t.seconds);
    RETURN dt;
END PackDateTime;

PROCEDURE fmtDate (plain:BOOLEAN;datedata:CARDINAL):str16;
CONST
    separator = dash;
    pad       = "0";
    tmonths   = "Jan Fv Mar Avr Mai Jun Jui Ao Sep Oct Nov Dc ???";
    tmonths2  = "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ???";
VAR
    y :CARDINAL;
    m :CARDINAL;
    d :CARDINAL;
    R,S :str16; (* "##-###-####" *)
BEGIN
    unpackDate(datedata, y,m,d);

    Str.Concat(R, using(d,2,pad), separator);
    IF plain THEN
        Str.ItemS(S,tmonths2," ",m-1);
    ELSE
        Str.Copy(S,using(m,2,pad));
    END;
    Str.Append(R,S);
    Str.Append(R,separator);
    Str.Append(R, using( y,4,pad ));
    RETURN R;
END fmtDate;

PROCEDURE fmtTime (complete:BOOLEAN;timedata:CARDINAL):str16;
CONST
    separator = colon;
    pad       = "0";
VAR
    h,m,s : CARDINAL;
    R:str16; (* "##:##:##" *)
BEGIN
    unpackTime(timedata, h,m,s);

    Str.Concat(R, using(h,2,pad), separator);
    Str.Append(R, using(m,2,pad));
    IF complete THEN
        Str.Append(R, separator);
        Str.Append(R, using(s,2,pad));
    END;
    RETURN R;
END fmtTime;

(* ------------------------------------------------------------ *)

CONST
    fixnone = 0;
    fix1900 = 1;
    fix2000 = 2;

PROCEDURE parseDate (VAR date:datetype; yearfix:CARDINAL; S:ARRAY OF CHAR):BOOLEAN;
CONST
    separator=dash;
    legaldateset = strdigits+separator+stralphabet;
    mindd=1;
    maxdd=31;
    minmm=1;
    maxmm=12;
    minyy=DOSbaseyear; (* 1980 = base year for messdos *)
    maxyy=2099;
CONST
    century1900 = 1900;
    century2000 = 2000;
VAR
    i : CARDINAL;
    R : str80;
    v : LONGCARD;
    ok: BOOLEAN;
BEGIN
    UpperCase(S); (* in case months would be letters *)
    ReplaceChar(S,slash,separator);
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legaldateset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    IF CharCount(S,separator) # 2 THEN RETURN FALSE; END;

    Str.ItemS(R,S,separator,0);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < mindd) OR (v > maxdd) THEN RETURN FALSE; END;
    date.day := CARDINAL(v);

    Str.ItemS(R,S,separator,1);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN
        Str.Prepend(R,dash); (* fake command line parameter ! *)
        i := GetOptIndex(R,"JAN"+delim+"JAN"+delim+
                           "FEB"+delim+"FEV"+delim+
                           "MAR"+delim+"MAR"+delim+
                           "APR"+delim+"AVR"+delim+
                           "MAY"+delim+"MAI"+delim+
                           "JUN"+delim+"JUN"+delim+
                           "JUL"+delim+"JUI"+delim+
                           "AUG"+delim+"AOU"+delim+
                           "SEP"+delim+"SEP"+delim+
                           "OCT"+delim+"OCT"+delim+
                           "NOV"+delim+"NOV"+delim+
                           "DEC"+delim+"DEC");
        CASE i OF
        | 1..24 :
            v := LONGCARD(i+1) DIV 2;
        ELSE
            RETURN FALSE;
        END;
    END;
    IF (v < minmm) OR (v > maxmm) THEN RETURN FALSE; END;
    date.month := CARDINAL(v);

    Str.ItemS(R,S,separator,2);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    CASE yearfix OF
    | fixnone : ;
    | fix2000 :
        IF v < 100 THEN
            IF v < 80 THEN
                INC(v,century2000);
            ELSE
                INC(v,century1900);
            END;
        END;
    | fix1900 :
        IF v < 100 THEN INC(v,century1900); END;
    END;

    IF (v < minyy) OR (v > maxyy) THEN RETURN FALSE; END;
    date.year := CARDINAL(v);
    RETURN TRUE;
END parseDate;

PROCEDURE parseTime (VAR time : timetype; S : ARRAY OF CHAR) : BOOLEAN;
CONST
    separator=colon;
    legaltimeset = strdigits+separator;
    minhh=0;
    maxhh=23;
    minmm=0;
    maxmm=59;
    minss=0;
    maxss=59;
VAR
    i : CARDINAL;
    R : str80;
    v : LONGCARD;
    ok: BOOLEAN;
BEGIN
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legaltimeset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    i := CharCount(S,separator);
    IF (i # 1) AND (i # 2) THEN RETURN FALSE; END;

    Str.ItemS(R,S,separator,0);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < minhh) OR (v > maxhh) THEN RETURN FALSE; END;
    time.hours := CARDINAL(v);

    Str.ItemS(R,S,separator,1);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < minmm) OR (v > maxmm) THEN RETURN FALSE; END;
    time.minutes := CARDINAL(v);

    CASE i OF
    | 1 :
        time.seconds := minss;
    | 2 :
        Str.ItemS(R,S,separator,2);
        v := Str.StrToCard(R,10,ok);
        IF ok=FALSE THEN RETURN FALSE; END;
        IF (v < minss) OR (v > maxss) THEN RETURN FALSE; END;
        time.seconds := CARDINAL(v);
    END;

    RETURN TRUE;
END parseTime;

PROCEDURE getDateNow (VAR d : datetype);
VAR
    dayOfWeek : Lib.DayType;
BEGIN
    Lib.GetDate(d.year,d.month,d.day,dayOfWeek);
END getDateNow;

(* ignore hundredth ! *)

PROCEDURE getTimeNow (VAR t : timetype);
VAR
    hundredth : CARDINAL;
BEGIN
    Lib.GetTime(t.hours,t.minutes,t.seconds,hundredth);
END getTimeNow;

(* ------------------------------------------------------------ *)

(* see GetLongCard() in QD_Box *)

PROCEDURE removeOption (VAR R:ARRAY OF CHAR);
VAR
    p:CARDINAL;
BEGIN
    Str.Subst(R,equal,colon); (* command line option xxx= becomes xxx: *)
    p := Str.CharPos(R,colon);
    IF p # MAX(CARDINAL) THEN Str.Delete(R,0,p+1); END;
END removeOption;

PROCEDURE parseMask (S:ARRAY OF CHAR; VAR m:masktype):BOOLEAN;
VAR
    included,excluded,indifferent:str16;
    i,len:CARDINAL;
    ch,keepch:CHAR;
    state:(wait,grab);
    rc:BOOLEAN;
    attrstate:attrstatetype;
BEGIN
    removeOption(S);

    (* accept ?+, ?-, ?x, ? (assuming ?+) : exemple A+R-H+ *)

    len:=Str.Length(S);
    FOR i:= 1 TO len DO
        ch:=S[i-1];
        IF Belongs(allowedAttr,ch) THEN
            IF CharCount(S,ch) > 1 THEN RETURN FALSE; END; (* avoid repeated or contradictory *)
        END;
    END;
    included := "";
    excluded := "";
    indifferent:="";
    i:=1;
    rc:=FALSE;
    state:=wait;
    LOOP
        IF i > len THEN rc:=TRUE; EXIT; END;
        ch:=S[i-1];
        CASE state OF
        | wait:
            CASE ch OF
            | "D","R","H","S","A" : keepch:=ch;
                IF (i+1) > len THEN Str.Append(included,keepch);END;
            ELSE
                EXIT;
            END;
            state:=grab;
        | grab:
            CASE ch OF
            | "+"         : Str.Append(included,keepch);
            | "-"         : Str.Append(excluded,keepch);
            | "?","!","X" : Str.Append(indifferent,keepch);
            | "D","R","H","S","A": Str.Append(included,keepch); (* fake "+" for previous flag *)
                DEC(i); (* but cancel next advance *)
            ELSE
                EXIT;
            END;
            state:=wait;
        END;
        INC(i);
    END;
    (* included and excluded don't share any attribute *)
    FOR i:=firstattr TO lastattr DO
        ch:=allowedAttr[i-1];
        IF Belongs(included,ch) THEN
            attrstate:=required;
        ELSIF Belongs(excluded,ch) THEN
            attrstate:=unwanted;
        ELSIF Belongs(indifferent,ch) THEN
            attrstate:=dontcare;
        ELSE
            attrstate:=dontcare;
        END;
        CASE ch OF
        | "D" : m.stateD := attrstate;
        | "R" : m.stateR := attrstate;
        | "H" : m.stateH := attrstate;
        | "S" : m.stateS := attrstate;
        | "A" : m.stateA := attrstate;
        END;
    END;
    RETURN rc;
END parseMask;

PROCEDURE fmtMask (pack:BOOLEAN;m:masktype):str16;
VAR
    R:str16;
    i:CARDINAL;
    ch:CHAR;
    attrstate:attrstatetype;
BEGIN
    R:="";
    FOR i:=firstattr TO lastattr DO
        ch:=allowedAttr[i-1];
        CASE ch OF
        | "D" : attrstate:=m.stateD;
        | "R" : attrstate:=m.stateR;
        | "H" : attrstate:=m.stateH;
        | "S" : attrstate:=m.stateS;
        | "A" : attrstate:=m.stateA;
        END;
        Str.Append(R,ch);
        CASE attrstate OF
        | dontcare: ch:="?";
        | required: ch:="+";
        | unwanted: ch:="-";
        END;
        Str.Append(R,ch);
        Str.Append(R," ");
    END;
    Rtrim(R," ");
    IF pack THEN ReplaceChar(R," ","");END;
    (* Str.Lows(R); *)
    RETURN R;
END fmtMask;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE isReservedPattern (S:ARRAY OF CHAR):BOOLEAN;
VAR
    i   : CARDINAL;
    pat : str16; (* "*\"+8+1+3 = 14 *)
BEGIN
    FOR i:=firstpat TO lastpat DO
        CASE i OF
        | 1 : pat:=specSWP;
        | 2 : pat:=specPAR;
        | 3 : pat:=specPAGEFILE;    (*   $ *)
        | 4 : pat:=specROLLBACK;    (*   $ *)
        | 5 : pat:=specPAGEFILEpat; (* *\$ *)
        | 6 : pat:=specROLLBACKpat; (* *\$ *)
        END;
        IF Str.Match(S,pat) THEN RETURN TRUE; END; (* Str.Match is not case sensitive *)
    END;
    RETURN FALSE;
END isReservedPattern;

PROCEDURE isReservedEntry (S : ARRAY OF CHAR) : BOOLEAN; (* assume uppercase *)
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    IF same(S,dotdot) THEN RETURN TRUE; END;
    RETURN isReservedPattern(S);
END isReservedEntry;

CONST
    firstEntry = 1; (* 1..count *)
TYPE
    ptrToEntry = POINTER TO entryType;
    entryType = RECORD
        next   : ptrToEntry;
        slen   : CARDINAL; (* a SHORTCARD would do fine in almost all cases  *)
        string : CHAR;      (* here, after other data, because variable length *)
    END;

PROCEDURE buildMatchList (VAR anchor:ptrToEntry;
                          useLFN:BOOLEAN;spec:pathtype):CARDINAL;
VAR
    S,u,d,n,e,dirbase,filespec:pathtype; (* some are oversized but safety first ! *)
    newInList : ptrToEntry;
    lastEntry,len,needed : CARDINAL;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    DOSentry     : FIO.DirEntry;
    rc,found:BOOLEAN;
    dosattr:FIO.FileAttr;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Lib.MakeAllPath(dirbase,u,d,"","");
    fixDirectory(dirbase);
    Lib.MakeAllPath(filespec,"","",n,e);

    lastEntry:=firstEntry-1;
    anchor:=NIL;

    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,DOSentry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(S,w9Xentry.fullfilename);
        ELSE
            Str.Copy(S,DOSentry.Name);
        END;
        IF isReservedEntry(S) THEN (* skip "." ".." "*.SWP" "*.PAR" *)
            ; (* silently ignore this spec *)
        ELSE
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=DOSentry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                INC(lastEntry);
                IF lastEntry=MAX(CARDINAL) THEN
                    IF useLFN THEN rc:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL);
                END; (* too many files but let's fake ALLOCATE failure *)
                Str.Prepend(S,dirbase); (* yes, yes, we know it's a waste of RAM... but we DO NOT care ! *)
                len:=Str.Length(S);
                needed:=SIZE(entryType)-SIZE(CHAR)+len;
                IF Available(needed)=FALSE THEN
                    IF useLFN THEN rc:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL);
                END; (* storage ALLOCATE failure *)
                CASE lastEntry OF
                | firstEntry :
                    ALLOCATE( anchor,needed);
                    newInList := anchor;
                ELSE
                    ALLOCATE(newInList^.next,needed);
                    newInList :=newInList^.next;
                END;
                Lib.FastMove( ADR(S),ADR(newInList^.string),len);
                newInList^.slen := len;
                newInList^.next := NIL;
            END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(DOSentry);
        END;
    END;
    IF useLFN THEN rc:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN lastEntry;
END buildMatchList;

PROCEDURE freeMatchList (anchor:ptrToEntry);
VAR
    len,needed      : CARDINAL;
    firstInList,newInList   : ptrToEntry;
BEGIN
    firstInList := anchor;
    newInList := firstInList;
    WHILE newInList # NIL DO
        len         := CARDINAL(newInList^.slen);
        needed      := SIZE(entryType)-SIZE(CHAR)+len;
        firstInList := firstInList^.next;
        DEALLOCATE (newInList,needed);
        newInList := firstInList;
    END
END freeMatchList;

PROCEDURE getMatchEntry (VAR R:pathtype;
                         n:CARDINAL; anchor:ptrToEntry);
VAR
    i,len:CARDINAL;
    newInList:ptrToEntry;
    S:pathtype;
BEGIN
    newInList := anchor;
    DEC(n); (* trick to force to anchor if 1, and locate correct string if > 1 *)
    FOR i:=firstEntry TO n DO
         newInList := newInList^.next;
    END;
    len         := newInList^.slen;
    Lib.FastMove( ADR(newInList^.string),ADR(S),len);
    S[len]      := nullchar; (* REQUIRED safety ! *)
    Str.Copy(R,S); (* yep, compiler won't let us fill R directly *)
END getMatchEntry;

PROCEDURE getEntryStr ( ps:ptrToEntry; VAR R:pathtype);
VAR
    len:CARDINAL;
    S:pathtype;
BEGIN
    len         := ps^.slen;
    Lib.FastMove( ADR(ps^.string),ADR(S),len);
    S[len]      := nullchar; (* REQUIRED safety ! *)
    Str.Copy(R,S); (* yep, compiler won't let us fill R directly *)
END getEntryStr;

(* ------------------------------------------------------------ *)

PROCEDURE unquote (VAR S:pathtype);
CONST
    quotedpat = dquote+star+dquote;
BEGIN
    IF Str.Match(S,quotedpat) THEN ReplaceChar(S,dquote,"");END; (* brutal and possibly wrong, eh eh... *)
END unquote;

PROCEDURE buildFromList (VAR anchor:ptrToEntry; VAR pb:CARDINAL; VAR R:pathtype;
                        useLFN:BOOLEAN;list:pathtype):CARDINAL;
VAR
    hlist:FIO.File;
    S:pathtype;
    needed,len,lastEntry:CARDINAL;
    newInList:ptrToEntry;
BEGIN
    pb:=errNone;
    lastEntry:=firstEntry-1;
    anchor:=NIL;

    hlist:=fileOpenRead(useLFN,list);
    FIO.AssignBuffer(hlist,ioBuffer);
    LOOP
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hlist,S);
        IF FIO.EOF THEN EXIT; END;
        LtrimBlanks(S);
        RtrimBlanks(S);
        CASE S[0] OF
        | nullchar,semicolon,pound:
            ;
        ELSE
            IF chkAnyJoker(FALSE,S) THEN pb:=errJokerInEntry; Str.Copy(R,S);EXIT;END;
            unquote(S);
            IF fileIsDirectorySpec(useLFN,S) THEN pb:=errEntryIsDir;Str.Copy(R,S);EXIT;END;

            IF isReservedEntry(S) = FALSE THEN
                INC(lastEntry);
                IF lastEntry=MAX(CARDINAL) THEN EXIT;END; (* too many files but let's fake ALLOCATE failure *)
                len:=Str.Length(S);
                needed:=SIZE(entryType)-SIZE(CHAR)+len;
                IF Available(needed)=FALSE THEN lastEntry:=MAX(CARDINAL);EXIT;END; (* storage ALLOCATE failure *)
                CASE lastEntry OF
                | firstEntry :
                    ALLOCATE( anchor,needed);
                    newInList := anchor;
                ELSE
                    ALLOCATE(newInList^.next,needed);
                    newInList :=newInList^.next;
                END;
                Lib.FastMove( ADR(S),ADR(newInList^.string),len);
                newInList^.slen := len;
                newInList^.next := NIL;
            END;
        END;
    END;
    fileClose(useLFN,hlist);
    RETURN lastEntry;
END buildFromList;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE getfilestamp (VAR h:dttype  ; useLFN:BOOLEAN;S:pathtype):BOOLEAN;
VAR
    hnd:FIO.File;
    rc:BOOLEAN;
BEGIN
    hnd:=fileOpenRead(useLFN,S);
    IF hnd = MAX(CARDINAL) THEN (* not trapped by fileGetFileStamp() *)
        rc:=FALSE;
    ELSE
        h.dt := FIO.GetFileDate(hnd);
        rc:=TRUE;
    END;
    fileClose(useLFN,hnd);
    RETURN rc;
END getfilestamp;

PROCEDURE setFileStamp (useLFN:BOOLEAN;h:dttype;S:pathtype):BOOLEAN;
VAR
    hnd:FIO.File;
    rc:BOOLEAN;
BEGIN
    hnd:=fileOpen(useLFN,S);
    IF hnd=MAX(CARDINAL) THEN
        rc:=FALSE;
    ELSE
        FIO.SetFileDate(hnd,h.dt);
        rc:=TRUE;
    END;
    fileClose(useLFN,hnd);
    RETURN rc;
END setFileStamp;

CONST
    mintime             = LONGCARD(0);
    defaultmaxtime      = LONGCARD( (24*60)*60 ); (* 86400 *)

PROCEDURE secondsToHMS (seconds:LONGCARD;VAR hh,mm,ss:CARDINAL);
CONST
    secondsPerHour = 60*60;
    minutesPerHour = 60;
    secondsPerMinute=60;
BEGIN
    hh := CARDINAL (seconds DIV secondsPerHour);
    mm := CARDINAL (seconds DIV minutesPerHour) MOD secondsPerMinute;
    ss := CARDINAL (seconds MOD 60 );
END secondsToHMS;

PROCEDURE buildstamp (VAR dt:dttype; seconds:LONGCARD);
VAR
    hh,mm,ss:CARDINAL;
BEGIN
    secondsToHMS(seconds,hh,mm,ss);
    dt.t := packTime(hh,mm,ss);
END buildstamp;

(*
Bitfields for file attributes:
Bit(s)  Description     (Table 01420)
 7      shareable (Novell NetWare)
 7      pending deleted files (Novell DOS, OpenDOS)
 6      unused
 5      archive
 4      directory
 3      volume label
        execute-only (Novell NetWare)
 2      system
 1      hidden
 0      read-only
*)

PROCEDURE getfileattr (VAR orgattr:FIO.FileAttr; useLFN:BOOLEAN;S:pathtype);
VAR
    R : SYSTEM.Registers;
    filename : str256;
    attribute : CARDINAL;
    shortform:pathtype;
    rc:CARDINAL;
BEGIN
    IF useLFN THEN
        IF w9XlongToShort(S,rc,shortform) THEN
            Str.Copy(S,shortform);
        END;
    END;
    Str.Copy(filename,S);
    filename[Str.Length(filename)] := nullchar;

    R.AX := 4300H;                            (* get file attributes *)
    R.DS := Seg(filename);
    R.DX := Ofs(filename);
    Lib.Dos(R);
    attribute := R.CX;
    (* assume everything goes without error ! *)
    orgattr:=FIO.FileAttr(attribute AND 000FFH);
END getfileattr;

PROCEDURE setFileAttr (useLFN:BOOLEAN;newattr:FIO.FileAttr;S:pathtype );
VAR
    R : SYSTEM.Registers;
    filename : str256;
    attribute : CARDINAL;
    shortform:pathtype;
    rc:CARDINAL;
BEGIN
    attribute := CARDINAL( newattr );
    IF useLFN THEN
        IF w9XlongToShort(S,rc,shortform) THEN
            Str.Copy(S,shortform);
        END;
    END;
    Str.Copy(filename,S);
    filename[Str.Length(filename)] := nullchar;

    R.AX := 4301H;
    R.DS := Seg(filename);
    R.DX := Ofs(filename);
    R.CX := attribute;
    Lib.Dos(R);
    (* assume everything goes without error ! *)
END setFileAttr;

PROCEDURE buildMask (attr:FIO.FileAttr):masktype;
VAR
    m:masktype;
    i:CARDINAL;
BEGIN
    m.stateD:=unwanted;
    m.stateR:=unwanted;
    m.stateH:=unwanted;
    m.stateS:=unwanted;
    m.stateA:=unwanted;

    FOR i:=firstattr TO lastattr DO
        CASE i OF
        | 1 : IF ( aD IN attr ) THEN m.stateD:=required;END;
        | 2 : IF ( aR IN attr ) THEN m.stateR:=required;END;
        | 3 : IF ( aH IN attr ) THEN m.stateH:=required;END;
        | 4 : IF ( aS IN attr ) THEN m.stateS:=required;END;
        | 5 : IF ( aA IN attr ) THEN m.stateA:=required;END;
        END;
    END;
    RETURN m;
END buildMask;

(* if state is dontcare, use original status *)

PROCEDURE buildAttr (m:masktype;attr:FIO.FileAttr):FIO.FileAttr;
VAR
    ma:masktype;
    primary,secondary:attrstatetype;
    r:FIO.FileAttr;
    i:CARDINAL;
    yes:BOOLEAN;
BEGIN
    r:=FIO.FileAttr{};
    ma:=buildMask(attr);
    FOR i:=firstattr TO lastattr DO
        CASE i OF
        | 1 : primary := m.stateD; secondary := ma.stateD;
        | 2 : primary := m.stateR; secondary := ma.stateR;
        | 3 : primary := m.stateH; secondary := ma.stateH;
        | 4 : primary := m.stateS; secondary := ma.stateS;
        | 5 : primary := m.stateA; secondary := ma.stateA;
        END;
        CASE primary OF
        | dontcare: yes:=(secondary=required);
        | required: yes:=TRUE;
        | unwanted: yes:=FALSE;
        END;
        CASE i OF
        | 1: IF yes THEN INCL(r, aD);END;
        | 2: IF yes THEN INCL(r, aR);END;
        | 3: IF yes THEN INCL(r, aH);END;
        | 4: IF yes THEN INCL(r, aS);END;
        | 5: IF yes THEN INCL(r, aA);END;
        END;
    END;

    RETURN r;
END buildAttr;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE show (wi:CARDINAL;S1,S2:ARRAY OF CHAR);
CONST
    sep = " : ";
VAR
    i:CARDINAL;
BEGIN
    WrStr(S1);
    FOR i:=( Str.Length(S1)+1) TO wi DO WrStr(" ");END;
    WrStr(sep);WrStr(S2);WrLn;
END show;

PROCEDURE showbool (wi:CARDINAL;tf:BOOLEAN;S1,S2,S3:ARRAY OF CHAR);
BEGIN
    IF tf THEN
        show(wi,S1,S2);
    ELSE
        show(wi,S1,S3);
    END;
END showbool;

(* ------------------------------------------------------------ *)

CONST
    msgTested     = ';-) "@"';
    msgProcessed  = '+++ "@"';
    msgSkippedRO  = '--- (read-only) "@"';
    msgNotFound   = '::: (not found) "@"';
    msgOpening    = '--- (opening) "@"';
    msgRollover   = '--- (24h rollover) "@"';
CONST
    spad          = 16;
    sOperation    = "Operation";
    sReference    = "Reference file";
    sNewDate      = "New date";
    sNewTime      = "New time";
    sBaseDate     = "Base date";
    sInterval     = "Time increment";
    sTarget       = "Target";
    sNewMask      = "New attributes";
    sChangeRO     = "Process RO files";
    sYearFix      = "2000 date fix";
    sActualUpdate = "Update data";
    sCreateUndo   = "Create undo";
    sUseLFN       = "Use LFN";
CONST
    UNDEFINED = MAX(CARDINAL);
    minParm = 1;
    maxParm = 3;
VAR
    parm : ARRAY [minParm..maxParm] OF pathtype;
    parmcount,i,opt:CARDINAL;
    S,R:pathtype;
    lastparm:CARDINAL;
    fromlist,createundo,useLFN,verbose,changeRO,testmode,verboser:BOOLEAN;
    yearfix:CARDINAL;
    dateref : datetype;
    timeref : timetype;
    orgstamp,refstamp: dttype;
    sDMY : str16; (* "##-###-####" *)
    sHMS : str16; (* "##:##:##"    *)
    sourcespec,targetspec,listspec:pathtype;
    cmd:(notdefinedyet,restamp,touchfromfile,touchfromcli,reattr);
    anchor,ps : ptrToEntry;
    pb:CARDINAL;
    newmask,attrmask,orgmask : masktype;
    strmask  : str16;   (* "D.S.H.R.A." *)
    ok,doit,wasRO:BOOLEAN;
    msg:str1024; (* must be bigger than pathtype *)
    seconds,timeinterval,maxtime:LONGCARD;
    hout:FIO.File;
    newattr,orgattr : FIO.FileAttr;
    updated:BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;

    WrLn; (* required here for pretty display *)

    IF AtLeastDosVersion(3,20)=FALSE THEN abort(errDosVersion,"3.20"); END;

    parmcount := Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,"");END;

    timeinterval:= UNDEFINED;
    verbose    := TRUE;
    verboser   := FALSE;
    changeRO   := FALSE;
    testmode   := TRUE;
    yearfix    := fix1900;
    useLFN     := TRUE;
    createundo := FALSE;

    cmd        := notdefinedyet;
    sDMY       := sDMYrestampDefault; (* default *)
    sHMS       := sHMSrestampDefault;

    lastparm := minParm-1;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S); cleantabs(R);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "Q"+delim+"QUIET"+delim+
                                   "R"+delim+"READONLY"+delim+
                                   "Y"+delim+"APPLY"+delim+
                                   "C"+delim+"2000"+delim+"XXI"+delim+"2K"+delim+
                                   "X"+delim+"LFN"+delim+
                                   "YY"+delim+
                                   "D:"+delim+"DATE:"+delim+
                                   "M:"+delim+"MASK:"+delim+"A:"+delim+"ATTRIBUTES:"+delim+
                                   "B"+delim+"BATCH"+delim+"U"+delim+"UNDO"+delim+
                                   "I:"+delim+"INCREMENT:"+delim+
                                   "V"+delim+"VERBOSE");
            CASE opt OF
            | 1,2,3 :     abort(errHelp,"");
            | 4,5:        verbose  := FALSE;
            | 6,7:        changeRO := TRUE;
            | 8,9:        testmode := FALSE;
            |10,11,12,13: yearfix  := fix2000;
            |14,15:       useLFN   := FALSE;
            |16:          testmode := FALSE; createundo:=TRUE;
            |17,18:       CASE cmd OF
                          | notdefinedyet,restamp:
                              GetString(R, sDMY);
                              (* parse later because of yearfix
                              IF same(sDMY,nowplaceholder)=FALSE THEN
                                  IF parseDate(dateref,yearfix,sDMY)=FALSE THEN
                                      abort(errBadDate,sDMY);
                                  END;
                              END;
                              *)
                              cmd:=restamp;
                          ELSE
                              abort(errCmdConflict,"-d:$");
                          END;
            |19,20,21,22: CASE cmd OF
                          | notdefinedyet,reattr:
                              GetString(R, strmask);
                              IF parseMask(strmask,attrmask)=FALSE THEN
                                  abort(errMask,strmask);
                              END;
                              cmd:=reattr;
                          ELSE
                              abort(errCmdConflict,"-m:$");
                          END;
            |23,24,25,26: createundo:=TRUE;
            |27,28 :      IF GetLongCard(R,timeinterval)=FALSE THEN abort(errBadVal,S);END;
                          IF timeinterval < mintimeinterval THEN abort(errInterval,S);END;
                          IF timeinterval > maxtimeinterval THEN abort(errInterval,S);END;
            |29,30:       verboser := TRUE;
            ELSE
                abort(errUnknownOption,S);
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxParm THEN abort(errParmOverflow,S);END;
            Str.Copy( parm[lastparm],S);
        END;
    END;

    useLFN := ( useLFN AND w9XsupportLFN() );

    IF NOT(useLFN) THEN
        FOR i:=minParm TO lastparm DO UpperCase(parm[i]);END;
    END;

    CASE lastparm OF
    | 0 : abort(errHelp,"");
    | 1 :
        CASE cmd OF
        | restamp,notdefinedyet: (* syntax 3 *)
            IF same(sDMY, nowplaceholder) THEN
                getDateNow(dateref);
            ELSE
                IF parseDate(dateref,yearfix,sDMY)=FALSE THEN abort(errBadDate,sDMY);END;
            END;
            IF parseTime(timeref,sHMS)=FALSE THEN abort(errBadTime,sHMS);END;
            refstamp := PackDateTime (dateref,timeref);
            IF timeinterval=UNDEFINED THEN timeinterval:=defaulttimeinterval;END;
            maxtime := defaultmaxtime-timeinterval;
            cmd:=restamp;
        | reattr:  (* syntax 4 *)
            IF yearfix=fix2000 THEN abort(errSyntaxOption,"-c");END;
            IF timeinterval # UNDEFINED THEN abort(errSyntaxOption,"-i:#");END;
            changeRO:=TRUE; (* force it here *)
        ELSE
            abort(errSyntax,""); (* never ! *)
        END;
        Str.Copy(targetspec,parm[minParm]);
    | 2 :
        CASE cmd OF
        | restamp,reattr:
            abort(errSyntax,"");
        ELSE (* syntax 1 *)
            IF yearfix=fix2000 THEN abort(errSyntaxOption,"-c");END;
            IF timeinterval # UNDEFINED THEN abort(errSyntaxOption,"-i:#");END;
        END;
        Str.Copy(sourcespec,parm[minParm]);
        Str.Copy(targetspec,parm[minParm+1]);

        IF chkAnyJoker(FALSE,sourcespec) THEN abort(errSourceJokers,sourcespec); END;
        IF chkValidName(sourcespec)=FALSE THEN abort(errSourceBadName,sourcespec); END;
        IF fileExists(useLFN,sourcespec)=FALSE THEN abort(errSourceNoMatch,sourcespec); END;
        IF getfilestamp(refstamp, useLFN,sourcespec)=FALSE THEN abort(errOpening,sourcespec); END;
        cmd := touchfromfile;
    | 3 :
        CASE cmd OF
        | restamp,reattr:
            abort(errSyntax,"");
        ELSE (* syntax 2 *)
            IF timeinterval # UNDEFINED THEN abort(errSyntaxOption,"-i:#");END;
        END;
        Str.Copy(sDMY      ,parm[minParm]);
        Str.Copy(sHMS      ,parm[minParm+1]);
        Str.Copy(targetspec,parm[minParm+2]);

        IF same(sDMY, nowplaceholder) THEN
            getDateNow(dateref);
        ELSE
            IF parseDate(dateref,yearfix,sDMY)=FALSE THEN abort(errBadDate,sDMY);END;
        END;
        IF same(sHMS, nowplaceholder) THEN
            getTimeNow(timeref);
        ELSE
            IF parseTime(timeref,sHMS)=FALSE THEN abort(errBadTime,sHMS);END;
        END;
        refstamp := PackDateTime (dateref,timeref);
        cmd := touchfromcli;
    ELSE
        abort(errHelp,""); (* never ! *)
    END;

    (* time to work now : retrieve filenames from directory or filelist *)

    CASE targetspec[0] OF
    | arobas:
        fromlist:=TRUE;
        Str.Copy(listspec,targetspec);
        Str.Delete(listspec,0,1);
        IF chkValidName(listspec)=FALSE THEN abort(errListBadName,listspec);END;
        IF Str.RCharPos(listspec,dot)=MAX(CARDINAL) THEN Str.Append(listspec,extLST);END;
        IF chkAnyJoker(FALSE,listspec) THEN abort(errJokerList,listspec);END;
        IF fileIsDirectorySpec(useLFN,listspec) THEN abort(errNotFile,listspec);END;
        IF fileExists(useLFN,listspec)=FALSE THEN abort(errNotFound,listspec);END;
        i:=buildFromList (anchor,pb,S,  useLFN,listspec);
        CASE i OF
        | 0:             abort(errNoEntry,listspec);
        | MAX(CARDINAL): abort(errTooManyEntries,listspec);
        END;
        IF pb # errNone THEN abort(pb,S);END;
    ELSE
        fromlist:=FALSE;
        IF chkValidName(targetspec)=FALSE THEN abort(errTargetBadName,targetspec);END;
        IF same(targetspec,dot) THEN Str.Copy(targetspec,stardotstar);END;
        IF chkAnyJoker(TRUE,targetspec) THEN abort(errJokerPath,targetspec);END;
        i:=buildMatchList (anchor,  useLFN,targetspec);
        CASE i OF
        | 0:             abort(errNoMatch,targetspec);
        | MAX(CARDINAL): abort(errTooMany,targetspec);
        END;
    END;

    IF verbose THEN
        WrStr(Banner);WrLn;
        WrLn;
        S:="Operation";
        CASE cmd OF
        | touchfromfile: show(spad,S,"synchronize date and time from file");
                         show(spad,sReference,sourcespec);
                         show(spad,sNewDate,fmtDate(TRUE,refstamp.d));
                         show(spad,sNewTime,fmtTime(TRUE,refstamp.t));
        | touchfromcli:  show(spad,S,"synchronize date and time from command line");
                         show(spad,sNewDate,fmtDate(TRUE,refstamp.d));
                         show(spad,sNewTime,fmtTime(TRUE,refstamp.t));
                         showbool(spad,(yearfix=fix2000),sYearFix,"yes","no");
        | restamp:       show(spad,S,"change and renumber time");
                         show(spad,sBaseDate,fmtDate(TRUE,refstamp.d));
                         Str.Concat(S,using(CARDINAL(timeinterval),1,"")," seconds");
                         show(spad,sInterval,S);
                         showbool(spad,(yearfix=fix2000),sYearFix,"yes","no");
        | reattr:        show(spad,S,"change attributes");
                         show(spad,sNewMask,fmtMask(FALSE,attrmask));
        END;
        IF fromlist THEN
            S:='files from "@"';    Str.Subst(S,"@",listspec);
        ELSE
            S:='files matching "@"';Str.Subst(S,"@",targetspec);
        END;
        show(spad,sTarget,S);
        showbool(spad,changeRO,sChangeRO,"yes","no");
        showbool(spad,useLFN,sUseLFN,"yes","no");

        showbool(spad,NOT(testmode),sActualUpdate,"yes","no");
        showbool(spad,createundo,sCreateUndo,"yes","no");
        WrLn;
    END;

    (* we'll ignore any opening problem : fed up with error handling ! *)

    IF createundo THEN
        hout := FIO.Create(sUNDOBATCH);
        FIO.AssignBuffer(hout,ioBuffer);
        S:= "@ECHO OFF"+nl+
            "PAUSE Hit Ctrl-C or Ctrl-Break to abort !"+nl+
            nl;
        FIO.WrStr(hout,S);
    END;

    seconds:=mintime;
    ps:=anchor;
    WHILE ps # NIL DO
        getEntryStr(ps,S);
        updated:=FALSE;
        IF fileExists(useLFN,S) THEN
            wasRO:=fileIsRO(useLFN,S);
            IF wasRO THEN
                doit:=changeRO;
            ELSE
                doit:=TRUE;
            END;
            IF doit THEN
                CASE cmd OF
                | reattr : getfileattr(orgattr, useLFN,S);
                | restamp: IF seconds < maxtime THEN
                               buildstamp(refstamp,seconds);
                               INC(seconds,timeinterval);
                           END;
                END;
                IF createundo THEN ok:=getfilestamp(orgstamp, useLFN,S); END;
                IF testmode THEN
                    msg:=msgTested;
                    updated:=TRUE; (* well... *)
                ELSE
                    msg:=msgProcessed; (* default *)
                    IF wasRO THEN fileSetRW(useLFN,S);END;
                    CASE cmd OF
                    | touchfromfile,touchfromcli:
                        ok:=setFileStamp(useLFN,refstamp,S);
                        IF wasRO THEN fileSetRO(useLFN,S);END;
                        updated:=TRUE;
                    | restamp:
                        IF seconds < maxtime THEN
                            ok:=setFileStamp(useLFN,refstamp,S);
                            updated:=TRUE;
                        ELSE
                            msg:=msgRollover;
                        END;
                        IF wasRO THEN fileSetRO(useLFN,S);END;
                    | reattr:
                        newattr:=buildAttr(attrmask,orgattr);
                        setFileAttr(useLFN,newattr,S);
                        updated:=TRUE;
                    END;
                END;
            ELSE
                msg:=msgSkippedRO;
            END;
        ELSE
            msg:=msgNotFound;
        END;
        IF verbose THEN
            IF (verboser AND updated) THEN
                IF cmd = reattr THEN
                    Str.Subst(msg,'"@"','@ --> @ "@"');
                    orgmask:=buildMask(orgattr);
                    Str.Subst(msg,"@",fmtMask(FALSE,orgmask));
                    Str.Subst(msg,"@",fmtMask(FALSE,attrmask));
                ELSE
                    Str.Subst(msg,'"@"','@ @ "@"');
                    Str.Subst(msg,"@",fmtDate(FALSE,refstamp.d));
                    Str.Subst(msg,"@",fmtTime(TRUE ,refstamp.t));
                END;
            END;
            Str.Subst(msg,"@",S);
            IF NOT(useLFN) THEN Str.Subst(S,dquote,"");Str.Subst(S,dquote,"");END;
            WrStr(msg);WrLn;
        END;
        IF createundo THEN
            IF cmd=reattr THEN
                msg:=ProgEXEname+' -y -m:@ "@"';
                newmask:=buildMask(orgattr);
                Str.Subst(msg,"@",fmtMask(TRUE,newmask));
            ELSE
                msg:=ProgEXEname+' -y @ @ "@"';
                Str.Subst(msg,"@",fmtDate(FALSE,orgstamp.d));
                Str.Subst(msg,"@",fmtTime(TRUE ,orgstamp.t));
            END;
            Str.Subst(msg,"@",S);
            IF NOT(useLFN) THEN Str.Subst(S,dquote,"");Str.Subst(S,dquote,"");END;
            FIO.WrStr(hout,msg);FIO.WrLn(hout);
        END;

        ps:=ps^.next;
    END;

    IF createundo THEN
        FIO.Close(hout);
        IF verbose THEN WrLn;END;
        WrStr(sUNDOBATCH);WrStr(" created.");WrLn;
    END;

    freeMatchList(anchor);

    abort(errNone,"");
END dtSync.
