(* ---------------------------------------------------------------
Title         Q&D Trim
Overview      remove leading or trailing blanks and tabs
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
Bugs
Wish List     LFN support while we're at it ? bah...
              protect remarks ( ";*" "#*" ) ?

--------------------------------------------------------------- *)

MODULE Trim;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM IO IMPORT WrStr, WrLn;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname     = "TRIM";
    ProgTitle       = "Q&D Trim";
    ProgVersion     = "v1.0k";
    ProgCopyright   = "by PhG";
    Banner          = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    cr              = CHR(13);
    lf              = CHR(10);
    nl              = cr+lf;
    doublequote     = '"';
    singlequote     = "'";
    dot             = ".";
    space           = " ";
    tab             = CHR(9);
    star            = "*";
    stardotstar     = star+dot+star;
    extBAK          = ".BK!";
    extCOM          = ".COM";
    extEXE          = ".EXE";
    extDLL          = ".DLL";
    extOVR          = ".OVR";
    extOVL          = ".OVL";
    extDRV          = ".DRV";
    extZIP          = ".ZIP";
    extARJ          = ".ARJ";
    extLZH          = ".LZH";
    extensions      = extBAK+delim+extCOM+delim+extEXE+delim+
                      extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                      extZIP+delim+extARJ+delim+extLZH;
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errTooManyParms = 3;
    errBadCard      = 4;
    errTabRange     = 5;
    errMissingSpec  = 6;
    errCmdNeeded    = 7;
    errTooManyFiles = 8;
    errNotFound     = 9;
    errFileNames    = 10;
    errQuotes       = 11;
    errRelaxMode    = 12;
    (* errQuotesCmd    = 14; *)

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" <file> [option]..."+nl+
nl+
"This program removes trailing, leading or embedded spaces and tabs,"+nl+
"encloses each line with double quotes, or removes enclosing double quotes."+nl+
nl+
"-r    remove trailing spaces and tabs"+nl+
"-l    remove smallest possible number of leading spaces and tabs"+nl+
"-ll   remove all leading spaces and tabs"+nl+
"-b    remove both leading and trailing spaces and tabs (same as -r -ll)"+nl+
"-t:#  tabulation width ([1..256], default is 8)"+nl+
"-n    replace internal spaces and tabs with a dot for filenames (-b forced)"+nl+
"-nn   remove spaces and tabs preceding a dot for filenames (-b forced)"+nl+
"-e[e] enclose each line with double quotes (-ee = -b then -q)"+nl+
"-q[q] remove enclosing double quotes from each line (-qq = -b then -q)"+nl+
"-p    remove empty lines"+nl+
"-k    ignore opening/closing match check when removing double quotes (-q[q])"+nl+
nl+
"a) Default options are -r -l. With -l[l], leading tabs are expanded to spaces."+nl+
"b) Note -e and -q will process lines only when necessary."+nl+
"c) "+extensions+" files will be ignored."+nl+
"d) As a safety, backups (with "+extBAK+" extension) are always created."+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," parameter is one too many !");
    | errBadCard:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," value !");
    | errTabRange:
        S := "Tabulation count must be in the [1..256] range !";
    | errMissingSpec:
        S := "Missing file specification !";
    | errCmdNeeded:
        S := "Missing command !";
    | errTooManyFiles:
        Str.Concat(S,"Too many files match ",einfo);Str.Append(S," specification !");
    | errNotFound:
        Str.Concat(S,"No file matches ",einfo);Str.Append(S," specification !");
    | errFileNames:
        S := "-n and -nn options are mutually exclusive !";
    | errQuotes:
        S := "-e and -q options are mutually exclusive !";
    | errRelaxMode:
        S := "-k option requires -q[q] option !";
    (*
    | errQuotesCmd:
        S := "-e and -q options cannot be mixed with any other option !";
    *)
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp:
        ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;
    bufferOut : ioBuffer;
    hugestr   : ARRAY[0..32768-1] OF CHAR; (* does not like to be local ! *)
    hugetmp   : ARRAY[0..32768-1] OF CHAR;

(* ------------------------------------------------------------ *)

(* we use global hugestr and hugetmp here as not to fill local procedure stack *)

PROCEDURE countDetabbedLeft (tabwidth:CARDINAL;createtmp:BOOLEAN):CARDINAL ;
VAR
    len,i,j,k,add: CARDINAL;
    c : CHAR;
BEGIN
    IF createtmp THEN Str.Copy(hugetmp,"");END;
    len:=Str.Length(hugestr);
    IF len=0 THEN RETURN MAX(CARDINAL); END; (* ignore empty lines *)
    j:=0; (* yes, 0 and not 1 ! *)
    i:=1;
    LOOP
        c:=hugestr[i-1];
        IF ORD(c) > ORD(space) THEN EXIT; END;
        IF c = tab THEN
            add := tabwidth - (j MOD tabwidth);
            WHILE add > 0 DO
                IF createtmp THEN Str.Append(hugetmp,space);END;
                INC(j);
                DEC(add);
            END;
        ELSE
            IF createtmp THEN Str.Append(hugetmp,c);END;
            INC(j);
        END;
        INC(i);
        IF i > len THEN EXIT; END;
    END;
    IF createtmp THEN
        (* yes, we could do something smarter... *)
        FOR k:= i TO len DO
            Str.Append(hugetmp,hugestr[k-1]);
        END;
    END;
    RETURN j; (* yes, j and not i *)
END countDetabbedLeft;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE unquote(bothquotes,relaxmode:BOOLEAN ; VAR S:ARRAY OF CHAR);
VAR
    len,last,i,performed:CARDINAL;
    ch:CHAR;
    pat:str16;
BEGIN
    IF bothquotes THEN
        last:=2;
    ELSE
        last:=1;
    END;
    i:=1;
    LOOP
        CASE i OF
        | 1: ch:=doublequote;
        | 2: ch:=singlequote;
        END;
        IF relaxmode THEN
            performed:=0;
            Str.Concat(pat,ch,"*");
            IF Str.Match(S,pat) THEN Str.Delete(S,0,1); INC(performed); END;
            Str.Concat(pat,"*",ch);
            IF Str.Match(S,pat) THEN
                len:=Str.Length(S);
                Str.Delete(S,len-1,1);
                INC(performed);
            END;
            IF performed # 0 THEN EXIT; END;
        ELSE
            Str.Concat(pat,ch,"*");Str.Append(pat,ch);
            IF Str.Match(S,pat) THEN
                Str.Delete(S,0,1); (* first doublequote *)
                len:=Str.Length(S);
                Str.Delete(S,len-1,1);
                EXIT;
            END;
        END;
        INC(i);
        IF i > last THEN EXIT; END;
    END;
END unquote;

(* ------------------------------------------------------------ *)

PROCEDURE doTrim (oldname,newname:ARRAY OF CHAR;
                  doRight,doLeft,doFull,relaxmode,
                  doNames,doPackNames,doEnclose,doUnclose,doKillEmpty:BOOLEAN;
                  tabwidth:CARDINAL);
CONST
    msgProcessing = "Processing ";
    msgOK         = " OK !";
VAR
    hin,hout : FIO.File;
    leading,smallest,i,len : CARDINAL;
    S:str16;
    dmpme:BOOLEAN;
BEGIN
    WrStr(msgProcessing); WrStr(newname);
    Work(cmdInit);

    hin := FIO.OpenRead(oldname);
    FIO.AssignBuffer(hin,bufferIn);
    hout:= FIO.Create(newname);
    FIO.AssignBuffer(hout,bufferOut);

    IF ((doLeft=TRUE) AND (doFull=FALSE)) THEN
        smallest := MAX(CARDINAL);
        FIO.EOF := FALSE;
        LOOP
            IF FIO.EOF THEN EXIT; END;
            Work(cmdShow);
            FIO.RdStr(hin,hugestr);
            IF FIO.EOF THEN EXIT; END;
            IF doRight THEN RtrimBlanks(hugestr); END;
            leading:=countDetabbedLeft(tabwidth,FALSE );
            IF leading < smallest THEN smallest := leading; END;
        END;
        IF smallest = 0 THEN doLeft := FALSE; END;
        FIO.Seek(hin,0);
    END;

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        Work(cmdShow);
        FIO.RdStr(hin,hugestr);
        IF FIO.EOF THEN EXIT; END;
        IF doRight THEN RtrimBlanks(hugestr); END;
        IF doFull THEN (* precedence over doLeft which can be set to false *)
            LtrimBlanks(hugestr);

            (* caller checked doNames and doPackNames cannot be both true *)

            IF doNames THEN (* doFull, doRight, doLeft have been forced when parsing *)
                IF Str.CharPos(hugestr,dot) = MAX(CARDINAL) THEN (* safety *)
                    ReplaceChar(hugestr,tab,space);
                    Str.Subst(hugestr,space,dot); (* first one *)
                    ReplaceChar(hugestr,space,"");
                END;
            END;
            IF doPackNames THEN (* doFull, doRight, doLeft have been forced when parsing *)
                FOR i:=1 TO 2 DO
                    CASE i OF
                    | 1: S:=space+dot;
                    | 2: S:=tab+dot;
                    END;
                    LOOP
                        IF Str.Pos(hugestr,S)=MAX(CARDINAL) THEN EXIT; END;
                        Str.Subst(hugestr,S,dot);
                    END;
                END;
            END;
        ELSE
            IF doLeft THEN
                leading:=countDetabbedLeft(tabwidth,TRUE);
                Str.Delete(hugetmp,0,smallest);
                Str.Copy(hugestr,hugetmp);
            END;
        END;

        len:=Str.Length(hugestr);
        IF len = 0 THEN
            dmpme := NOT(doKillEmpty);
        ELSE

            (* caller check for logic *)

            IF doEnclose THEN
                unquote(FALSE,relaxmode,hugestr);
                Str.Prepend (hugestr,doublequote);
                Str.Append  (hugestr,doublequote);
            END;
            IF doUnclose THEN unquote(FALSE,relaxmode, hugestr); END;

            dmpme:=TRUE;
        END;

        IF dmpme THEN FIO.WrStr(hout,hugestr);FIO.WrLn(hout); END;
    END;

    FIO.Flush(hout);
    FIO.Close(hout);
    FIO.Close(hin);

    Work(cmdStop);
    WrStr(msgOK); WrLn;
END doTrim;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE legalextension (S:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN

    Str.Caps(S); (* ah, lowercase LFNs... *)

    rc:=TRUE;
    n:=0;
    LOOP
        isoleItemS(e3, extensions,delim,n);
        IF same(e3,"") THEN EXIT; END;
        IF Str.Pos(S,e3) # MAX(CARDINAL) THEN rc:=FALSE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END legalextension;

TYPE
    f8e3 = ARRAY [0..8+1+3-1] OF CHAR;
CONST
    firstFile = 1;
    lastFile  = 2000;
    maxFile   = lastFile-firstFile+1;
VAR
    fileArray : ARRAY [firstFile..lastFile] OF f8e3;

PROCEDURE buildList (spec:ARRAY OF CHAR):CARDINAL;
VAR
    countFile : CARDINAL;
    found     : BOOLEAN;
    entry     : FIO.DirEntry;
BEGIN
    FIO.IOcheck := FALSE;
    countFile := 0;
    found := FIO.ReadFirstEntry(spec,allfiles,entry);
    WHILE found DO
        IF countFile = maxFile THEN RETURN MAX(CARDINAL); END;
        IF legalextension(entry.Name) THEN (* skip *.bk!, *.com and *.exe entries *)
            fileArray[firstFile+countFile]:=f8e3(entry.Name);
            Str.Caps(fileArray[firstFile+countFile]); (* useless ! *)
            INC (countFile);
        END;
        found :=FIO.ReadNextEntry(entry);
    END;
    RETURN countFile;
END buildList;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    mintabwidth = 1;
    maxtabwidth = 64;
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : str128;
    state           : (waiting,gotparm1);
    v               : LONGCARD;
    parm1           : str128;
    u,path,f8,e3    : str128; (* oversized just in case *)
    basepath        : str128;
    oldname,newname : str128;

    doRight,doLeft,doFull,doNames,doPackNames:BOOLEAN;
    doEnclose,doUnclose,doKillEmpty,relaxmode : BOOLEAN;
    tabwidth        : CARDINAL;
    count           : CARDINAL;
BEGIN
    Lib.DisableBreakCheck();
    WrLn;

    doRight     := FALSE;
    doLeft      := FALSE;
    doFull      := FALSE;
    doNames     := FALSE;
    doPackNames := FALSE;
    doEnclose   := FALSE;
    doUnclose   := FALSE;
    doKillEmpty := FALSE;
    relaxmode   := FALSE;
    tabwidth    := 8;

    state   := waiting;
    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "R"+delim+"RIGHT"+delim+"TRAILING"+delim+
                                   "L"+delim+"LEFT"+delim+"LEADING"+delim+
                                   "LL"+delim+"FULL"+delim+
                                   "B"+delim+"BOTH"+delim+
                                   "T:"+delim+"TAB:"+delim+
                                   "N"+delim+"NAMES"+delim+
                                   "NN"+delim+
                                   "E"+delim+"ENCLOSE"+delim+
                                   "Q"+delim+"QUOTES"+delim+
                                   "P"+delim+"PACK"+delim+
                                   "QQ"+delim+
                                   "EE"+delim+
                                   "K"+delim+"RELAXED"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5,6 : doRight := TRUE;
            | 7,8,9 : doLeft  := TRUE;
            | 10,11 : doLeft  := TRUE; doFull := TRUE;
            | 12,13:  doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 14,15:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadCard,S);END;
                IF ((v < mintabwidth) OR (v > maxtabwidth)) THEN abort(errTabRange,S);END;
                tabwidth := CARDINAL(v);
            | 16,17:  doNames := TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 18:     doPackNames:=TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 19,20:  doEnclose:= TRUE;
            | 21,22:  doUnclose:= TRUE;
            | 23,24:  doKillEmpty:=TRUE;
            | 25:     doUnclose:= TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 26:     doEnclose:= TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 27,28:  relaxmode := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting : Str.Copy(parm1,S);
            | gotparm1: abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;
    (* check nonsense *)
    IF state=waiting THEN abort(errMissingSpec,"");END;
    IF (doNames AND doPackNames) THEN abort(errFileNames,"");END;
    IF (doEnclose AND doUnclose) THEN abort(errQuotes,"");END;
    IF (doEnclose OR doUnclose) THEN
        ; (* IF (doRight OR doLeft) THEN abort(errQuotesCmd,"");END; *)
    ELSE
        IF ((doRight=FALSE) AND (doLeft=FALSE)) THEN
            doRight := TRUE;
            doLeft  := TRUE;
            (* abort(errCmdNeeded,""); *)
        END;
    END;
    IF NOT(doUnclose) THEN
        IF relaxmode THEN abort(errRelaxMode,"");END;
    END;

    IF same(parm1,dot) THEN Str.Copy(parm1,stardotstar);END;
    count := buildList(parm1);
    CASE count OF
    | 0 : abort(errNotFound,parm1);
    | MAX(CARDINAL): abort(errTooManyFiles,parm1);
    END;

    Lib.SplitAllPath(parm1,u,path,f8,e3);
    Str.Concat(basepath,u,path);

    WrStr(Banner);WrLn;
    WrLn;

    FOR i := 1 TO count DO
        Str.Concat(newname,basepath,fileArray[firstFile+i-1]);
        Lib.SplitAllPath(newname,u,path,f8,e3);
        Lib.MakeAllPath(oldname,u,path,f8,extBAK);
        IF FIO.Exists(oldname) THEN
            IF isReadOnly(oldname) THEN setReadWrite(oldname);END;
            FIO.Erase(oldname);
        END;
        FIO.Rename(newname,oldname);
        doTrim (oldname,newname,doRight,doLeft,doFull,relaxmode,
               doNames,doPackNames,doEnclose,doUnclose,doKillEmpty,tabwidth);
    END;
    abort(errNone,"");
END Trim.





(*

PROCEDURE countLeading (tabwidth:CARDINAL; VAR S : ARRAY OF CHAR):CARDINAL;
CONST
    espace=ORD(space);
VAR
    i,len,n : CARDINAL;
BEGIN
    len := Str.Length (S);
    IF len = 0 THEN RETURN MAX(CARDINAL); END; (* ignore empty lines *)
    i   := 0;
    n   := 0;
    LOOP
        IF i = len THEN EXIT; END;
        IF ORD(S[i]) > espace THEN EXIT; END;
        IF S[i] = tab THEN
            INC(n,tabwidth);
        ELSE
            INC(n);
        END;
        INC (i);
    END;
    RETURN n;
END countLeading;

PROCEDURE leftTrim (smallest,tabwidth:CARDINAL;VAR S : ARRAY OF CHAR);
CONST
    espace=ORD(space);
VAR
    i,len,n : CARDINAL;
BEGIN
    len := Str.Length (S);
    IF len = 0 THEN RETURN;END;
    i   := 0;
    n   := 0;
    LOOP
        IF i = len THEN EXIT; END;
        IF ORD(S[i]) > espace THEN EXIT; END;
        IF S[i] = tab THEN
            INC(n,tabwidth);
        ELSE
            INC(n);
        END;
        INC (i);
    END;
    (* i is position of first non-blank in S *)
    (* n is count of leading spaces *)
    IF n > 0 THEN
        Str.Delete(S,0,i);
        FOR i :=1 TO (n-smallest) DO
            Str.Prepend(S,space);
        END;
    END;
END leftTrim;

PROCEDURE detab (tabwidth:CARDINAL;S:ARRAY OF CHAR; VAR R : ARRAY OF CHAR  );
VAR
    i,j,add: CARDINAL;
    c : CHAR;
BEGIN
    Str.Copy(R,"");
    j:=0; (* yes, 0 and not 1 ! *)
    FOR i:=1 TO Str.Length(S) DO
        c := S[i-1];
        IF c = tab THEN
            add := tabwidth - (j MOD tabwidth);
            WHILE add > 0 DO
                Str.Append(R,space); INC(j);
                DEC(add);
            END;
        ELSE
            Str.Append(R,c); INC(j);
        END;
    END;
END detab;

*)
