(* ---------------------------------------------------------------
Title         Q&D Characters Count
Overview
Usage         see help
Notes
Bugs
Wish List     ignore also graphics and the like?
              a graphical representation of codes distribution ?
              (bar, circle)
              statistics (entropy, chi-square, etc.) ?

--------------------------------------------------------------- *)

MODULE cCount;

IMPORT IO;
IMPORT Lib;
IMPORT Str;
IMPORT FIO;
IMPORT MsMouse;
IMPORT Graph;
IMPORT SYSTEM;
IMPORT BiosIO;

FROM IO IMPORT WrStr,WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

(* ------------------------------------------------------------ *)

CONST
    MOUSEDONE   = FALSE;
CONST
    cr          = CHR(13);
    lf          = CHR(10);
    nl          = cr+lf;
    quote       = '"';
    tab         = CHR(9);
    dot         = ".";
    comma       = ",";
    space       = " ";
    star        = "*";
    stardotstar = star+dot+star;
    backslash   = "\";
    blank       = ORD(space); (* yes, we use character value here *)
    firstcode   = 0;
    maxcode     = 255;
    countcode   = maxcode - firstcode + 1;

(* ------------------------------------------------------------ *)

CONST
    progEXEname   = "CCOUNT";
    progTitle     = "Q&D Character Counter";
    progVersion   = "v1.0h";
    progCopyright = "by PhG";
    banner        = progTitle+" "+progVersion+" "+progCopyright;
CONST
    errNone          = 0;
    errHelp          = 1;
    errUnknownOption = 2;
    errTooManyParms  = 3;
    errMissingSpec   = 4;
    errTooManyMatches= 5;
    errNoMatch       = 6;
    errNonsense      = 7;
    errNonsenseVal   = 8;
    errBadNumber     = 9;
    errBarWidthRange = 10;
    errVGAneeded     = 11;
    errCannotSet     = 12;

PROCEDURE abort (e : CARDINAL; einfo:ARRAY OF CHAR);
CONST
(*
 0        1         2         3         4         5         6         7         8
 12345678901234567890123456789012345678901234567890123456789012345678901234567890
*)
    msgHelp =
banner+nl+
nl+
"Syntax 1 : "+progEXEname+" <file(s)> [option]..."+nl+
"Syntax 2 : "+progEXEname+" <file(s)> <-v> [-s|-r]"+nl+
nl+
"This program counts bytes used in up to 1000 files."+nl+
nl+
"-a    show data for all bytes (no restriction to only bytes appearing in file)"+nl+
"-p    add percentage to byte count"+nl+
"-c    compute percentages excluding control characters ([00..31]) from total"+nl+
"-i[i] ignore case (-ii = ignore case and accents)"+nl+
"-s    sort from higher to lower byte count"+nl+
"-r    sort from lower to higher byte count"+nl+
"-x    prefix each character with its hexadecimal value"+nl+
"-d    prefix each character with its decimal value"+nl+
"-t    terse report (without header)"+nl+
"-g[g] show global cumulative data (-gg = global only)"+nl+
"-b[b] show bar graph (-bb = no semigraphics)"+nl+
"-w:#  bar graph width (default is 25 characters)"+nl+
"-e    eye candy"+nl+
(*%F MOUSEDONE *)
"-v    graphical distribution view"+nl+
(*%E *)
(*%T MOUSEDONE *)
"-v[v] graphical distribution view (-vv = -v -m)"+nl+
"-m    use mouse instead of keyboard (if mouse is available)"+nl+
(*%E  *)
nl+
'a) Syntax 2 is roughly similar to "-a -p -x -d -gg -b" syntax 1 display ;'+nl+
"   it requires a VGA card allowing 320x200x8 standard video mode ;"+nl+
(*%T MOUSEDONE *)
"   mouse is supported though it is optional ;"+nl+
(*%E *)
"   any option other than -<s|r|e|m> is silently ignored."+nl+
"   While viewing, hit Tab key to toggle real or proportional dusplay."+nl+
"b) -i[i] option assumes DOS character set."+nl+
nl+
"Examples : "+progEXEname+" *.t -a"+nl+
"           "+progEXEname+" e:\fun\fortune\*.txt -v -ii"+nl;

VAR
    S : str256; (* oversized *)
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msgHelp);
    | errUnknownOption :
        Str.Concat(S,"Unknown "+quote,einfo);
        Str.Append(S,quote+" option !");
    | errTooManyParms:
        Str.Concat(S,"Useless "+quote,einfo);
        Str.Append(S,quote+" parameter !");
    | errMissingSpec:
        S := "Missing <file(s)> specification !";
    | errTooManyMatches:
        Str.Concat(S,"Too many files match "+quote,einfo);
        Str.Append(S,quote+" specification !");
    | errNoMatch:
        Str.Concat(S,"No file matches "+quote,einfo);
        Str.Append(S,quote+" specification !");
    | errNonsense:
        S := "-s and -r options are mutually exclusive !";
    | errNonsenseVal:
        S := "-x and -d options are mutually exclusive !";
    | errBadNumber:
        Str.Concat(S,"Illegal number in ",einfo);
        Str.Append(S," parameter !");
    | errBarWidthRange:
        S := "Bar graph width should belong to the [1..200] range !";
    | errVGAneeded :
        S := "VGA adapter needed !" ;
    | errCannotSet :
        Str.Concat(S,"Cannot set ",einfo);Str.Append(S," video mode !");
    ELSE
        S := "How did you get THERE ? This is illogical, Captain ! ;-)";
    END;
    CASE e OF
    | errNone, errHelp:
        ;
    ELSE
        WrStr(progEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE sound (freq,duration,pause:CARDINAL);
BEGIN
    Lib.Sound(freq);
    Lib.Delay(duration);
    Lib.NoSound();
    Lib.Delay(pause);
END sound;

PROCEDURE errbip (okbeep:BOOLEAN);
CONST
    freq     = 55;
    duration = 55; (* was 300 *)
    tempo    = 100;
BEGIN
    IF okbeep THEN sound (freq,duration,tempo); END;
END errbip;

(* ------------------------------------------------------------ *)

(* sep ok if pad is space *)

PROCEDURE fmtlc (v : LONGCARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.CardToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtlc;

PROCEDURE fmtstring (S:ARRAY OF CHAR; pad:CHAR; n:INTEGER ) : str80;
VAR
    R : str80;
BEGIN
    Str.Copy(R,S);
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(n) THEN EXIT; END;
        IF n < 0 THEN
            Str.Append(R,pad); (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtstring;

PROCEDURE fmtpercent (v:LONGREAL;pad:CHAR;sep:CHAR;prec:CARDINAL;field:INTEGER):str80;
VAR
    R : str80;
    ok: BOOLEAN;
BEGIN
    Str.FixRealToStr(v,prec,R,ok);
    Str.Subst(R,dot,sep);
    RETURN fmtstring(R,pad,field);
END fmtpercent;

PROCEDURE fmtbar (percentused:LONGREAL;used,free:CHAR;field:CARDINAL):str80;
VAR
    R : str80;
    p,i : CARDINAL;
BEGIN
    percentused := (percentused / 100.0 ) * LONGREAL(field);
    p := CARDINAL(percentused + 0.5); (* round ! *)
    R := "";
    FOR i := 1 TO field DO
        IF i <= p THEN
           Str.Append(R,used);
        ELSE
           Str.Append(R,free);
        END;
    END;
    RETURN R;
END fmtbar;

(* ------------------------------------------------------------ *)

CONST
    IObufferSize = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = IObufferSize;
VAR
    ioBuffer : ARRAY [1..IObufferSize] OF BYTE;

(* ------------------------------------------------------------ *)

CONST
    hexdigits = "0123456789abcdef";

PROCEDURE WriteCharacter ( i : CARDINAL );
VAR
    ok : BOOLEAN;
BEGIN
    CASE i OF
    | 0..31, 127, 255 :
       WrStr("$");
       WrStr(hexdigits[i DIV 16]); (* assume $00..$FF range, of course ! *)
       WrStr(hexdigits[i MOD 16]);
    | ORD(quote):
       WrStr( "'" + '"' + "'");
    ELSE
       WrStr(quote);
       IO.WrChar( CHR(i) );
       WrStr(quote);
    END;
END WriteCharacter;

PROCEDURE showbargraph (semigraphics:BOOLEAN;wi:CARDINAL;count,maxcount:LONGCARD  );
CONST
    chUsed = CHR(219);  chUsedTxt = "+";
    chFree = CHR(176);  chFreeTxt = "-";
VAR
    v      : LONGREAL;
    charUsed,charFree:CHAR;
BEGIN
    IF semigraphics THEN
        charUsed := chUsed ;     charFree := chFree;
    ELSE
        charUsed := chUsedTxt ;  charFree := chFreeTxt;
    END;
    v := ( LONGREAL(count) / LONGREAL(maxcount) ) * 100.0;
    WrStr(fmtbar(v,charUsed,charFree,wi));
END showbargraph;

PROCEDURE wrval (i:CARDINAL;hex:BOOLEAN);
CONST
    digits    = "0123456789";
    hexdigits = digits+"abcdef";
BEGIN
    IF hex THEN
        WrStr("$");
        WrStr(hexdigits[i DIV 16]); (* assume $00..$FF range, of course ! *)
        WrStr(hexdigits[i MOD 16]);
    ELSE
        WrStr(digits[i DIV 100]);
        WrStr(digits[ (i MOD 100) DIV 10 ]);
        WrStr(digits[i MOD  10]);
    END;
END wrval;

(* ------------------------------------------------------------ *)

(* ugly globerks ! *)

VAR
    code      ,gcode      : ARRAY [firstcode..maxcode] OF CARDINAL;
    count     ,gcount     : ARRAY [firstcode..maxcode] OF LONGCARD;
    total     ,gtotal     : LONGCARD;
    ctrlchars ,gctrlchars : LONGCARD;
    filesize  ,gfilesize  : LONGCARD;
CONST
    wilc          = (11+3);
    msgRem        = ";";
    msgFile       = msgRem+" File       : ";
    msgBytes      = msgRem+" Bytes read : ";
    msgTotal      = msgRem+" Total      : ";
    msgCtrl       = msgRem+" [00..31]   : ";
    msgRestricted = msgRem+" Show data only for bytes appearing in file";
    msgAll        = msgRem+" Show data for all bytes";
    msgNoCtrl     = msgRem+" Percentages computed excluding [00..31] from bytes read";
    msgWithCtrl   = msgRem+" Percentages computed using all bytes read";
    msgWarning    = msgRem+" Warning ! As total equals 0, percentages will NOT be computed !";
    msgCase       = msgRem+" Warning ! Case is ignored";
    msgCaseAccents= msgRem+" Warning ! Case and accents are ignored";

PROCEDURE process ( path:ARRAY OF CHAR;
                    excludecontrol,eyecandy,showglobal,
                    ignorecase,ignoreaccents:BOOLEAN;
                    VAR  filesize, total, ctrlchars,
                        gfilesize,gtotal,gctrlchars:LONGCARD);
VAR
    hnd    : FIO.File;
    prompt : str128;
    i,got  : CARDINAL;
    ch     : BYTE;
    charac : str2;
    ok     : BOOLEAN;
BEGIN
    total     := 0;
    ctrlchars := 0;
    filesize  := getFileSize(path);
    IF filesize = 0 THEN RETURN; END;

    FOR i := firstcode TO maxcode DO
        count[i]:=0;
    END;

    Str.Concat(prompt,"Processing ",path);Str.Append(prompt," ");
    video(prompt,TRUE);
    IF eyecandy THEN Work(cmdInit);END;
    hnd:=FIO.OpenRead(path);
    FIO.AssignBuffer(hnd,ioBuffer);
    FIO.EOF:=TRUE; (* safety ! *)
    LOOP
        IF eyecandy THEN Work(cmdShow);END;
        got := FIO.RdBin(hnd,ch,1);
        IF got = 0 THEN EXIT; END;

        ok:=ch # BYTE(0);
        ok:=( ok AND ignorecase );
        IF ok THEN
            Str.Copy(charac, CHR( ch ) );
            IF ignoreaccents THEN
                LowerCase(charac);
            ELSE
                LowerCaseAlt(charac);
            END;
            ch := charac[0];
        END;

        INC( count[CARDINAL(ch)] ); (* could use ORD(ch) if CHAR TYPE *)
        IF showglobal THEN
            INC( gcount[CARDINAL(ch)] );
        END;
    END;
    FIO.Close(hnd);
    IF eyecandy THEN Work(cmdStop);END;

    FOR i := firstcode TO (blank-1) DO
        INC(ctrlchars,count[i]);
        IF showglobal THEN INC(gctrlchars,count[i]); END;
    END;
    IF excludecontrol THEN
        total := filesize-ctrlchars;
    ELSE
        total := filesize;
    END;
    IF showglobal THEN INC(gtotal,total); INC(gfilesize,filesize); END;
    video(prompt,FALSE);
END process;

PROCEDURE dump (path:ARRAY OF CHAR;restricted,percentage,excludecontrol,
                hexval,decval,sort,reversesort,eyecandy,
                showheader,showglobal,showbar,semigraphics,
                ignorecase,ignoreaccents:BOOLEAN;
                filesize,total,ctrlchars:LONGCARD;barwi:CARDINAL);
VAR
    i : CARDINAL;
    exchange,test,dmpall : BOOLEAN;
    ln1,ln2:LONGCARD;
    n1,n2:CARDINAL;
    v : LONGREAL;
    S : str16;
    (* alphanum,symbols,graphics,others:LONGCARD; *)
BEGIN
    IF showheader THEN
        WrStr(msgFile);WrStr(path);WrLn;
        WrStr(msgBytes);WrStr(fmtlc(filesize,space,dot,wilc));WrLn;
        WrStr(msgTotal);WrStr(fmtlc(total,space,dot,wilc));WrLn;
        WrStr(msgCtrl);WrStr(fmtlc(ctrlchars,space,dot,wilc));WrLn;
        WrStr(msgRem);WrLn;
        IF restricted THEN
            WrStr(msgRestricted);
        ELSE
            WrStr(msgAll);
        END;
        WrLn;
        IF ignorecase THEN
            IF ignoreaccents THEN
                WrStr(msgCaseAccents);
            ELSE
                WrStr(msgCase);
            END;
            WrLn;
        END;
        IF percentage THEN
            IF excludecontrol THEN
                WrStr(msgNoCtrl);
            ELSE
                WrStr(msgWithCtrl);
            END;
            WrLn;
            IF total=0 THEN
                WrStr(msgWarning);WrLn;
                percentage := FALSE;
            END;
        END;
        WrLn;
    END;

    FOR i := firstcode TO maxcode DO
        code [i]:=i;
    END;

    IF (sort OR reversesort) THEN
        IF eyecandy THEN Animation(cmdInit);END;
        exchange := TRUE;
        WHILE exchange DO
            IF eyecandy THEN Animation(cmdShow);END;
            exchange := FALSE;
            FOR i := firstcode TO maxcode-1 DO
                ln1 := count[i];
                ln2 := count[i+1];
                IF sort THEN
                    test := (ln1 < ln2);  (* keep A, B, C,... sequence *)
                ELSIF reversesort THEN
                    test := (ln2 < ln1); (* <= instead of < forces A, B, C,... sequence, but would provoke an infinite loop *)
                END;
                IF test THEN
                    count[i]   := ln2;
                    count[i+1] := ln1;
                    n1         := code[i];
                    n2         := code[i+1];
                    code[i]    := n2;
                    code[i+1]  := n1;
                    exchange   := TRUE;
                END;
            END;
        END;
        IF eyecandy THEN Animation(cmdStop);END;
    END;

    dmpall := NOT(restricted);
    FOR i := firstcode TO maxcode DO
        IF ( dmpall OR (count[i] # 0) ) THEN

            IF hexval THEN wrval(ORD(code[i]),TRUE); WrStr(tab); END;
            IF decval THEN wrval(ORD(code[i]),FALSE);WrStr(tab); END;

            WriteCharacter(code[i]);
            WrStr(tab);
            IO.WrLngCard(count[i],11);
            IF percentage THEN
                WrStr(tab);
                v := ( LONGREAL(count[i]) / LONGREAL(total) ) * 100.0;
                Str.Concat(S, fmtpercent(v,space,comma,2, 3+1+2) , "%");
                IF excludecontrol THEN
                    CASE i OF
                    | 0..31 : S := "nonsense";
                    END;
                END;
                WrStr(S);
            END;
            IF showbar THEN
                WrStr(tab);
                showbargraph (semigraphics, barwi, count[i],total);
            END;
            WrLn;
        END;
    END;
END dump;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE realToCard (v:LONGREAL ):CARDINAL ;
CONST
    k = LONGREAL(0.5); (* was 0.5 *)
BEGIN
    RETURN VAL(CARDINAL, v + k);
END realToCard;

PROCEDURE plot (x,ybase,color:CARDINAL; v:LONGREAL );
VAR
    p,y:CARDINAL;
BEGIN
    p:=realToCard(v);
    y:=ybase-p+1;
    CASE p OF
    | 0 : ;
    (* | 1 : Graph.Plot(x,y,color); *)
    ELSE
          Graph.Line(x,ybase,x,y,color);
    END;
END plot;

CONST
    cBLACK	       =0;
    cBLUE 	       =1;
    cGREEN         =2;
    cCYAN          =3;
    cRED           =4;
    cMAGENTA       =5;
    cBROWN         =6;
    cWHITE         =7;
    cGRAY          =8;
    cLIGHTBLUE     =9;
    cLIGHTGREEN    =10;
    cLIGHTCYAN     =11;
    cLIGHTRED      =12;
    cLIGHTMAGENTA  =13;
    cLIGHTYELLOW   =14;
    cBRIGHTWHITE   =15;
    barcolor1      = cLIGHTRED;
    barcolor2      = cLIGHTYELLOW;

PROCEDURE WaitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END WaitVGAretrace;

PROCEDURE centerme (minhtab,wi:CARDINAL;S:ARRAY OF CHAR):CARDINAL;
VAR
    len,hpos:CARDINAL;
BEGIN
    len := Str.Length(S);
    IF len < wi THEN
        hpos:=minhtab+(wi-len) DIV 2;
    ELSE
        hpos:=minhtab;
    END;
    RETURN hpos;
END centerme;

VAR
    gvideoconfig        : Graph.VideoConfig;
    gvmode              : CARDINAL;
VAR
    mouseXmul           : CARDINAL;
    screenX, screenY    : CARDINAL;
    htabMin,htabWidth   : CARDINAL;
    vtabMin,vtabHeight  : CARDINAL;
    numColors           : CARDINAL;
    xmin,xmax,ymin,ymax : CARDINAL;
    screenCX,screenCY   : CARDINAL;

PROCEDURE setupVideoScreen();
VAR
    k : CARDINAL;
BEGIN
    Graph.GetVideoConfig(gvideoconfig);
    screenX  := gvideoconfig.numxpixels; (* 320 *)
    screenY  := gvideoconfig.numypixels; (* 200 *)
    CASE screenX OF
    | 320 : k :=2;
    | 640 : k :=1;
    ELSE
            k :=1;
    END;
    mouseXmul := k;

    htabMin  := 1;
    htabWidth:= gvideoconfig.numtextcols; (* screenX DIV 8 i.e. 40 here *)
    vtabMin  := 1;
    vtabHeight:=gvideoconfig.numtextrows;

    numColors:= gvideoconfig.numcolors; (* 16 *)

    xmin     := 0;
    xmax     := screenX-1;
    ymin     := 0;
    ymax     := screenY-1;

    screenCX := screenX DIV 2;
    screenCY := screenY DIV 2;
END setupVideoScreen;

PROCEDURE isKeyWaiting (  ) : BOOLEAN;
BEGIN
    RETURN BiosIO.KeyPressed();
END isKeyWaiting;

CONST
    keyEscape = 1B00H;
    keyLeft   = 004BH;
    keyRight  = 004DH;
    keyHome   = 0047H;
    keyEnd    = 004FH;
    keyPgUp   = 0049H;
    keyPgDn   = 0051H;
    keyTab    = 0900H;

PROCEDURE ReadKeycode (  ) : CARDINAL;
VAR
    cl,ch : CARDINAL;
BEGIN
    ch := ORD(BiosIO.RdKey());
    IF ch = 0 THEN
        cl:=ORD(BiosIO.RdKey());
    ELSE
        cl:=0;
    END;
    RETURN cl+(ch << 8);
END ReadKeycode;

PROCEDURE nextcolor (VAR c :CARDINAL);
BEGIN
    CASE c OF
    | barcolor1 : c:=barcolor2;
    | barcolor2 : c:=barcolor1;
    END;
END nextcolor;

PROCEDURE newcode (VAR v:CARDINAL; k:INTEGER);
VAR
    i:CARDINAL;
BEGIN
    IF k < 0 THEN
        FOR i:=1 TO ABS(k) DO
            IF v > firstcode THEN
                DEC(v);
            ELSE
                v:=maxcode;
            END;
        END;
    ELSE
        FOR i:=1 TO k DO
            IF v < maxcode THEN
                INC(v);
            ELSE
                v:=firstcode;
            END;
        END;
    END;
END newcode;

PROCEDURE procSort (sort,reversesort:BOOLEAN   );
VAR
    i : CARDINAL;
    exchange,test:BOOLEAN;
    ln1,ln2:LONGCARD;
    n1,n2:CARDINAL;
BEGIN
    FOR i := firstcode TO maxcode DO
        gcode [i]:=i;
    END;
    IF (sort OR reversesort) THEN
        exchange := TRUE;
        WHILE exchange DO
            exchange := FALSE;
            FOR i := firstcode TO maxcode-1 DO
                ln1 := gcount[i];
                ln2 := gcount[i+1];
                IF sort THEN
                    test := (ln1 < ln2);  (* keep A, B, C,... sequence *)
                ELSIF reversesort THEN
                    test := (ln2 < ln1); (* <= instead of < forces A, B, C,... sequence, but would provoke an infinite loop *)
                END;
                IF test THEN
                    gcount[i]   := ln2;
                    gcount[i+1] := ln1;
                    n1          := gcode[i];
                    n2          := gcode[i+1];
                    gcode[i]    := n2;
                    gcode[i+1]  := n1;
                    exchange    := TRUE;
                END;
            END;
        END;
    END;
END procSort;

PROCEDURE procView (usemouse,sort,reversesort:BOOLEAN);
CONST
    (*                    0        1         2         3         4 *)
    (*                    1234567890123456789012345678901234567890 *)
    msgHelpMouse       = "Mouse-cursor  Escape-exit";
    msgHelpKbd         = "Arrows Home End PgUp PgDn Tab Escape";
CONST
    msgHex             = "Hexadecimal : $~";
    msgDec             = "Decimal     : ~";
    msgChar            = "Character   : ~";
    msgPercentage      = "Percentage  : ~%";
    msgCount           = "Count       : ~";
    msgTotal           = "Total       : ~";
    nlines             = 6;
    dquote             = '"';
    squote             = "'";
CONST
    widata = countcode; (* codes [0..255] *)
    hedata = 100; (* percentage *)
    k      = 1+2+1+1+1;
VAR
    realcode,prevcode,currcode,hpos,vpos,charcode : CARDINAL;
    color,i,x,mx,my:CARDINAL;
    xbase,ybase:CARDINAL;
    v,highest:LONGREAL;
    S:str128;
    xy:Graph.TextCoords;
    R:str80;
    msinfo : MsMouse.MsData;
    ok,realsize:BOOLEAN;
BEGIN
    setupVideoScreen();

    xbase := xmin + (screenX-widata) DIV 2;
    ybase := ymin + hedata + k*2;

    color := cLIGHTGREEN;
    i:=Graph.SetTextColor(color);
    vpos  := vtabMin+vtabHeight-1-1;
    IF usemouse THEN
        S:=msgHelpMouse;
    ELSE
        S:=msgHelpKbd;
    END;
    hpos:=centerme(htabMin,htabWidth,S);
    xy:=Graph.SetTextPosition(vpos,hpos);
    Graph.OutText(S);
    DEC(vpos,nlines+1);

    color := cWHITE;
    Graph.Rectangle(xbase-k,ybase+k,xbase+widata+k,ybase-hedata-k,color,FALSE);
    FOR i:=firstcode TO maxcode DO
        IF i MOD 8 = 0 THEN (* inner hex *)
            color := cLIGHTYELLOW;
            Graph.Plot(xbase+i,ybase+k-1,color);
        END;
        IF i MOD 10 = 0 THEN (* outer dec *)
            color := cWHITE;
            Graph.Plot(xbase+i,ybase+k+1,color);
        END;
    END;

    color:=cWHITE;
    FOR i:=0 TO hedata DO
        IF i MOD 10 = 0 THEN
            Graph.Plot (xbase-k-1,ybase-i,color);
            Graph.Plot (xbase+widata+k+1,ybase-i,color);
        END;
    END;

    procSort(sort,reversesort);

    highest:=0.0;
    color := barcolor1;
    FOR i:=firstcode TO maxcode DO
        v := ( LONGREAL(gcount[i]) / LONGREAL(gtotal) ) * 100.0;
        IF v > highest THEN highest:=v; END;
        x:=xbase+i;
        plot(x,ybase,color,v);
        nextcolor(color);
    END;

    color := cBRIGHTWHITE;
    i:=Graph.SetTextColor(color);

    IF usemouse THEN
        mx := xbase * mouseXmul;
        my := ybase;
        MsMouse.SetPosition(INTEGER(my),INTEGER(mx) );
        MsMouse.Cursor(MsMouse._MS_SHOW);
    END;

    realsize:=TRUE; (* 0..100 *)
    currcode := firstcode;
    LOOP
        prevcode:=currcode;
        IF usemouse THEN
(*%T MOUSEDONE *)
            IF isKeyWaiting() THEN
                charcode:=ReadKeycode();
                CASE charcode OF
                | keyEscape:
                    EXIT;
                ELSE
                    errbip(TRUE);
                END;
            END;
            MsMouse.GetStatus(msinfo);
            mx:=CARDINAL(msinfo.col) DIV mouseXmul;
            my:=CARDINAL(msinfo.row);

            IF ( (mx >= xbase ) AND (mx <= (xbase+widata) ) ) THEN
                IF ( (my <= ybase) AND (my >= (ybase-hedata) ) ) THEN
                    currcode:=mx-xbase;
                END;
            END;
(*%E  *)
        ELSE
            IF isKeyWaiting() THEN
                charcode:=ReadKeycode();
                CASE charcode OF
                | keyHome: currcode:=firstcode;
                | keyEnd:  currcode:=maxcode;
                | keyEscape:
                    EXIT;
                | keyRight: newcode(currcode,1);
                | keyLeft:  newcode(currcode,-1);
                | keyPgUp:  newcode(currcode,-16);
                | keyPgDn:  newcode(currcode,16);
                | keyTab:
                     realsize := NOT(realsize);
                     color := barcolor1;
                     FOR i:=firstcode TO maxcode DO
                         v := ( LONGREAL(gcount[i]) / LONGREAL(gtotal) ) * 100.0;
                         IF  NOT(realsize) THEN v:=v * 100.0 / highest; END;
                         x:=xbase+i;
                         Graph.Line(x,ybase+1,x,ybase-hedata-1,cBLACK); (* erase old bar plus safety margins *)
                         plot(x,ybase,color,v);
                         nextcolor(color);
                     END;
                     FOR i:=0 TO hedata DO
                         CASE i OF
                         | 0,hedata: ok:=TRUE; color:=cWHITE; (* force 0 and 100 *)
                         ELSE
                             ok:=( ( i MOD 10 ) = 0);
                             IF realsize THEN
                                 color:=cWHITE;
                             ELSE
                                 color:=cBLACK;
                             END;
                         END;
                         IF ok THEN
                             Graph.Plot (xbase-k-1,ybase-i,color);
                             Graph.Plot (xbase+widata+k+1,ybase-i,color);
                         END;
                     END;
                ELSE
                    errbip(TRUE);
                END;
            END;
        END;

        FOR i:=1 TO 2 DO
            CASE i OF
            | 1: color:=cBLACK;         x:=prevcode;
            | 2: color:=cBRIGHTWHITE;   x:=currcode;
            END;
            Graph.Plot(xbase+x,ybase+2,color);
            Graph.Plot(xbase+x-1,ybase+2+1,color);
            Graph.Plot(xbase+x+1,ybase+2+1,color);
        END;

        realcode:=gcode[currcode];
        FOR i:=1 TO nlines DO
            CASE i OF
            | 1: S:=msgHex;
                 Str.Copy(R,hexdigits[realcode DIV 16]); (* assume $00..$FF range, of course ! *)
                 Str.Append(R,hexdigits[realcode MOD 16]);
            | 2: S:=msgDec;
                 R:=fmtlc ( LONGCARD(realcode), space, "",3);
            | 3: S:=msgChar;
                 CASE realcode OF
                 | 0..31:
                     Str.Concat(R,"Ctrl-",CHR(realcode+ORD("A")-1));
                 | 34:
                     Str.Concat(R,squote,CHR(realcode));
                     Str.Append(R,squote+"   "); (* same length as "Ctrl-?" *)
                 | 255:
                     R:=dquote+" "+dquote+"   "; (* invisible space *)
                 ELSE
                     Str.Concat(R,dquote,CHR(realcode));
                     Str.Append(R,dquote+"   "); (* same length as "Ctrl-?" *)
                 END;
            | 4: S:=msgPercentage;
                 v := ( LONGREAL(gcount[currcode]) / LONGREAL(gtotal) ) * 100.0;
                 R:= fmtpercent (v, space, comma , 2,3+1+2);
            | 5: S:=msgCount;
                 R:=fmtlc ( gcount[currcode], space, dot, wilc);
            | 6: S:=msgTotal;

                 R:=fmtlc ( gtotal, space, dot, wilc);
            END;
            Str.Subst(S,"~",R);
            hpos:=htabMin+2;
            xy:=Graph.SetTextPosition(vpos+i-1,hpos);
            Graph.OutText(S);
        END;
    END;
(*%T MOUSEDONE  *)
    IF usemouse THEN MsMouse.Cursor(MsMouse._MS_HIDE); END;
(*%E  *)
END procView;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    firstfile = 1;
    maxfile   = 1000;
VAR
    filename  : ARRAY [firstfile..maxfile] OF str16;
    lastfile  : CARDINAL;

PROCEDURE readMatching(spec:ARRAY OF CHAR;VAR lastfile:CARDINAL);
VAR
    found : BOOLEAN;
    entry : FIO.DirEntry;
BEGIN
    lastfile := firstfile-1;
    found := FIO.ReadFirstEntry(spec,allfiles,entry);
    WHILE found DO
        INC(lastfile);
        IF lastfile > maxfile THEN RETURN;END;
        Str.Copy(filename[lastfile],entry.Name);
        found := FIO.ReadNextEntry(entry);
    END;
END readMatching;

(* fix common cases  : "." -- "*\" -- "\*.*" *)

PROCEDURE fixdirspec (VAR R : ARRAY OF CHAR  );
VAR
    len : CARDINAL;
BEGIN
    len := Str.Length(R);
    IF len = 0 THEN RETURN; END; (* no risk ! *)
    IF same(R,dot) THEN Str.Copy(R,stardotstar); RETURN; END;        (* "." becomes "*.*" *)
    IF R[len-1]=backslash THEN Str.Append(R,stardotstar);RETURN;END; (* "*\" becomes "*\*.*" *)
END fixdirspec;

PROCEDURE buildbase (spec:ARRAY OF CHAR;VAR base:ARRAY OF CHAR);
VAR
    u,d,f8,e3 : str128; (* oversized just in CASE -- e3 include dot ! *)
BEGIN
    Lib.SplitAllPath(spec,u,d,f8,e3);
    Lib.MakeAllPath(base,u,d,"","");
    UpperCase(base);
END buildbase;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    defaultBarWidth = 25;
    minBarWidth     = 1;
    maxBarWidth     = 200;
VAR
    restricted,percentage,excludecontrol,sort,reversesort,semigraphics : BOOLEAN;
    eyecandy,hexval,decval,showheader,showglobal,showall,showbar:BOOLEAN;
    barwi:CARDINAL;
    hiresview,usemouse,mousehere,ignorecase,ignoreaccents : BOOLEAN;
VAR
    parmcount : CARDINAL;
    i,opt     : CARDINAL;
    S,R       : str128;
    status    : (waiting,gotspec);
    spec,base,path : str128;
    lc:LONGCARD;
BEGIN
    FIO.IOcheck := FALSE;
    WrLn; (* yes, here now for help, errors, etc. *)

    restricted     := TRUE;
    percentage     := FALSE;
    excludecontrol := FALSE;
    sort           := FALSE;
    reversesort    := FALSE;
    eyecandy       := FALSE;
    hexval         := FALSE;
    decval         := FALSE;
    showheader     := TRUE;
    showglobal     := FALSE;
    showall        := TRUE;
    showbar        := FALSE;
    semigraphics   := TRUE;
    hiresview      := FALSE;
    usemouse       := FALSE; (* default *)
    mousehere      := FALSE;
    ignorecase     := FALSE;
    ignoreaccents  := FALSE;
    barwi          := defaultBarWidth;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    status:=waiting;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "A"+delim+"ALL"+delim+
                                  "P"+delim+"PERCENTAGE"+delim+
                                  "C"+delim+"CONTROL"+delim+
                                  "E"+delim+"EYECANDY"+delim+
                                  "S"+delim+"SORT"+delim+
                                  "R"+delim+"REVERSE"+delim+
                                  "X"+delim+"HEX"+delim+
                                  "D"+delim+"DEC"+delim+
                                  "T"+delim+"TERSE"+delim+
                                  "G"+delim+"GLOBAL"+delim+
                                  "GG"+delim+
                                  "B"+delim+"BARGRAPH"+delim+
                                  "BB"+delim+
                                  "W:"+delim+"WIDTH:"+delim+
                                  "V"+delim+"VIEW"+delim+
                                  "I"+delim+"CASE"+delim+
                                  "II"+delim+"ACCENTS"
(*%T MOUSEDONE *)
                                                  +delim+
                                  "M"+delim+"MOUSE"+delim+
                                  "VV"+delim+"VM"
(*%E *)
                              );
            CASE opt OF
            | 1,2,3 :   abort(errHelp,"");
            | 4,5:      restricted     := FALSE;
            | 6,7:      percentage     := TRUE;
            | 8,9:      excludecontrol := TRUE;
            | 10,11:    eyecandy       := TRUE;
            | 12,13:    sort           := TRUE;
            | 14,15:    reversesort    := TRUE;
            | 16,17:    hexval         := TRUE;
            | 18,19:    decval         := TRUE;
            | 20,21:    showheader     := FALSE;
            | 22,23:    showglobal     := TRUE;
            | 24:       showglobal     := TRUE; showall:=FALSE;
            | 25,26:    showbar        := TRUE;
            | 27:       showbar        := TRUE; semigraphics:=FALSE;
            | 28,29:    IF GetLongCard(S,lc)=FALSE THEN abort(errBadNumber,S);END;
                        IF (lc < LONGCARD(minBarWidth)) OR (lc > LONGCARD(maxBarWidth)) THEN
                            abort(errBarWidthRange,"");
                        END;
                        barwi:=CARDINAL(lc);
            | 30,31:                   hiresview:=TRUE;
            | 32,33:    ignorecase := TRUE;
            | 34,35:    ignorecase := TRUE; ignoreaccents:=TRUE;
(*%T MOUSEDONE  *)
            | 36,37:    usemouse:=TRUE;
            | 38,39:    usemouse:=TRUE;hiresview:=TRUE;
(*%E *)
            ELSE
                abort(errUnknownOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE status OF
            | waiting : Str.Copy(spec,S);
            | gotspec : abort(errTooManyParms,S);
            END;
            INC (status);
        END;
    END;
    IF status=waiting THEN abort(errMissingSpec,"");END;
    IF (sort AND reversesort) THEN abort(errNonsense,"");END;
    IF (hexval AND decval) THEN abort(errNonsenseVal,"");END;

    fixdirspec(spec);
    readMatching(spec,lastfile);
    IF lastfile > maxfile THEN abort(errTooManyMatches,spec);END;
    IF lastfile < firstfile THEN abort(errNoMatch,spec);END;
    buildbase(spec,base);

    IF hiresview THEN
        Graph.GetVideoConfig(gvideoconfig);
        IF gvideoconfig.adapter # Graph._VGA THEN abort(errVGAneeded,""); END;

(*%T MOUSEDONE  *)
        mousehere := ( MsMouse.Reset() # MAX(INTEGER) );
        usemouse  := ( usemouse AND mousehere );
(*%E  *)
        showheader     := FALSE; (* -t *)
        showglobal     := TRUE;  (* -g  *)
        showall        := FALSE; (* -gg *)
        restricted     := FALSE; (* -a *)
        excludecontrol := FALSE;
    END;

    IF showheader THEN
        WrStr(msgRem+" "+banner);WrLn;
        WrLn;
    END;

    IF showglobal THEN
        FOR i:=firstcode TO maxcode DO
            gcount[i]:=0;
        END;
        gtotal     :=0;
        gctrlchars :=0;
        gfilesize  :=0;
    END;

    FOR i := firstfile TO lastfile DO
        Str.Concat(path,base,filename[i]);
        process (path,excludecontrol,eyecandy,showglobal,
                 ignorecase,ignoreaccents,
                filesize,total,ctrlchars, gfilesize,gtotal,gctrlchars);
        IF showall THEN
            IF filesize # 0 THEN
                IF i # firstfile THEN WrLn; END;
                dump (path,restricted,percentage,excludecontrol,
                     hexval,decval,sort,reversesort,
                     eyecandy,showheader,showglobal,showbar,semigraphics,
                     ignorecase,ignoreaccents,
                         filesize,total,ctrlchars,barwi);
            END;
        END;
    END;

    IF showglobal THEN
        FOR i:=firstcode TO maxcode DO
            count[i]:=gcount[i];
        END;
        IF hiresview THEN
            gvmode   := Graph._MRES16COLOR;
            S        := "320x200x8";
            IF Graph.SetVideoMode(gvmode)=FALSE THEN abort(errCannotSet,S); END;

            procView(usemouse,sort,reversesort);

            gvmode   := Graph._DEFAULTMODE;
            S        := "original";
            IF Graph.SetVideoMode(gvmode)=FALSE THEN abort(errCannotSet,S); END;
        ELSE
            Str.Copy(path,"all matching files");
            WrLn;
            dump (path,restricted,percentage,excludecontrol,
                 hexval,decval,sort,reversesort,
                 eyecandy,showheader,showglobal,showbar,semigraphics,
                 ignorecase,ignoreaccents,
                    gfilesize,gtotal,gctrlchars,barwi);
        END;
    END;

    abort(errNone,"");
END cCount.
