
(* ---------------------------------------------------------------
Title         Q&D CRC
Author        PhG
Overview      self-explanatory !
Notes         >= 386 !
              Ethernet CRC32 (also used by PKZIP)
              polynomial is $EDB88320 i.e.
              x32+x26+x23+x22+x16+x12+x11+x10+x8+x7+x5+x4+x2+x+1
              preconditioning value is FFFFFFFFh
              postconditioning procedure is bit-inversion

              X^32+X^26+X^23+X^22+X^16+X^12+X^11+X^10+X^8
              +X^7+X^5+X^4+X^2+X^1+X^0
              We take it "backwards" and put the highest-order term in
              the lowest-order bit. The X^32 term is "implied" ; the LSB
              is the X^31 term, etc. The X^0 term (usually shown as "+1")
              results in the MSB being 1.

              crc = 0xFFFFFFFF;
              for (i=0; i<packetlength; i++) {
                  index = (crc ^ *(packet++)) & 0x000000FF;
                  crc = ((crc >> 8) & 0x00FFFFFF) ^ CRCTable[index];
              }
              crc = ~crc;

              crcval is computed on count of bytes as added safety
Bugs
Wish List     assembly ? ah ah, only serious ! more general a subrs (buffer, count)

--------------------------------------------------------------- *)

IMPLEMENTATION MODULE QD_CRC;

IMPORT FIO;
FROM QD_CRC_A IMPORT asmCRC32withVal, asmCRC32;

CONST
    useASM    = TRUE; (* yes, now it is safe *)
CONST
    useINLINE = TRUE; (* dword xor *)

(* ------------------------------------------------------------ *)

(* all indexes must be from 0, and end at 255 *)
(* both arrays are LONGCARD *)

CONST
    firstByteVal    = 0;         (* for CRCsigma *)
    lastByteVal     = 255;
    firstTableIndex = 0;         (* for CRC *)
    lastTableIndex  = 255;

VAR
    CRC32table : ARRAY [firstTableIndex..lastTableIndex] OF LONGCARD;
    countTable : ARRAY [firstByteVal..lastByteVal] OF LONGCARD;
    computeSigma: BOOLEAN; (* ugly global hack... *)

(* ------------------------------------------------------------ *)

PROCEDURE InitCountTable ();
VAR
    n : CARDINAL;
BEGIN
    FOR n := firstByteVal TO lastByteVal DO
        countTable[n] := 0;
    END;
END InitCountTable;

(* ------------------------------------------------------------ *)

(*

w1+w2 = OR
w1*w2 = AND
w1/w2 = XOR
{0..15} - 16 = NOT
i IN w = true if bit i set in word w
INCL(w,i) = set
EXCL(w,i) = clear

*)

TYPE
    dBITSET = SET OF [0..31];

(*%F useINLINE *)
PROCEDURE xorDWORD (v1,v2 : LONGCARD  ) : LONGCARD;
BEGIN
    RETURN LONGCARD( dBITSET(v1) / dBITSET(v2) );
END xorDWORD;
(*%E *)
(*%T useINLINE *)
(* en code inline, le gain est... drisoire (exemple : 234s au lieu de 246 !) *)
TYPE
    code4 = ARRAY [1..4] OF BYTE;

(*# save                              *)
(*# call(reg_param =>(ax,bx,cx,dx))   *)
(*# call(reg_return=>(ax,bx)      )   *)
(*# call(inline    =>on           )   *)

INLINE PROCEDURE xorDWORD (v1,v2:LONGCARD) : LONGCARD = code4 (
033H, 0C1H, (* xor ax,cx *)
033H, 0DAH  (* xor bx,dx *)
);
(*# restore                           *)

(*%E  *)

(* ------------------------------------------------------------ *)

(*%F useASM  *)
PROCEDURE InitCRC32 ();
CONST
    polynomial      = 3988292384; (* $EDB88320 *)
VAR
    n   : CARDINAL;
    i   : CARDINAL;
    crc : LONGCARD;
BEGIN
    FOR n := firstTableIndex TO lastTableIndex DO
        crc := LONGCARD(n);
        FOR i := 1 TO 8 DO
            IF ( 0 IN dBITSET(crc) )=TRUE THEN (* crc AND 1 *)
                crc := xorDWORD( (crc >> 1) , polynomial);
            ELSE
                crc := crc >> 1;
            END;
        END;
        CRC32table[n]:=crc;
    END;
END InitCRC32;
(*%E  *)

(* ------------------------------------------------------------ *)

(*

we use a work buffer as to be MUCH faster :

       1Mb  200Kb  32Kb
no  153.35  28.23  4.67
*2   17.19   3.30  0.66
*4   17.02   3.24  0.66
*8   16.76   3.19  0.66
*16  16.26   3.13  0.66
*32  16.53   3.19  0.66

*)

CONST
    ioBufferSize    = (8*512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
VAR
    ioBuffer  : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
    TMPbuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)

CONST
    seedCRC   = LONGCARD(0FFFFFFFFH);
    invertCRC = LONGCARD(0FFFFFFFFH);
    (* mask24    = LONGCARD(000FFFFFFH); *)
    mask16    = LONGCARD(00000FFFFH);
    mask8     = LONGCARD(0000000FFH);

PROCEDURE ComputeCRC32 (S : ARRAY OF CHAR) : LONGCARD;
VAR
    hnd  : FIO.File;
    ch   : BYTE;
    got  : CARDINAL;
    tmp1 : LONGCARD;
    tmp2 : LONGCARD;
    i    : CARDINAL;
    crc  : LONGCARD;
    buffhere,buffsigmahere:FarADDRESS;
BEGIN
(*%T useASM *)
    buffhere      := FarADR(TMPbuffer);
    buffsigmahere := FarADR(countTable);
(*%E  *)
    InitCountTable;
    crc := seedCRC;
    hnd :=FIO.OpenRead(S);
    FIO.AssignBuffer(hnd,ioBuffer);
    LOOP
        got := FIO.RdBin(hnd,TMPbuffer,ioBufferSize);
        IF got = 0 THEN EXIT; END; (* size was a multiple of iobufferSize *)
(*%F useASM  *)
        FOR i := firstBufferByte TO got DO
            ch := TMPbuffer[i];
            IF computeSigma THEN INC(countTable[CARDINAL(ch)]); END;

            (* no speed gain if these lines are packed *)
            (*
            tmp1 := (crc >> 8);
            tmp2 := crc AND mask16;
            tmp2 := xorDWORD(tmp2,LONGCARD(ch));
            tmp2 := CRC32table[CARDINAL(tmp2 AND mask8)];
            crc  := xorDWORD(tmp1,tmp2);
            *)

            (* previous is safe but possibly overcomplicated, use alternate instead *)

            tmp2 := xorDWORD(crc,LONGCARD(ch)) AND mask8;
            tmp1 := (crc >> 8); (* AND mask24; is USELESS !!! *)
            crc  := xorDWORD(tmp1,CRC32table[CARDINAL(tmp2)]);

        END;
(*%E  *)
(*%T useASM  *)
        IF computeSigma THEN
            crc := asmCRC32withVal(buffhere,got,crc,buffsigmahere); (* // remember : got is a count ! *)
        ELSE
            crc := asmCRC32(buffhere,got,crc); (* // remember : got is a count ! *)
        END;
(*%E  *)
        IF got < ioBufferSize THEN EXIT; END;
    END;
    FIO.Close(hnd);
    RETURN xorDWORD(crc,invertCRC);
END ComputeCRC32;

(* ------------------------------------------------------------ *)

PROCEDURE ComputeCRCval () : LONGCARD;
CONST
    first = 0;
    last = (lastByteVal-firstByteVal+1) * SIZE(LONGCARD) -1;
VAR
    p    : POINTER TO ARRAY [first..last] OF BYTE;
    ch   : BYTE;
    tmp1 : LONGCARD;
    tmp2 : LONGCARD;
    i    : CARDINAL;
    crc  : LONGCARD;
    buffhere : FarADDRESS;
BEGIN
    IF computeSigma=FALSE THEN RETURN 0;END;
    crc := seedCRC;
(*%F useASM  *)
    p := ADR(countTable); (* not always a far so we use ADR *)
    FOR i := first TO last DO
        ch := p^[i];
        tmp2 := xorDWORD(crc,LONGCARD(ch)) AND mask8;
        tmp1 := (crc >> 8);
        crc  := xorDWORD(tmp1,CRC32table[CARDINAL(tmp2)]);
    END;
(*%E  *)
(*%T useASM  *)
    buffhere := FarADR(countTable);
    crc := asmCRC32(buffhere,(last+1),crc); (* // remember : last+1 because it is a count ! *)
(*%E  *)
    RETURN xorDWORD(crc,invertCRC);
END ComputeCRCval;

(* ------------------------------------------------------------ *)

PROCEDURE setSigmaUse (flag:BOOLEAN);
BEGIN
    computeSigma := flag;
END setSigmaUse;

(* ------------------------------------------------------------ *)

(* use current file position, and assume count is valid *)

PROCEDURE SegmentFileComputeCRC32 (hnd:FIO.File;count:LONGCARD) : LONGCARD;
VAR
    ch   : BYTE;
    got  : CARDINAL;
    tmp1 : LONGCARD;
    tmp2 : LONGCARD;
    i    : CARDINAL;
    crc  : LONGCARD;
    buffhere,buffsigmahere:FarADDRESS;
    wanted:CARDINAL;
BEGIN
(*%T useASM *)
    buffhere      := FarADR(TMPbuffer);
    buffsigmahere := FarADR(countTable);
(*%E  *)
    InitCountTable;
    crc := seedCRC;

    LOOP
        IF count=0 THEN EXIT; END;
        IF count < ioBufferSize THEN
            wanted:=CARDINAL(count);
            count:=0;
        ELSE
            wanted:=ioBufferSize;
            DEC(count,ioBufferSize);
        END;
        got := FIO.RdBin(hnd,TMPbuffer,wanted);
        IF got = 0 THEN EXIT; END; (* size was a multiple of iobufferSize *)
(*%F useASM  *)
        FOR i := firstBufferByte TO got DO
            ch := TMPbuffer[i];

            tmp2 := xorDWORD(crc,LONGCARD(ch)) AND mask8;
            tmp1 := (crc >> 8); (* AND mask24; is USELESS !!! *)
            crc  := xorDWORD(tmp1,CRC32table[CARDINAL(tmp2)]);

        END;
(*%E  *)
(*%T useASM  *)
        crc := asmCRC32(buffhere,got,crc); (* // remember : got is a count ! *)
(*%E  *)
        IF got < wanted THEN EXIT; END;
    END;

    RETURN xorDWORD(crc,invertCRC);
END SegmentFileComputeCRC32;

(* ------------------------------------------------------------ *)

BEGIN
(*%F useASM  *)
    InitCRC32; (* init table then forget it ! *)
(*%E  *)
    setSigmaUse(TRUE); (* default is compute sigma too, for compatibility *)
END QD_CRC.
