(* ---------------------------------------------------------------
Title         Q&D World's Smallest Political Quiz
Overview      see help
Usage         see help
Notes         as usual, Q&D rules ! :-(
              we only accept 640x480x256 PCX file with 16 colors
Bugs
Wish List     yes, we should use dynamic allocation... but what for ?
              append data files to EXE ? bah...

--------------------------------------------------------------- *)

MODULE Quiz;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;
IMPORT MATHLIB;

IMPORT Graph;

FROM IO IMPORT WrStr, WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim,
completedInit, completedShow, completedSHOW, completedEnd, completed,
cleantabs;

(* ------------------------------------------------------------ *)

CONST
    cr              = CHR(13);
    lf              = CHR(10);
    nl              = cr+lf;
    extEXE          = ".EXE";
    extDAT          = ".DAT";
    extPCX          = ".PCX";
    star            = "*";
    semicolon       = ";";
    colon           = ":";
    doublequote     = '"';
    singlequote     = "'";
    dot             = ".";
    period          = dot; (* ! *)
    equal           = "=";
    percent         = "%";
    dollar          = "$";
    escape          = CHR(27);
    infinite        = 0; (* for pause *)
CONST
    ProgEXEname     = "QUIZ";
    ProgTitle       = "Q&D World's Smallest Political Quiz";
    ProgVersion     = "v1.0b";
    (* ProgCopyright   = "(program by PhG)"; *)
    Banner          = ProgTitle+" "+ProgVersion; (* +" "+ProgCopyright; *)
CONST
    errNone             = 0;
    errHelp             = 1;
    errOption           = 2;
    errParameter        = 3;
    errDataNotFound     = 4;
    errRedirected       = 5;
    errAborted          = 6;
    errBadWidth         = 7;
    errCorrupted        = 8;
    errBadPause         = 9;
    errVGAneeded        = 10;
    errVGAmode          = 11;
    errPicNotFound      = 12;
    errBadCoor          = 13;
    errBadCoeff         = 14;
    errBadInk           = 15;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [-w:#] [-i:#] [-m:#] [-d:datafile["+extDAT+"]] [-t] [-c:#]"+nl+
nl+
"  -w:# line width (default is 74)"+nl+
"  -i:# maximum pause for intro ([0..30], 0=infinite, default is 5)"+nl+
"  -m:# maximum pause for map ([0..30], 0=infinite, default is 0)"+nl+
"  -d:$ data file (default is "+ProgEXEname+extDAT+")"+nl+
"  -t   do not show graph (640x480x256 PCX background file with 16 colors)"+nl+
"  -c:# dot color"+nl+
nl+
"If needed, you may adjust score dot position with -x:#, -y:# and -k:# options."+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errParameter:
        Str.Concat(S,"Useless ",einfo);Str.Append(S," parameter !");
    | errDataNotFound:
        Str.Concat(S,einfo," data file does not exist !");
    | errRedirected:
        S:="Output redirection is a nonsense !";
    | errAborted:
        S:="Aborted by user !";
    | errBadWidth :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," line width !");
    | errCorrupted:
        Str.Concat(S,"Corrupted ",einfo);Str.Append(S," data file !");
    | errBadPause :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," pause !");
    | errVGAneeded:
        S := "Map requires a VGA card !";
    | errVGAmode:
        Str.Concat(S,"VGA card would not set ",einfo);
        Str.Append(S," video mode !");
    | errPicNotFound:
        Str.Concat(S,einfo," picture file does not exist !");
    | errBadCoor :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," value !");
    | errBadCoeff :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," coefficient !");
    | errBadInk :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," ink value !");

    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp: ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

PROCEDURE warn4 (S0,S1,S2,S3:ARRAY OF CHAR );
BEGIN
    WrStr(S0);WrStr(S1);WrStr(S2);WrStr(S3);WrLn;
    WrLn;
END warn4;

PROCEDURE getCard (VAR wi:CARDINAL;
                   lower,upper:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    lc:LONGCARD;
BEGIN
    IF GetLongCard(S,lc)=FALSE THEN RETURN FALSE;END;
    IF lc > MAX(CARDINAL) THEN RETURN FALSE; END;
    wi:=CARDINAL(lc);
    IF ( (wi < lower) OR (wi > upper) ) THEN RETURN FALSE; END;
    RETURN TRUE;
END getCard;

PROCEDURE getInt (VAR wi:INTEGER;
                 lower,upper:INTEGER;S:ARRAY OF CHAR):BOOLEAN;
VAR
    lc:LONGINT;
BEGIN
    IF GetLongInt(S,lc)=FALSE THEN RETURN FALSE;END;
    IF lc > MAX(INTEGER) THEN RETURN FALSE; END;
    wi:=INTEGER(lc);
    IF ( (wi < lower) OR (wi > upper) ) THEN RETURN FALSE; END;
    RETURN TRUE;
END getInt;

PROCEDURE getReal (VAR wi:LONGREAL; S:ARRAY OF CHAR):BOOLEAN;
VAR
    p:CARDINAL;
    ok:BOOLEAN;
BEGIN
    p:=Str.CharPos(S,colon);
    IF p=MAX(CARDINAL) THEN
        p:=Str.CharPos(S,equal);
        IF p=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    Str.Delete(S,0,p+1);
    wi:=Str.StrToReal(S,ok);
    RETURN ok;
END getReal;

PROCEDURE assumeEXEpath (VAR pcxpath:ARRAY OF CHAR);
VAR
    u,d,n,e,S,n0,e0:str128;
BEGIN
    Lib.SplitAllPath(pcxpath,u,d,n,e);
    Lib.MakeAllPath(S,u,d,"","");
    IF same(S,"") THEN (* only f8e3 was specified *)
        Lib.ParamStr(S,0);
        Lib.SplitAllPath(S,u,d,n0,e0);
        Lib.MakeAllPath(pcxpath,u,d,n,e);
    END;
END assumeEXEpath;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
VAR
    ioBuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    PI      : LONGREAL;
    deg2rad : LONGREAL;

PROCEDURE inittrigo (  );
BEGIN
    PI      := 4.0 * MATHLIB.ATan(1.0);
    deg2rad := PI / 180.0;
END inittrigo;

PROCEDURE DegreesToRadians ( alphadeg : LONGREAL ) : LONGREAL ;
BEGIN
    RETURN (alphadeg * deg2rad);
END DegreesToRadians;

PROCEDURE RadiansToDegrees( alpha : LONGREAL ) : LONGREAL ;
BEGIN
    RETURN (alpha / deg2rad);
END RadiansToDegrees;

PROCEDURE PolarToRectangular (alphadeg:LONGREAL;radius:CARDINAL;
                              VAR x,y : INTEGER);
VAR
    alpha,dx,dy,co,si : LONGREAL;
BEGIN
    alpha := alphadeg;   (* +origin *)
    alpha := DegreesToRadians(alpha);
    co    := MATHLIB.Cos(alpha);
    si    := MATHLIB.Sin(alpha);
    dx    := LONGREAL(radius) * co;
    dy    := LONGREAL(radius) * si;
    x     := VAL(INTEGER,dx);
    y     := VAL(INTEGER,dy);
END PolarToRectangular;

PROCEDURE RectangularToPolar (x,y:INTEGER;
                              VAR radius:CARDINAL;VAR alphadeg:LONGREAL);
VAR
    xx,yy,numerator,divisor:LONGREAL;
    si,co,r,alpha:LONGREAL;
BEGIN
    xx:=LONGREAL(x);
    yy:=LONGREAL(y);
    numerator := xx;
    divisor   := yy;
    alpha     := MATHLIB.ATan2(numerator,divisor);
    r         := MATHLIB.Sqrt (xx * xx + yy * yy);
    radius    := CARDINAL(r);
    alphadeg  := RadiansToDegrees(alpha);
END RectangularToPolar;

(* ------------------------------------------------------------ *)

CONST
    answerYes      = 20;
    answerMaybe    = 10;
    answerNo       = 0;
CONST
    iPersonal      = 1;
    IEconomic      = 2;
    iPersonalScore = 3;
    iEconomicScore = 4;
    iProfile       = 5;

    iChYes         = 6;
    iChNo          = 7;
    iChMaybe       = 8;
    iChQuit        = 9;

    iYes           = 10;
    iNo            = 11;
    iMaybe         = 12;
    iQuit          = 13;
    iEscape        = 14;
    iQuestion      = 15;
    iOutOf         = 16;
    iAnswer        = 17;
    iPrefix        = 18;
    iPrefixAnswer  = 19;
    iPrefixWarn    = 20;
    iHitAnyKey     = 21;
    iPCXfile       = 22;
    firstMessage   = iPersonal;
    maxMessage     = iPCXfile;
    firstchoice    = iChYes;
    lastchoice     = iChMaybe;
CONST
    iLibertarian        = 1;
    iLeftLiberal        = 2;
    iCentrist           = 3;
    iRightConservative  = 4;
    iAuthoritarian      = 5;
    firstProfile        = iLibertarian;
    maxProfile          = iAuthoritarian;
CONST
    firstQuestion         = 1;
    maxQuestion           = 10;
    firstEconomicQuestion = (maxQuestion DIV 2) + 1;
VAR
    question    : ARRAY [firstQuestion..maxQuestion] OF str256; (* oversized *)
    message     : ARRAY [firstMessage..maxMessage] OF str256;
    profile     : ARRAY [firstProfile..maxProfile] OF str256;
    fullprofile : ARRAY [firstProfile..maxProfile] OF str1024;

(* ------------------------------------------------------------ *)

CONST
    graphrotation = LONGREAL(45); (* 360-45 rotate diagram clockwise by 45 degrees *)
    half          = 100 DIV 2; (* 0..100 -> -50..+50 *)

(* check if coordinates are in centrist area *)
(* fixed : isCentrist did not work, when P20E60 was centrist ! *)

PROCEDURE isCentrist (p,e:CARDINAL):BOOLEAN ;
CONST
    (* y=ax+b for quadrant with x >= 0 and y >= 0 *)
    a = INTEGER(-1);
    b = INTEGER(30); (* 50-20 *)
VAR
    xs,ys, ydiag:INTEGER;
    alphaScore,alphaCentrist:LONGREAL;
    radiusScore,radiusCentrist:CARDINAL;
BEGIN
    (* new coordinates centered on diagram center : 0..100 -> -50..+50 *)
    (* p=y, e=x *)
    xs:=INTEGER(e)-half;
    ys:=INTEGER(p)-half;
    (* remap coordinates TO quadrant with x >=0 and y >= 0 *)
    IF xs < 0 THEN xs:=ABS(xs); END;
    IF ys < 0 THEN ys:=ABS(ys); END;

    ydiag := a * xs + b;

    RETURN (ydiag >= ys);
END isCentrist;

PROCEDURE findProfile (p,e:CARDINAL):CARDINAL;
VAR
    i : CARDINAL;
BEGIN
    IF p < half THEN
        IF e < half THEN
            i:=iAuthoritarian;
        ELSE
            i:=iRightConservative;
        END;
    ELSE
        IF e < half THEN
            i:=iLeftLiberal;
        ELSE
            i:=iLibertarian;
        END;
    END;
    IF isCentrist(p,e) THEN i:=iCentrist; END;
    RETURN i;
END findProfile;

PROCEDURE readentry (hnd:FIO.File;
                     VAR R:ARRAY OF CHAR):BOOLEAN ;
VAR
    len,i:CARDINAL;
    c:CHAR;
BEGIN
    LOOP
        FIO.RdStr(hnd,R);
        LtrimBlanks(R);
        RtrimBlanks(R);
        CASE R[0] OF
        | CHR(0): ; (* empty string *)
        | semicolon : ; (* comment *)
        | dollar : RETURN TRUE; (* end of section *)
        | doublequote,singlequote:
            len:=Str.Length(R);
            IF len >= 2 THEN
                IF R[len-1]=R[0] THEN
                    R[len-1]:=0C;
                    Str.Delete(R,0,1);
                END;
            END;
            RETURN FALSE;
        ELSE
            RETURN FALSE; (* not done yet *)
        END;
        IF FIO.EOF THEN RETURN TRUE; END;
    END;
END readentry;

PROCEDURE readtext (hnd:FIO.File;
                    VAR R:ARRAY OF CHAR):BOOLEAN;
CONST
    opConcat    = "\+";
    lenOpConcat = 2;
VAR
    done:BOOLEAN;
    S:str1024;
    len,p:CARDINAL;
BEGIN
    Str.Copy(R,"");
    LOOP
        done:=readentry(hnd,S);
        IF done THEN EXIT; END;
        len:=Str.Length(S);
        p:=Str.Pos(S,opConcat);
        IF p < len THEN
            IF p # (len-lenOpConcat) THEN EXIT; END;
            S[p]:=0C;
            Str.Append(R,S);
        ELSE
            Str.Append(R,S);
            EXIT;
        END;
    END;
    RETURN done;
END readtext;

PROCEDURE dump (wi:CARDINAL; lastCR:BOOLEAN; para:ARRAY OF CHAR);
VAR
    ok:BOOLEAN;
    R:str128;
BEGIN
    ok:=dmpTTX (para,wi,R,TRUE);
    WHILE ok DO
        WrStr(R);
        ok:=dmpTTX(para,wi,R,FALSE);
        IF ok THEN
            WrLn;
        ELSE
            IF lastCR THEN WrLn;END;
        END;
    END;
END dump;

PROCEDURE keyflush (  );
VAR
    ch:CHAR;
BEGIN
    WHILE IO.KeyPressed() DO
        ch:=IO.RdKey();
    END;
END keyflush;

PROCEDURE eyecandypause (v:CARDINAL;S:ARRAY OF CHAR);
VAR
    i,j:CARDINAL;
    ch:CHAR;
BEGIN
IF v=infinite THEN
    video(S,TRUE);
    keyflush;
    REPEAT
    UNTIL IO.KeyPressed();
    keyflush;
    video(S,FALSE);
ELSE
    video(S,TRUE);
    keyflush;
    (* animInit(v, "[", "]", CHR(249) , CHR(254) ,"-\|/"); *)
    animInit(v, "(", ")", "-", "+" ,"-\|/"); (* could be ugly with windoze charset *)
    FOR i:=1 TO v DO
        FOR j:=1 TO 10 DO
           anim(animSHOW);
           Lib.Delay(100); (* 10*100=1000=1second *)
           IF IO.KeyPressed() THEN
               keyflush;
               anim(animEnd);
               anim(animClear);
               video(S,FALSE);
               RETURN;
           END;
        END;
        anim(animAdvance);
    END;
    anim(animEnd);
    anim(animClear);
    video(S,FALSE);
END;
END eyecandypause;

PROCEDURE waitpause (v:CARDINAL);
VAR
    i,j:CARDINAL;
    ch:CHAR;
BEGIN
IF v=infinite THEN
    keyflush;
    REPEAT
    UNTIL IO.KeyPressed();
    keyflush;
ELSE
    keyflush;
    FOR i:=1 TO v DO (* +1 needed if v=0 ! *)
        FOR j:=1 TO 10 DO
           Lib.Delay(100); (* 10*100=1000=1second *)
           IF IO.KeyPressed() THEN
               keyflush;
               RETURN;
           END;
        END;
    END;
END;
END waitpause;

(* uses a few globerks ! ;-) *)

PROCEDURE loadData (DEBUG:BOOLEAN;datafile,exefile:ARRAY OF CHAR;
                    useinternal:BOOLEAN;datapos:LONGCARD;
                    wi,pause:CARDINAL):CARDINAL;
CONST
    msgLoading = "Loading...";
VAR
    hnd:FIO.File;
    para:str4096; (* should do ! *)
    i,imax,rc:CARDINAL;
    done:BOOLEAN;
BEGIN

    IF useinternal THEN Str.Copy(datafile,exefile); END;

    hnd:=FIO.OpenRead(datafile);
    FIO.AssignBuffer(hnd,ioBuffer);

    IF useinternal THEN FIO.Seek(hnd,datapos);END;

    (* title *)
    done:=readentry(hnd,para);
    dump(wi,TRUE,para);
    WrLn;

    (* author *)
    done:=readentry(hnd,para);
    dump(wi,TRUE,para);
    WrLn;

    (* intro *)
    LOOP
        done:=readtext(hnd,para);
        IF done THEN EXIT;END;
        dump(wi,TRUE,para);
    END;
    WrLn;

    (* instructions *)
    LOOP
        done:=readtext(hnd,para);
        IF done THEN EXIT;END;
        dump(wi,TRUE,para);
    END;
    WrLn;

    IF NOT(DEBUG) THEN video(msgLoading,TRUE); END;

    (* questions *)
    FOR i:=firstQuestion TO maxQuestion DO
        IF readentry(hnd,para) THEN
            FIO.Close(hnd);
            video(msgLoading,FALSE);
            RETURN errCorrupted;
        END;
        Str.Copy(question[i],str256(para) );
        IF DEBUG THEN WrStr(para);WrLn;END;
    END;

    (* profiles *)
    FOR i:=firstProfile TO maxProfile DO
        IF readentry(hnd,para) THEN
            FIO.Close(hnd);
            video(msgLoading,FALSE);
            RETURN errCorrupted;
        END;
        Str.Copy(profile[i],str256(para) );
        IF DEBUG THEN WrStr(para);WrLn;END;
        done:=readtext(hnd,para);
        Str.Copy(fullprofile[i],para);
        IF DEBUG THEN WrStr(para);WrLn;END;
    END;

    (* messages *)
    FOR i:=firstMessage TO maxMessage DO
        IF readentry(hnd,para) THEN
            FIO.Close(hnd);
            video(msgLoading,FALSE);
            RETURN errCorrupted;
        END;
        Str.Copy(message[i],str256(para) );
        IF DEBUG THEN WrStr(para);WrLn;END;
    END;



    FIO.Close(hnd);
    IF NOT(DEBUG) THEN video(msgLoading,FALSE); END;

    eyecandypause(pause,message[iHitAnyKey]);

    RETURN errNone;
END loadData;

PROCEDURE getchoice (legal:ARRAY OF CHAR):CHAR;
VAR
    ch:CHAR;
BEGIN
    LOOP
        ch:=IO.RdKey();
        IF ch=CHR(0) THEN
            ch:=IO.RdKey();
        ELSE
            IF Str.CharPos(legal,ch) # MAX(CARDINAL) THEN EXIT; END;
            ch:=CAP(ch);
            IF Str.CharPos(legal,ch) # MAX(CARDINAL) THEN EXIT; END;
        END;
    END;
    RETURN ch;
END getchoice;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    cBLACK	       =0;
    cPALEBLUE 	   =1;
    cPALEGREEN     =2;
    cPALECYAN      =3;
    cPALERED       =4;
    cPALEMAGENTA   =5;
    cBROWN         =6;
    cPALEWHITE     =7; (* was WHITE *)
    cGRAY          =8;
    cBLUE          =9;
    cGREEN         =10;
    cCYAN          =11;
    cRED           =12;
    cMAGENTA       =13;
    cYELLOW        =14;
    cWHITE         =15; (* was BRIGHTWHITE *)

CONST
    gpaper = Graph._WHITE; (* same as PCX background *)
    gink   = cGRAY;
VAR
    cx,cy:INTEGER; (* lazyness ! *)
    xmin,xmax,ymin,ymax:INTEGER;

PROCEDURE isVGA (  ):BOOLEAN ;
VAR
    video:Graph.VideoConfig;
BEGIN
    Graph.GetVideoConfig(video);
    IF video.adapter # Graph._VGA THEN RETURN FALSE;END;
    RETURN TRUE;
END isVGA;

PROCEDURE Hires(paper:LONGCARD;ink:CARDINAL);
VAR
    video : Graph.VideoConfig;
    oldpaper : LONGCARD;
    oldink   : CARDINAL;
BEGIN
    IF Graph.SetVideoMode(Graph._VRES16COLOR) THEN
    END;
    Graph.GetVideoConfig(video);
    xmin:=0;
    xmax:=INTEGER(video.numxpixels);
    ymin:=0;
    ymax:=INTEGER(video.numypixels);
    cx := xmax DIV 2;
    cy := ymax DIV 2;
    DEC(xmax);
    DEC(ymax);
    oldpaper:=Graph.SetBkColor(paper);
    Graph.ClearScreen(Graph._GWINDOW);
    oldink:=Graph.SetTextColor(ink);
END Hires;

PROCEDURE TextMode ( );
BEGIN
    IF Graph.SetVideoMode(Graph._DEFAULTMODE) THEN
    END;
END TextMode;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    PCXrgbType = RECORD
        red,green,blue:BYTE;
    END;

    (* (almost) useless fields are indented again *)

    PCXheaderType = RECORD
     manufacturer    : BYTE; (* constant 10=$0A=ZSoft *)
     version         : BYTE; (* constant 5=PC Paintbrush v3.0+ *)
     encoding        : BYTE; (* constant 1=RLE *)
    bitsPerPixel     : BYTE; (* 1, 2, 4 or 8, i.e. mono, 4, 16 OR 256 colors  *)
	                  (* window, i.e. image dimensions : xmin, ymin, xmax, ymax *)
    leftmargin       : WORD;
    uppermargin      : WORD;
    rightmargin      : WORD; (* do not use xmax-xmin ! *)
    lowermargin      : WORD;
     hDPIresolution  : WORD; (* put 300 for instance *)
     vDPIresolution  : WORD; (* idem *)
    colormap         : ARRAY[0..15] OF PCXrgbType;
     reserved1       : BYTE; (* must be 0 *)
    NCP              : BYTE; (* number of color planes, 1 or 4 *)
    NBS              : WORD; (* number of bytes per scanline, always even *)
                            (* from here, fields may contain rubbish *)
     paletteInfo     : WORD; (* 1=color or BW, 2=grayscale *)
     hscreensize     : WORD;
     vscreensize     : WORD;
     reserved2       : ARRAY[0..53] OF BYTE; (* must be 0 *)
    END;

CONST
    palID = BYTE(0CH); (* 12=$0C *)
    palidsize=1;
    palsize = 3 * 256;

PROCEDURE waitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END waitVGAretrace;

PROCEDURE setDAC(index:CARDINAL;red,green,blue:BYTE);
CONST
    DACWriteIndex  = 03C8H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACWriteIndex,SHORTCARD(index));
    SYSTEM.Out (DACDataRegister, red);
    SYSTEM.Out (DACDataRegister, green);
    SYSTEM.Out (DACDataRegister, blue);
END setDAC;

PROCEDURE resetdac (index:CARDINAL;r,g,b,pcxshift:BYTE);
BEGIN
    r := r >> pcxshift;
    g := g >> pcxshift;
    b := b >> pcxshift;
    setDAC(index,r,g,b);
END resetdac;

PROCEDURE newpalette (h:PCXheaderType;colors:CARDINAL;hnd:FIO.File);
CONST
    palsize = 3 * 256;
VAR
    i : CARDINAL;
    r,g,b,id:BYTE;
    p:LONGCARD;
    n:CARDINAL;
BEGIN
    waitVGAretrace;
    CASE colors OF
    | 2:
        FOR i:= 0 TO 1 DO
            r:=h.colormap[i].red;
            g:=h.colormap[i].green;
            b:=h.colormap[i].blue;
            resetdac(i,r,g,b,2);
        END;
    | 256:
        p := FIO.Size(hnd);
        FIO.Seek(hnd,p-1 -palsize-palidsize+1);
        n := FIO.RdBin(hnd,id,1);
        IF id = palID THEN
            FOR i := 0 TO 255 DO
                n:=FIO.RdBin(hnd,r,1);
                n:=FIO.RdBin(hnd,g,1);
                n:=FIO.RdBin(hnd,b,1);
                resetdac(i,r,g,b,2);
            END;
        ELSE
            FOR i:= 0 TO 15 DO
                r:=h.colormap[i].red;
                g:=h.colormap[i].green;
                b:=h.colormap[i].blue;
                resetdac(i,r,g,b,2);
            END;
        END;
        p := SIZE(PCXheaderType);
        FIO.Seek(hnd,p);
    END;
END newpalette;

TYPE
    masktype = ARRAY [0..7] OF CARDINAL;
CONST
    bitmask = masktype(080H,040H,020H,010H,008H,004H,002H,001H);

CONST
    PCXCOUNTFLAG = BYTE(0C0H); (* 11000000 *)
    PCXCOUNTMASK = BYTE(03FH); (* 00111111 *)
    PCXMAXCOUNT  = 03FH;
CONST
    maxPlanes = 4;
    maxpixels = 1280;
    firstcode = 0;
    (* maxcode   = (1280 DIV 8) * maxPlanes -1; (* in bytes *) *)
    maxcode   = maxpixels * maxPlanes-1;
TYPE
    imgtype = RECORD
        picwidth : CARDINAL ;
        picheight: CARDINAL ;
        decodebuffer : ARRAY [firstcode..maxcode] OF BYTE;
    END;
VAR
    img : imgtype;

PROCEDURE decodeScanline (lastcode:CARDINAL;hnd:FIO.File);
VAR
    databyte:BYTE;
    p,count,n : CARDINAL;
BEGIN
    p := firstcode;
    WHILE p <= lastcode DO
        n:=FIO.RdBin(hnd,databyte,1);
        IF (databyte AND PCXCOUNTFLAG) = PCXCOUNTFLAG THEN
            count := CARDINAL(databyte AND PCXCOUNTMASK);
            n := FIO.RdBin(hnd,databyte,1);
            FOR n := 1 TO count DO
                img.decodebuffer[p]:=databyte;
                INC(p);
            END;
        ELSE
            img.decodebuffer[p]:=databyte;
            INC(p);
        END;
    END;
END decodeScanline;

PROCEDURE showBackground (S:ARRAY OF CHAR);
CONST
    xmin = 0;
    ymin = 0;
VAR
    h : PCXheaderType;
    colors,n,n0,lastcode:CARDINAL;
    lastx,lasty,x,y,x0:CARDINAL;
    bpp : CARDINAL;
    ndxmask,mask:CARDINAL;
    hnd:FIO.File;
BEGIN
    hnd:=FIO.OpenRead(S);
    FIO.AssignBuffer(hnd,ioBuffer);
    n:=FIO.RdBin(hnd,h,SIZE(PCXheaderType));
    (* colors 2 or 256 already checked for *)
    CASE CARDINAL(h.bitsPerPixel) OF
    | 1 : colors :=2;
    | 8 : colors :=256;
    END;

    newpalette(h,colors,hnd);

    lastx    := CARDINAL (h.rightmargin-h.leftmargin);
    lasty    := CARDINAL (h.lowermargin-h.uppermargin);
    lastcode := CARDINAL(h.NCP) * CARDINAL(h.NBS) -1;
    IF lastcode > maxcode THEN
        FIO.Close(hnd);
        RETURN;
    END;
    (*
    p := firstcode;
    FOR n := 1 TO CARDINAL(h.NCP) DO
        ndxPlane[n]:=p;
        INC(p,lastx+1);
    END;
    *)
    bpp:=CARDINAL(h.bitsPerPixel);
    waitVGAretrace;
    (*
        alternatively, we could directly decode to video screen,
        using Graph.Line for runs of the same color value...
    *)
    FOR y := ymin TO lasty DO
        decodeScanline(lastcode,hnd);
        CASE bpp OF
        | 1:
            FOR x := xmin TO lastx DO
                n := CARDINAL(img.decodebuffer[x DIV 8]);
                ndxmask := x MOD 8;
                mask := bitmask[ndxmask];
                IF (n AND mask)=mask THEN
                    n:=1;
                ELSE
                    n:=0;
                END;
                Graph.Plot(x,y,n);
            END;
        | 8:
            (*
            FOR x := xmin TO lastx DO
                n := CARDINAL(img.decodebuffer[x]);
                Graph.Plot(x,y,n);
            END;
            *)
            x0:=xmin;
            n0:=CARDINAL(img.decodebuffer[x0]);
            x:=xmin;
            LOOP
                INC(x);
                IF x > lastx THEN
                    Graph.HLine(x0,y,lastx,n0);
                    EXIT;
                END;
                n:=CARDINAL(img.decodebuffer[x]);
                IF n # n0 THEN
                    Graph.HLine(x0,y,x-1,n0);
                    x0:=x;
                    n0:=n;
                END;
            END;
        END;
    END;
    FIO.Close(hnd);
END showBackground;

(* assume 640x480x256 with a 16 colors palette *)

PROCEDURE chkPCX (S:ARRAY OF CHAR;VAR h:PCXheaderType):BOOLEAN ;
CONST
    kTen = BYTE(0AH);
    kOne = BYTE(1);
    kZero= BYTE(0);
VAR
    n : CARDINAL;
    hnd:FIO.File;
BEGIN
    hnd:=FIO.OpenRead(S);
    FIO.AssignBuffer(hnd,ioBuffer);
    n:=FIO.RdBin(hnd,h,SIZE(PCXheaderType));
    FIO.Close(hnd);
    IF n < SIZE(PCXheaderType) THEN RETURN FALSE; END;
    IF h.manufacturer # kTen THEN RETURN FALSE; END;
    IF h.encoding # kOne THEN RETURN FALSE;END;
    IF h.reserved1 # kZero THEN RETURN FALSE; END;
    (*
    FOR n:=0 TO 53 DO
        IF h.reserved2[n] # kZero THEN RETURN FALSE; END;
    END;
    *)
    RETURN TRUE;
END chkPCX;

PROCEDURE chkSupportedPCX (h:PCXheaderType):BOOLEAN;
CONST
    kFour= BYTE(4);
    kOne = WORD(1);
    kEight=BYTE(8);
VAR
    colors : CARDINAL;
BEGIN
    IF h.bitsPerPixel > kEight THEN RETURN FALSE; END;
    IF h.NCP > kFour THEN RETURN FALSE; END;
    IF h.paletteInfo # kOne THEN RETURN FALSE; END;
    CASE CARDINAL(h.NCP) OF
    | 1:
         CASE CARDINAL(h.bitsPerPixel) OF
         | 1 : colors :=2;
         | 8 : colors :=256;
         ELSE
             RETURN FALSE;
         END;
    | 4:
         CASE CARDINAL(h.bitsPerPixel) OF
         | 1 : (* colors :=16; *) RETURN FALSE;
         ELSE
             RETURN FALSE;
         END;
    ELSE
        RETURN FALSE;
    END;
    RETURN TRUE;
END chkSupportedPCX;

PROCEDURE getPictureSize(h:PCXheaderType;VAR picwidth,picheight:CARDINAL);
BEGIN
    picwidth := CARDINAL(h.rightmargin-h.leftmargin)+1;
    picheight:= CARDINAL(h.lowermargin-h.uppermargin)+1;
END getPictureSize;

PROCEDURE chkValidBackground (pic:ARRAY OF CHAR  ):BOOLEAN;
CONST
    expectedpicw = 640;
    expectedpich = 480;
VAR
    h : PCXheaderType;
    picw,pich:CARDINAL;
BEGIN
    IF chkPCX(pic,h)=FALSE THEN RETURN FALSE; END;
    IF chkSupportedPCX(h)=FALSE THEN RETURN FALSE; END;
    (* we know here we have 2 or 256 colors, let's check 640x480 *)
    getPictureSize(h,picw,pich);
    IF picw # expectedpicw THEN RETURN FALSE;END;
    IF pich # expectedpich THEN RETURN FALSE;END;
    RETURN TRUE;
END chkValidBackground;

PROCEDURE newxy (mycx,mycy:INTEGER; k:LONGREAL; VAR x,y:INTEGER);
VAR
    r:CARDINAL;
    alpha:LONGREAL;
BEGIN
    x:=INTEGER (LONGREAL(x-half)*k);
    y:=INTEGER (LONGREAL(y-half)*k);
    RectangularToPolar(x,y, r,alpha);
    PolarToRectangular (alpha+graphrotation,r, x,y);
    x:=mycx+x;
    y:=mycy-y;
END newxy;

PROCEDURE drawScore (p,e,radius,ink:CARDINAL; mycx,mycy:INTEGER; myK:LONGREAL);
VAR
    x,y:INTEGER;
BEGIN
    x:=INTEGER(e);
    y:=INTEGER(p);
    newxy(mycx,mycy,myK, x,y);
    Graph.Disc(x,y,radius,ink);
    Graph.Circle(x,y,radius+2,ink);
END drawScore;

PROCEDURE diagProfiles (  );
VAR
    p,e,n:CARDINAL;
    c:CHAR;
    S:str128;
BEGIN
    FOR p:=100 TO 0 BY -10 DO
        CASE p OF
        | 0:  c:="0";
        | 50: c:="5";
        | 100:c:="!";
        ELSE  c:=".";
        END;
        WrStr(c);
        FOR e:=0 TO 100 BY 10 DO
            n:=findProfile(p,e);
            CASE n OF
            | iLibertarian       : c:="*";
            | iLeftLiberal       : c:="l";
            | iCentrist          : c:="c";
            | iRightConservative : c:="r";
            | iAuthoritarian     : c:="a";
            END;
            WrStr(c);
        END;
        WrLn;
    END;
    WrStr(" 0....5....!");WrLn;
    WrLn;
    FOR p:=0 TO 100 BY 10 DO
        FOR e:=0 TO 100 BY 10 DO
            n:=findProfile(p,e);
            CASE n OF
            | iLibertarian       : S:="libertarian";
            | iLeftLiberal       : S:="left liberal";
            | iCentrist          : S:="centrist";
            | iRightConservative : S:="right conservative";
            | iAuthoritarian     : S:="authoritarian";
            END;
            WrStr("P");   IO.WrCard(p,3);
            WrStr(", E"); IO.WrCard(e,3);
            WrStr(" : ");
            WrStr(S);WrLn;
        END;
    END;
END diagProfiles;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    defaultInk   = cPALECYAN;
    scoreRadius  = 5;
    defaultK     = 2.28; (* based upon QUIZ.PCX *)
    minpos       = 0;
    maxpos       = 640-1;
    minWidth     = 1;
    maxWidth     = 256; (* longest possible line, eh eh *)
    defaultWidth = 74;
    minPause     = 0;
    maxPause     = 30;
    defaultIpause= 5;
    defaultMpause= 0;
    digitsCount  = 1; (* was 2 *)
    digitsScore  = 1; (* was 3 *)
VAR
    wi,ipause,mpause,ink: CARDINAL;
    showmap : BOOLEAN;
    exefile,datafile,pcxfile: str128;
    scorePersonal,scoreEconomic:INTEGER;
    para:str1024; (* more than enough here *)
    msgChoice,legalchoices : str16;
    mycx,mycy:INTEGER;
    myK:LONGREAL;
    datapos:LONGCARD;
VAR
    parmcount,i,opt,p:CARDINAL;
    S,R:str128;
    state:(waiting);
    DEBUG,DEBUGDIAGRAM : BOOLEAN;
    ch:CHAR;
    value:INTEGER;
    useinternal:BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    inittrigo();
    DEBUG := FALSE;
    DEBUGDIAGRAM:=FALSE;
    wi     := defaultWidth;
    ipause := defaultIpause;
    mpause := defaultMpause;
    showmap:= TRUE;
    mycx   := MAX(INTEGER);
    mycy   := MAX(INTEGER);
    myK    := defaultK;
    ink    := defaultInk;
    useinternal:=FALSE;
    datapos:=MAX(LONGCARD); (* safety default *)

    Lib.ParamStr(exefile,0); UpperCase(exefile); (* useless but just in case... *)
    Str.Copy(datafile,exefile);
    Str.Subst(datafile,extEXE,extDAT);

    state:=waiting;
    parmcount := Lib.ParamCount();
    (* IF parmcount=0 THEN abort(errHelp,"");END; *)
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R); (* try and fix Yet Another TopSpeed bug ! *)
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "DEBUG"+delim+
                                   "W:"+delim+"WIDTH:"+delim+
                                   "I:"+delim+"INTRO:"+delim+
                                   "T"+delim+"TEXT"+delim+
                                   "D:"+delim+"DATAFILE:"+delim+
                                   "M:"+delim+"MAP:"+delim+
                                   "X:"+delim+
                                   "Y:"+delim+
                                   "K:"+delim+
                                   "C:"+delim+"INK:"+delim+"COLOR:"+delim+
                                   "DIAGRAM"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4     : DEBUG := TRUE;
            | 5,6   : IF getCard(wi,minWidth,maxWidth,S)=FALSE THEN
                          abort(errBadWidth,S);
                      END;
            | 7,8   : IF getCard(ipause,minPause,maxPause,S)=FALSE THEN
                          abort(errBadPause,S);
                      END;
            | 9,10  : showmap:=FALSE;
            | 11,12 : GetString(R,datafile); (* uppercase *)
            | 13,14 : IF getCard(mpause,minPause,maxPause,S)=FALSE THEN
                          abort(errBadPause,S);
                      END;
            | 15 :    IF getInt(mycx,minpos,maxpos,S)=FALSE THEN
                          abort(errBadCoor,S);
                      END;
            | 16 :    IF getInt(mycy,minpos,maxpos,S)=FALSE THEN
                          abort(errBadCoor,S);
                      END;
            | 17 :    IF getReal(myK,S)=FALSE THEN
                          abort(errBadCoeff,S);
                      END;
            | 18,19,20 :
                      IF getCard(ink,cBLACK,cWHITE,S)=FALSE THEN
                          abort(errBadInk,S);
                      END;
            | 21:
                      DEBUGDIAGRAM:=TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting:
                abort(errParameter,S);
            END;
            INC(state);
        END;
    END;

    IF DEBUGDIAGRAM THEN
        diagProfiles();
        abort(errNone,"");
    END;
    IF ( (DEBUG=FALSE) AND (IsRedirected()) ) THEN abort(errRedirected,"");END;

    IF showmap THEN
        IF isVGA()=FALSE THEN abort(errVGAneeded,"");END;
    END;

    IF Str.CharPos(datafile,dot)=MAX(CARDINAL) THEN
        Str.Append(datafile,extDAT);
    END;

    IF FIO.Exists(datafile)=FALSE THEN
        abort(errDataNotFound,S);
    END;

    WrStr(Banner);WrLn;
    WrLn;

    IF loadData(DEBUG,datafile,exefile,useinternal,datapos,wi,ipause) # errNone THEN abort(errCorrupted,datafile);END;
    IF showmap THEN
        Str.Copy(pcxfile,message[iPCXfile]);
        UpperCase(pcxfile);
        IF Str.CharPos(pcxfile,dot)=MAX(CARDINAL) THEN
            Str.Append(pcxfile,extPCX);
        END;
        assumeEXEpath(pcxfile); (* assume extension has been set *)
        IF FIO.Exists(pcxfile) THEN
            IF chkValidBackground(pcxfile)=FALSE THEN
                showmap:=FALSE;
                warn4(message[iPrefixWarn],"Ignoring ",pcxfile," unsupported format background !");
            END;
        ELSE
            (* abort(errPicNotFound,pcxfile); *)
            showmap:=FALSE;
            warn4(message[iPrefixWarn],"Ignoring missing ",pcxfile," background !");
        END;
    END;

    Str.Copy(legalchoices,escape);
    Str.Copy(msgChoice," [");
    FOR i:=firstchoice TO lastchoice DO
        ch:=CHAR(message[i]);
        Str.Append(legalchoices,ch);
        Str.Append(msgChoice,ch);
        IF i < lastchoice THEN Str.Append(msgChoice,"/"); END;
    END;
    Str.Append(msgChoice,"]"); (* "] ?" was not so good an idea after all *)

    scorePersonal := 0;
    scoreEconomic := 0;
    i:= firstQuestion;
    LOOP
        WrStr(message[iPrefix]);WrStr(message[iQuestion]);
        IO.WrCard(i,digitsCount);
        WrStr(message[iOutOf]);IO.WrCard(maxQuestion,digitsCount);
        WrStr(period);WrLn;
        WrLn;
        Str.Concat(para,question[i],msgChoice);
        dump(wi,FALSE,para);
        ch:=getchoice(legalchoices);
        video(msgChoice,FALSE);
        WrLn;
        IF ch=CHAR(message[iChYes]) THEN
            value:=answerYes;
            p:=iYes;
        ELSIF ch=CHAR(message[iChNo]) THEN
            value:=answerNo;
            p:=iNo;
        ELSIF ch=CHAR(message[iChMaybe]) THEN
            value:=answerMaybe;
            p:=iMaybe;
        ELSIF ( (ch=CHAR(message[iChQuit])) OR (ch=escape) ) THEN
            (* WrStr(message[iQuit]);WrLn; *)
            WrLn;
            abort(errAborted,"");
        END;
        IF i < firstEconomicQuestion THEN
            INC(scorePersonal,value);
        ELSE
            INC(scoreEconomic,value);
        END;
        Str.Concat(S,doublequote,message[p]);  Str.Append(S,doublequote);
        Str.Prepend(S,message[iAnswer]); Str.Append(S,period);
        Str.Prepend(S,message[iPrefixAnswer]);
        WrLn;
        dump(wi,FALSE,S);
        WrLn;
        WrLn;
        INC(i);
        IF i > maxQuestion THEN EXIT; END;
    END;

    IF showmap THEN
        Hires(gpaper,gink);
        showBackground(pcxfile);
        IF mycx=MAX(INTEGER) THEN mycx:=cx;END;
        IF mycy=MAX(INTEGER) THEN mycy:=cy;END;
        drawScore(scorePersonal,scoreEconomic, scoreRadius,ink,mycx,mycy,myK);
        waitpause(mpause);
        TextMode();
        WrLn; (* just in case ! *)
    END;

    (* score ! *)

    WrStr(message[iPrefix]);WrStr(message[iPersonalScore]);
    IO.WrCard(scorePersonal,digitsScore);WrStr(percent+period);WrLn;
    WrStr(message[iPrefix]);WrStr(message[iEconomicScore]);
    IO.WrCard(scoreEconomic,digitsScore);WrStr(percent+period);WrLn;
    WrLn;

    i:=findProfile(scorePersonal,scoreEconomic);

    WrStr(message[iPrefix]);WrStr(message[iProfile]);
    WrStr(doublequote);WrStr(profile[i]);WrStr(doublequote+period);WrLn;
    WrLn;
    dump(wi,TRUE,fullprofile[i]);

    abort(errNone,"");
END Quiz.

