
(* ---------------------------------------------------------------
Title         Q&D Plasma demo
Author        PhG
Overview      tsk tsk...
Usage         see help
Notes         
              minimal error messages and checking, etc.
              LARGE model (and a big stack for safety)
              plasma algorithm is rather good (clearly not VERY good :
              not enough control on parms such as roughness etc.)
Bugs

Wish List     tsk tsk...

--------------------------------------------------------------- *)

MODULE Plasma;

IMPORT Lib;
IMPORT SYSTEM;
IMPORT IO;
IMPORT Str;
IMPORT BiosIO;
IMPORT MATHLIB;
IMPORT MsMouse;

FROM IO IMPORT WrStr, WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows, cleantabs,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode;

FROM Lib IMPORT Fill;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    maxWait    = 10;

CONST
    cr         = CHR(13);
    lf         = CHR(10);
    nl         = cr+lf;

CONST
    ProgEXEname   = "PLASMA";
    ProgTitle     = "Q&D Plasma";
    ProgVersion   = "v1.0a";
    ProgCopyright = "adapted by PhG"; (* rewritten and enhanced, too ! *)
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;

CONST
    errNone           = 0;
    errHelp           = 1;
    errIllegalParm    = 2;
    errUnknownOpt     = 3;
    errRange          = 4;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [-p] [-w] [-z] [-c:#]"+nl+
nl+
"  -p   show palette until keypress or 10 seconds"+nl+
"  -w   show world until keypress or 10 seconds"+nl+
"  -z   end on mouseclick too"+nl+
"  -c:# color palette"+nl;
VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errIllegalParm :
        Str.Concat(S,"Illegal ",einfo); Str.Append(S," parameter !");
    | errUnknownOpt :
        Str.Concat(S,"Unknown ",einfo); Str.Append(S," option !");
    | errRange :
        Str.Concat(S,"Value for ",einfo);
        Str.Append(S," not in legal range !");

    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ; (* nada *)
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

CONST
    hires = 13H; (* 320x200x256 *)
    text  = 03H; (* 80x25 *)
    xcount = 320;
    ycount = 200;
    xmin = 0;
    xmax = xcount-1;
    ymin = 0;
    ymax = ycount-1;
    xcenter = xmax DIV 2;
    ycenter = ymax DIV 2;
    screensize = xcount * ycount;
    mincolorindex = 0;
    maxcolorindex = 255;

VAR
    videoscreen [0A000H:0000H] : ARRAY [0..screensize-1] OF BYTE;
    (* no structure for faster access *)
    rRGB : ARRAY [mincolorindex..maxcolorindex] OF BYTE;
    gRGB : ARRAY [mincolorindex..maxcolorindex] OF BYTE;
    bRGB : ARRAY [mincolorindex..maxcolorindex] OF BYTE;

PROCEDURE setVideoMode (mode:CARDINAL);
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AH := 00H;
    R.AL := BYTE(mode);
    Lib.Intr(R,10H);
END setVideoMode;

PROCEDURE setDACandStore(index:CARDINAL;red,green,blue:BYTE);
CONST
    DACWriteIndex  = 03C8H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACWriteIndex,SHORTCARD(index));
    SYSTEM.Out (DACDataRegister, red);
    SYSTEM.Out (DACDataRegister, green);
    SYSTEM.Out (DACDataRegister, blue);
    (* special : store color components *)
    rRGB[index] := red;
    gRGB[index] := green;
    bRGB[index] := blue;
END setDACandStore;

PROCEDURE setDAC(index:CARDINAL;red,green,blue:BYTE);
CONST
    DACWriteIndex  = 03C8H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACWriteIndex,SHORTCARD(index));
    SYSTEM.Out (DACDataRegister, red);
    SYSTEM.Out (DACDataRegister, green);
    SYSTEM.Out (DACDataRegister, blue);
END setDAC;

PROCEDURE WaitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END WaitVGAretrace;

PROCEDURE getrndrange (lower,upper:INTEGER):INTEGER;
VAR
    range : INTEGER;
    rnd   : REAL;
BEGIN
    range := upper-lower+1;
    rnd := (REAL(range) * Lib.RAND()) + REAL(lower);
    RETURN INTEGER(rnd);
END getrndrange;

PROCEDURE waitKeyboardCode (  ):CARDINAL;
VAR
    c1,c2:CHAR;
BEGIN
    REPEAT
    UNTIL BiosIO.KeyPressed();
    c1 := BiosIO.RdKey();
    IF c1 = CHR(0) THEN
        c2 := BiosIO.RdKey();
    ELSE
        c2 := CHR(0);
    END;
    RETURN ORD(c1)*256 + ORD(c2);
END waitKeyboardCode;

PROCEDURE getKeyboardCode (VAR keycode:CARDINAL):BOOLEAN;
VAR
    c1,c2:CHAR;
BEGIN
    IF BiosIO.KeyPressed()=FALSE THEN RETURN FALSE; END;
    c1 := BiosIO.RdKey();
    IF c1 = CHR(0) THEN
        c2 := BiosIO.RdKey();
    ELSE
        c2 := CHR(0);
    END;
    keycode := ORD(c1)*256 + ORD(c2);
    RETURN TRUE;
END getKeyboardCode;

PROCEDURE flushKeyboard (  );
VAR
    c : CHAR;
BEGIN
    LOOP
        IF BiosIO.KeyPressed()=FALSE THEN EXIT; END;
        c := BiosIO.RdKey();
        IF c = CHR(0) THEN c := BiosIO.RdKey(); END;
    END;
END flushKeyboard;

PROCEDURE input (  );
VAR
    keycode:CARDINAL;
    h,m,s,ss:CARDINAL;
    start : LONGINT;
    now   : LONGINT;
BEGIN
    Lib.GetTime(h,m,s,ss);
    start := (LONGINT(h)*60+LONGINT(m))*60+LONGINT(s);
    LOOP
        IF getKeyboardCode(keycode) THEN EXIT; END;
        Lib.GetTime(h,m,s,ss);
        now := (LONGINT(h)*60+LONGINT(m))*60+LONGINT(s);
        IF ABS(now-start) > maxWait THEN EXIT; END;
    END;
END input;

(* ------------------------------------------------------------ *)

PROCEDURE blend (ndx,count:CARDINAL; startink, endink:LONGCARD);
CONST
    rshift = LONGCARD(16);
    gshift = LONGCARD(8);
VAR
    r1,g1,b1:INTEGER;
    r2,g2,b2:INTEGER;
    r,g,b:INTEGER;
    i : CARDINAL;
BEGIN
    r1 := INTEGER (startink >> rshift) MOD 40H;
    g1 := INTEGER (startink >> gshift) MOD 40H;
    b1 := INTEGER (startink          ) MOD 40H;
    r2 := INTEGER (endink   >> rshift) MOD 40H;
    g2 := INTEGER (endink   >> gshift) MOD 40H;
    b2 := INTEGER (endink            ) MOD 40H;

    FOR i := 1 TO count DO
        r := r1 + ((r2 - r1) * INTEGER(i) ) DIV INTEGER(count);
        g := g1 + ((g2 - g1) * INTEGER(i) ) DIV INTEGER(count);
        b := b1 + ((b2 - b1) * INTEGER(i) ) DIV INTEGER(count);
        IF r < 0 THEN r:=0; END;
        IF g < 0 THEN g:=0; END;
        IF b < 0 THEN b:=0; END;
        IF r > 3FH THEN r:=3FH; END;
        IF g > 3FH THEN g:=3FH; END;
        IF b > 3FH THEN b:=3FH; END;
        setDACandStore(ndx+i-1,BYTE(r),BYTE(g),BYTE(b));
    END;
END blend;

CONST
    black       = LONGCARD(0000000H);
    white       = LONGCARD(03F3F3FH); (* ega/vga range is $00..$3F *)
    red         = LONGCARD(03F0000H);
    green       = LONGCARD(0003F00H);
    darkgreen   = LONGCARD(0002000H);
    blue        = LONGCARD(000003FH);
    darkblue    = LONGCARD(0000020H);
    yellow      = LONGCARD(03F3F00H);
    darkred     = LONGCARD(0200000H);
    orange      = LONGCARD(02F2F00H);
    cyan        = LONGCARD(0003F3FH);

TYPE
    paltype = (normalpal,redpal,greenpal,bluepal,descentpal,weirdpal,lastpal);

PROCEDURE newPalette (pal:paltype);
VAR
    ndx,n, alpha : CARDINAL;
    i,r,g,b : CARDINAL;
    pi,v:LONGREAL;
BEGIN
    CASE pal OF
    | normalpal,lastpal:
        ndx := 0;         n := 64;
        blend (ndx,n,red,yellow);

        INC(ndx,n);       n := 64;
        blend (ndx,n,yellow,blue);

        INC(ndx,n);       n := 64;
        blend (ndx,n,blue,cyan);

        INC(ndx,n);       n := 64;
        blend (ndx,n,cyan,red);
    | redpal:
        ndx := 0;         n := 64;
        blend (ndx,n,darkred,red);
        INC(ndx,n);       n := 64;
        blend (ndx,n,red,white);
        INC(ndx,n);       n := 64;
        blend (ndx,n,white,red);
        INC(ndx,n);       n := 64;
        blend (ndx,n,red,darkred);
    | greenpal:
        ndx := 0;         n := 64;
        blend (ndx,n,darkgreen,green);
        INC(ndx,n);       n := 64;
        blend (ndx,n,green,white);
        INC(ndx,n);       n := 64;
        blend (ndx,n,white,green);
        INC(ndx,n);       n := 64;
        blend (ndx,n,green,darkgreen);
    | bluepal:
        ndx := 0;         n := 64;
        blend (ndx,n,darkblue,blue);
        INC(ndx,n);       n := 64;
        blend (ndx,n,blue,white);
        INC(ndx,n);       n := 64;
        blend (ndx,n,white,blue);
        INC(ndx,n);       n := 64;
        blend (ndx,n,blue,darkblue);
    | descentpal:
	    FOR i:=0 TO 64 DO
		    (* Make 0-63 be red shades *)
		    r:= i; g:= 0; b:= 0;
            setDACandStore(i,BYTE(r),BYTE(g),BYTE(b));
		    (* Make 64-127 be green shades *)
		    r:= 0; g:= i; b:= 0;
            setDACandStore(i+64, BYTE(r),BYTE(g),BYTE(b));
		    (* Make 128-191 be blue shades *)
		    r:= 0; g:= 0; b:= i;
            setDACandStore(i+128, BYTE(r),BYTE(g),BYTE(b));
		    (* Make 192-255 be greyscale *)
		    r:= i; g:= i; b:= i;
            setDACandStore(i+192,BYTE(r),BYTE(g),BYTE(b));
        END;
    | weirdpal: (* not useful for now... *)
        alpha:=256;
        pi   :=4.0 * MATHLIB.ATan(1.0);
        FOR i:= mincolorindex TO maxcolorindex DO
            v:=pi * LONGREAL(alpha) / 128.0;
            v:=MATHLIB.Cos(v) * 127.0;
            n:=CARDINAL(v);
            INC(n,128);
            n:=n >> 2;
            r:=n;
            g:=n;
            b:=n;
            setDACandStore(i,BYTE(r),BYTE(g),BYTE(b));
            DEC(alpha);
        END;
    END;
    setDACandStore(0,00H,00H,00H); (* 0 stays black *)
END newPalette;

PROCEDURE showPalette (  );
VAR
   i    : CARDINAL;
   xpos : CARDINAL;
   ypos : CARDINAL;
   p    : CARDINAL;
BEGIN
    Fill(ADR(videoscreen[0]),screensize,00H); (* index of black color *)
    FOR i := 0 TO 255 DO
        FOR ypos := ymin TO ymax DO
            p := ypos * xcount + i ;
            videoscreen[p]:=BYTE(i);
        END;
    END;
    input;
END showPalette;

PROCEDURE rotatePalette (  ):CARDINAL;
VAR
    i,ndx,p:CARDINAL;
    keycode:CARDINAL;
BEGIN
    FOR i := mincolorindex TO maxcolorindex DO
        WaitVGAretrace;
        IF getKeyboardCode(keycode) THEN RETURN keycode; END;
        FOR ndx := (mincolorindex+1) TO maxcolorindex DO
            p := (ndx+i) MOD (maxcolorindex+1);
            setDAC(ndx,rRGB[p],gRGB[p],bRGB[p]);
        END;
    END;
    RETURN MAX(CARDINAL);
END rotatePalette;

(* ------------------------------------------------------------ *)

VAR
    Ybase : ARRAY [ymin..ymax] OF CARDINAL;

PROCEDURE initYbase (  );
VAR
    p,i : CARDINAL;
BEGIN
    p := 0;
    FOR i := ymin TO ymax DO
        Ybase[i] := p;
        INC(p,xcount);
    END;
END initYbase;

(* ------------------------------------------------------------ *)

PROCEDURE newcolorIndex (xa,ya,x,y,xb,yb:CARDINAL);
CONST
    rough = 1; (* 1 *)
    deltacolorrange = 1; (* 1 *)
VAR
    avg        : INTEGER;
    colorindex : INTEGER;
    inka,inkb  : INTEGER;
BEGIN
    IF videoscreen[ x+Ybase[y] ] # BYTE(0) THEN RETURN; END;
    avg := ABS(xa-xb)+ABS(ya-yb);
    inka := INTEGER( videoscreen [xa + Ybase[ya] ] );
    inkb := INTEGER( videoscreen [xb + Ybase[yb] ] );
    colorindex := ( inka + inkb ) >> 1; (* fast DIV 2 *)
    colorindex := colorindex + getrndrange(-deltacolorrange,+deltacolorrange) * avg * rough;

    IF colorindex < (mincolorindex+1) THEN colorindex:=mincolorindex+1; END;
    IF colorindex > maxcolorindex THEN colorindex:=maxcolorindex; END;
    videoscreen[ x+Ybase[y] ] := BYTE(colorindex);
END newcolorIndex;

PROCEDURE doPlasma (x1,y1,x2,y2:CARDINAL);
VAR
    x,y:CARDINAL;
    ink1,ink2,ink3,ink4:INTEGER;
    colorindex:INTEGER;
BEGIN
    IF (x2-x1<2) AND (y2-y1<2) THEN RETURN; END;

    x:=(x1+x2) >> 1; (* fast DIV 2 *)
    y:=(y1+y2) >> 1; (* fast DIV 2 *)

    newcolorIndex(x1, y1, x, y1, x2, y1);
    newcolorIndex(x2, y1, x2, y, x2, y2);
    newcolorIndex(x1, y2, x, y2, x2, y2);
    newcolorIndex(x1, y1, x1, y, x1, y2);

    IF videoscreen[ x+Ybase[y] ] = BYTE(0) THEN
        ink1:=INTEGER( videoscreen[ x1 + Ybase[y1] ]);
        ink2:=INTEGER( videoscreen[ x2 + Ybase[y1] ]);
        ink3:=INTEGER( videoscreen[ x2 + Ybase[y2] ]);
        ink4:=INTEGER( videoscreen[ x1 + Ybase[y2] ]);
        colorindex:= (ink1+ink2+ink3+ink4) >> 2; (* fast DIV 4 *)
        IF colorindex < (mincolorindex+1) THEN colorindex:=mincolorindex+1; END;
        IF colorindex > maxcolorindex THEN colorindex:=maxcolorindex; END;
        videoscreen[ x+Ybase[y] ] := BYTE(colorindex);
    END;
    doPlasma(x1,y1,x ,y );
    doPlasma(x ,y1,x2,y );
    doPlasma(x ,y ,x2,y2);
    doPlasma(x1,y ,x ,y2);
END doPlasma;

PROCEDURE newWorld (  );
BEGIN
    Fill(ADR(videoscreen[0]),screensize,00H); (* index of black color *)

    (* set the corners even though probably useless *)

    videoscreen[ xmin+Ybase[ymin] ]:= BYTE(getrndrange(mincolorindex+1,maxcolorindex));
    videoscreen[ xmax+Ybase[ymin] ]:= BYTE(getrndrange(mincolorindex+1,maxcolorindex));
    videoscreen[ xmin+Ybase[ymax] ]:= BYTE(getrndrange(mincolorindex+1,maxcolorindex));
    videoscreen[ xmax+Ybase[ymax] ]:= BYTE(getrndrange(mincolorindex+1,maxcolorindex));

    doPlasma(xmin,ymin,xmax,ymax);

END newWorld;

PROCEDURE showWorld (  );
BEGIN
    input;
END showWorld;

(* ------------------------------------------------------------ *)

PROCEDURE newval (VAR val:INTEGER; plus, min,max : INTEGER);
VAR
    v:INTEGER;
BEGIN
    v := val+plus;
    IF ((v < min) OR (v > max)) THEN

    ELSE
        val := v;
    END;
END newval;

PROCEDURE ivalue (S:ARRAY OF CHAR;min,max:INTEGER;VAR r:INTEGER):BOOLEAN;
VAR
    v : LONGINT;
BEGIN
    IF GetLongInt(S,v)=FALSE THEN RETURN FALSE; END;
    IF v < LONGINT(min) THEN RETURN FALSE; END;
    IF v > LONGINT(max) THEN RETURN FALSE; END;
    r := INTEGER(v);
    RETURN TRUE;
END ivalue;

PROCEDURE mouseclick (  ):BOOLEAN;
VAR
    msdata:MsMouse.MsData;
BEGIN
    MsMouse.GetStatus(msdata);
    IF msdata.left_pressed THEN RETURN TRUE; END;
    IF msdata.right_pressed THEN RETURN TRUE; END;
    RETURN msdata.middle_pressed;
END mouseclick;

(* ------------------------------------------------------------ *)

CONST
    escape     = 01B00H;
    space      = 02000H;
    keyCR      = 00D00H;
VAR
    flagPalette : BOOLEAN;
    flagWorld   : BOOLEAN;
    stopmouse   : BOOLEAN;
    palette : paltype;
    keycode     : CARDINAL;
VAR
    parmcount   : CARDINAL;
    i           : CARDINAL;
    opt         : CARDINAL;
    S           : str128;
    R           : str128;
    v           : INTEGER;
BEGIN
    Lib.DisableBreakCheck();
    WrLn;                       (* must be here for pretty ulterior display ! *)

    flagPalette  := FALSE;
    flagWorld    := FALSE;
    stopmouse    := FALSE;
    palette      := normalpal;

    parmcount := Lib.ParamCount();

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);cleantabs(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "P"+delim+"PALETTE"+delim+
                                   "W"+delim+"WORLD"+delim+
                                   "Z"+delim+"MOUSE"+delim+
                                   "C:"+delim+"COLOR:"
                               );
            CASE opt OF
            | 1,2,3 :
                abort(errHelp,"");
            | 4,5 :
                flagPalette := TRUE;
            | 6,7 :
                flagWorld   := TRUE;
            | 8,9:
                stopmouse:=TRUE;
            | 10,11:
                IF ivalue(R,ORD(normalpal),ORD(lastpal),v)=FALSE THEN
                    abort(errRange,"color palette");
                END;
                palette:= paltype(v);

            ELSE
                abort(errUnknownOpt,S);
            END;
        ELSE
            abort(errIllegalParm,S);
        END;
    END;

    IF stopmouse THEN
        IF MsMouse.Reset()=MAX(INTEGER) THEN
            stopmouse := FALSE;
        END;
    END;

    initYbase();

    Lib.RANDOMIZE;
    setVideoMode(hires);

    newPalette(palette);
    IF flagPalette THEN showPalette; END;

    newWorld;
    IF flagWorld THEN showWorld; END;

    flushKeyboard;

    LOOP
        keycode := rotatePalette(); (* now include wait for vga retrace *)
        CASE keycode OF
        | space        : input;
        | escape,keyCR : EXIT;
        END;
        IF stopmouse THEN
            IF mouseclick() THEN EXIT; END;
        END;
    END;
    setVideoMode(text);

    abort(errNone,"");
END Plasma.
