(* ---------------------------------------------------------------
Title         Q&D XOSL BootItem dump
Author        PhG
Overview      see help
Usage         see help
Notes
Bugs
Wish List

--------------------------------------------------------------- *)

MODULE XOSLdump;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT IO;

FROM IO IMPORT WrStr, WrLn;

(* ------------------------------------------------------------ *)

TYPE
    str128 = ARRAY [0..127] OF CHAR;
    str16  = ARRAY [0..15] OF CHAR;
CONST
    ProgEXEname   = "XOSLDUMP";
    ProgTitle     = "Q&D XOSL BootItem dump";
    ProgVersion   = "v1.0a";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;

CONST
    errNone         = 0;
    errHelp         = 1;
    errJoker        = 2;
    errNotFound     = 3;
    errFormat       = 4;
    errOption       = 5;
    errParmOverflow = 6;
CONST
    defaultFile     = "C:\BOOTITEM";
    defaultExtension= ".XDF";
CONST
    gremlin = CHR(254); (* was blank *)

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    nl   = CHR(13)+CHR(10);
    help = Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [file["+defaultExtension+"]] [-z] [-v]"+nl+
nl+
'  -z truncate strings at CHR(0) (default is to show CHR(0) as "'+gremlin+'")'+nl+
"  -v dump all data"+nl+
nl+
'This program dumps XOSL infos (default file is "'+defaultFile+defaultExtension+'").'+nl;


VAR
    S : str128;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(help);
    | errJoker:
        S:="Jokers are not supported !";
    | errNotFound :
        Str.Concat(S,einfo," does not exist !");
    | errFormat:
        Str.Concat(S,einfo," is not expected size !");
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errParmOverflow:
        S := "Too many parameters !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp:
        ;
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

(*
items.h

class CBootItem {
	char ItemName[48];
	unsigned char PartIndex;
	unsigned short Hotkey;
	char Activate;
	char FixDriveNum;
	char Disabled;
		unsigned long Password;
	unsigned short Keys[16];
	unsigned char HideList[56];
	char SwapDrives;
	char Reserved[11]; // reserved for future use
    }; // sizeof(CBootItem) == 158

class CPartDesc {
	unsigned char Drive;
	unsigned long StartSector;
    }; // sizeof(CPartDesc) == 5

class CBootItemFile {
	CBootItem BootItems[24];
	CPartDesc PartList[56];
	unsigned char BootItemCount;
	unsigned char DefaultItem;
	unsigned char Reserved0; // was timeout
	unsigned char PartCount;
	unsigned short Timeout;
	char Reserved[18]; // reserved for future use
    }; //  sizeof(CBootItemFile) == 4096
*)

CONST
    countBootItem = 24;
    countPartDesc = 56;
TYPE
    CBootItem = RECORD (* 158 *)
	    ItemName    : ARRAY [0..47] OF CHAR;
	    PartIndex   : SHORTCARD;
	    Hotkey      : CARDINAL;
	    Activate    : SHORTCARD;
	    FixDriveNum : SHORTCARD;
	    Disabled    : SHORTCARD;
		Password    : LONGCARD;
	    Keys        : ARRAY [1..16] OF CARDINAL;
	    HideList    : ARRAY [1..56] OF SHORTCARD;
	    SwapDrives  : SHORTCARD;
	    Reserved    : ARRAY [1..11] OF SHORTCARD;
	END;
    CPartDesc = RECORD (* 5 *)
	    Drive       : SHORTCARD;
	    StartSector : LONGCARD;
    END;
    CBootItemFile = RECORD (* 4096 *)
        BootItems    : ARRAY [1..countBootItem] OF CBootItem;
	    PartList     : ARRAY [1..countPartDesc] OF CPartDesc;
	    BootItemCount: SHORTCARD;
	    DefaultItem  : SHORTCARD;
	    Reserved0    : SHORTCARD;
	    PartCount    : SHORTCARD;
	    Timeout      : CARDINAL;
	    Reserved     : ARRAY [1..18] OF SHORTCARD;
    END;

(* ------------------------------------------------------------ *)

PROCEDURE same (S1,S2:ARRAY OF CHAR ):BOOLEAN;
BEGIN
    RETURN ( Str.Compare(S1,S2) = 0 );
END same;

PROCEDURE num2str (wi,base:CARDINAL; v : LONGCARD):str16;
VAR
    ok:BOOLEAN;
    i:CARDINAL;
    R:str16;
    pad:CHAR;
BEGIN
    IF base = 16 THEN
        pad:="0";
    ELSE
        pad:=" ";
    END;
    Str.CardToStr (v,R,base,ok);
    FOR i:=Str.Length(R)+1 TO wi DO
        Str.Prepend(R,pad);
    END;
    IF base = 16 THEN Str.Lows(R); Str.Prepend(R,"$");END;
    RETURN R;
END num2str;

PROCEDURE dmppad (S:ARRAY OF CHAR);
CONST
    wi = 16;
VAR
    i:CARDINAL;
BEGIN
    WrStr(S);
    FOR i:=Str.Length(S)+1 TO wi DO
        WrStr(" ");
    END;
    WrStr(" : ");
END dmppad;

PROCEDURE dmp (S:ARRAY OF CHAR; wi,base:CARDINAL; v:LONGCARD);
VAR
    R:str16;
BEGIN
    dmppad(S);
    R:=num2str(wi,base,v);
    WrStr(R);WrLn;
END dmp;

PROCEDURE dmpC (S:ARRAY OF CHAR;wi,base:CARDINAL; v : CARDINAL);
BEGIN
    dmp(S,wi,base, LONGCARD(v));
END dmpC;

PROCEDURE dmpb (S:ARRAY OF CHAR;wi,base:CARDINAL; v : SHORTCARD);
BEGIN
    dmp(S,wi,base, LONGCARD(v));
END dmpb;

PROCEDURE dmpbuf (S:ARRAY OF CHAR;wi,base:CARDINAL; buf:ARRAY OF SHORTCARD);
VAR
    i,last:CARDINAL;
BEGIN
    dmppad(S);
    last:=SIZE(buf)-1;
    FOR i := 0 TO last DO
        WrStr( num2str(wi,base,LONGCARD(buf[i]) ) );
        IF i < last THEN WrStr(" ");END;
    END;
    WrLn;
END dmpbuf;

PROCEDURE dmpbufcard (S:ARRAY OF CHAR; buf:ARRAY OF CARDINAL);
VAR
    i,last:CARDINAL;
BEGIN
    dmppad(S);
    last:=( SIZE(buf) DIV 2 )-1;
    FOR i := 0 TO last DO
        WrStr( num2str(4,16,LONGCARD(buf[i]) ) );
        IF i < last THEN WrStr(" ");END;
    END;
    WrLn;
END dmpbufcard;

PROCEDURE dmpstr (truncate:BOOLEAN; S:ARRAY OF CHAR; buf:ARRAY OF CHAR);
VAR
    i,last:CARDINAL;
BEGIN
    dmppad(S);
    WrStr('"');
    last:=SIZE(buf)-1;
    (* FOR i := 0 TO last DO *)
    i:=0;
    LOOP
        IF i > last THEN EXIT; END;
        IF buf[i] = 0C THEN
            IF truncate THEN EXIT; END;
            WrStr(gremlin);
        ELSE
            WrStr(buf[i]);
        END;
        INC(i);
    END;
    WrStr('"') ; WrLn;
END dmpstr;

PROCEDURE dmphexstr (S:ARRAY OF CHAR; buf:ARRAY OF CHAR);
VAR
    i,last:CARDINAL;
BEGIN
    dmppad(S);
    last:=SIZE(buf)-1;
    FOR i := 0 TO last DO
        WrStr( num2str(2,16,LONGCARD(buf[i]) ) );
        IF i < last THEN WrStr(" ");END;
    END;
    WrLn;
END dmphexstr;

PROCEDURE dmpItem (DEBUG,truncate:BOOLEAN; i:CARDINAL; d : CBootItem  );
VAR
    H:str16;
    S:str128;
BEGIN
    H := "[~] ";
    Str.Subst(H, "~" , num2str(2,10,LONGCARD(i)) );

    Str.Concat(S,H,"ItemName");
    dmpstr(truncate,S,d.ItemName);

    IF DEBUG THEN
        Str.Concat(S,H,"ItemNAME");
        dmphexstr(S,d.ItemName);
    END;

    Str.Concat(S,H,"PartIndex"); (* //WARNING ! 8-ung ! we must add 1 for 1-based partdescs *)
    dmpb(S,1,10,d.PartIndex+1);

    Str.Concat(S,H,"Hotkey");
    dmpC(S,4,16,d.Hotkey);

    Str.Concat(S,H,"Activate");
    dmpb(S,2,16,d.Activate);

    Str.Concat(S,H,"FixDriveNum");
    dmpb(S,2,16,d.FixDriveNum);

    Str.Concat(S,H,"Disabled");
    dmpb(S,2,16,d.Disabled);

    Str.Concat(S,H,"Password");
    dmp(S,8,16,d.Password);

    Str.Concat(S,H,"Keys");
    dmpbufcard(S,d.Keys);

    Str.Concat(S,H,"HideList");
    dmpbuf(S,1,10,d.HideList);

    Str.Concat(S,H,"SwapDrives");
    dmpb(S,2,16,d.SwapDrives);

    Str.Concat(S,H,"Reserved");
    dmpbuf(S,2,16,d.Reserved);

END dmpItem;

PROCEDURE dmpDesc (i:CARDINAL; d : CPartDesc);
VAR
    H:str16;
    S:str128;
BEGIN
    H := "[~] ";
    Str.Subst(H, "~" , num2str(2,10,LONGCARD(i)) );

    Str.Concat(S,H,"Drive");
    dmpb(S,2,16,d.Drive);

    Str.Concat(S,H,"StartSector");
    dmp(S,1,10,d.StartSector);
END dmpDesc;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize  = (8 * 512) + FIO.BufferOverhead;
    firstIObuffer = 1;
    lastIObuffer  = ioBufferSize;
TYPE
    ioBufferType  = ARRAY[firstIObuffer..lastIObuffer] OF BYTE;
VAR
    ioBuffer    : ioBufferType;
    hnd         : FIO.File;
    i,wanted,got,parmcount: CARDINAL;
    S,F,R       : str128;
    data        : CBootItemFile;
    fsize       : LONGCARD;
    items,parts : CARDINAL;
    truncate,verbose,gotfile,DEBUG : BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    gotfile := FALSE;
    truncate:= FALSE;
    verbose := FALSE;
    DEBUG   := FALSE;
    Str.Copy(F,defaultFile);

    parmcount := Lib.ParamCount();
    FOR i:=1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Caps(S);
        CASE S[0] OF
        | "/" , "-":
           Str.Copy(R,S);
           Str.Delete(R,0,1); (* remove option marker *)
           (* boy, this is ugly ! *)
           IF    same(R,"Z") THEN     truncate := TRUE;
           ELSIF same(R,"V") THEN     verbose  := TRUE;
           ELSIF same(R,"DEBUG") THEN DEBUG    := TRUE;
           ELSIF same(R,"?") THEN     abort(errHelp,"");
           ELSIF same(R,"H") THEN     abort(errHelp,"");
           ELSE                       abort(errOption,S);
           END;
        ELSE
           IF gotfile THEN abort(errParmOverflow,"");END;
           Str.Copy(F,S);
           gotfile:=TRUE;
        END;
    END;

    IF Str.CharPos(F,".")=MAX(CARDINAL) THEN Str.Append(F,defaultExtension);END;
    IF Str.CharPos(F,"?") # MAX(CARDINAL) THEN abort(errJoker,F);END;
    IF Str.CharPos(F,"*") # MAX(CARDINAL) THEN abort(errJoker,F);END;

    IF FIO.Exists(F)=FALSE THEN abort(errNotFound,F); END;

    hnd:=FIO.OpenRead(F);
    FIO.AssignBuffer(hnd,ioBuffer);
    fsize:=FIO.Size(hnd);
    wanted:=SIZE(data);
    got:=FIO.RdBin(hnd,data,wanted);
    FIO.Close(hnd);
    IF fsize # LONGCARD(wanted) THEN abort(errFormat,F);END;

    IF verbose THEN
        items:= countBootItem;
        parts:= countPartDesc;
    ELSE
        items:= CARDINAL( data.BootItemCount );
        parts:= CARDINAL( data.PartCount );
    END;

    FOR i:=1 TO items DO
        dmpItem(DEBUG,truncate, i,data.BootItems[i]);
        WrLn;
    END;
    FOR i:=1 TO parts DO
        dmpDesc(i,data.PartList[i]);
        WrLn;
    END;

	dmpb   ("BootItemCount" ,1,10,data.BootItemCount);
	dmpb   ("DefaultItem"   ,2,16,data.DefaultItem);
	dmpb   ("Reserved0"     ,2,16,data.Reserved0);
	dmpb   ("PartCount"     ,1,10,data.PartCount);
	dmpC   ("Timeout"       ,1,10,data.Timeout);
	dmpbuf ("Reserved"      ,2,16,data.Reserved);

    abort(errNone,"");
END XOSLdump.
