(* ---------------------------------------------------------------
Title         Q&D XLAT filter
Overview      Yet Another Use<full|less> Program to remap chars to chars
Usage         see help
Notes         revising 15 years old code on a whim was not so difficult after all :
              m2 is neither asm6502 nor forth, eh eh
Bugs          wrong error message about string not being the same length
              when culprit line was a range with a mispelled ~command~
Wish List

--------------------------------------------------------------- *)

MODULE xlat;

IMPORT Lib;
IMPORT Str;
IMPORT IO;
IMPORT FIO;

FROM IO IMPORT WrStr,WrLn;

FROM Storage IMPORT ALLOCATE, DEALLOCATE, Available;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileIsDirectorySpec,
fileClose;

(* ------------------------------------------------------------ *)

CONST
    what        = "Q&D Xlat";
    version     = "v1.3b";
    who         = "by PhG";
    prog        = "XLAT";
    banner      = what+" "+version+" "+who;
CONST
    rem1        = ";";
    rem2        = "#";
CONST
    header      = prog+" : ";
    extGLO      = ".GLO";
    extLOG      = ".LOG";
    extBAK      = ".BK!";
    extTST      = ".TST";
    extXLT      = ".XLT";
    extCOM      = ".COM";
    extEXE      = ".EXE";
    extSYS      = ".SYS";
    extDLL      = ".DLL";
    extOVR      = ".OVR";
    extOVL      = ".OVL";
    extDRV      = ".DRV";
    extVXD      = ".VXD";
    extBIN      = ".BIN";
    extZIP      = ".ZIP";
    extARJ      = ".ARJ";
    extLZH      = ".LZH";
    skippedextensions = extBAK+delim+extCOM+delim+extEXE+delim+extSYS+delim+
                        extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                        extVXD+delim+extBIN+delim+
                        extZIP+delim+extARJ+delim+extLZH;
CONST
    idem        = "~IDEM~";
    nihil       = "~NIL~";
    reset       = "~RESET~";
    bit7off     = "~BIT7OFF~";
    bit7on      = "~BIT7ON~";
    plus1       = "~PLUS1~";
    minus1      = "~MINUS1~";
    dosnl       = "~NL~";
    doscrlf     = "~CRLF~";
    (* 1..9 *)
    TokenList   = idem+delim+
                  nihil+delim+
                  reset+delim+
                  bit7off+delim+
                  bit7on+delim+
                  plus1+delim+
                  minus1+delim+
                  dosnl+delim+
                  doscrlf;
CONST
    cr = CHR(13);
    lf = CHR(10);
    nl = cr+lf;
CONST
    sectionHead = "[_[_";
    sectionTail = "_]_]";
    sectionpattern = sectionHead+"*"+sectionTail;
CONST
    dot         = ".";
    dash        = "-";
    slash       = "/";
    question    = "?";
    doublequote = '"';
    colon       = ":";
    star        = "*";
    antislash   = "\";
    dirdotdot   = dot+dot;
    stardotstar = star+dot+star;
CONST
    nullchar    = 0C; (* CHR(0) *)
    blank       = ORD(" ");
CONST
    errNone         = 0;
    errHelp         = 1;
    errIn           = 2;
    errOut          = 3;
    errNotFound     = 4;
    errLineTooLong  = 5;
    errMissingQuote = 6;
    errBadHexVal    = 7;
    errBadVal       = 8;
    errEmptyString  = 9;
    errMissingSecondParm = 10;
    errThirdParm    = 11;
    errMissingClose = 12;
    errCloseNotEnd  = 13;
    errNotRange     = 14;
    errBadRange     = 15;
    errRanges       = 16;
    errRedefined    = 17;
    errRangeString  = 18;
    errStringRange  = 19;
    errValueRange   = 20;
    errStrings      = 21;
    errValueString  = 22;
    errUNdefined    = 23;
    errREdefinedWarn= 24;
    errLetterRange  = 25;
    errNoMatch      = 26;
    errTooMany      = 27;
    errBadExt       = 28;
    errOpt          = 29;
    errParm         = 30;
    errSyntax       = 31;
    errNoExtInSection=32;
    errList          =33;
    errSectionNotFound=34;
    errBadSpec      = 35;
    errListGen      = 36;
    errGen          = 37;
    errCharToString = 38;
    errJokerDest    = 39;
    errDestBadExt   = 40;
    errImpossible   = 255;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msghelp= banner+nl+
nl+
"This program transcodes characters from one set to another."+nl+
nl+
"Syntax 1 : "+prog+" [-r] [-u] [-z] [-debug] <glossary> < infile > outfile"+nl+
"Syntax 2 : "+prog+" [-r] [-u] [-z] [-debug] <glossary> <filespec> [destfile["+extXLT+"]]"+nl+
"Syntax 3 : "+prog+" <-list|-l>"+nl+
"Syntax 4 : "+prog+" <-gen|-!>"+nl+
nl+
"  -r      no redefinition allowed (default is to allow redefinitions)"+nl+
"  -u      no undefinition allowed (default is to allow undefinitions)"+nl+
"  -z      allow character to string assignment (default is char to char)"+nl+
"  -list   list available glossaries"+nl+
"  -gen    create "+prog+extTST+" test file in current directory"+nl+
"  -k      keep original date/time stamp"+nl+
"  -x      disable LFN support even if available"+nl+
(* "  -debug  create default "+prog+extLOG+" definition file in current directory"+nl+ *)
nl+
"a) Program will not process files with any of these extensions :"+nl+
"   "+skippedextensions+" for safety."+nl+
"b) Each original file is saved with "+extBAK+" extension ([destfile] is not)."+nl+
"c) Glossaries are defined in "+prog+extGLO+" file, located in executable directory."+nl+
'   A glossary section is between "'+sectionHead+'" and "'+sectionTail+'", without blanks.'+nl+
'   Definition format is "source target", where both elements can be a string,'+nl+
"   a quoted string, a decimal number, a hexadecimal ($) number or a range."+nl+
"   Target can also be any of the following self-explanatory tokens :"+nl+
"   "+TokenList+"."+nl+
"   If -z option was specified, target can be a string ([1..16] characters)."+nl+
"d) "+prog+extGLO+" uses DOS ASCII character set."+nl+
'e) Lines beginning with either "'+rem1+'" or "'+rem2+'" are comments :'+nl+
"   should these characters be redefined, enclose them with double quotes."+nl;

VAR
    S : str128;
BEGIN
    CASE e OF
    | errHelp:
        WrStr(msghelp);
    | errIn :
        S:="StandardInput error !";
    | errOut :
        S:="StandardOutput error !";
    | errNotFound :
        Str.Concat(S,einfo," does not exist !");
    | errLineTooLong :
        S:="Line longer than 128 characters !";
    | errMissingQuote :
        S:="Quote not matched by another quote !";
    | errBadHexVal :
        S:="Hexadecimal number either invalid or outside the [$00..$FF] range !";
    | errBadVal :
        S:="Decimal number either invalid or outside the [0..255] range !";
    | errEmptyString :
        S:="Quoted string empty !";
    | errMissingSecondParm :
        S:="Missing translation element !";
    | errThirdParm :
        S:="More than 2 elements in line !";
    | errMissingClose :
        S:="Missing ']' in range definition !";
    | errCloseNotEnd :
        S:="']' not at the end of range definition !";
    | errNotRange :
        S:="Missing '..' in range definition !";
    | errBadRange :
        S:="Invalid range definition !";
    | errRanges :
        S:="Different lengths for source and target ranges !";
    | errRedefined :
        S:="Attempt to redefine a code !";
    | errRangeString :
        S:="Different lengths for source range and target string !";
    | errStringRange :
        S:="Different lengths for source string and target range !";
    | errValueRange :
        S:="Range assigned to a value !";
    | errStrings :
        S:="Different lengths for source and target strings !";
    | errValueString :
        S:="Single character expected as a match to a code !";
    | errUNdefined :
        S:="At least one undefined code !";
    | errREdefinedWarn :
        S:="At least one code defined more than once !";
    | errLetterRange :
        S:="Single character expected as range limit !";
    | errNoMatch :
        S:="No match (or reserved extension) for <filespec> !";
    | errTooMany :
        S:="Too many files match <filespec> !";
    | errBadExt :
        S:="Source file extension cannot be "+extBAK+" !";
    | errOpt:
        S:="Illegal option !";
    | errParm:
        S:="Useless parameter !";
    | errSyntax:
        S:="Syntax error !";
    | errNoExtInSection:
        S:="Glossaries are no longer individual external files !";
    | errList:
        S:="-list command and processing are mutually exclusive !";
    | errSectionNotFound :
        Str.Copy(S,einfo);
    | errBadSpec:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," specification !");
    | errListGen:
        S:="-list and -gen commands are mutually exclusive !";
    | errGen:
        S:="-gen command and processing are mutually exclusive !";
    | errCharToString:
        S:="Target string is more than 16 characters long !";
    | errJokerDest:
        S:="<filespec> cannot contain any joker if [destfile] is specified !";
    | errDestBadExt:
        S:="Reserved [destfile] extension !";
    ELSE
        S:="How did you get there ???";
    END;
    CASE e OF
    | errHelp,errNone: ;
    ELSE
        WrStr(header); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    pFname = POINTER TO fnameType;
    fnameType = RECORD
        next      : pFname;
        slen      : CARDINAL; (* a SHORTCARD could do but who knows ? *)
        str       : CHAR;
    END;

PROCEDURE initList (VAR anchor : pFname );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pFname);
VAR
    needed : CARDINAL;
    p      : pFname;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(fnameType) - SIZE(anchor^.str) + anchor^.slen;
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pFname; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(fnameType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pFname);
VAR
    len:CARDINAL;
BEGIN
    len := p^.slen;
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

(* ------------------------------------------------------------ *)

(* Str.Match is not case sensitive *)

PROCEDURE isReservedPattern (S,skipthem:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN
    rc:=FALSE;
    n:=0;
    LOOP
        isoleItemS(e3, skipthem,delim,n);
        IF same(e3,"") THEN EXIT; END;
        Str.Prepend(e3,"*");
        IF Str.Match(S,e3) THEN rc:=TRUE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END isReservedPattern;

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dirdotdot);
END isReservedEntry;

PROCEDURE buildFileList (VAR anchor:pFname;
                        useLFN,DEBUG :BOOLEAN;spec:pathtype):CARDINAL;
VAR
    count:CARDINAL; (* should do ! *)
    ok,found:BOOLEAN;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
    dosattr:FIO.FileAttr;
    entryname:pathtype;
    len : CARDINAL;
    pp:pFname;
    excludeme1,excludeme2:BOOLEAN;
BEGIN
    count:=0;
    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        excludeme1 := isReservedEntry   (entryname);  (* skip "." and ".." *)
        excludeme2 := isReservedPattern (entryname,skippedextensions );
        IF NOT(excludeme1 OR excludeme2) THEN
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                (* if file has no extension, add it as a marker *)
                IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                    Str.Append(entryname,".");
                END;
                IF DEBUG THEN WrStr("Included : ");WrStr(entryname);WrLn; END;
                len:=Str.Length(entryname);
                IF buildNewPtr(anchor,pp,len)=FALSE THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL); (* errStorage *)
                END;
                INC(count);
                pp^.slen      := len;
                Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
            ELSE
                IF DEBUG THEN WrStr("Ignored  : ");WrStr(entryname);WrLn;END;
            END;
        ELSE
            IF DEBUG THEN WrStr("Excluded : ");WrStr(entryname);WrLn;END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN count;
END buildFileList;

PROCEDURE doGetCurrent (useLFN:BOOLEAN;drive:SHORTCARD;
                       VAR unit:str2; VAR current:pathtype);
VAR
    rc:CARDINAL;
    longform:pathtype;
BEGIN
    Str.Concat(unit, CHR( ORD("A")-1+ORD(drive) ),colon);

    FIO.GetDir(drive,current); (* \path without u: nor trailing \ except at root *)
    IF current[1] # colon THEN Str.Prepend(current,unit); END; (* safety *)
    IF useLFN THEN
        IF w9XshortToLong(current,rc,longform) THEN (* if error, keep DOS current *)
            Str.Copy(current,longform);
        END;
    END;
    (* LFN function seems to always return "u:\*" form except at root *)
    IF current[1] = colon THEN Str.Delete(current,0,2);END; (* safety *)
    fixDirectory(current);
END doGetCurrent;

PROCEDURE makebase (useLFN:BOOLEAN;spec:pathtype;VAR basepath:pathtype);
VAR
    u,d,n,e,current:pathtype;
    unit:str2;
    drive:SHORTCARD;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Str.Concat(basepath,u,d);
    IF same(basepath,"") THEN
        drive:=FIO.GetDrive(); (* yes we could use 0 as default drive *)
        doGetCurrent(useLFN,drive,  unit,current); (* "u:" and "\" or "\*\" *)
        Str.Concat(basepath, unit,current);
    END;
END makebase;

PROCEDURE WrQuoted (S:ARRAY OF CHAR);
BEGIN
    WrStr('"');  WrStr(S);  WrStr('"');
END WrQuoted;

PROCEDURE WrFname (useLFN:BOOLEAN;S:pathtype );
BEGIN
    IF useLFN THEN
        WrQuoted(S);
    ELSE
        WrStr(S);
    END;
END WrFname;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBufferType  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn : ioBufferType;
    bufferOut: ioBufferType;

(* ------------------------------------------------------------ *)

PROCEDURE LeftTrim (VAR S : ARRAY OF CHAR) : CARDINAL;
(* strip leading blanks in S, returning its new length *)
VAR
    i   : CARDINAL;
    len : CARDINAL;
BEGIN
    len := Str.Length(S);
    i   := 0;
    LOOP
        IF i = len THEN     (* a long line of blanks *)
            Str.Copy(S,"");
            len := 0;
            EXIT;
        END;
        IF ORD(S[i]) > blank THEN
            Str.Delete(S,0,i); (* procedure handles removal of 0 char *)
            DEC(len,i);
            EXIT;
        END;
        INC(i);
    END;
    RETURN len;
END LeftTrim;

PROCEDURE TrimQuotes (VAR S : ARRAY OF CHAR);
(* from "xxx" or even "" to xxx *)
VAR
    len : CARDINAL;
BEGIN
    len := Str.Length(S);
    Str.Delete(S,len-1,1); (* remove trailing doublequote *)
    Str.Delete(S,0,1);     (* remove leading doublequote *)
END TrimQuotes;

(* ------------------------------------------------------------ *)

PROCEDURE GetArg (VAR S : ARRAY OF CHAR; VAR R : ARRAY OF CHAR);
(* read string until first non-blank to R, removing it from S,
and ltriming S for next parsing *)
VAR
    i   : CARDINAL;
    len : CARDINAL;
BEGIN
    len := Str.Length(S);
    i   := 0;
    LOOP
        IF i = len THEN             (* no blanks in S or S="" *)
            Str.Copy(R,S);
            Str.Copy(S,"");
            EXIT;
        END;
        IF ORD(S[i]) <= blank THEN
            Str.Slice(R,S,0,i);
            Str.Delete(S,0,i);
            len := LeftTrim(S);
            EXIT;
        END;
        INC(i);
    END;
END GetArg;

PROCEDURE GetThisString (VAR S : ARRAY OF CHAR; VAR R : ARRAY OF CHAR) : BOOLEAN;
(* return first "string" without quotes, leaving S ready for next parsing
FALSE if no matching " was found for the first one *)
VAR
    i   : CARDINAL;
    len : CARDINAL;
    p   : CARDINAL;
BEGIN
    len := Str.Length(S);
    IF len < 2 THEN             (* cannot be empty, can be 1 char only *)
        RETURN FALSE;
    ELSE
        p := Str.NextPos(S,doublequote,1); (* we already know S[0] is a doublequote *)
        IF p = MAX(CARDINAL) THEN
            RETURN FALSE;
        ELSIF p = (len-1) THEN
            Str.Copy(R,S);
            Str.Copy(S,"");
            RETURN TRUE;
        ELSE
            Str.Slice(R,S,0,p+1);
            Str.Delete(S,0,p+1);
            len := LeftTrim(S);
            RETURN TRUE;
        END;
    END;
END GetThisString;

(* ------------------------------------------------------------ *)

PROCEDURE GetHexVal (S : ARRAY OF CHAR; VAR value : CARDINAL) : BOOLEAN;
VAR
    v  : LONGCARD;
    ok : BOOLEAN;
BEGIN
    Str.Delete(S,0,1); (* remove leading $ *)
    IF Str.Length(S)=0 THEN RETURN FALSE; END;
    Str.Caps(S); (* this routine does not like $ff and the like ! *)
    v := Str.StrToCard(S,16,ok);
    IF ok = FALSE THEN RETURN FALSE; END;
    IF v > MAX(SHORTCARD) THEN RETURN FALSE; END;
    value := CARDINAL(v);
    RETURN TRUE;
END GetHexVal;

PROCEDURE GetVal (S : ARRAY OF CHAR; VAR value : CARDINAL) : BOOLEAN;
VAR
    v  : LONGCARD;
    ok : BOOLEAN;
BEGIN
    v := Str.StrToCard(S,10,ok);
    IF ok = FALSE THEN RETURN FALSE; END;
    IF v > MAX(SHORTCARD) THEN RETURN FALSE; END;
    value := CARDINAL(v);
    RETURN TRUE;
END GetVal;

(* ------------------------------------------------------------ *)

CONST
    firstCode = MIN(SHORTCARD); (* ues, it's just 0 *)
    lastCode  = MAX(SHORTCARD); (* pretty complicated way of saying : $FF ! *)
    clearBit7 = 07FH; (* AND 01111111 *)
    setBit7   = 080H; (* OR  10000000 *)
CONST
    ONECHAR      = SIZE(SHORTCARD);
    MAXUSERTOKEN = 16;
    firstbincode = 1;
    maxbincode   = MAXUSERTOKEN;
TYPE
    chardeftype= RECORD
         enabled: BOOLEAN;
         defined: CARDINAL;
         count  : CARDINAL; (* yes, SHORTCARD would do too *)
         bindef : ARRAY[firstbincode..maxbincode] OF SHORTCARD; (* fake str16 *)
    END;
VAR
    newCode : ARRAY[firstCode..lastCode] OF chardeftype; (* globerk *)

(* ------------------------------------------------------------ *)

TYPE
    format = (nada,range,string,numvalue);

CONST
    openset = "[";
    hex     = "$";
    digit0  = "0";
    digit9  = "9";
    closeset= "]";
    dotdot  = "..";

PROCEDURE ParseParm (VAR SSS : ARRAY OF CHAR;
                     VAR fmt : format;
                     VAR R   : ARRAY OF CHAR;
                     VAR v1  : CARDINAL;
                     VAR v2  : CARDINAL) : CARDINAL;
VAR
    p   : CARDINAL;
    N   : str128;
BEGIN
    Str.Copy(R,""); (* silly safety but Murphy is always lurking... *)
    fmt := nada;
    v1  := 0;
    v2  := 0;
    CASE SSS[0] OF
    | openset :         (* find interval [##..##] *)
        GetArg(SSS,R);
        p := Str.CharPos(R,closeset);
        IF p = MAX(CARDINAL) THEN RETURN errMissingClose; END;
        IF p # (Str.Length(R)-1) THEN RETURN errCloseNotEnd; END;
        Str.Delete(R,p,1); (* now we have [##..## *)
        Str.Delete(R,0,1); (* now we have ##..## *)
        p := Str.Pos(R,dotdot);
        IF p = MAX(CARDINAL) THEN RETURN errNotRange; END;
        Str.Subst(R,dotdot," "); (* ## ## *)
        GetArg(R,N); (* first ## *)
        CASE N[0] OF
        | hex :
            IF GetHexVal(N,v1)=FALSE THEN RETURN errBadHexVal; END;
        | digit0..digit9 :
            IF GetVal(N,v1)=FALSE THEN RETURN errBadVal; END;
        ELSE
            IF Str.Length(N) # 1 THEN RETURN errLetterRange; END;
            v1 := ORD ( N[0] );
        END;
        GetArg(R,N); (* second ## *)
        CASE N[0] OF
        | hex :
            IF GetHexVal(N,v2)=FALSE THEN RETURN errBadHexVal; END;
        | digit0..digit9 :
            IF GetVal(N,v2)=FALSE THEN RETURN errBadVal; END;
        ELSE
            IF Str.Length(N) # 1 THEN RETURN errLetterRange; END;
            v2 := ORD ( N[0] );
        END;
        IF v1 >= v2 THEN RETURN errBadRange; END;
        fmt := range;
    | doublequote :           (* find quoted string "xxx" *)
        IF GetThisString(SSS,R) = FALSE THEN RETURN errMissingQuote; END;
        TrimQuotes(R);
        IF Str.Length(R) = 0 THEN RETURN errEmptyString; END;
        fmt := string;
    | hex :
        GetArg(SSS,R);
        IF GetHexVal(R,v1)=FALSE THEN RETURN errBadHexVal; END;
        fmt := numvalue;
    | digit0..digit9 :
        GetArg(SSS,R);
        IF GetVal(R,v1)=FALSE THEN RETURN errBadVal; END;
        fmt := numvalue;
    ELSE                (* find string xxx *)
        GetArg(SSS,R);
        fmt := string;
    END;
    RETURN errNone;
END ParseParm;

(* ------------------------------------------------------------ *)

PROCEDURE dumpdef (code:CARDINAL; hnd:FIO.File);
VAR
    i:CARDINAL;
    c:CHAR;
BEGIN
    (*
    FOR i:= firstbincode TO newCode[code].count DO
        c := CHR ( newCode[code].bindef[i] );
        FIO.WrChar(hnd,c);
    END;
    *)
    FIO.WrBin(hnd, newCode[code].bindef, newCode[code].count );
END dumpdef;

PROCEDURE charToChar (index,code:CARDINAL);
BEGIN
    INC( newCode[index].defined );
    newCode[index].bindef[firstbincode] := SHORTCARD(code);
    newCode[index].count := ONECHAR;
END charToChar;

PROCEDURE lazare (index : CARDINAL);
BEGIN
    charToChar(index, index);
    newCode[index].enabled  := TRUE;
END lazare;

PROCEDURE ignore (index : CARDINAL) ;
BEGIN
    INC(newCode[index].defined);
    newCode[index].enabled := FALSE;
END ignore;

PROCEDURE assignstring (index:CARDINAL; S:ARRAY OF CHAR);
VAR
    p,i,len,code:CARDINAL;
BEGIN
    len:=Str.Length(S);
    INC( newCode[index].defined );
    newCode[index].count := len;
    p:=firstbincode;
    FOR i:= 0 TO len-1 DO
        code:=ORD ( S[i] );
        newCode[index].bindef[p] := SHORTCARD( code);
        INC(p);
    END;
END assignstring;

PROCEDURE assign (index : CARDINAL; newcode : CARDINAL);
BEGIN
    charToChar(index, newcode);
END assign;

PROCEDURE assignit (index : CARDINAL; value : INTEGER);
VAR
    newcode : CARDINAL;
BEGIN
    newcode := index;   (* default will be reset character to its value *)
    IF value = -1 THEN
        IF newcode=MIN(SHORTCARD) THEN
            newcode:=MAX(SHORTCARD);
        ELSE
            DEC(newcode);
        END;
    ELSIF value = 1 THEN
        IF newcode=MAX(SHORTCARD) THEN
            newcode:=MIN(SHORTCARD);
        ELSE
            INC(newcode);
        END;
    END;
    charToChar(index,newcode);
END assignit;

(* ------------------------------------------------------------ *)

PROCEDURE doRange (start : CARDINAL;
                   fin   : CARDINAL;
                   SP2   : ARRAY OF CHAR;
                   type  : format;
                   startP2 : CARDINAL;
                   endP2   : CARDINAL) : CARDINAL;

VAR
    i,err,code,ndxtok,pp : CARDINAL;
BEGIN
    CASE type OF
    | range :
        IF (endP2-startP2+1) # (fin-start+1) THEN RETURN errRanges; END;
        FOR i := start TO fin DO
            assign(i,startP2);
            INC(startP2);
        END;
    | string :
        ndxtok:= getStrIndex (delim, SP2, TokenList);
        IF ndxtok = 0 THEN
            IF Str.Length(SP2) # (fin-start+1) THEN RETURN errRangeString; END;
            pp := 0;
            FOR i := start TO fin DO
                code := ORD(SP2[pp]);
                assign(i,code);
                INC(pp);
            END;
        ELSE
            FOR i := start TO fin DO
                CASE ndxtok OF
                | 1 : assign(i,i);
                | 2 : ignore(i);
                | 3 : lazare(i);
                | 4 : assign(i, (i AND clearBit7) );
                | 5 : assign(i, (i OR setBit7) );
                | 6 : assignit(i,  1 );
                | 7 : assignit(i, -1 );
                | 8 : assignstring(i,nl);
                | 9 : assignstring(i,nl);
                END;
            END;
        END;
    | numvalue :
        FOR i := start TO fin DO
            assign(i,startP2);
        END;
    END;
    RETURN errNone;
END doRange;

(* ------------------------------------------------------------ *)

PROCEDURE doString (S : ARRAY OF CHAR;
                    SP2  : ARRAY OF CHAR;
                    type : format;
                    startP2 : CARDINAL;
                    endP2   : CARDINAL;
                    CHARtoSTRING:BOOLEAN) : CARDINAL;
VAR
    err,code,codenow,fin,pp,ndxtok : CARDINAL;
BEGIN
    fin:=Str.Length(S)-1;
    CASE type OF
    | range :
        IF (fin+1) # (endP2-startP2+1) THEN RETURN errStringRange; END;
        FOR pp := 0 TO fin DO
            code := ORD(S[pp]);
            assign(code,startP2);
            INC(startP2);
        END;
    | string :
        ndxtok:= getStrIndex (delim, SP2, TokenList);
        IF ndxtok = 0 THEN
            IF ( CHARtoSTRING AND ( (fin+1) = 1 ) ) THEN
                IF Str.Length(SP2) > MAXUSERTOKEN THEN RETURN errCharToString;END;
                code:=ORD (S[0]);
                assignstring(code, SP2);
            ELSE
                IF (fin+1) # Str.Length(SP2) THEN RETURN errStrings; END;
                FOR pp := 0 TO fin DO
                    code   :=ORD(S[pp]);
                    codenow:=ORD(SP2[pp]);
                    assign(code,codenow);
                END;
            END;
        ELSE
            FOR pp := 0 TO fin DO
                code:=ORD(S[pp]);
                CASE ndxtok OF
                | 1 : assign(code,code);
                | 2 : ignore(code);
                | 3 : lazare(code);
                | 4 : assign(code, (code AND clearBit7) );
                | 5 : assign(code, (code OR setBit7) );
                | 6 : assignit (code,  1 );
                | 7 : assignit (code, -1 );
                | 8 : assignstring(code, nl);
                | 9 : assignstring(code, nl);
                END;
            END;
        END;
    | numvalue :
        FOR pp := 0 TO fin DO
            code := ORD(S[pp]);
            assign(code,startP2);
        END;
    END;
    RETURN errNone;
END doString;

(* ------------------------------------------------------------ *)

PROCEDURE doValue (value : CARDINAL;
                   SP2   : ARRAY OF CHAR;
                   type  : format;
                   startP2 : CARDINAL;
                   CHARtoSTRING : BOOLEAN) : CARDINAL;
VAR
    err,code,ndxtok : CARDINAL;
BEGIN
    CASE type OF
    | range :
        RETURN errValueRange;
    | string :
        ndxtok:= getStrIndex (delim, SP2, TokenList);
        IF ndxtok = 0 THEN
            IF CHARtoSTRING THEN
                IF Str.Length(SP2) > MAXUSERTOKEN THEN RETURN errCharToString;END;
                assignstring(value,SP2);
            ELSE
                IF Str.Length(SP2) # 1 THEN RETURN errValueString; END;
                code := ORD(SP2[0]);
                assign(value,code);
            END;
        ELSE
            CASE ndxtok OF
            | 1 : assign(value,value);
            | 2 : ignore(value);
            | 3 : lazare(value);
            | 4 : assign(value, (value AND clearBit7) );
            | 5 : assign(value, (value OR setBit7) );
            | 6 : assignit(value,  1 );
            | 7 : assignit(value, -1 );
            | 8 : assignstring(value, nl);
            | 9 : assignstring(value, nl);
            END;
        END;
    | numvalue :
        assign(value,startP2);
    END;
    RETURN errNone;
END doValue;

(* ------------------------------------------------------------ *)

PROCEDURE Parse (CHARtoSTRING:BOOLEAN;S : ARRAY OF CHAR) : CARDINAL;
VAR
    err     : CARDINAL;
    SP1     : str128;
    typeP1  : format;
    startP1 : CARDINAL;
    endP1   : CARDINAL;
    SP2     : str128;
    typeP2  : format;
    startP2 : CARDINAL;
    endP2   : CARDINAL;
BEGIN
    IF S[0] = rem1 THEN RETURN errNone; END;
    IF S[0] = rem2 THEN RETURN errNone; END;

    err := ParseParm(S,typeP1,SP1,startP1,endP1);
    IF err # errNone THEN RETURN err; END;

    IF Str.Length(S)=0 THEN RETURN errMissingSecondParm; END;

    err := ParseParm(S,typeP2,SP2,startP2,endP2);
    IF err # errNone THEN RETURN err; END;

    IF Str.Length(S)#0 THEN RETURN errThirdParm; END;

    CASE typeP1 OF
    | range :
        RETURN doRange (    startP1,endP1,SP2,typeP2,startP2,endP2);
    | string :
        RETURN doString(SP1,              SP2,typeP2,startP2,endP2,CHARtoSTRING);
    | numvalue :
        RETURN doValue (    startP1,      SP2,typeP2,startP2      ,CHARtoSTRING);
    END;
END Parse;

(* ------------------------------------------------------------ *)

PROCEDURE printable (c : CHAR):CHAR;
BEGIN
    IF ORD(c) < blank THEN
        RETURN ".";
    ELSE
        RETURN c;
    END;
END printable;

PROCEDURE num2str (n,digits,base: CARDINAL;prefix,padchar : CHAR):str16;
VAR
    ok   : BOOLEAN;
    v    : LONGCARD;
    len  : CARDINAL;
    R    : str16;
BEGIN
    v := LONGCARD(n);
    Str.CardToStr(v,R,base,ok);
    len := Str.Length(R);
    LOOP
        IF Str.Length(R) >= digits THEN EXIT; END;
        Str.Prepend(R,padchar);
    END;
    Str.Prepend(R,prefix);
    IF base=16 THEN Str.Lows(R);END;
    RETURN R;
END num2str;

PROCEDURE Report (VAR R : ARRAY OF CHAR; index : CARDINAL);
CONST
    sep2 = "  ";
    (* we can't use subst here for obvious reasons *)
    header =
(*
"###  $##  "?"    ###  $##  "?"    Y  (###)  !!! ++ "*"
*)
"Index            Code             Status        String (if applicable)";

VAR
    code : CARDINAL;
    v : CARDINAL;
    S : str128;
    ch: CHAR;
BEGIN
    IF index = MAX(CARDINAL) THEN
        Str.Copy(R,header);
        RETURN;
    END;

    ch:=CHR(index);
    Str.Concat(S, num2str(index,3,10,"" ," "), sep2);
    Str.Append(S, num2str(index,2,16,"$","0"));
    Str.Append(S, sep2+doublequote);
    Str.Append(S, printable ( ch ));
    Str.Append(S, doublequote+sep2+sep2);

    code:=ORD(newCode[index].bindef[firstbincode]);
    ch:=CHR(code);
    Str.Append(S, num2str(code,3,10,"" ," "));
    Str.Append(S, sep2);
    Str.Append(S, num2str(code,2,16,"$","0"));
    Str.Append(S, sep2+doublequote);
    Str.Append(S, printable ( ch ) );
    Str.Append(S, doublequote+sep2+sep2);

    IF newCode[index].enabled THEN
        Str.Append(S,"Y");
    ELSE
        Str.Append(S,"n");
    END;
    Str.Append(S,sep2+"(");

    v := newCode[index].defined;
    Str.Append(S,num2str(v,3,16,"" ," "));
    Str.Append(S,")"+sep2);

    CASE v OF
    | 0 : Str.Append(S,"#-#"); (* undefined *)
    | 1 : Str.Append(S,"..."); (* defined once *)
    ELSE
          Str.Append(S,"#+#"); (* defined ### times *)
    END;

    IF newCode[index].count # ONECHAR THEN
        Str.Append(S,sep2+doublequote);
        FOR  v:=firstbincode TO newCode[index].count DO
             Str.Append(S, printable( CHR(newCode[index].bindef[v])) );
        END;
        Str.Append(S,doublequote);
    END;

    Str.Copy(R,S);
END Report;

PROCEDURE Warning (count : CARDINAL;err:CARDINAL );
VAR
    S : str128;
BEGIN
    Str.Concat(S, num2str(count,3,10,"" ," ")," code");
    IF count > 1 THEN Str.Append(S,"s"); END;
    CASE err OF
    | errUNdefined :     Str.Append(S," undefined !");
    | errREdefinedWarn : Str.Append(S," defined more than once !");
    END;
    Str.Prepend(S,"Warning ! ");
    FIO.WrStr(FIO.ErrorOutput,S);
    FIO.WrLn(FIO.ErrorOutput);
END Warning;

(* ------------------------------------------------------------ *)

PROCEDURE LoadGlossary (LOG,REDEFINED,UNDEFINED,CHARtoSTRING,useLFN:BOOLEAN;
                       glofile:pathtype; F : ARRAY OF CHAR):CARDINAL;
VAR
    i    : CARDINAL;
    hnd  : FIO.File;
    S    : str128;
    orgS : str128;
    len  : CARDINAL;
    err  : CARDINAL;
    count : CARDINAL;
    udne,u,d,n,e:pathtype;
    id:str128;
BEGIN
    FIO.WrStr(FIO.ErrorOutput,banner); FIO.WrLn(FIO.ErrorOutput);
    FIO.WrLn(FIO.ErrorOutput);

    FOR i := firstCode TO lastCode DO
        newCode[i].defined := 0; (* reset definition count *)
        charToChar(i, i);        (* default : keep the same *)
        newCode[i].defined := 0; (* reset definition count again which was just incremented *)
        newCode[i].enabled := TRUE;
    END;

    Str.Concat(id,sectionHead,F);Str.Append(id,sectionTail);
    Str.Caps(id); (* useless in fact ! *)

    hnd := fileOpenRead (useLFN,glofile);
    FIO.AssignBuffer(hnd,bufferIn);

    LOOP
        IF FIO.EOF THEN fileClose(useLFN,hnd);RETURN errSectionNotFound;END;
        FIO.RdStr(hnd,S);
        IF Str.Match(S,sectionpattern) THEN
            Str.Caps(S);
            IF same(id,S) THEN EXIT; END;
        END;
    END;

    LOOP
        IF FIO.EOF THEN
            err := errNone;
            EXIT;
        END;
        FIO.RdStr(hnd,S);

        IF Str.Match(S,sectionpattern) THEN
            err := errNone;
            EXIT;
        END;

        len := Str.Length(S);
        IF len > HIGH(S) THEN
            Str.Copy(orgS,S);
            err := errLineTooLong;
            EXIT;
        END;
        len := LeftTrim(S);
        Str.Copy(orgS,S);
        IF len # 0 THEN
            err := Parse(CHARtoSTRING,S);
            IF err # errNone THEN EXIT; END;
        END;
    END;
    fileClose(useLFN,hnd);
    IF err # errNone THEN
        WrStr(header+"Error in this glossary line : ");WrStr(orgS);WrLn;
        RETURN err;
    END;

    (* now check for codes not defined *)

    count := 0;
    FOR i := firstCode TO lastCode DO
        IF newCode[i].defined = 0 THEN INC(count); END;
    END;

    IF count # 0 THEN
        IF UNDEFINED THEN
            Warning(count,errUNdefined);
        ELSE
            RETURN errUNdefined;
        END;
    END;

    (* now check for codes defined more than once *)

    count := 0;
    FOR i := firstCode TO lastCode DO
        IF newCode[i].defined > 1 THEN INC(count); END;
    END;

    IF count # 0 THEN
        IF REDEFINED THEN
            Warning(count,errREdefinedWarn);
        ELSE
            RETURN errREdefinedWarn;
        END;
    END;

    (* now always keep a copy of state of new codes *)

    IF LOG THEN
        Lib.ParamStr(S,0);
        Str.Caps(S);                    (* just in case... *)
        Str.Subst(S,extEXE,extLOG);
        Lib.SplitAllPath(S,u,d,n,e);
        Lib.MakeAllPath(udne,"","",n,e); (* always in current directory *)
        hnd := fileCreate(useLFN,udne);
        FIO.WrLn(hnd);
        FIO.WrStr(hnd,F); FIO.WrLn(hnd);
        FIO.WrLn(hnd);
        Report(S,MAX(CARDINAL));
        FIO.WrStr(hnd,S); FIO.WrLn(hnd);
        FIO.WrLn(hnd);

        FOR i := firstCode TO lastCode DO
            Report(S,i);
            FIO.WrStr(hnd,S); FIO.WrLn(hnd);
        END;
        fileClose(useLFN,hnd);
    END;
    RETURN errNone;
END LoadGlossary;

PROCEDURE newExtension (VAR R:pathtype;S:pathtype;e3:ARRAY OF CHAR);
VAR
    p : CARDINAL;
BEGIN
    Str.Copy(R,S);
    p := Str.RCharPos(R,dot);
    IF p = MAX(CARDINAL) THEN
        ;
    ELSE
        R[p]:=CHR(0);
    END;
    Str.Append(R,e3);
END newExtension;

PROCEDURE chkUD (S:pathtype):BOOLEAN ;
VAR
    u,d,n,e:pathtype;
    pb:CARDINAL;
BEGIN
    Lib.SplitAllPath(S , u,d,n,e);
    pb:=0;
    IF chkJoker(u) THEN INC(pb);END;
    IF chkJoker(d) THEN INC(pb);END;
    RETURN (pb=0);
END chkUD;

(* ------------------------------------------------------------ *)

CONST
    steps = 10;
    animcmd=animShow;
VAR
    lastportion,portion,currportion:LONGCARD;

CONST
    msgSkipping   = "Skipped    ";
    msgProcessing = "Processing ";
    msgCreating   = "Creating ";
    msgNow        = " ... "; (* was " : " *)
    msgOK         = "Done.";

VAR
    hin,hout  : FIO.File;
    c    : CHAR;
    S,R,filespec,destfile,glofile,backup,file,basedir,source,target : pathtype;
    i,opt    : CARDINAL;
    code : CARDINAL;
    parmcount : CARDINAL;
    p         : CARDINAL;
    got       : CARDINAL;
    state     : (waiting,gotglossary,gotspec,gotdest);
    cmd       : (oplist,opredir,opfile,opgen);
    section   : str128;
    flagREDEFINED,flagUNDEFINED,flagCHARtoSTRING,DEBUG,useLFN:BOOLEAN;
    doList,doGen,doCreate,doKeepStamp:BOOLEAN;
    ptr,anchor  : pFname;
    countFile : CARDINAL;
    stamp : LONGCARD;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    flagREDEFINED    := TRUE; (* allow redefinition *)
    flagUNDEFINED    := TRUE; (* allow undefinition *)
    flagCHARtoSTRING := FALSE;
    useLFN           := TRUE;
    DEBUG            := FALSE;
    doList           := FALSE;
    doGen            := FALSE;
    doCreate         := FALSE;
    doKeepStamp      := FALSE;

    state:=waiting;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "R"+delim+
                                   "U"+delim+
                                   "L"+delim+"LIST"+delim+
                                   "X"+delim+"LFN"+delim+
                                   "Z"+delim+
                                   "!"+delim+"GEN"+delim+
                                   "K"+delim+"STAMP"+delim+
                                   "DEBUG"
                               );
            CASE opt OF
            | 1,2,3: abort(errHelp,"");
            | 4:     flagREDEFINED    := FALSE;
            | 5:     flagUNDEFINED    := FALSE;
            | 6,7:   doList           := TRUE;
            | 8,9:   useLFN           := FALSE;
            | 10:    flagCHARtoSTRING := TRUE;
            | 11,12: doGen            := TRUE;
            | 13,14: doKeepStamp      := TRUE;
            | 15:    DEBUG            := TRUE;
            ELSE
                abort(errOpt,S);
            END;
        ELSE
            CASE state OF
            | waiting:      IF Str.CharPos(R,dot) # MAX(CARDINAL) THEN
                                 abort(errNoExtInSection,"");
                            END;
                            Str.Copy(section,R);
            | gotglossary:  Str.Copy(filespec,R);
            | gotspec:      Str.Copy(destfile,R);
            | gotdest:      abort(errParm,R);
            END;
            INC(state);
        END;
    END;
    IF (doList AND doGen) THEN abort(errListGen,"");END;
    CASE state OF
    | waiting:
         IF doList THEN
             cmd:=oplist;
         ELSE
             IF doGen THEN
                 cmd:=opgen;
             ELSE
                 abort(errSyntax,"");
             END;
         END;
    | gotglossary:
         IF doList THEN abort(errList,"");END;
         IF doGen  THEN abort(errGen,"");END;
         cmd:=opredir;
    | gotspec:
         IF doList THEN abort(errList,"");END;
         IF doGen  THEN abort(errGen,"");END;
         cmd:=opfile;
    | gotdest:
         IF chkJoker(filespec) THEN abort(errJokerDest,"");END;
         IF Str.RCharPos(destfile,dot)=MAX(CARDINAL) THEN Str.Append(destfile,extXLT);END;
         IF isReservedPattern (destfile,skippedextensions ) THEN abort(errDestBadExt,"");END;
         doCreate:=TRUE;

         IF doList THEN abort(errList,"");END;
         IF doGen  THEN abort(errGen,"");END;
         cmd:=opfile;
    END;

    useLFN:=( useLFN AND w9XsupportLFN() );
    IF NOT(useLFN) THEN UpperCase(filespec);END;

    CASE cmd OF
    | opgen :
        ; (* don't care about glossary here *)
    ELSE
        Lib.ParamStr(glofile,0);
        Str.Caps(glofile);                    (* just in case... *)
        Str.Subst(glofile,extEXE,extGLO);
        IF fileExists(useLFN,glofile)=FALSE THEN abort(errNotFound,glofile); END;
    END;

    CASE cmd OF
    | oplist:
        WrStr("Glossaries found in ");WrStr(glofile);WrLn;WrLn;
        hin:=fileOpenRead(useLFN,glofile);
        FIO.AssignBuffer(hin,bufferIn);
        i:=0;
        LOOP
            IF FIO.EOF THEN EXIT;END;
            FIO.RdStr(hin,S);
            IF Str.Match(S,sectionpattern) THEN
                Str.Subst(S,sectionHead,"");
                Str.Subst(S,sectionTail,"");
                Str.Caps(S);
                WrStr(S);WrLn;
                INC(i);
            END;
        END;
        fileClose(useLFN,hin);
        IF i=0 THEN WrStr("None !");WrLn;END;

    | opgen:
        hout:=fileCreate(useLFN,prog+extTST);
        FOR i:=0 TO 255 DO
            FIO.WrChar( hout, printable(CHR(i)) );
            IF ((i+1) MOD 16) = 0 THEN FIO.WrLn(hout); END;
        END;
        fileClose(useLFN,hout);
        WrStr(prog+extTST+" has been created.");WrLn;
    | opredir:

        i:=LoadGlossary (DEBUG,flagREDEFINED,flagUNDEFINED,flagCHARtoSTRING,
                        useLFN,glofile,section);
        CASE i OF
        | errNone:
            ;
        | errSectionNotFound:
            Str.Concat(S,section," glossary was not found in ");
            Str.Append(S,glofile);Str.Append(S," !");
            abort(i,S);
        ELSE
            abort(i,"");
        END;

        hin  := FIO.StandardInput;
        hout := FIO.StandardOutput;
        LOOP
            c := FIO.RdChar(hin);
            IF FIO.IOresult() # 0 THEN
                abort(errIn,"");
            END;
            IF FIO.EOF THEN EXIT; END;
            code := ORD(c);
            IF newCode[code].enabled THEN
                dumpdef(code,hout);
                IF FIO.IOresult() # 0 THEN abort(errOut,""); END;
            END;
        END;

    | opfile:

        i:=LoadGlossary (DEBUG,flagREDEFINED,flagUNDEFINED,flagCHARtoSTRING,
                        useLFN,glofile,section);
        CASE i OF
        | errNone:
            ;
        | errSectionNotFound:
            Str.Concat(S,section," glossary was not found in ");
            Str.Append(S,glofile);Str.Append(S," !");
            abort(i,S);
        ELSE
            abort(i,"");
        END;

        (* a few q&d sanity checks *)

        IF chkUD(filespec)=FALSE THEN abort(errBadSpec,filespec);END;
        IF same(filespec,dot) THEN Str.Copy(filespec,stardotstar); END;
        IF same(filespec,star) THEN Str.Copy(filespec,stardotstar); END;
        IF Str.Match(filespec,"*"+antislash) THEN Str.Append(filespec,stardotstar); END;
        IF chkJoker(filespec)=FALSE THEN
            IF fileIsDirectorySpec ( useLFN,filespec) THEN
                fixDirectory(filespec);
                Str.Append(filespec,stardotstar);
            END;
        END;
        makebase(useLFN,filespec,basedir);

        initList(anchor);
        countFile := buildFileList(anchor,useLFN,DEBUG,filespec);
        CASE countFile OF
        | MAX(CARDINAL): abort(errTooMany,filespec);
        | 0 :            abort(errNoMatch,filespec);
        END;

        ptr:=anchor;
        WHILE ptr # NIL DO
            getStr(filespec,ptr);
            Str.Concat(file,basedir,filespec);

            IF doCreate THEN (* "there can be only one..." here *)
                Str.Copy(source,file);
                Str.Copy(target,destfile);

                IF fileExists(useLFN,target) THEN fileErase(useLFN,target); END;

                WrStr(msgCreating); WrFname(useLFN,target);WrStr(msgNow);
            ELSE
                newExtension(backup,file,extBAK);

                IF fileExists(useLFN,backup) THEN fileErase(useLFN,backup); END;
                fileRename(useLFN,file,backup);

                Str.Copy(source,backup);
                Str.Copy(target,file);
                WrStr(msgProcessing); WrFname(useLFN,target);WrStr(msgNow);
            END;

            (* completed (completedInit, getFileSize(backup)); *)
            animInit(steps, "[", "]", CHR(46), "", "\/" );
            portion:=getFileSize(source) DIV steps; INC(portion); (* avoid DIV 0 ! *)
            lastportion := steps+1;

            hin :=fileOpenRead(useLFN,source);
            FIO.AssignBuffer(hin,bufferIn);
            IF doKeepStamp THEN
                stamp:=FIO.GetFileDate(hin);
            END;
            hout:=fileCreate(useLFN,target);
            FIO.AssignBuffer(hout,bufferOut);
            FIO.EOF:=FALSE;
            LOOP
                IF FIO.EOF THEN EXIT; END;
                got:=FIO.RdBin(hin,c,1);
                IF got # 1 THEN EXIT;END;
                code := ORD(c);
                IF newCode[code].enabled THEN dumpdef(code,hout); END;
                (* completed(completedSHOW,FIO.GetPos(hin)); *)
                anim(animcmd);
                currportion:=FIO.GetPos(hin) DIV portion;
                IF currportion # lastportion THEN
                    anim(animAdvance);
                    lastportion:=currportion;
                END;

            END;
            FIO.Flush(hout);
            fileClose(useLFN,hout);
            fileClose(useLFN,hin);

            (* NOW and only NOW, we may update again date and time if needed *)
            IF doKeepStamp THEN
                hout:=fileOpen(useLFN,target);
                FIO.SetFileDate(hout,stamp);
                fileClose(useLFN,hout);
            END;

            (* completed(completedEnd,0); *)
            anim(animEnd);anim(animClear);

            WrStr(msgOK);WrLn;

            ptr:=ptr^.next;
        END;
        freeList(anchor);

    END;

    abort(errNone,"");
END xlat.



(*

// this is XLAT.GLO companion file

; A glossary section is between "[_[_" and "_]_]", without blanks.
; Definition format is "source target", where both elements can be a string,
; a quoted string, a decimal number, a hexadecimal ($) number or a range.
; Target can be any of the following self-explanatory tokens :
; ~IDEM~, ~NIL~, ~RESET~, ~BIT7OFF~, ~BIT7ON~, ~PLUS1~, ~MINUS1~, ~NL~, ~CRLF~

[_[_PC2WIN_]_]
;
; PC to vindoze
;
$14     $B6
$15     $A7
$7F     
$80     $C7
$81     $FC
$82     $E9
$83     $E2
$84     $E4
$85     $E0
$86     $E5
$87     $E7
$88     $EA
$89     $EB
$8A     $E8
$8B     $EF
$8C     $EE
$8D     $EC
$8E     $C4
$8F     $C5
$90     $C9
$91     $E6
$92     $C6
$93     $F4
$94     $F6
$95     $F2
$96     $FB
$97     $F9
$98     $FF
$99     $D6
$9A     $DC
$9B     $A2
$9C     $A3
$9D     $A5
$9E     
$9F     
$A0     $E1
$A1     $ED
$A2     $F3
$A3     $FA
$A4     $F1
$A5     $D1
$A6     $AA
$A7     $BA
$A8     $BF
$A9     
$AA     $AC
$AB     $BD
$AC     $BC
$AD     $A1
$AE     $AB
$AF     $BB
; graphics
[$B0..$DF]  "                                                "
$E0     
$E1     
$E2     
$E3     
$E4     
$E5     
$E6     $B5
$E7     
$E8     
$E9     
$EA     
$EB     
$EC     
$ED     $D8
$EE     
$EF     
$F0     
$F1     $B1
$F2     
$F3     
$F4     
$F5     
$F6     $F7
$F7     
$F8     $B0
$F9     
$FA     
$FB     
$FC     
$FD     $B2
$FE     
$FF     " "



[_[_WIN2PC_]_]
;
; vindoze to PC
;
[$7F..$90]  
$91         '
$92         '
[$93..$A0]  
$A1         
$A2         
$A3         
$A4         
$A5         
$A6         |
$A7         $15
$A8         
$A9         
$AA         
$AB         
$AC         
$AD         -
$AE         
$AF         -
$B0         
$B1         
$B2         
$B3         
$B4         
$B5         
$B6         $14
$B7         
$B8         
$B9         
$BA         
$BB         
$BC         
$BD         
$BE         
$BF         
$C0         A
$C1         A
$C2         A
$C3         A
$C4         
$C5         
$C6         
$C7         
$C8         E
$C9         
$CA         E
$CB         E
$CC         I
$CD         I
$CE         I
$CF         I
$D0         
$D1         
$D2         O
$D3         O
$D4         O
$D5         O
$D6         
$D7         X
$D8         O
$D9         U
$DA         U
$DB         U
$DC         
$DD         Y
$DE         
$DF         
$E0         
$E1         
$E2         
$E3         a
$E4         
$E5         
$E6         
$E7         
$E8         
$E9         
$EA         
$EB         
$EC         
$ED         
$EE         
$EF         
$F0         
$F1         
$F2         
$F3         
$F4         
$F5         o
$F6         
$F7         
$F8         o
$F9         
$FA         
$FB         
$FC         
$FD         y
$FE         
$FF         



[_[_UPPER_]_]
;
; upper case
;
abcdefghijklmnopqrstuvwxyz  ABCDEFGHIJKLMNOPQRSTUVWXYZ
 CUEAAAACEEEIIIAAEOOOUUYOU
                      AIOUNN



[_[_LOWER_]_]
;
; lower case
;
ABCDEFGHIJKLMNOPQRSTUVWXYZ  abcdefghijklmnopqrstuvwxyz
 ueaaaaeeeiiiaaeooouuyou
                      aiounn



[_[_NOCTRL_]_]
;
; clean any control character in a textfile except cr, lf and tab
;
[0..31]     ~NIL~
$09         ~RESET~
$0A         ~RESET~
$0D         ~RESET~



[_[_NOGRAPH_]_]

""                           "+++"
""           "!!!!!+!!++++++--!-+"
""         "!!+++++-+++++++++!+++"
""                         "+-!!-"



[_[_ROT13_]_]
;
; ROT13
;
[A..M]  [N..Z]
[N..Z]  [A..M]
[a..m]  [n..z]
[n..z]  [a..m]



[_[_ASCII_]_]
;
; clean any file to let only text
;
; would be a better idea if...
; ctrl-z did not meant a nasty stop when reading a binary file !
; maybe for v1.1 ? ;-)
;
[0..255]    ~NIL~
$0A         ~RESET~
$0D         ~RESET~
$20         ~RESET~

!"#$%&'()*+,-./0123456789:;<=>?     ~RESET~
@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_    ~RESET~
`abcdefghijklmnopqrstuvwxyz{|}~     ~RESET~
                 ~RESET~
                 ueaaaeeeiiEoouu



[_[_WEIRD_]_]
;
; clean any file to let only weird characters
;
[0..255]    ~RESET~

$0A         ~NIL~
$0D         ~NIL~
$20         ~NIL~
!"#$%&'()*+,-./0123456789:;<=>?     ~NIL~
@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_    ~NIL~
`abcdefghijklmnopqrstuvwxyz{|}~     ~NIL~



[_[_NOACCENT_]_]
;
; no accents nor cedilla !
;
  Cueaaaaceeeiii
   AAEooouuyOU
          aiounN



[_[_ALPHANUM_]_]
;
; leave only letters and digits
;
[0..255]    ~NIL~
$0A         ~RESET~
$0D         ~RESET~
$20         ~RESET~

"0123456789"                        ~RESET~
ABCDEFGHIJKLMNOPQRSTUVWXYZ          ~RESET~
abcdefghijklmnopqrstuvwxyz          ~RESET~
                 ~RESET~
                 ueaaaeeeiiEoouu



[_[_NODIGITS_]_]
;
; remove digits
;
[0..255]        ~RESET~
"0123456789"    ~NIL~



[_[_NODQUOTE_]_]
;
; clean any double quote (ASCII code 34) in a textfile
;
[0..255]    ~RESET~
$22         ~NIL~



[_[_MARTIAN_]_]
;
; martian
;
!"                          
.                           
?                           
ABCDEFGHIJKLMNOPQRSTUVWXYZ  G׺JkmNPqR$њwxz
abcdefghijklmnopqrstuvwxyz  뉟ghKMQWXZ



[_[_PLUS1_]_]
;
; add 1
;
[0..255]    ~PLUS1~



[_[_MINUS1_]_]
;
; substract 1
;
[0..255]    ~MINUS1~



[_[_BIT7ON_]_]
;
; set bit 7
;
[$00..$7F]  ~BIT7ON~



[_[_BIT7OFF_]_]
;
; clear bit 7
;
[$80..$FF]  ~BIT7OFF~


[_[_UPPERTEXT_]_]
;
; leave only digits, uppercase letters, spaces, cr/lf/tab
;
[0..255]    ~NIL~
$0A         ~RESET~
$0D         ~RESET~
$20         ~RESET~

"0123456789"                        ~RESET~
ABCDEFGHIJKLMNOPQRSTUVWXYZ          ~RESET~

[_[_NOCTRLZ_]_]
;
; remove Ctrl-Z
;
[0..255]        ~RESET~
$1A             ~NIL~



[_[_PC2MAC_]_]
;
; DOS PC to Macintrash (unchecked)
;
$00    $00
$01    $01
$02    $02
$03    $03
$04    $04
$05    $05
$06    $06
$07    $07
$08    $08
$09    $09
$0A    ~NIL~
$0B    $0B
$0C    $0C
$0D    $0D
$0E    $0E
$0F    $0F
$10    $10
$11    $11
$12    $12
$13    $13
$14    $A6
$15    $A4
$16    $16
$17    $17
$18    $18
$19    $19
$1A    $1A
$1B    $1B
$1C    $1C
$1D    $1D
$1E    $1E
$1F    $1F
$20    " "
$21    "!"
$22    $22
$23    "#"
$24    "$"
$25    "%"
$26    "&"
$27    "'"
$28    "("
$29    ")"
$2A    "*"
$2B    "+"
$2C    ","
$2D    "-"
$2E    "."
$2F    "/"
$30    "0"
$31    "1"
$32    "2"
$33    "3"
$34    "4"
$35    "5"
$36    "6"
$37    "7"
$38    "8"
$39    "9"
$3A    ":"
$3B    ";"
$3C    "<"
$3D    "="
$3E    ">"
$3F    "?"
$40    "@"
$41    "A"
$42    "B"
$43    "C"
$44    "D"
$45    "E"
$46    "F"
$47    "G"
$48    "H"
$49    "I"
$4A    "J"
$4B    "K"
$4C    "L"
$4D    "M"
$4E    "N"
$4F    "O"
$50    "P"
$51    "Q"
$52    "R"
$53    "S"
$54    "T"
$55    "U"
$56    "V"
$57    "W"
$58    "X"
$59    "Y"
$5A    "Z"
$5B    "["
$5C    "\"
$5D    "]"
$5E    "^"
$5F    "_"
$60    "`"
$61    "a"
$62    "b"
$63    "c"
$64    "d"
$65    "e"
$66    "f"
$67    "g"
$68    "h"
$69    "i"
$6A    "j"
$6B    "k"
$6C    "l"
$6D    "m"
$6E    "n"
$6F    "o"
$70    "p"
$71    "q"
$72    "r"
$73    "s"
$74    "t"
$75    "u"
$76    "v"
$77    "w"
$78    "x"
$79    "y"
$7A    "z"
$7B    "{"
$7C    "|"
$7D    "}"
$7E    "~"
$7F    $C6
$80    $82
$81    $9F
$82    $8E
$83    $89
$84    $8A
$85    $88
$86    $8C
$87    $8D
$88    $90
$89    $91
$8A    $8F
$8B    $95
$8C    $94
$8D    $93
$8E    $80
$8F    "A"
$90    $83
$91    $BE
$92    $AE
$93    $99
$94    $9A
$95    $98
$96    $9E
$97    $9D
$98    $D8
$99    $85
$9A    $86
$9B    $A2
$9C    $A3
$9D    $B4
$9E    "."
$9F    $C4
$A0    $87
$A1    $92
$A2    $97
$A3    $9C
$A4    $96
$A5    $84
$A6    $BB
$A7    $BC
$A8    $C0
$A9    "~"
$AA    $C2
$AB    "H"
$AC    "Q"
$AD    $C1
$AE    $C7
$AF    $C8
[$B0..$BF]    "++++++++++++++++"
[$C0..$CF]    "++++++++++++++++"
[$D0..$DF]    "++++++++++++++++"
$E0    "a"
$E1    $A7
$E2    "G"
$E3    $B9
$E4    $B7
$E5    "s"
$E6    $B5
$E7    "t"
$E8    "F"
$E9    "h"
$EA    $BD
$EB    $B6
$EC    $B0
$ED    $AF
$EE    "E"
$EF    "U"
$F0    "="
$F1    $B1
$F2    $B3
$F3    $B2
$F4    "("
$F5    ")"
$F6    $D6
$F7    $C5
$F8    $A2
$F9    $13
$FA    $D7
$FB    $C3
$FC    "n"
$FD    "2"
$FE    $A5
$FF    "_"



[_[_WIN2MAC_]_]
;
; Vindoze OEM to Macintoy (unchecked))
;
$00    $00
$01    $01
$02    $02
$03    $03
$04    $04
$05    $05
$06    $06
$07    $07
$08    $08
$09    $09
$0A    ~NIL~
$0B    $0B
$0C    $0C
$0D    $0D
$0E    $0E
$0F    $0F
$10    $10
$11    $11
$12    $12
$13    $13
$14    $14
$15    $15
$16    $16
$17    $17
$18    $18
$19    $19
$1A    $1A
$1B    $1B
$1C    $1C
$1D    $1D
$1E    $1E
$1F    $1F
$20    " "
$21    "!"
$22    $22
$23    "#"
$24    "$"
$25    "%"
$26    "&"
$27    "'"
$28    "("
$29    ")"
$2A    "*"
$2B    "+"
$2C    ","
$2D    "-"
$2E    "."
$2F    "/"
$30    "0"
$31    "1"
$32    "2"
$33    "3"
$34    "4"
$35    "5"
$36    "6"
$37    "7"
$38    "8"
$39    "9"
$3A    ":"
$3B    ";"
$3C    "<"
$3D    "="
$3E    ">"
$3F    "?"
$40    "@"
$41    "A"
$42    "B"
$43    "C"
$44    "D"
$45    "E"
$46    "F"
$47    "G"
$48    "H"
$49    "I"
$4A    "J"
$4B    "K"
$4C    "L"
$4D    "M"
$4E    "N"
$4F    "O"
$50    "P"
$51    "Q"
$52    "R"
$53    "S"
$54    "T"
$55    "U"
$56    "V"
$57    "W"
$58    "X"
$59    "Y"
$5A    "Z"
$5B    "["
$5C    "\"
$5D    "]"
$5E    "^"
$5F    "_"
$60    "`"
$61    "a"
$62    "b"
$63    "c"
$64    "d"
$65    "e"
$66    "f"
$67    "g"
$68    "h"
$69    "i"
$6A    "j"
$6B    "k"
$6C    "l"
$6D    "m"
$6E    "n"
$6F    "o"
$70    "p"
$71    "q"
$72    "r"
$73    "s"
$74    "t"
$75    "u"
$76    "v"
$77    "w"
$78    "x"
$79    "y"
$7A    "z"
$7B    "{"
$7C    "|"
$7D    "}"
$7E    "~"
$7F    
[$80..$8F]    
$90    
$91    $D4
$92    $D5
[$93..$9F]    
$A0    "_"
$A1    $C1
$A2    $A2
$A3    $A3
$A4    $D7
$A5    $B4
$A6    "|"
$A7    $A4
$A8    $AC
$A9    $A9
$AA    $BB
$AB    $C7
$AC    $C2
$AD    $D0
$AE    $A8
$AF    $D1
$B0    $A1
$B1    $B1
$B2    "2"
$B3    "3"
$B4    $AB
$B5    $B5
$B6    $A6
$B7    $22
$B8    ","
$B9    "1"
$BA    $BC
$BB    $C8
$BC    "Q"
$BD    "H"
$BE    "T"
$BF    $C0
$C0    $CB
$C1    "A"
$C2    $81
$C3    $CC
$C4    $80
$C5    "A"
$C6    $AE
$C7    $82
$C8    "E"
$C9    $83
$CA    "E"
$CB    "E"
$CC    $93
$CD    $92
$CE    $94
$CF    $95
$D0    "]"
$D1    $84
$D2    "O"
$D3    "O"
$D4    "O"
$D5    $CD
$D6    $85
$D7    
$D8    $AF
$D9    "U"
$DA    "U"
$DB    "U"
$DC    $86
$DD    "Y"
$DE    "}"
$DF    $A7
$E0    $88
$E1    $87
$E2    $89
$E3    $8B
$E4    $8A
$E5    $8C
$E6    $BE
$E7    $8D
$E8    $8F
$E9    $8E
$EA    $90
$EB    $91
$EC    $93
$ED    $92
$EE    $94
$EF    $95
$F0    "&"
$F1    $96
$F2    $98
$F3    $97
$F4    $99
$F5    $9B
$F6    $9A
$F7    $D6
$F8    $BF
$F9    $9D
$FA    $9C
$FB    $9E
$FC    $9F
$FD    "y"
$FE    "$"
$FF    $D8



[_[_MAC2PC_]_]
;
; Macintrash to DOS PC (unchecked))
;
$00    $00
$01    $01
$02    $02
$03    $03
$04    $04
$05    $05
$06    $06
$07    $07
$08    $08
$09    $09
$0A    $0A
$0B    $0B
$0C    $0C
$0D    ~NL~
$0E    $0E
$0F    $0F
$10    $10
$11    "X"
$12    "/"
$13    "x"
$14    "@"
$15    $15
$16    $16
$17    $17
$18    $18
$19    $19
$1A    $1A
$1B    $1B
$1C    $1C
$1D    $1D
$1E    $1E
$1F    $1F
$20    " "
$21    "!"
$22    $22
$23    "#"
$24    "$"
$25    "%"
$26    "&"
$27    "'"
$28    "("
$29    ")"
$2A    "*"
$2B    "+"
$2C    ","
$2D    "-"
$2E    "."
$2F    "/"
$30    "0"
$31    "1"
$32    "2"
$33    "3"
$34    "4"
$35    "5"
$36    "6"
$37    "7"
$38    "8"
$39    "9"
$3A    ":"
$3B    ";"
$3C    "<"
$3D    "="
$3E    ">"
$3F    "?"
$40    "@"
$41    "A"
$42    "B"
$43    "C"
$44    "D"
$45    "E"
$46    "F"
$47    "G"
$48    "H"
$49    "I"
$4A    "J"
$4B    "K"
$4C    "L"
$4D    "M"
$4E    "N"
$4F    "O"
$50    "P"
$51    "Q"
$52    "R"
$53    "S"
$54    "T"
$55    "U"
$56    "V"
$57    "W"
$58    "X"
$59    "Y"
$5A    "Z"
$5B    "["
$5C    "\"
$5D    "]"
$5E    "^"
$5F    "_"
$60    "`"
$61    "a"
$62    "b"
$63    "c"
$64    "d"
$65    "e"
$66    "f"
$67    "g"
$68    "h"
$69    "i"
$6A    "j"
$6B    "k"
$6C    "l"
$6D    "m"
$6E    "n"
$6F    "o"
$70    "p"
$71    "q"
$72    "r"
$73    "s"
$74    "t"
$75    "u"
$76    "v"
$77    "w"
$78    "x"
$79    "y"
$7A    "z"
$7B    "{"
$7C    "|"
$7D    "}"
$7E    "~"
$7F    $C6
$80    ""
$81    "A"
$82    ""
$83    ""
$84    ""
$85    ""
$86    ""
$87    ""
$88    ""
$89    ""
$8A    ""
$8B    "a"
$8C    ""
$8D    ""
$8E    ""
$8F    ""
$90    ""
$91    ""
$92    ""
$93    ""
$94    ""
$95    ""
$96    ""
$97    ""
$98    ""
$99    ""
$9A    ""
$9B    "o"
$9C    ""
$9D    ""
$9E    ""
$9F    ""
$A0    "^"
$A1    ""
$A2    ""
$A3    ""
$A4    $15
$A5    ""
$A6    $14
$A7    ""
$A8    ""
$A9    ""
$AA    ""
$AB    "'"
$AC    ":"
$AD    "#"
$AE    ""
$AF    "O"
$B0    ""
$B1    ""
$B2    ""
$B3    ""
$B4    ""
$B5    ""
$B6    ""
$B7    ""
$B8    ""
$B9    ""
$BA    ""
$BB    ""
$BC    ""
$BD    ""
$BE    ""
$BF    "o"
$C0    ""
$C1    ""
$C2    ""
$C3    ""
$C4    ""
$C5    ""
$C6    ""
$C7    ""
$C8    ""
$C9    "."
$CA    "_"
$CB    "A"
$CC    "A"
$CD    "O"
$CE    ""
$CF    ""
$D0    "-"
$D1    "-"
$D2    $22
$D3    $22
$D4    "'"
$D5    "'"
$D6    ""
$D7    ""
$D8    ""
[$D9..$DF]    
[$E0..$EF]    
[$F0..$FF]    



[_[_MAC2WIN_]_]
;
; Macintoy to Vindoze OEM (unchecked))
;
$00    $00
$01    $01
$02    $02
$03    $03
$04    $04
$05    $05
$06    $06
$07    $07
$08    $08
$09    $09
$0A    $0A
$0B    $0B
$0C    $0C
$0D    ~NL~
$0E    $0E
$0F    $0F
$10    $10
$11    "X"
$12    "/"
$13    "x"
$14    "+"
$15    $15
$16    $16
$17    $17
$18    $18
$19    $19
$1A    $1A
$1B    $1B
$1C    $1C
$1D    $1D
$1E    $1E
$1F    $1F
$20    " "
$21    "!"
$22    $22
$23    "#"
$24    "$"
$25    "%"
$26    "&"
$27    "'"
$28    "("
$29    ")"
$2A    "*"
$2B    "+"
$2C    ","
$2D    "-"
$2E    "."
$2F    "/"
$30    "0"
$31    "1"
$32    "2"
$33    "3"
$34    "4"
$35    "5"
$36    "6"
$37    "7"
$38    "8"
$39    "9"
$3A    ":"
$3B    ";"
$3C    "<"
$3D    "="
$3E    ">"
$3F    "?"
$40    "@"
$41    "A"
$42    "B"
$43    "C"
$44    "D"
$45    "E"
$46    "F"
$47    "G"
$48    "H"
$49    "I"
$4A    "J"
$4B    "K"
$4C    "L"
$4D    "M"
$4E    "N"
$4F    "O"
$50    "P"
$51    "Q"
$52    "R"
$53    "S"
$54    "T"
$55    "U"
$56    "V"
$57    "W"
$58    "X"
$59    "Y"
$5A    "Z"
$5B    "["
$5C    "\"
$5D    "]"
$5E    "^"
$5F    "_"
$60    "`"
$61    "a"
$62    "b"
$63    "c"
$64    "d"
$65    "e"
$66    "f"
$67    "g"
$68    "h"
$69    "i"
$6A    "j"
$6B    "k"
$6C    "l"
$6D    "m"
$6E    "n"
$6F    "o"
$70    "p"
$71    "q"
$72    "r"
$73    "s"
$74    "t"
$75    "u"
$76    "v"
$77    "w"
$78    "x"
$79    "y"
$7A    "z"
$7B    "{"
$7C    "|"
$7D    "}"
$7E    "~"
$7F    $D0
$80    $C4
$81    $C2
$82    $C7
$83    $C9
$84    $D1
$85    $D6
$86    $DC
$87    $E1
$88    $E0
$89    $E2
$8A    $E4
$8B    $E3
$8C    $E5
$8D    $E7
$8E    $E9
$8F    $E8
$90    $EA
$91    $EB
$92    $ED
$93    $EC
$94    $EE
$95    $EF
$96    $F1
$97    $F3
$98    $F2
$99    $F4
$9A    $F6
$9B    $F5
$9C    $FA
$9D    $F9
$9E    $FB
$9F    $FC
$A0    "^"
$A1    $B0
$A2    $A2
$A3    $A3
$A4    $A7
$A5    $A4
$A6    $B6
$A7    $DF
$A8    $AE
$A9    $A9
$AA    "T"
$AB    $B4
$AC    $A8
$AD    "#"
$AE    $C6
$AF    $D8
$B0    "="
$B1    $B1
$B2    "<"
$B3    ">"
$B4    $A5
$B5    $B5
$B6    $F0
$B7    "S"
$B8    "P"
$B9    "p"
$BA    "{"
$BB    $AA
$BC    $BA
$BD    "O"
$BE    $E6
$BF    $F8
$C0    $BF
$C1    $A1
$C2    $AC
$C3    "V"
$C4    "f"
$C5    "~"
$C6    "D"
$C7    $AB
$C8    $BB
$C9    "."
$CA    "_"
$CB    $C0
$CC    $C3
$CD    $D5
$CE    $C6
$CF    $E6
$D0    $AD
$D1    $AF
$D2    $22
$D3    $22
$D4    $91
$D5    $92
$D6    $B1
$D7    $A4
$D8    $FF
[$D9..$DF]    
[$E0..$EF]    
[$F0..$FF]    



[_[_UNICODE_]_]
;
; should fix most unicrap characters
;
$00         ~NIL~
[$01..$ff]  ~RESET~


[_[_WEIRDEST_]_]
;
; as weird but cleaning only a..z, 0..9 and common chars
;
[0..255]    ~RESET~

$20                             ~NIL~
!'(),-.0123456789:;             ~NIL~
ABCDEFGHIJKLMNOPQRSTUVWXYZ      ~NIL~
abcdefghijklmnopqrstuvwxyz      ~NIL~
                  ~NIL~


// end of XLAT.GLO


*)



(*

// ugly batches to check nothing was broken from 1.2c to 1.3

// all.bat

@echo off
c:
cd \modula
if exist xlat.chk del xlat.rpt

call one PC2WIN
call one WIN2PC
call one UPPER
call one LOWER
call one NOCTRL
call one NOGRAPH
call one ROT13
call one ASCII
call one WEIRD
call one NOACCENT
call one ALPHANUM
call one NODIGITS
call one NODQUOTE
call one MARTIAN
call one PLUS1
call one MINUS1
call one BIT7ON
call one BIT7OFF
call one UPPERTEXT
call one NOCTRLZ
// macintoy conversions were added with 1.3 for too many HTML files

// one.bat

@echo off
xlat -!
xlat %1 xlat.tst %1.new
xlat -!
c:\bat\xlat %1 xlat.tst
copy xlat.tst %1.old

fcomp %1.old %1.new >> xlat.rpt

*)







