;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; FreeDOS Resident Calculator
; Copyright (c) 2006, 2007 Oleg O. Chukaev
;
; This program is free software; you can redistribute it and/or
; modify it under the terms of the GNU General Public License
; as published by the Free Software Foundation; either version 2
; of the License, or (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
; 02111-1307, USA.
;
; FreeDOS is a trademark of Jim Hall
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedures:
;	int09h_handler	update_window	add_proc	shr_proc
;	int10h_handler	update_results	sub_proc	sar_proc
;	calculator	xchg_window	mul_proc	push_dw
;	change_sign	draw_window	div_proc	pop_dw
;	enter_handler	parse		mod_proc	show_status
;	move_left	type_of_char	and_proc	out_dec
;	move_right	calculate	or_proc		sqrt_proc
;	move_up		op_handler	xor_proc	out_bin
;	move_down	eol_handler	shl_proc	out_hex
;	out_char	print_bin	not_proc	print_hex
;	get_addr	update_line	ins_char	backspace
;	delete		clear_str	clear_eol	shift_str
;	cur_right	cur_left	cur_home	cur_end
;	ltoa		atol		atolx		atold
;	atolb		atols		ishexdigit	isdecdigit
;	isbindigit	install		insert_char	opt_a_proc
;	opt_e_proc	opt_h_proc	opt_n_proc	opt_u_proc
;	opt_l_proc	opt_t_proc	opt_k_proc	opt_s_proc
;	history_prev	history_next	get_string	put_string
;	beep		sleep
;
; Abbreviations:
;	Modf:	AX	-- AX destroyed by this proc
;	Modf:	!AX	-- AX contains result
;	Modf:	~AX	-- AX destroyed by proc that called by this proc
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Includes
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;
%include	"config.inc"
%include	"fdrc.inc"
;
%include	"dos_prt.inc"
%include	"cmdline.inc"
%include	"tsr.inc"
%include	"copyrigh.inc"
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Code
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
		cpu	8086
		section	.text
		org	100h
offset_0100:
;-----------------------------------------------------------------------------
		jmp	install
;-----------------------------------------------------------------------------


;=============================================================================
; int09h_handler
;-----------------------------------------------------------------------------
; Handler for INT 09h. Checks for activation keys and calls main procedure
; (calculator) if these keys was pressed.
; In:	---
; Out:	---
; Modf:	---
; Call:	calculator
; Use:	actkey, run_flag, shiftkey, old_int09h
;
int09h_handler:
		push	ds
		push	ax
		cmp	byte [cs:run_flag],1
		je	@@to_old_handler
;-----------------------------------------------------------------------------
; Check for activation keys
;-----------------------------------------------------------------------------
		sti
		in	al,60h
		cmp	al,[cs:actkey]
		jne	@@to_old_handler
		xor	ax,ax
		mov	ds,ax
		mov	al,[0417h]
		and	al,[cs:shiftkey]
		cmp	al,[cs:shiftkey]
		jne	@@to_old_handler
;-----------------------------------------------------------------------------
; Check for usable video mode
;-----------------------------------------------------------------------------
		mov	al,[0449h]	;Video mode
		and	al,7fh
		cmp	al,03h		;80x25
		je	@@usable
		cmp	al,55h		;132x25
		je	@@usable
		sti			;We must enable ints for `beep'
		call	beep
		call	beep
		cli
		jmp	@@to_old_handler
@@usable:
;-----------------------------------------------------------------------------
		key_received
		send_eoi
;-----------------------------------------------------------------------------
		mov	byte [cs:run_flag],1

		call	calculator

		pop	ax
		pop	ds
		mov	byte [cs:run_flag],0
		iret

@@to_old_handler:
		pop	ax
		pop	ds
		jmp	far [cs:old_int09h]
;=============================================================================
; int10h_handler
;-----------------------------------------------------------------------------
; Handler for INT 10h. Adds function MAGICK_CODE (0FDCAh) to INT 10h.
; In:	AX == MAGICK_CODE
; Out:	AL:AH == MAGICK_CODE if installed
;	CX == TSR's CS
; Modf:	AX, CX
; Call:	---
; Use:	old_int10h
;
int10h_handler:
		cmp	ax,MAGICK_CODE
		je	@@return
		jmp	far [cs:old_int10h]
@@return:
		xchg	al,ah
		mov	cx,cs
		iret
;=============================================================================
; calculator
;-----------------------------------------------------------------------------
; Main procedure. Draws window, read keystrokes, performs calculations, etc.
; In:	---
; Out:	---
; Modf:	DS, AX
; Call:	update_window, update_results, xchg_window, set_cursor
; Use:	scr_rows, top, scr_cols, left, old_ss, old_sp, stack_bottom,
;	key_table, fucking_proc
;
calculator:
		mov	ax,cs
		mov	ds,ax
		mov	[old_ss],ss	;Save old SS:SP
		mov	[old_sp],sp
		mov	ss,ax		;Set new stack
		mov	sp,stack_bottom
		push	es
		push	bx
		push	cx
		push	dx
		push	di
		push	si
		push	bp
		cld

		push	ax		;<--- AX == CS
		xor	ax,ax
		mov	es,ax
		mov	al,[es:0484h]	;Get number of rows on screen
		inc	ax
		mov	[scr_rows],al

		sub	al,CALCHEIGHT
		cmp	al,[top]
		jae	@@top_ok
		mov	byte [top],0
@@top_ok:
		mov	ax,[es:044ah]	;Get number of columns on screen
		mov	[scr_cols],ax

		sub	al,CALCWIDTH
		cmp	al,[left]
		jae	@@left_ok
		mov	byte [left],0
@@left_ok:
		mov	word [video_seg],COLOR_SEG
		pop	es		;<--- ES := CS
;-----------------------------------------------------------------------------
		mov	ah,03h		;Get cursor position and shape
		mov	bh,0		;Video page
		int	10h
		push	cx
		push	dx
;-----------------------------------------------------------------------------
		cmp	byte [autoclear],0
		je	@@skip_clear
		or	byte [autoclear],1
@@skip_clear:
;-----------------------------------------------------------------------------
		call	update_window
@@next_key:
		int	28h
		mov	ah,1
		int	16h
		jz	@@next_key
		mov	ah,0
		int	16h
		cmp	ah,1
		je	@@esc
;-----------------------------------------------------------------------------
		xchg	ax,bx		;OPTIMIZE: instead of MOV BX,AX
		mov	si,key_table-2
@@next:
		lodsw			;Skip address
		lodsw
		or	ax,ax
		jz	@@found		;Found last proc -- insert_char
		cmp	ax,bx
		jne	@@next		;Any other proc clears flag...
		and	byte [autoclear],~1
@@found:
;-----------------------------------------------------------------------------
		call	[si]
		call	word [fucking_proc]
		call	update_results
		jmp	@@next_key
@@esc:
		call	xchg_window	;Restore window
;-----------------------------------------------------------------------------
		pop	dx
		pop	cx
		call	set_cursor	;Set cursor position
;-----------------------------------------------------------------------------
		pop	bp
		pop	si
		pop	di
		pop	dx
		pop	cx
		pop	bx
		pop	es
		mov	ss,[old_ss]	;Restore old stack
		mov	sp,[old_sp]
		ret
;=============================================================================
; change_sign
;-----------------------------------------------------------------------------
; Changes sign of result representation.
; In:	---
; Out:	---
; Modf:	---
; Call:	---
; Use:	s_flag, byte [print_table+4]
;
change_sign:
		xor	byte [print_table+4],'u' ^ 'd'
		xor	byte [s_flag],1
		ret
;=============================================================================
; enter_handler, enter_handler_2
;-----------------------------------------------------------------------------
; enter_handler called when ENTER pressed. Stores string in history,
; stores '\0' at the end of string in buffer, then calls calculation procedure.
; enter_handler_2 called after each entered digit or letter if switch -a
; specified on command line.
; In:	---
; Out:	---
; Modf:	~AX, BX, ~CX, ~DX, ~DI, ~SI
; Call:	calculate
; Use:	eol_ptr
;
enter_handler:
		cmp	byte [hist_items],0
		je	enter_handler_2
		call	put_string
		mov	byte [current_item],0
enter_handler_2:
		mov	bx,[eol_ptr]
		mov	byte [bx],0
		call	calculate
return:
		ret
;=============================================================================
; move_left
;-----------------------------------------------------------------------------
; Moves calculator's window left.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	left
;
move_left:
		cmp	byte [left],0
		je	return
		call	xchg_window
		dec	byte [left]
		jmp	update_window
;=============================================================================
; move_right
;-----------------------------------------------------------------------------
; Moves calculator's window right.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	left, scr_cols
;
move_right:
		mov	al,[scr_cols]
		sub	al,CALCWIDTH
		cmp	[left],al
		jae	return
		call	xchg_window
		inc	byte [left]
		jmp	update_window
;=============================================================================
; move_up
;-----------------------------------------------------------------------------
; Moves calculator's window up.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	top
;
move_up:
		cmp	byte [top],0
		je	return
		call	xchg_window
		dec	byte [top]
		jmp	update_window
;=============================================================================
; move_down
;-----------------------------------------------------------------------------
; Moves calculator's window down.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	top, scr_rows
;
move_down:
		mov	al,[scr_rows]
		sub	al,CALCHEIGHT
		cmp	[top],al
		jae	return
		call	xchg_window
		inc	byte [top]
		jmp	update_window
;=============================================================================
; update_window
;-----------------------------------------------------------------------------
; Updates calculator's window.
; In:	---
; Out:	---
; Modf:	~AX, ~BX, ~CX, ~DX, ~DI, ~SI
; Call:	xchg_window, draw_window, update_results
; Use:	---
;
update_window:
		call	xchg_window
		call	draw_window
		call	update_results
		ret
;=============================================================================
; update_results
;-----------------------------------------------------------------------------
; Updates results and input line, prints error message (if necessary).
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	video_seg, print_table, top_left, error_code
;
update_results:
		push	es
		mov	si,print_table
@@loop01:
		lodsw			;Shift from top_left
		call	get_addr
		lodsw			;Procedure address
		push	si
		call	ax
		pop	si
		lodsw			;Char and it's color
		or	al,al
		jz	@@quit
		stosw
		jmp	@@loop01
@@quit:
		mov	[error_code],al	;OPTIMIZE: AL instead of `ERR_ALL_OK'
		pop	es
		ret
;=============================================================================
; xchg_window
;-----------------------------------------------------------------------------
; Exchanges rectangular area of screen and buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	scr_buf, scr_cols
;
xchg_window:
		push	es
		xor	ax,ax
		call	get_addr
		mov	bx,CALCHEIGHT
		mov	si,scr_buf

@@next_line:
		mov	di,ax
		mov	cx,CALCWIDTH
@@loop:
		mov	dx,[si]		;|
		xchg	dx,[es:di]	;| rep xchg word [es:di],[ds:si] ;-)
		mov	[si],dx		;|
		cmpsw			;OPTIMIZE: instead of (INC DI; INC SI)*2
		loop	@@loop

		add	ax,[scr_cols]
		add	ax,[scr_cols]
		dec	bx
		jnz	@@next_line

		pop	es
		ret
;=============================================================================
; draw_window
;-----------------------------------------------------------------------------
; Draws calculator's window.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	video_seg, top_left, scr_cols, calc_title
;
draw_window:
		push	es
		xor	ax,ax
		call	get_addr
		mov	dx,-CALCWIDTH
		add	dx,[scr_cols]
		shl	dx,1

		mov	ax,COL_BG*100h+CHAR_TOP_LEFT
		stosw

		mov	si,calc_title
		mov	ah,COL_TITLE
		mov	cx,TITLE_LENGTH
@@loop00:
		lodsb
		stosw
		loop	@@loop00

		mov	cl,CALCWIDTH - TITLE_LENGTH - 2
		mov	ax,COL_BG*100h+CHAR_TOP
	rep	stosw
		mov	al,CHAR_TOP_RIGHT
		stosw
		add	di,dx

		mov	bx,CALCHEIGHT-2
@@loop01:
		mov	al,CHAR_LEFT
		stosw
		mov	cl,CALCWIDTH-2	;OPTIMIZE: CL instead of CX
		mov	al,' '
	rep	stosw
		mov	al,CHAR_RIGHT
		stosw
		add	di,dx
		dec	bx
		jnz	@@loop01

		mov	al,CHAR_BOTT_LEFT
		stosw
		mov	cl,CALCWIDTH-2	;OPTIMIZE: CL instead of CX
		mov	al,CHAR_BOTTOM
	rep	stosw
		mov	al,CHAR_BOTT_RIGHT
		stosw

		pop	es
		ret
;=============================================================================
; parse
;-----------------------------------------------------------------------------
; Parsed input buffer using following table.
;
;		0    1    2    3    4    5    6    7    8    9
;		0-1  2-9  a-f  \.   @    op   \"   ' '  eol  *
; ----------------------------------------------------------------
; eol,0		exit exit exit exit exit exit exit exit exit exit
; op,1		exit exit exit exit exit exit exit exit exit exit
; endhex,2	exit exit exit exit exit exit exit exit exit exit
; enddec,3	err  err  err  err  err  exit err  exit exit err
; endbin,4	err  err  err  err  err  exit err  exit exit err
; endstr,5	err  err  err  err  err  exit str  exit exit err
; start,6	bin  dec  hex  err  err  op   str  err  eol  err
; hex,7		hex  hex  hex  err  err  endh err  endh endh err
; dec,8		dec  dec  hex  endd err  endh err  endh endh err
; bin,9		bin  dec  hex  endd endb endh err  endh endh err
; str,10	str  str  str  str  str  str  ends str  err  str
; exit,11
;
; In:	SI -> string
; Out:	SI -> next char after token
;	CF clear if no errors
;	CF set if error
;	AX -- type of token (finite automaton state)
; Modf:	AX, BX, CX, SI
; Call:	type_of_char
; Use:	state_table, error_offs, error_code
;
parse:
		mov	cx,P_START*100h+4	;CH -- initial state
@@next_char:					;CL == 4 for shifts
		mov	al,5			;Length of record in state_table
		mul	ch
		add	ax,state_table
		xchg	ax,bx

		lodsb
		call	type_of_char
		shr	ax,1
		xlat
		jnc	@@shr
		shl	al,cl
@@shr:
		shr	al,cl
		cmp	al,P_EX
		je	@@quit
		mov	ch,al
		cmp	al,P_ERR
		jne	@@next_char

		dec	si
		mov	[error_offs],si
		mov	byte [error_code],ERR_PARSE
		stc
		ret
@@quit:
		mov	al,ch
		cbw
		clc
		ret
;=============================================================================
; type_of_char
;-----------------------------------------------------------------------------
; Determines type of char.
; In:	AL -- char
; Out:	AX -- type of char:
;	0	0-1		1	2-9
;	2	a-f		3	\.
;	4	@		5	operator
;	6	\"		7	' '
;	8	eol		9	* (any other character)
;
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
type_of_char:
		push	si
		mov	dx,-1
		mov	ah,al
		mov	si,chars
@@underscore:
		inc	dx
@@next:
		lodsb
		cmp	al,'_'
		je	@@underscore
		cmp	al,ah
		je	@@quit
		or	al,al
		jnz	@@next
		inc	dx
@@quit:
		xchg	ax,dx
		pop	si
		ret
;=============================================================================
; calculate
;-----------------------------------------------------------------------------
; Parses input buffer and calculates result.
; In:	---
; Out:	---
; Modf:	AX, BX, ~CX, ~DX, ~DI, SI
; Call:	parse
; Use:	input_buf, proc_table, math_sp
;
calculate:
		mov	si,input_buf-1
@@next:
		inc	si
		cmp	byte [si],' '	;Skip spaces
		je	@@next

		push	si		;parse destroys SI
		call	parse
		pop	si
		jc	@@quit		;Quit if parse error
		shl	ax,1
		xchg	bx,ax		;OPTIMIZE: instead of MOV BX,AX
		call	[proc_table+bx]	;Retrieve token from input buffer
		dec	si		;CF not destroyed by DEC
		jnc	@@next		;Next token if no errors
@@quit:
		mov	word [math_sp],math_stack	;Flush stack
		ret
;=============================================================================
; op_handler
;-----------------------------------------------------------------------------
; Called when parser encountered operator in the input line.
; In:	SI -> operator
;	math stack: 2 values on top
; Out:	SI -> next char
;	math stack: result on top
;	CF clear if no errors
;	CF set if error
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	pop_dw, push_dw
; Use:	error_offs, error_code, op_table
;
op_handler:
		mov	[error_offs],si			;SI -> operator
		mov	byte [error_code],ERR_ALL_OK	;Assume no errors
		lodsb					;Get operator
		mov	di,op_table-2
@@next:
		scasw
		scasb			;Find operator in table
		ja	@@next

		cmp	al,'\'		;Handle unary operators: `sqrt'
		je	@@pop
		cmp	al,'~'		;and `not'
		je	@@pop

		call	pop_dw
		jc	@@quit
		mov	cx,dx
		xchg	bx,ax		;OPTIMIZE: instead of MOV BX,AX
@@pop:
		call	pop_dw
		jc	@@quit
		call	word [di]
		call	push_dw
		cmp	byte [error_code],ERR_ALL_OK
		je	@@quit
		stc
@@quit:
		ret
;=============================================================================
; eol_handler
;-----------------------------------------------------------------------------
; Called when parser encountered end of line.
; In:	---
; Out:	CF is _always_ set
; Modf:	AX, DX
; Call:	pop_dw
; Use:	math_sp, result, error_code
;
eol_handler:
		call	pop_dw
		jc	@@exit
		mov	[result],ax
		mov	[result+2],dx
		mov	al,ERR_ALL_OK
		cmp	word [math_sp],math_stack
		je	@@quit
		mov	al,ERR_NOT_EMPTY
@@quit:
		mov	[error_code],al
@@exit:
		stc
		ret
;=============================================================================
; overflow
;-----------------------------------------------------------------------------
; Overflow handler.
; In:	---
; Out:	DX:AX == ULONG_MAX
;	[error_code] == ERR_OVERFLOW
; Modf:	!AX, !DX
; Call:	---
; Use:	---
;
overflow:
		mov	byte [error_code],ERR_OVERFLOW
		mov	ax,-1
		cwd
		ret
;=============================================================================
; add_proc, sub_proc, mul_proc, div_proc, mod_proc
; and_proc, or_proc, xor_proc, shl_proc, shr_proc, sar_proc
; sqrt_proc, not_proc
;-----------------------------------------------------------------------------
; Performs arithmetical/logical operations.
; THIS PROCEDURES MUST PRESERVE SI!
; In:	DX:AX -- first operand
;	CX:BX -- second operand (only for binary operators)
; Out:	DX:AX -- result
; Modf:	!AX, ?BX, ?CX, !DX, ?DI, ?BP
; Call:	---
; Use:	---
;
add_proc:
		add	ax,bx
		adc	dx,cx
		jc	overflow
		ret
;=============================================================================
; sub_proc
;-----------------------------------------------------------------------------
sub_proc:
		sub	ax,bx
		sbb	dx,cx
		ret
;=============================================================================
; mul_proc
;-----------------------------------------------------------------------------
; This code borrowed from FreeDOS kernel sources and slightly modified.
; Original comment follows:
; > this one adapted from elks, http://elks.sourceforge.net
; > multiply cx:bx * dx:ax, result in dx:ax
; > optimized by Arkady Belousov:
; >   dx:ax * cx:bx
; > = xh:xl * yh:yl
; > = xh:xl*yh*w + xh:xl*yl
; > = [xh*yh*w*w +] (xl*yh + xh*yl)*w + xl*yl
;
mul_proc:
		xchg	bp,ax		; bp=xl		(XCHG instead MOV)
		xchg	ax,dx		; ax=xh		(XCHG instead MOV)
		mul	bx		; dx:ax=xh*yl (forget dx)
		xchg	cx,ax		; cx=low(xh*yl), ax=yh
		mul	bp		; dx:ax=xl*yh (forget dx)
		add	cx,ax		; cx=low(xl*yh+xh*yl)
		xchg	ax,bp		; ax=xl		(XCHG instead MOV)
		mul	bx		; dx:ax=xl*yl
		add	dx,cx
		ret
;=============================================================================
; div_proc
;-----------------------------------------------------------------------------
div_proc:
		push	bx
		or	bx,cx
		pop	bx
		jz	overflow
@@m0:
		push	si
		mov	si,bx
		mov	di,cx
		mov	cx,32
		xor	bx,bx
		xor	bp,bp
		shl	ax,1
		rcl	dx,1
@@m1:
		rcl	bx,1
		rcl	bp,1
		push	bx
		push	bp
		sub	bx,si
		sbb	bp,di
		jnc	@@m2
		pop	bp
		pop	bx
		jmp	@@m3
@@m2:
		add	sp,4
@@m3:
		cmc
		rcl	ax,1
		rcl	dx,1
		loop	@@m1

		pop	si
		ret
;=============================================================================
; mod_proc
;-----------------------------------------------------------------------------
; x % y == x - (x / y) * y
;
mod_proc:
		push	ax
		push	dx
		push	bx
		push	cx
		call	div_proc
		pop	cx
		pop	bx
		call	mul_proc
		xchg	ax,bx
		xchg	dx,cx
		pop	dx
		pop	ax
		call	sub_proc
		ret
;=============================================================================
; and_proc
;-----------------------------------------------------------------------------
and_proc:
		and	ax,bx
		and	dx,cx
		ret
;=============================================================================
; or_proc
;-----------------------------------------------------------------------------
or_proc:
		or	ax,bx
		or	dx,cx
		ret
;=============================================================================
; xor_proc
;-----------------------------------------------------------------------------
xor_proc:
		xor	ax,bx
		xor	dx,cx
null_proc:
		ret
;=============================================================================
; shl_proc
;-----------------------------------------------------------------------------
shl_proc:
		jcxz	@@cx_ok
		xor	ax,ax
		cwd
		ret
@@cx_ok:
		mov	cx,bx
		jcxz	@@quit
@@loop:
		shl	ax,1
		rcl	dx,1
		loop	@@loop
@@quit:
		ret
;=============================================================================
; shr_proc
;-----------------------------------------------------------------------------
shr_proc:
		jcxz	@@cx_ok
		xor	ax,ax
		cwd
		ret
@@cx_ok:
		mov	cx,bx
		jcxz	@@quit
@@loop:
		shr	dx,1
		rcr	ax,1
		loop	@@loop
@@quit:
		ret
;=============================================================================
; sar_proc
;-----------------------------------------------------------------------------
sar_proc:
		jcxz	@@cx_ok
		xchg	ax,dx
		cwd
		ret
@@cx_ok:
		mov	cx,bx
		jcxz	@@quit
@@loop:
		sar	dx,1
		rcr	ax,1
		loop	@@loop
@@quit:
		ret
;=============================================================================
; sqrt_proc
;-----------------------------------------------------------------------------
sqrt_proc:
		mov	bx,1
		xor	cx,cx
@@next:
		sub	ax,bx
		sbb	dx,0
		jc	@@done
		inc	cx
		inc	bx
		inc	bx
		jmp	@@next
@@done:
		xor	dx,dx
		xchg	ax,cx
		ret
;=============================================================================
; not_proc
;-----------------------------------------------------------------------------
not_proc:
		not	ax
		not	dx
		ret
;=============================================================================
; push_dw
;-----------------------------------------------------------------------------
; Saves double word on the top of math stack.
; In:	DX:AX -- double word to push
; Out:	---
; Modf:	---
; Call:	---
; Use:	math_sp
;
push_dw:
		push	bx
		mov	bx,[math_sp]
		mov	[bx],ax
		mov	[bx+2],dx
		add	word [math_sp],4
		pop	bx
		clc
		ret
;=============================================================================
; pop_dw
;-----------------------------------------------------------------------------
; Stores in DX:AX double word from the top of math stack.
; In:	---
; Out:	DX:AX -- double word
;	CF clear if no errors
;	CF set and error_code == ERR_UNDERFLOW if attempt to pop from
;		empty stack
; Modf:	!AX, !DX
; Call:	---
; Use:	math_sp, error_code
;
pop_dw:
		cmp	word [math_sp],math_stack
		jbe	@@error
		push	bx
		sub	word [math_sp],4
		mov	bx,[math_sp]
		mov	dx,[bx+2]
		mov	ax,[bx]
		pop	bx
		clc
		ret
@@error:
		mov	byte [error_code],ERR_UNDERFLOW
		stc
		ret
;=============================================================================
; show_status
;-----------------------------------------------------------------------------
; Prints status message.
; In:	error code:
;		0 -- no errors
;		1 -- 'Parse error'
;		2 -- 'Stack underflow'
;		3 -- 'Stack not empty'
;		4 -- 'Overflow'
;	ES:DI -> video memory
; Out:	---
; Modf:	AX, BX, SI
; Call:	---
; Use:	---
;
show_status:
		mov	ax,MSG_ERR_LENGTH
		mov	cx,ax
		mul	byte [error_code]
		add	ax,error_messages
		xchg	ax,si
		mov	ah,COL_ERRMSG
@@loop00:
		lodsb
		stosw
		loop	@@loop00

		add	di,30
		mov	al,[current_item]
		mov	ah,COL_BG
		call	print_hex
		scasw
		mov	al,[num_items]
		cmp	al,0
		jne	@@001
		mov	ax,COL_BG*256+04h
		stosw
		ret
@@001:
		mov	al,[current_item]
		cmp	al,[num_items]
		jb	@@002
		mov	ax,COL_BG*256+19h
		stosw
		ret
@@002:
		cmp	al,0
		jne	@@003
		mov	ax,COL_BG*256+18h
		stosw
		ret
@@003:
		mov	ax,COL_BG*256+12h
		stosw
		ret
;=============================================================================
; out_dec
;-----------------------------------------------------------------------------
; Outputs result in decimal form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, BX, ~CX, ~DX, DI, SI
; Call:	ltoa
; Use:	---
;
out_dec:
		call	ltoa	;     vvvvv------- MUST BE 0
		mov	bx,1011100011100000b
@@loop:				;    ^------------ MUST BE 1
		mov	ah,COL_NUM
		shl	bx,1
		adc	ah,0
		lodsb
		stosw
		or	bx,bx
		jnz	@@loop
		ret
;=============================================================================
; out_bin
;-----------------------------------------------------------------------------
; Outputs result in binary form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
out_bin:
		mov	ax,[result+2]
		call	print_bin
		mov	ax,[result]
print_bin:
		xchg	ax,dx
		mov	bx,00ffh
@@loop01:
		mov	ax,COL_NUM*100h
		shl	bx,1
		adc	ah,al
		rcl	dx,1
		adc	al,'0'
		stosw
		or	bx,bx
		jnz	@@loop01
		ret
;=============================================================================
; out_hex
;-----------------------------------------------------------------------------
; Outputs result in hexadecimal form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, ~CX, ~DI
; Call:	print_hex
; Use:	result
;
out_hex:
		mov	bp,print_hex
print_color:
		mov	si,result+3
		mov	bl,01010000b
@@loop:
		mov	ah,COL_NUM
		shl	bl,1
		adc	ah,0
		mov	al,[si]
		dec	si
		call	bp
		or	bl,bl
		jnz	@@loop
		ret
;=============================================================================
; out_char
;-----------------------------------------------------------------------------
; Outputs result in string form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
out_char:
		mov	ax,COL_LETTER*100h+'"'
		stosw
		mov	bp,@@print_char
		jmp	print_color
@@print_char:
		stosw
		ret
;=============================================================================
; print_hex
;-----------------------------------------------------------------------------
; Prints hex value of AL
; In:	AL -- byte
;	AH -- color
;	ES:DI -> place in video RAM
; Out:	---
; Modf:	AX, CX, DI
; Call:	---
; Use:	---
;
print_hex:
		mov	cl,4
@@next:
		push	ax
		shr	al,cl
		and	al,0fh
		cmp	al,10
		sbb	al,69h
		das
		stosw
		pop	ax
		sub	cl,4
		jz	@@next
		ret
;=============================================================================
; get_addr
;-----------------------------------------------------------------------------
; Returns address of char in video buffer
; In:	AH/AL -- additions to top/left row/column of char
; Out:	ES:AX, ES:DI -- address
; Modf:	!AX, !DI, !ES
; Call:	---
; Use:	---
;
get_addr:
		add	ax,[top_left]
		push	bx
		xchg	bx,ax
		mov	al,[scr_cols]
		mul	bh
		mov	bh,0
		add	ax,bx
		pop	bx
		shl	ax,1
		mov	di,ax
		mov	es,[video_seg]
		ret
;=============================================================================
; update_line
;-----------------------------------------------------------------------------
; Updates input line.
; In:	ES:DI -> videobuffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	start_ptr, end_ptr, cur_ptr, autoclear, error_code, error_offs, top_left
;
update_line:
		mov	si,[start_ptr]
@@loop01:
		cmp	si,[end_ptr]
		je	@@end_loop
;-----------------------------------------------------------------------------
; Select color for input line.
;-----------------------------------------------------------------------------
		mov	ah,COL_INPUT
		test	byte [autoclear],1
		jz	@@using_col_input
		mov	ah,COL_INPUT0
@@using_col_input:
;-----------------------------------------------------------------------------
; Check for errors.
;-----------------------------------------------------------------------------
		cmp	byte [error_code],ERR_ALL_OK
		je	@@no_parse_error
		cmp	byte [error_code],ERR_NOT_EMPTY
		je	@@no_parse_error
		cmp	si,[error_offs]
		jne	@@no_parse_error
		mov	ah,COL_ERRCHR
@@no_parse_error:
;-----------------------------------------------------------------------------
		lodsb
		stosw
		jmp	@@loop01
@@end_loop:
		mov	cx,[start_ptr]
		add	cx,LEN+1
		sub	cx,[end_ptr]
		jle	@@skip_fill
		mov	ax,COL_FILL*100h+INPUT_FILL_CHAR
	rep	stosw			;Fill input line
@@skip_fill:
		mov	ax,[cur_ptr]
		sub	ax,[start_ptr]
		add	ax,[top_left]
		add	ax,0301h
		xchg	ax,dx
		mov	cx,0e0fh	;Cursor like `_'
;-----------------------------------------------------------------------------
; set_cursor -- sets cursor shape (CX) and position (DX)
;-----------------------------------------------------------------------------
set_cursor:
		mov	ah,02h
		mov	bh,0
		int	10h		;Set cursor position
		mov	ah,01h
		int	10h		;Set cursor shape
		ret
;=============================================================================
; ins_char
;-----------------------------------------------------------------------------
; Inserts char into input buffer.
; if (no_free_space_in_buffer)
;     return
; if (cur_ptr < eol_ptr)
;     shift_string_to_right_from_current
; put_char
; if (cursor_at_the_end_of_window) {
;     ++start_ptr
;     ++end_ptr
; } else {
;     if (end_of_string_in_window) {
;         ++end_ptr
;     }
; }
; ++eol_ptr
; ++cur_ptr
;
; In:	BL -- char
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	eol_ptr, cur_ptr, start_ptr, end_ptr
;
ins_char:
		cmp	word [eol_ptr],input_buf+BUFLENGTH-1
		jae	@@exit		;Exit if buffer is full

		mov	si,[eol_ptr]	;Shift string
@@loop:
		cmp	si,[cur_ptr]
		je	@@end_loop
		mov	ah,[si-1]
		mov	[si],ah
		dec	si
		jmp	@@loop
@@end_loop:
		mov	[si],bl

		mov	ax,[start_ptr]
		add	ax,LEN
		cmp	[cur_ptr],ax
		jb	@@no_shift2
		inc	word [start_ptr]
		inc	word [end_ptr]
		jmp	@@quit
@@no_shift2:
		cmp	[eol_ptr],ax
		adc	word [end_ptr],0	;Instead of the following
;		jae	@@no_shift3		;<-- instructions
;		inc	[end_ptr]		;<--
;@@no_shift3:					;<--
@@quit:
		inc	word [cur_ptr]
		inc	word [eol_ptr]
@@exit:
		ret
;=============================================================================
; insert_char
;-----------------------------------------------------------------------------
; Front-end for ins_char.
; In:	BL -- char
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	clear_str, ins_char, cur_left
; Use:	autoclear, result
;
insert_char:
		cmp	bl,0
		je	@@exit		;Exit if char is invalid
		cmp	byte [autoclear],0
		je	@@ins_char
		test	byte [autoclear],1
		jz	@@ins_char
		call	clear_str
		mov	di,result	;OPTIMIZE: instead of
		xor	ax,ax		;mov word [result],0
		stosw			;mov word [result+2],0
		stosw
@@ins_char:
		and	byte [autoclear],~1
		cmp	bl,'"'
		jne	@@ins
		call	ins_char
		call	cur_left
@@ins:
		call	ins_char
@@exit:
		ret
;=============================================================================
; backspace
;-----------------------------------------------------------------------------
; Deletes one char before cursor.
; if (no_chars_before_cursor)
;     return
; shift_string_to_left_from_current
; --eol_ptr
; if (beginning_of_string_in_window) {
;     if (end_of_string_in_window) {
;         --end_ptr
;     }
; } else {
;     --start_ptr
;     --end_ptr
; }
; --cur_ptr
;
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	shift_str
; Use:	eol_ptr, cur_ptr, start_ptr, end_ptr
;
backspace:
		mov	ax,[cur_ptr]
		cmp	ax,input_buf
		je	@@quit

		call	shift_str
		dec	word [cur_ptr]
		mov	ax,[start_ptr]
		cmp	ax,input_buf
		jne	@@0001
		add	ax,LEN
		cmp	[eol_ptr],ax
		jb	@@0003
		ret
@@0001:
		dec	word [start_ptr]
@@0003:
		dec	word [end_ptr]
@@quit:
		ret
;=============================================================================
; delete
;-----------------------------------------------------------------------------
; Deletes char under cursor.
; if (cursor_at_the_end_of_string)
;     return
; shift_string_to_left_from_current
; --eol_ptr
; if (end_of_string_in_window) {
;     if (beginning_of_string_not_in_window) {
;         --start_ptr
;     }
;     --end_ptr
; }
;
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	shift_str
; Use:	cur_ptr, eol_ptr, start_ptr, end_ptr
;
delete:
		mov	ax,[cur_ptr]
		cmp	ax,[eol_ptr]
		je	@@quit

		inc	ax
		call	shift_str
		mov	ax,[start_ptr]
		add	ax,LEN
		cmp	[eol_ptr],ax
		jae	@@quit
		dec	word [end_ptr]
		mov	ax,[start_ptr]
		cmp	ax,input_buf
		je	@@quit
		dec	word [start_ptr]
@@quit:
		ret
;=============================================================================
; clear_str
;-----------------------------------------------------------------------------
; Clears input buffer.
; In:	---
; Out:	---
; Modf:	AX, DI
; Call:	---
; Use:	---
;
clear_str:
		mov	ax,input_buf
		mov	di,ed_pointers
		stosw			;start_ptr
		stosw			;cur_ptr
		stosw			;end_ptr
		stosw			;eol_ptr
		ret
;=============================================================================
; clear_eol
;-----------------------------------------------------------------------------
; Clears input buffer from current position to the end of bufer.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	cur_ptr, end_ptrm eol_ptr, start_ptr
;
clear_eol:
		mov	ax,[cur_ptr]
		mov	[end_ptr],ax
		mov	[eol_ptr],ax
		sub	ax,LEN
		cmp	ax,input_buf
		jae	@@quit
		mov	ax,input_buf
@@quit:
		mov	[start_ptr],ax
		ret
;=============================================================================
; shift_str
;-----------------------------------------------------------------------------
; Shifts string, from current position to end, on 1 char left.
; (For use w/ `backspace' and `delete'.)
; In:	AX -> string
; Out:	---
; Modf:	AX, SI
; Call:	---
; Use:	eol_ptr
;
shift_str:
		xchg	si,ax		;OPTIMIZE: instead of MOV SI,AX
@@loop:
		lodsb
		mov	[si-2],al
		cmp	si,[eol_ptr]
		jb	@@loop

		dec	word [eol_ptr]
		ret
;=============================================================================
; cur_right
;-----------------------------------------------------------------------------
; Moves cursor right.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	cur_ptr, eol_ptr, end_ptr, start_ptr
;
cur_right:
		mov	ax,[cur_ptr]
		cmp	ax,[eol_ptr]
		je	@@quit

		inc	word [cur_ptr]
		cmp	ax,[end_ptr]
		jb	@@quit
		inc	word [start_ptr]
		inc	word [end_ptr]
@@quit:
		ret
;=============================================================================
; cur_left
;-----------------------------------------------------------------------------
; Moves cursor left.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	cur_ptr, start_ptr, end_ptr
;
cur_left:
		mov	ax,[cur_ptr]
		cmp	ax,input_buf
		je	@@quit

		dec	word [cur_ptr]
		cmp	ax,[start_ptr]
		ja	@@quit
		dec	word [start_ptr]
		dec	word [end_ptr]
@@quit:
		ret
;=============================================================================
; cur_home
;-----------------------------------------------------------------------------
; Moves cursor to the beginning of input line.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	start_ptr, cur_ptr, eol_ptr, end_ptr
;
cur_home:
		mov	ax,input_buf
		mov	[start_ptr],ax
		mov	[cur_ptr],ax
		add	ax,LEN
		cmp	[eol_ptr],ax
		ja	@@0001
		mov	ax,[eol_ptr]
@@0001:
		mov	[end_ptr],ax
		ret
;=============================================================================
; cur_end
;-----------------------------------------------------------------------------
; Moves cursor to the end of input line.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	start_ptr, cur_ptr, eol_ptr, end_ptr
;
cur_end:
		mov	ax,[eol_ptr]
		mov	[cur_ptr],ax
		mov	[end_ptr],ax
		sub	ax,LEN
		cmp	ax,input_buf
		jae	@@0001
		mov	ax,input_buf
@@0001:
		mov	[start_ptr],ax
		ret
;=============================================================================
; history_prev
;-----------------------------------------------------------------------------
; Gets previous line from history.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_string, beep
; Use:	current_item, num_items
;
history_prev:
		mov	al,[current_item]
		cmp	al,[num_items]
		je	@@quit

		inc	byte [current_item]
		call	get_string
@@quit2:
		ret
@@quit:
		call	beep
		ret
;=============================================================================
; history_next
;-----------------------------------------------------------------------------
; Gets next line from history.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_string, beep
; Use:	current_item
;
history_next:
		mov	al,[current_item]
		cmp	al,0
		je	@@quit
		dec	byte [current_item]
		call	get_string
		ret
@@quit:
		call	beep
		ret
;=============================================================================
; put_string
;-----------------------------------------------------------------------------
; Stores current line to histroy buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	history_buffer, hist_items, num_items, eol_ptr, input_buf
;
put_string:
		mov	di,history_buffer+BUFLENGTH-1
		mov	al,[hist_items]
		cmp	al,[num_items]
		je	@@001
		inc	byte [num_items]
@@001:
		dec	ax
		mov	ah,BUFLENGTH
		mul	ah
		add	di,ax
		lea	si,[di-BUFLENGTH]
		std
		xchg	cx,ax
	rep	movsb
		cld

		mov	di,history_buffer
		mov	si,input_buf
		mov	cx,[eol_ptr]
		sub	cx,si
		jcxz	@@quit
	rep	movsb
@@quit:
		mov	byte [di],0
		ret
;=============================================================================
; get_string
;-----------------------------------------------------------------------------
; Copies line from history to buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	clear_str
; Use:	result, current_item, history_buffer, input_buf, eol_ptr, end_ptr
;
get_string:
		call	clear_str
		mov	word [result],0
		mov	word [result+2],0
		mov	al,[current_item]
		or	al,al
		jz	@@quit

		mov	si,history_buffer
		dec	al
		mov	ah,BUFLENGTH
		mul	ah
		add	si,ax
		mov	di,input_buf
		xor	cx,cx
		dec	cx
@@loop:
		lodsb
		stosb
		inc	cx
		or	al,al
		jnz	@@loop

		add	[eol_ptr],cx
		cmp	cx,LEN
		jae	@@001
		add	[end_ptr],cx
		ret
@@001:
		add	word [end_ptr],LEN
@@quit:
		ret
;=============================================================================
; ltoa	
;-----------------------------------------------------------------------------
; Converts dword [result] to 11-bytes string, 1st byte is `-' or ` '.
; Puts spaces to bytes before digits.
; In:	---
; Out:	SI -> ltoa_buf
; Modf:	AX, BX, DX, CX, !SI
; Call:	---
; Use:	ltoa_buf, result
;
ltoa:
		mov	si,ltoa_buf
		push	si
		mov	cx,11

@@fillbuf:
		mov	byte [si],' '	;We can't use stosb because ES:DI
		inc	si		;points to video memory
		loop	@@fillbuf	;Fill w/ spaces
		dec	si

		mov	ax,[result]
		mov	dx,[result+2]

		cmp	[s_flag],cl	;OPTIMIZE: CL instead of 0
		je	@@unsigned
		or	dx,dx
		jns	@@unsigned
		neg	dx
		neg	ax
		sbb	dx,cx		;OPTIMIZE: CX instead of 0
		mov	byte [si-10],'-'
@@unsigned:
		mov	cl,10
@@next_digit:
		mov	bx,ax
		mov	ax,dx
		xor	dx,dx
		div	cx
		xchg	ax,bx
		div	cx
		xchg	dx,bx
		add	bl,'0'
		mov	[si],bl		;Store digit
		dec	si
		mov	bx,dx
		or	bx,ax
		jnz	@@next_digit
		pop	si
		ret
;=============================================================================
; atol
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word.
; In:	SI -> string
;	BP -> checking procedure that returns CY if BL is not a valid
;		digit, NC and BL == number if BL is a valid digit
; Out:	DX:AX -- number
;	SI -> 2 chars after number
; Modf:	!AX, !DX, SI
; Call:	---
; Use:	BP -> procedure
;
atol:
		xor	ax,ax
		xor	bx,bx
		xor	di,di
@@loop:
		xchg	ax,di		;Mul DI:AX by base (CX)
		mul	cx
		xchg	ax,di
		mul	cx
		add	ax,bx		;Char, converted to digit
		adc	di,dx
		mov	bl,[si]		;Load char
		inc	si
		call	bp		;Check char and get base
		jnc	@@loop

		mov	dx,di
		ret			;Return
;=============================================================================
; atolx
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 16.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atolx:
		mov	bp,ishexdigit
		call	atol
		call	push_dw
		dec	si
		ret
;=============================================================================
; atold
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 10.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atold:
		mov	bp,isdecdigit
		call	atol
		call	push_dw
		ret
;=============================================================================
; atolb
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 2.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atolb:
		mov	bp,isbindigit
		call	atol
		call	push_dw
		ret
;=============================================================================
; atols
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
atols:
		push	bx
		xor	bx,bx
		xor	dx,dx
		lodsb			;Skip '"'
@@next:
		lodsb
		cmp	al,'"'
		jne	@@shift
		lodsb
		cmp	al,'"'
		jne	@@quit
@@shift:
		mov	dh,dl
		mov	dl,bh
		mov	bh,bl
		mov	bl,al
		jmp	@@next
@@quit:
		dec	si
		xchg	ax,bx		;OPTIMIZE: instead of MOV AX,BX
		call	push_dw
		pop	bx
		ret
;=============================================================================
; ishexdigit, isbindigit, isdecdigit
;-----------------------------------------------------------------------------
; Checks if char is:
;	ishexdigit -- hexadecimal digit
;	isdecdigit -- decimal digit
;	isbindigit -- binary digit
; In:	BL -- char
; Out:	CF clear if BL is valid
;		BL -- number
;	CF set otherwise
;	CX -- base
; Modf:	!BL, !CX
; Call:	---
; Use:	---
;
ishexdigit:
		mov	cx,16
		cmp	bl,'f'
		ja	..@no
		cmp	bl,'a'
		jae	..@yes_a
		cmp	bl,'F'
		ja	..@no
		cmp	bl,'A'
		jae	..@yes_aa
		jmp	..@check_dec
isdecdigit:				;<---------------------
		mov	cx,10
..@check_dec:
		cmp	bl,'9'
		ja	..@no
		cmp	bl,'0'
		jae	..@yes
		jmp	..@no
isbindigit:				;<---------------------
		mov	cx,2
		cmp	bl,'1'
		ja	..@no
		cmp	bl,'0'
		jae	..@yes
..@no:
		stc
		ret
..@yes_a:
		sub	bl,'a'-'A'
..@yes_aa:
		sub	bl,'A'-0ah-'0'
..@yes:
		sub	bl,'0'
		clc
		ret
;=============================================================================
; beep
;-----------------------------------------------------------------------------
; Generates beep. Code from Freemacs.
; In:	---
; Out:	---
; Modf:	AX
; Call:	sleep
; Use:	opt_q
;
beep:
		cmp	byte [cs:opt_q],0
		jne	@@quit
		call	sleep
		mov	al,0b6h		;Channel 2, LSB then MSB, Square Wave, Binary
		out	43h,al		;Program 8253 command register
		mov	ax,2000		;Get the frequency to be generated
		out	42h,al		;Load Channel 2 count register LSB
		mov	al,ah
		out	42h,al		;Load Channel 2 count register MSB
		in	al,61h		;Read settings from 8255 PPI I/O Port "PB"
		push	ax		;Save original settings in AH
		or	al,3		;Enable Timer Channel 2 & Speaker data
		out	61h,al		;program the 8255 with new setting-speaker on
		call	sleep		;Wait for a while.
		pop	ax		;Get original 8255 Port "PB" settings
		out	61h,al		;Reset port to original values-speaker off
@@quit:
		ret
;=============================================================================
; sleep
;-----------------------------------------------------------------------------
; Stops program execution by 1/18.2 sec.
; In:	---
; Out:	---
; Modf:	AX
; Call:	---
; Use:	---
;
sleep:
		push	ds
		xor	ax,ax
		mov	ds,ax
		mov	ax,[46ch]	;Get the current timer value.
@@wait:
		cmp	ax,[46ch]	;Did the timer value "tick"?
		je	@@wait		;No - keep waiting for a tick.
		pop	ds
		ret
;
;-----------------------------------------------------------------------------


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Initialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;
top_left:				;Col/row of calc's top left corner:
left		db	0		;column
top		db	0		;row
actkey		db	DEF_ACTKEY	;Default activation key
shiftkey	db	DEF_SHIFTKEY	;Default shift value
run_flag	db	0		;1 if calculator is active
autoclear	db	0		;
result		dw	0, 0		;Result of calculations
error_code	db	0		;Error code
s_flag		db	1		;1 if result have sign
hist_items	db	0		;Number of items in history
num_items	db	0		;
current_item	db	0		;Current item in history
opt_q		db	0		;1 if no sound
math_sp		dw	math_stack	;Math stack pointer
fucking_proc	dw	null_proc	;Call this proc after pressing any key
;-----------------------------------------------------------------------------
; Pointers for editing procedures
;-----------------------------------------------------------------------------
; Input buffer:		This_is_a_string_in_the_input_buffer
; Edit window:		^		   _   ^
; input_buf ------------'                  ^  ^\        ^   |
;					   |  | `cursor |   |
ed_pointers:				   ;  |         |   |
start_ptr	dw	input_buf	;--'  |         |   |
cur_ptr		dw	input_buf	;-----'         |   |
end_ptr		dw	input_buf	;---------------'   |
eol_ptr		dw	input_buf	;-------------------'
;-----------------------------------------------------------------------------
; Table for printing parts of calculator's window
;-----------------------------------------------------------------------------
print_table:
		prtstr	0101h,	out_dec,	'd',	COL_LETTER
;This char used in `change_sign' ----------------^
		prtstr	0110h,	out_hex,	'h',	COL_LETTER
		prtstr	011ch,	out_char,	'"',	COL_LETTER
		prtstr	0201h,	out_bin,	'b',	COL_LETTER
		prtstr	0301h,	update_line, CHAR_RIGHT, COL_BG
		prtstr	0401h,	show_status,	0,	COL_BG
;This char used in `update_results'             ^
;as end-of-table marker ------------------------'
;
;-----------------------------------------------------------------------------
; Macro from fdrc-msg.??
;
		resident_messages
;
;-----------------------------------------------------------------------------
; Table for `type_of_char'
; Classes of characters delimited by '_'.
;-----------------------------------------------------------------------------
chars		db	'01_23456789_abcdefABCDEF_._@_+-*/%&|^<>{}\~_"_ _', 0
;-----------------------------------------------------------------------------
; Editing and movement keys
;-----------------------------------------------------------------------------
key_table:
	%ifdef	VI_KEYS
		keystr	KEY_ALT_H,	move_left	;A-h (<--)
		keystr	KEY_ALT_L,	move_right	;A-l (-->)
		keystr	KEY_ALT_K,	move_up		;A-k (^)
		keystr	KEY_ALT_J,	move_down	;A-j (v)
	%else
		keystr	KEY_CTRL_S,	move_left	;C-s (<--)
		keystr	KEY_CTRL_D,	move_right	;C-d (-->)
		keystr	KEY_CTRL_E,	move_up		;C-e (^)
		keystr	KEY_CTRL_X,	move_down	;C-x (v)
	%endif
		keystr	KEY_LEFT,	cur_left	;<--
		keystr	KEY_RIGHT,	cur_right	;-->
		keystr	KEY_UP,		history_prev	;^
		keystr	KEY_DOWN,	history_next	;v
		keystr	KEY_BACKSPACE,	backspace	;bksp
		keystr	KEY_DELETE,	delete		;del
		keystr	KEY_HOME,	cur_home	;home
		keystr	KEY_END,	cur_end		;end
		keystr	KEY_CTRL_Y,	clear_str	;C-y
		keystr	KEY_CTRL_K,	clear_eol	;C-k
		keystr	KEY_ALT_S,	change_sign	;A-s
		keystr	KEY_ENTER,	enter_handler	;Enter
		keystr	0000h,		insert_char	;Any key
;-----------------------------------------------------------------------------
; This table must be ordered by 'operator' field!
;-----------------------------------------------------------------------------
op_table:
		opstr	'%',	mod_proc
		opstr	'&',	and_proc
		opstr	'*',	mul_proc
		opstr	'+',	add_proc
		opstr	'-',	sub_proc
		opstr	'/',	div_proc
		opstr	'<',	shl_proc
		opstr	'>',	shr_proc
		opstr	'\',	sqrt_proc
		opstr	'^',	xor_proc
		opstr	'{',	shl_proc
		opstr	'|',	or_proc
		opstr	'}',	sar_proc
		opstr	'~',	not_proc
		opstr	0ffh,	null_proc	;End of table
;-----------------------------------------------------------------------------
; Table for finite automaton
;-----------------------------------------------------------------------------
state_table:
		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX	;EOL

		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX	;OP

		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX	;ENDHEX

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR,	\
			P_EX,  P_ERR, P_EX,  P_EX,  P_ERR	;ENDDEC

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR,	\
			P_EX,  P_ERR, P_EX,  P_EX,  P_ERR	;ENDBIN

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR,	\
			P_EX,  P_STR, P_EX,  P_EX,  P_ERR	;ENDSTR

		state	P_BIN, P_DEC, P_HEX, P_ERR, P_ERR,	\
			P_OP,  P_STR, P_ERR, P_EOL, P_ERR	;START

		state	P_HEX, P_HEX, P_HEX, P_ERR, P_ERR,	\
			P_EH,  P_ERR, P_EH,  P_EH,  P_ERR	;HEX

		state	P_DEC, P_DEC, P_HEX, P_ED,  P_ERR,	\
			P_EH,  P_ERR, P_EH,  P_EH,  P_ERR	;DEC

		state	P_BIN, P_DEC, P_HEX, P_ED,  P_EB,	\
			P_EH,  P_ERR, P_EH,  P_EH,  P_ERR	;BIN

		state	P_STR, P_STR, P_STR, P_STR, P_STR,	\
			P_STR, P_ES,  P_STR, P_ERR, P_STR	;STR
;-----------------------------------------------------------------------------
proc_table:
		dw	eol_handler	;P_EOL
		dw	op_handler	;P_OP
		dw	atolx		;P_EH
		dw	atold		;P_ED
		dw	atolb		;P_EB
		dw	atols		;P_ES
;-----------------------------------------------------------------------------
;
init_data_end:
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of initialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;=============================================================================
; Startup code
;-----------------------------------------------------------------------------
; Return codes:
;	0 -- all ok, requested functions performed
;	1 -- unload fails because FDRC was not loaded
;	2 -- unload fails because some INTs intercepted by other programs
;	3 -- _not used_
;	4 -- installation fails because FDRC already installed
;	5 -- invalid command line option used
;
install:
		cld
		mov	si,copyright
		call	dos_print	;Print copyright message
;-----------------------------------------------------------------------------
; Parse command line
;-----------------------------------------------------------------------------
		call	parse_cmdline
		jnc	@@parsed_ok
		mov	[unk_opt_char],al
		mov	si,unexp_param
		cmp	ah,0ffh
		je	@@unexp_param
		mov	si,unk_opt_msg
@@unexp_param:
		call	dos_print
		mov	si,bad_opt_tail
		call	dos_print
		mov	al,RC_BAD_OPT
		jmp	@@exit		;Unknown option found, exit
@@parsed_ok:
;-----------------------------------------------------------------------------
; -h
;-----------------------------------------------------------------------------
		rcr	byte [opt_h],1
		jnc	@@skip_help
		mov	si,usage
		call	dos_print
		mov	al,RC_ALL_OK
		jmp	@@exit		;Print help and exit

@@skip_help:
;-----------------------------------------------------------------------------
; -y
;-----------------------------------------------------------------------------
		mov	al,[hist_items]
		mov	ah,BUFLENGTH
		mul	ah
		add	ax,15
		mov	cl,4
		shr	ax,cl
		mov	[extra_memory],ax
;-----------------------------------------------------------------------------
; -u
;-----------------------------------------------------------------------------
; REWRITE!!!
		rcr	byte [opt_u],1
		jnc	@@skip_unload

		mov	ax,MAGICK_CODE
		int	10h
		cmp	ax,(MAGICK_CODE & 0ffh) * 100h + (MAGICK_CODE >> 8)
		je	@@unload	;FDRC installed, uninstall it...

		mov	si,not_inst_msg
		call	dos_print
		mov	al,RC_NOT_INST
		jmp	@@exit		;FDRC not installed, exit
@@unload:
;-----------------------------------------------------------------------------
		xor	ax,ax
		mov	es,ax
		cmp	cx,[es:09h*4+2]
		jne	@@int09h_hooked	;INT 09h intercepted...
		cmp	cx,[es:10h*4+2]
		je	@@real_unload
@@int09h_hooked:
		mov	si,cant_unload_msg
		call	dos_print	;INT 09h and/or INT 10h intercepted
		mov	al,RC_INTERC	;by other program, exit
		jmp	@@exit
;-----------------------------------------------------------------------------
@@real_unload:
		mov	es,cx
		push	ds
		cli
		lds	dx,[es:old_int09h]
		mov	ax,2509h
		int	21h		;Restore INT 09h
		lds	dx,[es:old_int10h]
		mov	ax,2510h
		int	21h		;Restore INT 10h
		sti
		pop	ds

		mov	ah,49h		;Free used memory
		int	21h

		mov	si,uninstalled_msg
		call	dos_print
		mov	al,RC_ALL_OK
		jmp	@@exit		;FDRC uninstalled successfully, exit

@@skip_unload:
;-----------------------------------------------------------------------------
; -n
;-----------------------------------------------------------------------------
; REWRITE!!!
		rcr	byte [opt_n],1
		jnc	@@resident
		jmp	calculator	;Non-resident mode
@@exit:
		mov	ah,4ch		;Return to DOS
		int	21h

@@resident:
;-----------------------------------------------------------------------------
; -e
;-----------------------------------------------------------------------------
		rcr	byte [opt_e],1
		jc	@@opt_e_is_set

		mov	ax,MAGICK_CODE
		int	10h
		cmp	ax,(MAGICK_CODE & 0ffh)*100h+(MAGICK_CODE>>8)
		jne	@@opt_e_is_set

		mov	si,already_ins_msg
		call	dos_print
		mov	al,RC_ALR_INST
		jmp	@@exit

@@opt_e_is_set:
;-----------------------------------------------------------------------------
; Save interrupt vectors
;-----------------------------------------------------------------------------
		mov	ax,3509h
		int	21h
		mov	[old_int09h],bx
		mov	[old_int09h+2],es
		mov	ax,3510h
		int	21h
		mov	[old_int10h],bx
		mov	[old_int10h+2],es
;-----------------------------------------------------------------------------
; Allocate UMB
;-----------------------------------------------------------------------------
		mov	ax,cs
		cmp	ax,0a000h	;If we loaded via LH, skip
		jb	@@check_opt_w
		mov	word [tail_msg],inst_umb_msg
		jmp	@@set_vectors	;allocating UMB
@@check_opt_w:
		rcr	byte [opt_w],1	;Option `w' -- don't load into UMB
		jc	@@set_vectors
		mov	ax,5802h
		int	21h
		cbw
		push	ax		;UMB link state
		mov	ax,5800h
		int	21h
		push	ax		;Memory allocation strategy
		mov	ax,5803h	;Set UMB link state:
		mov	bx,1		;link UMB for allocations
		int	21h
		jc	@@restore_umb	;No UMB?
		mov	ax,5801h
		mov	bx,40h		;First fit high only
		int	21h
		jc	@@restore_umb
		mov	ah,48h		;Allocate memory block
		mov	bx,((end_of_resident-offset_0100+100h)/16)+1	;Size
		add	bx,[extra_memory];of resident part in paragraphs
		int	21h
		jc	@@restore_umb	;Restore UMB if not enough free memory
;-----------------------------------------------------------------------------
; Copy interrupt handlers into UMB and patch MCB
;-----------------------------------------------------------------------------
		dec	ax		;Seg. addr. of MCB
		mov	es,ax
		inc	ax
		mov	[es:1],ax	;Seg. addr. of PSP
		mov	bx,cs
		dec	bx
		mov	ds,bx
		mov	cx,8+(end_of_resident-offset_0100+100h)
		mov	di,8
		mov	si,di
	rep	movsb			;Copy owner's name and code+data
		push	cs
		pop	ds
		mov	word [tail_msg],inst_umb_msg
		mov	word [dos_func],4c00h
		mov	ds,ax
@@restore_umb:
;-----------------------------------------------------------------------------
; Restore allocation strategy and UMB link state
;-----------------------------------------------------------------------------
		pop	bx
		mov	ax,5801h
		int	21h		;Restore mem. alloc. strat.
		pop	bx
		mov	ax,5803h
		int	21h		;Restore UMB link state
@@set_vectors:
;-----------------------------------------------------------------------------
; Set new handlers for INT 09h and INT 10h
;-----------------------------------------------------------------------------
		cli
		mov	ax,2509h
		mov	dx,int09h_handler
		int	21h
		mov	ax,2510h
		mov	dx,int10h_handler
		int	21h
		sti

		mov	ax,[cs:2ch]
		mov	es,ax
		mov	ah,49h		;Free environment segment
		int	21h
		
		push	cs
		pop	ds

		mov	si,inst_ok_msg
		call	dos_print
		mov	si,[tail_msg]
		call	dos_print

		mov	ax,[dos_func]
		mov	dx,((end_of_resident-offset_0100+100h)/16)+1
		add	dx,[extra_memory]
		int	21h		;Terminate (and stay resident?)
;=============================================================================
; Procedures from NASMLiB
;-----------------------------------------------------------------------------
		define_dos_print
		define_parse_cmdline
		define_get_param
;=============================================================================
;
opt_a_proc:
		mov	word [fucking_proc],enter_handler_2
		clc
		ret
;-----------------------------------------------------------------------------
opt_c_proc:
		dec	byte [autoclear]
		ret
;-----------------------------------------------------------------------------
opt_e_proc:
		dec	byte [opt_e]
		ret
;-----------------------------------------------------------------------------
opt_h_proc:
		dec	byte [opt_h]
		ret
;-----------------------------------------------------------------------------
opt_n_proc:
		dec	byte [opt_n]
		ret
;-----------------------------------------------------------------------------
opt_u_proc:
		dec	byte [opt_u]
		ret
;-----------------------------------------------------------------------------
opt_w_proc:
		dec	byte [opt_w]
		ret
;-----------------------------------------------------------------------------
opt_q_proc:
		dec	byte [opt_q]
		ret
;-----------------------------------------------------------------------------
opt_y_proc:
		mov	bp,isdecdigit
		call	atol
		or	dx,dx
		js	@@quit		;Quit if negative value
		mov	[hist_items],al
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_l_proc:
		mov	bp,isdecdigit
		call	atol
		or	dx,dx
		js	@@quit		;Quit if negative value
		mov	[left],al
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_t_proc:
		mov	bp,isdecdigit
		call	atol
		or	dx,dx
		js	@@quit		;Quit if negative value
		mov	[top],al
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_k_proc:
		mov	bp,ishexdigit
		call	atol
		mov	[actkey],al
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_s_proc:
		mov	bp,ishexdigit
		call	atol
		mov	[shiftkey],al
		dec	si
		clc
		ret
;
;-----------------------------------------------------------------------------


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Non-resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

;-----------------------------------------------------------------------------
; This table must be ordered by first field!
;-----------------------------------------------------------------------------
		options_begin
		optstr	'?',	opt_h_proc
		optstr	'a',	opt_a_proc
		optstr	'c',	opt_c_proc
		optstr	'e',	opt_e_proc
		optstr	'h',	opt_h_proc
		optstr	'k',	opt_k_proc
		optstr	'l',	opt_l_proc
		optstr	'n',	opt_n_proc
		optstr	'q',	opt_q_proc
		optstr	's',	opt_s_proc
		optstr	't',	opt_t_proc
		optstr	'u',	opt_u_proc
		optstr	'w',	opt_w_proc
		optstr	'y',	opt_y_proc
		optstr	0ffh,	null_proc
		options_end
;-----------------------------------------------------------------------------
opt_h		db	0		;Help
opt_e		db	0		;Allow loading 2nd, 3rd,... TSR
opt_n		db	0		;Non-resident mode
opt_u		db	0		;Unload
opt_w		db	0		;Prevent loading into UMB
;-----------------------------------------------------------------------------
tail_msg	dw	inst_conv_msg
dos_func	dw	3100h
extra_memory	dw	0
;-----------------------------------------------------------------------------
; Macro from fdrc-msg.??
;
		non_resident_messages
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of non-resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Uninitialized resident data. This variables overlaps installation code.
; Note that some variables initialized by that code!
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
		absolute	install
;-----------------------------------------------------------------------------
old_int09h	resw	2		;Old INT 09h vector
old_int10h	resw	2		;Old INT 10h vector
video_seg	resw	1		;Segment of videobuffer
scr_rows	resb	1		;Number of rows on screen
scr_cols	resw	1		;Number of columns on screen
error_offs	resw	1		;Offset of error char in buffer
old_ss		resw	1		;Saved SS
old_sp		resw	1		;Saved SP
scr_buf		resw	SCRBUFLENGTH	;Buffer for saving screen
input_buf	resb	BUFLENGTH	;Input buffer
ltoa_buf	resb	11		;Buffer for `ltoa'
math_stack	resw	2*MSTACK	;Math stack
new_stack	resw	PSTACK		;Programm's stack
stack_bottom:				;Bottom of stack
;
history_buffer:				;History buffer (if present)
					;resides here.
end_of_resident:			;Last byte of program in memory
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of uninitialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; E0F
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

