/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2013. All rights reserved.
 */

#include <stdio.h>
#include <ctype.h>    /* tolower() */
#include <stdlib.h>   /* atoi(), qsort() - not using it after all, redefining it manually later */
#include <string.h>   /* strlen() */
#include <sys/types.h>
#include <dirent.h>
#include "kprintf.h"
#include "helpers.h"  /* fdnpkg_strcasestr() */
#include "lsm.h"
#include "pkgdb.h"
#include "showinst.h"  /* include self for control */


/* this is a wrapper around strcasecmp(), to be used by qsort() */
static int strcompare(const void *str1, const void *str2) {
  char **s1 = (char **) str1;
  char **s2 = (char **) str2;
  return(strcasecmp(*s1, *s2));
}


/* translates a version string into a array of integer values.
   returns 0 if parsing was successful, non-zero otherwise.
   Accepted formats follow:
    300.12.1
    1
    12.2.34.2-4.5
    2013-12-31   */
static int versiontointarr(char *verstr, int *arr) {
  int i, vlen, dotcounter = 1;
  char *digits[8];
  char verstrcopy[16];
  /* first extensively validate the input */
  if (verstr == NULL) return(-1);
  vlen = strlen(verstr);
  if (vlen == 0) return(-1);
  if (vlen > 15) return(-1);
  if ((verstr[0] < '0') || (verstr[0] > '9')) return(-1); /* a version string must start with a 0..9 digit */
  if ((verstr[vlen - 1] < '0') || (verstr[vlen - 1] > '9')) return(-1); /* a version string must end with a 0..9 digit */
  digits[0] = verstrcopy;
  for (i = 0; i < vlen; i++) {
    verstrcopy[i] = verstr[i];
    switch (verstr[i]) {
      case '.':
      case '-':
        if (i == 0) return(-1);
        if (verstrcopy[i-1] == 0) return(-1); /* do not allow two separators in a row */
        if (dotcounter > 7) return(-1);
        digits[dotcounter++] = &verstrcopy[i + 1];
        verstrcopy[i] = 0;
        break;
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
        break;
      default: /* do not allow any character different than 0..9 or '.' */
        return(-1);
        break;
    }
  }
  verstrcopy[i] = 0;
  /* now that we know the input is sane, let's process it */
  for (i = 0; i < dotcounter; i++) {
    int tmpint;
    tmpint = atoi(digits[i]);
    if ((tmpint < 0) || (tmpint > 0xFFFFFFF)) return(-1);
    arr[i] = tmpint;
  }
  /* pad free space with zeroes */
  for (; i < 8; i++) arr[i] = 0;
  return(0);
}


/* compares version strings v1 and v2. Returns 1 if v2 is newer than v1, 0 otherwise, and -1 if comparison is unable to tell */
static int isversionnewer(char *v1, char *v2) {
  int x1[8], x2[8], i;
  if ((v1 == NULL) || (v2 == NULL)) return(-1); /* if input is invalid (NULL), don't continue */
  if (strcasecmp(v1, v2) == 0) return(0);  /* if both versions are the same, don't continue */
  /* check versions of the decimal format 1.23... */
  if ((versiontointarr(v1, x1) == 0) && (versiontointarr(v2, x2) == 0)) {
    for (i = 0; i < 8; i++) {
      if (x2[i] > x1[i]) return(1);
      if (x2[i] < x1[i]) return(0);
    }
    return(0);
  }
  return(-1);
}


static int loadinstpkglist(char **packagelist, char **packagelist_ver, int packagelist_maxlen, char *filterstr, char *dosdir) {
  DIR *dp;
  int packagelist_len = 0, x;
  struct dirent *ep;
  #define verstr_maxlen 64
  char pkgdir[512], lsmfilename[1024], verstr[verstr_maxlen];
  sprintf(pkgdir, "%s\\packages", dosdir);
  dp = opendir(pkgdir);
  if (dp != NULL) {
      while ((ep = readdir(dp)) != NULL) {
        if (ep->d_name[0] != '.') { /* ignore '.', '..', and hidden directories */
          if (strlen(ep->d_name) > 4) {
            int tlen = strlen(ep->d_name);
            if ((ep->d_name[tlen - 4] != '.') || (tolower(ep->d_name[tlen - 3]) != 'l') || (tolower(ep->d_name[tlen - 2]) != 's') || (tolower(ep->d_name[tlen - 1]) != 't')) continue;  /* if it's not an .lst file, skip it silently */

            if (filterstr != NULL) {
              if (fdnpkg_strcasestr(ep->d_name, filterstr) == NULL) continue; /* if it's not matching the non-NULL filter, skip it */
            }
            if (packagelist_len >= packagelist_maxlen) return(-1); /* if not enough place in the list - return an error */
            packagelist[packagelist_len] = strdup(ep->d_name);
            packagelist[packagelist_len][strlen(packagelist[packagelist_len]) - 4] = 0; /* cut out the .lst extension */
            packagelist_len += 1;
          }
        }
      }
      closedir(dp);
      qsort(packagelist, packagelist_len, sizeof (char **), strcompare); /* sort the package list */
      for (x = 0; x < packagelist_len; x++) {
        /* for each package, load the metadata from %DOSDIR\APPINFO\*.lsm */
        sprintf(lsmfilename, "%s\\appinfo\\%s.lsm", dosdir, packagelist[x]);
        if (readlsm(lsmfilename, verstr, verstr_maxlen) != 0) sprintf(verstr, "(unknown version)");
        packagelist_ver[x] = strdup(verstr);
      }
      return(packagelist_len);
    } else {
      kitten_printf(9, 0, "Error: Could not access the %s directory.");
      puts("");
      return(-1);
  }
}

#define packagelist_maxlen 1024

void showinstalledpkgs(char *filterstr, char *dosdir) {
  char *packagelist[packagelist_maxlen];
  char *packagelist_ver[packagelist_maxlen];
  int packagelist_len, x;

  /* load the list of packages */
  packagelist_len = loadinstpkglist(packagelist, packagelist_ver, packagelist_maxlen, filterstr, dosdir);   /* Populate the packages list */
  if (packagelist_len < 0) return;
  if (packagelist_len == 0) {
    kitten_puts(5, 0, "No package matched the search.");
    return;
  }

  /* iterate through all packages */
  for (x = 0; x < packagelist_len; x++) {
    /* print the package/version couple on screen */
    printf("%s %s\n", packagelist[x], packagelist_ver[x]);
  }
}


/* displays the list of available updates for local packages (or a single package if pkg is not NULL). Returns 0 if some updates are found, non zero otherwise. */
int checkupdates(char *dosdir, struct pkgdb *pkgdb, char **repolist, char *pkg) {
  struct pkgdb *curpkg;
  struct pkgrepo *currepo;
  char *packagelist[packagelist_maxlen];
  char *packagelist_ver[packagelist_maxlen];
  int packagelist_len, x, foundupdate = 0;
  packagelist_len = loadinstpkglist(packagelist, packagelist_ver, packagelist_maxlen, NULL, dosdir);
  for (x = 0; x < packagelist_len; x++) {
    if (pkg != NULL) {
      if (strcasecmp(pkg, packagelist[x]) != 0) continue; /* if we got asked for a specific package, skip all other packages. */
    }
    for (curpkg = pkgdb; curpkg != NULL; curpkg = curpkg->nextpkg) { /* iterate through packages */
      if (strcasecmp(curpkg->name, packagelist[x]) == 0) {
        foundupdate = 0;
        for (currepo = curpkg->repolist; currepo != NULL; currepo = currepo->nextrepo) { /* check for possible newer version in every repo */
          if (isversionnewer(packagelist_ver[x], currepo->version) != 0) {
            if (foundupdate == 0) {
              foundupdate = 1;
              if (pkg == NULL) kitten_printf(10, 0, "%s (local version: %s)", packagelist[x], packagelist_ver[x]);
              puts("");
            }
            printf("  ");
            if (pkg == NULL) kitten_printf(10, 1, "version %s at %s", currepo->version, repolist[currepo->repo]);
            puts("");
          }
        }
        break; /* get out of the loop to get over the next local package */
      }
    }
  }
  if (foundupdate != 0) {
      return(0);
    } else {
      return(-1);
  }
}
