/*
 * This file is part of FDNPKG
 * Copyright (C) Mateusz Viste 2012, 2013
 */

#include <stdio.h>
#include <stdlib.h> /* system() */
#include <string.h> /* strcpy() */
#include "htget.h"
#include "helpers.h"   /* slash2backslash(), strtolower() */
#include "fileexst.h"
#include "kprintf.h"
#include "pkginst.h"
#include "libunzip.h"  /* callunzip(), zip_listfiles()... */



void pkginstall(struct pkgdb *pkgdb, char *pkgname, char *dosdir, char *tempdir, char *localfile, int nosourceflag, char **repolist) {
  char fname[512];
  char shellcmd[512];
  char zipfile[512];
  char appinfofile[64];
  char packageslst[64];
  int appinfopresence;
  char *instrepo;
  struct pkgdb *pkgnode, *lastnode;
  struct pkgrepo *pkgrepo;
  int repoid;
  struct ziplist *ziplinkedlist, *curzipnode;
  FILE *fd;

  strtolower(pkgname); /* convert pkgname to lower case, because the http repo is probably case sensitive! */
  sprintf(appinfofile, "appinfo\\%s.lsm", pkgname); /* Prepare the appinfo/xxxx.lsm filename string for later use */
  sprintf(packageslst, "packages\\%s.lst", pkgname); /* Prepare the packages/xxxx.lst filename string for later use */

  if (localfile != NULL) strcpy(zipfile, localfile); /* if it's a local file, then skip all network stuff */

  /* check if not already installed, if already here, print a message "you need to remove it first" */
  sprintf(fname, "%s\\packages\\%s.lst", dosdir, pkgname);
  if (fileexists(fname) != 0) { /* file exists -> package already installed */
    kitten_printf(3, 0, "Package %s is already installed! Remove it first if you need to upgrade.", pkgname);
    puts("");
    return;
  }

  if (zipfile[0] == 0) {
    /* look into the db to find the package */
    pkgnode = findpkg(pkgdb, pkgname, &lastnode);
    if (pkgnode == NULL) { /* no such package found in repositories */
      kitten_printf(3, 1, "No package '%s' found in online repositories.", pkgname);
      puts("");
      return;
    }

    /* if found - check the list of repositories */
    if (pkgnode->repolist == NULL) {
      kitten_printf(3, 2, "Package '%s' is not available in repositories.", pkgname);
      puts("");
      return;
    }

    if (pkgnode->repolist->nextrepo != NULL) { /* available from more than 1 repo.. */
        char userchoicestr[8];
        int userchoice;
        struct pkgrepo *xrepo;
        puts("");
        kitten_printf(3, 3, "%s is available from several repositories. Choose which one to use:", pkgname);
        puts("");
        repoid = 1;
        for (xrepo = pkgnode->repolist; xrepo != NULL; xrepo = xrepo->nextrepo) {
          printf(" ");
          printf("%d) %s %s (%s)\r\n", repoid, pkgnode->name, xrepo->version, repolist[xrepo->repo]);
          repoid += 1;
        }
        for (;;) {
          kitten_printf(3, 4, "Your choice:");
          printf(" ");
          fgets(userchoicestr, 6, stdin);
          userchoice = atoi(userchoicestr);
          if ((userchoice < 1) || (userchoice >= repoid)) {
              kitten_puts(3, 5, "Invalid choice!");
            } else {
              break;
          }
        }
        repoid = userchoice;
      } else { /* available only from one repository - get it there */
        repoid = 1;
    }
    pkgrepo = pkgnode->repolist;
    for (; repoid > 1; repoid--) pkgrepo = pkgrepo->nextrepo;
    instrepo = repolist[pkgrepo->repo];

    /* download the package from repoid into the temp directory */
    sprintf(fname, "%s%s.zip", instrepo, pkgname);
    sprintf(zipfile, "%s\\fdnpkg.tmp", tempdir);
    kitten_printf(3, 6, "Downloading package %s...", fname);
    puts("");
    if (htget(fname, NULL, NULL, 0, zipfile) != 0) {
      kitten_puts(3, 7, "Error downloading package. Aborted.");
      return;
    }
  } /* if (zipfile[0] == 0) */

  /* list content of the ZIP file and put it into pkgname.lst - used also to check the validity of the zip file */
  ziplinkedlist = zip_listfiles(zipfile);
  if (ziplinkedlist == NULL) {
    kitten_puts(3, 8, "Error: Invalid zip archive! Package not installed.");
    return;
  }
  /* Verify that there's no collision with existing local files, and look for the appinfo presence */
  appinfopresence = 0;
  for (curzipnode = ziplinkedlist; curzipnode != NULL; curzipnode = curzipnode->nextfile) {
    slash2backslash(curzipnode->filename); /* change all slashes to backslashes */
    strtolower(curzipnode->filename);  /* put everything into lowercase */
    sprintf(shellcmd, "%s\\%s", dosdir, curzipnode->filename);
    if (strcmp(curzipnode->filename, appinfofile) == 0) appinfopresence = 1;
    if (fileexists(shellcmd)) {
      kitten_puts(3, 9, "Error: Package contains a file that already exists locally:");
      printf(" %s\r\n", shellcmd);
      zip_freelist(&ziplinkedlist);
      return;
    }
  }
  if (appinfopresence != 1) { /* if appinfo file not found, this is not a real FreeDOS package */
    kitten_printf(3, 12, "Error: Package do not contain the %s file! Not a valid FreeDOS package.", appinfofile);
    zip_freelist(&ziplinkedlist);
    return;
  }
  /* open the lst file */
  sprintf(shellcmd, "%s\\%s", dosdir, packageslst);
  fd = fopen(shellcmd, "w");
  if (fd == NULL) {
    kitten_printf(3, 10, "Error: Could not create %s!", shellcmd);
    puts("");
    zip_freelist(&ziplinkedlist);
    return;
  }
  /* write list of files in zip into the lst */
  for (curzipnode = ziplinkedlist; curzipnode != NULL; curzipnode = curzipnode->nextfile) {
    if (nosourceflag != 0) { /* if --nosource specified, skip sources */
      if (fdnpkg_strcasestr(curzipnode->filename, "source\\") == curzipnode->filename) continue;  /* skip this file */
    }
    if (strcasecmp(curzipnode->filename, packageslst) == 0) continue; /* skip silently the package.lst file */
    fprintf(fd, "%s\r\n", curzipnode->filename); /* log the filename to packages\pkg.lst */
    /* Now unzip files into %DOSDIR% (via the unzip -d directory switch) */
    sprintf(shellcmd, "unzip -qq -C %s %s -d %s", zipfile, curzipnode->filename, dosdir);
    printf("-> %s\r\n", shellcmd);
    callunzip(shellcmd);
  }
  /* free the ziplist and close file descriptor */
  zip_freelist(&ziplinkedlist);
  fclose(fd);
  if (nosourceflag == 0) {
      kitten_printf(3, 11, "Package %s installed (with possible sources).", pkgname);
    } else {
      kitten_printf(3, 11, "Package %s installed (without sources).", pkgname);
  }
  puts("");
}
