/*
 * This file is part of the FDNPKG project.
 * Copyright (C) Mateusz Viste 2013
 */

#include <stdlib.h>    /* NULL */
#include <stdio.h>     /* printf(), FILE, fclose()... */
#include "crc32.h"
#include "tinfl.h"     /* DEFLATE support */
#include "libgz.h"     /* include self for control */

#define GZ_FLAG_ASCII 1
#define GZ_FLAG_MULTIPART_CONTINUTATION 2
#define GZ_FLAG_EXTRA_FIELD_PRESENT 4
#define GZ_FLAG_ORIG_FILENAME_PRESENT 8
#define GZ_FLAG_FILE_COMMENT_PRESENT 16
#define GZ_FLAG_FILE_IS_ENCRYPTED 32


/* gunzips a file stored in *fd into a file at desfile. Returns 0 on success, non-zero otherwise. */
int ungz(char *srcfile, char *destfile) {
  #define buffinsize 32 * 1024   /* the input buffer must be at least 32K, because that's the (usual) dic size in deflate, apparently */
  #define buffoutsize 128 * 1024 /* it's better for the output buffer to be significantly larger than the input buffer (we are decompressing here, remember? */
  FILE *fd, *dstfd;
  unsigned char compmethod;
  unsigned long cksum, cksum_from_gz;
  unsigned char *buffout;
  unsigned char *buffin;
  unsigned char flags;
  int extract_res;
  int filelen, compressedfilelen, gztotalfilelen;

  /* open the src file */
  fd = fopen(srcfile, "rb");
  if (fd == NULL) return(-3);

  /* allocate buffers for data I/O */
  buffin = malloc(buffinsize);
  buffout = malloc(buffoutsize);
  if ((buffin == NULL) || (buffout == NULL)) {
    if (buffin != NULL) free(buffin);
    if (buffout != NULL) free(buffout);
    return(-6);
  }

  /* read the uncompressed file length */
  fseek(fd, -4, SEEK_END); /* set the reading position inside the gz file */
  if (fread(buffin, 4, 1, fd) != 1) return(-1);
  filelen = buffin[0] | buffin[1] << 8 | buffin[2] << 16 | buffin[3] << 24;

  /* printf("Uncompressed file len is %d\n", filelen); */

  /* compute the total length of the gz file */
  gztotalfilelen = ftell(fd);

  /* printf("Total gz file len is %d\n", gztotalfilelen); */

  /* set the cursor of *fd to initial position */
  rewind(fd);

  /* check the magic header (2 bytes) - should be 0x1F 0x8B */
  if ((fread(buffin, 2, 1, fd) != 1) || (buffin[0] != 0x1F) || (buffin[1] != 0x8B)) return(-7);

  /* load the compression method (1 byte) - should be 0 (stored) or 8 (deflate) */
  if ((fread(&compmethod, 1, 1, fd) != 1) || ((compmethod != 0) && (compmethod != 8))) return(-8);

  /* load flags (1 byte) */
  if (fread(&flags, 1, 1, fd) != 1) return(-9);

  /* check that the file is not a continuation of a multipart gzip */
  if (flags & GZ_FLAG_MULTIPART_CONTINUTATION) return(-10);

  /* check that the file is not encrypted */
  if (flags & GZ_FLAG_FILE_IS_ENCRYPTED) return(-11);

  /* load (and discard) the file modification timestamp (4 bytes), the extra flags (1 byte) as well as OS type (1 byte) */
  if (fread(buffin, 6, 1, fd) != 1) return(-12);

  /* skip the extra field (if present) */
  if (flags & GZ_FLAG_EXTRA_FIELD_PRESENT) {
    int extrafieldlen;
    /* load the length of the extra field (2 bytes) */
    if (fread(buffin, 2, 1, fd) != 1) return(-13);
    extrafieldlen = buffin[1];
    extrafieldlen <<= 8;
    extrafieldlen |= buffin[0];
    /* skip the extra field */
    if (fread(buffin, extrafieldlen, 1, fd) != 1) return(-14);
  }
  /* skip the filename, if present (null terminated string) */
  if (flags & GZ_FLAG_ORIG_FILENAME_PRESENT) {
    int x;
    for (x = 0;; x++) {
      if ((fread(buffin, 1, 1, fd) != 1) || (x > 1024)) return(-15);
      if (buffin[0] == 0) break;
    }
  }
  /* skip the file comment, if present (null terminated string) */
  if (flags & GZ_FLAG_FILE_COMMENT_PRESENT) {
    int x;
    for (x = 0;; x++) {
      if ((fread(buffin, 1, 1, fd) != 1) || (x > 1024)) return(-16);
      if (buffin[0] == 0) break;
    }
  }

  /* compute the length of the compressed stream */
  compressedfilelen = gztotalfilelen - (ftell(fd) + 8);

  /* printf("compressed stream length is %d\n", compressedfilelen); */

  /* open the dst file */
  dstfd = fopen(destfile, "wb");
  if (dstfd == NULL) return(-2);  /* failed to open the dst file */

  /* start reading and uncompressing the compressed data, computing CRC32 at the same time */
  cksum = crc32_init(); /* init the crc32 */
  if (compmethod == 0) { /* if the file is stored, copy it over */
      unsigned long i, toread;
      extract_res = 0;   /* assume we will succeed */
      for (i = 0; (signed)i < filelen;) {
        toread = filelen - i;
        if (toread > buffoutsize) toread = buffoutsize;
        if (fread(buffout, toread, 1, fd) != 1) extract_res = -3;   /* read a chunk of data */
        crc32_feed(&cksum, buffout, toread); /* update the crc32 checksum */
        if (fwrite(buffout, toread, 1, fd) != 1) extract_res = -4; /* write data chunk to dst file */
        i += toread;
      }
    } else if (compmethod == 8) { /* the file is deflated */
      size_t total_in = 0, total_out = 0;
      unsigned int infile_remaining = compressedfilelen;
      size_t avail_in = 0;
      tinfl_decompressor *tinflhandler;
      size_t avail_out = buffoutsize;
      void *next_out = buffout;
      const void *next_in = buffin;

      extract_res = 0; /* assume we will succeed */
      tinflhandler = malloc(sizeof(tinfl_decompressor));
      if (tinflhandler == NULL) {
          extract_res = -19;
        } else {
          tinfl_init(tinflhandler);
      }

      while (extract_res == 0) {
         size_t in_bytes, out_bytes;
         tinfl_status status;
         if (!avail_in) {
            /* Input buffer is empty, so read more bytes from input file. */
            unsigned int n = buffinsize;
            if (n > infile_remaining) n = infile_remaining;
            fread(buffin, 1, n, fd);
            next_in = buffin;
            avail_in = n;
            infile_remaining -= n;
         }

         in_bytes = avail_in;
         out_bytes = avail_out;
         status = tinfl_decompress(tinflhandler, (const mz_uint8 *)next_in, &in_bytes, buffout, (mz_uint8 *)next_out, &out_bytes, (infile_remaining ? TINFL_FLAG_HAS_MORE_INPUT : 0));

         avail_in -= in_bytes;
         next_in = (const mz_uint8 *)next_in + in_bytes;
         total_in += in_bytes;

         avail_out -= out_bytes;
         next_out = (mz_uint8 *)next_out + out_bytes;
         total_out += out_bytes;

         if ((status <= TINFL_STATUS_DONE) || (!avail_out)) {
            /* Output buffer is full, or decompression is done, so write buffer to output file. */
            unsigned int n = buffoutsize - (unsigned int)avail_out;
            fwrite(buffout, 1, n, dstfd);
            crc32_feed(&cksum, buffout, n); /* update the crc32 checksum */
            next_out = buffout;
            avail_out = buffoutsize;
         }

         /* If status is <= TINFL_STATUS_DONE then either decompression is done or something went wrong. */
         if (status <= TINFL_STATUS_DONE) {
            if (status == TINFL_STATUS_DONE) { /* Decompression completed successfully. */
                extract_res = 0;
              } else {  /* Decompression failed. */
                extract_res = -15;
            }
            break;
         }
      }
      if (tinflhandler != NULL) free(tinflhandler);
  }

  /* read the CRC32 */
  if (fread(buffin, 4, 1, fd) != 1) return(-21);
  cksum_from_gz = buffin[0] | buffin[1] << 8 | buffin[2] << 16 | buffin[3] << 24;

  /* close the dst file, and compute the final (real) CRC */
  fclose(dstfd);
  crc32_finish(&cksum);

  /* check the CRC validity */
  if (cksum != cksum_from_gz) return(-30);

  return(0);
}
