/*
 * CMDELETE.C
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <io.h>
#include <arch.h>
#include <dir.h>
#include <progress.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <ifsmgr.h>
#include <dzmain.h>

#define ID_DELETE	1
#define ID_DELETEALL	2
#define ID_SKIPFILE	3
#define ID_CANCEL	4

static char
	__spath		[WMAXDIR],
	__cdir		[256];

static int callback_file(const char *, wfblk *);
static int callback_directory(const char *);
static int remove_file(const char *, const char *, char);
static int remove_directory(const char *);
static int cmdeletezip(void);

#define MSG_DELSINGEL	 1
#define MSG_DELRDONLY	-1
#define MSG_DELSYSTEM	-2

int return_update_fcb(int result)
{
	progress_close();
	if (panel_state(&panela))
		panel_update_fcb(&panela);
	if (panel_state(&panelb))
		panel_update_fcb(&panelb);
	return result;
}

void set_confirmflag(void)
{
	confirmflag = CFSELECTED;
	if (config.confirm & _C_CONFDELETE)
		confirmflag |= CFDELETEALL;
	if (config.confirm & _C_CONFDELSUB)
		confirmflag |= CFDIRECTORY;
	if (config.confirm & _C_CONFSYSTEM)
		confirmflag |= CFSYSTEM;
	if (config.confirm & _C_CONFRDONLY)
		confirmflag |= CFREADONY;
}

int cmdelete(void)
{
	FBLK *b;
	int result;

	if (!(b = cpanel_findfirst()))
		return 0;
	if (ISARCHIVE(b->flag))
		return cmdeletezip();
	if ((*cpanel->flag & _W_NETWORK) == _W_NETWORK) {
		strcpy(__spath, cpanel->ide->path);
	} else {
		if (fullpath(__spath, 0) == NULL)
			return 0;
	}
	fp_maskp = cp_stdmask;
	fp_fileblock = callback_file;
	fp_directory = callback_directory;
	set_confirmflag();
	progress_open(cp_delete, NULL);
	if (ISSELECTED(b->flag)) {
		do {
			if (ISSUBDIR(b->flag))
				result = remove_directory(b->name);
			else
				result = remove_file(__spath, wshortname(b->name), b->flag);
			if (result)
				break;
			b->flag &= ~(_A_SELECTED);
		} while ((b = panel_ffselected(cpanel)) != NULL);
	} else {
		if (ISSUBDIR(b->flag))
			result = remove_directory(b->name);
		else
			result = remove_file(__spath, wshortname(b->name), b->flag);
	}
	return return_update_fcb(result);
}

int remove_file(const char *p, const char *name, char attrib)
{
	char fname[WMAXPATH];

	if (progress_set(NULL, name, 1))
		return _ER_USERABORT;
	switch (confirm_delete_file(name, attrib)) {
		case -1: return 1;
		case  0: return 0;
		default: break;
	}
	strfcat(fname, p, name);
	if (ISRDONLY(attrib))
		setfattr(fname, 0);
	if (remove(wlongpath(fname, NULL))) {
		ermsg(cp_edel, cp_emdel, fname, sys_errlist[errno]);
		return -1;
	}
	return 0;
}

int remove_directory(const char *name)
{
	strfcat(__cdir, __spath, wshortname(name));
	switch (confirm_delete_sub(__cdir)) {
	case ID_DELETEALL:
	case ID_DELETE:
		break;
	case ID_SKIPFILE:
		return 0;
	default:
		return -1;
	}
	return scan_directory(0, __cdir);
}

int callback_file(const char *p, wfblk *q)
{
	return remove_file(p, q->name, q->attrib);
}

int callback_directory(const char *p)
{
	int result;

	if (progress_set(NULL, p, 0))
		return _ER_USERABORT;
	result = scan_files(p);
	rmdir(p);
	return result;
}

/*** .ZIP File Delete ***/

int cmdeletezip(void)
{
	FBLK *b;
	int result;

	b = cpanel_findfirst();
	if (!ISZIP(b->flag))
		return notsup();
	set_confirmflag();
	progress_open(cp_delete, NULL);
	if (ISSELECTED(b->flag)) {
		do {
			if ((result = wzipdel(cpanel->ide, b)) != 0)
				break;
			b->flag &= ~(_A_SELECTED);
		} while ((b = panel_ffselected(cpanel)) != NULL);
	} else {
		result = wzipdel(cpanel->ide, b);
	}
	return return_update_fcb(result);
}
