; ****************************************************************************
; `dialogs' - subroutines used in user interface dialogs.
;   Written by Michal H. Tyc
;
; This file is part of `BOOTMGR - multi-BOOT ManaGeR'.
;
; Copyright (c) 1997-2006 BTTR Software.  All rights reserved.
;
; This program is free software; you can redistribute it and/or modify it
; under the terms of the `MODIFIED' BSD LICENSE.  Please see `legal.txt'
; for details.
; ****************************************************************************


; Print message, wait for [Enter] or [Esc] key.
; On entry:
;   ds:si = message text
; On exit:
;   ax = 100h: [Enter] pressed
;   ax = 0: [Esc] pressed, return two levels up
;   (+ see printtext)

acknowledge:
  push si
  mov si, entertext
  call printinv                     ; initial part of the prompt
  pop si
  mov di, SCROFS(4 + ENTERLEN, 22)  ; position on the screen
  mov cl, 72 - (ENTERLEN + 1)
  call printtext                    ; variable part of the prompt
  sub di, 2 * (ESCCANLEN + 2)       ; adjust screen position
  mov si, esccantext
  mov cl, ESCCANLEN
  call printtext                    ; constant part of the prompt
.key:
  call getkey
  cmp ah, 1
  je .rtn      ; [Enter]
  test ax, ax
  jne .key     ; not [Esc]
  pop di       ; throw away return address
.rtn:
  ret


; Save the Master Boot Record to a disk file.

savembr:
  call inputfn        ; read file name
  call swapbuff       ; MBR (partition table may be modified) to file buffer
  call readmbr        ; re-read MBR from disk
  call swapbuff       ; MBR from disk to file buffer, edited one back on place
  call diskerrmsg     ; display message if error (cf still holds status)
  jb acknowledge.rtn
  stc                 ; write


; Read or write buffer from/to file.
; Check and prompt for disk change in logical drives
; (as A:/B: on single-floppy machines).
; On entry:
;   cf = nc: read
;   cf = cy: write

filerw:
  sbb ax, ax          ; write: ah = -1 (=> fn 3ch)
  push ax             ; save r/w flag
  mov ax, [linebuff]  ; get two first characters
  cmp ah, ':'
  je .drv             ; drive letter in al
  mov ah, 19h         ; otherwise get default drive
  int DOSINT
  add al, 'A'         ; convert to letter
.drv:
  cmp al, 'a'
  jb .upc
  cmp al, 'z'
  ja .upc
  and al, 0dfh  ; letter to uppercase
.upc:
  mov [dskletter], al  ; prepare message text
  sub al, 'A' - 1      ; drive number (a: = 1, etc.)
  xchg ax, bx          ; to bx
  mov ax, 440eh        ; query logical drive
  int DOSINT
  jb .open             ; ignore error
  test al, al
  je .open             ; only 1 letter for this drive
  cmp al, bl           ; bl = last letter referenced
  je .open             ; was the same
  push bx              ; save drive number from path
  mov si, insdsktext   ; `Insert disk for drive D:'
  call printinv        ; print prompt
  pop bx               ; restore drive number
.key:
  call getkey          ; read key
  test ax, ax          ; [Esc]?
  je errmessage.abort  ; give up with error
  cmp al, ' '          ; [Space]?
  jne .key             ; no, read again
  mov ax, 440fh        ; set logical drive
  int DOSINT
.open:
  mov si, copytext    ; `Copying data...'
  call printqhelp     ; print information
  pop ax              ; get back r/w flag
  push ax             ; and save for later
  add ah, 3dh         ; read: ah = 3dh, al = 0 (r/o); write: ah = 3ch
  mov cx, 20h         ; attribute: archive
  mov dx, linebuff    ; ds:dx = pathname
  int DOSINT
  xchg ax, bx         ; bx = file handle
  pop ax              ; get r/w flag
  jb .err             ; if an error occured
  neg ah
  add ah, 3fh         ; fn 3fh/40h: file read/write
  mov cx, 512         ; buffer size
  mov dx, [filebuff]  ; ds:dx = buffer
  int DOSINT
  jb .eclose          ; close file if error
  cmp ax, cx          ; all bytes transferred?
  je .rwok            ; if so, ok
.eclose:
  mov ah, 3eh  ; close file
  int DOSINT
  JMPS .err    ; finished
.rwok:
  mov ah, 3eh          ; close file
  int DOSINT
  jnb errmessage.done  ; if no error, exit
.err:
  mov si, fderrtext                                  ; message
  mov byte [byte si + fdtext - fderrtext], 'f'       ;   with word
  mov word [byte si + fdtext - fderrtext + 2], 'le'  ;   `file'
  JMPS errmessage


; Print `Incorrect value' message and wait for [Esc].
; (+see errmessage)

invalerr:
  mov si, incorrtext


; Print error message and wait for pressing [Esc].
; On entry:
;   ds:si = message text
; On exit
;   cf = cy
;   ax = cx = 0
;   dh = -1
;   si, di = ?

errmessage:
  call printinv  ; print prompt
.key:
  call getkey  ; read key
  test ax, ax  ; [Esc]?
  jne .key     ; no, wait
  push si      ; push something
.abort:
  pop si  ; throw away r/w flag if abort, or pop si back
  stc     ; mark an error
.done:
  ret


; Restore the Master Boot Record from a disk file.

restorembr:
  call inputfn                   ; read file name
; clc                            ; not needed after inputfn
  call filerw                    ; read from file
  jb errmessage.done             ; error
  call swapbuff                  ; temporarily exchange file and MBR buffers
  call updateinfo                ; preview MBR from the file
  call swapbuff                  ; restore normal function of buffers
  call loadbufp                  ; pointers to buffers
  mov bx, rsttext1
  mov cx, NTVOLBYTES - 0         ; bootstrap loader code area
  call restoreblk
  mov cl, PARTTABL - NTVOLBYTES  ; WinNT/reserved part
  call restoreblk
  mov cl, 200h - PARTTABL        ; Partition Table
  call restoreblk
  jmp installmgr.write           ; write, reload and show all data


; Restore or skip given part of MBR.
; On entry:
;   si = source pointer
;   di = destination pointer
;   cx = block length
;   bx = prompt address
; On exit:
;   si, di = advanced by cx
;   cx = 0
;   bx = next prompt
;   ah = 0
;   al = undefined

restoreblk:
  push cx
  push di                       ; save destination and count
  xchg bx, si                   ; source saved in bx, prompt address to si
  call printinv                 ; print variable part of the prompt
  push si                       ; save next prompt address
  mov si, rstskiptext           ; next part of the prompt
  sub di, 2 * (RSTSKIPLEN + 2)  ; adjust screen position
  mov cx, RSTSKIPLEN
  call printtext                ; print constant part of the prompt
  pop si
  xchg bx, si                   ; source to si, next prompt to bx
  pop di
  pop cx                        ; restore destination and count
.key:
  call getkey
  test ax, ax        ; [Esc]?
  jz .skip
  and al, 0dfh       ; letters to uppercase
  cmp al, 'S'
  je .skip
  cmp al, 'R'
  jne .key           ; neither `S' nor `R'
.rest:
  rep movsb  ; restore given block, zero cx
.skip:
  add di, cx  ; skip over the block,
  rep lodsb   ;   or do nothing if rep movsb executed; always zero cx
  ret


; Install BOOTMGR/GMBLDR in the Master Boot Record, update Partition Table.
; On entry:
;   ch = 0
; On exit:
;   all registers destroyed

installmgr:
  mov si, insttext
  call printinv
.key:
  call getkey
  jne .opt     ; ASCII character entered
  test ax, ax
  jne .key
  ret          ; [Esc] pressed
.opt:
  and al, 0dfh         ; letters to uppercase
  cmp al, 'B'
  je .instb            ; install BOOTMGR
  mov bx, mbrgen
  mov si, instgeneric
  cmp al, 'G'
  je .inst             ; install GMBLDR
  mov bx, [mbrbuff]    ; dummy copy on itself
  mov si, instptable
  cmp al, 'K'
  jne .key
  cmp [isinst], ch     ; when BOOTMGR installed, cannot just update
  je .inst             ; partition table, as hiding map has to be adjusted

; Prepare for BOOTMGR installation.

.instb:
  mov si, mbr
  call checksett        ; verify settings in boot loader code
  mov si, wrongsettext  ; error message
  jnb .ok
.jerr:
  jmp errmessage  ; errors found
.ok:
  cmp byte [activ], 80h
  jb .jerr               ; no active partition: not allowed for BOOTMGR
  mov si, hidedata       ; source to build partition flags
  mov di, mbr + sparf
.it:
  mov cl, 4          ; hiding table columns counter
  xor ah, ah         ; initialize hiding bitmap
  mov bx, [mbrbuff]
.hm:
  mov al, [bx + PARTTABL + 4]  ; get partition ID
  add bx, byte 16              ; next partition
  call ishideable              ; can this type be hidden?
  je .tbl                      ; yes, use hiding table
  and al, 10h                  ; no, use bit 4
  add al, 0f0h
  inc si                       ; skip table entry
  JMPS .bit
.tbl:
  lodsb         ; get table entry
  add al, -'1'  ; '1'..'4' -> cf, ' ' -> nc
.bit:
  rcr ah, 1                          ; build bitmap
  loop .hm
  mov cl, [byte di + sdevf - sparf]  ; get device code
  test cl, cl
  js .af                             ; HD partition: active = boot
  mov cl, [activ]                    ; other device or unused
.af:
  and cl, 0fh                   ; mask out b7
  mov al, 1
  shl al, cl                    ; al = 1, 2, 4, 8 for partitions #1..#4
  or al, ah                     ; combine with hiding bitmap
  stosb                         ; write to partition flag table
  cmp di, mbr + sparf + NITEMS  ; all items?
  jb .it
  mov bx, mbr                   ; source address for copying
  mov [bx + sitems], dl         ; set number of items (from checksett)
  mov [isinst], dl              ; make install flag nonzero
  mov si, instbootmgr           ; acknowledgement prompt

; Install (verify first). Copy loader and normalize active partition flags.

.inst:
  call acknowledge               ; break if [Esc] pressed
  mov si, bx                     ; source address
  mov di, [mbrbuff]              ; destination: MBR buffer
; mov cx, NTVOLBYTES / 2         ; ch = 0 after acknowledge
  mov cl, NTVOLBYTES / 2
  rep movsw                      ; copy the loader
  mov si, instntvol
  push di
  call printinv                  ; should we clear NT Volume Bytes?
  pop di
  mov cl, PARTTABL - NTVOLBYTES  ; ch = 0 after rep movsw
.ntkey:
  call getkey
  jnz .ntkey   ; normal key
  dec ah
  je .skipnt   ; [Enter]
  cmp ah, 8
  jne .ntkey   ; not [Del]
  rep stosb    ; zero reserved bytes
.skipnt:
  add di, cx              ; advance di anyway
  mov cl, 4               ; ch = 0 after rep movsw
  mov al, 84h
  sub al, [activ]         ; 80h..83h -> 4..1 (loop counter)
.clr:
  mov byte [di], 80h  ; set active flag
  cmp cl, al
  je .nxp             ; this one not active
  mov [di], ch        ; clear active flag
.nxp:
  add di, byte 16               ; for all 4 partitions
  loop .clr
  mov word [di], ISVALIDPART    ; set valid partition table signature
.write:
  call writembr
  mov dl, [driveno]
  jmp identhd        ; reload and show all data (needed if GMBLDR chosen)


; Write the Master Boot Record to current drive.

writembr:
  mov ah, 3       ; function: write
  call accessmbr  ; write new MBR


; Display disk error message if Carry set

diskerrmsg:
  mov si, fderrtext                                  ; error message
  mov byte [byte si + fdtext - fderrtext], 'd'       ;   with word
  mov word [byte si + fdtext - fderrtext + 2], 'sk'  ;   `disk'
  jnb .done
  jmp errmessage                                     ; if error
.done:
  ret


; Load pointers to file and MBR buffers and its size.

loadbufp:
  mov cx, 200h        ; size of whole MBR
  mov si, [filebuff]  ; source: file
  mov di, [mbrbuff]   ; destination: MBR
  ret


; Input file name from keyboard and return if valid;
; pop return address and return two levels up if invalid.
; (+ see below)

inputfn:
  mov si, fntext  ; prompt for filename


; Input text line from keyboard to linebuff and return if valid;
; pop return address and return two levels up if invalid.
; On entry:
;   si = prompt
; On exit:
;   cf = nc
;   ah = ch = 0
;   all registers except bp destroyed

inputln:
  mov di, linebuff        ; input buffer
  push di                 ; needed later
  mov cx, 60              ; buffer length
  mov al, ' '
  rep stosb               ; fill with spaces
  mov ah, 70h             ; black on white
  mov cl, 10              ; prompt length
  mov dx, SCRCRD(16, 22)  ; coordinates for inputtext
  call printprompt        ; print prompt
  pop si                  ; input buffer address
  xor cx, cx              ; initial cursor position
  mov bx, SCRCRD(60, 1)   ; field size
  stc
  push bp
  call inputtext          ; `input' mode, exit on [Esc]/[Enter]
  pop bp
  test al, al
  je .esc                 ; exit with [Esc]
  std
; mov cx, 60              ; ch = cursor row = 0
  mov cl, 60              ; length
  mov di, linebuff + 59   ; end of filename
  mov al, ' '
  repe scasb              ; scan backward for non-space
  je .esc                 ; all spaces, treat as [Esc]
  and byte [di + 2], ch   ; put \0 at the end, cf = nc
  push di                 ; push something
.esc:
  pop di     ; throw away return address, or pop di back
  cld        ; restore normal direction
; mov ah, 0  ; ax = special key index after inputtext
.done:
  ret


; Zero tracks 0 and 1.
;   cf = nc
; On exit:
;   all registers destroyed
zapdisk:
  mov si, zapwarn
  call printinv    ; are you really sure?
.key:
  call getkey
  test ax, ax
  jz inputln.done   ; [Esc]
  cmp ax, 900h
  jne .key          ; not [Del]
  call clrmbrbuf    ; zero MBR buffer
  mov si, zaptext   ; `Erasing disk...'
  call printqhelp   ; print information
; Accessing sectors in reverse order would result in somewhat shorter
; but MUCH slower code (on some machines, the delay is easily seen!)
  mov dh, 0  ; start with head 0
.head:
  xor cx, cx  ; start with sector 1
.sect:
  inc cx              ; next sector
  mov ah, 3           ; write
  call accessmbr.any
  cmp cl, [nsectors]
  jb .sect
  xor dh, 1           ; the other head
  jne .head
  jmp identhd         ; reload and show all data


; Edit BOOTMGR menu items.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

editmenu:
  mov dx, SCRCRD(4, 15)  ; field coordinates for cursor
  mov cx, [editpos]      ; get menu cursor position (row same for editassoc)
  call menucoords        ; load coordinates
; clc
  call inputtext         ; input-type dialog, `field' mode
  mov [editpos], cx      ; save cursor position for the next time
  ret


; Load menu coordinates/pointers into registers.

menucoords:
  mov di, SCROFS(3, 15)      ; menu position on the screen,
  mov bx, SCRCRD(8, NITEMS)  ;   its size
  mov si, mbr + smenu        ;   and text
  ret


; Edit partition table.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

editpart:
  mov ch, [partpos]  ; cursor row for this dialog (and selactive)
  mov bp, partdlg    ; dialog data
; clc                ; `field' mode
  call dialog        ; process dialog
  mov [partpos], ch  ; save cursor row for the next time
  ret


; Select active partition for booting from floppy or other HD.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

selactive:
  mov ch, [partpos]  ; cursor row for this dialog (and editpart)
  mov bp, activedlg  ; dialog data
; clc                ; `field' mode
  call dialog        ; process dialog
  mov [partpos], ch  ; save cursor row for the next time
  ret


; Edit associations of partitions and menu items.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

editassoc:
  mov bp, assocdlg  ; dialog data
  JMPS edithide.dlg


; Edit partition hiding table.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

edithide:
  mov bp, hidedlg  ; dialog data
.dlg:
  mov ch, [editpos + 1]  ; cursor row for this dialog (and editmenu)
; clc                    ; `field' mode
  call dialog            ; process dialog
  mov [editpos + 1], ch  ; save cursor row for the next time
  ret


; Toggle arrow marks for active partitions and default items on [Space].
; On entry:
;   ax = key code
;   (+ see below)
; On exit:
;   zf = nz: [Space] not pressed
;   zf = zr: (+ see below)

arrowssp:
  cmp al, ' '
  jne arrows.done  ; ignore other keys


; Toggle arrow marks for active partitions and default items.
; On entry:
;   di = buffer for marks
;   ch = line with arrow
;   bh = number of lines
; On exit:
;   di = ?
;   cl = number of lines
;   al = marked item number
;   cf = nc

arrows:
  xor cl, cl  ; counter
.nxt:
  mov word [di], '  '
  cmp cl, ch
  jne .noarr           ; unselected partition -- spaces
  mov word [di], '<'  ; else -- arrow
.noarr:
  add di, byte 6  ; next line (6 columns in box)
  inc cx
  cmp cl, bh      ; last line?
  jb .nxt
  mov al, ch      ; al = marked item
.done:
  ret


; Exchange device codes (or other values) if needed
; On entry:
;   dl, dh = codes to exchange
;   di = address

xchidx:
  cmp [di], dh
  je .xch
  cmp [di], dl
  jne .done
.xch:
  xor [di], dh
  xor [di], dl  ; replace the value with the other one
.done:
  ret


; Edit partition table -- keyboard handler
; On entry:
;   ax = key code
;   ch = cursor row
;   bh = 4 (number of lines)
; (+ see partprint, dialog)

partkeys:
  mov [partpos], ch
  mov cl, ch         ; save and copy cursor row
  cmp al, ' '
  je .edid           ; edit ID
  sub ah, 10         ; [PgUp]?
  je .up
  dec ah             ; [PgDn]?
  stc
;*jne .jexit         ; ignore other keys
  jne xchidx.done    ; ignore other keys
.dn:
  inc cx      ; new row
  cmp cl, bh
  jne .swap   ; not at the bottom
.jexit:
  jmp .exit         ;*!!! move
.up:
  dec cx     ; new row
  js .jexit  ; at the top, ignore key
  dec ch     ; top row
.swap:
  mov [partpos], cl  ; save new row
  mov si, hidedata   ; si = row index
  xor bh, bh         ; bx = column index
.swhd:
  mov bl, ch
  mov al, [si + bx]      ; character from left column
  xor al, [si + bx + 1]  ; and from right one
  test al, 10h
  je .nxhd               ; both spaces or both digits, no change
  mov al, ch
  mov ah, ch
  add ax, 1211h          ; create mask for each column
  xor [si + bx], ax      ; space <-> digit
.nxhd:
  add si, 4
  cmp si, hidedata + 4 * NITEMS
  jne .swhd                      ; all rows?
  mov si, [mbrbuff]
  mov al, 16
  mul ch
  add ax, PARTTABL               ; si = partition table in MBR buffer
  add si, ax                     ; si = `top' partition
  lea di, [si + 16]              ; di = next partition (to be exchanged with)
  mov bx, 16
  call swapblks                  ; swap partitions
  mov dx, 8001h
  add dh, ch                     ; dh = `top' device code
  add dl, dh                     ; dl = next device (to be exchanged with)
  mov di, mbr + sdevf
  mov cx, NITEMS                 ; association table to be scanned
.nxt:
  call xchidx    ; exchange code if needed
  inc di
  loop .nxt
  mov di, activ
  call xchidx    ; same with active partition code
  JMPS .done
.edid:
  mov si, .exit
  push si               ; if [Esc] in inputln, go to .exit
  mov si, setidtext
  call inputln          ; read text from keyboard to linebuff
  pop si                ; throw away on-error target addres
  mov cl, 16            ; base for hexadecimal number
  call asc2byte
  jc .bad
  xchg ax, cx           ; al = 16
  mul byte [partpos]
  add ax, PARTTABL + 4
  xchg ax, bx           ; bx = partpos-th type index in partition table
  add bx, [mbrbuff]
  mov [bx], al          ; set ID in partition table
  JMPS .done
.bad:
  call invalerr
.done:
  push bp           ; save dialog data pointer
  call checkhiding  ; correct hiding table
  call decodepart   ; decode modified partition table into text
  call decodeassoc  ; decode modified BOOTMGR settings
  call showsett2    ;   and show them
  pop bp
.exit:
  mov si, parthelp
  call printqhelp
  mov ch, [partpos]
; clc                ; printqhelp returns nc
  ret


; Active partition selection dialog -- keyboard handler (toggles on [Space]).
; On entry:
;   ax = key code
;   ch = cursor row
;   bh = 4 (number of lines)
; (+ see activeprint, dialog)

activekeys:
  mov di, activea  ; marks to be printed
  call arrowssp    ; change arrows if needed
  je .set          ; [Space], set active partition
  cmp ah, 9        ; [Del], set no active partition
  stc
  jne .done        ; ignore other keys
  push cx          ; save cursor row
  mov ch, -1
  call arrows      ; delete arrow, al = -1
  pop cx
.set:
  or al, 80h       ; clear Carry BTW
  mov [activ], al
; clc
.done:
  ret


; Menu item associations and default item selection dialog --
; keyboard handler
; (toggles default on [Space], associates on [1]-[4], [A], [D]).
; On entry:
;   ax = key code
;   ch = cursor row
;   bh = NITEMS (number of lines)
; (+ see activeprint, dialog)

assockeys:
  mov di, assocdata + 4   ; marks to be printed
  call arrowssp           ; change arrows if needed
  jne .partno             ; not [Space]
  mov [mbr + sdefit], ch  ; default = highlighted
; clc                     ; arrowsp returns nc
  JMPS .done
.partno:
  cmp al, '1'
  jb .done           ; ignore wrong keys
  cmp al, '4'
  ja .disks          ; not a partition number
  mov dl, '#'
  mov dh, al         ; dx = `#n' to be printed
  mov ah, 80h - '1'
  add ah, al         ; ah = partition code
  JMPS .set
.disks:
  and al, 0dfh       ; letters to uppercase
  mov dx, 'A:'       ; `A:' to be printed
; mov ah, FLOPPYID   ; ah is 0 (key = ASCII character)
  cmp al, dl
  je .set            ; floppy selected
  mov dl, 'D'        ; `D:' to be printed
  mov ah, OTHERID
  cmp al, dl
  stc
  jne .done          ; ignore wrong keys
.set:
  mov bl, ch
  xor bh, bh                  ; bx = menu item
  mov [bx + mbr + sdevf], ah  ; modify boot loader tables
  mov al, 6
  mul bl                      ; ax = 6 * menu_item
  xchg ax, bx                 ; bx = index
  mov [di + bx - 4], dx       ; put `#n' or `d:' in the highlihted line
; clc                         ; mul haven't overflown
.done:
  ret


; Partition hiding dialog -- keyboard handler
; (toggles hiding on [1]-[4]).
; On entry:
;   ax = key code
;   ch = cursor row
;   bh = NITEMS (number of lines)
; (+ see activeprint, dialog)

hidekeys:
  cmp al, '1'
  jb assockeys.done
  cmp al, '5'
  cmc
  jb assockeys.done              ; ignore wrong keys
  mov bl, ch
  xor bh, bh
  add bx, bx
  add bx, bx                     ; array row index
  and al, 1fh                    ; 11h..14h
  add bl, al
  xor [bx + hidedata - 11h], al


; Check and correct partition hiding table
; On exit:
;   al, si, di destroyed
;   cf = nc

checkhiding:
  push cx
  mov si, hidedata
.row:
  mov bx, [mbrbuff]
  mov cx, 4          ; number of columns
.col:
  lodsb
  cmp al, ' '
  je .ok                       ; not hidden -- ok
  mov al, [bx + PARTTABL + 4]  ; get partition ID
  call ishideable              ; can this type be hidden?
  je .ok
.bad:
  mov byte [si - 1], ' '  ; unhide if wrong
.ok:
  add bx, byte 16                ; next partition table entry
  loop .col
  cmp si, hidedata + NITEMS * 4  ; end of table?
  jb .row
  pop cx
  ret


; Verify if partition type can be hidden (i.e., whether it is FAT or NTFS)
; On entry:
;   al = partition ID
;   ch = 0
; On exit:
;   zf = zr if hideable, zf = nz if not

ishideable:
  push ax
  push cx
  push di            ; save registers
  and al, 0efh       ; mask `hiding'
  mov cl, NHIDEABLE
  mov di, hideable
  repne scasb        ; check if hiding allowed
  pop di
  pop cx
  pop ax             ; restore registers
  ret


; Keyboard handler for `Timeout' dialog.
; Value changed in range from 1 to 99 with [+-] keys.

timkeys:
  mov dl, [delay]           ; get current delay
  cmp al, '+'
  je .plus
  cmp al, '-'
  je .minus
  cmp al, ' '
  stc
  jne assockeys.done        ; ignore other keys
  mov si, timecounter.done
  push si                   ; if [Esc] in inputln, exit
  mov si, settmtext
  call inputln              ; read text from keyboard to linebuff
  pop si                    ; throw away on-error target addres
  mov cl, 10                ; base for decimal number
  call asc2byte
  jc .bad
  xchg ax, bx               ; ax = number
  cmp al, 100
  jb timecounter            ; in range (0..99), set and display
.bad:
  call invalerr          ; error message
  JMPS timecounter.done
.minus:
  dec dl
  jnl .set  ; nonnegative values are ok
.plus:
  inc dx
  cmp dl, 100
  je .minus    ; must be < 100
.set:
  xchg ax, dx


; Convert timeout counter to 2 ASCII digits (with leading zero),
; and prepare for printing. Modify loader for finite/infinite waiting.
; On entry:
;   al = value
; On exit:
;   (see printqhelp)

timecounter:
  mov [delay], al
  push ax                       ; save value
  aam 10
  xchg al, ah                   ; ah = delay % 10, al = delay / 10
  or ax, '00'                   ; convert to ASCII digits
  mov [timetext], ax            ; store in buffer
  pop ax                        ; restore value
  mov ah, al
  mov al, 0b2h                  ; `mov dl, #' for finite timeout
  test ah, ah
  jnz .setd
  mov ax, (sdjump << 8) + 0ebh  ; `jmp short' for infinite timeout
.setd:
  mov [mbr + sdelay - 1], ax  ; set the opcode for delay
.done:
  mov si, timeouthelp
; call printqhelp
; clc                  ; printqhelp returns nc
; ret
  jmp printqhelp       ; refresh help text (needed if number entered)


; Set timeout for boot manager.
; On entry:
;   cf = nc
; On exit:
;   all registers destroyed

settimeout:
  xor ch, ch       ; single line, so cursor fixed there
  mov bp, timedlg  ; dialog data
  JMPS dialog


; Input text from keyboard.
; On entry:
;   ds:si = text buffer in memory
;   dx = row:(col + 1) of the upper-left corner of the input field
;   cl = current cursor column in the input field
;   ch = current cursor row in the input field
;   bl = length of the input field line
;   bh = number of lines in the input field
;   es = screen buffer segment
;   di = (dh * 80 + dl - 1) * 2 = upper-left corner screen address
;   cf = dialog field/input mode
; On exit:
;   cx = new cursor position
;   bp destroyed

inputtext:
  mov bp, inputdlg  ; dialog data


; General dialog box handler.
; Handles vertical movement of highlight bar with arrow keys
; and exit from the box via [Tab]/[Shift][Tab]/[Enter]/[Esc]/[Alt][...]
; or arrow keys. For other keys, the specified handler is called.
; The box is redrawn after each key press by the specified print routine.
; On entry:
;   ss:bp = dialog information:
;     word [ss:bp + 0] = print routine
;     word [ss:bp + 2] = keyboard handler
;     Both routines can't modify bp, ch and can
;     communicate using other registers.
;     The print routine has to return box height in bh.
;     The keyboard handler gets key code in ax.
;   ch = cursor row
;   cf = nc: field mode (exit on [Tab], [Shift][Tab] or [Alt][...]),
;   cf = cy: input mode (exit on [Enter] or [Esc]), single line only
; On exit:
;   ch = new cursor row
;   ax = index of the special key that caused exit
;   all other registers except bp possibly destroyed

dialog:
  push si  ; save text address
.loop:
  pop si
  push si             ; get text address
  pushf               ; save dialog mode
  call word [bp + 0]  ; specific print routine
  call getkey
  xchg al, ah         ; ax = special key index
  jne .hand           ; ASCII characters passed to the handler
  popf
  pushf               ; get mode
  jnc .fm             ; `field' mode
  cmp al, 1
  jbe .exit           ; [Enter] or [Esc]
  JMPS .hand          ; pass other keys to the handler
.fm:
  cmp al, FIRSTALT
  jnb .exit         ; [Alt][...]
  cmp al, 3
  je .exit          ; [Shift][Tab]
  cmp al, 2
  je .exit          ; [Tab]
  cmp al, 1
  jb .next          ; ignore [Esc]
  je .ent           ; [Enter]
  cmp al, 5         ; [Up Arrow]?
  je .up
  cmp al, 6         ; [Down Arrow]?
  je .dn
.hand:
  push ax             ; keep (swapped) key code
  xchg al, ah         ; restore key code
  call word [bp + 2]  ; specific key handler
  pop ax
  jnc .next
  popf
  pushf               ; get mode
  jc .next            ; `input' mode, no exit with cursor keys
  cmp al, 7           ; [Left Arrow]?
  je .exit            ; go to previous field
  cmp al, 8           ; [Right Arrow]?
  je .exit            ; go to next field
  JMPS .next
.ent:
  xor cl, cl  ; go to first column
  inc ch      ; down
  cmp ch, bh
  jb .next    ; not in the last line
  dec ch      ; up back
  inc ax      ; translate [Enter] in last line to [Tab]
  JMPS .exit
.dn:
  inc ch      ; cursor down
  cmp ch, bh
  jb .next    ; not in the last line
  dec ch
  JMPS .exit  ; leave cursor here, exit this field
.up:
  dec ch      ; cursor up
  jns .next   ; not in the first line
  inc ch      ; leave cursor here, exit this field
.exit:
  pop si          ; throw away flags (mode)
  pop si          ; original si
  push ax         ; save exit condition
  push cx         ;   and cursor position
  mov ch, 7fh     ; none of lines highlighted
  call [bp + 0]   ; redraw
  call cursorout  ; hide cursor
  pop cx
  pop ax
  ret
.next:
  popf        ; mode
  JMPS .loop


; Print routine for input dialog.
; On entry:
;   ds:si = text buffer
; On exit:
;   ds:si = character under the cursor in buffer
; (+ see inputtext above)

inpprint:
  push dx          ; save field coordinates
  add dx, cx       ; row:col plus the position inside field
  push bx          ; save
  mov ah, 2        ; `Set Cursor Position'
  xor bx, bx       ; bh = page
  call videobios   ; int BIOSVIDEOINT
  pop bx
  push si          ; text address
  call printfield  ; print all lines
  pop si           ;
  mov al, bl       ; set text index according to cursor position:
  mul ch           ;   i = y * l + x
  add al, cl
; adc ah, 0        ; not needed, our input fields < 256 bytes
  add si, ax
  pop dx           ; field coordinates
.done:
  ret


; Keyboard handler for input dialog.
; On entry:
;   (ds, es):si = character under the cursor in buffer
; On exit:
;   cx = updated cursor position
;   ax, si destroyed
;   (+ see inputtext above)

inpkeys:
  push di      ; save screen location,
  push bx      ;   field size
  push dx      ;   and coordinates
  test al, al  ; special key?
  jne .noctl
  xchg al, ah  ; ax = key index
  sub al, 7    ; any of the cursor keys?
  jnb .crsr
  dec cl       ; [Backspace]: move left
  js .rgt      ; already in the 1st column
  dec si       ; back in the buffer
  JMPS .del
.crsr:
  je .lft
  dec ax      ; [Right Arrow]?
  je .rgt
  dec ax      ; [Del]?
  jne .pg
.del:
  mov bh, 1      ; delete mode
  call insdelch
.jdone:
  jmp .done
.noctl:
  cmp cl, bl
  jnb .jdone     ; last column, no more space
  push ax
  push si
  push bx        ; save character, address and field size
  xor bh, bh     ; insert mode
  call insdelch
  pop bx
  pop di
  pop ax
  stosb          ; store character in buffer
.rgt:
  inc cl      ; move right
  cmp cl, bl
  jbe .jdone  ; not yet in the last column
  dec cx      ; restore position
  JMPS .jign
.lft:
  dec cl      ; move left
  jns .jdone  ; not yet in the 1st column
  inc cl      ; restore position
.jign:
  stc          ; ignore the key (handled by the caller)
  jmp .ignore
.pg:
  dec bh                                  ; multiline? (i.e., editmenu?)
  je .jign                                ; else no [PgUp PgDn F7 F8]
  dec ax                                  ; [PgUp]?
  je .pgup
  dec ax                                  ; [PgDn]?
  je .pgdn
  dec ax
  xchg bh, al                             ; bh = 0/1 (insert/delete)
  sub al, ch                              ; number of lines to move
  mov si, mbr + smenu + (NITEMS - 1) * 8  ; start of menu last line
  call insdel.any                         ; insert or delete menu line
  mov bl, 6                               ; Assoc. line size
  mov si, assocdata + (NITEMS - 1) * 6    ; last Assoc. line
  call insdel.any
  mov bl, 4                               ; Hide line size
  mov si, hidedata + (NITEMS - 1) * 4     ; last Hide line
  call insdel.any
  mov si, mbr + sdevf + NITEMS - 1
  call insdel                             ; move device flags
  xor cl, cl                              ; go to the first column
  push cx                                 ; save cursor position
  mov al, [mbr + sdefit]
  cmp al, ch
  jb .showa                               ; default item above cursor
  test bh, bh
  jne .delln                              ; if not, and [F8], delete
  inc ax                                  ; [F7]: default item moved down
  cmp al, NITEMS                          ; does it fall out of the table?
  je .rst0                                ; if so, topmost item is now default
  JMPS .setd                              ; else store new default
.delln:
  dec ax      ; default item moved up
  cmp al, ch  ; maybe deleted?
  jnl .setd   ; if not, store new default
.rst0:
  xor al, al                      ; top item number
  mov word [assocdata + 4], '<'  ; put arrow in top line
.setd:
  mov [mbr + sdefit], al  ; save selection
  JMPS .showa
.pgdn:
  cmp ch, bh
  je .done    ; last row, nothing to do
  inc ch      ; go down
  push cx     ; new cursor row
  dec ch      ; `top' row
  JMPS .swap
.pgup:
  test ch, ch
  je .done     ; first row, nothing to do
  dec ch       ; new cursor row and `top' row
  push cx      ; save cursor row
.swap:
  mov si, mbr + smenu
  mov bx, 8
  call swaprows         ; swap menu items (8 characters per row),
  mov si, assocdata
  mov bl, 6
  call swaprows         ;   associations (6),
  mov si, hidedata
  mov bl, 4
  call swaprows         ;   partition hiding map (4),
  mov si, mbr + sdevf
  mov bl, 1
  call swaprows         ;   and device flags (1 byte)
  mov dh, ch
  mov dl, cl
  inc dx                ; dh = `top' item, dl = next one
  mov di, mbr + sdefit
  call xchidx           ; change default item if needed
.showa:
  mov ch, 7fh      ; no line highlighted
  call assocprint  ; show modified associations
  call hideprint   ; show modified associations
  pop cx           ; restore cursor row
.done:
  clc
.ignore:
  pop dx  ; restore field coordinates,
  pop bx  ;   field size
  pop di  ;   and screen location
  ret


; Insert/delete one character at cursor
; On entry:
;   ah = 0
;   bl = number of columns
;   cl = cursor column
;   (ds, es):si = current address in buffer
; (+ see insdel below)

insdelch:
  mov al, bl
  dec ax
  sub al, cl  ; number of moved columns
  add si, ax  ; point to the last column


; Insert/delete a record into/from an array, moving the records after it
; down/up, and clearing the inserted/last line.
; On entry:
;   bh = 0 (insert), 1 (delete)
;   al = number of records to move
;   bl = length of single record (insdel.any)
;   (ds, es):si = last record in the array
; On exit:
;   si, di = ?

insdel:
  mov bl, 1
.any:
  push bx
  push cx
  push ax
  mul bl             ; ax = (number of records) * (record size)
  xchg ax, cx        ; cx = total size of moved block
  shr bh, 1          ; bx = line length (bh = 0)
  jb .del
  dec si
  lea di, [si + bx]  ; end of next (last) record
  std                ; moving block towards higher addresses
  JMPS .move
.del:
  sub si, cx  ; start of moved block
  mov di, si  ;   is destination address
  add si, bx  ; next line is source address
.move:
  rep movsb    ; move block
  mov cx, bx   ; cx = record size
  mov al, ' '
  rep stosb    ; fill the freed record with spaces
  cld          ; restore normal direction
  pop ax
  pop cx
  pop bx
  ret


; Exchange contents of file and MBR buffers
; On exit:
;   bx = 0
;   cx = bx on entry
;   si, di destroyed

swapbuff:
  call loadbufp
  JMPS swapblks.nxt


; Swap n-th and (n + 1)-st array rows
; On entry:
;   si = array base
;   ch = n
;   bx = row size (< 256)
; On exit:
;   ax, si, di = modified
;   bx = 0

swaprows:
  mov al, bl
  mul ch
  add si, ax         ; si = n-th row
  lea di, [si + bx]  ; next row


; Swap memory blocks
; On entry:
;   si, di = blocks to swap
;   bx = length of blocks
; On exit:
;   si, di = ends of respective blocks
;   bx = 0

swapblks:
  xchg bx, cx
.nxt:
  mov al, [di]
  movsb
  mov [si - 1], al
  loop .nxt
  xchg bx, cx
  ret


; (end of dialogs.inc)
