/*
 * cd2iso.c: Turn a CD into a disk image (.ISO file).
 *
 * Jason Hood, 27 & 28 September, 2004.
 *
 * A program to create a disk image from the contents of a CD. It is intended
 * as an alternative to CDCACHER, which uses a commercial windowing library,
 * so recompiling is only available to those with the library. I've also used
 * Borland C, not Microsoft C.
 *
 * Usage is simple: specify the drive letter of the CD (if you have more than
 * one) and the name of the image (which will default to the CD label with an
 * extension of ".ISO"). This is different to CDCACHER, which uses the device
 * driver; this program requires the redirector (MSCDEX or SHSUCDX).
 *
 * Parts of this program are taken directly from CDCACHER.C by John McCoy.
 *
 * v1.01, 25 October, 2004:
 *   abort if the image would be greater than 2GB;
 *   recognise --help;
 *   option to set the number of sectors to image.
 */

#define PVERS "1.01"
#define PDATE "25 October, 2004"


#include <stdlib.h>
#include <stdio.h>
#include <alloc.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <io.h>
#include <dos.h>
#include <string.h>
#include <conio.h>
#include <time.h>

#define PriVolDescSector 16
#define ISO_ID		 "CD001"
#define HSF_ID		 "CDROM"
#define CD_ISO		 'I'
#define CD_HSF		 'H'
#define CD_Unknown	 'U'

#define OCTETS(from,to) (to - from + 1)

typedef unsigned char BYTE;
typedef unsigned int  WORD;
typedef unsigned long DWORD;

struct ISO_CD
{
  BYTE	Fill	[OCTETS(   1,	 1 )];
  char	cdID	[OCTETS(   2,	 6 )];
  BYTE	Fill2	[OCTETS(   7,	40 )];
  char	volLabel[OCTETS(  41,	72 )];
  BYTE	Fill3	[OCTETS(  73,	80 )];
  DWORD volSize;
  BYTE	Fill4	[OCTETS(  85,  830 )];
  char	modDate [OCTETS( 831,  847 )];
  BYTE	Fill5	[OCTETS( 848, 2048 )];
} far* iso;

struct HSF_CD
{
  BYTE	Fill	[OCTETS(   1,	 9 )];
  char	cdID	[OCTETS(  10,	14 )];
  BYTE	Fill2	[OCTETS(  15,	48 )];
  char	volLabel[OCTETS(  49,	80 )];
  BYTE	Fill3	[OCTETS(  81,	88 )];
  DWORD volSize;
  BYTE	Fill4	[OCTETS(  93,  806 )];
  char	modDate [OCTETS( 807,  822 )];
  BYTE	Fill5	[OCTETS( 823, 2048 )];
} far* hsf;


void  GetCDLabel( void );
int   CacheCD( int action );
void  SetFTime( int handle );
int   CDReadLong( WORD SectorCount, DWORD StartSector );
void  ReadPVD( void );
void  progress( DWORD cur, DWORD max );
int   Abort( const char* msg1, const char* msg2 );
char* thousep( DWORD num );


char far* dta;
char  CDfmt = CD_Unknown;
int   CD = -1;
char  CacheName[80];
DWORD sectors;

enum
{
  E_OK, 		// No problems
  E_MEM,		// Not enough memory
  E_NOCD,		// Not a CD drive, MSCDEX/SHSUCDX not installed,
			//  unknown CD format or no CD present
  E_EXISTS,		// Image file already exists
  E_CREATE,		// Unable to create image file or not enough free space
  E_ABORTED		// User aborted (with read/write error)
};


int main( int argc, char** argv )
{
  union REGS regs;
  int handle;
  int j;

  if (argc > 1)
  {
    if (argv[1][0] == '?' || argv[1][1] == '?' || !strcmp( argv[1], "--help" ))
    {
      puts( "CD2ISO by Jason Hood <jadoxa@yahoo.com.au>.\n"
	    "Version "PVERS" ("PDATE"). Freeware.\n"
	    "http://shsucdx.adoxa.cjb.net/\n"
	    "\n"
	    "cd2iso [Drive] [Image] [Sectors]\n"
	    "\n"
	    "Drive:   drive letter of CD (default is first CD)\n"
	    "Image:   name of image (default is CD's label + \".ISO\")\n"
	    "Sectors: number of sectors (default is entire CD)"
	  );
      return E_OK;
    }

    for (j = 1; j < argc; ++j)
    {
      char* bp;
      DWORD num;

      if (argv[j][1] == '\0' || (argv[j][1] == ':' && argv[j][2] == '\0'))
	CD = (argv[j][0] | 0x20) - 'a';
      else
      {
	num = strtoul( argv[j], &bp, 0 );
	if (*bp == '\0')
	  sectors = num;
	else
	  strcpy( CacheName, argv[j] );
      }
    }
  }

  dta = farmalloc( 30u << 11 ); // transfer up to 30 blocks at a time
  if (dta == NULL)
  {
    fputs( "Not enough memory.\n", stderr );
    return E_MEM;
  }

  if (CD == -1)
  {
    regs.x.ax = 0x1500;
    regs.x.bx = 0;
    int86( 0x2f, &regs, &regs );
    CD = regs.x.cx;
  }
  else
  {
    regs.x.ax = 0x150B;
    regs.x.cx = CD;
    int86( 0x2f, &regs, &regs );
    if (regs.x.bx != 0xADAD)
      regs.x.bx = 0;
    else if (regs.x.ax == 0)
    {
      fprintf( stderr, "%c: is not a CD drive.\n", CD + 'A' );
      return E_NOCD;
    }
  }
  if (regs.x.bx == 0)
  {
    fputs( "No CD available (install MSCDEX/SHSUCDX).\n", stderr );
    return E_NOCD;
  }

  iso = (struct ISO_CD far*)dta;
  hsf = (struct HSF_CD far*)dta;
  ReadPVD();
  if (CDfmt == CD_Unknown)
  {
    fputs( "Unknown CD format or drive not ready.\n", stderr );
    return E_NOCD;
  }
  if (!*CacheName)
    GetCDLabel();

  handle = open( CacheName, O_BINARY );
  if (handle != -1)
  {
    close( handle );
    printf( "\"%s\" already exists.\n"
	    "Press 'O' to overwrite, 'R' to resume, anything else to exit.\n",
	    CacheName );
    j = getch() | 0x20;
    if (j != 'o' && j != 'r')
      return E_EXISTS;
  }
  else
    j = 0;

  return CacheCD( j );
}


int CacheCD( int action )
{
  DWORD i, n, volSize;
  int	drv;
  struct diskfree_t df;
  int  handle;
  WORD w;
  int  rc;

  volSize = (sectors) ? sectors :
	    (CDfmt == CD_ISO) ? iso->volSize : hsf->volSize;
  if (volSize >= 1048576L)
  {
    fputs( "CD/DVD contains too much data.\n", stderr );
    return E_CREATE;
  }
  if (CacheName[1] == ':')
    drv = (*CacheName | 0x20) - 'a' + 1;
  else
    _dos_getdrive( (WORD*)&drv );
  if (_dos_getdiskfree( drv, &df ) == 0)
  {
    // FAT32 will max out at 2GB, which is sufficient.
    DWORD free = (DWORD)df.avail_clusters
			* df.sectors_per_cluster * df.bytes_per_sector;
    if (action)
    {
      handle = open( CacheName, O_BINARY );
      free += filelength( handle );
      close( handle );
    }
    if ((free >> 11) < volSize)
    {
      fprintf( stderr, "Not enough free space on %c: (%s required, ",
	       drv + 'A' - 1, thousep( volSize << 11 ) );
      fprintf( stderr, "%s available).\n", thousep( free ) );
      return E_CREATE;
    }
  }

  rc = O_BINARY | O_CREAT | O_WRONLY;
  if (action == 'o')
    rc |= O_TRUNC;
  handle = open( CacheName, rc, S_IWRITE );
  if (handle == -1)
  {
    fprintf( stderr, "\"%s\" could not be created.\n", CacheName );
    return E_CREATE;
  }
  if (action)
  {
    gotoxy( 1, wherey() - 1 );
    clreol();
    gotoxy( 1, wherey() - 1 );
    clreol();
  }
  printf( "Writing \"%s\"; size: %s.\n", CacheName, thousep( volSize << 11 ) );

  _setcursortype( _NOCURSOR );
  rc = E_OK;
  if (action == 'r')
  {
    // Back up a bit, since there may have been a write error.
    i = filelength( handle ) >> 11;
    i = (i <= 30) ? 0 : i - 30;
    lseek( handle, i << 11, SEEK_SET );
  }
  else
    i = 0;
  while (i < volSize)
  {
    progress( i, volSize );

    n = volSize - i;
    if (n > 30)
      n = 30;

    while (!CDReadLong( (WORD)n, i ))
    {
      if (Abort( "Read error", "try again" ))
	goto aborted;
    }

    for (;;)
    {
      _dos_write( handle, dta, (WORD)n << 11, (WORD*)&w );
      if (w == ((WORD)n << 11))
	break;
      if (Abort( "Write error", "try again" ))
	goto aborted;
      lseek( handle, i << 11, SEEK_SET );
    }

    if (kbhit())
    {
      if (Abort( "Paused", "continue" ))
	goto aborted;
    }

    i += n;
  }
  if (rc == E_OK)
  {
    putch( '\r' );
    clreol();
    SetFTime( handle );
  }
  else
  {
  aborted:
    rc = E_ABORTED;
    cputs( "\r\n" );
  }
  _setcursortype( _NORMALCURSOR );

  close( handle );
  return rc;
}


void progress( DWORD cur, DWORD max )
{
  static int old_pc;
  static int len;
	 int pc;

  static clock_t begin_time, total_time, rate_time;
  static WORD	 old_time;
  static DWORD	 rate_val;
	 clock_t elapsed;

  if (len == 0)
  {
    char buf[16];
    len = sprintf( buf, "%ld", max );
    cprintf( "  0.0%% [..................................................] "
	     "%*c0/%s", len - 1, ' ', buf );
    begin_time = clock();
    rate_val   = cur;
  }

  // This is number of sectors, so there's no problem with overflow.
  pc = (int)(1000 * cur / max);
  cprintf( "\r%3d.%d", pc / 10, pc % 10 );

  pc /= 10;
  gotoxy( 9 + (old_pc >> 1), wherey() );
  while ((old_pc >> 1) < (pc >> 1))
  {
    putch( '*' );
    old_pc += 2;
  }
  if (pc & 1)
    putch( '+' );

  gotoxy( 61, wherey() );
  cprintf( "%*ld", len, cur );

  // Borland defines CLK_TCK as 18.2; let's avoid floating point.
  elapsed = clock() - begin_time;
  if (elapsed >= 91)	// five seconds
  {
    if (elapsed - rate_time >= 91)
    {
      total_time = elapsed + (max - cur) * (elapsed - rate_time)
					 / (cur - rate_val);
      rate_time  = elapsed;
      rate_val	 = cur;
    }
    elapsed = (elapsed >= total_time) ? 0 : (total_time - elapsed) * 5 / 91;
    if ((WORD)elapsed != old_time)
    {
      gotoxy( 61 + 1 + (len << 1) + 1, wherey() );
      cprintf( "%2d:%02d", (WORD)elapsed / 60, (WORD)elapsed % 60 );
      old_time = (WORD)elapsed;
    }
  }
}


int Abort( const char* msg1, const char* msg2 )
{
  int rc;

  cprintf( "\r\n%s: press ESC to abort, anything else to %s.", msg1, msg2 );
  while (kbhit()) getch();
  rc = getch();
  if (kbhit()) getch(); // skip second key of function keys
  if (rc != 27)
  {
    putch( '\r' );
    clreol();
    gotoxy( 1, wherey() - 1 );
  }

  return (rc == 27);
}


int CDReadLong( WORD SectorCount, DWORD StartSector )
{
  struct REGPACK regs;

  regs.r_ax = 0x1508;
  regs.r_es = FP_SEG( dta );
  regs.r_bx = FP_OFF( dta );
  regs.r_cx = CD;
  regs.r_si = (WORD)(StartSector >> 16);
  regs.r_di = (WORD)StartSector;
  regs.r_dx = SectorCount;
  intr( 0x2f, &regs );
  return !(regs.r_flags & 1);	// carry flag set if error
}


void ReadPVD( void )
{
  if (CDReadLong( 1, PriVolDescSector ))
  {
    if (_fmemcmp( iso->cdID, ISO_ID, sizeof(iso->cdID) ) == 0)
      CDfmt = CD_ISO;
    else if (_fmemcmp( hsf->cdID, HSF_ID, sizeof(hsf->cdID) ) == 0)
      CDfmt = CD_HSF;
  }
}


void GetCDLabel( void )
{
  char far* label;
  int j;

  label = (CDfmt == CD_ISO) ? iso->volLabel : hsf->volLabel;
  for (j = 0; j < 8; ++j)
  {
    if (*label == ' ')
      break;
    CacheName[j] = *label++;
  }
  if (j == 0)
  {
    CacheName[0] = 'C';
    CacheName[1] = 'D';
    j = 2;
  }
  strcpy( CacheName + j, ".ISO" );
}


void SetFTime( int handle )
{
  char far* mod = (CDfmt == CD_ISO) ? iso->modDate : hsf->modDate;
  char buf[16];
  int  year, month, day, hour, min, sec;
  struct ftime ft;

  CDReadLong( 1, PriVolDescSector );
  _fstrncpy( buf, mod, 14 );
  sscanf( buf, "%4d%2d%2d%2d%2d%2d", &year, &month, &day,
				     &hour, &min,   &sec );
  ft.ft_year  = year - 1980;
  ft.ft_month = month;
  ft.ft_day   = day;
  ft.ft_hour  = hour;
  ft.ft_min   = min;
  ft.ft_tsec  = sec >> 1;
  setftime( handle, &ft );
}


char* thousep( DWORD num )
{
  static char sep;
  static char buf[16];
  char* pos;
  int	th;

  if (!sep)
  {
    struct COUNTRY c;
    sep = (country( 0, &c )) ? c.co_thsep[0] : ',';
  }

  pos = buf + 15;
  th  = 0;
  do
  {
    *--pos = (num % 10) + '0';
    num /= 10;
    if (++th == 3 && num)
    {
      *--pos = sep;
      th = 0;
    }
  } while (num);

  return pos;
}
