/* Move

    Copyright (c) Joe Cosentino 1997-2002.
    Copyright (c) Imre Leber 2003.
    All Rights Reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Library General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    Some code written by Rene Ableidinger.

    2003 Imre Leber - major fixes to make it MS-DOS compatible.

*/

// I N C L U D E S
//////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

#include "version.h"
#include "switchch.h"
#include "truname.h"

// D E F I N E S
////////////////////////////////////////////////////////////

#define DIR_SEPARATOR "\\"
#define R_OK 4 // Readable.

#define OVERWRITE 0
#define SKIP 1
#define ASK 2

// G L O B A L S
////////////////////////////////////////////////////////////

char opt_verify=0,opt_help=0,opt_prompt=ASK,old_verify=0,dest_drive;
int found=0;

// P R O T O T Y P E S
//////////////////////////////////////////////////////

void classify_args(char, char *[], char *, char *[], char *, char *[]);
void error(const char *);
void exit_fn(void);
char *addsep(char *);
int dir_exists(const char *);
int makedir(char *);
void build_filename(char *, const char *, const char *);
void build_name(char *, const char *, const char *);
void move_files(const char *, const char *, const char *, const char *, int);
void prepare_move(char *, char *);
void do_move(const char *,const char *);
void SplitPath(const char* path, char* drive, char* dir, char* fname, char*
ext);
char *strmcpy(char *dest, const char *src, const unsigned int maxlen);
char *strmcat(char *dest, const char *src, const unsigned int maxlen);
int FullPath(char* buffer, char* path);
int copy_file(const char *src_filename, const char *dest_filename);
int MoveDirectory(const char* src_filename, const char* dest_filename);
int RenameTree(const char* src_filename, const char* dest_filename);

#define MAXSOURCES 50
static char SourcePaths[MAXSOURCES][MAXPATH];

static int AmofSources=0;

// F U N C T I O N S
////////////////////////////////////////////////////////

/*-------------------------------------------------------------------------*/
/* Returns wether the given string contains wildcards (* or ? chars)       */
/*-------------------------------------------------------------------------*/

int ContainsWildCards(const char* text)
{
   return strchr(text, '*') || strchr(text, '?');
}

/*-------------------------------------------------------------------------*/
/* Takes a string with filenames seperated by a ',' and puts the filenames */
/* in the global array SourcePaths					   */
/*-------------------------------------------------------------------------*/

int extract_sources(int fileargc, char** fileargv)
{
    int i;
    char *sep;
    char tempname[MAXPATH];
    char argv[MAXPATH], *pargv;

    for (i= 0; i < fileargc; i++)
    {
	strmcpy(argv, fileargv[i], MAXPATH);

        /* Strip leading and ending ','s */
	pargv = strchr(argv, 0)	- 1;
	while (*pargv && (*pargv == ',')) pargv--;
	*(pargv+1) = 0;
	pargv = argv;	
	while (*pargv && (*pargv == ',')) pargv++;
	if (!*pargv) continue;

	sep = strchr(pargv, ',');
	while (sep)
	{
	    if (sep-pargv < MAXPATH)
	    {
		memcpy(tempname, pargv, sep-pargv);
		tempname[sep-pargv] = '\0';
		if (!FullPath(SourcePaths[AmofSources++], tempname))
		{
		    error("Invalid source file");
		    exit(1);
		}
	    }
	    else
		return 0;

	    sep = strchr(pargv = (sep + 1), ',');
	}

	if (strlen(pargv) < MAXPATH)
	{
	    if (!FullPath(SourcePaths[AmofSources++], pargv))
	    {
		error("Invalid source file");
		return 0;
	    }
	}
	else
	    return 0;
    }

    return 1;
}

/*-------------------------------------------------------------------------*/
/* Splits the program arguments into file and switch arguments.            */
/*-------------------------------------------------------------------------*/

void classify_args(char argc, char *argv[], char *fileargc, char *fileargv[],
char *optargc, char *optargv[])
{
    char i, *argptr;

    *fileargc=0;
    *optargc=0;
    for (i=1;i<argc;i++)
    {
        argptr=argv[i];
        if (argptr[0] == '/')
	{
	    optargv[*optargc]=argptr+1;
	    *optargc=*optargc+1;
	} // end if.
        else
	{
	    fileargv[*fileargc]=argptr;
	    *fileargc=*fileargc+1;
	} // end else.

    } // end for.

} // end classify_args.

/*-------------------------------------------------------------------------*/
/* Converts a relative path to an absolute one.            		   */
/*-------------------------------------------------------------------------*/

int FullPath(char* buffer, char* path)
{
    return Truename(buffer, path) ? 1 : 0;
}

/*-------------------------------------------------------------------------*/
/* Extracts the different parts of a path name            		   */
/*-------------------------------------------------------------------------*/

void SplitPath(const char* path, char* drive, char* dir, char* fname, char* ext)
{
    fnsplit(path, drive, dir, fname, ext);
}

/*-------------------------------------------------------------------------*/
/* Prints an error message			           		   */
/*-------------------------------------------------------------------------*/

void error(const char *error)
{
    printf(" [%s]\n", error);

} // end error.

/*-------------------------------------------------------------------------*/
/* Gets called by the "exit" command 				           */
/*-------------------------------------------------------------------------*/

void exit_fn(void)
{
    if (opt_verify)
    {
	setverify(old_verify); // Restore value of the verify flag.
    } // end if.

} // end exit_fn.

/*-------------------------------------------------------------------------*/
/* Works like function strcpy() but stops copying characters into          */
/* destination when the specified maximum number of characters (including  */
/* the terminating null character) is reached to prevent bounds violation. */
/*-------------------------------------------------------------------------*/

char *strmcpy(char *dest, const char *src, const unsigned int maxlen)
{
    unsigned int i,tmp_maxlen;

    tmp_maxlen=maxlen-1;
    i=0;
    while (src[i] != '\0' && i<tmp_maxlen)
        {
        dest[i]=src[i];
        i=i+1;
        } // end while.

    dest[i]='\0';
    return dest;

} // end strmcpy.

/*-------------------------------------------------------------------------*/
/* Works like function strcat() but stops copying characters into          */
/* destination when the specified maximum number of characters (including  */
/* the terminating null character) is reached to prevent bounds violation. */
/*-------------------------------------------------------------------------*/

char *strmcat(char *dest, const char *src, const unsigned int maxlen)
{
    unsigned int i,tmp_maxlen;
    char *src_ptr;

    tmp_maxlen=maxlen-1;
    src_ptr=(char *)src;
    i=strlen(dest);
    while (*src_ptr != '\0' && i<tmp_maxlen)
        {
        dest[i]=*src_ptr;
        src_ptr=src_ptr+1;
        i=i+1;
        } // end while.

    dest[i]='\0';
    return dest;

} // end strmcat.

/*-------------------------------------------------------------------------*/
/* Appends a trailing directory separator to the path, but only if it is   */
/* missing.                                                                */
/*-------------------------------------------------------------------------*/

char *addsep(char *path)
{
    int path_length;

    path_length=strlen(path);
    if (path[path_length-1] != *DIR_SEPARATOR)
    {
	path[path_length]=*DIR_SEPARATOR;
        path[path_length+1]='\0';
    } // end if.

    return path;

} // end addsep.

/*-------------------------------------------------------------------------*/
/* Checks if the specified path is valid. The pathname may contain a       */
/* trailing directory separator.                                           */
/*-------------------------------------------------------------------------*/

int dir_exists(const char *path)
{
    char tmp_path[MAXPATH],i;
    int attrib;

    strmcpy(tmp_path, path, sizeof(tmp_path));
    i=strlen(tmp_path);
    if (i<3)
    {
	strmcat(tmp_path, DIR_SEPARATOR, sizeof(tmp_path));
    } // end if.
    else if (i>3)
    {
	i=i-1;
	if (tmp_path[i] == *DIR_SEPARATOR)
	{
	    tmp_path[i]='\0';
	} // end if.

    } // end else.

    attrib=_chmod(tmp_path, 0);
    if (attrib == -1 || (attrib & FA_DIREC) == 0)
    {
	return 0;
    } // end if.

    return -1;

} // end dir_exists.

/*--------------------------------------------------------------------------*/
/* Builds a complete path 	                           		    */
/*--------------------------------------------------------------------------*/

int makedir(char *path)
{
    char tmp_path1[MAXPATH],tmp_path2[MAXPATH],i,length,mkdir_error;

    if (path[0] == '\0')
    {
	return -1;
    } // end if.

    strmcpy(tmp_path1, path, sizeof(tmp_path1));
    addsep(tmp_path1);
    length=strlen(tmp_path1);
    strncpy(tmp_path2, tmp_path1, 3);
    i=3;
    while (i<length)
    {
	if (tmp_path1[i]=='\\')
	{
	    tmp_path2[i]='\0';
	    if (!dir_exists(tmp_path2))
	    {
		mkdir_error=mkdir(tmp_path2);
		if (mkdir_error)
		{
		    path[i]='\0';
                    return mkdir_error;
		} // end if.

	    } // end if.

	} // end if.

        tmp_path2[i]=tmp_path1[i];
        i++;
    } // end while.

    return 0;

} // end makedir.

/*-------------------------------------------------------------------------*/
/* Uses the source filename and the filepattern (which may contain the     */
/* wildcards '?' and '*' in any possible combination) to create a new      */
/* filename. The source filename may contain a pathname.                   */
/*-------------------------------------------------------------------------*/

void build_filename(char *dest_filename,const char *src_filename,const char
*filepattern)
{
    char
drive[MAXDRIVE],dir[MAXDIR],filename_file[MAXFILE],filename_ext[MAXEXT],filepattern_file[MAXFILE],filepattern_ext[MAXEXT],tmp_filename[MAXFILE],tmp_ext[MAXEXT];

    SplitPath(src_filename, drive, dir, filename_file, filename_ext);
    SplitPath(filepattern, drive, dir, filepattern_file, filepattern_ext);
    build_name(tmp_filename, filename_file, filepattern_file);
    build_name(tmp_ext, filename_ext, filepattern_ext);
    strmcpy(dest_filename, drive, MAXPATH);
    strmcat(dest_filename, dir, MAXPATH);
    strmcat(dest_filename, tmp_filename, MAXPATH);
    strmcat(dest_filename, tmp_ext, MAXPATH);

} // end build_filename.

/////////////////////////////////////////////////////////////////////////////

void build_name(char *dest, const char *src, const char *pattern)
{
    int i,pattern_i,src_length,pattern_length;

    src_length=strlen(src);
    pattern_length=strlen(pattern);
    i=0;
    pattern_i=0;
    while ((i<src_length || (pattern[pattern_i] != '\0' &&
	   pattern[pattern_i] != '?' && pattern[pattern_i] != '*')) &&
	   pattern_i<pattern_length)
    {
        switch (pattern[pattern_i])
	{
	    case '*':
                dest[i]=src[i];
                break;
            case '?':
                dest[i]=src[i];
                pattern_i++;
                break;
            default:
                dest[i]=pattern[pattern_i];
                pattern_i++;
                break;

	} // end switch.

        i++;
    } // end while.

    dest[i]='\0';

} // end build_name.

/*-------------------------------------------------------------------------*/
/* First part get all the files to be moved, then call prepare_move on     */
/* them 							           */
/*-------------------------------------------------------------------------*/

void move_files(const char *src_pathname, const char *src_filename,
		const char *dest_pathname, const char *dest_filename,
		int movedirs)
{
    char filepattern[MAXPATH],src_path_filename[MAXPATH],dest_path_filename[MAXPATH];
    char tmp_filename[MAXFILE+MAXEXT],tmp_pathname[MAXPATH];
    struct ffblk fileblock;
    int fileattrib,done;

    fileattrib=FA_RDONLY+FA_ARCH+FA_SYSTEM;
    
    if (movedirs || !ContainsWildCards(src_filename))
	fileattrib +=FA_DIREC;     

    // Find first source file.
    strmcpy(filepattern, src_pathname, sizeof(filepattern));
    strmcat(filepattern, src_filename, sizeof(filepattern));
    done=findfirst(filepattern, &fileblock, fileattrib);
    while ((!done) && (fileblock.ff_name[0] == '.'))
	  done = findnext(&fileblock);
    
    if (done)
    {
       char buffer[80];
       sprintf(buffer, "%s%s does not exist!", src_pathname, src_filename);
       error(buffer);
    }    

    // Check if destination directory has to be created.
    if ((!done) && !dir_exists(dest_pathname))
    {
        strmcpy(tmp_pathname, dest_pathname, sizeof(tmp_pathname));
        if (makedir(tmp_pathname) != 0)
	{
	    error("Unable to create directory");
	    exit(4);
	} // end if.

    } // end if.

    // Copy files.
    while (!done)
    {
        // Build source filename including path.
        strmcpy(src_path_filename, src_pathname, sizeof(src_path_filename));
        strmcat(src_path_filename, fileblock.ff_name,
	        sizeof(src_path_filename));

        // Build destination filename including path.
        strmcpy(dest_path_filename, dest_pathname, sizeof(dest_path_filename));
        build_filename(tmp_filename, fileblock.ff_name, dest_filename);
        strmcat(dest_path_filename, tmp_filename, sizeof(dest_path_filename));
        prepare_move(src_path_filename, dest_path_filename);

	do {
	  done = findnext(&fileblock);
	} while ((!done) && (fileblock.ff_name[0] == '.'));
    } // end while.

} // end move_files.

/*-------------------------------------------------------------------------*/
/* Second part do some checks to see wether it is save to move the files   */
/* then call do_move on them 						   */
/*-------------------------------------------------------------------------*/

void prepare_move(char *src_filename, char *dest_filename)
{
    struct stat src_statbuf;
    struct dfree disktable;
    unsigned long free_diskspace;
    char input[5], ch;

    if (src_filename[strlen(src_filename)-1] == '.')
       src_filename[strlen(src_filename)-1] = 0;
    if (dest_filename[strlen(dest_filename)-1] == '.')
       dest_filename[strlen(dest_filename)-1] = 0;

    if (strcmpi(src_filename, dest_filename) == 0)
    {
       error("File cannot be copied onto itself");
       return;
    }

    if (access(dest_filename, 0) == 0)
    {
	if (!dir_exists(src_filename) &&
	    (dir_exists(dest_filename)))
	{
	   error("Cannot move a file to a directory");
	   return;
	}

	if (opt_prompt == ASK)
        {
            do {
                // Ask for confirmation to create file.
		printf("%s already exists!", dest_filename);
		printf(" Overwrite file [Y/N/All/None]? ");
		scanf("%4s", &input);
		puts("");
                fflush(stdin);
				
		if (strlen(input) == 1)
		{		
		    ch=toupper(input[0]);
		    if (ch == 'N') // No-skip file.
		    {
			return;
		    } // end if.		
		}
		else
		{
		    ch = 0;
		    if (stricmp(input, "ALL") == 0)
		    {
			opt_prompt = OVERWRITE;
		    }
		    if (stricmp(input, "NONE") == 0)
		    {
			opt_prompt = SKIP;
			return;
		    }
		}


            } while ((ch != 'Y') && (stricmp(input, "ALL") != 0));
        }
	else if (opt_prompt == SKIP)
	{
	    error("File already exists");
	    return;
	}
    }

    // Check if source and destination file are equal.
    if (stricmp(src_filename, dest_filename) == 0)
    {
	error("File cannot be copied onto itself");
	exit(4);
    } // end if.

    // Check source file for read permission.
    if (access(src_filename, R_OK) != 0)
    {
	error("Access denied");
	exit(5);
    } // end if.

    // Get info of source and destination file.
    stat((char *)src_filename, &src_statbuf);

    // Get amount of free disk space in destination drive.
    getdfree(dest_drive, &disktable);
    free_diskspace=(unsigned long) disktable.df_avail * disktable.df_sclus * disktable.df_bsec;

    // Check free space on destination disk.
    if (src_statbuf.st_size>free_diskspace)
    {
	error("Insufficient disk space in destination path");
	exit(39);
    } // end if.

    // Check free space on destination disk.
    if (src_statbuf.st_size>free_diskspace)
    {
	error("Insufficient disk space");
	exit(39);
    } // end if.

    do_move(src_filename, dest_filename);

} // end move_file.

/*-------------------------------------------------------------------------*/
/* Finaly realy move the files						   */
/*-------------------------------------------------------------------------*/

void do_move(const char *src_filename,const char *dest_filename)
{
    printf("%s => %s", src_filename, dest_filename);

    if ((src_filename[0] == dest_filename[0]) &&
        (src_filename[1] == dest_filename[1]) &&
        (src_filename[1] == ':'))
    {
	if (dir_exists(src_filename))
	{
	   if (!RenameTree(src_filename, dest_filename))
	   {
	       printf("\nProblem moving directory: %s\n", src_filename);
	       return;
	   }
	}
	else
	{
	   unlink(dest_filename);
           if (rename(src_filename, dest_filename) == 1)
	   {
	       printf("\nProblem moving file: %s\n", src_filename);
	       return;
	   }	   
	}
	
	printf(" [ok]\n");
        return;
    }

    // Check wether we need to move a directory
    if (dir_exists(src_filename))
    {
	if (!MoveDirectory(src_filename, dest_filename))
	{
	    printf("\nProblem moving file: %s\n", src_filename);
	    return;
	}
    }
    else
    {
       if (!copy_file(src_filename, dest_filename))
       {   
          printf("\nProblem moving file: %s\n", src_filename);
	  return;
       }

       unlink(src_filename); // Delete file.
    }
    printf(" [ok]\n");

} // end do_move.

/*-------------------------------------------------------------------------*/
/* Copies the source into the destination file including file attributes   */
/* and timestamp.                                                          */
/*-------------------------------------------------------------------------*/
int copy_file(const char *src_filename,
               const char *dest_filename)
{
  FILE *src_file,
       *dest_file;
  static char buffer[16384];
  unsigned int buffersize;
  int readsize, fileattrib;
  struct ftime filetime;


  /* open source file */
  src_file = fopen(src_filename, "rb");
  if (src_file == NULL)
  {
    error("Cannot open source file");
    return 0;
  }

  /* open destination file */
  dest_file = fopen(dest_filename, "wb");
  if (dest_file == NULL)
  {
    error("Cannot create destination file");
    fclose(src_file);
    return 0;
  }

  /* copy file data */
  buffersize = sizeof(buffer);
  readsize = fread(buffer, sizeof(char), buffersize, src_file);
  while (readsize > 0)
  {
    if (fwrite(buffer, sizeof(char), readsize, dest_file) != readsize)
    {
      error("Write error on destination file");
      fclose(src_file);
      fclose(dest_file);
      return 0;
    }
    readsize = fread(buffer, sizeof(char), buffersize, src_file);
  }

  /* copy file timestamp */
  getftime(fileno(src_file), &filetime);
  setftime(fileno(dest_file), &filetime);

  /* close files */
  fclose(src_file);
  fclose(dest_file);

  /* copy file attributes */
  fileattrib = _chmod(src_filename, 0);
  _chmod(dest_filename, 1, fileattrib);

  return 1;
}

/*-------------------------------------------------------------------------*/
/* Show help message	               					   */
/*-------------------------------------------------------------------------*/

void Usage(char switchch)
{
    printf("Move " VERSION "\n"
	   "Moves a file/directory to another location.\n"
	   "(C) 1997-2002 by Joe Cosentino\n"
	   "(C) 2003-2004 by Imre Leber\n\n"
	   "Syntax: MOVE [%cY | %c-Y] source1[, source2[,...]] destination [%c?]\n"
	   " source      The name of the file or directory you want to move (rename)\n"
	   " destination Where you want to move the file(s) to\n"
	   " %cY          Supresses prompting to confirm you want to overwrite\n"
	   "             an existing destination file.\n"
	   " %c-Y         Causes prompting to confirm you want to overwrite an\n"
	   "             existing destination file.\n"
	   " %cV          Verifies each file as it is written to the destination file\n"
	   "             to make sure that the destination files are identical to the\n"
	   "             source files\n"
	   " %cS	     Be sure to move directories also\n"
	   " %c?          Displays this help message\n"
	   "\n"
	   "Remark:\n"
	   "\tYou can move directories with this tool\n",
	   switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
	   switchch,
           switchch);
}

/*-------------------------------------------------------------------------*/
/* main function	               					   */
/*-------------------------------------------------------------------------*/

int main(int argc, char *argv[])
{
    char switchch = SwitchChar();
    char fileargc,*fileargv[255],optargc,*optargv[255],dest_pathname[MAXPATH]="";
    char *ptr,option[255]="",i,environ[255],src_filename[MAXFILE+MAXEXT]="",dest_filename[MAXFILE+MAXEXT]="";
    int length, movedirs = 0;

    classify_args(argc, argv, &fileargc, fileargv, &optargc, optargv);
    atexit(exit_fn);

    // Read COPYCMD to set confirmation.
    strncpy(environ,getenv("COPYCMD"),sizeof(environ));
    if (environ[0] != '\0')
    {
        strupr(environ);
        if ((environ[0] == '/') || (environ[0] == switchch))
        {
            if ((environ[1] == 'Y') && (environ[2] == '\0'))
                opt_prompt = OVERWRITE;	// Overwrite existing files.
            else if (((environ[1] == 'N') && (environ[2] == '\0')) ||
                     ((environ[1] == '-') && (environ[2] == 'Y') && (environ[3] == '\0')))
                opt_prompt = SKIP;		// Skip existing files.
        }
    } // end if.

    if ((optargc == 1) && (stricmp(optargv[0], "?") == 0))
    {
        Usage(switchch);
        exit(0);
    }

    for (i=0;i<optargc;i++) // Get options.
    {
	strcpy(option, optargv[i]);
	if (stricmp(option, "v") == 0)
	{
	    old_verify=getverify();
            opt_verify=-1;
	} // end if.
	else if (stricmp(option, "y") == 0)
	    opt_prompt=OVERWRITE;
	else if (stricmp(option, "-y") == 0)
	    opt_prompt=SKIP;
	else if (stricmp(option, "s") == 0)
	    movedirs = 1;
	else
	{
            printf("Invalid parameter-%s\n", optargv[i]);
            exit(4);
	} // end else.

    } // end for.

    if (fileargc<2)
    {
	error("Required parameter missing");
	return 1;
    } // end if.

    if (!extract_sources(fileargc-1, fileargv))
    {
	error("Invalid source specification");
	return 4;
    }


    for (i=0; i<AmofSources; i++)
    {
        if (SourcePaths[i][0] == '\0')
        {
            printf("Invalid source drive specification\n");
            exit(4);
        } // end if.

        // Check source path.
        // Source path contains a filename/-pattern -> separate it.
        ptr=strrchr(SourcePaths[i], *DIR_SEPARATOR);
        ptr++;
        strncpy(src_filename, ptr, sizeof(src_filename));
        *ptr='\0';
        if (!dir_exists(SourcePaths[i]))
        {
            error("Source path not found");
            exit(4);
        } // end if.

	addsep(SourcePaths[i]);
	length=strlen(SourcePaths[i]);
	if (length>(MAXDRIVE-1+MAXDIR-1))
	{
	   error("Source path too long\n");
	   exit(4);
        } // end if.

	// Get destination pathname (with trailing backspace) and filename/-pattern.
	if (fileargc<2)
	{
	   // No destination path specified -> use current.
	   getcwd(dest_pathname, MAXPATH);
	   strncpy(dest_filename, "*.*", sizeof(dest_filename));
        } // end if.
	else
        {
	   // Destination path specified.
	   length=strlen(fileargv[fileargc-1]);
	   if (length>(MAXPATH-1))
	   {
	      error("Destination path too long\n");
	      exit(4);
	    } // end if.

	    if (!FullPath(dest_pathname, fileargv[fileargc-1]))
	    {
		printf("Invalid destination file\n");
		exit(1);
	     }

	    if (dest_pathname[0] == '\0')
	    {
		error("Invalid destination drive specification\n");
		exit(4);
	    } // end if.

	    // Check destination path.
	    if (fileargv[fileargc-1][length-1] != *DIR_SEPARATOR && !dir_exists(dest_pathname))
	    {
		// If more then one file was used for the source
		// see wether the destination contains wild cards,
		// if it does not try to create the destination as
		// a directory.
		if (((AmofSources > 1) ||
		     (ContainsWildCards(src_filename))) &&
		    (!ContainsWildCards(fileargv[fileargc-1])))
		{
		      char ch = 0;

		      while ((ch != 'n') && (ch != 'N'))
		      {
			  printf("%s does not exist as directory. Create it? [Y/N] ",
				 fileargv[fileargc-1]);
			  scanf("%c", &ch);
			  puts("");

			  if ((ch == 'Y') || (ch == 'y'))
			  {
			     strncpy(dest_filename, "*.*", sizeof(dest_filename));
			     break;
			  }
		      }

		      if ((ch == 'n') || (ch == 'N'))
		      {
			 exit(0);
		      }
		}
		else
		{
		    ptr=strrchr(dest_pathname, *DIR_SEPARATOR);
		    ptr++;
		    strncpy(dest_filename, ptr, sizeof(dest_filename));
		    *ptr='\0';
		}
	    } // end if.
	    else
            {
		// Destination is a directory.
		strncpy(dest_filename, "*.*", sizeof(dest_filename));
	    } // end else.

        } // end else.

	addsep(dest_pathname);
	length=strlen(dest_pathname);
	if (length>(MAXDRIVE-1+MAXDIR-1))
	{
	   error("Destination path too long\n");
	   exit(4);
        } // end if.

	dest_drive=toupper(dest_pathname[0])-64;
	if (opt_verify)
	   setverify(1);

	move_files(SourcePaths[i], src_filename, dest_pathname, dest_filename, movedirs);
    }

    return 0;

} // end main.
