/* find.c */

/* Copyright (C) 1994-2002, Jim Hall <jhall@freedos.org> */

/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

/* This program locates a string in a text file and prints those lines
 * that contain the string.  Multiple files are clearly separated.
 */

#include <stdio.h>
#include <stdlib.h>			/* borland needs this for 'exit' */
#include <string.h>
#include <ctype.h>

#include <dir.h>			/* for findfirst, findnext */
#include <dos.h>			/* for findfirst, findnext */

#include "getopt.h"			/* GNU Getopt */
#include "find_str.h"			/* find_str() back-end */
#include "catgets.h"			/* Cats message library */


/* Functions */

void usage (nl_catd cat);


/* Main program */

int
main (int argc, char **argv)
{
  char *s, *needle;
  int c, i, done, ret;

  unsigned drive, thisdrive;		/* used to store, change drive */
  char cwd[MAXDIR], thiscwd[MAXDIR];	/* used to store, change cwd */

  char drv[MAXDRIVE];			/* temporary buffer */
  unsigned maxdrives;			/* not used */

  int invert_search = 0;		/* flag to invert the search */
  int count_lines = 0;			/* flag to whether/not count lines */
  int number_output = 0;		/* flag to print line numbers */
  int ignore_case = 0;			/* flag to be case insensitive */

  FILE *pfile;				/* file pointer */
  nl_catd cat;				/* message catalog */
  struct ffblk ffblk;			/* findfirst, findnext block */

  /* Message catalog */

  cat = catopen ("find", 0);

  /* Scan the command line */

  while ((c = getopt (argc, argv, "cCiInNvV?")) != EOF)
    {
      switch (c)
	{
	case 'c':
	case 'C':		/* Count */
	  count_lines = 1;
	  break;

	case 'i':
	case 'I':		/* Ignore */
	  ignore_case = 1;
	  break;

	case 'n':
	case 'N':		/* Number */
	  number_output = 1;
	  break;

	case 'v':
	case 'V':		/* Not with */
	  invert_search = 1;
	  break;

	default:
	  usage (cat);
	  catclose (cat);
	  exit (2);		/* syntax error .. return errorlevel 2 */
	  break;
	}
    }

  /* Get the string */

  if ((c = optind) >= argc)
    {
      /* No string? */
      usage (cat);
      catclose (cat);
      exit (1);
    }

  else
    {
      /* Assign the string to find */
      needle = argv[c++];
    }

  /* Store the drive and cwd */

  /* findfirst/findnext do not return drive and cwd information, so we
     have to store the drive & cwd at the beginning, then chdir for
     each file we scan using findfirst, then chdir back to the initial
     drive & cwd at the end.  This is probably not the most efficient
     way of doing it, but it works.  -jh */

  _dos_getdrive (&drive);		/* 1 for A, 2 for B, ... */
  getcwd (cwd, MAXDIR);			/* also stores drive letter */

#if 0 /* debugging */
  fprintf (stderr, "drive=%c\n", (drive+'A'-1));
  fprintf (stderr, "cwd=%s\n", cwd);
#endif /* debugging */

  /* Scan the files for the string */

  if ((argc - c) == 0)
    {
      /* No files on command line - scan stdin */
      ret = find_str (needle, stdin, invert_search, count_lines, number_output, ignore_case);
    }

  else
    {
      for (i = c; i < argc; i++)
	{
	  /* find drive and wd for each file when using findfirst */

	  fnsplit (argv[i], drv, thiscwd, NULL, NULL);

	  /* use findfirst/findnext to expand the filemask */

	  done = findfirst (argv[i], &ffblk, 0);

	  if (done)
	    {
	      /* We were not able to find a file. Display a message and
		 set the exit status. */

	      s = catgets (cat, 2, 1, "No such file");
	      fprintf (stderr, "FIND: %s: %s\n", argv[i], s);
	    }

	  while (!done)
	    {
	      /* We have found a file, so try to open it */

	      /* set cwd to the filemask */

	      _dos_setdrive ((drv[0] - 'A' + 1), &maxdrives);
	      chdir (thiscwd);

	      /* open the file, or not */

	      if ((pfile = fopen (ffblk.ff_name, "r")) != NULL)
		{
		  printf ("---------------- %s\n", ffblk.ff_name);
		  ret = find_str (needle, pfile, invert_search, count_lines, number_output, ignore_case);
		  fclose (pfile);
		}

	      else
		{
		  s = catgets (cat, 2, 0, "Cannot open file");
		  fprintf (stderr, "FIND: %s: %s\n", argv[i], s);
		}

	      /* return the cwd */

	      _dos_setdrive (drive, &maxdrives);
	      chdir (cwd);

	      /* find next file to match the filemask */

	      done = findnext (&ffblk);
	    } /* while !done */
	} /* for each argv */
    } /* else */

  /* Done */

  catclose (cat);


 /* RETURN: If the string was found at least once, returns 0.
  * If the string was not found at all, returns 1.
  * (Note that find_str.c returns the exact opposite values.)
  */

  exit ( (ret ? 0 : 1) );
}

/* Show usage */

void
usage (nl_catd cat)
{
  char *s;

  fprintf (stderr, "FreeDOS Find, version 2.6\n");
  fprintf (stderr, "GNU GPL - copyright 1994-2002 Jim Hall <jhall@freedos.org>\n\n");

  s = catgets (cat, 0, 0, "Prints all lines of a file that contain a string");
  fprintf (stderr, "FIND: %s\n", s);

  s = catgets (cat, 1, 1, "string");
  fprintf (stderr, "FIND [ /C ] [ /I ] [ /N ] [ /V ] \"%s\" ", s);
  s = catgets (cat, 1, 0, "file");
  fprintf (stderr, "[ %s... ]\n", s);

  s = catgets (cat, 0, 1, "Count the number of lines that contain string");
  fprintf (stderr, "  /C  %s\n", s);

  s = catgets (cat, 0, 2, "Ignore case");
  fprintf (stderr, "  /I  %s\n", s);

  s = catgets (cat, 0, 3, "Number the displayed lines, starting at 1");
  fprintf (stderr, "  /N  %s\n", s);

  s = catgets (cat, 0, 4, "Print lines that do not contain the string");
  fprintf (stderr, "  /V  %s\n", s);
}
