/*
   boot.c - boot sector manipulation code.
   Copyright (C) 2000 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>

#include "..\header\rdwrsect.h"
#include "..\header\boot.h"
#include "..\header\bool.h"

int ReadBootSector(RDWRHandle handle, struct BootSectorStruct* buffer)
{
    if (ReadSectors(handle, 1, 0, buffer) != -1)
    {
       memcpy(&(handle->SectorsPerCluster), &(buffer->SectorsPerCluster), 15);
       return TRUE;
    }
    else
       return FALSE;
}

int WriteBootSector(RDWRHandle handle, struct BootSectorStruct* buffer)
{
    /* Synchronize cache values. */
    memcpy(&(handle->SectorsPerCluster), &(buffer->SectorsPerCluster), 15);

    return (WriteSectors(handle, 1, 0, buffer) != -1);
}

static int ReadBootSectorIfNeeded(RDWRHandle handle)
{
    struct BootSectorStruct boot;

    if ((handle->SectorsPerCluster == 0) && (!ReadBootSector(handle, &boot)))
       return FALSE;
    else
       return TRUE;
}

void InvalidateBootInfo(RDWRHandle handle)
{
    handle->SectorsPerCluster = 0;
}

unsigned char GetSectorsPerCluster(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->SectorsPerCluster;
    else
       return FALSE;
}

unsigned short GetReservedSectors(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->ReservedSectors;
    else
       return FALSE;
}

unsigned char GetNumberOfFats(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->Fats;
    else
       return FALSE;
}

unsigned short GetNumberOfRootEntries(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->NumberOfFiles;
    else
       return FALSE;
}

unsigned char GetMediaDescriptor(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->descriptor;
    else
       return FALSE;
}

unsigned short GetNumberOfSectors(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->NumberOfSectors;
    else
       return FALSE;
}

unsigned short GetSectorsPerFat(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->SectorsPerFat;
    else
       return FALSE;
}

unsigned short GetSectorsPerTrack(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->SectorsPerTrack;
    else
       return FALSE;
}

unsigned short GetReadWriteHeads(RDWRHandle handle)
{
    if (ReadBootSectorIfNeeded(handle))
       return handle->Heads;
    else
       return FALSE;
}

unsigned short GetClustersInDataArea(RDWRHandle handle)
{
    unsigned char sectorspercluster = GetSectorsPerCluster(handle);

    if (!sectorspercluster) return FALSE;

    return (GetNumberOfSectors(handle) -
	    GetReservedSectors(handle) -
            (GetNumberOfRootEntries(handle) / 16) -
            (GetNumberOfFats(handle) * GetSectorsPerFat(handle))) /
           GetSectorsPerCluster(handle);
}
