/*
 *  DIR.C - dir internal command
 *
 *  Comments:
 *
 *  01/29/97 (Tim Norman)
 *    started.
 *
 *  06/13/97 (Tim Norman)
 *    Fixed code.
 *
 *  07/12/97 (Tim Norman)
 *    Fixed bug that caused the root directory to be unlistable
 *
 *  07/12/97 (Marc Desrochers)
 *    Changed to use maxx, maxy instead of findxy()
 *
 *  06/08/98 (Rob Lake)
 *    Added compatibility for /w in dir
 *
 * Compatibility for dir/s started 06/09/98 -- Rob Lake
 * 06/09/98 (Rob Lake)
 * -Tested that program finds directories off root fine
 *
 *
 * 06/10/98 (Rob Lake)
 *      -do_recurse saves the cwd and also stores it in Root
 *      -build_tree adds the cwd to the beginning of
 *   its' entries
 *      -Program runs fine, added print_tree -- works fine.. as EXE,
 *   program won't work properly as COM.
 *
 * 06/11/98 (Rob Lake)
 * -Found problem that caused COM not to work
 *
 * 06/12/98 (Rob Lake)
 *      -debugged...
 *      -added free mem routine
 *
 * 06/13/98 (Rob Lake)
 *      -debugged the free mem routine
 *      -debugged whole thing some more
 *      Notes:
 *      -ReadDir stores Root name and _Read_Dir does the hard work
 *      -PrintDir prints Root and _Print_Dir does the hard work
 *      -KillDir kills Root _after_ _Kill_Dir does the hard work
 *      -Integrated program into DIR.C(this file) and made some same
 *       changes throughout
 *
 * 06/14/98 (Rob Lake)
 *      -Cleaned up code a bit, added comments
 *
 * 06/16/98 (Rob Lake)
 * - Added error checking to my previously added routines
 *
 * 06/17/98 (Rob Lake)
 *      - Rewrote recursive functions, again! Most other recursive
 *      functions are now obsolete -- ReadDir, PrintDir, _Print_Dir,
 *      KillDir and _Kill_Dir.  do_recurse does what PrintDir did
 *      and _Read_Dir did what it did before along with what _Print_Dir
 *      did.  Makes /s a lot faster!
 *  - Reports 2 more files/dirs that MS-DOS actually reports
 *      when used in root directory(is this because dir defaults
 *      to look for read only files?)
 *      - Added support for /b, /a and /l
 *      - Made error message similar to DOS error messages
 * - Added help screen
 *
 * 06/20/98 (Rob Lake)
 * - Added check for /-(switch) to turn off previously defined
 * switches
 * - Added ability to check for DIRCMD in environment and
 * process it
 *
 * 06/21/98 (Rob Lake)
 * - Fixed up /B
 * - Now can dir *.ext/X, no spaces!
 *
 * 06/29/98 (Rob Lake)
 *      - error message now found in command.h
 *
 * 07/08/1998 (John P. Price)
 * - removed extra returns; closer to MSDOS
 * - fixed wide display so that an extra return is not displayed when
 *   there is five filenames in the last line.
 *
 * 07/12/98 (Rob Lake)
 * - Changed error messages
 *
 * 27-Jul-1998 (John P Price <linux-guru@gcfl.net>)
 * - added config.h include
 *
 * 09-Aug-1998 (Rob Lake <rlake@cs.mun.ca>)
 * - fixed bug that caused blank lines to be printed
 * - renamed _Read_Dir to Read_Dir
 *
 * 10-Aug-1998 ska
 * - added checks for ^break
 *
 * 03-Dec-1998 John P Price
 * - Rewrote DIR command.  fixed problem with "DIR .COM" and other bugs
 *   Recursive switch does not work now, but it will be added soon.
 *
 * 31-Jan-1999 (John P. Price)
 * - Changed dir_print_header to use function INT21,AH=69 instead of the
 *   function it was using.  I'm not sure if this will fix anything or not.
 *   Also fixed bug with changing and restoring the current drive.
 *
 */

#include "config.h"

#ifdef INCLUDE_CMD_DIR

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <dir.h>
#include <dirent.h>
#include <dos.h>
#include <io.h>
#include <conio.h>
#include <string.h>
#include <alloc.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "command.h"
#include "strings.h"

/* useful macros */
#define MEM_ERR error_out_of_memory(); return 1;

/* flag definitions */
/* Changed hex to decimal, hex wouldn't work
 * if > 8, Rob Lake 06/17/98.
 */
enum
{
  DIR_RECURSE = 0x01,
  DIR_PAGE = 0x02,
  DIR_WIDE = 0x04,              /* Rob Lake */
  DIR_BARE = 0x08,              /* Rob Lake */
  DIR_ALL = 0x10,               /* Rob Lake */
  DIR_LWR = 0x20                /* Rob Lake */
};

/* Globally save the # of dirs, files and bytes,
 * probabaly later pass them to functions. Rob Lake  */
long recurse_dir_cnt;
long recurse_file_cnt;
long recurse_bytes;

/*
 * dir_read_param
 *
 * read the parameters from the command line
 */
int dir_read_param(char *line, char **param, unsigned *flags)
{
  int slash = 0;

	*param = NULL;

  /* scan the command line, processing switches */
  while (*line)
  {
    /* process switch */
    if (*line == '/' || slash)
    {
      if (!slash)
        line++;
      slash = 0;
      if (*line == '-')
      {
        line++;
      }
      if (toupper(*line) == 'S')
        *flags |= DIR_RECURSE;
      else if (toupper(*line) == 'P')
        *flags |= DIR_PAGE;
      else if (toupper(*line) == 'W')
				*flags |= DIR_WIDE;
      else if (toupper(*line) == 'B')
        *flags |= DIR_BARE;
			else if (toupper(*line) == 'A')
        *flags |= DIR_ALL;
      else if (toupper(*line) == 'L')
        *flags |= DIR_LWR;
      else
      {
        error_invalid_switch(toupper(*line));
        return 1;
      }
      line++;
      continue;
		}

    /* process parameter */
    if (!isspace(*line))
    {
      if (*param)
      {
        error_too_many_parameters(*param);
        return 1;
      }

			*param = line;

      /* skip to end of line or next whitespace or next / */
      while (*line && !isspace(*line) && *line != '/')
        line++;

      /* if end of line, return */
      if (!*line)
        return 0;

      /* if parameter, remember to process it later */
      if (*line == '/')
        slash = 1;

      *line++ = 0;
      continue;
    }

    line++;
  }

  if (slash)
  {
		error_invalid_switch(toupper(*line));
    return 1;
  }

  return 0;
}

/*
 * pause
 *
 * pause until a key is pressed
 */
int pause(void)
{
  cmd_pause(NULL);

  return 0;
}

/*
 * incline
 *
 * increment our line if paginating, display message at end of screen
 */
int incline(int *line, unsigned flags)
{
  if (!(flags & DIR_PAGE))
    return 0;

  (*line)++;

  if (*line >= *maxy)
  {
    *line = 0;
    return pause();
  }

  return 0;
}

/*
 * dir_print_header
 *
 * print the header for the dir command
 */
int dir_print_header(int drive, int *line, unsigned flags)
{
	struct media_id
	{
		int info_level;
		int serial1;
		int serial2;
		char vol_id[11];
		char file_sys[8];
	}
	media;
	struct ffblk f;
	struct SREGS s;
	union REGS r;
	int disk;

	if (cbreak)
		return 1;

	if (flags & DIR_BARE)
		return 0;

	disk = getdisk();
	setdisk(drive);
	if (getdisk() != drive)
	{
		setdisk(disk);
		error_invalid_drive();
		return 1;
	}

	/* get the media ID of the drive */
/*	media.info_level = 0;
	r.x.bx = drive + 1;
	r.x.cx = 0x866;
	s.ds = FP_SEG(&media);
	r.x.dx = FP_OFF(&media);
	r.x.ax = 0x440d;
	int86x(0x21, &r, &r, &s);
	media.vol_id[10] = NULL;
*/

/*
Format of disk info:
Offset  Size    Description     (Table 01766)
00h    WORD    0000h (info level)
02h    DWORD   disk serial number (binary)
06h 11 BYTEs   volume label or "NO NAME    " if none present
11h  8 BYTEs   (AL=00h only) filesystem type (see #01767)

*/

	r.x.ax = 0x6900;
	r.x.bx = drive + 1;
	s.ds = FP_SEG(&media);
	r.x.dx = FP_OFF(&media);
	int86x(0x21, &r, &r, &s);

	/* print drive info */
	printf("\n Volume in drive %c", drive + 'A');

	if (FINDFIRST("\\*.*", &f, FA_LABEL) == 0)
	{
		printf(" is %s\n", f.ff_name);
	}
	else
	{
		printf(" has no label\n");
	}

	setdisk(disk);

	if (incline(line, flags) != 0)
		return 1;

	/* print the volume serial number if the return was successful */
	if (!r.x.cflag)
	{
		printf(" Volume Serial Number is %04X-%04X\n", media.serial2, media.serial1);
		if (incline(line, flags) != 0)
			return 1;
	}

	return 0;
}

/*
 * convert
 *
 * insert commas into a number
 */
int convert(long num, char *des)
{
  char temp[32];
  int c = 0;
  int n = 0;

  if (num == 0)
  {
    des[0] = 0x30;
    des[1] = 0;
    n = 1;
  }
  else
  {
    temp[31] = 0;
    while (num > 0)
    {
      if (((c + 1) % 4) == 0)
        temp[30 - c++] = ',';
      temp[30 - c++] = (char)(num % 10) + 0x30;
      num /= 10;
    }
    for (n = 0; n <= c; n++)
      des[n] = temp[31 - c + n];
  }
  return n;
}

/*

 * print_summary: prints dir summary
 * Added by Rob Lake 06/17/98 to compact code
 * Just copied Tim's Code and patched it a bit
 *
 */
int print_summary(int drive, long files, long dirs, long bytes,
                  unsigned flags, int *line)
{
  char buffer[32];
  union REGS r;

  if (flags & DIR_BARE)
    return 0;

  convert(files, buffer);
  printf("   %6s file%c", buffer, files == 1 ? ' ' : 's');
  convert(bytes, buffer);
  printf("   %12s byte%c\n", buffer, bytes == 1 ? ' ' : 's');
  if (incline(line, flags) != 0)
    return 1;

  /* print number of dirs and bytes free */
  printf("%9d dirs", dirs);
  r.h.ah = 0x36;
  r.h.dl = drive + 1;
  int86(0x21, &r, &r);
  convert((long)r.x.ax * r.x.bx * r.x.cx, buffer);
  if ((flags & DIR_RECURSE) == 0)
    printf(" %15s bytes free\n", buffer);
  if (incline(line, flags) != 0)
    return 1;
  return 0;
}

/*
 * dir_list
 *
 * list the files in the directory
 */
int dir_list(char *pathspec, int *line, unsigned flags)
{
  char *ext;
  char buffer[32];
  struct ffblk file;
  long bytecount = 0;
  long filecount = 0;
  long dircount = 0;
  int time;
  int count;
  unsigned mode = FA_RDONLY | FA_ARCH | FA_DIREC;

  /* if the user wants all files listed RL 06/17/98 */
  if (flags & DIR_ALL)
    mode |= FA_HIDDEN | FA_SYSTEM;

  if (FINDFIRST(pathspec, &file, mode) != 0)
  {
    /* Don't want to print anything if scanning recursively
     * for a file. RL
     */
    if ((flags & DIR_RECURSE) == 0)
    {
      error_file_not_found();
      incline(line, flags);
      return 1;
    }
    return 0;
  }

  /* moved down here because if we are recursively searching and
   * don't find any files, we don't want just to print
   * Directory of C:\SOMEDIR
   * with nothing else
   * Rob Lake 06/13/98
   */
  if ((flags & DIR_BARE) == 0)
  {
    printf(" Directory of %s\n", pathspec);
    if (incline(line, flags) != 0)
      return 1;
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }

/* For counting columns of output */
  count = 0;

  do
  {
/* begin Rob Lake */
    if (flags & DIR_LWR)
      strlwr(file.ff_name);

    if (flags & DIR_WIDE && !(flags & DIR_BARE))
    {
      if (file.ff_attrib & FA_DIREC)
      {
        sprintf(buffer, "[%s]", file.ff_name);
        dircount++;
      }
      else
      {
        sprintf(buffer, "%s", file.ff_name);
        filecount++;
      }
      printf("%-15s", buffer);
      count++;
      if (count == 5)
      {
        /* outputted 5 columns */
        printf("\n");
        if (incline(line, flags) != 0)
          return 1;
        count = 0;
      }
      bytecount += file.ff_fsize;
      /* next block 06/17/98 */
    }
    else if (flags & DIR_BARE)
    {
      if (strcmp(file.ff_name, ".") == 0 || strcmp(file.ff_name, "..") == 0)
        continue;
      if (flags & DIR_RECURSE)
      {
        char dir[128];
        sprintf(dir, "%s", pathspec);
        if (flags & DIR_LWR)
          strlwr(dir);
        printf(dir);
      }
      printf("%-13s\n", file.ff_name);
      if (file.ff_attrib & FA_DIREC)
        dircount++;
      else
        filecount++;
      if (incline(line, flags) != 0)
        return 1;
      bytecount += file.ff_fsize;
    }
    else
    {
/* end Rob Lake */
      if (file.ff_name[0] == '.')
        printf("%-13s", file.ff_name);
      else
      {
        ext = strchr(file.ff_name, '.');
        if (!ext)
          ext = "";
        else
          *ext++ = 0;

        printf("%-8s %-3s ", file.ff_name, ext);
      }

      if (file.ff_attrib & FA_DIREC)
      {
        printf("%-14s", "<DIR>");
        dircount++;
      }
      else
      {
        convert(file.ff_fsize, buffer);
        printf("   %10s ", buffer);
        bytecount += file.ff_fsize;
        filecount++;
      }

      printf("%.2d-%.2d-%d", ((file.ff_fdate >> 5) & 0x000f),
             (file.ff_fdate & 0x001f), ((file.ff_fdate >> 9) + 80));
      time = file.ff_ftime >> 5 >> 6;
      printf("  %2d:%.2u%c\n",
             (time == 0 ? 12 : (time <= 12 ? time : time - 12)),
             ((file.ff_ftime >> 5) & 0x003f),
             (time <= 11 ? 'a' : 'p'));

      if (incline(line, flags) != 0)
        return 1;

    }
    if (cbreak)
      return 1;
  }
  while (FINDNEXT(&file) == 0);

/* Rob Lake, need to make clean output */
/* JPP 07/08/1998 added check for count != 0 */
  if ((flags & DIR_WIDE) && (count != 0))
  {
    printf("\n");
    if (incline(line, flags) != 0)
      return 1;
  }

  if (filecount || dircount)
  {
    recurse_dir_cnt += dircount;
    recurse_file_cnt += filecount;
    recurse_bytes += bytecount;
    /* The code that was here is now in print_summary */
    if (print_summary(pathspec[0] - 'A', filecount, dircount,
                      bytecount, flags, line) != 0)
      return 1;
  }
  else
  {
    error_file_not_found();
    return 1;
  }
  return 0;
}

///*
// * Read_Dir: Actual function that does recursive listing
// */
//int Read_Dir(int drive, char *parent, char *filespec, int *lines,
//             unsigned flags)
//{
//  DIR *dir;
//  struct dirent *ent;
//
//  if ((dir = opendir(parent)) == NULL)
//    return 1;
//
//  strupr(parent);
//  if (parent[strlen(parent) - 1] == '\\')
//    parent[strlen(parent) - 1] = '\0';
//
//  while ((ent = readdir(dir)) != NULL && !cbreak)
//  {
//    char buffer[128];
//    struct stat stbuf;
//    if (strcmp(ent->d_name, ".") == 0)
//      continue;
//    if (strcmp(ent->d_name, "..") == 0)
//      continue;
//    sprintf(buffer, "%s\\%s", parent, ent->d_name);
//    /* changed call to _chmod to a call to stat,
//     * faster? or is it another interrupt call?
//     */
//    if (stat(buffer, &stbuf) == -1)
//      return 1;
//    if ((stbuf.st_mode & S_IFMT) == S_IFDIR)
//    {
//      if (dir_list(drive, buffer, filespec, lines, flags) != 0)
//      {
//        closedir(dir);
//        return 1;
//      }
//      /* RL: This caused blanks lines being printed..
//         if ((flags & DIR_BARE) == 0)
//         {
//         printf("\n");
//         if (incline(lines, flags) != 0)
//         return 1;
//         } */
//      if (chdir(buffer) == -1)
//      {
//        closedir(dir);
//        return 1;
//      }
//      if (Read_Dir(drive, buffer, filespec, lines, flags) == 1)
//      {
//        closedir(dir);
//        return 1;
//      }
//      chdir("..");
//    }
//  }
//
//  if (closedir(dir) != 0)
//    return 1;
//  return 0;
//}

///*
// * do_recurse: Sets up for recursive directory listing
// */
//int do_recurse(int drive, char *directory, char *filespec,
//               int *line, unsigned flags)
//{
//  char cur_dir[MAXDIR];
//
//  recurse_dir_cnt = recurse_file_cnt = recurse_bytes = 0L;
//  setdisk(drive);
//  getcwd(cur_dir, sizeof(cur_dir));
//
//  if (chdir(directory) == -1)
//    return 1;
//  if (dir_print_header(drive, line, flags) != 0 || cbreak)
//    return 1;
//  if (dir_list(drive, directory, filespec, line, flags) != 0)
//    return 1;
//  if ((flags & DIR_BARE) == 0)
//  {
//    printf("\n");
//    if (incline(line, flags) != 0)
//      return 1;
//  }
//  if (Read_Dir(drive, directory, filespec, line, flags) != 0)
//    return 1;
//  if ((flags & DIR_BARE) == 0)
//    printf("\nTotal files listed:\n");
//  flags &= ~DIR_RECURSE;
//  if (print_summary(drive, recurse_file_cnt,
//                    recurse_dir_cnt, recurse_bytes, flags, line) != 0)
//    return 1;
//  if ((flags & DIR_BARE) == 0)
//  {
//    printf("\n");
//    if (incline(line, flags) != 0)
//      return 1;
//  }
//  chdir(cur_dir);
//  return 0;
//}
//
//
//

/* looks for wildcards in a path.  Returns 1 if any are found,
   else returns 0 */
int wildcards(char *path)
{
  if (strchr(path, '*'))
    return 1;
  if (strchr(path, '?'))
    return 1;
  return 0;
}

char *fullfilespec(const char *path, char *newpath)
{
  char drive[MAXDRIVE];
  char dir[MAXDIR];
  char fname[MAXFILE];
  char ext[MAXEXT];
  struct ffblk ffblk;

  _fullpath(newpath, path, MAXPATH);

//  printf("Fullpath = %s\n",newpath);

  if (!wildcards(newpath))
  {
//    printf("no wildcards.\n");
    if (findfirst(newpath, &ffblk, FA_DIREC | FA_ARCH) == 0)
    {
//      printf("findfirst returned 0; attrib=%d\n",ffblk.ff_attrib);
      //      printf("name = %s\n",ffblk.ff_name);
      if (ffblk.ff_attrib & FA_DIREC)
      {
        strcat(newpath, "\\");
      }
    }
  }
//  printf("Fullpath = %s\n",newpath);

  _splitpath(newpath, drive, dir, fname, ext);

//  printf("drive = %s\n",drive);
  //  printf("dir = %s\n",dir);
  //  printf("name = %s\n",fname);
  //  printf("ext = %s\n",ext);

  if (!*fname && !*ext)
  {
    strcpy(fname, "*");
    strcpy(ext, ".*");
  }

  if (!*fname)
  {
    strcpy(fname, "*");
  }

  _makepath(newpath, drive, dir, fname, ext);
  return newpath;
}

/*
 * dir
 *
 * internal dir command
 */
#pragma argsused
int cmd_dir(char *rest)
{
  unsigned flags = 0;
  char *param;
  char *dircmd;
  int line = 0;
  int rv;                       /* return value */

  char path[MAXPATH];

  /* read the parameters from env */
  dircmd = getEnv("DIRCMD");
  if (dir_read_param(dircmd, &param, &flags) != 0)
    return 1;
  /* read the parameters */
  if (dir_read_param(rest, &param, &flags) != 0)
    return 1;

  /* default to current directory */
  if (!param)
    param = ".";

  fullfilespec(param, path);

  rv = 1;                       /* failed */
	/* print the header */
	if (dir_print_header(toupper(path[0]) - 'A', &line, flags) != 0)
		goto ret;

	rv = dir_list(path, &line, flags);

ret:

	return rv;
}

#endif

#ifdef INCLUDE_CMD_VOL

/*
 * vol
 *
 * internal vol command
 */
#pragma argsused
int cmd_vol(char *rest)
{
  int orig_drive;
  int d;
  int line = 0;

  /* save the current directory info */
  orig_drive = getdisk();

  if (*rest)
  {
    if ((strlen(rest) == 2) && (rest[1] == ':'))
    {
      d = toupper(rest[0]) - 'A';
      dir_print_header(d, &line, 0);
      setdisk(orig_drive);
    }
    else
    {
      display_string(TEXT_ERROR_INVALID_PARAMETER, rest);
      return 1;
    }
  }
  else
    dir_print_header(orig_drive, &line, 0);

  return 0;
}
#endif
