/*
 * COMMAND.C - command-line interface.
 *
 * Comments:
 *
 * 17 Jun 1994 (Tim Norman)
 *   started.
 *
 * 08 Aug 1995 (Matt Rains)
 *   I have cleaned up the source code. changes now bring this source
 *   into guidelines for recommended programming practice.
 *
 *   A added the the standard FreeDOS GNU licence test to the
 *   initialize() function.
 *
 *   Started to replease puts() with printf(). this will help
 *   standardize output. please follow my lead.
 *
 *   I have added some constants to help making changes easier.
 *
 * 15 Dec 1995 (Tim Norman)
 *   major rewrite of the code to make it more efficient and add
 *   redirection support (finally!)
 *
 * 6 Jan 1996 (Tim Norman)
 *   finished adding redirection support!  Changed to use our own exec
 *   code (MUCH thanks to Svante Frey!!)
 *
 * 29 Jan 1996 (Tim Norman)
 *   added support for CHDIR, RMDIR, MKDIR, and ERASE, as per suggestion
 *   of Steffan Kaiser
 *
 *   changed "file not found" error message to "bad command or filename"
 *   thanks to Dustin Norman for noticing that confusing message!
 *
 *   changed the format to call internal commands (again) so that if they
 *   want to split their commands, they can do it themselves (none of the
 *   internal functions so far need that much power, anyway)
 *
 *
 * 27 Aug 1996 (Tim Norman)
 *   added in support for Oliver Mueller's ALIAS command
 *
 * 14 Jun 1997 (Steffan Kaiser)
 *   added ctrl-break handling and error level
 *
 * 16 Jun 1998 (Rob Lake)
 *   Runs command.com if /P is specified in command line.  Command.com
 *   also stays permanent.  If /C is in the command line, starts the
 *   program next in the line.
 *
 * 21 Jun 1998 (Rob Lake)
 *   Fixed up /C so that arguments for the program
 *
 * 08-Jul-1998 (John P. Price)
 *   - Now sets COMSPEC environment variable
 *   - misc clean up and optimization
 *   - added date and time commands
 *   - changed to using spawnl instead of exec.  exec does not copy the
 *     environment to the child process!
 *
 * 14 Jul 1998 (Hans B Pufal)
 *   Reorganised source to be more efficient and to more closely follow
 *   MS-DOS conventions. (eg %..% environment variable replacement
 *   works form command line as well as batch file.
 *
 *   New organisation also properly support nested batch files.
 *
 *   New command table structure is half way towards providing a
 *   system in which COMMAND will find out what internal commands
 *   are loaded
 *
 * 24 Jul 1998 (Hans B Pufal) [HBP_003]
 *   Fixed return value when called with /C option
 *
 * 27 Jul 1998  John P. Price
 * - added config.h include
 *
 * 28 Jul 1998  John P. Price
 * - added showcmds function to show commands and options available
 *
 * 07-Aug-1998 (John P Price <linux-guru@gcfl.net>)
 * - Fixed carrage return output to better match MSDOS with echo on or off.
 *   (marked with "JPP 19980708")
 *
 * 10-Aug-1998 ska
 * - fixed exit code of "/C" options
 * - readded ^Break catcher (^Break without catcher doesn't make much sense)
 *
 * 27-Oct-1998 ska
 * - changed: creation of name of temporary file of pipe (tmpnam()), no
 *  need for *tempdir, no need for malloc()'ed fname0&1
 * - bugfix: in pipe "cmd1 | cmd2 | cmd3" when switching to cmd3
 *  the tempfile of cmd2's input is removed, before it's closed.
 *  That's not recommended.
 * - bugfix: 'abort:' label sometimes missed a reset-option
 * - bugfix: moved the removal of fname1 behind the 'close()' of the
 *  opened file.
 *
 * 19-Nov-1998 (Rob Green <robg@sundial.net>)
 * - changed line in is_delim function to include backslash and period.
 *
 * 30-Nov-1998 (Rob Green <robg@sundial.net>)
 * - Removed previous change (Opps)
 *
 * 30-Nov-1998 (John P Price <linux-guru@gcfl.net>)
 * - Added code to remove white space from end of command in docommand
 *   function.  Also added nextcmd char pointer so we don't loose the other
 *   commands in the line.
 *   This fixes the bug with "type filename.c | more"
 *   (marked with "JPP 19981130")
 *
 * 1998/12/04 ska
 * - chg: moved tracemode interaction from  process_input() into
 *  paresecommandline(): This won't miss _any_ commands now and does
 *  now display the line after variable and alias expansion (if ECHO
 *  is ON, this adds an useful amount of information for the tracing
 *  user).
 * - chg: use vcgetcstr() to gather user input.
 * - add: tracemode interaction is now ^Break aware
 *
 * 1998/12/05 ska
 * - add: forceLow variable
 *
 * 1999/01/24 ska
 * add: support for CP/M style device names (openf.h)
 *
 * 09-Feb-1999 (John P Price <linux-guru@gcfl.net>)
 * - changed the spawnl call back to using the exec function.  Now that we
 *   are handling the environment correctly, this seems to work better.  It
 *   passes new environment variables (even ones that were created by this
 *   copy of command.com) to the child process.  This was suggested by ska.
 *
 * 24-Mar-1999 (John P Price <linux-guru@gcfl.net>)
 * - added support for swapable exec.
 * - changed the way we run autoexec.bat or a batch file on the command line.
 *
 */

#include "config.h"

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <dos.h>
#include <process.h>
#include <time.h>
#include <errno.h>
#include <dir.h>
#include <fcntl.h>
#include <io.h>
#include <sys\stat.h>

#include "command.h"
#include "batch.h"
#include "strings.h"

#include "openf.h"

#ifdef FEATURE_SWAP_EXEC
#include "swapexec.h"
#endif
extern struct CMD cmds[];       /* The internal command table */

int exitflag = 0;               /* indicates EXIT was typed */
int canexit = 1;                /* indicates if this shell is exitable */
int ctrlBreak = 0;              /* Ctrl-Break or Ctrl-C hit */
int errorlevel = 0;             /* Errorlevel of last launched external program */
int forceLow = 0;               /* load resident copy into low memory */

void fatal_error(char *s)
{
  /*
   *  fatal error handler.
   *
   */

  printf("fatal_error() : %s\n", s);

  exit(100);
}

static char is_delim(char c)
{
  /*
   *  is character a delimeter when used on first word?
   *
   */

  return (c == '/' || c == '=' || c == 0 || isspace(c));
}

static void execute(char *first, char *rest)
{
  /*
   * This command (in first) was not found in the command table
   *
   *
   * first - first word on command line
   * rest  - rest of command line
   *
   */

  char *fullname;

  if (strlen(first) + strlen(rest) + 1 > MAX_EXTERNAL_COMMAND_SIZE)
  {
    error_line_too_long();
    return;
  }

  /* check for a drive change */
  if ((strcmp(first + 1, ":") == 0) && isalpha(*first))
  {
    setdisk(toupper(*first) - 'A');

    if (getdisk() != toupper(*first) - 'A')
      display_string(TEXT_ERROR_INVALID_DRIVE);

    return;
  }

  /* get the PATH environment variable and parse it */
  /* search the PATH environment variable for the binary */
  fullname = find_which(first);
  dprintf(("[find_which returned %s]\n", fullname));

  if (!fullname)
  {
    error_bad_command();
    return;
  }

  /* check if this is a .BAT file */

  if (stricmp(strrchr(fullname, '.'), ".bat") == 0)
  {
    dprintf(("[BATCH: %s %s]\n", fullname, rest));
    batch(fullname, first, rest);
  }
  else
    /* exec the program */
  {
    int result;

    dprintf(("[EXEC: %s %s]\n", fullname, rest));
//    result = spawnl(P_WAIT, fullname, fullname, rest, NULL);
#ifdef FEATURE_SWAP_EXEC
    result = do_exec(fullname, rest, USE_ALL, 0xFFFF, environ);
#else
    result = exec(fullname, rest, 0);
#endif
    if (result == -1)
      perror("executing spawnl function");
    else
      errorlevel = result;
  }
}

static void docommand(char *line)
{
  /*
   * look through the internal commands and determine whether or not this
   * command is one of them.  If it is, call the command.  If not, call
   * execute to run it as an external program.
   *
   * line - the command line of the program to run
   *
   */

  char *com;                    /* the first word in the command */
  char *cp;
  char *cstart;
  char *rest = line;            /* pointer to the rest of the command line */

  int cl;

  struct CMD *cmdptr;

  if ((com = malloc(strlen(line) + 1)) == NULL)
  {
    error_out_of_memory();
    return;
  }

  cp = com;

  while (isspace(*rest))        /* Skip over initial white space */
    rest++;

  /* JPP 19981130 */
  while (isspace(rest[strlen(rest) - 1]))	/* delete white space at end of command */
    rest[strlen(rest) - 1] = '\0';

  cstart = rest;

  if (*rest)                    /* Anything to do ? */
  {
    while (!is_delim(*rest))    /* Copy over 1st word as lower case */
      *cp++ = tolower(*rest++);

    *cp = '\0';                 /* Terminate first word */

    while (isspace(*rest))      /* Skip over whitespace to rest of line */
      rest++;

    for (cmdptr = cmds;; cmdptr++)	/* Scan internal command table */
    {
      if (cmdptr->name == NULL) /* If end of table execute ext cmd */
      {
        execute(com, rest);
        break;
      }

      if (strcmp(com, cmdptr->name) == 0)
      {
        /* JPP this will print help for any command */
        if (strstr(rest, "/?"))
        {
          display_string(cmdptr->help_id);
        }
        else
        {
          dprintf(("CMD '%s' : '%s'\n", com, rest));
          cmdptr->func(rest);
        }
        break;
      }

      /* The following code handles the case of commands like CD which
       * are recognised even when the command name and parameter are
       * not space separated.
       *
       * e.g dir..
       * cd\freda
       */

      cl = strlen(cmdptr->name);	/* Get length of command name */

      if ((cmdptr->flags & CMD_SPECIAL) &&
          (strncmp(cmdptr->name, com, cl) == 0) &&
          (strchr("\\.", *(com + cl))))
      {
        // OK its one of the specials...

        com[cl] = '\0';         /* Terminate first word properly */

        cmdptr->func(cstart + cl);	/* Call with new rest */
        break;
      }
    }
  }
  free(com);
}

/*
 * process the command line and execute the appropriate functions
 * full input/output redirection and piping are supported
 *
 */
void parsecommandline(char *s)
{
  char *in = NULL;
  char *out = NULL;
  char *fname0 = NULL;
  char *fname1 = NULL;
  char *nextcmd;

  int of_attrib = O_CREAT | O_TRUNC | O_TEXT | O_WRONLY;
  int num;
  int oldinfd = -1;
  int oldoutfd = -1;

  dprintf(("[parsecommandline (%s)]\n", s));

  /* first thing we do is alias expansion */

#ifdef FEATURE_ALIASES
  aliasexpand(s, MAX_INTERNAL_COMMAND_SIZE);
  dprintf(("[alias expanded to (%s)]\n", s));
#endif

  if (tracemode)
  {                             /* Question after the variables expansion
                                   and make sure _all_ executed commands will honor the
                                   trace mode -- ska */
    printf("%s [Enter=Yes, ESC=No] ", s);
    /* If the user hits ^Break, it has the same effect as
       usually: If he is in a batch file, he is asked if
       to abort all the batchfiles or just the current one */
    if (!strchr("Y\r\n", vcgetcstr("\x1bYN\r\n")))
      /* Pressed either "No" or ^Break */
      return;
  }

  num = get_redirection(s, &in, &out, &of_attrib);
  if (num < 0)                  /* error */
    goto abort;

  /* Set up the initial conditions ... */

  if (in || (num > 1))          /* Need to preserve stdin */
    oldinfd = dup(0);

  if (in)                       /* redirect input from this file name */
  {
    close(0);
    if (0 != devopen(in, O_TEXT | O_RDONLY, S_IREAD))
    {
      display_string(TEXT_ERROR_REDIRECT_FROM_FILE, in);
      goto abort;
    }
  }

  if (out || (num > 1))         /* Need to preserve stdout */
    oldoutfd = dup(1);

  /* Now do all but the last pipe command */
  while (num-- > 1)
  {
    close(1);                   /* Close current output file */
    if ((fname0 = tmpfn()) == NULL)
      goto abort;
    open(fname0, O_CREAT | O_TRUNC | O_TEXT | O_WRONLY, S_IREAD | S_IWRITE);

    /* JPP 19981130 */
    nextcmd = s + strlen(s) + 1;
    docommand(s);

    close(0);
    killtmpfn(fname1);          /* fname1 can by NULL */
    fname1 = fname0;
    fname0 = NULL;
    open(fname1, O_TEXT | O_RDONLY, S_IREAD);

    /* JPP 19981130 */
    s = nextcmd;
  }

  /* Now set up the end conditions... */

  if (out)                      /* Final output to here */
  {
    close(1);
    if (1 != devopen(out, of_attrib, S_IREAD | S_IWRITE))
    {
      display_string(TEXT_ERROR_REDIRECT_TO_FILE, out);
      goto abort;
    }

    if (of_attrib & O_APPEND)
      lseek(1, 0, SEEK_END);

  }
  else if (oldoutfd != -1)      /* Restore original stdout */
  {
    close(1);
    dup2(oldoutfd, 1);
    close(oldoutfd);
    oldoutfd = -1;
  }

  docommand(s);                 /* process final command */

abort:
  if (oldinfd != -1)            /* Restore original STDIN */
  {
    close(0);
    dup2(oldinfd, 0);
    close(oldinfd);
  }

  if (oldoutfd != -1)           /* Restore original STDOUT */
  {
    close(1);
    dup2(oldoutfd, 1);
    close(oldoutfd);
  }

  killtmpfn(fname1);
  killtmpfn(fname0);

  if (out)
    free(out);

  if (in)
    free(in);
}

/*
 * do the prompt/input/process loop
 *
 */

static int process_input(int xflag, char *commandline)
{
/* If xflag == 1, then finish processing any batch file, then exit.
   This is used for the /c flag.
 */

  char *readline;
  char *evar;
  char *tp;
  char *ip;
  char *cp;

  /* JPP 19980807 - changed name so not to conflict with echo global */
  int echothisline = 0;

  do
  {
    if ((readline = malloc(MAX_INTERNAL_COMMAND_SIZE + 1)) == NULL)
    {
      error_out_of_memory();
      return 1;
    }

    if (*commandline == '\0')
    {
      if (NULL == (ip = readbatchline(&echothisline, readline,
                                      MAX_INTERNAL_COMMAND_SIZE)))	/* if no batch input then... */
      {
        if (xflag)
        {
          free(readline);
          break;                // out of the do loop
        }

        readcommand(readline, MAX_INTERNAL_COMMAND_SIZE);
        tracemode = 0;          //reset trace mode

        ip = readline;
        echothisline = 0;
      }
    }
    else
    {
      strcpy(readline, commandline);
    }

    cp = commandline;
    while (*ip)
    {
      if (*ip == '%')
      {
        switch (*++ip)
        {
          case '%':
            *cp++ = *ip++;
            break;

          case '0':
          case '1':
          case '2':
          case '3':
          case '4':
          case '5':
          case '6':
          case '7':
          case '8':
          case '9':
            if (NULL != (tp = find_arg(*ip - '0')))
            {
              cp = stpcpy(cp, tp);
              ip++;
            }
            else
              *cp++ = '%';

            break;

          case '?':
            cp += sprintf(cp, "%u", errorlevel);
            ip++;
            break;

          default:
            if ((tp = strchr(ip, '%')) != NULL)
            {
              *tp = '\0';

              if ((evar = getEnv(ip)) != NULL)
                cp = stpcpy(cp, evar);

              ip = tp + 1;
            }
            break;
        }
        continue;
      }

      if (iscntrl(*ip))
        *ip = ' ';

      *cp++ = *ip++;
    }

    free(readline);

    *cp = '\0';

    while ((--cp >= commandline) && isspace(*cp))	/* strip trailing spaces */
      ;

    *(cp + 1) = '\0';

    /* JPP 19980807 */
    if (echothisline)           /* Echo batch file line */
    {
      printprompt();
      puts(commandline);
    }

    if (*commandline)
    {
      parsecommandline(commandline);
      if (echothisline || echo)
        putchar('\n');          /* JPP 19980807 */
      *commandline = '\0';
    }
  }
  while (!canexit || !exitflag);

  return 0;
}

int main(int argc, char *argv[])
{
  char *commandline;
  int xflag;

  /*
   * * main function
   */

  if ((commandline = malloc(MAX_INTERNAL_COMMAND_SIZE + 1)) == NULL)
  {
    error_out_of_memory();
    return 1;
  }

  *commandline = '\0';
  /* check switches on command-line */

  /*JPP 19981201 if /c switch, then execute command and get out.
     initialize returns 1 if /c, else returns 0 */
  xflag = initialize(argc, argv, commandline);
  process_input(xflag, commandline);
  free(commandline);
  return 0;
}
