/*
 *  INTERNAL.C - command.com internal commands.
 *
 *  Comments:
 *
 *  17/08/94 (Tim Norman) ---------------------------------------------------
 *    started.
 *
 *  08/08/95 (Matt Rains) ---------------------------------------------------
 *    i have cleaned up the source code. changes now bring this source into
 *    guidelines for recommended programming practice.
 *
 *  cd()
 *    started.
 *
 *  dir()
 *    i have added support for file attributes to the DIR() function. the
 *    routine adds "d" (directory) and "r" (read only) output. files with the
 *    system attribute have the filename converted to lowercase. files with
 *    the hidden attribute are not displayed.
 *
 *    i have added support for directorys. now if the directory attribute is
 *    detected the file size if replaced with the string "<dir>".
 *
 *  ver()
 *    started.
 *
 *  md()
 *    started.
 *
 *  rd()
 *    started.
 *
 *  del()
 *    started.
 *
 *  does not support wildcard selection.
 *
 *  todo: add delete directory support.
 *        add recursive directory delete support.
 *
 *  ren()
 *    started.
 *
 *  does not support wildcard selection.
 *
 *    todo: add rename directory support.
 *
 *  a general structure has been used for the cd, rd and md commands. this
 *  will be better in the long run. it is too hard to maintain such diverse
 *  functions when you are involved in a group project like this.
 *
 *  12/14/95 (Tim Norman) -----------------------------------------------------
 *    fixed DIR so that it will stick \*.* if a directory is specified and
 *    that it will stick on .* if a file with no extension is specified or
 *    *.* if it ends in a \
 *
 *  1/6/96 (Tim Norman) -----------------------------------------------------
 *    added an isatty call to DIR so it won't prompt for keypresses unless
 *    stdin and stdout are the console.
 *
 *    changed parameters to be mutually consistent to make calling the
 *    functions easier
 *
 *  rem()
 *    started.
 *
 *  doskey()
 *    started.
 *
 *  1/22/96 (Oliver Mueller) -------------------------------------------------
 *    error messages are now handled by perror.
 *
 *  02/05/96 (Tim Norman) ----------------------------------------------------
 *    converted all functions to accept first/rest parameters
 *
 *  07/26/96 (Tim Norman) ----------------------------------------------------
 *     changed return values to int instead of void
 *
 *  path()
 *     started.
 *
 *  12/23/96 (Aaron Kaufman) -------------------------------------------------
 *     rewrote dir() to mimic MS-DOS's dir
 *
 *  01/28/97 (Tim Norman) ----------------------------------------------------
 *     cleaned up Aaron's DIR code
 *
 *  06/13/97 (Tim Norman) ----------------------------------------------------
 *     moved DIR code to dir.c
 *     re-implemented Aaron's DIR code
 *
 *  06/14/97 (Steffan Kaiser) ------------------------------------------------
 *     ctrl-break handling
 *     bug fixes
 *
 *  06/14/97 (Tim Norman) ----------------------------------------------------
 *     changed static var in set() to a malloc'd space to pass to putenv.
 *     need to find a better way to do this, since it seems it is wasting
 *     memory when variables are redefined.
 *
 *  06/29/98 (Rob Lake) ------------------------------------------------------
 *      rewrote del to support wildcards
 *      added my name to the contributors
 *
 *  06/30/98 (Rob Lake) ------------------------------------------------------
 *      rewrote ver command to accept switches, now ver alone prints
 *      copyright notice only.
 *  07/08/1998 (John P. Price)
 *  - removed call to show_environment in set command.
 *  - moved test for syntax before allocating memory in set command.
 *  - misc clean up and optimization.
 */

#include <stdlib.h>
#include <dos.h>
#include <direct.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <dir.h>
#include <conio.h>
#include <io.h>
#include <sys/stat.h>
#include "command.h"

#define SHELLINFO    "FreeDOS Command Line Interface"
#define SHELLVER     "version 0.74b"
#define BADCMDLINE   "bad or incorrect command line"
#define USAGE        "usage"
#define CD           "change to directory   CD [d:][path]"
#define MD           "make directory   MD [d:]path"
#define RD           "remove directory   RD [d:]path"
#define DIR          "display directory listing   DIR [d:][path][filespec]"
#define VER          "display shell version info   VER [/C/R/W/?]"
#define DEL          "delete file   DEL [d:][path]filespec"
#define REN          "rename file   REN [d:][path]filespec1 [d:][path]filespec2"
#define SET          "SET"
#define PROMPTEQUAL  "PROMPT="
#define PATHEQUAL    "PATH="

enum
{
  FALSE, TRUE
};

/*
 * get a character out-of-band and honor Ctrl-Break characters
 */
int
cgetchar(void)
{
  int c;

  if ((c = getch()) == 0)
    c = getch() << 8;

  if (c == 3)
    ctrlBreak = 1;

  return c;
}

/*
 * Check if Ctrl-Break was pressed during the last calls
 */
int
chkCBreak(int mode)
{
  static int leaveAll = 0;      /* leave all batch files */
  int c;

  switch (mode)
  {
    case BREAK_OUTOFBATCH:
      leaveAll = 0;
      return 0;
    case BREAK_BATCHFILE:
      if (leaveAll)
        return 1;
      if (!ctrlBreak)
        return 0;

      /* we need to be sure the string arrives on the screen! */
      do
        cputs("\r\nCtrl-Break pressed.  Cancel batch file? (Yes/No/All) ");
      while (!strchr("YNA\3", c = toupper(cgetchar())) || !c);

      cputs("\r\n");

      if (c == 'N')
        return ctrlBreak = 0;   /* ignore */

      leaveAll = c == 'A' || c == '\3';  /* leave all batch files */

      break;
    case BREAK_INPUT:
      if (!ctrlBreak)
        return 0;
      break;
  }

  ctrlBreak = 0;                /* state processed */
  return 1;
}

/*
 * function to destructively split a string into an array of strings
 *
 * Changed return value to int to determine number of args
 *   -- Rob Lake (06/29/98)
 *
 */
int
split(char *s, char **p)
{
  int sc = 0, pc = 0;           /* string and parameter counters */

  while (s[sc])
  {
    if (sc && isspace(s[sc]) && !isspace(s[sc - 1]))
      s[sc] = 0;

    else if (!isspace(s[sc]) && (sc == 0 || isspace(s[sc - 1]) ||
                                 s[sc - 1] == 0))
      p[pc++] = &s[sc];

    sc++;
  }

  p[pc] = NULL;
  return pc;
}

/*
 * set environment variables
 *
 *
 */
#pragma argsused
int
set(char *first, char *rest)
{
	unsigned char count;          /* counter */
	char *env_temp;               /* temporary copy for putenv */

	/* if no parameters, show the environment */
	if (rest[0] == 0)
	{
		/* JPP 07/08/1998 removed call to show_environment */
		for (count = 0; environ[count]; count++)
		{
	    puts(environ[count]);
		}
		return 0;
	}

	/* make sure there is an = in the command */
	/* JPP 07/08/1998 moved test for syntax before allocating memory */
	if (strchr(rest, '=') == NULL)
	{
		puts("Syntax error");
		return 1;
	}

	if ((env_temp = strdup(rest)) == NULL)
	{
		puts("Memory error");
		return 1;
	}

	/* capitalize name of env. var. */
	for (count = 0; env_temp[count] && env_temp[count] != '='; count++)
	{
		env_temp[count] = toupper(env_temp[count]);
	}

	if (putenv(env_temp) < 0)
	{
		puts("Environment error");
	}

	return 0;
}

/*
 * internal function to get the first argument from the parameter list
 * and return a pointer to the beginning of the next argument, or NULL if
 * there are no more arguments
 *
 */
char *
parse_firstarg(char *s)
{
	char *place;

	/* skip over first argument */
	place = s;
	while (*place && !isspace(*place))
		place++;

	if (*place)
	{
		/* mark the end of the first parameter */
		*place++ = 0;

		/* skip over whitespace before next argument */
		while (isspace(*place))
			place++;

		/* if there is something here, return a pointer to it, else NULL */
		if (*place)
			return place;
		else
			return NULL;
	}
	else
		return NULL;
}

/*
 *  generic function to handle cd, md, and rd (and their verbose names)
 *
 *
 */
int
directory_handler(char *first, char *rest,
									int (*func) (const char *),
									char *func_name, char *usage)
{
	char *dir;                    /* pointer to the directory to change to          */
	char *place;                  /* used to search for the \ when no space is used */

	/* check if there is no space between the command and the path */
	if (rest[0] == 0)
	{
		/* search for the \ or . so that both short & long names will work */
		for (place = first; *place; place++)
			if (*place == '.' || *place == '\\')
				break;

		if (*place)
			dir = place;
		else
			dir = NULL;               /* signal that there are no parameters */
	}
	else
	{
		/* if there is more than 1 parameter */
		if (parse_firstarg(rest) != NULL)
		{
			puts(BADCMDLINE);    /* JPP 07/08/1998 clean up */
			printf(USAGE ": %s\n", usage); /*JPP 07/08/1998 clean up */
			return 1;
		}
    else
			dir = rest;
  }

	/* if doing a CD and no parameters given, print out current directory */
	if (func == chdir && (!dir || !dir[0]))
  {
		char direc[128];
    char temp[128];

		direc[0] = getdisk() + 'A';
		direc[1] = ':';
		getcurdir(0, temp);

    if (temp[0] == '\\')
    {
			strcpy(&direc[2], temp);
    }
    else
    {
			direc[2] = '\\';
			strcpy(&direc[3], temp);
    }

		printf("%s\n", direc);

		return 0;
  }

	/* take off trailing \ if any, but ONLY if dir is not the root dir */
	if (strlen(dir) >= 2 && dir[strlen(dir) - 1] == '\\')
		dir[strlen(dir) - 1] = 0;

	if (func(dir) != 0)
  {
    perror(func_name);
    return 1;
  }

  return 0;
}

/*
 * CD / CHDIR - makes a call to directory_handler to do its work
 *
 *
 */
int
cd(char *first, char *rest)
{
	return directory_handler(first, rest, chdir, "cd()", CD);
}

/*
 * MD / MKDIR - makes a call to directory_handler to do its work
 *
 *
 */
int
md(char *first, char *rest)
{
	return directory_handler(first, rest, mkdir, "md()", MD);
}

/*
 * RD / RMDIR - makes a call to directory_handler to do its work
 *
 *
 */
int
rd(char *first, char *rest)
{
	return directory_handler(first, rest, rmdir, "rd()", RD);
}

/*
 *  display shell version info internal command.
 *
 *
 */
#pragma argsused
int
ver(char *first, char *rest)
{
	int i;

	/* JPP 07/08/1998 clean up and shortened info. */

	printf("\n" SHELLINFO " " SHELLVER ", (C) 1994-1998 Tim Norman\n");
	/* Basic copyright notice */
	if (rest[0] == 0)
	{
		printf("\n"
"%s comes with ABSOLUTELY NO WARRANTY; for details\n"
"type: `ver /w'. This is free software, and you are welcome to redistribute\n"
"it under certain conditions; type `ver /r' for details. Type `ver /c' for a\n"
"listing of credits.\n"
					 "\n", SHELLINFO);
		return 0;
	}

	/* MS-DOS ver prints just help if /? is alone or not */
	if (strstr(rest, "/?") != NULL)
	{
		printf("%s: %s\n", USAGE, VER);
		return 0;
	}

	for (i = 0; rest[i]; i++)
	{
		/* skip spaces */
		if (rest[i] == ' ')
			continue;
		if (rest[i] == '/')
		{
			/* is this a lone '/' ? */
			if (rest[i + 1] == 0)
			{
				fprintf(stderr, INVALID_SWITCH, ' ');
				return 1;
			}
			continue;
		}
		if (toupper(rest[i]) == 'W')
		{                           /* Warranty notice */
			/* JPP 07/08/1998 removed extra printf calls */
			puts("\n This program is distributed in the hope that it will be useful,\n"
					 " but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
					 " MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
					 " GNU General Public License for more details.");
		}
		else if (toupper(rest[i]) == 'R')
		{                           /* Redistribution notice */
			/* JPP 07/08/1998 removed extra printf calls */
			puts("\n This program is free software; you can redistribute it and/or modify\n"
					 " it under the terms of the GNU General Public License as published by\n"
					 " the Free Software Foundation; either version 2 of the License, or\n"
					 " (at your option) any later version.");
		}
		else if (toupper(rest[i]) == 'C')
		{                           /* Developer listing */
			/* JPP 07/08/1998 removed extra printf calls; rearranged names */
			puts("\ndeveloped by:\n"
			"    Tim Norman      Matt Rains\n"
			"    Evan Jeffrey    Steffen Kaiser\n"
			"    Svante Frey     Oliver Mueller\n"
			"    Aaron Kaufman   Marc Desrochers\n"
			"    Rob Lake        John P. Price");
		}
		else
		{
			fprintf(stderr, INVALID_SWITCH, toupper(rest[i]));
			return 1;
		}
	}
	return 0;
}

/*
 *
 *  simple file delete internal command.
 *  Rewritten by Rob Lake (06/29/98)
 *
 */

#pragma argsused
int
del(char *first, char *rest)
{
  int done, flags, orig_disk, args, inter;
  struct ffblk f;
	static char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
	char fn[MAXFILE + MAXEXT + 1], orig_dir[MAXPATH];
  char **arg;

  drive[0] = '\0';
	dir[0] = '\0';

  args = split(rest, arg);
  inter = 0;

  if (args > 2)
  {
    fprintf(stderr, TOO_MANY_PARAMETERS, arg[args - 1]);
    return 1;
  }
  else if (args >= 1)
  {
    struct stat stbuf;
    /* check for /? anywhere in command line */
    if (stricmp(arg[2], "/?") == 0 ||
        stricmp(arg[1], "/?") == 0 ||
				stricmp(arg[0], "/?") == 0)
    {
      printf("%s: %s\n", USAGE, DEL);
      return 0;
    }
    /* check for /p as the first or second, MS-DOS
     * only allows for it as the second arg, so do we :) */
    if (stricmp(arg[1], "/p") == 0 ||
        stricmp(arg[0], "/p") == 0)
      inter = 1;
    /* only one argument and that's /P */
    if (args == 1 && inter)
    {
      fprintf(stderr, REQ_PARAM_MISSING);
			return 1;
    }
    /* check whether the file actually exists */
    if (stat(arg[0], &stbuf) == -1)
    {
      fprintf(stderr, FILE_NOT_FOUND);
      return 1;
    }
		/* check if it is a directory */
    if ((stbuf.st_mode & S_IFMT) == S_IFDIR
        && strchr(arg[0], '*') == NULL)
      strcat(arg[0], "\\");

		flags = fnsplit(arg[0], drive, dir, file, ext);
    /* build the filename, wildcards allowed */
    if (flags & FILENAME && flags && EXTENSION)
      sprintf(fn, "%s%s", file, ext);
    else if (flags & FILENAME && (flags && EXTENSION) == 0)
      sprintf(fn, "%s", file);
    else
      sprintf(fn, "*.*");

    /* uppercase the strings, for prettier output */
    strupr(drive);
		strupr(dir);
    strupr(file);
    strupr(ext);

    if (flags & DRIVE)
    {
      orig_disk = getdisk();
			_chdrive(drive[0] - 'A' + 1);
    }

    if (flags & DIRECTORY)
    {
			if (strlen(dir) > 1)
				dir[strlen(dir) - 1] = '\0';
			getcwd(orig_dir, MAXPATH);
			if (chdir(dir))
      {
        fprintf(stderr, PATH_NOT_FOUND);
        return 1;
      }
			if (dir[0] != '\0' && strlen(dir) > 1)
				strcat(dir, "\\");
    }
  }
  else
  {
    /* only command given */
    fprintf(stderr, REQ_PARAM_MISSING);
    return 1;
  }

  /* needed? */
  if (inter && args == 1)
  {
    fprintf(stderr, REQ_PARAM_MISSING);
    return 1;
  }

  /* make sure user is sure if all files are to be
   * deleted */
  if (strcmp(fn, "*.*") == 0 && !inter)
  {
    int yes;
    char c;
    c = 0;
    yes = 0;
		printf("All files in directory will be deleted!\n");
    printf("Are you sure (Y/N)?");
    while (c != 13)
    {
      c = getch();
      while (toupper(c) != 'N' && toupper(c) != 'Y')
				c = getch();
      putchar(toupper(c));
      if (toupper(c) == 'Y')
        yes = 1;
      else
        yes = 0;
      c = getch();
      if (c == 8)
        printf("\b \b");
    }
    printf("\n");
    if (yes == 0)
    {
			chdir(orig_dir);
			return 1;
    }
  }

  done = findfirst(fn, &f, FA_NORMAL | FA_ARCH);
  if (done)
  {
		chdir(orig_dir);
    fprintf(stderr, FILE_NOT_FOUND);
    return 0;
  }

  while (!done)
  {
    int okay = 1;
    if (inter)
    {
      char c;
      okay = -1;
      printf(drive);
			printf(dir);
      printf(f.ff_name);
      while (okay == -1)
      {
        printf(", Delete(Y/N)?");
        c = toupper(getche());
        if (c == 'Y')
          okay = 1;
        else if (c == 'N')
          okay = 0;
        else if (c == 3)
				{                       /* ^C pressed */
          if (flags & DRIVE)
            _chdrive(orig_disk + 1);
					chdir(orig_dir);
          printf("\n");
          return 1;
        }
        else
        {
          printf(drive);
					printf(dir);
          printf(f.ff_name);
        }
      }
			printf("\n");
    }
    if (okay)
    {
#ifdef NODEL
/* define NODEL if you want to debug */
      printf(drive);
			printf(dir);
      printf(f.ff_name);
      printf("\n");
#else
      if (unlink(f.ff_name) != 0)
      {
        perror("del()");
        return 1;
      }
#endif
    }
    done = findnext(&f);
  }

  if (flags & DRIVE)
    _chdrive(orig_disk + 1);
	chdir(orig_dir);
  return 0;
}

/*
 *
 *  simple file rename internal command.
 *
 */
#pragma argsused
int
ren(char *first, char *rest)
{
  char *arg[2];

  /* set the first argument */
  arg[0] = rest;

  /* split off the first argument and get the second argument start */
  arg[1] = parse_firstarg(rest);

  /* check if there are the wrong number of arguments */
	if (!arg[0][0] || !arg[1] || parse_firstarg(arg[1]) != NULL)
  {
    printf("%s\n", BADCMDLINE);
    printf("%s: %s\n", USAGE, REN);
    return 1;
  }
  else if (rename(arg[0], arg[1]) != 0)
  {
    perror("ren()");
    return 1;
  }

  return 0;
}

/*
 *
 * set the exitflag to true
 *
 */
#pragma argsused
int
internal_exit(char *first, char *rest)
{
  exitflag = 1;

  return 0;
}

/*
 *
 * does nothing
 *
 */
#pragma argsused
int
rem(char *first, char *rest)
{
  return 0;
}

/*
 *
 * prints DOSKEY message...  will soon emulate DOSKEY macros
 *
 */
#pragma argsused
int
doskey(char *first, char *rest)
{
  printf("DOSKEY features are already enabled in the shell.\n");

  return 0;
}

/*
 *
 * changes the PROMPT env. var.
 *
 */
#pragma argsused
int
prompt(char *first, char *rest)
{
  char *from, tempcommand[256];

  /* create a fake command to pass to set() */
	strcpy(tempcommand, PROMPTEQUAL);
  if (*rest == '=')
  {
    from = &rest[1];
    while (isspace(*from))
      from++;
    strcat(tempcommand, from);
  }
  else
		strcat(tempcommand, rest);

  return set(SET, tempcommand);
}

/*
 *
 * changes the PATH env. var.
 *
 */
#pragma argsused
int
path(char *first, char *rest)
{
	char *from, tempcommand[256];

  if (!rest || !*rest)
  {
    printf("PATH=%s\n", getenv("PATH"));
    return 0;
  }

  /* create a fake command to pass to set() */
  strcpy(tempcommand, PATHEQUAL);
  if (*rest == '=')
  {
    from = &rest[1];
    while (isspace(*from))
      from++;
    strcat(tempcommand, from);
  }
  else
    strcat(tempcommand, rest);

  return set(SET, tempcommand);
}
