/*
    This file is the source for an implementation of the DOS ATTRIB program.
    Copyright (C) 1998 by Phil Brutsche

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
Abbreviations:
plb - Phil Brutsche (the original author)

Version history:
2-12-1998 - Initial version - 0.1 (plb)
2-13-1998 - Started documentation.  Fixed bug in display_attribs where if
   there were no files found, the program exited regardless of any remaining
   file specifications in the argument list. (plb)
2-18-1998 - Finished documentation.  Changend calls to _dos_setfileattr and
   _dos_getfileattr to use the internal function setfileattr.
   This function was created by me for use in this program; It will be
   submitted to the maintainer of the FreeDOS C Library shortly. Version
   increased to 0.2. (plb)
2-22-1998 - Added MS undocumented function:  `attrib ,` will unset all
   file attributes.  (plb)
2-24-1998 - Added ability to change attributes on directories.  Increased
   to version 0.3.  (plb)
3-08-1998 - Rewrote to be able to use file lists (ie `attrib @filename` or
   `attrib @-`) per a suggestion from Charles Dye <raster@highfiber.com>.
   Moved up to version 0.4.  (plb)
3-16-1998 - Updated these friggin' comments :-).  Added a command-line
   parameter (`/?` in case yer interested.  `-?` works too.) to display
   online help.  Now version 0.5.  (plb)
7-19-1998 - Fixed bugs:
   * If files in another directory are specified, or are on another drive,
     it will set the attributes for the appropriately names file in the
     current directory.
   * Also: if only a drive is specified (ie `attrib d:`) it will assume `*.*`
     was meant.
   Now version 0.60.  (plb)
7-21-1998 - Fixed bugs:
   * In 4DOS attrib and MS-DOS attrib, if you give some attributes but
     no file names it will give all files and directories in the current
     directory those attributes.  (eg `attrib +r` would set all the files
     in the current directory as read-only).  This attrib didn't do that.
     (plb)
8-29-1998 - Changed behavior:
   * As per a suggestion from Charles Dye <raster@highfiber.com>, specifying
     only a directory on the command line will display the attributes for
     that directory.  (plb)
8-30-1998 - Changed behavior:
   * As per a suggestion from Charles Dye <raster@highfiber.com>, it now
     ignores the directory entries `..` and `.`.  (plb)
9-2-1998 - New version released as 0.63.  (plb)
*/

#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define RD_ONLY 0
#define HIDDEN 1
#define SYSTEM 2
#define ARCHIVE 3
#define INDENT "     "
#define SET_ATTRIBS 0
#define DISPLAY_ATTRIBS 1

char attrib_version [] = "0.60";

void set_file_attribs (int [], char *);
void get_attrib_str (char [], char);
void set_attribs (int [], char *);
int setfileattr (char *, int);
void secret_comma_change (void);
void display_attribs (char *);
void invalid_attrib_change (char *);
void process_file_list (int [], char *, int);
int attribs_unchanged (int []);
void display_online_help (void);
void make_name (char *, char *, char *);

char *modifications [] = {
   "+S", "+H",
   "+R", "+A",
   "-S", "-H",
   "-R", "-A",
   NULL };

char *mod_text [] = {
   "Sets the system attribute", "Sets the hidden attribute",
   "Sets the read-only attribute", "Sets the archive attribute",
   "Clears the system attribute", "Clears the hidden attribute",
   "Clears the read-only attribute", "Clears the archive attribute" };

int main (int argc, char *argv [])
{
   int attr [4] = { 0, 0, 0, 0 };
   int i, has_filespec = 0;
   if (argc == 1)
   {
      display_attribs ("*.*");
      return EXIT_SUCCESS;
   }
   for (i = 1; i < argc; i++)
   {
      strupr (argv [i]);
      /* Check for the command line help! */
      if (strcmp (argv [i], "/?") == 0 || strcmp (argv [i], "-?") == 0)
      {
         display_online_help ();
         return 0;
      }
      /* This is an `undocumented feature` from MS's ATTRIB.  It basically
         unsets all the attributes of all the files in a directory */
      if (strcmp (argv [i], ",") == 0)
      {
         secret_comma_change ();
         return 0;
      }
      if (strlen (argv [i]) == 2)
      {
         /* Not so fast, bucko.  It may be an attribute, but it must be
            a VALID attribute to continue */
         if (strchr ("-+", argv [i][0]))
         {
            int change;
            if (argv [i][0] == '-')
               change = -1;
            else if (argv [i][0] == '+')
               change = 1;
            switch (argv [i][1])
            {
               case 'R':
                  attr [RD_ONLY] = change;
                  break;
               case 'S':
                  attr [SYSTEM] = change;
                  break;
               case 'A':
                  attr [ARCHIVE] = change;
                  break;
               case 'H':
                  attr [HIDDEN] = change;
                  break;
               default:
                  invalid_attrib_change (argv [i]);
                  return EXIT_FAILURE;
            }
            argv [i] = NULL;
         }
      }
      else
         has_filespec = 1;
   }
   if (! has_filespec && ! attribs_unchanged (attr))
   {
      set_file_attribs (attr, "*.*");
      return EXIT_SUCCESS;
   }
   for (i = 1; i < argc; i++)
   {
      if (argv [i] != NULL)
      {
         if (attribs_unchanged (attr))
         {
            if (argv [i][0] == '@')
               process_file_list (attr, argv [i], DISPLAY_ATTRIBS);
            else
               display_attribs (argv [i]);
         }
         else if (argv [i][0] == '@')
            process_file_list (attr, argv [i], SET_ATTRIBS);
         else
            set_file_attribs (attr, argv [i]);
      }
   }
   return EXIT_SUCCESS;
}

void process_file_list (int attribs [], char *filelist, int mode)
{
   FILE *infile;
   char *result, data [80];
   if (strcmp (filelist + 1, "-") == 0)
      infile = stdin;
   else
   {
      infile = fopen (filelist + 1, "rt");
      if (infile == NULL)
      {
         fprintf (stderr, "Unable to open file list \"%s\"\n", filelist + 1);
         return;
      }
   }
   do
   {
      result = fgets (data, 80, infile);
      data [strlen (data) - 1] = '\0';
      if (result != NULL)
         if (mode == SET_ATTRIBS)
            set_file_attribs (attribs, data);
         else
            display_attribs (data);
   } while (result != NULL);
   if (infile != stdin)
      fclose (infile);
}

void set_file_attribs (int attribs [], char *search_spec)
{
   struct ffblk file_info_block;
   int attributes = FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_ARCH | FA_DIREC;
   unsigned int result;
   char drive [3];
   char dir [66];
   char name [9];
   char ext [5];
   char searchpath [80];
   strcpy (searchpath, search_spec);
   fnsplit (search_spec, drive, dir, name, ext);
   if (strcmp (name, "") == 0)
      strcat (searchpath, "\*.*");
   result = findfirst (searchpath, &file_info_block, attributes);
   if (result != 0)
      perror (search_spec);
   while (result == 0)  /* There are files to process! */
   {
      char old_attrib_data [] = "----";
      char new_attrib_data [] = "----";
      char filename [80];
      if (strcmp (file_info_block.ff_name, "..") == 0 ||
          strcmp (file_info_block.ff_name, ".") == 0)
      {
         result = findnext (&file_info_block);
         continue;
      }
      get_attrib_str (old_attrib_data, file_info_block.ff_attrib);
      set_attribs (attribs, &file_info_block.ff_attrib);
      get_attrib_str (new_attrib_data, file_info_block.ff_attrib);
      if (file_info_block.ff_attrib & FA_DIREC)
         file_info_block.ff_attrib ^= FA_DIREC;
      strcpy (filename, drive);
      strcat (filename, dir);
      strcat (filename, file_info_block.ff_name);
      result = setfileattr (filename, file_info_block.ff_attrib);
      if (result != 0)
         perror (filename);
      else
         fprintf (stdout, "[%s] -> [%s] %s\n", old_attrib_data, new_attrib_data,
                  filename);
      result = findnext (&file_info_block);
   }
}

void set_attribs (int attrib_list [], char *attrib)
{
   if ((attrib_list [RD_ONLY] > 0) && ((*attrib & FA_RDONLY) == 0))
      *attrib ^= FA_RDONLY;
   if ((attrib_list [RD_ONLY] < 0) && (*attrib & FA_RDONLY))
      *attrib ^= FA_RDONLY;
   if ((attrib_list [ARCHIVE] > 0) && ((*attrib & FA_ARCH) == 0))
      *attrib ^= FA_ARCH;
   if ((attrib_list [ARCHIVE] < 0) && (*attrib & FA_ARCH))
      *attrib ^= FA_ARCH;
   if ((attrib_list [HIDDEN] > 0) && ((*attrib & FA_HIDDEN) == 0))
      *attrib ^= FA_HIDDEN;
   if ((attrib_list [HIDDEN] < 0) && (*attrib & FA_HIDDEN))
      *attrib ^= FA_HIDDEN;
   if ((attrib_list [SYSTEM] > 0) && ((*attrib & FA_SYSTEM) == 0))
      *attrib ^= FA_SYSTEM;
   if ((attrib_list [SYSTEM] < 0) && (*attrib & FA_SYSTEM))
      *attrib ^= FA_SYSTEM;
}

void get_attrib_str (char string [], char attribs)
{
   if (FA_RDONLY & attribs)
      string [0] = 'R';
   if (FA_HIDDEN & attribs)
      string [1] = 'H';
   if (FA_SYSTEM & attribs)
      string [2] = 'S';
   if (FA_ARCH & attribs)
      string [3] = 'A';
}

int setfileattr (char *filename, int attributes)
{
   struct REGPACK regs;
   regs.r_ax = 0x4301;
   regs.r_cx = attributes;
   regs.r_ds = FP_SEG (filename);
   regs.r_dx = FP_OFF (filename);
   intr (0x21, &regs);
   if ((regs.r_flags & 0x0001) == 0)
      return 0;
   else
      return regs.r_ax;
}

void secret_comma_change (void)
{
   int attribs [] = { -1, -1, -1, -1 };
   set_file_attribs (attribs, "*.*");
}

void display_attribs (char *searchspec)
{
   struct ffblk file_info_block;
   int attributes = FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_ARCH | FA_DIREC;
   unsigned int result;
   char drive [3];
   char dir [66];
   char name [9];
   char ext [5];
   char searchpath [80];
   strcpy (searchpath, searchspec);
   fnsplit (searchspec, drive, dir, name, ext);
   if (strcmp (name, "") == 0)
      strcat (searchpath, "\*.*");
   result = findfirst (searchpath, &file_info_block, attributes);
   if (result != 0)
      perror (searchspec);
   while (result == 0)  /* There are files to process! */
   {
      char attrib_data [] = "----";
      char filename [80];
      if (strcmp (file_info_block.ff_name, "..") == 0 ||
          strcmp (file_info_block.ff_name, ".") == 0)
      {
         result = findnext (&file_info_block);
         continue;
      }
      get_attrib_str (attrib_data, file_info_block.ff_attrib);
      make_name (filename, searchspec, file_info_block.ff_name);
      fprintf (stdout, "[%s] %s\n", attrib_data, filename);
      result = findnext (&file_info_block);
   }
}

void invalid_attrib_change (char *invalid_attrib)
{
   int i = 0;
   fprintf (stdout, "Sorry.  \"%s\" is an invalid attribute change.\n",
            invalid_attrib);
   fprintf (stdout, "The valid attributes are:\n");
   while (modifications [i] != NULL)
   {
      fprintf (stdout, "%s%s%s%s\n", INDENT, modifications [i],
                                     INDENT, modifications [i + 1]);
      i += 2;
   }
}

int attribs_unchanged (int attribs [])
{
   return (attribs [0] == 0) && (attribs [1] == 0) &&
          (attribs [2] == 0) && (attribs [3] == 0);
}

void display_online_help (void)
{
   int i = 0;
   fprintf (stdout, "ATTRIB v%s:\n", attrib_version);
   fprintf (stdout, INDENT "Views/Changes the attributes for files and "
                    "directories\n");
   fprintf (stdout, "Command line format:\n");
   fprintf (stdout, INDENT "ATTRIB [attributes] [filename | @filelist] ...\n");
   fprintf (stdout, "\n");
   fprintf (stdout, "Attribute flags:\n");
   while (modifications [i] != NULL)
   {
      fprintf (stdout, "%s%s%s%s\n", INDENT, modifications [i], INDENT, mod_text [i]);
      i += 1;
   }
}

void make_name (char *result, char *searchpath, char *filename)
{
   char drive [3];
   char dir [66];
   char name [9];
   char ext [5];
   fnsplit (searchpath, drive, dir, name, ext);
   strcpy (result, drive);
   strcat (result, dir);
   strcat (result, filename);
}
