\RAY.XPL	16-Jan-2005	Loren Blaney	loren_blaney@idcomm.com
\Simple Ray Tracer. Use arrow keys etc. to adjust image. (See DoKey.)
\
\Understanding vectors is the key to understanding this program.
\
\A right handed, 3-D coordinate system is used with the origin (O) at the
\ upper-left corner of the display screen.
\
\				  Z (into screen)
\			        /
\			      /
\			    /
\			  /
\		      O +------------------- X
\			|
\			|
\			|
\			|		RV
\			|	      /
\			|	    /
\			Y	  /
\			     RP *
\
\The viewer is centered in front of the screen and back some distance, which
\ is negative Z. Objects are behind the screen with positive Z coordinates.
\
\A light ray is represented by a starting position (RPx, RPy, RPz), and a
\ direction vector (RVx, RVy, RVz). Rays are traced backwards, from the eye
\ to the objects. This makes the program much faster by eliminating rays that
\ the viewer never sees. This works because the path that a ray traces is the
\ same whether it's coming or going.
\
\Colors are also three-dimensional vectors (red, green, blue).

include	C:\CXPL\CODESI.XPL;

def	TVW=640, TVH=480; \Screen width, height, depth (see SetVid)
def	Intsize=4;	\Number of bytes in an integer (2 or 4)
def	Size0=16;	\Initial resolution step size (power of 2: [1..32])
def	Big=1E20, Small=0.01;	\Relative to size of our objects
def	CR=$0D, Esc=$1B, Ctrl=$40;	\ASCII characters

int	CpuReg,		\Address of CPU registers, for calls to BIOS routines
	II,		\Scratch for Main
	Objects,	\Number of objects (spheres)
	Redraw,		\Flag: redraw the image (because the view has changed)
	SpSurf,		\Array: sphere surface type
	SpSurf2;	\Array: sphere surface color
real	Eye,		\Array: coordinates of viewer
	Zoom,		\Magnification of screen image
	Refraction,	\Index of refraction
	SpCenter,	\2-D array: center coordinates for each sphere
	SpRadius,	\Array: radius of each sphere
	Sp2Radius,	\Array: radius squared of each sphere
	OffsetX,	\Screen image position (horizontal and vertical)
	OffsetY;

def	\Surf\	Sky, Light, Metal, Glass;	\Types of surface materials

\===============================================================================

func	GetKey;		\Get character from keyboard
int	Ch;		\This is a low-level routine with no echo,
begin			\ no Ctrl-C, and no cursor.
CpuReg(0):= 0;		\Requires: CpuReg:= GetReg;   Function = $00
SoftInt($16);		\Call BIOS routine
Ch:= CpuReg(0) & $FF;
if Ch = 0 then		\Convert scan code to control code
	case CpuReg(0)>>8 of	\Arrows:
	  $48:	Ch:= $0B;	\ up
	  $50:	Ch:= $0A;	\ down
	  $4B:	Ch:= $08;	\ left
	  $4D:	Ch:= $15	\ right
	other	Ch:= -(CpuReg(0)>>8);
return Ch;
end;	\GetKey

\-------------------------------------------------------------------------------

proc	DoKey;		\Handle keyboard commands
int	Ch;
begin
Ch:= GetKey;
Redraw:= true;		\Anticipate that a redraw will be necessary
case Ch of
  ^.:	Zoom:= Zoom * 1.1;			\Zoom in a little
  ^>:	Zoom:= Zoom * 2.0;			\Zoom in to double size
  ^,:	Zoom:= Zoom / 1.1;			\Zoom out a little
  ^<:	Zoom:= Zoom / 2.0;			\Zoom out to half size
  $15:	OffsetX:= OffsetX + float(Size0);	\Shift view to the right
  $08:	OffsetX:= OffsetX - float(Size0);	\Shift view to the left
  $0A:	OffsetY:= OffsetY + float(Size0);	\Shift view downward
  $0B:	OffsetY:= OffsetY - float(Size0);	\Shift view upward
  ^I:	Refraction:= Refraction * 1.1;		\Adjust index of refraction
  ^i:	Refraction:= Refraction / 1.1;
  $20:	[];					\Redraw
  ^C-Ctrl, Esc:  [SetVid($03);   exit];		\Restore video mode and exit
  ^S-Ctrl: [Ch:= GetKey;   Redraw:=false]	\Stop/start program
other	Redraw:= false;				\Illegal command
end;	\DoKey

\===============================================================================

proc	DrawPixel(X, Y, Color);	\Draw a pixel
int	X, Y,		\Screen coordinates
	Color;		\Vector: color of pixel (R,G,B)
int	R, G, B;
begin
\Convert color vector to the 6-bit range that the VGA hardware can handle
R:= Color(0)>>2;   G:= Color(1)>>2;   B:= Color(2)>>2;
if R>$FF then R:= $FF;   if G>$FF then G:= $FF;   if B>$FF then B:= $FF;
Point(X, Y, R<<16+G<<8+B);
end;	\DrawPixel

\-------------------------------------------------------------------------------

proc	Reflect(I, N, R);		\Reflect a ray
real	I, N, R;	\Unit vectors: Incident, Normal, Reflected
real	D;
\		                 N
\		                 |        R
\\		          \      |      /
\\		            \    |    /
\\		              \ I|  /
\\		                \|/
\		        ---------*----------
begin
D:= I(0)*N(0) + I(1)*N(1) + I(2)*N(2);		\R = I - 2*(I.N)*N
D:= D + D;
R(0):= I(0) - N(0)*D;
R(1):= I(1) - N(1)*D;
R(2):= I(2) - N(2)*D;
end;	\Reflect

\-------------------------------------------------------------------------------

proc	Refract(I, N, R, N1, N2);	\Refract a ray
real	I, N, R,	\Unit vectors: Incident, Normal, Refracted
	N1, N2;		\Indexes of refraction (air = 1.0)
real	C, D, E;
\		                N
\		                |
\\		         \      |
\\		           \    |
\\		             \ I|
\\		               \|       N1
\		       ---------*----------
\\		                 \      N2
\\		                  \
\\		                   \
\\		                    \
\		                     R
begin
E:= N2 / N1;
C:= -(I(0)*N(0) + I(1)*N(1) + I(2)*N(2));	\C = -I.N    (dot product)
D:= C*C + E*E - 1.0;		\R:= ((C - Sqrt(C*C + E*E - 1.0))*N + I) / E
if D < 0.0 then			\Total internal reflection; Reflect(I, N, R)
	begin					\R = I - 2*(I.N)*N
	C:= C + C;				\R = I + 2*C*N
	R(0):= I(0) + N(0)*C;
	R(1):= I(1) + N(1)*C;
	R(2):= I(2) + N(2)*C;
	return;
	end;

D:= C - sqrt(D);
R(0):= (D*N(0) + I(0))/E;			\R:= (D*N + I) / E
R(1):= (D*N(1) + I(1))/E;
R(2):= (D*N(2) + I(2))/E;
end;	\Refract

\===============================================================================

func real HitSphere(RP, RV, S, R2);	\Intersection of a ray and a sphere.
\Returns the distance from RP to the point of intersection. The coordinates
\ of this point of intersection can be obtained by: P = RP + RV*DIST.
\Returns 0.0 if there is no intersection.
real	RP,	\Array: Ray point (X,Y,Z)
	RV,	\Array: Ray vector (U,V,W)
	S,	\Array: Sphere center (X,Y,Z)
	R2;	\Radius of sphere squared
real	G0, G1, G2, B, D,	\Intermediate variables
	Dist, Dist1, Dist2;	\Distances to points of intersection
begin
G0:= RP(0) - S(0);			\G:= RP - S;
G1:= RP(1) - S(1);
G2:= RP(2) - S(2);

B:= RV(0)*G0 + RV(1)*G1 + RV(2)*G2;	\B:= RV.G   (dot product)
D:= B*B - (G0*G0 + G1*G1 + G2*G2 - R2);
if D <= 0.0 then return 0.0;		\Ray does not intersect sphere

D:= sqrt(D);
Dist1:= +D - B;
Dist2:= -D - B;

Dist:= if Dist1 < Dist2 then Dist1 else Dist2;	\Get the closer point
if Dist > Small then return Dist;		\The usual case
if Dist1 > Small then return Dist1;	\The ray starts inside the sphere
if Dist2 > Small then return Dist2;
return 0.0;				\The sphere is behind the ray
end;	\HitSphere

\-------------------------------------------------------------------------------

func real HitPlane(RP, RV, Y0);	\Intersection of a ray and the X-Z plane.
\Returns the distance from RP to the point of intersection. The coordinates
\ of this point of intersection can be obtained by: P = RP + RV*Dist.
\Returns 0.0 if there is no intersection.
real	RP,	\Array: Ray point (X,Y,Z)
	RV,	\Array: Ray vector (U,V,W)
	Y0;	\Y position of X-Z plane
real	Dist;	\Distance along RV from RP to point of intersection
begin
\Dist = (Y0 - RPy) / RVy
if abs(RV(1)) < Small then return 0.0; \Avoid divide by 0.0 error
Dist:= (Y0 - RP(1)) / RV(1);

if Dist > 0.0 then return Dist;		\The usual case
return 0.0;				\The plane is behind the ray
end;	\HitPlane

\===============================================================================

proc	TraceRays;	\Determine the correct color for each pixel and draw it
real	RayV,		\Array: Ray vector from eye to point on screen
	RaV,		\Array: Ray unit vector from eye to point on screen
	Len, Len1, Len2, \Length of ray vector (RayV)
	DRX;		\Delta Ray vector X
int	X, Y,		\Pixel coordinates on the screen
	Size,		\Step size (resolution control)
	Color,		\Vector: Pixel color (R,G,B)
	Level,		\Level of recursion
	XF, YF,		\Flags used for resolution control
	FirstTime;	\Flag: first time through loop

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

proc	Trace(RP, RV);	\Trace ray from point RP in direction RV, and set
\ Color to the color of the closest object intersected by this ray.
real	RP, RV;		\Ray point (X,Y,Z), Ray vector (U,V,W)
real	PX, PZ,
	RP2, RV2,	\Ray point, Ray vector for reflected ray (if any)
	RP3, RV3,	\Ray point, Ray vector for refracted ray (if any)
	Norm,		\Vector normal (perpendicular) to sphere at point
			\ of intersection
	Dist, DistMin,	\Distance to point of intersection, Closest distance
	SpC, SpR,	\Array: closest sphere center, Closest sphere radius
	D;
int	IPX, IPZ,
	Obj, ObjMin,	\Object, Closest object
	Surf,		\Surface type
	SColor;		\Save reflected color (added to refracted color)
def	CheckSize= 1.0/18.75;	\Reciprocal of size of checker squares
begin
if Level >= 66 then return;		\Limit the recursion depth
Level:= Level + 1;			\Next level of recursion down

RP2:= RlRes(3);

Surf:= Sky;
DistMin:= Big;				\Infinitely far away (more or less)
for Obj:= 0, Objects-1 do		\See if ray intersects an object
	begin
	Dist:= HitSphere(RP, RV, SpCenter(Obj), Sp2Radius(Obj));
	if Dist > Small then \Prevent infinite reflections due to rounding errors
	    if Dist < DistMin then	\Ray intersects -- save closest point
		begin
		DistMin:= Dist;
		Surf:= SpSurf(Obj);	\Save surface type of closest point
		ObjMin:= Obj;
		end;
	end;

if Surf = Light then
	begin
	Color(0):= SpSurf2(ObjMin, 0);
	Color(1):= SpSurf2(ObjMin, 1);
	Color(2):= SpSurf2(ObjMin, 2);
	end;

if DistMin = Big then			\Ray didn't hit sphere
	begin				\See if it hits the bottom plane
	Dist:= HitPlane(RP, RV, 340.0);	\Y distance to bottom plane
	if Dist > Small then \Prevent infinite reflections due to rounding errors
	    if Dist < DistMin then	\Ray intersects -- save closest point
		begin
		\Make checker-board pattern for X-Z plane
		\Px = RPx + RVx*Dist		Find point of intersection
		PX:= RP(0) + RV(0)*Dist;
		PZ:= RP(2) + RV(2)*Dist;
		IPX:= FIX(PX*CheckSize - 0.51);	\(the extra .01 avoids fuzz
		IPZ:= FIX(PZ*CheckSize - 0.51);	\ on the mortar lines)

		\RP2 = RP + RV*Dist		Point where ray intersects plane
		RP2(0):= PX;
		RP2(1):= RP(1) + RV(1)*Dist;
		RP2(2):= PZ;

		RV(1):= -RV(1);			\Reflect
		Trace(RP2, RV);			\Trace reflected ray

		Color(0):= Color(0)>>2;		\Tiles reflect attenuated color
		Color(1):= Color(1)>>2;
		Color(2):= Color(2)>>2;
		if (IPX&$0F)=$0F ! (IPZ&$0F)=$0F then
			begin			\Add white for mortar
			Color(0):= Color(0) + $100;
			Color(1):= Color(1) + $100;
			Color(2):= Color(2) + $C0;
			end;
		return;
		end;
	end;

if Surf = Sky then
	begin				\Sky color
	Color(0):= $200;   Color(1):= $200;   Color(2):= $200;
	end

else if Surf = Glass then		\Reflect and refract rays and trace them
	begin
	RV2:= RlRes(3);   Norm:= RlRes(3);
	RP3:= RlRes(3);   RV3:= RlRes(3);
	SColor:= Reserve(3*IntSize);

	\Point where ray intersects sphere
	\RP2 = RP + RV*DistMin
	RP2(0):= RP(0) + RV(0)*DistMin;
	RP2(1):= RP(1) + RV(1)*DistMin;
	RP2(2):= RP(2) + RV(2)*DistMin;

	SpC:= SpCenter(ObjMin);
	SpR:= SpRadius(ObjMin);

	\Normal unit vector of sphere at point of intersection
	\Norm:= (RP2 - C) / R
	Norm(0):= (RP2(0) - SpC(0)) / SpR;
	Norm(1):= (RP2(1) - SpC(1)) / SpR;
	Norm(2):= (RP2(2) - SpC(2)) / SpR;

	D:= RV(0)*Norm(0) + RV(1)*Norm(1) + RV(2)*Norm(2);    \R = I - 2*(I.N)*N
	D:= D + D;
	RV2(0):= RV(0) - Norm(0)*D;
	RV2(1):= RV(1) - Norm(1)*D;
	RV2(2):= RV(2) - Norm(2)*D;

	Trace(RP2, RV2);		\Trace reflected ray

	SColor(0):= Color(0) >> 2;	\Attenuate reflected color and save it
	SColor(1):= Color(1) >> 2;
	SColor(2):= Color(2) >> 2;

	\Refract ray going into glass sphere (incident, normal, refracted)
	Refract(RV, Norm, RV2, 1.0, Refraction);

	Dist:= HitSphere(RP2, RV2, SpC, SpR*SpR);

	\RP3 = RP2 + RV2*Dist = Point where ray intersects sphere from inside
	RP3(0):= RP2(0) + RV2(0)*Dist;
	RP3(1):= RP2(1) + RV2(1)*Dist;
	RP3(2):= RP2(2) + RV2(2)*Dist;

	\Norm:= -(RP3 - C) / R		Normal to sphere @ point of intersection
	Norm(0):= (SpC(0) - RP3(0)) / SpR;
	Norm(1):= (SpC(1) - RP3(1)) / SpR;
	Norm(2):= (SpC(2) - RP3(2)) / SpR;

	\Refract ray going out of glass sphere (incident, normal, refracted)
	Refract(RV2, Norm, RV3, Refraction, 1.0);
	Trace(RP3, RV3);			\Trace refracted ray

	\Combine doubly refracted color, reflected color, and color of sphere
	Color(0):= Color(0) + SColor(0) - SpSurf2(ObjMin,  0);
	Color(1):= Color(1) + SColor(1) - SpSurf2(ObjMin,  1);
	Color(2):= Color(2) + SColor(2) - SpSurf2(ObjMin,  2);
	if Color(0) < 0 then Color(0):= 0;
	if Color(1) < 0 then Color(1):= 0;
	if Color(2) < 0 then Color(2):= 0;
	end;
end;	\Trace

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

begin	\TraceRays
RayV:= RlRes(3);
RaV:= RlRes(3);
Color:= Reserve(3*IntSize);

FirstTime:= true;
Size:= Size0;
RayV(2):= \0.0\ - Eye(2);
Len2:= RayV(2) * RayV(2);

loop	begin
	YF:= false;
	Y:= 0;
	while Y < TVH do
		begin
		if ChkKey then DoKey;			\Check for key commands
		if Redraw then return;

		RayV(1):= (float(Y) - Eye(1) + OffsetY) / Zoom;
		Len1:= RayV(1)*RayV(1) + Len2;

		RayV(0):= (OffsetX - Eye(0)) / Zoom;
		DRX:= float(Size) / Zoom;

		XF:= false;
		X:= 0;
		while X < TVW do
			begin
			if XF ! YF ! FirstTime then	\(Resolution control)
				begin	\Project ray from viewer to pixel X,Y
				\Make it a unit vector (U,V,W)
				Len:= sqrt(RayV(0)*RayV(0) + Len1);
				RaV(0):= RayV(0) / Len;
				RaV(1):= RayV(1) / Len;
				RaV(2):= RayV(2) / Len;

				Level:= 0;
				Trace(Eye, RaV);	\Trace ray and get color

				DrawPixel(X, Y, Color);
				end;
			X:= X + Size;			\Skip Size many pixels
			XF:= ~XF;
			RayV(0):= RayV(0) + DRX;
			end;
		Y:= Y + Size;				\Skip Size many pixels
		YF:= ~YF;
		end;
	Size:= Size >> 1;				\Chop size in half
	if Size = 0 then quit;
	FirstTime:= false;
	end;
end;	\TraceRays

\===============================================================================

begin	\Main
CpuReg:= GetReg;	\Get address of CPU registers for BIOS calls
SetVid($112);		\Set VGA mode (also see def TVW, TVH)

\Make some spheres:
Objects:= 5;						\Number of spheres
SpSurf:= [Glass, Glass, Glass, Glass, Light];		\Type of each sphere
SpSurf2:= [[$300,  $40,  $40],
	   [$200,  $40, $3F0],				\Color of each sphere
	   [$200, $3F0,  $40],				\ (scaled to 10 bits)
	   [ $40, $3F0, $200],				\ (R,G,B)
	   [$FF0, $FF0, $FF0]];				\ (extra brite)
SpRadius:= [100.0, 100.0, 100.0, 100.0, 1000.0];	\Radii
SpCenter:=[[000.0, 240.0, 300.0],			\Centers (X,Y,Z)
	   [300.0, 240.0, 600.0],
	   [600.0, 240.0, 900.0],
	   [000.0, 240.0, 900.0],
	   [-6000.0, -4000.0, 8000.0]];
Eye:= [-250.0, -100.0, -500.0];			\Coordinates of viewer (X,Y,Z)

OffsetX:= -232.0;   OffsetY:= -16.0;   Zoom:= 1.65;
Refraction:= 1.65;			\Dense flint glass (looks good)

Sp2Radius:= RlRes(Objects);		\Set up squares of radii (for speed)
for II:= 0, Objects-1 do
	Sp2Radius(II):= SpRadius(II) * SpRadius(II);

loop	begin
	loop	begin
		Redraw:= false;		\This flag is set to true by DoKey
		Clear;
		TraceRays;
		if ~Redraw then quit;
		end;
	repeat DoKey until Redraw;	\Wait for key press to exit or redraw
	end;
end;	\Main
