\SEE.XPL	21-Jul-2011	Ver 1.4		loren.blaney@gmail.com
\Graphic File Viewer for BMP, LBM, IFF, PCX, and GIF Image Files
\This uses a file navigator similar to Window's "Open" dialog box.
\Compile with XPLPX (xpx.bat)

\REVISIONS:
\V1.0: 28-Feb-2005, Released.
\V1.1: 04-Mar-2005, Protect against bad data in a GIF (missing EOF) that caused
\ the program to blow up due to an array bounds error.
\V1.2: 20-Jul-2006, Fixed small bug that allowed an extra pixel to be displayed
\ in the lower left corner of GIF images. Also used new version of XPLPX that
\ fixed a bug where some bad pixels could be displayed for 24-bit images.
\V1.3: 12-Mar-2011, Added capability to delete files.
\V1.4: 21-Jul-2011, Greatly speed up GIF display, check array bounds, fix memory
\ leak.
\
\This program is free software; you can redistribute it and/or modify it under
\ the terms of the GNU General Public License version 2 as published by the
\ Free Software Foundation.
\This program is distributed in the hope that it will be useful, but WITHOUT
\ ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
\ FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
\ details.
\
include	C:\CXPL\CODES;	\intrinsic code definitions
string 0;		\all strings are terminated with 0 here (ASCIIZ)

int	CpuReg,		\address of CPU register array (from GetReg)
	DataSeg,	\segment address where the .exe loader put our data
	FileHandle,	\image file handle (DOS needs this to read files)
	OrigPath,	\segment address of string containing original directory
	Selection,	\user's file selection (index into FileName array, etc.)
	SelectionViewed,\image selection last viewed (displays status info)
	WinStart;	\first file name displayed in window
int	Width, Height,	\image dimensions in pixels
	Depth,		\number of bits of color (4 bits gives 16 colors, etc.)
	X0, Y0;		\offset to center image on screen (pixels)

def	WidthMax=1280,	\maximum dimensions of image that can be displayed
	HeightMax=1024;	\ (pixels)
def	FilesMax=2000;	\maximum number of file names allowed
char	FileName(FilesMax, 14); \(8+1+3+1=13 characters long maximum)
def	ID = 13;	\the 14th byte is a file type identifier: 0=dir, 1=BMP,
			\ 2=LBM, 3=BBM, 4=IFF, 5=PCX, 6=GIF, $FF=deleted
int	FileDate(FilesMax), \date of file (compressed format)
	FileTime(FilesMax), \time of file (compressed format)
	FileSize(FilesMax); \size of file (bytes)

int	CLUT(256);	\Color Look-Up Table (R<<16 + G<<8 + B)

def	Black, Blue, Green, Cyan, Red, Magenta, Brown, White,  \attribute colors
	Gray, LBlue, LGreen, LCyan, LRed, LMagenta, Yellow, BWhite; \EGA palette

def	FileColor = Blue<<4+Yellow,	\attribute for file name colors
	CurFileColor = Brown<<4+Yellow,	\attribute for file name colors @ cursor
	DirColor = Blue<<4+LMagenta,	\attribute for directory name colors
	CurDirColor = Brown<<4+LMagenta,\attribute for dir name colors at cursor
	BorderColor = White<<4+Black,	\attribute for borders
	TitleColor = White<<4+Red,	\attribute for program's title & version
	InfoColor = Cyan<<4+BWhite;	\attribute for file information

def	Nul=$00, Bel=$07, BS=$08, Tab=$09, LF=$0A, FF=$0C,	\control chars
	CR=$0D, EOF=$1A, Esc=$1B, Ctrl=$40;

def	UpArrow=$48, DnArrow=$50, LtArrow=$4B, RtArrow=$4D,	\scan codes
	PageUp=$49, PageDn=$51, Home=$47, End=$4F, Insert=$52, Delete=$53,
	Func1=$3B;



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(unused arguments need not be passed)
begin
CpuReg(0):= AX;
CpuReg(1):= BX;
CpuReg(2):= CX;
CpuReg(3):= DX;
CpuReg(6):= BP;
CpuReg(9):= DS;
CpuReg(11):= ES;
SoftInt(Int);
return CpuReg(0);		\return AX register
end;	\CallInt



func	GetKey;			\Get character from keyboard (wait if necessary)
int	SC, Ch;			\this is a low-level routine with no echo, no
begin				\ Ctrl-C, and no flashing cursor.
SC:= CallInt($16, $0000);	\read character and scan code from keyboard
Ch:= SC & $00FF;
if Ch = 0 then Ch:= -(SC>>8);	\return non-ASCII chars as negative scan code
return Ch;
end;	\GetKey



proc	CursorOff;		\Turn off flashing cursor
CallInt($10, $0100, 0, $2000);



func	StrNCmp(A, B, N);	\Compare string A to string B up to N bytes long
\This returns:
\	>0 if A > B
\	=0 if A = B
\	<0 if A < B
\This provides a general string compare, for example:
\ if StrNCmp(A, B, N) >= 0 then...	(if A >= B then...)
char	A, B;	\strings to be compared (must be right justified)
int	N;	\number of bytes to compare
int	I;
begin
for I:= 0, N-1 do
	if A(I) # B(I) then
		return A(I) - B(I);
return 0;			\they're equal
end;	\StrNCmp



proc	Beep;			\A not-too-obnoxious beep
begin
Sound(false, 1, 1000);		\synchronize with system timer to make tone a
Sound(true, 1, 3000);		\ consistent duration and a consistent sound
end;	\Beep



proc	WaitVB;			\Wait for vertical blank
begin
while port($3DA) & $08 do;	\wait for vertical blank to go away
repeat until port($3DA) & $08;	\wait for vertical blank
end;	\WaitVB



proc	TimeOut(Time);		\Display time e.g: 14:25
int	Time;	\time in DOS packed format
int	M, H;	\minute, hour
begin
M:= Time>>5 & $003F;		\0-59
H:= (Time>>11 & $1F);		\0-23
if H < 10 then ChOut(6, ^0);
IntOut(6, H);
ChOut(6, ^:);
if M < 10 then ChOut(6, ^0);
IntOut(6, M);
end;	\TimeOut



proc	DateOut(Date);		\Display date e.g: 10-Feb-2005
int	Date;	\date in DOS packed format
int	D, M, Y,\day, month, year
	I, J;
char	Str;
begin
D:= Date & $001F;		\1-31
M:= Date>>5 & $000F;		\1-12
Y:= (Date>>9 & $7F) + 1980;

IntOut(6, D);
ChOut(6, ^-);
Str:= "JanFebMarAprMayJunJulAugSepOctNovDec ";
J:= 3*(M-1);
for I:= 0, 3-1 do ChOut(6, Str(I+J));
ChOut(6, ^-);
IntOut(6, Y);
end;	\DateOut



func	GetDate;		\Returns the current date in DOS packed format
begin
CallInt($21, $2A00);
return ((CpuReg(2)-1980)<<9 ! (CpuReg(3)&$FF00)>>3 ! (CpuReg(3)&$00FF)) & $FFFF;
end;	\GetDate



func	GetTimeX;		\Returns the current time in DOS packed format
begin
CallInt($21, $2C00);
return (CpuReg(2)&$1F00)<<3 ! (CpuReg(2)&$003F)<<5 ! (CpuReg(3)&$0F00)>>9;
end;	\GetTimeX

\===============================================================================

proc	Exit;			\Make a clean exit
begin
CallInt($21, $3B00, 0, 0, 0, 0, OrigPath);     \set current dir back to original
SetVid(3);			\clear screen and restore flashing cursor
exit;
end;	\Exit

\-------------------------------------------------------------------------------

proc	SetMode;	\Set video mode and compute coordinates for centering
int	F256;	\flag: use 256 color mode instead of 24-bit color mode
int	I;
\Inputs: Height, Width, Depth, CLUT
\Outputs: Video Mode, CLUT, X0, Y0	offsets to center image (pixels)
begin
F256:= Depth <= 8;

case of
  Width=320 & Height=200 & F256:
	begin
	SetVid($13);
	X0:= 0;
	Y0:= 0;
	end;
  Width<=640 & Height<=480:
	begin
	SetVid(if F256 then $101 else $112);
	X0:= (640-Width)/2;		\coordinates to center image on screen
	Y0:= (480-Height)/2;
	end;
  Width<=800 & Height<=600:
	begin
	SetVid(if F256 then $103 else $115);
	X0:= (800-Width)/2;		\coordinates to center image on screen
	Y0:= (600-Height)/2;
	end;
  Width<=1024 & Height<=768:
	begin
	SetVid(if F256 then $105 else $118);
	X0:= (1024-Width)/2;		\coordinates to center image on screen
	Y0:= (768-Height)/2;
	end;
  Width<=1280 & Height<=1024:
	begin
	SetVid(if F256 then $107 else $11B);
	X0:= (1280-Width)/2;		\coordinates to center image on screen
	Y0:= (1024-Height)/2;
	end
other	Beep;

if F256 then
	begin
	port($3C8):= 0;			\starting color register 
	for I:= 0, 256-1 do
		begin
		port($3C9):= CLUT(I)>>18;	\R (8 bits into 6 bits)
		port($3C9):= CLUT(I)>>10;	\G
		port($3C9):= CLUT(I)>>2;	\B
		CLUT(I):= I;
		end;
	end;
end;	\SetMode

\###############################################################################

proc	ShowGIF;	\Display a GIF image file
\Based on CompuServe document 89a, and GifSlow.pas (1/19/88) by Jim Griebel
\"GIF" and "Graphics Interchange Format" are trademarks of CompuServe, Inc.
\Note that there are a maximum of 256 colors in a GIF image.
int	Animate,	\flag: NETSCAPE Application Ext: loop entire GIF file
	FirstTime,	\flag: if animation, set video mode only once
	Iteration,	\if Animate then number of times to loop (0=infinite)
	DelayTime,	\time delay between animation frames (vertical blanks)
			\ Technically this should 0.01 seconds per count)
			\ Note that vertical blank intervals vary 50..120~/sec.
	Transparent,	\color assigned to be transparent (0..255; -1 if none)
	Background,	\background color (normally = 0)
	Erase;		\flag: dispose of image by overwriting with background

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

proc	ShowImage;	\Display one GIF image
int	XC, YC,		\coordinates of current pixel to be plotted
	X1, Y1,		\offset to upper-left corner of image (usually 0,0)
	Pass,		\pass counter, used for interlace (0..3)
	HaveCTbl,	\flag: color table follows current block
	Interlaced,	\flag: scan lines are interlaced (usually false)
	ColorTblSize,	\number of bytes in color table
	ByteCtr,	\number of bytes of data remaining to be read from block
	SP,		\stack pointer
	J, T,		\index and temporary scratch

	Code,		\compressed code (up to 12 bits long)
	CodeBuf,	\3-byte input buffer
	ReadMask,	\Code AND mask for current CodeSize
	ShiftCtr,	\8-counter; counts number of bits shifted in CodeBuf

	ClearCode,	\clear code resets CodeSize and Inx
	EOFCode,	\end-of-image code
	OldCode,	\old Code
	NewCode,	\new Code
	Inx0,		\initial empty location in StrTbl and PixTbl
	Inx,		\table index: next empty location in table
	CodeSize,	\Code size read from header blk (LZW Minimum Code Size)
	CodeSize0,	\initial code size
	FinPix,		\final pixel color (not a code)
	BitMask;	\AND mask for pixel colors (has Depth number of 1 bits)

int	StrTbl(4096),	\table of indexes to substrings
	PixTbl(4096);	\pixel colors
def	StackSize = 4096; \maximum number of items in a repeated sequence
int	Stack(StackSize);



	proc	PlotPix(Color);	\Plot pixel in current raster position
	int	Color;
	int	Tbl1, Tbl2;
	begin
	if Color # Transparent then
	    if YC < Height then	\eliminate occasional extra pixel (from GWS)
		Point(X0+X1+XC, Y0+Y1+YC, Color);
	XC:= XC + 1;		\move to next raster position
	if XC >= Width then
		begin
		XC:= 0;
		if Interlaced then
			begin
			Tbl1:= [8, 8, 4, 2];
			Tbl2:= [0, 4, 2, 1];
			YC:= YC + Tbl1(Pass);
			if YC >= Height then
				begin
				Pass:= Pass+1 & $03;	\limit to array size
				YC:= Tbl2(Pass);
				end;
			end
		else	YC:= YC + 1;
		end;
	end;	\PlotPix



	proc	Push(B);	\Push an item onto the Stack
	int	B;
	begin
	if SP >= StackSize then [Beep;   Exit];
	Stack(SP):= B;
	SP:= SP + 1;
	end;	\Push



	func	Pop;		\Pop item off Stack
	begin
	SP:= SP - 1;
	return Stack(SP);
	end;	\Pop



	func	GetByte;	\Read next byte of image data from block
	begin
	if ByteCtr = 0 then		\get the number of bytes of image data
		ByteCtr:= ChIn(3);	\ that will follow in the current block
		\if ByteCtr gets set to 0 then there's a terminator block and
		\ no more bytes should be read from device 3
	ByteCtr:= ByteCtr - 1;
	return if ByteCtr < 0 then 0 else ChIn(3);
	end;	\GetByte



	func	GetCode;	\Read next image code
	int	C, I;
	begin
	C:= CodeBuf & ReadMask;
	for I:= 0, CodeSize-1 do
		begin
		CodeBuf:= CodeBuf >> 1;
		ShiftCtr:= ShiftCtr - 1;
		if ShiftCtr <= 0 then
			begin
			CodeBuf:= CodeBuf + GetByte<<16;
			ShiftCtr:= 8;
			end;
		end;
	return C;
	end;	\GetCode



	proc	OpenGetCode;	\Initialize GetCode procedure
	begin
	ByteCtr:= 0;		\initialize for GetByte
	CodeBuf:= GetByte + GetByte<<8 + GetByte<<16;
	ShiftCtr:= 8;
	end;	\OpenGetCode



	proc	ReadColorTbl;	\Read in the color table block
	int	I;
	begin
	port($3C8):= 0;
	for I:= 0, ColorTblSize*3-1 do
		port($3C9):= ChIn(3)>>2;	\RGB
	end;	\ReadColorTbl



	func	ReadWord;		\Read in a 16-bit word
	return	ChIn(3) + ChIn(3)<<8;



	proc	ReadImage;	\Read in image data and display it
	int	T;
	begin
	\Read in Image Descriptor Block:
	X1:= ReadWord;				\image position
	Y1:= ReadWord;				\ (mostly used by animations)
	Width:= ReadWord;			\5
	Height:= ReadWord;			\7
	if Width > WidthMax then [Beep;   return];
	if Height > HeightMax then [Beep;   return];

	T:= ChIn(3);				\8, packed bits
	HaveCTbl:= T & $80;
	Interlaced:= T & $40;
	if HaveCTbl then
		begin				\read local color table
		Depth:= (T&7) + 1;
		ColorTblSize:= 1 << Depth;	\number of colors in palette
		ReadColorTbl;
		end;

	\Read in image data blocks and display them
	CodeSize:= ChIn(3);			\LZW minimum code size (bits)
	ClearCode:= 1<<CodeSize;
	EOFCode:= ClearCode + 1;
	Inx0:= ClearCode + 2;			\initial table index
	Inx:= Inx0;
	BitMask:= 1<<Depth - 1;

	\The GIF spec states that the code size used to compute the above values
	\is the code size given in the file, but it's actually the code size + 1
	CodeSize:= CodeSize + 1;
	CodeSize0:= CodeSize;
	ReadMask:= 1<<CodeSize - 1;

	XC:= 0;   YC:= 0;			\start plotting at upper-left
	Pass:= 0;
	SP:= 0;					\empty stack
	OpenGetCode;

	\Decompress image data
	loop	begin
		Code:= GetCode;
		if Code = EOFCode then quit;
		\if ChkKey then quit;

		\The Clear code sets everything back to its initial value then
		\ reads the next code as uncompressed data
		if Code = ClearCode then
			begin
			CodeSize:= CodeSize0;
			ReadMask:= 1<<CodeSize - 1;
			Inx:= Inx0;

			Code:= GetCode;
			OldCode:= Code;
			FinPix:= Code & BitMask;
			PlotPix(FinPix);
			end
		else	begin			\not a clear code, must be data
			NewCode:= Code;
			if Code >= Inx then
				begin		\not in table yet
				Push(FinPix);	\handle this exceptional case
				Code:= OldCode;
				end;
			while Code > BitMask do
				begin	       \follow chain of links thru table
				Push(PixTbl(Code)); \push associated output code
				Code:= StrTbl(Code);
				end;
			FinPix:= Code & BitMask;  \the last code is raw data
			PlotPix(FinPix);

			while SP > 0 do PlotPix(Pop);	\plot stacked pixels

			if Inx<0 ! Inx>=4096 then [Beep; return];
			StrTbl(Inx):= OldCode;	 \rebuild table on the fly
			PixTbl(Inx):= FinPix;	 \(table is not stored with GIF)
			OldCode:= NewCode;

			\Point to next location in table. If the current
			\ maximum value is exceeded, increment code size unless
			\ it's already 12. If it is then do nothing. The next
			\ code decompressed better be a ClearCode
			Inx:= Inx + 1;
			if Inx>ReadMask & CodeSize<12 then
				begin
				CodeSize:= CodeSize + 1;
				ReadMask:= 1<<CodeSize - 1;
				end;
			end;
		end;
	end;	\ReadImage



	proc	EatBlks;	\Skip data sub-blocks
	int	Size, I, T;
	begin
	repeat	Size:= ChIn(3);
		for I:= 0, Size-1 do T:= ChIn(3);
	until Size = 0;			\block terminator
	end;	\EatBlks



	proc	AppExtBlk;	\Application Extension Block: check for Animate
	int	Size, I, T, A;
	char	Str;
	begin
	Size:= ChIn(3);			\should be 11
	Str:= "NETSCAPE ";
	A:= true;
	for I:= 0, 8-1 do
		if ChIn(3) # Str(I) then A:= false;
	for I:= I, Size-1 do T:= ChIn(3); \ignore authentication code
	if A then
		begin			\NETSCAPE application: read sub-block
		Animate:= true;		\turn on animation
		Size:= ChIn(3);
		T:= ChIn(3);		\type
		I:= ReadWord;
		if T=1 & Iteration<=0 then Iteration:= I;
		for I:= 3, Size-1 do T:= ChIn(3);
		end;
	EatBlks;
	end;	\AppExtBlk



	proc	CtrlExtBlk;	\Graphic Control Extension Block
	int	T;
	begin
	T:= ChIn(3);			\size (better be = 4)
	T:= ChIn(3);			\packed bit fields
	DelayTime:= ReadWord; 		\(unsigned)
	Transparent:= ChIn(3);
	if (T&$01) = 0 then Transparent:= -1;	\bit must be set else undefined
	Erase:= (T&$1C) = $08;		\disposal = restore to background color
	T:= ChIn(3);			\termainator (should be = 0)
	end;	\CtrlExtBlk



begin	\ShowImage
\Read in Header Block and Logical Screen Descriptor:
if ChIn(3) # ^G then [Beep;   return];	\file must begin with "GIF"
if ChIn(3) # ^I then [Beep;   return];
if ChIn(3) # ^F then [Beep;   return];
for J:= 3, 5 do T:= ChIn(3);		\skip version info

Width:= ReadWord;			\get screen height and width
Height:= ReadWord;			\the image is <= to these dimensions
if FirstTime then			\only do this once, in case of animation
	begin
	FirstTime:= false;
	case of
	  Width=320 & Height=200:
		begin
		SetVid($13);
		X0:= 0;
		Y0:= 0;
		end;
	  Width<=640 & Height<=480:
		begin
		SetVid($101);
		X0:= (640-Width)/2;	\coordinates to center image
		Y0:= (480-Height)/2;
		end;
	  Width<=800 & Height<=600:
		begin
		SetVid($103);
		X0:= (800-Width)/2;
		Y0:= (600-Height)/2;
		end;
	  Width<=1024 & Height<=768:
		begin
		SetVid($105);
		X0:= (1024-Width)/2;
		Y0:= (768-Height)/2;
		end;
	  Width<=1280 & Height<=1024:
		begin
		SetVid($107);
		X0:= (1280-Width)/2;
		Y0:= (1024-Height)/2;
		end
	other	begin			\default to 640x480
		SetVid($101);
		X0:= (640-Width)/2;
		Y0:= (480-Height)/2;
		end;
	end;

T:= ChIn(3);				\get packed bits
HaveCTbl:= T & $80;
Background:= ChIn(3);			\background color
if HaveCTbl then
	begin
	Depth:= (T&7) + 1;
	ColorTblSize:= 1 << Depth;	\number of colors in palette
	end
else	Background:= 0;
T:= ChIn(3);				\aspect ratio (not used)
if HaveCTbl then ReadColorTbl;		\global color table

loop	begin
	case ChIn(3) of
	  $21:	begin			\"!"
		case ChIn(3) of
		  $F9:	CtrlExtBlk;	\Graphic Ctrl Ext [S ] 00
		  $FF:	AppExtBlk;	\Application Ext 11 [S ]...[S ] 00
		  $01:	EatBlks;	\Plain Text Ext [S ]...[S ] 00
		  $FE:	EatBlks		\Comment Ext [S ]...[S ] 00
		other	EatBlks;	\ignore any unknown $21 labels
		end;
	  $2C:	begin			\","
		ReadImage;
		if Width > WidthMax then return;
		if Height > HeightMax then return;
		if Inx<0 ! Inx>=4096 then return;
		for J:= 0, DelayTime+3-1 do	\there's a minimum delay
			begin
			if ChkKey then quit;	\delays can be 65538/100 seconds
			WaitVB;
			end;
		if Erase then
			begin		\erase image by overwriting w background
			if Background # Transparent then
				for YC:= 0, Height-1 do
				    for XC:= 0, Width-1 do
					Point(X0+X1+XC, Y0+Y1+YC, Background);
			end;
		end;
	  $3B:	quit			\";" trailer
	other	[Beep;   return];	\(probably out of sync)

	if ChkKey then quit;		\abort animation immediately
	end;
end;	\ShowImage

\ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

begin	\ShowGIF
Animate:= false;
Iteration:= 0;
DelayTime:= 0;
Transparent:= -1;
Background:= 0;
Erase:= false;

FirstTime:= true;

loop	begin
	ShowImage;
	if not Animate then quit;
	if ChkKey then quit;		\(CloseImageFile eats keystroke)
	Iteration:= Iteration - 1;
	if Iteration = 0 then quit;	\(don't quit if -1)
	OpenI(3);			\reopen (restart) input file (loop)
	end;
end;	\ShowGIF

\###############################################################################

proc	ShowPCX;	\Display a .PCX file
\Reference: "Graphics File Formats" by Kay & Levine; Lots of hex dumps.
int	BitsPerPixel,	\number of bits per pixel per plane (1 or 8)
	Planes,		\number of planes (1..4)
	BytesPerLine;	\number of bytes per scan line per plane
int	I, J;		\indexes

\-------------------------------------------------------------------------------

proc	Show24;		\Display a 24-bit color PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat (for compression)
	X, Y,	\coordinates (pixels)
	P,	\plane
	S;	\amount to shift color byte (16, 8, or 0)
int	LineBuf(WidthMax); \buffer for one scan line
begin
SetMode;
for Y:= 0, Height-1 do
	begin
	for X:= 0, Width-1 do		\initialize line buffer
		LineBuf(X):= 0;
	Cnt:= 0;			\initialize repeat count
	for P:= 0, 3-1 do		\for 3 planes...
		begin
		S:= (2-P)*8;		\amount to shift for R,G,B: 16, 8, 0
		for X:= 0, BytesPerLine-1 do
			begin
			if Cnt > 0 then
				Cnt:= Cnt - 1
			else	begin
				Byte:= ChIn(3);
				if (Byte & $C0) = $C0 then
					begin		       \set repeat count
					Cnt:= (Byte & $3F) - 1;
					Byte:= ChIn(3);
					end;
				end;
			LineBuf(X):= LineBuf(X) ! Byte<<S;	\RGB
			end;
		end;
	for X:= 0, Width-1 do		\display the scan line
		Point(X0+X, Y0+Y, LineBuf(X));
	end;
end;	\Show24



proc	Show8;		\Display an 8-bit (256-color) PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat
	X, Y,	\coordinates (pixels)
	I;	\index
begin
SetMode;
for Y:= 0, Height-1 do
    for X:= 0, BytesPerLine-1 do
	begin
	Byte:= ChIn(3);
	if (Byte & $C0) # $C0 then
		[if X < Width then Point(X0+X, Y0+Y, Byte)]
	else	begin			\decompress multiple copies
		Cnt:= Byte & $3F;
		Byte:= ChIn(3);
		for I:= 0, Cnt-1 do
			if X+I < Width then Point(X0+X+I, Y0+Y, Byte);
		X:= X + Cnt - 1;
		end;
	end;
end;	\Show8

\-------------------------------------------------------------------------------

proc	ShowPlanar;	\Display a planar PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat (to decompress)
	B,	\working copy of image byte
	X, Y,	\coordinates (pixels)
	XB,	\horizontal (scan line) byte
	P,	\plane
	I;	\index
int	LineBuf(WidthMax); \buffer for one scan line
begin
SetMode;
for Y:= 0, Height-1 do
	begin
	for X:= 0, Width-1 do LineBuf(X):= 0;	\initialize line buffer
	Cnt:= 0;

	for P:= 0, Planes-1 do
		begin
		X:= 0;
		for XB:= 0, BytesPerLine-1 do
			begin
			if Cnt > 0 then Cnt:= Cnt - 1	\use current Byte
			else	begin
				Byte:= ChIn(3);
				if (Byte & $C0) = $C0 then
					begin		\decompress
					Cnt:= (Byte & $3F) - 1;
					Byte:= ChIn(3);
					end;
				end;
			B:= Byte;			\get working copy
			for I:= 0, 8-1 do		\for all of the bits...
				begin
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<P;
				B:= B << 1;		\next bit
				X:= X + 1;
				end;
			end;	\for XB
		end;	\for P

	for X:= 0, Width-1 do
		Point(X0+X, Y0+Y, CLUT(LineBuf(X)&$0F));
	end;
end;	\ShowPlanar

\===============================================================================

begin	\ShowPCX
\Read image file header information
if ChIn(3) # $0A then [Beep;   return];	\0 ignore file if invalid format
I:= ChIn(3);				\1 skip version
I:= ChIn(3);				\2 skip
BitsPerPixel:= ChIn(3);			\3
Width:= ChIn(3) + ChIn(3)<<8;		\4,5 XMin
Height:= ChIn(3) + ChIn(3)<<8;		\6,7 YMin
Width:= ChIn(3) + ChIn(3)<<8 \XMax\ - Width + 1;	\8,9
Height:= ChIn(3) + ChIn(3)<<8 \YMax\ - Height + 1;	\10,11
for I:= 12, 15 do J:= ChIn(3);		\12-15 skip
for I:= 0, 16-1 do			\16-63
	CLUT(I):= ChIn(3)<<16 + ChIn(3)<<8 + ChIn(3);
I:= ChIn(3);				\64 skip
Planes:= ChIn(3);			\65
BytesPerLine:= ChIn(3) + ChIn(3)<<8;	\66,67
for I:= 68, 127 do J:= ChIn(3);		\68-127 skip
Depth:= BitsPerPixel * Planes;

if Width > WidthMax then [Beep;   return];
if BytesPerLine > WidthMax then [Beep;   return];

if BitsPerPixel = 8 then
	begin
	if Planes = 3 then Show24
	else	begin
		Show8;
		\Load CPU's color registers with image's palette info
		if ChIn(3) # $0C then	\check for if valid palette info
			[Beep;   return];\don't change colors if invalid
		port($3C8):= 0;		\starting color register 
		for I:= 0, 256*3-1 do
			port($3C9):= ChIn(3)>>2;
		end;
	end
else	ShowPlanar;
end;	\ShowPCX

\###############################################################################

proc	ShowLBM;	\Display the opened .LBM (or .BBM or .IFF) file
\ Reference: "Amiga ROM Kernel Reference Manual", IFF, p I-28. "Supercharged
\ Bitmapped Graphics" and "Graphics Workshop" by Rimmer don't handle widths with
\ odd numbers of pixels correctly. Lots of hex dumps were used. Deluxe Paint II
\ is the standard. VPic 6.1e.6 also has problems similar to GWS 7.0d.
\ILBM (InterLeaved BitMap) is the most common file format. It uses separate
\ planes for each color bit. PBM (Deluxe Paint's Proprietary BitMap) uses a
\ byte (like BMP and PCX) to represent one of 256 colors in a palette defined
\ by CMAP. PBM doesn't seem to be used for 24-bit color.

def	FORM= $464F524D, ILBM= $494C424D, PBM = $50424D20,	\chunk headers
	BMHD= $424D4844, CMAP= $434D4150, BODY= $424F4459;

int	HaveBMHD,	\flag: bit map header chunk has been read in
	HavePBM,	\flag: image data are in PBM format, not ILBM
	Masking,	\masking type (0=none, 1=extra bit plane, 2=transparent)
	Compressed,	\compression algorithm (0= none, 1=ByteRun1)
	Transparent,	\color designated as transparent (black) when masking=2
	DW;		\double word (32-bit value)



func	ReadWord;	\Return word from disk file (big endian style)
return ChIn(3)<<8 + ChIn(3);



func	ReadDWord;	\Return double word from disk file (big endian)
return ReadWord <<16 + ReadWord;

\-------------------------------------------------------------------------------

proc	EatChunk;	\Read in and dispose of a chunk
int	CSize, N, T;
begin
CSize:= ReadDWord;
for N:= 0, CSize-1 do
	T:= ChIn(3);
if CSize & 1 then		\make sure it's ending on an even-byte boundary
	T:= ChIn(3);
end;	\EatChunk

\-------------------------------------------------------------------------------

proc	DoPBM;		\Load and display image data in PBM format
int	CSize,	\chunk size (bytes) (ignored)
	X, Y,	\coordinates (pixels)
	P;	\padding byte (discarded)



proc	DoLine;		\Read in a compressed scan line in PBM format
int	N, B,	\bytes from input file
	J;	\index
begin
X:= 0;					\start at left side
repeat	begin
	N:= ChIn(3);			\get number N
	if N <= $7F then
		begin			\copy the next N+1 bytes literally
		for J:= 0, N do		\ (i.e. no compression)
			begin
			Point(X0+X, Y0+Y, CLUT(ChIn(3)));
			X:= X + 1;
			end;
		end

	\else if N = $80 then []	\\incorrectly used by Adobe Photoshop

	else	begin			\N is in the range: $81..$FF
					\replicate next byte -extend(N)+1 times
		B:= ChIn(3);		\read next byte
		for J:= 0, -ext(N) do	\(byte run compression)
			begin
			Point(X0+X, Y0+Y, CLUT(B));
			X:= X + 1;
			end;
		end;
	end;
until X >= ((Width+1) & $FFFE);		\an even number of PIXELS must be read
end;	\DoLine				 "transparent" pixels pad the width



begin	\DoPBM
Depth:= 8;				\always 8 bits per color (VPic bug)
CSize:= ReadDWord;			\get chunk size and ignore it
for Y:= 0, Height-1 do			\for all of the scan lines...
	begin
	if Compressed then DoLine
	else	begin
		for X:= 0, Width-1 do	\PBM (chunky) mode is straightforward
			Point(X0+X, Y0+Y, CLUT(ChIn(3)));
		if Width & 1 then P:= ChIn(3);		\eat odd padding byte
		end;
	end;
end;	\DoPBM

\-------------------------------------------------------------------------------

proc	DoILBM;		\Load and display image data in ILBM format
int	CSize,	\chunk size (bytes) (ignored)
	WWidth,	\image width in words (rounded up)
	X, Y,	\coordinates (pixels)
	D,	\depth of bit plane (0..Depth-1)
	W,	\word
	XW,	\horizontal word
	I;	\index
int	LineBuf(WidthMax); \buffer for one scan line



func	BGR2RGB(BGR);	\Swap the red and blue bytes
int	BGR;
int	R, G, B;
begin
R:= BGR & $0000FF;
G:= BGR & $00FF00;
B:= BGR & $FF0000;
return R<<16 + G + B>>16;
end;	\BGR2RGB



proc	DoLine;		\Read in a compressed scan line for a single bit plane
int	N, B, B0,	\bytes from input file
	J;		\index
begin
X:= 0;					\start at left side
repeat	begin
	N:= ChIn(3);			\get number N
	if N <= $7F then
		begin			\copy the next N+1 bytes literally
		for J:= 0, N do		\ (i.e. no compression)
			begin
			B:= ChIn(3);
			for I:= 0, 8-1 do
				begin
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<D;
				B:= B << 1;
				X:= X + 1;
				end;
			end;
		end

	\else if N = $80 then []	\\incorrectly used by Adobe Photoshop

	else	begin			\N is in the range: $81..$FF
					\replicate next byte -extend(N)+1 times
		B0:= ChIn(3);		\read next byte
		for J:= 0, -ext(N) do	\(byte-run compression)
			begin
			B:= B0;		\working copy
			for I:= 0, 8-1 do
				begin	\distribute bits in B into 8 bytes
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<D;
				B:= B << 1;
				X:= X + 1;
				end;
			end;
		end;
	end;
until X >= Width;
end;	\DoLine



begin	\DoILBM
WWidth:= (Width+15)/16;			\width of scan line in words rounded up
CSize:= ReadDWord;			\get chunk size and ignore it
for Y:= 0, Height-1 do			\for all of the scan lines...
	begin
	for X:= 0, WidthMax-1 do LineBuf(X):= 0;	\initialize line buffer

	for D:= 0, Depth-1 do		\for all of the bit planes...
		begin
		if Compressed then DoLine
		else	begin
			X:= 0;				\start at left side
			for XW:= 0, WWidth-1 do		\for all of the words...
				begin
				W:= ReadWord;
				for I:= 0, 16-1 do	\for all of the bits...
					begin
					if W & $8000 then	\set bit
					    if X < WidthMax then
						LineBuf(X):= LineBuf(X) ! 1<<D;
					W:= W << 1;
					X:= X + 1;
					end;
				end;
			end;
		end;	\for loop

	if Depth = 24 then			\display the scan line
		for X:= 0, Width-1 do
			Point(X0+X, Y0+Y, BGR2RGB(LineBuf(X)))
	else	for X:= 0, Width-1 do
			Point(X0+X, Y0+Y, CLUT(LineBuf(X)&$FF));

	if Masking = 1 then			\read mask in and discard it
		if Compressed then DoLine
		else	for XW:= 0, WWidth-1 do	\for all of the words...
				W:= ReadWord;
	end;
end;	\DoILBM

\-------------------------------------------------------------------------------

proc	DoCMAP;		\Read in color map chunk and set up color lookup table
int	CSize,	\number of bytes in CMAP field
	ColRegs,\number of color registers to use
	N, T;
begin
CSize:= ReadDWord;
ColRegs:= CSize/3;
if ColRegs > 256 then ColRegs:= 256;	\limit maximum number of color registers
for N:= 0, ColRegs-1 do
	CLUT(N):= ChIn(3)<<16 ! ChIn(3)<<8 ! ChIn(3);	\R, G, B order

for N:= N*3, CSize-1 do		\eat the rest of the chunk, if any
	T:= ChIn(3);
if N & 1 then T:= ChIn(3);	\must end on an even-byte boundary
end;	\DoCMAP

\-------------------------------------------------------------------------------

proc	DoBMHD;		\Read in BitMap Header chunk
int	T;
begin
T:= ReadDWord;		\chunk size in bytes (not used, it's always 20)
Width:= ReadWord;	\raster width and height in pixels
Height:= ReadWord;
T:= ReadWord;		\offset (in pixels) for this image
T:= ReadWord;		\ (not used)
Depth:= ChIn(3);	\number of bit planes (unless PBM, which is always 8)
Masking:= ChIn(3);	\masking technique (1=mask stored in separate bit plane)
Compressed:= ChIn(3);	\compression algorithm (0=none, 1=byte run)
T:= ChIn(3);		\unused by IFF standard
Transparent:= ReadWord;	\transparent color (only used with masking = 2)
T:= ChIn(3);		\aspect ratio (1:1 is assumed here)
T:= ChIn(3);
T:= ReadWord;		\source page size (not used here)
T:= ReadWord;

HaveBMHD:= true;	\indicate that BMHD chunk has been read in
end;	\DoBMHD

\-------------------------------------------------------------------------------

begin	\ShowLBM
HaveBMHD:= false;
if ReadDWord # FORM then [Beep;   return];	\file did not start with "FORM"

DW:= ReadDWord;		\read in chunk size and ignore it
DW:= ReadDWord;		\read in 4-character chunk ID
if DW#ILBM & DW#PBM then [Beep;   return];	\"ILBM" or "PBM " missing
HavePBM:= DW = PBM;

for DW:= 0, 256-1 do	\set up default gray-scale palette, in case no CMAP
	CLUT(DW):= DW<<16 ! DW<<8 ! DW;

loop	begin
	case ReadDWord of		\handle chunk
	  BMHD:	begin
		DoBMHD;
		if Width > WidthMax then [Beep;   return];  \image is too wide
		end;
	  CMAP:	DoCMAP;
	  BODY:	begin
		if ~HaveBMHD then [Beep;   return];	\"BMHD" missing
		if Masking = 2 then CLUT(Transparent&$FF):= $FFFFFF; \br. white
		SetMode;
		if HavePBM then DoPBM else DoILBM;
		quit;
		end
	other	EatChunk;		\ignore any unrecognized chunks
	if GetErr \#0\ then [Beep;   return];		\BODY missing
	end;
end;	\ShowLBM

\###############################################################################

proc	ShowBMP;	\Display the opened .BMP file
\Reference: Microsoft MSDN Library; and "Graphics File Formats" by Kay & Levine
int	Offset,		\offset from start of file to image data (bytes)
	HeadSize,	\header size: 40=Windows format; 12=OS/2 (old) format
	R, G, B,	\red, green, blue (0..255)
	X, Y,		\screen coordinates (pixels)
	I, J, T,	\indexes and temporary scratch
	CTblSize;	\number of bytes in BMP's color table
begin
if ChIn(3) # ^B then [Beep;   return];	   \file must begin with "BM" (Bit Map)
if ChIn(3) # ^M then [Beep;   return];
for I:= 2, 9 do X:= ChIn(3);	\2, skip unused header info
Offset:= ChIn(3) + ChIn(3)<<8;	\10, offset from start of file to image data
T:= ChIn(3);   T:= ChIn(3);	\12, skip unused header info (32767 max. is ok)

HeadSize:= ChIn(3) + ChIn(3)<<8;	   \14, size of header (40 or 12)
T:= ChIn(3);   T:= ChIn(3);		   \16, skip high bytes of size
Width:= ChIn(3) + ChIn(3)<<8;	   	   \18
if HeadSize = 12 then
	begin				   \OS/2 (old) format:
	Height:= ChIn(3) + ChIn(3)<<8;	   \20
	T:= ChIn(3);   T:= ChIn(3);	   \22, skip "number of image planes"
	Depth:= ChIn(3) + ChIn(3)<<8;	   \24, bits per pixel
	end
else	begin				   \Windows (normal) format:
	T:= ChIn(3);   T:= ChIn(3);	   \20, skip unused high bytes of Width
	Height:= ChIn(3) + ChIn(3)<<8;	   \22
	T:= ChIn(3);   T:= ChIn(3);	   \24, skip unused high bytes of Height
	T:= ChIn(3);   T:= ChIn(3);	   \26, skip "number of image planes"
	Depth:= ChIn(3) + ChIn(3)<<8;	   \28, bits per pixel
	for I:= 30, 53 do T:= ChIn(3);	   \30, skip rest of header
	end;
if Width > WidthMax then [Beep;   return];
if Height > HeightMax then [Beep;   return];

\Read in the color table up to a maximum of 256 (4-byte) entries
CTblSize:= Offset - 14 - HeadSize;	\number of bytes of color-table data
if CTblSize > 256*4 then CTblSize:= 256*4;
I:= 0;   J:= 0;
while J < CTblSize do
	begin
	B:= ChIn(3);			\(things tend to be backwards in BMPs)
	G:= ChIn(3);
	R:= ChIn(3);
	J:= J + 3;
	if HeadSize # 12 then		\if not OS/2 (old) format
		[T:= ChIn(3);   J:= J + 1];
	if I < 256 then			\(for safety)
		[CLUT(I):= R<<16 + G<<8 + B;   I:= I + 1];
	end;

SetMode;
for Y:= -(Height-1), 0 do		\they're upside down (!)
	begin		\note that lines are padded to end on a 4-byte boundary
	case Depth of
	  32:	begin
		for X:= 0, Width-1 do
			begin
			T:= ChIn(3) + ChIn(3)<<8 + ChIn(3)<<16;	\xxRRGGBB
			Point(X0+X, Y0-Y, T);
			T:= ChIn(3);	\discard unused high byte
			end;
		end;
	  24:	begin
		for X:= 0, Width-1 do
			begin
			T:= ChIn(3) + ChIn(3)<<8 + ChIn(3)<<16;	\xxRRGGBB
			Point(X0+X, Y0-Y, T);
			end;
		for I:= Width*3, ((Width*3+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  16:	begin
		for X:= 0, Width-1 do
			begin
			T:= ChIn(3) + ChIn(3)<<8;	\5 bits of R, G, and B
			R:= (T&$7C00)<<9;		\repackage in 24-bit
			G:= (T&$03E0)<<6;		\ format; (5:6:5 format
			B:= (T&$001F)<<3;		\ is not an issue here:
			Point(X0+X, Y0-Y, R+G+B);	\ hardware-independent)
			end;
		for I:= Width*2, ((Width*2+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  8:	begin
		for X:= 0, Width-1 do
			Point(X0+X, Y0-Y, CLUT(ChIn(3)));
		for I:= Width, ((Width+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  4:	for X:= 0, ((Width+7)&$FFF8)-1 do	\4-byte boundary =
			begin				\ 8-nibble boundary
			T:= ChIn(3);
			if X < Width then Point(X0+X, Y0-Y, CLUT(T>>4));
			X:= X + 1;
			if X < Width then Point(X0+X, Y0-Y, CLUT(T&$0F));
			end;
	  1:	for X:= 0, ((Width+31)&$FFE0)-1 do	\4-byte boundary =
			begin				\ 32-bit boundary
			T:= ChIn(3);
			for I:= 0, 7 do
				begin
				if X < Width then
					Point(X0+X, Y0-Y, CLUT(T>>7&1));
				X:= X + 1;
				T:= T << 1;
				end;
			X:= X - 1;
			end
	other	[Beep;   return];
	end;
end;	\ShowBMP

\###############################################################################

func	OpenImageFile(FN);	\Open (initialize) an image file for viewing
char	FN;			\string containing name of file to open (ASCIIZ)
begin
Trap(false);			\turn off error trapping; if error don't abort
FileHandle:= FOpen(FN, 0);	\get input handle
if GetErr \#0\ then
	[Beep;   return false]	\just beep if file was not found (somehow)
else	[FSet(FileHandle, ^I);   OpenI(3);   return true];	\else open file
end;	\OpenImageFile		 leave error trapping turned off



proc	CloseImageFile;		\Finish viewing an image file
begin
FClose(FileHandle);		\close out file handle so it can be reused
if GetErr \#0\ then Beep;	\notify if there was a read beyond EOF, etc.
Trap(true);			\reenable error trapping
if ChIn(1) then [];		\wait for keystroke (Ctrl+C is ignored)
OpenI(0);			\flush (arrow keys, etc., return multiple bytes)
SetVid(3);			\restore normal text display mode
CursorOff;			\ but without the flashing cursor
SelectionViewed:= Selection;	\record selection viewed (to display its info)
end;	\CloseImageFile

\===============================================================================

proc	GoToSubdir(Dir);     \Change current directory to specified subdirectory
char	Dir;	\string containing name of subdirectory (ASCIIZ)
char	Dir2;	\string in lower memory containing complete path name of sub dir
int	I, J;
begin
Dir2:= MAlloc(64/16+1);			\provide some memory that DOS can access
Poke(Dir2, 0, ^\);			\insert leading slash for root directory
CpuReg(5\SI\):= 1;			\get current directory path into Dir2
CallInt($21, $4700, 0, 0, 0\default drive\, 0, Dir2);

loop	begin				\locate end of Dir2 string
	for I:= 1, 64 do
		if Peek(Dir2,I) = 0 then quit;
	quit;	\(for safety)
	end;
if I > 1 then				\append backslash, but not two in a row
	[Poke(Dir2, I, ^\);   I:= I + 1];
for J:= 0, 13-1 do			\append directory name (8+1+3+1 maximum)
	Poke(Dir2, I+J, Dir(J));

CallInt($21, $3B00, 0, 0, 0, 0, Dir2);	\set new current directory
Release(Dir2);				\(don't accumulate chunks of memory)
end;	\GoToSubdir



proc	GoToParent;	\Change current directory to parent directory
char	Dir2;	\string in lower memory containing complete path name
int	I;
begin
Dir2:= MAlloc(64/16+1);			\provide some memory that DOS can access
Poke(Dir2, 0, ^\);			\insert leading slash for root directory
CpuReg(5\SI\):= 1;			\get current directory path into Dir2
CallInt($21, $4700, 0, 0, 0\default drive\, 0, Dir2);

\Search back from end of string to first backslash and replace it with a 0
loop	begin				\locate end of Dir2
	for I:= 1, 64 do
		if Peek(Dir2,I) = 0 then quit;
	quit;	\(for safety)
	end;
repeat I:= I - 1 until Peek(Dir2,I) = ^\;

if I = 0 then I:= 1;			\if root dir then leave backslash alone
Poke(Dir2, I, 0);			\terminate string (with a 0)
CallInt($21, $3B00, 0, 0, 0, 0, Dir2);	\set new current directory
Release(Dir2);				\(don't accumulate chunks of memory)
end;	\GoToParent

\===============================================================================

proc	GetFileSelection;	\Get file selection from user. Outputs Selection
\Create a list of files from the current directory, store them into FileName,
\ FileTime (etc.) and return an index into the array of the selected file
int	Extension,	\list of file extensions
	ExtID,		\Extension ID (index)
	FMax;		\index to last file name + 1
def	WinX=2, WinY=1; \upper left corner of text in display window
int	WinW, WinH,	\width and height of text in window (in characters)
	ScreenW, ScreenH, \screen width and height (in characters)
	Cols;		\number of columns of file names

\-------------------------------------------------------------------------------

proc	SelectFile;	\Use arrow keys to select a file. Outputs Selection.
int	I, J,		\indexes
	X, Y,		\screen coordinates (character cells)
	Key;		\keystroke (ASCII or negative value of the scan code)
def	NameW = 13;	\width of displayed file names (in characters) (8+1+3+1)



proc	ShowFrame;	\Display a frame around the file selection window
int	X, Y, 		\screen coordinates (character cells)
	S1, S2;		\graphic file sizes (bytes)
char	Path,		\ASCIIZ string containing path name of current directory
	Path2(64);	\Path in upper memory (which is in normal heap space)
begin
ScreenW:= Peek(0, $44A);			\get actual screen dimensions
ScreenH:= Peek(0, $484) + 1;
if ScreenH<24 ! ScreenH>65 then ScreenH:= 25;	\(for safety)

WinW:= ScreenW - 2 - 2;	\there are TWO borders on the sides of the text window
Cols:= WinW/NameW;	\number of columns of file names that will fit
WinW:= Cols*NameW;	\truncate window width to last column
WinH:= ScreenH - 1 - 3;	\there is ONE border on the top and 3 on the bottom

\Draw borders
OpenO(6);		\reset existing SetWind to full screen, so borders can
Attrib(BorderColor);	\ be drawn
for Y:= 0, ScreenH-1 do
	begin
	Cursor(0, Y);   ChOut(6, ^ );
	Cursor(2+WinW+2-1, Y);   ChOut(6, ^ );
	end;
for X:= 1, 2+WinW+2-1-1 do
	begin
	Cursor(X, 0);   ChOut(6, ^ );
	Cursor(X, ScreenH-1);   ChOut(6, ^ );
	end;

Cursor(3, 0);   Text(6, "S E E   1.4");		\title and version number
Cursor(2+WinW-12-2-5, 0);  DateOut(GetDate);  Text(6, "  ");  TimeOut(GetTimeX);

\Show list of key commands
Cursor(2, ScreenH-1);
Text(6, 
"Arrows   PgUp   PgDn   Home   End   Del   1stChar   Enter   Esc");

Attrib(InfoColor);
for X:= 1, 2+WinW+2-1-1 do
	begin
	Cursor(X, ScreenH-2);   ChOut(6, ^ );
	Cursor(X, ScreenH-3);   ChOut(6, ^ );
	end;

\Display drive and path name of current directory
Path:= MAlloc(64/16+1);			\use memory space that DOS can access
X:= CallInt($21, $1900) + ^A;		\get letter for the current drive
Poke(Path, 0, X);			\insert it into the path string
Poke(Path, 1, ^:);
Poke(Path, 2, ^\);			\insert leading slash for root directory
CpuReg(5\SI\):= 3;			\get current directory path name string
CallInt($21, $4700, 0, 0, 0\default drive\, 0, Path);
Blit(Path, 0, DataSeg, Path2, 64);	\copy it into our data space
Release(Path);				\(be careful not to accumulate memory)
Cursor(2, ScreenH-3);			\display complete path name
Text(6, Path2);

\Show info for the last file viewed
if Selection = SelectionViewed then
	begin
	\ C:/LONG/PATH/FILENAME.EXE
	\ 14-Feb-2005    14:28    27520/64000=0.43    320x200x8
	if Path2(3) \#0\ then ChOut(6, ^\);	\if not root directory
	Text(6, addr FileName(Selection,0));
	Cursor(2, ScreenH-2);
	DateOut(FileDate(Selection));
	Text(6, "     ");
	TimeOut(FileTime(Selection));
	Text(6, "     ");
	S1:= FileSize(Selection);
	IntOut(6, S1);
	ChOut(6, ^/);
	S2:= Width*Height*Depth/8;		\image size
	if S2 <= 0 then S2:= 1;			\protect against div-by-zero
	IntOut(6, S2);
	ChOut(6, ^=);
	Format(1,2);				\ratio (compression)
	RlOut(6, float(S1) / float(S2));
	Text(6, "     ");
	IntOut(6, Width);			\image dimensions
	ChOut(6, ^x);
	IntOut(6, Height);
	ChOut(6, ^x);
	IntOut(6, Depth);
	end;
end;	\ShowFrame



proc	ShowFileNames;	\Show a list of file names in a window
int	J, X, Y;
begin
ShowFrame;

Attrib(FileColor);
SetWind(WinX-1, WinY, WinX-1+WinW+2-1, WinY+WinH-1, 0, true);	\erase window

J:= WinStart;		\display file names, starting with the first in window
for X:= 0, Cols-1 do		\for all of the columns of names...
    for Y:= WinY, WinY+WinH-1 do	\for all of the lines in the window...
	begin
	Cursor(WinX+X*NameW, Y);
	Attrib(if FileName(J,ID) = 0 then DirColor else FileColor);
	if J < FMax then Text(6, addr FileName(J,0)) else return;
	J:= J + 1;			\next name
	end;
end;	\ShowFileNames



func	Verify;		\Show message and return 'true' if verified
def	ScrWidth=80;
begin
Attrib(CurFileColor);
SetWind((ScrWidth-16)>>1, 7, (ScrWidth+16)>>1-1, 7+4, 0, true);

Text(6, "
  Delete file
  ");
Text(6, addr FileName(Selection,0));
Text(6, "
  (Y/N)? ");

OpenI(0);
repeat until KeyHit;				\Wait for keystroke
ShowFileNames;
if (GetKey ! $20) = ^y then return true;	\(don't echo, especially Ctrl+C)
return false;
end;	\Verify



func	DeleteFile;	\Returns 'true' if file was deleted
int	Del, DosMemSeg;
begin
Del:= Verify;
if not Del then return false;
DosMemSeg:= MAlloc(1);
FileName(Selection,ID):= $FF;
Blit(DataSeg, addr FileName(Selection,0), DosMemSeg, 0, 16);
\WARNING: FileName must reside within the first 64K of DataSeg
CpuReg(0):= $4100;
CpuReg(3):= 0;
CpuReg(9):= DosMemSeg;
SoftInt($21);

Release(DosMemSeg);
return true;		\file was deleted
end;	\DeleteFile



begin	\SelectFile
ShowFileNames;
loop	begin
	\Make sure Selection is within legal range
	if Selection < 0 then Selection:= 0
	else if Selection >= FMax then Selection:= FMax-1;

	while Selection >= WinStart+Cols*WinH do  \shift window if necessary
		begin			\move window forward a column at a time
		WinStart:= WinStart + WinH;
		ShowFileNames;		\(rarely moves more than 1 or 2 columns)
		end;
	while Selection < WinStart do
		begin			\move window back a column at a time
		WinStart:= WinStart - WinH;
		if WinStart < 0 then WinStart:= 0;
		ShowFileNames;
		end;

	X:= (Selection-WinStart)/WinH;	\highlight item at cursor position
	Y:= rem(0);
	X:= X*NameW + WinX;
	Y:= Y + WinY;
	Hilight(X, Y, X+NameW-2, Y,
		if FileName(Selection,ID)=0 then CurDirColor else CurFileColor);

	Key:= GetKey;			\wait for key command

	Hilight(X, Y, X+NameW-2, Y,	\unhighlight item
		if FileName(Selection,ID)=0 then DirColor else FileColor);

	case Key of			\execute key command
	  -UpArrow: Selection:= Selection - 1;
	  -DnArrow: Selection:= Selection + 1;
	  -LtArrow: Selection:= Selection - WinH;	\(may go out of range)
	  -RtArrow: Selection:= Selection + WinH;
	  -Home:    Selection:= 0;
	  -End:     Selection:= FMax-1;
	  -Delete:  [if DeleteFile then quit];
	  -PageUp:  Selection:= Selection - Cols*WinH;
	  -PageDn:  Selection:= Selection + Cols*WinH;
	  CR:	    quit;
	  Esc, ^C-Ctrl: Exit
	other	begin			\select file name by its first character
		I:= Selection;		\search forward from current selection
		for J:= 0, FMax-1 do	\for all of the files...
			begin
			I:= I + 1;
			if I >= FMax then I:= 0;	\wrap to top of list
			if (Key&$DF) = (FileName(I,0)&$DF) then	\use uppercase
				[Selection:= I;   J:= FMax];
			end;
		end
	end;	\loop
end;	\SelectFile

\-------------------------------------------------------------------------------

proc	Sort(A, N);	\Sort array A (etc.) using the shell sort method
int	A,		\array FileName (in an integer instead of char)
	N;		\number of elements in array (size)
int	J, Gap, I, JG, T;
begin
Gap:= N/2;
while Gap > 0 do
	begin
	for I:= Gap, N-1 do
	    begin
	    J:= I - Gap;
	    loop begin
		if J < 0 then quit;
		JG:= J + Gap;
		if StrNCmp(A(J), A(JG), 8) <= 0 then quit;
		T:= A(J);   A(J):= A(JG);   A(JG):= T; \swap entries at J and JG
		T:= FileDate(J);  FileDate(J):= FileDate(JG);  FileDate(JG):= T;
		T:= FileTime(J);  FileTime(J):= FileTime(JG);  FileTime(JG):= T;
		T:= FileSize(J);  FileSize(J):= FileSize(JG);  FileSize(JG):= T;
		J:= J - Gap;
		end;
	    end;
	    Gap:= Gap/2;
	end;
end;	\Sort



func	ReadFileNames(J, Ext);
\Read the current directory and store the file names that match Ext into
\ the FileName (etc.) arrays. Returns the index of the last name stored (+1).
int	J;	\(starting) index for FileName array
char	Ext;	\string containing file extension (or *.*)
int	I;	\scratch index
char	DTA;	\Disk Transfer Area, in conventional memory
begin	
DTA:= MAlloc(43/16+1);
CallInt($21, $1A00, 0, 0, 0, 0, DTA);		\set up Disk Transfer Area

if CallInt($21, $4E00, 0, $37, Ext, 0, DataSeg) = 0 then \look up 1st file name
  loop	begin			\copy file names into the FileName (etc.) arrays
	if J >= FilesMax then quit;		\protect against overflows
	if ExtID>0 ! (Peek(DTA,21) & $10) then	\file or directory (incl .. & .)
		begin
		for I:= 0, 12 do FileName(J,I):= Peek(DTA,30+I);\get file name,
		FileName(J,ID):= ExtID;				\ file type
		FileDate(J):= Peek(DTA,24) + Peek(DTA,25)<<8;	\ date
		FileTime(J):= Peek(DTA,22) + Peek(DTA,23)<<8;	\ time
		FileSize(J):= Peek(DTA,26) + Peek(DTA,27)<<8 +	\ size
			Peek(DTA,28)<<16 + Peek(DTA,29)<<24;
		if Peek(DTA,30)#^. ! Peek(DTA,31)#0 then J:= J+1; \eliminate "."
		end;
	if CallInt($21, $4F00) then quit;	\look up next file name, if one
	end;

Release(DTA);
return J;	\return index (used to append additional file names)
end;	\ReadFileNames



begin	\GetFileSelection
\List of file extensions to read in:
\ "*.*" is used for subdirectories (and ".."), and it must be first in the list.
\ The items in this array must correspond with the ID numbers used in Main.
Extension:= ["*.*", "*.BMP", "*.LBM", "*.BBM", "*.IFF", "*.PCX", "*.GIF"];
FMax:= 0;
for ExtID:= 0, 7-1 do			\for each item in Extension...
	FMax:= ReadFileNames(FMax, Extension(ExtID)); \read list of file names
Sort(FileName, FMax);			\put them into alphabetical order
SelectFile;				\get selection from user and return it
end;	\GetFileSelection

\===============================================================================

begin	\Main
CpuReg:= GetReg;
DataSeg:= CpuReg(12);

TrapC(true);				\disable Ctrl+C (always use Exit proc)
\Save current directory path name so Exit restores current directory if needed
OrigPath:= MAlloc(64/16+1);		\use memory space that DOS can access
Poke(OrigPath, 0, ^\);			\insert leading slash for root directory
CpuReg(5\SI\):= 1;			\save current directory path in OrigPath
CallInt($21, $4700, 0, 0, 0\default drive\, 0, OrigPath);

Clear;					\clear screen (graphics or text)
CursorOff;
Selection:= 0;				\initialize to first file name
WinStart:= 0;
SelectionViewed:= -1;			\set to illegal value to suppress status
loop	begin
	GetFileSelection;
	case FileName(Selection,ID) of	\file type (determined by extension)
	  0:	begin
		if FileName(Selection,0) = ^. then GoToParent	\".."
	  	else GoToSubdir(addr FileName(Selection,0));	\sub-directory
	  	Selection:= 0;		\initialize to first file name
	  	SelectionViewed:= -1;
	  	WinStart:= 0;
	  	end;
	  1:	if OpenImageFile(addr FileName(Selection,0)) then
			[ShowBMP;   CloseImageFile];
	  2, 3, 4:	\LBM, BBM, IFF
		if OpenImageFile(addr FileName(Selection,0)) then
			[ShowLBM;   CloseImageFile];
	  5:	if OpenImageFile(addr FileName(Selection,0)) then
			[ShowPCX;   CloseImageFile];
	  6:	if OpenImageFile(addr FileName(Selection,0)) then
			[ShowGIF;   CloseImageFile];
	  $FF:	[]			\file was deleted
	other	Beep;			\should be impossible, but don't abort
	end;
end;	\Main
