/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: SCANLNGS.C $
   $Locker: ska $	$Name:  $	$State: Rel $

	int msg2ScanLngDefs([msgCat *cat,] MSGFILE f, dword pos, dword limit)

	Read all language definitions fromout the catalogue and join them
	into the catalogue.

	If MSG_MULTIPLE_CATALOGUES is not defined, there is no first argument,
	what causes that the 'msg_glbCat' identifier refers to the global
	catalogue; otherwise it refers to the passed catalogue.

	"Joining" means that if the catalogue already contains entries, they
	may superceed the language definitions of the current catalogue.
	This is weighted with a "match strength" algorithm as follows:
		+ The highest strength has a language definition that
			a) has at least one associated character set that is
			identical to the current one, and
			b) the language code is equal to the current one.
		+ Then: language definitions that:
			a) has at least one associated character set that is
			identical to the current one, and
			b) one of the alias language codes is equal to the current one.
		+ Then: language definitions that:
			a) has at least one associated character set that is
			identical to the current one, and
			b) is the physically first definition within the chain.
		+ The lowest strength has a language that:
			a) is the physically first definition within the chain.

	Any language definitions, that have no strength at all, are ignored.

	Language definitions of the same strength are weighted again:
		+ A definition of the _current_ catalogue superceeds definitions
			from previous catalogues.
		+ Within a catalogue the first definition superceeds any other
			definition of the same catalogue with the same strength.

	A match is performed for each message group individually.

	Return:
		<0: an error occured while reading the file or the item chain;
			abs(rv) == the number of successfully scaned language definitions

		>0: no error; number of successfully scanned definitions

		==0: no definition found; either empty catalogue or error before first
			language definition

*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include "msg2.h"
#include "cntry.h"
#include "fileIO.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: SCANLNGS.C 1.4 1998/10/07 04:42:53 ska Rel ska $";
#endif /*#	defined(RCS_Version) */

/*
 *	Fetch this control area of the current language definition
 *	and join it with msg_glbCat
 *
 *	Return: 0 on failure
 */
static int fetchLngDef(DCL_GCAT	MSGFILE f, MSGFILE fd)
{	msgLng lng;
	MsgStru2 *msg2;
	msgGrpDef *grp;
	int strength;
	int i, rv;
	MSGFILE f1;		/* file descriptor to be removed */

/* 1. read in all the MsgStru1 and variant fields */
	if(msg2ScanMsg1(f, aS(lng)) != 0)
		return 0;

	rv = 0;

/* 2. calculate the strength of the current definition */
	/* All except STR_ANY must match in the charset */
	/* set bit0 as we are at the current catalogue superceeding previous
		catalogues */
	strength = STR_FIRST | 1;
	for(i = lng.msg1.ms1_charsets; i--;)
		if(msgMatchCharset(lng.charsets[i])) {		/* found */
			/* Now try to match the language code */
			if(msgMatchLngcode(lng.msg1.ms1_lng))
				strength = STR_EXACT | 1;
			else {
				/* Try to match the alias language codes, otherwise
					default to "matching charset" */
				strength = STR_CHARSET | 1;
				for(i = lng.msg1.ms1_aliases; i--;)
					if(msgMatchLngcode(lng.aliases[i])) {
						strength = STR_ALIAS | 1;
						break;
					}
			}
			break;
		}

/* 3. Check all included groups, if the current language code superceeds
	the already found definition */
	for(i = lng.msg1.ms1_groups; i--;) {
		msg2 = sIdx(lng.msg2, i);
		if(msg2->ms2_gid >= MSGGROUPMAX)
			goto errRet;
		grp = aS(GCAT groups[msg2->ms2_gid]);
		if(grp->strength < strength) {	/* OK, superceeded this group */
			grp->strength = strength;
			grp->entries = msg2->ms2_num;
			longcpy(grp->pos, msg2->ms2_ms3);	/* already absolute pos */
			f1 = grp->file;
			grp->file = fd;
			msg_closefd(PAS_ARG f1);
		}
	}

	rv = 1;

errRet:
	free(lng.aliases);
	free(lng.charsets);
	free(lng.msg2);

	return rv;
}


int msg2ScanLngDefs(DCL_GCAT MSGFILE f, dword iM(*)start, dword iM(*)limit)
{	MSGFILE fd;		/* duplicated file that is stored within msgCat */
	MsgStruMN mmn;
	int cnt;
	dword pos;

	assert(aS(msg_glbCat));
	assert(f != MSGNOFILE);

	fd = msg_dupfd(PAS_ARG f);

	/* reset the bit0 of all strengths, as the catalogue file does no
		longer contain the *current* catalogue's definitions */
	for(cnt = MSGGROUPMAX; cnt;)
		GCAT groups[--cnt].strength &= ~1;

	longcpy(pos, start);
	/* cnt = 0;	for() sets cnt := 0 */

	while(longcmp1(pos, limit) < 0) {	/* successfully read all the chain? */

		if(!Msetpos(f, pos)		/* goto next item in chain */
		 || !MSread(f, mmn)		/* just read the magic number and next field */
		 || !Msetpos(f, pos)	/* reset to start of structure */
		 || !msgValidMagicNumber(mmn.mmn_magicnumber))
			goto errRet;

		switch(mmn.mmn_magicnumber) {
		/* ignore all other valid magic numbers */
		case MSG_MN_STRU0:		/* end of chain reached */
			goto ret;

		case MSG_MN_STRU1:		/* one language definition */
			if(fetchLngDef(PAS_ARG  f, fd)) {
				--cnt;
				break;
			}
			goto errRet;
		}

		if(!longtst(mmn.mmn_next))	/* check if next item is available */
			break;
		longadd(pos, mmn.mmn_next);	/* calculate addr of next item */
	}
ret:
	cnt = -cnt;

errRet:

	msg_closefd(PAS_ARG fd);		/* close the duplicate if unused */

	return cnt;
}
