/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: MSGPRNTF.INC $
   $Locker: ska $	$Name:  $	$State: Rel $

	Due to the useage of the MR depend msgLock() function, these functions
	must be generated once per access method.

    Target compilers:
    	+ Micro-C 3.14
    	+ Borland C++ v3.1+

	Emit message strings:
	 special syntaxes within format string:
	 	+ %-%* insert special format *, following formats are currently
	 		available:
	 		+ m<MSGID>: include message string right at this place,
	 					<MSGID> is in binary form and may contain NUL's.
	 		+ M:		take one argument of the type (MSGID) and include
	 					this message string.
	 		+ r<repeat><char>: both are 1 byte wide: insert the character
						<char> <repeat> times.
	 		+ R<repeat>: <repeat> is 1 byte wide: take one argument of the
	 					type (int) and insert it <repeat> times.
			+ A<count>???: <count> is 1 byte wide: Number of characters
						in ???. ANSI escape sequence enclosure.
			+ W<count>???: <count> is 1 byte wide: Number of characters
						in ???. Raw byte enclosure.
		+ %fmt		(fmt == normal format specification)
			=> is replaced by the next argument, which has the specified 
			   type
			fmt 'b' == binary is added to non-Micro-C
			fmt 's' == string is re-implemented for Micro-C
			fmt 'X' == upper-cased hex is added to Micro-C
			fmt 'x' == lower-cased hex is added to Micro-C

		+ %#%fmt	(# == decimal; fmt == format specification as printf() would accept)
			=> is replaced by the #th argument, which has the specified
			   type; first argument with # == 1
		+ %-#%fmt	(# == decimal; is number of the skipped argument)
			=> defines, that the #th argument has this type, but is ignored
			   fully; first argument with # == 1

		+ there can be multiple statements with the same number for:
		  %#%fmt and %-#%fmt, but always the last is used for defining
		  the argument size.
		+ the "next argument" is the first undefined argument, e.g.:
		  "...%1%s...%s...%2%s"
		  ==> both %s and %2%s expand to the string given as the 2nd argument
		  whereas:
		  "...%s..%2%s..%d"
		  ==> %s is 1st, %2%s is 2nd, %d is 3rd
		  and
		  "...%2%s...%s..%d"
		  ==> %2%s is 2nd, %s is 1st, %d is 3rd
		  and
		  "..%1%s..%1%p..%1%u..%1%c..%s"
		  ==> in near data model: %1%s, %1%p, %1%u,& %1%c will define the
		      first argument as 2 bytes wide
		  ==> in far data model: 
		  		arg2-byte 3
		  		arg2-byte 2
		  		arg2-byte 1 \
		  		arg2-byte 0  \(3)
		  		arg1-byte 3  /    \
		  		arg1-byte 2	/      \(2)
		  		arg1-byte 1 \ (1)  /
		  		arg1-byte 0 /     /

		  		(1) used by %1%u,& %1%c
		  		(2) used by %1%s,& %1%p
		  		(3) used by %s, because the 2 bytes-wide definition for
		  		    argument 1 is define last (with %1%c).


	 examples: (double quotes shall mark the C string)
	 	"disk %c: is full"

	 	Three-argument message:
	 	"Disk %c: has the label '%s' and is %s"
		If this message should be altered to:
		"Disk %c: is %s"	the second argument is given to the message() function,
							but should be avoided.
		=> correct: "Disk %c: is %3%s%-2%s"
		                             ^^^^^ defines argument 2 as char*, but is replaced by ""
		                         ^^^^ is replaced by the third argument

		"Disk with label '%-%A\4\33[7m%s%-%A\4\33[0m' is full"
					      |     |     | |     ^^^^^^ ANSI escape sequence
						  |     |     | ^^^^^^ define, that next 4 characters are ANSI
						  |     |     ^^ normal format string to emit a char*
					      |     ^^^^^^ ANSI escape sequence
						  ^^^^^^ define, that next 4 characters are ANSI



	Algorithm:
	The normal printf() function determines the length of each argument
	by the supplied format string. When the arguments may differ in
	length, it is impossible to calculate the address of an unread argument.
	There are three ways to come along with that restriction:
	1) Write a brand-new printf() function.
	   That would require to much overhead right now and is non-portable,
	   because of the usage of floating point numbers.
	2) Define, that all arguments must have the same length.
	   Hmm, that's sounds unacceptable, because the only reason to
	   use a non-Micro-C compiler is to use its advantages over
	   Micro-C, e.g. floating point arethmetics, far data model.
	3) Prescan the format string and do the swapping with that knowledge.
	   Seems the best here, therefore it is implemented. (Note: The limit
	   of one argument per message was refused, which would be another way.)

	How to realize the argument swapping?
	3.1) Swap the space allocated for ach argument within the stack.
	     There are optimizations, which can re-use that arguments on the
	     stacks -> refused.
	3.2) Duplicate the necessary space, then 3.1).
	     To use dynamic memory is not wise here, therefore the allocated
	     memory had to be within the stack or, by the defining the maximal
	     amount of arguments, is the worst-case (MAXARG * sizeof(long double))
	     as a static buffer. Refused because:
	     a) Maybe, there is a longer type than long double somewhere.
	     b) The format string must be readible by the original
	        printf() function. The argument swapping itself could be
	        stored away from the format string, but not the colour
	        support.
	     Therefore, the format string itself had to be copied
	     => not acceptable without dynamic memory.
	3.3) Call printf() for each argument.
	     Two more advantages:
	     a) Allow to add new format specifications, e.g. %b for consistency
	        between Micro-C and non-Micro-C.
	     b) Handle more than 132 bytes long output in Micro-C.



	Special notes to the Micro-C implementation:
		In Micro-C all arguments have the same length (2 bytes). Therefore
		it is not necessary to calculate the addresses through the format
		string.



	Known bugs:
		+ on non-Micro-C: the %b format was added for consistency purpose
		  to Micro-C. It supports (unsigned) and (unsigned long). The
		  size modifier 'l' must immediatly preceed the 'b'. The "blank
		  when zero" option is not implemented. There is no alternate form.
		+ on Micro-C: the %s format was re-implemented. Therefore, to enable
		  the "Fill leading space with zeros" option the zro must immediatly
		  follow the '%', e.g. "%0123s".

*/


#include <stdio.h>
#ifndef _MICROC_
#include <string.h>
#include <stdarg.h>		/* definition of type va_list */
#include <io.h>
#define skfwrite(buf,len,fp) fwrite(buf,1,len,fp)
#else /*#	!(!defined(_MICROC_)) */
#include <file.h>
#define skfwrite(buf,len,fp) fput(buf,len,fp)
#endif /*#	!defined(_MICROC_) */
#include <portable.h>
#include "yerror.h"

#ifdef __TURBOC__
	/* Disable: "Mixing pointer of different char type" warning */
#pragma option -w-ucp
#endif /*#	defined(__TURBOC__) */


/*ska better no error report 
   ANSI defines that an corrupt format string is *implementor defined*;
   so I define: ignore it most times, but return with EOF if it's a 
   fatal error.

   int __formatStringError(const char *str);
*/

#define MAXARG NL_ARGMAX		/* maximal number of arguments */
#define FL_DASH ((unsigned)1 << (sizeof(unsigned) * 8 - 1))
#define FL_MISC (FL_DASH >> 1)


/*
	Message enclosures require some sort of recursation (or an
	iteration). Also there are functions that need to know about
	the output destination.

	Therefore, all variables helding information that is not recursion-
	depend is put into a structure. That will also reduce the amount of
	pointer generations when invoking functions like chkLen() and
	dumpStr().

	This structure could be made static lossing the possiblity of
	threading this function.

*/

struct SKPRINTF {
	union {
		byte *sk_cpoi;		/* pointer to (char) output buffer */
		FILE *sk_fpoi;		/* pointer to output file */
	} sk_p;
	unsigned sk_size;		/* size of output buffer or 0 for file */
	unsigned sk_cnt;		/* output counter */
	unsigned sk_idx;		/* index of non-numbered arguments */
	const unsigned *sk_args;		/* pointer to arguments */
	FLAG sk_ansi;			/* !=0 if output ANSI enclosures */
#ifdef _MICROC_
	byte sk_arg[MAXARG];/* indicate, that argument [n] was already used
							   for shifting arguments only */
#else /*#	!(defined(_MICROC_)) */
	unsigned sk_argm[MAXARG];	/* arg idx ==> swapped to arg argMap[idx] */
	unsigned sk_arg[MAXARG], *sk_as;	/* points to next argument slot */
#endif /*#	defined(_MICROC_) */
};
#define Sref var->
#define Scpoi	Sref sk_p.sk_cpoi
#define Sfpoi	Sref sk_p.sk_fpoi
#define Ssize	Sref sk_size
#define Scnt	Sref sk_cnt
#define Sansi	Sref sk_ansi
#define Sarg	Sref sk_arg
#define Sidx	Sref sk_idx
#define Sargs	Sref sk_args
#define Sargm	Sref sk_argm
#define Sas		Sref sk_as


#ifdef _MICROC_
							/* prescan & do-duty function */
#define skdomsg(fmt) skdomsg_(fmt, var)
static int skdomsg_(unsigned char *fmt, struct SKPRINTF *var);
#else /*#	!(defined(_MICROC_)) */
							/* prescan function */
#define skdomsg1(fmt) skdomsg1_(fmt, var)
static int skdomsg1_(unsigned char *fmt, struct SKPRINTF *var);
							/* do-duty function */
#define skdomsg2(fmt) skdomsg2_(fmt, var)
static int skdomsg2_(unsigned char *fmt, struct SKPRINTF *var);
#endif /*#	defined(_MICROC_) */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSGPRNTF.INC 1.6 1998/10/07 04:42:53 ska Rel ska $";
#endif /*#	defined(RCS_Version) */


/*
 * What to do with ANSI escape sequence enclosures?
 */
static int ansiTest = ANSI_TEST;

int msgGetAnsiFlag(void)
{	return ansiTest;	}
void msgSetAnsiFlag(int flag)
{	ansiTest = flag;	}


/* 
   arg[] holds the offset of the arguments as they are supplied on the
		 caller's function parameter list.

   argMap[] specifies, that the idx'th argument of the format string
		is the argMap[idx]'th argument within the arg[] array.
		This is necessary for the "next arguments" without the
		preceeding %# and for unused arguments.

		While emitting the arguments, argMap[] holds the corresponding
		offsets fromout arg[].
*/


/* The colour support

	bases upon ANSI to allow to use the stream functions. Further
	implementations may also differ between ANSI/direct/no colour.

	ANSI is necessary for redirecting (remote access), direct for 
	colour without an ANSI driver, and none for redirecting into
	a file. When skprintf() is used as sprintf(), ANSI sequences
	are NOT included!

	To differ between ANSI/non-ANSI the current implementation uses
	the global variable ansiTest:
		ANSI_TEST: The function will use ANSI only , if the output stream
			is connected to a CON device.
		ANSI_ALLOW: Use ANSI ever.
		ANSI_DENY: Use ANSI never.


	The test is made with the isatty() function.

*/

#ifndef _MICROC_
#define adjustOffsets() adjustOffsets_(var)
static int adjustOffsets_(struct SKPRINTF *var)
{	int size, flags, anr, nr;

	/* evaluate offsets of the arguments in the original order */
	for(size = flags = 0, anr = -1; ++anr < MAXARG;) {
		if((nr = Sarg[anr]) != 0) {
			if(flags) /* with a missing argument, offset not calculatable */
				/* return __formatStringError(fmt); */	/* a missing argument */
				return 1;	/* unable to evaluate the argument address */
		}
		else flags = 1;

		Sarg[anr] = size;
		size += nr;
	}

	/* 2.1. map indexes to offsets, so argMap[] points correctly */
	while(--Sas >= Sargm) {
		*Sas = Sarg[*Sas];
	}

	return 0;		/* OK */
}
#endif /*#	!defined(_MICROC_) */


#define chkLen(len) chkLen_(len, var)
static void chkLen_(unsigned sLen, struct SKPRINTF *var)
{	
	if(sLen >= Ssize) 		/* error, because local buffer overflow may */
		fatal(E_msgBuffer);					/* corrupted the stack */
	Ssize -= sLen;
}

#define dumpStr(str,sLen) if(dumpStr_(str, sLen, var)) \
							return 1
static int dumpStr_(char *str, unsigned sLen, struct SKPRINTF *var)
{	Scnt += sLen;
	if(!Ssize) 	/* dump into stream */
		return skfwrite(str, sLen, Sfpoi) != sLen;

	/* dump into buffer */
	chkLen(sLen);				/* string fits into buffer? */
	memcpy(Scpoi, str, sLen);
	Scpoi += sLen;
	return 0;
}


int skprintf(unsigned char *poi, unsigned len, char *fmt, const unsigned *args)
/* poi: output stream
   len: length of the output buffer:
   	== 0 => output buffer is stream (poi is FILE*)
   	else => output buffer is memory (terminating NUL is added)
   		the buffer is filled with up to (len-1) bytes
   		is used to indicate, how much space is left in the buffer
   fmt: enhanced format string
   args: pointer to the first argument

   return: number of written characters or EOF on failure
*/
{	struct SKPRINTF var;

	memset(aS(var), 0, sizeof(var));

	switch(ansiTest) {
	case ANSI_TEST:		/* check if output is stream && is TTY */
		var.sk_ansi = !len && isatty(fileno((FILE*)poi));
	case ANSI_DENY:		/* already zero */
		break;
	default:
		var.sk_ansi = 1;
		break;
	}
	var.sk_p.sk_cpoi = poi;
	var.sk_size = len;
	var.sk_args = args;

#ifdef _MICROC_
	return skdomsg_(fmt, aS(var))? EOF: var.sk_cnt;

#else /*#	!(defined(_MICROC_)) */
	var.sk_as = var.sk_argm;

	/* 1. Prescan */
	if(skdomsg1_((byte*)fmt, aS(var)))
		return EOF;					/* error in prescan */

	/* 2. Evaluate argument offsets; first argument starts with 0 */
	if(adjustOffsets_(aS(var))) {		/* error? */
		dumpStr_(fmt, strlen(fmt), aS(var));
		return var.sk_cnt;
	}

	/* 3. Perform the output */
	var.sk_idx = 0;

	return skdomsg2_((byte*)fmt, aS(var))? EOF: var.sk_cnt;
#endif /*#	defined(_MICROC_) */
}



int fputmc(int c, int rep, FILE *fp);
#define dumpFiller(c,rep) if(dumpFiller_(c, rep, var)) \
							return 1
static int dumpFiller_(char c, unsigned rep, struct SKPRINTF *var)
{	Scnt += rep;
	if(!Ssize) 	/* dump into stream */
		return fputmc(c, rep, Sfpoi);

	/* dump into buffer */
	chkLen(rep);						/* string fits into buffer? */
	memset(Scpoi, c, rep);
	Scpoi += rep;
	return 0;
}

#ifdef _MICROC_

	/* Special for Micro-C

		+ argUsed[] indicates used arguments to skip for
		  the "next argument" calculation.

		+ The whole part of determining the offsets and swapping is
		  cut, because all arguments are two bytes wide.

		+ The %s-format is not passed to printf().
			That enables to output strings longer than 130 bytes or so.

		+ printf() is not called via vprintf(), but by duplicating
		  the current argument.

		+ the long specs (%#l?) are mainly handled by ltoa(), unfortunately
			this function is unsigned only. To allow signed decimals
			a temporary variable is needed. The array size of this variable
			depends on the LSIZE macro and defaults to 4 if absend.
	*/

#ifndef LSIZE
#define LSIZE 4
#endif /*#	!defined(LSIZE) */


#define justify(string,preceed,filler) 							\
	if(justify_(string, preceed, filler, nr, flags & FL_DASH	\
	, var))				 										\
		return 1

static int justify_(char *str, unsigned preceed, int filler, int nr, int left
 , struct SKPRINTF *var)
{	int size;

	size = strlen(str);
	if(preceed) {
		if(nr) --nr;
		if(!isspace(filler)) {		/* preceed preceeds even fillers */
			dumpFiller(preceed, 1);
			preceed = NUL;
		}
	}

	nr = nr > size? nr - size: 0;

	if(nr && !left)
		dumpFiller(filler, nr);

	if(preceed)
		dumpFiller(preceed, 1);

	dumpStr(str, size);

	return (nr && left) ? dumpFiller_(' ', nr, var) : 0;
}


static int skdomsg_(unsigned char *p, struct SKPRINTF *var)
{	unsigned char *s;		/* pointer into the format string */
	unsigned arg;			/* all possible arguments fit into this var */
	int anr, nr;			/* argument index, arbitary numbers */
	unsigned flags;
	FLAG skip;				/* 1: skip the current item */
	unsigned preceed;		/* character preceeding a justify field */
	int radix;				/* radix of integer number to dump */
	byte Xbuf[1 + 8 * LSIZE];
	char longArg[LSIZE];
	int filler;				/* filler depending on format string */

	--p;

	while(p = strchr(s = p + 1, '%')) {	/* at least one special */

		if(s != p) {	/* emit constant text */
			dumpStr(s, p - s);
			s = p;		/* mark beginning of format item */
		}

		anr = Sidx;		/* argument index which has been used up */
		arg = Sargs[-anr];
		filler = skip = flags = nr = 0;
		do switch(*++p) {
		case '%':
			if(p[-1] == '-') {
				if(p[-2] == '%') {	/* special %-%* formats */
					switch(*++p) {	/* all * formats has the format ID right in the first place */
					case 'r':	/* r<repeat><char> */
						dumpFiller(p[2], p[1]);
						p += 2;
						goto noArg1;
					case 'R':	/* R<repeat> */
						++p;	/* skip onto <repeat> byte */
						if(!skip) {
							dumpFiller(arg, *p);
						}
						goto Arg1;
					case 'm':	/* m<MSGID> */
						p += 2;				/* skip MSGID */
#ifndef MSG_MI_STANDALONE
						if(!skip) {			/* display message */
							nr = skdomsg(msgLock(*(MSGID*)(p - 1)));
							msgUnlock(*(MSGID*)(p - 1));
							if(nr) return 1;		/* failed */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						/*FALL THROUGH*/
					default:
						/* quirk => just ignore %-% and restart with next char */
						goto noArg1;
					case 'M':			/* message include by argument */
#ifndef MSG_MI_STANDALONE
						if(!skip) {				/* display message */
							nr = skdomsg(msgLock(arg));
							msgUnlock(arg);
							if(nr) return 1;		/* failed */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						goto Arg1;				/* one argument used */
					case 'A':	/* A<count>* ANSI enclosure */
						if(!Sansi) skip = 1;
						/*FALL THROUGH*/
					case 'W':	/* W<count>* Raw data enclosure */
						nr = *++p;			/* number of characters to dump */
						if(!skip) {
							dumpStr(p + 1, nr);
						}
						p += nr;
						goto noArg1;
					}
				}
				else { /* special %#-%XXXX ignore! */
					p += nr;
					goto noArg1;	/* did not consumed an arg */
				}
			}
			else if(p[-1] == '%') {	/* special %% */
				dumpStr(p, 1);
				goto noArg1;
			}
			else if((flags & ~FL_DASH) == 0) {	/* special %[-]#% => argument switching/suppressing */
				if((anr = nr - 1) > MAXARG - 1)
					/* return __formatStringError(p); */
					return EOF;
				arg = Sargs[-anr];		/* the argument itself */
				nr = 0;					/* recieves the field length */
				s = p;					/* save start of format spec */
				if(!filler && s[1] == '0')
					filler = '0';
				if(flags & FL_DASH)		/* skip current item ? */
					skip = 1;
				continue;
			}

		default:
			/* return __formatStringError(s); */
			goto noArg1;

		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			nr = nr * 10 + *p - '0';
			continue;

		case '-': flags |= FL_DASH; continue;

		/* The misc flags are of no interest */

		case 'l':	/* a long modifier can be applied to (int) */
			++p;	/* skip to format specifier */
			if(skip) goto Arg1;

			/* unfortunately lota() handles unsigned long only */
			preceed = NUL;
			longcpy(longArg, arg);
			switch(*p) {
			case 'X': case 'x': radix = 16; break;
			case 'b': radix = 2; break;
			case 'o': radix = 8; break;
			default:			/* signed int */
				if(longshl(longArg)) {			/* negative! */
					preceed = '-';
					longset(longArg, 0);
					longsub(longArg, arg);
				}
				else longcpy(longArg, arg);
				/*FALL THROUGH*/
			case 'u':
				radix = 10;
				break;
			}

			ltoa(longArg, Xbuf, radix);
			if(*p == 'x') strlwr(Xbuf);
			justify(Xbuf, preceed, filler);

			goto Arg1;

		case 'd':
			if((int)arg < 0) {			/* negative argument */
				preceed = '-';
				arg = -(int)arg;
				radix = 10;
				goto intFrm;
			}
			/*FALL THROUGH*/
		case 'u': radix = 10; goto intFrm2;
		case 'o': radix = 8; goto intFrm2;
		case 'b': radix = 2; goto intFrm2;
		case 'X':
		case 'x': radix = 16;
		intFrm2:
			preceed = NUL;

		intFrm:
			if(skip)	/* do not display this argument */
				goto Arg1;
			
			itoa(arg, Xbuf, radix);
			if(p[-1] == 'x') strlwr(Xbuf);
			justify(Xbuf, preceed, filler);

			goto Arg1;

		case 'c':

			/* setup arg to point to "\<arg>" using Xbuf */
			*Xbuf = arg;
			Xbuf[1] = NUL;
			arg = Xbuf;
			/*FALL THROUGH*/
		case 's':	/* special string handling */
			if(!skip) {		/* do display this argument */
				justify(arg, NUL, ' ');
			}
			goto Arg1;

		} while(1);

Arg1:
		Sarg[anr] = 1;	/* Mark this arg as used */
		while(Sarg[Sidx])	/* advance to next possible argument */
			if(++Sidx >= MAXARG)
				/* __formatStringError(p); */
				return EOF;
noArg1:	/* no argument was used */

	}

	/* Emit any trailing string */
	if(*s)	{ 						/* There're some chars left */
		dumpStr(s, strlen(s));
	}

	if(Ssize)					/* place the string terminator */
		*Scpoi = NUL;

	return 0;
}


register int fd_sprintf(char *buf, const char *msg)	/* need two args */
{
	asm {
		mov cx, 0fffeh	; != 0: indicating poi is memory buffer
		jmp short callSkprintf
	}
}

register int fd_fprintf(FILE *fp, const char *msg)	/* need two args */
{
	asm {
		xor cx, cx	; 0: indicating poi is file pointer
	callSkprintf:
		shl ax, 1	; amount of bytes used by all arguments
		lea bx, 2[bp]	; address below last arg
		add bx, ax	; address of file pointer
		push Word Ptr [bx]	; 1st arg: file pointer
		dec bx
		dec bx		; address of format string
		push cx		; type of poi: file pointer or memory buffer?
		push Word Ptr [bx]	; formt string
		dec bx
		dec bx		; address of 1st printf()''s arg
		push bx		; addrss of 1st arg for printf()
		call _skprintf
		add sp, 8
	}
}

#else /*#	!(defined(_MICROC_)) */

#define dumpPrintf(fmt,addr) if(dumpPrintf_(fmt, p, addr, var)) \
								return 1
static int dumpPrintf_(char *fmtS, char *fmtE, va_list addr, struct SKPRINTF *var)
{	int c;
	unsigned size;

	c = fmtE[1];
	fmtE[1] = NUL;	/* terminate format string */

	if(Ssize) {	/* dump into buffer */
		chkLen(size = vsprintf(Scpoi, fmtS, addr));
		Scpoi += size;
	}
	else size = vfprintf(Sfpoi, fmtS, addr);

	Scnt += size;
	fmtE[1] = c;	/* restore format string */
	return 0;
}


#define argAt(off) (((byte*)Sargs) + off)
#define mappedArg(idx) argAt(Sargm[idx])

/*
 * Prescan function.
 * Checks which argument points to which offset
 */
static int skdomsg1_(byte *p, struct SKPRINTF *var)
{
	int anr, nr;			/* argument indexes */
	unsigned flags, size;
	FLAG skip;				/* 1: skip current item */

	/* 1. Create the argument lengths */

	--p;

	while((p = strchr(p + 1, '%')) != NULL) {	/* at least one special */
		anr = Sidx;						/* actually used argument */
			/* size: size of current argument
			   nr: arbitary number while scanning numericals
			   flags: what has been scanned between '%' and format char
			   skip: don't display this argument */
		skip = flags = size = nr = 0;
		do switch(*++p) {
		case '%':
			if(flags == FL_DASH && p[-1] == '-') {
				if(p[-2] == '%') {	/* special %-%* */
					switch(*++p) {	/* the format ID always immediately folloing the %-% */
					case 'r':	/* r<repeat><char> */
						p += 2;
						goto noArg;
					case 'R':	/* R<repeat> */
						++p;	/* skip over <repeat> byte */
						size = sizeof(int);	/* use an (int) argument */
						goto Arg;
					case 'm':	/* m<MSGID> */
						p += 2;						/* skip MSGID */
#ifndef MSG_MI_STANDALONE
						if(!skip) {					/* free slot found */
							nr = skdomsg1(msgLock(*(MSGID*)(p - 1)));
							msgUnlock(*(MSGID*)(p - 1));
							if(nr) return 1;				/* error */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						/*FALL THROUGH*/
					default:
						/* quirk => just ignore %-% and restart with next char */
						goto noArg;
					case 'M':					/* message include by argument */
#ifndef MSG_MI_STANDALONE
						if(!skip) {				/* go to included message */
							for(size = nr = 0; nr < anr; size += Sarg[anr++])
								if(!Sarg[anr])	/* hole -> error */
									return 1;
							nr = skdomsg1(msgLock(*(MSGID*)argAt(size)));
							msgUnlock(*(MSGID*)argAt(size));
							if(nr) return 1;			/* error */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						size = sizeof(MSGID);
						goto Arg;			/* one argument used */
					case 'A':	/* A<count>* ANSI enclosure */
					case 'W':	/* W<count>* Raw data enclosure */
					/* there is no test, if the string is less than the 
					   requested length. This will allow '\0's (for what
					   reason ever). */
						nr = *++p;			/* number of characters to dump */
						p += nr;
						goto noArg;
					}
				}
				else { /* special %#-%XXXX */

					/* there is no test, if the string is less than the 
					   requested length. This will allow '\0's (for what
					   reason ever). */
					p += nr;
					goto noArg;	/* did not consumed an arg */
				}
			}
			else if(p[-1] == '%')	/* special %% */
				goto noArg;
			else if((flags & ~FL_DASH) == 0) {	/* special %#% or %-#% => argument switching or suppressing */
				anr = nr - 1;
				if(flags & FL_DASH)		/* skip current item? */
					skip = 1;
				continue;
			}

			/* fall through to quirk */
		default:
			/*!!!ska better no error reporting */
			/* return __formatStringError(p); */	/* format string quirk */
			goto noArg;

		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			nr = nr * 10 + *p - '0';
			continue;

		case '-': flags |= FL_DASH; continue;

		case ' ': case '+': case '#': case '.':
			flags |= FL_MISC;	/* these are valid flags, but for no use now */
			continue;

		/* This flag indicates an additional (int) item used up by this
			argument */
		case '*': ++flags; continue;

		case 'N': case 'F': case 'l': case 'h': case 'L':	/* size modificating prefixes */
			size = *p;
			continue;

		case 'd': case 'i': case 'o': case 'u':
		case 'x': case 'X':
/* Micro-C does not accept #if 0 */
#ifdef NICHT_UEBERGEHEN		/* automatically promoted to  int */
			if(sizeof(short) != sizeof(int)) {	/* I hope, your compiler removes unreachable code */
				if(size == 'h')	{
					size = sizeof(short);
					goto Arg;
				}
			}
#endif /*#	defined(NICHT_UEBERGEHEN) */
			if(sizeof(long) != sizeof(int)) {	/* I hope, your compiler removes unreachable code */
				if(size == 'l')	{
					size = sizeof(long);
					goto Arg;
				}
			}
			/* default: sizeof(int) */

		case 'c': size = sizeof(int); goto Arg;

		case 'f': case 'e': case 'E': case 'g': case 'G':
/* Micro-C does not accept #if 0 */
#ifdef NICHT_UEBERGEHEN			/* float is automatically promoted to double */
			size = size == 'l'? sizeof(double):
				   size == 'L'? sizeof(long double): sizeof(float);
#else /*#	!(defined(NICHT_UEBERGEHEN)) */
			size = size == 'L'? sizeof(long double): sizeof(double);
#endif /*#	defined(NICHT_UEBERGEHEN) */
			goto Arg;
		case 's': case 'p': case 'n':
			if(sizeof(void*) == sizeof(void near*)) {	/* I hope, your compiler removes unreachable code */
				size = size == 'F'? sizeof(void far*): sizeof(void*);
			} else {
				size = size == 'N'? sizeof(void near*): sizeof(void*);
			}
			goto Arg;

		} while(1);

Arg:	/* an argument was specified */

		if(anr > MAXARG-1) 	/* argument anr is out of bound */
			/* return __formatStringError(p); */	/* argument amount overflow */
			return EOF;	/* try to avoid rubish */

		/* the size of the argument and its additional (int) arguments */
		Sarg[anr] = size + (flags & ~(FL_DASH | FL_MISC)) * sizeof(int);

		if(!skip)	/* this argument uses the specified offset */
			*Sas++ = anr;		/* record useage */
		/*  ^^^^^^^^^^^       current argument is mapped to anr */

		if(Sarg[Sidx] != 0)	/* advance to next argument */
			while(++Sidx < MAXARG && Sarg[Sidx]);
			/* allow idx == MAXARG, for exactly MAXARG arguments */

noArg:;
	}
	return 0;					/* all OK so far */
}

/*
 * do-duty function
 * At this point the Sargm maps like this:
 *	The n-th argument of the format string (regardless of any %# prefix)
 *	starts at offset Sargm[n].
 */
static int skdomsg2_(unsigned char *p, struct SKPRINTF *var)
{	unsigned char *s;		/* pointer within the format string */
	int nr;					/* arbitary number */
	unsigned flags;
	int filler;
	FLAG skip;				/* 1: skip current item */

	/* 3. Emit message by using the "original" printf */
	/* here should be no errors anymore */

	--p;

	while((p = strchr(s = p + 1, '%')) != NULL) {	/* at least one special */

		if(s != p) {	/* emit constant text */
			dumpStr(s, p - s);
			s = p;
		}

		filler = skip = flags = nr = 0;
		do switch(*++p) {
		case '%':
			if(p[-1] == '-') {
				if(p[-2] == '%') {	/* special %-%* */
					switch(*++p) {	/* the format ID always immediately folloing the %-% */
					case 'r':	/* r<repeat><char> */
						dumpFiller(p[2], p[1]);
						p += 2;
						goto noArg1;
					case 'R':	/* R<repeat> */
						++p;	/* skip over <repeat> byte */
						if(!skip) {
							dumpFiller(*(char*)mappedArg(Sidx), *p);
						}
						goto Arg1;
					case 'm':	/* m<MSGID> */
						p += 2;						/* skip MSGID */
#ifndef MSG_MI_STANDALONE
						if(!skip) {					/* free slot found */
							nr = skdomsg2(msgLock(*(MSGID*)(p - 1)));
							msgUnlock(*(MSGID*)(p - 1));
							if(nr) return 1;				/* error */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						/*FALL THROUGH*/
					default:
						/* quirk => just ignore %-% and restart with next char */
						goto noArg1;
					case 'M':					/* message include by argument */
#ifndef MSG_MI_STANDALONE
						if(!skip) {				/* go to included message */
							nr = skdomsg2(msgLock(*(MSGID*)mappedArg(Sidx)));
							msgUnlock(*(MSGID*)mappedArg(Sidx));
							if(nr) return 1;			/* error */
						}
#endif /*#	defined(USE_FEAT_MSGRETRIEVER) */
						goto Arg1;			/* one argument used */
					case 'A':	/* A<count>* ANSI enclosure */
						if(!Sansi) skip = 1;	/* if no ANSI available -> skip */
					case 'W':	/* W<count>* Raw data enclosure */
					/* there is no test, if the string is less than the 
					   requested length. This will allow '\0's (for what
					   reason ever). */
						nr = *++p;			/* number of characters to dump */
						if(!skip) {
							dumpStr(p + 1, nr);
						}
						p += nr;
						goto noArg1;
					}
				}
				else { /* special %#-%XXXX */
					p += nr;
					goto noArg1;	/* did not consumed an arg */
				}
			}
			else if(p[-1] == '%') {	/* special %% */
				dumpStr(p, 1);
				goto noArg1;
			}
			else if((flags & ~FL_DASH) == 0) {	/* special %[-]#% => argument switching/suppressing */
				/* anr = nr - 1; */
				/* nr = 0;	not necessary, because field width is generated by printf() */
				s = p;	/* save start of format spec */
				if(!filler && s[1] == '0')
					filler = '0';
				if(flags & FL_DASH)
					skip = 1;
				continue;
			}

		default:
			continue;	/* misc flags and size modifier */

		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			nr = nr * 10 + *p - '0';	/* for ANSI enclosures only */
			continue;

		case '-': flags |= FL_DASH; continue;

		/* The misc flags are of no interest anymore */

		case 'd': case 'i': case 'o': case 'u':
		case 'x': case 'X':
		case 'c': 
		case 'f': case 'e': case 'E': case 'g': case 'G':
		case 's': case 'p': case 'n':

			if(skip)	/* do not display this argument */
				goto noArg1;

			dumpPrintf(s, (va_list)mappedArg(Sidx));

			goto Arg1;

		case 'b':	/* for consistency to Micro-C */
			if(skip)		/* do not display this argument */
				goto noArg1;
			{
				char buf[sizeof(unsigned long) * 8 + 1];	/* assume long */
				char *bufp;
				unsigned long it;

				*(bufp = buf + sizeof(buf) - 1) = NUL; /* end of buf */
				it = p[-1] == 'l'
				 ?  *(unsigned long*)mappedArg(Sidx)
				 :	(long)(*(unsigned*)mappedArg(Sidx));

				if(it) /* create ASCII representation */
					do *--bufp = '0' + ((unsigned)it & 1);
					while(it >>= 1);
				else *--bufp = '0';

				*p = 's';	/* dump a string instead of %b */
				dumpPrintf(s, (va_list)bufp);
				*p = 'b';	/* restore format string */
			}
			goto Arg1;

		} while(1);

Arg1: ++Sidx;
noArg1:;

	}

	/* 4. Emit any trailing string */
	if(*s)	/* There're some chars left */
		dumpStr(s, strlen(s));

	if(Ssize) *Scpoi = NUL;	/* write string terminator */

	return 0;				/* all OK */
}

int fd_fprintf(FILE *fp, char *msg, ...)
{	return skprintf((unsigned char*)fp, 0, msg, (const unsigned*)(&msg + 1));	}

int fd_sprintf(char *buf, char *msg, ...)
{	return skprintf((unsigned char*)buf, 0xfffe, msg, (const unsigned*)(&msg + 1));	}

#endif /*#	defined(_MICROC_) */
