/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    MSGCOMP2 - utility for the message handling library
    Copyright (C) 1995,97,98  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: MSGCOMP2.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	The message compiler 2nd generation.

   ----
	command line usage:

	msgComp2 [{option}] pathGlbMsg lng locHeader [msgDcl [msgFeatures]]


	pathGlbMsg  : path to the global message definition files
	lng			: language to be compiled (message description file)
	locHeader	: local header file that includes the local msg decl file
	msgDcl		: message declarations
	msgFeature	: list of features used within the description file

	If the extension of <lng> is omited, ".MSG" is assumed.

	All local messages are read from <lng>.
	Internally used names are read from both:
			<pathGlbMsg>\<lng>[.msg]
	and		<pathGlbMsg>\english.msg
	But if one word resides in both files, the meaning out of
	<lng>[.msg] takes precedance.
	In the english.msg message description file the internally used
	names are of the form:
		>>>>>>I_internalKeywords
		<number of keywords>
		<keyword> | <comment>
	The <number of keywords> defines how many keywords exist.
	A <comment> is any line that does not start with a digit.
	A <keyword> definition looks like this:
		<numerical ID> <symbolical ID> <string> \n
	The <numerical ID> is the number that is internally used to identify
	this keyword. It must be equal to the #define'ed macro 'IN_????'.
	The <symbolic ID> is the word with what the English keywords are
	linked to the language specific keywords. The <symbolic ID> must
	start with 'IN_'!
	The rest of the line is interpreted as the <string> that identifies
	the keyword. It may contain spaces.
	The I_internalKeywords message may appear more than once within the
	file, but the first line will be ignored except for the first one.

	Options:
		/!	switches the name to "msg!####", because VAL complains about
			multiple modules. So the global messages can be generated
			with this switch.
			This also implies /g=8.
		/b	open the message description files in binary mode. That allows
			to include '\0's and ^Z characters in the file. Although there
			will be made no translation of the "newline" character!
		/f=	Sets the output format to
				1 -> numereous .C source files to be compiled into
					a library (method #1) [default]
				2 -> binary file <lng>.MS2 (method #2)
		/g=	Assigns a message group prefix. Range 0..15. This is the
			most-highest half-byte of the message number.
			Currently assigned groups:
				0: local message
				8: global message
				others: reserved for future use
		/l=	causes to output the library response file into the specified
			file. This file looks like (only applicable if method #1):
				+- msg_0000.obj &
				+- msg_0001.obj &
				   ...
				+- msg_####.obj
		/c=	causes to output the compiler response file into the specified
			file. This file looks like (only applicable if method #1):
				msg_0000.c
				msg_0001.c
				   ...
				msg_####.c
		/m=	causes to output a make dependiency structure into the
			specified file. This file looks like (only applicable if method #1):
				msg_0000.obj : msg_0000.c
				msg_0001.obj : msg_0001.c
				   ...
				msg_####.obj : msg_####.c


   Files:
   	All files will be overwritten without further notice. If they are
   	preceeded by a '+' sign, the new information will be appended.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----

	When generating output format #1:

	The message compiler creates numerous message definition files
	"msg_####.c" containing the message string definitions and the
	header file (msgFile with the extension ".msg") containing the
	message string declarations, the error level definitions, and
	the mapping to the native language (see below).
	These files are garantied to be plain 8-bit ASCII. All characters
	00 through 0x1F are represented by their octal form, e.g.:
		ASCII(24) -> \30
	While all characters except NUL (ASCII(0)) are translated immediately
	before writing them into the output file, NUL is translated while
	reading. This is necessary, because NUL is internally used as the string
	terminator (well, that's the C style); there is only one problem I can
	see so far: The %--%A and %--%W specials treats the next character as
	the delimiter; if one choose the NUL here ...

   ----

   	When generating output format #2 (see msg2lock.c for more details):

   	The message compiler generates a single file of the name:
   		<lng>.MS2
   	It consists of one structure #1 and all associated structures and
   	message strings. The stru #1's next pointer address the byte
   	immediately following the last byte of <lng>.MS2, thus, it
   	can be easily combined with other files.
   	NO structure #0 is included!

   	During the creation of the file two temporary files are created:
   	tempfile1: holds stru #1, stru #2 and all stru #3's [Note: One message
   		declaration file defines the messages of one group.]
   	tempfile2: holds the message strings, not packed!

   	One new internally used message is read out of the global message
   	definition file: I_countrycodes
   	Its body consists of at least one numerical country codes. The first
   	one is the code that specifies the language code, the others define
   	the aliases of that language. It's recommended to place this definition
   	as one of the first.

   	Because format #2 binary message library is to support language
   	switching at startup/runtime of the program, there is the problem of
   	identifiers uniquely associated with the identical message for all
   	the languages.

   	Therefore MSGCOMP supports (updates/creates) an index of the association
   	of the symbolic message IDs with the numerical ones. This file's name
   	defaults to MSGLIB.ID. The message IDs used within this particular
   	binary message library. This file is an ASCII file of the following
   	format:
   		<symMsgID> <numMsgID> <reserved>
   	Any line starting with '#', ':' or a non-printable character (incl.
   	whitespaces) are ignored completely.
   	<symMsgID> is the symbolic, <numMsgID> is the numerical mssage ID.
   	The rest of the line is reserved for future enhancements.

   	The message IDs are temporarily stored within another temporary hash
   	file structure, see msgIDs.h for more details.

   ----

	The message declarations:

	To declare a sequence of preprocessor statements is provided:
	#if MSG_METHOD == 1
		DCL_MSG(msgId)
	#else
		#define msgId mkMsgID(msgGroup, msgNr)
	#endif

	The msgId is the source code identificator of the message as it
	apears in the message head of a message definition file.
	The msgGroup is the group, in which the message is located (see /g
	option).
	The msgNr is a number, that uniquely identifies the message ID within
	its message group. The current implementation allows up to 4095
	(0xFFF) message per group. msgNr is always positive.

   ----

   	The message description file syntax:

   	Each message consists of a message head and a message body. The message
   	head is the whole line of the format: ">>>>>>?_name [#]\n". The message
   	body consists of all lines up to the next message header or the EOF.

   	name :: is the name of the message in English
   	?    :: is the message group and must be a single upper-cased letter
   	#    :: error level (for ? == 'E' only)

	The message body may contain special strings: "%-%*%", where '*' stands
	for a comma delimited list of the following ANSI keywords:
		cls			clear screen
		clreol		clear from cursor to end of line
		
		home		cursor to upper left corner
		up #		cursor up # rows		\
		down #		cursor down # rows		 | neither scroll nor change
		left #		cursor left # columns	 | of the other coordiante
		right #		cursor right # columns	/
		goto # #	move cursor to position row,column (one based!!)

		save		save one cursor position
		restore		restore saved cursor position

	  colour related keywords:
		bright		set attribute: foreground colour highlighted
		blink		set attribute: blinking character
		inverse		set attribute: inverse
		underline	set attribute: underline
		hidden		set attribute: hidden
		normal		unset all attributes
		black		foreground colour: black 
		red			foreground colour: red 
		green		foreground colour: green 
		yellow		foreground colour: yellow 
		blue		foreground colour: blue 
		magenta		foreground colour: magenta 
		cyan		foreground colour: cyan 
		white		foreground colour: white 

		Those keywords may be combined in any order WITHOUT a delimiting
		comma. If more than one colour is given, the first is treated as
		a foreground colour, all others as background colours.

	These keywords must be defined in the global message file within
	message group 'I' == internal or in the global English message file.


	The special string: %--% includes the following functionality:
		%--%A?****?
			ANSI enclosure. **** is any string that is emitted, if the
			output of ANSI sequences is allowed. ? is any character that
			is not contained within ****. The string may be longer than
			255 characters, but it will be broken into pieces smaller
			than 256 characters.
		%--%W?****?
			Raw data enclosure; as %--%A, but the string is always emitted.
		%--%r?###%
			Output the single character ? ### times. Although ### may be
			greater than 255, it will be broken into pieces less than
			256.
		%--%R###%
			As above, but the character to be emitted is an argument of the
			type (int).
		%--%m****%
			Recursively include the message with the ID ****. The text of
			the message **** is interpreted, as if it would replace this
			special token, that means the "%1%fmt" means the same argument
			in both the including and the included message.
		%--%M
			As above, but the message to be included is specified by one
			argument of the type (MSGID). Because the included message may
			also contain argument specifications all these argument should
			preceed all other arguments in both conjunctions the argument
			type declaration and as parameter, e.g.:
				>>>>>>M_include
				%-1%MAny arguments %s %d, and the message %1%M, and other
				arguments %s %c.
				>>>>>>M_includedMessage
				%s

				To output the message use that parameter sequence:
					<message to be included>, <para of 1st %s>, <para of %d>,
					<para of %s of M_includedMessage>,
					<para of 2nd %s>, <para of %c>
				e.g.:
					smessage(M_include, M_includedMessage, "s1", 1,
						"s of included msg", "s2", 'c');
				If M_includedMessage would contain "%1%s", "s1" would
				be emitted.

   ----

	Because the message body is transformed into a C string definition,
	any non-printable characters are replaced by their C-style and already
	enterred C-style is interpreted by the C compiler, the last '\n' is
	stripped, e.g.:

>>>>>>M_dummy
	line\\	1
	line\r	2
>>>>>>

	is transformed into:

	"\tline\\\t1\n\tline\r\t2"


   ----
	
	Known bugs:
		Most compilers have problems with '\0' characters within
		the IN stream. Therefore, it's not generally possible to
		place them e.g. in the %#-%??? special. Also, because of
		the text mode reading '\x1a' (^Z) is not allowed as a
		legal character.

		The string attached to the %#-%??? special is not
		processed in any way, especially no C-style syntax is
		translated here.

   ----
	
*/

#define MSG_METHOD 2

#include <assert.h>
#include <stdio.h>
#ifndef _MICROC_
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#else /*#	!(!defined(_MICROC_)) */
#include <file.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>
#include <suppl.h>
#include <dfn.h>
#include <getopt.h>

#include "saveMem.h"
#include "fileIO.h"
#include "msgIDs.h"
#include "copyF.h"

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#include "yerror.h"

#include "../msgconf2.h"		/* method #2 structures */


#ifndef NUL
#define NUL '\0'
#endif /*#	!defined(NUL) */
#define BUF 2048 	/* buffer size while copying a file */

#include "keywords.inc"

#define MSG_ID_INTERNAL 'I'		/* prefix of internal used messages */
#define MSG_ID_ERROR 'E'		/* prefix of error messages */
#define GLOBAL_GROUP 8		/* group of global messages */

struct DumpMsgID {		/* where to dump msgID mappings to */
	FILE *fp;
	const char *fn;
};

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSGCOMP2.C 2.9 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */


#ifndef NDEBUG			/* suppress TEST if no debugging */
//#define TEST			/* TEST TEST TEST */
#endif /*#	!defined(NDEBUG) */


#ifdef TEST
FILE *LOG = NULL;
#endif /*#	defined(TEST) */

char fmode[] = "rt";						/* open mode for message file */


/* opening of all message definition files */
const char msgOpening[] = "\
#ifndef _MICROC_\n\
#include <algnbyte.h>\n\
#else \n\
#define const \n\
#endif\n\
#include \"%s\"\n";						/* local message header file */
										/* needed for message includes */

const char msgErrOpening[] = "\
static unsigned char errNr = %u;\n";	/* message error number */

const char msgMsgDef[] = "\
const unsigned char %s[] = {\n";		/* message head */

/* closing of all message definition files */
const char msgClosing[] = "\
'\\0' } ;\n";

/* Id of special enclosures */
const char specId[] = "\
%%-%%%c";

/* include of a message id:
	1. line: finish previous string portion
	2. line: include message ID
	3. line: declare next string portion
	The 2. and 3. lines are unified by contiguous numbers to allow
	more than one message include. */
const char msgInclude[] = "\
};\n\
const static MSGID mid%u = %s;\n\
const static unsigned char sid%u[] = {\n";

/* how to declare a message */
char msgDCL[] = "\n\
#if MSG_METHOD == 1\n\
DCL_MSG(%s)\n\
#else\n\
#define %s mkMsgID(%u,%u)\n\
#endif\n";

/*
 *	Buffer for internal commands
 *	internal[] is for language specific keywords
 *	internalE[] is for English keywords
 *
 *	The keywords are identified in the English message description file
 *	then read from the language specific description file.
 */
char **internal = NULL;
char **internalE = NULL;
unsigned keywords = 0;
const char iKeywords[] = "I_internalKeywords";
const char symID[] = "IN_";			/* how symbolic IDs start */


/*
 *	various default extensions
 */
const char extBak[] = "BAK";		/* backup extension */
const char extMsg[] = "MSG";		/* message file */
const char extDcl[] = "DCL";		/* message declaration file */
const char extFtr[] = "FTR";		/* message feature file */
const char extMethod2[] = "MS2";	/* binary message library */
const char fnMsg[] = "ENGLISH";		/* default filename of msg file */
const char fnMsgID[] = "msglib.id";	/* default num<->sym msgID check file */


char msgDefFile[] = "%smsg_%04x%s";	/* name of the message definition files */
#define padDefFile 5		/* offset of '_' */

char *msgFile; /* extension-less local message file; with space for ext */
char *msgBody = NULL;			/* current message body */
unsigned msgIncCnt = 0;			/* counter within 1 msg def file */
char *locHead;					/* local header file */
unsigned msgDefFileNr = 0;	/* number of message definition files */
unsigned msgDefColumn;		/* counter within a line in file */
char msgDefFileStr[13];

unsigned errNr = 0;			/* error level of current message header */
unsigned lineNr = 0;		/* keep it NOT (long) because of Micro-C */
FLAG binary = 0;
int dclCnt = 0;				/* message counter */
int lastID = -1;			/* last found num msgID */
int group = 0;				/* message group */
int method;					/* output format */
struct MsgStruct3 msg3={0};	/* incomplete structure of previous message */
TempFile *ms2Ctrl;			/* temporary file for the control area */
#define ms2Fp ms2Ctrl->fp
#define ms2Fn ms2Ctrl->fn
TempFile *ms2Strg;			/* temporary file for the string area */

int autoCharSet = 1;		/* automatically detect 7bitASCII character set */
int numCharsets = 0;		/* number of preserved character sets */
word *charsets = NULL;		/* preserved iCharSets message */

/*	Flags, which features has been used; one per file */
const char iFeatures[] = "I_features";
/*	Where to aquire the aliases from; in global file only */
const char iAliases[] = "I_languagecodes";
/*	Where the supported character sets are defined in; one per file */
const char iCharSets[] = "I_charsets";

/*
 *	Features, which are available & how to keep track of it
 */
#define featAnsi features[0]				/* %-%A */
#define featRaw features[1]					/* %-%W */
#define featRep features[2]					/* %-%r */
#define featRepArg features[3]				/* %-%R */
#define featMInc features[4]				/* %-%m */
#define featMIncArg features[5]				/* %-%M */
#define featReorder features[6]				/* %-#%fmt or %#%fmt */
#define featNUL features[7]					/* '\0' within message */
											/* set even if _possibly_
												a NUL is in msg, e.g. %-%m */
#define feat8bit features[8]				/* no pure 7bit ASCII */
#define FEATURES 9

FLAG features[FEATURES] = {0};
char *featDef[FEATURES] = {
	"#define MSG_FEAT_ANSI",
	"#define MSG_FEAT_RAW",
	"#define MSG_FEAT_REP",
	"#define MSG_FEAT_REPA",
	"#define MSG_FEAT_MINC",
	"#define MSG_FEAT_MINCA",
	"#define MSG_FEAT_REORDER",
	"#define MSG_FEAT_NUL",
	"#define MSG_FEAT_8BIT"	};

struct MsgIDs *msgIDs;			/* current available msgID mappings */
dword firstStru3;		/* offset to the first stru3 within the ms2Fp
			file to help to seek to the stru3 of the message, because
			num<->sym msgID mapping information may cause a non-sequential
			write of messages */


/************************************************************************/
/*********** Helper functions for strings *******************************/
/************************************************************************/

/*
 *	Skip whitespaces
 */
char *skipws(char *s)
{	assert(s);

	while(isspace(*s)) ++s;
	return s;
}


/*
 *	Skip to the end of word (one byte behind the word)
 */
char *eow(char *s)
{	assert(s);

	while(*s && *s != '\n' && !isspace(*s)) ++s;
	return s;
}

/*
 *	Skip to the end of line (trailing spaces ignored)
 */
char *eol(char *s)
{	char *e;

	assert(s);

	if((e = strchr(s, '\n')) == NULL)
		e = strchr(s, NUL);
	while(--e >= s && isspace(*e));
	return e + 1;
}

/*
 *	Return the end of a special format
 *	On failure: Terminate with error message
 */
char *eoe(char *p, char id)
{
	assert(p);

	if((p = strchr(p, id)) == NULL)
		error(E_openEncl, id, lineNr);
	return p;
}

char *getNum(char *p, unsigned *num)
{	unsigned n;

	assert(p);
	assert(num);

	p = skipws(p);
	if(!isdigit(*p))
		return NULL;
	n = 0;
	if(*p != '0' || toupper(p[1]) != 'X')
		do n = n * 10 + *p - '0';
		while(isdigit(*++p));
	else if(isxdigit(*(p += 2)))
		do n = (n << 4) + *p - (isdigit(*p)? '0'
		                        : isupper(*p)? 'A' - 10
		                        : 'a' - 10);
		while(isxdigit(*++p));
	*num = n;
	return skipws(p);
}


unsigned copyNumber(char *dest, char **src, char nam)
{	unsigned len;
	char *s;

	assert(dest);
	assert(src && *src);

	len = 0;
	s = *src;
	if(isdigit(*s)) {
		while(*s == '0') ++s;
		while(isdigit(*s)) {
			++len;
			*dest++ = *s++;
		}
		if(!len) {
			*dest++ = '0';
			++len;
		}
	}
	*dest++ = nam;
	*dest++ = NUL;
	while(isspace(*s)) ++s;
	*src = s;
	return 1 + len;
}


/************************************************************************/
/************ Filename related functions ********************************/
/************************************************************************/


/*
 *	Makes sure that the file has an extension attached.
 */
char *attachExt(char *fn, const char * const ext)
{	char *dr, *pa, *fi, *ex;

	assert(fn);
	assert(ext);

	if(!dfnsplit(fn, &dr, &pa, &fi, &ex)
	 || (!ex && (fn = dfnmerge(NULL, dr, pa, fi, ext)) == NULL))
	 	fatal(E_noMem);

	free(dr);
	free(pa);
	free(fi);
	free(ex);

	return fn;
}



char *mkFname(const char *fname, const char *ex)
{	char *dr, *pa, *fi, *fn;

	if(!dfnsplit((char*)fname, &dr, &pa, &fi, NULL)
	 || (fn = dfnmerge(NULL, dr, pa, fi, ex)) == NULL)
	 	fatal(E_noMem);

	free(dr);
	free(pa);
	free(fi);
	return fn;
}

/************************************************************************/
/************ Message file reading functions ****************************/
/************************************************************************/

char *readMsgBody(FILE *fp)
/*	read next lines, until EOF or an message header encountered */
{	char line[80];
	char *p;
	int cnt;
	int c;
#ifdef _MICROC_
	static int waiting = 0;		/* ungetc() not implemented in Micro-C */
	static char wChar;
#endif /*#	defined(_MICROC_) */

	assert(fp);

	free(msgBody);
	msgBody = NULL;
	do {
		p = line - 1;
		cnt = 1;
		++lineNr;
		while((c = iM(waiting? (waiting = 0, wChar):) fgetc(fp)) != EOF
		 && (*++p = c) != '\n')
			if(p > line + 78) { 
				p[1] = NUL;
				addStr(msgBody, line);
				p = line - 1;
			} 
			else if(c == '>') {
				if(cnt && ++cnt == 7) {	/* message header found */
					p[-5] = NUL;	/* cut out '>>>>>>' */
					c = EOF;	/* stop reading */
					break;
				}
			}
			else {
				cnt = 0;	/* no message header */
				if(binary) {
					if(c == NUL) {	/* translate to \0 */
						featNUL = !NUL;
						*p = '\\';
#ifndef _MICROC_
						ungetc('0', fp);
#else /*#	!(!defined(_MICROC_)) */
						waiting = wChar = '0';
#endif /*#	!defined(_MICROC_) */
					}
					else if(c == '\r') {
						/* find 'newlne' cod.
							As a newline two character strings are accepted:
								[{ \r }] \n
							and { \r }
							both are replaced by a single \n
						*/
						while((c = fgetc(fp)) == '\r');
						*p = '\n';
						if(c != '\n')
#ifndef _MICROC_
							ungetc(c, fp);
#else /*#	!(!defined(_MICROC_)) */
							waiting = 1;	/* for c == NUL */
							wChar = c;
#endif /*#	!defined(_MICROC_) */
						break;
					}
				}
			}

		p[1] = NUL;
		addStr(msgBody, line);
	} while(c != EOF);

	/* strip last '\n' */
	if(*(p = strchr(msgBody, NUL) - 1) == '\n')
		*p = NUL;
	return msgBody;
}

char *readMsgHeader(FILE *fp)
/* treat the next line as a message header, the '>>>>>>" are already
   read */
{	static char msgHeader[40];
	char *p;
	int c;

	assert(fp);

	*(p = msgHeader) = NUL;
	while((isalnum(c = fgetc(fp)) || c == '_') && p < &msgHeader[39])
		*p++ = c;
	*p = NUL;

	while(c != '\n' && isspace(c))
		c = fgetc(fp);

	if(*msgHeader == MSG_ID_ERROR
	 || *msgHeader == MSG_ID_INTERNAL) { 	/* scan error level */
		if(isdigit(c)) {
			errNr = 0;
			do errNr = errNr * 10 + c - '0';
			while(isdigit(c = fgetc(fp)));

			while(c != '\n' && isspace(c))
				c = fgetc(fp);
		}
		else if(*msgHeader == MSG_ID_INTERNAL)
			errNr = -1;
		else
			error(E_msgFileHeadErrLvl, lineNr);
	}

	if(*msgHeader == NUL && c == EOF)
		return NULL;

	if(*msgHeader < 'A' || *msgHeader > 'Z' || msgHeader[1] != '_' 
	  || c != '\n' && c != EOF)
		error(E_msgFileHeadStruc, lineNr);

	readMsgBody(fp);	/* scan in the body and supply it in the global msgBody */
	return msgHeader;
}

/************************************************************************/
/************ Message file writing functions ****************************/
/************************************************************************/

static const char Cfrom[] = "abcdefghijklmnopqrstuvwyz";
static const char Cto[] = {
	 '\a', '\b', '\c', '\d', '\e', '\f', '\g', '\h', '\i', '\j'
	,'\k', '\l', '\m', '\n', '\o', '\p', '\q', '\r', '\s', '\t'
	,'\u', '\v', '\w', '\y', '\z', '\0' };
static const char Cquote[] = "\'\\";

/*
 *	Write a single character and check if it's NUL
 */
void writeChar(int ch, FILE *fp)
{	const char *p;

	assert(fp);

	switch(method) {
	case 1:
		putc('\'', fp);
		if(ch < ' ' || strchr(Cquote, ch))
			putc('\\', fp);
		if(ch < ' ') {
			if(ch && (p = strchr(Cto, ch)) != NULL)
				putc(Cfrom[p - Cto], fp);
			else
				fprintf(fp, "%o", ch);
		}
		else
			putc(ch, fp);
		fputs("', ", fp);
		if(ch == '\n' || ++msgDefColumn >= 8) {
			putc('\n', fp);
			msgDefColumn = 0;
		}
		break;
	case 2:
		putc(ch, fp);
		break;
	}

	if(ch >= 0x7f)
		feat8bit = 1;		/* no 7bit ASCII character */
	else if(!ch)
		featNUL = !NUL;
}


void writeMsgBody(FILE *fp, byte *s, int length)
/* Write the string s onto the file fp and transform all control characters
   into C-style. With method #2 we interprete C-style strings. */
{	int num;
	const char *p;

	assert(fp);
	assert(s);

	if(length) {
		--s;
		do {
			if(*++s == '\\') {
				if(!--length)
					return;
				switch(*++s) {
				case 'x': case 'X':
					num = 0;
					while(isxdigit(*++s) && length > 0) {
						num = (num << 4) + *s
						 - (isdigit(*s)? '0'
							: (isupper(*s)? 'A' - 10
							   : 'a' - 10));
						--length;
					}
					--s;
					writeChar(num, fp);
				case '\n': break;
				case '0': case '1': case '2': case '3': case '4':
				case '5': case '6': case '7': case '8': case '9':
					num = 0;
					while(isdigit(*++s) && *s < '8' && length > 0) {
						num = (num << 3) + *s - '0';
						--length;
					}
					--s;
					writeChar(num, fp);
					break;
				default:
					if((p = strchr(Cfrom, *s)) == NULL)
						writeChar(*s, fp);
					else writeChar(Cto[p - Cfrom], fp);
					break;
				}
			}
			else writeChar(*s, fp);
		} while(--length);
	}
}

FILE *openDefFile(char *head, int errNr)
/* open next message definition file with nme head and error level errNr */
{	FILE *fp;

	assert(head);

	sprintf(msgDefFileStr, msgDefFile, "", msgDefFileNr++, ".c");
	fp = Efopen(msgDefFileStr, "wt");

	fprintf(fp, msgOpening, locHead);
	if(*head == MSG_ID_ERROR) 
		fprintf(fp, msgErrOpening, errNr);
	fprintf(fp, msgMsgDef, head);

	msgDefColumn = 0;

	return fp;
}

void closeDefFile(FILE *fp)
/* close a message definition file */
{	assert(fp);

	fputs(msgClosing, fp);
	Efclose(fp, msgDefFileStr);
}


/*
 *	Write a message declaration
 */
void writeMsgDcl(FILE *dclFp, char *head, int group)
{	
	assert(dclFp);
	assert(head);

	if(method == 2) {			/* aquire numerical message ID */
		lastID = addMsgID(msgIDs, head, strlen(head), group);
		if(lastID > dclCnt)
			dclCnt = lastID;
	}
	else lastID = ++dclCnt;
	fprintf(dclFp, msgDCL, head, head, group, lastID);
	if(lastID >= 0x1000)
		fatal(E_tooManyMsg);
}


/*
 *	Write a special format id with a length number
 */
void writeSpec(FILE *fp, char id)
{	char buf[sizeof(specId)];

	assert(fp);

	sprintf(buf, specId, id);
	writeMsgBody(fp, buf, strlen(buf));
}
void writeSpecNum(FILE *fp, char id, unsigned length)
{	
	assert(fp);

	writeSpec(fp, id);
	if(!length) featNUL = !NUL;
	if(method == 2)
		putc(length, fp);
	else fprintf(fp, "'\\%o', ", length);
}


/*
 *	Write the C stuff to inlcude a message
 */
void writeMsgInc(FILE *fp, char *p)
{	struct MsgIDhash hash;

	assert(fp);
	assert(p);

	writeSpec(fp, 'm');
	switch(method) {
	case 1:
		fprintf(fp, msgInclude, msgIncCnt, p, msgIncCnt);
		++msgIncCnt;
		msgDefColumn = 0;
		break;
	case 2:
		if(!getSymNam(msgIDs, p, strlen(p), aS(hash))) {
			warning(W_inclMSGID, p, lineNr);
			hash.msgid = 0;
		}
		writeMsgBody(fp, (byte*)&hash.msgid, sizeof(hash.msgid));
		break;
	}
	featNUL = !NUL;		/* as we cannot assume what the compiler puts here */
}

/*
 *	Write a string but chunk it into pieces of maximal 255 characters
 */
void writeChunked(FILE *fp, char id, char *str, int i)
{	
	assert(fp);
	assert(str);

	if(i) {			/* is there a string at all? */
		while(i > 255) {
			writeSpecNum(fp, id, 255);
			writeMsgBody(fp, (byte*)str, 255);
			str += 255;
			i -= 255;
		}
		writeSpecNum(fp, id, i);
		writeMsgBody(fp, (byte*)str, i);
	}
}
void writeChunkedBody(FILE *fp, char id, char *str)
{
	assert(fp);
	assert(str);

	writeChunked(fp, id, str, strlen(str));
}

/*
 *	Create the initial part of the binary file for method #2
 *	Input: 
 *		msgBody: the country code and the list of aliases
 *		ms2Fp: the opened file to dump the info into
 *	The following steps are performed:
 *	1) Dumping a dummy structure #1 
 *	2) Dumping all aliases
 *	3) Re-dumping structure #1 leaving the member "next" unoccupied
 *	4) Seeking to the end of the file and save the file position
 *	5) Dumping structure #2 leaving the member "num" unoccupied
 */
void createMsg2(const char *fn, const char *msg)
{	struct MsgStruct2 msg2;
	struct MsgStruct1 msg1;
	struct MsgStruct3 msg3;
	char *p;
	word code;
	int i;

	assert(fn);
	assert(msg);

	memset(aS(msg1), 0, sizeof(msg1));
	setMode(ms2Fp, TF_WR);
	Ewrite(aS(msg1), sizeof(msg1), ms2Fp, ms2Fn);
	msg1.ms1_magicnumber = MSG_MN_STRU1;
	msg1.ms1_groups = 1;		/* we define only ONE group per run */
	if((p = getNum(msgBody, &code)) == NULL)	/* no code */
		error(E_noCountryCodes, fn, msg);
	msg1.ms1_lng = code;
	while(*p) {
		if((p = getNum(p, &code)) == NULL)	/* no code */
			error(E_illformedCountryCodes, fn, msg);
		Ewrite(&code, sizeof(code), ms2Fp, ms2Fn);
		++msg1.ms1_aliases;
	}
	Eseek(0, SEEK_SET, ms2Fp, ms2Fn);
	Ewrite(aS(msg1), sizeof(msg1), ms2Fp, ms2Fn);
	Eseek(0, SEEK_END, ms2Fp, ms2Fn);

	msg2.ms2_gid = group;
	longset(msg2.ms2_ms3, 0);	/* as exactly ONE group is parsed by
									the message compiler and, thus, exactly
									ONE struct2 is written, the 1st struct3
									is immediately following it. */
	Ewrite(aS(msg2), sizeof(msg2), ms2Fp, ms2Fn);
	Egetpos(firstStru3, ms2Fp, ms2Fn);	/* store for newMsg2() */

	/* possibly there are num<->sym msgID mapping information. We
		must make sure that all pre-defined messages do have an stru3
		entry. It's possible that the mapping info is incomplete
		and some numericals are not defined with the file. */

	if(msgIDs && (i = msgIDs->grpIDs[group]) != 0) {
		dclCnt = i;	/* Is used to calculate the next unused entry */
		memset(aS(msg3), 0, sizeof(msg3));	/* mark all stru3's as undefined */
		do Ewrite(aS(msg3), sizeof(msg3), ms2Fp, ms2Fn);
		while(--i);
	}
	setMode(ms2Fp, TF_RD);
}


/*
 *	If this is _not_ the first message, we write the structure #3
 *	of the _previous_ message.
 *	Then we save the current errNr and the start of the message
 *	string.
 */
void newMsg2(char *head, int errNr)
{	dword pos, tmp;
	static char *prevName = NULL;
	static int prevID;
#define msgFp ms2Strg->fp
#define msgFn ms2Strg->fn

	assert(ms2Fp);
	assert(msgFp);
	assert(msgFn);

	Etell(aS(pos), msgFp, msgFn);
	if(prevName) {			/* not the first message */
		longcpy(tmp, pos);
		longsub(tmp, msg3.ms3_string);
		if(DW_HI(tmp))
			error(E_msgTooLong, prevName);
		msg3.ms3_length = DW_LO(tmp);
		/* Seek to the lastID's stru3. The ms2Fp file holds:
			1x stru1
			1x stru2
			Nx stru3

			N == msgIDs.grpIDs[group]

			lastID is the stru3 we must overwrite
		*/
		longsetu(tmp, sizeof(MsgStru3));
		longmulu(tmp, prevID - 1);		/* msgIndex > 0! */
		longadd(tmp, firstStru3);
#ifdef TEST
fprintf(LOG, "Dump Msg #0x%04x \"%s\" errNr %u @0x%lx\n", prevID
	, prevName, msg3.ms3_exitcode, tmp);
#endif /*#	defined(TEST) */
		Esetpos(tmp, ms2Fp, ms2Fn);
		setMode(ms2Fp, TF_WR);
		Ewrite(aS(msg3), sizeof(msg3), ms2Fp, ms2Fn);
		setMode(ms2Fp, TF_RD);
	}
	prevID = lastID;		/* preserve the current values */
	newstr(prevName, head);
	longcpy(msg3.ms3_string, pos);
	msg3.ms3_exitcode = errNr;
	msg3.ms3_class = *head;
#undef msgFp
#undef msgFn
}


/*
 *	Finalize the binary file ms2Fp
 *	+ flush the last structure #3
 *	+ Update date structure #1 and all structures #2's
 */
void closeMsg2(char *lng)
{	dword ofs;			/* offset of the message strings */
	dword tmp;
	dword strLength;
	FILE *fp;			/* final file */
	char *fn;			/* final filename */
	struct MsgStruct1 msg1;
	struct MsgStruct2 msg2;
	word code;
#define msgFp ms2Strg->fp
#define msgFn ms2Strg->fn

	assert(msgFp);
	assert(msgFn);
	assert(lng);

	if(!numCharsets)
		error(E_noCharset);

	fn = mkFname(lng, extMethod2);
	informative(M_creatMsgFile, fn);
	fp = Efopen(fn, "wb");

/* append the numeric <-> symbolic message ID mapping */
	dumpBinMsgIDS(msgIDs, fp, fn);

/* append the language definition */
	/* 1st, dump the pending message */
	newMsg2(NULL, 0);

	/* 2nd, copy & update the control area */
	Etell(aS(ofs), ms2Fp, ms2Fn);	/* length of control area of this group */
	Etell(aS(strLength), msgFp, msgFn);	/* length of string area */

	/* bring both temporary files into read mode.
		Because Micro-C has some problems with Read & Write files,
		we revoke the write permission before we actually seek back
		to the beginning of the files. */
	setMode(msgFp, TF_RD);
	Eseek(0, SEEK_SET, ms2Fp, ms2Fn);
	Eseek(0, SEEK_SET, msgFp, msgFn);

	/* begin to copy the control area */
	/* first the structure #1 and we replace the next field with
		the correct value! */
	Eread(aS(msg1), sizeof(msg1), ms2Fp, ms2Fn);
	longcpy(msg1.ms1_next, ofs);		/* length of control area */
	longadd(msg1.ms1_next, strLength);	/* Add the length of string area */
	longset(tmp, sizeof(word) * numCharsets);	/* add the length of the */
	longadd(msg1.ms1_next, tmp);		/* character sets */
	msg1.ms1_charsets = numCharsets;
	Ewrite(aS(msg1), sizeof(msg1), fp, fn);

	Ewrite(charsets, DW_LO(tmp), fp, fn);	/* append the character sets */

	/* second the alias codes */
	while(msg1.ms1_aliases--) {
		Eread(&code, sizeof(code), ms2Fp, ms2Fn);
		Ewrite(&code, sizeof(code), fp, fn);
	}

	/* third the structure #2 and we fill it with values */
	Eread(aS(msg2), sizeof(msg2), ms2Fp, ms2Fn);
	msg2.ms2_num = dclCnt;
	Ewrite(aS(msg2), sizeof(msg2), fp, fn);

	/* forth the structures #3 and we correct the "string" member.
		Initially "string" is the offset within the temporary file
		holding the message strings. They are dumped immediately
		following the last struc3. Within the struc2 the "string"
		is relatively referenced to the end of the struc2, that means
		the final value is:
			offset_of_first_string - offset_of_end_of_struct2
			+ offset_from_first_string_to_current_string

		Offset of the first string is already saved within 'ofs'.
		The other one is the offset returned by ftell() when the
		structure has been read out of the stream.
		The third one is already stored within the "string" member.
	*/
	code = 0;
	while(++code <= dclCnt) {
		Eread(aS(msg3), sizeof(msg3), ms2Fp, ms2Fn);
		Etell(aS(tmp), ms2Fp, ms2Fn);	/* offset of end of struc2 */
		longadd(msg3.ms3_string, ofs);	/* add the displacement */
		longsub(msg3.ms3_string, tmp);	/* subtract the ofs to 1st string */
		Ewrite(aS(msg3), sizeof(msg3), fp, fn);
	}

	/* 3rd, append all the strings
		we buffer the read stuff within a dynamic buffer */
	copyfilel(strLength, fp, fn, msgFp, msgFn);

	Efclose(fp, fn);
	Efclose(msgFp, msgFn);
	remove(msgFn);
	Efclose(ms2Fp, ms2Fn);
	remove(ms2Fn);
#undef msgFp
#undef msgFn
}

/************************************************************************/
/************ Scanning functions ****************************************/
/************************************************************************/

/*
 *	Search an (char *[]) array for a string
 *	Return:
 *		-1: not found
 *		else: index into array
 */
int searchArr(char *arr[], const char str[], unsigned size)
{	int nr;

	assert(arr);
	assert(str);

	nr = size + 1;
	while(--nr) {
		assert(*arr);
		if(strcmp(*arr++, str) == 0)
			return size - nr;
	}
	return -1;
}

int internalName(char **str)
{	int i, c;
	char *s;

	assert(str && *str);

	s = *str;
	if(!*s) return -1;

	while(isalpha(*s)) ++s;
	c = *s;
	*s = NUL;
	if((i = searchArr(internal, *str, keywords)) < 0)
		i = searchArr(internalE, *str, keywords);
	*s = c;

	if(i < 0) return -1;	/* error */

	while(isspace(*s)) ++s;
	*str = s;
	return i;
}

unsigned translateANSI(char *dest, char *src)
/* translate a ANSI sequence into its escape code representation */
{	int i, j, flag;
	char *s, ansi[32], nam;
	unsigned len;

	assert(dest);
	assert(src);

	len = 0;
	if((s = src = strtok(src, ",")) != NULL) do {
		while(isspace(*src)) ++src;
		if((i = internalName(&s)) < 0)
			error(E_ANSIToken, src, lineNr);

#define sAnsi(a) stpcpy(ansi, a); break
#define tAnsi(a) nam = a; goto singleArg
#define cAnsi(a) nam = a; goto colourArg
#define aAnsi(a) nam = a; goto argArg
		switch(i) {
			case IN_cls: 		sAnsi("2J");
			case IN_clreol: 	sAnsi("K");
			case IN_home: 		sAnsi("H");
			case IN_save:		sAnsi("s");
			case IN_restore:	sAnsi("u");
			case IN_up:		tAnsi('A');
			case IN_down:		tAnsi('B');
			case IN_left:		tAnsi('D');
			case IN_right:		tAnsi('C');
			singleArg:
				if((i = copyNumber(ansi, &s, nam)) == 1 || i > 15)
					error(E_ANSInumericArg, src, lineNr);
				break;
			case IN_goto:
				if((i = copyNumber(ansi, &s, ';')) == 1 || i > 31
				  || (j = copyNumber(ansi + i, &s, 'H')) == i + 1 || j > 31)
					error(E_ANSIdoubleNumericArg, src, lineNr);
				break;
			default:
				flag = j = 0;	/* first colour is foreground */
				do switch(i) {
					case IN_bright:	aAnsi(1);
					case IN_blink:		aAnsi(5);
					case IN_inverse:	aAnsi(7);
					case IN_underline:	aAnsi(4);
					case IN_hidden:	aAnsi(8);
					case IN_normal:	aAnsi(0);
					case IN_black:		cAnsi(0);
					case IN_red:		cAnsi(1);
					case IN_green:		cAnsi(2);
					case IN_yellow:	cAnsi(3);
					case IN_blue:		cAnsi(4);
					case IN_magenta:	cAnsi(5);
					case IN_cyan:		cAnsi(6);
					case IN_white:		cAnsi(7);
					colourArg:
						nam |= flag? 0x40: 0x30;
						flag = 1;
					argArg:
						if(j < 29) {
							if(nam & 0xf0)
								ansi[j++] = 0x30 | (nam >> 4);
							ansi[j++] = 0x30 | (nam & 0xf);
							ansi[j++] = ';';
							break;
						}
					default:
						error(E_ANSIToken, src, lineNr);
				} while((i = internalName(&s)) >= 0);
				ansi[j - 1] = 'm';
				ansi[j] = NUL;
				break;
		}
		if(*s)
			error(E_ANSIToken, src, lineNr);
		len += 2 + strlen(ansi);
		dest = stpcpy(stpcpy(dest, "\33["), ansi);
	} while((src = s = strtok(NULL, ",")) != NULL);
	return len;
}


/*
 *	Read a pair of words
 */
void readWrdPair(char **Xs, char **w1, char **w2)
{	char *s, *e;

	assert(Xs && *Xs);
	assert(w1);
	assert(w2);

	*w1 = *w2 = NULL;
	e = eow(s = skipws(*Xs));
	*w1 = strdupe(s, e);
	s = skipws(e);
	*w2 = strdupe(s, e = eol(s));
	if(!*w1 || !*w2 || !**w1)
		error(E_synSKeyword, *Xs);
	*Xs = e;
}

/*
 *	Read the current character set internal message and preserve its
 *	contents for later dumping into the binary message library.
 *	It supports (case-insensitive):
 *		0x<headecimal number>:	DOS code page number
 *		<decimal number>: DOS code page number
 *		CP<decimal number>: DOS code page number
 *		7bitASCII: characters of the range 0x00..0x7e
 *		UNICode16: UNI Code 16bit
 *	The message compiler is allowed to automatically add "7bitascii" to
 *	the list of character sets. The special 8BITASCII disables this feature.
 */
void scanCharSets(void)
{	word code;
	char *p, *e, *q;
#define chkCS(cp) ((e - p) == sizeof(cp) - 1 && memcmp(p, cp, sizeof(cp) - 1) == 0)

	strupr(e = msgBody);		/* make string compares more easier */
	while(*(q = p = skipws(e))) {
		e = eow(p);
		switch(*p) {
		case 'C':
			if(p[1] != 'P' || !isdigit(p[2]))
				goto errRet;
			p += 2;
			/* fall through */
			/* However, we support CP0xnumber, It's no bug, it's a feature! */
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '9':
		codepage:
			code = 0;
			if(p[1] == 'X') {	/* hexadecimal */
				++p;
				while(isxdigit(*++p))
					code = (code << 4) + *p - (isdigit(*p)? '0': 'A' - 10);
			}
			else
				do code = code * 10 + *p - '0';
				while(isdigit(*++p));
			if(p != e) goto errRet;
		addCode:
			if((charsets = realloc(charsets, sizeof(word) * ++numCharsets))
			 == NULL)
			 	error(E_noMem);
			charsets[numCharsets - 1] = code;
			break;

		case '7': 
			if(isdigit(p[1])) goto codepage;
			if(!chkCS("7BITASCII")) goto errRet;
			code = CHARSET_7BITASCII;
			autoCharSet = 0;
			goto addCode;

		case '8': 
			if(isdigit(p[1])) goto codepage;
			if(!chkCS("8BITASCII")) goto errRet;
			autoCharSet = 0;
			goto addCode;

		case 'U':
			if(chkCS("UNICODE16")) {
				code = CHARSET_UNICODE16;
				goto addCode;
			}

		case 'N':
			if(chkCS("NONE")) {
				code = CHARSET_NONE;
				goto addCode;
			}

		errRet:
			*e = '\0';
			error(E_unknownCS, q);
		}
	}
#undef chkCS
}



/*
 *	Read the global message description files and extract the internally
 *	used names into internal[] and internalE[].
 *	If method #2 parsing, prepare the binary file as soon as the
 *	first I_countrycodes message is found in the global language
 *	description file.
 */
void readGlbFile(char *path, char *lng, FILE *dclFp)
{	char *fi, *fn, *head, *p, *h;
	FILE *fp;
	int nr;
	int aliasesFound;

	assert(path);
	assert(lng);
	assert(dclFp);

	if(!dfnsplit(lng, NULL, NULL, &fi, NULL)
	 || (fn = dfnmerge(NULL, NULL, path, fnMsg, extMsg)) == NULL)
		error(E_noMem);

	/* Read the keyword table */
	fp = Efopen(fn, fmode);	/* global English message file */
	lineNr = 0;
	readMsgBody(fp);	/* skip header */

    while((head = readMsgHeader(fp)) != NULL)
		if(*head == MSG_ID_INTERNAL && strcmp(head, iKeywords) == 0) {
			/* message containing the English keywords. This file also
				contains the core information of the keywords:
					numerical & symbolical ID
				The message is structured like this:
				1st line: amount of keywords (numerical)
				...     : Any line starting with no digit -> comment
				Nth line: <numerical ID> <symbolical ID> <English keyword>

				The numerical ID is the index where to store the keyword and
				must be equal to the corresponding IN_???? macro.
				The symbolical ID links the keyword to the one of the
				keyword in the other languages.

				This message may appear more than once in the file, but
				the first line is ignored for all messages except the
				first one.
			*/
			
			if(!keywords) {
				if((keywords = atoi(msgBody)) <= 0)
					error(E_keywords);
				internalE = (char**)cgetmem(keywords * sizeof(char *));
				internal  = (char**)cgetmem(keywords * sizeof(char *));
			}

			head = strchr(msgBody, '\n');
			while(head && *++head) {			/* advance to next line */
				if(isdigit(*head)) {					/* keyword line */
					nr = *head - '0';
					while(isdigit(*++head))
						nr = nr * 10 + *head - '0';
					if((unsigned)nr >= keywords)
						error(E_numKeyword, nr);
					if(!isspace(*head))
						error(E_synKeyword, nr);
					readWrdPair(&head, &internal[nr], &internalE[nr]);
				}
				head = strchr(head, '\n');
			}
		}

	Efclose(fp, fn);
	free(fn);

	if(!keywords)
		fatal(E_noKeywords);

	for(nr = 0; nr < keywords; ++nr)
		if(!internalE[nr])
			error(E_missingInternal, nr);

	if((fn = dfnmerge(NULL, NULL, path, fi, extMsg)) == NULL)
		error(E_noMem);

	/* Read the language specific keyword table and the country codes */
	fp = Efopen(fn, fmode);		/* global language specific message file */
	lineNr = 0;
	aliasesFound = 0;
	readMsgBody(fp);	/* skip header */

    while((head = readMsgHeader(fp)) != NULL) {
    	if(strcmp(head, iCharSets) == 0)	/* ignore character sets */
    		continue;
    	if(group != GLOBAL_GROUP)
			writeMsgDcl(dclFp, head, GLOBAL_GROUP);
    	if(*head != MSG_ID_INTERNAL)
    		continue;
		if(strcmp(head, iKeywords) == 0) {
			head = msgBody - 1;
			while(head && *++head) {			/* advance to next line */
				if(memcmp(head, symID, sizeof(symID) - 1) == 0) {
					readWrdPair(&head, &p, &h);
					if((nr = searchArr(internal, p, keywords)) < 0)
						error(E_noKeyword, p);
					free(internal[nr]);
					internal[nr] = h;
					free(p);
				}
				head = strchr(head, '\n');
			}
		}
		else if(group != GLOBAL_GROUP && strcmp(head, iFeatures) == 0) {
			/* read the features of the global messages */
			head = msgBody - 1;
			while(head && *++head) {
				nr = 0;
				do {
					if(memcmp(featDef[nr], head, strlen(featDef[nr])) == 0) {
						features[nr] = !NUL;
						break;
					}
				} while(++nr < FEATURES);
				head = strchr(head, '\n');
			}
		}
		else if(method == 2 && strcmp(head, iAliases) == 0) {
		/* language specific codes */
			if(aliasesFound)
				fatal(E_multipleCountryCodes, fn, iAliases);
			aliasesFound = 1;
			createMsg2(fn, iAliases);
		}
	}

	if(method == 2 && !aliasesFound)
		fatal(E_noCountryCodes, fn, iAliases);
	Efclose(fp, fn);
	free(fn);
	free(fi);
}

/*
 *	See msgIDs.h
 */
#ifdef _MICROC_
#define fp arg->fp
#define fn arg->fn
#else /*#	!(defined(_MICROC_)) */
#define fp ((struct DumpMsgID*)arg)->fp
#define fn ((struct DumpMsgID*)arg)->fn
#endif /*#	defined(_MICROC_) */
static int scanNxtMsgID(inM(struct DumpMsgID, void) *arg
	, MSGID2 *id, char **name, int *length, int *buflen)
{	char *p;
	unsigned num;

	assert(arg);
	assert(id);
	assert(name);
	assert(length);
	assert(buflen);

	if(*buflen != BUF + 1)
		*name = chgmem(*name, *buflen = BUF + 1);

	do {
		if(!fgets(p = *name, BUF, fp))
			return 0;
#ifndef _MICROC_		/* MC strips '\n' */
		if(!strchr(p, '\n'))
			error(E_longLine, fn, BUF);
#endif /*#	!defined(_MICROC_) */
	} while(*p == '#' || *p == ':' || !isprint(*p));		/* comments */

	while(isalnum(*p) || *p == '_')
		++p;

	if((*length = p - *name) < 3 || !isspace(*p))
		error(E_syntaxID, fn, *name);

	if(!getNum(p, &num))
		error(E_syntaxID, fn, *name);

	*id = num;

	return 1;
}

static int dumpNxtMsgID(inM(struct DumpMsgID, void) *arg
	, MSGID2 id, char *name, int length)
{	assert(name);

	Ewrite(name, length, fp, fn);
	fprintf(fp, " 0x%04x\n", id);
	if(ferror(fp))
		error(E_writeFile, fn);
	return 1;
}
#undef fp
#undef fn

void readSharedMsgIDs(const char *fn)
{
	struct DumpMsgID arg;

	assert(fn);

	if(fileExists(fn)) {
		informative(M_scanMsgIDs);
		arg.fp = Efopen(arg.fn = fn, "rt");
		msgIDs = scanMsgIDS(NULL, aF(scanNxtMsgID), aS(arg));
		Efclose(arg.fp, arg.fn);
	}
	else msgIDs = scanMsgIDS(NULL, NULL, NULL);
}

void dumpSharedMsgIDs(const char *fn)
{	char *bak;
	struct DumpMsgID arg;

	assert(fn);
	
	if(fileExists(fn)) {
		/* create backup */
		bak = mkFname(fn, extBak);
		remove(bak);
		rename(fn, bak);
		if(fileExists(fn))
			error(E_renameFile, fn, bak);
		free(bak);
	}
	arg.fp = Efopen(arg.fn = fn, "wt");
	dumpMsgIDS(msgIDs, aF(dumpNxtMsgID), aS(arg));
	Efclose(arg.fp, arg.fn);
}

/*
 *	Dump a lower-cased defFile for use with DMake's Makefile.
 *	DMake 7-bit-lower-cases a globbed filename, but performs a case-
 *	sensitive match on that globbed name and the name within a Makefile.
 */
void lowerCaseDefFile(FILE *fp, char *prefix, int num, char *postfix)
{	char buf[sizeof(msgDefFile) + 10];

	assert(fp);
	assert(prefix);
	assert(postfix);

	sprintf(buf, msgDefFile, prefix, num, postfix);

	assert(strlen(buf) < sizeof(buf));

	strlwr(buf);
	fputs(buf, fp);
}

void errRet(void)
{	clMsgIDs(msgIDs);
	msgIDs = NULL;
	if(ms2Strg) {
		clTempFile(ms2Strg);
		ms2Strg = NULL;
	}
	if(ms2Ctrl) {
		clTempFile(ms2Ctrl);
		ms2Ctrl = NULL;
	}
}


#undef msgFp
#undef msgFn

char *opt1st = "!B";
char *opt2nd = "CFGILM";
main(int argc, char **argv)
{	char *head, *tail, *p;
	FILE *fp, *dclFp, *msgFp;
	int c;

    char  *path				/* path to the global message files */
    	 ,*lng				/* language to be compiled */
    	 ,*cRspFn			/* compiler response file */
    	 ,*lRspFn			/* librarian response file */
    	 ,*mRspFn			/* Makefile dependency file */
    	 ,*featFn			/* used feature logfilename */
    	 ,*dclFn			/* message declaration filename */
		 ;
	const char
		  *idFn				/* method #2 num<->sym msgID mapper */
    	 ;

    unsigned i;

    msgInit();
    msgErrFct(errRet);

#ifdef TEST
	puts("MSGCOMP2: TEST support enabled!");
	if((LOG = fopen("msgComp2.log", "wt")) == NULL) {
		puts("Could not open logfile \"msgComp2.log\"");
		exit(126);
	}
#endif /*#	defined(TEST) */

	idFn = cRspFn = lRspFn = mRspFn = featFn = dclFn = NULL;
	method = 1;

	while((int)(i = getoptG(argc, argv)) != EOF) switch(i) {
	default:	hlpScreen();
	case '!':				/* use msg!#### instead of msg_#### */
		msgDefFile[padDefFile] = '!';
		group = GLOBAL_GROUP;
		break;
	case 'B':				/* use binary mode */
		binary = !binary;
		break;
	case 'C':				/* compiler response file */
		cRspFn = optarg;
		break;
	case 'L':				/* librarian response file */
		lRspFn = optarg;
		break;
	case 'M':				/* Makefile dependencies */
		mRspFn = optarg;
		break;
	case 'I':				/* num<->sym msgID mapper */
		idFn = optarg;
		break;
	case 'G':				/* message group */
		if((group = atoi(optarg)) < 0 || group > 15)
			error(E_rangeOfGroup);
		break;
	case 'F':				/* output file format */
		if((method = atoi(optarg)) < 1 || method > 2)
			error(E_rangeOfMethod);
		break;
	}

	if(binary)								/* read in binary mode */
		fmode[1] = 'b';

	switch(method) {
	case 2:						/* no response files */
		lRspFn = mRspFn = cRspFn = NULL;
		if(!idFn)
			informative(M_dfltMsgIDfn, idFn = fnMsgID);
		break;
	case 1:
		idFn = NULL;
		break;
	}

	switch(argc - optind) {
	default:	hlpScreen();
	case 5:		featFn = argv[optind + 4];
	case 4:		dclFn = argv[optind + 3];
	case 3:		locHead = argv[optind + 2];
				lng = argv[optind + 1];
				path = argv[optind + 0];
				break;
	}

	if(!dclFn)
		dclFn = mkFname(lng, extDcl);
	dclFp = Efopen1(dclFn);

	dclCnt = lineNr = 0;

	if(method == 2) {
		ms2Ctrl = tmpFile(TF_RW);
		setMode(ms2Fp, TF_RD);
		ms2Strg = tmpFile(TF_RW);
		msgFp = ms2Strg->fp;
#define	msgFn ms2Strg->fn
		readSharedMsgIDs(idFn);
	}

	/* read internally used keywords and the features and declarations
		and aliases out of the global message file */
	informative(M_readInternal);
	readGlbFile(path, lng, dclFp);


	/* read message file & create the message definition files */
	lng = attachExt(lng, extMsg);
	informative(M_readMsgFile, lng);
	fp = Efopen(lng, fmode);

	readMsgBody(fp);	/* skip header */

	while((head = readMsgHeader(fp)) != NULL) {
		if(strcmp(head, iCharSets) == 0) {
			/* either scan or ignore completely */
			if(method == 2)		/* scan & preserve character sets */
				scanCharSets();
			continue;
		}

		writeMsgDcl(dclFp, head, group);
		switch(method) {	/* initialize new message */
		case 1:	msgFp = openDefFile(head, errNr); break;
		case 2: newMsg2(head, errNr); break;
		}
		p = (tail = msgBody) - 1;
		while((p = strchr(p + 1, '%')) != NULL) {	/* specil format spec */
			if(p != tail) {			/* emit previous text */
				writeMsgBody(msgFp, (byte*)tail, p - tail);
				tail = p;
			}

			/* tail points to the opening percent sign '%', p to the
				working place behind it */

			if(*++p == '%') {	/* skip %% */
				continue;
			}
			if(*p == '-') {
				if(p[1] == '%') {		/* %-% found : ANSI enclosure */
					featAnsi = !NUL;
					*(p = eoe(p + 2, '%')) = NUL;
					writeChunked(msgFp, 'A', tail
					 , translateANSI(tail, tail + 3));
					tail = p + 1;
					continue;
				}
				else if(p[1] == '-') {
					if(p[2] == '%') {
						switch(*(p += 3)) {
						case 'A':
							featAnsi = !NUL;
							goto copyChunk;
						case 'W':
							featRaw = !NUL;
						copyChunk:
							c = *++p;
							*(p = eoe(tail = p, c)) = NUL;
							writeChunkedBody(msgFp, 'A', tail);
							tail = p + 1;
							break;
						case 'r':				/* %--%r?###% */
							featRep = !NUL;
							tail = eoe(p + 2, '%') + 1;
							if(!(unsigned)(i = atoi(p + 2)))
								error(E_numberZero);
							if(p[1] >= 0x7f)
								feat8bit = !NUL;
							while((unsigned)i > 255) {
								writeSpecNum(msgFp, 'r', 255);
								putc(p[1], msgFp);
								i -= 255;
							}
							if(i) {
								writeSpecNum(msgFp, 'r', i);
								putc(p[1], msgFp);
							}
							break;
						case 'R':				/* %--%R###% */
							featRepArg = !NUL;
							tail = eoe(p + 1, '%') + 1;
							if((unsigned)(i = atoi(p + 1)) > 255 || !i)
								error(E_invalidSpec, 'R');
							writeSpecNum(msgFp, 'R', i);
							break;
						case 'm':				/* %--%m****% */
							featMInc = !NUL;
							tail = eoe(++p, '%');
							*tail++ = NUL;
							writeMsgInc(msgFp, p);
							break;
						case 'M':				/* %--%M */
							featMIncArg = !NUL;
							writeSpec(msgFp, 'M');
							tail = p + 1;
							break;
						}
					}
					continue;
				}
				++p;
			}
			while(isdigit(*p)) ++p;
			if(*p == '%') {
				--p;
				featReorder = !NUL;
			}
		}
		writeMsgBody(msgFp, (byte*)tail, strlen(tail));
		if(method == 1)
			closeDefFile(msgFp);
	}
	Efclose(fp, lng);
	Efclose(dclFp, dclFn);

	if(!feat8bit && autoCharSet) {
		/* 7bit ASCII */
		if((charsets = realloc(charsets, sizeof(word) * ++numCharsets))
		 == NULL)
		 	error(E_noMem);
		charsets[numCharsets - 1] = CHARSET_7BITASCII;
	}

	if(method == 2) {		/* finaly compose the binary message file */
		closeMsg2(lng);
		dumpSharedMsgIDs(idFn);
	}
	else {

		/* create the compiler response file */
		if(cRspFn) {	/* response file */
			informative(M_createCompRsp);
			fp = Efopen1(cRspFn);
			for(i = 0; i < msgDefFileNr; ++i)
				fprintf(fp, msgDefFile, "", i, ".c\n");
			Efclose(fp, cRspFn);
		}
		else warning(W_compRsp);

		/* create the librarian response file */
		if(lRspFn) {	/* response file */
			informative(M_createLibRsp);
			fp = Efopen1(lRspFn);
			head = "+ ";
			for(i = 0; i < msgDefFileNr; ++i)
				fprintf(fp, msgDefFile, head, i, ""), head = ".obj &\n+ ";
			if(*head == '.') fputs(".obj\n", fp);
			Efclose(fp, lRspFn);
		}
		else warning(W_libRsp);

		/* create the Makefile dependency file */
		if(mRspFn) {	/* response file */
			informative(M_createMakedep);
			strlwr(locHead);		/* for DMake's case-sensitive math against
										lower-cased globbed filenames. */
			fp = Efopen1(mRspFn);
			fputs("all : ", fp);	/* create the default target */
			for(i = 0; i < msgDefFileNr; ++i)
				lowerCaseDefFile(fp, " \\\n\t", i, ".obj");
			fputs("\n\n\n", fp);
			for(i = 0; i < msgDefFileNr; ++i) {
				lowerCaseDefFile(fp, "", i, ".obj");
				lowerCaseDefFile(fp, " : ", i, ".c ");
				fputs(locHead, fp);
				putc('\n', fp);
				putc('\n', fp);
			}
			Efclose(fp, mRspFn);
		}
		else warning(W_makRsp);
	}

	/* create the feature file */
	if(!featFn)
		featFn = mkFname(lng, extFtr);
	informative(M_createFeature, featFn);
	fp = Efopen1(featFn);
	c = 0;
	do {
		if(features[c]) {
			fputs(featDef[c], fp);
			putc('\n', fp);
		}
	} while(++c < FEATURES);
	Efclose(fp, featFn);

	errRet();

#ifndef _MICROC_
	return 0;	/* standard for Micro-C */
#endif /*#	!defined(_MICROC_) */
#undef msgFn
}


/************************************************************************/
/************************************************************************/
/************************************************************************/

