/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    FILEIO - Common file I/O functions
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: FILEIO.H $
   $Locker: ska $	$Name:  $	$State: Exp $

	Wrapper functions of <stdio.h> functions.

	In general, where a (FILE*) is passed into a <stdio.h> function, the
	file's name is passed, too. On failure of the particular <stdio.h>
	function, an error message containing the filename is issued and
	the program is terminated.

   ----

   Defined functions:

	void setMode(FILE *fp, int mode)
		Micro-C has problems with seeking within file with granted write
		permission (WRONLY or RDWR). Therefore setMode() should encapsule
		non-sequential file access, or, to put it in other words, the
		file mode should be set to "read" by default and changed to "write"
		before some data is to be written.

	FILE *Efopen(char *fname, const char *mode)
		Open the file "fname" with the requested mode (same parameters
		as fopen()). It will generate an error message and terminate
		the program, when the file could not be opened.

	FILE *Efopen1(char *fn)
		Open file "fn" with mode "wt". If the first character of 'fn'
		is equal to '+', in which case "at" is used.

	void Efclose(FILE *fp, char *fnam)
		Same as fclose(). If ferror(fp) returns non-zero (meaning previous
		file IO had failed), an error is issued and the program is
		terminated.

	void Ewrite(void *buf, int size, FILE *fp, char *fnam)
		Wrapper of fwrite(). On failure an error message is issued and the
		program is terminated. The EINTR case is not handled.

	void Eread(void *buf, int size, FILE *fp, char *fnam)
		Wrapper of fread(). As Ewrite().

	void Eseek(int ofs, int whence, FILE *fp, char *fnam)
	void Esetpos(pos,fp,fn) Eseekx(pos,0,fp,fn)
	void Eseekc(pos,fp,fn) Eseekx(pos,1,fp,fn)
	void Eseekx(dword iM(*)pos, int whence, FILE *fp, char *fnam)
		Wrappers of fseek(). On failure an error message is issued and
		the program is terminated.
		Eseek() directly wraps fseek(), but with an _int_ offset.
		Esetpos() sets the file position (seeking from the beginning of
			the file). Opposite of Egetpos().
		Eseekc() seeks the file position based upon the current position.
		Eseekx() directly wraps fseek() with a (long) offset.

		The (dword) 'pos' parameters can be passed directly into these
		functions for both Micro-C and Borland C.

	void Egetpos(pos,fp,fnam) Etell(aS(pos), fp, fnam)
	void Etell(dword nM(*)pos, FILE *fp, char *fn)
		Wrapper of ftell(). On failure an error message is issued and the
		program is terminated.

		Egetpos() is the opposite of Esetpos() and retrieves the current
		file position. The 'pos' parameter must be a (dword) lvalue.
		Etell() is a direct wrapper of ftell(), 'pos' requires aS(pos) or
		something similiar if necessary.

	void Epeek(void *buf, int size, FILE *fp, char *fnam)
		Wrapper of fread(). On failure an error message is issued and
		the program is terminated.
		Epeek() reads the requested block and returns the same file
		position as before the read. It's recommended to set the file
		mode to 'read' if the file mode actually is 'read&write'.

	int fileExists(char *fnam)
		Check if the file 'fnam' exists.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
   Known bugs:

   ----
	
*/

#ifndef __FILEIO__H_
#define __FILEIO__H_

#ifndef _MICROC_
#include <io.h>			/* access() */
#endif /*#	!defined(_MICROC_) */

#ifndef CLROBJ_MACROS
#define CLROBJ_MACROS
/*
 *	Some useful definitions
 */
#define clrobj(obj) memset(aS(obj), 0, sizeof(obj))		/* Clear object */
#define clrpobj(obj) memset(obj, 0, sizeof(*obj))	/* Clear pointered object */
#define clraobj(obj) memset(obj, 0, sizeof(obj))	/* Clear array object */
#define clrarray(array,items) memset(array, 0, sizeof(array[0]) * items)
#endif /*#	!defined(CLROBJ_MACROS) */ 


/* Open/access modes */
#define TF_RD	0	/* read-only */
#define TF_WR	1	/* write-only */
#define TF_RW	2	/* read-write */


extern FILE *Efopen(const char *fname, const char *mode);
extern FILE *Efopen1(const char *fn);
extern void Efclose(FILE *fp, const char *fnam);
extern void Ewrite(void *buf, int size, FILE *fp, const char *fnam);
extern void Eread(void *buf, int size, FILE *fp, const char *fnam);
extern void Eseek(int ofs, int whence, FILE *fp, const char *fnam);
#define Esetpos(pos,fp,fn) Eseekx(pos,0,fp,fn)
#define Eseekc(pos,fp,fn) Eseekx(pos,1,fp,fn)
extern void Eseekx(dword iM(*)pos, int whence, FILE *fp, const char *fnam);
#define Egetpos(pos,fp,fnam) Etell(aS(pos), fp, fnam)
extern void Etell(dword nM(*)pos, FILE *fp, const char *fn);
extern void Epeek(void *buf, int size, FILE *fp, const char *fnam);

#ifdef _MICROC_
extern int fileExists(const char *fnam);
extern void setMode(FILE *fp, int mode);
#else /*#	!(defined(_MICROC_)) */ 
#define fileExists(fnam) (access(fnam, 0) == 0)
#define setMode(fp,mode)
#endif /*#	defined(_MICROC_) */ 

#ifndef _MICROC_
#define appendMode "at"
#else /*#	!(!defined(_MICROC_)) */
#define appendMode "wat"
#endif /*#	!defined(_MICROC_) */

#endif /*#	!defined(__FILEIO__H_) */
